module sub_calc
! サブ処理ルーチン

  use Derivation

contains

subroutine calc_wind( x, y, psi, u, v, undef )
! x, y と流線関数から水平風速を計算するルーチン

  implicit none

  real, intent(in) :: x(:)  ! x 方向の座標 [m]
  real, intent(in) :: y(:)  ! y 方向の座標 [m]
  real, intent(in) :: psi(size(x),size(y))  ! 流線関数 [m2/s]
  real, intent(inout) :: u(size(x),size(y))  ! x 方向速度 [m/s]
  real, intent(inout) :: v(size(x),size(y))  ! y 方向速度 [m/s]
  real, intent(in), optional :: undef  ! 未定義値

  integer :: i, j, k, nx, ny

  nx=size(x)
  ny=size(y)

  do j=1,ny
     call grad_1d( x, psi(:,j), v(:,j), undef=undef )
  end do

  do i=1,nx
     call grad_1d( y, psi(i,:), u(i,:), undef=undef )
  end do

  do j=1,ny
     do i=1,nx
        if(psi(i,j)/=undef)then
           v(i,j)=-v(i,j)
           u(i,j)=u(i,j)/x(i)
        else
           v(i,j)=undef
           u(i,j)=undef
        end if
     end do
  end do

end subroutine

subroutine bound_set( x, y, val, ib, bnd, undef )
! 境界条件を設定する.
  implicit none
  real, intent(in) :: x(:)  ! x 方向座標 [m]
  real, intent(in) :: y(:)  ! y 方向座標 [m]
  real, intent(inout) :: val(size(x),size(y))  ! 境界条件を設定する変数
  integer, intent(in) :: ib(size(x),size(y))  ! 内部を含む全境界条件
  real, intent(in) :: bnd(size(x),size(y))  ! 境界条件の値
  real, intent(in) :: undef  ! 計算内部領域での未定義値.

  integer :: i, j, nx, ny, ix, jy
  real, allocatable, dimension(:) :: dx, dy, x_inv

  nx=size(x)
  ny=size(y)

  allocate(dx(nx))
  allocate(dy(ny))
  allocate(x_inv(nx))

  x_inv=1.0/x

  do i=2,nx-1
     dx(i)=0.5*(x(i+1)-x(i-1))
  end do

  do j=2,ny-1
     dy(j)=0.5*(y(j+1)-y(j-1))
  end do

  do j=1,ny
     do i=1,nx
        select case (ib(i,j))
        case(1)
           val(i,j)=bnd(i,j)
        case(2)
           val(i,j)=val(i+1,j)-bnd(i,j)*dx(i)
        case(-2)
           val(i,j)=val(i-1,j)+bnd(i,j)*dx(i)
        case (3)  ! 周期境界
           if(i==1)then
              ix=nx-1
           else if(i==nx)then
              ix=2
           else
              ix=i
           end if
           if(j==1)then
              jy=ny-1
           else if(j==ny)then
              jy=2
           else
              jy=j
           end if
           val(i,j)=val(ix,jy)
        case(4)
           val(i,j)=val(i,j+1)-bnd(i,j)*dy(j)*x_inv(i)
        case(-4)
           val(i,j)=val(i,j-1)+bnd(i,j)*dy(j)*x_inv(i)
        case (8)  ! 両方フラックス一定で左下角か右上角
           if(i==1.and.j==1)then  ! -- 評価 1
              val(i,j)=val(i+1,j+1)-0.5*(bnd(i+1,j)*dy(j)*x_inv(i)+bnd(i,j+1)*dx(i))

           else if(i==nx.and.j==ny)then  ! -- 評価 2
              val(i,j)=val(i-1,j-1)+0.5*(bnd(i-1,j)*dy(j)*x_inv(i)+bnd(i,j-1)*dx(i))

           else if(ib(i-1,j)==10.or.ib(i,j-1)==10)then
              ! -- 評価 1 と同じ
              val(i,j)=val(i+1,j+1)-0.5*(bnd(i+1,j)*dy(j)*x_inv(i)+bnd(i,j+1)*dx(i))

           else if(ib(i+1,j)==10.or.ib(i,j+1)==10)then
              ! -- 評価 2 と同じ
              val(i,j)=val(i-1,j-1)+0.5*(bnd(i-1,j)*dy(j)*x_inv(i)+bnd(i,j-1)*dx(i))

           end if

        case (-8)  ! 両方フラックス一定で右下角か左上角
           if(i==1.and.j==ny)then  ! -- 評価 1
              val(i,j)=val(i+1,j+1)+0.5*(bnd(i+1,j)*dy(j)*x_inv(i)-bnd(i,j-1)*dx(i))

           else if(i==nx.and.j==1)then  ! -- 評価 2
              val(i,j)=val(i-1,j-1)+0.5*(-bnd(i-1,j)*dy(j)*x_inv(i)+bnd(i,j+1)*dx(i))

           else if(ib(i-1,j)==10.or.ib(i,j+1)==10)then
              ! -- 評価 1 と同じ
              val(i,j)=val(i+1,j+1)+0.5*(bnd(i+1,j)*dy(j)*x_inv(i)-bnd(i,j-1)*dx(i))

           else if(ib(i+1,j)==10.or.ib(i,j-1)==10)then
              ! -- 評価 2 と同じ
              val(i,j)=val(i-1,j-1)+0.5*(-bnd(i-1,j)*dy(j)*x_inv(i)+bnd(i,j+1)*dx(i))

           end if
        end select
     end do
  end do

end subroutine

subroutine set_omega_inner( x, y, psi, inner_bound, omega )
! psi の値をもとに, omega の内部境界値を設定する.
  implicit none
  real, intent(in) :: x(:)  ! x 方向座標 [m]
  real, intent(in) :: y(:)  ! y 方向座標 [m]
  real, intent(in) :: psi(size(x),size(y))  ! 境界条件設定についての参照変数
  integer, intent(in) :: inner_bound(size(x),size(y))  ! 内部を含む全境界条件
  real, intent(inout) :: omega(size(x),size(y))  ! 設定する境界条件の値が入る

  integer :: i, j, nx, ny
  real, allocatable, dimension(:) :: dx, dy, x2_inv

  nx=size(x)
  ny=size(y)

  allocate(dx(nx))
  allocate(dy(ny))
  allocate(x2_inv(nx))

  do i=2,nx-1
     dx(i)=0.5*(x(i+1)-x(i-1))
  end do

  do j=2,ny-1
     dy(j)=0.5*(y(j+1)-y(j-1))
  end do

  x2_inv=1.0/(x**2)

!-- 境界の設定方法
!-- 上 (+), 下 (-) : 壁では速度がゼロ.
!-- omega_壁 = -du/dy = -(u_{j+1}/dy), -(-u_{j-1}/dy)
!-- u_{j+1} = dp/dy = (psi_{j+2}-psi_壁)/(2dy)
!-- u_{j-1} = dp/dy = (psi_壁-psi_{j-2})/(2dy)
!-- まとめると, omega_壁 = -(psi_{j+-2}-psi_壁)/(2dy^2)
!-- 右 (+), 左 (-) : 壁では速度がゼロ.
!-- omega_壁 = dv/dx = (v_{i+1}/dx), (-v_{i-1}/dx)
!-- v_{i+1} = -dp/dx = -(psi_{i+2}-psi_壁)/(2dx)
!-- v_{i-1} = -dp/dx = -(psi_壁-psi_{i-2})/(2dx)
!-- まとめると, omega_壁 = -(psi_{i+-2}-psi_壁)/(2dx^2)

  do j=2,ny-1
     do i=2,nx-1
        if(inner_bound(i,j)==1)then
           if(inner_bound(i-1,j)==10)then
              omega(i,j)=0.5*(psi(i,j)-psi(i+2,j))/(dx(i)*dx(i))
           else if(inner_bound(i+1,j)==10)then
              omega(i,j)=0.5*(psi(i,j)-psi(i-2,j))/(dx(i)*dx(i))
           else if(inner_bound(i,j-1)==10)then
              omega(i,j)=0.5*(psi(i,j)-psi(i,j+2))/(dy(j)*dy(j))*x2_inv(i)
           else if(inner_bound(i,j+1)==10)then
              omega(i,j)=0.5*(psi(i,j)-psi(i,j-2))/(dy(j)*dy(j))*x2_inv(i)
           else if(inner_bound(i,j-1)==1.and.inner_bound(i+1,j)==1)then  ! 左上
              omega(i,j)=0.25*((psi(i,j)-psi(i,j+2))/(dy(j)*dy(j))*x2_inv(i)  &
  &                           +(psi(i,j)-psi(i-2,j))/(dx(i)*dx(i)))
           else if(inner_bound(i,j-1)==1.and.inner_bound(i-1,j)==1)then  ! 右上
              omega(i,j)=0.25*((psi(i,j)-psi(i,j+2))/(dy(j)*dy(j))*x2_inv(i)  &
  &                           +(psi(i,j)-psi(i+2,j))/(dx(i)*dx(i)))
           else if(inner_bound(i,j+1)==1.and.inner_bound(i+1,j)==1)then  ! 左下
              omega(i,j)=0.25*((psi(i,j)-psi(i,j-2))/(dy(j)*dy(j))*x2_inv(i)+  &
  &                            (psi(i,j)-psi(i-2,j))/(dx(i)*dx(i)))
           else if(inner_bound(i,j+1)==1.and.inner_bound(i-1,j)==1)then  ! 右下
              omega(i,j)=0.25*((psi(i,j)-psi(i,j-2))/(dy(j)*dy(j))*x2_inv(i)  &
  &                           +(psi(i,j)-psi(i+2,j))/(dx(i)*dx(i)))
           end if
        end if
     end do
  end do

end subroutine

end module
