require "numru/gphys/subsetmapping"
require "numru/gphys/attribute"
require "narray_miss"

module NumRu

=begin
= VArray

Virtual Array class, in which a multi-dimentional array data is stored
on memory (NArray, NArrayMiss) or in file (NetCDFVar etc). 
The in-file data handlign is realized in a subclass, and 
data are stored on memory in this base class.
A VArray can be a reference to a subset of another VArray.

=end

   class VArray

      ### < basic parts to be redefined in subclasses > ###

      def initialize(narray=nil, attr=nil, name=nil)
	 # initialize with an actual array --- initialization by subset
	 # mapping is made with VArray.new.initialize_mapping(...)
	 @name = ( name || "noname" )
	 @mapping = nil
	 @varray = nil
	 @ary = __check_ary_class(narray)
	 if attr.is_a?(Attribute)
	    @attr = attr
	 else
	    @attr = NumRu::Attribute.new
	    if attr
	       attr.each{|key,val| @attr[key]=val}
	    end
	 end
      end

      attr_reader :mapping, :varray, :ary
      protected :mapping, :varray, :ary

      def inspect
	 if !@mapping
	    "<'#{@name}' #{ntype}#{shape.inspect} val=[#{(0..(3<length ? (l=true; 3) : (l=false; length))).collect do |i| @ary[i].to_s+',' end}#{'...' if l}]>"
	 else
	    "<'#{@name}' shape=#{shape.inspect}  subset of a #{@varray.class}>"
	 end
      end

      def VArray.new2(ntype, shape, attr=nil, name=nil)
	 ary = NArray.new(ntype, *shape)
	 VArray.new(ary, attr, name)
      end

      def val
	 if @mapping
	    @varray.ary[*@mapping.slicer]  # 1-step mapping is assumed
	 else
	    @ary
	 end
      end

      def val=(narray)
	 if @mapping
	    @varray.ary[*@mapping.slicer] = __check_ary_class2(narray)
	 else
	    @ary[] = __check_ary_class2(narray)
	 end
	 narray
      end

      def ntype
	 if !@mapping
	    __ntype(@ary)
	 else
	    __ntype(@varray.ary)
	 end
      end

      ### < basic parts invariant in subclasses > ###

      def copy(to=nil)
	 attr = self.attr.copy( (to ? to.attr : to) )
	 val = self.val
	 if self.class==VArray && !self.mapped? && (to==nil || to.class==VArray)
	    val = val.dup
	 end
	 if to
	    to.val = val
	    to
	 else
	    VArray.new(val, attr, self.name)
	 end
      end

      def reshape( *args )
	 # reshape method that returns a new entire copy (deep one).
	 # ToDo :: prepare another reshape method that does not make the
	 # entire copy (you could use NArray.refer, but be careful not 
	 # to make it public, because it's easily misused)
	 newobj = self.copy
	 newobj.ary.reshape!( *args )
	 newobj
      end

      def mapped?
	 @mapping != nil
      end

      def initialize_mapping(mapping, varray)
	 # protected method
	 raise ArgumentError if ! mapping.is_a?(SubsetMapping)
	 raise ArgumentError if ! varray.is_a?(VArray)
	 if ! varray.mapping
	    @mapping = mapping
	    @varray = varray
	 else
	    # To keep the mapping within one step
	    @mapping = varray.mapping.composite(mapping)
	    @varray = varray.varray
	 end
	 @attr = NumRu::Attribute.new
	 @ary = nil
	 self
      end
      protected :initialize_mapping

      def attr
	 if @mapping
	    @varray.attr
	 else
	    @attr
	 end
      end

      def [](*slicer)
	 mapping = SubsetMapping.new(self.shape_current, slicer)
	 VArray.new.initialize_mapping(mapping, self)
      end

      def []=(*args)
	 val = args.pop
	 slicer = args
	 if val.is_a?(VArray)
	    val = val.val
	 else
	    val = __check_ary_class2(val)
	 end
	 if @mapping
	    sl= @mapping.composite(SubsetMapping.new(self.shape,slicer)).slicer
	    @varray[*sl]=val
	 else
	    @ary[*slicer]=val
	 end
	 val
      end

      def name=(nm)
	 raise ArgumentError, "name should be a String" if ! nm.is_a?(String)
	 self.rename(nm)
	 nm
      end
      def rename(nm)
	 if @mapping
	    @varray.name = nm
	 else
	    @name = nm
	 end
	 self
      end
      def name
	 if @mapping
	    @varray.name
	 else
	    @name
	 end
      end

      ### < NArray methods > ###

      ## ToDo: implement units handling
      ## ToDo: coerce

      def coerce(other)
	 other = NArray.new(self.typecode, 1).coerce(other)[0]
	 [VArray.new(other,nil,self.name), self]
      end

      Math_funcs = ["sqrt","exp","log","log10","log2","sin","cos","tan",
	    "sinh","cosh","tanh","asin","acos","atan","asinh","acosh",
	    "atanh","csc","sec","cot","csch","sech","coth","acsc","asec",
	    "acot","acsch","asech","acoth"]
      Binary_operators = ["+","-","*","/","%","**", 
	                 ".add!",".sub!",".mul!",".div!","mod!"]
      Binary_operatorsL = [">",">=","<","<=","&","|","^",
	    ".eq",".ne",".gt",".ge",".lt",".le",".and",".or",".xor",".not"]
      Unary_operators = ["-@","~"]
      NArray_methods1 = ["all?","any?","none?","where","where2",
            "floor","ceil","round","to_f","to_i","to_a"]
      NArray_methods2 = ["rank", "shape", "total","length"]
      NArray_methods3 = ["typecode"]

      NArray_methods = Array.new.push(*NArray_methods1).
	                         push(*NArray_methods2).
	                         push(*NArray_methods3)

      for f in Math_funcs
	 eval <<-EOS
	 def VArray.#{f}(vary)
            raise ArgumentError, "Not a VArray" if !vary.is_a?(VArray)
	    VArray.new( NMath.#{f}(vary.val), vary.attr.copy, vary.name )
	 end
	 def #{f}
	    VArray.new( NMath.#{f}(self.val), self.attr.copy, self.name )
	 end
         EOS
      end
      for f in Binary_operators
	 eval <<-EOS
	 def #{f.delete(".")}(other)
            ary = self.val#{f} (other.is_a?(VArray) ? other.val : other)
	    VArray.new( ary, self.attr.copy, self.name )
	 end
	 EOS
      end
      for f in Binary_operatorsL
	 eval <<-EOS
	 def #{f.delete(".")}(other)
            # returns NArray
            self.val#{f} (other.is_a?(VArray) ? other.val : other)
	 end
	 EOS
      end
      for f in Unary_operators
	 eval <<-EOS
	 def #{f}
            ary = #{f.delete("@")} self.val
	    VArray.new( ary, self.attr.copy, self.name )
	 end
	 EOS
      end
      for f in NArray_methods1
	 eval <<-EOS
	 def #{f}(*args)
	    self.val.#{f}(*args)
	 end
	 EOS
      end
      for f in NArray_methods2
	 eval <<-EOS
	 def #{f}
            if @mapping
	       @mapping.#{f}
	    else
	       @ary.#{f}
            end
	 end
	 EOS
      end
      for f in NArray_methods3
	 eval <<-EOS
	 def #{f}
            if @mapping
	       @varray.ary.#{f}
	    else
	       @ary.#{f}
            end
	 end
	 EOS
      end

      alias shape_current shape

      ## < private methods >
      private
      def __check_ary_class(narray)
	 case narray
	 when NArray, NArrayMiss, nil
	 else
	    raise ArgumentError, "Invalid array class: #{narray.class}" 
	 end
	 narray
      end
      def __check_ary_class2(narray)
	 case narray
	 when NArray, NArrayMiss, nil, Numeric
	 else
	    raise ArgumentError, "Invalid array class: #{narray.class}" 
	 end
	 narray
      end
      def __ntype(na)
	 case na.typecode
	 when NArray::BYTE
	    "byte"
	 when NArray::SINT
	    "sint"
	 when NArray::LINT
	    "lint"
	 when NArray::SFLOAT
	    "sfloat"
	 when NArray::DFLOAT
	    "dfloat"
	 when NArray::SCOMPLEX
	    "scomplex"
	 when NArray::DCOMPLEX
	    "dcomplex"
	 when NArray::ROBJ
	    "robj"
	 end
      end

   end

end

##################################
### < test > ###

if $0 == __FILE__
   include NumRu
   p va = VArray.new( NArray.int(6,2,3).indgen! )
   vs = va[2..4,0,0..1]
   vs.attr["units"]="m"
   p "@@@",vs.rank,vs.shape,vs.total,vs.val,vs.attr["name"]
   vs.val=999
   p "*1*",va
   p "*2*",vt = vs/9, vs + vt
   p "*3*",vt.class.log10(vt)
   p "*4*",(vt < vs)
   vt.name='vvvttt'
   p "*5*",(3+vt), VArray.sin(vt), vt.cos
   vc = vt.copy
   p "eq",vc.eq(vt),vc.equal?(vt)

   vd = VArray.new( NArray.int(6).indgen!+10 )
   p "+++",vd[1].val
   p va
   p vs
end
