!-----------------------------------------------------------------------
! Copyright (c) 2000-2019 Gtool Development Group. All rights reserved.
!-----------------------------------------------------------------------
!>
!> @author    Youhei SASAKI, Yasuhiro MORIKAWA, Eizi TOYODA
!> @copyright Copyright (C) GFD Dennou Club, 2000-2017. All rights reserved. <br/>
!>            License is BSD-2-Clause. see [COPYRIGHT](@ref COPYRIGHT) in detail
!> @en
!> @brief Provides kind type parameter values.
!> @details
!> This module using F2003 feature, especially `ISO_C_BINDING, ISO_FORTRAN_ENV`.
!>
!> dc_types | intrinsic modules |  types
!> ---------|------------------ |---------------
!> `SP`     | ISO_C_BINDING     | `C_FLOAT`
!> `DP`     | ISO_C_BINDING     | `C_DOUBLE`
!> `STDIN`  | ISO_FORTRAN_ENV   | `INPUT_UNIT`
!> `STDOUT` | ISO_FORTRAN_ENV   | `OUTPUT_UNIT`
!> `STRERR` | ISO_FORTRAN_ENV   | `ERROR_UNIT`
!>
!> @warning If you want to try mixed-language programing
!>          (e.g., Fortran calling C function),
!>          Please check type of variables before use this module.
!> @enden
!>
!> @ja
!> @brief 種別型パラメタを提供します。
!> @details
!>
!> このモジュールは Fortran 2003 の機能, 具体的には `ISO_C_BINDING` と
!> `ISO_FORTRAN_ENV` と同じ型を提供します.
!>
!> dc_types | intrinsic modules |  types
!> ---------|------------------ |---------------
!> `SP`     | ISO_C_BINDING     | `C_FLOAT`
!> `DP`     | ISO_C_BINDING     | `C_DOUBLE`
!> `STDIN`  | ISO_FORTRAN_ENV   | `INPUT_UNIT`
!> `STDOUT` | ISO_FORTRAN_ENV   | `OUTPUT_UNIT`
!> `STRERR` | ISO_FORTRAN_ENV   | `ERROR_UNIT`
!>
!> などです．
!> @warning 言語混在プログラミング(例えば Fortran から C の関数を呼ぶなど)
!>          を行ないたい場合には本モジュールの使用前に,
!>          変数の型を確認して下さい.
!>
!> @endja
!>

module dc_types
  use, intrinsic :: ISO_C_BINDING,    &
       & only: C_INT,                 &
       &       C_FLOAT,               &
       &       C_DOUBLE
  use, intrinsic :: ISO_FORTRAN_ENV,  &
       & only: INPUT_UNIT,            &
       &       OUTPUT_UNIT,           &
       &       ERROR_UNIT
  implicit none
  private
  public :: SP
  public :: SP_EPS
  public :: DP
  public :: DP_EPS
  public :: TOKEN
  public :: STRING
  public :: STDIN
  public :: STDOUT
  public :: STDERR

  !> @var SP
  !> @en Single Precision Real number. @enden
  !> @ja 単精度実数型変数 @endja
  integer, parameter  :: SP     = C_FLOAT

  !> @var SP_EPS
  !> @en Machine epsilon for single precision real number. @enden
  !> @ja 単精度実数型変数のマシンイプシロン. @endja
  real(SP), parameter :: SP_EPS = EPSILON(0.0e0)

  !> @var DP
  !> @en Double Precision Real number @enden
  !> @ja 倍精度実数型変数 @endja
  integer, parameter  :: DP     = C_DOUBLE

  !> @var DP_EPS
  !> @en Machine epsilon for dobule precision real number. @enden
  !> @ja 倍精度実数型変数のマシンイプシロン. @endja
  real(DP), parameter :: DP_EPS = EPSILON(0.0d0)

  !> @var STDIN
  !> @en Unit number for Standard INPUT @enden
  !> @ja 標準入力の装置番号 @endja
  integer, parameter  :: STDIN  = INPUT_UNIT

  !> @var STDOUT
  !> @en Unit number for Standard OUTPUT @enden
  !> @ja 標準出力の装置番号 @endja
  integer, parameter  :: STDOUT = OUTPUT_UNIT

  !> @var STDERR
  !> @en Unit number for Standard ERROR @enden
  !> @ja 標準エラー出力の装置番号 @endja
  integer, parameter  :: STDERR = ERROR_UNIT


  !> @var TOKEN
  !> @en Character length for word, token @enden
  !> @ja 単語やキーワードを保持する文字型変数の種別型パラメタ @endja
  integer, parameter  :: TOKEN  = 32

  !> @var STRING
  !> @en Character length for string @enden
  !> @ja 文字列を保持する 文字型変数の種別型パラメタ
  !> @note 256 という値に深い理由はありません.
  !>       必要ならばより大きな値を設定しても構いません.
  !>       ただし 8 バイト境界となるよう, 8 の倍数となっていることを推奨します.
  !> @endja
  integer, parameter  :: STRING = 256

  !> @namespace dc_types
end module dc_types
