% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aaa.R
\name{on_load}
\alias{on_load}
\alias{run_on_load}
\alias{on_package_load}
\title{Run expressions on load}
\usage{
on_load(expr, env = parent.frame(), ns = topenv(env))

run_on_load(ns = topenv(parent.frame()))

on_package_load(pkg, expr, env = parent.frame())
}
\arguments{
\item{expr}{An expression to run on load.}

\item{env}{The environment in which to evaluate \code{expr}. Defaults to
the current environment, which is your package namespace if you
run \code{on_load()} at top level.}

\item{ns}{The namespace in which to hook \code{expr}.}

\item{pkg}{Package to hook expression into.}
}
\description{
\itemize{
\item \code{on_load()} registers expressions to be run on the user's machine
each time the package is loaded in memory. This is by contrast to
normal R package code which is run once at build time on the
packager's machine (e.g. CRAN).

\code{on_load()} expressions require \code{run_on_load()} to be called
inside \code{\link[=.onLoad]{.onLoad()}}.
\item \code{on_package_load()} registers expressions to be run each time
another package is loaded.
}

\code{on_load()} is for your own package and runs expressions when the
namespace is not \emph{sealed} yet. This means you can modify existing
binding or create new ones. This is not the case with
\code{on_package_load()} which runs expressions after a foreign package
has finished loading, at which point its namespace is sealed.
}
\section{When should I run expressions on load?}{

There are two main use cases for running expressions on load:
\enumerate{
\item When a side effect, such as registering a method with
\code{s3_register()}, must occur in the user session rather than the
package builder session.
\item To avoid hard-coding objects from other packages in your
namespace. If you assign \code{foo::bar} or the result of
\code{foo::baz()} in your package, they become constants. Any
upstream changes in the \code{foo} package will not be reflected in
the objects you've assigned in your namespace. This often breaks
assumptions made by the authors of \code{foo} and causes all sorts of
issues.

Recreating the foreign objects each time your package is loaded
makes sure that any such changes will be taken into account. In
technical terms, running an expression on load introduces
\emph{indirection}.
}
}

\section{Comparison with \code{.onLoad()}}{

\code{on_load()} has the advantage that hooked expressions can appear in
any file, in context. This is unlike \code{.onLoad()} which gathers
disparate expressions in a single block.

\code{on_load()} is implemented via \code{.onLoad()} and requires
\code{run_on_load()} to be called from that hook.

The expressions inside \code{on_load()} do not undergo static analysis
by \verb{R CMD check}. Therefore, it is advisable to only use
simple function calls inside \code{on_load()}.
}

\examples{
quote({  # Not run

# First add `run_on_load()` to your `.onLoad()` hook,
# then use `on_load()` anywhere in your package
.onLoad <- function(lib, pkg) {
  run_on_load()
}

# Register a method on load
on_load({
  s3_register("foo::bar", "my_class")
})

# Assign an object on load
var <- NULL
on_load({
  var <- foo()
})

# To use `on_package_load()` at top level, wrap it in `on_load()`
on_load({
  on_package_load("foo", message("foo is loaded"))
})

# In functions it can be called directly
f <- function() on_package_load("foo", message("foo is loaded"))

})
}
