/* ************************************************************************
 * Copyright (C) 2023 Advanced Micro Devices, Inc. All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell cop-
 * ies of the Software, and to permit persons to whom the Software is furnished
 * to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IM-
 * PLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNE-
 * CTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * ************************************************************************ */

#include "hipblas_data.hpp"
#include "hipblas_test.hpp"
#include "solver/testing_getri_batched.hpp"
#include "solver/testing_getri_npvt_batched.hpp"
#include "type_dispatch.hpp"

namespace
{
    // possible getri test cases
    enum getri_test_type
    {
        GETRI_BATCHED,
        GETRI_NPVT_BATCHED
    };

    //getri test template
    template <template <typename...> class FILTER, getri_test_type GETRI_TYPE>
    struct getri_template : HipBLAS_Test<getri_template<FILTER, GETRI_TYPE>, FILTER>
    {
        template <typename... T>
        struct type_filter_functor
        {
            bool operator()(const Arguments& args)
            {
                // additional global filters applied first
                if(!hipblas_client_global_filters(args))
                    return false;

                // type filters
                return static_cast<bool>(FILTER<T...>{});
            }
        };

        // Filter for which types apply to this suite
        static bool type_filter(const Arguments& arg)
        {
            return hipblas_simple_dispatch<getri_template::template type_filter_functor>(arg);
        }

        // Filter for which functions apply to this suite
        static bool function_filter(const Arguments& arg)
        {
            switch(GETRI_TYPE)
            {
            case GETRI_BATCHED:
                return !strcmp(arg.function, "getri_batched")
                       || !strcmp(arg.function, "getri_batched_bad_arg");
            case GETRI_NPVT_BATCHED:
                return !strcmp(arg.function, "getri_npvt_batched")
                       || !strcmp(arg.function, "getri_npvt_batched_bad_arg");
            }
            return false;
        }

        // Google Test name suffix based on parameters
        static std::string name_suffix(const Arguments& arg)
        {
            std::string name;
            if constexpr(GETRI_TYPE == GETRI_BATCHED)
                testname_getri_batched(arg, name);
            else if constexpr(GETRI_TYPE == GETRI_NPVT_BATCHED)
                testname_getri_npvt_batched(arg, name);
            return std::move(name);
        }
    };

    // By default, arbitrary type combinations are invalid.
    // The unnamed second parameter is used for enable_if_t below.
    template <typename, typename = void>
    struct getri_testing : hipblas_test_invalid
    {
    };

    // When the condition in the second argument is satisfied, the type combination
    // is valid. When the condition is false, this specialization does not apply.
    template <typename T>
    struct getri_testing<
        T,
        std::enable_if_t<
            std::is_same_v<
                T,
                float> || std::is_same_v<T, double> || std::is_same_v<T, hipblasComplex> || std::is_same_v<T, hipblasDoubleComplex>>>
        : hipblas_test_valid
    {
        void operator()(const Arguments& arg)
        {
            if(!strcmp(arg.function, "getri_batched"))
                testing_getri_batched<T>(arg);
            else if(!strcmp(arg.function, "getri_batched_bad_arg"))
                testing_getri_batched_bad_arg<T>(arg);
            else if(!strcmp(arg.function, "getri_npvt_batched"))
                testing_getri_npvt_batched<T>(arg);
            else if(!strcmp(arg.function, "getri_npvt_batched_bad_arg"))
                testing_getri_npvt_batched_bad_arg<T>(arg);
            else
                FAIL() << "Internal error: Test called with unknown function: " << arg.function;
        }
    };

    using getri_batched = getri_template<getri_testing, GETRI_BATCHED>;
    TEST_P(getri_batched, solver)
    {
        CATCH_SIGNALS_AND_EXCEPTIONS_AS_FAILURES(
            hipblas_simple_dispatch<getri_testing>(GetParam()));
    }
    INSTANTIATE_TEST_CATEGORIES(getri_batched);

    using getri_npvt_batched = getri_template<getri_testing, GETRI_NPVT_BATCHED>;
    TEST_P(getri_npvt_batched, solver)
    {
        CATCH_SIGNALS_AND_EXCEPTIONS_AS_FAILURES(
            hipblas_simple_dispatch<getri_testing>(GetParam()));
    }
    INSTANTIATE_TEST_CATEGORIES(getri_npvt_batched);

} // namespace
