% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/iterators.R
\name{igraph-es-indexing}
\alias{igraph-es-indexing}
\alias{[.igraph.es}
\alias{\%--\%}
\alias{\%<-\%}
\alias{\%->\%}
\title{Indexing edge sequences}
\usage{
\method{[}{igraph.es}(x, ...)
}
\arguments{
\item{x}{An edge sequence}

\item{...}{Indices, see details below.}
}
\value{
Another edge sequence, referring to the same graph.
}
\description{
Edge sequences can be indexed very much like a plain numeric R vector,
with some extras.
}
\section{Multiple indices}{

When using multiple indices within the bracket, all of them
are evaluated independently, and then the results are concatenated
using the \code{c()} function. E.g. \code{E(g)[1, 2, .inc(1)]}
is equivalent to \code{c(E(g)[1], E(g)[2], E(g)[.inc(1)])}.
}

\section{Index types}{

Edge sequences can be indexed with positive numeric vectors,
negative numeric vectors, logical vectors, character vectors:
\itemize{
  \item When indexed with positive numeric vectors, the edges at the
    given positions in the sequence are selected. This is the same as
    indexing a regular R atomic vector with positive numeric vectors.
  \item When indexed with negative numeric vectors, the edges at the
    given positions in the sequence are omitted. Again, this is the same
    as indexing a regular R atomic vector.
  \item When indexed with a logical vector, the lengths of the edge
    sequence and the index must match, and the edges for which the
    index is \code{TRUE} are selected.
  \item Named graphs can be indexed with character vectors,
    to select edges with the given names. Note that a graph may
    have edge names and vertex names, and both can be used to select
    edges. Edge names are simply used as names of the numeric
    edge id vector. Vertex names effectively only work in graphs without
    multiple edges, and must be separated with a \code{|} bar character
    to select an edges that incident to the two given vertices. See
    examples below.
}
}

\section{Edge attributes}{

When indexing edge sequences, edge attributes can be referred
to simply by using their names. E.g. if a graph has a \code{weight} edge
attribute, then \code{E(G)[weight > 1]} selects all edges with a weight
larger than one. See more examples below. Note that attribute names mask the
names of variables present in the calling environment; if you need to look up
a variable and you do not want a similarly named edge attribute to mask it,
use the \code{.env} pronoun to perform the name lookup in the calling
environment. In other words, use \code{E(g)[.env$weight > 1]} to make sure
that \code{weight} is looked up from the calling environment even if there is
an edge attribute with the same name. Similarly, you can use \code{.data} to
match attribute names only.
}

\section{Special functions}{

There are some special igraph functions that can be used
only in expressions indexing edge sequences: \describe{
  \item{\code{.inc}}{takes a vertex sequence, and selects
    all edges that have at least one incident vertex in the vertex
    sequence.}
  \item{\code{.from}}{similar to \code{.inc()}, but only
    the tails of the edges are considered.}
  \item{\code{.to}}{is similar to \code{.inc()}, but only
    the heads of the edges are considered.}
  \item{\code{\%--\%}}{a special operator that can be
    used to select all edges between two sets of vertices. It ignores
    the edge directions in directed graphs.}
  \item{\code{\%->\%}}{similar to \code{\%--\%},
    but edges \emph{from} the left hand side argument, pointing
    \emph{to} the right hand side argument, are selected, in directed
    graphs.}
  \item{\code{\%<-\%}}{similar to \code{\%--\%},
    but edges \emph{to} the left hand side argument, pointing
    \emph{from} the right hand side argument, are selected, in directed
    graphs.}
}
Note that multiple special functions can be used together, or with
regular indices, and then their results are concatenated. See more
examples below.
}

\examples{
# -----------------------------------------------------------------
# Special operators for indexing based on graph structure
g <- sample_pa(100, power = 0.3)
E(g) [ 1:3 \%--\% 2:6 ]
E(g) [ 1:5 \%->\% 1:6 ]
E(g) [ 1:3 \%<-\% 2:6 ]

# -----------------------------------------------------------------
# The edges along the diameter
g <- sample_pa(100, directed = FALSE)
d <- get_diameter(g)
E(g, path = d)

# -----------------------------------------------------------------
# Select edges based on attributes
g <- sample_gnp(20, 3/20) \%>\%
  set_edge_attr("weight", value = rnorm(gsize(.)))
E(g)[[ weight < 0 ]]

# Indexing with a variable whose name matches the name of an attribute
# may fail; use .env to force the name lookup in the parent environment
E(g)$x <- E(g)$weight
x <- 2
E(g)[.env$x]

}
\seealso{
Other vertex and edge sequences: 
\code{\link{E}()},
\code{\link{V}()},
\code{\link{igraph-es-attributes}},
\code{\link{igraph-es-indexing2}},
\code{\link{igraph-vs-attributes}},
\code{\link{igraph-vs-indexing2}},
\code{\link{igraph-vs-indexing}},
\code{\link{print.igraph.es}()},
\code{\link{print.igraph.vs}()}

Other vertex and edge sequence operations: 
\code{\link{c.igraph.es}()},
\code{\link{c.igraph.vs}()},
\code{\link{difference.igraph.es}()},
\code{\link{difference.igraph.vs}()},
\code{\link{igraph-es-indexing2}},
\code{\link{igraph-vs-indexing2}},
\code{\link{igraph-vs-indexing}},
\code{\link{intersection.igraph.es}()},
\code{\link{intersection.igraph.vs}()},
\code{\link{rev.igraph.es}()},
\code{\link{rev.igraph.vs}()},
\code{\link{union.igraph.es}()},
\code{\link{union.igraph.vs}()},
\code{\link{unique.igraph.es}()},
\code{\link{unique.igraph.vs}()}
}
\concept{vertex and edge sequence operations}
\concept{vertex and edge sequences}
