#!/usr/bin/env python3

# Libervia CLI
# Copyright (C) 2009-2025 Jérôme Poisson (goffi@goffi.org)

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.

# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from typing import TYPE_CHECKING

from libervia.frontends.tools import aio
from libervia.frontends.tools.webrtc import CallData, WebRTCCall
from libervia.frontends.tools.webrtc_models import SinksPipeline

aio.install_glib_asyncio_iteration()

if TYPE_CHECKING:
    from .cmd_call import Common


class AVCallUI:
    """This class doesn't create any UI."""

    def __init__(self, host, webrtc=None) -> None:
        self.host = host
        self.webrtc = webrtc

    @staticmethod
    def parse_output_opts(parent: "Common") -> dict:
        """Parse output options.

        @return: keyword argument to use to instanciate WebRTCCall
        """
        kwargs = {}
        for oo in parent.args.output_opts:
            if oo.startswith("audio="):
                kwargs["audio_pipeline"] = oo[6:].strip()
            elif oo.startswith("video="):
                kwargs["video_pipeline"] = oo[6:].strip()
            else:
                parent.parser.error(f"Unknown output option: {oo!r}")
        return kwargs

    @classmethod
    async def run(cls, parent: "Common", call_data: CallData) -> None:
        sinks_kwargs = cls.parse_output_opts(parent)
        if (
            call_data.kwargs.get("record_fd") is not None
            and not "audio_pipeline" in sinks_kwargs
            and not "video_pipeline" in sinks_kwargs
        ):
            # If we have `record_fd` set, we are piping out the stream. If no pipeline is
            # explictely specified, we disable the UI by using empty string.
            sinks_kwargs["audio_pipeline"] = sinks_kwargs["video_pipeline"] = ""
        call_data.kwargs["sinks_data"] = SinksPipeline(**sinks_kwargs)

        webrtc_call = await WebRTCCall.make_webrtc_call(
            parent.host.bridge,
            parent.profile,
            call_data,
            merge_pip=False,
            on_call_setup_cb=lambda sid, profile: parent.host.add_on_quit_callback(
                parent.host.bridge.call_end, sid, "", profile
            ),
            on_call_ended_cb=lambda sid, profile: parent.host.a_quit(),
        )

        cls(parent.host, webrtc_call.webrtc)
