/* SPDX-License-Identifier: GPL-2.0-or-later */
/*
 *  Copyright 2003-2005 Red Hat, Inc.  All rights reserved.
 *  Copyright 2003-2005 Jeff Garzik
 *
 *  libata documentation is available via 'make {ps|pdf}docs',
 *  as Documentation/driver-api/libata.rst
 */

#ifndef __LINUX_LIBATA_H__
#define __LINUX_LIBATA_H__

#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/interrupt.h>
#include <linux/dma-mapping.h>
#include <linux/scatterlist.h>
#include <linux/io.h>
#include <linux/ata.h>
#include <linux/workqueue.h>
#include <scsi/scsi_host.h>
#include <linux/acpi.h>
#include <linux/cdrom.h>
#include <linux/sched.h>
#include <linux/async.h>

/*
 * Define if arch has non-standard setup.  This is a _PCI_ standard
 * not a legacy or ISA standard.
 */
#ifdef CONFIG_ATA_NONSTANDARD
#include <asm/libata-portmap.h>
#else
#define ATA_PRIMARY_IRQ(dev)	14
#define ATA_SECONDARY_IRQ(dev)	15
#endif

/*
 * compile-time options: to be removed as soon as all the drivers are
 * converted to the new debugging mechanism
 */
#undef ATA_IRQ_TRAP		/* define to ack screaming irqs */

/* defines only for the constants which don't work well as enums */
#define ATA_TAG_POISON		0xfafbfcfdU

/*
 * Quirk flags bits.
 * ata_device->quirks is an unsigned int, so __ATA_QUIRK_MAX must not exceed 32.
 */
enum ata_quirks {
	__ATA_QUIRK_DIAGNOSTIC,		/* Failed boot diag */
	__ATA_QUIRK_NODMA,		/* DMA problems */
	__ATA_QUIRK_NONCQ,		/* Don't use NCQ */
	__ATA_QUIRK_MAX_SEC_128,	/* Limit max sects to 128 */
	__ATA_QUIRK_BROKEN_HPA,		/* Broken HPA */
	__ATA_QUIRK_DISABLE,		/* Disable it */
	__ATA_QUIRK_HPA_SIZE,		/* Native size off by one */
	__ATA_QUIRK_IVB,		/* cbl det validity bit bugs */
	__ATA_QUIRK_STUCK_ERR,		/* Stuck ERR on next PACKET */
	__ATA_QUIRK_BRIDGE_OK,		/* No bridge limits */
	__ATA_QUIRK_ATAPI_MOD16_DMA,	/* Use ATAPI DMA for commands that */
					/* are not a multiple of 16 bytes */
	__ATA_QUIRK_FIRMWARE_WARN,	/* Firmware update warning */
	__ATA_QUIRK_1_5_GBPS,		/* Force 1.5 Gbps */
	__ATA_QUIRK_NOSETXFER,		/* Skip SETXFER, SATA only */
	__ATA_QUIRK_BROKEN_FPDMA_AA,	/* Skip AA */
	__ATA_QUIRK_DUMP_ID,		/* Dump IDENTIFY data */
	__ATA_QUIRK_MAX_SEC_LBA48,	/* Set max sects to 65535 */
	__ATA_QUIRK_ATAPI_DMADIR,	/* Device requires dmadir */
	__ATA_QUIRK_NO_NCQ_TRIM,	/* Do not use queued TRIM */
	__ATA_QUIRK_NOLPM,		/* Do not use LPM */
	__ATA_QUIRK_WD_BROKEN_LPM,	/* Some WDs have broken LPM */
	__ATA_QUIRK_ZERO_AFTER_TRIM,	/* Guarantees zero after trim */
	__ATA_QUIRK_NO_DMA_LOG,		/* Do not use DMA for log read */
	__ATA_QUIRK_NOTRIM,		/* Do not use TRIM */
	__ATA_QUIRK_MAX_SEC_1024,	/* Limit max sects to 1024 */
	__ATA_QUIRK_MAX_TRIM_128M,	/* Limit max trim size to 128M */
	__ATA_QUIRK_NO_NCQ_ON_ATI,	/* Disable NCQ on ATI chipset */
	__ATA_QUIRK_NO_LPM_ON_ATI,	/* Disable LPM on ATI chipset */
	__ATA_QUIRK_NO_ID_DEV_LOG,	/* Identify device log missing */
	__ATA_QUIRK_NO_LOG_DIR,		/* Do not read log directory */
	__ATA_QUIRK_NO_FUA,		/* Do not use FUA */

	__ATA_QUIRK_MAX,
};

enum {
	/* various global constants */
	LIBATA_MAX_PRD		= ATA_MAX_PRD / 2,
	LIBATA_DUMB_MAX_PRD	= ATA_MAX_PRD / 4,	/* Worst case */
	ATA_DEF_QUEUE		= 1,
	ATA_MAX_QUEUE		= 32,
	ATA_TAG_INTERNAL	= ATA_MAX_QUEUE,
	ATA_SHORT_PAUSE		= 16,

	ATAPI_MAX_DRAIN		= 16 << 10,

	ATA_ALL_DEVICES		= (1 << ATA_MAX_DEVICES) - 1,

	ATA_SHT_EMULATED	= 1,
	ATA_SHT_THIS_ID		= -1,

	/* struct ata_taskfile flags */
	ATA_TFLAG_LBA48		= (1 << 0), /* enable 48-bit LBA and "HOB" */
	ATA_TFLAG_ISADDR	= (1 << 1), /* enable r/w to nsect/lba regs */
	ATA_TFLAG_DEVICE	= (1 << 2), /* enable r/w to device reg */
	ATA_TFLAG_WRITE		= (1 << 3), /* data dir: host->dev==1 (write) */
	ATA_TFLAG_LBA		= (1 << 4), /* enable LBA */
	ATA_TFLAG_FUA		= (1 << 5), /* enable FUA */
	ATA_TFLAG_POLLING	= (1 << 6), /* set nIEN to 1 and use polling */

	/* struct ata_device stuff */
	ATA_DFLAG_LBA		= (1 << 0), /* device supports LBA */
	ATA_DFLAG_LBA48		= (1 << 1), /* device supports LBA48 */
	ATA_DFLAG_CDB_INTR	= (1 << 2), /* device asserts INTRQ when ready for CDB */
	ATA_DFLAG_NCQ		= (1 << 3), /* device supports NCQ */
	ATA_DFLAG_FLUSH_EXT	= (1 << 4), /* do FLUSH_EXT instead of FLUSH */
	ATA_DFLAG_ACPI_PENDING	= (1 << 5), /* ACPI resume action pending */
	ATA_DFLAG_ACPI_FAILED	= (1 << 6), /* ACPI on devcfg has failed */
	ATA_DFLAG_AN		= (1 << 7), /* AN configured */
	ATA_DFLAG_TRUSTED	= (1 << 8), /* device supports trusted send/recv */
	ATA_DFLAG_FUA		= (1 << 9), /* device supports FUA */
	ATA_DFLAG_DMADIR	= (1 << 10), /* device requires DMADIR */
	ATA_DFLAG_NCQ_SEND_RECV = (1 << 11), /* device supports NCQ SEND and RECV */
	ATA_DFLAG_NCQ_PRIO	= (1 << 12), /* device supports NCQ priority */
	ATA_DFLAG_CDL		= (1 << 13), /* supports cmd duration limits */
	ATA_DFLAG_CFG_MASK	= (1 << 14) - 1,

	ATA_DFLAG_PIO		= (1 << 14), /* device limited to PIO mode */
	ATA_DFLAG_NCQ_OFF	= (1 << 15), /* device limited to non-NCQ mode */
	ATA_DFLAG_SLEEPING	= (1 << 16), /* device is sleeping */
	ATA_DFLAG_DUBIOUS_XFER	= (1 << 17), /* data transfer not verified */
	ATA_DFLAG_NO_UNLOAD	= (1 << 18), /* device doesn't support unload */
	ATA_DFLAG_UNLOCK_HPA	= (1 << 19), /* unlock HPA */
	ATA_DFLAG_INIT_MASK	= (1 << 20) - 1,

	ATA_DFLAG_NCQ_PRIO_ENABLED = (1 << 20), /* Priority cmds sent to dev */
	ATA_DFLAG_CDL_ENABLED	= (1 << 21), /* cmd duration limits is enabled */
	ATA_DFLAG_RESUMING	= (1 << 22),  /* Device is resuming */
	ATA_DFLAG_DETACH	= (1 << 24),
	ATA_DFLAG_DETACHED	= (1 << 25),
	ATA_DFLAG_DA		= (1 << 26), /* device supports Device Attention */
	ATA_DFLAG_DEVSLP	= (1 << 27), /* device supports Device Sleep */
	ATA_DFLAG_ACPI_DISABLED = (1 << 28), /* ACPI for the device is disabled */
	ATA_DFLAG_D_SENSE	= (1 << 29), /* Descriptor sense requested */

	ATA_DFLAG_FEATURES_MASK	= (ATA_DFLAG_TRUSTED | ATA_DFLAG_DA |	\
				   ATA_DFLAG_DEVSLP | ATA_DFLAG_NCQ_SEND_RECV | \
				   ATA_DFLAG_NCQ_PRIO | ATA_DFLAG_FUA | \
				   ATA_DFLAG_CDL),

	ATA_DEV_UNKNOWN		= 0,	/* unknown device */
	ATA_DEV_ATA		= 1,	/* ATA device */
	ATA_DEV_ATA_UNSUP	= 2,	/* ATA device (unsupported) */
	ATA_DEV_ATAPI		= 3,	/* ATAPI device */
	ATA_DEV_ATAPI_UNSUP	= 4,	/* ATAPI device (unsupported) */
	ATA_DEV_PMP		= 5,	/* SATA port multiplier */
	ATA_DEV_PMP_UNSUP	= 6,	/* SATA port multiplier (unsupported) */
	ATA_DEV_SEMB		= 7,	/* SEMB */
	ATA_DEV_SEMB_UNSUP	= 8,	/* SEMB (unsupported) */
	ATA_DEV_ZAC		= 9,	/* ZAC device */
	ATA_DEV_ZAC_UNSUP	= 10,	/* ZAC device (unsupported) */
	ATA_DEV_NONE		= 11,	/* no device */

	/* struct ata_link flags */
	/* NOTE: struct ata_force_param currently stores lflags in u16 */
	ATA_LFLAG_NO_HRST	= (1 << 1), /* avoid hardreset */
	ATA_LFLAG_NO_SRST	= (1 << 2), /* avoid softreset */
	ATA_LFLAG_ASSUME_ATA	= (1 << 3), /* assume ATA class */
	ATA_LFLAG_ASSUME_SEMB	= (1 << 4), /* assume SEMB class */
	ATA_LFLAG_ASSUME_CLASS	= ATA_LFLAG_ASSUME_ATA | ATA_LFLAG_ASSUME_SEMB,
	ATA_LFLAG_NO_RETRY	= (1 << 5), /* don't retry this link */
	ATA_LFLAG_DISABLED	= (1 << 6), /* link is disabled */
	ATA_LFLAG_SW_ACTIVITY	= (1 << 7), /* keep activity stats */
	ATA_LFLAG_NO_LPM	= (1 << 8), /* disable LPM on this link */
	ATA_LFLAG_RST_ONCE	= (1 << 9), /* limit recovery to one reset */
	ATA_LFLAG_CHANGED	= (1 << 10), /* LPM state changed on this link */
	ATA_LFLAG_NO_DEBOUNCE_DELAY = (1 << 11), /* no debounce delay on link resume */

	/* struct ata_port flags */
	ATA_FLAG_SLAVE_POSS	= (1 << 0), /* host supports slave dev */
					    /* (doesn't imply presence) */
	ATA_FLAG_SATA		= (1 << 1),
	ATA_FLAG_NO_LPM		= (1 << 2), /* host not happy with LPM */
	ATA_FLAG_NO_LOG_PAGE	= (1 << 5), /* do not issue log page read */
	ATA_FLAG_NO_ATAPI	= (1 << 6), /* No ATAPI support */
	ATA_FLAG_PIO_DMA	= (1 << 7), /* PIO cmds via DMA */
	ATA_FLAG_PIO_LBA48	= (1 << 8), /* Host DMA engine is LBA28 only */
	ATA_FLAG_PIO_POLLING	= (1 << 9), /* use polling PIO if LLD
					     * doesn't handle PIO interrupts */
	ATA_FLAG_NCQ		= (1 << 10), /* host supports NCQ */
	ATA_FLAG_NO_POWEROFF_SPINDOWN = (1 << 11), /* don't spindown before poweroff */
	ATA_FLAG_NO_HIBERNATE_SPINDOWN = (1 << 12), /* don't spindown before hibernation */
	ATA_FLAG_DEBUGMSG	= (1 << 13),
	ATA_FLAG_FPDMA_AA		= (1 << 14), /* driver supports Auto-Activate */
	ATA_FLAG_IGN_SIMPLEX	= (1 << 15), /* ignore SIMPLEX */
	ATA_FLAG_NO_IORDY	= (1 << 16), /* controller lacks iordy */
	ATA_FLAG_ACPI_SATA	= (1 << 17), /* need native SATA ACPI layout */
	ATA_FLAG_AN		= (1 << 18), /* controller supports AN */
	ATA_FLAG_PMP		= (1 << 19), /* controller supports PMP */
	ATA_FLAG_FPDMA_AUX	= (1 << 20), /* controller supports H2DFIS aux field */
	ATA_FLAG_EM		= (1 << 21), /* driver supports enclosure
					      * management */
	ATA_FLAG_SW_ACTIVITY	= (1 << 22), /* driver supports sw activity
					      * led */
	ATA_FLAG_NO_DIPM	= (1 << 23), /* host not happy with DIPM */
	ATA_FLAG_SAS_HOST	= (1 << 24), /* SAS host */

	/* bits 24:31 of ap->flags are reserved for LLD specific flags */


	/* struct ata_port pflags */
	ATA_PFLAG_EH_PENDING	= (1 << 0), /* EH pending */
	ATA_PFLAG_EH_IN_PROGRESS = (1 << 1), /* EH in progress */
	ATA_PFLAG_FROZEN	= (1 << 2), /* port is frozen */
	ATA_PFLAG_RECOVERED	= (1 << 3), /* recovery action performed */
	ATA_PFLAG_LOADING	= (1 << 4), /* boot/loading probe */
	ATA_PFLAG_SCSI_HOTPLUG	= (1 << 6), /* SCSI hotplug scheduled */
	ATA_PFLAG_INITIALIZING	= (1 << 7), /* being initialized, don't touch */
	ATA_PFLAG_RESETTING	= (1 << 8), /* reset in progress */
	ATA_PFLAG_UNLOADING	= (1 << 9), /* driver is being unloaded */
	ATA_PFLAG_UNLOADED	= (1 << 10), /* driver is unloaded */

	ATA_PFLAG_RESUMING	= (1 << 16),  /* port is being resumed */
	ATA_PFLAG_SUSPENDED	= (1 << 17), /* port is suspended (power) */
	ATA_PFLAG_PM_PENDING	= (1 << 18), /* PM operation pending */
	ATA_PFLAG_INIT_GTM_VALID = (1 << 19), /* initial gtm data valid */

	ATA_PFLAG_PIO32		= (1 << 20),  /* 32bit PIO */
	ATA_PFLAG_PIO32CHANGE	= (1 << 21),  /* 32bit PIO can be turned on/off */
	ATA_PFLAG_EXTERNAL	= (1 << 22),  /* eSATA/external port */

	/* struct ata_queued_cmd flags */
	ATA_QCFLAG_ACTIVE	= (1 << 0), /* cmd not yet ack'd to scsi lyer */
	ATA_QCFLAG_DMAMAP	= (1 << 1), /* SG table is DMA mapped */
	ATA_QCFLAG_RTF_FILLED	= (1 << 2), /* result TF has been filled */
	ATA_QCFLAG_IO		= (1 << 3), /* standard IO command */
	ATA_QCFLAG_RESULT_TF	= (1 << 4), /* result TF requested */
	ATA_QCFLAG_CLEAR_EXCL	= (1 << 5), /* clear excl_link on completion */
	ATA_QCFLAG_QUIET	= (1 << 6), /* don't report device error */
	ATA_QCFLAG_RETRY	= (1 << 7), /* retry after failure */
	ATA_QCFLAG_HAS_CDL	= (1 << 8), /* qc has CDL a descriptor set */

	ATA_QCFLAG_EH		= (1 << 16), /* cmd aborted and owned by EH */
	ATA_QCFLAG_SENSE_VALID	= (1 << 17), /* sense data valid */
	ATA_QCFLAG_EH_SCHEDULED = (1 << 18), /* EH scheduled (obsolete) */
	ATA_QCFLAG_EH_SUCCESS_CMD = (1 << 19), /* EH should fetch sense for this successful cmd */

	/* host set flags */
	ATA_HOST_SIMPLEX	= (1 << 0),	/* Host is simplex, one DMA channel per host only */
	ATA_HOST_STARTED	= (1 << 1),	/* Host started */
	ATA_HOST_PARALLEL_SCAN	= (1 << 2),	/* Ports on this host can be scanned in parallel */
	ATA_HOST_IGNORE_ATA	= (1 << 3),	/* Ignore ATA devices on this host. */

	ATA_HOST_NO_PART	= (1 << 4), /* Host does not support partial */
	ATA_HOST_NO_SSC		= (1 << 5), /* Host does not support slumber */
	ATA_HOST_NO_DEVSLP	= (1 << 6), /* Host does not support devslp */

	/* bits 24:31 of host->flags are reserved for LLD specific flags */

	/* Various lengths of time */
	ATA_TMOUT_INTERNAL_QUICK = 5000,
	ATA_TMOUT_MAX_PARK	= 30000,

	/*
	 * GoVault needs 2s and iVDR disk HHD424020F7SV00 800ms.  2s
	 * is too much without parallel probing.  Use 2s if parallel
	 * probing is available, 800ms otherwise.
	 */
	ATA_TMOUT_FF_WAIT_LONG	=  2000,
	ATA_TMOUT_FF_WAIT	=   800,

	/* Spec mandates to wait for ">= 2ms" before checking status
	 * after reset.  We wait 150ms, because that was the magic
	 * delay used for ATAPI devices in Hale Landis's ATADRVR, for
	 * the period of time between when the ATA command register is
	 * written, and then status is checked.  Because waiting for
	 * "a while" before checking status is fine, post SRST, we
	 * perform this magic delay here as well.
	 *
	 * Old drivers/ide uses the 2mS rule and then waits for ready.
	 */
	ATA_WAIT_AFTER_RESET	=  150,

	/* If PMP is supported, we have to do follow-up SRST.  As some
	 * PMPs don't send D2H Reg FIS after hardreset, LLDs are
	 * advised to wait only for the following duration before
	 * doing SRST.
	 */
	ATA_TMOUT_PMP_SRST_WAIT	= 10000,

	/* When the LPM policy is set to ATA_LPM_MAX_POWER, there might
	 * be a spurious PHY event, so ignore the first PHY event that
	 * occurs within 10s after the policy change.
	 */
	ATA_TMOUT_SPURIOUS_PHY	= 10000,

	/* ATA bus states */
	BUS_UNKNOWN		= 0,
	BUS_DMA			= 1,
	BUS_IDLE		= 2,
	BUS_NOINTR		= 3,
	BUS_NODATA		= 4,
	BUS_TIMER		= 5,
	BUS_PIO			= 6,
	BUS_EDD			= 7,
	BUS_IDENTIFY		= 8,
	BUS_PACKET		= 9,

	/* SATA port states */
	PORT_UNKNOWN		= 0,
	PORT_ENABLED		= 1,
	PORT_DISABLED		= 2,

	/* encoding various smaller bitmaps into a single
	 * unsigned int bitmap
	 */
	ATA_NR_PIO_MODES	= 7,
	ATA_NR_MWDMA_MODES	= 5,
	ATA_NR_UDMA_MODES	= 8,

	ATA_SHIFT_PIO		= 0,
	ATA_SHIFT_MWDMA		= ATA_SHIFT_PIO + ATA_NR_PIO_MODES,
	ATA_SHIFT_UDMA		= ATA_SHIFT_MWDMA + ATA_NR_MWDMA_MODES,
	ATA_SHIFT_PRIO		= 6,

	ATA_PRIO_HIGH		= 2,
	/* size of buffer to pad xfers ending on unaligned boundaries */
	ATA_DMA_PAD_SZ		= 4,

	/* ering size */
	ATA_ERING_SIZE		= 32,

	/* return values for ->qc_defer */
	ATA_DEFER_LINK		= 1,
	ATA_DEFER_PORT		= 2,

	/* desc_len for ata_eh_info and context */
	ATA_EH_DESC_LEN		= 80,

	/* reset / recovery action types */
	ATA_EH_REVALIDATE	= (1 << 0),
	ATA_EH_SOFTRESET	= (1 << 1), /* meaningful only in ->prereset */
	ATA_EH_HARDRESET	= (1 << 2), /* meaningful only in ->prereset */
	ATA_EH_RESET		= ATA_EH_SOFTRESET | ATA_EH_HARDRESET,
	ATA_EH_ENABLE_LINK	= (1 << 3),
	ATA_EH_PARK		= (1 << 5), /* unload heads and stop I/O */
	ATA_EH_GET_SUCCESS_SENSE = (1 << 6), /* Get sense data for successful cmd */
	ATA_EH_SET_ACTIVE	= (1 << 7), /* Set a device to active power mode */

	ATA_EH_PERDEV_MASK	= ATA_EH_REVALIDATE | ATA_EH_PARK |
				  ATA_EH_GET_SUCCESS_SENSE | ATA_EH_SET_ACTIVE,
	ATA_EH_ALL_ACTIONS	= ATA_EH_REVALIDATE | ATA_EH_RESET |
				  ATA_EH_ENABLE_LINK,

	/* ata_eh_info->flags */
	ATA_EHI_HOTPLUGGED	= (1 << 0),  /* could have been hotplugged */
	ATA_EHI_NO_AUTOPSY	= (1 << 2),  /* no autopsy */
	ATA_EHI_QUIET		= (1 << 3),  /* be quiet */
	ATA_EHI_NO_RECOVERY	= (1 << 4),  /* no recovery */

	ATA_EHI_DID_SOFTRESET	= (1 << 16), /* already soft-reset this port */
	ATA_EHI_DID_HARDRESET	= (1 << 17), /* already soft-reset this port */
	ATA_EHI_PRINTINFO	= (1 << 18), /* print configuration info */
	ATA_EHI_SETMODE		= (1 << 19), /* configure transfer mode */
	ATA_EHI_POST_SETMODE	= (1 << 20), /* revalidating after setmode */
	ATA_EHI_DID_PRINT_QUIRKS = (1 << 21), /* already printed quirks info */

	ATA_EHI_DID_RESET	= ATA_EHI_DID_SOFTRESET | ATA_EHI_DID_HARDRESET,

	/* mask of flags to transfer *to* the slave link */
	ATA_EHI_TO_SLAVE_MASK	= ATA_EHI_NO_AUTOPSY | ATA_EHI_QUIET,

	/* max tries if error condition is still set after ->error_handler */
	ATA_EH_MAX_TRIES	= 5,

	/* sometimes resuming a link requires several retries */
	ATA_LINK_RESUME_TRIES	= 5,

	/* how hard are we gonna try to probe/recover devices */
	ATA_EH_DEV_TRIES	= 3,
	ATA_EH_PMP_TRIES	= 5,
	ATA_EH_PMP_LINK_TRIES	= 3,

	SATA_PMP_RW_TIMEOUT	= 3000,		/* PMP read/write timeout */

	/* This should match the actual table size of
	 * ata_eh_cmd_timeout_table in libata-eh.c.
	 */
	ATA_EH_CMD_TIMEOUT_TABLE_SIZE = 8,

	/*
	 * Quirk flags: may be set by libata or controller drivers on drives.
	 * Some quirks may be drive/controller pair dependent.
	 */
	ATA_QUIRK_DIAGNOSTIC		= (1U << __ATA_QUIRK_DIAGNOSTIC),
	ATA_QUIRK_NODMA			= (1U << __ATA_QUIRK_NODMA),
	ATA_QUIRK_NONCQ			= (1U << __ATA_QUIRK_NONCQ),
	ATA_QUIRK_MAX_SEC_128		= (1U << __ATA_QUIRK_MAX_SEC_128),
	ATA_QUIRK_BROKEN_HPA		= (1U << __ATA_QUIRK_BROKEN_HPA),
	ATA_QUIRK_DISABLE		= (1U << __ATA_QUIRK_DISABLE),
	ATA_QUIRK_HPA_SIZE		= (1U << __ATA_QUIRK_HPA_SIZE),
	ATA_QUIRK_IVB			= (1U << __ATA_QUIRK_IVB),
	ATA_QUIRK_STUCK_ERR		= (1U << __ATA_QUIRK_STUCK_ERR),
	ATA_QUIRK_BRIDGE_OK		= (1U << __ATA_QUIRK_BRIDGE_OK),
	ATA_QUIRK_ATAPI_MOD16_DMA	= (1U << __ATA_QUIRK_ATAPI_MOD16_DMA),
	ATA_QUIRK_FIRMWARE_WARN		= (1U << __ATA_QUIRK_FIRMWARE_WARN),
	ATA_QUIRK_1_5_GBPS		= (1U << __ATA_QUIRK_1_5_GBPS),
	ATA_QUIRK_NOSETXFER		= (1U << __ATA_QUIRK_NOSETXFER),
	ATA_QUIRK_BROKEN_FPDMA_AA	= (1U << __ATA_QUIRK_BROKEN_FPDMA_AA),
	ATA_QUIRK_DUMP_ID		= (1U << __ATA_QUIRK_DUMP_ID),
	ATA_QUIRK_MAX_SEC_LBA48		= (1U << __ATA_QUIRK_MAX_SEC_LBA48),
	ATA_QUIRK_ATAPI_DMADIR		= (1U << __ATA_QUIRK_ATAPI_DMADIR),
	ATA_QUIRK_NO_NCQ_TRIM		= (1U << __ATA_QUIRK_NO_NCQ_TRIM),
	ATA_QUIRK_NOLPM			= (1U << __ATA_QUIRK_NOLPM),
	ATA_QUIRK_WD_BROKEN_LPM		= (1U << __ATA_QUIRK_WD_BROKEN_LPM),
	ATA_QUIRK_ZERO_AFTER_TRIM	= (1U << __ATA_QUIRK_ZERO_AFTER_TRIM),
	ATA_QUIRK_NO_DMA_LOG		= (1U << __ATA_QUIRK_NO_DMA_LOG),
	ATA_QUIRK_NOTRIM		= (1U << __ATA_QUIRK_NOTRIM),
	ATA_QUIRK_MAX_SEC_1024		= (1U << __ATA_QUIRK_MAX_SEC_1024),
	ATA_QUIRK_MAX_TRIM_128M		= (1U << __ATA_QUIRK_MAX_TRIM_128M),
	ATA_QUIRK_NO_NCQ_ON_ATI		= (1U << __ATA_QUIRK_NO_NCQ_ON_ATI),
	ATA_QUIRK_NO_LPM_ON_ATI		= (1U << __ATA_QUIRK_NO_LPM_ON_ATI),
	ATA_QUIRK_NO_ID_DEV_LOG		= (1U << __ATA_QUIRK_NO_ID_DEV_LOG),
	ATA_QUIRK_NO_LOG_DIR		= (1U << __ATA_QUIRK_NO_LOG_DIR),
	ATA_QUIRK_NO_FUA		= (1U << __ATA_QUIRK_NO_FUA),

	/* User visible DMA mask for DMA control. DO NOT renumber. */
	ATA_DMA_MASK_ATA	= (1 << 0),	/* DMA on ATA Disk */
	ATA_DMA_MASK_ATAPI	= (1 << 1),	/* DMA on ATAPI */
	ATA_DMA_MASK_CFA	= (1 << 2),	/* DMA on CF Card */

	/* ATAPI command types */
	ATAPI_READ		= 0,		/* READs */
	ATAPI_WRITE		= 1,		/* WRITEs */
	ATAPI_READ_CD		= 2,		/* READ CD [MSF] */
	ATAPI_PASS_THRU		= 3,		/* SAT pass-thru */
	ATAPI_MISC		= 4,		/* the rest */

	/* Timing constants */
	ATA_TIMING_SETUP	= (1 << 0),
	ATA_TIMING_ACT8B	= (1 << 1),
	ATA_TIMING_REC8B	= (1 << 2),
	ATA_TIMING_CYC8B	= (1 << 3),
	ATA_TIMING_8BIT		= ATA_TIMING_ACT8B | ATA_TIMING_REC8B |
				  ATA_TIMING_CYC8B,
	ATA_TIMING_ACTIVE	= (1 << 4),
	ATA_TIMING_RECOVER	= (1 << 5),
	ATA_TIMING_DMACK_HOLD	= (1 << 6),
	ATA_TIMING_CYCLE	= (1 << 7),
	ATA_TIMING_UDMA		= (1 << 8),
	ATA_TIMING_ALL		= ATA_TIMING_SETUP | ATA_TIMING_ACT8B |
				  ATA_TIMING_REC8B | ATA_TIMING_CYC8B |
				  ATA_TIMING_ACTIVE | ATA_TIMING_RECOVER |
				  ATA_TIMING_DMACK_HOLD | ATA_TIMING_CYCLE |
				  ATA_TIMING_UDMA,

	/* ACPI constants */
	ATA_ACPI_FILTER_SETXFER	= 1 << 0,
	ATA_ACPI_FILTER_LOCK	= 1 << 1,
	ATA_ACPI_FILTER_DIPM	= 1 << 2,
	ATA_ACPI_FILTER_FPDMA_OFFSET = 1 << 3,	/* FPDMA non-zero offset */
	ATA_ACPI_FILTER_FPDMA_AA = 1 << 4,	/* FPDMA auto activate */

	ATA_ACPI_FILTER_DEFAULT	= ATA_ACPI_FILTER_SETXFER |
				  ATA_ACPI_FILTER_LOCK |
				  ATA_ACPI_FILTER_DIPM,
};

enum ata_xfer_mask {
	ATA_MASK_PIO		= ((1U << ATA_NR_PIO_MODES) - 1) << ATA_SHIFT_PIO,
	ATA_MASK_MWDMA		= ((1U << ATA_NR_MWDMA_MODES) - 1) << ATA_SHIFT_MWDMA,
	ATA_MASK_UDMA		= ((1U << ATA_NR_UDMA_MODES) - 1) << ATA_SHIFT_UDMA,
};

enum hsm_task_states {
	HSM_ST_IDLE,		/* no command on going */
	HSM_ST_FIRST,		/* (waiting the device to)
				   write CDB or first data block */
	HSM_ST,			/* (waiting the device to) transfer data */
	HSM_ST_LAST,		/* (waiting the device to) complete command */
	HSM_ST_ERR,		/* error */
};

enum ata_completion_errors {
	AC_ERR_OK		= 0,	    /* no error */
	AC_ERR_DEV		= (1 << 0), /* device reported error */
	AC_ERR_HSM		= (1 << 1), /* host state machine violation */
	AC_ERR_TIMEOUT		= (1 << 2), /* timeout */
	AC_ERR_MEDIA		= (1 << 3), /* media error */
	AC_ERR_ATA_BUS		= (1 << 4), /* ATA bus error */
	AC_ERR_HOST_BUS		= (1 << 5), /* host bus error */
	AC_ERR_SYSTEM		= (1 << 6), /* system error */
	AC_ERR_INVALID		= (1 << 7), /* invalid argument */
	AC_ERR_OTHER		= (1 << 8), /* unknown */
	AC_ERR_NODEV_HINT	= (1 << 9), /* polling device detection hint */
	AC_ERR_NCQ		= (1 << 10), /* marker for offending NCQ qc */
};

/*
 * Link Power Management (LPM) policies.
 *
 * The default LPM policy to use for a device link is defined using these values
 * with the CONFIG_SATA_MOBILE_LPM_POLICY config option and applied through the
 * target_lpm_policy field of struct ata_port.
 *
 * If you alter this, you also need to alter the policy names used with the
 * sysfs attribute link_power_management_policy defined in libata-sata.c.
 */
enum ata_lpm_policy {
	/* Keep firmware settings */
	ATA_LPM_UNKNOWN,
	/* No power savings (maximum performance) */
	ATA_LPM_MAX_POWER,
	/* HIPM (Partial) */
	ATA_LPM_MED_POWER,
	/* HIPM (Partial) and DIPM (Partial and Slumber) */
	ATA_LPM_MED_POWER_WITH_DIPM,
	/* HIPM (Partial and DevSleep) and DIPM (Partial and Slumber) */
	ATA_LPM_MIN_POWER_WITH_PARTIAL,
	/* HIPM (Slumber and DevSleep) and DIPM (Partial and Slumber) */
	ATA_LPM_MIN_POWER,
};

enum ata_lpm_hints {
	ATA_LPM_EMPTY		= (1 << 0), /* port empty/probing */
	ATA_LPM_HIPM		= (1 << 1), /* may use HIPM */
	ATA_LPM_WAKE_ONLY	= (1 << 2), /* only wake up link */
};

/* forward declarations */
struct scsi_device;
struct ata_port_operations;
struct ata_port;
struct ata_link;
struct ata_queued_cmd;

/* typedefs */
typedef void (*ata_qc_cb_t) (struct ata_queued_cmd *qc);
typedef int (*ata_prereset_fn_t)(struct ata_link *link, unsigned long deadline);
typedef int (*ata_reset_fn_t)(struct ata_link *link, unsigned int *classes,
			      unsigned long deadline);
typedef void (*ata_postreset_fn_t)(struct ata_link *link, unsigned int *classes);

extern struct device_attribute dev_attr_unload_heads;
#ifdef CONFIG_SATA_HOST
extern struct device_attribute dev_attr_link_power_management_supported;
extern struct device_attribute dev_attr_link_power_management_policy;
extern struct device_attribute dev_attr_ncq_prio_supported;
extern struct device_attribute dev_attr_ncq_prio_enable;
extern struct device_attribute dev_attr_em_message_type;
extern struct device_attribute dev_attr_em_message;
extern struct device_attribute dev_attr_sw_activity;
#endif

enum sw_activity {
	OFF,
	BLINK_ON,
	BLINK_OFF,
};

struct ata_taskfile {
	unsigned long		flags;		/* ATA_TFLAG_xxx */
	u8			protocol;	/* ATA_PROT_xxx */

	u8			ctl;		/* control reg */

	u8			hob_feature;	/* additional data */
	u8			hob_nsect;	/* to support LBA48 */
	u8			hob_lbal;
	u8			hob_lbam;
	u8			hob_lbah;

	union {
		u8		error;
		u8		feature;
	};
	u8			nsect;
	u8			lbal;
	u8			lbam;
	u8			lbah;

	u8			device;

	union {
		u8		status;
		u8		command;
	};

	u32			auxiliary;	/* auxiliary field */
						/* from SATA 3.1 and */
						/* ATA-8 ACS-3 */
};

#ifdef CONFIG_ATA_SFF
struct ata_ioports {
	void __iomem		*cmd_addr;
	void __iomem		*data_addr;
	void __iomem		*error_addr;
	void __iomem		*feature_addr;
	void __iomem		*nsect_addr;
	void __iomem		*lbal_addr;
	void __iomem		*lbam_addr;
	void __iomem		*lbah_addr;
	void __iomem		*device_addr;
	void __iomem		*status_addr;
	void __iomem		*command_addr;
	void __iomem		*altstatus_addr;
	void __iomem		*ctl_addr;
#ifdef CONFIG_ATA_BMDMA
	void __iomem		*bmdma_addr;
#endif /* CONFIG_ATA_BMDMA */
	void __iomem		*scr_addr;
};
#endif /* CONFIG_ATA_SFF */

struct ata_host {
	spinlock_t		lock;
	struct device 		*dev;
	void __iomem * const	*iomap;
	unsigned int		n_ports;
	unsigned int		n_tags;			/* nr of NCQ tags */
	void			*private_data;
	struct ata_port_operations *ops;
	unsigned long		flags;
	struct kref		kref;

	struct mutex		eh_mutex;
	struct task_struct	*eh_owner;

	struct ata_port		*simplex_claimed;	/* channel owning the DMA */
	struct ata_port		*ports[];
};

struct ata_queued_cmd {
	struct ata_port		*ap;
	struct ata_device	*dev;

	struct scsi_cmnd	*scsicmd;
	void			(*scsidone)(struct scsi_cmnd *);

	struct ata_taskfile	tf;
	u8			cdb[ATAPI_CDB_LEN];

	unsigned long		flags;		/* ATA_QCFLAG_xxx */
	unsigned int		tag;		/* libata core tag */
	unsigned int		hw_tag;		/* driver tag */
	unsigned int		n_elem;
	unsigned int		orig_n_elem;

	int			dma_dir;

	unsigned int		sect_size;

	unsigned int		nbytes;
	unsigned int		extrabytes;
	unsigned int		curbytes;

	struct scatterlist	sgent;

	struct scatterlist	*sg;

	struct scatterlist	*cursg;
	unsigned int		cursg_ofs;

	unsigned int		err_mask;
	struct ata_taskfile	result_tf;
	ata_qc_cb_t		complete_fn;

	void			*private_data;
	void			*lldd_task;
};

struct ata_port_stats {
	unsigned long		unhandled_irq;
	unsigned long		idle_irq;
	unsigned long		rw_reqbuf;
};

struct ata_ering_entry {
	unsigned int		eflags;
	unsigned int		err_mask;
	u64			timestamp;
};

struct ata_ering {
	int			cursor;
	struct ata_ering_entry	ring[ATA_ERING_SIZE];
};

struct ata_cpr {
	u8			num;
	u8			num_storage_elements;
	u64			start_lba;
	u64			num_lbas;
};

struct ata_cpr_log {
	u8			nr_cpr;
	struct ata_cpr		cpr[] __counted_by(nr_cpr);
};

struct ata_cdl {
	/*
	 * Buffer to cache the CDL log page 18h (command duration descriptors)
	 * for SCSI-ATA translation.
	 */
	u8			desc_log_buf[ATA_LOG_CDL_SIZE];

	/*
	 * Buffer to handle reading the sense data for successful NCQ Commands
	 * log page for commands using a CDL with one of the limits policy set
	 * to 0xD (successful completion with sense data available bit set).
	 */
	u8			ncq_sense_log_buf[ATA_LOG_SENSE_NCQ_SIZE];
};

struct ata_device {
	struct ata_link		*link;
	unsigned int		devno;		/* 0 or 1 */
	unsigned int		quirks;		/* List of broken features */
	unsigned long		flags;		/* ATA_DFLAG_xxx */
	struct scsi_device	*sdev;		/* attached SCSI device */
	void			*private_data;
#ifdef CONFIG_ATA_ACPI
	union acpi_object	*gtf_cache;
	unsigned int		gtf_filter;
#endif
#ifdef CONFIG_SATA_ZPODD
	void			*zpodd;
#endif
	struct device		tdev;
	/* n_sector is CLEAR_BEGIN, read comment above CLEAR_BEGIN */
	u64			n_sectors;	/* size of device, if ATA */
	u64			n_native_sectors; /* native size, if ATA */
	unsigned int		class;		/* ATA_DEV_xxx */
	unsigned long		unpark_deadline;

	u8			pio_mode;
	u8			dma_mode;
	u8			xfer_mode;
	unsigned int		xfer_shift;	/* ATA_SHIFT_xxx */

	unsigned int		multi_count;	/* sectors count for
						   READ/WRITE MULTIPLE */
	unsigned int		max_sectors;	/* per-device max sectors */
	unsigned int		cdb_len;

	/* per-dev xfer mask */
	unsigned int		pio_mask;
	unsigned int		mwdma_mask;
	unsigned int		udma_mask;

	/* for CHS addressing */
	u16			cylinders;	/* Number of cylinders */
	u16			heads;		/* Number of heads */
	u16			sectors;	/* Number of sectors per track */

	union {
		u16		id[ATA_ID_WORDS]; /* IDENTIFY xxx DEVICE data */
		u32		gscr[SATA_PMP_GSCR_DWORDS]; /* PMP GSCR block */
	} ____cacheline_aligned;

	/* General Purpose Log Directory log page */
	u8			gp_log_dir[ATA_SECT_SIZE] ____cacheline_aligned;

	/* DEVSLP Timing Variables from Identify Device Data Log */
	u8			devslp_timing[ATA_LOG_DEVSLP_SIZE];

	/* NCQ send and receive log subcommand support */
	u8			ncq_send_recv_cmds[ATA_LOG_NCQ_SEND_RECV_SIZE];
	u8			ncq_non_data_cmds[ATA_LOG_NCQ_NON_DATA_SIZE];

	/* ZAC zone configuration */
	u32			zac_zoned_cap;
	u32			zac_zones_optimal_open;
	u32			zac_zones_optimal_nonseq;
	u32			zac_zones_max_open;

	/* Concurrent positioning ranges */
	struct ata_cpr_log	*cpr_log;

	/* Command Duration Limits support */
	struct ata_cdl		*cdl;

	/* error history */
	int			spdn_cnt;
	/* ering is CLEAR_END, read comment above CLEAR_END */
	struct ata_ering	ering;

	/* For EH */
	u8			sector_buf[ATA_SECT_SIZE] ____cacheline_aligned;
};

/* Fields between ATA_DEVICE_CLEAR_BEGIN and ATA_DEVICE_CLEAR_END are
 * cleared to zero on ata_dev_init().
 */
#define ATA_DEVICE_CLEAR_BEGIN		offsetof(struct ata_device, n_sectors)
#define ATA_DEVICE_CLEAR_END		offsetof(struct ata_device, ering)

struct ata_eh_info {
	struct ata_device	*dev;		/* offending device */
	u32			serror;		/* SError from LLDD */
	unsigned int		err_mask;	/* port-wide err_mask */
	unsigned int		action;		/* ATA_EH_* action mask */
	unsigned int		dev_action[ATA_MAX_DEVICES]; /* dev EH action */
	unsigned int		flags;		/* ATA_EHI_* flags */

	unsigned int		probe_mask;

	char			desc[ATA_EH_DESC_LEN];
	int			desc_len;
};

struct ata_eh_context {
	struct ata_eh_info	i;
	int			tries[ATA_MAX_DEVICES];
	int			cmd_timeout_idx[ATA_MAX_DEVICES]
					       [ATA_EH_CMD_TIMEOUT_TABLE_SIZE];
	unsigned int		classes[ATA_MAX_DEVICES];
	unsigned int		did_probe_mask;
	unsigned int		unloaded_mask;
	unsigned int		saved_ncq_enabled;
	u8			saved_xfer_mode[ATA_MAX_DEVICES];
	/* timestamp for the last reset attempt or success */
	unsigned long		last_reset;
};

struct ata_acpi_drive
{
	u32 pio;
	u32 dma;
} __packed;

