#!/bin/csh -f
#-----------------------------------------------------------------------------
# file:         runTests.cpp
#
# description:
#       Shell script to run HDFEOS test drivers.
#
# notes:
#       1) Any existing driver output files (except the samples) will be 
#          deleted.
#        
# usage:
#       runTests.cpp [GROUP] (ex. runTests.cpp point grid)
#
#       This runs the C++ test drivers.  These are the same drivers that are
#       used for 'C' but are compiled in C++.
#
# author:
#       Abe Taaheri
# history:
#       07-Jul-2014 AT  Used TOOLKIT testdrivers originally written by
#                       Curt Schafer / Emergent Information Syatems
#                       to create this and other scripts that this script runs
#-----------------------------------------------------------------------------
 
# initialize variables

set LANGS = ( C++ )
set SIMPLE_GRPS = ( )
set COMPLEX_GRPS = ( )
set ATTEPH_MODE = ( "" )

# initialize baselines

set SIMPLE_BASELINE = (  )
set COMPLEX_BASELINE = ( grid swath point )


# check the number of arguments coming in
    
if ($#argv < 1) then

    set SIMPLE_GRPS = ($SIMPLE_BASELINE)
    set COMPLEX_GRPS = ($COMPLEX_BASELINE)

else

    # setup simple groups according to input from command line ($argv)
    
    foreach GROUP ( ${SIMPLE_BASELINE} )
        if ( `echo $argv | grep $GROUP | wc -l` == "1" ) then
            set SIMPLE_GRPS = ( $SIMPLE_GRPS $GROUP )
        endif
    end
    
    
    # setup complex groups according to input from command line ($argv)
    
    foreach GROUP ( ${COMPLEX_BASELINE} )
        if ( `echo $argv | grep $GROUP | wc -l` == "1" ) then
            set COMPLEX_GRPS = ( $COMPLEX_GRPS $GROUP )
        endif
    end

    if ( "$SIMPLE_GRPS" == "" && "$COMPLEX_GRPS" == "" ) then
        set SIMPLE_GRPS = ($SIMPLE_BASELINE)
        set COMPLEX_GRPS = ($COMPLEX_BASELINE)
    endif
endif

echo ""; echo "The following languages will be used : ${LANGS}" 
echo ""; echo "The following tool groups will be processed :"
echo ""; echo $SIMPLE_GRPS $COMPLEX_GRPS |          \
              awk '{for(i=1;i<=NF;i++) print $i;}' | sort | \
              awk '{printf("%s ",$1);}' ; echo ""
echo ""; echo "If this isn't what you want, hit CTRL-C now."
sleep 5


# cleanup the previous test files (this only needs to be done
# for the SIMPLE groups, complex groups remove the files within
# their scripts)

if ( "$SIMPLE_GRPS" != "" ) then
    echo ""; echo "Cleaning up any previous test output files ..."
    $HDFEOS5_TST_SRC/Common/cleanup.csh $SIMPLE_GRPS >&! /dev/null
endif

echo ""; echo ""; echo ""; echo "Start running test drivers ..."; echo ""

# run the simple tool group drivers

foreach GROUP (${SIMPLE_GRPS})

    if (-d $HDFEOS5_TST_SRC/$GROUP) then

        # change to the group sub-directory

        cd $HDFEOS5_TST_SRC/$GROUP

        # run the drivers in that directory

        echo "";echo "Processing $GROUP drivers ..."
        ${HDFEOS5_TST_SRC}/Common/rundrivers.csh.cpp >&! ${HDFEOS5_TST_SRC}/Common/${GROUP}.log
        echo "";echo "Results are in ${HDFEOS5_TST_SRC}/Common/${GROUP}.log ...";echo ""

    else

        echo "";echo "Unable to find directory for $GROUP ...";echo ""

    endif

end


# change to the 'Common' sub-directory

cd $HDFEOS5_TST_SRC/Common

# run the complex tool group drivers

foreach GROUP (${COMPLEX_GRPS})

    echo ""; echo "Processing the ${GROUP} drivers ..."
    $HDFEOS5_TST_SRC/Common/run${GROUP}.csh.cpp $ATTEPH_MODE >&! $HDFEOS5_TST_SRC/Common/${GROUP}.log
    echo ""; echo "Results are in ${HDFEOS5_TST_SRC}/Common/${GROUP}.log ...";echo ""

end


echo ""; echo "Processing log files ..."; echo ""
 
set LOG_ERROR_FILE = $HDFEOS5_TST_SRC/Common/log_file_errors
 
\rm -f $LOG_ERROR_FILE

# setup the strings we need to look for

set ERROR_STRINGS = ( error      \
                      warning    \
                      undefined  \
                      fatal      \
                      failed     \
                      core       \
                      exit       \
                    )


# these drivers 'get in the way' because the 'error' strings we are looking for
# are included in these drivers, therefore we will invert the grep for these strings

set DRIVER_STRINGS = ( \
                     )

# setup the grep strings 

set GREP_STR = `echo $ERROR_STRINGS | awk '{for(i=1;i<NF;i++) printf("%s|",$i); printf("%s",$i);}' `

set INV_GREP_STR = `echo $DRIVER_STRINGS | awk '{for(i=1;i<NF;i++) printf("%s|",$i); printf("%s",$i);}' `

# grep all the log files

# First setup a dummy log file so that there is 
# always more than one file (this is to make sure
# that the filename is output from the grep command).

echo "" >! $$.log

egrep -i -n '('$GREP_STR')' *.log | egrep -v '('$INV_GREP_STR')' >&! ${LOG_ERROR_FILE}.$$

\rm -f $$.log

# if we have something in the log file then format it with 'awk'

if ( ! -z ${LOG_ERROR_FILE}.$$ ) then
    awk -F: 'BEGIN {printf("  LOGFILE    LINE#     MESSAGE\n\n")}   \
                   {printf("%9s    %4s      ",$1,$2);               \
                    for(i=3;i<NF;i++) printf("%s:",$i);             \
                    printf("%s\n",$i);       }' ${LOG_ERROR_FILE}.$$ >&! ${LOG_ERROR_FILE}
    \rm -f ${LOG_ERROR_FILE}.$$
    echo "Problems were found in some of the log files, check the"
    echo "file:  $LOG_ERROR_FILE for the errors."; echo ""
else
    mv ${LOG_ERROR_FILE}.$$ ${LOG_ERROR_FILE}
    echo "No problems found with the log files ..."; echo ""
endif

 
echo "Processing difference files ..."; echo ""
 
set DIFF_ERROR_FILE = $HDFEOS5_TST_SRC/Common/diff_file_errors
 
\rm -f $DIFF_ERROR_FILE
 
# search for 'c' difference files that have
# a number of lines different than whats expected

foreach diff_file ( `find $HDFEOS5_TST_SRC -name "*_c.diff" -type f -print` )
    if ( `grep "Machine Name" $diff_file | wc -l` == 2 ) then
        @ EXP_LINES = 16
    else
        @ EXP_LINES = 12
    endif
 
    if ( `wc -l $diff_file | awk '{print $1}'` != $EXP_LINES ) then
        echo $diff_file >>! ${DIFF_ERROR_FILE}.$$
    endif
end
 
if ( -e ${DIFF_ERROR_FILE}.$$ ) then
    awk 'BEGIN {printf("#\!/bin/csh -f\n")} \
               {printf("more %s\n",$0);}'    ${DIFF_ERROR_FILE}.$$ | sort >&! $DIFF_ERROR_FILE
    echo "Problems were found in some of the difference files, check the"
    echo "file:  $DIFF_ERROR_FILE for the filenames."; echo ""
    \rm -f ${DIFF_ERROR_FILE}.$$
    chmod 755 $DIFF_ERROR_FILE
else
    echo "No problems found with the difference files ..."; echo ""
endif
