// Copyright 2012-2015 Oliver Eilhard. All rights reserved.
// Use of this source code is governed by a MIT-license.
// See http://olivere.mit-license.org/license.txt for details.

package elastic

// ValueCountAggregation is a single-value metrics aggregation that counts
// the number of values that are extracted from the aggregated documents.
// These values can be extracted either from specific fields in the documents,
// or be generated by a provided script. Typically, this aggregator will be
// used in conjunction with other single-value aggregations.
// For example, when computing the avg one might be interested in the
// number of values the average is computed over.
// See: http://www.elasticsearch.org/guide/en/elasticsearch/reference/current/search-aggregations-metrics-valuecount-aggregation.html
type ValueCountAggregation struct {
	field           string
	script          string
	scriptFile      string
	lang            string
	format          string
	params          map[string]interface{}
	subAggregations map[string]Aggregation
}

func NewValueCountAggregation() ValueCountAggregation {
	a := ValueCountAggregation{
		params:          make(map[string]interface{}),
		subAggregations: make(map[string]Aggregation),
	}
	return a
}

func (a ValueCountAggregation) Field(field string) ValueCountAggregation {
	a.field = field
	return a
}

func (a ValueCountAggregation) Script(script string) ValueCountAggregation {
	a.script = script
	return a
}

func (a ValueCountAggregation) ScriptFile(scriptFile string) ValueCountAggregation {
	a.scriptFile = scriptFile
	return a
}

func (a ValueCountAggregation) Lang(lang string) ValueCountAggregation {
	a.lang = lang
	return a
}

func (a ValueCountAggregation) Format(format string) ValueCountAggregation {
	a.format = format
	return a
}

func (a ValueCountAggregation) Param(name string, value interface{}) ValueCountAggregation {
	a.params[name] = value
	return a
}

func (a ValueCountAggregation) SubAggregation(name string, subAggregation Aggregation) ValueCountAggregation {
	a.subAggregations[name] = subAggregation
	return a
}

func (a ValueCountAggregation) Source() interface{} {
	// Example:
	//	{
	//    "aggs" : {
	//      "grades_count" : { "value_count" : { "field" : "grade" } }
	//    }
	//	}
	// This method returns only the { "value_count" : { "field" : "grade" } } part.

	source := make(map[string]interface{})
	opts := make(map[string]interface{})
	source["value_count"] = opts

	// ValuesSourceAggregationBuilder
	if a.field != "" {
		opts["field"] = a.field
	}
	if a.script != "" {
		opts["script"] = a.script
	}
	if a.scriptFile != "" {
		opts["script_file"] = a.scriptFile
	}
	if a.lang != "" {
		opts["lang"] = a.lang
	}
	if a.format != "" {
		opts["format"] = a.format
	}
	if len(a.params) > 0 {
		opts["params"] = a.params
	}

	// AggregationBuilder (SubAggregations)
	if len(a.subAggregations) > 0 {
		aggsMap := make(map[string]interface{})
		source["aggregations"] = aggsMap
		for name, aggregate := range a.subAggregations {
			aggsMap[name] = aggregate.Source()
		}
	}

	return source
}
