require "rbconfig"
require "gtkglext"
require "vtk"


class GtkGLExtVTKRenderWindowInteractor < Gtk::DrawingArea

  def initialize
    super

    glconfig = Gdk::GLConfig.new(Gdk::GLConfig::MODE_RGB|
                                   Gdk::GLConfig::MODE_DEPTH|
                                   Gdk::GLConfig::MODE_DOUBLE)
    if !glconfig
      glconfig = Gdk::GLConfig.new(Gdk::GLConfig::MODE_RGB|
                                   Gdk::GLConfig::MODE_DEPTH)
    end
    set_gl_capability(glconfig)

    @RenderWindow = Vtk::RenderWindow.new

    # private attributes
    @Created = false
    @ActiveButton = 0

    @Iren = Vtk::GenericRenderWindowInteractor.new
    @Iren.SetRenderWindow(@RenderWindow)
    Vtk::InteractorStyleSwitch.SafeDownCast(@Iren.GetInteractorStyle).SetCurrentStyleToTrackballCamera

    createTimer = Proc.new{|obj, event|
      Gtk.timeout_add(10){ @Iren.TimerEvent }
    }
    command = Vtk::CallbackCommand.new
    command.SetCallback( createTimer )
    @Iren.AddObserver('CreateTimerEvent', command)
    destroyTimer = Proc.new{|obj, event|
    }
    command = Vtk::CallbackCommand.new
    command.SetCallback( destroyTimer )
    @Iren.AddObserver('DestroyTimerEvent', command)
    self.ConnectSignals
        
    # need this to be able to handle key_press events.
    self.set_flags(Gtk::Window::CAN_FOCUS)
  end

  def set_size_request(w, h)
    super
    @RenderWindow.SetSize(w, h)
    @Iren.SetSize(w, h)
    @Iren.ConfigureEvent
  end
	
  def ConnectSignals
    self.signal_connect("realize"){|wid,event| OnRealize(wid, event) }
    self.signal_connect("expose_event"){|wid,event| OnExpose(wid, event) }
    self.signal_connect("configure_event"){|wid,event| OnConfigure(wid,event) }
    self.signal_connect("button_press_event"){|wid,event| OnButtonDown(wid, event) }
    self.signal_connect("button_release_event"){|wid,event| OnButtonUp(wid, event) }
    self.signal_connect("motion_notify_event"){|wid,event| OnMouseMove(wid, event) }
    self.signal_connect("enter_notify_event"){|wid,event| OnEnter(wid, event) }
    self.signal_connect("leave_notify_event"){|wid,event| OnLeave(wid, event) }
    self.signal_connect("key_press_event"){|wid,event| OnKeyPress(wid, event) }
    self.signal_connect("delete_event"){|wid,event| OnDestroy }
    self.add_events(Gdk::Event::EXPOSURE_MASK| Gdk::Event::BUTTON_PRESS_MASK |
                        Gdk::Event::BUTTON_RELEASE_MASK |
                        Gdk::Event::KEY_PRESS_MASK |
                        Gdk::Event::POINTER_MOTION_MASK |
                        Gdk::Event::POINTER_MOTION_HINT_MASK |
                        Gdk::Event::ENTER_NOTIFY_MASK | Gdk::Event::LEAVE_NOTIFY_MASK)
  end

  def GetRenderWindow
    return @RenderWindow
  end

  def GetInteractor
    return @Iren
  end

  def Render
    if @Created
      @RenderWindow.Render
    end
  end

  def OnRealize(widget, event)
    until @Created
      self.realize
      if Config::CONFIG["host_os"] =~ /win32/
        win_id = self.window.handle.to_s
      else
        win_id = self.window.xid.to_s
      end

      @RenderWindow.SetWindowInfo(win_id)
      @Created = true
    end
    return true
  end
    
  def OnConfigure(widget, event)
    @Iren.SetSize(event.width, event.height)
    @Iren.ConfigureEvent
    self.Render
    return true
  end

  def OnExpose(widget, event)
    self.Render
    return true
  end

  def OnDestroy
    self.hide
    self.destroy
    return true
  end

  def GetCtrlShift(event)
      ctrl, shift = 0, 0        
    if ((event.state & Gdk::Window::ModifierType::CONTROL_MASK) == Gdk::Window::ModifierType::CONTROL_MASK)
      ctrl = 1
    end
    if ((event.state & Gdk::Window::ModifierType::SHIFT_MASK) == Gdk::Window::ModifierType::SHIFT_MASK)
      shift = 1
    end
    return ctrl, shift
  end

  def OnButtonDown(wid, event)
    m = self.pointer
    ctrl, shift = self.GetCtrlShift(event)
    @Iren.SetEventInformationFlipY(m[0], m[1], ctrl, shift)
    button = event.button
    if button == 3
      @Iren.RightButtonPressEvent
      return true
    elsif button == 1
      @Iren.LeftButtonPressEvent
      return true
    elsif button == 2
      @Iren.MiddleButtonPressEvent
      return true
    else
      return false
    end
  end
    
  def OnButtonUp(wid, event)
    m = self.pointer
    ctrl, shift = self.GetCtrlShift(event)
    @Iren.SetEventInformationFlipY(m[0], m[1], ctrl, shift)
    button = event.button
    if button == 3
      @Iren.RightButtonReleaseEvent
      return true
    elsif button == 1
      @Iren.LeftButtonReleaseEvent
      return true
    elsif button == 2
      @Iren.MiddleButtonReleaseEvent
      return true
    else
      return false
    end
  end

  def OnMouseMove(wid, event)
    m = self.pointer
    ctrl, shift = self.GetCtrlShift(event)
    @Iren.SetEventInformationFlipY(m[0], m[1], ctrl, shift)
    @Iren.MouseMoveEvent
    return true
  end

  def OnEnter(wid, event)
    self.grab_focus
    m = self.pointer
    ctrl, shift = self.GetCtrlShift(event)
    @Iren.SetEventInformationFlipY(m[0], m[1], ctrl, shift)
    @Iren.EnterEvent
    return true
  end

  def OnLeave(wid, event)
    m = self.pointer
    ctrl, shift = self.GetCtrlShift(event)
    @Iren.SetEventInformationFlipY(m[0], m[1], ctrl, shift)
    @Iren.LeaveEvent
    return true
  end

  def OnKeyPress(wid, event)
    m = self.pointer
    ctrl, shift = self.GetCtrlShift(event)
    keycode = event.keyval
#    keycode, keysym = event.keyval, event.string
    key = 0
    if keycode < 256
      key = keycode
    end
    @Iren.SetEventInformationFlipY(m[0], m[1], ctrl, shift, key.chr)
#    @Iren.SetEventInformationFlipY(m[0], m[1], ctrl, shift, key.chr, 0, keysym)
    @Iren.KeyPressEvent
    @Iren.CharEvent
    return true
  end

  def OnKeyRelease(wid, event)
    m = self.pointer
    ctrl, shift = self.GetCtrlShift(event)
    keycode = event.keyval
#    keycode, keysym = event.keyval, event.string
    key = chr(0)
    if keycode < 256
      key = chr(keycode)
    end
    @Iren.SetEventInformationFlipY(m[0], m[1], ctrl, shift, key.chr)
#    @Iren.SetEventInformationFlipY(m[0], m[1], ctrl, shift, key.chr, 0, keysym)
    @Iren.KeyReleaseEvent
    return true
  end

  def Initialize
    if @Created
      @Iren.Initialize
    end
  end

  def Start
    if @Created
      @Iren.Start
    end
  end
	    
  def SetPicker(picker)
    @Iren.SetPicker(picker)
  end

  def GetPicker(picker)
    return @Iren.GetPicker
  end

end

############################
if $0 == __FILE__

  Gtk.init
  Gtk::GL.init

  window = Gtk::Window.new(Gtk::Window::TOPLEVEL)
  window.set_title("A GtkVTKRenderWindow Demo!")
  window.signal_connect("destroy"){ Gtk.main_quit }
  window.signal_connect("delete_event"){ Gtk.main_quit }
  window.set_border_width(10)

  # A VBox into which widgets are packed.
  vbox = Gtk::VBox.new(false, 3)
  window.add(vbox)
  vbox.show

  # The GtkVTKRenderWindow
  gvtk = GtkGLExtVTKRenderWindowInteractor.new
  #gvtk.SetDesiredUpdateRate(1000)
  gvtk.set_size_request(400, 400)
  vbox.pack_start(gvtk)
  gvtk.show
  gvtk.Initialize
  gvtk.Start
  # prevents 'q' from exiting the app.
  exit_app = Proc.new{|obj, event|
  }
  command = Vtk::CallbackCommand.new
  command.SetCallback( exit_app )
#  gvtk.AddObserver("ExitEvent", command )

  # The VTK stuff.
  cone = Vtk::ConeSource.new
  cone.SetResolution(80)
  coneMapper = Vtk::PolyDataMapper.new
  coneMapper.SetInput(cone.GetOutput)
  #coneActor = Vtk::LODActor.new
  coneActor = Vtk::Actor.new
  coneActor.SetMapper(coneMapper)    
  coneActor.GetProperty.SetColor(0.5, 0.5, 1.0)
  ren = Vtk::Renderer.new
  gvtk.GetRenderWindow.AddRenderer(ren)
  ren.AddActor(coneActor)

  # A simple quit button
  quit = Gtk::Button.new("Quit!")
  quit.signal_connect("clicked"){ Gtk.main_quit }
  vbox.pack_start(quit)
  quit.show

  # show the main window and start event processing.
  window.show
  Gtk.main

end
