
// -*- mode: c++; c-basic-offset:4 -*-

// This file is part of libnc-dods, A C++ implementation of netCDF 3 which
// supports reading from local files and OPeNDAP servers.

// Copyright (c) 2004 OPeNDAP, Inc.
// Author: James Gallagher <jgallagher@opendap.org>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// You can contact OPeNDAP, Inc. at PO Box 112, Saunderstown, RI. 02874-0112.
 
#include <cppunit/TextTestRunner.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/extensions/HelperMacros.h>

// Tests for nc_util.c 2/25/05 jhrg

#include <netcdf.h>

// #define DODS_DEBUG 

#include <list>

#include "BaseType.h"
#include "NCSequence.h"
#include "NCStructure.h"
#include "NCArray.h"
#include "NCByte.h"
#include "NCFloat64.h"
#include "NCInt32.h"
#include "NCInt16.h"

#include "AttrTable.h"

#include "nc_util.h"

#include "debug.h"

using namespace CppUnit;
using namespace std;

class NCUtilTest : public TestFixture {
private:
    AttrTable *flat;
    AttrTable *nested;

public: 
    NCUtilTest() {}
    ~NCUtilTest() {}

    void setUp() {
        flat = new AttrTable;
        flat->append_attr("name", "String", "Zephyr");
        flat->append_attr("weight", "Float64", "92.5");
        
        nested = new AttrTable;
        nested->append_attr("name", "String", "Pets");
        nested->append_container(flat, "Pet1");
    }

    void tearDown() {
        // Deleting nested will also delete flat
        delete nested; nested = 0;
    }

    CPPUNIT_TEST_SUITE( NCUtilTest );

    CPPUNIT_TEST(number_of_elements_test);
    CPPUNIT_TEST(transfer_attributes_to_list_test1);
    CPPUNIT_TEST(transfer_attributes_to_list_test2);
    CPPUNIT_TEST(attribute_list_to_table_test1);
    CPPUNIT_TEST(find_child_sequence_test);

    CPPUNIT_TEST_SUITE_END();

    void find_child_sequence_test() {
        NCSequence *s = new NCSequence("top");
        
        NCByte *b = new NCByte("b");
        s->add_var(b);
        delete b; b = 0;
        CPPUNIT_ASSERT(s->find_child_sequence() == 0);

        NCFloat64 *f64 = new NCFloat64("f64");
        s->add_var(f64);
        delete f64; f64 = 0;
        CPPUNIT_ASSERT(s->find_child_sequence() == 0);

        NCStructure *str = new NCStructure("Structure");
        NCInt32 *i32 = new NCInt32("i32");
        str->add_var(i32);
        delete i32; i32 = 0;
        s->add_var(str);
        delete str; str = 0;
        CPPUNIT_ASSERT(s->find_child_sequence() == 0);

        NCArray *a1 = new NCArray("a1");
        NCInt16 *i16 = new NCInt16("i16");
        a1->add_var(i16);
        delete i16; i16 = 0;
        s->add_var(a1);
        delete a1; a1 = 0;
        CPPUNIT_ASSERT(s->find_child_sequence() == 0);
        
        // Set up a Sequence s with a Sequence
        NCSequence *s2 = new NCSequence("s2");
        i16 = new NCInt16("i16_2");
        s2->add_var(i16);
        delete i16; i16 = 0;
        s->add_var(s2);
        delete s2; s2 = 0;
        BaseType *btp = s->find_child_sequence();
        CPPUNIT_ASSERT(btp && btp->name() == "s2");
        
        // Set up a Sequence s with an Array of Sequence
        delete s;               // New top level sequence
        s = new NCSequence("top2");
        NCArray *a2 = new NCArray("a2");
        s2 = new NCSequence("s2");
        i16 = new NCInt16("i16_2");
        s2->add_var(i16);
        delete i16; i16 = 0;
        a2->add_var(s2);
        s->add_var(a2);
        delete a2; a2 = 0;
        delete s2; s2 = 0;
        btp = s->find_child_sequence();        
        CPPUNIT_ASSERT(btp && btp->name() == "s2");
        
        // Set up a Seq --> Structure --> Seq --> Int16
        delete s;
        s = new NCSequence("top3");
        
        str = new NCStructure("str2");
        s2 = new NCSequence("s2_2");
        i16 = new NCInt16("i16_2");
        s2->add_var(i16);
        delete i16; i16 = 0;
        str->add_var(s2);
        delete s2; s2 = 0;
        s->add_var(str);
        delete str; str = 0;

        btp = s->find_child_sequence();        
        CPPUNIT_ASSERT(btp && btp->name() == "s2_2");        
        
        delete s; s = 0;
    }
    
    void attribute_list_to_table_test2() {
        EntryList *el = transfer_attributes_to_list(nested);
        AttrTable *at = attributes_list_to_table(el);
        
        AttrTable::Attr_iter i = at->attr_begin();
        CPPUNIT_ASSERT(at->get_name(i) == "name");
        CPPUNIT_ASSERT(at->get_attr_type(i) == Attr_string);
        vector<string> *v = at->get_attr_vector(i);
        CPPUNIT_ASSERT(*(v->begin()) == "Pets");
        ++i;
        DBG(cerr << "i->name: " << i->name << endl);
        CPPUNIT_ASSERT(at->get_name(i) == "Pet1:name");
        CPPUNIT_ASSERT(at->get_attr_type(i) == Attr_string);
        v = at->get_attr_vector(i);
        CPPUNIT_ASSERT(*(v->begin()) == "Zephyr");
        ++i;
        CPPUNIT_ASSERT(at->get_name(i) == "Pet1:weight");
        CPPUNIT_ASSERT(at->get_attr_type(i) == Attr_float64);
        v = at->get_attr_vector(i);
        CPPUNIT_ASSERT(*(v->begin()) == "92.5");
        
        delete at; at = 0;
    }

    void attribute_list_to_table_test1() {
        EntryList *el = transfer_attributes_to_list(flat);
        AttrTable *at = attributes_list_to_table(el);
        delete el; el = 0;
        AttrTable::Attr_iter i = at->attr_begin();
        CPPUNIT_ASSERT(at->get_name(i) == "name");
        CPPUNIT_ASSERT(at->get_attr_type(i) == Attr_string);
        vector<string> *v = at->get_attr_vector(i);
        CPPUNIT_ASSERT(*(v->begin()) == "Zephyr");
        ++i;
        CPPUNIT_ASSERT(at->get_name(i) == "weight");
        CPPUNIT_ASSERT(at->get_attr_type(i) == Attr_float64);
        v = at->get_attr_vector(i);
        CPPUNIT_ASSERT(*(v->begin()) == "92.5");
        
        delete at; at = 0;
    }
    
    void transfer_attributes_to_list_test1() {
        EntryList *el = transfer_attributes_to_list(flat);
        EntryList::iterator i = el->begin();
        CPPUNIT_ASSERT(i->name == "name");
        CPPUNIT_ASSERT(i->type == Attr_string);
        CPPUNIT_ASSERT(*(i->attr->begin()) == "Zephyr");
        ++i;
        CPPUNIT_ASSERT(i->name == "weight");
        CPPUNIT_ASSERT(i->type == Attr_float64);
        CPPUNIT_ASSERT(*(i->attr->begin()) == "92.5");
        
        delete el; el = 0;
    }
    
    void transfer_attributes_to_list_test2() {
        EntryList *el = transfer_attributes_to_list(nested);
        EntryList::iterator i = el->begin();
        CPPUNIT_ASSERT(i->name == "name");
        CPPUNIT_ASSERT(i->type == Attr_string);
        CPPUNIT_ASSERT(*(i->attr->begin()) == "Pets");
        ++i;
        DBG(cerr << "i->name: " << i->name << endl);
        CPPUNIT_ASSERT(i->name == "Pet1:name");
        CPPUNIT_ASSERT(i->type == Attr_string);
        CPPUNIT_ASSERT(*(i->attr->begin()) == "Zephyr");
        ++i;
        CPPUNIT_ASSERT(i->name == "Pet1:weight");
        CPPUNIT_ASSERT(i->type == Attr_float64);
        CPPUNIT_ASSERT(*(i->attr->begin()) == "92.5");
        
        delete el; el = 0;
    }
    
    void number_of_elements_test() {
        CPPUNIT_ASSERT(number_of_elements("[0:1:19]") == 20);
        CPPUNIT_ASSERT(number_of_elements("[0:19]") == 20);
        CPPUNIT_ASSERT(number_of_elements("[0:2:19]") == 10);
        CPPUNIT_ASSERT(number_of_elements("[0:3:19]") == 7);
        CPPUNIT_ASSERT(number_of_elements("[0:19][0:19]") == 400);
        CPPUNIT_ASSERT(number_of_elements("[0:9][0:9][0:9][0:9][0:2:9]") == 50000);
    }
};

CPPUNIT_TEST_SUITE_REGISTRATION(NCUtilTest);

int 
main( int argc, char* argv[] )
{
    CppUnit::TextTestRunner runner;
    runner.addTest( CppUnit::TestFactoryRegistry::getRegistry().makeTest() );

    runner.run();

    return 0;
}
