
// -*- mode: c++; c-basic-offset:4 -*-

// This file is part of libdap, A C++ implementation of the OPeNDAP Data
// Access Protocol.

// Copyright (c) 2002,2003,2005 OPeNDAP, Inc.
// Author: James Gallagher <jgallagher@opendap.org>
//         Reza Nekovei <reza@intcomm.net>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// You can contact OPeNDAP, Inc. at PO Box 112, Saunderstown, RI. 02874-0112.

// (c) COPYRIGHT OPeNDAP, Inc 2004
// Please read the full copyright statement in the file COPYRIGHT.
//
// Authors:
//      jhrg            James Gallagher <jgallagher@opendap.org>

// Utility functions used by the netCDF CL code.

#include "config_nc.h"

static char rcsid[] not_used ={"$Id: nc_util.cc 11906 2005-08-08 19:51:43Z root $"};

#include <sstream>

// #define DODS_DEBUG

#include "AttrTable.h"
#include "InternalErr.h"
#include "debug.h"

#include "Dnetcdf.h"
#include "Dncx.h"
#include "nc_util.h"

#include "NCAccess.h"
#include "NCGrid.h"
#include "NCArray.h"
#include "NCSequence.h"
#include "NCStructure.h"

/** Defined in lnetcdf/lerror. This is not included in a header (like
    lnetcdf.h because doing so would require all sources that use lnetcdf.h
    to also include Error.h 03/02/05 jhrg */
extern void set_opendap_error(const Error &);

/** Starting with \e child, look 'up the chain' to find a parent, grandparent,
    et c., BaseType that is a Sequence.
    
    @brief Is there an ancestral Sequence?
    @param child Starting point for the search.
    @return A pointer to the Sequence that contains \e child or NULL if there 
    is no ancestral sequence. */
BaseType *
find_ancestral_sequence(BaseType *child)
{
    if (!child)
        return 0;
    else if (child->type() == dods_sequence_c)
        return child;
    else 
        return find_ancestral_sequence(child->get_parent());
}

/** Copy the values of an attribute into a supplied buffer. The buffer must 
    be allocated prior to calling this function.
    
    @param attr Read the attribute from this table.
    @param p The Attr_iter which references the attribute.
    @param values The buffer; a value-result parameter
    @return The number of bytes read.
    @exception InternalErr Thrown if the type of the attribute is not
    supported. */
int
copy_attributes(AttrTable &attr, AttrTable::Attr_iter &p, void *values) 
    throw(InternalErr)
{
    // Get the type and size of the attribute
    AttrType type = attr.get_attr_type(p);
    unsigned int attr_siz = attr.get_attr_num(p);
    
    unsigned int i;
    // integers and floats are converted to binary and added to the pointer
    switch (type) {
      case Attr_int16: {
	    short *sp = (short *) values;
	    for (i=0; i<attr_siz; i++)
	        *(sp+i) = atol(attr.get_attr(p, i).c_str());
        return attr_siz;
      }

      case Attr_uint16: {
	    unsigned short *usp = (unsigned short *) values;
	    for (i=0; i<attr_siz; i++)
	        *(usp+i) = atol(attr.get_attr(p, i).c_str());
        return attr_siz;
      }

      case Attr_int32: {
	    nclong *lp = (nclong *) values;
	    for (i=0; i<attr_siz; i++)
	        *(lp+i) = atol(attr.get_attr(p, i).c_str());
        return attr_siz;
      }

      case Attr_uint32: {
	    unsigned long *ulp = (unsigned long *) values;
	    for (i=0; i<attr_siz; i++)
	        *(ulp+i) = strtoul(attr.get_attr(p, i).c_str(), 0, 0);
        return attr_siz;
      }

      case Attr_float32: {
	    float *fp = (float *) values;
	    for (i=0; i<attr_siz; i++)
	        *(fp+i)=atof(attr.get_attr(p, i).c_str());
        return attr_siz;
      }

      case Attr_float64: {
	    double *dp = (double *) values;
	    for (i=0; i<attr_siz; i++)
	        *(dp+i)=atof(attr.get_attr(p, i).c_str());
        return attr_siz;
      }

      // bytes are added one at the time to the pointer address
      case Attr_byte: {
	    char *cp = (char *) values;
	    for (i=0; i<attr_siz; i++)
	        *(cp+i)=(char) atoi(attr.get_attr(p, i).c_str());
        return attr_siz;
      }

      // strings are added to the pointer (seperated by '\n' and ended by
      // '\0')
      case Attr_string: {
	    char *cp = (char *) values;
	    int offset = 0;
	    for (i=0; i<attr_siz; i++) {
	        string attrV = attr.get_attr(p, i);
	        size_t atsiz = attrV.length() - 2 ; // size minus quotes  
	        string Attribute = attrV.substr(1, atsiz); // remove quotes

                memcpy(cp+offset, Attribute.c_str(), atsiz); 
	        offset += atsiz; 
	        if (attr_siz > 1) {
		        // Separate strings by EOL for more than 1 string (not a
		        // DODS netcdf server).
		        *(cp+offset) = '\n'; 
		        offset++;
	        }
	    }
	    // End the string attribute by EOS, if array of string was sent
	    // (not a netCDF server).
	    if (attr_siz > 1)
	        *(cp+offset-1) = '\0';
            
        return offset;
      }
      
      default:
	    throw InternalErr("cast_attr_type called with an invalid type.");
    }
}

/** Transfer the entires in this AttrTable to a list. The caller must delete
    the returned list after it's done with it. */
EntryList *
transfer_attributes_to_list(AttrTable *a, const string &name)
{
    EntryList *el = new EntryList;
    
    AttrTable::Attr_iter a_iter;
    for (a_iter = a->attr_begin(); a_iter != a->attr_end(); a_iter++) {

        if (a->is_container(a_iter)) {
            EntryList *sel 
                = transfer_attributes_to_list(a->get_attr_table(a_iter),
                                              (name.empty()
                                               ? a->get_name(a_iter)
                                               : name + ":" + a->get_name(a_iter)));
            el->splice(el->end(), *sel);
            delete sel;
        }
        else {
            AttrTable::entry e;
          
            e.name = (name.empty()) ? a->get_name(a_iter) : name + ":" + a->get_name(a_iter);
            e.type = a->get_attr_type(a_iter);
            
            vector<string> *vs = a->get_attr_vector(a_iter);
            vector<string> *nvs = new vector<string>;
            *nvs = *vs;
            e.attr = nvs;
            el->push_back(e);
        }
    }
    
    return el;
}

AttrTable *
attributes_list_to_table(EntryList *el)
{
    AttrTable *at = new AttrTable;
    
    EntryList::iterator i = el->begin();
    EntryList::iterator end_i = el->end();
    while (i != end_i) {
        vector<string>::iterator j = i->attr->begin();
        vector<string>::iterator end_j = i->attr->end();
        while (j != end_j) {
            at->append_attr(i->name, AttrType_to_String(i->type), *j);
            ++j;
        }
        ++i;
    }
    
    return at;
}

/** What is the size (in bytes) of the given type.

    @param type The netCDF type number (0 == void, ..., 8 == double).
    @return The number of bytes used to represent \e type. */
static int
Dtypelen(int type) 
{
    switch(type) { // return the size of output type
      case Tvoid :
      case Ttext :
      case Tuchar :
      case Tschar :
        return((int)sizeof(char));
      case Tshort :
	    return((int)sizeof(short));
      case Tint : 
	    return((int)sizeof(int));
      case Tlong : 
	    return((int)sizeof(long));
      case Tfloat : 
	    return((int)sizeof(float));
      case Tdouble : 
	    return((int)sizeof(double));
      default:
	    return -1;
    }
}

/** Type conversion from DAP's on-the-wire representation to the local
    machine's representation. This function should rarely actually do
    anything since the DAP uses XDR to encode/decode data. In \e most cases
    this function will simply copy the buffer pointer. The caller must be
    careful to only free the buffer once; test the returned buffer to see if
    it different than \e tmpbufin.
    
    @note we need this function only for those rare cases when the OPeNDAP
    type is not what you'd expect. For example, suppose a \c short is really
    32-bits (I'm not even sure that can happen). The point is that the
    conversions in this function should rarely happen and that's why it makes
    sense to test the types and just copy the buffer pointer in the default
    case. The caller has to make sure to only delete the returned pointer
    when it's different than the pointer passed in!

    @note The string conversion code is now built into DODvario().

    @todo Change this to something that's only called when needed. For
    example, devise a compile-time test and have this code drop out on
    platforms where it's not needed.
    
    @param datatype The netCDF type of the data.
    @param nels Number of elements to convert.
    @param v_tmpbufin Pointer to a block of memory holding the data.
    @return Pointer to a block of memory holding the munged data. If this is 
    different than \e tmpbufin, the caller must free this block using delete[].
*/
static void *    
type_conv(nc_type datatype, int nels, void *v_tmpbufin)
{
    DBG(cerr << "Entering type_conv" << endl);
    
    // Cast void* to char* so pointer arithmetic will work in the code below.
    char *tmpbufin = static_cast<char*>(v_tmpbufin);
    
    if ((datatype == NC_FLOAT) 
	   && (sizeof(dods_float32) != nctypelen(datatype))) {
	   float *flt = reinterpret_cast<float *>(new char [(nels*nctypelen(datatype))]);
    	for (int id = 0; id < nels; id++) {
    	    *flt++ = float(*reinterpret_cast<dods_float32*>(tmpbufin));
            tmpbufin += sizeof(dods_float32);
        }
    	return flt;
    }
    else if ((datatype == NC_DOUBLE) 
    	     && (sizeof(dods_float64) != nctypelen(datatype))) {
    	double *dbl = reinterpret_cast<double*>(new char [(nels*nctypelen(datatype))]);
    	for (int id = 0; id < nels; id++) {
    	    *dbl++ = double(*reinterpret_cast<dods_float64*>(tmpbufin));
            tmpbufin += sizeof(dods_float64);
        }
    	return dbl;
    }
    else if ((datatype == NC_SHORT) 
    	     && (sizeof(dods_int16) != nctypelen(datatype))) {
    	short *sht = reinterpret_cast<short*>(new char [(nels*nctypelen(datatype))]);
    	for (int id = 0; id < nels; id++) {
    	    *sht++ = short(*reinterpret_cast<dods_int16*>(tmpbufin));
            tmpbufin += sizeof(dods_int16);
        }
    	return sht;
    }
    else if ((datatype == NC_LONG) 
    	     && (sizeof(dods_int32) != nctypelen(datatype))) {
    	nclong *lng = reinterpret_cast<nclong*>(new char [(nels*nctypelen(datatype))]);
    	for (int id = 0; id < nels; id++) {
    	    *lng++ = nclong(*reinterpret_cast<dods_int32*>(tmpbufin));
            tmpbufin += sizeof(nclong);
        }
    	return lng;
    }
    else {
    	return tmpbufin;
    }
}

/** Copy the values from tmpbufin to values. This function handles two
    kinds of type conversions. First it handles any conversion required
    to get the OPeNDAP types (the 'wire' types) to netCDF's types. Secondly,
    it converts from one netCDF type to another.
    
    @todo Improve this by converting directly from the opendap type to the
    requested netCDF type. In most cases, the netCDF type and the opendap
    type are one an the same.

    @param typep The type of the values in \e tmpbufin.
    @param outtype The desired type (for the values to be copied to \e values).
    @param nels The number of elements in \e tmpbufin.
    @param tmpbufin Input values.
    @param values Output values.
    @return A netCDF error code. */
int
convert_nc_type(nc_type typep, int outtype, int nels, void *tmpbufin, 
    void *values)
{
    // Convert from the OPeNDAP type to the netCDF type.
    // For most machines, this is a no-op and tmpbufout == tmpbufin
    char *tmpbufout =  (char *)type_conv(typep, nels, tmpbufin);
    int szof = nctypelen(typep);

    // Convert from the netCDF type matching the OPenDAP type to the
    // request type. The putn_into_*() functions convert between
    // various numeric types.
    int rcode = NC_NOERR;
    void *xp = (void *)tmpbufout;

    switch(outtype) {
      case Ttext:
        rcode = NC_ECHAR;
        break;
      case Tvoid:
        memcpy((void *)values, (void *)tmpbufout, (size_t)nels*szof);
        break;
      case Tuchar:
        rcode = putn_into_uchar(&xp, nels, (unsigned char *)values, typep);
        break;
      case Tschar:
        rcode = putn_into_schar(&xp, nels, (signed char *)values, typep);
        break;
      case Tshort:
        rcode = putn_into_short(&xp, nels, (short *)values, typep);
        break;
      case Tint:
        rcode = putn_into_int(&xp, nels, (int *)values, typep);
        break;
      case Tlong:
        rcode = putn_into_long(&xp, nels,(long *)values, typep);
        break;
      case Tfloat:
        rcode = putn_into_float(&xp, nels, (float *)values, typep);
        break;
      case Tdouble:
        rcode = putn_into_double(&xp, nels, (double *)values, typep);
        break;
      default:
        rcode = -1;
        break;
     }
     
     if (tmpbufin != tmpbufout)
        delete[] tmpbufout; tmpbufout = 0;

     return rcode;
}    

/** Use the complete projection CE to compute the size of the thing 
    the CL is asking for. This function assumes the CE is valid and
    that it describes one variable. No error checking is performed.
    
    @param clause The Projection CE to evaluate.
    @return The number of elements in this CE. */
int
number_of_elements(const string &clause)
{
    int noe = 1;
    
    // For each bracketed group of numbers, get information about a
    // dimension.
    string::size_type idx = 0;
    string::size_type clause_start = clause.find("[", idx);
    string::size_type clause_end = clause.find("]", idx);
    while (clause_start != string::npos || clause_end != string::npos) {
        string dim_proj = clause.substr(clause_start, clause_end - clause_start + 1);

        struct dim_proj_info dpi(dim_proj);

        noe *= ((dpi.stop - dpi.start) / dpi.stride) + 1;
        
        idx = clause_end + 1;
        clause_start = clause.find("[", idx);
        clause_end = clause.find("]", idx);
    }
      
    return noe;
}
    
/** This is the sole place in the client-library code where data are read
    from an OPeNDAP server. Read information from a DAP server for the
    specificed variable. Of course, in this Client-Library, the DAP server is
    masquerading as a netCDF file.

    @param cdfid Read data from this netCDF file.
    @param varid Read data for this variable.
    @param start An array of start indices.
    @param edges An array of ending indices.
    @param stride An array of stride values.
    @param values Return the values read in this array of bytes.
    @param outtype Type of the data to be put in \e values.
    @return The netCDF result code. NC_NOERR if all went OK. */
int
DODvario(int cdfid, int varid, const size_t *start, const size_t *edges,
	 const ptrdiff_t *stride, void *values, int outtype)
{
    NCConnect *c = (*conns)[cdfid];

    if ((varid < 0) || (varid > c->get_nvars()))
	   return NC_ENOTVAR;
    
    // Declare here so that we can delete in the catch block if needed.
    NCTypeFactory *nctf = new NCTypeFactory;

    // NCConnect::get_variable() uses the translated DDS.
    BaseType *cbt = c->get_variable(varid);
    try {
        NCAccess *nca = dynamic_cast<NCAccess*>(cbt);
        if (!nca)
            throw InternalErr(__FILE__, __LINE__, "Bad cast to NCAccess.");

        string expr;            // Holds complete CE
        
        if (nca->get_source()) {
            NCAccess *source_nca = dynamic_cast<NCAccess*>(nca->get_source());
            // Set version information; used when building constraint
            source_nca->set_implementation_version(c->get_version());
            // Move projection info to object
            source_nca->store_projection(c->get_proj_ce());
            // Build complete proj CE using version, command line info and 
            // start, edges and stride info from the API        
            expr = source_nca->build_constraint(outtype, start, edges, stride);
        }
        else {
            // Same as above but there's no source (i.e., this variable is
            // not the result of translation).
            nca->set_implementation_version(c->get_version());
            nca->store_projection(c->get_proj_ce());
            expr = nca->build_constraint(outtype, start, edges, stride);
        }

        // The netCDF CL can only ask for one variable at at time. Thus this 
        // must describe the number of elemens in that one variable.
        int elements = number_of_elements(expr);
        
        // Append any selection clauses provided to nc_open().
        expr += c->get_sel_ce();
        
        DBG(cerr << "CE: " << expr << endl);
        
        DataDDS Rdds(nctf);
    	c->request_data(Rdds, expr);

        // If the client-library is talking to a server and has translated the
        // DDS, then it may think it's getting back an Array, for example, but
        // actually be getting back a Structure that holds an array in a field.
        // If the CL grabs the first variable, it will get the Structure and not
        // the array (since it is working from a translated DDS, it knows nothing
        // about the Structure, only that the name 'has a funny dot in it').
        // So... Use DDS::var(const string &) to get the BaseType pointer to the
        // variable (passing it the fully qualified name for the variable). The
        // FQN is the same as the name generated in the translated DDS, so it's
        // easy to get using cbt (which comes from inside the translated DDS).
        // 02/24/04 jhrg

        BaseType *bt = Rdds.var(cbt->name());
        DBG(cerr << "Returned variable: " << bt->toString() << endl);

        dynamic_cast<NCAccess*>(bt)->extract_values(values, elements, outtype);
	
	delete nctf; nctf = 0;
    }
    catch (Error &e) {
	delete nctf; nctf = 0;
        int code = e.get_error_code();
        DBG(cerr << "Error: "  << e.get_error_message() << endl);
        set_opendap_error(e);
        // NCadvise(code, const_cast<char *>(e.get_error_message().c_str())) ;
        return code;
    }
            
    return NC_NOERR;
}

/** This function is used to read data when the imap value (non null)
    indicates that the values need to be 'arranged.' It uses DODvario() to
    actually read values.

    @param cdfid Read data from this netCDF file.
    @param varid Read data for this variable.
    @param start An array of start indices.
    @param edges An array of ending indices.
    @param stride An array of stride values.
    @param imap ??? 03/08/04 jhrg
    @param values Holds the return values.
    @param outtype The type of the values, a value parameter.
    @return The netCDF result code. NC_NOERR if all went OK. */
int
GenRead(int cdfid, int varid, const size_t *start, const size_t *count,
	const ptrdiff_t *stride, const ptrdiff_t *imap, void *values, 
	int outtype)
{
    if (( varid < 0) || (varid > (*conns)[cdfid]->get_nvars()))
        return NC_ENOTVAR;

    // If imap is null, just call DODvario and return its result.
    if (imap == NULL)
	return DODvario(cdfid, varid, start, count, stride, values, outtype);

    // ... otherwise the variable is an array with IMAP.

    //get the variable shape and type
    int shape[MAX_VAR_DIMS];
    int ndim;
    nc_type datatype;
    BaseType *bt = (*conns)[cdfid]->get_variable(varid);
    try {
        (*conns)[cdfid]->var_info(bt, &datatype, &ndim, shape);
    }
    catch (InternalErr &e) {
        DBG(cerr << "Internal Error: " << e.get_error_message() << endl);
        set_opendap_error(e);
        return NC_ENOTVAR;
    }
    
    int maxidim = ndim - 1;

    // scalar variable with an imap?
    if (maxidim < 0)
        return DODvario(cdfid, varid, 0, 0, 0, values, outtype);

    // Verify stride argument.
    int idim;
    for (idim = 0; idim < ndim; ++idim)
    	if (stride != NULL && stride[idim] < 1)
    	    return NC_ESTRIDE;
      
    // Initialize I/O parameters (set defaults).
    size_t mycount[MAX_NC_DIMS];
    size_t mystart[MAX_NC_DIMS];
    ptrdiff_t mystride[MAX_NC_DIMS];
    ptrdiff_t myimap[MAX_NC_DIMS];
    size_t stop[MAX_NC_DIMS];       /* stop indexes */
    size_t length[MAX_NC_DIMS];              /* edge lengths in bytes */
    long nels = 1;
    for (idim = maxidim; idim >= 0; --idim) {
    	mystart[idim]	= start != NULL
    	    ? start[idim]
    	    : 0;
    	mycount[idim]	= count != NULL
    	    ? count[idim]
    	    : shape[idim] - mystart[idim]; 
    	mystride[idim]	= stride != NULL 
    	    ? stride[idim]
    	    : 1;
    	myimap[idim]	= imap != NULL 
    	    ? imap[idim]
    	    : idim == maxidim
    	    ? 1
    	    : myimap[idim+1] * mycount[idim+1];
    	
    	nels = mycount[idim]*nels;
    	length[idim] = myimap[idim] * mycount[idim];
    	stop[idim] = mystart[idim] + mycount[idim];
    }
      
    // if nels is zero, we'll wind up allocating a zero length buffer for 
    // DODvario() below and then maybe reading out of that (down in the 
    // memcpy call in the for(;;) loop. jhrg 3/3/05
    if (!nels)
        return NC_NOERR;
        
    /*
     * As an optimization, adjust I/O parameters when the fastest 
     * dimension has unity stride both externally and internally.
     * In this case, the user could have called a simpler routine
     * (i.e. ncvarget() or ncvarput()).
     */
    int iocount = 1;
    if (mystride[maxidim] == 1 && myimap[maxidim] == 1) {
    	iocount = mycount[maxidim];
    	myimap[maxidim] = length[maxidim];
    }
      
    // Problem: szof (size of output type) is used to size the input buffer.
    // Should use datatype to size input buffer. 06/15/04 jhrg

    // sizeof each value (for output type)
    int szof = Dtypelen(outtype);  

    // read only once over the net into the temp. buffer
    char *valpBuf = new char [(nels*szof)];      
    int iostat = DODvario(cdfid, varid, mystart, mycount, mystride, 
			  (void *)valpBuf, outtype);

    // For data out of range error finish IMAP first 
    if ((iostat != 0) && (iostat != NC_ERANGE)) {
        delete [] valpBuf; valpBuf = 0;
    	return iostat;
    }

    // We need a char* for the pointer arithmetic below. jhrg 3/3/2005
    char *valp = (char*)values;
    int offset = 0;
    // Perform IMAP in the buffer.  Exit when done.
    for (;;) {
    	// valp has be set to point at the parameter values.
    	memcpy(valp, (void *)(valpBuf+offset), (size_t)szof*iocount);
    	offset += szof*iocount;	
    
    	/*
    	 * The following code permutes through the variable's external
    	 * start-index space and it's internal address space.  At the 
    	 * UPC, this algorithm is commonly called `odometer code'.
    	 */
    	idim = maxidim;
        carry:
    	valp += myimap[idim]*szof;
    	mystart[idim] += idim == maxidim
    	    ? iocount
    	    : 1;
    	if (mystart[idim] >= stop[idim]) {
    	    mystart[idim] = start[idim];
    	    valp -= length[idim]*szof;
    	    if (--idim < 0) {
    		delete [] valpBuf; // remove temp buffer for imap 
                valpBuf = 0;
    		return iostat;
    	    }
    	    goto carry;
    	}
    } // IMAP loop 
}

// $Log: nc_util.cc,v $
// Revision 1.20  2005/04/11 18:38:20  jimg
// Fixed a problem with NCSequence where nested sequences were not flagged
// but instead were translated. The extract_values software cannot process a
// nested sequence yet. Now the code inserts an attribute that notes that a
// nested sequence has been elided.
//
// Revision 1.19  2005/04/06 20:50:07  jimg
// Fixed a problem reported by Kevin O'Brien where Grids were not processed
// correctly. The fix is in NCConnect::var_info(). The case where a netCDF array
// is an Opendap Grid was not handled.
//
// Revision 1.18  2005/03/31 00:04:51  jimg
// Modified to use the factory class in libdap++ 3.5.
//
// Revision 1.17  2005/03/23 20:15:10  jimg
// Fixed a bug where attributes in tables nested more than one level deep
// were not properly renamed.
//
// Revision 1.16  2005/03/23 19:35:36  jimg
// Added functions to flatten an AttrTable (remove its hierarchy) using the
// STL list<> container class as an intermediary. Also removed the old
// implementation of var_info().
//
// Revision 1.15  2005/03/04 18:10:49  jimg
// At this point valgrind runs the Unidata tests for both local and remote access
// and shows no errors or leaks. There are 8 bytes still reachable from an
// exception, but that's it.
//
// Revision 1.14  2005/03/02 17:51:50  jimg
// Considerable reduction in memory leaks and fixed all errant memory
// accesses found with nc_test. OPeNDAP error codes and Error object
// message strings are now reported using the nc_strerrror() function!
//
// Revision 1.13  2005/02/26 00:43:20  jimg
// Check point: This version of the CL can now translate strings from the
// server into char arrays. This is controlled by two things: First a
// compile-time directive STRING_AS_ARRAY can be used to remove/include
// this feature. When included in the code, only Strings associated with
// variables created by the translation process will be turned into char
// arrays. Other String variables are assumed to be single character strings
// (although there may be a bug with the way these are handled, see
// NCAccess::extract_values()).
//
// Revision 1.12  2005/02/17 23:44:13  jimg
// Modifications for processing of command line projections combined
// with the limit stuff and projection info passed in from the API. I also
// consolodated some of the code by moving d_source from various
// classes to NCAccess. This may it so that DODvario() could be simplified
// as could build_constraint() and store_projection() in NCArray.
//
// Revision 1.11  2005/01/29 00:20:29  jimg
// Checkpoint: CEs ont he command line/ncopen() almost work.
//
// Revision 1.10  2005/01/26 23:25:51  jimg
// Implemented a fix for Sequence access by row number when talking to a
// 3.4 or earlier server (which contains a bug in is_end_of_rows()).
//
// Revision 1.9  2004/11/30 22:11:35  jimg
// I replaced the flatten_*() functions with a flatten() method in
// NCAccess. The default version of this method is in NCAccess and works
// for the atomic types; constructors must provide a specialization.
// Then I removed the code that copied the variables from vectors to
// lists. The translation code in NCConnect was modified to use the
// new method.
//
// Revision 1.8  2004/11/08 20:23:27  jimg
// Added list_to_variables().
//
// Revision 1.7  2004/11/05 17:13:57  jimg
// Added code to copy the BaseType pointers from the vector container into
// a list. This will enable more efficient translation software to be
// written.
//
// Revision 1.6  2004/10/28 16:38:19  jimg
// Added support for error handling to ClientParams. Added use of
// ClientParams to NCConnect, although that's not complete yet. NCConnect
// now has an instance of ClientParams. The instance is first built and
// then passed into NCConnect's ctor which stores a const reference to the CP
// object.
//
// Revision 1.5  2004/10/22 21:51:34  jimg
// More massive changes: Translation of Sequences now works so long as the
// Sequence contains only atomic types.
//
// Revision 1.4  2004/09/08 22:08:22  jimg
// More Massive changes: Code moved from the files that clone the netCDF
// function calls into NCConnect, NCAccess or nc_util.cc. Much of the
// translation functions are now methods. The netCDF type classes now
// inherit from NCAccess in addition to the DAP type classes.
//
