
// -*- mode: c++; c-basic-offset:4 -*-

// This file is part of libdap, A C++ implementation of the OPeNDAP Data
// Access Protocol.

// Copyright (c) 2002,2003,2005 OPeNDAP, Inc.
// Author: James Gallagher <jgallagher@opendap.org>
//         Reza Nekovei <reza@intcomm.net>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// You can contact OPeNDAP, Inc. at PO Box 112, Saunderstown, RI. 02874-0112.

// (c) COPYRIGHT URI/MIT 1994-1996
// Please read the full copyright statement in the file COPYRIGHT.
//
// Authors:
//      reza            Reza Nekovei (reza@intcomm.net)

// DODS-netCDF surrogate library for data variables. 
// 
// ReZa 10/6/94

#include "config_nc.h"

#include <cstring>

static char rcsid[] not_used ={"$Id: Dvar.cc 18782 2008-05-23 16:39:07Z jimg $"};

#include "Dnetcdf.h"
#include "nc_util.h"

#include <BaseType.h>

/** Defined in lnetcdf/lerror. This is not included in a header (like
    lnetcdf.h because doing so would require all sources that use lnetcdf.h
    to also include Error.h 03/02/05 jhrg */
extern void set_opendap_error(const Error &);

int
nc_def_var( int cdfid, const char *name, nc_type type,
         int ndims, const int *dimids, int *varidp)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
      rcode = lnc_def_var((*conns)[cdfid]->get_ncid(), name, type, ndims, dimids,
			  varidp);
      return rcode;
    }
#endif

    return NC_EPERM;
}

int
nc_inq_varid(int cdfid, const char *name, int *varid_ptr)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode=lnc_inq_varid((*conns)[cdfid]->get_ncid(), name, varid_ptr);
	return rcode;
    }
#endif

    DDS &dds = (*conns)[cdfid]->get_translated_dds();
    int i = 0;
    for (DDS::Vars_iter p = dds.var_begin(); p != dds.var_end(); ++p, ++i) {
	if ((*p)->name() == name) {
	    *varid_ptr = i;
	    return NC_NOERR;
	}
    }

    return NC_ENOTVAR;
}



int
nc_rename_var(int cdfid, int varid, const char *newname)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
      rcode=lnc_rename_var((*conns)[cdfid]->get_ncid(), varid, newname);
      return rcode;
    }
#endif

    return NC_EPERM;
}

int
nc_inq_var(int cdfid, int varid, char *name, nc_type *typep,
        int *ndimsp, int *dimids, int *nattsp)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
      rcode= lnc_inq_var((*conns)[cdfid]->get_ncid(), varid, name, typep,
			 ndimsp, dimids, nattsp);
      return rcode;
    }
#endif
    try {
	BaseType *bt = (*conns)[cdfid]->get_variable(varid);

	if (name != NULL)
	    (void) strcpy(name, bt->name().c_str());

	if ((typep != 0) || (ndimsp != 0) || (dimids != 0))
            (*conns)[cdfid]->var_info(bt, typep, ndimsp, dimids);
	    // var_info(cdfid, bt, typep, ndimsp, dimids);

	if (nattsp != 0)
	    *nattsp = (*conns)[cdfid]->get_num_attr(varid);
    
    }
    catch (Error &e) {
        set_opendap_error(e);
	return e.get_error_code();
    }

    return NC_NOERR;
}


int 
nc_inq_varname(int cdfid, int varid, char *name)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
      rcode= lnc_inq_varname((*conns)[cdfid]->get_ncid(), varid, name);
      return rcode;
    }
#endif

    try {
	// FIXME: what if nae *is* null? 03/08/04 jhrg
	if (name != NULL)
	    strcpy(name, (*conns)[cdfid]->get_variable(varid)->name().c_str());
    }
    catch (Error &e) {
        set_opendap_error(e);
	return e.get_error_code();
    }


    return NC_NOERR;
}


int 
nc_inq_vartype(int cdfid, int varid, nc_type *typep)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
      rcode= lnc_inq_vartype((*conns)[cdfid]->get_ncid(), varid, typep);
      return rcode;
    }
#endif

    try {
	BaseType *bt = (*conns)[cdfid]->get_variable(varid);

	if (typep != 0)
	    (*conns)[cdfid]->var_info(bt, typep, 0, 0);
    }
    catch (Error &e) {
        set_opendap_error(e);
	return e.get_error_code();
    }

    return NC_NOERR; 
}


int 
nc_inq_varndims(int cdfid, int varid, int *ndimsp)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

    if((*conns)[cdfid] == NULL) // does the file id exist?
      return NC_EBADID ;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
      rcode= lnc_inq_varndims((*conns)[cdfid]->get_ncid(), varid, ndimsp);
      return rcode;
    }
#endif

    try {
	BaseType *bt = (*conns)[cdfid]->get_variable(varid);

	if (ndimsp != 0)
	    (*conns)[cdfid]->var_info(bt, 0, ndimsp, 0);
    }
    catch (Error &e) {
        set_opendap_error(e);
	return e.get_error_code();
    }

    return NC_NOERR; 
}


int 
nc_inq_vardimid(int cdfid, int varid, int *dimids)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
      rcode= lnc_inq_vardimid((*conns)[cdfid]->get_ncid(), varid, dimids);
      return rcode;
    }
#endif

    try {
	BaseType *bt = (*conns)[cdfid]->get_variable(varid);

	if (dimids != 0)
	    (*conns)[cdfid]->var_info(bt, 0, 0, dimids);
    }
    catch (Error &e) {
        set_opendap_error(e);
	return e.get_error_code();
    }

    return NC_NOERR;
}


int 
nc_inq_varnatts(int cdfid, int varid, int *nattsp)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
      rcode= lnc_inq_varnatts((*conns)[cdfid]->get_ncid(), varid, nattsp);
      return rcode;
    }
#endif

    // varid = -1 is used for global atrributes and it is valid
    if(( varid+1 < 0) || (varid+1 > (*conns)[cdfid]->get_nvars()))
      return NC_ENOTVAR;

    if(nattsp != 0)         // Gets NC_GLOBAL when varid is -1
      *nattsp = (*conns)[cdfid]->get_num_attr(varid); 
    
    return NC_NOERR;
}

int
nc_copy_var(int ncid_in, int varid, int ncid_out)
{
    if (!conns || ncid_in < 0 || ncid_in > MAX_NC_OPEN 
	|| ncid_out < 0 || ncid_out > MAX_NC_OPEN 
	|| (*conns)[ncid_in] == NULL || (*conns)[ncid_out] == NULL) 
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[ncid_in]->is_local() && (*conns)[ncid_out]->is_local()) {
	rcode = lnc_copy_var((*conns)[ncid_in]->get_ncid(), varid, 
			  (*conns)[ncid_out]->get_ncid());
	return rcode;
    }
#endif

     // case 2, the output file is not local, Error writing over the net.

    if (!(*conns)[ncid_out]->is_local()) {
	NCadvise(NC_EPERM, "DODS network interface is read-only") ;
	return(-1);
    }

    // last case, input is from DODS file and output a local netcdf file

    // varid = -1 is used for global atrributes
    if (( varid < 0) || (varid > (*conns)[ncid_in]->get_nvars())) {
	NCadvise(NC_ENOTVAR, "%d is not a valid variable id", varid) ;
	return(-1) ;
    }

    // Make ncid_out be the real id by reading the index from conns and
    // assigning that to ncid_out. This saves calling the NCConnect::ncid()
    // method many times below. 03/05/04 jhrg
    ncid_out = (*conns)[ncid_out]->get_ncid();

    int status;			// return value of nc_*() calls.

    //get variable name and type, etc
    char name[MAX_NC_NAME];
    nc_type datatype;
    int ndims;
    int dims[MAX_VAR_DIMS];
    int nattrs;
    nc_inq_var(ncid_in, varid, name, &datatype, &ndims, dims, &nattrs);

    //get all the dimensions and make the missing ones in the output file
    int outdims[MAX_VAR_DIMS];
    char dname[MAX_NC_NAME];
    for (int i=0; i<ndims; i++) {
	size_t size, outsize;
	nc_inq_dim(ncid_in, dims[i], dname, &size);

	// is there such a dim. in the output file, if not make it
	int outdim_id;
	status = nc_inq_dimid(ncid_out, dname, &outdim_id);
	if (status != NC_NOERR) {
	    nc_redef(ncid_out); 
	    nc_def_dim(ncid_out, dname, size, &outdims[i]);
	    nc_enddef(ncid_out);
	}
	else {
	    //is it the correct size?
	    status = nc_inq_dim(ncid_out, outdim_id, (char *)0, &outsize);
	    if (size == outsize)
		outdims[i] = outdim_id;
	    else {
		return status;	// ??? FIXME
	    }
	}
    }

    //create the new variable
    nc_redef(ncid_out); 
    int outvar;
    status = nc_def_var(ncid_out, name, datatype, ndims, outdims, &outvar);
    if (status != NC_NOERR)
	return status;

    nc_enddef(ncid_out);

    //loop all the attributes and copy them
    for (int i=0; i<nattrs; i++) {
	nc_inq_attname(ncid_in, varid, i, name);
	status = nc_copy_att(ncid_in, varid, name, ncid_out, outvar);
	if (status != NC_NOERR)
	    return status;
    }

    //get variable shape and size
    long nels = 1;
    size_t cor[MAX_VAR_DIMS];
    size_t edge[MAX_VAR_DIMS];
    size_t size;
    for (int i=0; i<ndims; i++) {
	nc_inq_dim(ncid_in, dims[i], (char *)0, &size);
	nels *= size;
	cor[i] = 0;
	edge[i] = size;
    }

    char *tmp_values = new char [(nels*nctypelen(datatype))];

    //read the date values into memory
    nc_get_vara(ncid_in, varid, cor, edge, (void *)tmp_values);

    //Write the data to local file
    status = lnc_put_vara(ncid_out, outvar, cor, edge, (void *)tmp_values);

    delete [] tmp_values;
    return status;
}

// $Log: Dvar.cc,v $
// Revision 1.18  2005/03/02 17:51:50  jimg
// Considerable reduction in memory leaks and fixed all errant memory
// accesses found with nc_test. OPeNDAP error codes and Error object
// message strings are now reported using the nc_strerrror() function!
//
// Revision 1.17  2005/02/26 00:43:20  jimg
// Check point: This version of the CL can now translate strings from the
// server into char arrays. This is controlled by two things: First a
// compile-time directive STRING_AS_ARRAY can be used to remove/include
// this feature. When included in the code, only Strings associated with
// variables created by the translation process will be turned into char
// arrays. Other String variables are assumed to be single character strings
// (although there may be a bug with the way these are handled, see
// NCAccess::extract_values()).
//
// Revision 1.16  2005/01/26 23:25:51  jimg
// Implemented a fix for Sequence access by row number when talking to a
// 3.4 or earlier server (which contains a bug in is_end_of_rows()).
//
// Revision 1.15  2004/11/30 20:45:36  jimg
// Removed old code; indented.
//
// Revision 1.14  2004/09/08 22:08:21  jimg
// More Massive changes: Code moved from the files that clone the netCDF
// function calls into NCConnect, NCAccess or nc_util.cc. Much of the
// translation functions are now methods. The netCDF type classes now
// inherit from NCAccess in addition to the DAP type classes.
//
// Revision 1.13  2004/08/03 23:23:13  jimg
// Vast changes to get attribute support working for variables from
// data sources with Structures. The Structure variables are 'translated'
// to sets of variables with 'dots in their names.' These new changes
// correctly find the attribtues for the old variables and add them to
// the built in attribute table objects for the new variables. Global
// attributes are now handled as well. This software relies heavily on the
// heuristic code in DDS::transfer_attributes(). During the course of the
// work, I moved all of the software that interacts with the OPeNDAP servers
// to NCConnect or nc_util. The files Dvar.cc, Dattr.cc, ..., are now
// just the clones of the netCDF api functions.
//
// Revision 1.12  2004/07/26 19:10:44  jimg
// Moved netCDF CL <--> OPeNDAP server interface code to nc_util and
// NCConnect.
//
// Revision 1.11  2004/06/16 19:19:12  jimg
// Fixed up the code in var_info. Added a test for Structure, Sequence and
// unknown which throws InternalErr.
//
// Revision 1.10  2004/03/09 22:56:32  jimg
// Refactored so that Pix is no longer used. Some common code (in the
// glue routines) was also factored out to functions. The netCDF 2
// interface is now supplied by the lnetcdf/lv2i.c file (which is a mostly
// unaltered copy of the original v2i.c source file). See lnetcdf/README.
//
// Revision 1.9  2004/03/08 19:08:33  jimg
// This version of the code uses the Unidata netCDF 3.5.1 version of the
// netCDF 2 API emulation. This functions call our netCDF 3 API functions
// which may either interact with a DAP server r call the local netCDF 3
// functions.
//
// Revision 1.8  2004/03/01 22:30:58  jimg
// Update; more fixes for translation of Structure types. This code can
// be built with ncdump and that client can read from a host of HDF4
// datasets. The changes that enable this are filtering of Sequences.
//
// Revision 1.7  2004/02/25 00:47:52  jimg
// This code will translate Structures, including ones that are nested.
// Not tested much; needs work.
//
// Revision 1.6  2003/12/08 18:06:37  edavis
// Merge release-3-4 into trunk
//
// Revision 1.5  2003/01/28 07:08:24  jimg
// Merged with release-3-2-8.
//
// Revision 1.4.4.1  2002/12/05 20:32:39  pwest
// Corrected problems with IteratorAdapter code and cleaned up file descriptors
// and memory management problems.
//
// Revision 1.4  2000/10/06 01:22:02  jimg
// Moved the CVS Log entries to the ends of files.
// Modified the read() methods to match the new definition in the dap library.
// Added exception handlers in various places to catch exceptions thrown
// by the dap library.
//
// Revision 1.3  1999/11/05 05:15:04  jimg
// Result of merge woth 3-1-0
//
// Revision 1.1.2.2  1999/10/29 05:05:20  jimg
// Reza's fixes plus the configure & Makefile update
//
// Revision 1.2  1999/10/21 13:19:06  reza
// IMAP and other bug fixed for version3.
//
// Revision 1.1.2.1  1999/10/15 19:50:56  jimg
// Changed return values and conditions for NC API entry points
//
// Revision 1.1  1999/07/28 00:22:42  jimg
// Added
//
// Revision 1.13.2.1  1999/05/27 17:43:23  reza
// Fixed bugs in string changes
//
// Revision 1.13  1999/05/07 23:45:31  jimg
// String --> string fixes
//
// Revision 1.12  1999/04/08 13:45:52  reza
// Added support for more data types.
//
// Revision 1.11  1999/03/30 05:20:55  reza
// Added support for the new data types (Int16, UInt16, and Float32).
//
// Revision 1.10  1996/09/17 17:06:12  jimg
// Merge the release-2-0 tagged files (which were off on a branch) back into
// the trunk revision.
//
// Revision 1.9.2.3  1996/09/17 00:26:13  jimg
// Merged changes from a side branch which contained various changes from
// Reza and Charles.
// Removed ncdump and netexec since ncdump is now in its own directory and
// netexec is no longer used.
//
// Revision 1.9.2.2  1996/07/10 21:43:51  jimg
// Changes for version 2.06. These fixed lingering problems from the migration
// from version 1.x to version 2.x.
// Removed some (but not all) warning generated with gcc's -Wall option.
//
// Revision 1.9.2.1  1996/06/25 22:04:14  jimg
// Version 2.0 from Reza.
//
// Revision 1.9  1995/11/18  11:33:22  reza
// Updated member function names for DAP-1.1.1.
//
// Revision 1.8  1995/07/09  21:33:37  jimg
// Added copyright notice.
//
// Revision 1.7  1995/06/29  15:42:05  jimg
// Fixed instances of delete which should have included brackets
//
// Revision 1.6  1995/06/28  20:22:38  jimg
// Replaced malloc calls with calls to new (and calls to free with calls to
// delete).
//
// Revision 1.5  1995/06/23  14:00:46  reza
// Added ncvarcpy to the library.
//
// Revision 1.4  1994/12/22  04:51:40  reza
// Updated to use DODS new named dimension capability.
//
// Revision 1.3  1994/12/08  19:01:31  reza
// Added support for local netCDF file access through the original netCDF code.
//
// Revision 1.2  1994/11/23  21:06:04  reza
// First working version.
//
// Revision 1.1  1994/11/03  04:35:48  reza
// Preliminary version of netCDF -> DODS library.
