
// -*- mode: c++; c-basic-offset:4 -*-

// This file is part of libdap, A C++ implementation of the OPeNDAP Data
// Access Protocol.

// Copyright (c) 2002,2003,2005 OPeNDAP, Inc.
// Author: James Gallagher <jgallagher@opendap.org>
//         Reza Nekovei <reza@intcomm.net>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// You can contact OPeNDAP, Inc. at PO Box 112, Saunderstown, RI. 02874-0112.

// (c) COPYRIGHT URI/MIT 1994-1996
// Please read the full copyright statement in the file COPYRIGHT.
//
// Authors:
//      reza            Reza Nekovei (reza@intcomm.net)

// Implmentation for NCConnect: a subclass of Connect specialized for netcdf.
//
// jhrg 9/29/94

//#define DODS_DEBUG

#include "config_nc.h"

static char rcsid[] not_used =
    { "$Id: NCConnect.cc 16751 2007-06-26 17:54:03Z jimg $" };

#include "netcdf.h"

#include "DDS.h"
#include "AttrTable.h"
#include "InternalErr.h"
#include "debug.h"

#include "ClientParams.h"
#include "NCArray.h"
#include "NCStructure.h"
#include "NCSequence.h"
#include "NCGrid.h"
#include "NCConnect.h"
#include "nc_util.h"

using namespace std;

const string spr = ".";         // structure rename

static inline bool is_string(BaseType * btp)
{
    return (btp->type() == dods_str_c || btp->type() == dods_url_c);
}

static inline bool is_array(BaseType * btp)
{
    return (btp->type() == dods_array_c);
}

static inline bool is_grid(BaseType * btp)
{
    return (btp->type() == dods_grid_c);
}

static nc_type get_attr_nc_type(AttrType attrT) throw(Error)
{
    switch (attrT) {
    case Attr_byte:
        return NC_BYTE;

    case Attr_int32:
    case Attr_uint32:
        return NC_LONG;

    case Attr_int16:
    case Attr_uint16:
        return NC_SHORT;

    case Attr_float32:
        return NC_FLOAT;

    case Attr_float64:
        return NC_DOUBLE;

    case Attr_string:
    case Attr_url:
        return NC_CHAR;

    default:
        throw Error(NC_ENOTATT, "No such attribute");
    }
}

/** Initialize an instance. If \e name is a URL, the method 
    init_remote_source() must also be called.
    
    @param name A filename or URL. */
NCConnect::NCConnect(const string & name, ClientParams * cp)
    :AISConnect(name), d_client_params(cp), d_global_attributes(0),
    d_constrained_dds(0), d_translated_dds(0), d_factory(0),
    d_ncid(-1), d_nvars(-1), d_recdim(-1), d_ndims(0)
      // Note the trick above where d_factory is initialized to null and null
      // is passed to the two DDS objects. This prevents this ctor from trying
      // to call the default ctor for DDS (which doesn't exist). In the body 
      // of the ctor I make the real Factory and pass it to the two DDS 
      // objects. Using the new operator in these initialization code 
      // does not work. 1/27/06
{
    d_factory = new NCTypeFactory;
    d_constrained_dds.set_factory(d_factory);
    d_translated_dds.set_factory(d_factory);
}

NCConnect::~NCConnect()
{
    delete d_client_params;
    d_client_params = 0;
    delete d_global_attributes;
    d_global_attributes = 0;
    delete d_factory;
    d_factory = 0;
}

/** Split a constraint expression into the projection and selection
    parts and store them in this object.
    @param ce The constraint expression. */
void
 NCConnect::store_ce(const string & ce)
{
    if (ce.empty()) {
        d_proj_ce = d_sel_ce = "";
        return;
    }

    string::size_type amp = ce.find_first_of('&');
    if (amp == string::npos) {
        d_proj_ce = ce;
        d_sel_ce = "";
        return;
    }

    d_proj_ce = ce.substr(0, amp);
    d_sel_ce = ce.substr(amp);
}

/** Initialize a NCConnect instance when talking to a remote data source.

    @param ce Limit the DDS using this constraint expression.
    @exception Error Thrown by some of the NCConnect methods called here. 
 */
void NCConnect::init_remote_source(const string & ce) throw(Error)
{
    // Get the DDS. If a CE was appended to the URL, use it.
    request_dds(d_constrained_dds, ce);

    // Record the ce for later use.
    store_ce(ce);

    // Get the DAS.
    DAS das;
    request_das(das);

    // Combine the Attributes with the variables in the constrained DDS.
    d_constrained_dds.transfer_attributes(&das);

    // Search for global attributes. For netCDF, the global attributes must
    // all be atomic; no containers within the global attr container. These
    // lines of code transfer the global attributes from the constrained
    // DDS to the AttrTable held by this object and then flatten those
    // attributes. 
    set_global_attributes();
    AttrTable *at = flatten_attributes(d_global_attributes);
    delete d_global_attributes;
    d_global_attributes = at;

    // Mutate data types; from here on use the translated DDS! Note that
    // the variables' attributes are stored with the variables, so as they 
    // are translated the variable attributes will move to the 'new'
    // variables. Note that the global attributes have already been copied
    // from the constrained DDS to this (NCConnect) object.
    translate_dds();
    d_nvars = d_translated_dds.var_end() - d_translated_dds.var_begin();

    for (DDS::Vars_iter s = d_translated_dds.var_begin();
         s != d_translated_dds.var_end(); s++) {
        AttrTable & at = (*s)->get_attr_table();
        flatten_attributes(at);
    }

    // Build info about dimensions.    
    d_ndims = 0;
    parse_grid_dims(d_translated_dds);
    parse_array_dims(d_translated_dds);

    if (d_client_params->get_string_as_char_array())
        parse_string_dims(d_translated_dds);

    set_recdim(das);
}


int NCConnect::get_ncid()
{
    DBG(cerr << "Returning ncid: " << d_ncid << endl);
    return d_ncid;
}

void NCConnect::set_ncid(int id)
{
    DBG(cerr << "Setting ncid: " << id << endl);
    d_ncid = id;
}

int NCConnect::get_ndims()
{
    return d_ndims;
}

int NCConnect::get_nvars()
{
    return d_nvars;
}

void NCConnect::set_nvars(int n)
{
    d_nvars = n;
}

// Returns the dimension ID for the unlimited dimension
int NCConnect::recdim()
{
    return d_recdim;
}


// Returns the dimension size for the given dimension ID
int NCConnect::dim_size(int dimid)
{
    return d_dim_size[dimid];
}


// Returns the dimension name for the given dimension ID
const string & NCConnect::dim_name(int dimid)
{
    return d_dim_name[dimid];
}

DDS & NCConnect::get_constrained_dds()
{
    return d_constrained_dds;
}

DDS & NCConnect::get_translated_dds()
{
    return d_translated_dds;
}

/** Scan a DAS object looking for information about a NetCDF unlimited
    dimension. The OPeNDAP netCDF server adds this information by creating a
    new global attribute container at the top level named "DODS_EXTRA." The
    unlimited dimension is the value of the String attribute
    "Unlimited_Dimension." 

    @param das Search this DAS object. */
void NCConnect::set_recdim(DAS & das)
{
    // d_ndims and d_dim_name must be defined
    AttrTable *attr = das.find_container("DODS_EXTRA");
    if (attr) {
        string dim = attr->get_attr("Unlimited_Dimension");
        DBG2(cerr << "Found an ulimitied dimension attribute: " << dim <<
             endl);
        for (int i = 0; i < d_ndims; i++) {
            // A String attribute may be quoted
            DBG2(cerr << "Examining: " << d_dim_name[i] << endl);
            if (d_dim_name[i] == dim
                || dim.substr(1, d_dim_name[i].size()) == d_dim_name[i]) {
                DBG2(cerr << "Set record dim to: " << i << endl);
                d_recdim = i;
                return;
            }
        }
    }
}

/** Get the variable from this instance's DDS which corresponds to the
    given variable id.

    @note This method and others should check for overflow and throw Error.

    @param varid The netCDF variable id.
    @return The BaseType pointer to the variable */
BaseType *NCConnect::get_variable(int varid) throw(Error)
{
    if ((varid < 0) || (varid > get_nvars()))
        throw Error(NC_ENOTVAR, "Invalid variable id number.");

    return *(d_translated_dds.var_begin() + varid);
}

AttrTable & NCConnect::get_attribute_table(int varid) throw(Error)
{
    // varid = -1 is used for global atrributes
    if ((varid + 1 < 0) || (varid + 1 > get_nvars()))
        throw Error(NC_ENOTVAR, "No such variable.");;

    return (varid == -1) ? get_global_attributes()
        : get_variable(varid)->get_attr_table();
}

/** return the number of attributes for variable \e varid. if \e varid is 
    the special value \c NC_GLOBAL, then return the number of attributes in
    the global attribute container.
    
    @param varid The netCDF file id of the variable
    @return The number of attribtues */
int NCConnect::get_num_attr(int varid)
{
    return get_attribute_table(varid).get_size();
}

/** Read attribute values and load them in a char *. The caller is 
    responsible for freeing values unless an exception is thrown. Free the 
    values using delete[]. Be sure to read the note!
    
    @note Even though the attributes are held in the DAS object as character
    data, this function converts the values to the attribute's declared
    datatype. Don't be fooled by the char * return type; if the attributes
    are Int32s, that's the type of the return data.

    @param cdfid NetCDF file id
    @param varid NetCDF variable id
    @param name Name of the attribute.
    @param count Value-result parameter that holds the number of values. 
    @param datatype The datatype of the attribute; a value-result param.
    @return A pointer to the raw values. Delete using delete[].
    @exception Error Thrown if the values cannot be read or don't exist. 
*/
char *NCConnect::get_raw_values(int varid, const char *name,
                                size_t * count,
                                nc_type * datatype) throw(Error)
{
    char *values = 0;
    try {
        AttrTable & attr = get_attribute_table(varid);
        AttrTable::Attr_iter p;
        AttrTable *dummy;
        attr.find(name, &dummy, &p);

        if (p == attr.attr_end())
            throw Error(NC_ENOTATT, "No such attribute");
        //return NC_ENOTATT;

        *datatype = get_attr_nc_type(attr.get_attr_type(p));

        if (*datatype == NC_CHAR)
            *count = compute_string_attr_length(attr, p);
        else
            *count = static_cast < size_t > (attr.get_attr_num(p));

        DBG2(cerr << "name: " << name << ", count: " << *count << ", dt: "
             << *datatype << endl);

        if (*count == 0)
            throw Error(NC_NOERR, "Zero count.");

        values = new char[(*count + 1) * nctypelen(*datatype)];
	// Huh? *count is assigned above, then again here. In the case where
	// datatype is NC_CHAR, the first length may be longer than the real
	// length once escape characters are removed. The second assignment
	// corrects for that. In the cases of other types, it makes no
	// difference. jhrg 11/10/06
        *count = copy_attributes(attr, p, values);
	DBG(cerr << "count: " << *count << endl);
    }
    catch(Error & e) {
        delete[]values;
        values = 0;
        throw e;
    }
    catch (exception &e) {
	DBG(cerr << "NCConnect::get_raw_values: Exception: " << e.what()
                 << endl);
        delete[]values;
        values = 0;
	throw Error(e.what());
    }

    return values;
}

/** For the BaseType \e bt in \e cdfid, determine its type and dimension
    info.

    @note Currently only handles BaseTypes, Grid and Array. 
    @note If any of the value-result parameters are zero, no value will be
    returned. Thus the function will return only the type if \e ndimsp and \e
    dims are both zero. Also, in this latter case, cdfid will be ignored.
    
    @param cdfid netCDF library id.
    @param bt Pointer to the BaseType within a data source referenced by \e
    cdfid. 
    @param typep Value-result paramter; return the type of \e bt.
    @param ndimsp Value-result paramter; return the number of dimensions. For
    a scalar variable, *ndimsp is zero.
    @param dims Value-result paramter; return the size of each dimension. For
    a scalar variable, dims[0] is one. 
    @exception InternalErr Thrown if \e bt is a Structure, Sequence, or
    unknown type. */
void NCConnect::var_info(BaseType * bt, nc_type * typep, int *ndimsp,
                         int dims[]) throw(InternalErr)
{
    if (typep)
        *typep = dynamic_cast < NCAccess & >(*bt).get_nc_type();

    // Is this an Array? If not, is it a Grid? If so, grab the array
    // part of the grid. If not set ar to null and let the rest of the method
    // handle things. This could be turned into an NCAccess method that's
    // specialized by NCArray and NCGrid. 4/6/05 jhrg
    Array *ar = dynamic_cast < Array * >(bt);
    if (!ar) {
        Grid *g = dynamic_cast < Grid * >(bt);
        ar = (g) ? dynamic_cast < Array * >(g->array_var()) : 0;
    }
    // Special case arrays; note that there are special tests for arrays 
    // of strings.
    if (ar) {
        // set the number of dimentions and the shape for this array
        if (dims) {
            int ii = 0;
            for (Array::Dim_iter d = ar->dim_begin(); d != ar->dim_end();
                 ++d) {
                // get dim. IDs for shape construction
                // match using dimension's name and size, 
                // if not available use dim. size only
                DBG2(cerr << "var_info: Looking at dimension: " << ii <<
                     endl);
                for (int jj = 0; jj < d_ndims; jj++) {
                    if (ar->dimension_name(d).empty()) {
                        if (d_dim_size[jj] == ar->dimension_size(d))
                            dims[ii++] = jj;
                        break;
                    } else if ((d_dim_size[jj] == ar->dimension_size(d))
                               && (d_dim_name[jj] ==
                                   ar->dimension_name(d))) {
                        dims[ii++] = jj;
                        break;
                    }
                }
            }

            if (d_client_params->get_string_as_char_array()
                && dynamic_cast < NCAccess & >(*ar).get_translated()
                && is_string(ar->var())) {
                string dimname;
                int dimlen;
                get_dods_str_dim(ar, dimname, dimlen);
                for (int jj = 0; jj < d_ndims; jj++) {
                    if (d_dim_name[jj] == dimname) {
                        dims[ii++] = jj;
                        DBG2(cerr << "Scalar Dims: " << ar->
                             name() << " " << jj << endl);
                        break;
                    }
                }
            }

        }

        if (ndimsp) {
            // Why isn't ndimsp == ii??? jhrg 2/24/05
            if (d_client_params->get_string_as_char_array()
                && dynamic_cast < NCAccess & >(*ar).get_translated()
                && is_string(ar->var())) {
                *ndimsp = ar->dimensions() + 1;
            } else {
                *ndimsp = ar->dimensions();
            }
        }
    }                           // closes 'if (ar)'

    else if (d_client_params->get_string_as_char_array()
             && dynamic_cast < NCAccess & >(*bt).get_translated()
             && is_string(bt)) {
        if (dims) {
            // It's a string, but not an array. Find the dims index, et c.
            for (int jj = 0; jj < d_ndims; jj++) {
                if (d_dim_name[jj] == bt->name() + "-chars") {
                    dims[0] = jj;
                    DBG2(cerr << "Scalar Dims: " << bt->
                         name() << " " << jj << endl);
                    break;
                }
            }
        }
        if (ndimsp)
            *ndimsp = 1;
    }

    else {
        //set default values for single value base types (non-array)
        if (dims)
            dims[0] = 1;
        if (ndimsp)
            *ndimsp = 0;
    }
}

const char *likely_global_attrs[] =
    { "NC_GLOBAL", "HDF_GLOBAL", "FF_GLOBAL",
    "DSP_GLOBAL", "MAT_GLOBAL", "CoreMetadata", NULL
};

/** Set the global attributes. This method assumes that 
    DDS::transfer_attributes() has already been called and that 
    DDS::get_attr_table() when envoked on d_constrained_dds will return
    a valid AttrTable. It scans that AttrTable for likely netCDF global
    attributes, looking for attribute containers with certain special names.
    If no 'special' containers are found, it punts and sets the NCConnect
    global attribute object to an empty table.
    
    @note This method could do more, such as scan for more than one likely
    container and merge them. It could also read the names of likely 
    containers from the .dodsrc file.

    @note This method uses the 'constrained DDS' not the 'translated DDS'
    because it assumes that a previous call has transferred the attributes
    from a DAS into the constrained DDS (which is the DDS straight from the 
    server, with no translation applied). If we tried to merge the DAS info
    from the server with the translated DDS there's little chance it would
    work since the variable names are likely to be different in a translated
    DDS.
    
    @see likely_global_attrs */
void NCConnect::set_global_attributes()
{
    int i = 0;
    while (likely_global_attrs[i] && !d_global_attributes) {
        AttrTable *a =
            d_constrained_dds.get_attr_table().
            find_container(likely_global_attrs[i++]);
        // Choose this container only if it has elements. For some data sources
        // there is a container like HDF_GLOBAL, but it's empty and global
        // attributes are tucked away somewhere else (like CoreMetadata...).
        // Force a copy of the table's values into a newly allocated table.
        if (a && a->get_size() != 0)
            d_global_attributes = new AttrTable(*a);
    }

    // If there's nothing that matches, create an empty table.
    if (!d_global_attributes)
        d_global_attributes = new AttrTable;
}

/** Return the global attribute table for this data source.
    @return A reference to a container that holds the global attributes.
    @see set_global_attributes() */
AttrTable & NCConnect::get_global_attributes()
{
    if (!d_global_attributes)
        set_global_attributes();

    return *d_global_attributes;
}

/** Remove attribute containers from the attribute table by 'flattening' 
    the table. An attribute inside a container is renamed 
    <container name>:<attribtue name>.
        
    @param src The attribute table to flatten. 
    @return A pointer to a new AttrTable. It is the caller's responsibility
    to delete this object when done using it. */
AttrTable *NCConnect::flatten_attributes(AttrTable * src)
{
    EntryList *el = transfer_attributes_to_list(src);

    AttrTable *at = attributes_list_to_table(el);

    delete el;
    el = 0;

    return at;
}

void NCConnect::flatten_attributes(AttrTable & at)
{
    EntryList *el = transfer_attributes_to_list(&at);

    attributes_list_to_table(at, el);

    delete el;
    el = 0;
}


static inline void delete_basetype(BaseType * btp)
{
    delete btp;
    btp = 0;
}

/** This method translates the DDS held in the field d_constrained_dds and
    stores the result in d_translated_dds. It performs a deep copy of all
    elements in the process, even if no variables need to be translated.
    
    I think this could be improved upon, maybe by passing the d_translated_dds
    object to NCAccess::flatten() as a parameter. */
void NCConnect::translate_dds()
{

    DDS::Vars_iter field = d_constrained_dds.var_begin();
    DDS::Vars_iter field_end = d_constrained_dds.var_end();
    VarList new_vars;           // Store new vars here

    while (field != field_end) {
        DBG2((*field)->print_decl(stderr, ""));

        VarList embedded_vars =
            dynamic_cast < NCAccess * >(*field)->flatten(*d_client_params,
                                                         string(""));
        new_vars.splice(new_vars.end(), embedded_vars);

        // Some of the flatten() methods bind 'translation' attributes to
        // 'field' BaseTypes as a way of sending information about the
        // translation process 'up the chain.'
        //
        // If one of the 'field' variables has a translation attribute, 
        // move that attribute to the global table. The code does this because
        // all of the field BaseTypes will be lost when the d_translated_dds
        // is populated using variables stored on the new_vars list.
        string trans = (*field)->get_attr_table().get_attr("translation");
        if (!trans.empty())
            d_global_attributes->append_attr("translation", "String",
                                             trans);

        ++field;
    }

    // Load new_vars into d_translated_dds
    VarListIter i = new_vars.begin();
    VarListIter end = new_vars.end();
    while (i != end) {
        d_translated_dds.add_var(*i);
        delete *i;
        *i++ = 0;
    }
    DBG(cerr << "d_constrained_dds.get_dataset_name(): "
             << d_constrained_dds.get_dataset_name() << endl); 
    d_translated_dds.set_dataset_name(d_constrained_dds.
                                      get_dataset_name());
    DBG(cerr << "d_translated_dds.get_dataset_name(): "
             << d_translated_dds.get_dataset_name() << endl); 
}

// Adds dimensions to the look up table for Grid class 
void NCConnect::parse_grid_dims(DDS & dds)
{
    DDS::Vars_iter s = find_if(dds.var_begin(), dds.var_end(), is_grid);
    while (s != dds.var_end()) {
        Grid *gr = dynamic_cast < Grid * >(*s);

        for (Grid::Map_iter q = gr->map_begin(); q != gr->map_end(); ++q) {
            Array *ar = dynamic_cast < Array * >(*q);

            // See Trac tickets #50-53
            string mapdimname;
            int mapdimsize;
            if (d_client_params->get_scoped_grid_dims()) {
                // Get dimension name; all maps are vectors in DAP 2
                mapdimname =
                    gr->name() + "." + ar->dimension_name(ar->dim_begin());
                mapdimsize = ar->dimension_size(ar->dim_begin());

                if (mapdimname == "")   // if dim. name is missing use var. name 
                    mapdimname = gr->name() + "." + ar->name();
            } else {
                // Get dimension name; all maps are vectors in DAP 2
                mapdimname = ar->dimension_name(ar->dim_begin());
                mapdimsize = ar->dimension_size(ar->dim_begin());

                if (mapdimname == "")   // if dim. name is missing use var. name 
                    mapdimname = ar->name();
            }

            // replace code here with call to add_dim_if_new() and in other places: test. jhrg 3/29/06
            // Find a match to the dimension name and size ?
            bool match_found = false;
            for (int j = 0; j < d_ndims; j++) {
                if (d_dim_name[j] == mapdimname
                    && d_dim_size[j] == mapdimsize) {
                    match_found = true;
                    break;
                }
            }

            // No match found, add it to the dimension tables
            if (!match_found) {
                d_dim_name[d_ndims] = mapdimname;
                d_dim_size[d_ndims++] = mapdimsize;
            }
        }

        // Find the next grid
        s = find_if(++s, dds.var_end(), is_grid);
    }
}

// Note: the members _ndims, d_dim_name and d_dim_size are first modified by
// NCConnect::parse_grid_dims(). The initial value of _ndims is zero when
// parse_grid_dims() Is called (see parse_dims() below).

// Adds dimensions to the look up table for Array class 
void NCConnect::parse_array_dims(DDS & dds)
{
    DDS::Vars_iter s = find_if(dds.var_begin(), dds.var_end(), is_array);
    while (s != dds.var_end()) {
        Array *ar = dynamic_cast < Array * >(*s);
        int dim_cnt = 0;

        // search all dimensions of this array
        for (Array::Dim_iter d = ar->dim_begin(); d != ar->dim_end(); ++d) {
            // get dimension name and size
            string dimname = ar->dimension_name(d);
            int dim_size = ar->dimension_size(d);

            if (dimname == "")
                dimname = ar->name() + string("_") + long_to_string(dim_cnt);

            add_dim_if_new(dimname, dim_size);
            dim_cnt++;
        }
        
        // If this is an array of strings, add an extra dimension because
        // netCDF will store the string as an array of NC_CHARs. But only
        // do this if the URL parameter 'string_as_char_array is set.
        if (d_client_params->get_string_as_char_array()
            && is_string(ar->var())) {
            // determine the string dim if possible
            string dimname;
            int dimsize;
            get_dods_str_dim(*s, dimname, dimsize);

            // add new dimension if not already there
            add_dim_if_new(dimname, dimsize, dimname + "-chars");
        } 

        // Find the next array
        s = find_if(++s, dds.var_end(), is_array);
    }
}

// Since string dimensions are name <string-name>-chars, each is unique. 
// So there's no search for an existing dimension; we just append a new
// dimension for each string variable.
void NCConnect::parse_string_dims(DDS & dds)
{
    DDS::Vars_iter s = find_if(dds.var_begin(), dds.var_end(), is_string);
    while (s != dds.var_end()) {
        if (dynamic_cast < NCAccess & >(**s).get_translated()) {

            // determine the string dim if possible
            string dimname;
            int dimsize;
            get_dods_str_dim(*s, dimname, dimsize);

            // add new dimension if not already there
            add_dim_if_new(dimname, dimsize, dimname + "-chars");
        }

        s = find_if(++s, dds.var_end(), is_string);
    }
}

void NCConnect::get_dods_str_dim(BaseType * s, string & dimname, int &dimsize)
{
    AttrTable & attr = s->get_attr_table();
    AttrTable::Attr_iter pl;
    AttrTable::Attr_iter pd;
    AttrTable *dummy;

    // look for the special attributes
    attr.find("DODS:strlen", &dummy, &pl);
    attr.find("DODS:dimName", &dummy, &pd);

    // if not found, try longer version that results from flattening containers
    if ((pl == attr.attr_end()) && (pd == attr.attr_end())) {
        string deepname = s->name();
        size_t l = deepname.find_last_of('.');
        if (l) {
            deepname.replace(0, l + 1, "");
            attr.find(deepname + ":" + "DODS.strlen", &dummy, &pl);
            attr.find(deepname + ":" + "DODS.dimName", &dummy, &pd);
        }
    }
    
    // were they found?
    if ((pl != attr.attr_end()) && (pd != attr.attr_end())) {
        // extract the dim name and size from them
        dimsize = atoi(attr.get_attr(pl, 0).c_str());
        dimname = attr.get_attr(pd, 0);

    }
    else { // else, use a fixed size
        dimname = s->name() + "-chars";
        dimsize = STRING_ARRAY_SIZE;
    }

    dynamic_cast < NCAccess & >(*s).set_strdim(dimsize);

    return;
}

void NCConnect::add_dim_if_new(string dimname, int dimsize, string altname)
{

    if (altname == "")
        altname = dimname;

    int j;
    // loop thru existing dims
    for (j = 0; j < d_ndims; j++) {
        DBG(cerr << "Looking at dim name: " << d_dim_name[j] << ", index: "
                << j << endl);
        // find a matching name
        if (d_dim_name[j] == dimname) {
            // it must match in size too
            if (d_dim_size[j] != dimsize) {
                // already tried the altname?
                if (dimname == altname) {
                    // construct a unique name
                    dimname += long_to_string(d_ndims);
                }
                // otherwise try the altname
                else {
                    dimname = altname;
                }
                // restart search
                j = -1;
                continue;
            }
            // it does match, break out of loop
            else {
                break;
            }
        }
    }

    // j == d_ndims if match wasn't found -- add new dim
    if (j == d_ndims) {
        d_dim_name[d_ndims] = dimname;
        d_dim_size[d_ndims] = dimsize;
        d_ndims++;
    }
}

