
// -*- mode: c++; c-basic-offset:4 -*-

// This file is part of libnc-dods, A C++ implementation of netCDF 3 which
// supports reading from local files and OPeNDAP servers.

// Copyright (c) 2004 OPeNDAP, Inc.
// Author: James Gallagher <jgallagher@opendap.org>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// You can contact OPeNDAP, Inc. at PO Box 112, Saunderstown, RI. 02874-0112.

#include "ClientParams.h"

#include <sstream>

// #define DODS_DEBUG 1
#include "debug.h"

using namespace std;

static int
parse_int(stringstream &iparams, const string &param)
{
    string i_string;
    iparams >> ws >> i_string;
    if (i_string.empty())
        throw Error(string("Could not read a value for the '") + param
                    + string("' parameter."));

    int i;
    char *end = NULL;
    char **endptr = &end;
    i = strtol(i_string.c_str(), endptr, 0);
    if (**endptr != '\0')
        throw Error(string("Could not read a positive integer value for '")
                    + param + string("'. Instead I read: '")
                    + i_string + string("'."));
    return i;
}


/** Initialize an instance using the information in a URL.
    @param url The DAP URL with client-side parameters. */
ClientParams::ClientParams(const string &url) throw(Error)
    : d_url(url), d_preload(false), d_string_as_char_array(false),
    d_scoped_grid_dims(false), d_default_limit(0)
{
    // Get the params off the front of the URL and replace []= with spaces.
    string params(d_url.substr(0, d_url.find("http://")));

    string::size_type pos;
    while ((pos = params.find_first_of("[]=")) != string::npos)
        params.replace(pos, 1, " ");
        
    stringstream iparams(params);
    string param;

    while ((iparams >> ws >> param) && !iparams.eof()) {
        DBG(cerr << "param: '" << param << "'" << endl);
        if (param == "limit") {
            d_default_limit = parse_int(iparams, "limit");
        }
        else if (param == "preload") {
            string bool_val;
            iparams >> ws >> bool_val;
            DBG(cerr << "bool_val: '" << bool_val << "'" << endl);
            if (bool_val == "true")
                d_preload = true;
            else if (bool_val == "false")
                d_preload = false;
            else
                throw Error("The value: '" + bool_val 
                            + "' is not a valid boolean value. "
                            + "Please use 'true' or 'false'");
        }
        else if (param == "string_as_char_array") {
            string bool_val;
            iparams >> ws >> bool_val;
            DBG(cerr << "string_as_char_array, bool_val: '" << bool_val << "'" << endl);
            if (bool_val == "true")
                d_string_as_char_array = true;
            else if (bool_val == "false")
                d_string_as_char_array = false;
            else
                throw Error("The value: '" + bool_val 
                            + "' is not a valid boolean value. "
                            + "Please use 'true' or 'false'");
        }
        else if (param == "scoped_grid_dims") {
            string bool_val;
            iparams >> ws >> bool_val;
            DBG(cerr << "scoped_grid_dims, bool_val: '" << bool_val << "'" << endl);
            if (bool_val == "true")
                d_scoped_grid_dims = true;
            else if (bool_val == "false")
                d_scoped_grid_dims = false;
            else
                throw Error("The value: '" + bool_val 
                            + "' is not a valid boolean value. "
                            + "Please use 'true' or 'false'");
        }
        else if (param.find("limit-") != string::npos) {
            int val = parse_int(iparams, param);
            string var_name = param.substr(param.find("-") + 1);
            d_limits[var_name] = val;
        }
        else {
            throw Error("The client-side parameter: '" + param
                        + "' is not valid.\n"
                        + "Supported parameters are: limit, preload, "
                        + "limit-<varname>, scoped_grid_dims and string_as_char_array");
        }
    }
}

void
ClientParams::clone(const ClientParams &rhs) 
{
    DBG(cerr << "ClientParams::clone" << endl);
    d_url = rhs.d_url;
    d_preload = rhs.d_preload;
    d_string_as_char_array = rhs.d_string_as_char_array;
    d_scoped_grid_dims = rhs.d_scoped_grid_dims;
    d_default_limit = rhs.d_default_limit;
    d_limits = rhs.d_limits;
}

/** Get the value of the \c limit parameter for the variable \e var_name. 
    Return the default value if no specific value of \c limit was set for 
    \e var_name.
    @param var_name The name of the variable.
    @return The value of \c limit. */
int
ClientParams::get_limit(const string &var_name) const
{
    map<string,int>::const_iterator pos = d_limits.find(var_name);
    if (pos == d_limits.end())
        return d_default_limit;
    else
        return pos->second;
}

/** Given a url which may be prefixed with client-side params, return the 
    part of the URL without the parameters. This depends on a DAP URL
    beginning with "http://".
    @param url The URL
    @return The URL minus any client-side parameters. */
string
ClientParams::url_without_params() const
{
    string::size_type p = d_url.find("http://");
    if (p == string::npos)
        return d_url;
    else
        return d_url.substr(p);
}
