#include "GMatrix.h"
#include "../io_src/GError_Output.h"
#include "../io_src/GStd_Output.h"

#include <iostream>
#include <iomanip>
#include <string>
#include <vector>
#include <omp.h>

using namespace GMathLib;
using namespace GMathLib::IO;

GMatrix::GMatrix()
: GObject(), p_data(0), row(0), column(0), internal_new_flag(false)
{
    Class_Name("GMatrix");
    // std::cout << Class_Name() << ": default constructor" << std::endl;
}

GMatrix::~GMatrix()
{
	if(internal_new_flag){
		delete[] p_data;
		// std::cout << "GMatrix destructor : free memory :" << row << " * " << column << std::endl; 
	}
}

GMatrix::GMatrix(const GMatrix& obj)
{
    // 初期化時の行列オブジェクトのコピー
    row = obj.row;
    column = obj.column;
    
    if(obj.p_data != 0){
        double* original_d = obj.p_data;
        int data_size = row * column;
        this->p_data = new double[data_size];
        internal_new_flag = true;		

        for(int i=0; i < data_size; i++){
            this->p_data[i] = original_d[i];
        }
        // std::cout << "copy constructor" << std::endl;
    }
}

void GMatrix::allocate_initialize(int mrow, int mcolumn, double* mdata)
{
        Class_Name("GMatrix");
        row = mrow; column = mcolumn;
	
	// double 型配列のポインタである mdata が 0 でないならば,行列データを表す
	// 配列として, 引数で渡された mdata を使う. 0 ならば, 指定された行列サイズ
	// に合わせて, 内部的にメモリを確保する.
	if(mdata != 0){
		p_data = mdata;
		internal_new_flag = false;
                
		// std::cout << "GMatrix :" << row << " * " << column << " data set" << std::endl;
		// Print();
	}else{
		p_data = new double[row * column];
		internal_new_flag = true;
                
		// std::cout << "GMatrix :" << row << " * " << column << " allocate" << std::endl;
	}
}

int GMatrix::Copy(const GMatrix& obj)
{

   //コピー元と自身で行列サイズが一緒の場合は，単純に行列要素の値をコピーする
   //コピー元と自身で行列サイズが異なる場合は，新たにメモリを確保し行列サイズを変更する
   if(row != obj.row || column != obj.column){

       //既に行列オブジェクトが存在していて，
       //かつメモリが自動管理対象の場合は現在確保しているメモリを解放する
       if(obj.p_data != 0 && internal_new_flag){
           delete[] p_data;
           //std::cout << "free memory for copying matrix - " << row << "*" << column << std::endl;
       }

       row = obj.row;
       column = obj.column;
       
       p_data = new double[row * column];
       internal_new_flag = true;
       //std::cout << "copy() -allocate for copying matrix"<< row << "*" << column << std::endl;
           
   }
   
   double *original_d = obj.p_data;
   int d_size = row * column;
           
   //コピー元の行列要素をコピー先へ書き写す
   int i;
   #pragma omp parallel for private(i) if(d_size > 200)
   for(i=0; i < d_size; i++){
       this->p_data[i] = original_d[i];
   }
       
   return 0;
}

int GMatrix::Reshape(int new_row, int new_column)
{
    //指定された引数では行列サイズが変わってしまう場合は,エラーを返す
    if(row * column != new_row * new_column){
        
        const std::string errinfo = "The size of matrix must not be changed.";
        GError_Output::Puts(Class_Name() +"::Reshape",  errinfo);
        return 1;
    }

    //行数と列数を,引数で与えられた値に変更する
    row = new_row;
    column = new_column;

    return 0;
}

int GMatrix::Add(const GMatrix &a, const GMatrix  &b)
{
	int b_col = b.Column(); int b_row = b.Row();
	
	if(a.Row() != b_row || b_row != row ||
	a.Column() != b_col || b_col != column){
		
		const std::string errinfo = "The sizes of two matrixes are not consistent.";
		GError_Output::Puts(Class_Name() + "::Add",  errinfo);
		return 1;
		
	}else{
	    double tmp;

	    for(int i= 0; i < b_row; i++){
	        for(int j=0; j < b_col; j++){
		    tmp = a.Get(i, j) + b.Get(i, j);
		    Set(i, j, tmp);
		}
	     }
	     
	     return 0;
	}
}



int GMatrix::Sub(const GMatrix &a, const GMatrix  &b)
{
	int b_col = b.Column(); int b_row = b.Row();
	
	if(a.Row() != b_row || b_row != row ||
	a.Column() != b_col || b_col != column){
		
		const std::string errinfo = "The sizes of two matrixes are not consistent.";
		GError_Output::Puts(Class_Name() + "::Sub", errinfo);
		return 1;
		
	}else{
	   double tmp;

           for(int i= 0; i < b_row; i++){
	       for(int j=0; j < b_col; j++){
		    tmp = a.Get(i, j) - b.Get(i, j);
		    Set(i, j, tmp);
	        }
	    }
		
            return 0;
	}
}

int GMatrix::Multi(const GMatrix& a, const GMatrix& b)
{
	int this_row = Row();
	int this_col = Column();
	int a_col = a.Column();
	int b_col = b.Column();
	
	if(this_row != a.Row() || this_col != b_col || a_col != b.Row()){
		const std::string errinfo = "Error of matrix size in calculating Multi!";
                GError_Output::Puts(Class_Name() + "::Multi", errinfo);
		return 1;
		
	}else{
             double tmp_d = 0.0;
		
	     for(int i= 0; i < this_row; i++){
	         for(int j=0; j < this_col; j++){
		    for(int k = 0; k < a_col; k++){
		         tmp_d += a.Get(i, k) * b.Get(k, j);
		     }
				
		     Set(i, j, tmp_d);
		     tmp_d = 0.0;
                }
	    }

            return 0;
	}
}

/*
 * 行列の代入演算子の定義
 */
GMatrix GMatrix::operator=(const GMatrix &a)
{
    int a_row = a.Row();
    int a_col = a.Column();

    if( Row() != a_row || Column() != a_col ){
        const std::string errinfo = "The sizes of two matrixes are not consistent.";
	GError_Output::Puts(Class_Name() + "::operator=",  errinfo);
		
    }else{
        this->Copy(a);
    }

    return (*this);
}

/**
 * 行列の和の演算子の定義
 */
GMatrix GMatrix::operator+(const GMatrix &a)
{
    int a_row = a.Row();
    int a_col = a.Column();

    GMatrix r(a_row, a_col);
    
    if( Row() != a_row || Column() != a_col ){
        const std::string errinfo = "The sizes of two matrixes are not consistent.";
	GError_Output::Puts(Class_Name() + "::operator+",  errinfo);
	return r;
		
    }else{
	r.Add(*this, a);

	return (r);
    }
}

/**
 * 行列の差の演算子の定義
 */
GMatrix GMatrix::operator-(const GMatrix &a)
{
    int a_row = a.Row();
    int a_col = a.Column();

    GMatrix r(a_row, a_col);
    
    if( Row() != a_row || Column() != a_col ){
        const std::string errinfo = "The sizes of two matrixes are not consistent.";
	GError_Output::Puts(Class_Name() + "::operator-",  errinfo);
	return r;
		
    }else{
	r.Sub(*this, a);

	return r;
    }

}

/**
 * 行列同士の積の演算子の定義
 */
GMatrix GMatrix::operator*(const GMatrix &a)
{
    int a_row = a.Row();
    int a_col = a.Column();

    GMatrix r(Row(), a_col);
    
    if( Column() != a_row ){
        const std::string errinfo = "The sizes of two matrixes are not consistent.";
	GError_Output::Puts(Class_Name() + "::operator-",  errinfo);
	return r;
		
    }else{
	r.Multi(*this, a);

	return r;
    }
}

/**
 * 行列データをコンソールに出力する関数
 */
void GMatrix::Print(int turn_down, bool scientific_format) 
{
   
    GStd_Output::Write(Class_Name() + "::Print()", ToString());  


    //科学表記法のフラグが真ならばその設定をする
    if(scientific_format)
        std::cout.setf(std::ios::scientific);
    

    //行列要素を成形して標準出力する
    for(int i = 0; i < row; i++){
        std::cout << "| ";

	for(int j = 0; j < column-1; j++){
	    std::cout  << Get(i, j) << " ";

            // 指定された折り返し列数まで来ていれば,折り返す
	    if( turn_down != -1 && !( (j+1) % turn_down ))
	        std::cout << std::endl;
	}
	
	//行列の最終列に到達したので,折り返す
	std::cout << Get(i, column - 1) << " |\n";
	
    }
	
    std::cout << std::endl;
}

