#ifndef GLINEAR_HOMO_EQ_H
#define GLINEAR_HOMO_EQ_H

#include "GObject.h"
#include "GMatrix.h"
#include "GLU_Composition.h"

namespace GMathLib{


/** 
 * 多元連立一次方程式のソルバクラス 
 */
class GLinear_Homo_Eq : public GObject
{
public:
        /** 
	 * コンストラクタ
	 */
	GLinear_Homo_Eq();
	
	/**
	 * デストラクタ
	 */
        ~GLinear_Homo_Eq();

	/** 
	 * 部分ピボット選択を行うガウス消去法を用いた連立一次方程式ソルバ
	 * A を N*N の係数行列，B をN行の列ベクトルとしたときに，方程式 AX = B において未知の（ N 行の）列ベクトル X
	 * をもとめる．
	 * 第 1 引数に係数行列 A，第 2 引数に列ベクトル B を与える．方程式の解である列ベクトル X は，第3引数で
	 * 与えられた行列オブジェクトの参照に格納される．
	 * 引数の matrix, vec の中身はこの関数内において変更されるので注意が必要である．
	 * 引数で与える行列オブジェクトの中身を変更したくない場合には，代わりに定数オブジェクトを渡す
	 * S_Gauss_Partial_Pivot(const GMatrix&, const GMatrix&, GMatrix&) を呼び出せばよい．
	 * @see S_Gauss_Partial_Pivot(const GMatrix&, const GMatrix&, GMatrix&)
	 * @param matrix 係数行列 A を表す行列オブジェクトの参照
	 * @param vec 列ベクトル B を表す行列オブジェクトの参照
	 * @param answer_vec 求めた解を格納するための列ベクトルを表す行列オブジェクトの参照
	 */
	void S_Gauss_Partial_Pivot(GMatrix& matrix, GVector& vec, GVector& answer_vec);
	
	/** 
	 * 部分ピボット選択を行うガウスの消去法を用いた連立一次方程式ソルバ
	 * A を N*N の係数行列，B をN行の列ベクトルとしたときに，方程式 AX = B において未知の（ N 行の）列ベクトル X
	 * をもとめる．
	 * 第 1 引数に係数行列 A，第 2 引数に列ベクトル B を与える．方程式の解である列ベクトル X は，第3引数で
	 * 与えられた行列オブジェクトの参照に格納される．
	 * 引数で与えた行列オブジェクト matrix, vec の中身はこの関数内において変更されない．
	 * ただし，内部的に引数で与えた行列を複製するために，オブジェクトの生成, コピーのオーバヘッドがある．
	 * 引数で与える行列オブジェクトを変更してもよい場合は，
	 * S_Gauss_Partial_Pivot(const GMatrix&, const GMatrix&, GMatrix&)を呼び出せばよい．
	 * @see S_Gauss_Partial_Pivot(GMatrix&, GMatrix&, GMatrix&)
	 * @param matrix 係数行列Aを表す定数行列オブジェクトの参照
	 * @param vec 列ベクトルBを表す定数行列オブジェクトの参照
	 * @param answer_vec 求めた解を格納するための列ベクトルを表す行列オブジェクトの参照
	 */
	void S_Gauss_Partial_Pivot(const GMatrix& matrix, const GVector& vec, GVector& answer_vec);
        
        
        /**
	 * 部分ピボット選択および行列のスケーリングを行う LU 分解による連立一次方程式ソルバ
	 * A を N*N の係数行列，B をN行の列ベクトルとしたときに，方程式 AX = B において未知の（ N 行の）列ベクトル X
	 * をもとめる．
	 * 第 1 引数に係数行列 A，第 2 引数に列ベクトル B を与える．方程式の解である列ベクトル X は，第3引数で
	 * 与えられた行列オブジェクトの参照に格納される．
	 * 内部的に GLU_Composition クラスを使って係数行列 A を LU 分解する. 
	 * 係数行列 A が変化しなければ, 以前の LU 分解の結果を用いて解を求めた方が計算時間が短い. 
	 * そうしたい場合は, 第 3 引数のフラグを false にしてこの関数を呼び出せばよい. 
	 * @param matrix 係数行列Aを表す定数行列オブジェクトの参照
	 * @param vec 列ベクトルBを表す定数行列オブジェクトの参照
	 * @param answer_vec 求めた解を格納するための列ベクトルを表す行列オブジェクトの参照
	 * @param do_lu_cmp_flag 解を求める際に LU 分解を行うかのフラグ. デフォルトは真である. 
	 */
	void S_LU_Partial_Pivot(GMatrix& matrix, GVector& vec, GVector& answer_vec, bool do_lu_cmp_flag=true);


private:
    /**
     * 連立方程式を解く前に与えられた各行列のサイズが正しいかを調べる関数
     * @param matrix 係数行列Aを表す定数行列オブジェクトの参照
     * @param vec 列ベクトルBを表す定数行列オブジェクトの参照
     * @param answer_vec 求めた解を格納するための列ベクトルを表す行列オブジェクトの参照
     * @return 各行列サイズが正しければ, 真を返す
     */
    bool check_matrix_size(GMatrix& matrix, GVector& vec, GVector& answer_vec);
    

    GMatrix_Util::GLU_Composition* p_lu; /*<< LU 分解法用の GLU_Composition オブジェクトのポインタ */
};

}
#endif

