#ifndef GLU_COMPOSITION_H
#define GLU_COMPOSITION_H


#include "GObject.h"
#include "GMatrix.h"
#include "GVector.h"
#include "io/GError_Output.h"
#include <string>

namespace GMathLib{

namespace GMatrix_Util{

/**
 * LU 分解を提供するクラス
 */
class GLU_Composition : public GObject
{
public:
    /**
     * コンストラクタ
     */
    GLU_Composition();
    
    /**
     * デストラクタ
     */
    ~GLU_Composition();
    
    /**
     * 与えた行列オブジェクト A の LU 分解(Crout's algorithm, 部分ピボット選択あり)を行う関数. 
     * L (下三角行列)と U (上三角行列)を結合して記憶領域を節約した場合は, 
     * 第二引数で GLU_Composition::LU_COMBINED を指定すればよい. 
     * デフォルトは GLU_Composition::LU_NOT_COMBINED であり, L と U は結合せず別々の行列オブジェクトとして取り扱う.
     * LU 分解の結果の行列オブジェクト L や U は, Get_L() や Get_U() で取得できる. 
     * GLU_Composition::LU_COMBINED を指定した場合は, 結合した L と U の行列オブジェクトを  Get_LU() にて取得する. 
     * LU 分解の際に行う部分ピボット選択による行交換の情報は, 交換行列 P として保持される. これは Get_P() にて取得する. 
     * @see Get_L()
     * @see Get_U()
     * @see Get_LU()
     * @see Get_P()
     * @param A LU 分解を行いたい行列オブジェクト
     * @param combined_flag L と U を結合して取り扱うかの指定(デフォルトは結合しない)
     * @return LU 分解に成功したか否かをの真偽を返す
     */
    bool LU_Composition(GMatrix& A, int combined_flag=GLU_Composition::LU_NOT_COMBINED);
    
    /**
     * L (下三角行列) を取得する関数. 
     * @see Get_LU()
     * @return p_L 行列オブジェクト L のポインタ
     */
    inline GMatrix* Get_L(){ 
       if(combined_flag == GLU_Composition::LU_NOT_COMBINED){
           return p_L;
       }else{
           const std::string context = "GLU_Composition::LU_NOT_COMBINED is not specified in LU_Composition().";
           IO::GError_Output::Puts(Class_Name() + "Get_L()", context); 
           return 0;
       }
    }
    
    /**
     * U (上三角行列) を取得する関数. 
     * @see Get_LU()
     * @return p_L 行列オブジェクト U のポインタ
     */
    inline GMatrix* Get_U(){
       if(combined_flag == GLU_Composition::LU_NOT_COMBINED){
           return p_U;
       }else{
           const std::string context = "GLU_Composition::LU_NOT_COMBINED is not specified in LU_Composition().";
           IO::GError_Output::Puts(Class_Name() + "Get_L()", context);
	   return 0;
       }
    }
    
    /**
     * L と U を結合した行列 LU (L+U=LU) を取得する関数.
     * LU_Composition() の第二引数で GLU_Composition::LU_COMBINED を指定した場合には, 
     * LU 分解の結果の L, U をこの関数で取得する. 
     * @see Get_LU()
     * @return p_L 行列オブジェクト L, U を結合した行列オブジェクト LU のポインタ
     */
    inline GMatrix* Get_LU(){
       if(combined_flag == GLU_Composition::LU_COMBINED){
           return p_LU;
       }else{
           const std::string context = "GLU_Composition::LU_COMBINED is not specified in LU_Composition().";
           IO::GError_Output::Puts(Class_Name() + "Get_L()", context);   
           return 0;   
       }
    }
    
    /**
     * P (交換行列) を取得する関数.
     * @return p_P 行列オブジェクト P のポインタ
     */
    inline GMatrix* Get_P(){ return p_P; }

    
    const static int LU_NOT_COMBINED = 0; /*<< LU 分解の結果を L と U 別々の行列で保持する状態を示す定数 */
    const static int LU_COMBINED = 1; /*<< LU 分解の結果を 1 つの行列(L, U を結合)で保持する状態を示す定数 */

private:
    /**
     * このクラスが保持している行列オブジェクトを一括して破棄する関数. 
     */
    void delete_mx_object();

    bool lu_composition(GMatrix& target_mx, GVector& pv_mx);
    
    int combined_flag;

    GMatrix* p_L;
    GMatrix* p_U;
    GMatrix* p_LU;
    GMatrix* p_P;
};

}
}
#endif
