#ifndef GCIP_METHOD_CALC1D_h
#define GCIP_METHOD_CALC1D_h

#include "GObject.h"
#include "GMatrix.h"

namespace GMathLib{

/**
 * GCIP_Method_Calcクラスの移流相の計算で用いられる，各シミュレーションパラメータを保持する構造体の定義
 */
struct GCIP2D_Param
{
    double del_x; /**< x方向の格子点間の距離 */
    double del_y; /**< y方向の格子点間の距離 */
    double del_t; /**< 1タイムステップの増分 */ 
    int gridp_num_x; /**< x方向の格子点数 */
    int gridp_num_y; /**< y方向の格子点数 */
};


/**
 * GCIP_Method_Calc::Proceed_TimeStep() (移流相を計算する関数）が終わる時に呼び出されるイベントを受けるとる
 * リスナークラス.
 * このクラスを継承し,純粋仮想関数 Proceed_NonAdvect_Phase() を実装することで,非移流相の計算を定義する.
 * 実装した Proceed_NonAdvect_Phase() を呼び出すためには,このクラスを継承したサブクラスのオブジェクトを
 * GCIP_Method_Calc2D::Add_NonAdvect_Phase_Listener() によって,GCIP_Method_Calc2D クラスのオブジェクトに
 * 登録しなければならない.
 *
 */
class GNonAdvect_Phase2D_Listener
{
public:
    /**
     * GCIP_Method_Calc2D クラスの移流相を計算する関数の最後で呼び出されるイベント関数
     * このクラスを継承し,純粋仮想関数 Proceed_NonAdvect_Phase() を実装することで,GCIP_Method_Calc2D クラスを
     * 使った CIP 法のシミュレーションの中に非移流相の計算を定義できる.
     * 実装したイベントが呼び出されるためには,このリスナークラスのオブジェクトを
     * GCIP_Method_Clac2D::Add_NonAdvect_Phase2D_Listener() によって, GCIP_Method_Clac2D クラス のオブジェクトに
     * 登録しなければならない.
     * 移流相によって更新された新しい物理量およびその微分値のデータは，第 1,2,3 引数によって与えられる．
     * この非移流相で計算した物理量およびその微分値は，第4,5,6引数で得られる行列オブジェクトの参照に代入
     * することで更新されなければならない．
     * @see GCIP_Method_Clac1D::Add_NonAdvect_Phase_Listener()
     * @param advected_phys_mx 移流相のステップでCIP法によって時間積分した物理量をデータに持つ行列オブジェクトの参照
     * @param advected_diff_x_mx 移流相のステップでCIP法によって時間積分した物理量のx方向の空間微分値をデータに持つ行列オブジェクトの参照
     * @param advected_diff_y_mx 移流相のステップでCIP法によって時間積分した物理量のy方向の空間微分値をデータに持つ行列オブジェクトの参照
     * @param updated_phys_mx この非移流相のステップで時間積分して求めた物理量を格納しなければならない行列オブジェクトへの参照
     * @param updated_diff_x_mx この非移流相のステップで時間積分して求めた物理量のx方向の微分値を格納しなければならない行列オブジェクトへの参照
     * @param updated_diff_y_mx この非移流相のステップで時間積分して求めた物理量のy方向の微分値を格納しなければならない行列オブジェクトへの参照
     * @param vel_u_mx 移流相で用いられた速度場のx方向成分をもつ行列オブジェクトの参照
     * @param pm GCIP_Method_Clac2Dクラスが保持する構造体GCIP2D_Paramの参照
     */
    virtual void Proceed_NonAdvect2D_Phase(
        GMatrix& advected_phys_x,
	GMatrix& advected_diff_x_mx, GMatrix& advected_diff_y_mx, 
	GMatrix& updated_phys_mx, 
	GMatrix& updated_diff_x_mx, GMatrix& updated_diff_y_mx,
	GMatrix& vel_u_mx, GMatrix& vel_v_mx, GCIP2D_Param& pm
    ) = 0;
};

/**
 * 2 次元空間上のある物理量がｄρ/ｄt = -ρ∂u(∂u/∂ｘ＋∂ｖ/∂ｙ)+Q形式の支配方程式に従うときに，その
 * 物理量ρの移流相の時間発展を計算するための汎用クラス．
 * 物理量と速度の空間分布,及びシミュレーションパラメータを与えることで,移流相における物理量の中間的な時間発展を
 * 計算する関数を提供する.
 * 非移流相の計算は,移流相による中間的な時間発展がこのクラスの関数によって計算された前後に,
 * 組み込まれることが期待される.また,非移流相を組み込みたい場合は,GNonAdvect_Phase2D_Listener クラスのサブクラス
 * を作成し,その中のイベント関数を実装する.そして,そのリスナークラスのオブジェクトをGCIP_Method_Clac2D クラスに
 * 登録する. 
 */
class GCIP_Method_Calc2D : public GObject
{
public:
    /**
     * コンストラクタ
     */
    GCIP_Method_Calc2D();

    /**
     * デストラクタ
     */
    virtual ~GCIP_Method_Calc2D();

    /**
     * 非移流相の計算時に呼び出したいリスナークラスを登録するための関数
     * この関数によってリスナークラスが登録された場合,移流相の計算を Proceed_TimeStep() で行った後に, 
     * GNonAdvect_Phase_Listener::Proceed_TimeStep() を自動的に呼び出す.
     * なお,登録されたリスナークラスは,このクラスが破棄されるときに自動的には破棄されないので,
     * 登録元で破棄されなければならない.
     * @see GNonAdvect_Phase2D_Listener
     * @see GNonAdvect_Phase2D_Listener::Proceed_NonAdvect_Phase()
     * @param listener 登録するリスナークラスの参照
     */
    void Add_NonAdvect_Phase2D_Listener(GNonAdvect_Phase2D_Listener& listener_){
        listener = &listener_;
    }

    /**
     * シミュレーションパラメータ,物理量,速度場の初期状態を設定する関数
     * はじめて Proceed_TimeStep() が呼ばれる前に,必ずこの関数を通してシミュレーションに必要な
     * パラメータや初期値を設定しなければならない.
     * 引数で与えた行列オブジェクトのデータは,このクラス内部で別に生成される行列オブジェクトに完全に
     * コピーされるので,呼出し後に引数として与えた行列オブジェクトを破棄してもかまわない.
     * はじめてProceed_TimeStep()が呼ばれる前に，必ずこの関数を通してシミュレーションに必要な初期値を設定しなければならない．
     * @param param シミュレーションパラメータを保持している構造体 GCIP2D_Param の参照
     * @param init_phys_mx 物理量データを保持している行列オブジェクトの参照
     * @param init_vel_u_mx 速度場の x 成分を保持している行列オブジェクトの参照
     * @param init_vel_v_mx 速度場の y 成分を保持している行列オブジェクトの参照
     */
    virtual void Prepair(GCIP2D_Param& param, GMatrix& init_phys_mx, GMatrix& init_vel_u_mx, GMatrix& init_vel_v_mx);
    
    /**
     * 内部的に保持している速度場の行列データを更新するための関数
     * 引数で与えた行列オブジェクトは,このクラス内部で別に生成される行列オブジェクトに完全に
     * コピーされるので,呼出し後に引数として与えた行列オブジェクトを破棄してもかまわない.
     * @param updated_vel_u_mx 更新したい速度場のx成分を保持している行列オブジェクトの参照
     * @param updated_vel_v_mx 更新したい速度場のy成分を保持している行列オブジェクトの参照
     */
    void Update_Velocity_Field(GMatrix& updated_vel_u_mx, GMatrix& updated_vel_v_mx);


    /**
     * CIP法を用いて1タイムステップ分だけ x,y 方向の移流相を計算する（非移流相用のリスナーが登録されている場合は,
     * 1タイムステップの中間分に相当する）．
     * Add_NonAdvect_Phase2D_Listener() によって，リスナークラスである GNonAdvect_Phase2D_Listener がこのクラスに
     * 登録されている場合は，x,y 方向の移流相の計算が終わり次第，GNonAdvect_Phase_Listener::Proceed_NonAdvect_Phase()を呼び出す．
     * @see Add_NonAdvect_Phase2D_Listener()
     */
    virtual void Proceed_TimeStep();

    
    /**
     * 移流相の計算後（リスナークラスを登録した場合は，非移流相の計算後）に得られた現ステップにおける
     * 物理量データを取得するゲッター.
     * @ see Add_NonAdvect_Phase_Listener()
     * @ param x_no 参照したい物理量データ配列のx方向の位置（インデックス）
     * @ param y_no 参照したい物理量データ配列のy方向の位置（インデックス）
     * @return 指定された位置の物理量の値
     */
    inline double Get_PhysData(int x_no, int y_no){
        return phys_mx2->Get(x_no, y_no);
    }

    /**
     * 移流相の計算後（リスナークラスを登録した場合は，非移流相の計算後）に得られた現ステップにおける
     * 物理量データを保持する行列オブジェクトの参照を取得するゲッター.
     * @return 物理量の空間分布データを保持する行列オブジェクトの参照
     */
    inline GMatrix& Get_PhysMatrix(){
        return *phys_mx2;
    }
    

protected:
    /**
     * CIP 法を用いて 2 次元の移流相を計算する関数（直接解法）
     */
    virtual void calc_advect_phase_2D();


    GNonAdvect_Phase2D_Listener* listener;
   
    GMatrix *phys_mx1, *phys_mx2;
    GMatrix *diff_mx1_x, *diff_mx2_x;
    GMatrix *diff_mx1_y, *diff_mx2_y;
    GMatrix *vel_u_mx;
    GMatrix *vel_v_mx;
    GCIP2D_Param pm;

};

}

#endif
