#include "GCIP_Method_Calc1D.h"
#include "GDiff_Util.h"
#include <omp.h>
#include <iostream>

using namespace GMathLib;
using namespace GMathLib::GDiff_Util;

GCIP_Method_Calc1D::GCIP_Method_Calc1D()
: listener(0), phys_mx1(0), phys_mx2(0), diff_mx1(0), diff_mx2(0)
{
    Class_Name("GCIP_Method_Calc1D");
}

GCIP_Method_Calc1D::~GCIP_Method_Calc1D()
{
    if(phys_mx1 != 0){
        delete phys_mx1;
	delete phys_mx2;
	delete diff_mx1;
	delete diff_mx2;
	delete vel_mx;
    }
}

void GCIP_Method_Calc1D::Prepair(GCIP1D_Param& param, GVector& init_phys_mx, GVector& init_vel_mx)
{
    int size = init_phys_mx.Size();
    pm = param;

    if(phys_mx1 != 0){
        delete phys_mx1;
	delete phys_mx2;
	delete diff_mx1;
	delete diff_mx2;
	delete vel_mx;
    }

    //プライベート変数に必要なメモリを確保する
    phys_mx1 = new GVector(size);
    phys_mx2 = new GVector(size);
    diff_mx1 = new GVector(size);
    diff_mx2 = new GVector(size);
    vel_mx = new GVector(size);
    
    //物理量と速度場の空間データをコピーする
    phys_mx2->Copy(init_phys_mx);
    vel_mx->Copy(init_vel_mx);

    //初期の物理量の空間微分データをセットする(周期境界)
    double tmp;
    
    tmp = CentralDiff1D(phys_mx2->GetData(), 0, pm.del_x, true, pm.gridp_num);
    diff_mx2->Set(0, tmp);
    tmp = CentralDiff1D(phys_mx2->GetData(), pm.gridp_num-1, pm.del_x, true, pm.gridp_num);
    diff_mx2->Set(pm.gridp_num-1, tmp);
    
    for(int i=1; i < pm.gridp_num -1; i++){
        tmp = CentralDiff1D(phys_mx2->GetData(), i, pm.del_x);
        diff_mx2->Set(i, tmp);
    }
}

void GCIP_Method_Calc1D::Update_Velocity_Field(GVector& updated_vel_mx)
{
     vel_mx->Copy(updated_vel_mx);
}

void GCIP_Method_Calc1D::Proceed_TimeStep()
{

    //まず，移流相をCIP法で計算する
    calc_advect_phase();

    //非移流相計算用のリスナーが登録されて入れば，それを呼び出す．
    //登録されていなければ，行列オブジェクト1の値を行列オブジェクト2にコピーする
    if(listener == 0){
        phys_mx2->Copy(*phys_mx1);
	diff_mx2->Copy(*diff_mx1);
 
    }else{
        listener->Proceed_NonAdvect_Phase(*phys_mx1, *diff_mx1, *phys_mx2, *diff_mx2, *vel_mx, pm);
    }
}

void GCIP_Method_Calc1D::calc_advect_phase()
{
    double delx;
    double a, b;
    int iup;
    double X;

    //移流相で計算された値は行列オブジェクトの添字1のほうに代入する．
    //なのでここで使われるデータは行列オブジェクト2のほうである
    
    #pragma omp parallel for private (a, b, X, iup, delx)
    for(int i=0; i < pm.gridp_num; i++){
        if( (*vel_mx)(i) >= 0){
            iup = i - 1;
	    delx = - pm.del_x;

	    if(iup == -1){ 
	        iup = pm.gridp_num - 1;
	    }

	}else{
	    iup = i + 1;
	    delx = pm.del_x;
	
	    if(iup == pm.gridp_num){ 
	        iup = 0;
	    }
	}

	a = ( 
	      (*diff_mx2)(i) + (*diff_mx2)(iup)
	       + 2.0 * ( (*phys_mx2)(i) - (*phys_mx2)(iup) ) / delx 
	    ) / (delx * delx);

	b = ( 
	       3.0 * ( (*phys_mx2)(iup) - (*phys_mx2)(i) ) / delx
	       - ( 2.0 * (*diff_mx2)(i) + (*diff_mx2)(iup) )
	    ) / delx;

	X = - (*vel_mx)(i) * pm.del_t;

	phys_mx1->Set( i, 
	              ( ( a * X + b ) * X + (*diff_mx2)(i) ) * X + (*phys_mx2)(i));
        diff_mx1->Set( i, 
	              ( 3.0 * a * X + 2.0 * b) * X + (*diff_mx2)(i)); 

    }
}
