#ifndef GPROPERTIES_H
#define GPROPERTIES_H

#include <cstdlib>
#include <string>
#include <iostream>
#include <map>
#include "gmathcore/GObject.h"

namespace GMathLib{
namespace Util{

/**
 * プロパティセット(キーと値のセット)を取り扱うためのクラス
 * プロパティファイル(キーと値を1行ごとに定義したファイル.ex) Age=10 )からプロパティデータを読み込み,
 * このクラスが保持するマップオブジェクトにセットすることで,そのプロパティデータを管理する.
 */
class GProperties : public GObject
{
public:
    typedef std::map<std::string, std::string> PropertyMap;
    
    /**
     * コンストラクタ
     */
    GProperties();

    /**
     * デストラクタ
     */ 
    ~GProperties(){}

    /**
     * プロパティファイルを読み込みマップオブジェクトにプロパティデータをマッピングする
     * 指定されたファイルを読み込むために使われたファイルストリームは,この関数が終了するときに閉じられて,
     * システムリソースを開放するので利用者は注意しなくてもよい.
     * 読み込み時に例外が発生した場合は, エラー出力を行う.
     * @param filename 読み込みたいプロパティファイルの名前
     */
    void Load(const std::string filename);
    
    
    /**
     * キーに対応付けられたプロパティデータの値( std::string 型)を取得するためのゲッター. 
     * 対応する値が存在しない場合はエラーを出力し,空文字を返す.
     * @see GetProperty_d()
     * @param key 取得したいプロパティの値に対応するキーの名前
     * @return キーに対応するプロパティの値
     */
    std::string GetProperty(const std::string key);

    /**
     * キーに対応付けられたプロパティデータの値を double 型に変換して取得するためのゲッター.
     * 対応する値が存在しない場合はエラーを出力する.その時返す値は適当となる.
     * @see GetProperty()
     * @param key 取得したいプロパティの値に対応するキーの名前
     * @return キーに対応するプロパティの値を double 型に変換した値 
     */
    double GetProperty_d(const std::string key)
    {
           std::string val = GetProperty(key);
	   return std::atof(val.c_str());
	   //std::cout << "error" << std::endl;
	   //return 0;
    }
     
    /**
     * プロパティデータが格納されているマップオブジェクトを取得するためのゲッター.
     * @return プロパティデータが格納されているマップオブジェクト
     */
    PropertyMap Get_List() const{ return m_map;}

    /**
     * GProperties オブジェクトが保持している全プロパティデータを標準出力する関数. 
     */
    void Print_All_Properties();

private:
    PropertyMap m_map;
    bool load_flag;
};

}
}

#endif
