#ifndef GMATRIX_LIST_H
#define GMATRIX_LIST_H

#include <map>
#include "gmathcore/GObject.h"
#include "gmathcore/GMatrix.h"
#include "io/GError_Output.h"

namespace GMathLib{
namespace Util{

/**
 * 複数の GMatrix オブジェクトを内部的にリストとして保持し管理するクラス. 
 * このクラスに GMatrix オブジェクトを登録する際は, 内部的に GMatrix オブジェクトを新たに生成し完全にコピーする. 
 * 登録された GMatrix オブジェクトが暗黙的に破棄されるのは, このクラスのコンストラクタが呼び出された時である. 
 */
class GMatrixList_Manager : public GObject 
{
public:
    typedef std::map<int, GMatrix*> GMatrixList;
    
    /**
     * コンストラクタ
     */
    GMatrixList_Manager();
    
    /**
     * デストラクタ
     * 登録されている行列オブジェクトはデストラクタが呼び出されたときに全て破棄される. 
     */
    ~GMatrixList_Manager();

    /**
     * このクラスに GMatrix オブジェクトを登録する関数. 
     * 引数として与えられた GMatrix オブジェクトを内部的に作成した GMatrix オブジェクトに完全にコピーするため, 
     * この関数の呼び出し後に引数として与えた GMatrix オブジェクトを破棄してもかまわない. 
     * GMatrix オブジェクトを管理するために与えた引数 id の値が既に使われている場合は, エラー出力を行い登録を行わない.
     * 引数として与える行列オブジェクトの生成とこの関数内での行列オブジェクトの生成がオーバヘッドとなる場合は,
     * Create_Register_Matrix() を呼び出せばよい. 
     * @see Create_Register_Matrix()
     * @param id 登録した GMatrix オブジェクトを管理するための ID
     * @param mx 登録したい GMatrix オブジェクトの参照
     */
    void Register_Matrix(int id, GMatrix& mx);

    /**
     * 指定されたサイズの GMatrix オブジェクトを生成し, このクラスに登録する関数. 
     * GMatrix オブジェクトを管理するために与えた引数 id の値が既に使われている場合は, エラー出力を行い登録を行わない.
     * @see Register_Matrix()
     * @param id 登録した GMatrix オブジェクトを管理するための ID
     * @param row 作成したい行列オブジェクトの行数
     * @param column 作成したい行列オブジェクトの列数
     * @return 作成し登録した GMatrix オブジェクトのポインタ
     */
    GMatrix* Create_Register_Matrix(int id, int row, int column);

    /**
     * 登録している GMatrix オブジェクトを明示的に破棄するための関数. 
     * id に対応する GMatrix オブジェクトが登録されていない場合には, エラー出力を行う. 
     * @param id 削除したい行列オブジェクトの登録時の id
     */
    void Delete_Matrix(int id);

    /**
     * 登録している GMatrix オブジェクトのポインタを取得するゲッター.
     * id に対応する GMatrix オブジェクトが見つからない場合にはエラー出力を行う. 
     * その場合返り値は 0 となる. 
     * @see id 取得したい行列オブジェクトの登録時の id
     */
    inline GMatrix* Get_Matrix(int id){
        GMatrixList::iterator itr = matrix_list.find(id);

        if(itr == matrix_list.end()){
	    std::ostringstream oss;
	    oss << id;
            std::string errorinfo = "Specified id '" + oss.str() +"' is not found in registered IDs.";         
            IO::GError_Output::Puts(Class_Name() + "::Get_Matrix", errorinfo);

        }
        
	return itr->second;
    }

    /**
     * このクラスが保持する行列オブジェクトへのポインタを管理するリストを取得するための関数. 
     * @return ID と行列オブジェクトへのポインタの組を管理するマップオブジェクト. 
     */
    GMatrixList Get_List() const{return matrix_list;}
private:
    /** 
     * 実際に 登録したい GMatrix オブジェクトのポインタを map に登録する関数.
     * @param id 登録した GMatrix オブジェクトを管理するための ID
     * @param p_matrix 登録したい GMatrix オブジェクトのポインタ
     * @return 登録が成功した場合には真を返す. id が競合して登録されなかった場合には負を返す. 
     */
    bool register_matrix(int id, GMatrix* p_matrix);

    GMatrixList matrix_list;

};

}
}
#endif
