#ifndef GVECTOR_H
#define GVECTOR_H

#include "GObject.h"
#include "GMatrix.h"
#include "io/GError_Output.h"

#include <string>

namespace GMathLib{

/**
 * ベクトルを表現するための GMatrix クラスの派生クラス.
 * ベクトル特有の演算 (内積, 外積 ) などを提供する.
 */
class GVector : public GMatrix
{
public: 
    /**
     * コンストラクタ
     * 派生元の GMatrix クラスのコンストラクタを呼び出す.
     * @see GMatrix()
     */
    GVector();

    /**
     * デストラクタ
     * 派生元の GMatrix クラスのデストラクタを呼び出す.
     *
     * @see ~GMatrix()
     */
    virtual ~GVector();

    /**
     * コンストラクタ
     * ベクトルのサイズを指定し、ベクトルデータをセットする. または, そのベクトルサイズ分のメモリを確保する. 
     * 第 3 引数に double 型のベクトル配列のポインタを与えることで, ベクトルデータをセットする.
     * その際, ベクトルの形は第 2 引数で与えたフラグによって決まる.
     * フラグとして, 定数 GVector::COLUMN_VECTOR を指定した場合は列ベクトル, 定数 GVector::ROW_VECTOR を指定
     * した場合は行ベクトルとなる.(省略した場合は列ベクトルになる)
     * もし第 3 引数の double 型ポインタが 0 ならば, 指定されたベクトルサイズに合わせてメモリを自動確保する.
     * この自動確保されたメモリは, オブジェクトが破棄されるときに自動解放される.
     * 前者の方法でセットされたベクトルデータのメモリは,オブジェクトが破棄されるときに自動的には解放されない
     * ので注意が必要である.
     *
     * @see GVector()
     * @see ~GVector()
     * @param size ベクトルのサイズ
     * @param vec_type　列ベクトルか行ベクトルかの指定
     * @param p_data　ベクトル要素を表す 1 次元配列データ
     */
    GVector(int size, int vec_type=GVector::COLUMN_VECTOR, double *p_data=0);
    
    /**
     * GMatrix オブジェクトが保持している行列データの一部を, 他の行列オブジェクトの行列データの任意の位置へとコピーする関数. 
     * コピー元の行列オブジェクト obj がもつ行列データのインデックス begin_id から end_id までの範囲を, 
     * コピー先の行列オブジェクトの行列データ中の特定の位置 pos が先頭になるようにコピーする. 
     * コピー元の行列データにおいて, 指定されたデータ範囲が不適切な場合, 
     * およびコピー時に確保された配列サイズを超えてコピーされる場合はエラーを出力し, コピーは行われない. 
     * @see Copy(const GMatrix&)
     * @param obj コピー元の行列オブジェクトの参照
     * @param begin_id コピーされる行列データ範囲の先頭となるインデックス
     * @param end_id コピーされる行列データ範囲の末端となるインデックス
     * @param pos コピーしてきた行列データの先頭を埋め込む位置を示すコピー先の行列データ内のインデックス
     * @return 成功すれば0,失敗すれば1を返す
     */
    virtual int Copy(const GVector& obj, int begin_id, int end_id, int pos);

    /**
     * GVector オブジェクトを引数として, GMatrix::Copy(const GMatrix&) と同じ動作を提供する関数. 
     * @see GMatrix::Copy(const GMatrix&)
     * @param obj ベクトル要素をコピーしたいもとの GVector オブジェクト
     * @return 成功すれば0,失敗すれば1を返す
     */
    virtual int Copy(const GVector& obj){
        return GMatrix::Copy(obj);
    }

    /**
     * ベクトルの要素を設定する関数. 実際, SetElement と同じ動作をする. 
     * @see SetElement
     * @param no 設定したいベクトル要素の要素番号
     * @param value 指定したい値
     */
    inline void Set(int no, double value){
        if( Row() == 1){
            SetElement(0, no, value);
        }else if( Column() == 1){
            SetElement(no, 0, value);
        }
    }

    /**
      * ベクトル要素を取得する関数. 実際, GetElement と同じ動作をする. 
      * @see GetElement
      * @param 取得したいベクトル要素の要素番号
      * @return no 指定したベクトル要素の値
      */
    inline double Get(int no) const{
       if( Row() == 1){
             return GetElement(0, no);
        }else{
            return GetElement(no, 0);
        }

	return -1;
    }


    /**
     * ベクトルの内積を計算する関数.
     * 引数として与えられた 2 つのベクトルオブジェクトに対して, 内積演算を行う.
     * その計算結果のスカラー値は, 返り値として与えられる.
     * @param vec1 内積演算の対象となる片方のベクトルオブジェクトの参照
     * @param vec2 内積演算の対象となるもう一方のベクトルオブジェクトの参照
     * @return 内積の計算結果 
     */
    static double Dot_Product(GVector& vec1, GVector& vec2);

    /** 
     * 3次元ベクトル同士の外積を計算する関数.
     * 引数として与えられた 2  つの 3 次元ベクトルオブジェクトの, 外積を計算する.
     * その計算結果のベクトルデータは, この関数を呼び出したベクトルオブジェクトに格納される.
     * @param vec1 外積演算の対象となる片方のベクトルオブジェクトの参照
     * @param vec2 外積演算の対象となるもう一方のベクトルオブジェクトの参照
     */
    void Cross_Product(GVector& vec1, GVector& vec2);


    /**
     * ベクトルのノルムを計算する関数
     * @return 計算したノルムの値
     */
    double Norm();

    /**
     * ベクトルのサイズを返す関数
     * @return ベクトルのサイズ
     */
    inline int Size() const{
        if( Row() == 1){
	   return Column();
	}else{
	   return Row();
	}
    }
    
    /**
     * 列ベクトルか行ベクトルかの情報を取得するための関数
     * @return 列ベクトルならば GVector::COLUMN_VECTOR, 行ベクトルならば GVector::ROW_VECTOR 
     */
    inline int Shape(){
        if( Row() == 1){
	   return GVector::ROW_VECTOR;
	}else{
	   return GVector::COLUMN_VECTOR;
	}
    }

    /**
     * 添え字演算子の定義
     * @param i ベクトル成分のインデックス
     * @return ベクトルの i 成分に対応する double 型の参照
     */
     inline double& operator()(int i){
         return p_data[i];
     }

    const static int COLUMN_VECTOR = 0; /**< 列ベクトルであることを示すフラグ */
    const static int ROW_VECTOR = 1;    /**< 行ベクトルであることを示すフラグ */

};

}

#endif
