#ifndef GMATRIX_H
#define GMATRIX_H

#include "GObject.h"

#include <iostream>
#include <sstream>
#include <string>

namespace GMathLib{

/**
 * 行列データを表現し, 管理するクラス
 * GMathLib で用いる行列データを表現するためのクラス.
 * 行列データの管理, 行列の基本的な演算などを提供する.
 *
 */
class GMatrix : public GObject
{
public:
	/**
         * デフォルトコンストラクタ
         * @see GMatrix(int, int, double*)
         */
	GMatrix();


	/**
         * コンストラクタ
         * 行列サイズを指定し、行列データをセットする. または, その行列サイズ分のメモリを確保する. 
	 * 第 3 引数に double 型のベクトル配列のポインタを与えることで, 行列データをセットする.
	 * その際, 行列の形は第 1, 2 引数で与えた行列サイズによって決まる.
	 * もし第 3 引数の double 型ポインタが 0 ならば,指定された行列サイズに合わせてメモリを自動確保する.
	 * この自動確保されたメモリは, オブジェクトが破棄されるときに自動解放される.
	 * 前者の方法でセットされた行列データのメモリは,オブジェクトが破棄されるときに自動的には解放されない
	 * ので注意が必要である.
	 * @see GMatrix()
         * @see ~GMatrix()
         * @param mrow 行列の行数
         * @param mcolumn　行列の列数
         * @param p_data　行列要素を表す 1 次元配列データ
         */
        GMatrix(int mrow, int mcolumn, double* p_data = 0)
	 : GObject()
	{
	    allocate_initialize(mrow, mcolumn, p_data);
	}

	/**
         * コンストラクタ
         * オブジェクトの名前, 行列サイズを指定し, 行列データをセットする. またはその行列サイズ分のメモリを確保する. 
	 * 第 1 引数でオブジェクトの名前を指定する他は, GMatrix(int, int, double*)と同じ動作をする. 
	 * @see GMatrix(int, int, double*)
         * @see ~GMatrix()
	 * @param obj_name オブジェクトの名前
         * @param mrow 行列の行数
         * @param mcolumn　行列の列数
         * @param p_data　行列要素を表す 1 次元配列データ
         */
        GMatrix(std::string obj_name, int mrow, int mcolumn, double* p_data = 0)
	 : GObject()
	{
	    Object_Name(obj_name);
	    allocate_initialize(mrow, mcolumn, p_data);
	}

	/**
         * デストラクタ
	 * コンストラクタで自動確保した行列データのメモリはオブジェクトが廃棄されるときに自動解放される.
	 * 一方で,外部でメモリを確保され, 行列オブジェクトにセットされた行列データについては,
	 * その行列データのメモリ解放の責任は持たない.
	 * @see GMatrix(int, int, double*)
         */
        virtual ~GMatrix();


	/**
         * コピースコントラクタ
	 */
        GMatrix(const GMatrix& obj);


        /**
         * GMatrix オブジェクトが保持している行列データを, 他の行列オブジェクトへ完全にコピーするための関数.
	 * コピーしたい行列オブジェクトの行列サイズが, コピー元の行列オブジェクトと同じ場合はメモリが,
	 * 新たに再確保されることはない. 一方, 異なる場合には必要な行列サイズに合わせてメモリを再確保する.
	 * その際に不要になったメモリは, そのメモリが GMatrix クラス内で自動確保されたものならば自動的に解放するが,
	 * そうでない場合は自動的には解放されない.
         * @see GMatrix(const GMatrix&)
         * @param obj 行列要素をコピーしたいもとの行列オブジェクト
         * @return 成功すれば0,失敗すれば1を返す
         */
        virtual int Copy(const GMatrix& obj);


	/**
         * 行列要素をすべて同じ値で埋めるための関数.
	 * @param a 埋めたい値
         * @return 成功すれば0,失敗すれば1を返す
         */
        virtual int Fill(double a){
	    if(p_data == 0){
	        return 1;
	    }

	   for(int i=0; i < row; i++){
		for(int j=0; j < column; j++){
	            Set(i, j, a);
	        }
	    }	

	    return 0;
	}	


	/**
         * ２項演算関数(加法)
	 * 演算に関係する行列オブジェクトは、行列演算において適切な行列サイズに設定されていなければならない.
         * @param a 演算したい行列オブジェクト１
         * @param b 演算したい行列オブジェクト２
         * @return 成功すれば0,失敗すれば1を返す
	 */
        int Add(const GMatrix& a, const GMatrix& b);


	/**
         * ２項演算関数(減法)
	 * 演算に関係する行列オブジェクトは、行列演算において適切な行列サイズに設定されていなければならない.
         * @param a 演算したい行列オブジェクト１
         * @param b 演算したい行列オブジェクト２
         * @return 成功すれば0,失敗すれば1を返す
	 */
        int Sub(const GMatrix& a, const GMatrix& b);


	/**
         * ２項演算関数(行列の積)
	 * 演算に関係する行列オブジェクトは、行列演算において適切な行列サイズに設定されていなければならない.
         * @param a 演算したい行列オブジェクト１
         * @param b 演算したい行列オブジェクト２
         * @return 成功すれば0,失敗すれば1を返す
	 */
        int Multi(const GMatrix& a, const GMatrix& b);


        /**
	 * 関数を呼び出した行列オブジェクトに対し,引数の値でスカラー積を取る関数
	 * @param sc スカラー積をとる時に使うスカラー値 
	 */
	inline void Scalar_Product(double sc){
	    for(int i=0; i < row; i++){
	        for(int j=0; j < column; j++){
		    Set(i, j, sc * Get(i, j) ); 
		}
	     }
	}


	/**
         * 行列の行数を返す関数
         * @return 行列の行数
         */
	inline int Row() const{return row;}


	/**
         * 行列の列数を返す関数
         * @return 行列の列数
         */
        inline int Column() const{return column;}


	/**
         * 行列オブジェクトが保持している行列データの配列ポインタを取得する関数
	 * @return 行列要素を保持している double 型の配列のポインタ
         */ 
        inline double* GetData(){return p_data;}


	/**
	 * 行列要素の値を取得するための関数
	 * @param row_no 取得したい行列要素の行数
	 * @param column_no 取得したい行列要素の列数
	 * @return 指定された行列要素の値
	 */
	inline double GetElement(int row_no, int column_no) const{ 
	     return p_data[column_no + row_no * column];
	}
        
	/**
	 * GetElement と同じ動作をする関数.
	 * @param row_no 取得したい行列要素の行数
	 * @param column_no 取得したい行列要素の列数
	 * @return 指定された行列要素の値
	 */
	inline double Get(int row_no, int column_no) const{ 
	     return GetElement(row_no, column_no);
	}


	/**
	 * 行列要素の値を設定するための関数
	 * @param row_no 設定したい行列要素の行のインデックス
	 * @param column_no 設定したい行列要素の列のインデックス
	 * @param value 設定する行列要素の値
	 */
	inline void SetElement(int row_no, int column_no, double value){ 
	     p_data[column_no + row_no * column] = value;
	}


	/**
	 * SetElement と同じ動作をする関数.
	 * @param row_no 設定したい行列要素の行のインデックス
	 * @param column_no 設定したい行列要素の列のインデックス
	 * @param value 設定する行列要素の値
	 */
	inline void Set(int row_no, int column_no, double value){ 
	     SetElement(row_no, column_no, value);
	}

        virtual std::string ToString()
	{
	    std::string text = GObject::ToString();
	    std::ostringstream oss;

	    text.append("Matrix Size: (");
	    oss << Row();
	    text.append(oss.str()); text.append(","); 
	    
	    oss.str(std::string());
	    oss << Column();
	    text.append(oss.str()); text.append(")\n");

            return text;
	}

	/**
         * 行列要素をコンソールに表示するための関数
	 * 標準出力を通して行列要素を成形して表示する.強制的な折り返しは行わない.
	 * @see Print(int, bool)
         */
	void Print() {
	     Print(-1);
	}


	/**
         * 行列要素をコンソールに表示するための関数
	 * 標準出力を通して行列要素を成形して表示する.
	 * 強制的に折り返したい列のインデックスや数値の表示法を指定できる.強制的に折り返したくな場合は -1 を指定する.
	 * @param turn_down 強制的に折り返しを行いたい列のインデックス
	 * @scientific_format 数値を科学表記法で表示するかのフラグ.デフォルトはfalse
         */
	void Print(int turn_down, bool scientific_format=false);


	/**
	 * 行列オブジェクトを(列)ベクトル化する関数
	 * 実際のところ行列データ自体に影響を与えない.単に列数が,１として取り扱われるようになるだけである.
	 */
	inline void Vectorize(){
	    row = row * column;
	    column = 1;
	}

        /**
	 * 行列の全要素数は変えずに, 行列の行数と列数を変えるための関数
	 * 行列のサイズが変わる場合にはエラーを返して,行列の形は変更されない.
         * @param new_row 設定したい行列の行数
         * @param new_column 設定したい行列の列数
         * @return 成功すれば0,失敗すれば1を返す
	 */
        virtual int Reshape(int new_row, int new_column);

        /**
	 * 添え字演算子の定義
	 * @param i 行列の行インデックス
	 * @param j 行列の列インデックス
	 * @return 行列の(i, j)成分に対応する double 型の参照
	 */
        inline double& operator()(int i, int j){
	     return p_data[j + i * column];
	}
        
	/**
	 * 代入演算子の定義
	 */
	 GMatrix operator=(const GMatrix &a);
	
	/**
	 * + 演算子(行列の和)の定義
	 */
	 GMatrix operator+(const GMatrix &a);
	
	/**
	 * - 演算子(行列の差)の定義
	 */
	 GMatrix operator-(const GMatrix &a);
	
	/**
	 * * 演算子(行列同士の積)の定義
	 */
	 GMatrix operator*(const GMatrix &a);

	/**
	 * * 演算子(行列とスカラーの積)の定義
	 */
	 inline friend GMatrix operator*(double k, const GMatrix &a){
	 
             GMatrix r(a.Row(), a.Column());
             r.Copy(a);
             r.Scalar_Product(k);

             return r;
	 }
	 
	/**
	 * * 演算子(行列とスカラーの積)の定義
	 */
	 inline friend GMatrix operator*(const GMatrix &a, double k){
	     return k * a;
	 }

protected:
       
       /**
        * コンストラクタから呼び出される行列オブジェクトの領域確保と初期化を行う関数. 
        * @param mrow 行列の行数
        * @param mcolumn　行列の列数
        * @param p_data　行列要素を表す 1 次元配列データ
	*/
       void allocate_initialize(int mrow, int mcolumn, double* p_data = 0);

       double* p_data;                   /**< 行列要素を保持するdouble配列のポインタ */


private:
	int row;                         /**< 行列の行サイズを保持する変数 */
	int column;                      /**< 行列の列サイズを保持する変数 */
	bool internal_new_flag;          /**< クラス内で行列データ用のメモリを確保したかのフラグ */
};

}
#endif

