#ifndef GDIFF_UTIL_H
#define GDIFF_UTIL_H

#include "GMatrix.h"

namespace GMathLib
{

namespace GDiff_Util{

/**
 * 引数で与えた配列データの一部分の中央差分をとるインライン関数
 * 第 1 引数で物理量が格納されている配列ポインタを与えて，第 2 引数でどの空間位置の中央差分をとるかを指定する．
 * また，第 3 引数で空間差分幅を指定する．
 * 第 4 引数のフラグをtrueにした場合にのみ，第1引数の配列の両端に対し，周期境界条件を適用して中央差分をとる．
 * その場合，第5引数で第1引数で与えた配列のサイズを必ず与えなければならない．
 * デフォルトでは，第４引数のフラグは false である．
 * @param discre_data 物理量が格納されている配列の double 型ポインタ
 * @param no 空間中央差分をとりたい空間位置に対応する，第 1 引数の配列のインデックス
 * @param del_x 空間差分幅
 * @param periodic_bound_condition 周期境界条件を適用するかのフラグ
 * @param d_size 第１引数で与える配列のサイズ
 * @return 指定されたインデックスにおける中央差分の計算結果
 */
inline double CentralDiff1D(double* discre_data, int no, double del_x, bool periodic_bound_condition=false, int d_size=0){
       
       if(periodic_bound_condition){  
           if(no == 0){
               return (discre_data[d_size-1] - discre_data[1])/(2 * del_x);
           }else if(no == d_size-1){
               return (discre_data[0] - discre_data[d_size-2])/(2 * del_x);
	   }
       }

       return (discre_data[no+1] - discre_data[no-1])/(2 * del_x);
}




/**
 * 引数で与えた配列データから 指定されたインデックスにおける 2 階微分の差分商をとるインライン関数
 * 第 1 引数で物理量が格納されている配列ポインタを与えて，第 2 引数でどの空間位置の ２ 階の空間微分の差分をとるか
 * を指定する．また，第 3 引数で空間差分幅を指定する．
 * 第 4 引数のフラグを true にした場合にのみ，第 1 引数の配列の両端に対し，周期境界条件を適用して差分をとる．
 * その場合，第 5 引数で第 1 引数で与えた配列のサイズを必ず与えなければならない．
 * デフォルトでは，このフラグは false である．
 * @param discre_data 物理量が格納されている配列の double 型ポインタ
 * @param no ２ 階空間微分の差分をとりたい空間位置に対応する，第 1 引数の配列のインデックス
 * @param del_x 空間差分幅
 * @param periodic_bound_condition 周期境界条件を適用するかのフラグ
 * @param d_size 第１引数で与える配列のサイズ
 * @return 指定されたインデックスにおける 2 階微分の差分商の計算結果
 */
inline double Diff_2Times1D(double* discre_data, int no, double del_x, bool periodic_bound_condition=false, int d_size=0){
       
       if(periodic_bound_condition){  
           if(no == 0){
               return (discre_data[1] - 2 * discre_data[no] + discre_data[d_size-1])/(del_x * del_x);
           }else if(no == d_size-1){
               return (discre_data[0] - 2 * discre_data[d_size-1] + discre_data[d_size-2])/(del_x * del_x);
           }
       }
       
       return (discre_data[no+1] - 2 * discre_data[no] + discre_data[no-1])/(del_x * del_x);
}


/**
 * ２次元配列に指定するインデックスの前後のインデックスを, 周期境界条件に合わせて調整する関数
 * 例えば,第一引数のインデックス i が右端境界のインデックスであれば, i+1 (iup)は,
 * 左端境界のインデックスとなる.
 * 調整された i+1 , i-1 の値は引数で与えた iup, idown にそれぞれ格納される.
 * @param i 対象となる x 方向のインデックス i
 * @param j 対象となる y 方向のインデックス j
 * @param iup 周期境界の調整を考慮したインデックス i+1 が格納される参照
 * @param idown 周期境界の調整を考慮したインデックス i-1 が格納される参照
 * @param jup 周期境界の調整を考慮したインデックス j+1 が格納される参照
 * @param jdown 周期境界の調整を考慮したインデックス j-1 が格納される参照
 * @param matrix_isize x 方向の配列サイズ
 * @param matrix_jsize y 方向の配列サイズ
 */
inline void BoundaryCondition2D_For_MatrixId(
         int i, int j, int &iup, int &idown, int &jup, int &jdown, int matrix_isize, int matrix_jsize
	 )
{
     //インデックス i について調整する
     if(i == 0){
        idown = matrix_isize-1;
	iup = i+1;
     }else if(i == matrix_isize-1){
        iup = 0;
	idown=i-1;
     }else{
        iup = i+1;
	idown = i-1;
     }
		   
     if(j == 0){
         jdown = matrix_jsize-1;
	 jup = j+1;
     }else if(j == matrix_jsize-1){
         jup = 0;
	 jdown = j-1;
     }else{
         jup = j+1;
	 jdown = j-1;
     }	   
}

} //end namespace GDiff_Util
} //end namespace GMathLib
#endif
