# This file is part of Rubber and thus covered by the GPL
# (c) Emmanuel Beffara, 2004--2006
# vim: noet:ts=4
"""
Common support for makeindex and xindy external tools.
"""
import os.path

import rubber.depend
from rubber.util import _, msg

# FIXME: this class may probably be simplified a lot if inheriting
# from rubber.depend.Shell instead of rubber.depend.Node.

class Index (rubber.depend.Node):
	"""
	This class represents a single index.
	"""
	def __init__ (self, doc, source, target, transcript):
		"""
		Initialize the index, by specifying the source file (generated by
		LaTeX), the target file (the output of makeindex) and the transcript
		(e.g. .ilg) file.  Transcript is used by glosstex.py.
		"""
		super (Index, self).__init__ (doc.set)
		src = doc.basename (with_suffix = "." + source)
		tgt = doc.basename (with_suffix = "." + target)
		log = doc.basename (with_suffix = "." + transcript)
		doc.add_product (src)
		self.add_product (tgt)
		self.add_product (log)
		self.add_source (src, track_contents=True)
		doc.add_source (tgt, track_contents=True)
		self.doc = doc
		self.cmd = ["makeindex", src, "-q", "-o", tgt, "-t", log]
		self.lang = None   # only for xindy
		self.modules = []  # only for xindy
		self.opts = []
		self.path = []
		self.style = None  # only for makeindex
		self.command_env = None

	def do_language (self, lang):
		self.lang = lang

	def do_modules (self, *args):
		self.modules.extend(args)

	def do_order (self, *args):
		for opt in args:
			if opt == "standard": self.opts = []
			elif opt == "german": self.opts.append("-g")
			elif opt == "letter": self.opts.append("-l")
			else: msg.warn(
				_("unknown option '%s' for 'makeidx.order'") % opt)

	def do_path (self, path):
		self.path.append(path)

	def do_style (self, style):
		self.style = style

	def do_tool (self, tool):
		if tool not in ("makeindex", "xindy"):
			msg.error(_("unknown indexing tool '%s'") % tool)
		self.cmd [0] = tool

	def run (self):
		if not os.path.exists (self.cmd [1]):
			msg.info (_ ("%s not yet generated" % self.cmd [1]))
			return True

		# No more settings are expected, we compute the
		# command once and for all.
		if self.command_env == None:
			if self.cmd [0] == "makeindex":
				self.cmd.extend (self.opts)
				if self.style:
					self.cmd.extend (["-s", self.style])
				path_var = "INDEXSTYLE"
			else:   # self.cmd [0] == "texindy"
				for opt in self.opts:
					if opt == "-g":
						if self.lang != "":
							msg.warn(_("'language' overrides 'order german'"),
								 pkg="index")
						else:
							self.lang = "german-din"
					else: # opt == "-l"
						self.modules.append("letter-ordering")
						msg.warn(_("use 'module letter-ordering' instead of 'order letter'"),
							 pkg="index")
				for mod in self.modules:
					self.cmd.extend(["--module", mod])
				if self.lang:
					self.cmd.extend(["--language", self.lang])
				path_var = "XINDY_SEARCHPATH"

			if self.path != []:
				self.command_env = { path_var: ':'.join(self.path + [os.getenv(path_var, '')]) }
			else:
				self.command_env = {}

		# The actual run.
		return self.doc.env.execute(self.cmd, self.command_env) == 0
