/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/weibull/cdf' );
import Weibull = require( './../../../../../base/dists/weibull/ctor' );
import entropy = require( './../../../../../base/dists/weibull/entropy' );
import kurtosis = require( './../../../../../base/dists/weibull/kurtosis' );
import logcdf = require( './../../../../../base/dists/weibull/logcdf' );
import logpdf = require( './../../../../../base/dists/weibull/logpdf' );
import mean = require( './../../../../../base/dists/weibull/mean' );
import median = require( './../../../../../base/dists/weibull/median' );
import mgf = require( './../../../../../base/dists/weibull/mgf' );
import mode = require( './../../../../../base/dists/weibull/mode' );
import pdf = require( './../../../../../base/dists/weibull/pdf' );
import quantile = require( './../../../../../base/dists/weibull/quantile' );
import skewness = require( './../../../../../base/dists/weibull/skewness' );
import stdev = require( './../../../../../base/dists/weibull/stdev' );
import variance = require( './../../../../../base/dists/weibull/variance' );

/**
* Interface describing the `weibull` namespace.
*/
interface Namespace {
	/**
	* Weibull distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param k - scale parameter
	* @param lambda - shape parameter
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 2.0, 1.0, 1.0 );
	* // returns ~0.865
	*
	* var myCDF = ns.cdf.factory( 2.0, 10.0 );
	* y = myCDF( 12.0 );
	* // returns ~0.763
	*/
	cdf: typeof cdf;

	/**
	* Weibull distribution.
	*/
	Weibull: typeof Weibull;

	/**
	* Returns the differential entropy of a Weibull distribution.
	*
	* ## Notes
	*
	* -   If `k <= 0` or `lambda <= 0`, the function returns `NaN`.
	*
	* @param k - shape parameter
	* @param lambda - scale parameter
	* @returns entropy
	*
	* @example
	* var v = ns.entropy( 1.0, 1.0 );
	* // returns 1.0
	*
	* @example
	* var v = ns.entropy( 4.0, 12.0 );
	* // returns ~2.532
	*
	* @example
	* var v = ns.entropy( 8.0, 2.0 );
	* // returns ~0.119
	*
	* @example
	* var v = ns.entropy( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( NaN, 2.0 );
	* // returns NaN
	*/
	entropy: typeof entropy;

	/**
	* Returns the excess kurtosis of a Weibull distribution.
	*
	* ## Notes
	*
	* -   If `k <= 0` or `lambda <= 0`, the function returns `NaN`.
	*
	* @param k - shape parameter
	* @param lambda - scale parameter
	* @returns kurtosis
	*
	* @example
	* var v = ns.kurtosis( 1.0, 1.0 );
	* // returns 6.0
	*
	* @example
	* var v = ns.kurtosis( 4.0, 12.0 );
	* // returns ~-0.252
	*
	* @example
	* var v = ns.kurtosis( 8.0, 2.0 );
	* // returns ~0.328
	*
	* @example
	* var v = ns.kurtosis( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( NaN, 2.0 );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Weibull distribution logarithm of cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param k - scale parameter
	* @param lambda - shape parameter
	* @returns evaluated logCDF
	*
	* @example
	* var y = ns.logcdf( 2.0, 1.0, 1.0 );
	* // returns ~-0.145
	*
	* var mylogcdf = ns.logcdf.factory( 2.0, 10.0 );
	* y = mylogcdf( 12.0 );
	* // returns ~-0.27
	*/
	logcdf: typeof logcdf;

	/**
	* Weibull distribution natural logarithm of the probability density function (logPDF).
	*
	* @param x - input value
	* @param k - shape parameter
	* @param lambda - scale parameter
	* @returns evaluated logPDF
	*
	* @example
	* var y = ns.logpdf( 2.0, 1.0, 0.5 );
	* // returns ~-3.297
	*
	* var mylogpdf = ns.logpdf.factory( 7.0, 6.0 );
	* y = mylogpdf( 7.0 );
	* // returns ~-1.864
	*/
	logpdf: typeof logpdf;

	/**
	* Returns the expected value of a Weibull distribution.
	*
	* ## Notes
	*
	* -   If `k <= 0` or `lambda <= 0`, the function returns `NaN`.
	*
	* @param k - shape parameter
	* @param lambda - scale parameter
	* @returns expected value
	*
	* @example
	* var v = ns.mean( 1.0, 1.0 );
	* // returns 1.0
	*
	* @example
	* var v = ns.mean( 4.0, 12.0 );
	* // returns ~10.877
	*
	* @example
	* var v = ns.mean( 8.0, 2.0 );
	* // returns ~1.883
	*
	* @example
	* var v = ns.mean( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( NaN, 2.0 );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Returns the median of a Weibull distribution.
	*
	* ## Notes
	*
	* -   If `k <= 0` or `lambda <= 0`, the function returns `NaN`.
	*
	* @param k - shape parameter
	* @param lambda - scale parameter
	* @returns median
	*
	* @example
	* var v = ns.median( 1.0, 1.0 );
	* // returns ~0.693
	*
	* @example
	* var v = ns.median( 4.0, 12.0 );
	* // returns ~10.949
	*
	* @example
	* var v = ns.median( 8.0, 2.0 );
	* // returns ~1.91
	*
	* @example
	* var v = ns.median( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.median( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.median( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.median( NaN, 2.0 );
	* // returns NaN
	*/
	median: typeof median;

	/**
	* Weibull distribution moment-generating function (MGF).
	*
	* @param t - input value
	* @param k - shape parameter
	* @param lambda - scale parameter
	* @returns evaluated MGF
	*
	* @example
	* var y = ns.mgf( 1.0, 1.0, 0.5 );
	* // returns ~2.0
	*
	* y = ns.mgf( -1.0, 4.0, 4.0 );
	* // returns ~0.019
	*
	* var myMGF = ns.mgf.factory( 8.0, 10.0 );
	*
	* y = myMGF( 0.8 );
	* // returns ~3150.149
	*
	* y = myMGF( 0.08 );
	* // returns ~2.137s
	*/
	mgf: typeof mgf;

	/**
	* Returns the mode of a Weibull distribution.
	*
	* ## Notes
	*
	* -   If `k <= 0` or `lambda <= 0`, the function returns `NaN`.
	*
	* @param k - shape parameter
	* @param lambda - scale parameter
	* @returns mode
	*
	* @example
	* var v = ns.mode( 1.0, 1.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.mode( 4.0, 12.0 );
	* // returns ~11.167
	*
	* @example
	* var v = ns.mode( 8.0, 2.0 );
	* // returns ~1.967
	*
	* @example
	* var v = ns.mode( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( NaN, 2.0 );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Weibull distribution probability density function (PDF).
	*
	* @param x - input value
	* @param k - shape parameter
	* @param lambda - scale parameter
	* @returns evaluated PDF
	*
	* @example
	* var y = ns.pdf( 2.0, 1.0, 0.5 );
	* // returns ~0.037
	*
	* var myPDF = ns.pdf.factory( 7.0, 6.0 );
	* y = myPDF( 7.0 );
	* // returns ~0.155
	*/
	pdf: typeof pdf;

	/**
	* Weibull distribution quantile function.
	*
	* @param p - input value
	* @param k - scale parameter
	* @param lambda - shape parameter
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.8, 1.0, 1.0 );
	* // returns ~1.609
	*
	* var myQuantile = ns.quantile.factory( 2.0, 10.0 );
	* y = myQuantile( 0.4 );
	* // returns ~7.147
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness of a Weibull distribution.
	*
	* ## Notes
	*
	* -   If `k <= 0` or `lambda <= 0`, the function returns `NaN`.
	*
	* @param k - shape parameter
	* @param lambda - scale parameter
	* @returns skewness
	*
	* @example
	* var v = ns.skewness( 1.0, 1.0 );
	* // returns 2.0
	*
	* @example
	* var v = ns.skewness( 4.0, 12.0 );
	* // returns ~-0.087
	*
	* @example
	* var v = ns.skewness( 8.0, 2.0 );
	* // returns ~-0.534
	*
	* @example
	* var v = ns.skewness( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( NaN, 2.0 );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation of a Weibull distribution.
	*
	* ## Notes
	*
	* -   If `k <= 0` or `lambda <= 0`, the function returns `NaN`.
	*
	* @param k - shape parameter
	* @param lambda - scale parameter
	* @returns standard deviation
	*
	* @example
	* var v = ns.stdev( 1.0, 1.0 );
	* // returns 1.0
	*
	* @example
	* var v = ns.stdev( 4.0, 12.0 );
	* // returns ~3.051
	*
	* @example
	* var v = ns.stdev( 8.0, 2.0 );
	* // returns ~0.279
	*
	* @example
	* var v = ns.stdev( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( NaN, 2.0 );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance of a Weibull distribution.
	*
	* ## Notes
	*
	* -   If `k <= 0` or `lambda <= 0`, the function returns `NaN`.
	*
	* @param k - shape parameter
	* @param lambda - scale parameter
	* @returns variance
	*
	* @example
	* var v = ns.variance( 1.0, 1.0 );
	* // returns 1.0
	*
	* @example
	* var v = ns.variance( 4.0, 12.0 );
	* // returns ~9.311
	*
	* @example
	* var v = ns.variance( 8.0, 2.0 );
	* // returns ~0.078
	*
	* @example
	* var v = ns.variance( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( NaN, 2.0 );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Weibull distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
