/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var isnan = require( '@stdlib/math/base/assert/is-nan' );
var pow = require( '@stdlib/math/base/special/pow' );
var PI = require( '@stdlib/constants/float64/pi' );


// MAIN //

/**
* Evaluates the probability density function (PDF) for a Cauchy distribution with location parameter `x0` and scale parameter `gamma` at a value `x`.
*
* @param {number} x - input value
* @param {number} x0 - location parameter
* @param {PositiveNumber} gamma - scale parameter
* @returns {number} evaluated PDF
*
* @example
* var y = pdf( 2.0, 1.0, 1.0 );
* // returns ~0.159
*
* @example
* var y = pdf( 4.0, 3.0, 0.1 );
* // returns ~0.0315
*
* @example
* var y = pdf( 4.0, 3.0, 3.0 );
* // returns ~0.095
*
* @example
* var y = pdf( NaN, 1.0, 1.0 );
* // returns NaN
*
* @example
* var y = pdf( 2.0, NaN, 1.0 );
* // returns NaN
*
* @example
* var y = pdf( 2.0, 1.0, NaN );
* // returns NaN
*
* @example
* // Negative scale parameter:
* var y = pdf( 2.0, 1.0, -2.0 );
* // returns NaN
*/
function pdf( x, x0, gamma ) {
	var denom;
	if (
		isnan( x ) ||
		isnan( gamma ) ||
		isnan( x0 ) ||
		gamma <= 0.0
	) {
		return NaN;
	}
	denom = PI * gamma * ( 1.0 + pow( (x-x0)/gamma, 2.0 ) );
	return 1.0 / denom;
}


// EXPORTS //

module.exports = pdf;
