// bibiman - a TUI for managing BibLaTeX databases
// Copyright (C) 2024  lukeflo
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.
/////

use crate::bibiman::bibisetup::BibiData;
use ratatui::widgets::{ScrollbarState, TableState};

#[derive(Debug, Clone, PartialEq, Eq)]
pub enum EntryTableColumn {
    Authors,
    Title,
    Year,
    Pubtype,
}

// Define list containing entries as table
#[derive(Debug, PartialEq, Eq)]
pub struct EntryTable {
    pub entry_table_items: Vec<BibiData>,
    pub entry_table_at_search_start: Vec<BibiData>,
    pub entry_table_selected_column: EntryTableColumn,
    pub entry_table_sorted_by_col: Option<EntryTableColumn>,
    pub entry_table_reversed_sort: bool,
    pub entry_table_state: TableState,
    pub entry_scroll_state: ScrollbarState,
    pub entry_info_scroll: u16,
    pub entry_info_scroll_state: ScrollbarState,
}

impl EntryTable {
    pub fn new(entry_list: Vec<BibiData>) -> Self {
        let mut entry_table_items = entry_list;
        entry_table_items.sort_by(|a, b| a.authors.to_lowercase().cmp(&b.authors.to_lowercase()));
        // entry_table
        let entry_table_state = TableState::default()
            .with_selected(0)
            .with_selected_column(1)
            // other two values above are ignored, if selected cell isn't fitting
            .with_selected_cell(Some((0, 1)));
        let entry_scroll_state = ScrollbarState::new(entry_table_items.len());
        let entry_info_scroll_state = ScrollbarState::default();
        Self {
            entry_table_items,
            entry_table_at_search_start: Vec::new(),
            entry_table_selected_column: EntryTableColumn::Authors,
            entry_table_sorted_by_col: Some(EntryTableColumn::Authors),
            entry_table_reversed_sort: false,
            entry_table_state,
            entry_scroll_state,
            entry_info_scroll: 0,
            entry_info_scroll_state,
        }
    }

    pub fn sort_by_id(&mut self) {
        if self.entry_table_sorted_by_col.is_some() {
            self.entry_table_reversed_sort = false
        } else {
            self.entry_table_reversed_sort = !self.entry_table_reversed_sort;
        }
        if self.entry_table_reversed_sort {
            self.entry_table_items.sort_by(|a, b| b.id.cmp(&a.id));
        } else {
            self.entry_table_items.sort_by(|a, b| a.id.cmp(&b.id));
        }
        self.entry_table_sorted_by_col = None;
    }

    // Sort entry table by specific column.
    // Toggle sorting by hitting same key again
    pub fn sort_entry_table(&mut self, toggle: bool) {
        // Check if table was sorted by visible column before
        if toggle && self.entry_table_sorted_by_col.is_some() {
            self.entry_table_reversed_sort = !self.entry_table_reversed_sort;
        } else {
            self.entry_table_reversed_sort = false;
        }
        if self.entry_table_sorted_by_col.as_ref().is_some() {
            if &self.entry_table_selected_column != self.entry_table_sorted_by_col.as_ref().unwrap()
            {
                self.entry_table_reversed_sort = false
            }
        }
        self.entry_table_sorted_by_col = Some(self.entry_table_selected_column.clone());
        if self.entry_table_reversed_sort {
            match self.entry_table_selected_column {
                EntryTableColumn::Authors => self
                    .entry_table_items
                    .sort_by(|a, b| b.authors.to_lowercase().cmp(&a.authors.to_lowercase())),
                EntryTableColumn::Title => self
                    .entry_table_items
                    .sort_by(|a, b| b.title.to_lowercase().cmp(&a.title.to_lowercase())),
                EntryTableColumn::Year => self
                    .entry_table_items
                    .sort_by(|a, b| b.year.to_lowercase().cmp(&a.year.to_lowercase())),
                EntryTableColumn::Pubtype => self.entry_table_items.sort_by(|a, b| {
                    b.custom_field_value()
                        .to_lowercase()
                        .cmp(&a.custom_field_value().to_lowercase())
                }),
            }
        } else if !self.entry_table_reversed_sort {
            match self.entry_table_selected_column {
                EntryTableColumn::Authors => self
                    .entry_table_items
                    .sort_by(|a, b| a.authors.to_lowercase().cmp(&b.authors.to_lowercase())),
                EntryTableColumn::Title => self
                    .entry_table_items
                    .sort_by(|a, b| a.title.to_lowercase().cmp(&b.title.to_lowercase())),
                EntryTableColumn::Year => self
                    .entry_table_items
                    .sort_by(|a, b| a.year.to_lowercase().cmp(&b.year.to_lowercase())),
                EntryTableColumn::Pubtype => self.entry_table_items.sort_by(|a, b| {
                    a.custom_field_value()
                        .to_lowercase()
                        .cmp(&b.custom_field_value().to_lowercase())
                }),
            }
        }
    }
}

#[cfg(test)]
mod tests {
    use crate::{
        bibiman::{
            bibisetup::{BibiRow, CustomField},
            BibiData,
        },
        cliargs::CLIArgs,
        config::BibiConfig,
    };

    #[test]
    fn check_os() {
        let os = std::env::consts::OS;
        assert_eq!(
            os,
            "linux",
            "You're not coding on linux, but on {}... Switch to linux, now!",
            std::env::consts::OS
        )
    }

    #[test]
    fn shorten_authors() {
        let cfg = BibiConfig::new(&CLIArgs::default());
        let mut entry: BibiData = BibiData {
            id: 1,
            authors: "Miller, Schmitz, Bernard".to_string(),
            short_author: String::new(),
            title: "A title".to_string(),
            year: "2000".to_string(),
            custom_field: (CustomField::Pubtype, "article".to_string()),
            keywords: "key1, key2".to_string(),
            citekey: "miller_2000".to_string(),
            abstract_text: "An abstract".to_string(),
            doi_url: None,
            filepath: None,
            file_field: false,
            subtitle: None,
            notes: None,
            symbols: [None, None, None],
            sanitized_bibi_data: None,
        }
        .gen_sanitized();

        let entry_vec = BibiData::ref_vec(&mut entry, &cfg);

        let mut entry_editors: BibiData = BibiData {
            id: 2,
            authors: "Miller, Schmitz, Bernard (ed.)".to_string(),
            short_author: String::new(),
            title: "A title".to_string(),
            year: "2000".to_string(),
            custom_field: (CustomField::Pubtype, "article".to_string()),
            keywords: "key1, key2".to_string(),
            citekey: "miller_2000".to_string(),
            abstract_text: "An abstract".to_string(),
            doi_url: None,
            filepath: None,
            file_field: false,
            subtitle: None,
            notes: None,
            symbols: [None, None, None],
            sanitized_bibi_data: None,
        }
        .gen_sanitized();

        let entry_vec_editors = BibiData::ref_vec(&mut entry_editors, &cfg);

        assert_eq!(
            entry_vec,
            BibiRow {
                authors: "Miller et al.",
                title: "A title",
                year: "2000",
                custom_field_value: "article",
                symbols: &[None, None, None]
            }
        );
        assert_eq!(
            entry_vec_editors,
            BibiRow {
                authors: "Miller et al. (ed.)",
                title: "A title",
                year: "2000",
                custom_field_value: "article",
                symbols: &[None, None, None]
            }
        )
    }
}
