/* LibraryDeleteDialog.cpp */

/* Copyright (C) 2011-2024 Michael Lugmair (Lucio Carreras)
 *
 * This file is part of sayonara player
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "GUI_DeleteDialog.h"
#include "Utils/Language/Language.h"
#include "Gui/Utils/ui_GUI_DeleteDialog.h"
#include "Gui/Utils/Icons.h"

#include <QPushButton>

struct GUI_DeleteDialog::Private
{
	int trackCount;
	Library::TrackDeletionMode answer {Library::TrackDeletionMode::None};

	explicit Private(const int trackCount) :
		trackCount(trackCount) {}
};

GUI_DeleteDialog::GUI_DeleteDialog(const int trackCount, QWidget* parent) :
	Gui::Dialog(parent),
	m {Pimpl::make<Private>(trackCount)},
	ui {std::make_shared<Ui::GUI_DeleteDialog>()}
{
	ui->setupUi(this);

	auto* btnYes = ui->buttonBox->button(QDialogButtonBox::Yes);
	auto* btnNo = ui->buttonBox->button(QDialogButtonBox::No);

	connect(btnYes, &QPushButton::clicked, this, &GUI_DeleteDialog::yesClicked);
	connect(btnNo, &QPushButton::clicked, this, &GUI_DeleteDialog::noClicked);
}

GUI_DeleteDialog::~GUI_DeleteDialog() = default;

Library::TrackDeletionMode GUI_DeleteDialog::answer() const
{
	return m->answer;
}

void GUI_DeleteDialog::yesClicked()
{
	if(ui->cbOnlyFromLibrary->isChecked())
	{
		m->answer = Library::TrackDeletionMode::OnlyLibrary;
	}

	else
	{
		m->answer = Library::TrackDeletionMode::AlsoFiles;
	}

	close();
}

void GUI_DeleteDialog::noClicked()
{
	m->answer = Library::TrackDeletionMode::None;
	close();
}

void GUI_DeleteDialog::showEvent(QShowEvent* e)
{
	Gui::Dialog::showEvent(e);

	setFocus();

	const auto height = fontMetrics().height() * 2;
	ui->labIcon->setPixmap(Gui::pixmap(Gui::Info, QSize {height, height}));
	ui->cbOnlyFromLibrary->setText(tr("Only from library"));
	ui->labWarning->setText(Lang::get(Lang::Warning) + "!");
	ui->labInfo->setText(
		tr("You are about to delete %n file(s)", "", m->trackCount) +
		"!\n" +
		Lang::get(Lang::Continue).question());
}

void GUI_DeleteDialog::setTrackCount(const int trackCount)
{
	m->trackCount = trackCount;
}

