/*
 * Copyright 2014 Canonical Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef UCMOUSE_P_H
#define UCMOUSE_P_H

#include <QtCore/QBasicTimer>
#include <QtCore/QObject>
#include <QtQml/QtQml>
#include <QtQuick/QQuickItem>
#include <QtQuick/private/qquickevents_p_p.h>

#if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
#include <LomiriToolkit-Qt6/lomiritoolkitglobal.h>
#else
#include <LomiriToolkit/lomiritoolkitglobal.h>
#endif

UT_NAMESPACE_BEGIN

class LOMIRITOOLKIT_EXPORT ForwardedEvent : public QEvent {
public:
    enum EventType {
        MousePress,
        MouseRelease,
        MouseMove,
        MouseDblClick,
        HoverEnter,
        HoverExit,
        MouseClick,
        MouseLongPress,
    };
    ForwardedEvent(EventType type, QQuickItem *sender, QEvent *originalEvent, QQuickMouseEvent *quickEvent)
        : QEvent((QEvent::Type)m_eventBase)
        , m_subType(type)
        , m_sender(sender)
        , m_originalEvent(originalEvent)
        , m_quickEvent(quickEvent)
    {
        setAccepted(false);
    }

    static void registerForwardedEvent();

    EventType subType() const { return m_subType; }
    QQuickItem *sender() const { return m_sender; }
    QQuickMouseEvent *quickEvent() const { return m_quickEvent; }
    QEvent *originalEvent() const { return m_originalEvent; }
    static QEvent::Type baseType() { return m_eventBase; }
private:
    EventType m_subType;
    QPointer<QQuickItem> m_sender;
    QEvent *m_originalEvent;
    QPointer<QQuickMouseEvent> m_quickEvent;
    static QEvent::Type m_eventBase;
};

class LOMIRITOOLKIT_EXPORT UCMouse : public QObject
{
    Q_OBJECT

    Q_PROPERTY(bool enabled READ isEnabled WRITE setEnabled NOTIFY enabledChanged)
    Q_PROPERTY(Qt::MouseButtons acceptedButtons READ acceptedButtons NOTIFY acceptedButtonsChanged)
    Q_PROPERTY(bool hoverEnabled READ hoverEnabled NOTIFY hoverEnabledChanged)
    Q_PROPERTY(int clickAndHoldThreshold READ clickAndHoldThreshold WRITE setClickAndHoldThreshold NOTIFY clickAndHoldThresholdChanged)
    Q_PROPERTY(QQmlListProperty<QQuickItem> forwardTo READ forwardTo)
    Q_PROPERTY(Priority priority READ priority WRITE setPriority NOTIFY priorityChanged)

    //Added in 1.3, but we can't use REVISION because of QTBUG-40043
    Q_PROPERTY(bool ignoreSynthesizedEvents MEMBER m_ignoreSynthesizedEvents NOTIFY ignoreSynthesizedEventsChanged)

    Q_ENUMS(Priority)
public:
    enum Priority {
        BeforeItem,
        AfterItem
    };

    // keep in sync with QQuickMouseArea PressAndHoldDelay
    static constexpr int DefaultPressAndHoldDelay{800};

    explicit UCMouse(QObject *parent = 0);

    static UCMouse *qmlAttachedProperties(QObject *owner);

    bool isEnabled() const;
    virtual void setEnabled(bool enabled);
    Qt::MouseButtons acceptedButtons() const;
    bool hoverEnabled() const;
    int clickAndHoldThreshold() const;
    void setClickAndHoldThreshold(int threshold);
    QQmlListProperty<QQuickItem> forwardTo();
    Priority priority() const;
    virtual void setPriority(Priority priority);

Q_SIGNALS:
    void enabledChanged();
    void acceptedButtonsChanged();
    void hoverEnabledChanged();
    void clickAndHoldThresholdChanged();
    void priorityChanged();
    void ignoreSynthesizedEventsChanged();

    void pressed(QQuickMouseEvent *mouse, QQuickItem *host);
    void released(QQuickMouseEvent *mouse, QQuickItem *host);
    void clicked(QQuickMouseEvent *mouse, QQuickItem *host);
    void pressAndHold(QQuickMouseEvent *mouse, QQuickItem *host);
    void doubleClicked(QQuickMouseEvent *mouse, QQuickItem *host);
    void positionChanged(QQuickMouseEvent *mouse, QQuickItem *host);
    void entered(QQuickMouseEvent *event, QQuickItem *host);
    void exited(QQuickMouseEvent *event, QQuickItem *host);

protected:
    bool eventFilter(QObject *, QEvent *) override;
    void timerEvent(QTimerEvent *event) override;
    virtual bool mouseEvents(QObject *target, QMouseEvent *event);
    virtual bool hoverEvents(QObject *target, QHoverEvent *event);
    virtual bool forwardedEvents(ForwardedEvent *event);
    virtual bool hasAttachedFilter(QQuickItem *item);

    void setHovered(bool hovered, QEvent *hoverEvent);
    bool mousePressed(QMouseEvent *event);
    bool mouseReleased(QMouseEvent *event);
    bool mouseDblClick(QMouseEvent *event);
    bool mouseMoved(QMouseEvent *event);
    bool hoverEntered(QHoverEvent *event);
    bool hoverMoved(QHoverEvent *event);
    bool hoverExited(QHoverEvent *event);

    virtual void saveEvent(QMouseEvent *event);
    bool isDoubleClickConnected();
    bool isMouseEvent(QEvent::Type type);
    bool isHoverEvent(QEvent::Type type);
    bool forwardEvent(ForwardedEvent::EventType type, QEvent *event, QQuickMouseEvent *quickEvent);

protected:
    QQuickItem *m_owner;
    QList<QQuickItem*> m_forwardList;
    QBasicTimer m_pressAndHoldTimer;
    QRectF m_toleranceArea;
    QPointF m_lastPos;
    QPointF m_lastScenePos;
    Qt::MouseButton m_lastButton;
    Qt::MouseButtons m_lastButtons;
    Qt::KeyboardModifiers m_lastModifiers;
    Qt::MouseButtons m_pressedButtons;
    Priority m_priority;
    int m_moveThreshold;

    bool m_signalWhenContains:1;
    bool m_enabled: 1;
    bool m_moved:1;
    bool m_longPress:1;
    bool m_hovered:1;
    bool m_doubleClicked:1;
    bool m_ignoreSynthesizedEvents;
};

UT_NAMESPACE_END

QML_DECLARE_TYPEINFO(UT_PREPEND_NAMESPACE(UCMouse), QML_HAS_ATTACHED_PROPERTIES)

#endif // UCMOUSE_P_H
