/*
 * Copyright 2014-2016 Canonical Ltd.
 * Copyright 2023 Robert Tari
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authors:
 *   Charles Kerr <charles.kerr@canonical.com>
 *   Robert Tari <robert@tari.in>
 */

#pragma once

#include <core/property.h>

#include <gio/gio.h> // GIcon

#include <string>
#include <vector>

struct Header
{
  bool is_visible = false;
  std::string title;
  std::string label;
  std::string a11y;
  std::string tooltip;
  std::shared_ptr<GIcon> icon;

  bool operator== (const Header& that) const {
    return (is_visible == that.is_visible) &&
           (title == that.title) &&
           (label == that.label) &&
           (a11y == that.a11y) &&
           (icon == that.icon);
  }
  bool operator!= (const Header& that) const { return !(*this == that);}
};


class Profile
{
public:
  virtual std::string name() const =0;
  virtual core::Property<Header>& header() =0;
  virtual std::shared_ptr<GMenuModel> menu_model() const =0;
  virtual ~Profile();

protected:
  Profile();
};


class SimpleProfile: public Profile
{
public:
  SimpleProfile(const char* name, const std::shared_ptr<GMenuModel>& menu): m_name(name), m_menu(menu) {}
  virtual ~SimpleProfile() override;

  std::string name() const override {return m_name;}
  core::Property<Header>& header() override {return m_header;}
  std::shared_ptr<GMenuModel> menu_model() const override {return m_menu;}

protected:
  const std::string m_name;
  core::Property<Header> m_header;
  std::shared_ptr<GMenuModel> m_menu;
};


class Indicator
{
public:
  virtual ~Indicator();

  virtual const char* name() const =0;
  virtual GSimpleActionGroup* action_group() const =0;
  virtual std::vector<std::shared_ptr<Profile>> profiles() const =0;
};
