/* api.c API unit tests
 *
 * Copyright (C) 2006-2025 wolfSSL Inc.
 *
 * This file is part of wolfSSL.
 *
 * wolfSSL is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * wolfSSL is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1335, USA
 */

/* For AES-CBC, input lengths can optionally be validated to be a
 * multiple of the block size, by defining WOLFSSL_AES_CBC_LENGTH_CHECKS,
 * also available via the configure option --enable-aescbc-length-checks.
 */


/*----------------------------------------------------------------------------*
 | Includes
 *----------------------------------------------------------------------------*/

#include <tests/unit.h>

#include <wolfssl/wolfcrypt/logging.h>
#include <wolfssl/wolfcrypt/hash.h>

#if defined(WOLFSSL_STATIC_MEMORY)
    #include <wolfssl/wolfcrypt/memory.h>
#endif
#ifdef WOLFSSL_ASNC_CRYPT
    #include <wolfssl/wolfcrypt/async.h>
#endif
#ifdef HAVE_ECC
    #include <wolfssl/wolfcrypt/ecc.h>   /* wc_ecc_fp_free */
    #ifdef WOLFSSL_SM2
        #include <wolfssl/wolfcrypt/sm2.h>
    #endif
#endif
#ifndef NO_ASN
    #include <wolfssl/wolfcrypt/asn_public.h>
#endif

#include <stdlib.h>
#include <wolfssl/ssl.h>  /* compatibility layer */
#include <wolfssl/error-ssl.h>

#include <wolfssl/test.h>
#include <tests/utils.h>
#include <testsuite/utils.h>

/* for testing compatibility layer callbacks */
#include "examples/server/server.h"

#ifndef NO_MD5
    #include <wolfssl/wolfcrypt/md5.h>
#endif
#ifndef NO_SHA
    #include <wolfssl/wolfcrypt/sha.h>
#endif
#ifndef NO_SHA256
    #include <wolfssl/wolfcrypt/sha256.h>
#endif
#ifdef WOLFSSL_SHA512
    #include <wolfssl/wolfcrypt/sha512.h>
#endif
#ifdef WOLFSSL_SHA384
    #include <wolfssl/wolfcrypt/sha512.h>
#endif
#ifdef WOLFSSL_SHA3
    #include <wolfssl/wolfcrypt/sha3.h>
#endif
#ifdef WOLFSSL_SM3
    #include <wolfssl/wolfcrypt/sm3.h>
#endif
#ifndef NO_AES
    #include <wolfssl/wolfcrypt/aes.h>
    #ifdef HAVE_AES_DECRYPT
        #include <wolfssl/wolfcrypt/wc_encrypt.h>
    #endif
#endif
#ifdef WOLFSSL_SM4
    #include <wolfssl/wolfcrypt/sm4.h>
#endif
#ifdef WOLFSSL_RIPEMD
    #include <wolfssl/wolfcrypt/ripemd.h>
#endif
#ifndef NO_DES3
    #include <wolfssl/wolfcrypt/des3.h>
    #include <wolfssl/wolfcrypt/wc_encrypt.h>
#endif
#ifdef WC_RC2
    #include <wolfssl/wolfcrypt/rc2.h>
#endif

#ifndef NO_HMAC
    #include <wolfssl/wolfcrypt/hmac.h>
#endif

#ifdef HAVE_CHACHA
    #include <wolfssl/wolfcrypt/chacha.h>
#endif

#ifdef HAVE_POLY1305
    #include <wolfssl/wolfcrypt/poly1305.h>
#endif

#if defined(HAVE_CHACHA) && defined(HAVE_POLY1305)
    #include <wolfssl/wolfcrypt/chacha20_poly1305.h>
#endif

#ifdef HAVE_CAMELLIA
    #include <wolfssl/wolfcrypt/camellia.h>
#endif

#ifndef NO_RC4
    #include <wolfssl/wolfcrypt/arc4.h>
#endif

#ifdef HAVE_BLAKE2
    #include <wolfssl/wolfcrypt/blake2.h>
#endif

#ifndef NO_RSA
    #include <wolfssl/wolfcrypt/rsa.h>
#endif

#ifndef NO_SIG_WRAPPER
    #include <wolfssl/wolfcrypt/signature.h>
#endif

#ifdef HAVE_AESCCM
    #include <wolfssl/wolfcrypt/aes.h>
#endif

#ifdef HAVE_PKCS7
    #include <wolfssl/wolfcrypt/pkcs7.h>
    #include <wolfssl/wolfcrypt/asn.h>
    #ifdef HAVE_LIBZ
        #include <wolfssl/wolfcrypt/compress.h>
    #endif
#endif

#ifdef WOLFSSL_SMALL_CERT_VERIFY
    #include <wolfssl/wolfcrypt/asn.h>
#endif

#ifndef NO_DSA
    #include <wolfssl/wolfcrypt/dsa.h>
#endif

#ifdef WOLFSSL_CMAC
    #include <wolfssl/wolfcrypt/cmac.h>
#endif

#ifdef HAVE_ED25519
    #include <wolfssl/wolfcrypt/ed25519.h>
#endif
#ifdef HAVE_CURVE25519
    #include <wolfssl/wolfcrypt/curve25519.h>
#endif
#ifdef HAVE_ED448
    #include <wolfssl/wolfcrypt/ed448.h>
#endif
#ifdef HAVE_CURVE448
    #include <wolfssl/wolfcrypt/curve448.h>
#endif

#ifdef WOLFSSL_HAVE_MLKEM
    #include <wolfssl/wolfcrypt/mlkem.h>
#ifdef WOLFSSL_WC_MLKEM
    #include <wolfssl/wolfcrypt/wc_mlkem.h>
#endif
#endif
#ifdef HAVE_DILITHIUM
    #include <wolfssl/wolfcrypt/dilithium.h>
#endif

#ifdef HAVE_PKCS12
    #include <wolfssl/wolfcrypt/pkcs12.h>
#endif

#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL) || \
    defined(OPENSSL_ALL)
    #include <wolfssl/openssl/ssl.h>
    #ifndef NO_ASN
        /* for ASN_COMMON_NAME DN_tags enum */
        #include <wolfssl/wolfcrypt/asn.h>
    #endif
    #ifdef HAVE_OCSP
        #include <wolfssl/openssl/ocsp.h>
    #endif
#endif
#ifdef OPENSSL_EXTRA
    #include <wolfssl/openssl/cmac.h>
    #include <wolfssl/openssl/x509v3.h>
    #include <wolfssl/openssl/asn1.h>
    #include <wolfssl/openssl/crypto.h>
    #include <wolfssl/openssl/pkcs12.h>
    #include <wolfssl/openssl/evp.h>
    #include <wolfssl/openssl/dh.h>
    #include <wolfssl/openssl/bn.h>
    #include <wolfssl/openssl/buffer.h>
    #include <wolfssl/openssl/pem.h>
    #include <wolfssl/openssl/ec.h>
    #include <wolfssl/openssl/ecdh.h>
    #include <wolfssl/openssl/engine.h>
    #include <wolfssl/openssl/hmac.h>
    #include <wolfssl/openssl/objects.h>
    #include <wolfssl/openssl/rand.h>
    #include <wolfssl/openssl/modes.h>
    #include <wolfssl/openssl/fips_rand.h>
    #include <wolfssl/openssl/kdf.h>
    #include <wolfssl/openssl/x509_vfy.h>
#ifdef OPENSSL_ALL
    #include <wolfssl/openssl/txt_db.h>
    #include <wolfssl/openssl/lhash.h>
#endif
#ifndef NO_AES
    #include <wolfssl/openssl/aes.h>
#endif
#ifndef NO_DES3
    #include <wolfssl/openssl/des.h>
#endif
#ifndef NO_RC4
    #include <wolfssl/openssl/rc4.h>
#endif
#ifdef HAVE_ECC
    #include <wolfssl/openssl/ecdsa.h>
#endif
#ifdef HAVE_PKCS7
    #include <wolfssl/openssl/pkcs7.h>
#endif
#ifdef HAVE_CURVE25519
    #include <wolfssl/openssl/ec25519.h>
#endif
#ifdef HAVE_ED25519
    #include <wolfssl/openssl/ed25519.h>
#endif
#ifdef HAVE_CURVE448
    #include <wolfssl/openssl/ec448.h>
#endif
#ifdef HAVE_ED448
    #include <wolfssl/openssl/ed448.h>
#endif
#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) && defined(WOLFCRYPT_HAVE_SRP) && \
    !defined(NO_SHA256) && !defined(RC_NO_RNG)
        #include <wolfssl/wolfcrypt/srp.h>
#endif

#if (defined(SESSION_CERTS) && defined(TEST_PEER_CERT_CHAIN)) || \
    defined(HAVE_SESSION_TICKET) || (defined(OPENSSL_EXTRA) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN)) || \
    defined(WOLFSSL_TEST_STATIC_BUILD) || defined(WOLFSSL_DTLS) || \
    defined(HAVE_ECH) || defined(HAVE_EX_DATA) || !defined(NO_SESSION_CACHE) \
    || !defined(WOLFSSL_NO_TLS12) || defined(WOLFSSL_TLS13)
    /* for testing SSL_get_peer_cert_chain, or SESSION_TICKET_HINT_DEFAULT,
     * for setting authKeyIdSrc in WOLFSSL_X509, or testing DTLS sequence
     * number tracking */
    #include "wolfssl/internal.h"
#endif

/* include misc.c here regardless of NO_INLINE, because misc.c implementations
 * have default (hidden) visibility, and in the absence of visibility, it's
 * benign to mask out the library implementation.
 */
#define WOLFSSL_MISC_INCLUDED
#include <wolfcrypt/src/misc.c>

#include <tests/api/api.h>
#include <tests/api/api_decl.h>

/* Gather test declarations to include them in the testCases array */
#include <tests/api/test_md2.h>
#include <tests/api/test_md4.h>
#include <tests/api/test_md5.h>
#include <tests/api/test_sha.h>
#include <tests/api/test_sha256.h>
#include <tests/api/test_sha512.h>
#include <tests/api/test_sha3.h>
#include <tests/api/test_blake2.h>
#include <tests/api/test_sm3.h>
#include <tests/api/test_ripemd.h>
#include <tests/api/test_hash.h>
#include <tests/api/test_hmac.h>
#include <tests/api/test_cmac.h>
#include <tests/api/test_des3.h>
#include <tests/api/test_chacha.h>
#include <tests/api/test_poly1305.h>
#include <tests/api/test_chacha20_poly1305.h>
#include <tests/api/test_camellia.h>
#include <tests/api/test_arc4.h>
#include <tests/api/test_rc2.h>
#include <tests/api/test_aes.h>
#include <tests/api/test_ascon.h>
#include <tests/api/test_sm4.h>
#include <tests/api/test_wc_encrypt.h>
#include <tests/api/test_random.h>
#include <tests/api/test_wolfmath.h>
#include <tests/api/test_rsa.h>
#include <tests/api/test_dsa.h>
#include <tests/api/test_dh.h>
#include <tests/api/test_ecc.h>
#include <tests/api/test_sm2.h>
#include <tests/api/test_curve25519.h>
#include <tests/api/test_ed25519.h>
#include <tests/api/test_curve448.h>
#include <tests/api/test_ed448.h>
#include <tests/api/test_mlkem.h>
#include <tests/api/test_mldsa.h>
#include <tests/api/test_signature.h>
#include <tests/api/test_dtls.h>
#include <tests/api/test_ocsp.h>
#include <tests/api/test_evp.h>
#include <tests/api/test_tls_ext.h>

#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_RSA)        && !defined(SINGLE_THREADED) && \
    !defined(NO_WOLFSSL_SERVER) && !defined(NO_WOLFSSL_CLIENT)
    #define HAVE_IO_TESTS_DEPENDENCIES
#endif

#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_RSA) && \
    !defined(NO_WOLFSSL_SERVER) && !defined(NO_WOLFSSL_CLIENT) && \
    !defined(WOLFSSL_TIRTOS)
    #define HAVE_SSL_MEMIO_TESTS_DEPENDENCIES
#endif

#if !defined(NO_RSA) && !defined(NO_SHA) && !defined(NO_FILESYSTEM) && \
    !defined(NO_CERTS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(WOLFSSL_NO_CLIENT_AUTH))
    #define HAVE_CERT_CHAIN_VALIDATION
#endif

#if defined(WOLFSSL_STATIC_MEMORY) && !defined(WOLFCRYPT_ONLY)
    #if (defined(HAVE_ECC) && !defined(ALT_ECC_SIZE)) || defined(SESSION_CERTS)
        #ifdef OPENSSL_EXTRA
            #define TEST_TLS_STATIC_MEMSZ (400000)
        #else
            #define TEST_TLS_STATIC_MEMSZ (320000)
        #endif
    #else
            #define TEST_TLS_STATIC_MEMSZ (80000)
    #endif
#endif

#if defined(HAVE_PKCS7)
    typedef struct {
        const byte* content;
        word32      contentSz;
        int         contentOID;
        int         encryptOID;
        int         keyWrapOID;
        int         keyAgreeOID;
        byte*       cert;
        size_t      certSz;
        byte*       privateKey;
        word32      privateKeySz;
    } pkcs7EnvelopedVector;

    #ifndef NO_PKCS7_ENCRYPTED_DATA
        typedef struct {
            const byte*     content;
            word32          contentSz;
            int             contentOID;
            int             encryptOID;
            byte*           encryptionKey;
            word32          encryptionKeySz;
        } pkcs7EncryptedVector;
    #endif
#endif /* HAVE_PKCS7 */

#ifdef WOLFSSL_DUMP_MEMIO_STREAM
const char* currentTestName;
char tmpDirName[16];
int tmpDirNameSet = 0;
#endif

/*----------------------------------------------------------------------------*
 | Constants
 *----------------------------------------------------------------------------*/

#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_SERVER) || !defined(NO_WOLFSSL_CLIENT))
    static const char* bogusFile  =
    #ifdef _WIN32
        "NUL"
    #else
        "/dev/null"
    #endif
    ;
#endif /* !NO_FILESYSTEM && !NO_CERTS && (!NO_WOLFSSL_SERVER || !NO_WOLFSSL_CLIENT) */

enum {
    TESTING_RSA = 1,
    TESTING_ECC = 2
};

#ifdef WOLFSSL_QNX_CAAM
#include <wolfssl/wolfcrypt/port/caam/wolfcaam.h>
int testDevId = WOLFSSL_CAAM_DEVID;
#else
int testDevId = INVALID_DEVID;
#endif

/*----------------------------------------------------------------------------*
 | BIO with fixed read/write size
 *----------------------------------------------------------------------------*/

#if defined(OPENSSL_EXTRA) && !defined(NO_BIO)

static int wolfssl_bio_s_fixed_mem_write(WOLFSSL_BIO* bio, const char* data,
    int len)
{
    if ((bio == NULL) || (bio->ptr.mem_buf_data == NULL) || (data == NULL)) {
        len = 0;
    }
    else {
        if (bio->wrSz - bio->wrIdx < len) {
            len = bio->wrSz - bio->wrIdx;
        }
        XMEMCPY(bio->ptr.mem_buf_data + bio->wrIdx, data, (size_t)len);
        bio->wrIdx += len;
    }

    return len;
}

static int wolfssl_bio_s_fixed_mem_read(WOLFSSL_BIO* bio, char* data, int len)
{
    if ((bio == NULL) || (bio->ptr.mem_buf_data == NULL) || (data == NULL)) {
        len = 0;
    }
    else {
        if (bio->wrSz - bio->rdIdx < len) {
            len = bio->wrSz - bio->rdIdx;
        }
        XMEMCPY(data, bio->ptr.mem_buf_data + bio->rdIdx, (size_t)len);
        bio->rdIdx += len;
    }

    return len;
}

static WOLFSSL_BIO_METHOD* wolfSSL_BIO_s_fixed_mem(void)
{
    static WOLFSSL_BIO_METHOD meth;

    meth.type = WOLFSSL_BIO_BIO;
    XMEMCPY(meth.name, "Fixed Memory Size", 18);
    meth.writeCb = wolfssl_bio_s_fixed_mem_write;
    meth.readCb = wolfssl_bio_s_fixed_mem_read;

    return &meth;
}

#endif

/*----------------------------------------------------------------------------*
 | Setup
 *----------------------------------------------------------------------------*/

static int test_wolfSSL_Init(void)
{
    EXPECT_DECLS;
    ExpectIntEQ(wolfSSL_Init(), WOLFSSL_SUCCESS);
    return EXPECT_RESULT();
}


static int test_wolfSSL_Cleanup(void)
{
    EXPECT_DECLS;
    ExpectIntEQ(wolfSSL_Cleanup(), WOLFSSL_SUCCESS);
    return EXPECT_RESULT();
}


/*  Initialize the wolfCrypt state.
 *  POST: 0 success.
 */
static int test_wolfCrypt_Init(void)
{
    EXPECT_DECLS;
    ExpectIntEQ(wolfCrypt_Init(), 0);
    return EXPECT_RESULT();

} /* END test_wolfCrypt_Init */

static int test_wolfCrypt_Cleanup(void)
{
    EXPECT_DECLS;
    ExpectIntEQ(wolfCrypt_Cleanup(), 0);
    return EXPECT_RESULT();
}


#ifdef WOLFSSL_STATIC_MEMORY
    #define TEST_LSM_STATIC_SIZE 440000
    /* Create new bucket list, using the default list, adding
     * one dang large buffer size. */
    #define TEST_LSM_DEF_BUCKETS (WOLFMEM_DEF_BUCKETS+1)
    #define TEST_LSM_BUCKETS WOLFMEM_BUCKETS,(LARGEST_MEM_BUCKET*2)
    #define TEST_LSM_DIST WOLFMEM_DIST,1
#endif

static int test_wc_LoadStaticMemory_ex(void)
{
    EXPECT_DECLS;
#ifdef WOLFSSL_STATIC_MEMORY
    byte staticMemory[TEST_LSM_STATIC_SIZE];
    word32 sizeList[TEST_LSM_DEF_BUCKETS] = { TEST_LSM_BUCKETS };
    word32 distList[TEST_LSM_DEF_BUCKETS] = { TEST_LSM_DIST };
    WOLFSSL_HEAP_HINT* heap;

    /* For this test, the size and dist lists will be the ones configured
     * for the build, or default. The value of WOLFMEM_DEF_BUCKETS is 9,
     * so these lists are 10 long. For most tests, the value of
     * WOLFMEM_DEF_BUCKETS is used. There's a test case where one is added
     * to that, to make sure the list size is larger than
     * WOLFMEM_MAX_BUCKETS. */

    /* Pass in zero everything. */
    ExpectIntEQ(wc_LoadStaticMemory_ex(NULL, 0, NULL, NULL, NULL, 0, 0, 0),
            WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    /* Set the heap pointer to NULL. */
    ExpectIntEQ(wc_LoadStaticMemory_ex(NULL,
                WOLFMEM_DEF_BUCKETS, sizeList, distList,
                staticMemory, (word32)sizeof(staticMemory),
                0, 1),
            WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    /* Set other pointer values to NULL one at a time. */
    heap = NULL;
    ExpectIntEQ(wc_LoadStaticMemory_ex(&heap,
                WOLFMEM_DEF_BUCKETS, NULL, distList,
                staticMemory, (word32)sizeof(staticMemory),
                0, 1),
            WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    heap = NULL;
    ExpectIntEQ(wc_LoadStaticMemory_ex(&heap,
                WOLFMEM_DEF_BUCKETS, sizeList, NULL,
                staticMemory, (word32)sizeof(staticMemory),
                0, 1),
            WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    heap = NULL;
    ExpectIntEQ(wc_LoadStaticMemory_ex(&heap,
                WOLFMEM_DEF_BUCKETS, sizeList, distList,
                NULL, (word32)sizeof(staticMemory),
                0, 1),
            WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    /* Set the size of the static buffer to 0. */
    heap = NULL;
    ExpectIntEQ(wc_LoadStaticMemory_ex(&heap,
                WOLFMEM_DEF_BUCKETS, sizeList, distList,
                staticMemory, 0,
                0, 1),
            WC_NO_ERR_TRACE(BUFFER_E));

    /* Set the size of the static buffer to one less than minimum allowed. */
    heap = NULL;
    ExpectIntEQ(wc_LoadStaticMemory_ex(&heap,
                WOLFMEM_DEF_BUCKETS, sizeList, distList,
                staticMemory,
                (word32)(sizeof(WOLFSSL_HEAP) + sizeof(WOLFSSL_HEAP_HINT)) - 1,
                0, 1),
            WC_NO_ERR_TRACE(BUFFER_E));

    /* Set the size of the static buffer to exactly the minimum size. */
    heap = NULL;
    ExpectIntEQ(wc_LoadStaticMemory_ex(&heap,
                WOLFMEM_DEF_BUCKETS, sizeList, distList,
                staticMemory,
                (word32)(sizeof(WOLFSSL_HEAP) + sizeof(WOLFSSL_HEAP_HINT)),
                0, 1),
            0);
    wc_UnloadStaticMemory(heap);

    /* Use more buckets than able. Success case. */
    heap = NULL;
    ExpectIntEQ(wc_LoadStaticMemory_ex(&heap,
                WOLFMEM_DEF_BUCKETS*2, sizeList, distList,
                staticMemory, (word32)sizeof(staticMemory),
                0, 1),
            0);
    wc_UnloadStaticMemory(heap);

    /* Success case. */
    heap = NULL;
    ExpectIntEQ(wc_LoadStaticMemory_ex(&heap,
                WOLFMEM_DEF_BUCKETS, sizeList, distList,
                staticMemory, (word32)sizeof(staticMemory),
                0, 1),
            0);
    wc_UnloadStaticMemory(heap);
#endif /* WOLFSSL_STATIC_MEMORY */
    return EXPECT_RESULT();
}


static int test_wc_LoadStaticMemory_CTX(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_STATIC_MEMORY) && !defined(NO_WOLFSSL_CLIENT)
    byte staticMemory[TEST_LSM_STATIC_SIZE];
    word32 sizeList[TEST_LSM_DEF_BUCKETS] = { TEST_LSM_BUCKETS };
    word32 distList[TEST_LSM_DEF_BUCKETS] = { TEST_LSM_DIST };
    WOLFSSL_HEAP_HINT* heap;
    WOLFSSL_CTX *ctx1 = NULL, *ctx2 = NULL;


    /* Set the size of the static buffer to exactly the minimum size. */
    heap = NULL;
    ExpectIntEQ(wc_LoadStaticMemory_ex(&heap,
                WOLFMEM_DEF_BUCKETS, sizeList, distList,
                staticMemory, sizeof(staticMemory), 0, 1),
            0);

    /* Creating two WOLFSSL_CTX objects from the same heap hint and free'ing
     * them should not cause issues. */
    ExpectNotNull((ctx1 = wolfSSL_CTX_new_ex(wolfSSLv23_client_method_ex(heap),
        heap)));
    wolfSSL_CTX_free(ctx1);
    ExpectNotNull((ctx2 = wolfSSL_CTX_new_ex(wolfSSLv23_client_method_ex(heap),
        heap)));
    wolfSSL_CTX_free(ctx2);

    /* two CTX's at once */
    ExpectNotNull((ctx1 = wolfSSL_CTX_new_ex(wolfSSLv23_client_method_ex(heap),
        heap)));
    ExpectNotNull((ctx2 = wolfSSL_CTX_new_ex(wolfSSLv23_client_method_ex(heap),
        heap)));
    wolfSSL_CTX_free(ctx1);
    wolfSSL_CTX_free(ctx2);

    wc_UnloadStaticMemory(heap);
#endif /* WOLFSSL_STATIC_MEMORY */
    return EXPECT_RESULT();
}


/*----------------------------------------------------------------------------*
 | Platform dependent function test
 *----------------------------------------------------------------------------*/
static int test_fileAccess(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_TEST_PLATFORMDEPEND) && !defined(NO_FILESYSTEM)
    const char *fname[] = {
        svrCertFile, svrKeyFile, caCertFile,
        eccCertFile, eccKeyFile, eccRsaCertFile,
        cliCertFile, cliCertDerFile, cliKeyFile,
        dhParamFile,
        cliEccKeyFile, cliEccCertFile, caEccCertFile, edCertFile, edKeyFile,
        cliEdCertFile, cliEdKeyFile, caEdCertFile,
        NULL
    };
    const char derfile[] = "./certs/server-cert.der";
    XFILE f = XBADFILE;
    size_t sz;
    byte *buff = NULL;
    int i;

    ExpectTrue(XFOPEN("badfilename", "rb") == XBADFILE);
    for (i=0; EXPECT_SUCCESS() && fname[i] != NULL ; i++) {
        ExpectTrue((f = XFOPEN(fname[i], "rb")) != XBADFILE);
        XFCLOSE(f);
    }

    ExpectTrue((f = XFOPEN(derfile, "rb")) != XBADFILE);
    ExpectTrue(XFSEEK(f, 0, XSEEK_END) == 0);
    ExpectIntGE(sz = (size_t) XFTELL(f), sizeof_server_cert_der_2048);
    ExpectTrue(XFSEEK(f, 0, XSEEK_SET) == 0);
    ExpectTrue((buff = (byte*)XMALLOC(sz, NULL, DYNAMIC_TYPE_FILE)) != NULL);
    ExpectTrue(XFREAD(buff, 1, sz, f) == sz);
    ExpectIntEQ(XMEMCMP(server_cert_der_2048, buff, sz), 0);
    XFREE(buff, NULL, DYNAMIC_TYPE_FILE);
    XFCLOSE(f);
#endif
    return EXPECT_RESULT();
}

/*----------------------------------------------------------------------------*
 | Method Allocators
 *----------------------------------------------------------------------------*/

static int test_wolfSSL_Method_Allocators(void)
{
    EXPECT_DECLS;

    #define TEST_METHOD_ALLOCATOR(allocator, condition) \
        do {                                            \
            WOLFSSL_METHOD *method = NULL;              \
            condition(method = allocator());            \
            XFREE(method, 0, DYNAMIC_TYPE_METHOD);      \
        } while (0)

    #define TEST_VALID_METHOD_ALLOCATOR(a) \
            TEST_METHOD_ALLOCATOR(a, ExpectNotNull)

    #define TEST_INVALID_METHOD_ALLOCATOR(a) \
            TEST_METHOD_ALLOCATOR(a, ExpectNull)

#ifndef NO_TLS
#ifndef NO_OLD_TLS
    #ifdef WOLFSSL_ALLOW_SSLV3
        #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfSSLv3_server_method);
        #endif
        #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfSSLv3_client_method);
        #endif
    #endif
    #ifdef WOLFSSL_ALLOW_TLSV10
        #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_server_method);
        #endif
        #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_client_method);
        #endif
    #endif
    #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_1_server_method);
    #endif
    #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_1_client_method);
    #endif
#endif /* !NO_OLD_TLS */

#ifndef WOLFSSL_NO_TLS12
    #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_2_server_method);
    #endif
    #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_2_client_method);
    #endif
#endif /* !WOLFSSL_NO_TLS12 */

#ifdef WOLFSSL_TLS13
    #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_3_server_method);
    #endif
    #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_3_client_method);
    #endif
#endif /* WOLFSSL_TLS13 */

#ifndef NO_WOLFSSL_SERVER
    TEST_VALID_METHOD_ALLOCATOR(wolfSSLv23_server_method);
#endif
#ifndef NO_WOLFSSL_CLIENT
    TEST_VALID_METHOD_ALLOCATOR(wolfSSLv23_client_method);
#endif

#ifdef WOLFSSL_DTLS
    #ifndef NO_OLD_TLS
        #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfDTLSv1_server_method);
        #endif
        #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfDTLSv1_client_method);
        #endif
    #endif
    #ifndef WOLFSSL_NO_TLS12
        #ifndef NO_WOLFSSL_SERVER
        TEST_VALID_METHOD_ALLOCATOR(wolfDTLSv1_2_server_method);
        #endif
        #ifndef NO_WOLFSSL_CLIENT
        TEST_VALID_METHOD_ALLOCATOR(wolfDTLSv1_2_client_method);
        #endif
    #endif
#endif /* WOLFSSL_DTLS */

#if !defined(NO_OLD_TLS) && defined(OPENSSL_EXTRA)
    /* Stubs */
    #ifndef NO_WOLFSSL_SERVER
        TEST_INVALID_METHOD_ALLOCATOR(wolfSSLv2_server_method);
    #endif
    #ifndef NO_WOLFSSL_CLIENT
        TEST_INVALID_METHOD_ALLOCATOR(wolfSSLv2_client_method);
    #endif
#endif

    /* Test Either Method (client or server) */
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    TEST_VALID_METHOD_ALLOCATOR(wolfSSLv23_method);
    #ifndef NO_OLD_TLS
        #ifdef WOLFSSL_ALLOW_TLSV10
            TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_method);
        #endif
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_1_method);
    #endif /* !NO_OLD_TLS */
    #ifndef WOLFSSL_NO_TLS12
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_2_method);
    #endif /* !WOLFSSL_NO_TLS12 */
    #ifdef WOLFSSL_TLS13
        TEST_VALID_METHOD_ALLOCATOR(wolfTLSv1_3_method);
    #endif /* WOLFSSL_TLS13 */
    #ifdef WOLFSSL_DTLS
        TEST_VALID_METHOD_ALLOCATOR(wolfDTLS_method);
        #ifndef NO_OLD_TLS
            TEST_VALID_METHOD_ALLOCATOR(wolfDTLSv1_method);
        #endif /* !NO_OLD_TLS */
        #ifndef WOLFSSL_NO_TLS12
            TEST_VALID_METHOD_ALLOCATOR(wolfDTLSv1_2_method);
        #endif /* !WOLFSSL_NO_TLS12 */
        #ifdef WOLFSSL_DTLS13
            TEST_VALID_METHOD_ALLOCATOR(wolfDTLSv1_3_method);
        #endif /* WOLFSSL_DTLS13 */
    #endif /* WOLFSSL_DTLS */
#endif /* OPENSSL_EXTRA || WOLFSSL_EITHER_SIDE */
#endif /* !NO_TLS */
    return EXPECT_RESULT();
}

#if defined(WOLFSSL_DUAL_ALG_CERTS) && !defined(NO_FILESYSTEM)
/*----------------------------------------------------------------------------*
 | Dual algorithm Certificate Tests
 *----------------------------------------------------------------------------*/
#define LARGE_TEMP_SZ 4096

/* To better understand this, please see the X9.146 example in wolfssl-examples
 * repo. */
static int do_dual_alg_root_certgen(byte **out, char *caKeyFile,
                                    char *sapkiFile, char *altPrivFile)
{
    EXPECT_DECLS;
    FILE* file = NULL;
    Cert newCert;
    DecodedCert preTBS;

    byte caKeyBuf[LARGE_TEMP_SZ];
    word32 caKeySz = LARGE_TEMP_SZ;
    byte sapkiBuf[LARGE_TEMP_SZ];
    word32 sapkiSz = LARGE_TEMP_SZ;
    byte altPrivBuf[LARGE_TEMP_SZ];
    word32 altPrivSz = LARGE_TEMP_SZ;
    byte altSigAlgBuf[LARGE_TEMP_SZ];
    word32 altSigAlgSz = LARGE_TEMP_SZ;
    byte scratchBuf[LARGE_TEMP_SZ];
    word32 scratchSz = LARGE_TEMP_SZ;
    byte preTbsBuf[LARGE_TEMP_SZ];
    word32 preTbsSz = LARGE_TEMP_SZ;
    byte altSigValBuf[LARGE_TEMP_SZ];
    word32 altSigValSz = LARGE_TEMP_SZ;
    byte *outBuf = NULL;
    word32 outSz = LARGE_TEMP_SZ;
    WC_RNG rng;
    RsaKey caKey;
    ecc_key altCaKey;
    word32 idx = 0;

    XMEMSET(&rng, 0, sizeof(WC_RNG));
    XMEMSET(&caKey, 0, sizeof(RsaKey));
    XMEMSET(&altCaKey, 0, sizeof(ecc_key));

    ExpectNotNull(outBuf = (byte*)XMALLOC(outSz, NULL,
                  DYNAMIC_TYPE_TMP_BUFFER));
    ExpectIntEQ(wc_InitRng(&rng), 0);
    XMEMSET(caKeyBuf, 0, caKeySz);
    ExpectNotNull(file = fopen(caKeyFile, "rb"));
    ExpectIntGT(caKeySz = (word32)fread(caKeyBuf, 1, caKeySz, file), 0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    ExpectIntEQ(wc_InitRsaKey_ex(&caKey, NULL, INVALID_DEVID), 0);
    idx = 0;
    ExpectIntEQ(wc_RsaPrivateKeyDecode(caKeyBuf, &idx, &caKey, caKeySz),
                0);
    XMEMSET(sapkiBuf, 0, sapkiSz);
    ExpectNotNull(file = fopen(sapkiFile, "rb"));
    ExpectIntGT(sapkiSz = (word32)fread(sapkiBuf, 1, sapkiSz, file), 0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    XMEMSET(altPrivBuf, 0, altPrivSz);
    ExpectNotNull(file = fopen(altPrivFile, "rb"));
    ExpectIntGT(altPrivSz = (word32)fread(altPrivBuf, 1, altPrivSz, file), 0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    wc_ecc_init(&altCaKey);
    idx = 0;
    ExpectIntEQ(wc_EccPrivateKeyDecode(altPrivBuf, &idx, &altCaKey,
                                       (word32)altPrivSz), 0);
    XMEMSET(altSigAlgBuf, 0, altSigAlgSz);
    ExpectIntGT(altSigAlgSz = SetAlgoID(CTC_SHA256wECDSA, altSigAlgBuf,
                                         oidSigType, 0), 0);
    wc_InitCert(&newCert);
    strncpy(newCert.subject.country, "US", CTC_NAME_SIZE);
    strncpy(newCert.subject.state, "MT", CTC_NAME_SIZE);
    strncpy(newCert.subject.locality, "Bozeman", CTC_NAME_SIZE);
    strncpy(newCert.subject.org, "wolfSSL", CTC_NAME_SIZE);
    strncpy(newCert.subject.unit, "Engineering", CTC_NAME_SIZE);
    strncpy(newCert.subject.commonName, "www.wolfssl.com", CTC_NAME_SIZE);
    strncpy(newCert.subject.email, "root@wolfssl.com", CTC_NAME_SIZE);
    strncpy((char*)newCert.beforeDate, "\x18\x0f""20250101000000Z",
        CTC_DATE_SIZE);
    newCert.beforeDateSz = 17;
    strncpy((char*)newCert.afterDate, "\x18\x0f""20493112115959Z",
        CTC_DATE_SIZE);
    newCert.afterDateSz = 17;
    newCert.sigType = CTC_SHA256wRSA;
    newCert.isCA    = 1;

    ExpectIntEQ(wc_SetCustomExtension(&newCert, 0, "1.2.3.4.5",
                (const byte *)"This is NOT a critical extension", 32), 0);
    ExpectIntEQ(wc_SetCustomExtension(&newCert, 0, "2.5.29.72", sapkiBuf,
                sapkiSz), 0);
    ExpectIntEQ(wc_SetCustomExtension(&newCert, 0, "2.5.29.73", altSigAlgBuf,
                                altSigAlgSz), 0);

    XMEMSET(scratchBuf, 0, scratchSz);
    ExpectIntGT(scratchSz = wc_MakeSelfCert(&newCert, scratchBuf, scratchSz,
                &caKey, &rng), 0);

    wc_InitDecodedCert(&preTBS, scratchBuf, scratchSz, 0);
    ExpectIntEQ(wc_ParseCert(&preTBS, CERT_TYPE, NO_VERIFY, NULL), 0);

    XMEMSET(preTbsBuf, 0, preTbsSz);
    ExpectIntGT(preTbsSz = wc_GeneratePreTBS(&preTBS, preTbsBuf, preTbsSz), 0);
    XMEMSET(altSigValBuf, 0, altSigValSz);
    ExpectIntGT(altSigValSz = wc_MakeSigWithBitStr(altSigValBuf, altSigValSz,
                CTC_SHA256wECDSA, preTbsBuf, preTbsSz, ECC_TYPE, &altCaKey,
                &rng), 0);
    ExpectIntEQ(wc_SetCustomExtension(&newCert, 0, "2.5.29.74", altSigValBuf,
                altSigValSz), 0);

    /* Finally, generate the new certificate. */
    if (outBuf != NULL) {
        XMEMSET(outBuf, 0, outSz);
    }
    ExpectIntGT(outSz = wc_MakeSelfCert(&newCert, outBuf, outSz, &caKey, &rng),
                0);
    *out = outBuf;

    wc_FreeRsaKey(&caKey);
    wc_FreeRng(&rng);
    wc_FreeDecodedCert(&preTBS);
    return outSz;
}

static int do_dual_alg_server_certgen(byte **out, char *caKeyFile,
                                      char *sapkiFile, char *altPrivFile,
                                      char *serverKeyFile,
                                      byte *caCertBuf, int caCertSz)
{
    EXPECT_DECLS;
    FILE* file = NULL;
    Cert newCert;
    DecodedCert preTBS;

    byte serverKeyBuf[LARGE_TEMP_SZ];
    word32 serverKeySz = LARGE_TEMP_SZ;
    byte caKeyBuf[LARGE_TEMP_SZ];
    word32 caKeySz = LARGE_TEMP_SZ;
    byte sapkiBuf[LARGE_TEMP_SZ];
    word32 sapkiSz = LARGE_TEMP_SZ;
    byte altPrivBuf[LARGE_TEMP_SZ];
    word32 altPrivSz = LARGE_TEMP_SZ;
    byte altSigAlgBuf[LARGE_TEMP_SZ];
    word32 altSigAlgSz = LARGE_TEMP_SZ;
    byte scratchBuf[LARGE_TEMP_SZ];
    word32 scratchSz = LARGE_TEMP_SZ;
    byte preTbsBuf[LARGE_TEMP_SZ];
    word32 preTbsSz = LARGE_TEMP_SZ;
    byte altSigValBuf[LARGE_TEMP_SZ];
    word32 altSigValSz = LARGE_TEMP_SZ;
    byte *outBuf = NULL;
    word32 outSz = LARGE_TEMP_SZ;
    WC_RNG rng;
    RsaKey caKey;
    RsaKey serverKey;
    ecc_key altCaKey;
    word32 idx = 0;

    XMEMSET(&rng, 0, sizeof(WC_RNG));
    XMEMSET(&caKey, 0, sizeof(RsaKey));
    XMEMSET(&serverKey, 0, sizeof(RsaKey));
    XMEMSET(&altCaKey, 0, sizeof(ecc_key));

    ExpectNotNull(outBuf = (byte*)XMALLOC(outSz, NULL,
                  DYNAMIC_TYPE_TMP_BUFFER));
    ExpectIntEQ(wc_InitRng(&rng), 0);
    XMEMSET(serverKeyBuf, 0, serverKeySz);
    ExpectNotNull(file = fopen(serverKeyFile, "rb"));
    ExpectIntGT(serverKeySz = (word32)fread(serverKeyBuf, 1, serverKeySz, file),
                0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    ExpectIntEQ(wc_InitRsaKey_ex(&serverKey, NULL, INVALID_DEVID), 0);
    idx = 0;
    ExpectIntEQ(wc_RsaPrivateKeyDecode(serverKeyBuf, &idx, &serverKey,
                (word32)serverKeySz), 0);
    XMEMSET(caKeyBuf, 0, caKeySz);
    ExpectNotNull(file = fopen(caKeyFile, "rb"));
    ExpectIntGT(caKeySz = (word32)fread(caKeyBuf, 1, caKeySz, file), 0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    ExpectIntEQ(wc_InitRsaKey_ex(&caKey, NULL, INVALID_DEVID), 0);
    idx = 0;
    ExpectIntEQ(wc_RsaPrivateKeyDecode(caKeyBuf, &idx, &caKey,
                (word32)caKeySz), 0);
    XMEMSET(sapkiBuf, 0, sapkiSz);
    ExpectNotNull(file = fopen(sapkiFile, "rb"));
    ExpectIntGT(sapkiSz = (word32)fread(sapkiBuf, 1, sapkiSz, file), 0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    XMEMSET(altPrivBuf, 0, altPrivSz);
    ExpectNotNull(file = fopen(altPrivFile, "rb"));
    ExpectIntGT(altPrivSz = (word32)fread(altPrivBuf, 1, altPrivSz, file), 0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    wc_ecc_init(&altCaKey);
    idx = 0;
    ExpectIntEQ(wc_EccPrivateKeyDecode(altPrivBuf, &idx, &altCaKey,
                (word32)altPrivSz), 0);
    XMEMSET(altSigAlgBuf, 0, altSigAlgSz);
    ExpectIntGT(altSigAlgSz = SetAlgoID(CTC_SHA256wECDSA, altSigAlgBuf,
                oidSigType, 0), 0);
    wc_InitCert(&newCert);
    strncpy(newCert.subject.country, "US", CTC_NAME_SIZE);
    strncpy(newCert.subject.state, "MT", CTC_NAME_SIZE);
    strncpy(newCert.subject.locality, "Bozeman", CTC_NAME_SIZE);
    strncpy(newCert.subject.org, "wolfSSL", CTC_NAME_SIZE);
    strncpy(newCert.subject.unit, "Engineering", CTC_NAME_SIZE);
    strncpy(newCert.subject.commonName, "www.wolfssl.com", CTC_NAME_SIZE);
    strncpy(newCert.subject.email, "server@wolfssl.com", CTC_NAME_SIZE);
    strncpy((char*)newCert.beforeDate, "\x18\x0f""20250101000000Z",
        CTC_DATE_SIZE);
    newCert.beforeDateSz = 17;
    strncpy((char*)newCert.afterDate, "\x18\x0f""20493112115959Z",
        CTC_DATE_SIZE);
    newCert.afterDateSz = 17;

    newCert.sigType = CTC_SHA256wRSA;
    newCert.isCA    = 0;
    ExpectIntEQ(wc_SetIssuerBuffer(&newCert, caCertBuf, caCertSz), 0);
    ExpectIntEQ(wc_SetCustomExtension(&newCert, 0, "2.5.29.72", sapkiBuf,
                sapkiSz), 0);
    ExpectIntEQ(wc_SetCustomExtension(&newCert, 0, "2.5.29.73", altSigAlgBuf,
                altSigAlgSz), 0);
    XMEMSET(scratchBuf, 0, scratchSz);
    ExpectIntGT(wc_MakeCert(&newCert, scratchBuf, scratchSz, &serverKey, NULL,
                &rng), 0);
    ExpectIntGT(scratchSz = wc_SignCert(newCert.bodySz, newCert.sigType,
                scratchBuf, scratchSz, &caKey, NULL, &rng), 0);
    wc_InitDecodedCert(&preTBS, scratchBuf, scratchSz, 0);
    ExpectIntEQ(wc_ParseCert(&preTBS, CERT_TYPE, NO_VERIFY, NULL), 0);
    XMEMSET(preTbsBuf, 0, preTbsSz);
    ExpectIntGT(preTbsSz = wc_GeneratePreTBS(&preTBS, preTbsBuf, preTbsSz), 0);
    XMEMSET(altSigValBuf, 0, altSigValSz);
    ExpectIntGT(altSigValSz = wc_MakeSigWithBitStr(altSigValBuf, altSigValSz,
                CTC_SHA256wECDSA, preTbsBuf, preTbsSz, ECC_TYPE, &altCaKey,
                &rng), 0);
    ExpectIntEQ(wc_SetCustomExtension(&newCert, 0, "2.5.29.74",
                 altSigValBuf, altSigValSz), 0);
    /* Finally, generate the new certificate. */
    if (outBuf != NULL) {
        XMEMSET(outBuf, 0, outSz);
    }
    ExpectIntGT(wc_MakeCert(&newCert, outBuf, outSz, &serverKey, NULL, &rng),
                0);
    ExpectIntGT(outSz = wc_SignCert(newCert.bodySz, newCert.sigType, outBuf,
                outSz, &caKey, NULL, &rng), 0);
    *out = outBuf;
    wc_FreeRsaKey(&caKey);
    wc_FreeRsaKey(&serverKey);
    wc_FreeRng(&rng);
    wc_FreeDecodedCert(&preTBS);
    return outSz;
}

static int do_dual_alg_tls13_connection(byte *caCert, word32 caCertSz,
                                        byte *serverCert, word32 serverCertSz,
                                        byte *serverKey, word32 serverKeySz,
                                        int negative_test)
{
    EXPECT_DECLS;
    WOLFSSL_CTX *ctx_c = NULL;
    WOLFSSL_CTX *ctx_s = NULL;
    WOLFSSL *ssl_c = NULL;
    WOLFSSL *ssl_s = NULL;
    struct test_memio_ctx test_ctx;

    XMEMSET(&test_ctx, 0, sizeof(test_ctx));
    ExpectIntEQ(test_memio_setup_ex(&test_ctx, &ctx_c, &ctx_s, &ssl_c, &ssl_s,
                wolfTLSv1_3_client_method, wolfTLSv1_3_server_method,
                caCert, caCertSz, serverCert, serverCertSz,
                serverKey, serverKeySz), 0);
    if (negative_test) {
        ExpectTrue(test_memio_do_handshake(ssl_c, ssl_s, 10, NULL) != 0);
    }
    else {
        ExpectIntEQ(test_memio_do_handshake(ssl_c, ssl_s, 10, NULL), 0);
    }
    wolfSSL_free(ssl_c);
    wolfSSL_free(ssl_s);
    wolfSSL_CTX_free(ctx_c);
    wolfSSL_CTX_free(ctx_s);
    return EXPECT_RESULT();
}

/**
 * Function to generate a root certificate with dual algorithm support and
 * configurable criticality for extensions and path length constraints.
 *
 * @param out         [out] Pointer to store the generated certificate
 * @param caKeyFile   [in]  Path to the CA key file
 * @param sapkiFile   [in]  Path to the subject alternative public key info file
 * @param altPrivFile [in]  Path to the alternative private key file
 * @param setCrit     [in]  Flag to set criticality of extensions (1=critical, 0=non-critical)
 * @param setPathLen  [in]  Flag to set path length constraint (1=set, 0=don't set)
 * @param pathLen     [in]  Path length value (only used if setPathLen=1)
 * @return            Size of the generated certificate or negative on error
 */
static int do_dual_alg_root_certgen_crit(byte **out, char *caKeyFile,
                                      char *sapkiFile, char *altPrivFile,
                                      int setCrit, int setPathLen, int pathLen)
{
    EXPECT_DECLS;
    FILE* file = NULL;
    Cert newCert;
    DecodedCert preTBS;

    byte caKeyBuf[LARGE_TEMP_SZ];
    word32 caKeySz = LARGE_TEMP_SZ;
    byte sapkiBuf[LARGE_TEMP_SZ];
    word32 sapkiSz = LARGE_TEMP_SZ;
    byte altPrivBuf[LARGE_TEMP_SZ];
    word32 altPrivSz = LARGE_TEMP_SZ;
    byte altSigAlgBuf[LARGE_TEMP_SZ];
    word32 altSigAlgSz = LARGE_TEMP_SZ;
    byte scratchBuf[LARGE_TEMP_SZ];
    word32 scratchSz = LARGE_TEMP_SZ;
    byte preTbsBuf[LARGE_TEMP_SZ];
    word32 preTbsSz = LARGE_TEMP_SZ;
    byte altSigValBuf[LARGE_TEMP_SZ];
    word32 altSigValSz = LARGE_TEMP_SZ;
    byte *outBuf = NULL;
    word32 outSz = LARGE_TEMP_SZ;
    WC_RNG rng;
    RsaKey caKey;
    ecc_key altCaKey;
    word32 idx = 0;

    XMEMSET(&rng, 0, sizeof(WC_RNG));
    XMEMSET(&caKey, 0, sizeof(RsaKey));
    XMEMSET(&altCaKey, 0, sizeof(ecc_key));

    ExpectNotNull(outBuf = (byte*)XMALLOC(outSz, NULL,
                  DYNAMIC_TYPE_TMP_BUFFER));
    ExpectIntEQ(wc_InitRng(&rng), 0);
    XMEMSET(caKeyBuf, 0, caKeySz);
    ExpectNotNull(file = fopen(caKeyFile, "rb"));
    ExpectIntGT(caKeySz = (word32)fread(caKeyBuf, 1, caKeySz, file), 0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    ExpectIntEQ(wc_InitRsaKey_ex(&caKey, NULL, INVALID_DEVID), 0);
    idx = 0;
    ExpectIntEQ(wc_RsaPrivateKeyDecode(caKeyBuf, &idx, &caKey, caKeySz),
                0);
    XMEMSET(sapkiBuf, 0, sapkiSz);
    ExpectNotNull(file = fopen(sapkiFile, "rb"));
    ExpectIntGT(sapkiSz = (word32)fread(sapkiBuf, 1, sapkiSz, file), 0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    XMEMSET(altPrivBuf, 0, altPrivSz);
    ExpectNotNull(file = fopen(altPrivFile, "rb"));
    ExpectIntGT(altPrivSz = (word32)fread(altPrivBuf, 1, altPrivSz, file), 0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    wc_ecc_init(&altCaKey);
    idx = 0;
    ExpectIntEQ(wc_EccPrivateKeyDecode(altPrivBuf, &idx, &altCaKey,
                                       (word32)altPrivSz), 0);
    XMEMSET(altSigAlgBuf, 0, altSigAlgSz);
    ExpectIntGT(altSigAlgSz = SetAlgoID(CTC_SHA256wECDSA, altSigAlgBuf,
                                         oidSigType, 0), 0);
    wc_InitCert(&newCert);
    strncpy(newCert.subject.country, "US", CTC_NAME_SIZE);
    strncpy(newCert.subject.state, "MT", CTC_NAME_SIZE);
    strncpy(newCert.subject.locality, "Bozeman", CTC_NAME_SIZE);
    strncpy(newCert.subject.org, "wolfSSL", CTC_NAME_SIZE);
    strncpy(newCert.subject.unit, "Engineering", CTC_NAME_SIZE);
    strncpy(newCert.subject.commonName, "www.wolfssl.com", CTC_NAME_SIZE);
    strncpy(newCert.subject.email, "root@wolfssl.com", CTC_NAME_SIZE);
    strncpy((char*)newCert.beforeDate, "\x18\x0f""20250101000000Z",
        CTC_DATE_SIZE);
    newCert.beforeDateSz = 17;
    strncpy((char*)newCert.afterDate, "\x18\x0f""20493112115959Z",
        CTC_DATE_SIZE);
    newCert.afterDateSz = 17;
    newCert.sigType = CTC_SHA256wRSA;
    newCert.isCA    = 1;

    /* Set criticality of basic constraint extension if requested */
    if (setCrit) {
        newCert.basicConstCrit = 1;
    }

    /* Set pathlen if requested */
    if (setPathLen) {
        newCert.pathLen = pathLen;
        newCert.pathLenSet = 1;
    }

    ExpectIntEQ(wc_SetCustomExtension(&newCert, 0, "1.2.3.4.5",
                (const byte *)"This is NOT a critical extension", 32), 0);
    ExpectIntEQ(wc_SetCustomExtension(&newCert, setCrit, "2.5.29.72", sapkiBuf,
                sapkiSz), 0);
    ExpectIntEQ(wc_SetCustomExtension(&newCert, setCrit, "2.5.29.73",
                                      altSigAlgBuf, altSigAlgSz), 0);

    XMEMSET(scratchBuf, 0, scratchSz);
    ExpectIntGT(scratchSz = wc_MakeSelfCert(&newCert, scratchBuf, scratchSz,
                &caKey, &rng), 0);

    wc_InitDecodedCert(&preTBS, scratchBuf, scratchSz, 0);
    ExpectIntEQ(wc_ParseCert(&preTBS, CERT_TYPE, NO_VERIFY, NULL), 0);

    XMEMSET(preTbsBuf, 0, preTbsSz);
    ExpectIntGT(preTbsSz = wc_GeneratePreTBS(&preTBS, preTbsBuf, preTbsSz), 0);
    XMEMSET(altSigValBuf, 0, altSigValSz);
    ExpectIntGT(altSigValSz = wc_MakeSigWithBitStr(altSigValBuf, altSigValSz,
                CTC_SHA256wECDSA, preTbsBuf, preTbsSz, ECC_TYPE, &altCaKey,
                &rng), 0);
    ExpectIntEQ(wc_SetCustomExtension(&newCert, setCrit, "2.5.29.74",
                                      altSigValBuf, altSigValSz), 0);

    /* Finally, generate the new certificate. */
    if (outBuf != NULL) {
        XMEMSET(outBuf, 0, outSz);
        ExpectIntGT(outSz = wc_MakeSelfCert(&newCert, outBuf, outSz, &caKey,
                    &rng), 0);
        *out = outBuf;
    }
    else {
        outSz = 0;
    }

    wc_FreeDecodedCert(&preTBS);
    wc_ecc_free(&altCaKey);
    wc_FreeRsaKey(&caKey);
    wc_FreeRng(&rng);

    return (int)outSz;
}

/**
 * Function to generate a server certificate with dual algorithm support and
 * configurable criticality for extensions and path length constraints.
 *
 * @param out           [out] Pointer to store the generated certificate
 * @param caKeyFile     [in]  Path to the CA key file
 * @param sapkiFile     [in]  Path to the subject alternative public key info file
 * @param altPrivFile   [in]  Path to the alternative private key file
 * @param serverKeyFile [in]  Path to the server key file
 * @param caCertBuf     [in]  Buffer containing the CA certificate
 * @param caCertSz      [in]  Size of the CA certificate buffer
 * @param setCrit       [in]  Flag to set criticality of extensions (1=critical, 0=non-critical)
 * @param setPathLen    [in]  Flag to set path length constraint (1=set, 0=don't set)
 * @param pathLen       [in]  Path length value (only used if setPathLen=1)
 * @return              Size of the generated certificate or negative on error
 */
static int do_dual_alg_server_certgen_crit(byte **out, char *caKeyFile,
                                        char *sapkiFile, char *altPrivFile,
                                        char *serverKeyFile,
                                        byte *caCertBuf, int caCertSz,
                                        int setCrit)
{
    EXPECT_DECLS;
    FILE* file = NULL;
    Cert newCert;
    DecodedCert preTBS;

    byte serverKeyBuf[LARGE_TEMP_SZ];
    word32 serverKeySz = LARGE_TEMP_SZ;
    byte caKeyBuf[LARGE_TEMP_SZ];
    word32 caKeySz = LARGE_TEMP_SZ;
    byte sapkiBuf[LARGE_TEMP_SZ];
    word32 sapkiSz = LARGE_TEMP_SZ;
    byte altPrivBuf[LARGE_TEMP_SZ];
    word32 altPrivSz = LARGE_TEMP_SZ;
    byte altSigAlgBuf[LARGE_TEMP_SZ];
    word32 altSigAlgSz = LARGE_TEMP_SZ;
    byte scratchBuf[LARGE_TEMP_SZ];
    word32 scratchSz = LARGE_TEMP_SZ;
    byte preTbsBuf[LARGE_TEMP_SZ];
    word32 preTbsSz = LARGE_TEMP_SZ;
    byte altSigValBuf[LARGE_TEMP_SZ];
    word32 altSigValSz = LARGE_TEMP_SZ;
    byte *outBuf = NULL;
    word32 outSz = LARGE_TEMP_SZ;
    WC_RNG rng;
    RsaKey caKey;
    RsaKey serverKey;
    ecc_key altCaKey;
    word32 idx = 0;

    XMEMSET(&rng, 0, sizeof(WC_RNG));
    XMEMSET(&caKey, 0, sizeof(RsaKey));
    XMEMSET(&serverKey, 0, sizeof(RsaKey));
    XMEMSET(&altCaKey, 0, sizeof(ecc_key));

    ExpectNotNull(outBuf = (byte*)XMALLOC(outSz, NULL,
                  DYNAMIC_TYPE_TMP_BUFFER));
    ExpectIntEQ(wc_InitRng(&rng), 0);
    XMEMSET(serverKeyBuf, 0, serverKeySz);
    ExpectNotNull(file = fopen(serverKeyFile, "rb"));
    ExpectIntGT(serverKeySz = (word32)fread(serverKeyBuf, 1, serverKeySz, file),
                0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    ExpectIntEQ(wc_InitRsaKey_ex(&serverKey, NULL, INVALID_DEVID), 0);
    idx = 0;
    ExpectIntEQ(wc_RsaPrivateKeyDecode(serverKeyBuf, &idx, &serverKey,
                                       serverKeySz), 0);
    XMEMSET(caKeyBuf, 0, caKeySz);
    ExpectNotNull(file = fopen(caKeyFile, "rb"));
    ExpectIntGT(caKeySz = (word32)fread(caKeyBuf, 1, caKeySz, file), 0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    ExpectIntEQ(wc_InitRsaKey_ex(&caKey, NULL, INVALID_DEVID), 0);
    idx = 0;
    ExpectIntEQ(wc_RsaPrivateKeyDecode(caKeyBuf, &idx, &caKey, caKeySz), 0);
    XMEMSET(sapkiBuf, 0, sapkiSz);
    ExpectNotNull(file = fopen(sapkiFile, "rb"));
    ExpectIntGT(sapkiSz = (word32)fread(sapkiBuf, 1, sapkiSz, file), 0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    XMEMSET(altPrivBuf, 0, altPrivSz);
    ExpectNotNull(file = fopen(altPrivFile, "rb"));
    ExpectIntGT(altPrivSz = (word32)fread(altPrivBuf, 1, altPrivSz, file), 0);
    if (file) {
        fclose(file);
        file = NULL;
    }
    wc_ecc_init(&altCaKey);
    idx = 0;
    ExpectIntEQ(wc_EccPrivateKeyDecode(altPrivBuf, &idx, &altCaKey,
                                       (word32)altPrivSz), 0);
    XMEMSET(altSigAlgBuf, 0, altSigAlgSz);
    ExpectIntGT(altSigAlgSz = SetAlgoID(CTC_SHA256wECDSA, altSigAlgBuf,
                                         oidSigType, 0), 0);
    wc_InitCert(&newCert);
    strncpy(newCert.subject.country, "US", CTC_NAME_SIZE);
    strncpy(newCert.subject.state, "MT", CTC_NAME_SIZE);
    strncpy(newCert.subject.locality, "Bozeman", CTC_NAME_SIZE);
    strncpy(newCert.subject.org, "wolfSSL", CTC_NAME_SIZE);
    strncpy(newCert.subject.unit, "Engineering", CTC_NAME_SIZE);
    strncpy(newCert.subject.commonName, "www.wolfssl.com", CTC_NAME_SIZE);
    strncpy(newCert.subject.email, "server@wolfssl.com", CTC_NAME_SIZE);
    strncpy((char*)newCert.beforeDate, "\x18\x0f""20250101000000Z",
        CTC_DATE_SIZE);
    newCert.beforeDateSz = 17;
    strncpy((char*)newCert.afterDate, "\x18\x0f""20493112115959Z",
        CTC_DATE_SIZE);
    newCert.afterDateSz = 17;

    newCert.sigType = CTC_SHA256wRSA;
    newCert.isCA    = 0;
    ExpectIntEQ(wc_SetIssuerBuffer(&newCert, caCertBuf, caCertSz), 0);
    ExpectIntEQ(wc_SetCustomExtension(&newCert, setCrit, "2.5.29.72", sapkiBuf,
                                      sapkiSz), 0);
    ExpectIntEQ(wc_SetCustomExtension(&newCert, setCrit, "2.5.29.73",
                                      altSigAlgBuf, altSigAlgSz), 0);
    XMEMSET(scratchBuf, 0, scratchSz);
    ExpectIntGT(wc_MakeCert(&newCert, scratchBuf, scratchSz, &serverKey, NULL,
                &rng), 0);
    ExpectIntGT(scratchSz = wc_SignCert(newCert.bodySz, newCert.sigType,
                scratchBuf, scratchSz, &caKey, NULL, &rng), 0);
    wc_InitDecodedCert(&preTBS, scratchBuf, scratchSz, 0);
    ExpectIntEQ(wc_ParseCert(&preTBS, CERT_TYPE, NO_VERIFY, NULL), 0);
    XMEMSET(preTbsBuf, 0, preTbsSz);
    ExpectIntGT(preTbsSz = wc_GeneratePreTBS(&preTBS, preTbsBuf, preTbsSz), 0);
    XMEMSET(altSigValBuf, 0, altSigValSz);
    ExpectIntGT(altSigValSz = wc_MakeSigWithBitStr(altSigValBuf, altSigValSz,
                CTC_SHA256wECDSA, preTbsBuf, preTbsSz, ECC_TYPE, &altCaKey,
                &rng), 0);
    ExpectIntEQ(wc_SetCustomExtension(&newCert, setCrit, "2.5.29.74",
                                      altSigValBuf, altSigValSz), 0);

    /* Finally, generate the new certificate. */
    if (outBuf != NULL) {
        XMEMSET(outBuf, 0, outSz);
        ExpectIntGT(outSz = wc_SignCert(newCert.bodySz, newCert.sigType, scratchBuf,
                    outSz, &caKey, NULL, &rng), 0);
        *out = outBuf;
    }
    else {
        outSz = 0;
    }

    wc_FreeDecodedCert(&preTBS);
    wc_ecc_free(&altCaKey);
    wc_FreeRsaKey(&serverKey);
    wc_FreeRsaKey(&caKey);
    wc_FreeRng(&rng);

    return (int)outSz;
}

/**
 * Test dual-alg ECDSA + ML-DSA with critical extensions and path length
 * constraints:
 *  - keygen + certgen
 *
 * TLS tests not designed to pass with these extensions marked critical. No
 * TLS connection.
 * */
static int test_dual_alg_crit_ext_support(void)
{
    EXPECT_DECLS;
    /* Root CA and server keys will be the same. This is only appropriate for
     * testing. */
    char keyFile[] = "./certs/ca-key.der";
    char sapkiFile[] = "./certs/ecc-keyPub.der";
    char altPrivFile[] = "./certs/ecc-key.der";
    byte *serverKey = NULL;
    size_t serverKeySz = 0;
    byte *root = NULL;
    int rootSz = 0;
    byte *server = NULL;
    int serverSz = 0;

    ExpectIntEQ(load_file(keyFile, &serverKey, &serverKeySz), 0);

    /* Test with critical extensions and pathlen set to 1 */
    if (EXPECT_SUCCESS()) {
        rootSz = do_dual_alg_root_certgen_crit(&root, keyFile, sapkiFile,
            altPrivFile, 1, 1, 1);
    }
    ExpectNotNull(root);
    ExpectIntGT(rootSz, 0);
    if (EXPECT_SUCCESS()) {
        serverSz = do_dual_alg_server_certgen_crit(&server, keyFile, sapkiFile,
            altPrivFile, keyFile, root, rootSz, 1);
    }
    ExpectNotNull(server);
    ExpectIntGT(serverSz, 0);
    XFREE(root, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    root = NULL;
    XFREE(server, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    server = NULL;

    /* Test with critical extensions and pathlen set to 0 */
    if (EXPECT_SUCCESS()) {
        rootSz = do_dual_alg_root_certgen_crit(&root, keyFile, sapkiFile,
            altPrivFile, 1, 1, 0);
    }
    ExpectNotNull(root);
    ExpectIntGT(rootSz, 0);
    if (EXPECT_SUCCESS()) {
        serverSz = do_dual_alg_server_certgen_crit(&server, keyFile, sapkiFile,
            altPrivFile, keyFile, root, rootSz, 1);
    }
    ExpectNotNull(server);
    ExpectIntGT(serverSz, 0);
    XFREE(root, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    root = NULL;
    XFREE(server, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    server = NULL;

    /* Test with critical alt extensions and no pathlen set */
    if (EXPECT_SUCCESS()) {
        rootSz = do_dual_alg_root_certgen_crit(&root, keyFile, sapkiFile,
            altPrivFile, 1, 0, 0);
    }
    ExpectNotNull(root);
    ExpectIntGT(rootSz, 0);
    if (EXPECT_SUCCESS()) {
        serverSz = do_dual_alg_server_certgen_crit(&server, keyFile, sapkiFile,
            altPrivFile, keyFile, root, rootSz, 0);
    }
    ExpectNotNull(server);
    ExpectIntGT(serverSz, 0);
    XFREE(root, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(server, NULL, DYNAMIC_TYPE_TMP_BUFFER);

    free(serverKey);

    return EXPECT_RESULT();
}

static int test_dual_alg_support(void)
{
    EXPECT_DECLS;
    /* Root CA and server keys will be the same. This is only appropriate for
     * testing. */
    char keyFile[] = "./certs/ca-key.der";
    char sapkiFile[] = "./certs/ecc-keyPub.der";
    char altPrivFile[] = "./certs/ecc-key.der";
    char wrongPrivFile[] = "./certs/ecc-client-key.der";
    byte *serverKey = NULL;
    size_t serverKeySz = 0;
    byte *root = NULL;
    int rootSz = 0;
    byte *server = NULL;
    int serverSz = 0;

    ExpectIntEQ(load_file(keyFile, &serverKey, &serverKeySz), 0);

    /* Base normal case. */
    if (EXPECT_SUCCESS()) {
        rootSz = do_dual_alg_root_certgen(&root, keyFile, sapkiFile,
            altPrivFile);
    }
    ExpectNotNull(root);
    ExpectIntGT(rootSz, 0);
    if (EXPECT_SUCCESS()) {
        serverSz = do_dual_alg_server_certgen(&server, keyFile, sapkiFile,
            altPrivFile, keyFile, root, rootSz);
    }
    ExpectNotNull(server);
    ExpectIntGT(serverSz, 0);
    ExpectIntEQ(do_dual_alg_tls13_connection(root, rootSz,
                server, serverSz, serverKey, (word32)serverKeySz, 0),
                TEST_SUCCESS);
    XFREE(root, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    root = NULL;
    XFREE(server, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    server = NULL;

    /* Now we try a negative case. Note that we use wrongPrivFile to generate
     * the alternative signature and then set negative_test to true for the
     * call to do_dual_alg_tls13_connection(). Its expecting a failed connection
     * because the signature won't verify. The exception is if
     * WOLFSSL_TRUST_PEER_CERT is defined. In that case, no verification happens
     * and this is no longer a negative test. */
    if (EXPECT_SUCCESS()) {
        rootSz = do_dual_alg_root_certgen(&root, keyFile, sapkiFile,
            wrongPrivFile);
    }
    ExpectNotNull(root);
    ExpectIntGT(rootSz, 0);
    if (EXPECT_SUCCESS()) {
        serverSz = do_dual_alg_server_certgen(&server, keyFile, sapkiFile,
            wrongPrivFile, keyFile, root, rootSz);
    }
    ExpectNotNull(server);
    ExpectIntGT(serverSz, 0);
#ifdef WOLFSSL_TRUST_PEER_CERT
    ExpectIntEQ(do_dual_alg_tls13_connection(root, rootSz,
                server, serverSz, serverKey, (word32)serverKeySz, 0),
                TEST_SUCCESS);
#else
    ExpectIntEQ(do_dual_alg_tls13_connection(root, rootSz,
                server, serverSz, serverKey, (word32)serverKeySz, 1),
                TEST_SUCCESS);
#endif

    XFREE(root, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(server, NULL, DYNAMIC_TYPE_TMP_BUFFER);

    free(serverKey);

    return EXPECT_RESULT();
}
#else
static int test_dual_alg_support(void)
{
    return TEST_SKIPPED;
}

static int test_dual_alg_crit_ext_support(void)
{
    return TEST_SKIPPED;
}

#endif /* WOLFSSL_DUAL_ALG_CERTS && !NO_FILESYSTEM */

/**
 * Test dual-alg ECDSA + ML-DSA:
 *  - keygen + certgen + cert manager load
 * */
static int test_dual_alg_ecdsa_mldsa(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_DUAL_ALG_CERTS) && defined(HAVE_DILITHIUM) && \
    defined(HAVE_ECC) && !defined(WC_NO_RNG) && \
    defined(WOLFSSL_WC_DILITHIUM) && \
    !defined(WOLFSSL_DILITHIUM_NO_MAKE_KEY) && \
    !defined(WOLFSSL_DILITHIUM_NO_SIGN) && \
    !defined(WOLFSSL_DILITHIUM_NO_VERIFY) && !defined(WOLFSSL_SMALL_STACK)
    WOLFSSL_CERT_MANAGER * cm = NULL;
    MlDsaKey    alt_ca_key;
    ecc_key     ca_key;
    WC_RNG      rng;
    int         ret = 0;
    DecodedCert d_cert;
    Cert        new_cert;
    /* various tmp buffs.  */
    byte        alt_pub_der[LARGE_TEMP_SZ];
    word32      alt_pub_sz = LARGE_TEMP_SZ;
    byte        alt_sig_alg[LARGE_TEMP_SZ];
    word32      alt_sig_alg_sz = LARGE_TEMP_SZ;
    byte        tbs_der[LARGE_TEMP_SZ];
    word32      tbs_der_sz = LARGE_TEMP_SZ;
    byte        alt_sig[LARGE_TEMP_SZ];
    word32      alt_sig_sz = LARGE_TEMP_SZ;
    /* Intermediate der. */
    byte        der[LARGE_TEMP_SZ];
    word32      der_sz = LARGE_TEMP_SZ;
    /* The final der will be large because of ML-DSA signature. */
    byte        final_der[2 * LARGE_TEMP_SZ];
    word32      final_der_sz = 2 * LARGE_TEMP_SZ;

    XMEMSET(alt_pub_der, 0, alt_pub_sz);
    XMEMSET(alt_sig_alg, 0, alt_sig_alg_sz);
    XMEMSET(tbs_der, 0, tbs_der_sz);
    XMEMSET(alt_sig, 0, alt_sig_sz);
    XMEMSET(der, 0, der_sz);
    XMEMSET(final_der, 0, final_der_sz);

    ExpectIntEQ(wc_InitRng(&rng), 0);

    /**
     * ML-DSA key gen.
     * */
    ret = wc_MlDsaKey_Init(&alt_ca_key, NULL, INVALID_DEVID);
    ExpectIntEQ(ret, 0);
    ret = wc_MlDsaKey_SetParams(&alt_ca_key, WC_ML_DSA_44);
    ExpectIntEQ(ret, 0);
    ret = wc_MlDsaKey_MakeKey(&alt_ca_key, &rng);
    ExpectIntEQ(ret, 0);
    alt_pub_sz = wc_MlDsaKey_PublicKeyToDer(&alt_ca_key, alt_pub_der,
                                            alt_pub_sz, 1);
    ExpectIntGT(alt_pub_sz, 0);

    alt_sig_alg_sz = SetAlgoID(CTC_SHA256wECDSA, alt_sig_alg, oidSigType, 0);
    ExpectIntGT(alt_sig_alg_sz, 0);

    /**
     * ECC key gen.
     * */
    ret = wc_ecc_init(&ca_key);
    ExpectIntEQ(ret, 0);
    ret = wc_ecc_make_key(&rng, KEY32, &ca_key);
    ExpectIntEQ(ret, 0);

    /**
     * Cert gen.
     * */
    wc_InitCert(&new_cert);
    strncpy(new_cert.subject.country, "US", CTC_NAME_SIZE);
    strncpy(new_cert.subject.state, "MT", CTC_NAME_SIZE);
    strncpy(new_cert.subject.locality, "Bozeman", CTC_NAME_SIZE);
    strncpy(new_cert.subject.org, "wolfSSL", CTC_NAME_SIZE);
    strncpy(new_cert.subject.unit, "Engineering", CTC_NAME_SIZE);
    strncpy(new_cert.subject.commonName, "www.wolfssl.com", CTC_NAME_SIZE);
    strncpy(new_cert.subject.email, "root@wolfssl.com", CTC_NAME_SIZE);
    new_cert.sigType = CTC_SHA256wECDSA;
    new_cert.isCA    = 1;

    ret = wc_SetCustomExtension(&new_cert, 0, "1.2.3.4.5",
                         (const byte *)"This is NOT a critical extension", 32);
    ExpectIntEQ(ret, 0);

    ExpectIntEQ(wc_SetCustomExtension(&new_cert, 0, "2.5.29.72", alt_pub_der,
                alt_pub_sz), 0);
    ExpectIntEQ(wc_SetCustomExtension(&new_cert, 0, "2.5.29.73", alt_sig_alg,
                alt_sig_alg_sz), 0);

    ret = wc_MakeCert_ex(&new_cert, der, der_sz, ECC_TYPE, &ca_key, &rng);
    ExpectIntGT(ret, 0);

    der_sz = wc_SignCert_ex(new_cert.bodySz, new_cert.sigType, der, der_sz,
                            ECC_TYPE, &ca_key, &rng);
    ExpectIntGT(der_sz, 0);

    wc_InitDecodedCert(&d_cert, der, der_sz, 0);
    ret = wc_ParseCert(&d_cert, CERT_TYPE, NO_VERIFY, NULL);
    ExpectIntEQ(ret, 0);

    tbs_der_sz = wc_GeneratePreTBS(&d_cert, tbs_der, tbs_der_sz);
    ExpectIntGT(tbs_der_sz, 0);

    alt_sig_sz = wc_MakeSigWithBitStr(alt_sig, alt_sig_sz,
                                      CTC_ML_DSA_LEVEL2, tbs_der, tbs_der_sz,
                                      ML_DSA_LEVEL2_TYPE, &alt_ca_key, &rng);
    ExpectIntGT(alt_sig_sz, 0);

    ret = wc_SetCustomExtension(&new_cert, 0, "2.5.29.74", alt_sig, alt_sig_sz);
    ExpectIntEQ(ret, 0);

    /* Finally generate the new certificate. */
    ret = wc_MakeCert_ex(&new_cert, final_der, final_der_sz, ECC_TYPE, &ca_key,
                         &rng);
    ExpectIntGT(ret, 0);

    final_der_sz = wc_SignCert_ex(new_cert.bodySz, new_cert.sigType, final_der,
                                  final_der_sz, ECC_TYPE, &ca_key, &rng);
    ExpectIntGT(final_der_sz, 0);

    cm = wolfSSL_CertManagerNew();
    ExpectNotNull(cm);

    /* Load the certificate into CertManager. */
    if (cm != NULL && final_der_sz > 0) {
        ret = wolfSSL_CertManagerLoadCABuffer(cm, final_der, final_der_sz,
                                              WOLFSSL_FILETYPE_ASN1);
        ExpectIntEQ(ret, WOLFSSL_SUCCESS);
    }

    if (cm != NULL) {
        wolfSSL_CertManagerFree(cm);
        cm = NULL;
    }

    wc_FreeDecodedCert(&d_cert);
    wc_ecc_free(&ca_key);
    wc_MlDsaKey_Free(&alt_ca_key);
    wc_FreeRng(&rng);

#endif /* WOLFSSL_DUAL_ALG_CERTS && DILITHIUM and more */
    return EXPECT_RESULT();
}


/*----------------------------------------------------------------------------*
 | Context
 *----------------------------------------------------------------------------*/
#if !defined(NO_WOLFSSL_SERVER) && !defined(NO_TLS)
static int test_wolfSSL_CTX_new(void)
{
    EXPECT_DECLS;
    WOLFSSL_CTX *ctx;
    WOLFSSL_METHOD* method = NULL;

    ExpectNull(ctx = wolfSSL_CTX_new(NULL));
    ExpectNotNull(method = wolfSSLv23_server_method());
    if (method != NULL)
        ExpectNotNull(ctx = wolfSSL_CTX_new(method));

    wolfSSL_CTX_free(ctx);

    return EXPECT_RESULT();
}
#endif

#if (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)) && \
    !defined(NO_TLS) && \
    (!defined(NO_RSA) || defined(HAVE_ECC)) && !defined(NO_FILESYSTEM)
static int test_for_double_Free(void)
{
    EXPECT_DECLS;
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL*     ssl = NULL;
    int skipTest = 0;
    const char* testCertFile;
    const char* testKeyFile;
    char optionsCiphers[] = "RC4-SHA:RC4-MD5:DES-CBC3-SHA:AES128-SHA:AES256-SHA"
":NULL-SHA:NULL-SHA256:DHE-RSA-AES128-SHA:DHE-RSA-AES256-SHA:DHE-PSK-AES256-GCM"
"-SHA384:DHE-PSK-AES128-GCM-SHA256:PSK-AES256-GCM-SHA384:PSK-AES128-GCM-SHA256:"
"DHE-PSK-AES256-CBC-SHA384:DHE-PSK-AES128-CBC-SHA256:PSK-AES256-CBC-SHA384:PSK-"
"AES128-CBC-SHA256:PSK-AES128-CBC-SHA:PSK-AES256-CBC-SHA:DHE-PSK-AES128-CCM:DHE"
"-PSK-AES256-CCM:PSK-AES128-CCM:PSK-AES256-CCM:PSK-AES128-CCM-8:PSK-AES256-CCM-"
"8:DHE-PSK-NULL-SHA384:DHE-PSK-NULL-SHA256:PSK-NULL-SHA384:PSK-NULL-SHA256:PSK-"
"NULL-SHA:AES128-CCM-8:AES256-CCM-8:ECDHE-ECDSA-"
"AES128-CCM:ECDHE-ECDSA-AES128-CCM-8:ECDHE-ECDSA-AES256-CCM-8:ECDHE-RSA-AES128-"
"SHA:ECDHE-RSA-AES256-SHA:ECDHE-ECDSA-AES128-SHA:ECDHE-ECDSA-AES256-SHA:ECDHE-R"
"SA-RC4-SHA:ECDHE-RSA-DES-CBC3-SHA:ECDHE-ECDSA-RC4-SHA:ECDHE-ECDSA-DES-CBC3-SHA"
":AES128-SHA256:AES256-SHA256:DHE-RSA-AES128-SHA256:DHE-RSA-AES256-SHA256:ECDH-"
"RSA-AES128-SHA:ECDH-RSA-AES256-SHA:ECDH-ECDSA-AES128-SHA:ECDH-ECDSA-AES256-SHA"
":ECDH-RSA-RC4-SHA:ECDH-RSA-DES-CBC3-SHA:ECDH-ECDSA-RC4-SHA:ECDH-ECDSA-DES-CBC3"
"-SHA:AES128-GCM-SHA256:AES256-GCM-SHA384:DHE-RSA-AES128-GCM-SHA256:DHE-RSA-AES"
"256-GCM-SHA384:ECDHE-RSA-AES128-GCM-SHA256:ECDHE-RSA-AES256-GCM-SHA384:ECDHE-E"
"CDSA-AES128-GCM-SHA256:ECDHE-ECDSA-AES256-GCM-SHA384:ECDH-RSA-AES128-GCM-SHA25"
"6:ECDH-RSA-AES256-GCM-SHA384:ECDH-ECDSA-AES128-GCM-SHA256:ECDH-ECDSA-AES256-GC"
"M-SHA384:CAMELLIA128-SHA:DHE-RSA-CAMELLIA128-SHA:CAMELLIA256-SHA:DHE-RSA-CAMEL"
"LIA256-SHA:CAMELLIA128-SHA256:DHE-RSA-CAMELLIA128-SHA256:CAMELLIA256-SHA256:DH"
"E-RSA-CAMELLIA256-SHA256:ECDHE-RSA-AES128-SHA256:ECDHE-ECDSA-AES128-SHA256:ECD"
"H-RSA-AES128-SHA256:ECDH-ECDSA-AES128-SHA256:ECDHE-RSA-AES256-SHA384:ECDHE-ECD"
"SA-AES256-SHA384:ECDH-RSA-AES256-SHA384:ECDH-ECDSA-AES256-SHA384:ECDHE-RSA-CHA"
"CHA20-POLY1305:ECDHE-ECDSA-CHACHA20-POLY1305:DHE-RSA-CHACHA20-POLY1305:ECDHE-R"
"SA-CHACHA20-POLY1305-OLD:ECDHE-ECDSA-CHACHA20-POLY1305-OLD:DHE-RSA-CHACHA20-PO"
"LY1305-OLD:ECDHE-ECDSA-NULL-SHA:ECDHE-PSK-NULL-SHA256:ECDHE-PSK-A"
"ES128-CBC-SHA256:PSK-CHACHA20-POLY1305:ECDHE-PSK-CHACHA20-POLY1305:DHE-PSK-CHA"
"CHA20-POLY1305:EDH-RSA-DES-CBC3-SHA:TLS13-AES128-GCM-SHA256:TLS13-AES256-GCM-S"
"HA384:TLS13-CHACHA20-POLY1305-SHA256:TLS13-AES128-CCM-SHA256:TLS13-AES128-CCM-"
"8-SHA256:TLS13-SHA256-SHA256:TLS13-SHA384-SHA384";
    /* OpenVPN uses a "blacklist" method to specify which ciphers NOT to use */
#ifdef OPENSSL_EXTRA
    char openvpnCiphers[] = "DEFAULT:!EXP:!LOW:!MEDIUM:!kDH:!kECDH:!DSS:!PSK:"
                            "!SRP:!kRSA:!aNULL:!eNULL";
#endif

#ifndef NO_RSA
    testCertFile = svrCertFile;
    testKeyFile = svrKeyFile;
#elif defined(HAVE_ECC)
    testCertFile = eccCertFile;
    testKeyFile = eccKeyFile;
#else
    skipTest = 1;
#endif

    if (skipTest != 1) {
#ifndef NO_WOLFSSL_SERVER
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#endif
        ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, testCertFile,
            WOLFSSL_FILETYPE_PEM));
        ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, testKeyFile,
            WOLFSSL_FILETYPE_PEM));
        ExpectNotNull(ssl = wolfSSL_new(ctx));

        /* First test freeing SSL, then CTX */
        wolfSSL_free(ssl);
        ssl = NULL;
        wolfSSL_CTX_free(ctx);
        ctx = NULL;

#ifndef NO_WOLFSSL_CLIENT
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#endif
        ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, testCertFile,
            WOLFSSL_FILETYPE_PEM));
        ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, testKeyFile,
            WOLFSSL_FILETYPE_PEM));
        ExpectNotNull(ssl = wolfSSL_new(ctx));

        /* Next test freeing CTX then SSL */
        wolfSSL_CTX_free(ctx);
        ctx = NULL;
        wolfSSL_free(ssl);
        ssl = NULL;

#ifndef NO_WOLFSSL_SERVER
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#endif
        /* Test setting ciphers at ctx level */
        ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, testCertFile,
            WOLFSSL_FILETYPE_PEM));
        ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, testKeyFile,
            WOLFSSL_FILETYPE_PEM));
        ExpectTrue(wolfSSL_CTX_set_cipher_list(ctx, optionsCiphers));
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_TLS13) && defined(HAVE_AESGCM) && \
        defined(WOLFSSL_SHA384) && defined(WOLFSSL_AES_256)
        /* only update TLSv13 suites */
        ExpectTrue(wolfSSL_CTX_set_cipher_list(ctx, "TLS13-AES256-GCM-SHA384"));
#endif
#if defined(OPENSSL_EXTRA) && defined(HAVE_ECC) && defined(HAVE_AESGCM) && \
    !defined(NO_SHA256) && !defined(WOLFSSL_NO_TLS12) && \
    defined(WOLFSSL_AES_128) && !defined(NO_RSA)
        /* only update pre-TLSv13 suites */
        ExpectTrue(wolfSSL_CTX_set_cipher_list(ctx,
            "ECDHE-RSA-AES128-GCM-SHA256"));
#endif
#ifdef OPENSSL_EXTRA
        ExpectTrue(wolfSSL_CTX_set_cipher_list(ctx, openvpnCiphers));
#endif
        ExpectNotNull(ssl = wolfSSL_new(ctx));
        wolfSSL_CTX_free(ctx);
        ctx = NULL;
        wolfSSL_free(ssl);
        ssl = NULL;

#ifndef NO_WOLFSSL_CLIENT
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#endif
        ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, testCertFile,
            WOLFSSL_FILETYPE_PEM));
        ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, testKeyFile,
            WOLFSSL_FILETYPE_PEM));
        ExpectNotNull(ssl = wolfSSL_new(ctx));
        /* test setting ciphers at SSL level */
        ExpectTrue(wolfSSL_set_cipher_list(ssl, optionsCiphers));
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_TLS13) && defined(HAVE_AESGCM) && \
        defined(WOLFSSL_SHA384) && defined(WOLFSSL_AES_256)
        /* only update TLSv13 suites */
        ExpectTrue(wolfSSL_set_cipher_list(ssl, "TLS13-AES256-GCM-SHA384"));
#endif
#if defined(OPENSSL_EXTRA) && defined(HAVE_ECC) && defined(HAVE_AESGCM) && \
    !defined(NO_SHA256) && !defined(WOLFSSL_NO_TLS12) && \
    defined(WOLFSSL_AES_128) && !defined(NO_RSA)
        /* only update pre-TLSv13 suites */
        ExpectTrue(wolfSSL_set_cipher_list(ssl, "ECDHE-RSA-AES128-GCM-SHA256"));
#endif
        wolfSSL_CTX_free(ctx);
        ctx = NULL;
        wolfSSL_free(ssl);
        ssl = NULL;
    }

    return EXPECT_RESULT();
}
#endif


static int test_wolfSSL_CTX_set_cipher_list_bytes(void)
{
    EXPECT_DECLS;
#if (defined(OPENSSL_EXTRA) || defined(WOLFSSL_SET_CIPHER_BYTES)) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)) && \
    (!defined(NO_RSA) || defined(HAVE_ECC)) && !defined(NO_FILESYSTEM) && \
    !defined(NO_TLS)
    const char* testCertFile;
    const char* testKeyFile;
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL* ssl = NULL;

    const byte cipherList[] =
    {
        /* TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA */ 0x00, 0x16,
        /* TLS_DHE_RSA_WITH_AES_256_CBC_SHA  */ 0x00, 0x39,
        /* TLS_DHE_RSA_WITH_AES_128_CBC_SHA  */ 0x00, 0x33,
        /* TLS_DH_anon_WITH_AES_128_CBC_SHA  */ 0x00, 0x34,
        /* TLS_RSA_WITH_AES_256_CBC_SHA      */ 0x00, 0x35,
        /* TLS_RSA_WITH_AES_128_CBC_SHA      */ 0x00, 0x2F,
        /* TLS_RSA_WITH_NULL_MD5             */ 0x00, 0x01,
        /* TLS_RSA_WITH_NULL_SHA             */ 0x00, 0x02,
        /* TLS_PSK_WITH_AES_256_CBC_SHA      */ 0x00, 0x8d,
        /* TLS_PSK_WITH_AES_128_CBC_SHA256   */ 0x00, 0xae,
        /* TLS_PSK_WITH_AES_256_CBC_SHA384   */ 0x00, 0xaf,
        /* TLS_PSK_WITH_AES_128_CBC_SHA      */ 0x00, 0x8c,
        /* TLS_PSK_WITH_NULL_SHA256          */ 0x00, 0xb0,
        /* TLS_PSK_WITH_NULL_SHA384          */ 0x00, 0xb1,
        /* TLS_PSK_WITH_NULL_SHA             */ 0x00, 0x2c,
        /* SSL_RSA_WITH_RC4_128_SHA          */ 0x00, 0x05,
        /* SSL_RSA_WITH_RC4_128_MD5          */ 0x00, 0x04,
        /* SSL_RSA_WITH_3DES_EDE_CBC_SHA     */ 0x00, 0x0A,

        /* ECC suites, first byte is 0xC0 (ECC_BYTE) */
        /* TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA     */ 0xC0, 0x14,
        /* TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA     */ 0xC0, 0x13,
        /* TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA   */ 0xC0, 0x0A,
        /* TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA   */ 0xC0, 0x09,
        /* TLS_ECDHE_RSA_WITH_RC4_128_SHA         */ 0xC0, 0x11,
        /* TLS_ECDHE_ECDSA_WITH_RC4_128_SHA       */ 0xC0, 0x07,
        /* TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA    */ 0xC0, 0x12,
        /* TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA  */ 0xC0, 0x08,
        /* TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256  */ 0xC0, 0x27,
        /* TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256*/ 0xC0, 0x23,
        /* TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384  */ 0xC0, 0x28,
        /* TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384*/ 0xC0, 0x24,
        /* TLS_ECDHE_ECDSA_WITH_NULL_SHA          */ 0xC0, 0x06,
        /* TLS_ECDHE_PSK_WITH_NULL_SHA256         */ 0xC0, 0x3a,
        /* TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA256  */ 0xC0, 0x37,

        /* static ECDH, first byte is 0xC0 (ECC_BYTE) */
        /* TLS_ECDH_RSA_WITH_AES_256_CBC_SHA      */ 0xC0, 0x0F,
        /* TLS_ECDH_RSA_WITH_AES_128_CBC_SHA      */ 0xC0, 0x0E,
        /* TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA    */ 0xC0, 0x05,
        /* TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA    */ 0xC0, 0x04,
        /* TLS_ECDH_RSA_WITH_RC4_128_SHA          */ 0xC0, 0x0C,
        /* TLS_ECDH_ECDSA_WITH_RC4_128_SHA        */ 0xC0, 0x02,
        /* TLS_ECDH_RSA_WITH_3DES_EDE_CBC_SHA     */ 0xC0, 0x0D,
        /* TLS_ECDH_ECDSA_WITH_3DES_EDE_CBC_SHA   */ 0xC0, 0x03,
        /* TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256   */ 0xC0, 0x29,
        /* TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA256 */ 0xC0, 0x25,
        /* TLS_ECDH_RSA_WITH_AES_256_CBC_SHA384   */ 0xC0, 0x2A,
        /* TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA384 */ 0xC0, 0x26,

        /* WDM_WITH_NULL_SHA256 */ 0x00, 0xFE, /* wolfSSL DTLS Multicast */

        /* SHA256 */
        /* TLS_DHE_RSA_WITH_AES_256_CBC_SHA256 */ 0x00, 0x6b,
        /* TLS_DHE_RSA_WITH_AES_128_CBC_SHA256 */ 0x00, 0x67,
        /* TLS_RSA_WITH_AES_256_CBC_SHA256     */ 0x00, 0x3d,
        /* TLS_RSA_WITH_AES_128_CBC_SHA256     */ 0x00, 0x3c,
        /* TLS_RSA_WITH_NULL_SHA256            */ 0x00, 0x3b,
        /* TLS_DHE_PSK_WITH_AES_128_CBC_SHA256 */ 0x00, 0xb2,
        /* TLS_DHE_PSK_WITH_NULL_SHA256        */ 0x00, 0xb4,

        /* SHA384 */
        /* TLS_DHE_PSK_WITH_AES_256_CBC_SHA384 */ 0x00, 0xb3,
        /* TLS_DHE_PSK_WITH_NULL_SHA384        */ 0x00, 0xb5,

        /* AES-GCM */
        /* TLS_RSA_WITH_AES_128_GCM_SHA256      */ 0x00, 0x9c,
        /* TLS_RSA_WITH_AES_256_GCM_SHA384      */ 0x00, 0x9d,
        /* TLS_DHE_RSA_WITH_AES_128_GCM_SHA256  */ 0x00, 0x9e,
        /* TLS_DHE_RSA_WITH_AES_256_GCM_SHA384  */ 0x00, 0x9f,
        /* TLS_DH_anon_WITH_AES_256_GCM_SHA384  */ 0x00, 0xa7,
        /* TLS_PSK_WITH_AES_128_GCM_SHA256      */ 0x00, 0xa8,
        /* TLS_PSK_WITH_AES_256_GCM_SHA384      */ 0x00, 0xa9,
        /* TLS_DHE_PSK_WITH_AES_128_GCM_SHA256  */ 0x00, 0xaa,
        /* TLS_DHE_PSK_WITH_AES_256_GCM_SHA384  */ 0x00, 0xab,

        /* ECC AES-GCM, first byte is 0xC0 (ECC_BYTE) */
        /* TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256 */ 0xC0, 0x2b,
        /* TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384 */ 0xC0, 0x2c,
        /* TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256  */ 0xC0, 0x2d,
        /* TLS_ECDH_ECDSA_WITH_AES_256_GCM_SHA384  */ 0xC0, 0x2e,
        /* TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256   */ 0xC0, 0x2f,
        /* TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384   */ 0xC0, 0x30,
        /* TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256    */ 0xC0, 0x31,
        /* TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384    */ 0xC0, 0x32,

        /* AES-CCM, first byte is 0xC0 but isn't ECC,
         * also, in some of the other AES-CCM suites
         * there will be second byte number conflicts
         * with non-ECC AES-GCM */
        /* TLS_RSA_WITH_AES_128_CCM_8         */ 0xC0, 0xa0,
        /* TLS_RSA_WITH_AES_256_CCM_8         */ 0xC0, 0xa1,
        /* TLS_ECDHE_ECDSA_WITH_AES_128_CCM   */ 0xC0, 0xac,
        /* TLS_ECDHE_ECDSA_WITH_AES_128_CCM_8 */ 0xC0, 0xae,
        /* TLS_ECDHE_ECDSA_WITH_AES_256_CCM_8 */ 0xC0, 0xaf,
        /* TLS_PSK_WITH_AES_128_CCM           */ 0xC0, 0xa4,
        /* TLS_PSK_WITH_AES_256_CCM           */ 0xC0, 0xa5,
        /* TLS_PSK_WITH_AES_128_CCM_8         */ 0xC0, 0xa8,
        /* TLS_PSK_WITH_AES_256_CCM_8         */ 0xC0, 0xa9,
        /* TLS_DHE_PSK_WITH_AES_128_CCM       */ 0xC0, 0xa6,
        /* TLS_DHE_PSK_WITH_AES_256_CCM       */ 0xC0, 0xa7,

        /* Camellia */
        /* TLS_RSA_WITH_CAMELLIA_128_CBC_SHA        */ 0x00, 0x41,
        /* TLS_RSA_WITH_CAMELLIA_256_CBC_SHA        */ 0x00, 0x84,
        /* TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256     */ 0x00, 0xba,
        /* TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256     */ 0x00, 0xc0,
        /* TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA    */ 0x00, 0x45,
        /* TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA    */ 0x00, 0x88,
        /* TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256 */ 0x00, 0xbe,
        /* TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256 */ 0x00, 0xc4,

        /* chacha20-poly1305 suites first byte is 0xCC (CHACHA_BYTE) */
        /* TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256   */ 0xCC, 0xa8,
        /* TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256 */ 0xCC, 0xa9,
        /* TLS_DHE_RSA_WITH_CHACHA20_POLY1305_SHA256     */ 0xCC, 0xaa,
        /* TLS_ECDHE_PSK_WITH_CHACHA20_POLY1305_SHA256   */ 0xCC, 0xac,
        /* TLS_PSK_WITH_CHACHA20_POLY1305_SHA256         */ 0xCC, 0xab,
        /* TLS_DHE_PSK_WITH_CHACHA20_POLY1305_SHA256     */ 0xCC, 0xad,

        /* chacha20-poly1305 earlier version of nonce and padding (CHACHA_BYTE) */
        /* TLS_ECDHE_RSA_WITH_CHACHA20_OLD_POLY1305_SHA256   */ 0xCC, 0x13,
        /* TLS_ECDHE_ECDSA_WITH_CHACHA20_OLD_POLY1305_SHA256 */ 0xCC, 0x14,
        /* TLS_DHE_RSA_WITH_CHACHA20_OLD_POLY1305_SHA256     */ 0xCC, 0x15,

        /* ECDHE_PSK RFC8442, first byte is 0xD0 (ECDHE_PSK_BYTE) */
        /* TLS_ECDHE_PSK_WITH_AES_128_GCM_SHA256 */ 0xD0, 0x01,

        /* TLS v1.3 cipher suites */
        /* TLS_AES_128_GCM_SHA256       */ 0x13, 0x01,
        /* TLS_AES_256_GCM_SHA384       */ 0x13, 0x02,
        /* TLS_CHACHA20_POLY1305_SHA256 */ 0x13, 0x03,
        /* TLS_AES_128_CCM_SHA256       */ 0x13, 0x04,
        /* TLS_AES_128_CCM_8_SHA256     */ 0x13, 0x05,

        /* TLS v1.3 Integrity only cipher suites - 0xC0 (ECC) first byte */
        /* TLS_SHA256_SHA256 */ 0xC0, 0xB4,
        /* TLS_SHA384_SHA384 */ 0xC0, 0xB5
    };

#ifndef NO_RSA
    testCertFile = svrCertFile;
    testKeyFile = svrKeyFile;
#elif defined(HAVE_ECC)
    testCertFile = eccCertFile;
    testKeyFile = eccKeyFile;
#endif

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#endif

    ExpectTrue(wolfSSL_CTX_set_cipher_list_bytes(ctx, &cipherList[0U],
                                                           sizeof(cipherList)));

    wolfSSL_CTX_free(ctx);
    ctx = NULL;

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#endif

    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, testCertFile,
                                                         WOLFSSL_FILETYPE_PEM));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, testKeyFile,
                                                         WOLFSSL_FILETYPE_PEM));

    ExpectNotNull(ssl = wolfSSL_new(ctx));

    ExpectTrue(wolfSSL_set_cipher_list_bytes(ssl, &cipherList[0U],
                                                           sizeof(cipherList)));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif /* (OPENSSL_EXTRA || WOLFSSL_SET_CIPHER_BYTES) &&
    (!NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER) && (!NO_RSA || HAVE_ECC) */

    return EXPECT_RESULT();
}


static int test_wolfSSL_CTX_use_certificate(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) || defined(HAVE_LIGHTY) || \
    defined(WOLFSSL_MYSQL_COMPATIBLE) || defined(HAVE_STUNNEL) || \
    defined(WOLFSSL_NGINX) || defined(HAVE_POCO_LIB) || \
    defined(WOLFSSL_HAPROXY)
#if !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX* ctx = NULL;
    X509* x509 = NULL;

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#endif

    ExpectNotNull(x509 = wolfSSL_X509_new());

    /* Negative tests. */
    ExpectIntEQ(SSL_CTX_use_certificate(NULL, NULL), 0);
    ExpectIntEQ(SSL_CTX_use_certificate(ctx, NULL), 0);
    ExpectIntEQ(SSL_CTX_use_certificate(NULL, x509), 0);
    /* Empty certificate */
    ExpectIntEQ(SSL_CTX_use_certificate(ctx, x509), 0);

    wolfSSL_X509_free(x509);
    wolfSSL_CTX_free(ctx);
#endif /* !NO_TLS && (!NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER) */
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_use_certificate_file(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_SERVER)
    WOLFSSL_CTX *ctx = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    /* invalid context */
    ExpectFalse(wolfSSL_CTX_use_certificate_file(NULL, svrCertFile,
                                                         WOLFSSL_FILETYPE_PEM));
    /* invalid cert file */
    ExpectFalse(wolfSSL_CTX_use_certificate_file(ctx, bogusFile,
                                                         WOLFSSL_FILETYPE_PEM));
    /* invalid cert type */
    ExpectFalse(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile, 9999));

#ifdef NO_RSA
    /* rsa needed */
    ExpectFalse(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile,
                                                         WOLFSSL_FILETYPE_PEM));
#else
    /* success */
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile,
                                                         WOLFSSL_FILETYPE_PEM));
#endif

    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

#if (defined(OPENSSL_ALL) || defined(WOLFSSL_ASIO)) && !defined(NO_RSA)
static int test_wolfSSL_CTX_use_certificate_ASN1(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_WOLFSSL_SERVER) && !defined(NO_ASN) && \
    !defined(NO_TLS)
    WOLFSSL_CTX* ctx = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    /* Failure cases. */
    ExpectIntEQ(SSL_CTX_use_certificate_ASN1(NULL, 0, NULL                ),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_CTX_use_certificate_ASN1(ctx , 0, NULL                ),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_CTX_use_certificate_ASN1(NULL, 0, server_cert_der_2048),
       WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_CTX_use_certificate_ASN1(ctx , 0, server_cert_der_2048),
       WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    ExpectIntEQ(SSL_CTX_use_certificate_ASN1(ctx, sizeof_server_cert_der_2048,
        server_cert_der_2048), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}
#endif /* (OPENSSL_ALL || WOLFSSL_ASIO) && !NO_RSA */

/*  Test function for wolfSSL_CTX_use_certificate_buffer. Load cert into
 *  context using buffer.
 *  PRE: NO_CERTS not defined; USE_CERT_BUFFERS_2048 defined; compile with
 *  --enable-testcert flag.
 */
static int test_wolfSSL_CTX_use_certificate_buffer(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_TLS) && !defined(NO_WOLFSSL_SERVER) && \
    defined(USE_CERT_BUFFERS_2048) && !defined(NO_RSA)
    WOLFSSL_CTX* ctx = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    /* Invalid parameters. */
    ExpectIntEQ(wolfSSL_CTX_use_certificate_buffer(NULL, NULL, 0,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_use_certificate_buffer(ctx, NULL, 0,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_PARSE_E));
    ExpectIntEQ(wolfSSL_CTX_use_certificate_buffer(NULL, server_cert_der_2048,
        0, WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_use_certificate_buffer(ctx, server_cert_der_2048, 0,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_PARSE_E));

    ExpectIntEQ(wolfSSL_CTX_use_certificate_buffer(ctx,
        server_cert_der_2048, sizeof_server_cert_der_2048,
        WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();

} /* END test_wolfSSL_CTX_use_certificate_buffer */

static int test_wolfSSL_use_certificate_buffer(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_TLS) && !defined(NO_WOLFSSL_CLIENT) && \
    defined(USE_CERT_BUFFERS_2048) && !defined(NO_RSA)
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL*     ssl = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    /* Invalid parameters. */
    ExpectIntEQ(wolfSSL_use_certificate_buffer(NULL, NULL, 0,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_use_certificate_buffer(ssl, NULL, 0,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_PARSE_E));
    ExpectIntEQ(wolfSSL_use_certificate_buffer(NULL, client_cert_der_2048, 0,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_use_certificate_buffer(ssl, client_cert_der_2048, 0,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_PARSE_E));

    ExpectIntEQ(wolfSSL_use_certificate_buffer(ssl,
        client_cert_der_2048, sizeof_client_cert_der_2048,
        WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_use_PrivateKey_file(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_SERVER)
    WOLFSSL_CTX *ctx = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    /* invalid context */
    ExpectFalse(wolfSSL_CTX_use_PrivateKey_file(NULL, svrKeyFile,
                                                         WOLFSSL_FILETYPE_PEM));
    /* invalid key file */
    ExpectFalse(wolfSSL_CTX_use_PrivateKey_file(ctx, bogusFile,
                                                         WOLFSSL_FILETYPE_PEM));
    /* invalid key type */
    ExpectFalse(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, 9999));

    /* invalid key format */
    ExpectFalse(wolfSSL_CTX_use_PrivateKey_file(ctx, "./certs/dh-priv-2048.pem",
                                                         WOLFSSL_FILETYPE_PEM));

    /* success */
#ifdef NO_RSA
    /* rsa needed */
    ExpectFalse(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
                                                         WOLFSSL_FILETYPE_PEM));
#else
    /* success */
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
                                                         WOLFSSL_FILETYPE_PEM));
#endif

    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_use_RSAPrivateKey_file(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_SERVER) && defined(OPENSSL_EXTRA)
    WOLFSSL_CTX *ctx = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    /* invalid context */
    ExpectIntEQ(wolfSSL_CTX_use_RSAPrivateKey_file(NULL, svrKeyFile,
        WOLFSSL_FILETYPE_PEM), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    /* invalid key file */
    ExpectIntEQ(wolfSSL_CTX_use_RSAPrivateKey_file(ctx, bogusFile,
        WOLFSSL_FILETYPE_PEM), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    /* invalid key type */
    ExpectIntEQ(wolfSSL_CTX_use_RSAPrivateKey_file(ctx, svrKeyFile, 9999),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    /* success */
#ifdef NO_RSA
    /* rsa needed */
    ExpectIntEQ(wolfSSL_CTX_use_RSAPrivateKey_file(ctx, svrKeyFile,
        WOLFSSL_FILETYPE_PEM), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
#else
    /* success */
    ExpectIntEQ(wolfSSL_CTX_use_RSAPrivateKey_file(ctx, svrKeyFile,
        WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
#endif

    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_use_RSAPrivateKey_file(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_CLIENT) && defined(OPENSSL_EXTRA)
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL*     ssl = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    ExpectNotNull(ssl = SSL_new(ctx));

    /* invalid context */
    ExpectIntEQ(wolfSSL_use_RSAPrivateKey_file(NULL, svrKeyFile,
        WOLFSSL_FILETYPE_PEM), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    /* invalid key file */
    ExpectIntEQ(wolfSSL_use_RSAPrivateKey_file(ssl, bogusFile,
        WOLFSSL_FILETYPE_PEM), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    /* invalid key type */
    ExpectIntEQ(wolfSSL_use_RSAPrivateKey_file(ssl, svrKeyFile, 9999),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    /* success */
#ifdef NO_RSA
    /* rsa needed */
    ExpectIntEQ(wolfSSL_use_RSAPrivateKey_file(ssl, svrKeyFile,
        WOLFSSL_FILETYPE_PEM), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
#else
    /* success */
    ExpectIntEQ(wolfSSL_use_RSAPrivateKey_file(ssl, svrKeyFile,
        WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
#endif

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_use_PrivateKey(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_SERVER) && defined(OPENSSL_EXTRA)
    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL_EVP_PKEY* pkey = NULL;
    const unsigned char* p;

    (void)p;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    ExpectNotNull(pkey = wolfSSL_EVP_PKEY_new());
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey(NULL, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey(ctx, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey(NULL, pkey), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    /* No data. */
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey(ctx, pkey), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;

#if defined(USE_CERT_BUFFERS_2048)
#if !defined(NO_RSA)
    p = client_key_der_2048;
    ExpectNotNull(pkey = d2i_PrivateKey(EVP_PKEY_RSA, NULL, &p,
        sizeof_client_key_der_2048));
#if defined(WOLFSSL_KEY_GEN)
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey(ctx, pkey), WOLFSSL_SUCCESS);
#else
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey(ctx, pkey), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
#endif
    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;
#endif
#if defined(WOLFSSL_QT) || defined(OPENSSL_ALL) || defined(WOLFSSL_OPENSSH)
#ifndef NO_DSA
    p = dsa_key_der_2048;
    ExpectNotNull(pkey = d2i_PrivateKey(EVP_PKEY_DSA, NULL, &p,
        sizeof_dsa_key_der_2048));
#if !defined(HAVE_SELFTEST) && (defined(WOLFSSL_KEY_GEN) || \
    defined(WOLFSSL_CERT_GEN))
    /* Not supported in ProcessBuffer. */
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey(ctx, pkey), WC_NO_ERR_TRACE(WOLFSSL_BAD_FILE));
#else
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey(ctx, pkey), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
#endif
    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;
#endif
#endif /* WOLFSSL_QT || OPENSSL_ALL || WOLFSSL_OPENSSH */
#if !defined(NO_DH) && defined(OPENSSL_ALL) && \
    (!defined(HAVE_FIPS) || FIPS_VERSION_GT(2,0))
    p = dh_ffdhe_statickey_der_2048;
    ExpectNotNull(pkey = d2i_PrivateKey(EVP_PKEY_DH, NULL, &p,
        sizeof_dh_ffdhe_statickey_der_2048));
    /* Not supported. */
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey(ctx, pkey), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;
#endif
#endif /* USE_CERT_BUFFERS_2048 */
#if defined(HAVE_ECC) && defined(USE_CERT_BUFFERS_256)
    p = ecc_clikey_der_256;
    ExpectNotNull(pkey = d2i_PrivateKey(EVP_PKEY_EC, NULL, &p,
        sizeof_ecc_clikey_der_256));
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey(ctx, pkey), WOLFSSL_SUCCESS);
    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;
#endif
    ExpectNotNull(pkey = wolfSSL_EVP_PKEY_new_mac_key(EVP_PKEY_HMAC, NULL,
        (unsigned char*)"01234567012345670123456701234567", 32));
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey(ctx, pkey), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;

    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

/* test both file and buffer versions along with unloading trusted peer certs */
static int test_wolfSSL_CTX_trust_peer_cert(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && defined(WOLFSSL_TRUST_PEER_CERT) && \
    !defined(NO_TLS) && !defined(NO_WOLFSSL_CLIENT) && !defined(NO_RSA)
    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL* ssl = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    ExpectNotNull(ssl = wolfSSL_new(ctx));

#if !defined(NO_FILESYSTEM)
    /* invalid file */
    ExpectIntNE(wolfSSL_CTX_trust_peer_cert(ctx, NULL,
                                     WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    ExpectIntNE(wolfSSL_CTX_trust_peer_cert(ctx, bogusFile,
                                     WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    ExpectIntNE(wolfSSL_CTX_trust_peer_cert(ctx, cliCertFile,
                                     WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* success */
    ExpectIntEQ(wolfSSL_CTX_trust_peer_cert(ctx, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    /* unload cert */
    ExpectIntNE(wolfSSL_CTX_Unload_trust_peers(NULL), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CTX_Unload_trust_peers(ctx), WOLFSSL_SUCCESS);

    /* invalid file */
    ExpectIntNE(wolfSSL_trust_peer_cert(ssl, NULL,
                                     WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    ExpectIntNE(wolfSSL_trust_peer_cert(ssl, bogusFile,
                                     WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    ExpectIntNE(wolfSSL_trust_peer_cert(ssl, cliCertFile,
                                     WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* success */
    ExpectIntEQ(wolfSSL_trust_peer_cert(ssl, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    #ifdef WOLFSSL_LOCAL_X509_STORE
    /* unload cert */
    ExpectIntNE(wolfSSL_Unload_trust_peers(NULL), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_Unload_trust_peers(ssl), WOLFSSL_SUCCESS);
    #endif
#endif

    /* Test of loading certs from buffers */

    /* invalid buffer */
    ExpectIntNE(wolfSSL_CTX_trust_peer_buffer(ctx, NULL, -1,
                                     WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* success */
#ifdef USE_CERT_BUFFERS_1024
    ExpectIntEQ(wolfSSL_CTX_trust_peer_buffer(ctx, client_cert_der_1024,
          sizeof_client_cert_der_1024, WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif
#ifdef USE_CERT_BUFFERS_2048
    ExpectIntEQ(wolfSSL_CTX_trust_peer_buffer(ctx, client_cert_der_2048,
          sizeof_client_cert_der_2048, WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif

    /* unload cert */
    ExpectIntNE(wolfSSL_CTX_Unload_trust_peers(NULL), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CTX_Unload_trust_peers(ctx), WOLFSSL_SUCCESS);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_load_verify_locations(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX *ctx = NULL;
#ifndef NO_RSA
    WOLFSSL_CERT_MANAGER* cm = NULL;
#ifdef PERSIST_CERT_CACHE
    int cacheSz = 0;
    unsigned char* cache = NULL;
    int used = 0;
#ifndef NO_FILESYSTEM
    const char* cacheFile = "./tests/cert_cache.tmp";
#endif
    int i;
    int t;
    int* p;
#endif
#endif
#if !defined(NO_WOLFSSL_DIR) && !defined(WOLFSSL_TIRTOS)
    const char* load_certs_path = "./certs/external";
    const char* load_no_certs_path = "./examples";
    const char* load_expired_path = "./certs/test/expired";
#endif

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));

    /* invalid arguments */
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations(NULL, caCertFile, NULL),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx, NULL, NULL),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    /* invalid ca file */
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx, bogusFile, NULL),
        WS_RETURN_CODE(WC_NO_ERR_TRACE(WOLFSSL_BAD_FILE),
                       WC_NO_ERR_TRACE(WOLFSSL_FAILURE)));


#if !defined(NO_WOLFSSL_DIR) && !defined(WOLFSSL_TIRTOS) && \
  ((defined(WOLFSSL_QT) || defined(WOLFSSL_IGNORE_BAD_CERT_PATH)) && \
  !(WOLFSSL_LOAD_VERIFY_DEFAULT_FLAGS & WOLFSSL_LOAD_FLAG_IGNORE_BAD_PATH_ERR))
    /* invalid path */
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx, NULL, bogusFile),
        WS_RETURN_CODE(WC_NO_ERR_TRACE(BAD_PATH_ERROR),WC_NO_ERR_TRACE(WOLFSSL_FAILURE)));
#endif
#if defined(WOLFSSL_QT) || defined(WOLFSSL_IGNORE_BAD_CERT_PATH)
    /* test ignoring the invalid path */
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL, bogusFile,
        WOLFSSL_LOAD_FLAG_IGNORE_BAD_PATH_ERR), WOLFSSL_SUCCESS);
#endif

    /* load ca cert */
#ifdef NO_RSA
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx, caCertFile, NULL),
        WS_RETURN_CODE(WC_NO_ERR_TRACE(ASN_UNKNOWN_OID_E),WC_NO_ERR_TRACE(WOLFSSL_FAILURE)));
#else /* Skip the following test without RSA certs. */
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx, caCertFile, NULL),
        WOLFSSL_SUCCESS);

#ifdef PERSIST_CERT_CACHE
    /* Get cert cache size */
    ExpectIntGT(cacheSz = wolfSSL_CTX_get_cert_cache_memsize(ctx), 0);

    ExpectNotNull(cache = (byte*)XMALLOC((size_t)cacheSz, NULL,
                            DYNAMIC_TYPE_TMP_BUFFER));

    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(NULL, NULL, -1, NULL),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(ctx, NULL, -1, NULL),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(NULL, cache, -1, NULL),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(NULL, NULL, cacheSz, NULL),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(NULL, NULL, -1, &used),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(NULL, cache, cacheSz, &used),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(ctx, NULL, cacheSz, &used),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(ctx, cache, -1, &used),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(ctx, cache, cacheSz, NULL),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(ctx, cache, cacheSz - 10, &used),
        WC_NO_ERR_TRACE(BUFFER_E));
    ExpectIntEQ(wolfSSL_CTX_memsave_cert_cache(ctx, cache, cacheSz, &used), 1);
    ExpectIntEQ(cacheSz, used);

    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(NULL, NULL, -1),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, NULL, -1),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(NULL, cache, -1),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(NULL, NULL, cacheSz),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(NULL, cache, cacheSz),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, NULL, cacheSz),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, cache, -1),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    /* Smaller than header. */
    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, cache, 1), WC_NO_ERR_TRACE(BUFFER_E));
    for (i = 1; i < cacheSz; i++) {
        ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, cache, cacheSz - i),
            WC_NO_ERR_TRACE(BUFFER_E));
    }
    if (EXPECT_SUCCESS()) {
        /* Modify header for bad results! */
        p = (int*)cache;
        /* version */
        t = p[0]; p[0] = 0xff;
        ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, cache, cacheSz),
            WC_NO_ERR_TRACE(CACHE_MATCH_ERROR));
        p[0] = t; p++;
        /* rows */
        t = p[0]; p[0] = 0xff;
        ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, cache, cacheSz),
            WC_NO_ERR_TRACE(CACHE_MATCH_ERROR));
        p[0] = t; p++;
        /* columns[0] */
        t = p[0]; p[0] = -1;
        ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, cache, cacheSz),
            WC_NO_ERR_TRACE(PARSE_ERROR));
        p[0] = t; p += CA_TABLE_SIZE;
        /* signerSz*/
        t = p[0]; p[0] = 0xff;
        ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, cache, cacheSz),
            WC_NO_ERR_TRACE(CACHE_MATCH_ERROR));
        p[0] = t;
    }

    ExpectIntEQ(wolfSSL_CTX_memrestore_cert_cache(ctx, cache, cacheSz), 1);
    ExpectIntEQ(cacheSz = wolfSSL_CTX_get_cert_cache_memsize(ctx), used);

#ifndef NO_FILESYSTEM
    ExpectIntEQ(wolfSSL_CTX_save_cert_cache(NULL, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_save_cert_cache(ctx, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_save_cert_cache(NULL, cacheFile), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_save_cert_cache(ctx, cacheFile), 1);

    ExpectIntEQ(wolfSSL_CTX_restore_cert_cache(NULL, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_restore_cert_cache(ctx, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_restore_cert_cache(NULL, cacheFile), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_restore_cert_cache(ctx, "no-file"),
        WC_NO_ERR_TRACE(WOLFSSL_BAD_FILE));
    ExpectIntEQ(wolfSSL_CTX_restore_cert_cache(ctx, cacheFile), 1);
    /* File contents is not a cache. */
    ExpectIntEQ(wolfSSL_CTX_restore_cert_cache(ctx, "./certs/ca-cert.pem"),
        WC_NO_ERR_TRACE(CACHE_MATCH_ERROR));
#endif

    XFREE(cache, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif
    /* Test unloading CA's */
    ExpectIntEQ(wolfSSL_CTX_UnloadCAs(ctx), WOLFSSL_SUCCESS);

#ifdef PERSIST_CERT_CACHE
    /* Verify no certs (result is less than cacheSz) */
    ExpectIntGT(cacheSz, wolfSSL_CTX_get_cert_cache_memsize(ctx));
#endif

    /* load ca cert again */
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx, caCertFile, NULL),
        WOLFSSL_SUCCESS);

    /* Test getting CERT_MANAGER */
    ExpectNotNull(cm = wolfSSL_CTX_GetCertManager(ctx));

    /* Test unloading CA's using CM */
    ExpectIntEQ(wolfSSL_CertManagerUnloadCAs(cm), WOLFSSL_SUCCESS);

#ifdef PERSIST_CERT_CACHE
    /* Verify no certs (result is less than cacheSz) */
    ExpectIntGT(cacheSz, wolfSSL_CTX_get_cert_cache_memsize(ctx));
#endif
#endif

#if !defined(NO_WOLFSSL_DIR) && !defined(WOLFSSL_TIRTOS)
    /* Test loading CA certificates using a path */
    #ifdef NO_RSA
    /* failure here okay since certs in external directory are RSA */
    ExpectIntNE(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL, load_certs_path,
        WOLFSSL_LOAD_FLAG_PEM_CA_ONLY), WOLFSSL_SUCCESS);
    #else
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL, load_certs_path,
        WOLFSSL_LOAD_FLAG_PEM_CA_ONLY), WOLFSSL_SUCCESS);
    #endif

    /* Test loading path with no files */
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL,
        load_no_certs_path, WOLFSSL_LOAD_FLAG_PEM_CA_ONLY),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    /* Test loading expired CA certificates */
    #ifdef NO_RSA
    ExpectIntNE(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL,
        load_expired_path,
        WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY | WOLFSSL_LOAD_FLAG_PEM_CA_ONLY),
        WOLFSSL_SUCCESS);
    #else
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL,
        load_expired_path,
        WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY | WOLFSSL_LOAD_FLAG_PEM_CA_ONLY),
        WOLFSSL_SUCCESS);
    #endif

    /* Test loading CA certificates and ignoring all errors */
    #ifdef NO_RSA
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL, load_certs_path,
        WOLFSSL_LOAD_FLAG_IGNORE_ERR), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    #else
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, NULL, load_certs_path,
        WOLFSSL_LOAD_FLAG_IGNORE_ERR), WOLFSSL_SUCCESS);
    #endif
#endif

    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_load_system_CA_certs(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_SYS_CA_CERTS) && !defined(NO_WOLFSSL_CLIENT) && \
    !defined(NO_TLS) && (!defined(NO_RSA) || defined(HAVE_ECC))
    WOLFSSL_CTX* ctx;
    byte dirValid = 0;
    int ret = 0;

    ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    if (ctx == NULL) {
        fprintf(stderr, "wolfSSL_CTX_new failed.\n");
        ret = -1;
    }
    if (ret == 0) {
    #if defined(USE_WINDOWS_API) || defined(__APPLE__) || \
        defined(NO_WOLFSSL_DIR)
        dirValid = 1;
    #else
        word32 numDirs;
        const char** caDirs = wolfSSL_get_system_CA_dirs(&numDirs);

        if (caDirs == NULL || numDirs == 0) {
            fprintf(stderr, "wolfSSL_get_system_CA_dirs failed.\n");
            ret = -1;
        }
        else {
            ReadDirCtx dirCtx;
            word32 i;

            for (i = 0; i < numDirs; ++i) {
                if (wc_ReadDirFirst(&dirCtx, caDirs[i], NULL) == 0) {
                    /* Directory isn't empty. */
                    dirValid = 1;
                    wc_ReadDirClose(&dirCtx);
                    break;
                }
            }
        }
    #endif
    }
    /*
     * If the directory isn't empty, we should be able to load CA
     * certs from it. On Windows/Mac, we assume the CA cert stores are
     * usable.
     */
    if (ret == 0 && dirValid && wolfSSL_CTX_load_system_CA_certs(ctx) !=
        WOLFSSL_SUCCESS) {
        fprintf(stderr, "wolfSSL_CTX_load_system_CA_certs failed.\n");
        ret = -1;
    }
#ifdef OPENSSL_EXTRA
    if (ret == 0 &&
        wolfSSL_CTX_set_default_verify_paths(ctx) != WOLFSSL_SUCCESS) {
        fprintf(stderr, "wolfSSL_CTX_set_default_verify_paths failed.\n");
        ret = -1;
    }
#endif /* OPENSSL_EXTRA */

    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(ret == 0);
#endif /* WOLFSSL_SYS_CA_CERTS && !NO_WOLFSSL_CLIENT */

    return res;
}

#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS)
static int test_cm_load_ca_buffer(const byte* cert_buf, size_t cert_sz,
    int file_type)
{
    int ret;
    WOLFSSL_CERT_MANAGER* cm;

    cm = wolfSSL_CertManagerNew();
    if (cm == NULL) {
        fprintf(stderr, "test_cm_load_ca failed\n");
        return -1;
    }

    ret = wolfSSL_CertManagerLoadCABuffer(cm, cert_buf, (sword32)cert_sz, file_type);

    wolfSSL_CertManagerFree(cm);

    return ret;
}

static int test_cm_load_ca_file(const char* ca_cert_file)
{
    int ret = 0;
    byte* cert_buf = NULL;
    size_t cert_sz = 0;
#if defined(WOLFSSL_PEM_TO_DER)
    DerBuffer* pDer = NULL;
#endif

    ret = load_file(ca_cert_file, &cert_buf, &cert_sz);
    if (ret == 0) {
        /* normal test */
        ret = test_cm_load_ca_buffer(cert_buf, cert_sz, WOLFSSL_FILETYPE_PEM);

        if (ret == WOLFSSL_SUCCESS) {
            /* test including null terminator in length */
            byte* tmp = (byte*)realloc(cert_buf, cert_sz+1);
            if (tmp == NULL) {
                ret = MEMORY_E;
            }
            else {
                cert_buf = tmp;
                cert_buf[cert_sz] = '\0';
                ret = test_cm_load_ca_buffer(cert_buf, cert_sz+1,
                        WOLFSSL_FILETYPE_PEM);
            }

        }

    #if defined(WOLFSSL_PEM_TO_DER)
        if (ret == WOLFSSL_SUCCESS) {
            /* test loading DER */
            ret = wc_PemToDer(cert_buf, (sword32)cert_sz, CA_TYPE, &pDer,
                    NULL, NULL, NULL);
            if (ret == 0 && pDer != NULL) {
                ret = test_cm_load_ca_buffer(pDer->buffer, pDer->length,
                    WOLFSSL_FILETYPE_ASN1);

                wc_FreeDer(&pDer);
            }
        }
    #endif

    }
    free(cert_buf);

    return ret;
}

static int test_cm_load_ca_buffer_ex(const byte* cert_buf, size_t cert_sz,
                                     int file_type, word32 flags)
{
    int ret;
    WOLFSSL_CERT_MANAGER* cm;

    cm = wolfSSL_CertManagerNew();
    if (cm == NULL) {
        fprintf(stderr, "test_cm_load_ca failed\n");
        return -1;
    }

    ret = wolfSSL_CertManagerLoadCABuffer_ex(cm, cert_buf, (sword32)cert_sz, file_type,
                                             0, flags);

    wolfSSL_CertManagerFree(cm);

    return ret;
}

static int test_cm_load_ca_file_ex(const char* ca_cert_file, word32 flags)
{
    int ret = 0;
    byte* cert_buf = NULL;
    size_t cert_sz = 0;
#if defined(WOLFSSL_PEM_TO_DER)
    DerBuffer* pDer = NULL;
#endif

    ret = load_file(ca_cert_file, &cert_buf, &cert_sz);
    if (ret == 0) {
        /* normal test */
        ret = test_cm_load_ca_buffer_ex(cert_buf, cert_sz,
                                        WOLFSSL_FILETYPE_PEM, flags);

        if (ret == WOLFSSL_SUCCESS) {
            /* test including null terminator in length */
            byte* tmp = (byte*)realloc(cert_buf, cert_sz+1);
            if (tmp == NULL) {
                ret = MEMORY_E;
            }
            else {
                cert_buf = tmp;
                cert_buf[cert_sz] = '\0';
                ret = test_cm_load_ca_buffer_ex(cert_buf, cert_sz+1,
                        WOLFSSL_FILETYPE_PEM, flags);
            }

        }

    #if defined(WOLFSSL_PEM_TO_DER)
        if (ret == WOLFSSL_SUCCESS) {
            /* test loading DER */
            ret = wc_PemToDer(cert_buf, (sword32)cert_sz, CA_TYPE, &pDer,
                              NULL, NULL, NULL);
            if (ret == 0 && pDer != NULL) {
                ret = test_cm_load_ca_buffer_ex(pDer->buffer, pDer->length,
                    WOLFSSL_FILETYPE_ASN1, flags);

                wc_FreeDer(&pDer);
            }
        }
    #endif

    }
    free(cert_buf);

    return ret;
}

#endif /* !NO_FILESYSTEM && !NO_CERTS */

static int test_wolfSSL_CertManagerAPI(void)
{
    EXPECT_DECLS;
#ifndef NO_CERTS
    WOLFSSL_CERT_MANAGER* cm = NULL;
    unsigned char c;

    ExpectNotNull(cm = wolfSSL_CertManagerNew_ex(NULL));

    wolfSSL_CertManagerFree(NULL);
    ExpectIntEQ(wolfSSL_CertManager_up_ref(NULL), 0);
    ExpectIntEQ(wolfSSL_CertManagerUnloadCAs(NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
#ifdef WOLFSSL_TRUST_PEER_CERT
    ExpectIntEQ(wolfSSL_CertManagerUnload_trust_peers(NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
#endif

    ExpectIntEQ(wolfSSL_CertManagerLoadCABuffer_ex(NULL, &c, 1,
        WOLFSSL_FILETYPE_ASN1, 0, 0), WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));

#if !defined(NO_WOLFSSL_CLIENT) || !defined(WOLFSSL_NO_CLIENT_AUTH)
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(NULL, NULL, -1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, NULL, -1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(NULL, &c, -1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(NULL, NULL, 1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(NULL, &c, 1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, NULL, 1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, &c, -1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, &c, 1, -1),
        WC_NO_ERR_TRACE(WOLFSSL_BAD_FILETYPE));
#endif

#if !defined(NO_FILESYSTEM)
    {
        const char* ca_cert = "./certs/ca-cert.pem";
    #if !defined(NO_WOLFSSL_CLIENT) || !defined(WOLFSSL_NO_CLIENT_AUTH)
        const char* ca_cert_der = "./certs/ca-cert.der";
    #endif
        const char* ca_path = "./certs";

    #if !defined(NO_WOLFSSL_CLIENT) || !defined(WOLFSSL_NO_CLIENT_AUTH)
        ExpectIntEQ(wolfSSL_CertManagerVerify(NULL, NULL, -1),
            WC_NO_ERR_TRACE(BAD_FUNC_ARG));
        ExpectIntEQ(wolfSSL_CertManagerVerify(cm, NULL, WOLFSSL_FILETYPE_ASN1),
            WC_NO_ERR_TRACE(BAD_FUNC_ARG));
        ExpectIntEQ(wolfSSL_CertManagerVerify(NULL, ca_cert,
            WOLFSSL_FILETYPE_PEM), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
        ExpectIntEQ(wolfSSL_CertManagerVerify(cm, ca_cert, -1),
            WC_NO_ERR_TRACE(WOLFSSL_BAD_FILETYPE));
        ExpectIntEQ(wolfSSL_CertManagerVerify(cm, "no-file",
            WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(WOLFSSL_BAD_FILE));
        ExpectIntEQ(wolfSSL_CertManagerVerify(cm, ca_cert_der,
            WOLFSSL_FILETYPE_PEM), WC_NO_ERR_TRACE(ASN_NO_PEM_HEADER));
    #endif

        ExpectIntEQ(wolfSSL_CertManagerLoadCA(NULL, NULL, NULL),
            WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
        ExpectIntEQ(wolfSSL_CertManagerLoadCA(NULL, ca_cert, NULL),
            WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
        ExpectIntEQ(wolfSSL_CertManagerLoadCA(NULL, NULL, ca_path),
            WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
        ExpectIntEQ(wolfSSL_CertManagerLoadCA(NULL, ca_cert, ca_path),
            WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
    }
#endif

#ifdef OPENSSL_COMPATIBLE_DEFAULTS
    ExpectIntEQ(wolfSSL_CertManagerEnableCRL(cm, 0), 1);
#elif !defined(HAVE_CRL)
    ExpectIntEQ(wolfSSL_CertManagerEnableCRL(cm, 0), WC_NO_ERR_TRACE(NOT_COMPILED_IN));
#endif

    ExpectIntEQ(wolfSSL_CertManagerDisableCRL(NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerDisableCRL(cm), 1);
#ifdef HAVE_CRL
    /* Test APIs when CRL is disabled. */
#ifdef HAVE_CRL_IO
    ExpectIntEQ(wolfSSL_CertManagerSetCRL_IOCb(cm, NULL), 1);
#endif
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(cm, server_cert_der_2048,
        sizeof_server_cert_der_2048), 1);
    ExpectIntEQ(wolfSSL_CertManagerFreeCRL(cm), 1);
#endif

    /* OCSP */
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSP(NULL, 0), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerDisableOCSP(NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSPStapling(NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerDisableOCSPStapling(NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSPMustStaple(NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerDisableOCSPMustStaple(NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
#if !defined(HAVE_CERTIFICATE_STATUS_REQUEST) && \
    !defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2)
    ExpectIntEQ(wolfSSL_CertManagerDisableOCSPStapling(cm), WC_NO_ERR_TRACE(NOT_COMPILED_IN));
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSPMustStaple(cm), WC_NO_ERR_TRACE(NOT_COMPILED_IN));
    ExpectIntEQ(wolfSSL_CertManagerDisableOCSPMustStaple(cm), WC_NO_ERR_TRACE(NOT_COMPILED_IN));
#endif

#ifdef HAVE_OCSP
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(NULL, NULL, -1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(cm, NULL, -1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(NULL, &c, -1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(NULL, NULL, 1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(NULL, &c, 1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(cm, NULL, 1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(cm, &c, -1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(NULL, NULL, 0,
        NULL, NULL, NULL, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(cm, NULL, 1,
        NULL, NULL, NULL, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(NULL, &c, 1,
        NULL, NULL, NULL, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    ExpectIntEQ(wolfSSL_CertManagerSetOCSPOverrideURL(NULL, NULL),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerSetOCSPOverrideURL(NULL, ""),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerSetOCSPOverrideURL(cm, NULL), 1);

    ExpectIntEQ(wolfSSL_CertManagerSetOCSP_Cb(NULL, NULL, NULL, NULL),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerSetOCSP_Cb(cm, NULL, NULL, NULL), 1);

    ExpectIntEQ(wolfSSL_CertManagerDisableOCSP(cm), 1);
    /* Test APIs when OCSP is disabled. */
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(cm, &c, 1,
        NULL, NULL, NULL, NULL), 1);
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(cm, &c, 1), 1);

#endif

    ExpectIntEQ(wolfSSL_CertManager_up_ref(cm), 1);
    if (EXPECT_SUCCESS()) {
        wolfSSL_CertManagerFree(cm);
    }
    wolfSSL_CertManagerFree(cm);
    cm = NULL;

    ExpectNotNull(cm = wolfSSL_CertManagerNew_ex(NULL));

#ifdef HAVE_OCSP
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSP(cm, WOLFSSL_OCSP_URL_OVERRIDE |
         WOLFSSL_OCSP_CHECKALL), 1);
#if defined(HAVE_CERTIFICATE_STATUS_REQUEST) || \
    defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2)
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSPStapling(cm), 1);
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSPStapling(cm), 1);
    ExpectIntEQ(wolfSSL_CertManagerDisableOCSPStapling(cm), 1);
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSPStapling(cm), 1);
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSPMustStaple(cm), 1);
    ExpectIntEQ(wolfSSL_CertManagerDisableOCSPMustStaple(cm), 1);
#endif

    ExpectIntEQ(wolfSSL_CertManagerSetOCSPOverrideURL(cm, ""), 1);
    ExpectIntEQ(wolfSSL_CertManagerSetOCSPOverrideURL(cm, ""), 1);
#endif

#ifdef WOLFSSL_TRUST_PEER_CERT
    ExpectIntEQ(wolfSSL_CertManagerUnload_trust_peers(cm), 1);
#endif
    wolfSSL_CertManagerFree(cm);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CertManagerLoadCABuffer(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS)
    const char* ca_cert = "./certs/ca-cert.pem";
    const char* ca_expired_cert = "./certs/test/expired/expired-ca.pem";
    int ret;

    ExpectIntLE(ret = test_cm_load_ca_file(ca_cert), 1);
#if defined(NO_WOLFSSL_CLIENT) && defined(NO_WOLFSSL_SERVER)
    ExpectIntEQ(ret, WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
#elif defined(NO_RSA)
    ExpectIntEQ(ret, WC_NO_ERR_TRACE(ASN_UNKNOWN_OID_E));
#else
    ExpectIntEQ(ret, WOLFSSL_SUCCESS);
#endif

    ExpectIntLE(ret = test_cm_load_ca_file(ca_expired_cert), 1);
#if defined(NO_WOLFSSL_CLIENT) && defined(NO_WOLFSSL_SERVER)
    ExpectIntEQ(ret, WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
#elif defined(NO_RSA)
    ExpectIntEQ(ret, WC_NO_ERR_TRACE(ASN_UNKNOWN_OID_E));
#elif !(WOLFSSL_LOAD_VERIFY_DEFAULT_FLAGS & WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY) && \
      !defined(NO_ASN_TIME)
    ExpectIntEQ(ret, WC_NO_ERR_TRACE(ASN_AFTER_DATE_E));
#else
    ExpectIntEQ(ret, WOLFSSL_SUCCESS);
#endif
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CertManagerLoadCABuffer_ex(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS)
    const char* ca_cert = "./certs/ca-cert.pem";
    const char* ca_expired_cert = "./certs/test/expired/expired-ca.pem";
    int ret;

    ExpectIntLE(ret = test_cm_load_ca_file_ex(ca_cert, WOLFSSL_LOAD_FLAG_NONE),
        1);
#if defined(NO_WOLFSSL_CLIENT) && defined(NO_WOLFSSL_SERVER)
    ExpectIntEQ(ret, WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
#elif defined(NO_RSA)
    ExpectIntEQ(ret, WC_NO_ERR_TRACE(ASN_UNKNOWN_OID_E));
#else
    ExpectIntEQ(ret, WOLFSSL_SUCCESS);
#endif

    ExpectIntLE(ret = test_cm_load_ca_file_ex(ca_expired_cert,
        WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY), 1);
#if defined(NO_WOLFSSL_CLIENT) && defined(NO_WOLFSSL_SERVER)
    ExpectIntEQ(ret, WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
#elif defined(NO_RSA)
    ExpectIntEQ(ret, WC_NO_ERR_TRACE(ASN_UNKNOWN_OID_E));
#elif !(WOLFSSL_LOAD_VERIFY_DEFAULT_FLAGS & WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY) && \
      !defined(NO_ASN_TIME) && defined(WOLFSSL_TRUST_PEER_CERT) && \
      defined(OPENSSL_COMPATIBLE_DEFAULTS)
    ExpectIntEQ(ret, WC_NO_ERR_TRACE(ASN_AFTER_DATE_E));
#else
    ExpectIntEQ(ret, WOLFSSL_SUCCESS);
#endif

#endif

    return EXPECT_RESULT();
}


static int test_wolfSSL_CertManagerGetCerts(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(NO_CERTS) && \
    !defined(NO_FILESYSTEM) && !defined(NO_RSA) && \
     defined(WOLFSSL_SIGNER_DER_CERT)
    WOLFSSL_CERT_MANAGER* cm = NULL;
    WOLFSSL_STACK* sk = NULL;
    X509* x509 = NULL;
    X509* cert1 = NULL;
    FILE* file1 = NULL;
#ifdef DEBUG_WOLFSSL_VERBOSE
    WOLFSSL_BIO* bio = NULL;
#endif
    int i = 0;
    int ret = 0;
    const byte* der = NULL;
    int derSz = 0;

    ExpectNotNull(file1 = fopen("./certs/ca-cert.pem", "rb"));

    ExpectNotNull(cert1 = wolfSSL_PEM_read_X509(file1, NULL, NULL, NULL));
    if (file1 != NULL) {
        fclose(file1);
    }

    ExpectNull(sk = wolfSSL_CertManagerGetCerts(NULL));
    ExpectNotNull(cm = wolfSSL_CertManagerNew_ex(NULL));
    ExpectNull(sk = wolfSSL_CertManagerGetCerts(cm));

    ExpectNotNull(der = wolfSSL_X509_get_der(cert1, &derSz));
#if defined(OPENSSL_ALL) || defined(WOLFSSL_QT)
    /* Check that ASN_SELF_SIGNED_E is returned for a self-signed cert for QT
     * and full OpenSSL compatibility */
    ExpectIntEQ(ret = wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_SELF_SIGNED_E));
#else
    ExpectIntEQ(ret = wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_NO_SIGNER_E));
#endif

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CertManagerLoadCA(cm,
                "./certs/ca-cert.pem", NULL));

    ExpectNotNull(sk = wolfSSL_CertManagerGetCerts(cm));

    for (i = 0; EXPECT_SUCCESS() && i < sk_X509_num(sk); i++) {
        ExpectNotNull(x509 = sk_X509_value(sk, i));
        ExpectIntEQ(0, wolfSSL_X509_cmp(x509, cert1));

#ifdef DEBUG_WOLFSSL_VERBOSE
        bio = BIO_new(wolfSSL_BIO_s_file());
        if (bio != NULL) {
            BIO_set_fp(bio, stderr, BIO_NOCLOSE);
            X509_print(bio, x509);
            BIO_free(bio);
        }
#endif /* DEBUG_WOLFSSL_VERBOSE */
    }
    wolfSSL_X509_free(cert1);
    sk_X509_pop_free(sk, NULL);
    wolfSSL_CertManagerFree(cm);
#endif /* defined(OPENSSL_ALL) && !defined(NO_CERTS) && \
          !defined(NO_FILESYSTEM) && !defined(NO_RSA) && \
          defined(WOLFSSL_SIGNER_DER_CERT) */

    return EXPECT_RESULT();
}

static int test_wolfSSL_CertManagerSetVerify(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_CM_VERIFY) && !defined(NO_RSA) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(WOLFSSL_NO_CLIENT_AUTH))
    WOLFSSL_CERT_MANAGER* cm = NULL;
    int tmp = myVerifyAction;
    const char* ca_cert = "./certs/ca-cert.pem";
    const char* expiredCert = "./certs/test/expired/expired-cert.pem";

    wolfSSL_CertManagerSetVerify(NULL, NULL);
    wolfSSL_CertManagerSetVerify(NULL, myVerify);

    ExpectNotNull(cm = wolfSSL_CertManagerNew());

    wolfSSL_CertManagerSetVerify(cm, myVerify);

#if defined(NO_WOLFSSL_CLIENT) && defined(NO_WOLFSSL_SERVER)
    ExpectIntEQ(wolfSSL_CertManagerLoadCA(cm, ca_cert, NULL), -1);
#else
    ExpectIntEQ(wolfSSL_CertManagerLoadCA(cm, ca_cert, NULL),
                WOLFSSL_SUCCESS);
#endif
    /* Use the test CB that always accepts certs */
    myVerifyAction = VERIFY_OVERRIDE_ERROR;

    ExpectIntEQ(wolfSSL_CertManagerVerify(cm, expiredCert,
        WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

#ifdef WOLFSSL_ALWAYS_VERIFY_CB
    {
        const char* verifyCert = "./certs/server-cert.der";
        /* Use the test CB that always fails certs */
        myVerifyAction = VERIFY_FORCE_FAIL;

        ExpectIntEQ(wolfSSL_CertManagerVerify(cm, verifyCert,
            WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(VERIFY_CERT_ERROR));
    }
#endif

    wolfSSL_CertManagerFree(cm);
    myVerifyAction = tmp;
#endif

    return EXPECT_RESULT();
}

#if !defined(NO_FILESYSTEM) && defined(OPENSSL_EXTRA) && \
    defined(DEBUG_UNIT_TEST_CERTS)
/* Used when debugging name constraint tests. Not static to allow use in
 * multiple locations with complex define guards. */
void DEBUG_WRITE_CERT_X509(WOLFSSL_X509* x509, const char* fileName)
{
    BIO* out = BIO_new_file(fileName, "wb");
    if (out != NULL) {
        PEM_write_bio_X509(out, x509);
        BIO_free(out);
    }
}
void DEBUG_WRITE_DER(const byte* der, int derSz, const char* fileName)
{
    BIO* out = BIO_new_file(fileName, "wb");
    if (out != NULL) {
        BIO_write(out, der, derSz);
        BIO_free(out);
    }
}
#else
#define DEBUG_WRITE_CERT_X509(x509, fileName) WC_DO_NOTHING
#define DEBUG_WRITE_DER(der, derSz, fileName) WC_DO_NOTHING
#endif


static int test_wolfSSL_CertManagerNameConstraint(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CM_VERIFY) && !defined(NO_RSA) && \
    defined(OPENSSL_EXTRA) && defined(WOLFSSL_CERT_GEN) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_ALT_NAMES) && \
    !defined(NO_SHA256)
    WOLFSSL_CERT_MANAGER* cm = NULL;
    WOLFSSL_EVP_PKEY *priv = NULL;
    WOLFSSL_X509_NAME* name = NULL;
    const char* ca_cert = "./certs/test/cert-ext-nc.der";
    const char* server_cert = "./certs/test/server-goodcn.pem";
    int i = 0;
    static const byte extNameConsOid[] = {85, 29, 30};

    RsaKey  key;
    WC_RNG  rng;
    byte    *der = NULL;
    int     derSz = 0;
    word32  idx = 0;
    byte    *pt;
    WOLFSSL_X509 *x509 = NULL;
    WOLFSSL_X509 *ca = NULL;

    wc_InitRng(&rng);

    /* load in CA private key for signing */
    ExpectIntEQ(wc_InitRsaKey_ex(&key, HEAP_HINT, testDevId), 0);
    ExpectIntEQ(wc_RsaPrivateKeyDecode(server_key_der_2048, &idx, &key,
                sizeof_server_key_der_2048), 0);

    /* get ca certificate then alter it */
    ExpectNotNull(der =
            (byte*)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER));
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(ca_cert,
                WOLFSSL_FILETYPE_ASN1));
    ExpectNotNull(pt = (byte*)wolfSSL_X509_get_tbs(x509, &derSz));
    if (EXPECT_SUCCESS() && (der != NULL)) {
        XMEMCPY(der, pt, (size_t)derSz);

        /* find the name constraint extension and alter it */
        pt = der;
        for (i = 0; i < derSz - 3; i++) {
            if (XMEMCMP(pt, extNameConsOid, 3) == 0) {
                pt += 3;
                break;
            }
            pt++;
        }
        ExpectIntNE(i, derSz - 3); /* did not find OID if this case is hit */

        /* go to the length value and set it to 0 */
        while (i < derSz && *pt != 0x81) {
            pt++;
            i++;
        }
        ExpectIntNE(i, derSz); /* did not place to alter */
        pt++;
        *pt = 0x00;
    }

    /* resign the altered certificate */
    ExpectIntGT((derSz = wc_SignCert(derSz, CTC_SHA256wRSA, der,
                             FOURK_BUF, &key, NULL, &rng)), 0);

    ExpectNotNull(cm = wolfSSL_CertManagerNew());
    ExpectIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_PARSE_E));
    wolfSSL_CertManagerFree(cm);

    XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    wolfSSL_X509_free(x509);
    wc_FreeRsaKey(&key);
    wc_FreeRng(&rng);

    /* add email alt name to satisfy constraint */
    pt = (byte*)server_key_der_2048;
    ExpectNotNull(priv = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL,
                (const unsigned char**)&pt, sizeof_server_key_der_2048));

    ExpectNotNull(cm = wolfSSL_CertManagerNew());
    ExpectNotNull(ca = wolfSSL_X509_load_certificate_file(ca_cert,
                WOLFSSL_FILETYPE_ASN1));

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(ca, &derSz)));
    DEBUG_WRITE_DER(der, derSz, "ca.der");

    ExpectIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* Good cert test with proper alt email name */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "emailAddress", MBSTRING_UTF8,
                    (byte*)"support@info.wolfssl.com", 24, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "wolfssl@info.wolfssl.com", ASN_RFC822_TYPE);

    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);
    x509 = NULL;


    /* Cert with bad alt name list */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "emailAddress", MBSTRING_UTF8,
                    (byte*)"support@info.wolfssl.com", 24, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "wolfssl@info.com", ASN_RFC822_TYPE);
    wolfSSL_X509_add_altname(x509, "wolfssl@info.wolfssl.com", ASN_RFC822_TYPE);

    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_NAME_INVALID_E));

    wolfSSL_CertManagerFree(cm);
    wolfSSL_X509_free(x509);
    wolfSSL_X509_free(ca);
    wolfSSL_EVP_PKEY_free(priv);
#endif

    return EXPECT_RESULT();
}


static int test_wolfSSL_CertManagerNameConstraint2(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CM_VERIFY) && !defined(NO_RSA) && \
    defined(OPENSSL_EXTRA) && defined(WOLFSSL_CERT_GEN) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_ALT_NAMES)
    const char* ca_cert  = "./certs/test/cert-ext-ndir.der";
    const char* ca_cert2 = "./certs/test/cert-ext-ndir-exc.der";
    const char* server_cert = "./certs/server-cert.pem";
    WOLFSSL_CERT_MANAGER* cm = NULL;
    WOLFSSL_X509 *x509 = NULL;
    WOLFSSL_X509 *ca = NULL;

    const unsigned char *der = NULL;
    const unsigned char *pt;
    WOLFSSL_EVP_PKEY *priv = NULL;
    WOLFSSL_X509_NAME* name = NULL;
    int   derSz = 0;

    /* C=US*/
    char altName[] = {
        0x30, 0x0D, 0x31, 0x0B, 0x30, 0x09,
        0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53
    };

    /* C=ID */
    char altNameFail[] = {
        0x30, 0x0D, 0x31, 0x0B, 0x30, 0x09,
        0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x49, 0x44
    };

    /* C=US ST=California*/
    char altNameExc[] = {
        0x30, 0x22,
        0x31, 0x0B,
        0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53,
        0x31, 0x13,
        0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0C, 0x0A,
        0x43, 0x61, 0x6c, 0x69, 0x66, 0x6f, 0x72, 0x6e, 0x69, 0x61
    };
    /* load in CA private key for signing */
    pt = ca_key_der_2048;
    ExpectNotNull(priv = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL, &pt,
                                                sizeof_ca_key_der_2048));

    ExpectNotNull(cm = wolfSSL_CertManagerNew());
    ExpectNotNull(ca = wolfSSL_X509_load_certificate_file(ca_cert,
                WOLFSSL_FILETYPE_ASN1));
    ExpectNotNull((der = wolfSSL_X509_get_der(ca, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wolfSSL_X509_sign(x509, priv, EVP_sha3_256());
#else
    wolfSSL_X509_sign(x509, priv, EVP_sha256());
#endif
    ExpectNotNull((der = wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* Test no name case. */
    ExpectIntEQ(wolfSSL_X509_add_altname_ex(x509, NULL, 0, ASN_DIR_TYPE),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_add_altname(x509, "", ASN_DIR_TYPE),
        WOLFSSL_SUCCESS);
    /* IP not supported. */
    ExpectIntEQ(wolfSSL_X509_add_altname(x509, "127.0.0.1", ASN_IP_TYPE),
        WOLFSSL_FAILURE);

    /* add in matching DIR alt name and resign */
    wolfSSL_X509_add_altname_ex(x509, altName, sizeof(altName), ASN_DIR_TYPE);
#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wolfSSL_X509_sign(x509, priv, EVP_sha3_256());
#else
    wolfSSL_X509_sign(x509, priv, EVP_sha256());
#endif

    ExpectNotNull((der = wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* check verify fail */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);

    /* add in miss matching DIR alt name and resign */
    wolfSSL_X509_add_altname_ex(x509, altNameFail, sizeof(altNameFail),
            ASN_DIR_TYPE);

#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wolfSSL_X509_sign(x509, priv, EVP_sha3_256());
#else
    wolfSSL_X509_sign(x509, priv, EVP_sha256());
#endif
    ExpectNotNull((der = wolfSSL_X509_get_der(x509, &derSz)));
#ifndef WOLFSSL_NO_ASN_STRICT
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_NAME_INVALID_E));
#else
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif

    /* check that it still fails if one bad altname and one good altname is in
     * the certificate */
    wolfSSL_X509_free(x509);
    x509 = NULL;
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    wolfSSL_X509_add_altname_ex(x509, altName, sizeof(altName), ASN_DIR_TYPE);
    wolfSSL_X509_add_altname_ex(x509, altNameFail, sizeof(altNameFail),
            ASN_DIR_TYPE);

#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wolfSSL_X509_sign(x509, priv, EVP_sha3_256());
#else
    wolfSSL_X509_sign(x509, priv, EVP_sha256());
#endif
    ExpectNotNull((der = wolfSSL_X509_get_der(x509, &derSz)));
#ifndef WOLFSSL_NO_ASN_STRICT
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_NAME_INVALID_E));
#else
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif

    /* check it fails with switching position of bad altname */
    wolfSSL_X509_free(x509);
    x509 = NULL;
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    wolfSSL_X509_add_altname_ex(x509, altNameFail, sizeof(altNameFail),
            ASN_DIR_TYPE);
    wolfSSL_X509_add_altname_ex(x509, altName, sizeof(altName), ASN_DIR_TYPE);

#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wolfSSL_X509_sign(x509, priv, EVP_sha3_256());
#else
    wolfSSL_X509_sign(x509, priv, EVP_sha256());
#endif
    ExpectNotNull((der = wolfSSL_X509_get_der(x509, &derSz)));
#ifndef WOLFSSL_NO_ASN_STRICT
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_NAME_INVALID_E));
#else
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif
    wolfSSL_CertManagerFree(cm);

    wolfSSL_X509_free(x509);
    x509 = NULL;
    wolfSSL_X509_free(ca);
    ca = NULL;

    /* now test with excluded name constraint */
    ExpectNotNull(cm = wolfSSL_CertManagerNew());
    ExpectNotNull(ca = wolfSSL_X509_load_certificate_file(ca_cert2,
                WOLFSSL_FILETYPE_ASN1));
    ExpectNotNull((der = wolfSSL_X509_get_der(ca, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    wolfSSL_X509_add_altname_ex(x509, altNameExc, sizeof(altNameExc),
            ASN_DIR_TYPE);
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);

#if defined(WOLFSSL_SHA3) && !defined(WOLFSSL_NOSHA3_256)
    wolfSSL_X509_sign(x509, priv, EVP_sha3_256());
#else
    wolfSSL_X509_sign(x509, priv, EVP_sha256());
#endif
    ExpectNotNull((der = wolfSSL_X509_get_der(x509, &derSz)));
#ifndef WOLFSSL_NO_ASN_STRICT
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_NAME_INVALID_E));
#else
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif
    wolfSSL_CertManagerFree(cm);
    wolfSSL_X509_free(x509);
    wolfSSL_X509_free(ca);
    wolfSSL_EVP_PKEY_free(priv);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CertManagerNameConstraint3(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CM_VERIFY) && !defined(NO_RSA) && \
    defined(OPENSSL_EXTRA) && defined(WOLFSSL_CERT_GEN) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_ALT_NAMES) && \
    !defined(NO_SHA256)
    WOLFSSL_CERT_MANAGER* cm = NULL;
    WOLFSSL_EVP_PKEY *priv = NULL;
    WOLFSSL_X509_NAME* name = NULL;
    const char* ca_cert = "./certs/test/cert-ext-mnc.der";
    const char* server_cert = "./certs/test/server-goodcn.pem";

    byte    *der = NULL;
    int     derSz = 0;
    byte    *pt;
    WOLFSSL_X509 *x509 = NULL;
    WOLFSSL_X509 *ca = NULL;

    pt = (byte*)server_key_der_2048;
    ExpectNotNull(priv = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL,
                (const unsigned char**)&pt, sizeof_server_key_der_2048));

    ExpectNotNull(cm = wolfSSL_CertManagerNew());
    ExpectNotNull(ca = wolfSSL_X509_load_certificate_file(ca_cert,
                WOLFSSL_FILETYPE_ASN1));
    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(ca, &derSz)));
    DEBUG_WRITE_DER(der, derSz, "ca.der");

    ExpectIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* check satisfying .wolfssl.com constraint passes */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "emailAddress", MBSTRING_UTF8,
                    (byte*)"support@info.wolfssl.com", 24, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "wolfssl@info.wolfssl.com", ASN_RFC822_TYPE);

    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-1st-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* check satisfying .random.com constraint passes */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "emailAddress", MBSTRING_UTF8,
                    (byte*)"support@info.example.com", 24, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "wolfssl@info.example.com", ASN_RFC822_TYPE);

    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-2nd-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* check fail case when neither constraint is matched */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "emailAddress", MBSTRING_UTF8,
                     (byte*)"support@info.com", 16, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "wolfssl@info.com", ASN_RFC822_TYPE);

    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_NAME_INVALID_E));

    wolfSSL_CertManagerFree(cm);
    wolfSSL_X509_free(x509);
    wolfSSL_X509_free(ca);
    wolfSSL_EVP_PKEY_free(priv);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CertManagerNameConstraint4(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CM_VERIFY) && !defined(NO_RSA) && \
    defined(OPENSSL_EXTRA) && defined(WOLFSSL_CERT_GEN) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_ALT_NAMES) && \
    !defined(NO_SHA256)
    WOLFSSL_CERT_MANAGER* cm = NULL;
    WOLFSSL_EVP_PKEY *priv = NULL;
    WOLFSSL_X509_NAME* name = NULL;
    const char* ca_cert = "./certs/test/cert-ext-ncdns.der";
    const char* server_cert = "./certs/test/server-goodcn.pem";

    byte    *der = NULL;
    int     derSz;
    byte    *pt;
    WOLFSSL_X509 *x509 = NULL;
    WOLFSSL_X509 *ca = NULL;

    pt = (byte*)server_key_der_2048;
    ExpectNotNull(priv = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL,
                (const unsigned char**)&pt, sizeof_server_key_der_2048));

    ExpectNotNull(cm = wolfSSL_CertManagerNew());
    ExpectNotNull(ca = wolfSSL_X509_load_certificate_file(ca_cert,
                WOLFSSL_FILETYPE_ASN1));
    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(ca, &derSz)));
    DEBUG_WRITE_DER(der, derSz, "ca.der");

    ExpectIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* check satisfying wolfssl.com constraint passes */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "www.wolfssl.com", ASN_DNS_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-1st-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* check satisfying example.com constraint passes */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"example.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "www.example.com", ASN_DNS_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-2nd-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* check satisfying wolfssl.com constraint passes with list of DNS's */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "www.wolfssl.com", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "www.info.wolfssl.com", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "extra.wolfssl.com", ASN_DNS_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-multiple-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* check fail when one DNS in the list is bad */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "www.wolfssl.com", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "www.nomatch.com", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "www.info.wolfssl.com", ASN_DNS_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-multiple-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_NAME_INVALID_E));
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* check fail case when neither constraint is matched */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"common", 6, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "www.random.com", ASN_DNS_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_NAME_INVALID_E));

    wolfSSL_CertManagerFree(cm);
    wolfSSL_X509_free(x509);
    wolfSSL_X509_free(ca);
    wolfSSL_EVP_PKEY_free(priv);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CertManagerNameConstraint5(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CM_VERIFY) && !defined(NO_RSA) && \
    defined(OPENSSL_EXTRA) && defined(WOLFSSL_CERT_GEN) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_ALT_NAMES) && \
    !defined(NO_SHA256)
    WOLFSSL_CERT_MANAGER* cm = NULL;
    WOLFSSL_EVP_PKEY *priv = NULL;
    WOLFSSL_X509_NAME* name = NULL;
    const char* ca_cert = "./certs/test/cert-ext-ncmixed.der";
    const char* server_cert = "./certs/test/server-goodcn.pem";

    byte    *der = NULL;
    int     derSz;
    byte    *pt;
    WOLFSSL_X509 *x509 = NULL;
    WOLFSSL_X509 *ca = NULL;

    pt = (byte*)server_key_der_2048;
    ExpectNotNull(priv = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL,
                (const unsigned char**)&pt, sizeof_server_key_der_2048));

    ExpectNotNull(cm = wolfSSL_CertManagerNew());
    ExpectNotNull(ca = wolfSSL_X509_load_certificate_file(ca_cert,
                WOLFSSL_FILETYPE_ASN1));
    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(ca, &derSz)));
    DEBUG_WRITE_DER(der, derSz, "ca.der");

    ExpectIntEQ(wolfSSL_CertManagerLoadCABuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* check satisfying wolfssl.com constraint passes */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"example", 7, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "good.example", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "facts@into.wolfssl.com", ASN_RFC822_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* fail with DNS check because of common name */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName", MBSTRING_UTF8,
                             (byte*)"wolfssl.com", 11, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "example", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "facts@wolfssl.com", ASN_RFC822_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-cn-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_NAME_INVALID_E));
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* fail on permitted DNS name constraint */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "www.example", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "www.wolfssl", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "info@wolfssl.com", ASN_RFC822_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-1st-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_NAME_INVALID_E));
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* fail on permitted email name constraint */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);
    name = NULL;

    wolfSSL_X509_add_altname(x509, "example", ASN_DNS_TYPE);
    wolfSSL_X509_add_altname(x509, "info@wolfssl.com", ASN_RFC822_TYPE);
    wolfSSL_X509_add_altname(x509, "info@example.com", ASN_RFC822_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "bad-2nd-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_NAME_INVALID_E));
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* success with empty email name */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(server_cert,
                WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = wolfSSL_X509_get_subject_name(ca));
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509, name), WOLFSSL_SUCCESS);
    name = NULL;

    ExpectNotNull(name = X509_NAME_new());
    ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName", MBSTRING_UTF8,
                                       (byte*)"US", 2, -1, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509, name), WOLFSSL_SUCCESS);
    X509_NAME_free(name);

    wolfSSL_X509_add_altname(x509, "example", ASN_DNS_TYPE);
    ExpectIntGT(wolfSSL_X509_sign(x509, priv, EVP_sha256()), 0);
    DEBUG_WRITE_CERT_X509(x509, "good-missing-constraint-cert.pem");

    ExpectNotNull((der = (byte*)wolfSSL_X509_get_der(x509, &derSz)));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, der, derSz,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    wolfSSL_X509_free(x509);

    wolfSSL_CertManagerFree(cm);
    wolfSSL_X509_free(ca);
    wolfSSL_EVP_PKEY_free(priv);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CertManagerCRL(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && defined(HAVE_CRL) && \
    !defined(NO_RSA)
    const char* ca_cert = "./certs/ca-cert.pem";
    const char* crl1     = "./certs/crl/crl.pem";
    const char* crl2     = "./certs/crl/crl2.pem";
#ifdef WC_RSA_PSS
    const char* crl_rsapss = "./certs/crl/crl_rsapss.pem";
    const char* ca_rsapss  = "./certs/rsapss/ca-rsapss.pem";
#endif
    /* ./certs/crl/crl.der */
    const unsigned char crl_buff[] = {
        0x30, 0x82, 0x02, 0x04, 0x30, 0x81, 0xED, 0x02,
        0x01, 0x01, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86,
        0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x0B, 0x05,
        0x00, 0x30, 0x81, 0x94, 0x31, 0x0B, 0x30, 0x09,
        0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55,
        0x53, 0x31, 0x10, 0x30, 0x0E, 0x06, 0x03, 0x55,
        0x04, 0x08, 0x0C, 0x07, 0x4D, 0x6F, 0x6E, 0x74,
        0x61, 0x6E, 0x61, 0x31, 0x10, 0x30, 0x0E, 0x06,
        0x03, 0x55, 0x04, 0x07, 0x0C, 0x07, 0x42, 0x6F,
        0x7A, 0x65, 0x6D, 0x61, 0x6E, 0x31, 0x11, 0x30,
        0x0F, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x0C, 0x08,
        0x53, 0x61, 0x77, 0x74, 0x6F, 0x6F, 0x74, 0x68,
        0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04,
        0x0B, 0x0C, 0x0A, 0x43, 0x6F, 0x6E, 0x73, 0x75,
        0x6C, 0x74, 0x69, 0x6E, 0x67, 0x31, 0x18, 0x30,
        0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x0F,
        0x77, 0x77, 0x77, 0x2E, 0x77, 0x6F, 0x6C, 0x66,
        0x73, 0x73, 0x6C, 0x2E, 0x63, 0x6F, 0x6D, 0x31,
        0x1F, 0x30, 0x1D, 0x06, 0x09, 0x2A, 0x86, 0x48,
        0x86, 0xF7, 0x0D, 0x01, 0x09, 0x01, 0x16, 0x10,
        0x69, 0x6E, 0x66, 0x6F, 0x40, 0x77, 0x6F, 0x6C,
        0x66, 0x73, 0x73, 0x6C, 0x2E, 0x63, 0x6F, 0x6D,
        0x17, 0x0D, 0x32, 0x34, 0x30, 0x31, 0x30, 0x39,
        0x30, 0x30, 0x33, 0x34, 0x33, 0x30, 0x5A, 0x17,
        0x0D, 0x32, 0x36, 0x31, 0x30, 0x30, 0x35, 0x30,
        0x30, 0x33, 0x34, 0x33, 0x30, 0x5A, 0x30, 0x14,
        0x30, 0x12, 0x02, 0x01, 0x02, 0x17, 0x0D, 0x32,
        0x34, 0x30, 0x31, 0x30, 0x39, 0x30, 0x30, 0x33,
        0x34, 0x33, 0x30, 0x5A, 0xA0, 0x0E, 0x30, 0x0C,
        0x30, 0x0A, 0x06, 0x03, 0x55, 0x1D, 0x14, 0x04,
        0x03, 0x02, 0x01, 0x02, 0x30, 0x0D, 0x06, 0x09,
        0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01,
        0x0B, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00,
        0xB3, 0x6F, 0xED, 0x72, 0xD2, 0x73, 0x6A, 0x77,
        0xBF, 0x3A, 0x55, 0xBC, 0x54, 0x18, 0x6A, 0x71,
        0xBC, 0x6A, 0xCC, 0xCD, 0x5D, 0x90, 0xF5, 0x64,
        0x8D, 0x1B, 0xF0, 0xE0, 0x48, 0x7B, 0xF2, 0x7B,
        0x06, 0x86, 0x53, 0x63, 0x9B, 0xD8, 0x24, 0x15,
        0x10, 0xB1, 0x19, 0x96, 0x9B, 0xD2, 0x75, 0xA8,
        0x25, 0xA2, 0x35, 0xA9, 0x14, 0xD6, 0xD5, 0x5E,
        0x53, 0xE3, 0x34, 0x9D, 0xF2, 0x8B, 0x07, 0x19,
        0x9B, 0x1F, 0xF1, 0x02, 0x0F, 0x04, 0x46, 0xE8,
        0xB8, 0xB6, 0xF2, 0x8D, 0xC7, 0xC0, 0x15, 0x3E,
        0x3E, 0x8E, 0x96, 0x73, 0x15, 0x1E, 0x62, 0xF6,
        0x4E, 0x2A, 0xF7, 0xAA, 0xA0, 0x91, 0x80, 0x12,
        0x7F, 0x81, 0x0C, 0x65, 0xCC, 0x38, 0xBE, 0x58,
        0x6C, 0x14, 0xA5, 0x21, 0xA1, 0x8D, 0xF7, 0x8A,
        0xB9, 0x24, 0xF4, 0x2D, 0xCA, 0xC0, 0x67, 0x43,
        0x0B, 0xC8, 0x1C, 0xB4, 0x7D, 0x12, 0x7F, 0xA2,
        0x1B, 0x19, 0x0E, 0x94, 0xCF, 0x7B, 0x9F, 0x75,
        0xA0, 0x08, 0x9A, 0x67, 0x3F, 0x87, 0x89, 0x3E,
        0xF8, 0x58, 0xA5, 0x8A, 0x1B, 0x2D, 0xDA, 0x9B,
        0xD0, 0x1B, 0x18, 0x92, 0xC3, 0xD2, 0x6A, 0xD7,
        0x1C, 0xFC, 0x45, 0x69, 0x77, 0xC3, 0x57, 0x65,
        0x75, 0x99, 0x9E, 0x47, 0x2A, 0x20, 0x25, 0xEF,
        0x90, 0xF2, 0x5F, 0x3B, 0x7D, 0x9C, 0x7D, 0x00,
        0xEA, 0x92, 0x54, 0xEB, 0x0B, 0xE7, 0x17, 0xAF,
        0x24, 0x1A, 0xF9, 0x7C, 0x83, 0x50, 0x68, 0x1D,
        0xDC, 0x5B, 0x60, 0x12, 0xA7, 0x52, 0x78, 0xD9,
        0xA9, 0xB0, 0x1F, 0x59, 0x48, 0x36, 0xC7, 0xA6,
        0x97, 0x34, 0xC7, 0x87, 0x3F, 0xAE, 0xFD, 0xA9,
        0x56, 0x5D, 0x48, 0xCC, 0x89, 0x7A, 0x79, 0x60,
        0x8F, 0x9B, 0x2B, 0x63, 0x3C, 0xB3, 0x04, 0x1D,
        0x5F, 0xF7, 0x20, 0xD2, 0xFD, 0xF2, 0x51, 0xB1,
        0x96, 0x93, 0x13, 0x5B, 0xAB, 0x74, 0x82, 0x8B
    };

    WOLFSSL_CERT_MANAGER* cm = NULL;

    ExpectNotNull(cm = wolfSSL_CertManagerNew());

    ExpectIntEQ(wolfSSL_CertManagerEnableCRL(NULL, 0), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerEnableCRL(cm, WOLFSSL_CRL_CHECKALL), 1);
    ExpectIntEQ(wolfSSL_CertManagerEnableCRL(cm, WOLFSSL_CRL_CHECK), 1);
    ExpectIntEQ(wolfSSL_CertManagerEnableCRL(cm,
        WOLFSSL_CRL_CHECK | WOLFSSL_CRL_CHECKALL), 1);
    ExpectIntEQ(wolfSSL_CertManagerEnableCRL(cm, 16), 1);
    ExpectIntEQ(wolfSSL_CertManagerEnableCRL(cm, WOLFSSL_CRL_CHECKALL), 1);

    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(NULL, NULL, -1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(cm, NULL, -1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(NULL, server_cert_der_2048, -1),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(NULL, NULL, 1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(NULL, server_cert_der_2048, 1),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(cm, NULL, 1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(cm, server_cert_der_2048, -1),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(cm, server_cert_der_2048,
        sizeof_server_cert_der_2048), WC_NO_ERR_TRACE(ASN_NO_SIGNER_E));

    ExpectIntEQ(wolfSSL_CertManagerSetCRL_Cb(NULL, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerSetCRL_Cb(cm, NULL), 1);
#ifdef HAVE_CRL_IO
    ExpectIntEQ(wolfSSL_CertManagerSetCRL_IOCb(NULL, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerSetCRL_IOCb(cm, NULL), 1);
#endif

#ifndef NO_FILESYSTEM
    ExpectIntEQ(wolfSSL_CertManagerLoadCRL(NULL, NULL, WOLFSSL_FILETYPE_ASN1,
        0), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerLoadCRL(cm, NULL, WOLFSSL_FILETYPE_ASN1,
        0), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    /* -1 seen as !WOLFSSL_FILETYPE_PEM */
    ExpectIntEQ(wolfSSL_CertManagerLoadCRL(cm, "./certs/crl", -1, 0), 1);

    ExpectIntEQ(wolfSSL_CertManagerLoadCRLFile(NULL, NULL,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLFile(cm, NULL, WOLFSSL_FILETYPE_ASN1),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    /* -1 seen as !WOLFSSL_FILETYPE_PEM */
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLFile(cm, "./certs/crl/crl.pem", -1),
        WC_NO_ERR_TRACE(ASN_PARSE_E));
#endif

    ExpectIntEQ(wolfSSL_CertManagerLoadCRLBuffer(NULL, NULL, -1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLBuffer(cm, NULL, -1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLBuffer(NULL, crl_buff, -1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLBuffer(NULL, NULL, 1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLBuffer(NULL, crl_buff, 1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLBuffer(cm, NULL, 1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLBuffer(cm, crl_buff, -1,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    ExpectIntEQ(wolfSSL_CertManagerFreeCRL(NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    DoExpectIntEQ(wolfSSL_CertManagerFreeCRL(cm), 1);

    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCA(cm, ca_cert, NULL));
    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCRL(cm, crl1, WOLFSSL_FILETYPE_PEM, 0));
    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCRL(cm, crl2, WOLFSSL_FILETYPE_PEM, 0));
    wolfSSL_CertManagerFreeCRL(cm);

#ifndef WOLFSSL_CRL_ALLOW_MISSING_CDP
    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCRL(cm, crl1, WOLFSSL_FILETYPE_PEM, 0));
    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCA(cm, ca_cert, NULL));
    ExpectIntEQ(wolfSSL_CertManagerCheckCRL(cm, server_cert_der_2048,
        sizeof_server_cert_der_2048), WC_NO_ERR_TRACE(CRL_MISSING));
    ExpectIntEQ(wolfSSL_CertManagerVerifyBuffer(cm, server_cert_der_2048,
        sizeof_server_cert_der_2048, WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(CRL_MISSING));
#endif /* !WOLFSSL_CRL_ALLOW_MISSING_CDP */

    ExpectIntEQ(wolfSSL_CertManagerLoadCRLBuffer(cm, crl_buff, sizeof(crl_buff),
        WOLFSSL_FILETYPE_ASN1), 1);

#if !defined(NO_FILESYSTEM) && defined(WC_RSA_PSS)
    /* loading should fail without the CA set */
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLFile(cm, crl_rsapss,
        WOLFSSL_FILETYPE_PEM), WC_NO_ERR_TRACE(ASN_CRL_NO_SIGNER_E));

    /* now successfully load the RSA-PSS crl once loading in it's CA */
    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCA(cm, ca_rsapss, NULL));
    ExpectIntEQ(wolfSSL_CertManagerLoadCRLFile(cm, crl_rsapss,
        WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
#endif

    wolfSSL_CertManagerFree(cm);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CertManagerCheckOCSPResponse(void)
{
    EXPECT_DECLS;
#if defined(HAVE_OCSP) && !defined(NO_RSA) && !defined(NO_SHA)
/* Need one of these for wolfSSL_OCSP_REQUEST_new. */
#if defined(OPENSSL_ALL) || defined(WOLFSSL_NGINX) || \
    defined(WOLFSSL_HAPROXY) || defined(WOLFSSL_APACHE_HTTPD) || \
    defined(HAVE_LIGHTY)
    WOLFSSL_CERT_MANAGER* cm = NULL;
    /* Raw OCSP response bytes captured using the following setup:
     * - Run responder with
     *      openssl ocsp -port 9999 -ndays 9999
     *      -index certs/ocsp/index-intermediate1-ca-issued-certs.txt
     *      -rsigner certs/ocsp/ocsp-responder-cert.pem
     *      -rkey certs/ocsp/ocsp-responder-key.pem
     *      -CA certs/ocsp/intermediate1-ca-cert.pem
     * - Run client with
     *      openssl ocsp -host 127.0.0.1:9999 -respout resp.out
     *      -issuer certs/ocsp/intermediate1-ca-cert.pem
     *      -cert certs/ocsp/server1-cert.pem
     *      -CAfile certs/ocsp/root-ca-cert.pem -noverify
     * - Select the response packet in Wireshark, and export it using
     *   "File->Export Packet Dissection->As "C" Arrays".  Select "Selected
     *   packets only".  After importing into the editor, remove the initial
     *   ~148 bytes of header, ending with the Content-Length and the \r\n\r\n.
     */
    static const byte response[] = {
        0x30, 0x82, 0x07, 0x40, /* ....0..@ */
        0x0a, 0x01, 0x00, 0xa0, 0x82, 0x07, 0x39, 0x30, /* ......90 */
        0x82, 0x07, 0x35, 0x06, 0x09, 0x2b, 0x06, 0x01, /* ..5..+.. */
        0x05, 0x05, 0x07, 0x30, 0x01, 0x01, 0x04, 0x82, /* ...0.... */
        0x07, 0x26, 0x30, 0x82, 0x07, 0x22, 0x30, 0x82, /* .&0.."0. */
        0x01, 0x40, 0xa1, 0x81, 0xa1, 0x30, 0x81, 0x9e, /* .@...0.. */
        0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, /* 1.0...U. */
        0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x13, 0x30, /* ...US1.0 */
        0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0a, /* ...U.... */
        0x57, 0x61, 0x73, 0x68, 0x69, 0x6e, 0x67, 0x74, /* Washingt */
        0x6f, 0x6e, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, /* on1.0... */
        0x55, 0x04, 0x07, 0x0c, 0x07, 0x53, 0x65, 0x61, /* U....Sea */
        0x74, 0x74, 0x6c, 0x65, 0x31, 0x10, 0x30, 0x0e, /* ttle1.0. */
        0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x07, 0x77, /* ..U....w */
        0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, 0x31, 0x14, /* olfSSL1. */
        0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, /* 0...U... */
        0x0b, 0x45, 0x6e, 0x67, 0x69, 0x6e, 0x65, 0x65, /* .Enginee */
        0x72, 0x69, 0x6e, 0x67, 0x31, 0x1f, 0x30, 0x1d, /* ring1.0. */
        0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x16, 0x77, /* ..U....w */
        0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, 0x20, 0x4f, /* olfSSL O */
        0x43, 0x53, 0x50, 0x20, 0x52, 0x65, 0x73, 0x70, /* CSP Resp */
        0x6f, 0x6e, 0x64, 0x65, 0x72, 0x31, 0x1f, 0x30, /* onder1.0 */
        0x1d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, /* ...*.H.. */
        0x0d, 0x01, 0x09, 0x01, 0x16, 0x10, 0x69, 0x6e, /* ......in */
        0x66, 0x6f, 0x40, 0x77, 0x6f, 0x6c, 0x66, 0x73, /* fo@wolfs */
        0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x18, 0x0f, /* sl.com.. */
        0x32, 0x30, 0x32, 0x34, 0x31, 0x32, 0x32, 0x30, /* 20241220 */
        0x31, 0x37, 0x30, 0x37, 0x30, 0x34, 0x5a, 0x30, /* 170704Z0 */
        0x64, 0x30, 0x62, 0x30, 0x3a, 0x30, 0x09, 0x06, /* d0b0:0.. */
        0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1a, 0x05, 0x00, /* .+...... */
        0x04, 0x14, 0x71, 0x4d, 0x82, 0x23, 0x40, 0x59, /* ..qM.#@Y */
        0xc0, 0x96, 0xa1, 0x37, 0x43, 0xfa, 0x31, 0xdb, /* ...7C.1. */
        0xba, 0xb1, 0x43, 0x18, 0xda, 0x04, 0x04, 0x14, /* ..C..... */
        0x83, 0xc6, 0x3a, 0x89, 0x2c, 0x81, 0xf4, 0x02, /* ..:.,... */
        0xd7, 0x9d, 0x4c, 0xe2, 0x2a, 0xc0, 0x71, 0x82, /* ..L.*.q. */
        0x64, 0x44, 0xda, 0x0e, 0x02, 0x01, 0x05, 0x80, /* dD...... */
        0x00, 0x18, 0x0f, 0x32, 0x30, 0x32, 0x34, 0x31, /* ...20241 */
        0x32, 0x32, 0x30, 0x31, 0x37, 0x30, 0x37, 0x30, /* 22017070 */
        0x34, 0x5a, 0xa0, 0x11, 0x18, 0x0f, 0x32, 0x30, /* 4Z....20 */
        0x35, 0x32, 0x30, 0x35, 0x30, 0x36, 0x31, 0x37, /* 52050617 */
        0x30, 0x37, 0x30, 0x34, 0x5a, 0xa1, 0x23, 0x30, /* 0704Z.#0 */
        0x21, 0x30, 0x1f, 0x06, 0x09, 0x2b, 0x06, 0x01, /* !0...+.. */
        0x05, 0x05, 0x07, 0x30, 0x01, 0x02, 0x04, 0x12, /* ...0.... */
        0x04, 0x10, 0x12, 0x7c, 0x27, 0xbd, 0x22, 0x28, /* ...|'."( */
        0x5e, 0x62, 0x81, 0xed, 0x6d, 0x2c, 0x2d, 0x59, /* ^b..m,-Y */
        0x42, 0xd7, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, /* B.0...*. */
        0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, /* H....... */
        0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x6c, 0xce, /* ......l. */
        0xa8, 0xe8, 0xfe, 0xaf, 0x33, 0xe2, 0xce, 0x4e, /* ....3..N */
        0x63, 0x8d, 0x61, 0x16, 0x0f, 0x70, 0xb2, 0x0c, /* c.a..p.. */
        0x9a, 0xe3, 0x01, 0xd5, 0xca, 0xe5, 0x9b, 0x70, /* .......p */
        0x81, 0x6f, 0x94, 0x09, 0xe8, 0x88, 0x98, 0x1a, /* .o...... */
        0x67, 0xa0, 0xc2, 0xe7, 0x8f, 0x9b, 0x5f, 0x13, /* g....._. */
        0x17, 0x8d, 0x93, 0x8c, 0x31, 0x61, 0x7d, 0x72, /* ....1a}r */
        0x34, 0xbd, 0x21, 0x48, 0xca, 0xb2, 0xc9, 0xae, /* 4.!H.... */
        0x28, 0x5f, 0x97, 0x19, 0xcb, 0xdf, 0xed, 0xd4, /* (_...... */
        0x6e, 0x89, 0x30, 0x89, 0x11, 0xd1, 0x05, 0x08, /* n.0..... */
        0x81, 0xe9, 0xa7, 0xba, 0xf7, 0x16, 0x0c, 0xbe, /* ........ */
        0x48, 0x2e, 0xc0, 0x05, 0xac, 0x90, 0xc2, 0x35, /* H......5 */
        0xce, 0x6c, 0x94, 0x5d, 0x2b, 0xad, 0x4f, 0x19, /* .l.]+.O. */
        0xea, 0x7b, 0xd9, 0x4f, 0x49, 0x20, 0x8d, 0x98, /* .{.OI .. */
        0xa9, 0xe4, 0x53, 0x6d, 0xca, 0x34, 0xdb, 0x4a, /* ..Sm.4.J */
        0x28, 0xb3, 0x33, 0xfb, 0xfd, 0xcc, 0x4b, 0xfa, /* (.3...K. */
        0xdb, 0x70, 0xe1, 0x96, 0xc8, 0xd4, 0xf1, 0x85, /* .p...... */
        0x99, 0xaf, 0x06, 0xeb, 0xfd, 0x96, 0x21, 0x86, /* ......!. */
        0x81, 0xee, 0xcf, 0xd2, 0xf4, 0x83, 0xc9, 0x1d, /* ........ */
        0x8f, 0x42, 0xd1, 0xc1, 0xbc, 0x50, 0x0a, 0xfb, /* .B...P.. */
        0x95, 0x39, 0x4c, 0x36, 0xa8, 0xfe, 0x2b, 0x8e, /* .9L6..+. */
        0xc5, 0xb5, 0xe0, 0xab, 0xdb, 0xc0, 0xbf, 0x1d, /* ........ */
        0x35, 0x4d, 0xc0, 0x52, 0xfb, 0x08, 0x04, 0x4c, /* 5M.R...L */
        0x98, 0xf0, 0xb5, 0x5b, 0xff, 0x99, 0x74, 0xce, /* ...[..t. */
        0xb7, 0xc9, 0xe3, 0xe5, 0x70, 0x2e, 0xd3, 0x1d, /* ....p... */
        0x46, 0x38, 0xf9, 0x51, 0x17, 0x73, 0xd1, 0x08, /* F8.Q.s.. */
        0x8d, 0x3d, 0x12, 0x47, 0xd0, 0x66, 0x77, 0xaf, /* .=.G.fw. */
        0xfd, 0x4c, 0x75, 0x1f, 0xe9, 0x6c, 0xf4, 0x5a, /* .Lu..l.Z */
        0xde, 0xec, 0x37, 0xc7, 0xc4, 0x0a, 0xbe, 0x91, /* ..7..... */
        0xbc, 0x05, 0x08, 0x86, 0x47, 0x30, 0x2a, 0xc6, /* ....G0*. */
        0x85, 0x4b, 0x55, 0x6c, 0xef, 0xdf, 0x2d, 0x5a, /* .KUl..-Z */
        0xf7, 0x5b, 0xb5, 0xba, 0xed, 0x38, 0xb0, 0xcb, /* .[...8.. */
        0xeb, 0x7e, 0x84, 0x3a, 0x69, 0x2c, 0xa0, 0x82, /* .~.:i,.. */
        0x04, 0xc6, 0x30, 0x82, 0x04, 0xc2, 0x30, 0x82, /* ..0...0. */
        0x04, 0xbe, 0x30, 0x82, 0x03, 0xa6, 0xa0, 0x03, /* ..0..... */
        0x02, 0x01, 0x02, 0x02, 0x01, 0x04, 0x30, 0x0d, /* ......0. */
        0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, /* ..*.H... */
        0x01, 0x01, 0x0b, 0x05, 0x00, 0x30, 0x81, 0x97, /* .....0.. */
        0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, /* 1.0...U. */
        0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x13, 0x30, /* ...US1.0 */
        0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0c, 0x0a, /* ...U.... */
        0x57, 0x61, 0x73, 0x68, 0x69, 0x6e, 0x67, 0x74, /* Washingt */
        0x6f, 0x6e, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, /* on1.0... */
        0x55, 0x04, 0x07, 0x0c, 0x07, 0x53, 0x65, 0x61, /* U....Sea */
        0x74, 0x74, 0x6c, 0x65, 0x31, 0x10, 0x30, 0x0e, /* ttle1.0. */
        0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x07, 0x77, /* ..U....w */
        0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, 0x31, 0x14, /* olfSSL1. */
        0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x0c, /* 0...U... */
        0x0b, 0x45, 0x6e, 0x67, 0x69, 0x6e, 0x65, 0x65, /* .Enginee */
        0x72, 0x69, 0x6e, 0x67, 0x31, 0x18, 0x30, 0x16, /* ring1.0. */
        0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x77, /* ..U....w */
        0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, 0x20, 0x72, /* olfSSL r */
        0x6f, 0x6f, 0x74, 0x20, 0x43, 0x41, 0x31, 0x1f, /* oot CA1. */
        0x30, 0x1d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, /* 0...*.H. */
        0xf7, 0x0d, 0x01, 0x09, 0x01, 0x16, 0x10, 0x69, /* .......i */
        0x6e, 0x66, 0x6f, 0x40, 0x77, 0x6f, 0x6c, 0x66, /* nfo@wolf */
        0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x30, /* ssl.com0 */
        0x1e, 0x17, 0x0d, 0x32, 0x34, 0x31, 0x32, 0x31, /* ...24121 */
        0x38, 0x32, 0x31, 0x32, 0x35, 0x33, 0x31, 0x5a, /* 8212531Z */
        0x17, 0x0d, 0x32, 0x37, 0x30, 0x39, 0x31, 0x34, /* ..270914 */
        0x32, 0x31, 0x32, 0x35, 0x33, 0x31, 0x5a, 0x30, /* 212531Z0 */
        0x81, 0x9e, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, /* ..1.0... */
        0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, /* U....US1 */
        0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, /* .0...U.. */
        0x0c, 0x0a, 0x57, 0x61, 0x73, 0x68, 0x69, 0x6e, /* ..Washin */
        0x67, 0x74, 0x6f, 0x6e, 0x31, 0x10, 0x30, 0x0e, /* gton1.0. */
        0x06, 0x03, 0x55, 0x04, 0x07, 0x0c, 0x07, 0x53, /* ..U....S */
        0x65, 0x61, 0x74, 0x74, 0x6c, 0x65, 0x31, 0x10, /* eattle1. */
        0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, /* 0...U... */
        0x07, 0x77, 0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, /* .wolfSSL */
        0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, /* 1.0...U. */
        0x0b, 0x0c, 0x0b, 0x45, 0x6e, 0x67, 0x69, 0x6e, /* ...Engin */
        0x65, 0x65, 0x72, 0x69, 0x6e, 0x67, 0x31, 0x1f, /* eering1. */
        0x30, 0x1d, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, /* 0...U... */
        0x16, 0x77, 0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, /* .wolfSSL */
        0x20, 0x4f, 0x43, 0x53, 0x50, 0x20, 0x52, 0x65, /*  OCSP Re */
        0x73, 0x70, 0x6f, 0x6e, 0x64, 0x65, 0x72, 0x31, /* sponder1 */
        0x1f, 0x30, 0x1d, 0x06, 0x09, 0x2a, 0x86, 0x48, /* .0...*.H */
        0x86, 0xf7, 0x0d, 0x01, 0x09, 0x01, 0x16, 0x10, /* ........ */
        0x69, 0x6e, 0x66, 0x6f, 0x40, 0x77, 0x6f, 0x6c, /* info@wol */
        0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, /* fssl.com */
        0x30, 0x82, 0x01, 0x22, 0x30, 0x0d, 0x06, 0x09, /* 0.."0... */
        0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, /* *.H..... */
        0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, /* ........ */
        0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, /* 0....... */
        0x00, 0xb8, 0xba, 0x23, 0xb4, 0xf6, 0xc3, 0x7b, /* ...#...{ */
        0x14, 0xc3, 0xa4, 0xf5, 0x1d, 0x61, 0xa1, 0xf5, /* .....a.. */
        0x1e, 0x63, 0xb9, 0x85, 0x23, 0x34, 0x50, 0x6d, /* .c..#4Pm */
        0xf8, 0x7c, 0xa2, 0x8a, 0x04, 0x8b, 0xd5, 0x75, /* .|.....u */
        0x5c, 0x2d, 0xf7, 0x63, 0x88, 0xd1, 0x07, 0x7a, /* \-.c...z */
        0xea, 0x0b, 0x45, 0x35, 0x2b, 0xeb, 0x1f, 0xb1, /* ..E5+... */
        0x22, 0xb4, 0x94, 0x41, 0x38, 0xe2, 0x9d, 0x74, /* "..A8..t */
        0xd6, 0x8b, 0x30, 0x22, 0x10, 0x51, 0xc5, 0xdb, /* ..0".Q.. */
        0xca, 0x3f, 0x46, 0x2b, 0xfe, 0xe5, 0x5a, 0x3f, /* .?F+..Z? */
        0x41, 0x74, 0x67, 0x75, 0x95, 0xa9, 0x94, 0xd5, /* Atgu.... */
        0xc3, 0xee, 0x42, 0xf8, 0x8d, 0xeb, 0x92, 0x95, /* ..B..... */
        0xe1, 0xd9, 0x65, 0xb7, 0x43, 0xc4, 0x18, 0xde, /* ..e.C... */
        0x16, 0x80, 0x90, 0xce, 0x24, 0x35, 0x21, 0xc4, /* ....$5!. */
        0x55, 0xac, 0x5a, 0x51, 0xe0, 0x2e, 0x2d, 0xb3, /* U.ZQ..-. */
        0x0a, 0x5a, 0x4f, 0x4a, 0x73, 0x31, 0x50, 0xee, /* .ZOJs1P. */
        0x4a, 0x16, 0xbd, 0x39, 0x8b, 0xad, 0x05, 0x48, /* J..9...H */
        0x87, 0xb1, 0x99, 0xe2, 0x10, 0xa7, 0x06, 0x72, /* .......r */
        0x67, 0xca, 0x5c, 0xd1, 0x97, 0xbd, 0xc8, 0xf1, /* g.\..... */
        0x76, 0xf8, 0xe0, 0x4a, 0xec, 0xbc, 0x93, 0xf4, /* v..J.... */
        0x66, 0x4c, 0x28, 0x71, 0xd1, 0xd8, 0x66, 0x03, /* fL(q..f. */
        0xb4, 0x90, 0x30, 0xbb, 0x17, 0xb0, 0xfe, 0x97, /* ..0..... */
        0xf5, 0x1e, 0xe8, 0xc7, 0x5d, 0x9b, 0x8b, 0x11, /* ....]... */
        0x19, 0x12, 0x3c, 0xab, 0x82, 0x71, 0x78, 0xff, /* ..<..qx. */
        0xae, 0x3f, 0x32, 0xb2, 0x08, 0x71, 0xb2, 0x1b, /* .?2..q.. */
        0x8c, 0x27, 0xac, 0x11, 0xb8, 0xd8, 0x43, 0x49, /* .'....CI */
        0xcf, 0xb0, 0x70, 0xb1, 0xf0, 0x8c, 0xae, 0xda, /* ..p..... */
        0x24, 0x87, 0x17, 0x3b, 0xd8, 0x04, 0x65, 0x6c, /* $..;..el */
        0x00, 0x76, 0x50, 0xef, 0x15, 0x08, 0xd7, 0xb4, /* .vP..... */
        0x73, 0x68, 0x26, 0x14, 0x87, 0x95, 0xc3, 0x5f, /* sh&...._ */
        0x6e, 0x61, 0xb8, 0x87, 0x84, 0xfa, 0x80, 0x1a, /* na...... */
        0x0a, 0x8b, 0x98, 0xf3, 0xe3, 0xff, 0x4e, 0x44, /* ......ND */
        0x1c, 0x65, 0x74, 0x7c, 0x71, 0x54, 0x65, 0xe5, /* .et|qTe. */
        0x39, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x82, /* 9....... */
        0x01, 0x0a, 0x30, 0x82, 0x01, 0x06, 0x30, 0x09, /* ..0...0. */
        0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x02, 0x30, /* ..U....0 */
        0x00, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, /* .0...U.. */
        0x04, 0x16, 0x04, 0x14, 0x32, 0x67, 0xe1, 0xb1, /* ....2g.. */
        0x79, 0xd2, 0x81, 0xfc, 0x9f, 0x23, 0x0c, 0x70, /* y....#.p */
        0x40, 0x50, 0xb5, 0x46, 0x56, 0xb8, 0x30, 0x36, /* @P.FV.06 */
        0x30, 0x81, 0xc4, 0x06, 0x03, 0x55, 0x1d, 0x23, /* 0....U.# */
        0x04, 0x81, 0xbc, 0x30, 0x81, 0xb9, 0x80, 0x14, /* ...0.... */
        0x73, 0xb0, 0x1c, 0xa4, 0x2f, 0x82, 0xcb, 0xcf, /* s.../... */
        0x47, 0xa5, 0x38, 0xd7, 0xb0, 0x04, 0x82, 0x3a, /* G.8....: */
        0x7e, 0x72, 0x15, 0x21, 0xa1, 0x81, 0x9d, 0xa4, /* ~r.!.... */
        0x81, 0x9a, 0x30, 0x81, 0x97, 0x31, 0x0b, 0x30, /* ..0..1.0 */
        0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, /* ...U.... */
        0x55, 0x53, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, /* US1.0... */
        0x55, 0x04, 0x08, 0x0c, 0x0a, 0x57, 0x61, 0x73, /* U....Was */
        0x68, 0x69, 0x6e, 0x67, 0x74, 0x6f, 0x6e, 0x31, /* hington1 */
        0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07, /* .0...U.. */
        0x0c, 0x07, 0x53, 0x65, 0x61, 0x74, 0x74, 0x6c, /* ..Seattl */
        0x65, 0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, /* e1.0...U */
        0x04, 0x0a, 0x0c, 0x07, 0x77, 0x6f, 0x6c, 0x66, /* ....wolf */
        0x53, 0x53, 0x4c, 0x31, 0x14, 0x30, 0x12, 0x06, /* SSL1.0.. */
        0x03, 0x55, 0x04, 0x0b, 0x0c, 0x0b, 0x45, 0x6e, /* .U....En */
        0x67, 0x69, 0x6e, 0x65, 0x65, 0x72, 0x69, 0x6e, /* gineerin */
        0x67, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, /* g1.0...U */
        0x04, 0x03, 0x0c, 0x0f, 0x77, 0x6f, 0x6c, 0x66, /* ....wolf */
        0x53, 0x53, 0x4c, 0x20, 0x72, 0x6f, 0x6f, 0x74, /* SSL root */
        0x20, 0x43, 0x41, 0x31, 0x1f, 0x30, 0x1d, 0x06, /*  CA1.0.. */
        0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, /* .*.H.... */
        0x09, 0x01, 0x16, 0x10, 0x69, 0x6e, 0x66, 0x6f, /* ....info */
        0x40, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, /* @wolfssl */
        0x2e, 0x63, 0x6f, 0x6d, 0x82, 0x01, 0x63, 0x30, /* .com..c0 */
        0x13, 0x06, 0x03, 0x55, 0x1d, 0x25, 0x04, 0x0c, /* ...U.%.. */
        0x30, 0x0a, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, /* 0...+... */
        0x05, 0x07, 0x03, 0x09, 0x30, 0x0d, 0x06, 0x09, /* ....0... */
        0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, /* *.H..... */
        0x0b, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, /* ........ */
        0x4d, 0xa2, 0xd8, 0x55, 0xe0, 0x2b, 0xf4, 0xad, /* M..U.+.. */
        0x65, 0xe2, 0x92, 0x35, 0xcb, 0x60, 0xa0, 0xa2, /* e..5.`.. */
        0x6b, 0xa6, 0x88, 0xc1, 0x86, 0x58, 0x57, 0x37, /* k....XW7 */
        0xbd, 0x2e, 0x28, 0x6e, 0x1c, 0x56, 0x2a, 0x35, /* ..(n.V*5 */
        0xde, 0xff, 0x3e, 0x8e, 0x3d, 0x47, 0x21, 0x1a, /* ..>.=G!. */
        0xe9, 0xd3, 0xc6, 0xb4, 0xe2, 0xcb, 0x3e, 0xc6, /* ......>. */
        0xaf, 0x9b, 0xef, 0x23, 0x88, 0x56, 0x95, 0x73, /* ...#.V.s */
        0x2e, 0xb3, 0xed, 0xc5, 0x11, 0x4b, 0x69, 0xf7, /* .....Ki. */
        0x13, 0x3a, 0x05, 0xe1, 0xaf, 0xba, 0xc9, 0x59, /* .:.....Y */
        0xfd, 0xe2, 0xa0, 0x81, 0xa0, 0x4c, 0x0c, 0x2c, /* .....L., */
        0xcb, 0x57, 0xad, 0x96, 0x3a, 0x8c, 0x32, 0xa6, /* .W..:.2. */
        0x4a, 0xf8, 0x72, 0xb8, 0xec, 0xb3, 0x26, 0x69, /* J.r...&i */
        0xd6, 0x6a, 0x4c, 0x4c, 0x78, 0x18, 0x3c, 0xca, /* .jLLx.<. */
        0x19, 0xf1, 0xb5, 0x8e, 0x23, 0x81, 0x5b, 0x27, /* ....#.[' */
        0x90, 0xe0, 0x5c, 0x2b, 0x17, 0x4d, 0x78, 0x99, /* ..\+.Mx. */
        0x6b, 0x25, 0xbd, 0x2f, 0xae, 0x1b, 0xaa, 0xce, /* k%./.... */
        0x84, 0xb9, 0x44, 0x21, 0x46, 0xc0, 0x34, 0x6b, /* ..D!F.4k */
        0x5b, 0xb9, 0x1b, 0xca, 0x5c, 0x60, 0xf1, 0xef, /* [...\`.. */
        0xe6, 0x66, 0xbc, 0x84, 0x63, 0x56, 0x50, 0x7d, /* .f..cVP} */
        0xbb, 0x2c, 0x2f, 0x7b, 0x47, 0xb4, 0xfd, 0x58, /* .,/{G..X */
        0x77, 0x87, 0xee, 0x27, 0x20, 0x96, 0x72, 0x8e, /* w..' .r. */
        0x4c, 0x7e, 0x4f, 0x93, 0xeb, 0x5f, 0x8f, 0x9c, /* L~O.._.. */
        0x1e, 0x59, 0x7a, 0x96, 0xaa, 0x53, 0x77, 0x22, /* .Yz..Sw" */
        0x41, 0xd8, 0xd3, 0xf9, 0x89, 0x8f, 0xe8, 0x9d, /* A....... */
        0x65, 0xbd, 0x0c, 0x71, 0x3c, 0xbb, 0xa3, 0x07, /* e..q<... */
        0xbf, 0xfb, 0xa8, 0xd1, 0x18, 0x0a, 0xb4, 0xc4, /* ........ */
        0xf7, 0x83, 0xb3, 0x86, 0x2b, 0xf0, 0x5b, 0x05, /* ....+.[. */
        0x28, 0xc1, 0x01, 0x31, 0x73, 0x5c, 0x2b, 0xbd, /* (..1s\+. */
        0x60, 0x97, 0xa3, 0x36, 0x82, 0x96, 0xd7, 0x83, /* `..6.... */
        0xdf, 0x75, 0xee, 0x29, 0x42, 0x97, 0x86, 0x41, /* .u.)B..A */
        0x55, 0xb9, 0x70, 0x87, 0xd5, 0x02, 0x85, 0x13, /* U.p..... */
        0x41, 0xf8, 0x25, 0x05, 0xab, 0x6a, 0xaa, 0x57  /* A.%..j.W */
    };
    OcspEntry entry[1];
    CertStatus status[1];
    OcspRequest* request = NULL;
#ifndef NO_FILESYSTEM
    const char* ca_cert = "./certs/ca-cert.pem";
#endif

    byte serial[] = {0x05};
    byte issuerHash[] = {0x71, 0x4d, 0x82, 0x23, 0x40, 0x59, 0xc0, 0x96, 0xa1, 0x37, 0x43, 0xfa, 0x31, 0xdb, 0xba, 0xb1, 0x43, 0x18, 0xda, 0x04};
    byte issuerKeyHash[] = {0x83, 0xc6, 0x3a, 0x89, 0x2c, 0x81, 0xf4, 0x02, 0xd7, 0x9d, 0x4c, 0xe2, 0x2a, 0xc0, 0x71, 0x82, 0x64, 0x44, 0xda, 0x0e};


    XMEMSET(entry, 0, sizeof(OcspEntry));
    XMEMSET(status, 0, sizeof(CertStatus));

    ExpectNotNull(request = wolfSSL_OCSP_REQUEST_new());
    ExpectNotNull(request->serial = (byte*)XMALLOC(sizeof(serial), NULL,
                                                   DYNAMIC_TYPE_OCSP_REQUEST));

    if ((request != NULL) && (request->serial != NULL)) {
        request->serialSz = sizeof(serial);
        XMEMCPY(request->serial, serial, sizeof(serial));
        XMEMCPY(request->issuerHash, issuerHash, sizeof(issuerHash));
        XMEMCPY(request->issuerKeyHash, issuerKeyHash, sizeof(issuerKeyHash));
    }

    ExpectNotNull(cm = wolfSSL_CertManagerNew_ex(NULL));
    ExpectIntEQ(wolfSSL_CertManagerEnableOCSP(cm, 0), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CertManagerLoadCA(cm,
        "./certs/ocsp/intermediate1-ca-cert.pem", NULL), WOLFSSL_SUCCESS);

    /* Response should be valid. */
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(cm, (byte *)response,
        sizeof(response), NULL, status, entry, request), WOLFSSL_SUCCESS);

    /* Flip a byte in the request serial number, response should be invalid
     * now. */
    if ((request != NULL) && (request->serial != NULL))
        request->serial[0] ^= request->serial[0];
    ExpectIntNE(wolfSSL_CertManagerCheckOCSPResponse(cm, (byte *)response,
        sizeof(response), NULL, status, entry, request), WOLFSSL_SUCCESS);

#ifndef NO_FILESYSTEM
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(cm, server_cert_der_2048,
        sizeof(server_cert_der_2048)), WC_NO_ERR_TRACE(ASN_NO_SIGNER_E));
    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCA(cm, ca_cert, NULL));
    ExpectIntEQ(wolfSSL_CertManagerCheckOCSP(cm, server_cert_der_2048,
        sizeof(server_cert_der_2048)), 1);
#endif

    wolfSSL_OCSP_REQUEST_free(request);
    wolfSSL_CertManagerFree(cm);
#endif /* OPENSSL_ALL || WOLFSSL_NGINX ||  WOLFSSL_HAPROXY ||
        * WOLFSSL_APACHE_HTTPD || HAVE_LIGHTY */
#endif /* HAVE_OCSP */
    return EXPECT_RESULT();
}

static int test_wolfSSL_CheckOCSPResponse(void)
{
    EXPECT_DECLS;
#if defined(HAVE_OCSP) && defined(OPENSSL_EXTRA) && \
    !defined(NO_RSA) && !defined(NO_SHA)
    const char* responseFile = "./certs/ocsp/test-response.der";
    const char* responseMultiFile = "./certs/ocsp/test-multi-response.der";
    const char* responseNoInternFile =
        "./certs/ocsp/test-response-nointern.der";
    const char* caFile = "./certs/ocsp/root-ca-cert.pem";
    OcspResponse* res = NULL;
    byte data[4096];
    const unsigned char* pt;
    int  dataSz = 0; /* initialize to mitigate spurious maybe-uninitialized from
                      * gcc sanitizer with --enable-heapmath.
                      */
    XFILE f = XBADFILE;
    WOLFSSL_OCSP_BASICRESP* bs = NULL;
    WOLFSSL_X509_STORE* st = NULL;
    WOLFSSL_X509* issuer = NULL;


    ExpectTrue((f = XFOPEN(responseFile, "rb")) != XBADFILE);
    ExpectIntGT(dataSz = (word32)XFREAD(data, 1, sizeof(data), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }

    pt = data;
    ExpectNotNull(res = wolfSSL_d2i_OCSP_RESPONSE(NULL, &pt, dataSz));
    ExpectNotNull(issuer = wolfSSL_X509_load_certificate_file(caFile,
        SSL_FILETYPE_PEM));
    ExpectNotNull(st = wolfSSL_X509_STORE_new());
    ExpectIntEQ(wolfSSL_X509_STORE_add_cert(st, issuer), WOLFSSL_SUCCESS);
    ExpectNotNull(bs = wolfSSL_OCSP_response_get1_basic(res));
    ExpectIntEQ(wolfSSL_OCSP_basic_verify(bs, NULL, st, 0), WOLFSSL_SUCCESS);
    wolfSSL_OCSP_BASICRESP_free(bs);
    bs = NULL;
    wolfSSL_OCSP_RESPONSE_free(res);
    res = NULL;
    wolfSSL_X509_STORE_free(st);
    st = NULL;
    wolfSSL_X509_free(issuer);
    issuer = NULL;

    /* check loading a response with optional certs */
    ExpectTrue((f = XFOPEN(responseNoInternFile, "rb")) != XBADFILE);
    ExpectIntGT(dataSz = (word32)XFREAD(data, 1, sizeof(data), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);
    f = XBADFILE;

    pt = data;
    ExpectNotNull(res = wolfSSL_d2i_OCSP_RESPONSE(NULL, &pt, dataSz));
    wolfSSL_OCSP_RESPONSE_free(res);
    res = NULL;

    /* check loading a response with multiple certs */
    {
        WOLFSSL_CERT_MANAGER* cm = NULL;
        OcspEntry *entry = NULL;
        CertStatus* status = NULL;
        OcspRequest* request = NULL;

        byte serial1[] = {0x01};
        byte serial[] = {0x02};

        byte issuerHash[] = {
            0x44, 0xA8, 0xDB, 0xD1, 0xBC, 0x97, 0x0A, 0x83,
            0x3B, 0x5B, 0x31, 0x9A, 0x4C, 0xB8, 0xD2, 0x52,
            0x37, 0x15, 0x8A, 0x88
        };
        byte issuerKeyHash[] = {
            0x73, 0xB0, 0x1C, 0xA4, 0x2F, 0x82, 0xCB, 0xCF,
            0x47, 0xA5, 0x38, 0xD7, 0xB0, 0x04, 0x82, 0x3A,
            0x7E, 0x72, 0x15, 0x21
        };

        ExpectNotNull(entry = (OcspEntry*)XMALLOC(sizeof(OcspEntry), NULL,
             DYNAMIC_TYPE_OPENSSL));

        ExpectNotNull(status = (CertStatus*)XMALLOC(sizeof(CertStatus), NULL,
             DYNAMIC_TYPE_OPENSSL));

        if (entry != NULL)
            XMEMSET(entry, 0, sizeof(OcspEntry));
        if (status != NULL)
            XMEMSET(status, 0, sizeof(CertStatus));

        ExpectNotNull(request = wolfSSL_OCSP_REQUEST_new());
        ExpectNotNull(request->serial = (byte*)XMALLOC(sizeof(serial), NULL,
                                                    DYNAMIC_TYPE_OCSP_REQUEST));

        if (request != NULL && request->serial != NULL) {
             request->serialSz = sizeof(serial);
             XMEMCPY(request->serial, serial, sizeof(serial));
             XMEMCPY(request->issuerHash, issuerHash, sizeof(issuerHash));
             XMEMCPY(request->issuerKeyHash, issuerKeyHash,
                 sizeof(issuerKeyHash));
        }

        ExpectNotNull(cm = wolfSSL_CertManagerNew_ex(NULL));
        ExpectIntEQ(wolfSSL_CertManagerEnableOCSP(cm, 0), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CertManagerLoadCA(cm, caFile, NULL),
            WOLFSSL_SUCCESS);

        ExpectTrue((f = XFOPEN(responseMultiFile, "rb")) != XBADFILE);
        ExpectIntGT(dataSz = (word32)XFREAD(data, 1, sizeof(data), f), 0);
        if (f != XBADFILE)
            XFCLOSE(f);
        f = XBADFILE;

        ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(cm, data,
            dataSz, NULL, status, entry, request), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(cm, data,
            dataSz, NULL, entry->status, entry, request), WOLFSSL_SUCCESS);
        ExpectNotNull(entry->status);

        if (request != NULL && request->serial != NULL)
            XMEMCPY(request->serial, serial1, sizeof(serial1));
        ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(cm, data,
            dataSz, NULL, status, entry, request), WOLFSSL_SUCCESS);

        /* store both status's in the entry to check that "next" is not
         * overwritten */
        if (EXPECT_SUCCESS() && status != NULL && entry != NULL) {
            status->next  = entry->status;
            entry->status = status;
        }

        if (request != NULL && request->serial != NULL)
            XMEMCPY(request->serial, serial, sizeof(serial));
        ExpectIntEQ(wolfSSL_CertManagerCheckOCSPResponse(cm, data,
            dataSz, NULL, entry->status, entry, request), WOLFSSL_SUCCESS);
        ExpectNotNull(entry->status->next);

        /* compare the status found */
        ExpectIntEQ(status->serialSz, entry->status->serialSz);
        ExpectIntEQ(XMEMCMP(status->serial, entry->status->serial,
            status->serialSz), 0);

        if (status != NULL && entry != NULL && entry->status != status) {
            XFREE(status, NULL, DYNAMIC_TYPE_OPENSSL);
        }
        wolfSSL_OCSP_CERTID_free(entry);
        wolfSSL_OCSP_REQUEST_free(request);
        wolfSSL_CertManagerFree(cm);
    }

/* FIPS v2 and below don't support long salts. */
#if defined(WC_RSA_PSS) && \
    (!defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && \
     (HAVE_FIPS_VERSION > 2))) && (!defined(HAVE_SELFTEST) || \
     (defined(HAVE_SELFTEST_VERSION) && (HAVE_SELFTEST_VERSION > 2)))
    {
        const char* responsePssFile = "./certs/ocsp/test-response-rsapss.der";

        /* check loading a response with RSA-PSS signature */
        ExpectTrue((f = XFOPEN(responsePssFile, "rb")) != XBADFILE);
        ExpectIntGT(dataSz = (word32)XFREAD(data, 1, sizeof(data), f), 0);
        if (f != XBADFILE)
            XFCLOSE(f);

        pt = data;
        ExpectNotNull(res = wolfSSL_d2i_OCSP_RESPONSE(NULL, &pt, dataSz));

        /* try to verify the response */
        ExpectNotNull(issuer = wolfSSL_X509_load_certificate_file(caFile,
            SSL_FILETYPE_PEM));
        ExpectNotNull(st = wolfSSL_X509_STORE_new());
        ExpectIntEQ(wolfSSL_X509_STORE_add_cert(st, issuer), WOLFSSL_SUCCESS);
        ExpectNotNull(bs = wolfSSL_OCSP_response_get1_basic(res));
        ExpectIntEQ(wolfSSL_OCSP_basic_verify(bs, NULL, st, 0),
            WOLFSSL_SUCCESS);
        wolfSSL_OCSP_BASICRESP_free(bs);
        wolfSSL_OCSP_RESPONSE_free(res);
        wolfSSL_X509_STORE_free(st);
        wolfSSL_X509_free(issuer);
    }
#endif
#endif /* HAVE_OCSP */
    return EXPECT_RESULT();
}

static int test_wolfSSL_FPKI(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_FPKI) && !defined(NO_RSA) && !defined(NO_FILESYSTEM)
    XFILE f = XBADFILE;
    const char* fpkiCert = "./certs/fpki-cert.der";
    const char* fpkiCertPolCert = "./certs/fpki-certpol-cert.der";
    DecodedCert cert;
    byte buf[4096];
    byte* uuid = NULL;
    byte* fascn = NULL;
    word32 fascnSz;
    word32 uuidSz;
    int bytes = 0;

    ExpectTrue((f = XFOPEN(fpkiCert, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);

    wc_InitDecodedCert(&cert, buf, (word32)bytes, NULL);
    ExpectIntEQ(wc_ParseCert(&cert, CERT_TYPE, 0, NULL), 0);
    ExpectIntEQ(wc_GetFASCNFromCert(&cert, NULL, &fascnSz), WC_NO_ERR_TRACE(LENGTH_ONLY_E));
    ExpectNotNull(fascn = (byte*)XMALLOC(fascnSz, NULL,
        DYNAMIC_TYPE_TMP_BUFFER));
    ExpectIntEQ(wc_GetFASCNFromCert(&cert, fascn, &fascnSz), 0);
    XFREE(fascn, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    fascn = NULL;

    ExpectIntEQ(wc_GetUUIDFromCert(&cert, NULL, &uuidSz), WC_NO_ERR_TRACE(LENGTH_ONLY_E));
    ExpectNotNull(uuid = (byte*)XMALLOC(uuidSz, NULL, DYNAMIC_TYPE_TMP_BUFFER));
    ExpectIntEQ(wc_GetUUIDFromCert(&cert, uuid, &uuidSz), 0);
    XFREE(uuid, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    uuid = NULL;
    wc_FreeDecodedCert(&cert);

    XMEMSET(buf, 0, 4096);
    fascnSz = uuidSz = bytes = 0;
    f = XBADFILE;

    ExpectTrue((f = XFOPEN(fpkiCertPolCert, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);

    wc_InitDecodedCert(&cert, buf, (word32)bytes, NULL);
    ExpectIntEQ(wc_ParseCert(&cert, CERT_TYPE, 0, NULL), 0);
    ExpectIntEQ(wc_GetFASCNFromCert(&cert, NULL, &fascnSz), WC_NO_ERR_TRACE(LENGTH_ONLY_E));
    ExpectNotNull(fascn = (byte*)XMALLOC(fascnSz, NULL,
        DYNAMIC_TYPE_TMP_BUFFER));
    ExpectIntEQ(wc_GetFASCNFromCert(&cert, fascn, &fascnSz), 0);
    XFREE(fascn, NULL, DYNAMIC_TYPE_TMP_BUFFER);

    ExpectIntEQ(wc_GetUUIDFromCert(&cert, NULL, &uuidSz), WC_NO_ERR_TRACE(LENGTH_ONLY_E));
    ExpectNotNull(uuid = (byte*)XMALLOC(uuidSz, NULL, DYNAMIC_TYPE_TMP_BUFFER));
    ExpectIntEQ(wc_GetUUIDFromCert(&cert, uuid, &uuidSz), 0);
    XFREE(uuid, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    wc_FreeDecodedCert(&cert);
#endif

    return EXPECT_RESULT();
}

/* use RID in confuncture with other names to test parsing of unknown other
 * names */
static int test_wolfSSL_OtherName(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA) && !defined(NO_FILESYSTEM)
    XFILE f = XBADFILE;
    const char* ridCert = "./certs/rid-cert.der";
    DecodedCert cert;
    byte buf[4096];
    int bytes = 0;

    ExpectTrue((f = XFOPEN(ridCert, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);

    wc_InitDecodedCert(&cert, buf, (word32)bytes, NULL);
    ExpectIntEQ(wc_ParseCert(&cert, CERT_TYPE, 0, NULL), 0);
    wc_FreeDecodedCert(&cert);
#endif

    return EXPECT_RESULT();
}

#ifdef HAVE_CERT_CHAIN_VALIDATION
static int test_wolfSSL_CertRsaPss(void)
{
    EXPECT_DECLS;
/* FIPS v2 and below don't support long salts. */
#if !defined(NO_RSA) && defined(WC_RSA_PSS) && !defined(NO_FILESYSTEM) && \
    (!defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && \
     (HAVE_FIPS_VERSION > 2))) && (!defined(HAVE_SELFTEST) || \
     (defined(HAVE_SELFTEST_VERSION) && (HAVE_SELFTEST_VERSION > 2)))
    XFILE f = XBADFILE;
    const char* rsaPssSha256Cert = "./certs/rsapss/ca-rsapss.der";
    const char* rsaPssRootSha256Cert = "./certs/rsapss/root-rsapss.pem";
#if defined(WOLFSSL_SHA384) && defined(WOLFSSL_PSS_LONG_SALT) && \
    RSA_MAX_SIZE >= 3072
    const char* rsaPssSha384Cert = "./certs/rsapss/ca-3072-rsapss.der";
#endif
#if defined(WOLFSSL_SHA384) && RSA_MAX_SIZE >= 3072
    const char* rsaPssRootSha384Cert = "./certs/rsapss/root-3072-rsapss.pem";
#endif
    DecodedCert cert;
    byte buf[4096];
    int bytes = 0;
    WOLFSSL_CERT_MANAGER* cm = NULL;

    ExpectNotNull(cm = wolfSSL_CertManagerNew());
    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCA(cm, rsaPssRootSha256Cert, NULL));
#if defined(WOLFSSL_SHA384) && RSA_MAX_SIZE >= 3072
    ExpectIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CertManagerLoadCA(cm, rsaPssRootSha384Cert, NULL));
#endif

    ExpectTrue((f = XFOPEN(rsaPssSha256Cert, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    wc_InitDecodedCert(&cert, buf, (word32)bytes, NULL);
    ExpectIntEQ(wc_ParseCert(&cert, CERT_TYPE, VERIFY, cm), 0);
    wc_FreeDecodedCert(&cert);

#if defined(WOLFSSL_SHA384) && defined(WOLFSSL_PSS_LONG_SALT) && \
    RSA_MAX_SIZE >= 3072
    ExpectTrue((f = XFOPEN(rsaPssSha384Cert, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);
    wc_InitDecodedCert(&cert, buf, (word32)bytes, NULL);
    ExpectIntEQ(wc_ParseCert(&cert, CERT_TYPE, VERIFY, cm), 0);
    wc_FreeDecodedCert(&cert);
#endif

    wolfSSL_CertManagerFree(cm);
#endif

    return EXPECT_RESULT();
}
#endif

static int test_wolfSSL_CTX_load_verify_locations_ex(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_CLIENT) && !defined(NO_RSA)
    WOLFSSL_CTX* ctx = NULL;
    const char* ca_cert = "./certs/ca-cert.pem";
    const char* ca_expired_cert = "./certs/test/expired/expired-ca.pem";

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));

    /* test good CA */
    ExpectTrue(WOLFSSL_SUCCESS ==
        wolfSSL_CTX_load_verify_locations_ex(ctx, ca_cert, NULL,
            WOLFSSL_LOAD_FLAG_NONE));

    /* test expired CA */
#if !defined(OPENSSL_COMPATIBLE_DEFAULTS) && !defined(NO_ASN_TIME)
    ExpectIntNE(wolfSSL_CTX_load_verify_locations_ex(ctx, ca_expired_cert, NULL,
            WOLFSSL_LOAD_FLAG_NONE), WOLFSSL_SUCCESS);
#else
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, ca_expired_cert, NULL,
            WOLFSSL_LOAD_FLAG_NONE), WOLFSSL_SUCCESS);
#endif
    ExpectIntEQ(wolfSSL_CTX_load_verify_locations_ex(ctx, ca_expired_cert, NULL,
            WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_load_verify_buffer_ex(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_RSA) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX* ctx;
    const char* ca_expired_cert_file = "./certs/test/expired/expired-ca.der";
    byte ca_expired_cert[TWOK_BUF];
    word32 sizeof_ca_expired_cert = 0;
    XFILE fp = XBADFILE;

#ifndef NO_WOLFSSL_CLIENT
    ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
#else
    ctx = wolfSSL_CTX_new(wolfSSLv23_server_method());
#endif
    ExpectNotNull(ctx);

#if defined(USE_CERT_BUFFERS_2048)
    /* test good CA */
    ExpectTrue(WOLFSSL_SUCCESS ==
        wolfSSL_CTX_load_verify_buffer_ex(ctx, ca_cert_der_2048,
            sizeof_ca_cert_der_2048, WOLFSSL_FILETYPE_ASN1, 0,
            WOLFSSL_LOAD_FLAG_NONE));
#endif

    /* load expired CA */
    XMEMSET(ca_expired_cert, 0, sizeof(ca_expired_cert));
    ExpectTrue((fp = XFOPEN(ca_expired_cert_file, "rb")) != XBADFILE);
    ExpectIntGT(sizeof_ca_expired_cert = (word32)XFREAD(ca_expired_cert, 1,
        sizeof(ca_expired_cert), fp), 0);
    if (fp != XBADFILE)
        XFCLOSE(fp);

    /* test expired CA failure */


#if !defined(OPENSSL_COMPATIBLE_DEFAULTS) && !defined(NO_ASN_TIME)
    ExpectIntNE(wolfSSL_CTX_load_verify_buffer_ex(ctx, ca_expired_cert,
            sizeof_ca_expired_cert, WOLFSSL_FILETYPE_ASN1, 0,
            WOLFSSL_LOAD_FLAG_NONE), WOLFSSL_SUCCESS);
#else
    ExpectIntEQ(wolfSSL_CTX_load_verify_buffer_ex(ctx, ca_expired_cert,
            sizeof_ca_expired_cert, WOLFSSL_FILETYPE_ASN1, 0,
            WOLFSSL_LOAD_FLAG_NONE), WOLFSSL_SUCCESS);
#endif
    /* test expired CA success */
    ExpectIntEQ(wolfSSL_CTX_load_verify_buffer_ex(ctx, ca_expired_cert,
            sizeof_ca_expired_cert, WOLFSSL_FILETYPE_ASN1, 0,
            WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY), WOLFSSL_SUCCESS);

    /* Fail when ctx is NULL. */
    ExpectIntEQ(wolfSSL_CTX_load_verify_buffer_ex(NULL, ca_expired_cert,
            sizeof_ca_expired_cert, WOLFSSL_FILETYPE_ASN1, 0,
            WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    /* Load as modified cert - bad initial length. */
    ca_expired_cert[2] = 0x7f;
    ExpectIntEQ(wolfSSL_CTX_load_verify_buffer_ex(ctx, ca_expired_cert,
            sizeof_ca_expired_cert, WOLFSSL_FILETYPE_ASN1, 1,
            WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY), WC_NO_ERR_TRACE(ASN_PARSE_E));

    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_load_verify_chain_buffer_format(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_RSA) && defined(OPENSSL_EXTRA) && \
    defined(USE_CERT_BUFFERS_2048) && (WOLFSSL_MIN_RSA_BITS <= 1024) && \
    !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX* ctx = NULL;

  #ifndef NO_WOLFSSL_CLIENT
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
  #else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
  #endif

    /* Public key 140 bytes??? */
    ExpectIntEQ(wolfSSL_CTX_load_verify_chain_buffer_format(ctx,
        ca_cert_chain_der, sizeof_ca_cert_chain_der, WOLFSSL_FILETYPE_ASN1),
        WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_add1_chain_cert(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && defined(OPENSSL_EXTRA) && \
    defined(KEEP_OUR_CERT) && !defined(NO_RSA) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX*        ctx;
    WOLFSSL*            ssl = NULL;
    const char *certChain[] = {
            "./certs/intermediate/client-int-cert.pem",
            "./certs/intermediate/ca-int2-cert.pem",
            "./certs/intermediate/ca-int-cert.pem",
            "./certs/ca-cert.pem",
            NULL
    };
    const char** cert;
    WOLFSSL_X509* x509 = NULL;
    WOLF_STACK_OF(X509)* chain = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    ExpectNotNull(x509 = wolfSSL_X509_new());
    ExpectIntEQ(SSL_CTX_add1_chain_cert(ctx, x509), 0);
    ExpectIntEQ(SSL_CTX_add0_chain_cert(ctx, x509), 0);
    ExpectIntEQ(SSL_add1_chain_cert(ssl, x509), 0);
    ExpectIntEQ(SSL_add0_chain_cert(ssl, x509), 0);
    wolfSSL_X509_free(x509);
    x509 = NULL;

    for (cert = certChain; EXPECT_SUCCESS() && *cert != NULL; cert++) {
        ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(*cert,
            WOLFSSL_FILETYPE_PEM));

        /* Do negative tests once */
        if (cert == certChain) {
            /* Negative tests. */
            ExpectIntEQ(SSL_CTX_add1_chain_cert(NULL, NULL), 0);
            ExpectIntEQ(SSL_CTX_add1_chain_cert(ctx, NULL), 0);
            ExpectIntEQ(SSL_CTX_add1_chain_cert(NULL, x509), 0);
            ExpectIntEQ(SSL_CTX_add0_chain_cert(NULL, NULL), 0);
            ExpectIntEQ(SSL_CTX_add0_chain_cert(ctx, NULL), 0);
            ExpectIntEQ(SSL_CTX_add0_chain_cert(NULL, x509), 0);
        }

        ExpectIntEQ(SSL_CTX_add1_chain_cert(ctx, x509), 1);
        X509_free(x509);
        x509 = NULL;
    }
    for (cert = certChain; EXPECT_SUCCESS() && *cert != NULL; cert++) {
        ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(*cert,
            WOLFSSL_FILETYPE_PEM));

        /* Do negative tests once */
        if (cert == certChain) {
            /* Negative tests. */
            ExpectIntEQ(SSL_add1_chain_cert(NULL, NULL), 0);
            ExpectIntEQ(SSL_add1_chain_cert(ssl, NULL), 0);
            ExpectIntEQ(SSL_add1_chain_cert(NULL, x509), 0);
            ExpectIntEQ(SSL_add0_chain_cert(NULL, NULL), 0);
            ExpectIntEQ(SSL_add0_chain_cert(ssl, NULL), 0);
            ExpectIntEQ(SSL_add0_chain_cert(NULL, x509), 0);
        }

        ExpectIntEQ(SSL_add1_chain_cert(ssl, x509), 1);
        X509_free(x509);
        x509 = NULL;
    }

    ExpectIntEQ(SSL_CTX_get0_chain_certs(ctx, &chain), 1);
    ExpectIntEQ(sk_X509_num(chain), 3);
    ExpectIntEQ(SSL_get0_chain_certs(ssl, &chain), 1);
    ExpectIntEQ(sk_X509_num(chain), 3);

    SSL_free(ssl);
    SSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_use_certificate_chain_buffer_format(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_CLIENT) && !defined(NO_RSA) && \
    (!defined(NO_FILESYSTEM) || defined(USE_CERT_BUFFERS_2048))
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL* ssl = NULL;
#ifndef NO_FILESYSTEM
    const char* cert = "./certs/server-cert.pem";
    unsigned char* buf = NULL;
    size_t len = 0;

    ExpectIntEQ(load_file(cert, &buf, &len), 0);
#endif

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    /* Invalid parameters. */
#ifndef NO_FILESYSTEM
    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_buffer_format(NULL,
        NULL, 0, WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_buffer_format(ctx,
        NULL, 0, WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(ASN_PARSE_E));
    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_buffer(NULL, NULL, 0),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_buffer(ctx, NULL, 0),
        WC_NO_ERR_TRACE(ASN_NO_PEM_HEADER));
    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_buffer(NULL, buf,
        (sword32)len), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_use_certificate_chain_buffer(NULL, NULL, 0),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_use_certificate_chain_buffer(ssl, NULL, 0),
        WC_NO_ERR_TRACE(ASN_NO_PEM_HEADER));
    ExpectIntEQ(wolfSSL_use_certificate_chain_buffer(NULL, buf, (sword32)len),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_buffer_format(ctx, buf,
        (sword32)len, WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_buffer(ctx, buf, (sword32)len),
        WOLFSSL_SUCCESS);

    ExpectIntEQ(wolfSSL_use_certificate_chain_buffer(ssl, buf, (sword32)len),
        WOLFSSL_SUCCESS);
#endif /* !NO_FILESYSTEM */

    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_buffer_format(NULL,
        server_cert_der_2048, sizeof_server_cert_der_2048,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_buffer_format(ctx,
        server_cert_der_2048, sizeof_server_cert_der_2048,
        WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_buffer(ctx,
        server_cert_der_2048, sizeof_server_cert_der_2048),
        WC_NO_ERR_TRACE(ASN_NO_PEM_HEADER));

    ExpectIntEQ(wolfSSL_use_certificate_chain_buffer(ssl, server_cert_der_2048,
        sizeof_server_cert_der_2048), WC_NO_ERR_TRACE(ASN_NO_PEM_HEADER));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#ifndef NO_FILESYSTEM
    if (buf != NULL) {
        free(buf);
    }
#endif
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_use_certificate_chain_file_format(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_RSA) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    const char* server_chain_der = "./certs/server-cert-chain.der";
    const char* client_single_pem = "./certs/client-cert.pem";
    WOLFSSL_CTX* ctx = NULL;

    (void)server_chain_der;
    (void)client_single_pem;
    (void)ctx;

  #ifndef NO_WOLFSSL_CLIENT
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
  #else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
  #endif

    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_file_format(ctx,
        server_chain_der, WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_file_format(ctx,
        client_single_pem, WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_use_certificate_chain_file(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_CLIENT) && !defined(NO_RSA)
    const char* server_chain_der = "./certs/server-cert-chain.der";
    const char* client_single_pem = "./certs/client-cert.pem";
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL* ssl = NULL;

    (void)server_chain_der;
    (void)client_single_pem;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    /* Invalid parameters. */
    ExpectIntEQ(wolfSSL_use_certificate_chain_file_format(NULL, NULL,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_use_certificate_chain_file_format(ssl, NULL,
        WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_use_certificate_chain_file_format(NULL,
       server_chain_der, WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_use_certificate_chain_file(NULL, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_use_certificate_chain_file(ssl, NULL),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_use_certificate_chain_file(NULL, client_single_pem),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_use_certificate_chain_file(ssl, server_chain_der),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    ExpectIntEQ(wolfSSL_use_certificate_chain_file_format(ssl,
        server_chain_der, WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_use_certificate_chain_file_format(ssl,
        client_single_pem, WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_use_certificate_chain_file(ssl, client_single_pem),
        WOLFSSL_SUCCESS);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_SetTmpDH_file(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_DH) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX *ctx = NULL;
#if defined(WOLFSSL_WPAS) && !defined(NO_DSA)
    const char* dsaParamFile = "./certs/dsaparams.pem";
#endif

    (void)ctx;

  #ifndef NO_WOLFSSL_CLIENT
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
  #else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
  #endif

    /* invalid context */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_file(NULL,
                dhParamFile, WOLFSSL_FILETYPE_PEM));

    /* invalid dhParamFile file */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_file(ctx,
                NULL, WOLFSSL_FILETYPE_PEM));

    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_file(ctx,
                bogusFile, WOLFSSL_FILETYPE_PEM));

    /* success */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_file(ctx, dhParamFile,
                WOLFSSL_FILETYPE_PEM));
#if defined(WOLFSSL_WPAS) && !defined(NO_DSA)
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_file(ctx, dsaParamFile,
                WOLFSSL_FILETYPE_PEM));
#endif

    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_SetTmpDH_buffer(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_TLS) && !defined(NO_DH) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX *ctx = NULL;

  #ifndef NO_WOLFSSL_CLIENT
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
  #else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
  #endif

    /* invalid context */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(NULL,
                dh_key_der_2048, sizeof_dh_key_der_2048,
                WOLFSSL_FILETYPE_ASN1));

    /* invalid dhParamFile file */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(NULL, NULL,
                0, WOLFSSL_FILETYPE_ASN1));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(ctx, NULL,
                0, WOLFSSL_FILETYPE_ASN1));

    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(ctx,
                dsa_key_der_2048, sizeof_dsa_key_der_2048,
                WOLFSSL_FILETYPE_ASN1));

    /* invalid file format */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(ctx,
                dh_key_der_2048, sizeof_dh_key_der_2048, -1));

    /* success */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(ctx,
                dh_key_der_2048, sizeof_dh_key_der_2048,
                WOLFSSL_FILETYPE_ASN1));

    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_SetMinMaxDhKey_Sz(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_TLS) && !defined(NO_DH) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX *ctx;

    (void)ctx;

  #ifndef NO_WOLFSSL_CLIENT
    ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
  #else
    ctx = wolfSSL_CTX_new(wolfSSLv23_server_method());
  #endif
    ExpectNotNull(ctx);

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetMinDhKey_Sz(ctx, 3072));

    ExpectIntEQ(WC_NO_ERR_TRACE(DH_KEY_SIZE_E), wolfSSL_CTX_SetTmpDH_buffer(ctx, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetMinDhKey_Sz(ctx, 2048));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(ctx,
                dh_key_der_2048, sizeof_dh_key_der_2048,
                WOLFSSL_FILETYPE_ASN1));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetMaxDhKey_Sz(ctx, 1024));

    ExpectIntEQ(WC_NO_ERR_TRACE(DH_KEY_SIZE_E), wolfSSL_CTX_SetTmpDH_buffer(ctx,
                dh_key_der_2048, sizeof_dh_key_der_2048,
                WOLFSSL_FILETYPE_ASN1));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetMaxDhKey_Sz(ctx, 2048));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_buffer(ctx,
                dh_key_der_2048, sizeof_dh_key_der_2048,
                WOLFSSL_FILETYPE_ASN1));

    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_der_load_verify_locations(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && defined(WOLFSSL_DER_LOAD) && \
    !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX* ctx = NULL;
    const char* derCert = "./certs/server-cert.der";
    const char* nullPath = NULL;
    const char* invalidPath = "./certs/this-cert-does-not-exist.der";
    const char* emptyPath = "";

    /* der load Case 1 ctx NULL */
    ExpectIntEQ(wolfSSL_CTX_der_load_verify_locations(ctx, derCert,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

  #ifndef NO_WOLFSSL_CLIENT
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
  #else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
  #endif

    /* Case 2 filePath NULL */
    ExpectIntEQ(wolfSSL_CTX_der_load_verify_locations(ctx, nullPath,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    /* Case 3 invalid format */
    ExpectIntEQ(wolfSSL_CTX_der_load_verify_locations(ctx, derCert,
                WOLFSSL_FILETYPE_PEM), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    /* Case 4 filePath not valid */
    ExpectIntEQ(wolfSSL_CTX_der_load_verify_locations(ctx, invalidPath,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    /* Case 5 filePath empty */
    ExpectIntEQ(wolfSSL_CTX_der_load_verify_locations(ctx, emptyPath,
                WOLFSSL_FILETYPE_ASN1), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
#ifndef NO_RSA
    /* Case 6 success case */
    ExpectIntEQ(wolfSSL_CTX_der_load_verify_locations(ctx, derCert,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#endif

    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_enable_disable(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_TLS)
    WOLFSSL_CTX* ctx = NULL;

  #ifdef HAVE_CRL
    ExpectIntEQ(wolfSSL_CTX_DisableCRL(ctx), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_EnableCRL(ctx, 0), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
  #endif

  #ifdef HAVE_OCSP
    ExpectIntEQ(wolfSSL_CTX_DisableOCSP(ctx), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_EnableOCSP(ctx, 0), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
  #endif

  #if defined(HAVE_CERTIFICATE_STATUS_REQUEST) || \
      defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2)
    ExpectIntEQ(wolfSSL_CTX_DisableOCSPStapling(ctx), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_EnableOCSPStapling(ctx), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_DisableOCSPMustStaple(ctx), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_CTX_EnableOCSPMustStaple(ctx), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
  #endif

  #ifndef NO_WOLFSSL_CLIENT

    #ifdef HAVE_EXTENDED_MASTER
    ExpectIntEQ(wolfSSL_CTX_DisableExtendedMasterSecret(ctx), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    #endif
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));

    #ifdef HAVE_EXTENDED_MASTER
    ExpectIntEQ(wolfSSL_CTX_DisableExtendedMasterSecret(ctx), WOLFSSL_SUCCESS);
    #endif

  #elif !defined(NO_WOLFSSL_SERVER)
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
  #endif

  #if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)

    #ifdef HAVE_CRL
        ExpectIntEQ(wolfSSL_CTX_DisableCRL(ctx), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_EnableCRL(ctx, 0), WOLFSSL_SUCCESS);
    #endif

    #ifdef HAVE_OCSP
        ExpectIntEQ(wolfSSL_CTX_DisableOCSP(ctx), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_EnableOCSP(ctx, WOLFSSL_OCSP_URL_OVERRIDE),
                    WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_EnableOCSP(ctx, WOLFSSL_OCSP_NO_NONCE),
                    WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_EnableOCSP(ctx, WOLFSSL_OCSP_CHECKALL),
                    WOLFSSL_SUCCESS);
    #endif

    #if defined(HAVE_CERTIFICATE_STATUS_REQUEST) || \
        defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2)
        ExpectIntEQ(wolfSSL_CTX_DisableOCSPStapling(ctx), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_EnableOCSPStapling(ctx), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_DisableOCSPMustStaple(ctx), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_DisableOCSPMustStaple(ctx), WOLFSSL_SUCCESS);
    #endif
        wolfSSL_CTX_free(ctx);
  #endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif /* !NO_CERTS && !NO_CERTS */

    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_ticket_API(void)
{
    EXPECT_DECLS;
#if defined(HAVE_SESSION_TICKET) && !defined(NO_WOLFSSL_SERVER) && \
    !defined(NO_TLS)
    WOLFSSL_CTX* ctx = NULL;
    void *userCtx = (void*)"this is my ctx";

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_set_TicketEncCtx(ctx, userCtx));
    ExpectTrue(userCtx == wolfSSL_CTX_get_TicketEncCtx(ctx));

    wolfSSL_CTX_free(ctx);

    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_set_TicketEncCtx(NULL, userCtx));
    ExpectNull(wolfSSL_CTX_get_TicketEncCtx(NULL));
#endif /* HAVE_SESSION_TICKET && !NO_WOLFSSL_SERVER && !NO_TLS */
    return EXPECT_RESULT();
}

static int test_wolfSSL_set_minmax_proto_version(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_TLS)
    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL *ssl = NULL;

    (void)ssl;

#ifndef NO_WOLFSSL_CLIENT
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    ExpectIntEQ(wolfSSL_CTX_set_min_proto_version(NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_CTX_set_max_proto_version(NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_CTX_set_min_proto_version(ctx, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CTX_set_max_proto_version(ctx, 0), SSL_SUCCESS);

    ExpectIntEQ(wolfSSL_set_min_proto_version(NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_set_min_proto_version(ssl, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_set_max_proto_version(NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_set_max_proto_version(ssl, 0), SSL_SUCCESS);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
    ctx = NULL;
#endif
#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    ExpectIntEQ(wolfSSL_CTX_set_min_proto_version(NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_CTX_set_max_proto_version(NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_CTX_set_min_proto_version(ctx, 0), SSL_SUCCESS);
    ExpectIntEQ(wolfSSL_CTX_set_max_proto_version(ctx, 0), SSL_SUCCESS);

    wolfSSL_CTX_free(ctx);
#endif
#endif

    return EXPECT_RESULT();
}

#if defined(WOLFSSL_TLS13) && !defined(WOLFSSL_NO_TLS12) && \
    defined(OPENSSL_EXTRA) && defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
static int test_wolfSSL_CTX_set_max_proto_version_on_result(WOLFSSL* ssl)
{
    EXPECT_DECLS;
    ExpectStrEQ(wolfSSL_get_version(ssl), "TLSv1.2");
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_set_max_proto_version_ctx_ready(WOLFSSL_CTX* ctx)
{
    EXPECT_DECLS;
    /* Set TLS 1.2 */
    ExpectIntEQ(wolfSSL_CTX_set_max_proto_version(ctx, TLS1_2_VERSION),
            WOLFSSL_SUCCESS);
    return EXPECT_RESULT();
}

/* Test using wolfSSL_CTX_set_max_proto_version to limit the version below
 * what was set at ctx creation. */
static int test_wolfSSL_CTX_set_max_proto_version(void)
{
    EXPECT_DECLS;
    test_ssl_cbf client_cbs;
    test_ssl_cbf server_cbs;

    XMEMSET(&client_cbs, 0, sizeof(client_cbs));
    XMEMSET(&server_cbs, 0, sizeof(server_cbs));

    client_cbs.method = wolfTLS_client_method;
    server_cbs.method = wolfTLS_server_method;

    server_cbs.ctx_ready = test_wolfSSL_CTX_set_max_proto_version_ctx_ready;

    client_cbs.on_result = test_wolfSSL_CTX_set_max_proto_version_on_result;
    server_cbs.on_result = test_wolfSSL_CTX_set_max_proto_version_on_result;

    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cbs,
        &server_cbs, NULL), TEST_SUCCESS);

    return EXPECT_RESULT();
}
#else
static int test_wolfSSL_CTX_set_max_proto_version(void)
{
    return TEST_SKIPPED;
}
#endif

/*----------------------------------------------------------------------------*
 | SSL
 *----------------------------------------------------------------------------*/

static int test_server_wolfSSL_new(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_SERVER) && !defined(NO_RSA)

    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL_CTX *ctx_nocert = NULL;
    WOLFSSL *ssl = NULL;

    ExpectNotNull(ctx_nocert = wolfSSL_CTX_new(wolfSSLv23_server_method()));
    ExpectNotNull(ctx        = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile,
        WOLFSSL_FILETYPE_PEM));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
        WOLFSSL_FILETYPE_PEM));

    /* invalid context */
    ExpectNull(ssl = wolfSSL_new(NULL));
#if !defined(WOLFSSL_SESSION_EXPORT) && !defined(WOLFSSL_QT) && \
        !defined(OPENSSL_EXTRA) && !defined(WOLFSSL_NO_INIT_CTX_KEY)
    ExpectNull(ssl = wolfSSL_new(ctx_nocert));
#endif

    /* success */
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
    wolfSSL_CTX_free(ctx_nocert);
#endif

    return EXPECT_RESULT();
}


static int test_client_wolfSSL_new(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_CLIENT) && !defined(NO_RSA)

    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL_CTX *ctx_nocert = NULL;
    WOLFSSL *ssl = NULL;

    ExpectNotNull(ctx_nocert = wolfSSL_CTX_new(wolfSSLv23_client_method()));
    ExpectNotNull(ctx        = wolfSSL_CTX_new(wolfSSLv23_client_method()));

    ExpectTrue(wolfSSL_CTX_load_verify_locations(ctx, caCertFile, 0));

    /* invalid context */
    ExpectNull(ssl = wolfSSL_new(NULL));

    /* success */
    ExpectNotNull(ssl = wolfSSL_new(ctx_nocert));
    wolfSSL_free(ssl);
    ssl = NULL;

    /* success */
    ExpectNotNull(ssl = wolfSSL_new(ctx));
    wolfSSL_free(ssl);

    wolfSSL_CTX_free(ctx);
    wolfSSL_CTX_free(ctx_nocert);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_SetTmpDH_file(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_WOLFSSL_SERVER) && !defined(NO_DH)

    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL *ssl = NULL;
    const char* dhX942ParamFile = "./certs/x942dh2048.pem";
#if defined(WOLFSSL_WPAS) && !defined(NO_DSA)
    const char* dsaParamFile = "./certs/dsaparams.pem";
#endif

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#ifndef NO_RSA
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile,
               WOLFSSL_FILETYPE_PEM));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
               WOLFSSL_FILETYPE_PEM));
#elif defined(HAVE_ECC)
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, eccCertFile,
               WOLFSSL_FILETYPE_PEM));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, eccKeyFile,
               WOLFSSL_FILETYPE_PEM));
#elif defined(HAVE_ED25519)
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, edCertFile,
               WOLFSSL_FILETYPE_PEM));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, edKeyFile,
               WOLFSSL_FILETYPE_PEM));
#elif defined(HAVE_ED448)
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, ed448CertFile,
               WOLFSSL_FILETYPE_PEM));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, ed448KeyFile,
               WOLFSSL_FILETYPE_PEM));
#endif
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    /* invalid ssl */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_file(NULL,
                dhParamFile, WOLFSSL_FILETYPE_PEM));

    /* invalid dhParamFile file */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_file(ssl,
                NULL, WOLFSSL_FILETYPE_PEM));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_file(ssl,
                bogusFile, WOLFSSL_FILETYPE_PEM));

    /* success */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_file(ssl, dhParamFile,
                WOLFSSL_FILETYPE_PEM));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_file(ssl, dhX942ParamFile,
                WOLFSSL_FILETYPE_PEM));
#if defined(WOLFSSL_WPAS) && !defined(NO_DSA)
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetTmpDH_file(ctx, dsaParamFile,
                WOLFSSL_FILETYPE_PEM));
#endif

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_SetTmpDH_buffer(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_TLS) && !defined(NO_WOLFSSL_SERVER) && \
    !defined(NO_DH)
    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL *ssl = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
    ExpectTrue(wolfSSL_CTX_use_certificate_buffer(ctx, server_cert_der_2048,
               sizeof_server_cert_der_2048, WOLFSSL_FILETYPE_ASN1));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_buffer(ctx, server_key_der_2048,
               sizeof_server_key_der_2048, WOLFSSL_FILETYPE_ASN1));
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    /* invalid ssl */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(NULL, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    /* invalid dhParamFile file */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(NULL, NULL,
                0, WOLFSSL_FILETYPE_ASN1));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(ssl, NULL, 0,
                WOLFSSL_FILETYPE_ASN1));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(ssl, dsa_key_der_2048,
                sizeof_dsa_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    /* success */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(ssl, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_SetMinMaxDhKey_Sz(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_TLS) && !defined(NO_WOLFSSL_SERVER) && \
    !defined(NO_DH)
    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL_CTX *ctx2 = NULL;
    WOLFSSL *ssl = NULL;
    WOLFSSL *ssl2 = NULL;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
    ExpectTrue(wolfSSL_CTX_use_certificate_buffer(ctx, server_cert_der_2048,
                sizeof_server_cert_der_2048, WOLFSSL_FILETYPE_ASN1));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_buffer(ctx, server_key_der_2048,
                sizeof_server_key_der_2048, WOLFSSL_FILETYPE_ASN1));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetMinDhKey_Sz(ctx, 3072));
    ExpectNotNull(ssl = wolfSSL_new(ctx));
    ExpectNotNull(ctx2 = wolfSSL_CTX_new(wolfSSLv23_server_method()));
    ExpectTrue(wolfSSL_CTX_use_certificate_buffer(ctx2, server_cert_der_2048,
                sizeof_server_cert_der_2048, WOLFSSL_FILETYPE_ASN1));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_buffer(ctx2, server_key_der_2048,
                sizeof_server_key_der_2048, WOLFSSL_FILETYPE_ASN1));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_SetMaxDhKey_Sz(ctx, 1024));
    ExpectNotNull(ssl2 = wolfSSL_new(ctx2));

    ExpectIntEQ(WC_NO_ERR_TRACE(DH_KEY_SIZE_E), wolfSSL_SetTmpDH_buffer(ssl, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetMinDhKey_Sz(ssl, 2048));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(ssl, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetMinDhKey_Sz(ssl, 3072));
    ExpectIntEQ(WC_NO_ERR_TRACE(DH_KEY_SIZE_E), wolfSSL_SetTmpDH_buffer(ssl, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(ssl2, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetMaxDhKey_Sz(ssl2, 2048));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetTmpDH_buffer(ssl2, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SetMaxDhKey_Sz(ssl2, 1024));
    ExpectIntEQ(WC_NO_ERR_TRACE(DH_KEY_SIZE_E), wolfSSL_SetTmpDH_buffer(ssl, dh_key_der_2048,
                sizeof_dh_key_der_2048, WOLFSSL_FILETYPE_ASN1));

    wolfSSL_free(ssl2);
    wolfSSL_CTX_free(ctx2);
    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}


/* Test function for wolfSSL_SetMinVersion. Sets the minimum downgrade version
 * allowed.
 * POST: return 1 on success.
 */
static int test_wolfSSL_SetMinVersion(void)
{
    int                 res = TEST_SKIPPED;
#if !defined(NO_WOLFSSL_CLIENT) && !defined(NO_TLS)
    int                 failFlag = WOLFSSL_SUCCESS;
    WOLFSSL_CTX*        ctx = NULL;
    WOLFSSL*            ssl = NULL;
    int                 itr;

    #ifndef NO_OLD_TLS
        const int versions[]  =  {
                            #ifdef WOLFSSL_ALLOW_TLSV10
                                   WOLFSSL_TLSV1,
                            #endif
                                   WOLFSSL_TLSV1_1,
                                   WOLFSSL_TLSV1_2};
    #elif !defined(WOLFSSL_NO_TLS12)
        const int versions[]  =  { WOLFSSL_TLSV1_2 };
    #else
        const int versions[]  =  { WOLFSSL_TLSV1_3 };
    #endif

    ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    ssl = wolfSSL_new(ctx);

    for (itr = 0; itr < (int)(sizeof(versions)/sizeof(int)); itr++) {
       if (wolfSSL_SetMinVersion(ssl, *(versions + itr)) != WOLFSSL_SUCCESS) {
            failFlag = WOLFSSL_FAILURE;
        }
    }

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(failFlag == WOLFSSL_SUCCESS);
#endif
    return res;

} /* END test_wolfSSL_SetMinVersion */


#ifdef OPENSSL_EXTRA
static int test_EC25519(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CURVE25519) && defined(WOLFSSL_KEY_GEN)
    byte         priv[CURVE25519_KEYSIZE];
    unsigned int privSz = CURVE25519_KEYSIZE;
    byte         pub[CURVE25519_KEYSIZE];
    unsigned int pubSz = CURVE25519_KEYSIZE;
    byte         priv2[CURVE25519_KEYSIZE];
    unsigned int priv2Sz = CURVE25519_KEYSIZE;
    byte         pub2[CURVE25519_KEYSIZE];
    unsigned int pub2Sz = CURVE25519_KEYSIZE;
    byte         shared[CURVE25519_KEYSIZE];
    unsigned int sharedSz = CURVE25519_KEYSIZE;
    byte         shared2[CURVE25519_KEYSIZE];
    unsigned int shared2Sz = CURVE25519_KEYSIZE;

    /* Bad parameter testing of key generation. */
    ExpectIntEQ(wolfSSL_EC25519_generate_key(NULL,    NULL, NULL,   NULL), 0);
    ExpectIntEQ(wolfSSL_EC25519_generate_key(NULL, &privSz, NULL, &pubSz), 0);
    ExpectIntEQ(wolfSSL_EC25519_generate_key(NULL, &privSz,  pub, &pubSz), 0);
    ExpectIntEQ(wolfSSL_EC25519_generate_key(priv,    NULL,  pub, &pubSz), 0);
    ExpectIntEQ(wolfSSL_EC25519_generate_key(priv, &privSz, NULL, &pubSz), 0);
    ExpectIntEQ(wolfSSL_EC25519_generate_key(priv, &privSz,  pub,   NULL), 0);
    /*   Bad length */
    privSz = 1;
    ExpectIntEQ(wolfSSL_EC25519_generate_key(priv, &privSz, pub, &pubSz), 0);
    privSz = CURVE25519_KEYSIZE;
    pubSz = 1;
    ExpectIntEQ(wolfSSL_EC25519_generate_key(priv, &privSz, pub, &pubSz), 0);
    pubSz = CURVE25519_KEYSIZE;

    /* Good case of generating key. */
    ExpectIntEQ(wolfSSL_EC25519_generate_key(priv, &privSz, pub, &pubSz), 1);
    ExpectIntEQ(wolfSSL_EC25519_generate_key(priv2, &priv2Sz, pub2, &pub2Sz),
        1);
    ExpectIntEQ(privSz, CURVE25519_KEYSIZE);
    ExpectIntEQ(pubSz, CURVE25519_KEYSIZE);

    /* Bad parameter testing of shared key. */
    ExpectIntEQ(wolfSSL_EC25519_shared_key(  NULL,      NULL, NULL, privSz,
        NULL,  pubSz), 0);
    ExpectIntEQ(wolfSSL_EC25519_shared_key(  NULL, &sharedSz, NULL, privSz,
        NULL, pubSz), 0);
    ExpectIntEQ(wolfSSL_EC25519_shared_key(  NULL, &sharedSz, priv, privSz,
         pub, pubSz), 0);
    ExpectIntEQ(wolfSSL_EC25519_shared_key(shared, &sharedSz, NULL, privSz,
         pub, pubSz), 0);
    ExpectIntEQ(wolfSSL_EC25519_shared_key(shared, &sharedSz, priv, privSz,
        NULL, pubSz), 0);
    ExpectIntEQ(wolfSSL_EC25519_shared_key(  NULL, &sharedSz, priv, privSz,
         pub, pubSz), 0);
    ExpectIntEQ(wolfSSL_EC25519_shared_key(shared,      NULL, priv, privSz,
         pub, pubSz), 0);
    ExpectIntEQ(wolfSSL_EC25519_shared_key(shared, &sharedSz, NULL, privSz,
         pub, pubSz), 0);
    ExpectIntEQ(wolfSSL_EC25519_shared_key(shared, &sharedSz, priv, privSz,
        NULL, pubSz), 0);
    /*   Bad length. */
    sharedSz = 1;
    ExpectIntEQ(wolfSSL_EC25519_shared_key(shared, &sharedSz, priv, privSz,
         pub, pubSz), 0);
    sharedSz = CURVE25519_KEYSIZE;
    privSz = 1;
    ExpectIntEQ(wolfSSL_EC25519_shared_key(shared, &sharedSz, priv, privSz,
         pub, pubSz), 0);
    privSz = CURVE25519_KEYSIZE;
    pubSz = 1;
    ExpectIntEQ(wolfSSL_EC25519_shared_key(shared, &sharedSz, priv, privSz,
         pub, pubSz), 0);
    pubSz = CURVE25519_KEYSIZE;

    /* Good case of shared key. */
    ExpectIntEQ(wolfSSL_EC25519_shared_key(shared, &sharedSz, priv, privSz,
        pub2, pub2Sz), 1);
    ExpectIntEQ(wolfSSL_EC25519_shared_key(shared2, &shared2Sz, priv2, priv2Sz,
        pub, pubSz), 1);
    ExpectIntEQ(sharedSz, CURVE25519_KEYSIZE);
    ExpectIntEQ(shared2Sz, CURVE25519_KEYSIZE);
    ExpectIntEQ(XMEMCMP(shared, shared2, sharedSz), 0);
#endif /* HAVE_CURVE25519 && WOLFSSL_KEY_GEN */
    return EXPECT_RESULT();
}

static int test_ED25519(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT) && \
    defined(WOLFSSL_KEY_GEN)
    byte         priv[ED25519_PRV_KEY_SIZE];
    unsigned int privSz = (unsigned int)sizeof(priv);
    byte         pub[ED25519_PUB_KEY_SIZE];
    unsigned int pubSz = (unsigned int)sizeof(pub);
#if defined(HAVE_ED25519_SIGN) && defined(HAVE_ED25519_KEY_IMPORT)
    const char*  msg = TEST_STRING;
    unsigned int msglen = (unsigned int)TEST_STRING_SZ;
    byte         sig[ED25519_SIG_SIZE];
    unsigned int sigSz = (unsigned int)sizeof(sig);
#endif /* HAVE_ED25519_SIGN && HAVE_ED25519_KEY_IMPORT */

    /* Bad parameter testing of key generation. */
    ExpectIntEQ(wolfSSL_ED25519_generate_key(NULL,    NULL, NULL,   NULL), 0);
    ExpectIntEQ(wolfSSL_ED25519_generate_key(priv,    NULL, NULL,   NULL), 0);
    ExpectIntEQ(wolfSSL_ED25519_generate_key(NULL, &privSz, NULL,   NULL), 0);
    ExpectIntEQ(wolfSSL_ED25519_generate_key(NULL,    NULL,  pub,   NULL), 0);
    ExpectIntEQ(wolfSSL_ED25519_generate_key(NULL,    NULL, NULL, &pubSz), 0);
    ExpectIntEQ(wolfSSL_ED25519_generate_key(NULL, &privSz,  pub, &pubSz), 0);
    ExpectIntEQ(wolfSSL_ED25519_generate_key(priv,    NULL,  pub, &pubSz), 0);
    ExpectIntEQ(wolfSSL_ED25519_generate_key(priv, &privSz, NULL, &pubSz), 0);
    ExpectIntEQ(wolfSSL_ED25519_generate_key(priv, &privSz,  pub,   NULL), 0);
    /*   Bad length. */
    privSz = 1;
    ExpectIntEQ(wolfSSL_ED25519_generate_key(priv, &privSz, pub, &pubSz), 0);
    privSz = ED25519_PRV_KEY_SIZE;
    pubSz = 1;
    ExpectIntEQ(wolfSSL_ED25519_generate_key(priv, &privSz, pub, &pubSz), 0);
    pubSz = ED25519_PUB_KEY_SIZE;

    /* Good case of generating key. */
    ExpectIntEQ(wolfSSL_ED25519_generate_key(priv, &privSz, pub, &pubSz),
        1);
    ExpectIntEQ(privSz, ED25519_PRV_KEY_SIZE);
    ExpectIntEQ(pubSz, ED25519_PUB_KEY_SIZE);

#if defined(HAVE_ED25519_SIGN) && defined(HAVE_ED25519_KEY_IMPORT)
    /* Bad parameter testing of signing. */
    ExpectIntEQ(wolfSSL_ED25519_sign(      NULL, msglen, NULL, privSz, NULL,
          NULL), 0);
    ExpectIntEQ(wolfSSL_ED25519_sign((byte*)msg, msglen, NULL, privSz, NULL,
          NULL), 0);
    ExpectIntEQ(wolfSSL_ED25519_sign(      NULL, msglen, priv, privSz, NULL,
          NULL), 0);
    ExpectIntEQ(wolfSSL_ED25519_sign(      NULL, msglen, NULL, privSz, sig,
          NULL), 0);
    ExpectIntEQ(wolfSSL_ED25519_sign(      NULL, msglen, NULL, privSz, NULL,
        &sigSz), 0);
    ExpectIntEQ(wolfSSL_ED25519_sign(      NULL, msglen, priv, privSz,  sig,
        &sigSz), 0);
    ExpectIntEQ(wolfSSL_ED25519_sign((byte*)msg, msglen, NULL, privSz,  sig,
        &sigSz), 0);
    ExpectIntEQ(wolfSSL_ED25519_sign((byte*)msg, msglen, priv, privSz,  NULL,
        &sigSz), 0);
    ExpectIntEQ(wolfSSL_ED25519_sign((byte*)msg, msglen, priv, privSz,  sig,
          NULL), 0);
    /*   Bad length. */
    privSz = 1;
    ExpectIntEQ(wolfSSL_ED25519_sign((byte*)msg, msglen, priv, privSz, sig,
        &sigSz), 0);
    privSz = ED25519_PRV_KEY_SIZE;
    sigSz = 1;
    ExpectIntEQ(wolfSSL_ED25519_sign((byte*)msg, msglen, priv, privSz, sig,
        &sigSz), 0);
    sigSz = ED25519_SIG_SIZE;

    /* Good case of signing. */
    ExpectIntEQ(wolfSSL_ED25519_sign((byte*)msg, msglen, priv, privSz, sig,
        &sigSz), 1);
    ExpectIntEQ(sigSz, ED25519_SIG_SIZE);

#ifdef HAVE_ED25519_VERIFY
    /* Bad parameter testing of verification. */
    ExpectIntEQ(wolfSSL_ED25519_verify(      NULL, msglen, NULL, pubSz, NULL,
        sigSz), 0);
    ExpectIntEQ(wolfSSL_ED25519_verify((byte*)msg, msglen, NULL, pubSz, NULL,
        sigSz), 0);
    ExpectIntEQ(wolfSSL_ED25519_verify(      NULL, msglen,  pub, pubSz, NULL,
        sigSz), 0);
    ExpectIntEQ(wolfSSL_ED25519_verify(      NULL, msglen, NULL, pubSz,  sig,
        sigSz), 0);
    ExpectIntEQ(wolfSSL_ED25519_verify(      NULL, msglen,  pub, pubSz,  sig,
        sigSz), 0);
    ExpectIntEQ(wolfSSL_ED25519_verify((byte*)msg, msglen, NULL, pubSz,  sig,
        sigSz), 0);
    ExpectIntEQ(wolfSSL_ED25519_verify((byte*)msg, msglen,  pub, pubSz, NULL,
        sigSz), 0);
    /*   Bad length. */
    pubSz = 1;
    ExpectIntEQ(wolfSSL_ED25519_verify((byte*)msg, msglen, pub, pubSz, sig,
        sigSz), 0);
    pubSz = ED25519_PUB_KEY_SIZE;
    sigSz = 1;
    ExpectIntEQ(wolfSSL_ED25519_verify((byte*)msg, msglen, pub, pubSz, sig,
        sigSz), 0);
    sigSz = ED25519_SIG_SIZE;

    /* Good case of verification. */
    ExpectIntEQ(wolfSSL_ED25519_verify((byte*)msg, msglen, pub, pubSz, sig,
        sigSz), 1);
    /* Bad signature. */
    if (EXPECT_SUCCESS()) {
        sig[1] ^= 0x80;
    }
    ExpectIntEQ(wolfSSL_ED25519_verify((byte*)msg, msglen, pub, pubSz, sig,
        sigSz), 0);
#endif /* HAVE_ED25519_VERIFY */
#endif /* HAVE_ED25519_SIGN && HAVE_ED25519_KEY_IMPORT */
#endif /* HAVE_ED25519 && HAVE_ED25519_KEY_EXPORT && WOLFSSL_KEY_GEN */
    return EXPECT_RESULT();
}

static int test_EC448(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CURVE448) && defined(WOLFSSL_KEY_GEN)
    byte         priv[CURVE448_KEY_SIZE];
    unsigned int privSz = CURVE448_KEY_SIZE;
    byte         pub[CURVE448_KEY_SIZE];
    unsigned int pubSz = CURVE448_KEY_SIZE;
    byte         priv2[CURVE448_KEY_SIZE];
    unsigned int priv2Sz = CURVE448_KEY_SIZE;
    byte         pub2[CURVE448_KEY_SIZE];
    unsigned int pub2Sz = CURVE448_KEY_SIZE;
    byte         shared[CURVE448_KEY_SIZE];
    unsigned int sharedSz = CURVE448_KEY_SIZE;
    byte         shared2[CURVE448_KEY_SIZE];
    unsigned int shared2Sz = CURVE448_KEY_SIZE;

    /* Bad parameter testing of key generation. */
    ExpectIntEQ(wolfSSL_EC448_generate_key(NULL,    NULL, NULL,   NULL), 0);
    ExpectIntEQ(wolfSSL_EC448_generate_key(NULL, &privSz, NULL, &pubSz), 0);
    ExpectIntEQ(wolfSSL_EC448_generate_key(NULL, &privSz,  pub, &pubSz), 0);
    ExpectIntEQ(wolfSSL_EC448_generate_key(priv,    NULL,  pub, &pubSz), 0);
    ExpectIntEQ(wolfSSL_EC448_generate_key(priv, &privSz, NULL, &pubSz), 0);
    ExpectIntEQ(wolfSSL_EC448_generate_key(priv, &privSz,  pub,   NULL), 0);
    /*   Bad length. */
    privSz = 1;
    ExpectIntEQ(wolfSSL_EC448_generate_key(priv, &privSz, pub, &pubSz), 0);
    privSz = CURVE448_KEY_SIZE;
    pubSz = 1;
    ExpectIntEQ(wolfSSL_EC448_generate_key(priv, &privSz, pub, &pubSz), 0);
    pubSz = CURVE448_KEY_SIZE;

    /* Good case of generating key. */
    ExpectIntEQ(wolfSSL_EC448_generate_key(priv, &privSz, pub, &pubSz), 1);
    ExpectIntEQ(wolfSSL_EC448_generate_key(priv2, &priv2Sz, pub2, &pub2Sz), 1);
    ExpectIntEQ(privSz, CURVE448_KEY_SIZE);
    ExpectIntEQ(pubSz, CURVE448_KEY_SIZE);

    /* Bad parameter testing of shared key. */
    ExpectIntEQ(wolfSSL_EC448_shared_key(  NULL,      NULL, NULL, privSz,
        NULL,  pubSz), 0);
    ExpectIntEQ(wolfSSL_EC448_shared_key(  NULL, &sharedSz, NULL, privSz,
        NULL, pubSz), 0);
    ExpectIntEQ(wolfSSL_EC448_shared_key(  NULL, &sharedSz, priv, privSz,
         pub, pubSz), 0);
    ExpectIntEQ(wolfSSL_EC448_shared_key(shared, &sharedSz, NULL, privSz,
         pub, pubSz), 0);
    ExpectIntEQ(wolfSSL_EC448_shared_key(shared, &sharedSz, priv, privSz,
        NULL, pubSz), 0);
    ExpectIntEQ(wolfSSL_EC448_shared_key(  NULL, &sharedSz, priv, privSz,
         pub, pubSz), 0);
    ExpectIntEQ(wolfSSL_EC448_shared_key(shared,      NULL, priv, privSz,
         pub, pubSz), 0);
    ExpectIntEQ(wolfSSL_EC448_shared_key(shared, &sharedSz, NULL, privSz,
         pub, pubSz), 0);
    ExpectIntEQ(wolfSSL_EC448_shared_key(shared, &sharedSz, priv, privSz,
        NULL, pubSz), 0);
    /*   Bad length. */
    sharedSz = 1;
    ExpectIntEQ(wolfSSL_EC448_shared_key(shared, &sharedSz, priv, privSz,
         pub, pubSz), 0);
    sharedSz = CURVE448_KEY_SIZE;
    privSz = 1;
    ExpectIntEQ(wolfSSL_EC448_shared_key(shared, &sharedSz, priv, privSz,
         pub, pubSz), 0);
    privSz = CURVE448_KEY_SIZE;
    pubSz = 1;
    ExpectIntEQ(wolfSSL_EC448_shared_key(shared, &sharedSz, priv, privSz,
         pub, pubSz), 0);
    pubSz = CURVE448_KEY_SIZE;

    /* Good case of shared key. */
    ExpectIntEQ(wolfSSL_EC448_shared_key(shared, &sharedSz, priv, privSz,
        pub2, pub2Sz), 1);
    ExpectIntEQ(wolfSSL_EC448_shared_key(shared2, &shared2Sz, priv2, priv2Sz,
        pub, pubSz), 1);
    ExpectIntEQ(sharedSz, CURVE448_KEY_SIZE);
    ExpectIntEQ(shared2Sz, CURVE448_KEY_SIZE);
    ExpectIntEQ(XMEMCMP(shared, shared2, sharedSz), 0);
#endif /* HAVE_CURVE448 && WOLFSSL_KEY_GEN */
    return EXPECT_RESULT();
}

static int test_ED448(void)
{
    EXPECT_DECLS;
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT) && \
    defined(WOLFSSL_KEY_GEN)
    byte         priv[ED448_PRV_KEY_SIZE];
    unsigned int privSz = (unsigned int)sizeof(priv);
    byte         pub[ED448_PUB_KEY_SIZE];
    unsigned int pubSz = (unsigned int)sizeof(pub);
#if defined(HAVE_ED448_SIGN) && defined(HAVE_ED448_KEY_IMPORT)
    const char*  msg = TEST_STRING;
    unsigned int msglen = (unsigned int)TEST_STRING_SZ;
    byte         sig[ED448_SIG_SIZE];
    unsigned int sigSz = (unsigned int)sizeof(sig);
#endif /* HAVE_ED448_SIGN && HAVE_ED448_KEY_IMPORT */

    /* Bad parameter testing of key generation. */
    ExpectIntEQ(wolfSSL_ED448_generate_key(NULL,    NULL, NULL,   NULL), 0);
    ExpectIntEQ(wolfSSL_ED448_generate_key(priv,    NULL, NULL,   NULL), 0);
    ExpectIntEQ(wolfSSL_ED448_generate_key(NULL, &privSz, NULL,   NULL), 0);
    ExpectIntEQ(wolfSSL_ED448_generate_key(NULL,    NULL,  pub,   NULL), 0);
    ExpectIntEQ(wolfSSL_ED448_generate_key(NULL,    NULL, NULL, &pubSz), 0);
    ExpectIntEQ(wolfSSL_ED448_generate_key(NULL, &privSz,  pub, &pubSz), 0);
    ExpectIntEQ(wolfSSL_ED448_generate_key(priv,    NULL,  pub, &pubSz), 0);
    ExpectIntEQ(wolfSSL_ED448_generate_key(priv, &privSz, NULL, &pubSz), 0);
    ExpectIntEQ(wolfSSL_ED448_generate_key(priv, &privSz,  pub,   NULL), 0);
    /*   Bad length. */
    privSz = 1;
    ExpectIntEQ(wolfSSL_ED448_generate_key(priv, &privSz, pub, &pubSz), 0);
    privSz = ED448_PRV_KEY_SIZE;
    pubSz = 1;
    ExpectIntEQ(wolfSSL_ED448_generate_key(priv, &privSz, pub, &pubSz), 0);
    pubSz = ED448_PUB_KEY_SIZE;

    /* Good case of generating key. */
    ExpectIntEQ(wolfSSL_ED448_generate_key(priv, &privSz, pub, &pubSz), 1);
    ExpectIntEQ(privSz, ED448_PRV_KEY_SIZE);
    ExpectIntEQ(pubSz, ED448_PUB_KEY_SIZE);

#if defined(HAVE_ED448_SIGN) && defined(HAVE_ED448_KEY_IMPORT)
    /* Bad parameter testing of signing. */
    ExpectIntEQ(wolfSSL_ED448_sign(      NULL, msglen, NULL, privSz, NULL,
          NULL), 0);
    ExpectIntEQ(wolfSSL_ED448_sign((byte*)msg, msglen, NULL, privSz, NULL,
          NULL), 0);
    ExpectIntEQ(wolfSSL_ED448_sign(      NULL, msglen, priv, privSz, NULL,
          NULL), 0);
    ExpectIntEQ(wolfSSL_ED448_sign(      NULL, msglen, NULL, privSz, sig,
          NULL), 0);
    ExpectIntEQ(wolfSSL_ED448_sign(      NULL, msglen, NULL, privSz, NULL,
        &sigSz), 0);
    ExpectIntEQ(wolfSSL_ED448_sign(      NULL, msglen, priv, privSz,  sig,
        &sigSz), 0);
    ExpectIntEQ(wolfSSL_ED448_sign((byte*)msg, msglen, NULL, privSz,  sig,
        &sigSz), 0);
    ExpectIntEQ(wolfSSL_ED448_sign((byte*)msg, msglen, priv, privSz,  NULL,
        &sigSz), 0);
    ExpectIntEQ(wolfSSL_ED448_sign((byte*)msg, msglen, priv, privSz,  sig,
          NULL), 0);
    /*   Bad length. */
    privSz = 1;
    ExpectIntEQ(wolfSSL_ED448_sign((byte*)msg, msglen, priv, privSz, sig,
        &sigSz), 0);
    privSz = ED448_PRV_KEY_SIZE;
    sigSz = 1;
    ExpectIntEQ(wolfSSL_ED448_sign((byte*)msg, msglen, priv, privSz, sig,
        &sigSz), 0);
    sigSz = ED448_SIG_SIZE;

    /* Good case of signing. */
    ExpectIntEQ(wolfSSL_ED448_sign((byte*)msg, msglen, priv, privSz, sig,
        &sigSz), 1);
    ExpectIntEQ(sigSz, ED448_SIG_SIZE);

#ifdef HAVE_ED448_VERIFY
   /* Bad parameter testing of verification. */
    ExpectIntEQ(wolfSSL_ED448_verify(      NULL, msglen, NULL, pubSz, NULL,
        sigSz), 0);
    ExpectIntEQ(wolfSSL_ED448_verify((byte*)msg, msglen, NULL, pubSz, NULL,
        sigSz), 0);
    ExpectIntEQ(wolfSSL_ED448_verify(      NULL, msglen,  pub, pubSz, NULL,
        sigSz), 0);
    ExpectIntEQ(wolfSSL_ED448_verify(      NULL, msglen, NULL, pubSz,  sig,
        sigSz), 0);
    ExpectIntEQ(wolfSSL_ED448_verify(      NULL, msglen,  pub, pubSz,  sig,
        sigSz), 0);
    ExpectIntEQ(wolfSSL_ED448_verify((byte*)msg, msglen, NULL, pubSz,  sig,
        sigSz), 0);
    ExpectIntEQ(wolfSSL_ED448_verify((byte*)msg, msglen,  pub, pubSz, NULL,
        sigSz), 0);
    /*   Bad length. */
    pubSz = 1;
    ExpectIntEQ(wolfSSL_ED448_verify((byte*)msg, msglen, pub, pubSz, sig,
        sigSz), 0);
    pubSz = ED448_PUB_KEY_SIZE;
    sigSz = 1;
    ExpectIntEQ(wolfSSL_ED448_verify((byte*)msg, msglen, pub, pubSz, sig,
        sigSz), 0);
    sigSz = ED448_SIG_SIZE;

    /* Good case of verification. */
    ExpectIntEQ(wolfSSL_ED448_verify((byte*)msg, msglen, pub, pubSz, sig,
        sigSz), 1);
    /* Bad signature. */
    if (EXPECT_SUCCESS()) {
        sig[1] ^= 0x80;
    }
    ExpectIntEQ(wolfSSL_ED448_verify((byte*)msg, msglen, pub, pubSz, sig,
        sigSz), 0);
#endif /* HAVE_ED448_VERIFY */
#endif /* HAVE_ED448_SIGN && HAVE_ED448_KEY_IMPORT */
#endif /* HAVE_ED448 && HAVE_ED448_KEY_EXPORT && WOLFSSL_KEY_GEN */
    return EXPECT_RESULT();
}
#endif /* OPENSSL_EXTRA */

#include <wolfssl/openssl/pem.h>
/*----------------------------------------------------------------------------*
 | EVP
 *----------------------------------------------------------------------------*/

static int test_wolfSSL_EVP_PKEY_print_public(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_BIO)
    WOLFSSL_BIO* rbio = NULL;
    WOLFSSL_BIO* wbio = NULL;
    WOLFSSL_EVP_PKEY* pkey = NULL;
    char line[256] = { 0 };
    char line1[256] = { 0 };
    int i = 0;

    /* test error cases */
    ExpectIntEQ( EVP_PKEY_print_public(NULL,NULL,0,NULL),0L);

    /*
     *  test RSA public key print
     *  in this test, pass '3' for indent
     */
#if !defined(NO_RSA) && defined(USE_CERT_BUFFERS_1024)

    ExpectNotNull(rbio = BIO_new_mem_buf( client_keypub_der_1024,
        sizeof_client_keypub_der_1024));

    ExpectNotNull(wolfSSL_d2i_PUBKEY_bio(rbio, &pkey));

    ExpectNotNull(wbio = BIO_new(BIO_s_mem()));

    ExpectIntEQ(EVP_PKEY_print_public(wbio, pkey,3,NULL),1);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "   RSA Public-Key: (1024 bit)\n");
    ExpectIntEQ(XSTRNCMP(line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "   Modulus:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "       00:bc:73:0e:a8:49:f3:74:a2:a9:ef:18:a5:da:55:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);


    /* skip to the end of modulus element*/
    for (i = 0; i < 8 ;i++) {
        ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    }

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "   Exponent: 65537 (0x010001)\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);


    /* should reach EOF */
    ExpectIntLE(BIO_gets(wbio, line, sizeof(line)), 0);

    EVP_PKEY_free(pkey);
    pkey = NULL;
    BIO_free(rbio);
    BIO_free(wbio);
    rbio = NULL;
    wbio = NULL;

#endif  /* !NO_RSA && USE_CERT_BUFFERS_1024*/

    /*
     *  test DSA public key print
     */
#if !defined(NO_DSA) && defined(USE_CERT_BUFFERS_2048)
    ExpectNotNull(rbio = BIO_new_mem_buf( dsa_pub_key_der_2048,
        sizeof_dsa_pub_key_der_2048));

    ExpectNotNull(wolfSSL_d2i_PUBKEY_bio(rbio, &pkey));

    ExpectNotNull(wbio = BIO_new(BIO_s_mem()));

    ExpectIntEQ(EVP_PKEY_print_public(wbio, pkey,0,NULL),1);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "DSA Public-Key: (2048 bit)\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "pub:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1,
        "    00:C2:35:2D:EC:83:83:6C:73:13:9E:52:7C:74:C8:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of pub element*/
    for (i = 0; i < 17 ;i++) {
        ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    }

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "P:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of P element*/
    for (i = 0; i < 18 ;i++) {
        ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    }

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "Q:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of Q element*/
    for (i = 0; i < 3 ;i++) {
        ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    }
    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "G:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of G element*/
    for (i = 0; i < 18 ;i++) {
        ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    }
    /* should reach EOF */
    ExpectIntLE(BIO_gets(wbio, line, sizeof(line)), 0);

    EVP_PKEY_free(pkey);
    pkey = NULL;
    BIO_free(rbio);
    BIO_free(wbio);
    rbio = NULL;
    wbio = NULL;

#endif /* !NO_DSA && USE_CERT_BUFFERS_2048 */

    /*
     *  test ECC public key print
     */
#if defined(HAVE_ECC) && defined(USE_CERT_BUFFERS_256)

    ExpectNotNull(rbio = BIO_new_mem_buf( ecc_clikeypub_der_256,
        sizeof_ecc_clikeypub_der_256));

    ExpectNotNull(wolfSSL_d2i_PUBKEY_bio(rbio, &pkey));

    ExpectNotNull(wbio = BIO_new(BIO_s_mem()));

    ExpectIntEQ(EVP_PKEY_print_public(wbio, pkey,0,NULL),1);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    ExpectStrEQ(line, "Public-Key: (256 bit)\n");

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "pub:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1,
            "    04:55:BF:F4:0F:44:50:9A:3D:CE:9B:B7:F0:C5:4D:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of pub element*/
    for (i = 0; i < 4 ;i++) {
        ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    }

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "ASN1 OID: prime256v1\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "NIST CURVE: P-256\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);


    /* should reach EOF */
    ExpectIntLE(BIO_gets(wbio, line, sizeof(line)), 0);

    EVP_PKEY_free(pkey);
    pkey = NULL;
    BIO_free(rbio);
    BIO_free(wbio);
    rbio = NULL;
    wbio = NULL;

#endif /* HAVE_ECC && USE_CERT_BUFFERS_256 */

    /*
     *  test DH public key print
     */
#if defined(WOLFSSL_DH_EXTRA) && defined(USE_CERT_BUFFERS_2048)

    ExpectNotNull(rbio = BIO_new_mem_buf( dh_pub_key_der_2048,
        sizeof_dh_pub_key_der_2048));

    ExpectNotNull(wolfSSL_d2i_PUBKEY_bio(rbio, &pkey));

    ExpectNotNull(wbio = BIO_new(BIO_s_mem()));

    ExpectIntEQ(EVP_PKEY_print_public(wbio, pkey,0,NULL), 1);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "DH Public-Key: (2048 bit)\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "public-key:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1,
        "    34:41:BF:E9:F2:11:BF:05:DB:B2:72:A8:29:CC:BD:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of public-key element*/
    for (i = 0; i < 17 ;i++) {
        ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    }

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "prime:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1,
        "    00:D3:B2:99:84:5C:0A:4C:E7:37:CC:FC:18:37:01:\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* skip to the end of prime element*/
    for (i = 0; i < 17 ;i++) {
        ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    }

    ExpectIntGT(BIO_gets(wbio, line, sizeof(line)), 0);
    strcpy(line1, "generator: 2 (0x02)\n");
    ExpectIntEQ(XSTRNCMP( line, line1, XSTRLEN(line1)), 0);

    /* should reach EOF */
    ExpectIntLE(BIO_gets(wbio, line, sizeof(line)), 0);

    EVP_PKEY_free(pkey);
    pkey = NULL;
    BIO_free(rbio);
    BIO_free(wbio);
    rbio = NULL;
    wbio = NULL;

#endif /* WOLFSSL_DH_EXTRA && USE_CERT_BUFFERS_2048 */

    /* to prevent "unused variable" warning */
    (void)pkey;
    (void)wbio;
    (void)rbio;
    (void)line;
    (void)line1;
    (void)i;
#endif /* OPENSSL_EXTRA */
    return EXPECT_RESULT();
}
/* Test functions for base64 encode/decode */
static int test_wolfSSL_EVP_ENCODE_CTX_new(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && \
( defined(WOLFSSL_BASE64_ENCODE) || defined(WOLFSSL_BASE64_DECODE))
    EVP_ENCODE_CTX* ctx = NULL;

    ExpectNotNull(ctx = EVP_ENCODE_CTX_new());
    ExpectIntEQ(ctx->remaining,0);
    ExpectIntEQ(ctx->data[0],0);
    ExpectIntEQ(ctx->data[sizeof(ctx->data) -1],0);
    EVP_ENCODE_CTX_free(ctx);
#endif /* OPENSSL_EXTRA && (WOLFSSL_BASE64_ENCODE || WOLFSSL_BASE64_DECODE) */
    return EXPECT_RESULT();
}
static int test_wolfSSL_EVP_ENCODE_CTX_free(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && \
( defined(WOLFSSL_BASE64_ENCODE) || defined(WOLFSSL_BASE64_DECODE))
    EVP_ENCODE_CTX* ctx = NULL;

    ExpectNotNull(ctx = EVP_ENCODE_CTX_new());
    EVP_ENCODE_CTX_free(ctx);
#endif /* OPENSSL_EXTRA && (WOLFSSL_BASE64_ENCODE || WOLFSSL_BASE64_DECODE) */
    return EXPECT_RESULT();
}

static int test_wolfSSL_EVP_EncodeInit(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_BASE64_ENCODE)
    EVP_ENCODE_CTX* ctx = NULL;

    ExpectNotNull(ctx = EVP_ENCODE_CTX_new());
    ExpectIntEQ(ctx->remaining, 0);
    ExpectIntEQ(ctx->data[0], 0);
    ExpectIntEQ(ctx->data[sizeof(ctx->data) -1], 0);

    if (ctx != NULL) {
        /* make ctx dirty */
        ctx->remaining = 10;
        XMEMSET(ctx->data, 0x77, sizeof(ctx->data));
    }

    EVP_EncodeInit(ctx);

    ExpectIntEQ(ctx->remaining, 0);
    ExpectIntEQ(ctx->data[0], 0);
    ExpectIntEQ(ctx->data[sizeof(ctx->data) -1], 0);

    EVP_ENCODE_CTX_free(ctx);
#endif /* OPENSSL_EXTRA && WOLFSSL_BASE64_ENCODE*/
    return EXPECT_RESULT();
}
static int test_wolfSSL_EVP_EncodeUpdate(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_BASE64_ENCODE)
    int outl;
    int total;

    const unsigned char plain0[] = {"Th"};
    const unsigned char plain1[] = {"This is a base64 encodeing test."};
    const unsigned char plain2[] = {"This is additional data."};

    const unsigned char encBlock0[] = {"VGg="};
    const unsigned char enc0[]   = {"VGg=\n"};
    /* expected encoded result for the first output 64 chars plus trailing LF*/
    const unsigned char enc1[]   = {"VGhpcyBpcyBhIGJhc2U2NCBlbmNvZGVpbmcgdGVzdC5UaGlzIGlzIGFkZGl0aW9u\n"};

    const unsigned char enc2[]   =
    {"VGhpcyBpcyBhIGJhc2U2NCBlbmNvZGVpbmcgdGVzdC5UaGlzIGlzIGFkZGl0aW9u\nYWwgZGF0YS4=\n"};

    unsigned char encOutBuff[300];

    EVP_ENCODE_CTX* ctx = NULL;

    ExpectNotNull(ctx = EVP_ENCODE_CTX_new());

    EVP_EncodeInit(ctx);

    /* illegal parameter test */
    ExpectIntEQ(
        EVP_EncodeUpdate(
            NULL,            /* pass NULL as ctx */
            encOutBuff,
            &outl,
            plain1,
            sizeof(plain1)-1),
        0                    /* expected result code 0: fail */
    );

    ExpectIntEQ(
        EVP_EncodeUpdate(
            ctx,
            NULL,           /* pass NULL as out buff */
            &outl,
            plain1,
            sizeof(plain1)-1),
        0                    /* expected result code 0: fail */
    );

    ExpectIntEQ(
        EVP_EncodeUpdate(
            ctx,
            encOutBuff,
            NULL,            /* pass NULL as outl */
            plain1,
            sizeof(plain1)-1),
        0                    /* expected result code 0: fail */
    );

    ExpectIntEQ(
        EVP_EncodeUpdate(
            ctx,
            encOutBuff,
            &outl,
            NULL,            /* pass NULL as in */
            sizeof(plain1)-1),
        0                    /* expected result code 0: fail */
    );

    ExpectIntEQ(EVP_EncodeBlock(NULL, NULL, 0), -1);

    /* meaningless parameter test */

    ExpectIntEQ(
        EVP_EncodeUpdate(
            ctx,
            encOutBuff,
            &outl,
            plain1,
            0),              /* pass zero input */
        1                    /* expected result code 1: success */
    );

    /* very small data encoding test */

    EVP_EncodeInit(ctx);

    ExpectIntEQ(
        EVP_EncodeUpdate(
            ctx,
            encOutBuff,
            &outl,
            plain0,
            sizeof(plain0)-1),
        1                    /* expected result code 1: success */
    );
    ExpectIntEQ(outl,0);

    if (EXPECT_SUCCESS()) {
        EVP_EncodeFinal(
            ctx,
            encOutBuff + outl,
            &outl);
    }

    ExpectIntEQ( outl, sizeof(enc0)-1);
    ExpectIntEQ(
        XSTRNCMP(
            (const char*)encOutBuff,
            (const char*)enc0,sizeof(enc0) ),
    0);

    XMEMSET( encOutBuff,0, sizeof(encOutBuff));
    ExpectIntEQ(EVP_EncodeBlock(encOutBuff, plain0, sizeof(plain0)-1),
                sizeof(encBlock0)-1);
    ExpectStrEQ(encOutBuff, encBlock0);

    /* pass small size( < 48bytes ) input, then make sure they are not
     * encoded  and just stored in ctx
     */

    EVP_EncodeInit(ctx);

    total = 0;
    outl = 0;
    XMEMSET( encOutBuff,0, sizeof(encOutBuff));

    ExpectIntEQ(
    EVP_EncodeUpdate(
        ctx,
        encOutBuff,         /* buffer for output */
        &outl,              /* size of output */
        plain1,             /* input */
        sizeof(plain1)-1),  /* size of input */
        1);                 /* expected result code 1:success */

    total += outl;

    ExpectIntEQ(outl, 0);  /* no output expected */
    ExpectIntEQ(ctx->remaining, sizeof(plain1) -1);
    ExpectTrue(
        XSTRNCMP((const char*)(ctx->data),
                 (const char*)plain1,
                 ctx->remaining) ==0 );
    ExpectTrue(encOutBuff[0] == 0);

    /* call wolfSSL_EVP_EncodeUpdate again to make it encode
     * the stored data and the new input together
     */
    ExpectIntEQ(
    EVP_EncodeUpdate(
        ctx,
        encOutBuff + outl,  /* buffer for output */
        &outl,              /* size of output */
        plain2,             /* additional input */
        sizeof(plain2) -1), /* size of additional input */
        1);                 /* expected result code 1:success */

    total += outl;

    ExpectIntNE(outl, 0);   /* some output is expected this time*/
    ExpectIntEQ(outl, BASE64_ENCODE_RESULT_BLOCK_SIZE +1); /* 64 bytes and LF */
    ExpectIntEQ(
        XSTRNCMP((const char*)encOutBuff,(const char*)enc1,sizeof(enc1) ),0);

    /* call wolfSSL_EVP_EncodeFinal to flush all the unprocessed input */
    EVP_EncodeFinal(
        ctx,
        encOutBuff + outl,
        &outl);

    total += outl;

    ExpectIntNE(total,0);
    ExpectIntNE(outl,0);
    ExpectIntEQ(XSTRNCMP(
        (const char*)encOutBuff,(const char*)enc2,sizeof(enc2) ),0);

    /* test with illeagal parameters */
    outl = 1;
    EVP_EncodeFinal(NULL, encOutBuff + outl, &outl);
    ExpectIntEQ(outl, 0);
    outl = 1;
    EVP_EncodeFinal(ctx, NULL, &outl);
    ExpectIntEQ(outl, 0);
    EVP_EncodeFinal(ctx, encOutBuff + outl, NULL);
    EVP_EncodeFinal(NULL, NULL, NULL);

    EVP_ENCODE_CTX_free(ctx);
#endif /* OPENSSL_EXTRA && WOLFSSL_BASE64_ENCODE*/
    return EXPECT_RESULT();
}
static int test_wolfSSL_EVP_EncodeFinal(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_BASE64_ENCODE)
    /* tests for wolfSSL_EVP_EncodeFinal are included in
     * test_wolfSSL_EVP_EncodeUpdate
     */
    res = TEST_SUCCESS;
#endif /* OPENSSL_EXTRA && WOLFSSL_BASE64_ENCODE*/
    return res;
}


static int test_wolfSSL_EVP_DecodeInit(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_BASE64_DECODE)
    EVP_ENCODE_CTX* ctx = NULL;

    ExpectNotNull( ctx = EVP_ENCODE_CTX_new());
    ExpectIntEQ( ctx->remaining,0);
    ExpectIntEQ( ctx->data[0],0);
    ExpectIntEQ( ctx->data[sizeof(ctx->data) -1],0);

    if (ctx != NULL) {
        /* make ctx dirty */
        ctx->remaining = 10;
        XMEMSET( ctx->data, 0x77, sizeof(ctx->data));
    }

    EVP_DecodeInit(ctx);

    ExpectIntEQ( ctx->remaining,0);
    ExpectIntEQ( ctx->data[0],0);
    ExpectIntEQ( ctx->data[sizeof(ctx->data) -1],0);

    EVP_ENCODE_CTX_free(ctx);
#endif /* OPENSSL && WOLFSSL_BASE_DECODE */
    return EXPECT_RESULT();
}
static int test_wolfSSL_EVP_DecodeUpdate(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_BASE64_DECODE)
    int outl;
    unsigned char decOutBuff[300];

    EVP_ENCODE_CTX* ctx = NULL;

    static const unsigned char enc1[]   =
            {"VGhpcyBpcyBhIGJhc2U2NCBkZWNvZGluZyB0ZXN0Lg==\n"};
/*    const unsigned char plain1[] =
    {"This is a base64 decoding test."} */

    ExpectNotNull(ctx = EVP_ENCODE_CTX_new());

    EVP_DecodeInit(ctx);

    /* illegal parameter tests */

    /* pass NULL as ctx */
    ExpectIntEQ(
        EVP_DecodeUpdate(
            NULL,            /* pass NULL as ctx */
            decOutBuff,
            &outl,
            enc1,
            sizeof(enc1)-1),
        -1                    /* expected result code -1: fail */
    );
    ExpectIntEQ( outl, 0);

    /* pass NULL as output */
    ExpectIntEQ(
        EVP_DecodeUpdate(
            ctx,
            NULL,           /* pass NULL as out buff */
            &outl,
            enc1,
            sizeof(enc1)-1),
        -1                    /* expected result code -1: fail */
    );
    ExpectIntEQ( outl, 0);

    /* pass NULL as outl */
    ExpectIntEQ(
        EVP_DecodeUpdate(
            ctx,
            decOutBuff,
            NULL,            /* pass NULL as outl */
            enc1,
            sizeof(enc1)-1),
        -1                   /* expected result code -1: fail */
    );

    /* pass NULL as input */
    ExpectIntEQ(
        EVP_DecodeUpdate(
            ctx,
            decOutBuff,
            &outl,
            NULL,            /* pass NULL as in */
            sizeof(enc1)-1),
        -1                    /* expected result code -1: fail */
    );
    ExpectIntEQ( outl, 0);

    ExpectIntEQ(EVP_DecodeBlock(NULL, NULL, 0), -1);

    /* pass zero length input */

    ExpectIntEQ(
        EVP_DecodeUpdate(
            ctx,
            decOutBuff,
            &outl,
            enc1,
            0),              /* pass zero as input len */
        1                    /* expected result code 1: success */
    );

    /* decode correct base64 string */

    {
        static const unsigned char enc2[]   =
        {"VGhpcyBpcyBhIGJhc2U2NCBkZWNvZGluZyB0ZXN0Lg==\n"};
        static const unsigned char plain2[] =
        {"This is a base64 decoding test."};

        EVP_EncodeInit(ctx);

        ExpectIntEQ(
            EVP_DecodeUpdate(
                ctx,
                decOutBuff,
                &outl,
                enc2,
                sizeof(enc2)-1),
            0                    /* expected result code 0: success */
            );

        ExpectIntEQ(outl,sizeof(plain2) -1);

        ExpectIntEQ(
            EVP_DecodeFinal(
                ctx,
                decOutBuff + outl,
                &outl),
            1                    /* expected result code 1: success */
            );
        ExpectIntEQ(outl, 0);   /* expected DecodeFinal output no data */

        ExpectIntEQ(XSTRNCMP( (const char*)plain2,(const char*)decOutBuff,
                              sizeof(plain2) -1 ),0);
        ExpectIntEQ(EVP_DecodeBlock(decOutBuff, enc2, sizeof(enc2)),
                    sizeof(plain2)-1);
        ExpectIntEQ(XSTRNCMP( (const char*)plain2,(const char*)decOutBuff,
                              sizeof(plain2) -1 ),0);
    }

    /* decode correct base64 string which does not have '\n' in its last*/

    {
        static const unsigned char enc3[]   =
        {"VGhpcyBpcyBhIGJhc2U2NCBkZWNvZGluZyB0ZXN0Lg=="}; /* 44 chars */
        static const unsigned char plain3[] =
        {"This is a base64 decoding test."}; /* 31 chars */

        EVP_EncodeInit(ctx);

        ExpectIntEQ(
            EVP_DecodeUpdate(
                ctx,
                decOutBuff,
                &outl,
                enc3,
                sizeof(enc3)-1),
            0                    /* expected result code 0: success */
            );

        ExpectIntEQ(outl,sizeof(plain3)-1);   /* 31 chars should be output */

        ExpectIntEQ(XSTRNCMP( (const char*)plain3,(const char*)decOutBuff,
                              sizeof(plain3) -1 ),0);

        ExpectIntEQ(
            EVP_DecodeFinal(
                ctx,
                decOutBuff + outl,
                &outl),
            1                    /* expected result code 1: success */
            );

        ExpectIntEQ(outl,0 );

        ExpectIntEQ(EVP_DecodeBlock(decOutBuff, enc3, sizeof(enc3)-1),
                    sizeof(plain3)-1);
        ExpectIntEQ(XSTRNCMP( (const char*)plain3,(const char*)decOutBuff,
                              sizeof(plain3) -1 ),0);
    }

    /* decode string which has a padding char ('=') in the illegal position*/

    {
        static const unsigned char enc4[]   =
            {"VGhpcyBpcyBhIGJhc2U2N=CBkZWNvZGluZyB0ZXN0Lg==\n"};

        EVP_EncodeInit(ctx);

        ExpectIntEQ(
            EVP_DecodeUpdate(
                ctx,
                decOutBuff,
                &outl,
                enc4,
                sizeof(enc4)-1),
            -1                    /* expected result code -1: error */
            );
        ExpectIntEQ(outl,0);
        ExpectIntEQ(EVP_DecodeBlock(decOutBuff, enc4, sizeof(enc4)-1), -1);
    }

    /* small data decode test */

    {
        static const unsigned char enc00[]   = {"VG"};
        static const unsigned char enc01[]   = {"g=\n"};
        static const unsigned char plain4[]  = {"Th"};

        EVP_EncodeInit(ctx);

        ExpectIntEQ(
            EVP_DecodeUpdate(
                ctx,
                decOutBuff,
                &outl,
                enc00,
                sizeof(enc00)-1),
            1                    /* expected result code 1: success */
            );
        ExpectIntEQ(outl,0);

        ExpectIntEQ(
            EVP_DecodeUpdate(
                ctx,
                decOutBuff + outl,
                &outl,
                enc01,
                sizeof(enc01)-1),
            0                    /* expected result code 0: success */
            );

        ExpectIntEQ(outl,sizeof(plain4)-1);

        /* test with illegal parameters */
        ExpectIntEQ(EVP_DecodeFinal(NULL,decOutBuff + outl,&outl), -1);
        ExpectIntEQ(EVP_DecodeFinal(ctx,NULL,&outl), -1);
        ExpectIntEQ(EVP_DecodeFinal(ctx,decOutBuff + outl, NULL), -1);
        ExpectIntEQ(EVP_DecodeFinal(NULL,NULL, NULL), -1);

        if (EXPECT_SUCCESS()) {
            EVP_DecodeFinal(
                ctx,
                decOutBuff + outl,
                &outl);
        }

        ExpectIntEQ( outl, 0);
        ExpectIntEQ(
            XSTRNCMP(
                (const char*)decOutBuff,
                (const char*)plain4,sizeof(plain4)-1 ),
            0);
    }

    EVP_ENCODE_CTX_free(ctx);
#endif /* OPENSSL && WOLFSSL_BASE_DECODE */
    return EXPECT_RESULT();
}
static int test_wolfSSL_EVP_DecodeFinal(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_BASE64_DECODE)
    /* tests for wolfSSL_EVP_DecodeFinal are included in
     * test_wolfSSL_EVP_DecodeUpdate
     */
    res = TEST_SUCCESS;
#endif /* OPENSSL && WOLFSSL_BASE_DECODE */
    return res;
}

/* Test function for wolfSSL_EVP_get_cipherbynid.
 */

#ifdef OPENSSL_EXTRA
static int test_wolfSSL_EVP_get_cipherbynid(void)
{
    EXPECT_DECLS;
#ifndef NO_AES
    const WOLFSSL_EVP_CIPHER* c;

    c = wolfSSL_EVP_get_cipherbynid(419);
    #if (defined(HAVE_AES_CBC) || defined(WOLFSSL_AES_DIRECT)) && \
         defined(WOLFSSL_AES_128)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_128_CBC", c));
    #else
        ExpectNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(423);
    #if (defined(HAVE_AES_CBC) || defined(WOLFSSL_AES_DIRECT)) && \
         defined(WOLFSSL_AES_192)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_192_CBC", c));
    #else
        ExpectNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(427);
    #if (defined(HAVE_AES_CBC) || defined(WOLFSSL_AES_DIRECT)) && \
         defined(WOLFSSL_AES_256)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_256_CBC", c));
    #else
        ExpectNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(904);
    #if defined(WOLFSSL_AES_COUNTER) && defined(WOLFSSL_AES_128)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_128_CTR", c));
    #else
        ExpectNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(905);
    #if defined(WOLFSSL_AES_COUNTER) && defined(WOLFSSL_AES_192)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_192_CTR", c));
    #else
        ExpectNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(906);
    #if defined(WOLFSSL_AES_COUNTER) && defined(WOLFSSL_AES_256)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_256_CTR", c));
    #else
        ExpectNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(418);
    #if defined(HAVE_AES_ECB) && defined(WOLFSSL_AES_128)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_128_ECB", c));
    #else
        ExpectNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(422);
    #if defined(HAVE_AES_ECB) && defined(WOLFSSL_AES_192)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_192_ECB", c));
    #else
        ExpectNull(c);
    #endif

    c = wolfSSL_EVP_get_cipherbynid(426);
    #if defined(HAVE_AES_ECB) && defined(WOLFSSL_AES_256)
        ExpectNotNull(c);
        ExpectNotNull(XSTRCMP("EVP_AES_256_ECB", c));
    #else
        ExpectNull(c);
    #endif
#endif /* !NO_AES */

#ifndef NO_DES3
    ExpectNotNull(XSTRCMP("EVP_DES_CBC", wolfSSL_EVP_get_cipherbynid(31)));
#ifdef WOLFSSL_DES_ECB
    ExpectNotNull(XSTRCMP("EVP_DES_ECB", wolfSSL_EVP_get_cipherbynid(29)));
#endif
    ExpectNotNull(XSTRCMP("EVP_DES_EDE3_CBC", wolfSSL_EVP_get_cipherbynid(44)));
#ifdef WOLFSSL_DES_ECB
    ExpectNotNull(XSTRCMP("EVP_DES_EDE3_ECB", wolfSSL_EVP_get_cipherbynid(33)));
#endif
#endif /* !NO_DES3 */

#if defined(HAVE_CHACHA) && defined(HAVE_POLY1305)
    ExpectNotNull(XSTRCMP("EVP_CHACHA20_POLY13O5", EVP_get_cipherbynid(1018)));
#endif

    /* test for nid is out of range */
    ExpectNull(wolfSSL_EVP_get_cipherbynid(1));

    return EXPECT_RESULT();
}

static int test_wolfSSL_EVP_CIPHER_CTX(void)
{
    EXPECT_DECLS;
#if !defined(NO_AES) && defined(HAVE_AES_CBC) && defined(WOLFSSL_AES_128)
    EVP_CIPHER_CTX *ctx = EVP_CIPHER_CTX_new();
    const EVP_CIPHER *init = EVP_aes_128_cbc();
    const EVP_CIPHER *test;
    byte key[AES_BLOCK_SIZE] = {0};
    byte iv[AES_BLOCK_SIZE] = {0};

    ExpectNotNull(ctx);
    wolfSSL_EVP_CIPHER_CTX_init(ctx);
    ExpectIntEQ(EVP_CipherInit(ctx, init, key, iv, 1), WOLFSSL_SUCCESS);
    test = EVP_CIPHER_CTX_cipher(ctx);
    ExpectTrue(init == test);
    ExpectIntEQ(EVP_CIPHER_nid(test), NID_aes_128_cbc);

    ExpectIntEQ(EVP_CIPHER_CTX_reset(ctx), WOLFSSL_SUCCESS);
    ExpectIntEQ(EVP_CIPHER_CTX_reset(NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    EVP_CIPHER_CTX_free(ctx);
    /* test EVP_CIPHER_CTX_cleanup with NULL */
    ExpectIntEQ(EVP_CIPHER_CTX_cleanup(NULL), WOLFSSL_SUCCESS);
#endif /* !NO_AES && HAVE_AES_CBC && WOLFSSL_AES_128 */
    return EXPECT_RESULT();
}
#endif /* OPENSSL_EXTRA */

/*----------------------------------------------------------------------------*
 | IO
 *----------------------------------------------------------------------------*/

#if defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) || \
    defined(HAVE_IO_TESTS_DEPENDENCIES)
#ifdef WOLFSSL_HAVE_TLS_UNIQUE
    byte server_side_msg1[WC_MAX_DIGEST_SIZE]; /* msg sent by server */
    byte server_side_msg2[WC_MAX_DIGEST_SIZE]; /* msg received from client */
    byte client_side_msg1[WC_MAX_DIGEST_SIZE]; /* msg sent by client */
    byte client_side_msg2[WC_MAX_DIGEST_SIZE]; /* msg received from server */
#endif /* WOLFSSL_HAVE_TLS_UNIQUE */

/* TODO: Expand and enable this when EVP_chacha20_poly1305 is supported */
#if defined(HAVE_SESSION_TICKET) && defined(OPENSSL_EXTRA) && \
    defined(HAVE_AES_CBC)

    typedef struct openssl_key_ctx {
        byte name[WOLFSSL_TICKET_NAME_SZ]; /* server name */
        byte key[WOLFSSL_TICKET_KEY_SZ]; /* cipher key */
        byte hmacKey[WOLFSSL_TICKET_NAME_SZ]; /* hmac key */
        byte iv[WOLFSSL_TICKET_IV_SZ]; /* cipher iv */
    } openssl_key_ctx;

    static THREAD_LS_T openssl_key_ctx myOpenSSLKey_ctx;
    static THREAD_LS_T WC_RNG myOpenSSLKey_rng;

    static WC_INLINE int OpenSSLTicketInit(void)
    {
        int ret = wc_InitRng(&myOpenSSLKey_rng);
        if (ret != 0) return ret;

        ret = wc_RNG_GenerateBlock(&myOpenSSLKey_rng, myOpenSSLKey_ctx.name,
                sizeof(myOpenSSLKey_ctx.name));
        if (ret != 0) return ret;

        ret = wc_RNG_GenerateBlock(&myOpenSSLKey_rng, myOpenSSLKey_ctx.key,
                sizeof(myOpenSSLKey_ctx.key));
        if (ret != 0) return ret;

        ret = wc_RNG_GenerateBlock(&myOpenSSLKey_rng, myOpenSSLKey_ctx.hmacKey,
                sizeof(myOpenSSLKey_ctx.hmacKey));
        if (ret != 0) return ret;

        ret = wc_RNG_GenerateBlock(&myOpenSSLKey_rng, myOpenSSLKey_ctx.iv,
                sizeof(myOpenSSLKey_ctx.iv));
        if (ret != 0) return ret;

        return 0;
    }

    static int myTicketEncCbOpenSSL(WOLFSSL* ssl,
                             byte name[WOLFSSL_TICKET_NAME_SZ],
                             byte iv[WOLFSSL_TICKET_IV_SZ],
                             WOLFSSL_EVP_CIPHER_CTX *ectx,
                             WOLFSSL_HMAC_CTX *hctx, int enc) {
        (void)ssl;
        if (enc) {
            XMEMCPY(name, myOpenSSLKey_ctx.name, sizeof(myOpenSSLKey_ctx.name));
            XMEMCPY(iv, myOpenSSLKey_ctx.iv, sizeof(myOpenSSLKey_ctx.iv));
        }
        else if (XMEMCMP(name, myOpenSSLKey_ctx.name,
                            sizeof(myOpenSSLKey_ctx.name)) != 0 ||
                 XMEMCMP(iv, myOpenSSLKey_ctx.iv,
                            sizeof(myOpenSSLKey_ctx.iv)) != 0) {
            return 0;
        }
        HMAC_Init_ex(hctx, myOpenSSLKey_ctx.hmacKey, WOLFSSL_TICKET_NAME_SZ, EVP_sha256(), NULL);
        if (enc)
            EVP_EncryptInit_ex(ectx, EVP_aes_256_cbc(), NULL, myOpenSSLKey_ctx.key, iv);
        else
            EVP_DecryptInit_ex(ectx, EVP_aes_256_cbc(), NULL, myOpenSSLKey_ctx.key, iv);
        return 1;
    }

    static WC_INLINE void OpenSSLTicketCleanup(void)
    {
        wc_FreeRng(&myOpenSSLKey_rng);
    }
#endif
#endif

/* helper functions */
#ifdef HAVE_SSL_MEMIO_TESTS_DEPENDENCIES
static WC_INLINE int test_ssl_memio_write_cb(WOLFSSL *ssl, char *data, int sz,
    void *ctx)
{
    struct test_ssl_memio_ctx *test_ctx;
    byte *buf;
    int *len;

    test_ctx = (struct test_ssl_memio_ctx*)ctx;

    if (wolfSSL_GetSide(ssl) == WOLFSSL_SERVER_END) {
        buf = test_ctx->c_buff;
        len = &test_ctx->c_len;
    }
    else {
        buf = test_ctx->s_buff;
        len = &test_ctx->s_len;
    }

    if ((unsigned)(*len + sz) > TEST_SSL_MEMIO_BUF_SZ)
        return WOLFSSL_CBIO_ERR_WANT_WRITE;

    XMEMCPY(buf + *len, data, sz);
    *len += sz;

#ifdef WOLFSSL_DUMP_MEMIO_STREAM
    {
        /* This can be imported into Wireshark by transforming the file with
         *   od -Ax -tx1 -v test_output.dump > test_output.dump.hex
         * And then loading test_output.dump.hex into Wireshark using the
         * "Import from Hex Dump..." option ion and selecting the TCP
         * encapsulation option. */
        char dump_file_name[64];
        WOLFSSL_BIO *dump_file;
        sprintf(dump_file_name, "%s/%s.dump", tmpDirName, currentTestName);
        dump_file = wolfSSL_BIO_new_file(dump_file_name, "a");
        if (dump_file != NULL) {
            (void)wolfSSL_BIO_write(dump_file, data, sz);
            wolfSSL_BIO_free(dump_file);
        }
    }
#endif

    return sz;
}

static WC_INLINE int test_ssl_memio_read_cb(WOLFSSL *ssl, char *data, int sz,
    void *ctx)
{
    struct test_ssl_memio_ctx *test_ctx;
    int read_sz;
    byte *buf;
    int *len;

    test_ctx = (struct test_ssl_memio_ctx*)ctx;

    if (wolfSSL_GetSide(ssl) == WOLFSSL_SERVER_END) {
        buf = test_ctx->s_buff;
        len = &test_ctx->s_len;
    }
    else {
        buf = test_ctx->c_buff;
        len = &test_ctx->c_len;
    }

    if (*len == 0)
        return WOLFSSL_CBIO_ERR_WANT_READ;

    read_sz = sz < *len ? sz : *len;

    XMEMCPY(data, buf, read_sz);
    XMEMMOVE(buf, buf + read_sz, *len - read_sz);

    *len -= read_sz;

    return read_sz;
}

int test_ssl_memio_setup(test_ssl_memio_ctx *ctx)
{
    EXPECT_DECLS_NO_MSGS(-2000);
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    int c_sharedCtx = 0;
    int s_sharedCtx = 0;
#endif
    const char* clientCertFile = cliCertFile;
    const char* clientKeyFile = cliKeyFile;
    const char* serverCertFile = svrCertFile;
    const char* serverKeyFile = svrKeyFile;

    /********************************
     * Create WOLFSSL_CTX for client.
     ********************************/
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (ctx->c_ctx != NULL) {
        c_sharedCtx = ctx->c_cb.isSharedCtx;
    }
    else
#endif
    {
        WOLFSSL_METHOD* method  = NULL;
        if (ctx->c_cb.method != NULL) {
            method = ctx->c_cb.method();
        }
        else {
            method = wolfSSLv23_client_method();
        }
        ExpectNotNull(ctx->c_ctx = wolfSSL_CTX_new(method));
    }
    wolfSSL_SetIORecv(ctx->c_ctx, test_ssl_memio_read_cb);
    wolfSSL_SetIOSend(ctx->c_ctx, test_ssl_memio_write_cb);
#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx->c_ctx, PasswordCallBack);
#endif
    if (ctx->c_cb.caPemFile != NULL)
        ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx->c_ctx,
                ctx->c_cb.caPemFile, 0), WOLFSSL_SUCCESS);
    else
        ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx->c_ctx,
                caCertFile, 0), WOLFSSL_SUCCESS);
    if (ctx->c_cb.certPemFile != NULL) {
        clientCertFile = ctx->c_cb.certPemFile;
    }
    if (ctx->c_cb.keyPemFile != NULL) {
        clientKeyFile = ctx->c_cb.keyPemFile;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (!c_sharedCtx)
#endif
    {
        ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_file(ctx->c_ctx,
            clientCertFile), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_file(ctx->c_ctx, clientKeyFile,
            WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    }
#ifdef HAVE_CRL
    if (ctx->c_cb.crlPemFile != NULL) {
        ExpectIntEQ(wolfSSL_CTX_EnableCRL(ctx->c_ctx, WOLFSSL_CRL_CHECKALL),
            WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_CTX_LoadCRLFile(ctx->c_ctx, ctx->c_cb.crlPemFile,
            WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    }
#endif
    if (ctx->c_ciphers != NULL) {
        ExpectIntEQ(wolfSSL_CTX_set_cipher_list(ctx->c_ctx, ctx->c_ciphers),
            WOLFSSL_SUCCESS);
    }
    if (ctx->c_cb.ctx_ready != NULL) {
        ExpectIntEQ(ctx->c_cb.ctx_ready(ctx->c_ctx), TEST_SUCCESS);
    }


    /********************************
     * Create WOLFSSL_CTX for server.
     ********************************/
    if (ctx->s_ctx != NULL) {
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
        s_sharedCtx = 1;
#endif
        ctx->s_cb.isSharedCtx = 1;
    }
    else
    {
        WOLFSSL_METHOD* method  = NULL;
        if (ctx->s_cb.method != NULL) {
            method = ctx->s_cb.method();
        }
        else {
            method = wolfSSLv23_server_method();
        }
        ExpectNotNull(ctx->s_ctx = wolfSSL_CTX_new(method));
        ctx->s_cb.isSharedCtx = 0;
    }
    if (!ctx->s_cb.ticNoInit && (ctx->s_ctx != NULL)) {
#if defined(HAVE_SESSION_TICKET) && \
    ((defined(HAVE_CHACHA) && defined(HAVE_POLY1305)) || defined(HAVE_AESGCM))
#if defined(OPENSSL_EXTRA) && defined(HAVE_AES_CBC)
        OpenSSLTicketInit();
        wolfSSL_CTX_set_tlsext_ticket_key_cb(ctx->s_ctx, myTicketEncCbOpenSSL);
#elif defined(WOLFSSL_NO_DEF_TICKET_ENC_CB)
        TicketInit();
        wolfSSL_CTX_set_TicketEncCb(ctx->s_ctx, myTicketEncCb);
#endif
#endif
    }
    wolfSSL_SetIORecv(ctx->s_ctx, test_ssl_memio_read_cb);
    wolfSSL_SetIOSend(ctx->s_ctx, test_ssl_memio_write_cb);
    wolfSSL_CTX_set_verify(ctx->s_ctx, WOLFSSL_VERIFY_PEER |
        WOLFSSL_VERIFY_FAIL_IF_NO_PEER_CERT, 0);
    if (ctx->s_cb.caPemFile != NULL)
        ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx->s_ctx,
                ctx->s_cb.caPemFile, 0), WOLFSSL_SUCCESS);
    else
        ExpectIntEQ(wolfSSL_CTX_load_verify_locations(ctx->s_ctx,
                cliCertFile, 0), WOLFSSL_SUCCESS);
#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx->s_ctx, PasswordCallBack);
#endif
    if (ctx->s_cb.certPemFile != NULL) {
        serverCertFile = ctx->s_cb.certPemFile;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (!s_sharedCtx)
#endif
    {
        ExpectIntEQ(wolfSSL_CTX_use_certificate_chain_file(ctx->s_ctx,
            serverCertFile), WOLFSSL_SUCCESS);
    }
    if (ctx->s_cb.keyPemFile != NULL) {
        serverKeyFile = ctx->s_cb.keyPemFile;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (!s_sharedCtx)
#endif
    {
        ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_file(ctx->s_ctx, serverKeyFile,
            WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    }
    if (ctx->s_ciphers != NULL) {
        ExpectIntEQ(wolfSSL_CTX_set_cipher_list(ctx->s_ctx, ctx->s_ciphers),
            WOLFSSL_SUCCESS);
    }
    if (ctx->s_cb.ctx_ready != NULL) {
        ExpectIntEQ(ctx->s_cb.ctx_ready(ctx->s_ctx), TEST_SUCCESS);
    }


    /****************************
     * Create WOLFSSL for client.
     ****************************/
    ExpectNotNull(ctx->c_ssl = wolfSSL_new(ctx->c_ctx));
    wolfSSL_SetIOWriteCtx(ctx->c_ssl, ctx);
    wolfSSL_SetIOReadCtx(ctx->c_ssl, ctx);
    if (0
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
     || c_sharedCtx
#endif
        )
    {
        ExpectIntEQ(wolfSSL_use_certificate_chain_file(ctx->c_ssl,
                clientCertFile), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_use_PrivateKey_file(ctx->c_ssl, clientKeyFile,
            WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    }
    if (ctx->c_cb.ssl_ready != NULL) {
        ExpectIntEQ(ctx->c_cb.ssl_ready(ctx->c_ssl), TEST_SUCCESS);
    }

    /****************************
     * Create WOLFSSL for server.
     ****************************/
    ExpectNotNull(ctx->s_ssl = wolfSSL_new(ctx->s_ctx));
    wolfSSL_SetIOWriteCtx(ctx->s_ssl, ctx);
    wolfSSL_SetIOReadCtx(ctx->s_ssl, ctx);
    if (0
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
     || s_sharedCtx
#endif
        )
    {
        ExpectIntEQ(wolfSSL_use_certificate_chain_file(ctx->s_ssl,
                serverCertFile), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_use_PrivateKey_file(ctx->s_ssl, serverKeyFile,
            WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
    }
#if !defined(NO_FILESYSTEM) && !defined(NO_DH)
    wolfSSL_SetTmpDH_file(ctx->s_ssl, dhParamFile, WOLFSSL_FILETYPE_PEM);
#elif !defined(NO_DH)
    /* will repick suites with DHE, higher priority than PSK */
    SetDH(ctx->s_ssl);
#endif
    if (ctx->s_cb.ssl_ready != NULL) {
        ExpectIntEQ(ctx->s_cb.ssl_ready(ctx->s_ssl), TEST_SUCCESS);
    }

    return EXPECT_RESULT();
}

int test_ssl_memio_do_handshake(test_ssl_memio_ctx* ctx, int max_rounds,
    int* rounds)
{
    int handshake_complete = 0;
    int hs_c = 0;
    int hs_s = 0;
    int failing_s = 0;
    int failing_c = 0;
    int ret;
    int err;

    if (rounds != NULL) {
        *rounds = 0;
    }
    while ((!handshake_complete) && (max_rounds > 0)) {
        if (!hs_c) {
            wolfSSL_SetLoggingPrefix("client");
            ret = wolfSSL_connect(ctx->c_ssl);
            wolfSSL_SetLoggingPrefix(NULL);
            if (ret == WOLFSSL_SUCCESS) {
                hs_c = 1;
            }
            else {
                err = wolfSSL_get_error(ctx->c_ssl, ret);
                if (err != WOLFSSL_ERROR_WANT_READ &&
                    err != WOLFSSL_ERROR_WANT_WRITE) {
                    char buff[WOLFSSL_MAX_ERROR_SZ];
                    fprintf(stderr, "error = %d, %s\n", err,
                        wolfSSL_ERR_error_string((word32)err, buff));
                    failing_c = 1;
                    hs_c = 1;
                    if (failing_c && failing_s) {
                        break;
                    }
                }
            }
        }
        if (!hs_s) {
            wolfSSL_SetLoggingPrefix("server");
            ret = wolfSSL_accept(ctx->s_ssl);
            wolfSSL_SetLoggingPrefix(NULL);
            if (ret == WOLFSSL_SUCCESS) {
                hs_s = 1;
            }
            else {
                err = wolfSSL_get_error(ctx->s_ssl, ret);
                if (err != WOLFSSL_ERROR_WANT_READ &&
                    err != WOLFSSL_ERROR_WANT_WRITE) {
                    char buff[WOLFSSL_MAX_ERROR_SZ];
                    fprintf(stderr, "error = %d, %s\n", err,
                        wolfSSL_ERR_error_string((word32)err, buff));
                    failing_s = 1;
                    hs_s = 1;
                    if (failing_c && failing_s) {
                        break;
                    }
                }
            }
        }
        handshake_complete = hs_c && hs_s;
        max_rounds--;
        if (rounds != NULL) {
            *rounds += 1;
        }
    }

    if (!handshake_complete || failing_c || failing_s) {
        return TEST_FAIL;
    }

    return TEST_SUCCESS;
}

static int test_ssl_memio_read_write(test_ssl_memio_ctx* ctx)
{
    EXPECT_DECLS_NO_MSGS(-3000);
    char input[1024];
    int idx = 0;
    const char* msg_c = "hello wolfssl!";
    int msglen_c = (int)XSTRLEN(msg_c);
    const char* msg_s = "I hear you fa shizzle!";
    int msglen_s = (int)XSTRLEN(msg_s);

    if (ctx->c_msg != NULL) {
        msg_c = ctx->c_msg;
        msglen_c = ctx->c_msglen;
    }
    if (ctx->s_msg != NULL) {
        msg_s = ctx->s_msg;
        msglen_s = ctx->s_msglen;
    }

    wolfSSL_SetLoggingPrefix("client");
    ExpectIntEQ(wolfSSL_write(ctx->c_ssl, msg_c, msglen_c), msglen_c);
    wolfSSL_SetLoggingPrefix("server");
    ExpectIntGT(idx = wolfSSL_read(ctx->s_ssl, input, sizeof(input) - 1), 0);
    if (idx >= 0) {
        input[idx] = '\0';
    }
    ExpectIntGT(fprintf(stderr, "Client message: %s\n", input), 0);
    ExpectIntEQ(wolfSSL_write(ctx->s_ssl, msg_s, msglen_s), msglen_s);
    ctx->s_cb.return_code = EXPECT_RESULT();
    wolfSSL_SetLoggingPrefix("client");
    ExpectIntGT(idx = wolfSSL_read(ctx->c_ssl, input, sizeof(input) - 1), 0);
    wolfSSL_SetLoggingPrefix(NULL);
    if (idx >= 0) {
        input[idx] = '\0';
    }
    ExpectIntGT(fprintf(stderr, "Server response: %s\n", input), 0);
    ctx->c_cb.return_code = EXPECT_RESULT();
    if (ctx->c_cb.on_result != NULL) {
        ExpectIntEQ(ctx->c_cb.on_result(ctx->c_ssl), TEST_SUCCESS);
    }
    if (ctx->s_cb.on_result != NULL) {
        ExpectIntEQ(ctx->s_cb.on_result(ctx->s_ssl), TEST_SUCCESS);
    }

    return EXPECT_RESULT();
}

void test_ssl_memio_cleanup(test_ssl_memio_ctx* ctx)
{
    ctx->c_cb.last_err = wolfSSL_get_error(ctx->c_ssl, 0);
    ctx->s_cb.last_err = wolfSSL_get_error(ctx->s_ssl, 0);
    if (ctx->c_cb.on_cleanup != NULL) {
        ctx->c_cb.on_cleanup(ctx->c_ssl);
    }
    if (ctx->s_cb.on_cleanup != NULL) {
        ctx->s_cb.on_cleanup(ctx->s_ssl);
    }
    wolfSSL_shutdown(ctx->s_ssl);
    wolfSSL_shutdown(ctx->c_ssl);
    wolfSSL_free(ctx->s_ssl);
    wolfSSL_free(ctx->c_ssl);
    if (ctx->c_cb.on_ctx_cleanup != NULL) {
        ctx->c_cb.on_ctx_cleanup(ctx->c_ctx);
    }
    if (!ctx->c_cb.isSharedCtx) {
        wolfSSL_CTX_free(ctx->c_ctx);
        ctx->c_ctx = NULL;
    }
    if (ctx->s_cb.on_ctx_cleanup != NULL) {
        ctx->s_cb.on_ctx_cleanup(ctx->s_ctx);
    }
    if (!ctx->s_cb.isSharedCtx) {
        wolfSSL_CTX_free(ctx->s_ctx);
        ctx->s_ctx = NULL;
    }

    if (!ctx->s_cb.ticNoInit) {
#if defined(HAVE_SESSION_TICKET) && \
    ((defined(HAVE_CHACHA) && defined(HAVE_POLY1305)) || defined(HAVE_AESGCM))
#if defined(OPENSSL_EXTRA) && defined(HAVE_AES_CBC)
        OpenSSLTicketCleanup();
#elif defined(WOLFSSL_NO_DEF_TICKET_ENC_CB)
        TicketCleanup();
#endif
#endif
    }
}

static int test_wolfSSL_client_server_nofail_memio_ex(test_ssl_cbf* client_cb,
    test_ssl_cbf* server_cb, cbType client_on_handshake,
    cbType server_on_handshake)
{
    /* We use EXPECT_DECLS_NO_MSGS() here because this helper routine is used
     * for numerous but varied expected-to-fail scenarios that should not emit
     * error messages on the expected failures.  Instead, we return a distinct
     * code for each failure point, allowing the caller to assert on a
     * particular mode of expected failure.  On success, the usual TEST_SUCCESS
     * is returned.
     */
    EXPECT_DECLS_NO_MSGS(-1000);
    struct test_ssl_memio_ctx test_ctx;
#ifdef WOLFSSL_HAVE_TLS_UNIQUE
    size_t msg_len;
#endif /* WOLFSSL_HAVE_TLS_UNIQUE */

    XMEMSET(&test_ctx, 0, sizeof(test_ctx));
    XMEMCPY(&test_ctx.c_cb, client_cb, sizeof(test_ssl_cbf));
    XMEMCPY(&test_ctx.s_cb, server_cb, sizeof(test_ssl_cbf));

    test_ctx.c_ctx = client_cb->ctx;
    test_ctx.s_ctx = server_cb->ctx;
    test_ctx.c_cb.return_code = EXPECT_FAILURE_CODEPOINT_ID;
    test_ctx.s_cb.return_code = EXPECT_FAILURE_CODEPOINT_ID;

    ExpectIntEQ(test_ssl_memio_setup(&test_ctx), TEST_SUCCESS);
    ExpectIntEQ(test_ssl_memio_do_handshake(&test_ctx, 10, NULL), TEST_SUCCESS);

    if (client_on_handshake != NULL) {
        ExpectIntEQ(client_on_handshake(test_ctx.c_ctx, test_ctx.c_ssl),
            TEST_SUCCESS);
    }
    if (server_on_handshake != NULL) {
        ExpectIntEQ(server_on_handshake(test_ctx.s_ctx, test_ctx.s_ssl),
            TEST_SUCCESS);
    }
    if (client_cb->on_handshake != NULL) {
        ExpectIntEQ(client_cb->on_handshake(&test_ctx.c_ctx, &test_ctx.c_ssl),
            TEST_SUCCESS);
    }
    if (server_cb->on_handshake != NULL) {
        ExpectIntEQ(server_cb->on_handshake(&test_ctx.s_ctx, &test_ctx.s_ssl),
            TEST_SUCCESS);
    }
#ifdef WOLFSSL_HAVE_TLS_UNIQUE
    XMEMSET(server_side_msg2, 0, WC_MAX_DIGEST_SIZE);
    msg_len = wolfSSL_get_peer_finished(test_ctx.s_ssl, server_side_msg2,
        WC_MAX_DIGEST_SIZE);
    ExpectIntGE(msg_len, 0);

    XMEMSET(server_side_msg1, 0, WC_MAX_DIGEST_SIZE);
    msg_len = wolfSSL_get_finished(test_ctx.s_ssl, server_side_msg1,
        WC_MAX_DIGEST_SIZE);
    ExpectIntGE(msg_len, 0);
#endif /* WOLFSSL_HAVE_TLS_UNIQUE */

    ExpectIntEQ(test_ssl_memio_read_write(&test_ctx), TEST_SUCCESS);
    test_ssl_memio_cleanup(&test_ctx);

    client_cb->return_code = test_ctx.c_cb.return_code;
    client_cb->last_err = test_ctx.c_cb.last_err;
    server_cb->return_code = test_ctx.s_cb.return_code;
    server_cb->last_err = test_ctx.s_cb.last_err;

    return EXPECT_RESULT();
}

int test_wolfSSL_client_server_nofail_memio(test_ssl_cbf* client_cb,
    test_ssl_cbf* server_cb, cbType client_on_handshake)
{
    return (test_wolfSSL_client_server_nofail_memio_ex(client_cb, server_cb,
        client_on_handshake, NULL));
}
#endif

#ifdef HAVE_IO_TESTS_DEPENDENCIES

#ifdef WOLFSSL_SESSION_EXPORT
#ifdef WOLFSSL_DTLS
/* set up function for sending session information */
static int test_export(WOLFSSL* inSsl, byte* buf, word32 sz, void* userCtx)
{
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL*     ssl = NULL;

    AssertNotNull(inSsl);
    AssertNotNull(buf);
    AssertIntNE(0, sz);

    /* Set ctx to DTLS 1.2 */
    ctx = wolfSSL_CTX_new(wolfDTLSv1_2_server_method());
    AssertNotNull(ctx);

    ssl = wolfSSL_new(ctx);
    AssertNotNull(ssl);

    AssertIntGE(wolfSSL_dtls_import(ssl, buf, sz), 0);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
    (void)userCtx;
    return 0;
}
#endif

/* returns negative value on fail and positive (including 0) on success */
static int nonblocking_accept_read(void* args, WOLFSSL* ssl, SOCKET_T* sockfd)
{
    int ret, err, loop_count, count, timeout = 10;
    char msg[] = "I hear you fa shizzle!";
    char input[1024];

    loop_count = ((func_args*)args)->argc;

    #ifdef WOLFSSL_ASYNC_CRYPT
    err = 0; /* Reset error */
    #endif
    do {
    #ifdef WOLFSSL_ASYNC_CRYPT
        if (err == WC_NO_ERR_TRACE(WC_PENDING_E)) {
            ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
            if (ret < 0) { break; } else if (ret == 0) { continue; }
        }
    #endif
        ret = wolfSSL_accept(ssl);
        err = wolfSSL_get_error(ssl, 0);

        if (err == WOLFSSL_ERROR_WANT_READ ||
            err == WOLFSSL_ERROR_WANT_WRITE) {
            int select_ret;

            err = WC_PENDING_E;
            select_ret = tcp_select(*sockfd, timeout);
            if (select_ret == TEST_TIMEOUT) {
                return WOLFSSL_FATAL_ERROR;
            }
        }
    } while (err == WC_NO_ERR_TRACE(WC_PENDING_E));
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err,
            wolfSSL_ERR_error_string(err, buff));
        return ret;
    }

    for (count = 0; count < loop_count; count++) {
        int select_ret;

        select_ret = tcp_select(*sockfd, timeout);
        if (select_ret == TEST_TIMEOUT) {
            ret = WOLFSSL_FATAL_ERROR;
            break;
        }

        do {
            ret = wolfSSL_read(ssl, input, sizeof(input)-1);
            if (ret > 0) {
                input[ret] = '\0';
                fprintf(stderr, "Client message: %s\n", input);
            }
        } while (err == WOLFSSL_ERROR_WANT_READ && ret != WOLFSSL_SUCCESS);

        do {
            if ((ret = wolfSSL_write(ssl, msg, sizeof(msg))) != sizeof(msg)) {
                return WOLFSSL_FATAL_ERROR;
            }
            err = wolfSSL_get_error(ssl, ret);
        } while (err == WOLFSSL_ERROR_WANT_READ && ret != WOLFSSL_SUCCESS);
    }
    return ret;
}
#endif /* WOLFSSL_SESSION_EXPORT */

static THREAD_RETURN WOLFSSL_THREAD test_server_nofail(void* args)
{
    SOCKET_T sockfd = 0;
    SOCKET_T clientfd = 0;
    word16 port;

    callback_functions* cbf;
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL* ssl = NULL;
    func_args* opts = (func_args*)args;

    char msg[] = "I hear you fa shizzle!";
    char input[1024];
    int idx;
    int ret, err = 0;
    int sharedCtx = 0;
    int doUdp = 0;
    SOCKADDR_IN_T cliAddr;
    socklen_t     cliLen;
    const char* certFile = svrCertFile;
    const char* keyFile = svrKeyFile;

#ifdef WOLFSSL_HAVE_TLS_UNIQUE
    size_t msg_len = 0;
#endif

    wolfSSL_SetLoggingPrefix("server");

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    opts->return_code = TEST_FAIL;
    cbf = opts->callbacks;

    if (cbf != NULL && cbf->ctx) {
        ctx = cbf->ctx;
        sharedCtx = 1;
    }
    else
    {
        WOLFSSL_METHOD* method = NULL;
        if (cbf != NULL && cbf->method != NULL) {
            method = cbf->method();

        }
        else {
            method = wolfSSLv23_server_method();
        }
        ctx = wolfSSL_CTX_new(method);
    }
    if (ctx == NULL) {
        /* Release the wait for TCP ready. */
        signal_ready(opts->signal);
        goto done;
    }

    if (cbf == NULL || !cbf->ticNoInit) {
#if defined(HAVE_SESSION_TICKET) && \
    ((defined(HAVE_CHACHA) && defined(HAVE_POLY1305)) || defined(HAVE_AESGCM))
#if defined(OPENSSL_EXTRA) && defined(HAVE_AES_CBC)
        OpenSSLTicketInit();
        wolfSSL_CTX_set_tlsext_ticket_key_cb(ctx, myTicketEncCbOpenSSL);
#elif defined(WOLFSSL_NO_DEF_TICKET_ENC_CB)
        TicketInit();
        wolfSSL_CTX_set_TicketEncCb(ctx, myTicketEncCb);
#endif
#endif
    }

#if defined(USE_WINDOWS_API)
    port = opts->signal->port;
#elif defined(NO_MAIN_DRIVER) && !defined(WOLFSSL_SNIFFER) && \
     !defined(WOLFSSL_MDK_SHELL) && !defined(WOLFSSL_TIRTOS)
    /* Let tcp_listen assign port */
    port = 0;
#else
    /* Use default port */
    port = wolfSSLPort;
#endif

    if (cbf != NULL)
        doUdp = cbf->doUdp;

    /* do it here to detect failure */
    tcp_accept(
        &sockfd, &clientfd, opts, port, 0, doUdp, 0, 0, 1, 0, 0);

    if (doUdp) {
        cliLen = sizeof(cliAddr);

        idx = (int)recvfrom(sockfd, input, sizeof(input), MSG_PEEK,
                  (struct sockaddr*)&cliAddr, &cliLen);

        AssertIntGT(idx, 0);
    }
    else {
        CloseSocket(sockfd);
    }

    wolfSSL_CTX_set_verify(ctx,
                  WOLFSSL_VERIFY_PEER | WOLFSSL_VERIFY_FAIL_IF_NO_PEER_CERT, 0);

#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx, PasswordCallBack);
#endif

    if (wolfSSL_CTX_load_verify_locations(ctx, cliCertFile, 0)
                                                           != WOLFSSL_SUCCESS) {
        /*err_sys("can't load ca file, Please run from wolfSSL home dir");*/
        goto done;
    }

    if (cbf != NULL && cbf->certPemFile != NULL)
        certFile = cbf->certPemFile;
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (!sharedCtx && wolfSSL_CTX_use_certificate_file(ctx, certFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_CTX_use_certificate_file(ctx, certFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load server cert chain file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

    if (cbf != NULL && cbf->keyPemFile != NULL)
        keyFile = cbf->keyPemFile;
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (!sharedCtx && wolfSSL_CTX_use_PrivateKey_file(ctx, keyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_CTX_use_PrivateKey_file(ctx, keyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load server key file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

#ifdef HAVE_CRL
    if (cbf != NULL && cbf->crlPemFile != NULL) {
        if (wolfSSL_CTX_EnableCRL(ctx, WOLFSSL_CRL_CHECKALL) != WOLFSSL_SUCCESS)
            goto done;
        if (wolfSSL_CTX_LoadCRLFile(ctx, cbf->crlPemFile, WOLFSSL_FILETYPE_PEM)
                != WOLFSSL_SUCCESS)
            goto done;
    }
#endif

    /* call ctx setup callback */
    if (cbf != NULL && cbf->ctx_ready != NULL) {
        cbf->ctx_ready(ctx);
    }

    ssl = wolfSSL_new(ctx);
    if (ssl == NULL) {
        goto done;
    }

    if (doUdp) {
        err = wolfSSL_dtls_set_peer(ssl, &cliAddr, cliLen);
        if (err != WOLFSSL_SUCCESS)
            goto done;
    }

#ifdef WOLFSSL_SESSION_EXPORT
    /* only add in more complex nonblocking case with session export tests */
    if (args && opts->argc > 0) {
        /* set as nonblock and time out for waiting on read/write */
        tcp_set_nonblocking(&clientfd);
        wolfSSL_dtls_set_using_nonblock(ssl, 1);
    }
#endif
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (sharedCtx && wolfSSL_use_certificate_file(ssl, certFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_use_certificate_file(ssl, certFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load server cert chain file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (sharedCtx && wolfSSL_use_PrivateKey_file(ssl, keyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_use_PrivateKey_file(ssl, keyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load server key file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

    if (wolfSSL_set_fd(ssl, clientfd) != WOLFSSL_SUCCESS) {
        /*err_sys("SSL_set_fd failed");*/
        goto done;
    }

#if !defined(NO_FILESYSTEM) && !defined(NO_DH)
    wolfSSL_SetTmpDH_file(ssl, dhParamFile, WOLFSSL_FILETYPE_PEM);
#elif !defined(NO_DH)
    SetDH(ssl);  /* will repick suites with DHE, higher priority than PSK */
#endif

    /* call ssl setup callback */
    if (cbf != NULL && cbf->ssl_ready != NULL) {
        cbf->ssl_ready(ssl);
    }

#ifdef WOLFSSL_SESSION_EXPORT
    /* only add in more complex nonblocking case with session export tests */
    if (opts->argc > 0) {
        ret = nonblocking_accept_read(args, ssl, &clientfd);
        if (ret >= 0) {
            opts->return_code = TEST_SUCCESS;
        }
    #ifdef WOLFSSL_TIRTOS
        Task_yield();
    #endif
        goto done;
    }
#endif

    #ifdef WOLFSSL_ASYNC_CRYPT
    err = 0; /* Reset error */
    #endif
    do {
    #ifdef WOLFSSL_ASYNC_CRYPT
        if (err == WC_NO_ERR_TRACE(WC_PENDING_E)) {
            ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
            if (ret < 0) { break; } else if (ret == 0) { continue; }
        }
    #endif
        ret = wolfSSL_negotiate(ssl);
        err = wolfSSL_get_error(ssl, 0);
    } while (err == WC_NO_ERR_TRACE(WC_PENDING_E));
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err,
            wolfSSL_ERR_error_string((word32)err, buff));
        /*err_sys("SSL_accept failed");*/
        goto done;
    }

#ifdef WOLFSSL_HAVE_TLS_UNIQUE
    XMEMSET(server_side_msg2, 0, WC_MAX_DIGEST_SIZE);
    msg_len = wolfSSL_get_peer_finished(ssl, server_side_msg2, WC_MAX_DIGEST_SIZE);
    AssertIntGE(msg_len, 0);

    XMEMSET(server_side_msg1, 0, WC_MAX_DIGEST_SIZE);
    msg_len = wolfSSL_get_finished(ssl, server_side_msg1, WC_MAX_DIGEST_SIZE);
    AssertIntGE(msg_len, 0);
#endif /* WOLFSSL_HAVE_TLS_UNIQUE */

    idx = wolfSSL_read(ssl, input, sizeof(input)-1);
    if (idx > 0) {
        input[idx] = '\0';
        fprintf(stderr, "Client message: %s\n", input);
    }
    else if (idx < 0) {
        goto done;
    }

    if (wolfSSL_write(ssl, msg, sizeof(msg)) != sizeof(msg)) {
        /*err_sys("SSL_write failed");*/
        goto done;
    }

    if (cbf != NULL && cbf->on_result != NULL)
        cbf->on_result(ssl);

#ifdef WOLFSSL_TIRTOS
    Task_yield();
#endif

    opts->return_code = TEST_SUCCESS;

done:
    if (cbf != NULL)
        cbf->last_err = err;
    if (cbf != NULL && cbf->on_cleanup != NULL)
        cbf->on_cleanup(ssl);

    wolfSSL_shutdown(ssl);
    wolfSSL_free(ssl);
    if (!sharedCtx)
        wolfSSL_CTX_free(ctx);

    CloseSocket(clientfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

#if defined(NO_MAIN_DRIVER) && defined(HAVE_ECC) && defined(FP_ECC) \
                            && defined(HAVE_THREAD_LS)
    wc_ecc_fp_free();  /* free per thread cache */
#endif

    if (cbf == NULL || !cbf->ticNoInit) {
#if defined(HAVE_SESSION_TICKET) && \
    ((defined(HAVE_CHACHA) && defined(HAVE_POLY1305)) || defined(HAVE_AESGCM))
#if defined(OPENSSL_EXTRA) && defined(HAVE_AES_CBC)
        OpenSSLTicketCleanup();
#elif defined(WOLFSSL_NO_DEF_TICKET_ENC_CB)
        TicketCleanup();
#endif
#endif
    }

    wolfSSL_SetLoggingPrefix(NULL);

    WOLFSSL_RETURN_FROM_THREAD(0);
}

#if defined(OPENSSL_EXTRA) && !defined(NO_SESSION_CACHE) && \
    !defined(WOLFSSL_NO_TLS12)
static THREAD_RETURN WOLFSSL_THREAD test_server_loop(void* args)
{
    SOCKET_T sockfd;
    SOCKET_T clientfd = -1;
    word16 port;

    callback_functions* cbf;
    WOLFSSL_CTX* ctx = 0;
    WOLFSSL* ssl = 0;

    char msg[] = "I hear you fa shizzle!";
    char input[1024];
    int idx;
    int ret, err = 0;
    int sharedCtx = 0;
    func_args* opts = (func_args*)args;
    int loop_count = opts->argc;
    int count = 0;

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    opts->return_code = TEST_FAIL;
    cbf = opts->callbacks;

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (cbf != NULL && cbf->ctx) {
        ctx = cbf->ctx;
        sharedCtx = 1;
    }
    else
#endif
    {
        WOLFSSL_METHOD* method = NULL;
        if (cbf != NULL && cbf->method != NULL) {
            method = cbf->method();
        }
        else {
            method = wolfSSLv23_server_method();
        }
        ctx = wolfSSL_CTX_new(method);
    }

#if defined(USE_WINDOWS_API)
    port = opts->signal->port;
#elif defined(NO_MAIN_DRIVER) && !defined(WOLFSSL_SNIFFER) && \
     !defined(WOLFSSL_MDK_SHELL) && !defined(WOLFSSL_TIRTOS)
    /* Let tcp_listen assign port */
    port = 0;
#else
    /* Use default port */
    port = wolfSSLPort;
#endif

    wolfSSL_CTX_set_verify(ctx,
                  WOLFSSL_VERIFY_PEER | WOLFSSL_VERIFY_FAIL_IF_NO_PEER_CERT, 0);

#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx, PasswordCallBack);
#endif

    if (wolfSSL_CTX_load_verify_locations(ctx, cliCertFile, 0)
                                                           != WOLFSSL_SUCCESS) {
        /*err_sys("can't load ca file, Please run from wolfSSL home dir");*/
        /* Release the wait for TCP ready. */
        signal_ready(opts->signal);
        goto done;
    }
    if (!sharedCtx && wolfSSL_CTX_use_certificate_file(ctx, svrCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        /*err_sys("can't load server cert chain file, "
                "Please run from wolfSSL home dir");*/
        /* Release the wait for TCP ready. */
        signal_ready(opts->signal);
        goto done;
    }
    if (!sharedCtx && wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        /*err_sys("can't load server key file, "
                "Please run from wolfSSL home dir");*/
        /* Release the wait for TCP ready. */
        signal_ready(opts->signal);
        goto done;
    }
    /* call ctx setup callback */
    if (cbf != NULL && cbf->ctx_ready != NULL) {
        cbf->ctx_ready(ctx);
    }

    while (count != loop_count) {
        ssl = wolfSSL_new(ctx);
        if (ssl == NULL) {
            signal_ready(opts->signal);
            goto done;
        }
        if (sharedCtx && wolfSSL_use_certificate_file(ssl, svrCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
            /*err_sys("can't load server cert chain file, "
                    "Please run from wolfSSL home dir");*/
            /* Release the wait for TCP ready. */
            signal_ready(opts->signal);
            goto done;
        }
        if (sharedCtx && wolfSSL_use_PrivateKey_file(ssl, svrKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
            /*err_sys("can't load server key file, "
                    "Please run from wolfSSL home dir");*/
            /* Release the wait for TCP ready. */
            signal_ready(opts->signal);
            goto done;
        }

#if !defined(NO_FILESYSTEM) && !defined(NO_DH)
        wolfSSL_SetTmpDH_file(ssl, dhParamFile, WOLFSSL_FILETYPE_PEM);
#elif !defined(NO_DH)
        SetDH(ssl);  /* will repick suites with DHE, higher priority than PSK */
#endif
        /* call ssl setup callback */
        if (cbf != NULL && cbf->ssl_ready != NULL) {
            cbf->ssl_ready(ssl);
        }
        /* do it here to detect failure */
        tcp_accept(&sockfd, &clientfd, (func_args*)args, port, 0, 0, 0, 0, 1, 0,
            0);
        CloseSocket(sockfd);
        if (wolfSSL_set_fd(ssl, clientfd) != WOLFSSL_SUCCESS) {
            /*err_sys("SSL_set_fd failed");*/
            goto done;
        }

        #ifdef WOLFSSL_ASYNC_CRYPT
        err = 0; /* Reset error */
        #endif
        do {
        #ifdef WOLFSSL_ASYNC_CRYPT
            if (err == WC_NO_ERR_TRACE(WC_PENDING_E)) {
                ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
                if (ret < 0) { break; } else if (ret == 0) { continue; }
            }
        #endif
            ret = wolfSSL_accept(ssl);
            err = wolfSSL_get_error(ssl, 0);
        } while (err == WC_NO_ERR_TRACE(WC_PENDING_E));
        if (ret != WOLFSSL_SUCCESS) {
            char buff[WOLFSSL_MAX_ERROR_SZ];
            fprintf(stderr, "error = %d, %s\n", err,
                wolfSSL_ERR_error_string(err, buff));
            /*err_sys("SSL_accept failed");*/
            goto done;
        }

        idx = wolfSSL_read(ssl, input, sizeof(input)-1);
        if (idx > 0) {
            input[idx] = '\0';
            fprintf(stderr, "Client message: %s\n", input);
        }

        if (wolfSSL_write(ssl, msg, sizeof(msg)) != sizeof(msg)) {
            /*err_sys("SSL_write failed");*/
            goto done;
        }
        /* free ssl for this connection */
        wolfSSL_shutdown(ssl);
        wolfSSL_free(ssl); ssl = NULL;
        CloseSocket(clientfd);
        clientfd = -1;

        count++;
    }
#ifdef WOLFSSL_TIRTOS
    Task_yield();
#endif

    opts->return_code = TEST_SUCCESS;

done:
    if (ssl != NULL) {
        wolfSSL_shutdown(ssl);
        wolfSSL_free(ssl);
    }
    if (!sharedCtx)
        wolfSSL_CTX_free(ctx);

    if (clientfd != SOCKET_INVALID)
        CloseSocket(clientfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

#if defined(NO_MAIN_DRIVER) && defined(HAVE_ECC) && defined(FP_ECC) \
                            && defined(HAVE_THREAD_LS)
    wc_ecc_fp_free();  /* free per thread cache */
#endif

    WOLFSSL_RETURN_FROM_THREAD(0);
}
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_SESSION_CACHE) && !defined(WOLFSSL_TLS13) */

static int test_client_nofail(void* args, cbType cb)
{
#if !defined(NO_WOLFSSL_CLIENT)
    SOCKET_T sockfd = 0;
    callback_functions* cbf;

    WOLFSSL_CTX*     ctx     = 0;
    WOLFSSL*         ssl     = 0;
    WOLFSSL_CIPHER*  cipher;

    char msg[64] = "hello wolfssl!";
    char reply[1024];
    int  input;
    int  msgSz = (int)XSTRLEN(msg);
    int  ret, err = 0;
    int  cipherSuite;
    int  sharedCtx = 0;
    int  doUdp = 0;
    const char* cipherName1, *cipherName2;

    wolfSSL_SetLoggingPrefix("client");

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    ((func_args*)args)->return_code = TEST_FAIL;
    cbf = ((func_args*)args)->callbacks;

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (cbf != NULL && cbf->ctx) {
        ctx = cbf->ctx;
        sharedCtx = cbf->isSharedCtx;
    }
    else
#endif
    {
        WOLFSSL_METHOD* method  = NULL;
        if (cbf != NULL && cbf->method != NULL) {
            method = cbf->method();
        }
        else {
            method = wolfSSLv23_client_method();
        }
        ctx = wolfSSL_CTX_new(method);
    }

    if (cbf != NULL)
        doUdp = cbf->doUdp;

#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx, PasswordCallBack);
#endif

    /* Do connect here so server detects failures */
    tcp_connect(&sockfd, wolfSSLIP, ((func_args*)args)->signal->port,
                doUdp, 0, NULL);
    /* Connect the socket so that we don't have to set the peer later on */
    if (doUdp)
        udp_connect(&sockfd, wolfSSLIP, ((func_args*)args)->signal->port);

    if (wolfSSL_CTX_load_verify_locations(ctx, caCertFile, 0) != WOLFSSL_SUCCESS)
    {
        /* err_sys("can't load ca file, Please run from wolfSSL home dir");*/
        goto done;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (!sharedCtx && wolfSSL_CTX_use_certificate_file(ctx, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_CTX_use_certificate_file(ctx, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load client cert file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (!sharedCtx && wolfSSL_CTX_use_PrivateKey_file(ctx, cliKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_CTX_use_PrivateKey_file(ctx, cliKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#endif

        /*err_sys("can't load client key file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

#ifdef WOLFSSL_SRTP
    /* make sure that NULL (error condition) returns 1 */
    if (wolfSSL_CTX_set_tlsext_use_srtp(ctx, NULL) != 1) {
         goto done;
    }
#endif

#ifdef HAVE_CRL
    if (cbf != NULL && cbf->crlPemFile != NULL) {
        if (wolfSSL_CTX_EnableCRL(ctx, WOLFSSL_CRL_CHECKALL) != WOLFSSL_SUCCESS)
            goto done;
        if (wolfSSL_CTX_LoadCRLFile(ctx, cbf->crlPemFile, WOLFSSL_FILETYPE_PEM)
                != WOLFSSL_SUCCESS)
            goto done;
    }
#endif

    /* call ctx setup callback */
    if (cbf != NULL && cbf->ctx_ready != NULL) {
        cbf->ctx_ready(ctx);
    }

    ssl = wolfSSL_new(ctx);
    if (ssl == NULL) {
        goto done;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (sharedCtx && wolfSSL_use_certificate_file(ssl, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_use_certificate_file(ssl, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load client cert file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (sharedCtx && wolfSSL_use_PrivateKey_file(ssl, cliKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#else
    if (wolfSSL_use_PrivateKey_file(ssl, cliKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
#endif
        /*err_sys("can't load client key file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

#ifdef WOLFSSL_SRTP
    /* make sure that NULL (error condition) returns 1 */
    if (wolfSSL_set_tlsext_use_srtp(ssl, NULL) != 1) {
         goto done;
    }
#endif

    if (!doUdp) {
        if (wolfSSL_set_fd(ssl, sockfd) != WOLFSSL_SUCCESS) {
            /*err_sys("SSL_set_fd failed");*/
            goto done;
        }
    }
    else {
#ifdef WOLFSSL_DTLS
        if (wolfSSL_set_dtls_fd_connected(ssl, sockfd) != WOLFSSL_SUCCESS) {
            /*err_sys("SSL_set_fd failed");*/
            goto done;
        }
#else
        goto done;
#endif
    }

    /* call ssl setup callback */
    if (cbf != NULL && cbf->ssl_ready != NULL) {
        cbf->ssl_ready(ssl);
    }

    #ifdef WOLFSSL_ASYNC_CRYPT
    err = 0; /* Reset error */
    #endif
    do {
    #ifdef WOLFSSL_ASYNC_CRYPT
        if (err == WC_NO_ERR_TRACE(WC_PENDING_E)) {
            ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
            if (ret < 0) { break; } else if (ret == 0) { continue; }
        }
    #endif
        ret = wolfSSL_negotiate(ssl);
        err = wolfSSL_get_error(ssl, 0);
    } while (err == WC_NO_ERR_TRACE(WC_PENDING_E));
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err,
            wolfSSL_ERR_error_string((word32)err, buff));
        /*err_sys("SSL_connect failed");*/
        goto done;
    }

    /* test the various get cipher methods */
    /* Internal cipher suite names */
    cipherSuite = wolfSSL_get_current_cipher_suite(ssl);
    cipherName1 = wolfSSL_get_cipher_name(ssl);
    cipherName2 = wolfSSL_get_cipher_name_from_suite(
        (byte)(cipherSuite >> 8), cipherSuite & 0xFF);
    AssertStrEQ(cipherName1, cipherName2);

    /* IANA Cipher Suites Names */
    /* Unless WOLFSSL_CIPHER_INTERNALNAME or NO_ERROR_STRINGS,
        then it's the internal cipher suite name */
    cipher = wolfSSL_get_current_cipher(ssl);
    cipherName1 = wolfSSL_CIPHER_get_name(cipher);
    cipherName2 = wolfSSL_get_cipher(ssl);
    AssertStrEQ(cipherName1, cipherName2);
#if !defined(WOLFSSL_CIPHER_INTERNALNAME) && !defined(NO_ERROR_STRINGS) && \
    !defined(WOLFSSL_QT)
    cipherName1 = wolfSSL_get_cipher_name_iana_from_suite(
            (byte)(cipherSuite >> 8), cipherSuite & 0xFF);
    AssertStrEQ(cipherName1, cipherName2);
#endif

    if (cb != NULL)
        (cb)(ctx, ssl);

    if (wolfSSL_write(ssl, msg, msgSz) != msgSz) {
        /*err_sys("SSL_write failed");*/
        goto done;
    }

    input = wolfSSL_read(ssl, reply, sizeof(reply)-1);
    if (input > 0) {
        reply[input] = '\0';
        fprintf(stderr, "Server response: %s\n", reply);
    }

    if (cbf != NULL && cbf->on_result != NULL)
        cbf->on_result(ssl);

    ((func_args*)args)->return_code = TEST_SUCCESS;

done:
    if (cbf != NULL)
        cbf->last_err = err;
    if (cbf != NULL && cbf->on_cleanup != NULL)
        cbf->on_cleanup(ssl);

    wolfSSL_free(ssl);
    if (!sharedCtx)
        wolfSSL_CTX_free(ctx);

    CloseSocket(sockfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

#if defined(NO_MAIN_DRIVER) && defined(HAVE_ECC) && defined(FP_ECC) \
                            && defined(HAVE_THREAD_LS)
    wc_ecc_fp_free();  /* free per thread cache */
#endif

#else
    (void)args;
    (void)cb;
#endif /* !NO_WOLFSSL_CLIENT */

    wolfSSL_SetLoggingPrefix(NULL);

    return 0;
}

void test_wolfSSL_client_server_nofail_ex(callback_functions* client_cb,
    callback_functions* server_cb, cbType client_on_handshake)
{
    func_args client_args;
    func_args server_args;
    tcp_ready ready;
    THREAD_TYPE serverThread;

    XMEMSET(&client_args, 0, sizeof(func_args));
    XMEMSET(&server_args, 0, sizeof(func_args));

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    StartTCP();
    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    server_args.signal = &ready;
    server_args.callbacks = server_cb;
    client_args.signal = &ready;
    client_args.callbacks = client_cb;

    start_thread(test_server_nofail, &server_args, &serverThread);
    wait_tcp_ready(&server_args);
    test_client_nofail(&client_args, client_on_handshake);
    join_thread(serverThread);

    client_cb->return_code = client_args.return_code;
    server_cb->return_code = server_args.return_code;

    FreeTcpReady(&ready);

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif
}

void test_wolfSSL_client_server_nofail(callback_functions* client_cb,
    callback_functions* server_cb)
{
    test_wolfSSL_client_server_nofail_ex(client_cb, server_cb, NULL);
}


#if defined(OPENSSL_EXTRA) && !defined(NO_SESSION_CACHE) && \
   !defined(WOLFSSL_NO_TLS12) && !defined(NO_WOLFSSL_CLIENT)
static void test_client_reuse_WOLFSSLobj(void* args, cbType cb,
                                         void* server_args)
{
    SOCKET_T sockfd = 0;
    callback_functions* cbf;

    WOLFSSL_CTX*     ctx     = 0;
    WOLFSSL*         ssl     = 0;
    WOLFSSL_SESSION* session = NULL;

    char msg[64] = "hello wolfssl!";
    char reply[1024];
    int  input;
    int  msgSz = (int)XSTRLEN(msg);
    int  ret, err = 0;
    int  sharedCtx = 0;

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    ((func_args*)args)->return_code = TEST_FAIL;
    cbf = ((func_args*)args)->callbacks;

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    if (cbf != NULL && cbf->ctx) {
        ctx = cbf->ctx;
        sharedCtx = 1;
    }
    else
#endif
    {
        WOLFSSL_METHOD* method  = NULL;
        if (cbf != NULL && cbf->method != NULL) {
            method = cbf->method();
        }
        else {
            method = wolfSSLv23_client_method();
        }
        ctx = wolfSSL_CTX_new(method);
    }

#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx, PasswordCallBack);
#endif

    /* Do connect here so server detects failures */
    tcp_connect(&sockfd, wolfSSLIP, ((func_args*)args)->signal->port,
                0, 0, NULL);

    if (wolfSSL_CTX_load_verify_locations(ctx, caCertFile, 0) !=
                                                              WOLFSSL_SUCCESS) {
        /* err_sys("can't load ca file, Please run from wolfSSL home dir");*/
        goto done;
    }
    if (!sharedCtx && wolfSSL_CTX_use_certificate_file(ctx, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        /*err_sys("can't load client cert file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }
    if (!sharedCtx && wolfSSL_CTX_use_PrivateKey_file(ctx, cliKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        /*err_sys("can't load client key file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

    /* call ctx setup callback */
    if (cbf != NULL && cbf->ctx_ready != NULL) {
        cbf->ctx_ready(ctx);
    }

    ssl = wolfSSL_new(ctx);
    if (ssl == NULL) {
        goto done;
    }
    /* keep handshake resources for reusing WOLFSSL obj */
    wolfSSL_KeepArrays(ssl);
    if (wolfSSL_KeepHandshakeResources(ssl)) {
        /* err_sys("SSL_KeepHandshakeResources failed"); */
        goto done;
    }
    if (sharedCtx && wolfSSL_use_certificate_file(ssl, cliCertFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        /*err_sys("can't load client cert file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }
    if (sharedCtx && wolfSSL_use_PrivateKey_file(ssl, cliKeyFile,
                                     WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        /*err_sys("can't load client key file, "
                "Please run from wolfSSL home dir");*/
        goto done;
    }

    if (wolfSSL_set_fd(ssl, sockfd) != WOLFSSL_SUCCESS) {
        /*err_sys("SSL_set_fd failed");*/
        goto done;
    }

    /* call ssl setup callback */
    if (cbf != NULL && cbf->ssl_ready != NULL) {
        cbf->ssl_ready(ssl);
    }

    #ifdef WOLFSSL_ASYNC_CRYPT
    err = 0; /* Reset error */
    #endif
    do {
    #ifdef WOLFSSL_ASYNC_CRYPT
        if (err == WC_NO_ERR_TRACE(WC_PENDING_E)) {
            ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
            if (ret < 0) { break; } else if (ret == 0) { continue; }
        }
    #endif
        ret = wolfSSL_connect(ssl);
        err = wolfSSL_get_error(ssl, 0);
    } while (err == WC_NO_ERR_TRACE(WC_PENDING_E));
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err,
            wolfSSL_ERR_error_string(err, buff));
        /*err_sys("SSL_connect failed");*/
        goto done;
    }
    /* Build first session */
    if (cb != NULL)
        cb(ctx, ssl);

    if (wolfSSL_write(ssl, msg, msgSz) != msgSz) {
        /*err_sys("SSL_write failed");*/
        goto done;
    }

    input = wolfSSL_read(ssl, reply, sizeof(reply)-1);
    if (input > 0) {
        reply[input] = '\0';
        fprintf(stderr, "Server response: %s\n", reply);
    }

    /* Session Resumption by reusing WOLFSSL object */
    wolfSSL_set_quiet_shutdown(ssl, 1);
    if (wolfSSL_shutdown(ssl) != WOLFSSL_SUCCESS) {
        /* err_sys ("SSL shutdown failed"); */
        goto done;
    }
    session = wolfSSL_get1_session(ssl);
    if (wolfSSL_clear(ssl) != WOLFSSL_SUCCESS) {
        wolfSSL_SESSION_free(session);
        /* err_sys ("SSL_clear failed"); */
        goto done;
    }
    wolfSSL_set_session(ssl, session);
    wolfSSL_SESSION_free(session);
    session = NULL;
    /* close socket once */
    CloseSocket(sockfd);
    sockfd = 0;
    /* wait until server ready */
    wait_tcp_ready((func_args*)server_args);
    fprintf(stderr, "session resumption\n");
    /* Do re-connect  */
    tcp_connect(&sockfd, wolfSSLIP, ((func_args*)args)->signal->port,
                0, 0, NULL);
    if (wolfSSL_set_fd(ssl, sockfd) != WOLFSSL_SUCCESS) {
        /*err_sys("SSL_set_fd failed");*/
        goto done;
    }

    #ifdef WOLFSSL_ASYNC_CRYPT
    err = 0; /* Reset error */
    #endif
    do {
    #ifdef WOLFSSL_ASYNC_CRYPT
        if (err == WC_NO_ERR_TRACE(WC_PENDING_E)) {
            ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
            if (ret < 0) { break; } else if (ret == 0) { continue; }
        }
    #endif
        ret = wolfSSL_connect(ssl);
        err = wolfSSL_get_error(ssl, 0);
    } while (err == WC_NO_ERR_TRACE(WC_PENDING_E));
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err,
            wolfSSL_ERR_error_string(err, buff));
        /*err_sys("SSL_connect failed");*/
        goto done;
    }
    /* Build first session */
    if (cb != NULL)
        cb(ctx, ssl);

    if (wolfSSL_write(ssl, msg, msgSz) != msgSz) {
        /*err_sys("SSL_write failed");*/
        goto done;
    }

    input = wolfSSL_read(ssl, reply, sizeof(reply)-1);
    if (input > 0) {
        reply[input] = '\0';
        fprintf(stderr, "Server response: %s\n", reply);
    }

    ((func_args*)args)->return_code = TEST_SUCCESS;

done:
    wolfSSL_free(ssl);
    if (!sharedCtx)
        wolfSSL_CTX_free(ctx);

    CloseSocket(sockfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

    return;
}
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_SESSION_CACHE) &&
         !defined(WOLFSSL_TLS13) && !defined(NO_WOLFSSL_CLIENT) */

#if (defined(OPENSSL_ALL) || defined(WOLFSSL_NGINX) || \
     defined(WOLFSSL_HAPROXY) || defined(HAVE_LIGHTY)) && \
    defined(HAVE_ALPN) && defined(HAVE_SNI) && \
    defined(HAVE_IO_TESTS_DEPENDENCIES) && !defined(NO_BIO)
    #define HAVE_ALPN_PROTOS_SUPPORT
#endif

/* Generic TLS client / server with callbacks for API unit tests
 * Used by SNI / ALPN / crypto callback helper functions */
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && \
    (defined(HAVE_SNI) || defined(HAVE_ALPN) || defined(WOLF_CRYPTO_CB) || \
     defined(HAVE_ALPN_PROTOS_SUPPORT)) || defined(WOLFSSL_STATIC_MEMORY)
    #define ENABLE_TLS_CALLBACK_TEST
#endif

#if defined(ENABLE_TLS_CALLBACK_TEST) || \
    (defined(WOLFSSL_DTLS) && defined(WOLFSSL_SESSION_EXPORT))
/* TLS server for API unit testing - generic */
static THREAD_RETURN WOLFSSL_THREAD run_wolfssl_server(void* args)
{
    callback_functions* callbacks = ((func_args*)args)->callbacks;

    WOLFSSL_CTX*    ctx = NULL;
    WOLFSSL*        ssl = NULL;
    SOCKET_T        sfd = 0;
    SOCKET_T        cfd = 0;
    word16          port;

    char msg[] = "I hear you fa shizzle!";
    int  len   = (int) XSTRLEN(msg);
    char input[1024];
    int  idx;
    int  ret, err = 0;

    ((func_args*)args)->return_code = TEST_FAIL;

#if defined(USE_WINDOWS_API)
    port = ((func_args*)args)->signal->port;
#elif defined(NO_MAIN_DRIVER) && !defined(WOLFSSL_SNIFFER) && \
     !defined(WOLFSSL_MDK_SHELL) && !defined(WOLFSSL_TIRTOS)
    /* Let tcp_listen assign port */
    port = 0;
#else
    /* Use default port */
    port = wolfSSLPort;
#endif

#ifdef WOLFSSL_DTLS
    if (callbacks->method == wolfDTLS_server_method
#ifdef WOLFSSL_STATIC_MEMORY
     || callbacks->method_ex == wolfDTLS_server_method_ex
#endif
#ifndef NO_OLD_TLS
     || callbacks->method == wolfDTLSv1_server_method
#ifdef WOLFSSL_STATIC_MEMORY
     || callbacks->method_ex == wolfDTLSv1_server_method_ex
#endif
#endif
#ifndef WOLFSSL_NO_TLS12
     || callbacks->method == wolfDTLSv1_2_server_method
#ifdef WOLFSSL_STATIC_MEMORY
     || callbacks->method_ex == wolfDTLSv1_2_server_method_ex
#endif
#endif
#ifdef WOLFSSL_DTLS13
     || callbacks->method == wolfDTLSv1_3_server_method
#ifdef WOLFSSL_STATIC_MEMORY
     || callbacks->method_ex == wolfDTLSv1_3_server_method_ex
#endif
#endif
        ) {
        tcp_accept(&sfd, &cfd, (func_args*)args, port, 0, 1, 0, 0, 0, 0, 0);
    }
    else
#endif
    {
        tcp_accept(&sfd, &cfd, (func_args*)args, port, 0, 0, 0, 0, 1, 0, 0);
    }

#ifdef WOLFSSL_STATIC_MEMORY
    if (callbacks->method_ex != NULL && callbacks->mem != NULL &&
        callbacks->memSz > 0) {
        ret = wolfSSL_CTX_load_static_memory(&ctx, callbacks->method_ex,
            callbacks->mem, callbacks->memSz, 0, 1);
        if (ret != WOLFSSL_SUCCESS) {
            fprintf(stderr, "CTX static new failed %d\n", ret);
            goto cleanup;
        }
    }
#else
    ctx = wolfSSL_CTX_new(callbacks->method());
#endif
    if (ctx == NULL) {
        fprintf(stderr, "CTX new failed\n");
        goto cleanup;
    }

    /* set defaults */
    if (callbacks->caPemFile == NULL)
        callbacks->caPemFile = cliCertFile;
    if (callbacks->certPemFile == NULL)
        callbacks->certPemFile = svrCertFile;
    if (callbacks->keyPemFile == NULL)
        callbacks->keyPemFile = svrKeyFile;

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    wolfSSL_CTX_SetDevId(ctx, callbacks->devId);

    wolfSSL_CTX_set_verify(ctx,
                  WOLFSSL_VERIFY_PEER | WOLFSSL_VERIFY_FAIL_IF_NO_PEER_CERT, 0);

#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx, PasswordCallBack);
#endif
#if defined(WOLFSSL_SESSION_EXPORT) && defined(WOLFSSL_DTLS)
    if (callbacks->method == wolfDTLSv1_2_server_method) {
        if (wolfSSL_CTX_dtls_set_export(ctx, test_export) != WOLFSSL_SUCCESS)
            goto cleanup;
    }
#endif


    if (wolfSSL_CTX_load_verify_locations(ctx, callbacks->caPemFile, 0) !=
            WOLFSSL_SUCCESS) {
        goto cleanup;
    }

    if (wolfSSL_CTX_use_certificate_file(ctx, callbacks->certPemFile,
            WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        goto cleanup;
    }

    if (wolfSSL_CTX_use_PrivateKey_file(ctx, callbacks->keyPemFile,
            WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
        goto cleanup;
    }

#ifdef HAVE_CRL
    if (callbacks->crlPemFile != NULL) {
        if (wolfSSL_CTX_LoadCRLFile(ctx, callbacks->crlPemFile,
                WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
            goto cleanup;
        }
    }
#endif

    if (callbacks->ctx_ready)
        callbacks->ctx_ready(ctx);

    ssl = wolfSSL_new(ctx);
    if (ssl == NULL) {
        fprintf(stderr, "SSL new failed\n");
        goto cleanup;
    }
    if (wolfSSL_dtls(ssl)) {
        SOCKADDR_IN_T cliAddr;
        socklen_t     cliLen;

        cliLen = sizeof(cliAddr);
        idx = (int)recvfrom(sfd, input, sizeof(input), MSG_PEEK,
                (struct sockaddr*)&cliAddr, &cliLen);
        if (idx <= 0) {
            goto cleanup;
        }
        wolfSSL_dtls_set_peer(ssl, &cliAddr, cliLen);
    }
    else {
        CloseSocket(sfd);
    }

    if (wolfSSL_set_fd(ssl, cfd) != WOLFSSL_SUCCESS) {
        goto cleanup;
    }

    if (callbacks->loadToSSL) {
        wolfSSL_SetDevId(ssl, callbacks->devId);

        if (wolfSSL_use_certificate_file(ssl, callbacks->certPemFile,
                WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
            goto cleanup;
        }

        if (wolfSSL_use_PrivateKey_file(ssl, callbacks->keyPemFile,
                WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
            goto cleanup;
        }
    }

#ifdef NO_PSK
    #if !defined(NO_FILESYSTEM) && !defined(NO_DH)
        wolfSSL_SetTmpDH_file(ssl, dhParamFile, WOLFSSL_FILETYPE_PEM);
    #elif !defined(NO_DH)
        SetDH(ssl);  /* will repick suites with DHE, higher priority than PSK */
    #endif
#endif

    if (callbacks->ssl_ready)
        callbacks->ssl_ready(ssl);

#ifdef WOLFSSL_ASYNC_CRYPT
    err = 0; /* Reset error */
#endif
    do {
    #ifdef WOLFSSL_ASYNC_CRYPT
        if (err == WC_NO_ERR_TRACE(WC_PENDING_E)) {
            ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
            if (ret < 0) { break; } else if (ret == 0) { continue; }
        }
    #endif
        ret = wolfSSL_accept(ssl);
        err = wolfSSL_get_error(ssl, ret);
    } while (err == WC_NO_ERR_TRACE(WC_PENDING_E));
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "accept error = %d, %s\n", err,
            wolfSSL_ERR_error_string((word32)err, buff));
        /*err_sys("SSL_accept failed");*/
    }
    else {
    #ifdef WOLFSSL_ASYNC_CRYPT
        err = 0; /* Reset error */
    #endif
        do {
        #ifdef WOLFSSL_ASYNC_CRYPT
            if (err == WC_NO_ERR_TRACE(WC_PENDING_E)) {
                ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
                if (ret < 0) { break; } else if (ret == 0) { continue; }
            }
        #endif
            idx = wolfSSL_read(ssl, input, sizeof(input)-1);
            err = wolfSSL_get_error(ssl, idx);
        } while (err == WC_NO_ERR_TRACE(WC_PENDING_E));
        if (idx > 0) {
            input[idx] = 0;
            fprintf(stderr, "Client message: %s\n", input);
        }

    #ifdef WOLFSSL_ASYNC_CRYPT
        err = 0; /* Reset error */
    #endif
        do {
        #ifdef WOLFSSL_ASYNC_CRYPT
            if (err == WC_NO_ERR_TRACE(WC_PENDING_E)) {
                ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
                if (ret < 0) { break; } else if (ret == 0) { continue; }
            }
        #endif
            ret = wolfSSL_write(ssl, msg, len);
            err = wolfSSL_get_error(ssl, ret);
        } while (err == WC_NO_ERR_TRACE(WC_PENDING_E));
        if (len != ret) {
            goto cleanup;
        }

#if defined(WOLFSSL_SESSION_EXPORT) && !defined(HAVE_IO_POOL) && \
        defined(WOLFSSL_DTLS)
        if (wolfSSL_dtls(ssl)) {
            byte*  import;
            word32 sz;

            wolfSSL_dtls_export(ssl, NULL, &sz);
            import = (byte*)XMALLOC(sz, NULL, DYNAMIC_TYPE_TMP_BUFFER);
            if (import == NULL) {
                goto cleanup;
            }
            idx = wolfSSL_dtls_export(ssl, import, &sz);
            if (idx < 0) {
                goto cleanup;
            }
            if (wolfSSL_dtls_import(ssl, import, idx) < 0) {
                goto cleanup;
            }
            XFREE(import, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        }
#endif
#ifdef WOLFSSL_TIRTOS
        Task_yield();
#endif
        ((func_args*)args)->return_code = TEST_SUCCESS;
    }

    if (callbacks->on_result)
        callbacks->on_result(ssl);

    wolfSSL_shutdown(ssl);

cleanup:
    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
    CloseSocket(cfd);


#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

#if defined(NO_MAIN_DRIVER) && defined(HAVE_ECC) && defined(FP_ECC) \
                            && defined(HAVE_THREAD_LS)
    wc_ecc_fp_free();  /* free per thread cache */
#endif

    WOLFSSL_RETURN_FROM_THREAD(0);
}

/* TLS Client for API unit testing - generic */
static void run_wolfssl_client(void* args)
{
    callback_functions* callbacks = ((func_args*)args)->callbacks;

    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL*     ssl = NULL;
    SOCKET_T    sfd = 0;

    char msg[] = "hello wolfssl server!";
    int  len   = (int) XSTRLEN(msg);
    char input[1024];
    int  ret, err = 0;

    ((func_args*)args)->return_code = TEST_FAIL;

    /* set defaults */
    if (callbacks->caPemFile == NULL)
        callbacks->caPemFile = caCertFile;
    if (callbacks->certPemFile == NULL)
        callbacks->certPemFile = cliCertFile;
    if (callbacks->keyPemFile == NULL)
        callbacks->keyPemFile = cliKeyFile;

#ifdef WOLFSSL_STATIC_MEMORY
    if (callbacks->method_ex != NULL && callbacks->mem != NULL &&
        callbacks->memSz > 0) {
        ret = wolfSSL_CTX_load_static_memory(&ctx, callbacks->method_ex,
            callbacks->mem, callbacks->memSz, 0, 1);
        if (ret != WOLFSSL_SUCCESS) {
            fprintf(stderr, "CTX static new failed %d\n", ret);
            goto cleanup;
        }
    }
#else
    ctx = wolfSSL_CTX_new(callbacks->method());
#endif
    if (ctx == NULL) {
        fprintf(stderr, "CTX new failed\n");
        goto cleanup;
    }

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    if (!callbacks->loadToSSL) {
        wolfSSL_CTX_SetDevId(ctx, callbacks->devId);
    }

#ifdef WOLFSSL_ENCRYPTED_KEYS
    wolfSSL_CTX_set_default_passwd_cb(ctx, PasswordCallBack);
#endif

    if (wolfSSL_CTX_load_verify_locations(ctx, callbacks->caPemFile, 0) !=
            WOLFSSL_SUCCESS) {
        goto cleanup;
    }

    if (!callbacks->loadToSSL) {
        if (wolfSSL_CTX_use_certificate_file(ctx, callbacks->certPemFile,
                WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
            goto cleanup;
        }

        if (wolfSSL_CTX_use_PrivateKey_file(ctx, callbacks->keyPemFile,
                WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
            goto cleanup;
        }
    }

#ifdef HAVE_CRL
    if (callbacks->crlPemFile != NULL) {
        if (wolfSSL_CTX_LoadCRLFile(ctx, callbacks->crlPemFile,
                WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
            goto cleanup;
        }
    }
#endif

    if (callbacks->ctx_ready)
        callbacks->ctx_ready(ctx);

    ssl = wolfSSL_new(ctx);
    if (wolfSSL_dtls(ssl)) {
        tcp_connect(&sfd, wolfSSLIP, ((func_args*)args)->signal->port,
                    1, 0, ssl);
    }
    else {
        tcp_connect(&sfd, wolfSSLIP, ((func_args*)args)->signal->port,
                    0, 0, ssl);
    }
    if (wolfSSL_set_fd(ssl, sfd) != WOLFSSL_SUCCESS) {
        goto cleanup;
    }

    if (callbacks->loadToSSL) {
        wolfSSL_SetDevId(ssl, callbacks->devId);

        if (wolfSSL_use_certificate_file(ssl, callbacks->certPemFile,
                WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
            goto cleanup;
        }

        if (wolfSSL_use_PrivateKey_file(ssl, callbacks->keyPemFile,
                WOLFSSL_FILETYPE_PEM) != WOLFSSL_SUCCESS) {
            goto cleanup;
        }
    }

    if (callbacks->ssl_ready)
        callbacks->ssl_ready(ssl);

    #ifdef WOLFSSL_ASYNC_CRYPT
    err = 0; /* Reset error */
    #endif
    do {
    #ifdef WOLFSSL_ASYNC_CRYPT
        if (err == WC_NO_ERR_TRACE(WC_PENDING_E)) {
            ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
            if (ret < 0) { break; } else if (ret == 0) { continue; }
        }
    #endif
        ret = wolfSSL_connect(ssl);
        err = wolfSSL_get_error(ssl, ret);
    } while (err == WC_NO_ERR_TRACE(WC_PENDING_E));
    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err,
            wolfSSL_ERR_error_string((word32)err, buff));
        /*err_sys("SSL_connect failed");*/
    }
    else {
        #ifdef WOLFSSL_ASYNC_CRYPT
        err = 0; /* Reset error */
        #endif
        do {
        #ifdef WOLFSSL_ASYNC_CRYPT
            if (err == WC_NO_ERR_TRACE(WC_PENDING_E)) {
                ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
                if (ret < 0) { break; } else if (ret == 0) { continue; }
            }
        #endif
            ret = wolfSSL_write(ssl, msg, len);
            err = wolfSSL_get_error(ssl, ret);
        } while (err == WC_NO_ERR_TRACE(WC_PENDING_E));
        if (len != ret)
            goto cleanup;

        #ifdef WOLFSSL_ASYNC_CRYPT
        err = 0; /* Reset error */
        #endif
        do {
        #ifdef WOLFSSL_ASYNC_CRYPT
            if (err == WC_NO_ERR_TRACE(WC_PENDING_E)) {
                ret = wolfSSL_AsyncPoll(ssl, WOLF_POLL_FLAG_CHECK_HW);
                if (ret < 0) { break; } else if (ret == 0) { continue; }
            }
        #endif
            ret = wolfSSL_read(ssl, input, sizeof(input)-1);
            err = wolfSSL_get_error(ssl, ret);
        } while (err == WC_NO_ERR_TRACE(WC_PENDING_E));
        if (ret > 0) {
            input[ret] = '\0'; /* null term */
            fprintf(stderr, "Server response: %s\n", input);
        }
        ((func_args*)args)->return_code = TEST_SUCCESS;
    }

    if (callbacks->on_result)
        callbacks->on_result(ssl);

cleanup:
    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
    CloseSocket(sfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif
}

#endif /* ENABLE_TLS_CALLBACK_TEST */


static int test_wolfSSL_read_write(void)
{
    /* The unit testing for read and write shall happen simultaneously, since
     * one can't do anything with one without the other. (Except for a failure
     * test case.) This function will call all the others that will set up,
     * execute, and report their test findings.
     *
     * Set up the success case first. This function will become the template
     * for the other tests. This should eventually be renamed
     *
     * The success case isn't interesting, how can this fail?
     * - Do not give the client context a CA certificate. The connect should
     *   fail. Do not need server for this?
     * - Using NULL for the ssl object on server. Do not need client for this.
     * - Using NULL for the ssl object on client. Do not need server for this.
     * - Good ssl objects for client and server. Client write() without server
     *   read().
     * - Good ssl objects for client and server. Server write() without client
     *   read().
     * - Forgetting the password callback?
    */
    tcp_ready ready;
    func_args client_args;
    func_args server_args;
    THREAD_TYPE serverThread;
    EXPECT_DECLS;

    XMEMSET(&client_args, 0, sizeof(func_args));
    XMEMSET(&server_args, 0, sizeof(func_args));
#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    StartTCP();
    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    server_args.signal = &ready;
    client_args.signal = &ready;

    start_thread(test_server_nofail, &server_args, &serverThread);
    wait_tcp_ready(&server_args);
    test_client_nofail(&client_args, NULL);
    join_thread(serverThread);

    ExpectTrue(client_args.return_code);
    ExpectTrue(server_args.return_code);

    FreeTcpReady(&ready);

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_read_write_ex(void)
{
    EXPECT_DECLS;
    WOLFSSL_CTX *ctx_c = NULL;
    WOLFSSL_CTX *ctx_s = NULL;
    WOLFSSL *ssl_c = NULL;
    WOLFSSL *ssl_s = NULL;
    struct test_memio_ctx test_ctx;
    const char *test_str = "test";
    int test_str_size;
    size_t count;
    byte buf[255];

    XMEMSET(&test_ctx, 0, sizeof(test_ctx));
    ExpectIntEQ(test_memio_setup(&test_ctx, &ctx_c, &ctx_s, &ssl_c, &ssl_s,
        wolfSSLv23_client_method, wolfSSLv23_server_method), 0);
    ExpectIntEQ(test_memio_do_handshake(ssl_c, ssl_s, 10, NULL), 0);
    test_str_size = XSTRLEN("test") + 1;
    ExpectIntEQ(wolfSSL_write_ex(ssl_c, test_str, test_str_size, &count),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(count, test_str_size);
    count = 0;
    ExpectIntEQ(wolfSSL_read_ex(ssl_s, buf, sizeof(buf), &count), WOLFSSL_SUCCESS);
    ExpectIntEQ(count, test_str_size);
    ExpectIntEQ(XSTRCMP((char*)buf, test_str), 0);


    ExpectIntEQ(wolfSSL_shutdown(ssl_c), WOLFSSL_SHUTDOWN_NOT_DONE);
    ExpectIntEQ(wolfSSL_shutdown(ssl_s), WOLFSSL_SHUTDOWN_NOT_DONE);
    ExpectIntEQ(wolfSSL_shutdown(ssl_c), 1);
    ExpectIntEQ(wolfSSL_shutdown(ssl_s), 1);

    wolfSSL_free(ssl_c);
    wolfSSL_free(ssl_s);
    wolfSSL_CTX_free(ctx_c);
    wolfSSL_CTX_free(ctx_s);
    return TEST_SUCCESS;
}

static int test_wolfSSL_reuse_WOLFSSLobj(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_SESSION_CACHE) && \
    !defined(WOLFSSL_NO_TLS12)
    /* The unit test for session resumption by reusing WOLFSSL object.
     * WOLFSSL object is not cleared after first session. It reuse the object
     * for second connection.
    */
    tcp_ready ready;
    func_args client_args;
    func_args server_args;
    THREAD_TYPE serverThread;
    callback_functions client_cbf;
    callback_functions server_cbf;

    XMEMSET(&client_args, 0, sizeof(func_args));
    XMEMSET(&server_args, 0, sizeof(func_args));
    XMEMSET(&client_cbf, 0, sizeof(callback_functions));
    XMEMSET(&server_cbf, 0, sizeof(callback_functions));
#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    StartTCP();
    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    client_cbf.method = wolfTLSv1_2_client_method;
    server_cbf.method = wolfTLSv1_2_server_method;
    client_args.callbacks = &client_cbf;
    server_args.callbacks = &server_cbf;

    server_args.signal = &ready;
    client_args.signal = &ready;
    /* the var is used for loop number */
    server_args.argc = 2;

    start_thread(test_server_loop, &server_args, &serverThread);
    wait_tcp_ready(&server_args);
    test_client_reuse_WOLFSSLobj(&client_args, NULL, &server_args);
    join_thread(serverThread);

    ExpectTrue(client_args.return_code);
    ExpectTrue(server_args.return_code);

    FreeTcpReady(&ready);

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_SESSION_CACHE) &&
        * !defined(WOLFSSL_TLS13) */
    return EXPECT_RESULT();
}

#if defined(OPENSSL_EXTRA) && defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
static int test_wolfSSL_CTX_verifyDepth_ServerClient_1_ctx_ready(
    WOLFSSL_CTX* ctx)
{
    wolfSSL_CTX_set_verify(ctx, SSL_VERIFY_PEER, myVerify);
    myVerifyAction = VERIFY_USE_PREVERIFY;
    wolfSSL_CTX_set_verify_depth(ctx, 2);
    return TEST_SUCCESS;
}
#endif

static int test_wolfSSL_CTX_verifyDepth_ServerClient_1(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
    test_ssl_cbf client_cbf;
    test_ssl_cbf server_cbf;

    XMEMSET(&client_cbf, 0, sizeof(client_cbf));
    XMEMSET(&server_cbf, 0, sizeof(server_cbf));

#ifdef WOLFSSL_TLS13
    client_cbf.method = wolfTLSv1_3_client_method;
#endif /* WOLFSSL_TLS13 */
    client_cbf.ctx_ready =
        test_wolfSSL_CTX_verifyDepth_ServerClient_1_ctx_ready;

    /* test case 1 verify depth is equal to peer chain */
    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cbf,
        &server_cbf, NULL), TEST_SUCCESS);

    ExpectIntEQ(client_cbf.return_code, TEST_SUCCESS);
    ExpectIntEQ(server_cbf.return_code, TEST_SUCCESS);
#endif /* OPENSSL_EXTRA && HAVE_SSL_MEMIO_TESTS_DEPENDENCIES */

    return EXPECT_RESULT();
}

#if defined(OPENSSL_EXTRA) && defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
static int test_wolfSSL_CTX_verifyDepth_ServerClient_2_ctx_ready(
    WOLFSSL_CTX* ctx)
{
    wolfSSL_CTX_set_verify(ctx, SSL_VERIFY_PEER, myVerify);
    myVerifyAction = VERIFY_OVERRIDE_ERROR;
    wolfSSL_CTX_set_verify_depth(ctx, 0);
    return TEST_SUCCESS;
}
#endif

static int test_wolfSSL_CTX_verifyDepth_ServerClient_2(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
    test_ssl_cbf client_cbf;
    test_ssl_cbf server_cbf;

    XMEMSET(&client_cbf, 0, sizeof(client_cbf));
    XMEMSET(&server_cbf, 0, sizeof(server_cbf));

#ifdef WOLFSSL_TLS13
    client_cbf.method = wolfTLSv1_3_client_method;
#endif /* WOLFSSL_TLS13 */
    client_cbf.ctx_ready =
        test_wolfSSL_CTX_verifyDepth_ServerClient_2_ctx_ready;

    /* test case 2
     * verify depth is zero, number of peer's chain is 2.
     * verify result becomes MAX_CHAIN_ERROR, but it is overridden in
     * callback.
     */
    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cbf,
        &server_cbf, NULL), TEST_SUCCESS);

    ExpectIntEQ(client_cbf.return_code, TEST_SUCCESS);
    ExpectIntEQ(server_cbf.return_code, TEST_SUCCESS);
#endif /* OPENSSL_EXTRA && HAVE_SSL_MEMIO_TESTS_DEPENDENCIES */

    return EXPECT_RESULT();
}

#if defined(OPENSSL_EXTRA) && defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
static int test_wolfSSL_CTX_verifyDepth_ServerClient_3_ctx_ready(
    WOLFSSL_CTX* ctx)
{
    wolfSSL_CTX_set_verify(ctx, SSL_VERIFY_PEER, myVerify);
    myVerifyAction = VERIFY_USE_PREVERIFY;
    wolfSSL_CTX_set_verify_depth(ctx, 0);
    return TEST_SUCCESS;
}
#endif

static int test_wolfSSL_CTX_verifyDepth_ServerClient_3(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
    test_ssl_cbf client_cbf;
    test_ssl_cbf server_cbf;

    XMEMSET(&client_cbf, 0, sizeof(client_cbf));
    XMEMSET(&server_cbf, 0, sizeof(server_cbf));

#ifdef WOLFSSL_TLS13
    client_cbf.method = wolfTLSv1_3_client_method;
#endif /* WOLFSSL_TLS13 */
    client_cbf.ctx_ready =
        test_wolfSSL_CTX_verifyDepth_ServerClient_3_ctx_ready;

    /* test case 3
     * verify depth is zero, number of peer's chain is 2
     * verify result becomes MAX_CHAIN_ERRO. call-back returns failure.
     * therefore, handshake becomes failure.
     */
    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cbf,
        &server_cbf, NULL), -1001);

    ExpectIntEQ(client_cbf.return_code, -1000);
    ExpectIntEQ(server_cbf.return_code, -1000);
    ExpectIntEQ(client_cbf.last_err, WC_NO_ERR_TRACE(MAX_CHAIN_ERROR));
    ExpectIntEQ(server_cbf.last_err, WC_NO_ERR_TRACE(FATAL_ERROR));
#endif /* OPENSSL_EXTRA && HAVE_SSL_MEMIO_TESTS_DEPENDENCIES */

    return EXPECT_RESULT();
}

#if defined(OPENSSL_ALL) && defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && \
    !defined(WOLFSSL_NO_TLS12) && \
    defined(HAVE_ECC) && !defined(NO_AES) && !defined(NO_SHA256)
static int test_wolfSSL_CTX_set_cipher_list_server_ctx_ready(WOLFSSL_CTX* ctx)
{
    EXPECT_DECLS;
    ExpectTrue(wolfSSL_CTX_set_cipher_list(ctx, "DEFAULT:!NULL"));
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_set_cipher_list_client_ctx_ready(WOLFSSL_CTX* ctx)
{
    EXPECT_DECLS;
    ExpectTrue(wolfSSL_CTX_set_cipher_list(ctx, "ECDHE-RSA-AES128-SHA256"));
    return EXPECT_RESULT();
}
#endif

static int test_wolfSSL_CTX_set_cipher_list(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && \
    defined(HAVE_ECC) && !defined(NO_AES) && !defined(NO_SHA256)

    #if !defined(WOLFSSL_NO_TLS12)
    WOLFSSL_CTX* ctxClient = NULL;
    WOLFSSL*     sslClient = NULL;
    test_ssl_cbf client_cbf;
    test_ssl_cbf server_cbf;

    XMEMSET(&client_cbf, 0, sizeof(client_cbf));
    XMEMSET(&server_cbf, 0, sizeof(server_cbf));

    server_cbf.method = wolfTLSv1_2_server_method;
    server_cbf.ctx_ready = test_wolfSSL_CTX_set_cipher_list_server_ctx_ready;
    client_cbf.method = wolfTLSv1_2_client_method;
    client_cbf.ctx_ready = test_wolfSSL_CTX_set_cipher_list_client_ctx_ready;

    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cbf,
        &server_cbf, NULL), TEST_SUCCESS);

    ExpectIntEQ(client_cbf.return_code, TEST_SUCCESS);
    ExpectIntEQ(server_cbf.return_code, TEST_SUCCESS);

    /* check with cipher string that has '+' */
    ExpectNotNull((ctxClient = wolfSSL_CTX_new(wolfTLSv1_2_client_method())));
    /* Use trailing : with nothing to test for ASAN */
    ExpectTrue(wolfSSL_CTX_set_cipher_list(ctxClient, "ECDHE+AESGCM:"));
    ExpectNotNull((sslClient = wolfSSL_new(ctxClient)));

    /* check for the existence of an ECDHE ECDSA cipher suite */
    if (EXPECT_SUCCESS()) {
        int i = 0;
        int found = 0;
        const char* suite;

        WOLF_STACK_OF(WOLFSSL_CIPHER)* sk = NULL;
        WOLFSSL_CIPHER* current;

        ExpectNotNull((sk = wolfSSL_get_ciphers_compat(sslClient)));
        do {
            current = wolfSSL_sk_SSL_CIPHER_value(sk, i++);
            if (current) {
                suite = wolfSSL_CIPHER_get_name(current);
                if (suite && XSTRSTR(suite, "ECDSA")) {
                    found = 1;
                    break;
                }
            }
        } while (current);
        ExpectIntEQ(found, 1);
    }

    wolfSSL_free(sslClient);
    wolfSSL_CTX_free(ctxClient);

    #endif /* !WOLFSSL_NO_TLS12 */
#endif
    return EXPECT_RESULT();
}

#if defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && \
    defined(WOLFSSL_HAVE_TLS_UNIQUE)
static int test_wolfSSL_get_finished_client_on_handshake(WOLFSSL_CTX* ctx,
    WOLFSSL* ssl)
{
    EXPECT_DECLS;
    size_t msg_len;

    (void)ctx;

    /* get_finished test */
    /* 1. get own sent message */
    XMEMSET(client_side_msg1, 0, WC_MAX_DIGEST_SIZE);
    msg_len = wolfSSL_get_finished(ssl, client_side_msg1, WC_MAX_DIGEST_SIZE);
    ExpectIntGE(msg_len, 0);
    /* 2. get peer message */
    XMEMSET(client_side_msg2, 0, WC_MAX_DIGEST_SIZE);
    msg_len = wolfSSL_get_peer_finished(ssl, client_side_msg2, WC_MAX_DIGEST_SIZE);
    ExpectIntGE(msg_len, 0);

    return EXPECT_RESULT();
}
#endif

static int test_wolfSSL_get_finished(void)
{
    EXPECT_DECLS;
#if defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && \
    defined(WOLFSSL_HAVE_TLS_UNIQUE)
    test_ssl_cbf client_cbf;
    test_ssl_cbf server_cbf;

    XMEMSET(&client_cbf, 0, sizeof(client_cbf));
    XMEMSET(&server_cbf, 0, sizeof(server_cbf));

    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cbf,
        &server_cbf, test_wolfSSL_get_finished_client_on_handshake),
        TEST_SUCCESS);

    /* test received msg vs sent msg */
    ExpectIntEQ(0, XMEMCMP(client_side_msg1, server_side_msg2, WC_MAX_DIGEST_SIZE));
    ExpectIntEQ(0, XMEMCMP(client_side_msg2, server_side_msg1, WC_MAX_DIGEST_SIZE));
#endif /* HAVE_SSL_MEMIO_TESTS_DEPENDENCIES && WOLFSSL_HAVE_TLS_UNIQUE */

    return EXPECT_RESULT();
}

#if defined(HAVE_IO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
    !defined(SINGLE_THREADED) && defined(WOLFSSL_TLS13) && \
    !defined(NO_SESSION_CACHE)

/* Sessions to restore/store */
static WOLFSSL_SESSION* test_wolfSSL_CTX_add_session_client_sess;
static WOLFSSL_SESSION* test_wolfSSL_CTX_add_session_server_sess;
static WOLFSSL_CTX*     test_wolfSSL_CTX_add_session_server_ctx;

static void test_wolfSSL_CTX_add_session_ctx_ready(WOLFSSL_CTX* ctx)
{
    /* Don't store sessions. Lookup is still enabled. */
    AssertIntEQ(wolfSSL_CTX_set_session_cache_mode(ctx,
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE), WOLFSSL_SUCCESS);
#ifdef OPENSSL_EXTRA
    AssertIntEQ(wolfSSL_CTX_get_session_cache_mode(ctx) &
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE,
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE);
#endif
    /* Require both peers to provide certs */
    wolfSSL_CTX_set_verify(ctx, WOLFSSL_VERIFY_PEER, NULL);
}

static void test_wolfSSL_CTX_add_session_on_result(WOLFSSL* ssl)
{
    WOLFSSL_SESSION** sess;
#ifdef WOLFSSL_MUTEX_INITIALIZER
    static wolfSSL_Mutex m = WOLFSSL_MUTEX_INITIALIZER(m);

    (void)wc_LockMutex(&m);
#endif
    if (wolfSSL_is_server(ssl))
        sess = &test_wolfSSL_CTX_add_session_server_sess;
    else
        sess = &test_wolfSSL_CTX_add_session_client_sess;
    if (*sess == NULL) {
#ifdef NO_SESSION_CACHE_REF
        AssertNotNull(*sess = wolfSSL_get1_session(ssl));
#else
        /* Test for backwards compatibility */
        if (wolfSSL_is_server(ssl)) {
            AssertNotNull(*sess = wolfSSL_get1_session(ssl));
        }
        else {
            AssertNotNull(*sess = wolfSSL_get_session(ssl));
        }
#endif
        /* Now save the session in the internal store to make it available
         * for lookup. For TLS 1.3, we can't save the session without
         * WOLFSSL_TICKET_HAVE_ID because there is no way to retrieve the
         * session from cache. */
        if (wolfSSL_is_server(ssl)
#ifndef WOLFSSL_TICKET_HAVE_ID
                && wolfSSL_version(ssl) != TLS1_3_VERSION
#endif
                )
            AssertIntEQ(wolfSSL_CTX_add_session(wolfSSL_get_SSL_CTX(ssl),
                    *sess), WOLFSSL_SUCCESS);
    }
    else {
        /* If we have a session retrieved then remaining connections should be
         * resuming on that session */
        AssertIntEQ(wolfSSL_session_reused(ssl), 1);
    }
#ifdef WOLFSSL_MUTEX_INITIALIZER
    wc_UnLockMutex(&m);
#endif

    /* Save CTX to be able to decrypt tickets */
    if (wolfSSL_is_server(ssl) &&
            test_wolfSSL_CTX_add_session_server_ctx == NULL) {
        AssertNotNull(test_wolfSSL_CTX_add_session_server_ctx
                = wolfSSL_get_SSL_CTX(ssl));
        AssertIntEQ(wolfSSL_CTX_up_ref(wolfSSL_get_SSL_CTX(ssl)),
                WOLFSSL_SUCCESS);
    }
#ifdef SESSION_CERTS
#ifndef WOLFSSL_TICKET_HAVE_ID
    if (wolfSSL_version(ssl) != TLS1_3_VERSION &&
            wolfSSL_session_reused(ssl))
#endif
    {
        /* With WOLFSSL_TICKET_HAVE_ID the peer certs should be available
         * for all connections. TLS 1.3 only has tickets so if we don't
         * include the session id in the ticket then the certificates
         * will not be available on resumption. */
        WOLFSSL_X509* peer = wolfSSL_get_peer_certificate(ssl);
        AssertNotNull(peer);
        wolfSSL_X509_free(peer);
        AssertNotNull(wolfSSL_SESSION_get_peer_chain(*sess));
    #ifdef OPENSSL_EXTRA
        AssertNotNull(SSL_SESSION_get0_peer(*sess));
    #endif
    }
#endif /* SESSION_CERTS */
}

static void test_wolfSSL_CTX_add_session_ssl_ready(WOLFSSL* ssl)
{
    /* Set the session to reuse for the client */
    AssertIntEQ(wolfSSL_set_session(ssl,
            test_wolfSSL_CTX_add_session_client_sess), WOLFSSL_SUCCESS);
}
#endif

static int test_wolfSSL_CTX_add_session(void)
{
    EXPECT_DECLS;
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
    !defined(SINGLE_THREADED) && defined(WOLFSSL_TLS13) && \
    !defined(NO_SESSION_CACHE)
    tcp_ready ready;
    func_args client_args;
    func_args server_args;
    THREAD_TYPE serverThread;
    callback_functions client_cb;
    callback_functions server_cb;
    method_provider methods[][2] = {
#if !defined(NO_OLD_TLS) && ((!defined(NO_AES) && !defined(NO_AES_CBC)) || \
        !defined(NO_DES3))
        /* Without AES there are almost no ciphersuites available. This leads
         * to no ciphersuites being available and an error. */
        { wolfTLSv1_1_client_method, wolfTLSv1_1_server_method },
#endif
#ifndef WOLFSSL_NO_TLS12
        { wolfTLSv1_2_client_method, wolfTLSv1_2_server_method },
#endif
        /* Needs the default ticket callback since it is tied to the
         * connection context and this makes it easy to carry over the ticket
         * crypto context between connections */
#if defined(WOLFSSL_TLS13) && !defined(WOLFSSL_NO_DEF_TICKET_ENC_CB) && \
    defined(HAVE_SESSION_TICKET)
        { wolfTLSv1_3_client_method, wolfTLSv1_3_server_method },
#endif
    };
    const size_t methodsLen = sizeof(methods)/sizeof(*methods);
    size_t i, j;

    for (i = 0; i < methodsLen; i++) {
        /* First run creates a connection while the second+ run will attempt
         * to resume the connection. The trick is that the internal cache
         * is turned off. wolfSSL_CTX_add_session should put the session in
         * the cache anyway. */
        test_wolfSSL_CTX_add_session_client_sess = NULL;
        test_wolfSSL_CTX_add_session_server_sess = NULL;
        test_wolfSSL_CTX_add_session_server_ctx = NULL;

#ifdef NO_SESSION_CACHE_REF
        for (j = 0; j < 4; j++) {
#else
        /* The session may be overwritten in this case. Do only one resumption
         * to stop this test from failing intermittently. */
        for (j = 0; j < 2; j++) {
#endif
#ifdef WOLFSSL_TIRTOS
            fdOpenSession(Task_self());
#endif

            StartTCP();
            InitTcpReady(&ready);

            XMEMSET(&client_args, 0, sizeof(func_args));
            XMEMSET(&server_args, 0, sizeof(func_args));

            XMEMSET(&client_cb, 0, sizeof(callback_functions));
            XMEMSET(&server_cb, 0, sizeof(callback_functions));
            client_cb.method  = methods[i][0];
            server_cb.method  = methods[i][1];

            server_args.signal    = &ready;
            server_args.callbacks = &server_cb;
            client_args.signal    = &ready;
            client_args.callbacks = &client_cb;

            if (test_wolfSSL_CTX_add_session_server_ctx != NULL) {
                server_cb.ctx = test_wolfSSL_CTX_add_session_server_ctx;
                server_cb.isSharedCtx = 1;
            }
            server_cb.ctx_ready = test_wolfSSL_CTX_add_session_ctx_ready;
            client_cb.ctx_ready = test_wolfSSL_CTX_add_session_ctx_ready;
            if (j != 0)
                client_cb.ssl_ready = test_wolfSSL_CTX_add_session_ssl_ready;
            server_cb.on_result = test_wolfSSL_CTX_add_session_on_result;
            client_cb.on_result = test_wolfSSL_CTX_add_session_on_result;
            server_cb.ticNoInit = 1; /* Use default builtin */

            start_thread(test_server_nofail, &server_args, &serverThread);
            wait_tcp_ready(&server_args);
            test_client_nofail(&client_args, NULL);
            join_thread(serverThread);

            ExpectTrue(client_args.return_code);
            ExpectTrue(server_args.return_code);

            FreeTcpReady(&ready);

            if (EXPECT_FAIL())
                break;
        }
        wolfSSL_SESSION_free(test_wolfSSL_CTX_add_session_client_sess);
        wolfSSL_SESSION_free(test_wolfSSL_CTX_add_session_server_sess);
        wolfSSL_CTX_free(test_wolfSSL_CTX_add_session_server_ctx);

        if (EXPECT_FAIL())
            break;
    }
#endif

    return EXPECT_RESULT();
}
#if defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
     defined(WOLFSSL_TLS13) && !defined(NO_SESSION_CACHE) && \
     defined(OPENSSL_EXTRA) && defined(SESSION_CERTS) && \
     defined(HAVE_SESSION_TICKET) && \
    !defined(TITAN_SESSION_CACHE) && \
    !defined(HUGE_SESSION_CACHE) && \
    !defined(BIG_SESSION_CACHE) && \
    !defined(MEDIUM_SESSION_CACHE)

/* twcase - prefix for test_wolfSSL_CTX_add_session_ext */
/* Sessions to restore/store */
static WOLFSSL_SESSION* twcase_server_first_session_ptr;
static WOLFSSL_SESSION* twcase_client_first_session_ptr;
static WOLFSSL_CTX*     twcase_server_current_ctx_ptr;
static int twcase_new_session_called    = 0;
static int twcase_remove_session_called = 0;
static int twcase_get_session_called    = 0;

/* Test default, SESSIONS_PER_ROW*SESSION_ROWS = 3*11, see ssl.c */
#define SESSION_CACHE_SIZE 33

typedef struct {
    const byte* key;  /* key, altSessionID, session ID, NULL if empty */
    WOLFSSL_SESSION* value;
} hashTable_entry;

typedef struct {
    hashTable_entry entries[SESSION_CACHE_SIZE];  /* hash slots */
    size_t capacity;                     /* size of entries */
    size_t length;                       /* number of items in the hash table */
    wolfSSL_Mutex htLock;                /* lock */
}hashTable;

static hashTable server_sessionCache;

static int twcase_new_sessionCb(WOLFSSL *ssl, WOLFSSL_SESSION *sess)
{
    int i;
    unsigned int len;
    (void)ssl;

    /*
     * This example uses a hash table.
     * Steps you should take for a non-demo code:
     * - acquire a lock for the file named according to the session id
     * - open the file
     * - encrypt and write the SSL_SESSION object to the file
     * - release the lock
     *
     * Return:
     *  0: The callback does not wish to hold a reference of the sess
     *  1: The callback wants to hold a reference of the sess. The callback is
     *     now also responsible for calling wolfSSL_SESSION_free() on sess.
     */
    if (sess == NULL)
        return 0;

    if (wc_LockMutex(&server_sessionCache.htLock) != 0) {
        return 0;
    }
    for (i = 0; i < SESSION_CACHE_SIZE; i++) {
        if (server_sessionCache.entries[i].value == NULL) {
            server_sessionCache.entries[i].key = SSL_SESSION_get_id(sess, &len);
            server_sessionCache.entries[i].value = sess;
            server_sessionCache.length++;
            break;
        }
    }
    ++twcase_new_session_called;
    wc_UnLockMutex(&server_sessionCache.htLock);
    fprintf(stderr, "\t\ttwcase_new_session_called %d\n",
            twcase_new_session_called);
    return 1;
}

static void twcase_remove_sessionCb(WOLFSSL_CTX *ctx, WOLFSSL_SESSION *sess)
{
    int i;
    (void)ctx;
    (void)sess;

    if (sess == NULL)
        return;
    /*
     * This example uses a hash table.
     * Steps you should take for a non-demo code:
     * - acquire a lock for the file named according to the session id
     * - remove the file
     * - release the lock
     */
    if (wc_LockMutex(&server_sessionCache.htLock) != 0) {
        return;
    }
    for (i = 0; i < SESSION_CACHE_SIZE; i++) {
        if (server_sessionCache.entries[i].key != NULL &&
           XMEMCMP(server_sessionCache.entries[i].key,
                   sess->sessionID, SSL_MAX_SSL_SESSION_ID_LENGTH) == 0) {
            wolfSSL_SESSION_free(server_sessionCache.entries[i].value);
            server_sessionCache.entries[i].value = NULL;
            server_sessionCache.entries[i].key = NULL;
            server_sessionCache.length--;
            break;
        }
    }
    ++twcase_remove_session_called;
    wc_UnLockMutex(&server_sessionCache.htLock);
    fprintf(stderr, "\t\ttwcase_remove_session_called %d\n",
            twcase_remove_session_called);
}

static WOLFSSL_SESSION *twcase_get_sessionCb(WOLFSSL *ssl,
                                  const unsigned char *id, int len, int *ref)
{
    int i;
    (void)ssl;
    (void)id;
    (void)len;

    /*
     * This example uses a hash table.
     * Steps you should take for a non-demo code:
     * - acquire a lock for the file named according to the session id in the
     *   2nd arg
     * - read and decrypt contents of file and create a new SSL_SESSION
     * - object release the lock
     * - return the new session object
     */
    fprintf(stderr, "\t\ttwcase_get_session_called %d\n",
            ++twcase_get_session_called);
    /* This callback want to retain a copy of the object. If we want wolfSSL to
     * be responsible for the pointer then set to 0. */
    *ref = 1;

    for (i = 0; i < SESSION_CACHE_SIZE; i++) {
        if (server_sessionCache.entries[i].key != NULL &&
           XMEMCMP(server_sessionCache.entries[i].key, id,
                   SSL_MAX_SSL_SESSION_ID_LENGTH) == 0) {
           return server_sessionCache.entries[i].value;
        }
    }
    return NULL;
}
static int twcase_get_sessionCb_cleanup(void)
{
    int i;
    int cnt = 0;

    /* If  twcase_get_sessionCb sets *ref = 1, the application is responsible
     * for freeing sessions */

    for (i = 0; i < SESSION_CACHE_SIZE; i++) {
        if (server_sessionCache.entries[i].value != NULL) {
            wolfSSL_SESSION_free(server_sessionCache.entries[i].value);
            cnt++;
        }
    }

    fprintf(stderr, "\t\ttwcase_get_sessionCb_cleanup freed %d sessions\n",
            cnt);

    return TEST_SUCCESS;
}

static int twcase_cache_intOff_extOff(WOLFSSL_CTX* ctx)
{
    EXPECT_DECLS;
    /* off - Disable internal cache */
    ExpectIntEQ(wolfSSL_CTX_set_session_cache_mode(ctx,
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE), WOLFSSL_SUCCESS);
#ifdef OPENSSL_EXTRA
    ExpectIntEQ(wolfSSL_CTX_get_session_cache_mode(ctx) &
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE,
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE);
#endif
    /* off - Do not setup external cache */

    /* Require both peers to provide certs */
    wolfSSL_CTX_set_verify(ctx, WOLFSSL_VERIFY_PEER, NULL);
    return EXPECT_RESULT();
}

static int twcase_cache_intOn_extOff(WOLFSSL_CTX* ctx)
{
    /* on - internal cache is on by default */
    /* off - Do not setup external cache */
    /* Require both peers to provide certs */
    wolfSSL_CTX_set_verify(ctx, WOLFSSL_VERIFY_PEER, NULL);
    return TEST_SUCCESS;
}

static int twcase_cache_intOff_extOn(WOLFSSL_CTX* ctx)
{
    EXPECT_DECLS;
    /* off - Disable internal cache */
    ExpectIntEQ(wolfSSL_CTX_set_session_cache_mode(ctx,
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE), WOLFSSL_SUCCESS);
#ifdef OPENSSL_EXTRA
    ExpectIntEQ(wolfSSL_CTX_get_session_cache_mode(ctx) &
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE,
            WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE);
#endif
    /* on - Enable external cache */
    wolfSSL_CTX_sess_set_new_cb(ctx, twcase_new_sessionCb);
    wolfSSL_CTX_sess_set_remove_cb(ctx, twcase_remove_sessionCb);
    wolfSSL_CTX_sess_set_get_cb(ctx, twcase_get_sessionCb);

    /* Require both peers to provide certs */
    wolfSSL_CTX_set_verify(ctx, WOLFSSL_VERIFY_PEER, NULL);
    return EXPECT_RESULT();
}

static int twcase_cache_intOn_extOn(WOLFSSL_CTX* ctx)
{
    /* on - internal cache is on by default */
    /* on - Enable external cache */
    wolfSSL_CTX_sess_set_new_cb(ctx, twcase_new_sessionCb);
    wolfSSL_CTX_sess_set_remove_cb(ctx, twcase_remove_sessionCb);
    wolfSSL_CTX_sess_set_get_cb(ctx, twcase_get_sessionCb);

    /* Require both peers to provide certs */
    wolfSSL_CTX_set_verify(ctx, WOLFSSL_VERIFY_PEER, NULL);
    return TEST_SUCCESS;
}
static int twcase_cache_intOn_extOn_noTicket(WOLFSSL_CTX* ctx)
{
    /* on - internal cache is on by default */
    /* on - Enable external cache */
    wolfSSL_CTX_sess_set_new_cb(ctx, twcase_new_sessionCb);
    wolfSSL_CTX_sess_set_remove_cb(ctx, twcase_remove_sessionCb);
    wolfSSL_CTX_sess_set_get_cb(ctx, twcase_get_sessionCb);

    wolfSSL_CTX_set_options(ctx, WOLFSSL_OP_NO_TICKET);
    /* Require both peers to provide certs */
    wolfSSL_CTX_set_verify(ctx, WOLFSSL_VERIFY_PEER, NULL);
    return TEST_SUCCESS;
}
static int twcase_server_sess_ctx_pre_shutdown(WOLFSSL* ssl)
{
    EXPECT_DECLS;
    WOLFSSL_SESSION** sess;
    if (wolfSSL_is_server(ssl))
        sess = &twcase_server_first_session_ptr;
    else
        return TEST_SUCCESS;

    if (*sess == NULL) {
        ExpectNotNull(*sess = wolfSSL_get1_session(ssl));
        /* Now save the session in the internal store to make it available
         * for lookup. For TLS 1.3, we can't save the session without
         * WOLFSSL_TICKET_HAVE_ID because there is no way to retrieve the
         * session from cache. */
        if (wolfSSL_is_server(ssl)
#ifndef WOLFSSL_TICKET_HAVE_ID
                && wolfSSL_version(ssl) != TLS1_3_VERSION
                && wolfSSL_version(ssl) != DTLS1_3_VERSION
#endif
                ) {
            ExpectIntEQ(wolfSSL_CTX_add_session(wolfSSL_get_SSL_CTX(ssl),
                    *sess), WOLFSSL_SUCCESS);
        }
    }
    /* Save CTX to be able to decrypt tickets */
    if (twcase_server_current_ctx_ptr == NULL) {
        ExpectNotNull(twcase_server_current_ctx_ptr = wolfSSL_get_SSL_CTX(ssl));
        ExpectIntEQ(wolfSSL_CTX_up_ref(wolfSSL_get_SSL_CTX(ssl)),
                    WOLFSSL_SUCCESS);
    }
#ifdef SESSION_CERTS
#ifndef WOLFSSL_TICKET_HAVE_ID
    if (wolfSSL_version(ssl) != TLS1_3_VERSION &&
            wolfSSL_session_reused(ssl))
#endif
    {
        /* With WOLFSSL_TICKET_HAVE_ID the peer certs should be available
         * for all connections. TLS 1.3 only has tickets so if we don't
         * include the session id in the ticket then the certificates
         * will not be available on resumption. */
        WOLFSSL_X509* peer = NULL;
        ExpectNotNull(peer = wolfSSL_get_peer_certificate(ssl));
        wolfSSL_X509_free(peer);
        ExpectNotNull(wolfSSL_SESSION_get_peer_chain(*sess));
    }
#endif
    return EXPECT_RESULT();
}

static int twcase_client_sess_ctx_pre_shutdown(WOLFSSL* ssl)
{
    EXPECT_DECLS;
    WOLFSSL_SESSION** sess;
    sess = &twcase_client_first_session_ptr;
    if (*sess == NULL) {
        ExpectNotNull(*sess = wolfSSL_get1_session(ssl));
    }
    else {
        /* If we have a session retrieved then remaining connections should be
         * resuming on that session */
        ExpectIntEQ(wolfSSL_session_reused(ssl), 1);
    }

#ifdef SESSION_CERTS
#ifndef WOLFSSL_TICKET_HAVE_ID
    if (wolfSSL_version(ssl) != TLS1_3_VERSION &&
            wolfSSL_session_reused(ssl))
#endif
    {

        WOLFSSL_X509* peer = wolfSSL_get_peer_certificate(ssl);
        ExpectNotNull(peer);
        wolfSSL_X509_free(peer);
        ExpectNotNull(wolfSSL_SESSION_get_peer_chain(*sess));
#ifdef OPENSSL_EXTRA
        ExpectNotNull(wolfSSL_SESSION_get0_peer(*sess));
#endif
    }
#endif
    return EXPECT_RESULT();
}
static int twcase_client_set_sess_ssl_ready(WOLFSSL* ssl)
{
    EXPECT_DECLS;
    /* Set the session to reuse for the client */
    ExpectNotNull(ssl);
    ExpectNotNull(twcase_client_first_session_ptr);
    ExpectIntEQ(wolfSSL_set_session(ssl,twcase_client_first_session_ptr),
                WOLFSSL_SUCCESS);
    return EXPECT_RESULT();
}

struct test_add_session_ext_params {
    method_provider client_meth;
    method_provider server_meth;
    const char* tls_version;
};

static int test_wolfSSL_CTX_add_session_ext(
    struct test_add_session_ext_params* param)
{
    EXPECT_DECLS;
    /* Test the default 33 sessions */
    int j;

    /* Clear cache before starting */
    wolfSSL_CTX_flush_sessions(NULL, -1);

    XMEMSET(&server_sessionCache, 0, sizeof(hashTable));
    if (wc_InitMutex(&server_sessionCache.htLock) != 0)
        return BAD_MUTEX_E;
    server_sessionCache.capacity = SESSION_CACHE_SIZE;

    fprintf(stderr, "\tBegin %s\n", param->tls_version);
    for (j = 0; j < 5; j++) {
        int tls13 = XSTRSTR(param->tls_version, "TLSv1_3") != NULL;
        int dtls = XSTRSTR(param->tls_version, "DTLS") != NULL;
        test_ssl_cbf client_cb;
        test_ssl_cbf server_cb;

        (void)dtls;

        /* Test five cache configurations */
        twcase_client_first_session_ptr = NULL;
        twcase_server_first_session_ptr = NULL;
        twcase_server_current_ctx_ptr = NULL;
        twcase_new_session_called    = 0;
        twcase_remove_session_called = 0;
        twcase_get_session_called    = 0;

        /* connection 1 - first connection */
        fprintf(stderr, "\tconnect: %s: j=%d\n", param->tls_version, j);

        XMEMSET(&client_cb, 0, sizeof(client_cb));
        XMEMSET(&server_cb, 0, sizeof(server_cb));
        client_cb.method  = param->client_meth;
        server_cb.method  = param->server_meth;

        if (dtls)
            client_cb.doUdp = server_cb.doUdp = 1;

        /* Setup internal and external cache */
        switch (j) {
            case 0:
                /* SSL_OP_NO_TICKET stateful ticket case */
                server_cb.ctx_ready = twcase_cache_intOn_extOn_noTicket;
                break;
            case 1:
                server_cb.ctx_ready = twcase_cache_intOn_extOn;
                break;
            case 2:
                server_cb.ctx_ready = twcase_cache_intOff_extOn;
                break;
            case 3:
                server_cb.ctx_ready = twcase_cache_intOn_extOff;
                break;
            case 4:
                server_cb.ctx_ready = twcase_cache_intOff_extOff;
                break;
        }
        client_cb.ctx_ready = twcase_cache_intOff_extOff;

        /* Add session to internal cache and save SSL session for testing */
        server_cb.on_result = twcase_server_sess_ctx_pre_shutdown;
        /* Save client SSL session for testing */
        client_cb.on_result = twcase_client_sess_ctx_pre_shutdown;
        server_cb.ticNoInit = 1; /* Use default builtin */
        /* Don't free/release ctx */
        server_cb.ctx = twcase_server_current_ctx_ptr;
        server_cb.isSharedCtx = 1;

        ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cb,
            &server_cb, NULL), TEST_SUCCESS);

        ExpectIntEQ(twcase_get_session_called, 0);
        if (EXPECT_FAIL()) {
            wolfSSL_SESSION_free(twcase_client_first_session_ptr);
            wolfSSL_SESSION_free(twcase_server_first_session_ptr);
            wolfSSL_CTX_free(twcase_server_current_ctx_ptr);
            break;
        }

        switch (j) {
            case 0:
            case 1:
            case 2:
                /* cache cannot be searched with out a connection */
                /* Add a new session */
                ExpectIntEQ(twcase_new_session_called, 1);
                /* In twcase_server_sess_ctx_pre_shutdown
                 * wolfSSL_CTX_add_session which evicts the existing session
                 * in cache and adds it back in */
                ExpectIntLE(twcase_remove_session_called, 1);
                break;
            case 3:
            case 4:
                /* no external cache  */
                ExpectIntEQ(twcase_new_session_called, 0);
                ExpectIntEQ(twcase_remove_session_called, 0);
                break;
        }

        /* connection 2 - session resume */
        fprintf(stderr, "\tresume: %s: j=%d\n", param->tls_version, j);
        twcase_new_session_called    = 0;
        twcase_remove_session_called = 0;
        twcase_get_session_called    = 0;
        server_cb.on_result = 0;
        client_cb.on_result = 0;
        server_cb.ticNoInit = 1; /* Use default builtin */

        server_cb.ctx = twcase_server_current_ctx_ptr;

        /* try session resumption */
        client_cb.ssl_ready = twcase_client_set_sess_ssl_ready;

        ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cb,
            &server_cb, NULL), TEST_SUCCESS);

        /* Clear cache before checking */
        wolfSSL_CTX_flush_sessions(NULL, -1);

        switch (j) {
            case 0:
                if (tls13) {
                    /* (D)TLSv1.3 stateful case */
                    /* cache hit */
                    /* DTLS accesses cache once for stateless parsing and
                     * once for stateful parsing */
                    ExpectIntEQ(twcase_get_session_called, !dtls ? 1 : 2);

                    /* (D)TLSv1.3 creates a new ticket,
                     * updates both internal and external cache */
                    ExpectIntEQ(twcase_new_session_called, 1);
                    /* A new session ID is created for a new ticket */
                    ExpectIntEQ(twcase_remove_session_called, 2);

                }
                else {
                    /* non (D)TLSv1.3 case, no update */
                    /* DTLS accesses cache once for stateless parsing and
                     * once for stateful parsing */
#ifdef WOLFSSL_DTLS_NO_HVR_ON_RESUME
                    ExpectIntEQ(twcase_get_session_called, !dtls ? 1 : 2);
#else
                    ExpectIntEQ(twcase_get_session_called, 1);
#endif
                    ExpectIntEQ(twcase_new_session_called, 0);
                    /* Called on session added in
                     * twcase_server_sess_ctx_pre_shutdown */
                    ExpectIntEQ(twcase_remove_session_called, 1);
                }
                break;
            case 1:
                if (tls13) {
                    /* (D)TLSv1.3 case */
                    /* cache hit */
                    ExpectIntEQ(twcase_get_session_called, 1);
                    /* (D)TLSv1.3 creates a new ticket,
                     * updates both internal and external cache */
                    ExpectIntEQ(twcase_new_session_called, 1);
                    /* Called on session added in
                     * twcase_server_sess_ctx_pre_shutdown and by wolfSSL */
                    ExpectIntEQ(twcase_remove_session_called, 1);
                }
                else {
                    /* non (D)TLSv1.3 case */
                    /* cache hit */
                    /* DTLS accesses cache once for stateless parsing and
                     * once for stateful parsing */
#ifdef WOLFSSL_DTLS_NO_HVR_ON_RESUME
                    ExpectIntEQ(twcase_get_session_called, !dtls ? 1 : 2);
#else
                    ExpectIntEQ(twcase_get_session_called, 1);
#endif
                    ExpectIntEQ(twcase_new_session_called, 0);
                    /* Called on session added in
                     * twcase_server_sess_ctx_pre_shutdown */
                    ExpectIntEQ(twcase_remove_session_called, 1);
                }
                break;
            case 2:
                if (tls13) {
                    /* (D)TLSv1.3 case */
                    /* cache hit */
                    ExpectIntEQ(twcase_get_session_called, 1);
                    /* (D)TLSv1.3 creates a new ticket,
                     * updates both internal and external cache */
                    ExpectIntEQ(twcase_new_session_called, 1);
                    /* Called on session added in
                     * twcase_server_sess_ctx_pre_shutdown and by wolfSSL */
                    ExpectIntEQ(twcase_remove_session_called, 1);
                }
                else {
                    /* non (D)TLSv1.3 case */
                    /* cache hit */
                    /* DTLS accesses cache once for stateless parsing and
                     * once for stateful parsing */
#ifdef WOLFSSL_DTLS_NO_HVR_ON_RESUME
                    ExpectIntEQ(twcase_get_session_called, !dtls ? 1 : 2);
#else
                    ExpectIntEQ(twcase_get_session_called, 1);
#endif
                    ExpectIntEQ(twcase_new_session_called, 0);
                    /* Called on session added in
                     * twcase_server_sess_ctx_pre_shutdown */
                    ExpectIntEQ(twcase_remove_session_called, 1);
                }
                break;
            case 3:
            case 4:
                /* no external cache */
                ExpectIntEQ(twcase_get_session_called, 0);
                ExpectIntEQ(twcase_new_session_called, 0);
                ExpectIntEQ(twcase_remove_session_called, 0);
                break;
        }
        wolfSSL_SESSION_free(twcase_client_first_session_ptr);
        wolfSSL_SESSION_free(twcase_server_first_session_ptr);
        wolfSSL_CTX_free(twcase_server_current_ctx_ptr);

        if (EXPECT_FAIL())
            break;
    }
    twcase_get_sessionCb_cleanup();
    XMEMSET(&server_sessionCache.entries, 0,
            sizeof(server_sessionCache.entries));
    fprintf(stderr, "\tEnd %s\n", param->tls_version);

    wc_FreeMutex(&server_sessionCache.htLock);

    return EXPECT_RESULT();
}
#endif

static int test_wolfSSL_CTX_add_session_ext_tls13(void)
{
    EXPECT_DECLS;
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
     defined(WOLFSSL_TLS13) && !defined(NO_SESSION_CACHE) && \
     defined(OPENSSL_EXTRA) && defined(SESSION_CERTS) && \
     defined(HAVE_SESSION_TICKET) && \
    !defined(TITAN_SESSION_CACHE) && \
    !defined(HUGE_SESSION_CACHE) && \
    !defined(BIG_SESSION_CACHE) && \
    !defined(MEDIUM_SESSION_CACHE)
#if defined(WOLFSSL_TLS13) && !defined(WOLFSSL_NO_DEF_TICKET_ENC_CB) && \
    defined(HAVE_SESSION_TICKET) && defined(WOLFSSL_TICKET_HAVE_ID)
    struct test_add_session_ext_params param[1] =  {
        { wolfTLSv1_3_client_method, wolfTLSv1_3_server_method, "TLSv1_3" }
    };
    ExpectIntEQ(test_wolfSSL_CTX_add_session_ext(param), TEST_SUCCESS);
#endif
#endif
    return EXPECT_RESULT();
}
static int test_wolfSSL_CTX_add_session_ext_dtls13(void)
{
    EXPECT_DECLS;
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
     defined(WOLFSSL_TLS13) && !defined(NO_SESSION_CACHE) && \
     defined(OPENSSL_EXTRA) && defined(SESSION_CERTS) && \
     defined(HAVE_SESSION_TICKET) && \
    !defined(TITAN_SESSION_CACHE) && \
    !defined(HUGE_SESSION_CACHE) && \
    !defined(BIG_SESSION_CACHE) && \
    !defined(MEDIUM_SESSION_CACHE)
#if defined(WOLFSSL_TLS13) && !defined(WOLFSSL_NO_DEF_TICKET_ENC_CB) && \
    defined(HAVE_SESSION_TICKET) && defined(WOLFSSL_TICKET_HAVE_ID)
#ifdef WOLFSSL_DTLS13
    struct test_add_session_ext_params param[1] =  {
        { wolfDTLSv1_3_client_method, wolfDTLSv1_3_server_method, "DTLSv1_3" }
    };
    ExpectIntEQ(test_wolfSSL_CTX_add_session_ext(param), TEST_SUCCESS);
#endif
#endif
#endif
    return EXPECT_RESULT();
}
static int test_wolfSSL_CTX_add_session_ext_tls12(void)
{
    EXPECT_DECLS;
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
     defined(WOLFSSL_TLS13) && !defined(NO_SESSION_CACHE) && \
     defined(OPENSSL_EXTRA) && defined(SESSION_CERTS) && \
     defined(HAVE_SESSION_TICKET) && \
    !defined(TITAN_SESSION_CACHE) && \
    !defined(HUGE_SESSION_CACHE) && \
    !defined(BIG_SESSION_CACHE) && \
    !defined(MEDIUM_SESSION_CACHE)
#ifndef WOLFSSL_NO_TLS12
    struct test_add_session_ext_params param[1] =  {
        { wolfTLSv1_2_client_method, wolfTLSv1_2_server_method, "TLSv1_2" }
    };
    ExpectIntEQ(test_wolfSSL_CTX_add_session_ext(param), TEST_SUCCESS);
#endif
#endif
    return EXPECT_RESULT();
}
static int test_wolfSSL_CTX_add_session_ext_dtls12(void)
{
    EXPECT_DECLS;
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
     defined(WOLFSSL_TLS13) && !defined(NO_SESSION_CACHE) && \
     defined(OPENSSL_EXTRA) && defined(SESSION_CERTS) && \
     defined(HAVE_SESSION_TICKET) && \
    !defined(TITAN_SESSION_CACHE) && \
    !defined(HUGE_SESSION_CACHE) && \
    !defined(BIG_SESSION_CACHE) && \
    !defined(MEDIUM_SESSION_CACHE)
#ifndef WOLFSSL_NO_TLS12
#ifdef WOLFSSL_DTLS
    struct test_add_session_ext_params param[1] =  {
        { wolfDTLSv1_2_client_method, wolfDTLSv1_2_server_method, "DTLSv1_2" }
    };
    ExpectIntEQ(test_wolfSSL_CTX_add_session_ext(param), TEST_SUCCESS);
#endif
#endif
#endif
    return EXPECT_RESULT();
}
static int test_wolfSSL_CTX_add_session_ext_tls11(void)
{
    EXPECT_DECLS;
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
     defined(WOLFSSL_TLS13) && !defined(NO_SESSION_CACHE) && \
     defined(OPENSSL_EXTRA) && defined(SESSION_CERTS) && \
     defined(HAVE_SESSION_TICKET) && \
    !defined(TITAN_SESSION_CACHE) && \
    !defined(HUGE_SESSION_CACHE) && \
    !defined(BIG_SESSION_CACHE) && \
    !defined(MEDIUM_SESSION_CACHE)
#if !defined(NO_OLD_TLS) && ((!defined(NO_AES) && !defined(NO_AES_CBC)) || \
        !defined(NO_DES3))
    struct test_add_session_ext_params param[1] =  {
        { wolfTLSv1_1_client_method, wolfTLSv1_1_server_method, "TLSv1_1" }
    };
    ExpectIntEQ(test_wolfSSL_CTX_add_session_ext(param), TEST_SUCCESS);
#endif
#endif
    return EXPECT_RESULT();
}
static int test_wolfSSL_CTX_add_session_ext_dtls1(void)
{
    EXPECT_DECLS;
#if defined(HAVE_IO_TESTS_DEPENDENCIES) && defined(HAVE_EXT_CACHE) && \
     defined(WOLFSSL_TLS13) && !defined(NO_SESSION_CACHE) && \
     defined(OPENSSL_EXTRA) && defined(SESSION_CERTS) && \
     defined(HAVE_SESSION_TICKET) && \
    !defined(TITAN_SESSION_CACHE) && \
    !defined(HUGE_SESSION_CACHE) && \
    !defined(BIG_SESSION_CACHE) && \
    !defined(MEDIUM_SESSION_CACHE)
#if !defined(NO_OLD_TLS) && ((!defined(NO_AES) && !defined(NO_AES_CBC)) || \
        !defined(NO_DES3))
#ifdef WOLFSSL_DTLS
    struct test_add_session_ext_params param[1] =  {
        { wolfDTLSv1_client_method, wolfDTLSv1_server_method, "DTLSv1_0" }
    };
    ExpectIntEQ(test_wolfSSL_CTX_add_session_ext(param), TEST_SUCCESS);
#endif
#endif
#endif
    return EXPECT_RESULT();
}

#if defined(WOLFSSL_DTLS) && defined(WOLFSSL_SESSION_EXPORT)
/* canned export of a session using older version 3 */
static unsigned char version_3[] = {
    0xA5, 0xA3, 0x01, 0x88, 0x00, 0x3c, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x80, 0x0C, 0x00, 0x00, 0x00,
    0x00, 0x80, 0x00, 0x1C, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0x30,
    0x05, 0x09, 0x0A, 0x01, 0x01, 0x00, 0x0D, 0x05,
    0xFE, 0xFD, 0x01, 0x25, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x06, 0x00, 0x05, 0x00, 0x06, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x06, 0x00, 0x01, 0x00, 0x07, 0x00, 0x00,
    0x00, 0x30, 0x00, 0x00, 0x00, 0x10, 0x01, 0x01,
    0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x3F,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x05,
    0x12, 0xCF, 0x22, 0xA1, 0x9F, 0x1C, 0x39, 0x1D,
    0x31, 0x11, 0x12, 0x1D, 0x11, 0x18, 0x0D, 0x0B,
    0xF3, 0xE1, 0x4D, 0xDC, 0xB1, 0xF1, 0x39, 0x98,
    0x91, 0x6C, 0x48, 0xE5, 0xED, 0x11, 0x12, 0xA0,
    0x00, 0xF2, 0x25, 0x4C, 0x09, 0x26, 0xD1, 0x74,
    0xDF, 0x23, 0x40, 0x15, 0x6A, 0x42, 0x2A, 0x26,
    0xA5, 0xAC, 0x56, 0xD5, 0x4A, 0x20, 0xB7, 0xE9,
    0xEF, 0xEB, 0xAF, 0xA8, 0x1E, 0x23, 0x7C, 0x04,
    0xAA, 0xA1, 0x6D, 0x92, 0x79, 0x7B, 0xFA, 0x80,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x0C, 0x79, 0x7B, 0xFA, 0x80, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xAA, 0xA1, 0x6D,
    0x92, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x10, 0x00, 0x20, 0x00, 0x04, 0x00,
    0x10, 0x00, 0x10, 0x08, 0x02, 0x05, 0x08, 0x01,
    0x30, 0x28, 0x00, 0x00, 0x0F, 0x00, 0x02, 0x00,
    0x09, 0x31, 0x32, 0x37, 0x2E, 0x30, 0x2E, 0x30,
    0x2E, 0x31, 0xED, 0x4F
};
#endif /* defined(WOLFSSL_DTLS) && defined(WOLFSSL_SESSION_EXPORT) */

static int test_wolfSSL_dtls_export(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_DTLS) && defined(WOLFSSL_SESSION_EXPORT)
    tcp_ready ready;
    func_args client_args;
    func_args server_args;
    THREAD_TYPE serverThread;
    callback_functions server_cbf;
    callback_functions client_cbf;
#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    /* set using dtls */
    XMEMSET(&client_args, 0, sizeof(func_args));
    XMEMSET(&server_args, 0, sizeof(func_args));
    XMEMSET(&server_cbf, 0, sizeof(callback_functions));
    XMEMSET(&client_cbf, 0, sizeof(callback_functions));
    server_cbf.method = wolfDTLSv1_2_server_method;
    client_cbf.method = wolfDTLSv1_2_client_method;
    server_args.callbacks = &server_cbf;
    client_args.callbacks = &client_cbf;

    server_args.signal = &ready;
    client_args.signal = &ready;

    start_thread(run_wolfssl_server, &server_args, &serverThread);
    wait_tcp_ready(&server_args);
    run_wolfssl_client(&client_args);
    join_thread(serverThread);

    ExpectTrue(client_args.return_code);
    ExpectTrue(server_args.return_code);

    FreeTcpReady(&ready);

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    if (EXPECT_SUCCESS()) {
        SOCKET_T sockfd = 0;
        WOLFSSL_CTX* ctx = NULL;
        WOLFSSL*     ssl = NULL;
        char msg[64] = "hello wolfssl!";
        char reply[1024];
        int  msgSz = (int)XSTRLEN(msg);
        byte *session, *window;
        unsigned int sessionSz = 0;
        unsigned int windowSz = 0;

#ifndef TEST_IPV6
        struct sockaddr_in peerAddr;
#else
        struct sockaddr_in6 peerAddr;
#endif /* TEST_IPV6 */

        int i;


        /* Set ctx to DTLS 1.2 */
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfDTLSv1_2_server_method()));
        ExpectNotNull(ssl = wolfSSL_new(ctx));

        /* test importing version 3 */
        ExpectIntGE(wolfSSL_dtls_import(ssl, version_3, sizeof(version_3)), 0);

        /* test importing bad length and bad version */
        version_3[2]++;
        ExpectIntLT(wolfSSL_dtls_import(ssl, version_3, sizeof(version_3)), 0);
        version_3[2]--; version_3[1] = 0XA0;
        ExpectIntLT(wolfSSL_dtls_import(ssl, version_3, sizeof(version_3)), 0);
        wolfSSL_free(ssl);
        wolfSSL_CTX_free(ctx);


    /* check storing client state after connection and storing window only */
#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    /* set using dtls */
    XMEMSET(&server_args, 0, sizeof(func_args));
    XMEMSET(&server_cbf, 0, sizeof(callback_functions));
    server_cbf.method = wolfDTLSv1_2_server_method;
    server_cbf.doUdp = 1;
    server_args.callbacks = &server_cbf;
    server_args.argc = 3; /* set loop_count to 3 */


    server_args.signal = &ready;
    start_thread(test_server_nofail, &server_args, &serverThread);
    wait_tcp_ready(&server_args);

    /* create and connect with client */
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfDTLSv1_2_client_method()));
    ExpectIntEQ(WOLFSSL_SUCCESS,
            wolfSSL_CTX_load_verify_locations(ctx, caCertFile, 0));
    ExpectIntEQ(WOLFSSL_SUCCESS,
          wolfSSL_CTX_use_certificate_file(ctx, cliCertFile, SSL_FILETYPE_PEM));
    ExpectIntEQ(WOLFSSL_SUCCESS,
            wolfSSL_CTX_use_PrivateKey_file(ctx, cliKeyFile, SSL_FILETYPE_PEM));
    tcp_connect(&sockfd, wolfSSLIP, server_args.signal->port, 1, 0, NULL);
    ExpectNotNull(ssl = wolfSSL_new(ctx));
    ExpectIntEQ(wolfSSL_set_fd(ssl, sockfd), WOLFSSL_SUCCESS);

    /* store server information connected too */
    XMEMSET(&peerAddr, 0, sizeof(peerAddr));
#ifndef TEST_IPV6
    peerAddr.sin_family = AF_INET;
    ExpectIntEQ(XINET_PTON(AF_INET, wolfSSLIP, &peerAddr.sin_addr),1);
    peerAddr.sin_port = XHTONS(server_args.signal->port);
#else
    peerAddr.sin6_family = AF_INET6;
    ExpectIntEQ(
        XINET_PTON(AF_INET6, wolfSSLIP, &peerAddr.sin6_addr),1);
    peerAddr.sin6_port = XHTONS(server_args.signal->port);
#endif

    ExpectIntEQ(wolfSSL_dtls_set_peer(ssl, &peerAddr, sizeof(peerAddr)),
                    WOLFSSL_SUCCESS);

    ExpectIntEQ(wolfSSL_connect(ssl), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_dtls_export(ssl, NULL, &sessionSz), 0);
    session = (byte*)XMALLOC(sessionSz, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    ExpectIntGT(wolfSSL_dtls_export(ssl, session, &sessionSz), 0);
    ExpectIntEQ(wolfSSL_write(ssl, msg, msgSz), msgSz);
    ExpectIntGT(wolfSSL_read(ssl, reply, sizeof(reply)), 0);
    ExpectIntEQ(wolfSSL_dtls_export_state_only(ssl, NULL, &windowSz), 0);
    window = (byte*)XMALLOC(windowSz, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    ExpectIntGT(wolfSSL_dtls_export_state_only(ssl, window, &windowSz), 0);
    wolfSSL_free(ssl);

    for (i = 1; EXPECT_SUCCESS() && i < server_args.argc; i++) {
        /* restore state */
        ExpectNotNull(ssl = wolfSSL_new(ctx));
        ExpectIntGT(wolfSSL_dtls_import(ssl, session, sessionSz), 0);
        ExpectIntGT(wolfSSL_dtls_import(ssl, window, windowSz), 0);
        ExpectIntEQ(wolfSSL_set_fd(ssl, sockfd), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_dtls_set_peer(ssl, &peerAddr, sizeof(peerAddr)),
                    WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_write(ssl, msg, msgSz), msgSz);
        ExpectIntGE(wolfSSL_read(ssl, reply, sizeof(reply)), 0);
        ExpectIntGT(wolfSSL_dtls_export_state_only(ssl, window, &windowSz), 0);
        wolfSSL_free(ssl);
    }
    XFREE(session, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(window, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    wolfSSL_CTX_free(ctx);

    fprintf(stderr, "done and waiting for server\n");
    join_thread(serverThread);
    ExpectIntEQ(server_args.return_code, TEST_SUCCESS);

    FreeTcpReady(&ready);

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif
    }
#endif

    return EXPECT_RESULT();
}

#if defined(WOLFSSL_SESSION_EXPORT) && !defined(WOLFSSL_NO_TLS12)
#ifdef WOLFSSL_TLS13
static const byte canned_client_tls13_session[] = {
    0xA7, 0xA4, 0x01, 0x18, 0x00, 0x41, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x80, 0x04, 0x00, 0x00, 0x00,
    0x00, 0x80, 0x00, 0x1C, 0x01, 0x00, 0x00, 0x01,
    0x00, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
    0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x13,
    0x01, 0x0A, 0x0F, 0x10, 0x01, 0x02, 0x09, 0x00,
    0x05, 0x00, 0x00, 0x00, 0x00, 0x03, 0x04, 0x00,
    0xB7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x27, 0x00, 0x00, 0x00,
    0x11, 0x01, 0x01, 0x00, 0x20, 0x84, 0x4F, 0x18,
    0xD8, 0xC1, 0x24, 0xD8, 0xBB, 0x17, 0x9E, 0x31,
    0xA3, 0xF8, 0xA7, 0x3C, 0xBA, 0xEC, 0xFA, 0xB4,
    0x7F, 0xC5, 0x78, 0xEB, 0x6D, 0xE3, 0x2B, 0x7B,
    0x94, 0xBE, 0x20, 0x11, 0x7E, 0x17, 0x10, 0xA7,
    0x10, 0x19, 0xEC, 0x62, 0xCC, 0xBE, 0xF5, 0x01,
    0x35, 0x3C, 0xEA, 0xEF, 0x44, 0x3C, 0x40, 0xA2,
    0xBC, 0x18, 0x43, 0xA1, 0xA1, 0x65, 0x5C, 0x48,
    0xE2, 0xF9, 0x38, 0xEB, 0x11, 0x10, 0x72, 0x7C,
    0x78, 0x22, 0x13, 0x3B, 0x19, 0x40, 0xF0, 0x73,
    0xBE, 0x96, 0x14, 0x78, 0x26, 0xB9, 0x6B, 0x2E,
    0x72, 0x22, 0x0D, 0x90, 0x94, 0xDD, 0x78, 0x77,
    0xFC, 0x0C, 0x2E, 0x63, 0x6E, 0xF0, 0x0C, 0x35,
    0x41, 0xCD, 0xF3, 0x49, 0x31, 0x08, 0xD0, 0x6F,
    0x02, 0x3D, 0xC1, 0xD3, 0xB7, 0xEE, 0x3A, 0xA0,
    0x8E, 0xA1, 0x4D, 0xC3, 0x2E, 0x5E, 0x06, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C,
    0x35, 0x41, 0xCD, 0xF3, 0x49, 0x31, 0x08, 0xD0,
    0x6F, 0x02, 0x3D, 0xC1, 0xD3, 0xB7, 0xEE, 0x3A,
    0xA0, 0x8E, 0xA1, 0x4D, 0xC3, 0x2E, 0x5E, 0x06,
    0x00, 0x10, 0x00, 0x10, 0x00, 0x0C, 0x00, 0x10,
    0x00, 0x10, 0x07, 0x02, 0x04, 0x00, 0x00, 0x20,
    0x28, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x03
};

static const byte canned_server_tls13_session[] = {
    0xA7, 0xA4, 0x01, 0x18, 0x00, 0x41, 0x01, 0x00,
    0x01, 0x00, 0x00, 0x80, 0x04, 0x00, 0x00, 0x00,
    0x00, 0x80, 0x00, 0x1C, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x13,
    0x01, 0x0A, 0x0F, 0x10, 0x01, 0x02, 0x00, 0x0F,
    0x05, 0x00, 0x00, 0x00, 0x00, 0x03, 0x04, 0x00,
    0xB7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x17, 0x00, 0x00, 0x00,
    0x11, 0x01, 0x01, 0x00, 0x20, 0x84, 0x4F, 0x18,
    0xD8, 0xC1, 0x24, 0xD8, 0xBB, 0x17, 0x9E, 0x31,
    0xA3, 0xF8, 0xA7, 0x3C, 0xBA, 0xEC, 0xFA, 0xB4,
    0x7F, 0xC5, 0x78, 0xEB, 0x6D, 0xE3, 0x2B, 0x7B,
    0x94, 0xBE, 0x20, 0x11, 0x7E, 0x17, 0x10, 0xA7,
    0x10, 0x19, 0xEC, 0x62, 0xCC, 0xBE, 0xF5, 0x01,
    0x35, 0x3C, 0xEA, 0xEF, 0x44, 0x3C, 0x40, 0xA2,
    0xBC, 0x18, 0x43, 0xA1, 0xA1, 0x65, 0x5C, 0x48,
    0xE2, 0xF9, 0x38, 0xEB, 0x11, 0x10, 0x72, 0x7C,
    0x78, 0x22, 0x13, 0x3B, 0x19, 0x40, 0xF0, 0x73,
    0xBE, 0x96, 0x14, 0x78, 0x26, 0xB9, 0x6B, 0x2E,
    0x72, 0x22, 0x0D, 0x90, 0x94, 0xDD, 0x78, 0x77,
    0xFC, 0x0C, 0x2E, 0x63, 0x6E, 0xF0, 0x0C, 0x35,
    0x41, 0xCD, 0xF3, 0x49, 0x31, 0x08, 0xD0, 0x6F,
    0x02, 0x3D, 0xC1, 0xD3, 0xB7, 0xEE, 0x3A, 0xA0,
    0x8E, 0xA1, 0x4D, 0xC3, 0x2E, 0x5E, 0x06, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C,
    0xD3, 0xB7, 0xEE, 0x3A, 0xA0, 0x8E, 0xA1, 0x4D,
    0xC3, 0x2E, 0x5E, 0x06, 0x35, 0x41, 0xCD, 0xF3,
    0x49, 0x31, 0x08, 0xD0, 0x6F, 0x02, 0x3D, 0xC1,
    0x00, 0x10, 0x00, 0x10, 0x00, 0x0C, 0x00, 0x10,
    0x00, 0x10, 0x07, 0x02, 0x04, 0x00, 0x00, 0x20,
    0x28, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04
};
#endif /* WOLFSSL_TLS13 */

static const byte canned_client_session[] = {
    0xA7, 0xA4, 0x01, 0x40, 0x00, 0x41, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x80, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x80, 0x00, 0x1C, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0,
    0x27, 0x0A, 0x0D, 0x10, 0x01, 0x01, 0x0A, 0x00,
    0x05, 0x00, 0x01, 0x01, 0x01, 0x03, 0x03, 0x00,
    0xBF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x0A, 0x01, 0x01, 0x00, 0x20, 0x69, 0x11, 0x6D,
    0x97, 0x15, 0x6E, 0x52, 0x27, 0xD6, 0x1D, 0x1D,
    0xF5, 0x0D, 0x59, 0xA5, 0xAC, 0x2E, 0x8C, 0x0E,
    0xCB, 0x26, 0x1E, 0xE2, 0xCE, 0xBB, 0xCE, 0xE1,
    0x7D, 0xD7, 0xEF, 0xA5, 0x44, 0x80, 0x2A, 0xDE,
    0xBB, 0x75, 0xB0, 0x1D, 0x75, 0x17, 0x20, 0x4C,
    0x08, 0x05, 0x1B, 0xBA, 0x60, 0x1F, 0x6C, 0x91,
    0x8C, 0xAA, 0xBB, 0xE5, 0xA3, 0x0B, 0x12, 0x3E,
    0xC0, 0x35, 0x43, 0x1D, 0xE2, 0x10, 0xE2, 0x02,
    0x92, 0x4B, 0x8F, 0x05, 0xA9, 0x4B, 0xCC, 0x90,
    0xC3, 0x0E, 0xC2, 0x0F, 0xE9, 0x33, 0x85, 0x9B,
    0x3C, 0x19, 0x21, 0xD5, 0x62, 0xE5, 0xE1, 0x17,
    0x8F, 0x8C, 0x19, 0x52, 0xD8, 0x59, 0x10, 0x2D,
    0x20, 0x6F, 0xBA, 0xC1, 0x1C, 0xD1, 0x82, 0xC7,
    0x32, 0x1B, 0xBB, 0xCC, 0x30, 0x03, 0xD7, 0x3A,
    0xC8, 0x18, 0xED, 0x58, 0xC8, 0x11, 0xFE, 0x71,
    0x9C, 0x71, 0xD8, 0x6B, 0xE0, 0x25, 0x64, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x10, 0x00, 0x10, 0x00, 0x10, 0x00, 0x10,
    0x00, 0x00, 0x06, 0x01, 0x04, 0x08, 0x01, 0x20,
    0x28, 0x00, 0x09, 0xE1, 0x50, 0x70, 0x02, 0x2F,
    0x7E, 0xDA, 0xBD, 0x40, 0xC5, 0x58, 0x87, 0xCE,
    0x43, 0xF3, 0xC5, 0x8F, 0xA1, 0x59, 0x93, 0xEF,
    0x7E, 0xD3, 0xD0, 0xB5, 0x87, 0x1D, 0x81, 0x54,
    0x14, 0x63, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x03
};


static const byte canned_server_session[] = {
    0xA7, 0xA4, 0x01, 0x40, 0x00, 0x41, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x80, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x80, 0x00, 0x1C, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0,
    0x27, 0x08, 0x0F, 0x10, 0x01, 0x01, 0x00, 0x11,
    0x05, 0x00, 0x01, 0x01, 0x01, 0x03, 0x03, 0x00,
    0xBF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x0A, 0x01, 0x01, 0x00, 0x20, 0x69, 0x11, 0x6D,
    0x97, 0x15, 0x6E, 0x52, 0x27, 0xD6, 0x1D, 0x1D,
    0xF5, 0x0D, 0x59, 0xA5, 0xAC, 0x2E, 0x8C, 0x0E,
    0xCB, 0x26, 0x1E, 0xE2, 0xCE, 0xBB, 0xCE, 0xE1,
    0x7D, 0xD7, 0xEF, 0xA5, 0x44, 0x80, 0x2A, 0xDE,
    0xBB, 0x75, 0xB0, 0x1D, 0x75, 0x17, 0x20, 0x4C,
    0x08, 0x05, 0x1B, 0xBA, 0x60, 0x1F, 0x6C, 0x91,
    0x8C, 0xAA, 0xBB, 0xE5, 0xA3, 0x0B, 0x12, 0x3E,
    0xC0, 0x35, 0x43, 0x1D, 0xE2, 0x10, 0xE2, 0x02,
    0x92, 0x4B, 0x8F, 0x05, 0xA9, 0x4B, 0xCC, 0x90,
    0xC3, 0x0E, 0xC2, 0x0F, 0xE9, 0x33, 0x85, 0x9B,
    0x3C, 0x19, 0x21, 0xD5, 0x62, 0xE5, 0xE1, 0x17,
    0x8F, 0x8C, 0x19, 0x52, 0xD8, 0x59, 0x10, 0x2D,
    0x20, 0x6F, 0xBA, 0xC1, 0x1C, 0xD1, 0x82, 0xC7,
    0x32, 0x1B, 0xBB, 0xCC, 0x30, 0x03, 0xD7, 0x3A,
    0xC8, 0x18, 0xED, 0x58, 0xC8, 0x11, 0xFE, 0x71,
    0x9C, 0x71, 0xD8, 0x6B, 0xE0, 0x25, 0x64, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x10, 0x00, 0x10, 0x00, 0x10, 0x00, 0x10,
    0x00, 0x00, 0x06, 0x01, 0x04, 0x08, 0x01, 0x20,
    0x28, 0x00, 0xC5, 0x8F, 0xA1, 0x59, 0x93, 0xEF,
    0x7E, 0xD3, 0xD0, 0xB5, 0x87, 0x1D, 0x81, 0x54,
    0x14, 0x63, 0x09, 0xE1, 0x50, 0x70, 0x02, 0x2F,
    0x7E, 0xDA, 0xBD, 0x40, 0xC5, 0x58, 0x87, 0xCE,
    0x43, 0xF3, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04
};


static THREAD_RETURN WOLFSSL_THREAD tls_export_server(void* args)
{
    SOCKET_T sockfd = 0;
    SOCKET_T clientfd = 0;
    word16 port;

    callback_functions* cbf;
    WOLFSSL_CTX* ctx = 0;
    WOLFSSL* ssl = 0;

    char msg[] = "I hear you fa shizzle!";
    char input[1024];
    int idx;

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    ((func_args*)args)->return_code = TEST_FAIL;
    cbf = ((func_args*)args)->callbacks;

#if defined(USE_WINDOWS_API)
    port = ((func_args*)args)->signal->port;
#elif defined(NO_MAIN_DRIVER) && !defined(WOLFSSL_SNIFFER) && \
     !defined(WOLFSSL_MDK_SHELL) && !defined(WOLFSSL_TIRTOS)
    /* Let tcp_listen assign port */
    port = 0;
#else
    /* Use default port */
    port = wolfSSLPort;
#endif

    /* do it here to detect failure */
    tcp_accept(&sockfd, &clientfd, (func_args*)args, port, 0, 0, 0, 0, 1, 0, 0);
    CloseSocket(sockfd);

    {
        WOLFSSL_METHOD* method = NULL;
        if (cbf != NULL && cbf->method != NULL) {
            method = cbf->method();
        }
        else {
            method = wolfTLSv1_2_server_method();
        }
        ctx = wolfSSL_CTX_new(method);
    }
    if (ctx == NULL) {
        goto done;
    }
    wolfSSL_CTX_set_cipher_list(ctx, "ECDHE-RSA-AES128-SHA256");

    /* call ctx setup callback */
    if (cbf != NULL && cbf->ctx_ready != NULL) {
        cbf->ctx_ready(ctx);
    }

    ssl = wolfSSL_new(ctx);
    if (ssl == NULL) {
        goto done;
    }
    wolfSSL_set_fd(ssl, clientfd);

    /* call ssl setup callback */
    if (cbf != NULL && cbf->ssl_ready != NULL) {
        cbf->ssl_ready(ssl);
    }
    idx = wolfSSL_read(ssl, input, sizeof(input)-1);
    if (idx > 0) {
        input[idx] = '\0';
        fprintf(stderr, "Client message export/import: %s\n", input);
    }
    else {
        fprintf(stderr, "ret = %d error = %d\n", idx,
            wolfSSL_get_error(ssl, idx));
        goto done;
    }

    if (wolfSSL_write(ssl, msg, sizeof(msg)) != sizeof(msg)) {
        /*err_sys("SSL_write failed");*/
        WOLFSSL_RETURN_FROM_THREAD(0);
    }

#ifdef WOLFSSL_TIRTOS
    Task_yield();
#endif

    ((func_args*)args)->return_code = TEST_SUCCESS;

done:
    wolfSSL_shutdown(ssl);
    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    CloseSocket(clientfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

#if defined(NO_MAIN_DRIVER) && defined(HAVE_ECC) && defined(FP_ECC) \
                            && defined(HAVE_THREAD_LS)
    wc_ecc_fp_free();  /* free per thread cache */
#endif

#if defined(HAVE_SESSION_TICKET) && \
    ((defined(HAVE_CHACHA) && defined(HAVE_POLY1305)) || defined(HAVE_AESGCM))
#if defined(OPENSSL_EXTRA) && defined(HAVE_AESGCM)
    OpenSSLTicketCleanup();
#elif defined(WOLFSSL_NO_DEF_TICKET_ENC_CB)
    TicketCleanup();
#endif
#endif

    WOLFSSL_RETURN_FROM_THREAD(0);
}


static void load_tls12_canned_server(WOLFSSL* ssl)
{
    int clientfd = wolfSSL_get_fd(ssl);
    AssertIntEQ(wolfSSL_tls_import(ssl, canned_server_session,
                sizeof(canned_server_session)), sizeof(canned_server_session));
    wolfSSL_set_fd(ssl, clientfd);
}


#ifdef WOLFSSL_TLS13
static void load_tls13_canned_server(WOLFSSL* ssl)
{
    int clientfd = wolfSSL_get_fd(ssl);
    AssertIntEQ(wolfSSL_tls_import(ssl, canned_server_tls13_session,
            sizeof(canned_server_tls13_session)),
            sizeof(canned_server_tls13_session));
    wolfSSL_set_fd(ssl, clientfd);
}
#endif


/* v is for version WOLFSSL_TLSV1_2 or WOLFSSL_TLSV1_3 */
static int test_wolfSSL_tls_export_run(int v)
{
    EXPECT_DECLS;
    SOCKET_T sockfd = 0;
    WOLFSSL_CTX*     ctx     = 0;
    WOLFSSL*         ssl     = 0;
    char msg[64] = "hello wolfssl!";
    char reply[1024];
    word32 replySz;
    int  msgSz = (int)XSTRLEN(msg);
    const byte* clientSession = NULL;
    int   clientSessionSz = 0;

    tcp_ready ready;
    func_args server_args;
    THREAD_TYPE serverThread;
    callback_functions server_cbf;

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    XMEMSET(&server_args, 0, sizeof(func_args));
    XMEMSET(&server_cbf, 0, sizeof(callback_functions));
    switch (v) {
        case WOLFSSL_TLSV1_2:
            server_cbf.method     = wolfTLSv1_2_server_method;
            server_cbf.ssl_ready  = load_tls12_canned_server;

            /* setup the client side */
            ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_2_client_method()));
            wolfSSL_CTX_set_cipher_list(ctx, "ECDHE-RSA-AES128-SHA256");
            clientSession   = canned_client_session;
            clientSessionSz = sizeof(canned_client_session);
            break;
    #ifdef WOLFSSL_TLS13
        case WOLFSSL_TLSV1_3:
            server_cbf.method     = wolfTLSv1_3_server_method;
            server_cbf.ssl_ready  = load_tls13_canned_server;

            /* setup the client side */
            ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_3_client_method()));
            clientSession   = canned_client_tls13_session;
            clientSessionSz = sizeof(canned_client_tls13_session);
            break;
    #endif
    }
    server_args.callbacks = &server_cbf;
    server_args.signal    = &ready;

    start_thread(tls_export_server, &server_args, &serverThread);
    wait_tcp_ready(&server_args);


#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    ExpectNotNull(ssl = wolfSSL_new(ctx));
    tcp_connect(&sockfd, wolfSSLIP, ready.port, 0, 0, ssl);
    ExpectIntEQ(wolfSSL_tls_import(ssl, clientSession, clientSessionSz),
                clientSessionSz);
    replySz = sizeof(reply);
    ExpectIntGT(wolfSSL_tls_export(ssl, (byte*)reply, &replySz), 0);
#if !defined(NO_PSK) && defined(HAVE_ANON)
    /* index 20 has is setting if PSK was on and 49 is if anon is allowed */
    ExpectIntEQ(XMEMCMP(reply, clientSession, replySz), 0);
#endif
    wolfSSL_set_fd(ssl, sockfd);

    ExpectIntEQ(wolfSSL_write(ssl, msg, msgSz), msgSz);
    ExpectIntGT(wolfSSL_read(ssl, reply, sizeof(reply)-1), 0);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

    CloseSocket(sockfd);

#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

#if defined(NO_MAIN_DRIVER) && defined(HAVE_ECC) && defined(FP_ECC) \
                            && defined(HAVE_THREAD_LS)
    wc_ecc_fp_free();  /* free per thread cache */
#endif

    join_thread(serverThread);

    ExpectIntEQ(server_args.return_code, TEST_SUCCESS);
    FreeTcpReady(&ready);

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    return EXPECT_RESULT();
}
#endif

static int test_wolfSSL_tls_export(void)
{
    int res = TEST_SKIPPED;
#if defined(WOLFSSL_SESSION_EXPORT) && !defined(WOLFSSL_NO_TLS12)
    test_wolfSSL_tls_export_run(WOLFSSL_TLSV1_2);
    #ifdef WOLFSSL_TLS13
    test_wolfSSL_tls_export_run(WOLFSSL_TLSV1_3);
    #endif
    res = TEST_RES_CHECK(1);
#endif

    return res;
}

/*----------------------------------------------------------------------------*
 | TLS extensions tests
 *----------------------------------------------------------------------------*/

#ifdef ENABLE_TLS_CALLBACK_TEST
/* Connection test runner - generic */
static void test_wolfSSL_client_server(callback_functions* client_callbacks,
                                       callback_functions* server_callbacks)
{
    tcp_ready ready;
    func_args client_args;
    func_args server_args;
    THREAD_TYPE serverThread;

    XMEMSET(&client_args, 0, sizeof(func_args));
    XMEMSET(&server_args, 0, sizeof(func_args));

    StartTCP();

    client_args.callbacks = client_callbacks;
    server_args.callbacks = server_callbacks;

#ifdef WOLFSSL_TIRTOS
    fdOpenSession(Task_self());
#endif

    /* RUN Server side */
    InitTcpReady(&ready);

#if defined(USE_WINDOWS_API)
    /* use RNG to get random port if using windows */
    ready.port = GetRandomPort();
#endif

    server_args.signal = &ready;
    client_args.signal = &ready;
    start_thread(run_wolfssl_server, &server_args, &serverThread);
    wait_tcp_ready(&server_args);

    /* RUN Client side */
    run_wolfssl_client(&client_args);
    join_thread(serverThread);

    FreeTcpReady(&ready);
#ifdef WOLFSSL_TIRTOS
    fdCloseSession(Task_self());
#endif

    client_callbacks->return_code = client_args.return_code;
    server_callbacks->return_code = server_args.return_code;
}
#endif /* ENABLE_TLS_CALLBACK_TEST */


#ifdef HAVE_SNI
static int test_wolfSSL_UseSNI_params(void)
{
    EXPECT_DECLS;
#if !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX *ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    WOLFSSL     *ssl = wolfSSL_new(ctx);

    ExpectNotNull(ctx);
    ExpectNotNull(ssl);

    /* invalid [ctx|ssl] */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSNI(NULL, 0, "ctx", 3));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseSNI(    NULL, 0, "ssl", 3));
    /* invalid type */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSNI(ctx, (byte)-1, "ctx", 3));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseSNI(    ssl, (byte)-1, "ssl", 3));
    /* invalid data */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSNI(ctx,  0, NULL,  3));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseSNI(    ssl,  0, NULL,  3));
    /* success case */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSNI(ctx,  0, "ctx", 3));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseSNI(    ssl,  0, "ssl", 3));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif /* !NO_WOLFSSL_CLIENT */

    return EXPECT_RESULT();
}

/* BEGIN of connection tests callbacks */
static void use_SNI_at_ctx(WOLFSSL_CTX* ctx)
{
    AssertIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CTX_UseSNI(ctx, WOLFSSL_SNI_HOST_NAME, "www.wolfssl.com", 15));
}

static void use_SNI_at_ssl(WOLFSSL* ssl)
{
    AssertIntEQ(WOLFSSL_SUCCESS,
             wolfSSL_UseSNI(ssl, WOLFSSL_SNI_HOST_NAME, "www.wolfssl.com", 15));
}

static void different_SNI_at_ssl(WOLFSSL* ssl)
{
    AssertIntEQ(WOLFSSL_SUCCESS,
             wolfSSL_UseSNI(ssl, WOLFSSL_SNI_HOST_NAME, "ww2.wolfssl.com", 15));
}

static void use_SNI_WITH_CONTINUE_at_ssl(WOLFSSL* ssl)
{
    use_SNI_at_ssl(ssl);
    wolfSSL_SNI_SetOptions(ssl, WOLFSSL_SNI_HOST_NAME,
                                              WOLFSSL_SNI_CONTINUE_ON_MISMATCH);
}

static void use_SNI_WITH_FAKE_ANSWER_at_ssl(WOLFSSL* ssl)
{
    use_SNI_at_ssl(ssl);
    wolfSSL_SNI_SetOptions(ssl, WOLFSSL_SNI_HOST_NAME,
                                                WOLFSSL_SNI_ANSWER_ON_MISMATCH);
}

static void use_MANDATORY_SNI_at_ctx(WOLFSSL_CTX* ctx)
{
    use_SNI_at_ctx(ctx);
    wolfSSL_CTX_SNI_SetOptions(ctx, WOLFSSL_SNI_HOST_NAME,
                                                  WOLFSSL_SNI_ABORT_ON_ABSENCE);
}

static void use_MANDATORY_SNI_at_ssl(WOLFSSL* ssl)
{
    use_SNI_at_ssl(ssl);
    wolfSSL_SNI_SetOptions(ssl, WOLFSSL_SNI_HOST_NAME,
                                                  WOLFSSL_SNI_ABORT_ON_ABSENCE);
}

static void use_PSEUDO_MANDATORY_SNI_at_ctx(WOLFSSL_CTX* ctx)
{
    use_SNI_at_ctx(ctx);
    wolfSSL_CTX_SNI_SetOptions(ctx, WOLFSSL_SNI_HOST_NAME,
                 WOLFSSL_SNI_ANSWER_ON_MISMATCH | WOLFSSL_SNI_ABORT_ON_ABSENCE);
}

static void verify_UNKNOWN_SNI_on_server(WOLFSSL* ssl)
{
    AssertIntEQ(WC_NO_ERR_TRACE(UNKNOWN_SNI_HOST_NAME_E),
                wolfSSL_get_error(ssl, 0));
}

static void verify_SNI_ABSENT_on_server(WOLFSSL* ssl)
{
    AssertIntEQ(WC_NO_ERR_TRACE(SNI_ABSENT_ERROR), wolfSSL_get_error(ssl, 0));
}

static void verify_SNI_no_matching(WOLFSSL* ssl)
{
    byte type = WOLFSSL_SNI_HOST_NAME;
    void* request = (void*) &type; /* to be overwritten */

    AssertIntEQ(WOLFSSL_SNI_NO_MATCH, wolfSSL_SNI_Status(ssl, type));
    AssertNotNull(request);
    AssertIntEQ(0, wolfSSL_SNI_GetRequest(ssl, type, &request));
    AssertNull(request);
}

static void verify_SNI_real_matching(WOLFSSL* ssl)
{
    byte type = WOLFSSL_SNI_HOST_NAME;
    void* request = NULL;

    AssertIntEQ(WOLFSSL_SNI_REAL_MATCH, wolfSSL_SNI_Status(ssl, type));
    AssertIntEQ(15, wolfSSL_SNI_GetRequest(ssl, type, &request));
    AssertNotNull(request);
    AssertStrEQ("www.wolfssl.com", (char*)request);
}

static void verify_SNI_fake_matching(WOLFSSL* ssl)
{
    byte type = WOLFSSL_SNI_HOST_NAME;
    void* request = NULL;

    AssertIntEQ(WOLFSSL_SNI_FAKE_MATCH, wolfSSL_SNI_Status(ssl, type));
    AssertIntEQ(15, wolfSSL_SNI_GetRequest(ssl, type, &request));
    AssertNotNull(request);
    AssertStrEQ("ww2.wolfssl.com", (char*)request);
}

static void verify_FATAL_ERROR_on_client(WOLFSSL* ssl)
{
    AssertIntEQ(WC_NO_ERR_TRACE(FATAL_ERROR), wolfSSL_get_error(ssl, 0));
}
/* END of connection tests callbacks */

static int test_wolfSSL_UseSNI_connection(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_WOLFSSL_CLIENT) && !defined(NO_WOLFSSL_SERVER)
    callback_functions client_cb;
    callback_functions server_cb;
    size_t i;
#ifdef WOLFSSL_STATIC_MEMORY
    byte cliMem[TEST_TLS_STATIC_MEMSZ];
    byte svrMem[TEST_TLS_STATIC_MEMSZ];
#endif
    struct {
        method_provider client_meth;
        method_provider server_meth;
    #ifdef WOLFSSL_STATIC_MEMORY
        wolfSSL_method_func client_meth_ex;
        wolfSSL_method_func server_meth_ex;
    #endif
    } methods[] = {
#if defined(WOLFSSL_NO_TLS12) && !defined(WOLFSSL_TLS13)
        {wolfSSLv23_client_method, wolfSSLv23_server_method
        #ifdef WOLFSSL_STATIC_MEMORY
            ,wolfSSLv23_client_method_ex, wolfSSLv23_server_method_ex
        #endif
        },
#endif
#ifndef WOLFSSL_NO_TLS12
        {wolfTLSv1_2_client_method, wolfTLSv1_2_server_method
        #ifdef WOLFSSL_STATIC_MEMORY
            ,wolfTLSv1_2_client_method_ex, wolfTLSv1_2_server_method_ex
        #endif
        },
#endif
#ifdef WOLFSSL_TLS13
        {wolfTLSv1_3_client_method, wolfTLSv1_3_server_method
        #ifdef WOLFSSL_STATIC_MEMORY
            ,wolfTLSv1_3_client_method_ex, wolfTLSv1_3_server_method_ex
        #endif
        },
#endif
    };
    size_t methodsSz = sizeof(methods) / sizeof(*methods);

    for (i = 0; i < methodsSz; i++) {
        XMEMSET(&client_cb, 0, sizeof(callback_functions));
        XMEMSET(&server_cb, 0, sizeof(callback_functions));
        client_cb.method = methods[i].client_meth;
        server_cb.method = methods[i].server_meth;
        client_cb.devId = testDevId;
        server_cb.devId = testDevId;
    #ifdef WOLFSSL_STATIC_MEMORY
        client_cb.method_ex = methods[i].client_meth_ex;
        server_cb.method_ex = methods[i].server_meth_ex;
        client_cb.mem = cliMem;
        client_cb.memSz = (word32)sizeof(cliMem);
        server_cb.mem = svrMem;
        server_cb.memSz = (word32)sizeof(svrMem);;
    #endif

        /* success case at ctx */
        fprintf(stderr, "\n\tsuccess case at ctx\n");
        client_cb.ctx_ready = use_SNI_at_ctx; client_cb.ssl_ready = NULL; client_cb.on_result = NULL;
        server_cb.ctx_ready = use_SNI_at_ctx; server_cb.ssl_ready = NULL; server_cb.on_result = verify_SNI_real_matching;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* success case at ssl */
        fprintf(stderr, "\tsuccess case at ssl\n");
        client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_SNI_at_ssl; client_cb.on_result = verify_SNI_real_matching;
        server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_SNI_at_ssl; server_cb.on_result = verify_SNI_real_matching;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* default mismatch behavior */
        fprintf(stderr, "\tdefault mismatch behavior\n");
        client_cb.ctx_ready = NULL; client_cb.ssl_ready = different_SNI_at_ssl; client_cb.on_result = verify_FATAL_ERROR_on_client;
        server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_SNI_at_ssl;       server_cb.on_result = verify_UNKNOWN_SNI_on_server;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* continue on mismatch */
        fprintf(stderr, "\tcontinue on mismatch\n");
        client_cb.ctx_ready = NULL; client_cb.ssl_ready = different_SNI_at_ssl;         client_cb.on_result = NULL;
        server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_SNI_WITH_CONTINUE_at_ssl; server_cb.on_result = verify_SNI_no_matching;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* fake answer on mismatch */
        fprintf(stderr, "\tfake answer on mismatch\n");
        client_cb.ctx_ready = NULL; client_cb.ssl_ready = different_SNI_at_ssl;            client_cb.on_result = NULL;
        server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_SNI_WITH_FAKE_ANSWER_at_ssl; server_cb.on_result = verify_SNI_fake_matching;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* sni abort - success */
        fprintf(stderr, "\tsni abort - success\n");
        client_cb.ctx_ready = use_SNI_at_ctx;           client_cb.ssl_ready = NULL; client_cb.on_result = NULL;
        server_cb.ctx_ready = use_MANDATORY_SNI_at_ctx; server_cb.ssl_ready = NULL; server_cb.on_result = verify_SNI_real_matching;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* sni abort - abort when absent (ctx) */
        fprintf(stderr, "\tsni abort - abort when absent (ctx)\n");
        client_cb.ctx_ready = NULL;                     client_cb.ssl_ready = NULL; client_cb.on_result = verify_FATAL_ERROR_on_client;
        server_cb.ctx_ready = use_MANDATORY_SNI_at_ctx; server_cb.ssl_ready = NULL; server_cb.on_result = verify_SNI_ABSENT_on_server;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* sni abort - abort when absent (ssl) */
        fprintf(stderr, "\tsni abort - abort when absent (ssl)\n");
        client_cb.ctx_ready = NULL; client_cb.ssl_ready = NULL;                     client_cb.on_result = verify_FATAL_ERROR_on_client;
        server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_MANDATORY_SNI_at_ssl; server_cb.on_result = verify_SNI_ABSENT_on_server;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* sni abort - success when overwritten */
        fprintf(stderr, "\tsni abort - success when overwritten\n");
        client_cb.ctx_ready = NULL;                     client_cb.ssl_ready = NULL;           client_cb.on_result = NULL;
        server_cb.ctx_ready = use_MANDATORY_SNI_at_ctx; server_cb.ssl_ready = use_SNI_at_ssl; server_cb.on_result = verify_SNI_no_matching;
        test_wolfSSL_client_server(&client_cb, &server_cb);

        /* sni abort - success when allowing mismatches */
        fprintf(stderr, "\tsni abort - success when allowing mismatches\n");
        client_cb.ctx_ready = NULL;                            client_cb.ssl_ready = different_SNI_at_ssl; client_cb.on_result = NULL;
        server_cb.ctx_ready = use_PSEUDO_MANDATORY_SNI_at_ctx; server_cb.ssl_ready = NULL;                 server_cb.on_result = verify_SNI_fake_matching;
        test_wolfSSL_client_server(&client_cb, &server_cb);
    }

    res = TEST_RES_CHECK(1);
#endif /* !NO_WOLFSSL_CLIENT && !NO_WOLFSSL_SERVER */

    return res;
}

static int test_wolfSSL_SNI_GetFromBuffer(void)
{
    EXPECT_DECLS;
    byte buff[] = { /* www.paypal.com */
        0x00, 0x00, 0x00, 0x00, 0xff, 0x01, 0x00, 0x00, 0x60, 0x03, 0x03, 0x5c,
        0xc4, 0xb3, 0x8c, 0x87, 0xef, 0xa4, 0x09, 0xe0, 0x02, 0xab, 0x86, 0xca,
        0x76, 0xf0, 0x9e, 0x01, 0x65, 0xf6, 0xa6, 0x06, 0x13, 0x1d, 0x0f, 0xa5,
        0x79, 0xb0, 0xd4, 0x77, 0x22, 0xeb, 0x1a, 0x00, 0x00, 0x16, 0x00, 0x6b,
        0x00, 0x67, 0x00, 0x39, 0x00, 0x33, 0x00, 0x3d, 0x00, 0x3c, 0x00, 0x35,
        0x00, 0x2f, 0x00, 0x05, 0x00, 0x04, 0x00, 0x0a, 0x01, 0x00, 0x00, 0x21,
        0x00, 0x00, 0x00, 0x13, 0x00, 0x11, 0x00, 0x00, 0x0e, 0x77, 0x77, 0x77,
        0x2e, 0x70, 0x61, 0x79, 0x70, 0x61, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x00,
        0x0d, 0x00, 0x06, 0x00, 0x04, 0x04, 0x01, 0x02, 0x01
    };

    byte buff2[] = { /* api.textmate.org */
        0x16, 0x03, 0x01, 0x00, 0xc6, 0x01, 0x00, 0x00, 0xc2, 0x03, 0x03, 0x52,
        0x8b, 0x7b, 0xca, 0x69, 0xec, 0x97, 0xd5, 0x08, 0x03, 0x50, 0xfe, 0x3b,
        0x99, 0xc3, 0x20, 0xce, 0xa5, 0xf6, 0x99, 0xa5, 0x71, 0xf9, 0x57, 0x7f,
        0x04, 0x38, 0xf6, 0x11, 0x0b, 0xb8, 0xd3, 0x00, 0x00, 0x5e, 0x00, 0xff,
        0xc0, 0x24, 0xc0, 0x23, 0xc0, 0x0a, 0xc0, 0x09, 0xc0, 0x07, 0xc0, 0x08,
        0xc0, 0x28, 0xc0, 0x27, 0xc0, 0x14, 0xc0, 0x13, 0xc0, 0x11, 0xc0, 0x12,
        0xc0, 0x26, 0xc0, 0x25, 0xc0, 0x2a, 0xc0, 0x29, 0xc0, 0x05, 0xc0, 0x04,
        0xc0, 0x02, 0xc0, 0x03, 0xc0, 0x0f, 0xc0, 0x0e, 0xc0, 0x0c, 0xc0, 0x0d,
        0x00, 0x3d, 0x00, 0x3c, 0x00, 0x2f, 0x00, 0x05, 0x00, 0x04, 0x00, 0x35,
        0x00, 0x0a, 0x00, 0x67, 0x00, 0x6b, 0x00, 0x33, 0x00, 0x39, 0x00, 0x16,
        0x00, 0xaf, 0x00, 0xae, 0x00, 0x8d, 0x00, 0x8c, 0x00, 0x8a, 0x00, 0x8b,
        0x00, 0xb1, 0x00, 0xb0, 0x00, 0x2c, 0x00, 0x3b, 0x01, 0x00, 0x00, 0x3b,
        0x00, 0x00, 0x00, 0x15, 0x00, 0x13, 0x00, 0x00, 0x10, 0x61, 0x70, 0x69,
        0x2e, 0x74, 0x65, 0x78, 0x74, 0x6d, 0x61, 0x74, 0x65, 0x2e, 0x6f, 0x72,
        0x67, 0x00, 0x0a, 0x00, 0x08, 0x00, 0x06, 0x00, 0x17, 0x00, 0x18, 0x00,
        0x19, 0x00, 0x0b, 0x00, 0x02, 0x01, 0x00, 0x00, 0x0d, 0x00, 0x0c, 0x00,
        0x0a, 0x05, 0x01, 0x04, 0x01, 0x02, 0x01, 0x04, 0x03, 0x02, 0x03
    };

    byte buff3[] = { /* no sni extension */
        0x16, 0x03, 0x03, 0x00, 0x4d, 0x01, 0x00, 0x00, 0x49, 0x03, 0x03, 0xea,
        0xa1, 0x9f, 0x60, 0xdd, 0x52, 0x12, 0x13, 0xbd, 0x84, 0x34, 0xd5, 0x1c,
        0x38, 0x25, 0xa8, 0x97, 0xd2, 0xd5, 0xc6, 0x45, 0xaf, 0x1b, 0x08, 0xe4,
        0x1e, 0xbb, 0xdf, 0x9d, 0x39, 0xf0, 0x65, 0x00, 0x00, 0x16, 0x00, 0x6b,
        0x00, 0x67, 0x00, 0x39, 0x00, 0x33, 0x00, 0x3d, 0x00, 0x3c, 0x00, 0x35,
        0x00, 0x2f, 0x00, 0x05, 0x00, 0x04, 0x00, 0x0a, 0x01, 0x00, 0x00, 0x0a,
        0x00, 0x0d, 0x00, 0x06, 0x00, 0x04, 0x04, 0x01, 0x02, 0x01
    };

    byte buff4[] = { /* last extension has zero size */
        0x16, 0x03, 0x01, 0x00, 0xba, 0x01, 0x00, 0x00,
        0xb6, 0x03, 0x03, 0x83, 0xa3, 0xe6, 0xdc, 0x16, 0xa1, 0x43, 0xe9, 0x45,
        0x15, 0xbd, 0x64, 0xa9, 0xb6, 0x07, 0xb4, 0x50, 0xc6, 0xdd, 0xff, 0xc2,
        0xd3, 0x0d, 0x4f, 0x36, 0xb4, 0x41, 0x51, 0x61, 0xc1, 0xa5, 0x9e, 0x00,
        0x00, 0x28, 0xcc, 0x14, 0xcc, 0x13, 0xc0, 0x2b, 0xc0, 0x2f, 0x00, 0x9e,
        0xc0, 0x0a, 0xc0, 0x09, 0xc0, 0x13, 0xc0, 0x14, 0xc0, 0x07, 0xc0, 0x11,
        0x00, 0x33, 0x00, 0x32, 0x00, 0x39, 0x00, 0x9c, 0x00, 0x2f, 0x00, 0x35,
        0x00, 0x0a, 0x00, 0x05, 0x00, 0x04, 0x01, 0x00, 0x00, 0x65, 0xff, 0x01,
        0x00, 0x01, 0x00, 0x00, 0x0a, 0x00, 0x08, 0x00, 0x06, 0x00, 0x17, 0x00,
        0x18, 0x00, 0x19, 0x00, 0x0b, 0x00, 0x02, 0x01, 0x00, 0x00, 0x23, 0x00,
        0x00, 0x33, 0x74, 0x00, 0x00, 0x00, 0x10, 0x00, 0x1b, 0x00, 0x19, 0x06,
        0x73, 0x70, 0x64, 0x79, 0x2f, 0x33, 0x08, 0x73, 0x70, 0x64, 0x79, 0x2f,
        0x33, 0x2e, 0x31, 0x08, 0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31,
        0x75, 0x50, 0x00, 0x00, 0x00, 0x05, 0x00, 0x05, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x0d, 0x00, 0x12, 0x00, 0x10, 0x04, 0x01, 0x05, 0x01, 0x02,
        0x01, 0x04, 0x03, 0x05, 0x03, 0x02, 0x03, 0x04, 0x02, 0x02, 0x02, 0x00,
        0x12, 0x00, 0x00
    };

    byte buff5[] = { /* SSL v2.0 client hello */
        0x00, 0x2b, 0x01, 0x03, 0x01, 0x00, 0x09, 0x00, 0x00,
        /* dummy bytes below, just to pass size check */
        0xb6, 0x03, 0x03, 0x83, 0xa3, 0xe6, 0xdc, 0x16, 0xa1, 0x43, 0xe9, 0x45,
        0x15, 0xbd, 0x64, 0xa9, 0xb6, 0x07, 0xb4, 0x50, 0xc6, 0xdd, 0xff, 0xc2,
        0xd3, 0x0d, 0x4f, 0x36, 0xb4, 0x41, 0x51, 0x61, 0xc1, 0xa5, 0x9e, 0x00,
    };

    byte result[32] = {0};
    word32 length   = 32;

    ExpectIntEQ(0, wolfSSL_SNI_GetFromBuffer(buff4, sizeof(buff4),
                                                           0, result, &length));

    ExpectIntEQ(0, wolfSSL_SNI_GetFromBuffer(buff3, sizeof(buff3),
                                                           0, result, &length));

    ExpectIntEQ(0, wolfSSL_SNI_GetFromBuffer(buff2, sizeof(buff2),
                                                           1, result, &length));

    ExpectIntEQ(WC_NO_ERR_TRACE(BUFFER_ERROR), wolfSSL_SNI_GetFromBuffer(buff, sizeof(buff),
                                                           0, result, &length));
    buff[0] = 0x16;

    ExpectIntEQ(WC_NO_ERR_TRACE(BUFFER_ERROR), wolfSSL_SNI_GetFromBuffer(buff, sizeof(buff),
                                                           0, result, &length));
    buff[1] = 0x03;

    ExpectIntEQ(WC_NO_ERR_TRACE(SNI_UNSUPPORTED), wolfSSL_SNI_GetFromBuffer(buff,
                                           sizeof(buff), 0, result, &length));
    buff[2] = 0x03;

    ExpectIntEQ(WC_NO_ERR_TRACE(INCOMPLETE_DATA), wolfSSL_SNI_GetFromBuffer(buff,
                                           sizeof(buff), 0, result, &length));
    buff[4] = 0x64;

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SNI_GetFromBuffer(buff, sizeof(buff),
                                                           0, result, &length));
    if (EXPECT_SUCCESS())
        result[length] = 0;
    ExpectStrEQ("www.paypal.com", (const char*) result);

    length = 32;

    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_SNI_GetFromBuffer(buff2, sizeof(buff2),
                                                           0, result, &length));
    if (EXPECT_SUCCESS())
        result[length] = 0;
    ExpectStrEQ("api.textmate.org", (const char*) result);

    /* SSL v2.0 tests */
    ExpectIntEQ(WC_NO_ERR_TRACE(SNI_UNSUPPORTED), wolfSSL_SNI_GetFromBuffer(buff5,
                                          sizeof(buff5), 0, result, &length));

    buff5[2] = 0x02;
    ExpectIntEQ(WC_NO_ERR_TRACE(BUFFER_ERROR), wolfSSL_SNI_GetFromBuffer(buff5,
                                          sizeof(buff5), 0, result, &length));

    buff5[2] = 0x01; buff5[6] = 0x08;
    ExpectIntEQ(WC_NO_ERR_TRACE(BUFFER_ERROR), wolfSSL_SNI_GetFromBuffer(buff5,
                                          sizeof(buff5), 0, result, &length));

    buff5[6] = 0x09; buff5[8] = 0x01;
    ExpectIntEQ(WC_NO_ERR_TRACE(BUFFER_ERROR), wolfSSL_SNI_GetFromBuffer(buff5,
                                          sizeof(buff5), 0, result, &length));

    return EXPECT_RESULT();
}

#endif /* HAVE_SNI */

#endif /* HAVE_IO_TESTS_DEPENDENCIES */


#if defined(WOLFSSL_DTLS) && defined(WOLFSSL_SESSION_EXPORT) && \
    defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
/* Dummy peer functions to satisfy the exporter/importer */
static int test_wolfSSL_dtls_export_peers_get_peer(WOLFSSL* ssl, char* ip,
        int* ipSz, unsigned short* port, int* fam)
{
    (void)ssl;
    ip[0] = -1;
    *ipSz = 1;
    *port = 1;
    *fam = 2;
    return 1;
}

static int test_wolfSSL_dtls_export_peers_set_peer(WOLFSSL* ssl, char* ip,
        int ipSz, unsigned short port, int fam)
{
    (void)ssl;
    if (ip[0] != -1 || ipSz != 1 || port != 1 || fam != 2)
        return 0;
    return 1;
}

static int test_wolfSSL_dtls_export_peers_on_handshake(WOLFSSL_CTX **ctx,
        WOLFSSL **ssl)
{
    EXPECT_DECLS;
    unsigned char* sessionBuf = NULL;
    unsigned int sessionSz = 0;
    void* ioWriteCtx = wolfSSL_GetIOWriteCtx(*ssl);
    void* ioReadCtx = wolfSSL_GetIOReadCtx(*ssl);

    wolfSSL_CTX_SetIOGetPeer(*ctx, test_wolfSSL_dtls_export_peers_get_peer);
    wolfSSL_CTX_SetIOSetPeer(*ctx, test_wolfSSL_dtls_export_peers_set_peer);
    ExpectIntGE(wolfSSL_dtls_export(*ssl, NULL, &sessionSz), 0);
    ExpectNotNull(sessionBuf =
        (unsigned char*)XMALLOC(sessionSz, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER));
    ExpectIntGE(wolfSSL_dtls_export(*ssl, sessionBuf, &sessionSz), 0);
    wolfSSL_free(*ssl);
    *ssl = NULL;
    ExpectNotNull(*ssl = wolfSSL_new(*ctx));
    ExpectIntGE(wolfSSL_dtls_import(*ssl, sessionBuf, sessionSz), 0);
    wolfSSL_SetIOWriteCtx(*ssl, ioWriteCtx);
    wolfSSL_SetIOReadCtx(*ssl, ioReadCtx);

    XFREE(sessionBuf, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    return EXPECT_RESULT();
}
#endif

static int test_wolfSSL_dtls_export_peers(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_DTLS) && defined(WOLFSSL_SESSION_EXPORT) && \
    defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES)
    test_ssl_cbf client_cbf;
    test_ssl_cbf server_cbf;
    size_t i, j;
    struct test_params {
        method_provider client_meth;
        method_provider server_meth;
        const char* dtls_version;
    } params[] = {
#ifndef NO_OLD_TLS
        {wolfDTLSv1_client_method, wolfDTLSv1_server_method, "1.0"},
#endif
        {wolfDTLSv1_2_client_method, wolfDTLSv1_2_server_method, "1.2"},
        /* TODO DTLS 1.3 exporting not supported
#ifdef WOLFSSL_DTLS13
        {wolfDTLSv1_3_client_method, wolfDTLSv1_3_server_method, "1.3"},
#endif
         */
    };

    for (i = 0; i < sizeof(params)/sizeof(*params); i++) {
        for (j = 0; j <= 0b11; j++) {
            XMEMSET(&client_cbf, 0, sizeof(client_cbf));
            XMEMSET(&server_cbf, 0, sizeof(server_cbf));

            printf("\n\tTesting DTLS %s connection;", params[i].dtls_version);

            client_cbf.method = params[i].client_meth;
            server_cbf.method = params[i].server_meth;

            if (j & 0b01) {
                client_cbf.on_handshake =
                        test_wolfSSL_dtls_export_peers_on_handshake;
                printf(" With client export;");
            }
            if (j & 0b10) {
                server_cbf.on_handshake =
                        test_wolfSSL_dtls_export_peers_on_handshake;
                printf(" With server export;");
            }

            printf("\n");

            ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cbf,
                &server_cbf, NULL), TEST_SUCCESS);
            if (!EXPECT_SUCCESS())
                break;
        }
    }
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_UseTrustedCA(void)
{
    EXPECT_DECLS;
#if defined(HAVE_TRUSTED_CA) && !defined(NO_CERTS) && !defined(NO_FILESYSTEM) \
    && !defined(NO_RSA)
#if !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL     *ssl = NULL;
    byte        id[20];

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull((ctx = wolfSSL_CTX_new(wolfSSLv23_server_method())));
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile, WOLFSSL_FILETYPE_PEM));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, WOLFSSL_FILETYPE_PEM));
#else
    ExpectNotNull((ctx = wolfSSL_CTX_new(wolfSSLv23_client_method())));
#endif
    ExpectNotNull((ssl = wolfSSL_new(ctx)));
    XMEMSET(id, 0, sizeof(id));

    /* error cases */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(NULL, 0, NULL, 0));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_CERT_SHA1+1, NULL, 0));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_CERT_SHA1, NULL, 0));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_CERT_SHA1, id, 5));
#ifdef NO_SHA
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_KEY_SHA1, id, sizeof(id)));
#endif
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_X509_NAME, id, 0));

    /* success cases */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_PRE_AGREED, NULL, 0));
#ifndef NO_SHA
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_KEY_SHA1, id, sizeof(id)));
#endif
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseTrustedCA(ssl,
                WOLFSSL_TRUSTED_CA_X509_NAME, id, 5));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif /* !NO_TLS && (!NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER) */
#endif /* HAVE_TRUSTED_CA */
    return EXPECT_RESULT();
}

static int test_wolfSSL_UseMaxFragment(void)
{
    EXPECT_DECLS;
#if defined(HAVE_MAX_FRAGMENT) && !defined(NO_CERTS) && \
    !defined(NO_FILESYSTEM) && !defined(NO_RSA)

#if !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
  #ifndef NO_WOLFSSL_SERVER
    WOLFSSL_CTX* ctx = wolfSSL_CTX_new(wolfSSLv23_server_method());
  #else
    WOLFSSL_CTX* ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
  #endif
    WOLFSSL     *ssl = NULL;
  #ifdef OPENSSL_EXTRA
    int (*UseMaxFragment)(SSL *s, unsigned char mode);
    int (*CTX_UseMaxFragment)(SSL_CTX *c, unsigned char mode);
  #else
    int (*UseMaxFragment)(WOLFSSL *s, unsigned char mode);
    int (*CTX_UseMaxFragment)(WOLFSSL_CTX *c, unsigned char mode);
  #endif

  #ifndef NO_WOLFSSL_SERVER
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile, WOLFSSL_FILETYPE_PEM));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, WOLFSSL_FILETYPE_PEM));
  #endif

    ExpectNotNull(ctx);

    ExpectNotNull(ssl = wolfSSL_new(ctx));

  #ifdef OPENSSL_EXTRA
    CTX_UseMaxFragment = SSL_CTX_set_tlsext_max_fragment_length;
    UseMaxFragment = SSL_set_tlsext_max_fragment_length;
  #else
    UseMaxFragment = wolfSSL_UseMaxFragment;
    CTX_UseMaxFragment = wolfSSL_CTX_UseMaxFragment;
  #endif

    /* error cases */
    ExpectIntNE(WOLFSSL_SUCCESS, CTX_UseMaxFragment(NULL, WOLFSSL_MFL_2_9));
    ExpectIntNE(WOLFSSL_SUCCESS, UseMaxFragment(    NULL, WOLFSSL_MFL_2_9));
    ExpectIntNE(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx, WOLFSSL_MFL_MIN-1));
    ExpectIntNE(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx, WOLFSSL_MFL_MAX+1));
    ExpectIntNE(WOLFSSL_SUCCESS, UseMaxFragment(ssl, WOLFSSL_MFL_MIN-1));
    ExpectIntNE(WOLFSSL_SUCCESS, UseMaxFragment(ssl, WOLFSSL_MFL_MAX+1));

    /* success case */
  #ifdef OPENSSL_EXTRA
    ExpectIntEQ(WC_NO_ERR_TRACE(BAD_FUNC_ARG), CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_8));
  #else
    ExpectIntEQ(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_8));
  #endif
    ExpectIntEQ(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_9));
    ExpectIntEQ(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_10));
    ExpectIntEQ(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_11));
    ExpectIntEQ(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_12));
  #ifdef OPENSSL_EXTRA
    ExpectIntEQ(WC_NO_ERR_TRACE(BAD_FUNC_ARG), CTX_UseMaxFragment(ctx, WOLFSSL_MFL_2_13));

    ExpectIntEQ(WC_NO_ERR_TRACE(BAD_FUNC_ARG), UseMaxFragment(    ssl,  WOLFSSL_MFL_2_8));
  #else
    ExpectIntEQ(WOLFSSL_SUCCESS, CTX_UseMaxFragment(ctx,  WOLFSSL_MFL_2_13));

    ExpectIntEQ(WOLFSSL_SUCCESS, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_8));
  #endif
    ExpectIntEQ(WOLFSSL_SUCCESS, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_9));
    ExpectIntEQ(WOLFSSL_SUCCESS, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_10));
    ExpectIntEQ(WOLFSSL_SUCCESS, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_11));
    ExpectIntEQ(WOLFSSL_SUCCESS, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_12));

  #ifdef OPENSSL_EXTRA
    ExpectIntEQ(WC_NO_ERR_TRACE(BAD_FUNC_ARG), UseMaxFragment(    ssl,  WOLFSSL_MFL_2_13));
  #else
    ExpectIntEQ(WOLFSSL_SUCCESS, UseMaxFragment(    ssl,  WOLFSSL_MFL_2_13));
  #endif

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);

#if defined(OPENSSL_EXTRA) && defined(HAVE_MAX_FRAGMENT) && \
    defined(HAVE_MANUAL_MEMIO_TESTS_DEPENDENCIES)
    /* check negotiated max fragment size */
    {
        WOLFSSL *ssl_c = NULL;
        WOLFSSL *ssl_s = NULL;
        struct test_memio_ctx test_ctx;
        WOLFSSL_CTX *ctx_c = NULL;
        WOLFSSL_CTX *ctx_s = NULL;

        XMEMSET(&test_ctx, 0, sizeof(test_ctx));
        ExpectIntEQ(test_memio_setup(&test_ctx, &ctx_c, &ctx_s, &ssl_c, &ssl_s,
            wolfTLSv1_2_client_method, wolfTLSv1_2_server_method), 0);
        ExpectIntEQ(wolfSSL_UseMaxFragment(ssl_c, WOLFSSL_MFL_2_8),
            WOLFSSL_SUCCESS);
        ExpectIntEQ(test_memio_do_handshake(ssl_c, ssl_s, 10, NULL), 0);
#ifndef NO_SESSION_CACHE
        ExpectIntEQ(SSL_SESSION_get_max_fragment_length(
            wolfSSL_get_session(ssl_c)), WOLFSSL_MFL_2_8);
#endif

        wolfSSL_free(ssl_c);
        wolfSSL_free(ssl_s);
        wolfSSL_CTX_free(ctx_c);
        wolfSSL_CTX_free(ctx_s);
    }
#endif
#endif /* !NO_TLS && (!NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER) */
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_UseTruncatedHMAC(void)
{
    EXPECT_DECLS;
#if defined(HAVE_TRUNCATED_HMAC) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    !defined(NO_FILESYSTEM) && !defined(NO_RSA)
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
  #ifndef NO_WOLFSSL_SERVER
    WOLFSSL_CTX* ctx = wolfSSL_CTX_new(wolfSSLv23_server_method());
  #else
    WOLFSSL_CTX* ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
  #endif
    WOLFSSL     *ssl = NULL;

    ExpectNotNull(ctx);

  #ifndef NO_WOLFSSL_SERVER
    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, svrCertFile, WOLFSSL_FILETYPE_PEM));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, WOLFSSL_FILETYPE_PEM));
  #endif

    ExpectNotNull(ssl = wolfSSL_new(ctx));

    /* error cases */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_UseTruncatedHMAC(NULL));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseTruncatedHMAC(NULL));

    /* success case */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_UseTruncatedHMAC(ctx));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseTruncatedHMAC(ssl));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_UseSupportedCurve(void)
{
    EXPECT_DECLS;
#if defined(HAVE_SUPPORTED_CURVES) && !defined(NO_WOLFSSL_CLIENT) && \
    !defined(NO_TLS)
    WOLFSSL_CTX* ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    WOLFSSL     *ssl = wolfSSL_new(ctx);

    ExpectNotNull(ctx);
    ExpectNotNull(ssl);

    /* error cases */
    ExpectIntNE(WOLFSSL_SUCCESS,
                    wolfSSL_CTX_UseSupportedCurve(NULL, WOLFSSL_ECC_SECP256R1));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSupportedCurve(ctx,  0));

    ExpectIntNE(WOLFSSL_SUCCESS,
                        wolfSSL_UseSupportedCurve(NULL, WOLFSSL_ECC_SECP256R1));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseSupportedCurve(ssl,  0));

    /* success case */
    ExpectIntEQ(WOLFSSL_SUCCESS,
                     wolfSSL_CTX_UseSupportedCurve(ctx, WOLFSSL_ECC_SECP256R1));
    ExpectIntEQ(WOLFSSL_SUCCESS,
                         wolfSSL_UseSupportedCurve(ssl, WOLFSSL_ECC_SECP256R1));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif

    return EXPECT_RESULT();
}

#if defined(HAVE_ALPN) && defined(HAVE_IO_TESTS_DEPENDENCIES)

static void verify_ALPN_FATAL_ERROR_on_client(WOLFSSL* ssl)
{
    AssertIntEQ(WC_NO_ERR_TRACE(UNKNOWN_ALPN_PROTOCOL_NAME_E), wolfSSL_get_error(ssl, 0));
}

static void use_ALPN_all(WOLFSSL* ssl)
{
    /* http/1.1,spdy/1,spdy/2,spdy/3 */
    char alpn_list[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31, 0x2c,
                        0x73, 0x70, 0x64, 0x79, 0x2f, 0x31, 0x2c,
                        0x73, 0x70, 0x64, 0x79, 0x2f, 0x32, 0x2c,
                        0x73, 0x70, 0x64, 0x79, 0x2f, 0x33};
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, alpn_list, sizeof(alpn_list),
                                             WOLFSSL_ALPN_FAILED_ON_MISMATCH));
}

static void use_ALPN_all_continue(WOLFSSL* ssl)
{
    /* http/1.1,spdy/1,spdy/2,spdy/3 */
    char alpn_list[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31, 0x2c,
        0x73, 0x70, 0x64, 0x79, 0x2f, 0x31, 0x2c,
        0x73, 0x70, 0x64, 0x79, 0x2f, 0x32, 0x2c,
        0x73, 0x70, 0x64, 0x79, 0x2f, 0x33};
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, alpn_list, sizeof(alpn_list),
                                             WOLFSSL_ALPN_CONTINUE_ON_MISMATCH));
}

static void use_ALPN_one(WOLFSSL* ssl)
{
    /* spdy/2 */
    char proto[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x32};

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, proto, sizeof(proto),
                                             WOLFSSL_ALPN_FAILED_ON_MISMATCH));
}

static void use_ALPN_unknown(WOLFSSL* ssl)
{
    /* http/2.0 */
    char proto[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x32, 0x2e, 0x30};

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, proto, sizeof(proto),
                                             WOLFSSL_ALPN_FAILED_ON_MISMATCH));
}

static void use_ALPN_unknown_continue(WOLFSSL* ssl)
{
    /* http/2.0 */
    char proto[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x32, 0x2e, 0x30};

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, proto, sizeof(proto),
                                             WOLFSSL_ALPN_CONTINUE_ON_MISMATCH));
}

static void verify_ALPN_not_matching_spdy3(WOLFSSL* ssl)
{
    /* spdy/3 */
    char nego_proto[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x33};

    char *proto = NULL;
    word16 protoSz = 0;

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_ALPN_GetProtocol(ssl, &proto, &protoSz));

    /* check value */
    AssertIntNE(1, sizeof(nego_proto) == protoSz);
    if (proto) {
        AssertIntNE(0, XMEMCMP(nego_proto, proto, sizeof(nego_proto)));
    }
}

static void verify_ALPN_not_matching_continue(WOLFSSL* ssl)
{
    char *proto = NULL;
    word16 protoSz = 0;

    AssertIntEQ(WC_NO_ERR_TRACE(WOLFSSL_ALPN_NOT_FOUND),
                wolfSSL_ALPN_GetProtocol(ssl, &proto, &protoSz));

    /* check value */
    AssertIntEQ(1, (0 == protoSz));
    AssertIntEQ(1, (NULL == proto));
}

static void verify_ALPN_matching_http1(WOLFSSL* ssl)
{
    /* http/1.1 */
    char nego_proto[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31};
    char *proto;
    word16 protoSz = 0;

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_ALPN_GetProtocol(ssl, &proto, &protoSz));

    /* check value */
    AssertIntEQ(1, sizeof(nego_proto) == protoSz);
    AssertIntEQ(0, XMEMCMP(nego_proto, proto, protoSz));
}

static void verify_ALPN_matching_spdy2(WOLFSSL* ssl)
{
    /* spdy/2 */
    char nego_proto[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x32};
    char *proto;
    word16 protoSz = 0;

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_ALPN_GetProtocol(ssl, &proto, &protoSz));

    /* check value */
    AssertIntEQ(1, sizeof(nego_proto) == protoSz);
    AssertIntEQ(0, XMEMCMP(nego_proto, proto, protoSz));
}

static void verify_ALPN_client_list(WOLFSSL* ssl)
{
    /* http/1.1,spdy/1,spdy/2,spdy/3 */
    char alpn_list[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31, 0x2c,
                        0x73, 0x70, 0x64, 0x79, 0x2f, 0x31, 0x2c,
                        0x73, 0x70, 0x64, 0x79, 0x2f, 0x32, 0x2c,
                        0x73, 0x70, 0x64, 0x79, 0x2f, 0x33};
    char    *clist = NULL;
    word16  clistSz = 0;

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_ALPN_GetPeerProtocol(ssl, &clist,
                                                          &clistSz));

    /* check value */
    AssertIntEQ(1, sizeof(alpn_list) == clistSz);
    AssertIntEQ(0, XMEMCMP(alpn_list, clist, clistSz));

    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_ALPN_FreePeerProtocol(ssl, &clist));
}

#if defined(OPENSSL_ALL) || defined(WOLFSSL_NGINX) || \
    defined(WOLFSSL_HAPROXY) || defined(HAVE_LIGHTY)

/* ALPN select callback, success with spdy/2 */
static int select_ALPN_spdy2(WOLFSSL *ssl, const unsigned char **out,
    unsigned char *outlen, const unsigned char *in,
    unsigned int inlen, void *arg)
{
    /* spdy/2 */
    const char proto[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x32};

    (void)ssl;
    (void)arg;

    /* adding +1 since LEN byte comes first */
    if (inlen < sizeof(proto) + 1) {
        return SSL_TLSEXT_ERR_ALERT_FATAL;
    }

    if (XMEMCMP(in + 1, proto, sizeof(proto)) == 0) {
        *out = in + 1;
        *outlen = (unsigned char)sizeof(proto);
        return SSL_TLSEXT_ERR_OK;
    }

    return SSL_TLSEXT_ERR_ALERT_FATAL;
}

/* ALPN select callback, force failure */
static int select_ALPN_failure(WOLFSSL *ssl, const unsigned char **out,
    unsigned char *outlen, const unsigned char *in,
    unsigned int inlen, void *arg)
{
    (void)ssl;
    (void)out;
    (void)outlen;
    (void)in;
    (void)inlen;
    (void)arg;

    return SSL_TLSEXT_ERR_ALERT_FATAL;
}

static void use_ALPN_spdy2_callback(WOLFSSL* ssl)
{
    wolfSSL_set_alpn_select_cb(ssl, select_ALPN_spdy2, NULL);
}

static void use_ALPN_failure_callback(WOLFSSL* ssl)
{
    wolfSSL_set_alpn_select_cb(ssl, select_ALPN_failure, NULL);
}
#endif /* OPENSSL_ALL | NGINX | HAPROXY | LIGHTY | QUIC */

static int test_wolfSSL_UseALPN_connection(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_WOLFSSL_CLIENT) && !defined(NO_WOLFSSL_SERVER)
    callback_functions client_cb;
    callback_functions server_cb;

    XMEMSET(&client_cb, 0, sizeof(callback_functions));
    XMEMSET(&server_cb, 0, sizeof(callback_functions));
    client_cb.method = wolfSSLv23_client_method;
    server_cb.method = wolfSSLv23_server_method;
    client_cb.devId = testDevId;
    server_cb.devId = testDevId;

    /* success case same list */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_all; client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_all; server_cb.on_result = verify_ALPN_matching_http1;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* success case only one for server */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_all; client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_one; server_cb.on_result = verify_ALPN_matching_spdy2;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* success case only one for client */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_one; client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_all; server_cb.on_result = verify_ALPN_matching_spdy2;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* success case none for client */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = NULL;         client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_all; server_cb.on_result = NULL;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* success case mismatch behavior but option 'continue' set */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_all_continue;     client_cb.on_result = verify_ALPN_not_matching_continue;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_unknown_continue; server_cb.on_result = NULL;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* success case read protocol send by client */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_all; client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_one; server_cb.on_result = verify_ALPN_client_list;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* mismatch behavior with same list
        * the first and only this one must be taken */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_all; client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_all; server_cb.on_result = verify_ALPN_not_matching_spdy3;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* default mismatch behavior */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = use_ALPN_all;     client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = use_ALPN_unknown; server_cb.on_result = verify_ALPN_FATAL_ERROR_on_client;
    test_wolfSSL_client_server(&client_cb, &server_cb);

#if defined(OPENSSL_ALL) || defined(WOLFSSL_NGINX) || \
    defined(WOLFSSL_HAPROXY) || defined(HAVE_LIGHTY)

    /* WOLFSSL-level ALPN select callback tests */
    /* Callback: success (one protocol, spdy/2) */
    client_cb.ctx_ready = NULL;
    client_cb.ssl_ready = use_ALPN_one;
    client_cb.on_result = verify_ALPN_matching_spdy2;
    server_cb.ctx_ready = NULL;
    server_cb.ssl_ready = use_ALPN_spdy2_callback;
    server_cb.on_result = verify_ALPN_matching_spdy2;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* Callback: failure (one client protocol, spdy/2) */
    client_cb.ctx_ready = NULL;
    client_cb.ssl_ready = use_ALPN_one;
    client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL;
    server_cb.ssl_ready = use_ALPN_failure_callback;
    server_cb.on_result = verify_ALPN_FATAL_ERROR_on_client;
    test_wolfSSL_client_server(&client_cb, &server_cb);

#endif /* OPENSSL_ALL | NGINX | HAPROXY | LIGHTY */

    res = TEST_RES_CHECK(1);
#endif /* !NO_WOLFSSL_CLIENT && !NO_WOLFSSL_SERVER */
    return res;
}

static int test_wolfSSL_UseALPN_params(void)
{
    EXPECT_DECLS;
#ifndef NO_WOLFSSL_CLIENT
    /* "http/1.1" */
    char http1[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31};
    /* "spdy/1" */
    char spdy1[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x31};
    /* "spdy/2" */
    char spdy2[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x32};
    /* "spdy/3" */
    char spdy3[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x33};
    char buff[256];
    word32 idx;

    WOLFSSL_CTX *ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    WOLFSSL     *ssl = wolfSSL_new(ctx);

    ExpectNotNull(ctx);
    ExpectNotNull(ssl);

    /* error cases */
    ExpectIntNE(WOLFSSL_SUCCESS,
                wolfSSL_UseALPN(NULL, http1, sizeof(http1),
                                WOLFSSL_ALPN_FAILED_ON_MISMATCH));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, NULL, 0,
                                              WOLFSSL_ALPN_FAILED_ON_MISMATCH));

    /* success case */
    /* http1 only */
    ExpectIntEQ(WOLFSSL_SUCCESS,
                wolfSSL_UseALPN(ssl, http1, sizeof(http1),
                                WOLFSSL_ALPN_FAILED_ON_MISMATCH));

    /* http1, spdy1 */
    XMEMCPY(buff, http1, sizeof(http1));
    idx = sizeof(http1);
    buff[idx++] = ',';
    XMEMCPY(buff+idx, spdy1, sizeof(spdy1));
    idx += sizeof(spdy1);
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, buff, idx,
                                              WOLFSSL_ALPN_FAILED_ON_MISMATCH));

    /* http1, spdy2, spdy1 */
    XMEMCPY(buff, http1, sizeof(http1));
    idx = sizeof(http1);
    buff[idx++] = ',';
    XMEMCPY(buff+idx, spdy2, sizeof(spdy2));
    idx += sizeof(spdy2);
    buff[idx++] = ',';
    XMEMCPY(buff+idx, spdy1, sizeof(spdy1));
    idx += sizeof(spdy1);
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, buff, idx,
                                              WOLFSSL_ALPN_FAILED_ON_MISMATCH));

    /* spdy3, http1, spdy2, spdy1 */
    XMEMCPY(buff, spdy3, sizeof(spdy3));
    idx = sizeof(spdy3);
    buff[idx++] = ',';
    XMEMCPY(buff+idx, http1, sizeof(http1));
    idx += sizeof(http1);
    buff[idx++] = ',';
    XMEMCPY(buff+idx, spdy2, sizeof(spdy2));
    idx += sizeof(spdy2);
    buff[idx++] = ',';
    XMEMCPY(buff+idx, spdy1, sizeof(spdy1));
    idx += sizeof(spdy1);
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseALPN(ssl, buff, idx,
                                            WOLFSSL_ALPN_CONTINUE_ON_MISMATCH));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}
#endif /* HAVE_ALPN  */

#ifdef HAVE_ALPN_PROTOS_SUPPORT
static void CTX_set_alpn_protos(SSL_CTX *ctx)
{
    unsigned char p[] = {
       8, 'h', 't', 't', 'p', '/', '1', '.', '1',
       6, 's', 'p', 'd', 'y', '/', '2',
       6, 's', 'p', 'd', 'y', '/', '1',
    };

    unsigned char p_len = sizeof(p);
    int ret;

    ret = SSL_CTX_set_alpn_protos(ctx, p, p_len);

#ifdef WOLFSSL_ERROR_CODE_OPENSSL
    AssertIntEQ(ret, 0);
#else
    AssertIntEQ(ret, SSL_SUCCESS);
#endif
}

static void set_alpn_protos(SSL* ssl)
{
    unsigned char p[] = {
       6, 's', 'p', 'd', 'y', '/', '3',
       8, 'h', 't', 't', 'p', '/', '1', '.', '1',
       6, 's', 'p', 'd', 'y', '/', '2',
       6, 's', 'p', 'd', 'y', '/', '1',
    };

    unsigned char p_len = sizeof(p);
    int ret;

    ret = SSL_set_alpn_protos(ssl, p, p_len);

#ifdef WOLFSSL_ERROR_CODE_OPENSSL
    AssertIntEQ(ret, 0);
#else
    AssertIntEQ(ret, SSL_SUCCESS);
#endif

}

static void verify_alpn_matching_spdy3(WOLFSSL* ssl)
{
    /* "spdy/3" */
    char nego_proto[] = {0x73, 0x70, 0x64, 0x79, 0x2f, 0x33};
    const unsigned char *proto;
    unsigned int protoSz = 0;

    SSL_get0_alpn_selected(ssl, &proto, &protoSz);

    /* check value */
    AssertIntEQ(1, sizeof(nego_proto) == protoSz);
    AssertIntEQ(0, XMEMCMP(nego_proto, proto, protoSz));
}

static void verify_alpn_matching_http1(WOLFSSL* ssl)
{
    /* "http/1.1" */
    char nego_proto[] = {0x68, 0x74, 0x74, 0x70, 0x2f, 0x31, 0x2e, 0x31};
    const unsigned char *proto;
    unsigned int protoSz = 0;

    SSL_get0_alpn_selected(ssl, &proto, &protoSz);

    /* check value */
    AssertIntEQ(1, sizeof(nego_proto) == protoSz);
    AssertIntEQ(0, XMEMCMP(nego_proto, proto, protoSz));
}

static int test_wolfSSL_set_alpn_protos(void)
{
    int res = TEST_SKIPPED;
#if !defined(NO_WOLFSSL_CLIENT) && !defined(NO_WOLFSSL_SERVER)
    callback_functions client_cb;
    callback_functions server_cb;

    XMEMSET(&client_cb, 0, sizeof(callback_functions));
    XMEMSET(&server_cb, 0, sizeof(callback_functions));
    client_cb.method = wolfSSLv23_client_method;
    server_cb.method = wolfSSLv23_server_method;
    client_cb.devId = testDevId;
    server_cb.devId = testDevId;

    /* use CTX_alpn_protos */
    client_cb.ctx_ready = CTX_set_alpn_protos; client_cb.ssl_ready = NULL; client_cb.on_result = NULL;
    server_cb.ctx_ready = CTX_set_alpn_protos; server_cb.ssl_ready = NULL; server_cb.on_result = verify_alpn_matching_http1;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    /* use set_alpn_protos */
    client_cb.ctx_ready = NULL; client_cb.ssl_ready = set_alpn_protos; client_cb.on_result = NULL;
    server_cb.ctx_ready = NULL; server_cb.ssl_ready = set_alpn_protos; server_cb.on_result = verify_alpn_matching_spdy3;
    test_wolfSSL_client_server(&client_cb, &server_cb);

    res = TEST_SUCCESS;
#endif /* !NO_WOLFSSL_CLIENT && !NO_WOLFSSL_SERVER */
    return res;
}

#endif /* HAVE_ALPN_PROTOS_SUPPORT */

static int test_wolfSSL_wolfSSL_UseSecureRenegotiation(void)
{
    EXPECT_DECLS;
#if defined(HAVE_SECURE_RENEGOTIATION) && !defined(NO_WOLFSSL_CLIENT) && \
    !defined(NO_TLS)
    WOLFSSL_CTX *ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());
    WOLFSSL     *ssl = wolfSSL_new(ctx);

    ExpectNotNull(ctx);
    ExpectNotNull(ssl);

    /* error cases */
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSecureRenegotiation(NULL));
    ExpectIntNE(WOLFSSL_SUCCESS, wolfSSL_UseSecureRenegotiation(NULL));

    /* success cases */
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSecureRenegotiation(ctx));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseSecureRenegotiation(ssl));

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

/* Test reconnecting with a different ciphersuite after a renegotiation. */
static int test_wolfSSL_SCR_Reconnect(void)
{
    EXPECT_DECLS;
#if defined(HAVE_SECURE_RENEGOTIATION) && \
    defined(BUILD_TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384) && \
    defined(BUILD_TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256) && \
    defined(HAVE_MANUAL_MEMIO_TESTS_DEPENDENCIES)
    struct test_memio_ctx test_ctx;
    WOLFSSL_CTX *ctx_c = NULL, *ctx_s = NULL;
    WOLFSSL *ssl_c = NULL, *ssl_s = NULL;
    byte data;

    XMEMSET(&test_ctx, 0, sizeof(test_ctx));
    test_ctx.c_ciphers = "ECDHE-RSA-AES256-GCM-SHA384";
    test_ctx.s_ciphers =
        "ECDHE-RSA-AES256-GCM-SHA384:ECDHE-RSA-CHACHA20-POLY1305";
    ExpectIntEQ(test_memio_setup(&test_ctx, &ctx_c, &ctx_s, &ssl_c, &ssl_s,
        wolfTLSv1_2_client_method, wolfTLSv1_2_server_method), 0);
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSecureRenegotiation(ctx_c));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_UseSecureRenegotiation(ctx_s));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseSecureRenegotiation(ssl_c));
    ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_UseSecureRenegotiation(ssl_s));
    ExpectIntEQ(test_memio_do_handshake(ssl_c, ssl_s, 10, NULL), 0);
    /* WOLFSSL_FATAL_ERROR since it will block */
    ExpectIntEQ(wolfSSL_Rehandshake(ssl_s), WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
    ExpectIntEQ(wolfSSL_get_error(ssl_s, WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR)),
                WOLFSSL_ERROR_WANT_READ);
    ExpectIntEQ(wolfSSL_read(ssl_c, &data, 1), WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
    ExpectIntEQ(wolfSSL_get_error(ssl_s, WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR)),
                WOLFSSL_ERROR_WANT_READ);
    ExpectIntEQ(test_memio_do_handshake(ssl_c, ssl_s, 10, NULL), 0);
    wolfSSL_free(ssl_c);
    ssl_c = NULL;
    wolfSSL_free(ssl_s);
    ssl_s = NULL;
    wolfSSL_CTX_free(ctx_c);
    ctx_c = NULL;
    test_ctx.c_ciphers = "ECDHE-RSA-CHACHA20-POLY1305";
    ExpectIntEQ(test_memio_setup(&test_ctx, &ctx_c, &ctx_s, &ssl_c, &ssl_s,
        wolfTLSv1_2_client_method, wolfTLSv1_2_server_method), 0);
    ExpectIntEQ(test_memio_do_handshake(ssl_c, ssl_s, 10, NULL), 0);
    wolfSSL_free(ssl_s);
    wolfSSL_free(ssl_c);
    wolfSSL_CTX_free(ctx_s);
    wolfSSL_CTX_free(ctx_c);
#endif
    return EXPECT_RESULT();
}

#if !defined(NO_WOLFSSL_SERVER) && !defined(NO_TLS) && \
    !defined(NO_FILESYSTEM) && (!defined(NO_RSA) || defined(HAVE_ECC))
/* Called when writing. */
static int DummySend(WOLFSSL* ssl, char* buf, int sz, void* ctx)
{
    (void)ssl;
    (void)buf;
    (void)sz;
    (void)ctx;

    /* Force error return from wolfSSL_accept_TLSv13(). */
    return WANT_WRITE;
}
/* Called when reading. */
static int BufferInfoRecv(WOLFSSL* ssl, char* buf, int sz, void* ctx)
{
    WOLFSSL_BUFFER_INFO* msg = (WOLFSSL_BUFFER_INFO*)ctx;
    int len = (int)msg->length;

    (void)ssl;
    (void)sz;

    /* Pass back as much of message as will fit in buffer. */
    if (len > sz)
        len = sz;
    XMEMCPY(buf, msg->buffer, len);
    /* Move over returned data. */
    msg->buffer += len;
    msg->length -= (word32)len;

    /* Amount actually copied. */
    return len;
}
#endif

/* Test the detection of duplicate known TLS extensions.
 * Specifically in a ClientHello.
 */
static int test_tls_ext_duplicate(void)
{
    EXPECT_DECLS;
#if !defined(NO_WOLFSSL_SERVER) && !defined(NO_TLS) && \
    !defined(NO_FILESYSTEM) && (!defined(NO_RSA) || defined(HAVE_ECC))
    const unsigned char clientHelloDupTlsExt[] = {
        0x16, 0x03, 0x03, 0x00, 0x6a, 0x01, 0x00, 0x00,
        0x66, 0x03, 0x03, 0xf4, 0x65, 0xbd, 0x22, 0xfe,
        0x6e, 0xab, 0x66, 0xdd, 0xcf, 0xe9, 0x65, 0x55,
        0xe8, 0xdf, 0xc3, 0x8e, 0x4b, 0x00, 0xbc, 0xf8,
        0x23, 0x57, 0x1b, 0xa0, 0xc8, 0xa9, 0xe2, 0x8c,
        0x91, 0x6e, 0xf9, 0x20, 0xf7, 0x5c, 0xc5, 0x5b,
        0x75, 0x8c, 0x47, 0x0a, 0x0e, 0xc4, 0x1a, 0xda,
        0xef, 0x75, 0xe5, 0x21, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x13, 0x01,
        0x00, 0x9e, 0x01, 0x00,
        /* Extensions - duplicate signature algorithms. */
                                0x00, 0x19, 0x00, 0x0d,
        0x00, 0x04, 0x00, 0x02, 0x04, 0x01, 0x00, 0x0d,
        0x00, 0x04, 0x00, 0x02, 0x04, 0x01,
        /* Supported Versions extension for TLS 1.3. */
                                            0x00, 0x2b,
        0x00, 0x05, 0x04, 0x03, 0x04, 0x03, 0x03
    };
    WOLFSSL_BUFFER_INFO msg;
    const char* testCertFile;
    const char* testKeyFile;
    WOLFSSL_CTX *ctx = NULL;
    WOLFSSL     *ssl = NULL;

#ifndef NO_RSA
    testCertFile = svrCertFile;
    testKeyFile = svrKeyFile;
#elif defined(HAVE_ECC)
    testCertFile = eccCertFile;
    testKeyFile = eccKeyFile;
#endif

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));

    ExpectTrue(wolfSSL_CTX_use_certificate_file(ctx, testCertFile,
        WOLFSSL_FILETYPE_PEM));
    ExpectTrue(wolfSSL_CTX_use_PrivateKey_file(ctx, testKeyFile,
        WOLFSSL_FILETYPE_PEM));

    /* Read from 'msg'. */
    wolfSSL_SetIORecv(ctx, BufferInfoRecv);
    /* No where to send to - dummy sender. */
    wolfSSL_SetIOSend(ctx, DummySend);

    ssl = wolfSSL_new(ctx);
    ExpectNotNull(ssl);

    msg.buffer = (unsigned char*)clientHelloDupTlsExt;
    msg.length = (unsigned int)sizeof(clientHelloDupTlsExt);
    wolfSSL_SetIOReadCtx(ssl, &msg);

    ExpectIntNE(wolfSSL_accept(ssl), WOLFSSL_SUCCESS);
    /* can return duplicate ext error or socket error if the peer closed down
     * while sending alert */
    if (wolfSSL_get_error(ssl, 0) != WC_NO_ERR_TRACE(SOCKET_ERROR_E)) {
        ExpectIntEQ(wolfSSL_get_error(ssl, 0), WC_NO_ERR_TRACE(DUPLICATE_TLS_EXT_E));
    }

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

/*----------------------------------------------------------------------------*
 | X509 Tests
 *----------------------------------------------------------------------------*/
static int test_wolfSSL_X509_NAME_get_entry(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_RSA)
#if defined(OPENSSL_ALL) || \
        (defined(OPENSSL_EXTRA) && \
            (defined(KEEP_PEER_CERT) || defined(SESSION_CERTS)))
    /* use openssl like name to test mapping */
    X509_NAME_ENTRY* ne = NULL;
    X509_NAME* name = NULL;
    X509* x509 = NULL;
#ifndef NO_FILESYSTEM
    ASN1_STRING* asn = NULL;
    char* subCN = NULL;
#endif
    int idx = 0;
    ASN1_OBJECT *object = NULL;
#if defined(WOLFSSL_APACHE_HTTPD) || defined(OPENSSL_ALL) || \
    defined(WOLFSSL_NGINX)
#ifndef NO_BIO
    BIO* bio = NULL;
#endif
#endif

#ifndef NO_FILESYSTEM
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
        WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = X509_get_subject_name(x509));
    ExpectIntGE(idx = X509_NAME_get_index_by_NID(name, NID_commonName, -1), 0);
    ExpectNotNull(ne = X509_NAME_get_entry(name, idx));
    ExpectNull(X509_NAME_ENTRY_get_data(NULL));
    ExpectNotNull(asn = X509_NAME_ENTRY_get_data(ne));
    ExpectNotNull(subCN = (char*)ASN1_STRING_data(asn));
    wolfSSL_FreeX509(x509);
    x509 = NULL;
#endif

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
        WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(name = X509_get_subject_name(x509));
    ExpectIntGE(idx = X509_NAME_get_index_by_NID(name, NID_commonName, -1), 0);

#if defined(WOLFSSL_APACHE_HTTPD) || defined(OPENSSL_ALL) || \
    defined(WOLFSSL_NGINX)
#ifndef NO_BIO
    ExpectNotNull(bio = BIO_new(BIO_s_mem()));
    ExpectIntEQ(X509_NAME_print_ex(bio, name, 4,
                    (XN_FLAG_RFC2253 & ~XN_FLAG_DN_REV)), WOLFSSL_SUCCESS);
    ExpectIntEQ(X509_NAME_print_ex_fp(XBADFILE, name, 4,
                    (XN_FLAG_RFC2253 & ~XN_FLAG_DN_REV)), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_NAME_print_ex_fp(stderr, name, 4,
                    (XN_FLAG_RFC2253 & ~XN_FLAG_DN_REV)), WOLFSSL_SUCCESS);
    BIO_free(bio);
#endif
#endif

    ExpectNotNull(ne = X509_NAME_get_entry(name, idx));
    ExpectNotNull(object = X509_NAME_ENTRY_get_object(ne));
    wolfSSL_FreeX509(x509);
#endif /* OPENSSL_ALL || (OPENSSL_EXTRA && (KEEP_PEER_CERT || SESSION_CERTS) */
#endif /* !NO_CERTS && !NO_RSA */

    return EXPECT_RESULT();
}

/* Testing functions dealing with PKCS12 parsing out X509 certs */
static int test_wolfSSL_PKCS12(void)
{
    EXPECT_DECLS;
    /* .p12 file is encrypted with DES3 */
#ifndef HAVE_FIPS /* Password used in cert "wolfSSL test" is only 12-bytes
                   * (96-bit) FIPS mode requires Minimum of 14-byte (112-bit)
                   * Password Key
                   */
#if defined(OPENSSL_EXTRA) && !defined(NO_DES3) && !defined(NO_FILESYSTEM) && \
    !defined(NO_STDIO_FILESYSTEM) && !defined(NO_TLS) && \
    !defined(NO_ASN) && !defined(NO_PWDBASED) && !defined(NO_RSA) && \
    !defined(NO_SHA) && defined(HAVE_PKCS12) && !defined(NO_BIO)
    byte buf[6000];
    char file[] = "./certs/test-servercert.p12";
    char order[] = "./certs/ecc-rsa-server.p12";
#ifdef WC_RC2
    char rc2p12[] = "./certs/test-servercert-rc2.p12";
#endif
    char pass[] = "a password";
    const char goodPsw[] = "wolfSSL test";
    const char badPsw[] = "bad";
#ifdef HAVE_ECC
    WOLFSSL_X509_NAME *subject = NULL;
    WOLFSSL_X509      *x509 = NULL;
#endif
    XFILE f = XBADFILE;
    int  bytes = 0, ret = 0, goodPswLen = 0, badPswLen = 0;
    WOLFSSL_BIO      *bio = NULL;
    WOLFSSL_EVP_PKEY *pkey = NULL;
    WC_PKCS12        *pkcs12 = NULL;
    WC_PKCS12        *pkcs12_2 = NULL;
    WOLFSSL_X509     *cert = NULL;
    WOLFSSL_X509     *tmp = NULL;
    WOLF_STACK_OF(WOLFSSL_X509) *ca = NULL;
#if (defined(OPENSSL_ALL) || defined(WOLFSSL_ASIO) || defined(WOLFSSL_HAPROXY) \
    || defined(WOLFSSL_NGINX)) && defined(SESSION_CERTS)
    WOLFSSL_CTX      *ctx = NULL;
    WOLFSSL          *ssl = NULL;
    WOLF_STACK_OF(WOLFSSL_X509) *tmp_ca = NULL;
#endif

    ExpectTrue((f = XFOPEN(file, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }

    goodPswLen = (int)XSTRLEN(goodPsw);
    badPswLen = (int)XSTRLEN(badPsw);

    ExpectNotNull(bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));

    ExpectIntEQ(BIO_write(bio, buf, bytes), bytes); /* d2i consumes BIO */
    ExpectNotNull(d2i_PKCS12_bio(bio, &pkcs12));
    ExpectNotNull(pkcs12);
    BIO_free(bio);
    bio = NULL;

    /* check verify MAC directly */
    ExpectIntEQ(ret = PKCS12_verify_mac(pkcs12, goodPsw, goodPswLen), 1);

    /* check verify MAC fail case directly */
    ExpectIntEQ(ret = PKCS12_verify_mac(pkcs12, badPsw, badPswLen), 0);

    /* check verify MAC fail case */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "bad", &pkey, &cert, NULL), 0);
    ExpectNull(pkey);
    ExpectNull(cert);

    /* check parse with no extra certs kept */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, NULL),
        1);
    ExpectNotNull(pkey);
    ExpectNotNull(cert);

    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;
    wolfSSL_X509_free(cert);
    cert = NULL;

    /* check parse with extra certs kept */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, &ca),
        1);
    ExpectNotNull(pkey);
    ExpectNotNull(cert);
    ExpectNotNull(ca);

#if (defined(OPENSSL_ALL) || defined(WOLFSSL_ASIO) || defined(WOLFSSL_HAPROXY) \
    || defined(WOLFSSL_NGINX)) && defined(SESSION_CERTS)

    /* Check that SSL_CTX_set0_chain correctly sets the certChain buffer */
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
#if !defined(NO_WOLFSSL_CLIENT) && defined(SESSION_CERTS)
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#endif
    /* Copy stack structure */
    ExpectNotNull(tmp_ca = X509_chain_up_ref(ca));
    ExpectIntEQ(SSL_CTX_set0_chain(ctx, tmp_ca), 1);
    /* CTX now owns the tmp_ca stack structure */
    tmp_ca = NULL;
    ExpectIntEQ(wolfSSL_CTX_get_extra_chain_certs(ctx, &tmp_ca), 1);
    ExpectNotNull(tmp_ca);
    ExpectIntEQ(sk_X509_num(tmp_ca), sk_X509_num(ca));
    /* Check that the main cert is also set */
    ExpectNotNull(SSL_CTX_get0_certificate(ctx));
    ExpectNotNull(ssl = SSL_new(ctx));
    ExpectNotNull(SSL_get_certificate(ssl));
    SSL_free(ssl);
    SSL_CTX_free(ctx);
    ctx = NULL;
#endif
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
    /* should be 2 other certs on stack */
    ExpectNotNull(tmp = sk_X509_pop(ca));
    X509_free(tmp);
    ExpectNotNull(tmp = sk_X509_pop(ca));
    X509_free(tmp);
    ExpectNull(sk_X509_pop(ca));

    EVP_PKEY_free(pkey);
    pkey = NULL;
    X509_free(cert);
    cert = NULL;
    sk_X509_pop_free(ca, X509_free);
    ca = NULL;

    /* check PKCS12_create */
    ExpectNull(PKCS12_create(pass, NULL, NULL, NULL, NULL, -1, -1, -1, -1,0));
    ExpectIntEQ(PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, &ca),
            SSL_SUCCESS);
    ExpectNotNull((pkcs12_2 = PKCS12_create(pass, NULL, pkey, cert, ca,
                    -1, -1, 100, -1, 0)));
    EVP_PKEY_free(pkey);
    pkey = NULL;
    X509_free(cert);
    cert = NULL;
    sk_X509_pop_free(ca, NULL);
    ca = NULL;

    ExpectIntEQ(PKCS12_parse(pkcs12_2, "a password", &pkey, &cert, &ca),
            SSL_SUCCESS);
    PKCS12_free(pkcs12_2);
    pkcs12_2 = NULL;
    ExpectNotNull((pkcs12_2 = PKCS12_create(pass, NULL, pkey, cert, ca,
             NID_pbe_WithSHA1And3_Key_TripleDES_CBC,
             NID_pbe_WithSHA1And3_Key_TripleDES_CBC,
             2000, 1, 0)));
    EVP_PKEY_free(pkey);
    pkey = NULL;
    X509_free(cert);
    cert = NULL;
    sk_X509_pop_free(ca, NULL);
    ca = NULL;

    /* convert to DER then back and parse */
    ExpectNotNull(bio = BIO_new(BIO_s_mem()));
    ExpectIntEQ(i2d_PKCS12_bio(bio, pkcs12_2), SSL_SUCCESS);
    PKCS12_free(pkcs12_2);
    pkcs12_2 = NULL;

    ExpectNotNull(pkcs12_2 = d2i_PKCS12_bio(bio, NULL));
    BIO_free(bio);
    bio = NULL;
    ExpectIntEQ(PKCS12_parse(pkcs12_2, "a password", &pkey, &cert, &ca),
            SSL_SUCCESS);

    /* should be 2 other certs on stack */
    ExpectNotNull(tmp = sk_X509_pop(ca));
    X509_free(tmp);
    ExpectNotNull(tmp = sk_X509_pop(ca));
    X509_free(tmp);
    ExpectNull(sk_X509_pop(ca));


#ifndef NO_RC4
    PKCS12_free(pkcs12_2);
    pkcs12_2 = NULL;
    ExpectNotNull((pkcs12_2 = PKCS12_create(pass, NULL, pkey, cert, NULL,
             NID_pbe_WithSHA1And128BitRC4,
             NID_pbe_WithSHA1And128BitRC4,
             2000, 1, 0)));
    EVP_PKEY_free(pkey);
    pkey = NULL;
    X509_free(cert);
    cert = NULL;
    sk_X509_pop_free(ca, NULL);
    ca = NULL;

    ExpectIntEQ(PKCS12_parse(pkcs12_2, "a password", &pkey, &cert, &ca),
            SSL_SUCCESS);

#endif /* NO_RC4 */

    EVP_PKEY_free(pkey);
    pkey = NULL;
    X509_free(cert);
    cert = NULL;
    PKCS12_free(pkcs12);
    pkcs12 = NULL;
    PKCS12_free(pkcs12_2);
    pkcs12_2 = NULL;
    sk_X509_pop_free(ca, NULL);
    ca = NULL;

#ifdef HAVE_ECC
    /* test order of parsing */
    ExpectTrue((f = XFOPEN(order, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }

    ExpectNotNull(bio = BIO_new_mem_buf((void*)buf, bytes));
    ExpectNotNull(pkcs12 = d2i_PKCS12_bio(bio, NULL));
    ExpectIntEQ((ret = PKCS12_parse(pkcs12, "", &pkey, &cert, &ca)),
            WOLFSSL_SUCCESS);

    /* check use of pkey after parse */
#if (defined(OPENSSL_ALL) || defined(WOLFSSL_ASIO) || defined(WOLFSSL_HAPROXY) \
    || defined(WOLFSSL_NGINX)) && defined(SESSION_CERTS)
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
#if !defined(NO_WOLFSSL_CLIENT) && defined(SESSION_CERTS)
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#endif
    ExpectIntEQ(SSL_CTX_use_PrivateKey(ctx, pkey), WOLFSSL_SUCCESS);
    SSL_CTX_free(ctx);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif

    ExpectNotNull(pkey);
    ExpectNotNull(cert);
    ExpectNotNull(ca);

    /* compare subject lines of certificates */
    ExpectNotNull(subject = wolfSSL_X509_get_subject_name(cert));
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(eccRsaCertFile,
                SSL_FILETYPE_PEM));
    ExpectIntEQ(wolfSSL_X509_NAME_cmp((const WOLFSSL_X509_NAME*)subject,
            (const WOLFSSL_X509_NAME*)wolfSSL_X509_get_subject_name(x509)), 0);
    X509_free(x509);
    x509 = NULL;

    /* test expected fail case */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(eccCertFile,
                SSL_FILETYPE_PEM));
    ExpectIntNE(wolfSSL_X509_NAME_cmp((const WOLFSSL_X509_NAME*)subject,
            (const WOLFSSL_X509_NAME*)wolfSSL_X509_get_subject_name(x509)), 0);
    X509_free(x509);
    x509 = NULL;
    X509_free(cert);
    cert = NULL;

    /* get subject line from ca stack */
    ExpectNotNull(cert = sk_X509_pop(ca));
    ExpectNotNull(subject = wolfSSL_X509_get_subject_name(cert));

    /* compare subject from certificate in ca to expected */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(eccCertFile,
                SSL_FILETYPE_PEM));
    ExpectIntEQ(wolfSSL_X509_NAME_cmp((const WOLFSSL_X509_NAME*)subject,
            (const WOLFSSL_X509_NAME*)wolfSSL_X509_get_subject_name(x509)), 0);

    /* modify case and compare subject from certificate in ca to expected.
     * The first bit of the name is:
     * /C=US/ST=Washington
     * So we'll change subject->name[1] to 'c' (lower case) */
    if (subject != NULL) {
        subject->name[1] = 'c';
        ExpectIntEQ(wolfSSL_X509_NAME_cmp((const WOLFSSL_X509_NAME*)subject,
            (const WOLFSSL_X509_NAME*)wolfSSL_X509_get_subject_name(x509)), 0);
    }

    EVP_PKEY_free(pkey);
    pkey = NULL;
    X509_free(x509);
    x509 = NULL;
    X509_free(cert);
    cert = NULL;
    BIO_free(bio);
    bio = NULL;
    PKCS12_free(pkcs12);
    pkcs12 = NULL;
    sk_X509_pop_free(ca, NULL); /* TEST d2i_PKCS12_fp */
    ca = NULL;

    /* test order of parsing */
    ExpectTrue((f = XFOPEN(file, "rb")) != XBADFILE);
    ExpectNotNull(pkcs12 = d2i_PKCS12_fp(f, NULL));
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }

    /* check verify MAC fail case */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "bad", &pkey, &cert, NULL), 0);
    ExpectNull(pkey);
    ExpectNull(cert);

    /* check parse with no extra certs kept */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, NULL),
        1);
    ExpectNotNull(pkey);
    ExpectNotNull(cert);

    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;
    wolfSSL_X509_free(cert);
    cert = NULL;

    /* check parse with extra certs kept */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, &ca),
        1);
    ExpectNotNull(pkey);
    ExpectNotNull(cert);
    ExpectNotNull(ca);

    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;
    wolfSSL_X509_free(cert);
    cert = NULL;
    sk_X509_pop_free(ca, NULL);
    ca = NULL;

    PKCS12_free(pkcs12);
    pkcs12 = NULL;
#endif /* HAVE_ECC */

#ifdef WC_RC2
    /* test PKCS#12 with RC2 encryption */
    ExpectTrue((f = XFOPEN(rc2p12, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }

    ExpectNotNull(bio = BIO_new_mem_buf((void*)buf, bytes));
    ExpectNotNull(pkcs12 = d2i_PKCS12_bio(bio, NULL));

    /* check verify MAC fail case */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "bad", &pkey, &cert, NULL), 0);
    ExpectNull(pkey);
    ExpectNull(cert);

    /* check parse with not extra certs kept */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, NULL),
        WOLFSSL_SUCCESS);
    ExpectNotNull(pkey);
    ExpectNotNull(cert);

    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;
    wolfSSL_X509_free(cert);
    cert = NULL;

    /* check parse with extra certs kept */
    ExpectIntEQ(ret = PKCS12_parse(pkcs12, "wolfSSL test", &pkey, &cert, &ca),
        WOLFSSL_SUCCESS);
    ExpectNotNull(pkey);
    ExpectNotNull(cert);
    ExpectNotNull(ca);

    wolfSSL_EVP_PKEY_free(pkey);
    wolfSSL_X509_free(cert);
    sk_X509_pop_free(ca, NULL);

    BIO_free(bio);
    bio = NULL;
    PKCS12_free(pkcs12);
    pkcs12 = NULL;
#endif /* WC_RC2 */

    /* Test i2d_PKCS12_bio */
    ExpectTrue((f = XFOPEN(file, "rb")) != XBADFILE);
    ExpectNotNull(pkcs12 = d2i_PKCS12_fp(f, NULL));
    if (f != XBADFILE)
        XFCLOSE(f);

    ExpectNotNull(bio = BIO_new(BIO_s_mem()));

    ExpectIntEQ(ret = i2d_PKCS12_bio(bio, pkcs12), 1);

    ExpectIntEQ(ret = i2d_PKCS12_bio(NULL, pkcs12), 0);

    ExpectIntEQ(ret = i2d_PKCS12_bio(bio, NULL), 0);

    PKCS12_free(pkcs12);
    BIO_free(bio);

    (void)order;
#endif /* OPENSSL_EXTRA */
#endif /* HAVE_FIPS */
    return EXPECT_RESULT();
}


#if !defined(NO_FILESYSTEM) && !defined(NO_ASN) && defined(HAVE_PKCS8) && \
    defined(WOLFSSL_ENCRYPTED_KEYS) && !defined(NO_DES3) && !defined(NO_PWDBASED) && \
    (!defined(NO_RSA) || defined(HAVE_ECC)) && !defined(NO_MD5)
    #define TEST_PKCS8_ENC
#endif

#if !defined(NO_FILESYSTEM) && !defined(NO_ASN) && defined(HAVE_PKCS8) && \
    defined(HAVE_ECC) && defined(WOLFSSL_ENCRYPTED_KEYS) && !defined(NO_TLS)

/* used to keep track if FailTestCallback was called */
static int failTestCallbackCalled = 0;

static WC_INLINE int FailTestCallBack(char* passwd, int sz, int rw, void* userdata)
{
    (void)passwd;
    (void)sz;
    (void)rw;
    (void)userdata;

    /* mark called, test_wolfSSL_no_password_cb() will check and fail if set */
    failTestCallbackCalled = 1;

    return -1;
}
#endif

static int test_wolfSSL_no_password_cb(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_ASN) && defined(HAVE_PKCS8) && \
    defined(HAVE_ECC) && defined(WOLFSSL_ENCRYPTED_KEYS) && !defined(NO_TLS)
    WOLFSSL_CTX* ctx = NULL;
    byte buff[FOURK_BUF];
    const char eccPkcs8PrivKeyDerFile[] = "./certs/ecc-privkeyPkcs8.der";
    const char eccPkcs8PrivKeyPemFile[] = "./certs/ecc-privkeyPkcs8.pem";
    XFILE f = XBADFILE;
    int bytes = 0;

#ifndef NO_WOLFSSL_CLIENT
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLS_client_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLS_server_method()));
#endif
    wolfSSL_CTX_set_default_passwd_cb(ctx, FailTestCallBack);

    ExpectTrue((f = XFOPEN(eccPkcs8PrivKeyDerFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    ExpectIntLE(bytes, sizeof(buff));
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    ExpectTrue((f = XFOPEN(eccPkcs8PrivKeyPemFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);
    ExpectIntLE(bytes, sizeof(buff));
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);

    /* Password callback should not be called by default */
    ExpectIntEQ(failTestCallbackCalled, 0);
#endif
    return EXPECT_RESULT();
}

#if defined(TEST_PKCS8_ENC) && !defined(NO_TLS)
/* for PKCS8 test case */
static int PKCS8TestCallBack(char* passwd, int sz, int rw, void* userdata)
{
    int flag = 0;

    (void)rw;
    if (userdata != NULL) {
        flag = *((int*)userdata); /* user set data */
    }

    switch (flag) {
        case 1: /* flag set for specific WOLFSSL_CTX structure, note userdata
                 * can be anything the user wishes to be passed to the callback
                 * associated with the WOLFSSL_CTX */
            XSTRNCPY(passwd, "yassl123", sz);
            return 8;

        default:
            return BAD_FUNC_ARG;
    }
}
#endif /* TEST_PKCS8_ENC && !NO_TLS */

/* Testing functions dealing with PKCS8 */
static int test_wolfSSL_PKCS8(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && !defined(NO_ASN) && defined(HAVE_PKCS8) && \
    !defined(WOLFCRYPT_ONLY) && !defined(NO_TLS) && \
    (!defined(WOLFSSL_NO_TLS12) || defined(WOLFSSL_TLS13))
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
    byte buff[FOURK_BUF];
    byte der[FOURK_BUF];
    #ifndef NO_RSA
        const char serverKeyPkcs8PemFile[] = "./certs/server-keyPkcs8.pem";
        const char serverKeyPkcs8DerFile[] = "./certs/server-keyPkcs8.der";
    #endif
    const char eccPkcs8PrivKeyPemFile[] = "./certs/ecc-privkeyPkcs8.pem";
    #ifdef HAVE_ECC
        const char eccPkcs8PrivKeyDerFile[] = "./certs/ecc-privkeyPkcs8.der";
    #endif
    XFILE f = XBADFILE;
    int bytes = 0;
    WOLFSSL_CTX* ctx = NULL;
#if defined(HAVE_ECC) && !defined(NO_CODING) && !defined(WOLFSSL_NO_PEM)
    int ret;
    ecc_key key;
    word32 x = 0;
#endif
#ifdef TEST_PKCS8_ENC
    #if !defined(NO_RSA) && !defined(NO_SHA)
        const char serverKeyPkcs8EncPemFile[] = "./certs/server-keyPkcs8Enc.pem";
        const char serverKeyPkcs8EncDerFile[] = "./certs/server-keyPkcs8Enc.der";
    #endif
    #if defined(HAVE_ECC) && !defined(NO_SHA)
        const char eccPkcs8EncPrivKeyPemFile[] = "./certs/ecc-keyPkcs8Enc.pem";
        const char eccPkcs8EncPrivKeyDerFile[] = "./certs/ecc-keyPkcs8Enc.der";
    #endif
    int flag;
#endif

    (void)der;

#ifndef NO_WOLFSSL_CLIENT
    #ifndef WOLFSSL_NO_TLS12
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_2_client_method()));
    #else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_3_client_method()));
    #endif
#else
    #ifndef WOLFSSL_NO_TLS12
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_2_server_method()));
    #else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_3_server_method()));
    #endif
#endif

#ifdef TEST_PKCS8_ENC
    wolfSSL_CTX_set_default_passwd_cb(ctx, PKCS8TestCallBack);
    wolfSSL_CTX_set_default_passwd_cb_userdata(ctx, (void*)&flag);
    flag = 1; /* used by password callback as return code */

    #if !defined(NO_RSA) && !defined(NO_SHA)
    /* test loading PEM PKCS8 encrypted file */
    ExpectTrue((f = XFOPEN(serverKeyPkcs8EncPemFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    /* this next case should fail because of password callback return code */
    flag = 0; /* used by password callback as return code */
    ExpectIntNE(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    /* decrypt PKCS8 PEM to key in DER format with not using WOLFSSL_CTX */
    ExpectIntGT(wc_KeyPemToDer(buff, bytes, der, (word32)sizeof(der),
        "yassl123"), 0);

    /* test that error value is returned with a bad password */
    ExpectIntLT(wc_KeyPemToDer(buff, bytes, der, (word32)sizeof(der),
        "bad"), 0);

    /* test loading PEM PKCS8 encrypted file */
    ExpectTrue((f = XFOPEN(serverKeyPkcs8EncDerFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    flag = 1; /* used by password callback as return code */
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* this next case should fail because of password callback return code */
    flag = 0; /* used by password callback as return code */
    ExpectIntNE(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    #endif /* !NO_RSA && !NO_SHA */

    #if defined(HAVE_ECC) && !defined(NO_SHA)
    /* test loading PEM PKCS8 encrypted ECC Key file */
    ExpectTrue((f = XFOPEN(eccPkcs8EncPrivKeyPemFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    flag = 1; /* used by password callback as return code */
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    /* this next case should fail because of password callback return code */
    flag = 0; /* used by password callback as return code */
    ExpectIntNE(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

    /* decrypt PKCS8 PEM to key in DER format with not using WOLFSSL_CTX */
    ExpectIntGT(wc_KeyPemToDer(buff, bytes, der, (word32)sizeof(der),
        "yassl123"), 0);

    /* test that error value is returned with a bad password */
    ExpectIntLT(wc_KeyPemToDer(buff, bytes, der, (word32)sizeof(der),
        "bad"), 0);

    /* test loading DER PKCS8 encrypted ECC Key file */
    ExpectTrue((f = XFOPEN(eccPkcs8EncPrivKeyDerFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    flag = 1; /* used by password callback as return code */
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* this next case should fail because of password callback return code */
    flag = 0; /* used by password callback as return code */
    ExpectIntNE(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* leave flag as "okay" */
    flag = 1;
    #endif /* HAVE_ECC && !NO_SHA */
#endif /* TEST_PKCS8_ENC */


#ifndef NO_RSA
    /* test loading ASN.1 (DER) PKCS8 private key file (not encrypted) */
    ExpectTrue((f = XFOPEN(serverKeyPkcs8DerFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    /* test loading PEM PKCS8 private key file (not encrypted) */
    ExpectTrue((f = XFOPEN(serverKeyPkcs8PemFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);
#endif /* !NO_RSA */

    /* Test PKCS8 PEM ECC key no crypt */
    ExpectTrue((f = XFOPEN(eccPkcs8PrivKeyPemFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
#ifdef HAVE_ECC
    /* Test PKCS8 PEM ECC key no crypt */
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_PEM), WOLFSSL_SUCCESS);

#if !defined(NO_CODING) && !defined(WOLFSSL_NO_PEM)
    /* decrypt PKCS8 PEM to key in DER format */
    ExpectIntGT((bytes = wc_KeyPemToDer(buff, bytes, der,
        (word32)sizeof(der), NULL)), 0);
    ret = wc_ecc_init(&key);
    if (ret == 0) {
        ret = wc_EccPrivateKeyDecode(der, &x, &key, (word32)bytes);
        wc_ecc_free(&key);
    }
    ExpectIntEQ(ret, 0);
#endif

    /* Test PKCS8 DER ECC key no crypt */
    ExpectTrue((f = XFOPEN(eccPkcs8PrivKeyDerFile, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);

    /* Test using a PKCS8 ECC PEM */
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, buff, bytes,
                WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
#else
    /* if HAVE_ECC is not defined then BEGIN EC PRIVATE KEY is not found */
    ExpectIntEQ((bytes = wc_KeyPemToDer(buff, bytes, der,
        (word32)sizeof(der), NULL)), WC_NO_ERR_TRACE(ASN_NO_PEM_HEADER));
#endif /* HAVE_ECC */

    wolfSSL_CTX_free(ctx);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif /* !NO_FILESYSTEM && !NO_ASN && HAVE_PKCS8 */
    return EXPECT_RESULT();
}

static int test_wolfSSL_PKCS8_ED25519(void)
{
    EXPECT_DECLS;
#if !defined(NO_ASN) && defined(HAVE_PKCS8) && defined(HAVE_AES_CBC) && \
    defined(WOLFSSL_ENCRYPTED_KEYS) && defined(HAVE_ED25519) && \
    defined(HAVE_ED25519_KEY_IMPORT)
    const byte encPrivKey[] = \
    "-----BEGIN ENCRYPTED PRIVATE KEY-----\n"
    "MIGbMFcGCSqGSIb3DQEFDTBKMCkGCSqGSIb3DQEFDDAcBAheCGLmWGh7+AICCAAw\n"
    "DAYIKoZIhvcNAgkFADAdBglghkgBZQMEASoEEC4L5P6GappsTyhOOoQfvh8EQJMX\n"
    "OAdlsYKCOcFo4djg6AI1lRdeBRwVFWkha7gBdoCJOzS8wDvTbYcJMPvANu5ft3nl\n"
    "2L9W4v7swXkV+X+a1ww=\n"
    "-----END ENCRYPTED PRIVATE KEY-----\n";
    const char password[] = "abcdefghijklmnopqrstuvwxyz";
    byte der[FOURK_BUF];
#if !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX* ctx = NULL;
#endif
    int bytes;

    XMEMSET(der, 0, sizeof(der));
    ExpectIntGT((bytes = wc_KeyPemToDer(encPrivKey, sizeof(encPrivKey), der,
        (word32)sizeof(der), password)), 0);
#if !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#endif
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, der, bytes,
        WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);
#endif /* !NO_TLS && (!NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER) */
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_PKCS8_ED448(void)
{
    EXPECT_DECLS;
#if !defined(NO_ASN) && defined(HAVE_PKCS8) && defined(HAVE_AES_CBC) && \
    defined(WOLFSSL_ENCRYPTED_KEYS) && defined(HAVE_ED448) && \
    defined(HAVE_ED448_KEY_IMPORT)
    const byte encPrivKey[] = \
    "-----BEGIN ENCRYPTED PRIVATE KEY-----\n"
    "MIGrMFcGCSqGSIb3DQEFDTBKMCkGCSqGSIb3DQEFDDAcBAjSbZKnG4EPggICCAAw\n"
    "DAYIKoZIhvcNAgkFADAdBglghkgBZQMEASoEEFvCFWBBHBlJBsYleBJlJWcEUNC7\n"
    "Tf5pZviT5Btar4D/MNg6BsQHSDf5KW4ix871EsgDY2Zz+euaoWspiMntz7gU+PQu\n"
    "T/JJcbD2Ly8BbE3l5WHMifAQqNLxJBfXrHkfYtAo\n"
    "-----END ENCRYPTED PRIVATE KEY-----\n";
    const char password[] = "abcdefghijklmnopqrstuvwxyz";
    byte der[FOURK_BUF];
#if !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    WOLFSSL_CTX* ctx = NULL;
#endif
    int bytes;

    XMEMSET(der, 0, sizeof(der));
    ExpectIntGT((bytes = wc_KeyPemToDer(encPrivKey, sizeof(encPrivKey), der,
        (word32)sizeof(der), password)), 0);
#if !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_client_method()));
#endif
    ExpectIntEQ(wolfSSL_CTX_use_PrivateKey_buffer(ctx, der, bytes,
        WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);

    wolfSSL_CTX_free(ctx);
#endif /* !NO_TLS && (!NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER) */
#endif
    return EXPECT_RESULT();
}

/* Testing functions dealing with PKCS5 */
static int test_wolfSSL_PKCS5(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_SHA) && !defined(NO_PWDBASED)
#ifdef HAVE_FIPS /* Password minimum length is 14 (112-bit) in FIPS MODE */
    const char* passwd = "myfipsPa$$W0rd";
#else
    const char *passwd = "pass1234";
#endif
    const unsigned char *salt = (unsigned char *)"salt1234";
    unsigned char *out = (unsigned char *)XMALLOC(WC_SHA_DIGEST_SIZE, NULL,
                                                  DYNAMIC_TYPE_TMP_BUFFER);
    int ret = 0;

    ExpectNotNull(out);
    ExpectIntEQ(ret = PKCS5_PBKDF2_HMAC_SHA1(passwd,(int)XSTRLEN(passwd), salt,
        (int)XSTRLEN((const char *) salt), 10, WC_SHA_DIGEST_SIZE,out),
        WOLFSSL_SUCCESS);

#ifdef WOLFSSL_SHA512
    ExpectIntEQ(ret = PKCS5_PBKDF2_HMAC(passwd,(int)XSTRLEN(passwd), salt,
        (int)XSTRLEN((const char *) salt), 10, wolfSSL_EVP_sha512(),
        WC_SHA_DIGEST_SIZE, out), SSL_SUCCESS);
#endif

    XFREE(out, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_SHA) */
    return EXPECT_RESULT();
}

/* test parsing URI from certificate */
static int test_wolfSSL_URI(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_RSA) && !defined(NO_FILESYSTEM) \
    && (defined(KEEP_PEER_CERT) || defined(SESSION_CERTS) || \
    defined(OPENSSL_EXTRA))
    WOLFSSL_X509* x509 = NULL;
    const char uri[] = "./certs/client-uri-cert.pem";
    const char urn[] = "./certs/client-absolute-urn.pem";
    const char badUri[] = "./certs/client-relative-uri.pem";

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(uri,
        WOLFSSL_FILETYPE_PEM));
    wolfSSL_FreeX509(x509);
    x509 = NULL;

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(urn,
        WOLFSSL_FILETYPE_PEM));
    wolfSSL_FreeX509(x509);
    x509 = NULL;

#if !defined(IGNORE_NAME_CONSTRAINTS) && !defined(WOLFSSL_NO_ASN_STRICT) \
    && !defined(WOLFSSL_FPKI)
    ExpectNull(x509 = wolfSSL_X509_load_certificate_file(badUri,
        WOLFSSL_FILETYPE_PEM));
#else
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(badUri,
        WOLFSSL_FILETYPE_PEM));
#endif
    wolfSSL_FreeX509(x509);
#endif
    return EXPECT_RESULT();
}


static int test_wolfSSL_TBS(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_RSA) && !defined(NO_FILESYSTEM) \
    && defined(OPENSSL_EXTRA)
    WOLFSSL_X509* x509 = NULL;
    const unsigned char* tbs;
    int tbsSz;

    ExpectNotNull(x509 = wolfSSL_X509_new());
    ExpectNull(tbs = wolfSSL_X509_get_tbs(x509, &tbsSz));
    wolfSSL_X509_free(x509);
    x509 = NULL;

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(caCertFile,
        WOLFSSL_FILETYPE_PEM));

    ExpectNull(tbs = wolfSSL_X509_get_tbs(NULL, &tbsSz));
    ExpectNull(tbs = wolfSSL_X509_get_tbs(x509, NULL));
    ExpectNotNull(tbs = wolfSSL_X509_get_tbs(x509, &tbsSz));
    ExpectIntEQ(tbsSz, 1003);

    wolfSSL_FreeX509(x509);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_verify(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_RSA) && !defined(NO_FILESYSTEM) && \
    defined(OPENSSL_EXTRA)
    WOLFSSL_X509* ca = NULL;
    WOLFSSL_X509* serv = NULL;
    WOLFSSL_EVP_PKEY* pkey = NULL;
    unsigned char buf[2048];
    const unsigned char* pt = NULL;
    int bufSz = 0;

    ExpectNotNull(ca = wolfSSL_X509_load_certificate_file(caCertFile,
        WOLFSSL_FILETYPE_PEM));

    ExpectIntNE(wolfSSL_X509_get_pubkey_buffer(NULL, buf, NULL),
        WOLFSSL_SUCCESS);
    ExpectIntNE(wolfSSL_X509_get_pubkey_buffer(NULL, buf, &bufSz),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_get_pubkey_buffer(ca, NULL, &bufSz),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(bufSz, 294);

    bufSz--;
    ExpectIntNE(wolfSSL_X509_get_pubkey_buffer(ca, buf, &bufSz),
        WOLFSSL_SUCCESS);
    bufSz = 2048;
    ExpectIntEQ(wolfSSL_X509_get_pubkey_buffer(ca, buf, &bufSz),
        WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_get_pubkey_type(NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_X509_get_pubkey_type(ca), RSAk);


    ExpectNotNull(serv = wolfSSL_X509_load_certificate_file(svrCertFile,
        WOLFSSL_FILETYPE_PEM));

    /* success case */
    pt = buf;
    ExpectNotNull(pkey = wolfSSL_d2i_PUBKEY(NULL, &pt, bufSz));

    ExpectIntEQ(i2d_PUBKEY(pkey, NULL), bufSz);

    ExpectIntEQ(wolfSSL_X509_verify(serv, pkey), WOLFSSL_SUCCESS);
    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;

    /* fail case */
    bufSz = 2048;
    ExpectIntEQ(wolfSSL_X509_get_pubkey_buffer(serv, buf, &bufSz),
        WOLFSSL_SUCCESS);
    pt = buf;
    ExpectNotNull(pkey = wolfSSL_d2i_PUBKEY(NULL, &pt, bufSz));
    ExpectIntEQ(wolfSSL_X509_verify(serv, pkey), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    ExpectIntEQ(wolfSSL_X509_verify(NULL, pkey), WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
    ExpectIntEQ(wolfSSL_X509_verify(serv, NULL), WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));

#ifndef NO_WOLFSSL_STUB
    ExpectNull(wolfSSL_X509_get0_pubkey_bitstr(NULL));
    ExpectNull(wolfSSL_X509_get0_pubkey_bitstr(serv));
#endif

    wolfSSL_EVP_PKEY_free(pkey);

    wolfSSL_FreeX509(ca);
    wolfSSL_FreeX509(serv);
#endif
    return EXPECT_RESULT();
}

#if defined(WOLFSSL_ACERT) && !defined(NO_CERTS) && !defined(NO_RSA) && \
    defined(WC_RSA_PSS) && !defined(NO_FILESYSTEM) && defined(OPENSSL_EXTRA)
/* Given acert file and its pubkey file, read them and then
 * attempt to verify signed acert.
 *
 * If expect_pass is true, then verification should pass.
 * If expect_pass is false, then verification should fail.
 * */
static int do_acert_verify_test(const char * acert_file,
                                const char * pkey_file,
                                size_t       expect_pass)
{
    X509_ACERT * x509 = NULL;
    EVP_PKEY *   pkey = NULL;
    BIO *        bp = NULL;
    int          verify_rc = 0;

    /* First read the attribute certificate. */
    bp = BIO_new_file(acert_file, "r");
    if (bp == NULL) {
        return -1;
    }

    x509 = PEM_read_bio_X509_ACERT(bp, NULL, NULL, NULL);
    BIO_free(bp);
    bp = NULL;

    if (x509 == NULL) {
        return -1;
    }

    /* Next read the associated pub key. */
    bp = BIO_new_file(pkey_file, "r");

    if (bp == NULL) {
        X509_ACERT_free(x509);
        x509 = NULL;
        return -1;
    }

    pkey = PEM_read_bio_PUBKEY(bp, &pkey, NULL, NULL);
    BIO_free(bp);
    bp = NULL;

    if (pkey == NULL) {
        X509_ACERT_free(x509);
        x509 = NULL;
        return -1;
    }

    /* Finally, do verification. */
    verify_rc = X509_ACERT_verify(x509, pkey);

    X509_ACERT_free(x509);
    x509 = NULL;

    EVP_PKEY_free(pkey);
    pkey = NULL;

    if (expect_pass && verify_rc != 1) {
        return -1;
    }

    if (!expect_pass && verify_rc == 1) {
        return -1;
    }

    return 0;
}
#endif

static int test_wolfSSL_X509_ACERT_verify(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_ACERT) && !defined(NO_CERTS) && !defined(NO_RSA) && \
    defined(WC_RSA_PSS) && !defined(NO_FILESYSTEM) && defined(OPENSSL_EXTRA)
    /* Walk over list of signed ACERTs and their pubkeys.
     * All should load and pass verification. */
    const char * acerts[4] = {"certs/acert/acert.pem",
                              "certs/acert/acert_ietf.pem",
                              "certs/acert/rsa_pss/acert.pem",
                              "certs/acert/rsa_pss/acert_ietf.pem"};
    const char * pkeys[4] =  {"certs/acert/acert_pubkey.pem",
                              "certs/acert/acert_ietf_pubkey.pem",
                              "certs/acert/rsa_pss/acert_pubkey.pem",
                              "certs/acert/rsa_pss/acert_ietf_pubkey.pem"};
    int    rc = 0;
    size_t i = 0;
    size_t j = 0;

    for (i = 0; i < 4; ++i) {
        for (j = i; j < 4; ++j) {
            rc = do_acert_verify_test(acerts[i], pkeys[j], i == j);

            if (rc) {
                fprintf(stderr, "error: %s: i = %zu, j = %zu, rc = %d\n",
                        "do_acert_verify_test", i, j, rc);
                break;
            }
        }

        if (rc) { break; }
    }

    ExpectIntEQ(rc, 0);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_ACERT_misc_api(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_ACERT) && !defined(NO_CERTS) && !defined(NO_RSA) && \
    !defined(NO_FILESYSTEM) && defined(OPENSSL_EXTRA)
    const char * acerts[4] = {"certs/acert/acert.pem",
                              "certs/acert/acert_ietf.pem",
                              "certs/acert/rsa_pss/acert.pem",
                              "certs/acert/rsa_pss/acert_ietf.pem"};
    int          rc = 0;
    X509_ACERT * x509 = NULL;
    BIO *        bp = NULL;
    long         ver_long = 0;
    int          ver = 0;
    int          nid = 0;
    const byte * raw_attr = NULL;
    word32       attr_len = 0;
    size_t       i = 0;
    int          buf_len = 0;
    byte         ietf_serial[] = {0x03, 0xb5, 0x90, 0x59, 0x02,
                                  0xa2, 0xaa, 0xb5, 0x40, 0x21,
                                  0x44, 0xb8, 0x2c, 0x4f, 0xd9,
                                  0x80, 0x1b, 0x5f, 0x57, 0xc2};

    for (i = 0; i < 4; ++i) {
        const char * acert_file = acerts[i];
        int          is_rsa_pss = 0;
        int          is_ietf_acert = 0;
        byte         serial[64];
        int          serial_len = sizeof(serial);

        XMEMSET(serial, 0, sizeof(serial));

        is_rsa_pss = XSTRSTR(acert_file, "rsa_pss") != NULL ? 1 : 0;
        is_ietf_acert = XSTRSTR(acert_file, "ietf.pem") != NULL ? 1 : 0;

        /* First read the attribute certificate. */
        bp = BIO_new_file(acert_file, "r");
        ExpectNotNull(bp);

        x509 = PEM_read_bio_X509_ACERT(bp, NULL, NULL, NULL);
        ExpectNotNull(x509);

        /* We're done with the bio for now. */
        if (bp != NULL) {
            BIO_free(bp);
            bp = NULL;
        }

        /* Check version and signature NID. */
        ver_long = X509_ACERT_get_version(x509);
        ExpectIntEQ(ver_long, 1);

        ver = wolfSSL_X509_ACERT_version(x509);
        ExpectIntEQ(ver, 2);

        nid = X509_ACERT_get_signature_nid(x509);

        if (is_rsa_pss) {
            ExpectIntEQ(nid, NID_rsassaPss);
        }
        else {
            ExpectIntEQ(nid, NID_sha256WithRSAEncryption);
        }

        /* Get the serial number buffer.
         * The ietf acert example has a 20 byte serial number. */
        rc = wolfSSL_X509_ACERT_get_serial_number(x509, serial, &serial_len);
        ExpectIntEQ(rc, SSL_SUCCESS);

        if (is_ietf_acert) {
            ExpectIntEQ(serial_len, 20);
            ExpectIntEQ(XMEMCMP(serial, ietf_serial, sizeof(ietf_serial)), 0);
        }
        else {
            ExpectIntEQ(serial_len, 1);
            ExpectTrue(serial[0] == 0x01);
        }

        /* Repeat the same but with null serial buffer. This is ok. */
        rc = wolfSSL_X509_ACERT_get_serial_number(x509, NULL, &serial_len);
        ExpectIntEQ(rc, SSL_SUCCESS);

        if (is_ietf_acert) {
            ExpectIntEQ(serial_len, 20);
        }
        else {
            ExpectIntEQ(serial_len, 1);
            ExpectTrue(serial[0] == 0x01);
        }

        /* Get the attributes buffer. */
        rc = wolfSSL_X509_ACERT_get_attr_buf(x509, &raw_attr, &attr_len);
        ExpectIntEQ(rc, SSL_SUCCESS);

        if (is_ietf_acert) {
            /* This cert has a 65 byte attributes field. */
            ExpectNotNull(raw_attr);
            ExpectIntEQ(attr_len, 65);
        }
        else {
            /* This cert has a 237 byte attributes field. */
            ExpectNotNull(raw_attr);
            ExpectIntEQ(attr_len, 237);
        }

        /* Test printing acert to memory bio. */
        ExpectNotNull(bp = BIO_new(BIO_s_mem()));
        rc = X509_ACERT_print(bp, x509);
        ExpectIntEQ(rc, SSL_SUCCESS);

        /* Now do a bunch of invalid stuff with partially valid inputs. */
        rc = wolfSSL_X509_ACERT_get_attr_buf(x509, &raw_attr, NULL);
        ExpectIntEQ(rc, BAD_FUNC_ARG);

        rc = wolfSSL_X509_ACERT_get_attr_buf(x509, NULL, &attr_len);
        ExpectIntEQ(rc, BAD_FUNC_ARG);

        rc = wolfSSL_X509_ACERT_get_attr_buf(NULL, &raw_attr, &attr_len);
        ExpectIntEQ(rc, BAD_FUNC_ARG);

        ver_long = X509_ACERT_get_version(NULL);
        ExpectIntEQ(ver_long, 0);

        ver = wolfSSL_X509_ACERT_version(NULL);
        ExpectIntEQ(ver, 0);

        rc = wolfSSL_X509_ACERT_get_signature(x509, NULL, NULL);
        ExpectIntEQ(rc, WOLFSSL_FATAL_ERROR);

        rc = wolfSSL_X509_ACERT_get_signature(x509, NULL, &buf_len);
        ExpectIntEQ(rc, SSL_SUCCESS);
        ExpectIntEQ(buf_len, 256);

        rc = wolfSSL_X509_ACERT_get_serial_number(x509, serial, NULL);
        ExpectIntEQ(rc, BAD_FUNC_ARG);

        rc = X509_ACERT_print(bp, NULL);
        ExpectIntEQ(rc, WOLFSSL_FAILURE);

        rc = X509_ACERT_print(NULL, x509);
        ExpectIntEQ(rc, WOLFSSL_FAILURE);

        /* Finally free the acert and bio, we're done with them. */
        if (x509 != NULL) {
            X509_ACERT_free(x509);
            x509 = NULL;
        }

        if (bp != NULL) {
            BIO_free(bp);
            bp = NULL;
        }
    }
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_ACERT_buffer(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_ACERT) && !defined(NO_CERTS) && \
    !defined(NO_RSA) && defined(WC_RSA_PSS) && \
    (defined(OPENSSL_EXTRA_X509_SMALL) || defined(OPENSSL_EXTRA))
    const byte acert_ietf[] = \
    "-----BEGIN ATTRIBUTE CERTIFICATE-----\n"
    "MIICPTCCASUCAQEwN6AWMBGkDzANMQswCQYDVQQDDAJDQQIBAqEdpBswGTEXMBUG\n"
    "A1UEAwwOc2VydmVyLmV4YW1wbGWgLTArpCkwJzElMCMGA1UEAwwcQXR0cmlidXRl\n"
    "IENlcnRpZmljYXRlIElzc3VlcjANBgkqhkiG9w0BAQsFAAIUA7WQWQKiqrVAIUS4\n"
    "LE/ZgBtfV8IwIhgPMjAyMTA2MTUxMjM1MDBaGA8yMDMxMDYxMzEyMzUwMFowQTAj\n"
    "BggrBgEFBQcKBDEXMBWgCYYHVGVzdHZhbDAIDAZncm91cDEwGgYDVQRIMRMwEaEP\n"
    "gw1hZG1pbmlzdHJhdG9yMCwwHwYDVR0jBBgwFoAUYm7JaGdsZLtTgt0tqoCK2MrI\n"
    "i10wCQYDVR04BAIFADANBgkqhkiG9w0BAQsFAAOCAQEAlIOJ2Dj3TEUj6BIv6vUs\n"
    "GqFWms05i+d10XSzWrunlUTQPoJcUjYkifOWp/7RpZ2XnRl+6hH+nIbmwSmXWwBn\n"
    "ERw2bQMmw/""/nWuN4Qv9t7ltuovWC0pJX6VMT1IRTuTV4SxuZpFL37vkmnFlPBlb+\n"
    "mn3ESSxLTjThWFIq1tip4IaxE/i5Uh32GlJglatFHM1PCGoJtyLtYb6KHDlvknw6\n"
    "coDyjIcj0FZwtQw41jLwxI8jWNmrpt978wdpprB/URrRs+m02HmeQoiHFi/qvdv8\n"
    "d+5vHf3Pi/ulhz/+dvr0p1vEQSoFnYxLXuty2p5m3PJPZCFmT3gURgmgR3BN9d7A\n"
    "Bw==\n"
    "-----END ATTRIBUTE CERTIFICATE-----\n";
    X509_ACERT * x509 = NULL;
    int          rc = 0;
    byte         ietf_serial[] = {0x03, 0xb5, 0x90, 0x59, 0x02,
                                  0xa2, 0xaa, 0xb5, 0x40, 0x21,
                                  0x44, 0xb8, 0x2c, 0x4f, 0xd9,
                                  0x80, 0x1b, 0x5f, 0x57, 0xc2};
    byte         serial[64];
    int          serial_len = sizeof(serial);
    const byte * raw_attr = NULL;
    word32       attr_len = 0;

    x509 = wolfSSL_X509_ACERT_load_certificate_buffer_ex(acert_ietf,
                                                         sizeof(acert_ietf),
                                                         WOLFSSL_FILETYPE_PEM,
                                                         HEAP_HINT);

    rc = wolfSSL_X509_ACERT_get_serial_number(x509, serial, &serial_len);
    ExpectIntEQ(rc, SSL_SUCCESS);

    ExpectIntEQ(serial_len, 20);
    ExpectIntEQ(XMEMCMP(serial, ietf_serial, sizeof(ietf_serial)), 0);

    /* Get the attributes buffer. */
    rc = wolfSSL_X509_ACERT_get_attr_buf(x509, &raw_attr, &attr_len);
    ExpectIntEQ(rc, SSL_SUCCESS);

    /* This cert has a 65 byte attributes field. */
    ExpectNotNull(raw_attr);
    ExpectIntEQ(attr_len, 65);

    ExpectNotNull(x509);

    if (x509 != NULL) {
        wolfSSL_X509_ACERT_free(x509);
        x509 = NULL;
    }
#endif
    return EXPECT_RESULT();
}

/* note: when ACERT generation and signing are implemented,
 * this test will be filled out appropriately.
 * */
static int test_wolfSSL_X509_ACERT_new_and_sign(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_ACERT) && !defined(NO_CERTS) && \
    !defined(NO_RSA) && defined(WC_RSA_PSS) && \
    (defined(OPENSSL_EXTRA_X509_SMALL) || defined(OPENSSL_EXTRA))
    X509_ACERT * x509 = NULL;
    int          rc = 0;

    x509 = X509_ACERT_new();
    ExpectNotNull(x509);

    if (x509 != NULL) {
        wolfSSL_X509_ACERT_free(x509);
        x509 = NULL;
    }

    /* Same but with static memory hint. */
    x509 = wolfSSL_X509_ACERT_new_ex(HEAP_HINT);
    ExpectNotNull(x509);

    #ifndef NO_WOLFSSL_STUB
    /* ACERT sign not implemented yet. */
    if (x509 != NULL) {
        rc = wolfSSL_X509_ACERT_sign(x509, NULL, NULL);
        ExpectIntEQ(rc, WOLFSSL_NOT_IMPLEMENTED);
    }
    #else
    (void) rc;
    #endif /* NO_WOLFSSL_STUB */

    if (x509 != NULL) {
        wolfSSL_X509_ACERT_free(x509);
        x509 = NULL;
    }

#endif
    return EXPECT_RESULT();
}

/* Test ACERT support, but with ASN functions only.
 *
 * This example acert_ietf has both Holder IssuerSerial
 * and Holder entityName fields.
 * */
static int test_wolfSSL_X509_ACERT_asn(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_ACERT) && !defined(NO_CERTS)
    const byte     acert_ietf[] = \
    "-----BEGIN ATTRIBUTE CERTIFICATE-----\n"
    "MIICPTCCASUCAQEwN6AWMBGkDzANMQswCQYDVQQDDAJDQQIBAqEdpBswGTEXMBUG\n"
    "A1UEAwwOc2VydmVyLmV4YW1wbGWgLTArpCkwJzElMCMGA1UEAwwcQXR0cmlidXRl\n"
    "IENlcnRpZmljYXRlIElzc3VlcjANBgkqhkiG9w0BAQsFAAIUA7WQWQKiqrVAIUS4\n"
    "LE/ZgBtfV8IwIhgPMjAyMTA2MTUxMjM1MDBaGA8yMDMxMDYxMzEyMzUwMFowQTAj\n"
    "BggrBgEFBQcKBDEXMBWgCYYHVGVzdHZhbDAIDAZncm91cDEwGgYDVQRIMRMwEaEP\n"
    "gw1hZG1pbmlzdHJhdG9yMCwwHwYDVR0jBBgwFoAUYm7JaGdsZLtTgt0tqoCK2MrI\n"
    "i10wCQYDVR04BAIFADANBgkqhkiG9w0BAQsFAAOCAQEAlIOJ2Dj3TEUj6BIv6vUs\n"
    "GqFWms05i+d10XSzWrunlUTQPoJcUjYkifOWp/7RpZ2XnRl+6hH+nIbmwSmXWwBn\n"
    "ERw2bQMmw/""/nWuN4Qv9t7ltuovWC0pJX6VMT1IRTuTV4SxuZpFL37vkmnFlPBlb+\n"
    "mn3ESSxLTjThWFIq1tip4IaxE/i5Uh32GlJglatFHM1PCGoJtyLtYb6KHDlvknw6\n"
    "coDyjIcj0FZwtQw41jLwxI8jWNmrpt978wdpprB/URrRs+m02HmeQoiHFi/qvdv8\n"
    "d+5vHf3Pi/ulhz/+dvr0p1vEQSoFnYxLXuty2p5m3PJPZCFmT3gURgmgR3BN9d7A\n"
    "Bw==\n"
    "-----END ATTRIBUTE CERTIFICATE-----\n";
    int            rc = 0;
    int            n_diff = 0;
    byte           ietf_serial[] =      {0x03, 0xb5, 0x90, 0x59, 0x02,
                                         0xa2, 0xaa, 0xb5, 0x40, 0x21,
                                         0x44, 0xb8, 0x2c, 0x4f, 0xd9,
                                         0x80, 0x1b, 0x5f, 0x57, 0xc2};
    byte           holderIssuerName[] = {0x31, 0x0b, 0x30, 0x09, 0x06,
                                         0x03, 0x55, 0x04, 0x03, 0x0c,
                                         0x02, 0x43, 0x41};
    byte           holderEntityName[] = {0x31, 0x17, 0x30, 0x15, 0x06,
                                         0x03, 0x55, 0x04, 0x03, 0x0c,
                                         0x0e, 0x73, 0x65, 0x72, 0x76,
                                         0x65, 0x72, 0x2e, 0x65, 0x78,
                                         0x61, 0x6d, 0x70, 0x6c, 0x65};
    DerBuffer *    der = NULL;
#ifdef WOLFSSL_SMALL_STACK
    DecodedAcert * acert = NULL;
#else
    DecodedAcert   acert[1];
#endif

    rc = wc_PemToDer(acert_ietf, sizeof(acert_ietf), ACERT_TYPE, &der,
                     HEAP_HINT, NULL, NULL);

    ExpectIntEQ(rc, 0);
    ExpectNotNull(der);

    if (der != NULL) {
        ExpectNotNull(der->buffer);
    }

#ifdef WOLFSSL_SMALL_STACK
    acert = (DecodedAcert*)XMALLOC(sizeof(DecodedAcert), HEAP_HINT,
                                   DYNAMIC_TYPE_DCERT);
    ExpectNotNull(acert);
#else
    XMEMSET(acert, 0, sizeof(DecodedAcert));
#endif

    if (der != NULL && der->buffer != NULL
#ifdef WOLFSSL_SMALL_STACK
        && acert != NULL
#endif
    ) {
        wc_InitDecodedAcert(acert, der->buffer, der->length, HEAP_HINT);
        rc = wc_ParseX509Acert(acert, VERIFY_SKIP_DATE);
        ExpectIntEQ(rc, 0);

        ExpectIntEQ(acert->serialSz, 20);
        ExpectIntEQ(XMEMCMP(acert->serial, ietf_serial, sizeof(ietf_serial)),
                    0);

        /* This cert has a 65 byte attributes field. */
        ExpectNotNull(acert->rawAttr);
        ExpectIntEQ(acert->rawAttrLen, 65);

        ExpectNotNull(acert->holderIssuerName);
        ExpectNotNull(acert->holderEntityName);

        if ((acert->holderIssuerName != NULL) &&
            (acert->holderEntityName != NULL)) {
            ExpectNotNull(acert->holderEntityName->name);
            ExpectNotNull(acert->holderIssuerName->name);
        }

        if ((acert->holderIssuerName != NULL) &&
            (acert->holderEntityName != NULL) &&
            (acert->holderIssuerName->name != NULL) &&
            (acert->holderEntityName->name != NULL)) {
            ExpectIntEQ(acert->holderIssuerName->len,
                        sizeof(holderIssuerName));
            ExpectIntEQ(acert->holderEntityName->len,
                        sizeof(holderEntityName));

            ExpectIntEQ(acert->holderIssuerName->type, ASN_DIR_TYPE);
            ExpectIntEQ(acert->holderEntityName->type, ASN_DIR_TYPE);

            n_diff = XMEMCMP(acert->holderIssuerName->name, holderIssuerName,
                             sizeof(holderIssuerName));
            ExpectIntEQ(n_diff, 0);

            n_diff = XMEMCMP(acert->holderEntityName->name, holderEntityName,
                             sizeof(holderEntityName));
            ExpectIntEQ(n_diff, 0);
        }

        wc_FreeDecodedAcert(acert);
    }

#ifdef WOLFSSL_SMALL_STACK
    if (acert != NULL) {
        XFREE(acert, HEAP_HINT, DYNAMIC_TYPE_DCERT);
        acert = NULL;
    }
#endif

    if (der != NULL) {
        wc_FreeDer(&der);
        der = NULL;
    }

#endif
    return EXPECT_RESULT();
}

#if !defined(NO_DH) && !defined(NO_AES) && defined(WOLFSSL_CERT_GEN) && \
         defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && \
         defined(OPENSSL_EXTRA) && !defined(NO_ASN_TIME)
/* create certificate with version 2 */
static int test_set_x509_badversion(WOLFSSL_CTX* ctx)
{
    EXPECT_DECLS;
    WOLFSSL_X509 *x509 = NULL, *x509v2 = NULL;
    WOLFSSL_EVP_PKEY *priv = NULL, *pub = NULL;
    unsigned char *der = NULL, *key = NULL, *pt;
    char *header = NULL, *name = NULL;
    int derSz;
    long keySz;
    XFILE fp = XBADFILE;
    WOLFSSL_ASN1_TIME *notBefore = NULL, *notAfter = NULL;
    time_t t;

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
                WOLFSSL_FILETYPE_PEM));

    ExpectTrue((fp = XFOPEN(cliKeyFile, "rb")) != XBADFILE);
    ExpectIntEQ(wolfSSL_PEM_read(fp, &name, &header, &key, &keySz),
            WOLFSSL_SUCCESS);
    if (fp != XBADFILE)
        XFCLOSE(fp);
    pt = key;
    ExpectNotNull(priv = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL,
                (const unsigned char**)&pt, keySz));


    /* create the version 2 certificate */
    ExpectNotNull(x509v2 = X509_new());
    ExpectIntEQ(wolfSSL_X509_set_version(x509v2, 1), WOLFSSL_SUCCESS);

    ExpectIntEQ(wolfSSL_X509_set_subject_name(x509v2,
                wolfSSL_X509_get_subject_name(x509)), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_set_issuer_name(x509v2,
                wolfSSL_X509_get_issuer_name(x509)), WOLFSSL_SUCCESS);
    ExpectNotNull(pub = wolfSSL_X509_get_pubkey(x509));
    ExpectIntEQ(X509_set_pubkey(x509v2, pub), WOLFSSL_SUCCESS);

    t = time(NULL);
    ExpectNotNull(notBefore = wolfSSL_ASN1_TIME_adj(NULL, t, 0, 0));
    ExpectNotNull(notAfter = wolfSSL_ASN1_TIME_adj(NULL, t, 365, 0));
    ExpectTrue(wolfSSL_X509_set_notBefore(x509v2, notBefore));
    ExpectTrue(wolfSSL_X509_set_notAfter(x509v2, notAfter));

    ExpectIntGT(wolfSSL_X509_sign(x509v2, priv, EVP_sha256()), 0);
    derSz = wolfSSL_i2d_X509(x509v2, &der);
    ExpectIntGT(derSz, 0);
    ExpectIntEQ(wolfSSL_CTX_use_certificate_buffer(ctx, der, derSz,
                                     WOLFSSL_FILETYPE_ASN1), WOLFSSL_SUCCESS);
    /* TODO: Replace with API call */
    XFREE(der, HEAP_HINT, DYNAMIC_TYPE_OPENSSL);
    XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(name, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(header, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    wolfSSL_X509_free(x509);
    wolfSSL_X509_free(x509v2);
    wolfSSL_EVP_PKEY_free(priv);
    wolfSSL_EVP_PKEY_free(pub);
    wolfSSL_ASN1_TIME_free(notBefore);
    wolfSSL_ASN1_TIME_free(notAfter);

    return EXPECT_RESULT();
}


/* override certificate version error */
static int test_override_x509(int preverify, WOLFSSL_X509_STORE_CTX* store)
{
    EXPECT_DECLS;
#ifndef OPENSSL_COMPATIBLE_DEFAULTS
    ExpectIntEQ(store->error, WC_NO_ERR_TRACE(ASN_VERSION_E));
#else
    ExpectIntEQ(store->error, 0);
#endif
    ExpectIntEQ((int)wolfSSL_X509_get_version(store->current_cert), 1);
    (void)preverify;
    return EXPECT_RESULT() == TEST_SUCCESS;
}


/* set verify callback that will override bad certificate version */
static int test_set_override_x509(WOLFSSL_CTX* ctx)
{
    wolfSSL_CTX_set_verify(ctx, WOLFSSL_VERIFY_PEER, test_override_x509);
    return TEST_SUCCESS;
}
#endif


static int test_wolfSSL_X509_TLS_version_test_1(void)
{
    EXPECT_DECLS;
#if !defined(NO_DH) && !defined(NO_AES) && defined(WOLFSSL_CERT_GEN) && \
         defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && \
         defined(OPENSSL_EXTRA) && !defined(NO_ASN_TIME)
    test_ssl_cbf func_cb_client;
    test_ssl_cbf func_cb_server;

    /* test server rejects a client certificate that is not version 3 */
    XMEMSET(&func_cb_client, 0, sizeof(func_cb_client));
    XMEMSET(&func_cb_server, 0, sizeof(func_cb_server));

    func_cb_client.ctx_ready = &test_set_x509_badversion;
#ifndef WOLFSSL_NO_TLS12
    func_cb_client.method = wolfTLSv1_2_client_method;
#else
    func_cb_client.method = wolfTLSv1_3_client_method;
#endif

#ifndef WOLFSSL_NO_TLS12
    func_cb_server.method = wolfTLSv1_2_server_method;
#else
    func_cb_server.method = wolfTLSv1_3_server_method;
#endif

#ifndef OPENSSL_COMPATIBLE_DEFAULTS
    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&func_cb_client,
        &func_cb_server, NULL), -1001);
#else
    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&func_cb_client,
        &func_cb_server, NULL), TEST_SUCCESS);
#endif
#endif

    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_TLS_version_test_2(void)
{
    EXPECT_DECLS;
#if !defined(NO_DH) && !defined(NO_AES) && defined(WOLFSSL_CERT_GEN) && \
         defined(HAVE_SSL_MEMIO_TESTS_DEPENDENCIES) && \
         defined(OPENSSL_EXTRA) && !defined(NO_ASN_TIME)
    test_ssl_cbf func_cb_client;
    test_ssl_cbf func_cb_server;

    XMEMSET(&func_cb_client, 0, sizeof(func_cb_client));
    XMEMSET(&func_cb_server, 0, sizeof(func_cb_server));

    func_cb_client.ctx_ready = &test_set_x509_badversion;
    func_cb_server.ctx_ready = &test_set_override_x509;
#ifndef WOLFSSL_NO_TLS12
    func_cb_client.method = wolfTLSv1_2_client_method;
#else
    func_cb_client.method = wolfTLSv1_3_client_method;
#endif

#ifndef WOLFSSL_NO_TLS12
    func_cb_server.method = wolfTLSv1_2_server_method;
#else
    func_cb_server.method = wolfTLSv1_3_server_method;
#endif

    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&func_cb_client,
        &func_cb_server, NULL), TEST_SUCCESS);
#endif

    return EXPECT_RESULT();
}

/* Testing function  wolfSSL_CTX_SetMinVersion; sets the minimum downgrade
 * version allowed.
 * POST: 1 on success.
 */
static int test_wolfSSL_CTX_SetMinVersion(void)
{
    int                     res = TEST_SKIPPED;
#if !defined(NO_WOLFSSL_CLIENT) && !defined(NO_TLS)
    int                     failFlag = WOLFSSL_SUCCESS;
    WOLFSSL_CTX*            ctx;
    int                     itr;

    #ifndef NO_OLD_TLS
        const int versions[]  = {
                            #ifdef WOLFSSL_ALLOW_TLSV10
                                  WOLFSSL_TLSV1,
                            #endif
                                  WOLFSSL_TLSV1_1,
                                  WOLFSSL_TLSV1_2 };
    #elif !defined(WOLFSSL_NO_TLS12)
        const int versions[]  = { WOLFSSL_TLSV1_2 };
    #elif defined(WOLFSSL_TLS13)
        const int versions[]  = { WOLFSSL_TLSV1_3 };
    #else
        const int versions[0];
    #endif

    ctx = wolfSSL_CTX_new(wolfSSLv23_client_method());

    for (itr = 0; itr < (int)(sizeof(versions)/sizeof(int)); itr++) {
        if (wolfSSL_CTX_SetMinVersion(ctx, *(versions + itr))
                != WOLFSSL_SUCCESS) {
            failFlag = WOLFSSL_FAILURE;
        }
    }

    wolfSSL_CTX_free(ctx);

    res = TEST_RES_CHECK(failFlag == WOLFSSL_SUCCESS);
#endif
    return res;

} /* END test_wolfSSL_CTX_SetMinVersion */


/*----------------------------------------------------------------------------*
 | OCSP Stapling
 *----------------------------------------------------------------------------*/


/* Testing wolfSSL_UseOCSPStapling function. OCSP stapling eliminates the need
 * need to contact the CA, lowering the cost of cert revocation checking.
 * PRE: HAVE_OCSP and HAVE_CERTIFICATE_STATUS_REQUEST
 * POST: 1 returned for success.
 */
static int test_wolfSSL_UseOCSPStapling(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CERTIFICATE_STATUS_REQUEST) && defined(HAVE_OCSP) && \
    !defined(NO_TLS) && !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX*    ctx = NULL;
    WOLFSSL*        ssl = NULL;

#ifndef NO_WOLFSSL_CLIENT
    #ifndef WOLFSSL_NO_TLS12
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_2_client_method()));
    #else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_3_client_method()));
    #endif
#else
    #ifndef WOLFSSL_NO_TLS12
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_2_server_method()));
    #else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_3_server_method()));
    #endif
#endif
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    ExpectIntEQ(wolfSSL_UseOCSPStapling(NULL, WOLFSSL_CSR2_OCSP,
        WOLFSSL_CSR2_OCSP_USE_NONCE), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
#ifndef NO_WOLFSSL_CLIENT
    ExpectIntEQ(wolfSSL_UseOCSPStapling(ssl, WOLFSSL_CSR2_OCSP,
        WOLFSSL_CSR2_OCSP_USE_NONCE), 1);
#else
    ExpectIntEQ(wolfSSL_UseOCSPStapling(ssl, WOLFSSL_CSR2_OCSP,
        WOLFSSL_CSR2_OCSP_USE_NONCE), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
#endif

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
} /* END test_wolfSSL_UseOCSPStapling */


/* Testing OCSP stapling version 2, wolfSSL_UseOCSPStaplingV2 function. OCSP
 * stapling eliminates the need to contact the CA and lowers cert revocation
 * check.
 * PRE: HAVE_CERTIFICATE_STATUS_REQUEST_V2 and HAVE_OCSP defined.
 */
static int test_wolfSSL_UseOCSPStaplingV2(void)
{
    EXPECT_DECLS;
#if defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2) && defined(HAVE_OCSP) && \
    !defined(NO_TLS) && !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX*        ctx = NULL;
    WOLFSSL*            ssl = NULL;

#ifndef NO_WOLFSSL_CLIENT
    #ifndef WOLFSSL_NO_TLS12
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_2_client_method()));
    #else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_3_client_method()));
    #endif
#else
    #ifndef WOLFSSL_NO_TLS12
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_2_server_method()));
    #else
        ExpectNotNull(ctx = wolfSSL_CTX_new(wolfTLSv1_3_server_method()));
    #endif
#endif
    ExpectNotNull(ssl = wolfSSL_new(ctx));

    ExpectIntEQ(wolfSSL_UseOCSPStaplingV2(NULL, WOLFSSL_CSR2_OCSP,
        WOLFSSL_CSR2_OCSP_USE_NONCE), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
#ifndef NO_WOLFSSL_CLIENT
    ExpectIntEQ(wolfSSL_UseOCSPStaplingV2(ssl, WOLFSSL_CSR2_OCSP,
        WOLFSSL_CSR2_OCSP_USE_NONCE), 1);
#else
    ExpectIntEQ(wolfSSL_UseOCSPStaplingV2(ssl, WOLFSSL_CSR2_OCSP,
        WOLFSSL_CSR2_OCSP_USE_NONCE), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
#endif

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();

} /* END test_wolfSSL_UseOCSPStaplingV2 */

/*----------------------------------------------------------------------------*
 | Multicast Tests
 *----------------------------------------------------------------------------*/
static int test_wolfSSL_mcast(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_DTLS) && defined(WOLFSSL_MULTICAST) && \
    (defined(WOLFSSL_TLS13) || defined(WOLFSSL_SNIFFER))
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL* ssl = NULL;
    byte preMasterSecret[512];
    byte clientRandom[32];
    byte serverRandom[32];
    byte suite[2] = {0, 0xfe};  /* WDM_WITH_NULL_SHA256 */
    byte buf[256];
    word16 newId;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfDTLSv1_2_client_method()));

    ExpectIntEQ(wolfSSL_CTX_mcast_set_member_id(ctx, 0), WOLFSSL_SUCCESS);

    ExpectNotNull(ssl = wolfSSL_new(ctx));

    XMEMSET(preMasterSecret, 0x23, sizeof(preMasterSecret));
    XMEMSET(clientRandom, 0xA5, sizeof(clientRandom));
    XMEMSET(serverRandom, 0x5A, sizeof(serverRandom));
    ExpectIntEQ(wolfSSL_set_secret(ssl, 23, preMasterSecret,
        sizeof(preMasterSecret), clientRandom, serverRandom, suite),
        WOLFSSL_SUCCESS);

    ExpectIntLE(wolfSSL_mcast_read(ssl, &newId, buf, sizeof(buf)), 0);
    ExpectIntLE(newId, 100);

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif /* WOLFSSL_DTLS && WOLFSSL_MULTICAST && (WOLFSSL_TLS13 ||
        * WOLFSSL_SNIFFER) */
    return EXPECT_RESULT();
}


/*----------------------------------------------------------------------------*
 |  Wolfcrypt
 *----------------------------------------------------------------------------*/

/*
 * Testing wc_SetKeyUsage()
 */
static int test_wc_SetKeyUsage(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA) && defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN) && !defined(HAVE_FIPS)
    Cert myCert;

    ExpectIntEQ(wc_InitCert(&myCert), 0);

    ExpectIntEQ(wc_SetKeyUsage(&myCert, "keyEncipherment,keyAgreement"), 0);
    ExpectIntEQ(wc_SetKeyUsage(&myCert, "digitalSignature,nonRepudiation"), 0);
    ExpectIntEQ(wc_SetKeyUsage(&myCert, "contentCommitment,encipherOnly"), 0);
    ExpectIntEQ(wc_SetKeyUsage(&myCert, "decipherOnly"), 0);
    ExpectIntEQ(wc_SetKeyUsage(&myCert, "cRLSign,keyCertSign"), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_SetKeyUsage(NULL, "decipherOnly"), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_SetKeyUsage(&myCert, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_SetKeyUsage(&myCert, ""), WC_NO_ERR_TRACE(KEYUSAGE_E));
    ExpectIntEQ(wc_SetKeyUsage(&myCert, ","), WC_NO_ERR_TRACE(KEYUSAGE_E));
    ExpectIntEQ(wc_SetKeyUsage(&myCert, "digitalSignature, cRLSign"),
        WC_NO_ERR_TRACE(KEYUSAGE_E));
#endif
    return EXPECT_RESULT();
} /* END  test_wc_SetKeyUsage */

#if defined(OPENSSL_EXTRA) || defined(HAVE_WEBSERVER)
static void sample_mutex_cb (int flag, int type, const char* file, int line)
{
    (void)flag;
    (void)type;
    (void)file;
    (void)line;
}
#endif
/*
 * Testing wc_LockMutex_ex
 */
static int test_wc_LockMutex_ex(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) || defined(HAVE_WEBSERVER)
    int flag = CRYPTO_LOCK;
    int type = 0;
    const char* file = "./test-LockMutex_ex.txt";
    int line = 0;

    /* without SetMutexCb */
    ExpectIntEQ(wc_LockMutex_ex(flag, type, file, line), WC_NO_ERR_TRACE(BAD_STATE_E));
    /* with SetMutexCb */
    ExpectIntEQ(wc_SetMutexCb(sample_mutex_cb), 0);
    ExpectIntEQ(wc_LockMutex_ex(flag, type, file, line), 0);
    ExpectIntEQ(wc_SetMutexCb(NULL), 0);
#endif
    return EXPECT_RESULT();
} /* End test_wc_LockMutex_ex*/
/*
 * Testing wc_SetMutexCb
 */
static int test_wc_SetMutexCb(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) || defined(HAVE_WEBSERVER)
    ExpectIntEQ(wc_SetMutexCb(sample_mutex_cb), 0);
    ExpectIntEQ(wc_SetMutexCb(NULL), 0);
#endif
    return EXPECT_RESULT();
} /* End test_wc_SetMutexCb*/


/*
 * Testing ToTraditional
 */
static int test_ToTraditional(void)
{
    EXPECT_DECLS;
#if !defined(NO_ASN) && (defined(HAVE_PKCS8) || defined(HAVE_PKCS12)) && \
    (defined(WOLFSSL_TEST_CERT) || defined(OPENSSL_EXTRA) || \
     defined(OPENSSL_EXTRA_X509_SMALL)) && !defined(NO_FILESYSTEM)
    XFILE  f = XBADFILE;
    byte   input[TWOK_BUF];
    word32 sz = 0;

    ExpectTrue((f = XFOPEN("./certs/server-keyPkcs8.der", "rb")) != XBADFILE);
    ExpectTrue((sz = (word32)XFREAD(input, 1, sizeof(input), f)) > 0);
    if (f != XBADFILE)
        XFCLOSE(f);

    /* Good case */
    ExpectIntGT(ToTraditional(input, sz), 0);
    /* Bad cases */
    ExpectIntEQ(ToTraditional(NULL, 0), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(ToTraditional(NULL, sz), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
#ifdef WOLFSSL_ASN_TEMPLATE
    ExpectIntEQ(ToTraditional(input, 0), WC_NO_ERR_TRACE(BUFFER_E));
#else
    ExpectIntEQ(ToTraditional(input, 0), WC_NO_ERR_TRACE(ASN_PARSE_E));
#endif
#endif
    return EXPECT_RESULT();
} /* End test_ToTraditional*/


/*
 * Testing wc_SetSubjectBuffer
 */
static int test_wc_SetSubjectBuffer(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_CERT_GEN) && !defined(NO_RSA) && !defined(NO_FILESYSTEM)
    Cert   cert;
    XFILE  file = XBADFILE;
    byte*  der = NULL;
    word32 derSz;

    derSz = FOURK_BUF;
    ExpectNotNull(der = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT,
        DYNAMIC_TYPE_TMP_BUFFER));
    ExpectTrue((file = XFOPEN("./certs/ca-cert.der", "rb")) != XBADFILE);
    ExpectTrue((derSz = (word32)XFREAD(der, 1, FOURK_BUF, file)) > 0);
    if (file != XBADFILE)
        XFCLOSE(file);

    ExpectIntEQ(wc_InitCert(&cert), 0);
    ExpectIntEQ(wc_SetSubjectBuffer(&cert, der, (int)derSz), 0);
    ExpectIntEQ(wc_SetSubjectBuffer(NULL, der, (int)derSz), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif
    return EXPECT_RESULT();
} /* End test_wc_SetSubjectBuffer*/

/*
 * Testing wc_SetSubjectKeyIdFromPublicKey_ex
 */
static int test_wc_SetSubjectKeyIdFromPublicKey_ex(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN)
    WC_RNG      rng;
    Cert        cert;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN)
    RsaKey      rsaKey;
    int         bits = 2048;
#endif
#if defined(HAVE_ECC)
    ecc_key     eccKey;
    int         ret;
#endif
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT)
    ed25519_key ed25519Key;
#endif
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT)
    ed448_key   ed448Key;
#endif

#ifndef HAVE_FIPS
    ExpectIntEQ(wc_InitRng_ex(&rng, HEAP_HINT, testDevId), 0);
#else
    ExpectIntEQ(wc_InitRng(&rng), 0);
#endif

    ExpectIntEQ(wc_InitCert(&cert), 0);

#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN)
    /* RSA */
    XMEMSET(&rsaKey, 0, sizeof(RsaKey));
    ExpectIntEQ(wc_InitRsaKey(&rsaKey, HEAP_HINT), 0);
    ExpectIntEQ(MAKE_RSA_KEY(&rsaKey, bits, WC_RSA_EXPONENT, &rng), 0);
    ExpectIntEQ(wc_SetSubjectKeyIdFromPublicKey_ex(&cert, RSA_TYPE, &rsaKey),
        0);
    DoExpectIntEQ(wc_FreeRsaKey(&rsaKey), 0);
#endif

#if defined(HAVE_ECC)
    /* ECC */
    XMEMSET(&eccKey, 0, sizeof(ecc_key));
    ExpectIntEQ(wc_ecc_init(&eccKey), 0);
    ret = wc_ecc_make_key(&rng, KEY14, &eccKey);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &eccKey.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);
    ExpectIntEQ(wc_SetSubjectKeyIdFromPublicKey_ex(&cert, ECC_TYPE, &eccKey),
        0);
    DoExpectIntEQ(wc_ecc_free(&eccKey), 0);
#endif

#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT)
    /* ED25519 */
    XMEMSET(&ed25519Key, 0, sizeof(ed25519_key));
    ExpectIntEQ(wc_ed25519_init(&ed25519Key), 0);
    ExpectIntEQ(wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &ed25519Key), 0);
    ExpectIntEQ(wc_SetSubjectKeyIdFromPublicKey_ex(&cert, ED25519_TYPE,
        &ed25519Key), 0);
    wc_ed25519_free(&ed25519Key);
#endif

#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT)
    /* ED448 */
    XMEMSET(&ed448Key, 0, sizeof(ed448_key));
    ExpectIntEQ(wc_ed448_init(&ed448Key), 0);
    ExpectIntEQ(wc_ed448_make_key(&rng, ED448_KEY_SIZE, &ed448Key), 0);
    ExpectIntEQ(wc_SetSubjectKeyIdFromPublicKey_ex(&cert, ED448_TYPE,
        &ed448Key), 0);
    wc_ed448_free(&ed448Key);
#endif

    wc_FreeRng(&rng);
    DoExpectIntEQ(wc_FreeRng(&rng), 0);
#endif /* WOLFSSL_CERT_EXT && WOLFSSL_CERT_GEN */
    return EXPECT_RESULT();
} /* End test_wc_SetSubjectKeyIdFromPublicKey_ex*/

/*
 * Testing wc_SetAuthKeyIdFromPublicKey_ex
 */
static int test_wc_SetAuthKeyIdFromPublicKey_ex(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN)
    WC_RNG      rng;
    Cert        cert;
#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN)
    RsaKey      rsaKey;
    int         bits = 2048;
#endif
#if defined(HAVE_ECC)
    ecc_key     eccKey;
    int         ret;
#endif
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT)
    ed25519_key ed25519Key;
#endif
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT)
    ed448_key   ed448Key;
#endif

#ifndef HAVE_FIPS
    ExpectIntEQ(wc_InitRng_ex(&rng, HEAP_HINT, testDevId), 0);
#else
    ExpectIntEQ(wc_InitRng(&rng), 0);
#endif

    ExpectIntEQ(wc_InitCert(&cert), 0);

#if !defined(NO_RSA) && defined(WOLFSSL_KEY_GEN)
    /* RSA */
    XMEMSET(&rsaKey, 0, sizeof(RsaKey));
    ExpectIntEQ(wc_InitRsaKey(&rsaKey, HEAP_HINT), 0);
    ExpectIntEQ(MAKE_RSA_KEY(&rsaKey, bits, WC_RSA_EXPONENT, &rng), 0);
    ExpectIntEQ(wc_SetAuthKeyIdFromPublicKey_ex(&cert, RSA_TYPE, &rsaKey), 0);
    DoExpectIntEQ(wc_FreeRsaKey(&rsaKey), 0);
#endif

#if defined(HAVE_ECC)
    /* ECC */
    XMEMSET(&eccKey, 0, sizeof(ecc_key));
    ExpectIntEQ(wc_ecc_init(&eccKey), 0);
    ret = wc_ecc_make_key(&rng, KEY14, &eccKey);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &eccKey.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    ExpectIntEQ(ret, 0);
    ExpectIntEQ(wc_SetAuthKeyIdFromPublicKey_ex(&cert, ECC_TYPE, &eccKey), 0);
    DoExpectIntEQ(wc_ecc_free(&eccKey), 0);
#endif

#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT)
    /* ED25519 */
    XMEMSET(&ed25519Key, 0, sizeof(ed25519_key));
    ExpectIntEQ(wc_ed25519_init(&ed25519Key), 0);
    ExpectIntEQ(wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &ed25519Key), 0);
    ExpectIntEQ(wc_SetAuthKeyIdFromPublicKey_ex(&cert, ED25519_TYPE,
        &ed25519Key), 0);
    wc_ed25519_free(&ed25519Key);
#endif

#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT)
    /* ED448 */
    XMEMSET(&ed448Key, 0, sizeof(ed448_key));
    ExpectIntEQ(wc_ed448_init(&ed448Key), 0);
    ExpectIntEQ(wc_ed448_make_key(&rng, ED448_KEY_SIZE, &ed448Key), 0);
    ExpectIntEQ(wc_SetAuthKeyIdFromPublicKey_ex(&cert, ED448_TYPE, &ed448Key),
        0);
    wc_ed448_free(&ed448Key);
#endif

    DoExpectIntEQ(wc_FreeRng(&rng), 0);
#endif /* defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN)*/
    return EXPECT_RESULT();
} /* End test_wc_SetAuthKeyIdFromPublicKey_ex*/

/*
 * Testing wc_PKCS7_New()
 */
static int test_wc_PKCS7_New(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7)
    PKCS7* pkcs7 = NULL;

    ExpectNotNull(pkcs7 = wc_PKCS7_New(NULL, testDevId));
    wc_PKCS7_Free(pkcs7);
#endif
    return EXPECT_RESULT();
} /* END test-wc_PKCS7_New */

/*
 * Testing wc_PKCS7_Init()
 */
static int test_wc_PKCS7_Init(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7)
    PKCS7* pkcs7 = NULL;
    void*  heap = NULL;

    ExpectNotNull(pkcs7 = wc_PKCS7_New(heap, testDevId));

    ExpectIntEQ(wc_PKCS7_Init(pkcs7, heap, testDevId), 0);
    /* Pass in bad args. */
    ExpectIntEQ(wc_PKCS7_Init(NULL, heap, testDevId), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    wc_PKCS7_Free(pkcs7);
#endif
    return EXPECT_RESULT();
} /* END test-wc_PKCS7_Init */


/*
 * Testing wc_PKCS7_InitWithCert()
 */
static int test_wc_PKCS7_InitWithCert(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7)
    PKCS7* pkcs7 = NULL;

#ifndef NO_RSA
    #if defined(USE_CERT_BUFFERS_2048)
        unsigned char    cert[sizeof(client_cert_der_2048)];
        int              certSz = (int)sizeof(cert);

        XMEMSET(cert, 0, certSz);
        XMEMCPY(cert, client_cert_der_2048, sizeof(client_cert_der_2048));
    #elif defined(USE_CERT_BUFFERS_1024)
        unsigned char    cert[sizeof(client_cert_der_1024)];
        int              certSz = (int)sizeof(cert);

        XMEMSET(cert, 0, certSz);
        XMEMCPY(cert, client_cert_der_1024, sizeof_client_cert_der_1024);
    #else
        unsigned char   cert[ONEK_BUF];
        XFILE           fp = XBADFILE;
        int             certSz;

        ExpectTrue((fp = XFOPEN("./certs/1024/client-cert.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(certSz = (int)XFREAD(cert, 1, sizeof_client_cert_der_1024,
            fp), 0);
        if (fp != XBADFILE)
            XFCLOSE(fp);
    #endif
#elif defined(HAVE_ECC)
    #if defined(USE_CERT_BUFFERS_256)
        unsigned char    cert[sizeof(cliecc_cert_der_256)];
        int              certSz = (int)sizeof(cert);

        XMEMSET(cert, 0, certSz);
        XMEMCPY(cert, cliecc_cert_der_256, sizeof(cliecc_cert_der_256));
    #else
        unsigned char   cert[ONEK_BUF];
        XFILE           fp = XBADFILE;
        int             certSz;

        ExpectTrue((fp = XFOPEN("./certs/client-ecc-cert.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(certSz = (int)XFREAD(cert, 1, sizeof(cliecc_cert_der_256),
            fp), 0);
        if (fp != XBADFILE)
            XFCLOSE(fp);
    #endif
#else
        #error PKCS7 requires ECC or RSA
#endif

#ifdef HAVE_ECC
    {
    /* bad test case from ZD 11011, malformed cert gives bad ECC key */
        static unsigned char certWithInvalidEccKey[] = {
        0x30, 0x82, 0x03, 0x5F, 0x30, 0x82, 0x03, 0x04, 0xA0, 0x03, 0x02, 0x01,
        0x02, 0x02, 0x14, 0x61, 0xB3, 0x1E, 0x59, 0xF3, 0x68, 0x6C, 0xA4, 0x79,
        0x42, 0x83, 0x2F, 0x1A, 0x50, 0x71, 0x03, 0xBE, 0x31, 0xAA, 0x2C, 0x30,
        0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02, 0x30,
        0x81, 0x8D, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13,
        0x02, 0x55, 0x53, 0x31, 0x0F, 0x30, 0x0D, 0x06, 0x03, 0x55, 0x04, 0x08,
        0x0C, 0x06, 0x4F, 0x72, 0x65, 0x67, 0x6F, 0x6E, 0x31, 0x0E, 0x30, 0x0C,
        0x06, 0x03, 0x55, 0x04, 0x07, 0x0C, 0x05, 0x53, 0x61, 0x6C, 0x65, 0x6D,
        0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x0C, 0x0A, 0x43,
        0x6C, 0x69, 0x65, 0x6E, 0x74, 0x20, 0x45, 0x43, 0x43, 0x31, 0x0D, 0x30,
        0x0B, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x0C, 0x04, 0x46, 0x61, 0x73, 0x74,
        0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x0F, 0x77,
        0x77, 0x77, 0x2E, 0x77, 0x6F, 0x6C, 0x66, 0x73, 0x73, 0x6C, 0x2E, 0x63,
        0x6F, 0x6D, 0x31, 0x1F, 0x30, 0x1D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86,
        0xF7, 0x0D, 0x01, 0x09, 0x01, 0x16, 0x10, 0x69, 0x6E, 0x66, 0x6F, 0x40,
        0x77, 0x6F, 0x6C, 0x66, 0x73, 0x73, 0x6C, 0x2E, 0x63, 0x6F, 0x6D, 0x30,
        0x1E, 0x17, 0x0D, 0x32, 0x30, 0x30, 0x36, 0x31, 0x39, 0x31, 0x33, 0x32,
        0x33, 0x34, 0x31, 0x5A, 0x17, 0x0D, 0x32, 0x33, 0x30, 0x33, 0x31, 0x36,
        0x31, 0x33, 0x32, 0x33, 0x34, 0x31, 0x5A, 0x30, 0x81, 0x8D, 0x31, 0x0B,
        0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31,
        0x0F, 0x30, 0x0D, 0x06, 0x03, 0x55, 0x04, 0x08, 0x0C, 0x06, 0x4F, 0x72,
        0x65, 0x67, 0x6F, 0x6E, 0x31, 0x0E, 0x30, 0x0C, 0x06, 0x03, 0x55, 0x04,
        0x07, 0x0C, 0x05, 0x53, 0x61, 0x6C, 0x65, 0x6D, 0x31, 0x13, 0x30, 0x11,
        0x06, 0x03, 0x55, 0x04, 0x0A, 0x0C, 0x0A, 0x43, 0x6C, 0x69, 0x65, 0x6E,
        0x74, 0x20, 0x45, 0x43, 0x43, 0x31, 0x0D, 0x30, 0x0B, 0x06, 0x03, 0x55,
        0x04, 0x0B, 0x0C, 0x04, 0x46, 0x61, 0x73, 0x74, 0x31, 0x18, 0x30, 0x26,
        0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x0F, 0x77, 0x77, 0x77, 0x2E, 0x77,
        0x6F, 0x6C, 0x66, 0x73, 0x73, 0x6C, 0x2E, 0x63, 0x6F, 0x6D, 0x31, 0x1F,
        0x30, 0x1D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09,
        0x01, 0x16, 0x10, 0x69, 0x6E, 0x66, 0x6F, 0x40, 0x77, 0x6F, 0x6C, 0x66,
        0x73, 0x73, 0x6C, 0x2E, 0x63, 0x6F, 0x6D, 0x30, 0x59, 0x30, 0x13, 0x06,
        0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x08, 0x2A, 0x86,
        0x48, 0xCE, 0x3D, 0x03, 0x01, 0x07, 0x03, 0x02, 0x00, 0x04, 0x55, 0xBF,
        0xF4, 0x0F, 0x44, 0x50, 0x9A, 0x3D, 0xCE, 0x9B, 0xB7, 0xF0, 0xC5, 0x4D,
        0xF5, 0x70, 0x7B, 0xD4, 0xEC, 0x24, 0x8E, 0x19, 0x80, 0xEC, 0x5A, 0x4C,
        0xA2, 0x24, 0x03, 0x62, 0x2C, 0x9B, 0xDA, 0xEF, 0xA2, 0x35, 0x12, 0x43,
        0x84, 0x76, 0x16, 0xC6, 0x56, 0x95, 0x06, 0xCC, 0x01, 0xA9, 0xBD, 0xF6,
        0x75, 0x1A, 0x42, 0xF7, 0xBD, 0xA9, 0xB2, 0x36, 0x22, 0x5F, 0xC7, 0x5D,
        0x7F, 0xB4, 0xA3, 0x82, 0x01, 0x3E, 0x30, 0x82, 0x01, 0x3A, 0x30, 0x1D,
        0x06, 0x03, 0x55, 0x1D, 0x0E, 0x04, 0x16, 0x04, 0x14, 0xEB, 0xD4, 0x4B,
        0x59, 0x6B, 0x95, 0x61, 0x3F, 0x51, 0x57, 0xB6, 0x04, 0x4D, 0x89, 0x41,
        0x88, 0x44, 0x5C, 0xAB, 0xF2, 0x30, 0x81, 0xCD, 0x06, 0x03, 0x55, 0x1D,
        0x23, 0x04, 0x81, 0xC5, 0x30, 0x81, 0xC2, 0x80, 0x14, 0xEB, 0xD4, 0x4B,
        0x59, 0x72, 0x95, 0x61, 0x3F, 0x51, 0x57, 0xB6, 0x04, 0x4D, 0x89, 0x41,
        0x88, 0x44, 0x5C, 0xAB, 0xF2, 0xA1, 0x81, 0x93, 0xA4, 0x81, 0x90, 0x30,
        0x81, 0x8D, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13,
        0x02, 0x55, 0x53, 0x31, 0x0F, 0x30, 0x0D, 0x06, 0x03, 0x55, 0x08, 0x08,
        0x0C, 0x06, 0x4F, 0x72, 0x65, 0x67, 0x6F, 0x6E, 0x31, 0x0E, 0x30, 0x0C,
        0x06, 0x03, 0x55, 0x04, 0x07, 0x0C, 0x05, 0x53, 0x61, 0x6C, 0x65, 0x6D,
        0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x0C, 0x0A, 0x43,
        0x6C, 0x69, 0x65, 0x6E, 0x74, 0x20, 0x45, 0x43, 0x43, 0x31, 0x0D, 0x30,
        0x0B, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x0C, 0x04, 0x46, 0x61, 0x73, 0x74,
        0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x0F, 0x77,
        0x77, 0x77, 0x2E, 0x77, 0x6F, 0x6C, 0x66, 0x73, 0x73, 0x6C, 0x2E, 0x63,
        0x6F, 0x6D, 0x30, 0x1F, 0x30, 0x1D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86,
        0xF7, 0x0D, 0x01, 0x09, 0x01, 0x16, 0x10, 0x69, 0x6E, 0x66, 0x6F, 0x40,
        0x77, 0x6F, 0x6C, 0x66, 0x73, 0x73, 0x6C, 0x2E, 0x63, 0x6F, 0x6D, 0x82,
        0x14, 0x61, 0xB3, 0x1E, 0x59, 0xF3, 0x68, 0x6C, 0xA4, 0x79, 0x42, 0x83,
        0x2F, 0x1A, 0x50, 0x71, 0x03, 0xBE, 0x32, 0xAA, 0x2C, 0x30, 0x0C, 0x06,
        0x03, 0x55, 0x1D, 0x13, 0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xFF, 0x30,
        0x1C, 0x06, 0x03, 0x55, 0x1D, 0x11, 0x04, 0x15, 0x30, 0x13, 0x82, 0x0B,
        0x65, 0x78, 0x61, 0x6D, 0x70, 0x6C, 0x65, 0x2E, 0x63, 0x6F, 0x6D, 0x87,
        0x04, 0x23, 0x00, 0x00, 0x01, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x1D, 0x25,
        0x04, 0x16, 0x30, 0x14, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07,
        0x03, 0x01, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x02,
        0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02,
        0x03, 0x49, 0x00, 0x30, 0x46, 0x02, 0x21, 0x00, 0xE4, 0xA0, 0x23, 0x26,
        0x2B, 0x0B, 0x42, 0x0F, 0x97, 0x37, 0x6D, 0xCB, 0x14, 0x23, 0xC3, 0xC3,
        0xE6, 0x44, 0xCF, 0x5F, 0x4C, 0x26, 0xA3, 0x72, 0x64, 0x7A, 0x9C, 0xCB,
        0x64, 0xAB, 0xA6, 0xBE, 0x02, 0x21, 0x00, 0xAA, 0xC5, 0xA3, 0x50, 0xF6,
        0xF1, 0xA5, 0xDB, 0x05, 0xE0, 0x75, 0xD2, 0xF7, 0xBA, 0x49, 0x5F, 0x8F,
        0x7D, 0x1C, 0x44, 0xB1, 0x6E, 0xDF, 0xC8, 0xDA, 0x10, 0x48, 0x2D, 0x53,
        0x08, 0xA8, 0xB4
        };
#endif
        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        /* If initialization is not successful, it's free'd in init func. */
        ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, (byte*)cert, (word32)certSz),
            0);
        wc_PKCS7_Free(pkcs7);
        pkcs7 = NULL;

        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        /* Valid initialization usage. */
        ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);

        /* Pass in bad args. No need free for null checks, free at end.*/
        ExpectIntEQ(wc_PKCS7_InitWithCert(NULL, (byte*)cert, (word32)certSz),
            WC_NO_ERR_TRACE(BAD_FUNC_ARG));
        ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, (word32)certSz),
            WC_NO_ERR_TRACE(BAD_FUNC_ARG));

#ifdef HAVE_ECC
        ExpectIntLT(wc_PKCS7_InitWithCert(pkcs7, certWithInvalidEccKey,
            sizeof(certWithInvalidEccKey)), 0);
    }
#endif

    wc_PKCS7_Free(pkcs7);
#endif
    return EXPECT_RESULT();
} /* END test_wc_PKCS7_InitWithCert */


/*
 * Testing wc_PKCS7_EncodeData()
 */
static int test_wc_PKCS7_EncodeData(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7)
    PKCS7* pkcs7 = NULL;
    byte   output[FOURK_BUF];
    byte   data[] = "My encoded DER cert.";

#ifndef NO_RSA
    #if defined(USE_CERT_BUFFERS_2048)
        unsigned char cert[sizeof(client_cert_der_2048)];
        unsigned char key[sizeof(client_key_der_2048)];
        int certSz = (int)sizeof(cert);
        int keySz = (int)sizeof(key);

        XMEMSET(cert, 0, certSz);
        XMEMSET(key, 0, keySz);
        XMEMCPY(cert, client_cert_der_2048, certSz);
        XMEMCPY(key, client_key_der_2048, keySz);
    #elif defined(USE_CERT_BUFFERS_1024)
        unsigned char cert[sizeof(sizeof_client_cert_der_1024)];
        unsigned char key[sizeof_client_key_der_1024];
        int certSz = (int)sizeof(cert);
        int keySz = (int)sizeof(key);

        XMEMSET(cert, 0, certSz);
        XMEMSET(key, 0, keySz);
        XMEMCPY(cert, client_cert_der_1024, certSz);
        XMEMCPY(key, client_key_der_1024, keySz);
    #else
        unsigned char cert[ONEK_BUF];
        unsigned char key[ONEK_BUF];
        XFILE         fp = XBADFILE;
        int           certSz;
        int           keySz;

        ExpectTrue((fp = XFOPEN("./certs/1024/client-cert.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(certSz = (int)XFREAD(cert, 1, sizeof_client_cert_der_1024,
            fp), 0);
        if (fp != XBADFILE) {
            XFCLOSE(fp);
            fp = XBADFILE;
        }

        ExpectTrue((fp = XFOPEN("./certs/1024/client-key.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(keySz = (int)XFREAD(key, 1, sizeof_client_key_der_1024, fp),
            0);
        if (fp != XBADFILE)
            XFCLOSE(fp);
    #endif
#elif defined(HAVE_ECC)
    #if defined(USE_CERT_BUFFERS_256)
        unsigned char    cert[sizeof(cliecc_cert_der_256)];
        unsigned char    key[sizeof(ecc_clikey_der_256)];
        int              certSz = (int)sizeof(cert);
        int              keySz = (int)sizeof(key);
        XMEMSET(cert, 0, certSz);
        XMEMSET(key, 0, keySz);
        XMEMCPY(cert, cliecc_cert_der_256, sizeof_cliecc_cert_der_256);
        XMEMCPY(key, ecc_clikey_der_256, sizeof_ecc_clikey_der_256);
    #else
        unsigned char   cert[ONEK_BUF];
        unsigned char   key[ONEK_BUF];
        XFILE           fp = XBADFILE;
        int             certSz, keySz;

        ExpectTrue((fp = XFOPEN("./certs/client-ecc-cert.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(certSz = (int)XFREAD(cert, 1, sizeof_cliecc_cert_der_256,
            fp), 0);
        if (fp != XBADFILE) {
            XFCLOSE(fp);
            fp = XBADFILE;
        }

        ExpectTrue((fp = XFOPEN("./certs/client-ecc-key.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(keySz = (int)XFREAD(key, 1, sizeof_ecc_clikey_der_256, fp),
            0);
        if (fp != XBADFILE)
            XFCLOSE(fp);
    #endif
#endif

    XMEMSET(output, 0, sizeof(output));

    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);

    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, (byte*)cert, (word32)certSz), 0);

    if (pkcs7 != NULL) {
        pkcs7->content = data;
        pkcs7->contentSz = sizeof(data);
        pkcs7->privateKey = key;
        pkcs7->privateKeySz = (word32)keySz;
    }
    ExpectIntGT(wc_PKCS7_EncodeData(pkcs7, output, (word32)sizeof(output)), 0);

    /* Test bad args. */
    ExpectIntEQ(wc_PKCS7_EncodeData(NULL, output, (word32)sizeof(output)),
                                                            WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_PKCS7_EncodeData(pkcs7, NULL, (word32)sizeof(output)),
                                                            WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_PKCS7_EncodeData(pkcs7, output, 5), WC_NO_ERR_TRACE(BUFFER_E));

    wc_PKCS7_Free(pkcs7);
#endif
    return EXPECT_RESULT();
}  /* END test_wc_PKCS7_EncodeData */


#if defined(HAVE_PKCS7) && defined(HAVE_PKCS7_RSA_RAW_SIGN_CALLBACK) && \
    !defined(NO_RSA) && !defined(NO_SHA256)
/* RSA sign raw digest callback */
static int rsaSignRawDigestCb(PKCS7* pkcs7, byte* digest, word32 digestSz,
                              byte* out, word32 outSz, byte* privateKey,
                              word32 privateKeySz, int devid, int hashOID)
{
    /* specific DigestInfo ASN.1 encoding prefix for a SHA2565 digest */
    byte digInfoEncoding[] = {
        0x30, 0x31, 0x30, 0x0d, 0x06, 0x09, 0x60, 0x86,
        0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05,
        0x00, 0x04, 0x20
    };

    int ret;
    byte digestInfo[ONEK_BUF];
    byte sig[FOURK_BUF];
    word32 digestInfoSz = 0;
    word32 idx = 0;
    RsaKey rsa;

    /* SHA-256 required only for this example callback due to above
     * digInfoEncoding[] */
    if (pkcs7 == NULL || digest == NULL || out == NULL ||
        (sizeof(digestInfo) < sizeof(digInfoEncoding) + digestSz) ||
        (hashOID != SHA256h)) {
        return -1;
    }

    /* build DigestInfo */
    XMEMCPY(digestInfo, digInfoEncoding, sizeof(digInfoEncoding));
    digestInfoSz += sizeof(digInfoEncoding);
    XMEMCPY(digestInfo + digestInfoSz, digest, digestSz);
    digestInfoSz += digestSz;

    /* set up RSA key */
    ret = wc_InitRsaKey_ex(&rsa, pkcs7->heap, devid);
    if (ret != 0) {
        return ret;
    }

    ret = wc_RsaPrivateKeyDecode(privateKey, &idx, &rsa, privateKeySz);

    /* sign DigestInfo */
    if (ret == 0) {
        ret = wc_RsaSSL_Sign(digestInfo, digestInfoSz, sig, sizeof(sig),
                             &rsa, pkcs7->rng);
        if (ret > 0) {
            if (ret > (int)outSz) {
                /* output buffer too small */
                ret = -1;
            }
            else {
                /* success, ret holds sig size */
                XMEMCPY(out, sig, ret);
            }
        }
    }

    wc_FreeRsaKey(&rsa);

    return ret;
}
#endif

#if defined(HAVE_PKCS7) && defined(ASN_BER_TO_DER)
typedef struct encodeSignedDataStream {
    byte out[FOURK_BUF*3];
    int  idx;
    word32 outIdx;
    word32 chunkSz; /* max amount of data to be returned */
} encodeSignedDataStream;


/* content is 8k of partially created bundle */
static int GetContentCB(PKCS7* pkcs7, byte** content, void* ctx)
{
    int ret = 0;
    encodeSignedDataStream* strm = (encodeSignedDataStream*)ctx;

    if (strm->outIdx  < pkcs7->contentSz) {
        ret = (pkcs7->contentSz > strm->outIdx + strm->chunkSz)?
                strm->chunkSz : pkcs7->contentSz - strm->outIdx;
        *content = strm->out + strm->outIdx;
        strm->outIdx += ret;
    }

    (void)pkcs7;
    return ret;
}

static int StreamOutputCB(PKCS7* pkcs7, const byte* output, word32 outputSz,
    void* ctx)
{
    encodeSignedDataStream* strm = (encodeSignedDataStream*)ctx;

    XMEMCPY(strm->out + strm->idx, output, outputSz);
    strm->idx += outputSz;
    (void)pkcs7;
    return 0;
}
#endif


/*
 * Testing wc_PKCS7_EncodeSignedData()
 */
static int test_wc_PKCS7_EncodeSignedData(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7)
    PKCS7* pkcs7 = NULL;
    WC_RNG rng;
    byte   output[FOURK_BUF];
    byte   badOut[1];
    word32 outputSz = (word32)sizeof(output);
    word32 badOutSz = 0;
    byte   data[] = "Test data to encode.";
#ifndef NO_RSA
    int    encryptOid = RSAk;
    #if defined(USE_CERT_BUFFERS_2048)
        byte        key[sizeof(client_key_der_2048)];
        byte        cert[sizeof(client_cert_der_2048)];
        word32      keySz = (word32)sizeof(key);
        word32      certSz = (word32)sizeof(cert);
        XMEMSET(key, 0, keySz);
        XMEMSET(cert, 0, certSz);
        XMEMCPY(key, client_key_der_2048, keySz);
        XMEMCPY(cert, client_cert_der_2048, certSz);
    #elif defined(USE_CERT_BUFFERS_1024)
        byte        key[sizeof_client_key_der_1024];
        byte        cert[sizeof(sizeof_client_cert_der_1024)];
        word32      keySz = (word32)sizeof(key);
        word32      certSz = (word32)sizeof(cert);
        XMEMSET(key, 0, keySz);
        XMEMSET(cert, 0, certSz);
        XMEMCPY(key, client_key_der_1024, keySz);
        XMEMCPY(cert, client_cert_der_1024, certSz);
    #else
        unsigned char   cert[ONEK_BUF];
        unsigned char   key[ONEK_BUF];
        XFILE           fp = XBADFILE;
        int             certSz;
        int             keySz;

        ExpectTrue((fp = XFOPEN("./certs/1024/client-cert.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(certSz = (int)XFREAD(cert, 1, sizeof_client_cert_der_1024,
            fp), 0);
        if (fp != XBADFILE) {
            XFCLOSE(fp);
            fp = XBADFILE;
        }

        ExpectTrue((fp = XFOPEN("./certs/1024/client-key.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(keySz = (int)XFREAD(key, 1, sizeof_client_key_der_1024, fp),
            0);
        if (fp != XBADFILE)
            XFCLOSE(fp);
    #endif
#elif defined(HAVE_ECC)
    int    encryptOid = ECDSAk;
    #if defined(USE_CERT_BUFFERS_256)
        unsigned char    cert[sizeof(cliecc_cert_der_256)];
        unsigned char    key[sizeof(ecc_clikey_der_256)];
        int              certSz = (int)sizeof(cert);
        int              keySz = (int)sizeof(key);
        XMEMSET(cert, 0, certSz);
        XMEMSET(key, 0, keySz);
        XMEMCPY(cert, cliecc_cert_der_256, certSz);
        XMEMCPY(key, ecc_clikey_der_256, keySz);
    #else
        unsigned char   cert[ONEK_BUF];
        unsigned char   key[ONEK_BUF];
        XFILE           fp = XBADFILE;
        int             certSz;
        int             keySz;

        ExpectTrue((fp = XFOPEN("./certs/client-ecc-cert.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(certSz = (int)XFREAD(cert, 1, ONEK_BUF, fp), 0);
        if (fp != XBADFILE) {
            XFCLOSE(fp);
            fp = XBADFILE;
        }

        ExpectTrue((fp = XFOPEN("./certs/client-ecc-key.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(keySz = (int)XFREAD(key, 1, ONEK_BUF, fp), 0);
        if (fp != XBADFILE)
            XFCLOSE(fp);
    #endif
#endif

    XMEMSET(&rng, 0, sizeof(WC_RNG));

    XMEMSET(output, 0, outputSz);
    ExpectIntEQ(wc_InitRng(&rng), 0);

    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);

    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, certSz), 0);

    if (pkcs7 != NULL) {
        pkcs7->content = data;
        pkcs7->contentSz = (word32)sizeof(data);
        pkcs7->privateKey = key;
        pkcs7->privateKeySz = (word32)sizeof(key);
        pkcs7->encryptOID = encryptOid;
    #ifdef NO_SHA
        pkcs7->hashOID = SHA256h;
    #else
        pkcs7->hashOID = SHAh;
    #endif
        pkcs7->rng = &rng;
    }

    ExpectIntGT(wc_PKCS7_EncodeSignedData(pkcs7, output, outputSz), 0);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz), 0);

#if defined(ASN_BER_TO_DER) && !defined(NO_RSA)
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

    /* reinitialize and test setting stream mode */
    {
        int signedSz = 0, i;
        encodeSignedDataStream strm;
        static const int numberOfChunkSizes = 4;
        static const word32 chunkSizes[] = { 4080, 4096, 5000, 9999 };
        /* chunkSizes were chosen to test around the default 4096 octet string
         * size used in pkcs7.c */

        XMEMSET(&strm, 0, sizeof(strm));

        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);

        ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, certSz), 0);

        if (pkcs7 != NULL) {
            pkcs7->content = data;
            pkcs7->contentSz = (word32)sizeof(data);
            pkcs7->privateKey = key;
            pkcs7->privateKeySz = (word32)sizeof(key);
            pkcs7->encryptOID = encryptOid;
        #ifdef NO_SHA
            pkcs7->hashOID = SHA256h;
        #else
            pkcs7->hashOID = SHAh;
        #endif
            pkcs7->rng = &rng;
        }
        ExpectIntEQ(wc_PKCS7_GetStreamMode(pkcs7), 0);
        ExpectIntEQ(wc_PKCS7_SetStreamMode(pkcs7, 1, NULL, NULL, NULL), 0);
        ExpectIntEQ(wc_PKCS7_SetStreamMode(NULL, 1, NULL, NULL, NULL),
            WC_NO_ERR_TRACE(BAD_FUNC_ARG));
        ExpectIntEQ(wc_PKCS7_GetStreamMode(pkcs7), 1);

        ExpectIntGT(signedSz = wc_PKCS7_EncodeSignedData(pkcs7, output,
            outputSz), 0);
        wc_PKCS7_Free(pkcs7);
        pkcs7 = NULL;

        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);

        /* use exact signed buffer size since BER encoded */
        ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output,
            (word32)signedSz), 0);
        wc_PKCS7_Free(pkcs7);

        /* now try with using callbacks for IO */
        for (i = 0; i < numberOfChunkSizes; i++) {
            strm.idx    = 0;
            strm.outIdx = 0;
            strm.chunkSz = chunkSizes[i];

            ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
            ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);

            ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, certSz), 0);

            if (pkcs7 != NULL) {
                pkcs7->contentSz  = 10000;
                pkcs7->privateKey = key;
                pkcs7->privateKeySz = (word32)sizeof(key);
                pkcs7->encryptOID = encryptOid;
            #ifdef NO_SHA
                pkcs7->hashOID = SHA256h;
            #else
                pkcs7->hashOID = SHAh;
            #endif
                pkcs7->rng = &rng;
            }
            ExpectIntEQ(wc_PKCS7_SetStreamMode(pkcs7, 1, GetContentCB,
                StreamOutputCB, (void*)&strm), 0);

            ExpectIntGT(signedSz = wc_PKCS7_EncodeSignedData(pkcs7, NULL, 0),
                0);
            wc_PKCS7_Free(pkcs7);
            pkcs7 = NULL;

            ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
            ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);

            /* use exact signed buffer size since BER encoded */
            ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, strm.out,
                (word32)signedSz), 0);
            wc_PKCS7_Free(pkcs7);
            pkcs7 = NULL;
        }
    }
#endif
#ifndef NO_PKCS7_STREAM
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

    {
        word32 z;
        int ret;

        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);

        /* test for streaming mode */
        ret = -1;
        for (z = 0; z < outputSz && ret != 0; z++) {
            ret = wc_PKCS7_VerifySignedData(pkcs7, output + z, 1);
            if (ret < 0){
                ExpectIntEQ(ret, WC_NO_ERR_TRACE(WC_PKCS7_WANT_READ_E));
            }
        }
        ExpectIntEQ(ret, 0);
        ExpectIntNE(pkcs7->contentSz, 0);
        ExpectNotNull(pkcs7->contentDynamic);
    }
#endif /* !NO_PKCS7_STREAM */


    /* Pass in bad args. */
    ExpectIntEQ(wc_PKCS7_EncodeSignedData(NULL, output, outputSz),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_PKCS7_EncodeSignedData(pkcs7, NULL, outputSz), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_PKCS7_EncodeSignedData(pkcs7, badOut,
                                badOutSz), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    if (pkcs7 != NULL) {
        pkcs7->hashOID = 0; /* bad hashOID */
    }
    ExpectIntEQ(wc_PKCS7_EncodeSignedData(pkcs7, output, outputSz),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));

#if defined(HAVE_PKCS7) && defined(HAVE_PKCS7_RSA_RAW_SIGN_CALLBACK) && \
    !defined(NO_RSA) && !defined(NO_SHA256)
    /* test RSA sign raw digest callback, if using RSA and compiled in.
     * Example callback assumes SHA-256, so only run test if compiled in. */
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, certSz), 0);

    if (pkcs7 != NULL) {
        pkcs7->content = data;
        pkcs7->contentSz = (word32)sizeof(data);
        pkcs7->privateKey = key;
        pkcs7->privateKeySz = (word32)sizeof(key);
        pkcs7->encryptOID = RSAk;
        pkcs7->hashOID = SHA256h;
        pkcs7->rng = &rng;
    }

    ExpectIntEQ(wc_PKCS7_SetRsaSignRawDigestCb(pkcs7, rsaSignRawDigestCb), 0);

    ExpectIntGT(wc_PKCS7_EncodeSignedData(pkcs7, output, outputSz), 0);
#endif

    wc_PKCS7_Free(pkcs7);
    DoExpectIntEQ(wc_FreeRng(&rng), 0);

#endif
    return EXPECT_RESULT();
} /* END test_wc_PKCS7_EncodeSignedData */


/*
 * Testing wc_PKCS7_EncodeSignedData_ex() and wc_PKCS7_VerifySignedData_ex()
 */
static int test_wc_PKCS7_EncodeSignedData_ex(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7)
    int        i;
    PKCS7*     pkcs7 = NULL;
    WC_RNG     rng;
    byte       outputHead[FOURK_BUF/2];
    byte       outputFoot[FOURK_BUF/2];
    word32     outputHeadSz = (word32)sizeof(outputHead);
    word32     outputFootSz = (word32)sizeof(outputFoot);
    byte       data[FOURK_BUF];
    wc_HashAlg hash;
#ifdef NO_SHA
    enum wc_HashType hashType = WC_HASH_TYPE_SHA256;
#else
    enum wc_HashType hashType = WC_HASH_TYPE_SHA;
#endif
    byte        hashBuf[WC_MAX_DIGEST_SIZE];
    word32      hashSz = (word32)wc_HashGetDigestSize(hashType);

#ifndef NO_RSA
    #if defined(USE_CERT_BUFFERS_2048)
        byte        key[sizeof(client_key_der_2048)];
        byte        cert[sizeof(client_cert_der_2048)];
        word32      keySz = (word32)sizeof(key);
        word32      certSz = (word32)sizeof(cert);
        XMEMSET(key, 0, keySz);
        XMEMSET(cert, 0, certSz);
        XMEMCPY(key, client_key_der_2048, keySz);
        XMEMCPY(cert, client_cert_der_2048, certSz);
    #elif defined(USE_CERT_BUFFERS_1024)
        byte        key[sizeof_client_key_der_1024];
        byte        cert[sizeof(sizeof_client_cert_der_1024)];
        word32      keySz = (word32)sizeof(key);
        word32      certSz = (word32)sizeof(cert);
        XMEMSET(key, 0, keySz);
        XMEMSET(cert, 0, certSz);
        XMEMCPY(key, client_key_der_1024, keySz);
        XMEMCPY(cert, client_cert_der_1024, certSz);
    #else
        unsigned char  cert[ONEK_BUF];
        unsigned char  key[ONEK_BUF];
        XFILE          fp = XBADFILE;
        int            certSz;
        int            keySz;

        ExpectTure((fp = XFOPEN("./certs/1024/client-cert.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(certSz = (int)XFREAD(cert, 1, sizeof_client_cert_der_1024,
            fp), 0);
        if (fp != XBADFILE) {
            XFCLOSE(fp);
            fp = XBADFILE;
        }

        ExpectTrue((fp = XFOPEN("./certs/1024/client-key.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(keySz = (int)XFREAD(key, 1, sizeof_client_key_der_1024, fp),
            0);
        if (fp != XBADFILE)
            XFCLOSE(fp);
    #endif
#elif defined(HAVE_ECC)
    #if defined(USE_CERT_BUFFERS_256)
        unsigned char   cert[sizeof(cliecc_cert_der_256)];
        unsigned char   key[sizeof(ecc_clikey_der_256)];
        int             certSz = (int)sizeof(cert);
        int             keySz = (int)sizeof(key);

        XMEMSET(cert, 0, certSz);
        XMEMSET(key, 0, keySz);
        XMEMCPY(cert, cliecc_cert_der_256, sizeof_cliecc_cert_der_256);
        XMEMCPY(key, ecc_clikey_der_256, sizeof_ecc_clikey_der_256);
    #else
        unsigned char cert[ONEK_BUF];
        unsigned char key[ONEK_BUF];
        XFILE         fp = XBADFILE;
        int           certSz;
        int           keySz;

        ExpectTrue((fp = XFOPEN("./certs/client-ecc-cert.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(certSz = (int)XFREAD(cert, 1, sizeof_cliecc_cert_der_256,
            fp), 0);
        if (fp != XBADFILE) {
            XFCLOSE(fp);
            fp = XBADFILE;
        }

        ExpectTrue((fp = XFOPEN("./certs/client-ecc-key.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(keySz = (int)XFREAD(key, 1, sizeof_ecc_clikey_der_256, fp),
            0);
        if (fp != XBADFILE)
            XFCLOSE(fp);
    #endif
#endif

    XMEMSET(&rng, 0, sizeof(WC_RNG));

    /* initialize large data with sequence */
    for (i=0; i<(int)sizeof(data); i++)
        data[i] = i & 0xff;

    XMEMSET(outputHead, 0, outputHeadSz);
    XMEMSET(outputFoot, 0, outputFootSz);
    ExpectIntEQ(wc_InitRng(&rng), 0);

    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);

    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, certSz), 0);

    if (pkcs7 != NULL) {
        pkcs7->content = NULL; /* not used for ex */
        pkcs7->contentSz = (word32)sizeof(data);
        pkcs7->privateKey = key;
        pkcs7->privateKeySz = (word32)sizeof(key);
        pkcs7->encryptOID = RSAk;
    #ifdef NO_SHA
        pkcs7->hashOID = SHA256h;
    #else
        pkcs7->hashOID = SHAh;
    #endif
        pkcs7->rng = &rng;
    }

    /* calculate hash for content */
    XMEMSET(&hash, 0, sizeof(wc_HashAlg));
    ExpectIntEQ(wc_HashInit(&hash, hashType), 0);
    ExpectIntEQ(wc_HashUpdate(&hash, hashType, data, sizeof(data)), 0);
    ExpectIntEQ(wc_HashFinal(&hash, hashType, hashBuf), 0);
    DoExpectIntEQ(wc_HashFree(&hash, hashType), 0);

    /* Perform PKCS7 sign using hash directly */
    ExpectIntEQ(wc_PKCS7_EncodeSignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, &outputHeadSz, outputFoot, &outputFootSz), 0);
    ExpectIntGT(outputHeadSz, 0);
    ExpectIntGT(outputFootSz, 0);

    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);

    /* required parameter even on verify when using _ex, if using outputHead
     * and outputFoot */
    if (pkcs7 != NULL) {
        pkcs7->contentSz = (word32)sizeof(data);
    }
    ExpectIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, outputHeadSz, outputFoot, outputFootSz), 0);

    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

    /* assembly complete PKCS7 sign and use normal verify */
    {
        byte* output = NULL;
        word32 outputSz = 0;
    #ifndef NO_PKCS7_STREAM
        word32 z;
        int ret;
    #endif /* !NO_PKCS7_STREAM */

        ExpectNotNull(output = (byte*)XMALLOC(
            outputHeadSz + sizeof(data) + outputFootSz, HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER));
        if (output != NULL) {
            XMEMCPY(&output[outputSz], outputHead, outputHeadSz);
            outputSz += outputHeadSz;
            XMEMCPY(&output[outputSz], data, sizeof(data));
            outputSz += sizeof(data);
            XMEMCPY(&output[outputSz], outputFoot, outputFootSz);
            outputSz += outputFootSz;
        }

        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
        ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz), 0);

    #ifndef NO_PKCS7_STREAM
        wc_PKCS7_Free(pkcs7);
        pkcs7 = NULL;

        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);

        /* test for streaming mode */
        ret = -1;
        for (z = 0; z < outputSz && ret != 0; z++) {
            ret = wc_PKCS7_VerifySignedData(pkcs7, output + z, 1);
            if (ret < 0){
                ExpectIntEQ(ret, WC_NO_ERR_TRACE(WC_PKCS7_WANT_READ_E));
            }
        }
        ExpectIntEQ(ret, 0);
        ExpectIntNE(pkcs7->contentSz, 0);
        ExpectNotNull(pkcs7->contentDynamic);

        wc_PKCS7_Free(pkcs7);
        pkcs7 = NULL;
        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    #endif /* !NO_PKCS7_STREAM */

        XFREE(output, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }

    /* Pass in bad args. */
    ExpectIntEQ(wc_PKCS7_EncodeSignedData_ex(NULL, hashBuf, hashSz, outputHead,
        &outputHeadSz, outputFoot, &outputFootSz), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_PKCS7_EncodeSignedData_ex(pkcs7, NULL, hashSz, outputHead,
        &outputHeadSz, outputFoot, &outputFootSz), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_PKCS7_EncodeSignedData_ex(pkcs7, hashBuf, 0, outputHead,
        &outputHeadSz, outputFoot, &outputFootSz), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_PKCS7_EncodeSignedData_ex(pkcs7, hashBuf, hashSz, NULL,
        &outputHeadSz, outputFoot, &outputFootSz), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_PKCS7_EncodeSignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, NULL, outputFoot, &outputFootSz), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_PKCS7_EncodeSignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, &outputHeadSz, NULL, &outputFootSz), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_PKCS7_EncodeSignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, &outputHeadSz, outputFoot, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    if (pkcs7 != NULL) {
        pkcs7->hashOID = 0; /* bad hashOID */
    }
    ExpectIntEQ(wc_PKCS7_EncodeSignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, &outputHeadSz, outputFoot, &outputFootSz), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    ExpectIntEQ(wc_PKCS7_VerifySignedData_ex(NULL, hashBuf, hashSz, outputHead,
        outputHeadSz, outputFoot, outputFootSz), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    ExpectIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, NULL, hashSz, outputHead,
        outputHeadSz, outputFoot, outputFootSz), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
#ifndef NO_PKCS7_STREAM
    ExpectIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, 0, outputHead,
        outputHeadSz, outputFoot, outputFootSz), WC_NO_ERR_TRACE(WC_PKCS7_WANT_READ_E));
#else
    ExpectIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, 0, outputHead,
        outputHeadSz, outputFoot, outputFootSz), WC_NO_ERR_TRACE(BUFFER_E));
#endif
    ExpectIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz, NULL,
        outputHeadSz, outputFoot, outputFootSz), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
#ifndef NO_PKCS7_STREAM
    /* can pass in 0 buffer length with streaming API */
    ExpectIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, 0, outputFoot, outputFootSz), WC_NO_ERR_TRACE(WC_PKCS7_WANT_READ_E));
#else
    ExpectIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, 0, outputFoot, outputFootSz), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
#endif
    ExpectIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, outputHeadSz, NULL, outputFootSz), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
#ifndef NO_PKCS7_STREAM
    ExpectIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, outputHeadSz, outputFoot, 0), WC_NO_ERR_TRACE(WC_PKCS7_WANT_READ_E));
#else
    ExpectIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz,
        outputHead, outputHeadSz, outputFoot, 0), WC_NO_ERR_TRACE(BUFFER_E));
#endif

    wc_PKCS7_Free(pkcs7);
    DoExpectIntEQ(wc_FreeRng(&rng), 0);
#endif
    return EXPECT_RESULT();
} /* END test_wc_PKCS7_EncodeSignedData_ex */


#if defined(HAVE_PKCS7) && !defined(NO_FILESYSTEM)

/**
 * Loads certs/keys from files or buffers into the argument buffers,
 * helper function called by CreatePKCS7SignedData().
 *
 * Returns 0 on success, negative on error.
 */
static int LoadPKCS7SignedDataCerts(
        int useIntermediateCertChain, int pkAlgoType,
        byte* intCARoot, word32* intCARootSz,
        byte* intCA1, word32* intCA1Sz,
        byte* intCA2, word32* intCA2Sz,
        byte* cert, word32* certSz,
        byte* key, word32* keySz)
{
    EXPECT_DECLS;
    int ret = 0;
    XFILE fp = XBADFILE;

#ifndef NO_RSA
    const char* intCARootRSA   = "./certs/ca-cert.der";
    const char* intCA1RSA      = "./certs/intermediate/ca-int-cert.der";
    const char* intCA2RSA      = "./certs/intermediate/ca-int2-cert.der";
    const char* intServCertRSA = "./certs/intermediate/server-int-cert.der";
    const char* intServKeyRSA  = "./certs/server-key.der";

    #if !defined(USE_CERT_BUFFERS_2048) && !defined(USE_CERT_BUFFERS_1024)
        const char* cli1024Cert    = "./certs/1024/client-cert.der";
        const char* cli1024Key     = "./certs/1024/client-key.der";
    #endif
#endif
#ifdef HAVE_ECC
    const char* intCARootECC   = "./certs/ca-ecc-cert.der";
    const char* intCA1ECC      = "./certs/intermediate/ca-int-ecc-cert.der";
    const char* intCA2ECC      = "./certs/intermediate/ca-int2-ecc-cert.der";
    const char* intServCertECC = "./certs/intermediate/server-int-ecc-cert.der";
    const char* intServKeyECC  = "./certs/ecc-key.der";

    #ifndef USE_CERT_BUFFERS_256
        const char* cliEccCert     = "./certs/client-ecc-cert.der";
        const char* cliEccKey      = "./certs/client-ecc-key.der";
    #endif
#endif

    if (cert == NULL || certSz == NULL || key == NULL || keySz == NULL ||
        ((useIntermediateCertChain == 1) &&
        (intCARoot == NULL || intCARootSz == NULL || intCA1 == NULL ||
         intCA1Sz == NULL || intCA2 == NULL || intCA2Sz == NULL))) {
        return BAD_FUNC_ARG;
    }

    /* Read/load certs and keys to use for signing based on PK type and chain */
    switch (pkAlgoType) {
#ifndef NO_RSA
        case RSA_TYPE:
            if (useIntermediateCertChain == 1) {
                ExpectTrue((fp = XFOPEN(intCARootRSA, "rb")) != XBADFILE);
                *intCARootSz = (word32)XFREAD(intCARoot, 1, *intCARootSz, fp);
                if (fp != XBADFILE) {
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*intCARootSz, 0);

                ExpectTrue((fp = XFOPEN(intCA1RSA, "rb")) != XBADFILE);
                if (fp != XBADFILE) {
                    *intCA1Sz = (word32)XFREAD(intCA1, 1, *intCA1Sz, fp);
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*intCA1Sz, 0);

                ExpectTrue((fp = XFOPEN(intCA2RSA, "rb")) != XBADFILE);
                if (fp != XBADFILE) {
                    *intCA2Sz = (word32)XFREAD(intCA2, 1, *intCA2Sz, fp);
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*intCA2Sz, 0);

                ExpectTrue((fp = XFOPEN(intServCertRSA, "rb")) != XBADFILE);
                if (fp != XBADFILE) {
                    *certSz = (word32)XFREAD(cert, 1, *certSz, fp);
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*certSz, 0);

                ExpectTrue((fp = XFOPEN(intServKeyRSA, "rb")) != XBADFILE);
                if (fp != XBADFILE) {
                    *keySz = (word32)XFREAD(key, 1, *keySz, fp);
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*keySz, 0);
            }
            else {
            #if defined(USE_CERT_BUFFERS_2048)
                *keySz  = sizeof_client_key_der_2048;
                *certSz = sizeof_client_cert_der_2048;
                XMEMCPY(key, client_key_der_2048, *keySz);
                XMEMCPY(cert, client_cert_der_2048, *certSz);
            #elif defined(USE_CERT_BUFFERS_1024)
                *keySz  = sizeof_client_key_der_1024;
                *certSz = sizeof_client_cert_der_1024;
                XMEMCPY(key, client_key_der_1024, *keySz);
                XMEMCPY(cert, client_cert_der_1024, *certSz);
            #else
                ExpectTrue((fp = XFOPEN(cli1024Key, "rb")) != XBADFILE);
                if (fp != XBADFILE) {
                    *keySz = (word32)XFREAD(key, 1, *keySz, fp);
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*keySz, 0);

                ExpectTrue((fp = XFOPEN(cli1024Cert, "rb")) != XBADFILE);
                if (fp != XBADFILE) {
                    *certSz = (word32)XFREAD(cert, 1, *certSz, fp);
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*certSz, 0);
            #endif /* USE_CERT_BUFFERS_2048 */
            }
            break;
#endif /* !NO_RSA */
#ifdef HAVE_ECC
        case ECC_TYPE:
            if (useIntermediateCertChain == 1) {
                ExpectTrue((fp = XFOPEN(intCARootECC, "rb")) != XBADFILE);
                if (fp != XBADFILE) {
                    *intCARootSz = (word32)XFREAD(intCARoot, 1, *intCARootSz,
                                                  fp);
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*intCARootSz, 0);

                ExpectTrue((fp = XFOPEN(intCA1ECC, "rb")) != XBADFILE);
                if (fp != XBADFILE) {
                    *intCA1Sz = (word32)XFREAD(intCA1, 1, *intCA1Sz, fp);
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*intCA1Sz, 0);

                ExpectTrue((fp = XFOPEN(intCA2ECC, "rb")) != XBADFILE);
                if (fp != XBADFILE) {
                    *intCA2Sz = (word32)XFREAD(intCA2, 1, *intCA2Sz, fp);
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*intCA2Sz, 0);

                ExpectTrue((fp = XFOPEN(intServCertECC, "rb")) != XBADFILE);
                if (fp != XBADFILE) {
                    *certSz = (word32)XFREAD(cert, 1, *certSz, fp);
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*certSz, 0);

                ExpectTrue((fp = XFOPEN(intServKeyECC, "rb")) != XBADFILE);
                if (fp != XBADFILE) {
                    *keySz = (word32)XFREAD(key, 1, *keySz, fp);
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*keySz, 0);
            }
            else {
            #if defined(USE_CERT_BUFFERS_256)
                *keySz  = sizeof_ecc_clikey_der_256;
                *certSz = sizeof_cliecc_cert_der_256;
                XMEMCPY(key, ecc_clikey_der_256, *keySz);
                XMEMCPY(cert, cliecc_cert_der_256, *certSz);
            #else
                ExpectTrue((fp = XFOPEN(cliEccKey, "rb")) != XBADFILE);
                if (fp != XBADFILE) {
                    *keySz = (word32)XFREAD(key, 1, *keySz, fp);
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*keySz, 0);

                ExpectTrue((fp = XFOPEN(cliEccCert, "rb")) != XBADFILE);
                if (fp != XBADFILE) {
                    *certSz = (word32)XFREAD(cert, 1, *certSz, fp);
                    XFCLOSE(fp);
                    fp = XBADFILE;
                }
                ExpectIntGT(*certSz, 0);
            #endif /* USE_CERT_BUFFERS_256 */
            }
            break;
#endif /* HAVE_ECC */
        default:
            WOLFSSL_MSG("Unsupported SignedData PK type");
            ret = BAD_FUNC_ARG;
            break;
    }

    if (EXPECT_FAIL() && (ret == 0)) {
        ret = BAD_FUNC_ARG;
    }
    return ret;
}

/**
 * Creates a PKCS7/CMS SignedData bundle to use for testing.
 *
 * output          output buffer to place SignedData
 * outputSz        size of output buffer
 * data            data buffer to be signed
 * dataSz          size of data buffer
 * withAttribs     [1/0] include attributes in SignedData message
 * detachedSig     [1/0] create detached signature, no content
 * useIntCertChain [1/0] use certificate chain and include intermediate and
 *                 root CAs in bundle
 * pkAlgoType      RSA_TYPE or ECC_TYPE, choose what key/cert type to use
 *
 * Return size of bundle created on success, negative on error */
static int CreatePKCS7SignedData(unsigned char* output, int outputSz,
                                 byte* data, word32 dataSz,
                                 int withAttribs, int detachedSig,
                                 int useIntermediateCertChain,
                                 int pkAlgoType)
{
    EXPECT_DECLS;
    int ret = 0;
    WC_RNG rng;
    PKCS7* pkcs7 = NULL;

    static byte messageTypeOid[] =
               { 0x06, 0x0a, 0x60, 0x86, 0x48, 0x01, 0x86, 0xF8, 0x45, 0x01,
                 0x09, 0x02 };
    static byte messageType[] = { 0x13, 2, '1', '9' };

    PKCS7Attrib attribs[] =
    {
        { messageTypeOid, sizeof(messageTypeOid), messageType,
                                       sizeof(messageType) }
    };

    byte intCARoot[TWOK_BUF];
    byte intCA1[TWOK_BUF];
    byte intCA2[TWOK_BUF];
    byte cert[TWOK_BUF];
    byte key[TWOK_BUF];

    word32 intCARootSz = sizeof(intCARoot);
    word32 intCA1Sz    = sizeof(intCA1);
    word32 intCA2Sz    = sizeof(intCA2);
    word32 certSz      = sizeof(cert);
    word32 keySz       = sizeof(key);

    XMEMSET(intCARoot, 0, intCARootSz);
    XMEMSET(intCA1, 0, intCA1Sz);
    XMEMSET(intCA2, 0, intCA2Sz);
    XMEMSET(cert, 0, certSz);
    XMEMSET(key, 0, keySz);

    ret = LoadPKCS7SignedDataCerts(useIntermediateCertChain, pkAlgoType,
                intCARoot, &intCARootSz, intCA1, &intCA1Sz, intCA2, &intCA2Sz,
                cert, &certSz, key, &keySz);
    ExpectIntEQ(ret, 0);

    XMEMSET(output, 0, outputSz);
    ExpectIntEQ(wc_InitRng(&rng), 0);

    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, certSz), 0);

    if (useIntermediateCertChain == 1) {
        /* Add intermediate and root CA certs into SignedData Certs SET */
        ExpectIntEQ(wc_PKCS7_AddCertificate(pkcs7, intCA2, intCA2Sz), 0);
        ExpectIntEQ(wc_PKCS7_AddCertificate(pkcs7, intCA1, intCA1Sz), 0);
        ExpectIntEQ(wc_PKCS7_AddCertificate(pkcs7, intCARoot, intCARootSz), 0);
    }

    if (pkcs7 != NULL) {
        pkcs7->content = data;
        pkcs7->contentSz = dataSz;
        pkcs7->privateKey = key;
        pkcs7->privateKeySz = (word32)sizeof(key);
        if (pkAlgoType == RSA_TYPE) {
            pkcs7->encryptOID = RSAk;
        }
        else {
            pkcs7->encryptOID = ECDSAk;
        }
    #ifdef NO_SHA
        pkcs7->hashOID = SHA256h;
    #else
        pkcs7->hashOID = SHAh;
    #endif
        pkcs7->rng = &rng;
        if (withAttribs) {
            /* include a signed attribute */
            pkcs7->signedAttribs   = attribs;
            pkcs7->signedAttribsSz = (sizeof(attribs)/sizeof(PKCS7Attrib));
        }
    }

    if (detachedSig) {
        ExpectIntEQ(wc_PKCS7_SetDetached(pkcs7, 1), 0);
    }

    outputSz = wc_PKCS7_EncodeSignedData(pkcs7, output, (word32)outputSz);
    ExpectIntGT(outputSz, 0);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    if (detachedSig && (pkcs7 != NULL)) {
        pkcs7->content = data;
        pkcs7->contentSz = dataSz;
    }
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, (word32)outputSz), 0);

    wc_PKCS7_Free(pkcs7);
    wc_FreeRng(&rng);

    if (EXPECT_FAIL()) {
        outputSz = 0;
    }
    return outputSz;
}
#endif

/*
 * Testing wc_PKCS_VerifySignedData()
 */
static int test_wc_PKCS7_VerifySignedData_RSA(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7) && !defined(NO_FILESYSTEM) && !defined(NO_RSA)
    PKCS7* pkcs7 = NULL;
    byte   output[6000]; /* Large size needed for bundles with int CA certs */
    word32 outputSz = sizeof(output);
    byte   data[] = "Test data to encode.";
    byte   badOut[1];
    word32 badOutSz = 0;
    byte   badContent[] = "This is different content than was signed";
    wc_HashAlg hash;
#ifdef NO_SHA
    enum wc_HashType hashType = WC_HASH_TYPE_SHA256;
#else
    enum wc_HashType hashType = WC_HASH_TYPE_SHA;
#endif
    byte        hashBuf[WC_MAX_DIGEST_SIZE];
    word32      hashSz = (word32)wc_HashGetDigestSize(hashType);
#ifndef NO_RSA
    PKCS7DecodedAttrib* decodedAttrib = NULL;
    /* contentType OID (1.2.840.113549.1.9.3) */
    static const byte contentTypeOid[] =
        { 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xF7, 0x0d, 0x01, 0x09, 0x03 };

    /* PKCS#7 DATA content type (contentType defaults to DATA) */
    static const byte dataType[] =
        { 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01 };

    /* messageDigest OID (1.2.840.113549.1.9.4) */
    static const byte messageDigestOid[] =
        { 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x04 };
#ifndef NO_ASN_TIME
    /* signingTime OID () */
    static const byte signingTimeOid[] =
        { 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x05};
#endif
#if !defined(NO_ASN) && !defined(NO_ASN_TIME)
    int dateLength = 0;
    byte dateFormat;
    const byte* datePart = NULL;
    struct tm timearg;
    time_t now;
    struct tm* nowTm = NULL;
#ifdef NEED_TMP_TIME
    struct tm tmpTimeStorage;
    struct tm* tmpTime = &tmpTimeStorage;
#endif
#endif /* !NO_ASN && !NO_ASN_TIME */
#ifndef NO_PKCS7_STREAM
    word32 z;
    int ret;
#endif /* !NO_PKCS7_STREAM */

    XMEMSET(&hash, 0, sizeof(wc_HashAlg));

    /* Success test with RSA certs/key */
    ExpectIntGT((outputSz = (word32)CreatePKCS7SignedData(output, (int)outputSz, data,
        (word32)sizeof(data), 0, 0, 0, RSA_TYPE)), 0);

    /* calculate hash for content, used later */
    ExpectIntEQ(wc_HashInit(&hash, hashType), 0);
    ExpectIntEQ(wc_HashUpdate(&hash, hashType, data, sizeof(data)), 0);
    ExpectIntEQ(wc_HashFinal(&hash, hashType, hashBuf), 0);
    DoExpectIntEQ(wc_HashFree(&hash, hashType), 0);

    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz), 0);

#ifndef NO_PKCS7_STREAM
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);

    /* test for streaming */
    ret = -1;
    for (z = 0; z < outputSz && ret != 0; z++) {
        ret = wc_PKCS7_VerifySignedData(pkcs7, output + z, 1);
        if (ret < 0){
            ExpectIntEQ(ret, WC_NO_ERR_TRACE(WC_PKCS7_WANT_READ_E));
        }
    }
    ExpectIntEQ(ret, 0);
    ExpectIntNE(pkcs7->contentSz, 0);
    ExpectNotNull(pkcs7->contentDynamic);
#endif /* !NO_PKCS7_STREAM */

    /* Check that decoded signed attributes are correct */

    /* messageDigest should be first */
    if (pkcs7 != NULL) {
        decodedAttrib = pkcs7->decodedAttrib;
    }
    ExpectNotNull(decodedAttrib);
    ExpectIntEQ(decodedAttrib->oidSz, (word32)sizeof(messageDigestOid));
    ExpectIntEQ(XMEMCMP(decodedAttrib->oid, messageDigestOid,
        decodedAttrib->oidSz), 0);
    /* + 2 for OCTET STRING and length bytes */
    ExpectIntEQ(decodedAttrib->valueSz, hashSz + 2);
    ExpectNotNull(decodedAttrib->value);
    ExpectIntEQ(XMEMCMP(decodedAttrib->value + 2, hashBuf, hashSz), 0);

#ifndef NO_ASN_TIME
    /* signingTime should be second */
    if (decodedAttrib != NULL) {
        decodedAttrib = decodedAttrib->next;
    }
    ExpectNotNull(decodedAttrib);
    ExpectIntEQ(decodedAttrib->oidSz, (word32)sizeof(signingTimeOid));
    ExpectIntEQ(XMEMCMP(decodedAttrib->oid, signingTimeOid,
        decodedAttrib->oidSz), 0);

    ExpectIntGT(decodedAttrib->valueSz, 0);
    ExpectNotNull(decodedAttrib->value);
#endif

    /* Verify signingTime if ASN and time are available */
#if !defined(NO_ASN) && !defined(NO_ASN_TIME)
    ExpectIntEQ(wc_GetDateInfo(decodedAttrib->value, decodedAttrib->valueSz,
        &datePart, &dateFormat, &dateLength), 0);
    ExpectNotNull(datePart);
    ExpectIntGT(dateLength, 0);
    XMEMSET(&timearg, 0, sizeof(timearg));
    ExpectIntEQ(wc_GetDateAsCalendarTime(datePart, dateLength, dateFormat,
         &timearg), 0);

    /* Get current time and compare year/month/day against attribute value */
    ExpectIntEQ(wc_GetTime(&now, sizeof(now)), 0);
    nowTm = (struct tm*)XGMTIME((time_t*)&now, tmpTime);
    ExpectNotNull(nowTm);

    ExpectIntEQ(timearg.tm_year, nowTm->tm_year);
    ExpectIntEQ(timearg.tm_mon, nowTm->tm_mon);
    ExpectIntEQ(timearg.tm_mday, nowTm->tm_mday);
#endif /* !NO_ASN && !NO_ASN_TIME */

    /* contentType should be third */
    if (decodedAttrib != NULL) {
        decodedAttrib = decodedAttrib->next;
    }
    ExpectNotNull(decodedAttrib);
    ExpectIntEQ(decodedAttrib->oidSz, (word32)sizeof(contentTypeOid));
    ExpectIntEQ(XMEMCMP(decodedAttrib->oid, contentTypeOid,
        decodedAttrib->oidSz), 0);
    ExpectIntEQ(decodedAttrib->valueSz, (int)sizeof(dataType) + 2);
    ExpectNotNull(decodedAttrib->value);
    ExpectIntEQ(XMEMCMP(decodedAttrib->value + 2, dataType, sizeof(dataType)),
        0);
#endif /* !NO_RSA */

    /* Test bad args. */
    ExpectIntEQ(wc_PKCS7_VerifySignedData(NULL, output, outputSz),
                                          WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, NULL, outputSz),
                                          WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    #ifndef NO_PKCS7_STREAM
        /* can pass in 0 buffer length with streaming API */
        ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, badOut,
                                    badOutSz), WC_NO_ERR_TRACE(WC_PKCS7_WANT_READ_E));
    #else
        ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, badOut,
                                    badOutSz), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    #endif
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

#ifndef NO_RSA
    /* Try RSA certs/key/sig first */
    outputSz = sizeof(output);
    XMEMSET(output, 0, outputSz);
    ExpectIntGT((outputSz = (word32)CreatePKCS7SignedData(output, (int)outputSz, data,
                                                  (word32)sizeof(data),
                                                  1, 1, 0, RSA_TYPE)), 0);
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    if (pkcs7 != NULL) {
        pkcs7->content = badContent;
        pkcs7->contentSz = sizeof(badContent);
    }
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz),
                WC_NO_ERR_TRACE(SIG_VERIFY_E));

    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

#ifndef NO_PKCS7_STREAM
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    if (pkcs7 != NULL) {
        pkcs7->content = badContent;
        pkcs7->contentSz = sizeof(badContent);
    }
    /* test for streaming */
    ret = -1;
    for (z = 0; z < outputSz && ret != 0; z++) {
        ret = wc_PKCS7_VerifySignedData(pkcs7, output + z, 1);
        if (ret == WC_NO_ERR_TRACE(WC_PKCS7_WANT_READ_E)){
            continue;
        }
        else if (ret < 0) {
            break;
        }
    }
    ExpectIntEQ(ret, WC_NO_ERR_TRACE(SIG_VERIFY_E));
    ExpectIntNE(pkcs7->contentSz, 0);
    ExpectNotNull(pkcs7->contentDynamic);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
#endif /* !NO_PKCS7_STREAM */


    /* Test success case with detached signature and valid content */
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    if (pkcs7 != NULL) {
        pkcs7->content = data;
        pkcs7->contentSz = sizeof(data);
    }
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz), 0);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

#ifndef NO_PKCS7_STREAM
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    if (pkcs7 != NULL) {
        pkcs7->content = data;
        pkcs7->contentSz = sizeof(data);
    }

    /* test for streaming */
    ret = -1;
    for (z = 0; z < outputSz && ret != 0; z++) {
        ret = wc_PKCS7_VerifySignedData(pkcs7, output + z, 1);
        if (ret < 0){
            ExpectIntEQ(ret, WC_NO_ERR_TRACE(WC_PKCS7_WANT_READ_E));
        }
    }
    ExpectIntEQ(ret, 0);
    ExpectIntNE(pkcs7->contentSz, 0);
    ExpectNotNull(pkcs7->contentDynamic);

    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
#endif /* !NO_PKCS7_STREAM */

    /* verify using pre-computed content digest only (no content) */
    {
        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        ExpectIntEQ(wc_PKCS7_Init(pkcs7, NULL, 0), 0);
        ExpectIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz,
            output, outputSz, NULL, 0), 0);
        wc_PKCS7_Free(pkcs7);
        pkcs7 = NULL;
    }
#endif /* !NO_RSA */

    /* Test verify on signedData containing intermediate/root CA certs */
#ifndef NO_RSA
    outputSz = sizeof(output);
    XMEMSET(output, 0, outputSz);
    ExpectIntGT((outputSz = (word32)CreatePKCS7SignedData(output, (int)outputSz, data,
                                                  (word32)sizeof(data),
                                                  0, 0, 1, RSA_TYPE)), 0);
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz), 0);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

#ifndef NO_PKCS7_STREAM
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);

    /* test for streaming */
    ret = -1;
    for (z = 0; z < outputSz && ret != 0; z++) {
        ret = wc_PKCS7_VerifySignedData(pkcs7, output + z, 1);
        if (ret < 0){
            ExpectIntEQ(ret, WC_NO_ERR_TRACE(WC_PKCS7_WANT_READ_E));
        }
    }
    ExpectIntEQ(ret, 0);
    ExpectIntNE(pkcs7->contentSz, 0);
    ExpectNotNull(pkcs7->contentDynamic);

    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
#endif /* !NO_PKCS7_STREAM */

#endif /* !NO_RSA */
#if defined(ASN_BER_TO_DER) && !defined(NO_PKCS7_STREAM) && \
        !defined(NO_FILESYSTEM)
    {
        XFILE signedBundle = XBADFILE;
        int   signedBundleSz = 0;
        int   chunkSz = 1;
        int   i, rc = 0;
        byte* buf = NULL;

        ExpectTrue((signedBundle = XFOPEN("./certs/test-stream-sign.p7b",
            "rb")) != XBADFILE);
        ExpectTrue(XFSEEK(signedBundle, 0, XSEEK_END) == 0);
        ExpectIntGT(signedBundleSz = (int)XFTELL(signedBundle), 0);
        ExpectTrue(XFSEEK(signedBundle, 0, XSEEK_SET) == 0);
        ExpectNotNull(buf = (byte*)XMALLOC(signedBundleSz, HEAP_HINT,
            DYNAMIC_TYPE_FILE));
        if (buf != NULL) {
            ExpectIntEQ(XFREAD(buf, 1, (size_t)signedBundleSz, signedBundle),
                signedBundleSz);
        }
        if (signedBundle != XBADFILE) {
            XFCLOSE(signedBundle);
            signedBundle = XBADFILE;
        }

        if (buf != NULL) {
            ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
            ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
            for (i = 0; i < signedBundleSz;) {
                int sz = (i + chunkSz > signedBundleSz)? signedBundleSz - i :
                    chunkSz;
                rc = wc_PKCS7_VerifySignedData(pkcs7, buf + i, (word32)sz);
                if (rc < 0 ) {
                    if (rc == WC_NO_ERR_TRACE(WC_PKCS7_WANT_READ_E)) {
                        i += sz;
                        continue;
                    }
                    break;
                }
                else {
                    break;
                }
            }
            ExpectIntEQ(rc, WC_NO_ERR_TRACE(PKCS7_SIGNEEDS_CHECK));
            wc_PKCS7_Free(pkcs7);
            pkcs7 = NULL;
        }

        /* now try with malformed bundle */
        if (buf != NULL) {
            ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
            ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
            buf[signedBundleSz - 2] = buf[signedBundleSz - 2] + 1;
            for (i = 0; i < signedBundleSz;) {
                int sz = (i + chunkSz > signedBundleSz)? signedBundleSz - i :
                    chunkSz;
                rc = wc_PKCS7_VerifySignedData(pkcs7, buf + i, (word32)sz);
                if (rc < 0 ) {
                    if (rc == WC_NO_ERR_TRACE(WC_PKCS7_WANT_READ_E)) {
                        i += sz;
                        continue;
                    }
                    break;
                }
                else {
                    break;
                }
            }
            ExpectIntEQ(rc, WC_NO_ERR_TRACE(ASN_PARSE_E));
            wc_PKCS7_Free(pkcs7);
            pkcs7 = NULL;
        }

        if (buf != NULL)
            XFREE(buf, HEAP_HINT, DYNAMIC_TYPE_FILE);
    }
#endif /* BER and stream */
#endif
    return EXPECT_RESULT();
} /* END test_wc_PKCS7_VerifySignedData()_RSA */

/*
 * Testing wc_PKCS_VerifySignedData()
 */
static int test_wc_PKCS7_VerifySignedData_ECC(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7) && !defined(NO_FILESYSTEM) && defined(HAVE_ECC)
    PKCS7* pkcs7 = NULL;
    byte   output[6000]; /* Large size needed for bundles with int CA certs */
    word32 outputSz = sizeof(output);
    byte   data[] = "Test data to encode.";
    byte   badContent[] = "This is different content than was signed";
    wc_HashAlg hash;
#ifndef NO_PKCS7_STREAM
    word32 z;
    int ret;
#endif /* !NO_PKCS7_STREAM */
#ifdef NO_SHA
    enum wc_HashType hashType = WC_HASH_TYPE_SHA256;
#else
    enum wc_HashType hashType = WC_HASH_TYPE_SHA;
#endif
    byte        hashBuf[WC_MAX_DIGEST_SIZE];
    word32      hashSz = (word32)wc_HashGetDigestSize(hashType);

    XMEMSET(&hash, 0, sizeof(wc_HashAlg));

    /* Success test with ECC certs/key */
    outputSz = sizeof(output);
    XMEMSET(output, 0, outputSz);
    ExpectIntGT((outputSz = (word32)CreatePKCS7SignedData(output, (int)outputSz, data,
        (word32)sizeof(data), 0, 0, 0, ECC_TYPE)), 0);

    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz), 0);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

#ifndef NO_PKCS7_STREAM
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);

    /* test for streaming */
    ret = -1;
    for (z = 0; z < outputSz && ret != 0; z++) {
        ret = wc_PKCS7_VerifySignedData(pkcs7, output + z, 1);
        if (ret < 0){
            ExpectIntEQ(ret, WC_NO_ERR_TRACE(WC_PKCS7_WANT_READ_E));
        }
    }
    ExpectIntEQ(ret, 0);
    ExpectIntNE(pkcs7->contentSz, 0);
    ExpectNotNull(pkcs7->contentDynamic);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
#endif /* !NO_PKCS7_STREAM */

    /* Invalid content should error, use detached signature so we can
     * easily change content */
    outputSz = sizeof(output);
    XMEMSET(output, 0, outputSz);
    ExpectIntGT((outputSz = (word32)CreatePKCS7SignedData(output, (int)outputSz, data,
        (word32)sizeof(data), 1, 1, 0, ECC_TYPE)), 0);
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    if (pkcs7 != NULL) {
        pkcs7->content = badContent;
        pkcs7->contentSz = sizeof(badContent);
    }
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz),
        WC_NO_ERR_TRACE(SIG_VERIFY_E));
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

#ifndef NO_PKCS7_STREAM
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    if (pkcs7 != NULL) {
        pkcs7->content = badContent;
        pkcs7->contentSz = sizeof(badContent);
    }

    /* test for streaming */
    ret = -1;
    for (z = 0; z < outputSz && ret != 0; z++) {
        ret = wc_PKCS7_VerifySignedData(pkcs7, output + z, 1);
        if (ret == WC_NO_ERR_TRACE(WC_PKCS7_WANT_READ_E)){
            continue;
        }
        else if (ret < 0) {
            break;
        }
    }
    ExpectIntEQ(ret, WC_NO_ERR_TRACE(SIG_VERIFY_E));
    ExpectIntNE(pkcs7->contentSz, 0);
    ExpectNotNull(pkcs7->contentDynamic);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
#endif /* !NO_PKCS7_STREAM */


    /* Test success case with detached signature and valid content */
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    if (pkcs7 != NULL) {
        pkcs7->content = data;
        pkcs7->contentSz = sizeof(data);
    }
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz), 0);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

#ifndef NO_PKCS7_STREAM
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    if (pkcs7 != NULL) {
        pkcs7->content = data;
        pkcs7->contentSz = sizeof(data);
    }

    /* test for streaming */
    ret = -1;
    for (z = 0; z < outputSz && ret != 0; z++) {
        ret = wc_PKCS7_VerifySignedData(pkcs7, output + z, 1);
        if (ret < 0){
            ExpectIntEQ(ret, WC_NO_ERR_TRACE(WC_PKCS7_WANT_READ_E));
        }
    }
    ExpectIntEQ(ret, 0);
    ExpectIntNE(pkcs7->contentSz, 0);
    ExpectNotNull(pkcs7->contentDynamic);

    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
#endif /* !NO_PKCS7_STREAM */

    /* verify using pre-computed content digest only (no content) */
    {
        /* calculate hash for content */
        ExpectIntEQ(wc_HashInit(&hash, hashType), 0);
        ExpectIntEQ(wc_HashUpdate(&hash, hashType, data, sizeof(data)), 0);
        ExpectIntEQ(wc_HashFinal(&hash, hashType, hashBuf), 0);
        ExpectIntEQ(wc_HashFree(&hash, hashType), 0);

        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        ExpectIntEQ(wc_PKCS7_Init(pkcs7, NULL, 0), 0);
        ExpectIntEQ(wc_PKCS7_VerifySignedData_ex(pkcs7, hashBuf, hashSz,
            output, outputSz, NULL, 0), 0);
        wc_PKCS7_Free(pkcs7);
        pkcs7 = NULL;
    }

    /* Test verify on signedData containing intermediate/root CA certs */
    outputSz = sizeof(output);
    XMEMSET(output, 0, outputSz);
    ExpectIntGT((outputSz = (word32)CreatePKCS7SignedData(output, (int)outputSz, data,
        (word32)sizeof(data), 0, 0, 1, ECC_TYPE)), 0);
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, output, outputSz), 0);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

#ifndef NO_PKCS7_STREAM
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);

    /* test for streaming */
    ret = -1;
    for (z = 0; z < outputSz && ret != 0; z++) {
        ret = wc_PKCS7_VerifySignedData(pkcs7, output + z, 1);
        if (ret < 0){
            ExpectIntEQ(ret, WC_NO_ERR_TRACE(WC_PKCS7_WANT_READ_E));
        }
    }
    ExpectIntEQ(ret, 0);
    ExpectIntNE(pkcs7->contentSz, 0);
    ExpectNotNull(pkcs7->contentDynamic);

    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
#endif /* !NO_PKCS7_STREAM */

#endif
    return EXPECT_RESULT();
} /* END test_wc_PKCS7_VerifySignedData_ECC() */


#if defined(HAVE_PKCS7) && !defined(NO_AES) && defined(HAVE_AES_CBC) && \
    !defined(NO_AES_256)
static const byte defKey[] = {
    0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
    0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
    0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
    0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08
};
static byte aesHandle[32]; /* simulated hardware key handle */

/* return 0 on success */
static int myDecryptionFunc(PKCS7* pkcs7, int encryptOID, byte* iv, int ivSz,
        byte* aad, word32 aadSz, byte* authTag, word32 authTagSz,
        byte* in, int inSz, byte* out, void* usrCtx)
{
    int ret;
    Aes aes;

    if (usrCtx == NULL) {
        /* no simulated handle passed in */
        return -1;
    }

    switch (encryptOID) {
        case AES256CBCb:
            if (ivSz  != AES_BLOCK_SIZE)
                return BAD_FUNC_ARG;
            break;

        default:
            WOLFSSL_MSG("Unsupported content cipher type for test");
            return ALGO_ID_E;
    };

    /* simulate using handle to get key */
    ret = wc_AesInit(&aes, HEAP_HINT, INVALID_DEVID);
    if (ret == 0) {
        ret = wc_AesSetKey(&aes, (byte*)usrCtx, 32, iv, AES_DECRYPTION);
        if (ret == 0)
            ret = wc_AesCbcDecrypt(&aes, out, in, (word32)inSz);
        wc_AesFree(&aes);
    }

    (void)aad;
    (void)aadSz;
    (void)authTag;
    (void)authTagSz;
    (void)pkcs7;
    return ret;
}


/* returns key size on success */
static int myCEKwrapFunc(PKCS7* pkcs7, byte* cek, word32 cekSz, byte* keyId,
        word32 keyIdSz, byte* orginKey, word32 orginKeySz,
        byte* out, word32 outSz, int keyWrapAlgo, int type, int direction)
{
    int ret = -1;

    (void)cekSz;
    (void)cek;
    (void)outSz;
    (void)keyIdSz;
    (void)direction;
    (void)orginKey; /* used with KAKRI */
    (void)orginKeySz;

    if (out == NULL)
        return BAD_FUNC_ARG;

    if (keyId[0] != 0x00) {
        return -1;
    }

    if (type != (int)PKCS7_KEKRI) {
        return -1;
    }

    switch (keyWrapAlgo) {
        case AES256_WRAP:
            /* simulate setting a handle for later decryption but use key
             * as handle in the test case here */
            ret = wc_AesKeyUnWrap(defKey, sizeof(defKey), cek, cekSz,
                                      aesHandle, sizeof(aesHandle), NULL);
            if (ret < 0)
                return ret;

            ret = wc_PKCS7_SetDecodeEncryptedCtx(pkcs7, (void*)aesHandle);
            if (ret < 0)
                return ret;

            /* return key size on success */
            return sizeof(defKey);

        default:
            WOLFSSL_MSG("Unsupported key wrap algorithm in example");
            return BAD_KEYWRAP_ALG_E;
    };
}
#endif /* HAVE_PKCS7 && !NO_AES && HAVE_AES_CBC && !NO_AES_256 */


#if defined(HAVE_PKCS7) && defined(ASN_BER_TO_DER)
#define MAX_TEST_DECODE_SIZE 6000
static int test_wc_PKCS7_DecodeEnvelopedData_stream_decrypt_cb(wc_PKCS7* pkcs7,
    const byte* output, word32 outputSz, void* ctx) {
     WOLFSSL_BUFFER_INFO* out = (WOLFSSL_BUFFER_INFO*)ctx;

    if (out == NULL) {
        return -1;
    }

    if (outputSz + out->length > MAX_TEST_DECODE_SIZE) {
        printf("Example buffer size needs increased");
    }

    /* printf("Decoded in %d bytes\n", outputSz);
     * for (word32 z = 0; z < outputSz; z++) printf("%02X", output[z]);
     * printf("\n");
    */

    XMEMCPY(out->buffer + out->length, output, outputSz);
    out->length += outputSz;

    (void)pkcs7;
    return 0;
}
#endif /* HAVE_PKCS7 && ASN_BER_TO_DER */

/*
 * Testing wc_PKCS7_DecodeEnvelopedData with streaming
 */
static int test_wc_PKCS7_DecodeEnvelopedData_stream(void)
{
#if defined(HAVE_PKCS7) && defined(ASN_BER_TO_DER)
    EXPECT_DECLS;
    PKCS7*      pkcs7 = NULL;
    int ret = 0;
    XFILE f = XBADFILE;
    const char* testStream = "./certs/test-stream-dec.p7b";
    byte testStreamBuffer[100];
    size_t testStreamBufferSz = 0;
    byte decodedData[MAX_TEST_DECODE_SIZE]; /* large enough to hold result of decode, which is ca-cert.pem */
    WOLFSSL_BUFFER_INFO out;

    out.length = 0;
    out.buffer = decodedData;

    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, (byte*)client_cert_der_2048,
        sizeof_client_cert_der_2048), 0);

    ExpectIntEQ(wc_PKCS7_SetKey(pkcs7, (byte*)client_key_der_2048,
        sizeof_client_key_der_2048), 0);
    ExpectIntEQ(wc_PKCS7_SetStreamMode(pkcs7, 1, NULL,
        test_wc_PKCS7_DecodeEnvelopedData_stream_decrypt_cb, (void*)&out), 0);

    ExpectTrue((f = XFOPEN(testStream, "rb")) != XBADFILE);
    if (EXPECT_SUCCESS()) {
        do {
            testStreamBufferSz = XFREAD(testStreamBuffer, 1,
                sizeof(testStreamBuffer), f);
            if (testStreamBufferSz == 0) {
                break;
            }

            ret = wc_PKCS7_DecodeEnvelopedData(pkcs7, testStreamBuffer,
                (word32)testStreamBufferSz, NULL, 0);
            if (testStreamBufferSz < sizeof(testStreamBuffer)) {
                break;
            }
        } while (ret == WC_NO_ERR_TRACE(WC_PKCS7_WANT_READ_E));
    #ifdef NO_DES3
        ExpectIntEQ(ret, ALGO_ID_E);
    #else
        ExpectIntGT(ret, 0);
    #endif
    }

    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }

    wc_PKCS7_Free(pkcs7);
    return EXPECT_RESULT();
#else
    return TEST_SKIPPED;
#endif
} /* END test_wc_PKCS7_DecodeEnvelopedData_stream() */

/*
 * Testing wc_PKCS7_EncodeEnvelopedData()
 */
static int test_wc_PKCS7_EncodeDecodeEnvelopedData(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7)
    PKCS7*      pkcs7 = NULL;
#ifdef ASN_BER_TO_DER
    int encodedSz = 0;
#endif
#ifdef ECC_TIMING_RESISTANT
    WC_RNG      rng;
#endif
    word32      tempWrd32   = 0;
    byte*       tmpBytePtr = NULL;
    const char  input[] = "Test data to encode.";
    int         i;
    int         testSz = 0;
    #if !defined(NO_RSA) && (!defined(NO_AES) || (!defined(NO_SHA) || \
        !defined(NO_SHA256) || defined(WOLFSSL_SHA512)))
        byte*   rsaCert     = NULL;
        byte*   rsaPrivKey  = NULL;
        word32  rsaCertSz;
        word32  rsaPrivKeySz;
        #if !defined(NO_FILESYSTEM) && (!defined(USE_CERT_BUFFERS_1024) && \
                                           !defined(USE_CERT_BUFFERS_2048) )
            static const char* rsaClientCert = "./certs/client-cert.der";
            static const char* rsaClientKey = "./certs/client-key.der";
            rsaCertSz = (word32)sizeof(rsaClientCert);
            rsaPrivKeySz = (word32)sizeof(rsaClientKey);
        #endif
    #endif
    #if defined(HAVE_ECC) && (!defined(NO_AES) || (!defined(NO_SHA) ||\
        !defined(NO_SHA256) || defined(WOLFSSL_SHA512)))
        byte*   eccCert     = NULL;
        byte*   eccPrivKey  = NULL;
        word32  eccCertSz;
        word32  eccPrivKeySz;
        #if !defined(NO_FILESYSTEM) && !defined(USE_CERT_BUFFERS_256)
            static const char* eccClientCert = "./certs/client-ecc-cert.der";
            static const char* eccClientKey = "./certs/ecc-client-key.der";
        #endif
    #endif
    /* Generic buffer size. */
    byte    output[ONEK_BUF];
    byte    decoded[sizeof(input)/sizeof(char)];
    int     decodedSz = 0;
#ifndef NO_FILESYSTEM
    XFILE certFile = XBADFILE;
    XFILE keyFile = XBADFILE;
#endif

#ifdef ECC_TIMING_RESISTANT
    XMEMSET(&rng, 0, sizeof(WC_RNG));
#endif

#if !defined(NO_RSA) && (!defined(NO_AES) || (!defined(NO_SHA) ||\
    !defined(NO_SHA256) || defined(WOLFSSL_SHA512)))
    /* RSA certs and keys. */
    #if defined(USE_CERT_BUFFERS_1024)
        rsaCertSz = (word32)sizeof_client_cert_der_1024;
        /* Allocate buffer space. */
        ExpectNotNull(rsaCert = (byte*)XMALLOC(rsaCertSz, HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER));
        /* Init buffer. */
        if (rsaCert != NULL) {
            XMEMCPY(rsaCert, client_cert_der_1024, rsaCertSz);
        }
        rsaPrivKeySz = (word32)sizeof_client_key_der_1024;
        ExpectNotNull(rsaPrivKey = (byte*)XMALLOC(rsaPrivKeySz, HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER));
        if (rsaPrivKey != NULL) {
            XMEMCPY(rsaPrivKey, client_key_der_1024, rsaPrivKeySz);
        }
    #elif defined(USE_CERT_BUFFERS_2048)
        rsaCertSz = (word32)sizeof_client_cert_der_2048;
        /* Allocate buffer */
        ExpectNotNull(rsaCert = (byte*)XMALLOC(rsaCertSz, HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER));
        /* Init buffer. */
        if (rsaCert != NULL) {
            XMEMCPY(rsaCert, client_cert_der_2048, rsaCertSz);
        }
        rsaPrivKeySz = (word32)sizeof_client_key_der_2048;
        ExpectNotNull(rsaPrivKey = (byte*)XMALLOC(rsaPrivKeySz, HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER));
        if (rsaPrivKey != NULL) {
            XMEMCPY(rsaPrivKey, client_key_der_2048, rsaPrivKeySz);
        }
    #else
        /* File system. */
        ExpectTrue((certFile = XFOPEN(rsaClientCert, "rb")) != XBADFILE);
        rsaCertSz = (word32)FOURK_BUF;
        ExpectNotNull(rsaCert = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER));
        ExpectTrue((rsaCertSz = (word32)XFREAD(rsaCert, 1, rsaCertSz,
            certFile)) > 0);
        if (certFile != XBADFILE)
            XFCLOSE(certFile);
        ExpectTrue((keyFile = XFOPEN(rsaClientKey, "rb")) != XBADFILE);
        ExpectNotNull(rsaPrivKey = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER));
        rsaPrivKeySz = (word32)FOURK_BUF;
        ExpectTrue((rsaPrivKeySz = (word32)XFREAD(rsaPrivKey, 1, rsaPrivKeySz,
            keyFile)) > 0);
        if (keyFile != XBADFILE)
            XFCLOSE(keyFile);
    #endif /* USE_CERT_BUFFERS */
#endif /* NO_RSA */

/* ECC */
#if defined(HAVE_ECC) && (!defined(NO_AES) || (!defined(NO_SHA) ||\
    !defined(NO_SHA256) || defined(WOLFSSL_SHA512)))

    #ifdef USE_CERT_BUFFERS_256
        ExpectNotNull(eccCert = (byte*)XMALLOC(TWOK_BUF, HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER));
        /* Init buffer. */
        eccCertSz = (word32)sizeof_cliecc_cert_der_256;
        if (eccCert != NULL) {
            XMEMCPY(eccCert, cliecc_cert_der_256, eccCertSz);
        }
        ExpectNotNull(eccPrivKey = (byte*)XMALLOC(TWOK_BUF, HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER));
        eccPrivKeySz = (word32)sizeof_ecc_clikey_der_256;
        if (eccPrivKey != NULL) {
            XMEMCPY(eccPrivKey, ecc_clikey_der_256, eccPrivKeySz);
        }
    #else /* File system. */
        ExpectTrue((certFile = XFOPEN(eccClientCert, "rb")) != XBADFILE);
        eccCertSz = (word32)FOURK_BUF;
        ExpectNotNull(eccCert = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER));
        ExpectTrue((eccCertSz = (word32)XFREAD(eccCert, 1, eccCertSz,
            certFile)) > 0);
        if (certFile != XBADFILE) {
            XFCLOSE(certFile);
        }
        ExpectTrue((keyFile = XFOPEN(eccClientKey, "rb")) != XBADFILE);
        eccPrivKeySz = (word32)FOURK_BUF;
        ExpectNotNull(eccPrivKey = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER));
        ExpectTrue((eccPrivKeySz = (word32)XFREAD(eccPrivKey, 1, eccPrivKeySz,
            keyFile)) > 0);
        if (keyFile != XBADFILE) {
            XFCLOSE(keyFile);
        }
    #endif /* USE_CERT_BUFFERS_256 */
#endif /* END HAVE_ECC */

#ifndef NO_FILESYSTEM
    /* Silence. */
    (void)keyFile;
    (void)certFile;
#endif

    {
    const pkcs7EnvelopedVector testVectors[] = {
    /* DATA is a global variable defined in the makefile. */
#if !defined(NO_RSA)
    #ifndef NO_DES3
        {(byte*)input, (word32)(sizeof(input)/sizeof(char)), DATA, DES3b, 0, 0,
            rsaCert, rsaCertSz, rsaPrivKey, rsaPrivKeySz},
    #endif /* NO_DES3 */
    #if !defined(NO_AES) && defined(HAVE_AES_CBC)
        #ifndef NO_AES_128
        {(byte*)input, (word32)(sizeof(input)/sizeof(char)), DATA, AES128CBCb,
            0, 0, rsaCert, rsaCertSz, rsaPrivKey, rsaPrivKeySz},
        #endif
        #ifndef NO_AES_192
        {(byte*)input, (word32)(sizeof(input)/sizeof(char)), DATA, AES192CBCb,
            0, 0, rsaCert, rsaCertSz, rsaPrivKey, rsaPrivKeySz},
        #endif
        #ifndef NO_AES_256
        {(byte*)input, (word32)(sizeof(input)/sizeof(char)), DATA, AES256CBCb,
            0, 0, rsaCert, rsaCertSz, rsaPrivKey, rsaPrivKeySz},
        #endif
    #endif /* NO_AES && HAVE_AES_CBC */

#endif /* NO_RSA */
#if defined(HAVE_ECC)
    #if !defined(NO_AES) && defined(HAVE_AES_CBC)
        #if !defined(NO_SHA) && !defined(NO_AES_128)
            {(byte*)input, (word32)(sizeof(input)/sizeof(char)), DATA,
                AES128CBCb, AES128_WRAP, dhSinglePass_stdDH_sha1kdf_scheme,
                eccCert, eccCertSz, eccPrivKey, eccPrivKeySz},
        #endif
        #if !defined(NO_SHA256) && !defined(NO_AES_256)
            {(byte*)input, (word32)(sizeof(input)/sizeof(char)), DATA,
                AES256CBCb, AES256_WRAP, dhSinglePass_stdDH_sha256kdf_scheme,
                eccCert, eccCertSz, eccPrivKey, eccPrivKeySz},
        #endif
        #if defined(WOLFSSL_SHA512) && !defined(NO_AES_256)
            {(byte*)input, (word32)(sizeof(input)/sizeof(char)), DATA,
                AES256CBCb, AES256_WRAP, dhSinglePass_stdDH_sha512kdf_scheme,
                eccCert, eccCertSz, eccPrivKey, eccPrivKeySz},
        #endif
    #endif /* NO_AES && HAVE_AES_CBC*/
#endif /* END HAVE_ECC */
    }; /* END pkcs7EnvelopedVector */

#ifdef ECC_TIMING_RESISTANT
    ExpectIntEQ(wc_InitRng(&rng), 0);
#endif

    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, testDevId), 0);

    testSz = (int)sizeof(testVectors)/(int)sizeof(pkcs7EnvelopedVector);
    for (i = 0; i < testSz; i++) {
    #ifdef ASN_BER_TO_DER
        encodeSignedDataStream strm;

        /* test setting stream mode, the first one using IO callbacks */
        ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, (testVectors + i)->cert,
                                    (word32)(testVectors + i)->certSz), 0);
        if (pkcs7 != NULL) {
        #ifdef ECC_TIMING_RESISTANT
            pkcs7->rng = &rng;
        #endif

            if (i != 0)
                pkcs7->content       = (byte*)(testVectors + i)->content;
            pkcs7->contentSz     = (testVectors + i)->contentSz;
            pkcs7->contentOID    = (testVectors + i)->contentOID;
            pkcs7->encryptOID    = (testVectors + i)->encryptOID;
            pkcs7->keyWrapOID    = (testVectors + i)->keyWrapOID;
            pkcs7->keyAgreeOID   = (testVectors + i)->keyAgreeOID;
            pkcs7->privateKey    = (testVectors + i)->privateKey;
            pkcs7->privateKeySz  = (testVectors + i)->privateKeySz;
        }

        if (i == 0) {
            XMEMSET(&strm, 0, sizeof(strm));
            strm.chunkSz = FOURK_BUF;
            ExpectIntEQ(wc_PKCS7_SetStreamMode(pkcs7, 1, GetContentCB,
                StreamOutputCB, (void*)&strm), 0);
            encodedSz = wc_PKCS7_EncodeEnvelopedData(pkcs7, NULL, 0);
        }
        else {
            ExpectIntEQ(wc_PKCS7_SetStreamMode(pkcs7, 1, NULL, NULL, NULL), 0);
            encodedSz = wc_PKCS7_EncodeEnvelopedData(pkcs7, output,
                (word32)sizeof(output));
        }

        switch ((testVectors + i)->encryptOID) {
        #ifndef NO_DES3
            case DES3b:
            case DESb:
                ExpectIntEQ(encodedSz, WC_NO_ERR_TRACE(BAD_FUNC_ARG));
                break;
        #endif
        #ifdef HAVE_AESCCM
        #ifdef WOLFSSL_AES_128
            case AES128CCMb:
                ExpectIntEQ(encodedSz, WC_NO_ERR_TRACE(BAD_FUNC_ARG));
                break;
        #endif
        #ifdef WOLFSSL_AES_192
            case AES192CCMb:
                ExpectIntEQ(encodedSz, WC_NO_ERR_TRACE(BAD_FUNC_ARG));
                break;
        #endif
        #ifdef WOLFSSL_AES_256
            case AES256CCMb:
                ExpectIntEQ(encodedSz, WC_NO_ERR_TRACE(BAD_FUNC_ARG));
                break;
        #endif
        #endif
            default:
                ExpectIntGE(encodedSz, 0);
        }

        if (encodedSz > 0) {
            if (i == 0) {
                decodedSz = wc_PKCS7_DecodeEnvelopedData(pkcs7,
                    strm.out, (word32)encodedSz, decoded,
                    (word32)sizeof(decoded));
            }
            else {
                decodedSz = wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
                    (word32)encodedSz, decoded, (word32)sizeof(decoded));
            }
            ExpectIntGE(decodedSz, 0);
            /* Verify the size of each buffer. */
            ExpectIntEQ((word32)sizeof(input)/sizeof(char), decodedSz);
        }
        wc_PKCS7_Free(pkcs7);
        pkcs7 = NULL;
        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    #endif

        ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, (testVectors + i)->cert,
                                    (word32)(testVectors + i)->certSz), 0);
        if (pkcs7 != NULL) {
#ifdef ECC_TIMING_RESISTANT
            pkcs7->rng = &rng;
#endif

            pkcs7->content       = (byte*)(testVectors + i)->content;
            pkcs7->contentSz     = (testVectors + i)->contentSz;
            pkcs7->contentOID    = (testVectors + i)->contentOID;
            pkcs7->encryptOID    = (testVectors + i)->encryptOID;
            pkcs7->keyWrapOID    = (testVectors + i)->keyWrapOID;
            pkcs7->keyAgreeOID   = (testVectors + i)->keyAgreeOID;
            pkcs7->privateKey    = (testVectors + i)->privateKey;
            pkcs7->privateKeySz  = (testVectors + i)->privateKeySz;
        }

    #ifdef ASN_BER_TO_DER
        /* test without setting stream mode */
        ExpectIntEQ(wc_PKCS7_GetStreamMode(pkcs7), 0);
    #endif

        ExpectIntGE(wc_PKCS7_EncodeEnvelopedData(pkcs7, output,
            (word32)sizeof(output)), 0);

        decodedSz = wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
            (word32)sizeof(output), decoded, (word32)sizeof(decoded));
        ExpectIntGE(decodedSz, 0);
        /* Verify the size of each buffer. */
        ExpectIntEQ((word32)sizeof(input)/sizeof(char), decodedSz);

        /* Don't free the last time through the loop. */
        if (i < testSz - 1) {
            wc_PKCS7_Free(pkcs7);
            pkcs7 = NULL;
            ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        }
    }  /* END test loop. */
    }

    /* Test bad args. */
    ExpectIntEQ(wc_PKCS7_EncodeEnvelopedData(NULL, output,
                    (word32)sizeof(output)), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_PKCS7_EncodeEnvelopedData(pkcs7, NULL,
                    (word32)sizeof(output)), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_PKCS7_EncodeEnvelopedData(pkcs7, output, 0), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    /* Decode.  */
    ExpectIntEQ(wc_PKCS7_DecodeEnvelopedData(NULL, output,
        (word32)sizeof(output), decoded, (word32)sizeof(decoded)),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
        (word32)sizeof(output), NULL, (word32)sizeof(decoded)), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
        (word32)sizeof(output), decoded, 0), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, NULL,
        (word32)sizeof(output), decoded, (word32)sizeof(decoded)),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, output, 0, decoded,
        (word32)sizeof(decoded)), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    /* Should get a return of BAD_FUNC_ARG with structure data. Order matters.*/
#if defined(HAVE_ECC) && !defined(NO_AES) && defined(HAVE_AES_CBC)
    /* only a failure for KARI test cases */
    if (pkcs7 != NULL) {
        tempWrd32 = pkcs7->singleCertSz;
        pkcs7->singleCertSz = 0;
    }
    #if defined(WOLFSSL_ASN_TEMPLATE)
    ExpectIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
        (word32)sizeof(output), decoded, (word32)sizeof(decoded)),
        WC_NO_ERR_TRACE(BUFFER_E));
    #else
    ExpectIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
        (word32)sizeof(output), decoded, (word32)sizeof(decoded)),
        WC_NO_ERR_TRACE(ASN_PARSE_E));
    #endif
    if (pkcs7 != NULL) {
        pkcs7->singleCertSz = tempWrd32;

        tmpBytePtr = pkcs7->singleCert;
        pkcs7->singleCert = NULL;
    }
  #ifndef NO_RSA
    #if defined(NO_PKCS7_STREAM)
    /* when none streaming mode is used and PKCS7 is in bad state buffer error
     * is returned from kari parse which gets set to bad func arg */
    ExpectIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
        (word32)sizeof(output), decoded, (word32)sizeof(decoded)),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    #else
    ExpectIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
        (word32)sizeof(output), decoded, (word32)sizeof(decoded)),
        WC_NO_ERR_TRACE(ASN_PARSE_E));
    #endif
  #endif /* !NO_RSA */
    if (pkcs7 != NULL) {
        pkcs7->singleCert = tmpBytePtr;
    }
#endif
    if (pkcs7 != NULL) {
        tempWrd32 = pkcs7->privateKeySz;
        pkcs7->privateKeySz = 0;
    }

    ExpectIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
        (word32)sizeof(output), decoded, (word32)sizeof(decoded)),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    if (pkcs7 != NULL) {
        pkcs7->privateKeySz = tempWrd32;

        tmpBytePtr = pkcs7->privateKey;
        pkcs7->privateKey = NULL;
    }
    ExpectIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
        (word32)sizeof(output), decoded, (word32)sizeof(decoded)),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    if (pkcs7 != NULL) {
        pkcs7->privateKey = tmpBytePtr;
    }

    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

#if !defined(NO_AES) && defined(HAVE_AES_CBC) && !defined(NO_AES_256)
    /* test of decrypt callback with KEKRI enveloped data */
    {
        int envelopedSz = 0;
        const byte keyId[] = { 0x00 };

        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        if (pkcs7 != NULL) {
            pkcs7->content      = (byte*)input;
            pkcs7->contentSz    = (word32)(sizeof(input)/sizeof(char));
            pkcs7->contentOID   = DATA;
            pkcs7->encryptOID   = AES256CBCb;
        }
        ExpectIntGT(wc_PKCS7_AddRecipient_KEKRI(pkcs7, AES256_WRAP,
                    (byte*)defKey, sizeof(defKey), (byte*)keyId,
                    sizeof(keyId), NULL, NULL, 0, NULL, 0, 0), 0);
        ExpectIntEQ(wc_PKCS7_SetSignerIdentifierType(pkcs7, CMS_SKID), 0);
        ExpectIntGT((envelopedSz = wc_PKCS7_EncodeEnvelopedData(pkcs7, output,
                        (word32)sizeof(output))), 0);
        wc_PKCS7_Free(pkcs7);
        pkcs7 = NULL;

        /* decode envelopedData */
        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        ExpectIntEQ(wc_PKCS7_SetWrapCEKCb(pkcs7, myCEKwrapFunc), 0);
        ExpectIntEQ(wc_PKCS7_SetDecodeEncryptedCb(pkcs7, myDecryptionFunc), 0);
        ExpectIntGT((decodedSz = wc_PKCS7_DecodeEnvelopedData(pkcs7, output,
                        (word32)envelopedSz, decoded, sizeof(decoded))), 0);
        wc_PKCS7_Free(pkcs7);
        pkcs7 = NULL;
    }
#endif /* !NO_AES && !NO_AES_256 */

#ifndef NO_RSA
    XFREE(rsaCert, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(rsaPrivKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif /* NO_RSA */
#ifdef HAVE_ECC
    XFREE(eccCert, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(eccPrivKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif /* HAVE_ECC */

#ifdef ECC_TIMING_RESISTANT
    DoExpectIntEQ(wc_FreeRng(&rng), 0);
#endif

#if defined(USE_CERT_BUFFERS_2048) && !defined(NO_DES3) && \
    !defined(NO_RSA) && !defined(NO_SHA)
    {
        byte   out[7];
        byte   *cms = NULL;
        word32 cmsSz;
        XFILE  cmsFile = XBADFILE;

        XMEMSET(out, 0, sizeof(out));
        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        ExpectTrue((cmsFile = XFOPEN("./certs/test/ktri-keyid-cms.msg", "rb"))
            != XBADFILE);
        cmsSz = (word32)FOURK_BUF;
        ExpectNotNull(cms = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER));
        ExpectTrue((cmsSz = (word32)XFREAD(cms, 1, cmsSz, cmsFile)) > 0);
        if (cmsFile != XBADFILE)
            XFCLOSE(cmsFile);

        ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, (byte*)client_cert_der_2048,
            sizeof_client_cert_der_2048), 0);
        if (pkcs7 != NULL) {
            pkcs7->privateKey   = (byte*)client_key_der_2048;
            pkcs7->privateKeySz = sizeof_client_key_der_2048;
        }
        ExpectIntLT(wc_PKCS7_DecodeEnvelopedData(pkcs7, cms, cmsSz, out,
            2), 0);
        ExpectIntGT(wc_PKCS7_DecodeEnvelopedData(pkcs7, cms, cmsSz, out,
            sizeof(out)), 0);
        XFREE(cms, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        ExpectIntEQ(XMEMCMP(out, "test", 4), 0);
        wc_PKCS7_Free(pkcs7);
        pkcs7 = NULL;
    }
#endif /* USE_CERT_BUFFERS_2048 && !NO_DES3 && !NO_RSA && !NO_SHA */
#endif /* HAVE_PKCS7 */
    return EXPECT_RESULT();
} /* END test_wc_PKCS7_EncodeDecodeEnvelopedData() */


/*
 * Testing wc_PKCS7_EncodeEncryptedData()
 */
static int test_wc_PKCS7_EncodeEncryptedData(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7) && !defined(NO_PKCS7_ENCRYPTED_DATA)
    PKCS7*      pkcs7 = NULL;
    byte*       tmpBytePtr = NULL;
    byte        encrypted[TWOK_BUF];
    byte        decoded[TWOK_BUF];
    word32      tmpWrd32 = 0;
    int         tmpInt = 0;
    int         decodedSz = 0;
    int         encryptedSz = 0;
    int         testSz = 0;
    int         i = 0;
    const byte data[] = { /* Hello World */
        0x48,0x65,0x6c,0x6c,0x6f,0x20,0x57,0x6f,
        0x72,0x6c,0x64
    };
    #ifndef NO_DES3
        byte desKey[] = {
            0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef
        };
        byte des3Key[] = {
            0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,
            0xfe,0xde,0xba,0x98,0x76,0x54,0x32,0x10,
            0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67
        };
    #endif
    #if !defined(NO_AES) && defined(HAVE_AES_CBC)
        #ifndef NO_AES_128
        byte aes128Key[] = {
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08
        };
        #endif
        #ifndef NO_AES_192
        byte aes192Key[] = {
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08
        };
        #endif
        #ifndef NO_AES_256
        byte aes256Key[] = {
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
            0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08
        };
        #endif
    #endif /* !NO_AES && HAVE_AES_CBC */
    const pkcs7EncryptedVector testVectors[] =
    {
    #ifndef NO_DES3
        {data, (word32)sizeof(data), DATA, DES3b, des3Key, sizeof(des3Key)},

        {data, (word32)sizeof(data), DATA, DESb, desKey, sizeof(desKey)},
    #endif /* !NO_DES3 */
    #if !defined(NO_AES) && defined(HAVE_AES_CBC)
        #ifndef NO_AES_128
        {data, (word32)sizeof(data), DATA, AES128CBCb, aes128Key,
         sizeof(aes128Key)},
        #endif

        #ifndef NO_AES_192
        {data, (word32)sizeof(data), DATA, AES192CBCb, aes192Key,
         sizeof(aes192Key)},
        #endif

        #ifndef NO_AES_256
        {data, (word32)sizeof(data), DATA, AES256CBCb, aes256Key,
         sizeof(aes256Key)},
        #endif

    #endif /* !NO_AES && HAVE_AES_CBC */
    };

    testSz = sizeof(testVectors) / sizeof(pkcs7EncryptedVector);

    for (i = 0; i < testSz; i++) {
        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, testDevId), 0);
        if (pkcs7 != NULL) {
            pkcs7->content              = (byte*)testVectors[i].content;
            pkcs7->contentSz            = testVectors[i].contentSz;
            pkcs7->contentOID           = testVectors[i].contentOID;
            pkcs7->encryptOID           = testVectors[i].encryptOID;
            pkcs7->encryptionKey        = testVectors[i].encryptionKey;
            pkcs7->encryptionKeySz      = testVectors[i].encryptionKeySz;
            pkcs7->heap                 = HEAP_HINT;
        }

        /* encode encryptedData */
        ExpectIntGT(encryptedSz = wc_PKCS7_EncodeEncryptedData(pkcs7, encrypted,
            sizeof(encrypted)), 0);

        /* Decode encryptedData */
        ExpectIntGT(decodedSz = wc_PKCS7_DecodeEncryptedData(pkcs7, encrypted,
            (word32)encryptedSz, decoded, sizeof(decoded)), 0);

        ExpectIntEQ(XMEMCMP(decoded, data, decodedSz), 0);
        /* Keep values for last itr. */
        if (i < testSz - 1) {
            wc_PKCS7_Free(pkcs7);
            pkcs7 = NULL;
        }
    }
    if (pkcs7 == NULL || testSz == 0) {
        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, testDevId), 0);
    }

    ExpectIntEQ(wc_PKCS7_EncodeEncryptedData(NULL, encrypted,
        sizeof(encrypted)),WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_PKCS7_EncodeEncryptedData(pkcs7, NULL,
        sizeof(encrypted)), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_PKCS7_EncodeEncryptedData(pkcs7, encrypted,
        0), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    /* Testing the struct. */
    if (pkcs7 != NULL) {
        tmpBytePtr = pkcs7->content;
        pkcs7->content = NULL;
    }
    ExpectIntEQ(wc_PKCS7_EncodeEncryptedData(pkcs7, encrypted,
        sizeof(encrypted)), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    if (pkcs7 != NULL) {
        pkcs7->content = tmpBytePtr;
        tmpWrd32 = pkcs7->contentSz;
        pkcs7->contentSz = 0;
    }
    ExpectIntEQ(wc_PKCS7_EncodeEncryptedData(pkcs7, encrypted,
        sizeof(encrypted)), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    if (pkcs7 != NULL) {
        pkcs7->contentSz = tmpWrd32;
        tmpInt = pkcs7->encryptOID;
        pkcs7->encryptOID = 0;
    }
    ExpectIntEQ(wc_PKCS7_EncodeEncryptedData(pkcs7, encrypted,
        sizeof(encrypted)), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    if (pkcs7 != NULL) {
        pkcs7->encryptOID = tmpInt;
        tmpBytePtr = pkcs7->encryptionKey;
        pkcs7->encryptionKey = NULL;
    }
    ExpectIntEQ(wc_PKCS7_EncodeEncryptedData(pkcs7, encrypted,
        sizeof(encrypted)), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    if (pkcs7 != NULL) {
        pkcs7->encryptionKey = tmpBytePtr;
        tmpWrd32 = pkcs7->encryptionKeySz;
        pkcs7->encryptionKeySz = 0;
    }
    ExpectIntEQ(wc_PKCS7_EncodeEncryptedData(pkcs7, encrypted,
        sizeof(encrypted)), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    if (pkcs7 != NULL) {
        pkcs7->encryptionKeySz = tmpWrd32;
    }

    ExpectIntEQ(wc_PKCS7_DecodeEncryptedData(NULL, encrypted, (word32)encryptedSz,
        decoded, sizeof(decoded)), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_PKCS7_DecodeEncryptedData(pkcs7, NULL, (word32)encryptedSz,
        decoded, sizeof(decoded)), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_PKCS7_DecodeEncryptedData(pkcs7, encrypted, 0,
        decoded, sizeof(decoded)), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_PKCS7_DecodeEncryptedData(pkcs7, encrypted, (word32)encryptedSz,
        NULL, sizeof(decoded)), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_PKCS7_DecodeEncryptedData(pkcs7, encrypted, (word32)encryptedSz,
        decoded, 0), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    /* Test struct fields */

    if (pkcs7 != NULL) {
        tmpBytePtr = pkcs7->encryptionKey;
        pkcs7->encryptionKey = NULL;
    }
    ExpectIntEQ(wc_PKCS7_DecodeEncryptedData(pkcs7, encrypted, (word32)encryptedSz,
        decoded, sizeof(decoded)), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    if (pkcs7 != NULL) {
        pkcs7->encryptionKey = tmpBytePtr;
        pkcs7->encryptionKeySz = 0;
    }
    ExpectIntEQ(wc_PKCS7_DecodeEncryptedData(pkcs7, encrypted, (word32)encryptedSz,
        decoded, sizeof(decoded)), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    wc_PKCS7_Free(pkcs7);
#endif
    return EXPECT_RESULT();
} /* END test_wc_PKCS7_EncodeEncryptedData() */

/*
 * Testing wc_PKCS7_Degenerate()
 */
static int test_wc_PKCS7_Degenerate(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7) && !defined(NO_FILESYSTEM)
    PKCS7* pkcs7 = NULL;
    char   fName[] = "./certs/test-degenerate.p7b";
    XFILE  f = XBADFILE;
    byte   der[4096];
    word32 derSz = 0;
#ifndef NO_PKCS7_STREAM
    word32 z;
    int ret;
#endif /* !NO_PKCS7_STREAM */
    ExpectTrue((f = XFOPEN(fName, "rb")) != XBADFILE);
    ExpectTrue((derSz = (word32)XFREAD(der, 1, sizeof(der), f)) > 0);
    if (f != XBADFILE)
        XFCLOSE(f);

    /* test degenerate success */
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
#ifndef NO_RSA
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, der, derSz), 0);

    #ifndef NO_PKCS7_STREAM
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);

    /* test for streaming */
    ret = -1;
    for (z = 0; z < derSz && ret != 0; z++) {
        ret = wc_PKCS7_VerifySignedData(pkcs7, der + z, 1);
        if (ret < 0){
            ExpectIntEQ(ret, WC_NO_ERR_TRACE(WC_PKCS7_WANT_READ_E));
        }
    }
    ExpectIntEQ(ret, 0);
    #endif /* !NO_PKCS7_STREAM */
#else
    ExpectIntNE(wc_PKCS7_VerifySignedData(pkcs7, der, derSz), 0);
#endif /* NO_RSA */
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

    /* test with turning off degenerate cases */
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    wc_PKCS7_AllowDegenerate(pkcs7, 0); /* override allowing degenerate case */
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, der, derSz),
        WC_NO_ERR_TRACE(PKCS7_NO_SIGNER_E));

    #ifndef NO_PKCS7_STREAM
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    wc_PKCS7_AllowDegenerate(pkcs7, 0); /* override allowing degenerate case */

    /* test for streaming */
    ret = -1;
    for (z = 0; z < derSz && ret != 0; z++) {
        ret = wc_PKCS7_VerifySignedData(pkcs7, der + z, 1);
        if (ret == WC_NO_ERR_TRACE(WC_PKCS7_WANT_READ_E)){
            continue;
        }
        else
            break;
    }
    ExpectIntEQ(ret, WC_NO_ERR_TRACE(PKCS7_NO_SIGNER_E));
    #endif /* !NO_PKCS7_STREAM */

    wc_PKCS7_Free(pkcs7);
#endif
    return EXPECT_RESULT();
} /* END test_wc_PKCS7_Degenerate() */

#if defined(HAVE_PKCS7) && !defined(NO_FILESYSTEM) && \
    defined(ASN_BER_TO_DER) && !defined(NO_DES3) && !defined(NO_SHA)
static byte berContent[] = {
    0x30, 0x80, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86,
    0xF7, 0x0D, 0x01, 0x07, 0x03, 0xA0, 0x80, 0x30,
    0x80, 0x02, 0x01, 0x00, 0x31, 0x82, 0x01, 0x48,
    0x30, 0x82, 0x01, 0x44, 0x02, 0x01, 0x00, 0x30,
    0x81, 0xAC, 0x30, 0x81, 0x9E, 0x31, 0x0B, 0x30,
    0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
    0x55, 0x53, 0x31, 0x10, 0x30, 0x0E, 0x06, 0x03,
    0x55, 0x04, 0x08, 0x0C, 0x07, 0x4D, 0x6F, 0x6E,
    0x74, 0x61, 0x6E, 0x61, 0x31, 0x10, 0x30, 0x0E,
    0x06, 0x03, 0x55, 0x04, 0x07, 0x0C, 0x07, 0x42,
    0x6F, 0x7A, 0x65, 0x6D, 0x61, 0x6E, 0x31, 0x15,
    0x30, 0x13, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x0C,
    0x0C, 0x77, 0x6F, 0x6C, 0x66, 0x53, 0x53, 0x4C,
    0x5F, 0x31, 0x30, 0x32, 0x34, 0x31, 0x19, 0x30,
    0x17, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x0C, 0x10,
    0x50, 0x72, 0x6F, 0x67, 0x72, 0x61, 0x6D, 0x6D,
    0x69, 0x6E, 0x67, 0x2D, 0x31, 0x30, 0x32, 0x34,
    0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04,
    0x03, 0x0C, 0x0F, 0x77, 0x77, 0x77, 0x2E, 0x77,
    0x6F, 0x6C, 0x66, 0x73, 0x73, 0x6C, 0x2E, 0x63,
    0x6F, 0x6D, 0x31, 0x1F, 0x30, 0x1D, 0x06, 0x09,
    0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09,
    0x01, 0x16, 0x10, 0x69, 0x6E, 0x66, 0x6F, 0x40,
    0x77, 0x6F, 0x6C, 0x66, 0x73, 0x73, 0x6C, 0x2E,
    0x63, 0x6F, 0x6D, 0x02, 0x09, 0x00, 0xBB, 0xD3,
    0x10, 0x03, 0xE6, 0x9D, 0x28, 0x03, 0x30, 0x0D,
    0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D,
    0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x81, 0x80,
    0x2F, 0xF9, 0x77, 0x4F, 0x04, 0x5C, 0x16, 0x62,
    0xF0, 0x77, 0x8D, 0x95, 0x4C, 0xB1, 0x44, 0x9A,
    0x8C, 0x3C, 0x8C, 0xE4, 0xD1, 0xC1, 0x14, 0x72,
    0xD0, 0x4A, 0x1A, 0x94, 0x27, 0x0F, 0xAA, 0xE8,
    0xD0, 0xA2, 0xE7, 0xED, 0x4C, 0x7F, 0x0F, 0xC7,
    0x1B, 0xFB, 0x81, 0x0E, 0x76, 0x8F, 0xDD, 0x32,
    0x11, 0x68, 0xA0, 0x13, 0xD2, 0x8D, 0x95, 0xEF,
    0x80, 0x53, 0x81, 0x0E, 0x1F, 0xC8, 0xD6, 0x76,
    0x5C, 0x31, 0xD3, 0x77, 0x33, 0x29, 0xA6, 0x1A,
    0xD3, 0xC6, 0x14, 0x36, 0xCA, 0x8E, 0x7D, 0x72,
    0xA0, 0x29, 0x4C, 0xC7, 0x3A, 0xAF, 0xFE, 0xF7,
    0xFC, 0xD7, 0xE2, 0x8F, 0x6A, 0x20, 0x46, 0x09,
    0x40, 0x22, 0x2D, 0x79, 0x38, 0x11, 0xB1, 0x4A,
    0xE3, 0x48, 0xE8, 0x10, 0x37, 0xA0, 0x22, 0xF7,
    0xB4, 0x79, 0xD1, 0xA9, 0x3D, 0xC2, 0xAB, 0x37,
    0xAE, 0x82, 0x68, 0x1A, 0x16, 0xEF, 0x33, 0x0C,
    0x30, 0x80, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86,
    0xF7, 0x0D, 0x01, 0x07, 0x01, 0x30, 0x14, 0x06,
    0x08, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x03,
    0x07, 0x04, 0x08, 0xAD, 0xD0, 0x38, 0x9B, 0x16,
    0x4B, 0x7F, 0x99, 0xA0, 0x80, 0x04, 0x82, 0x03,
    0xE8, 0x6D, 0x48, 0xFB, 0x8A, 0xBD, 0xED, 0x6C,
    0xCD, 0xC6, 0x48, 0xFD, 0xB7, 0xB0, 0x7C, 0x86,
    0x2C, 0x8D, 0xF0, 0x23, 0x12, 0xD8, 0xA3, 0x2A,
    0x21, 0x6F, 0x8B, 0x75, 0xBB, 0x47, 0x7F, 0xC9,
    0xBA, 0xBA, 0xFF, 0x91, 0x09, 0x01, 0x7A, 0x5C,
    0x96, 0x02, 0xB8, 0x8E, 0xF8, 0x67, 0x7E, 0x8F,
    0xF9, 0x51, 0x0E, 0xFF, 0x8E, 0xE2, 0x61, 0xC0,
    0xDF, 0xFA, 0xE2, 0x4C, 0x50, 0x90, 0xAE, 0xA1,
    0x15, 0x38, 0x3D, 0xBE, 0x88, 0xD7, 0x57, 0xC0,
    0x11, 0x44, 0xA2, 0x61, 0x05, 0x49, 0x6A, 0x94,
    0x04, 0x10, 0xD9, 0xC2, 0x2D, 0x15, 0x20, 0x0D,
    0xBD, 0xA2, 0xEF, 0xE4, 0x68, 0xFA, 0x39, 0x75,
    0x7E, 0xD8, 0x64, 0x44, 0xCB, 0xE0, 0x00, 0x6D,
    0x57, 0x4E, 0x8A, 0x17, 0xA9, 0x83, 0x6C, 0x7F,
    0xFE, 0x01, 0xEE, 0xDE, 0x99, 0x3A, 0xB2, 0xFF,
    0xD3, 0x72, 0x78, 0xBA, 0xF1, 0x23, 0x54, 0x48,
    0x02, 0xD8, 0x38, 0xA9, 0x54, 0xE5, 0x4A, 0x81,
    0xB9, 0xC0, 0x67, 0xB2, 0x7D, 0x3C, 0x6F, 0xCE,
    0xA4, 0xDD, 0x34, 0x5F, 0x60, 0xB1, 0xA3, 0x7A,
    0xE4, 0x43, 0xF2, 0x89, 0x64, 0x35, 0x09, 0x32,
    0x51, 0xFB, 0x5C, 0x67, 0x0C, 0x3B, 0xFC, 0x36,
    0x6B, 0x37, 0x43, 0x6C, 0x03, 0xCD, 0x44, 0xC7,
    0x2B, 0x62, 0xD6, 0xD1, 0xF4, 0x07, 0x7B, 0x19,
    0x91, 0xF0, 0xD7, 0xF5, 0x54, 0xBC, 0x0F, 0x42,
    0x6B, 0x69, 0xF7, 0xA3, 0xC8, 0xEE, 0xB9, 0x7A,
    0x9E, 0x3D, 0xDF, 0x53, 0x47, 0xF7, 0x50, 0x67,
    0x00, 0xCF, 0x2B, 0x3B, 0xE9, 0x85, 0xEE, 0xBD,
    0x4C, 0x64, 0x66, 0x0B, 0x77, 0x80, 0x9D, 0xEF,
    0x11, 0x32, 0x77, 0xA8, 0xA4, 0x5F, 0xEE, 0x2D,
    0xE0, 0x43, 0x87, 0x76, 0x87, 0x53, 0x4E, 0xD7,
    0x1A, 0x04, 0x7B, 0xE1, 0xD1, 0xE1, 0xF5, 0x87,
    0x51, 0x13, 0xE0, 0xC2, 0xAA, 0xA3, 0x4B, 0xAA,
    0x9E, 0xB4, 0xA6, 0x1D, 0x4E, 0x28, 0x57, 0x0B,
    0x80, 0x90, 0x81, 0x4E, 0x04, 0xF5, 0x30, 0x8D,
    0x51, 0xCE, 0x57, 0x2F, 0x88, 0xC5, 0x70, 0xC4,
    0x06, 0x8F, 0xDD, 0x37, 0xC1, 0x34, 0x1E, 0x0E,
    0x15, 0x32, 0x23, 0x92, 0xAB, 0x40, 0xEA, 0xF7,
    0x43, 0xE2, 0x1D, 0xE2, 0x4B, 0xC9, 0x91, 0xF4,
    0x63, 0x21, 0x34, 0xDB, 0xE9, 0x86, 0x83, 0x1A,
    0xD2, 0x52, 0xEF, 0x7A, 0xA2, 0xEE, 0xA4, 0x11,
    0x56, 0xD3, 0x6C, 0xF5, 0x6D, 0xE4, 0xA5, 0x2D,
    0x99, 0x02, 0x10, 0xDF, 0x29, 0xC5, 0xE3, 0x0B,
    0xC4, 0xA1, 0xEE, 0x5F, 0x4A, 0x10, 0xEE, 0x85,
    0x73, 0x2A, 0x92, 0x15, 0x2C, 0xC8, 0xF4, 0x8C,
    0xD7, 0x3D, 0xBC, 0xAD, 0x18, 0xE0, 0x59, 0xD3,
    0xEE, 0x75, 0x90, 0x1C, 0xCC, 0x76, 0xC6, 0x64,
    0x17, 0xD2, 0xD0, 0x91, 0xA6, 0xD0, 0xC1, 0x4A,
    0xAA, 0x58, 0x22, 0xEC, 0x45, 0x98, 0xF2, 0xCC,
    0x4C, 0xE4, 0xBF, 0xED, 0xF6, 0x44, 0x72, 0x36,
    0x65, 0x3F, 0xE3, 0xB5, 0x8B, 0x3E, 0x54, 0x9C,
    0x82, 0x86, 0x5E, 0xB0, 0xF2, 0x12, 0xE5, 0x69,
    0xFA, 0x46, 0xA2, 0x54, 0xFC, 0xF5, 0x4B, 0xE0,
    0x24, 0x3B, 0x99, 0x04, 0x1A, 0x7A, 0xF7, 0xD1,
    0xFF, 0x68, 0x97, 0xB2, 0x85, 0x82, 0x95, 0x27,
    0x2B, 0xF4, 0xE7, 0x1A, 0x74, 0x19, 0xEC, 0x8C,
    0x4E, 0xA7, 0x0F, 0xAD, 0x4F, 0x5A, 0x02, 0x80,
    0xC1, 0x6A, 0x9E, 0x54, 0xE4, 0x8E, 0xA3, 0x41,
    0x3F, 0x6F, 0x9C, 0x82, 0x9F, 0x83, 0xB0, 0x44,
    0x01, 0x5F, 0x10, 0x9D, 0xD3, 0xB6, 0x33, 0x5B,
    0xAF, 0xAC, 0x6B, 0x57, 0x2A, 0x01, 0xED, 0x0E,
    0x17, 0xB9, 0x80, 0x76, 0x12, 0x1C, 0x51, 0x56,
    0xDD, 0x6D, 0x94, 0xAB, 0xD2, 0xE5, 0x15, 0x2D,
    0x3C, 0xC5, 0xE8, 0x62, 0x05, 0x8B, 0x40, 0xB1,
    0xC2, 0x83, 0xCA, 0xAC, 0x4B, 0x8B, 0x39, 0xF7,
    0xA0, 0x08, 0x43, 0x5C, 0xF7, 0xE8, 0xED, 0x40,
    0x72, 0x73, 0xE3, 0x6B, 0x18, 0x67, 0xA0, 0xB6,
    0x0F, 0xED, 0x8F, 0x9A, 0xE4, 0x27, 0x62, 0x23,
    0xAA, 0x6D, 0x6C, 0x31, 0xC9, 0x9D, 0x6B, 0xE0,
    0xBF, 0x9D, 0x7D, 0x2E, 0x76, 0x71, 0x06, 0x39,
    0xAC, 0x96, 0x1C, 0xAF, 0x30, 0xF2, 0x62, 0x9C,
    0x84, 0x3F, 0x43, 0x5E, 0x19, 0xA8, 0xE5, 0x3C,
    0x9D, 0x43, 0x3C, 0x43, 0x41, 0xE8, 0x82, 0xE7,
    0x5B, 0xF3, 0xE2, 0x15, 0xE3, 0x52, 0x20, 0xFD,
    0x0D, 0xB2, 0x4D, 0x48, 0xAD, 0x53, 0x7E, 0x0C,
    0xF0, 0xB9, 0xBE, 0xC9, 0x58, 0x4B, 0xC8, 0xA8,
    0xA3, 0x36, 0xF1, 0x2C, 0xD2, 0xE1, 0xC8, 0xC4,
    0x3C, 0x48, 0x70, 0xC2, 0x6D, 0x6C, 0x3D, 0x99,
    0xAC, 0x43, 0x19, 0x69, 0xCA, 0x67, 0x1A, 0xC9,
    0xE1, 0x47, 0xFA, 0x0A, 0xE6, 0x5B, 0x6F, 0x61,
    0xD0, 0x03, 0xE4, 0x03, 0x4B, 0xFD, 0xE2, 0xA5,
    0x8D, 0x83, 0x01, 0x7E, 0xC0, 0x7B, 0x2E, 0x0B,
    0x29, 0xDD, 0xD6, 0xDC, 0x71, 0x46, 0xBD, 0x9A,
    0x40, 0x46, 0x1E, 0x0A, 0xB1, 0x00, 0xE7, 0x71,
    0x29, 0x77, 0xFC, 0x9A, 0x76, 0x8A, 0x5F, 0x66,
    0x9B, 0x63, 0x91, 0x12, 0x78, 0xBF, 0x67, 0xAD,
    0xA1, 0x72, 0x9E, 0xC5, 0x3E, 0xE5, 0xCB, 0xAF,
    0xD6, 0x5A, 0x0D, 0xB6, 0x9B, 0xA3, 0x78, 0xE8,
    0xB0, 0x8F, 0x69, 0xED, 0xC1, 0x73, 0xD5, 0xE5,
    0x1C, 0x18, 0xA0, 0x58, 0x4C, 0x49, 0xBD, 0x91,
    0xCE, 0x15, 0x0D, 0xAA, 0x5A, 0x07, 0xEA, 0x1C,
    0xA7, 0x4B, 0x11, 0x31, 0x80, 0xAF, 0xA1, 0x0A,
    0xED, 0x6C, 0x70, 0xE4, 0xDB, 0x75, 0x86, 0xAE,
    0xBF, 0x4A, 0x05, 0x72, 0xDE, 0x84, 0x8C, 0x7B,
    0x59, 0x81, 0x58, 0xE0, 0xC0, 0x15, 0xB5, 0xF3,
    0xD5, 0x73, 0x78, 0x83, 0x53, 0xDA, 0x92, 0xC1,
    0xE6, 0x71, 0x74, 0xC7, 0x7E, 0xAA, 0x36, 0x06,
    0xF0, 0xDF, 0xBA, 0xFB, 0xEF, 0x54, 0xE8, 0x11,
    0xB2, 0x33, 0xA3, 0x0B, 0x9E, 0x0C, 0x59, 0x75,
    0x13, 0xFA, 0x7F, 0x88, 0xB9, 0x86, 0xBD, 0x1A,
    0xDB, 0x52, 0x12, 0xFB, 0x6D, 0x1A, 0xCB, 0x49,
    0x94, 0x94, 0xC4, 0xA9, 0x99, 0xC0, 0xA4, 0xB6,
    0x60, 0x36, 0x09, 0x94, 0x2A, 0xD5, 0xC4, 0x26,
    0xF4, 0xA3, 0x6A, 0x0E, 0x57, 0x8B, 0x7C, 0xA4,
    0x1D, 0x75, 0xE8, 0x2A, 0xF3, 0xC4, 0x3C, 0x7D,
    0x45, 0x6D, 0xD8, 0x24, 0xD1, 0x3B, 0xF7, 0xCF,
    0xE4, 0x45, 0x2A, 0x55, 0xE5, 0xA9, 0x1F, 0x1C,
    0x8F, 0x55, 0x8D, 0xC1, 0xF7, 0x74, 0xCC, 0x26,
    0xC7, 0xBA, 0x2E, 0x5C, 0xC1, 0x71, 0x0A, 0xAA,
    0xD9, 0x6D, 0x76, 0xA7, 0xF9, 0xD1, 0x18, 0xCB,
    0x5A, 0x52, 0x98, 0xA8, 0x0D, 0x3F, 0x06, 0xFC,
    0x49, 0x11, 0x21, 0x5F, 0x86, 0x19, 0x33, 0x81,
    0xB5, 0x7A, 0xDA, 0xA1, 0x47, 0xBF, 0x7C, 0xD7,
    0x05, 0x96, 0xC7, 0xF5, 0xC1, 0x61, 0xE5, 0x18,
    0xA5, 0x38, 0x68, 0xED, 0xB4, 0x17, 0x62, 0x0D,
    0x01, 0x5E, 0xC3, 0x04, 0xA6, 0xBA, 0xB1, 0x01,
    0x60, 0x5C, 0xC1, 0x3A, 0x34, 0x97, 0xD6, 0xDB,
    0x67, 0x73, 0x4D, 0x33, 0x96, 0x01, 0x67, 0x44,
    0xEA, 0x47, 0x5E, 0x44, 0xB5, 0xE5, 0xD1, 0x6C,
    0x20, 0xA9, 0x6D, 0x4D, 0xBC, 0x02, 0xF0, 0x70,
    0xE4, 0xDD, 0xE9, 0xD5, 0x5C, 0x28, 0x29, 0x0B,
    0xB4, 0x60, 0x2A, 0xF1, 0xF7, 0x1A, 0xF0, 0x36,
    0xAE, 0x51, 0x3A, 0xAE, 0x6E, 0x48, 0x7D, 0xC7,
    0x5C, 0xF3, 0xDC, 0xF6, 0xED, 0x27, 0x4E, 0x8E,
    0x48, 0x18, 0x3E, 0x08, 0xF1, 0xD8, 0x3D, 0x0D,
    0xE7, 0x2F, 0x65, 0x8A, 0x6F, 0xE2, 0x1E, 0x06,
    0xC1, 0x04, 0x58, 0x7B, 0x4A, 0x75, 0x60, 0x92,
    0x13, 0xC6, 0x40, 0x2D, 0x3A, 0x8A, 0xD1, 0x03,
    0x05, 0x1F, 0x28, 0x66, 0xC2, 0x57, 0x2A, 0x4C,
    0xE1, 0xA3, 0xCB, 0xA1, 0x95, 0x30, 0x10, 0xED,
    0xDF, 0xAE, 0x70, 0x49, 0x4E, 0xF6, 0xB4, 0x5A,
    0xB6, 0x22, 0x56, 0x37, 0x05, 0xE7, 0x3E, 0xB2,
    0xE3, 0x96, 0x62, 0xEC, 0x09, 0x53, 0xC0, 0x50,
    0x3D, 0xA7, 0xBC, 0x9B, 0x39, 0x02, 0x26, 0x16,
    0xB5, 0x34, 0x17, 0xD4, 0xCA, 0xFE, 0x1D, 0xE4,
    0x5A, 0xDA, 0x4C, 0xC2, 0xCA, 0x8E, 0x79, 0xBF,
    0xD8, 0x4C, 0xBB, 0xFA, 0x30, 0x7B, 0xA9, 0x3E,
    0x52, 0x19, 0xB1, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00
};
#endif /* HAVE_PKCS7 && !NO_FILESYSTEM && ASN_BER_TO_DER &&
        * !NO_DES3 && !NO_SHA
        */

/*
 * Testing wc_PKCS7_BER()
 */
static int test_wc_PKCS7_BER(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7) && !defined(NO_FILESYSTEM) && \
    !defined(NO_SHA) && defined(ASN_BER_TO_DER)
    PKCS7* pkcs7 = NULL;
    char   fName[] = "./certs/test-ber-exp02-05-2022.p7b";
    XFILE  f = XBADFILE;
    byte   der[4096];
#ifndef NO_DES3
    byte   decoded[2048];
#endif
    word32 derSz = 0;
#if !defined(NO_PKCS7_STREAM) && !defined(NO_RSA)
    word32 z;
    int ret;
#endif /* !NO_PKCS7_STREAM && !NO_RSA */

    ExpectTrue((f = XFOPEN(fName, "rb")) != XBADFILE);
    ExpectTrue((derSz = (word32)XFREAD(der, 1, sizeof(der), f)) > 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }

    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
#ifndef NO_RSA
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, der, derSz), 0);

    #ifndef NO_PKCS7_STREAM
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, INVALID_DEVID), 0);
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);

    /* test for streaming */
    ret = -1;
    for (z = 0; z < derSz && ret != 0; z++) {
        ret = wc_PKCS7_VerifySignedData(pkcs7, der + z, 1);
        if (ret < 0){
            ExpectIntEQ(ret, WC_NO_ERR_TRACE(WC_PKCS7_WANT_READ_E));
        }
    }
    ExpectIntEQ(ret, 0);
    #endif /* !NO_PKCS7_STREAM */
#else
    ExpectIntNE(wc_PKCS7_VerifySignedData(pkcs7, der, derSz), 0);
#endif
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

#ifndef NO_DES3
    /* decode BER content */
    ExpectTrue((f = XFOPEN("./certs/1024/client-cert.der", "rb")) != XBADFILE);
    ExpectTrue((derSz = (word32)XFREAD(der, 1, sizeof(der), f)) > 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
#ifndef NO_RSA
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, der, derSz), 0);
#else
    ExpectIntNE(wc_PKCS7_InitWithCert(pkcs7, der, derSz), 0);
#endif

    ExpectTrue((f = XFOPEN("./certs/1024/client-key.der", "rb")) != XBADFILE);
    ExpectTrue((derSz = (word32)XFREAD(der, 1, sizeof(der), f)) > 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    if (pkcs7 != NULL) {
        pkcs7->privateKey   = der;
        pkcs7->privateKeySz = derSz;
    }
#ifndef NO_RSA
#ifdef WOLFSSL_SP_MATH
    ExpectIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, berContent,
        sizeof(berContent), decoded, sizeof(decoded)), WC_NO_ERR_TRACE(WC_KEY_SIZE_E));
#else
    ExpectIntGT(wc_PKCS7_DecodeEnvelopedData(pkcs7, berContent,
        sizeof(berContent), decoded, sizeof(decoded)), 0);
#endif
#else
    ExpectIntEQ(wc_PKCS7_DecodeEnvelopedData(pkcs7, berContent,
        sizeof(berContent), decoded, sizeof(decoded)), WC_NO_ERR_TRACE(NOT_COMPILED_IN));
#endif
    wc_PKCS7_Free(pkcs7);
#endif /* !NO_DES3 */
#endif
    return EXPECT_RESULT();
} /* END test_wc_PKCS7_BER() */

static int test_wc_PKCS7_signed_enveloped(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7) && !defined(NO_RSA) && !defined(NO_AES) && \
    !defined(NO_FILESYSTEM)
    XFILE  f = XBADFILE;
    PKCS7* pkcs7 = NULL;
#ifdef HAVE_AES_CBC
    PKCS7* inner = NULL;
#endif
    WC_RNG rng;
    unsigned char key[FOURK_BUF/2];
    unsigned char cert[FOURK_BUF/2];
    unsigned char env[FOURK_BUF];
    int envSz  = FOURK_BUF;
    int keySz = 0;
    int certSz = 0;
    unsigned char sig[FOURK_BUF * 2];
    int sigSz = FOURK_BUF * 2;
#ifdef HAVE_AES_CBC
    unsigned char decoded[FOURK_BUF];
    int decodedSz = FOURK_BUF;
#endif
#ifndef NO_PKCS7_STREAM
    int z;
    int ret;
#endif /* !NO_PKCS7_STREAM */

    XMEMSET(&rng, 0, sizeof(WC_RNG));

    /* load cert */
    ExpectTrue((f = XFOPEN(cliCertDerFile, "rb")) != XBADFILE);
    ExpectIntGT((certSz = (int)XFREAD(cert, 1, sizeof(cert), f)), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }

    /* load key */
    ExpectTrue((f = XFOPEN(cliKeyFile, "rb")) != XBADFILE);
    ExpectIntGT((keySz = (int)XFREAD(key, 1, sizeof(key), f)), 0);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }
    ExpectIntGT(keySz = wolfSSL_KeyPemToDer(key, keySz, key, keySz, NULL), 0);

    /* sign cert for envelope */
    ExpectNotNull(pkcs7 = wc_PKCS7_New(NULL, 0));
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, (word32)certSz), 0);
    if (pkcs7 != NULL) {
        pkcs7->content    = cert;
        pkcs7->contentSz  = (word32)certSz;
        pkcs7->contentOID = DATA;
        pkcs7->privateKey   = key;
        pkcs7->privateKeySz = (word32)keySz;
        pkcs7->encryptOID   = RSAk;
        pkcs7->hashOID      = SHA256h;
        pkcs7->rng          = &rng;
    }
    ExpectIntGT((sigSz = wc_PKCS7_EncodeSignedData(pkcs7, sig, (word32)sigSz)), 0);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
    DoExpectIntEQ(wc_FreeRng(&rng), 0);

#ifdef HAVE_AES_CBC
    /* create envelope */
    ExpectNotNull(pkcs7 = wc_PKCS7_New(NULL, 0));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, (word32)certSz), 0);
    if (pkcs7 != NULL) {
        pkcs7->content   = sig;
        pkcs7->contentSz = (word32)sigSz;
        pkcs7->contentOID = DATA;
        pkcs7->encryptOID = AES256CBCb;
        pkcs7->privateKey   = key;
        pkcs7->privateKeySz = (word32)keySz;
    }
    ExpectIntGT((envSz = wc_PKCS7_EncodeEnvelopedData(pkcs7, env, (word32)envSz)), 0);
    ExpectIntLT(wc_PKCS7_EncodeEnvelopedData(pkcs7, env, 2), 0);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
#endif

    /* create bad signed enveloped data */
    sigSz = FOURK_BUF * 2;
    ExpectNotNull(pkcs7 = wc_PKCS7_New(NULL, 0));
    ExpectIntEQ(wc_InitRng(&rng), 0);
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, (word32)certSz), 0);
    if (pkcs7 != NULL) {
        pkcs7->content    = env;
        pkcs7->contentSz  = (word32)envSz;
        pkcs7->contentOID = DATA;
        pkcs7->privateKey   = key;
        pkcs7->privateKeySz = (word32)keySz;
        pkcs7->encryptOID   = RSAk;
        pkcs7->hashOID      = SHA256h;
        pkcs7->rng = &rng;
    }

    /* Set no certs in bundle for this test. */
    if (pkcs7 != NULL) {
        ExpectIntEQ(wc_PKCS7_SetNoCerts(pkcs7, 1), 0);
        ExpectIntEQ(wc_PKCS7_SetNoCerts(NULL, 1), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
        ExpectIntEQ(wc_PKCS7_GetNoCerts(pkcs7), 1);
    }
    ExpectIntGT((sigSz = wc_PKCS7_EncodeSignedData(pkcs7, sig, (word32)sigSz)), 0);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

    /* check verify fails */
    ExpectNotNull(pkcs7 = wc_PKCS7_New(NULL, 0));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, sig, (word32)sigSz),
            WC_NO_ERR_TRACE(PKCS7_SIGNEEDS_CHECK));

    /* try verifying the signature manually */
    {
        RsaKey rKey;
        word32 idx = 0;
        byte digest[MAX_SEQ_SZ + MAX_ALGO_SZ + MAX_OCTET_STR_SZ +
            WC_MAX_DIGEST_SIZE];
        int  digestSz = 0;

        ExpectIntEQ(wc_InitRsaKey(&rKey, HEAP_HINT), 0);
        ExpectIntEQ(wc_RsaPrivateKeyDecode(key, &idx, &rKey, (word32)keySz), 0);
        ExpectIntGT(digestSz = wc_RsaSSL_Verify(pkcs7->signature,
            pkcs7->signatureSz, digest, sizeof(digest), &rKey), 0);
        ExpectIntEQ(digestSz, pkcs7->pkcs7DigestSz);
        ExpectIntEQ(XMEMCMP(digest, pkcs7->pkcs7Digest, digestSz), 0);
        ExpectIntEQ(wc_FreeRsaKey(&rKey), 0);
        /* verify was success */
    }

    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

    /* initializing the PKCS7 struct with the signing certificate should pass */
    ExpectNotNull(pkcs7 = wc_PKCS7_New(NULL, 0));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, (word32)certSz), 0);
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, sig, (word32)sigSz), 0);

#ifndef NO_PKCS7_STREAM
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
    ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, (word32)certSz), 0);

    /* test for streaming */
    ret = -1;
    for (z = 0; z < sigSz && ret != 0; z++) {
        ret = wc_PKCS7_VerifySignedData(pkcs7, sig + z, 1);
        if (ret < 0){
            ExpectIntEQ(ret, WC_NO_ERR_TRACE(WC_PKCS7_WANT_READ_E));
        }
    }
    ExpectIntEQ(ret, 0);
#endif /* !NO_PKCS7_STREAM */

    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

    /* create valid degenerate bundle */
    sigSz = FOURK_BUF * 2;
    ExpectNotNull(pkcs7 = wc_PKCS7_New(NULL, 0));
    if (pkcs7 != NULL) {
        pkcs7->content    = env;
        pkcs7->contentSz  = (word32)envSz;
        pkcs7->contentOID = DATA;
        pkcs7->privateKey   = key;
        pkcs7->privateKeySz = (word32)keySz;
        pkcs7->encryptOID   = RSAk;
        pkcs7->hashOID      = SHA256h;
        pkcs7->rng = &rng;
    }
    ExpectIntEQ(wc_PKCS7_SetSignerIdentifierType(pkcs7, DEGENERATE_SID), 0);
    ExpectIntGT((sigSz = wc_PKCS7_EncodeSignedData(pkcs7, sig, (word32)sigSz)), 0);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
    wc_FreeRng(&rng);

    /* check verify */
    ExpectNotNull(pkcs7 = wc_PKCS7_New(NULL, 0));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, testDevId), 0);
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, sig, (word32)sigSz), 0);
    ExpectNotNull(pkcs7->content);

#ifndef NO_PKCS7_STREAM
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

    /* create valid degenerate bundle */
    sigSz = FOURK_BUF * 2;
    ExpectNotNull(pkcs7 = wc_PKCS7_New(NULL, 0));
    if (pkcs7 != NULL) {
        pkcs7->content    = env;
        pkcs7->contentSz  = (word32)envSz;
        pkcs7->contentOID = DATA;
        pkcs7->privateKey   = key;
        pkcs7->privateKeySz = (word32)keySz;
        pkcs7->encryptOID   = RSAk;
        pkcs7->hashOID      = SHA256h;
        pkcs7->rng = &rng;
    }
    ExpectIntEQ(wc_PKCS7_SetSignerIdentifierType(pkcs7, DEGENERATE_SID), 0);
    ExpectIntGT((sigSz = wc_PKCS7_EncodeSignedData(pkcs7, sig, (word32)sigSz)), 0);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
    wc_FreeRng(&rng);

    /* check verify */
    ExpectNotNull(pkcs7 = wc_PKCS7_New(NULL, 0));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, HEAP_HINT, testDevId), 0);
    /* test for streaming */
    ret = -1;
    for (z = 0; z < sigSz && ret != 0; z++) {
        ret = wc_PKCS7_VerifySignedData(pkcs7, sig + z, 1);
        if (ret < 0){
            ExpectIntEQ(ret, WC_NO_ERR_TRACE(WC_PKCS7_WANT_READ_E));
        }
    }
    ExpectIntEQ(ret, 0);
#endif /* !NO_PKCS7_STREAM */

#ifdef HAVE_AES_CBC
    /* check decode */
    ExpectNotNull(inner = wc_PKCS7_New(NULL, 0));
    ExpectIntEQ(wc_PKCS7_InitWithCert(inner, cert, (word32)certSz), 0);
    if (inner != NULL) {
        inner->privateKey   = key;
        inner->privateKeySz = (word32)keySz;
    }
    ExpectIntGT((decodedSz = wc_PKCS7_DecodeEnvelopedData(inner, pkcs7->content,
                   pkcs7->contentSz, decoded, (word32)decodedSz)), 0);
    wc_PKCS7_Free(inner);
    inner = NULL;
#endif
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

#ifdef HAVE_AES_CBC
    /* check cert set */
    ExpectNotNull(pkcs7 = wc_PKCS7_New(NULL, 0));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    ExpectIntEQ(wc_PKCS7_VerifySignedData(pkcs7, decoded, (word32)decodedSz), 0);
    ExpectNotNull(pkcs7->singleCert);
    ExpectIntNE(pkcs7->singleCertSz, 0);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

#ifndef NO_PKCS7_STREAM
    ExpectNotNull(pkcs7 = wc_PKCS7_New(NULL, 0));
    ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, NULL, 0), 0);
    /* test for streaming */
    ret = -1;
    for (z = 0; z < decodedSz && ret != 0; z++) {
        ret = wc_PKCS7_VerifySignedData(pkcs7, decoded + z, 1);
        if (ret < 0){
            ExpectIntEQ(ret, WC_NO_ERR_TRACE(WC_PKCS7_WANT_READ_E));
        }
    }
    ExpectIntEQ(ret, 0);
    ExpectNotNull(pkcs7->singleCert);
    ExpectIntNE(pkcs7->singleCertSz, 0);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;
#endif /* !NO_PKCS7_STREAM */
#endif

    {
        /* arbitrary custom SKID */
        const byte customSKID[] = {
            0x40, 0x25, 0x77, 0x56
        };

        ExpectIntEQ(wc_InitRng(&rng), 0);
        sigSz = FOURK_BUF * 2;
        ExpectNotNull(pkcs7 = wc_PKCS7_New(HEAP_HINT, testDevId));
        if (pkcs7 != NULL) {
            ExpectIntEQ(wc_PKCS7_InitWithCert(pkcs7, cert, (word32)certSz), 0);
            pkcs7->content    = cert;
            pkcs7->contentSz  = (word32)certSz;
            pkcs7->contentOID = DATA;
            pkcs7->privateKey   = key;
            pkcs7->privateKeySz = (word32)keySz;
            pkcs7->encryptOID   = RSAk;
            pkcs7->hashOID      = SHA256h;
            pkcs7->rng          = &rng;
            ExpectIntEQ(wc_PKCS7_SetSignerIdentifierType(pkcs7, CMS_SKID), 0);
            ExpectIntEQ(wc_PKCS7_SetCustomSKID(pkcs7, customSKID,
                        sizeof(customSKID)), 0);
            ExpectIntGT((sigSz = wc_PKCS7_EncodeSignedData(pkcs7, sig,
                (word32)sigSz)), 0);
        }
        wc_PKCS7_Free(pkcs7);
        pkcs7 = NULL;
        wc_FreeRng(&rng);
    }
#endif /* HAVE_PKCS7 && !NO_RSA && !NO_AES */
    return EXPECT_RESULT();
}

static int test_wc_PKCS7_NoDefaultSignedAttribs(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7) && !defined(NO_FILESYSTEM) && !defined(NO_RSA) \
    && !defined(NO_AES)
    PKCS7* pkcs7 = NULL;
    void*  heap = NULL;

    ExpectNotNull(pkcs7 = wc_PKCS7_New(heap, testDevId));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, heap, testDevId), 0);

    ExpectIntEQ(wc_PKCS7_NoDefaultSignedAttribs(NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_PKCS7_NoDefaultSignedAttribs(pkcs7), 0);

    wc_PKCS7_Free(pkcs7);
#endif
    return EXPECT_RESULT();
}

static int test_wc_PKCS7_SetOriEncryptCtx(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7) && !defined(NO_FILESYSTEM) && !defined(NO_RSA) \
    && !defined(NO_AES)
    PKCS7*       pkcs7 = NULL;
    void*        heap = NULL;
    WOLFSSL_CTX* ctx = NULL;

    ExpectNotNull(pkcs7 = wc_PKCS7_New(heap, testDevId));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, heap, testDevId), 0);

    ExpectIntEQ(wc_PKCS7_SetOriEncryptCtx(NULL, ctx), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_PKCS7_SetOriEncryptCtx(pkcs7, ctx), 0);

    wc_PKCS7_Free(pkcs7);
#endif
    return EXPECT_RESULT();
}

static int test_wc_PKCS7_SetOriDecryptCtx(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7) && !defined(NO_FILESYSTEM) && !defined(NO_RSA) \
    && !defined(NO_AES)
    PKCS7*       pkcs7 = NULL;
    void*        heap = NULL;
    WOLFSSL_CTX* ctx = NULL;

    ExpectNotNull(pkcs7 = wc_PKCS7_New(heap, testDevId));
    ExpectIntEQ(wc_PKCS7_Init(pkcs7, heap, testDevId), 0);

    ExpectIntEQ(wc_PKCS7_SetOriDecryptCtx(NULL, ctx), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_PKCS7_SetOriDecryptCtx(pkcs7, ctx), 0);

    wc_PKCS7_Free(pkcs7);
#endif
    return EXPECT_RESULT();
}

static int test_wc_PKCS7_DecodeCompressedData(void)
{
    EXPECT_DECLS;
#if defined(HAVE_PKCS7) && !defined(NO_FILESYSTEM) && !defined(NO_RSA) \
    && !defined(NO_AES) && defined(HAVE_LIBZ)
    PKCS7* pkcs7 = NULL;
    void*  heap = NULL;
    byte   out[4096];
    byte*  decompressed = NULL;
    int    outSz;
    int    decompressedSz;
    const char* cert = "./certs/client-cert.pem";
    byte*  cert_buf = NULL;
    size_t cert_sz = 0;

    ExpectIntEQ(load_file(cert, &cert_buf, &cert_sz), 0);
    ExpectNotNull((decompressed = (byte*)XMALLOC(cert_sz, heap,
        DYNAMIC_TYPE_TMP_BUFFER)));
    decompressedSz = (int)cert_sz;
    ExpectNotNull((pkcs7 = wc_PKCS7_New(heap, testDevId)));

    if (pkcs7 != NULL) {
        pkcs7->content    = (byte*)cert_buf;
        pkcs7->contentSz  = (word32)cert_sz;
        pkcs7->contentOID = DATA;
    }

    ExpectIntGT((outSz = wc_PKCS7_EncodeCompressedData(pkcs7, out,
        sizeof(out))), 0);
    wc_PKCS7_Free(pkcs7);
    pkcs7 = NULL;

    /* compressed key should be smaller than when started */
    ExpectIntLT(outSz, cert_sz);

    /* test decompression */
    ExpectNotNull((pkcs7 = wc_PKCS7_New(heap, testDevId)));
    ExpectIntEQ(pkcs7->contentOID, 0);

    /* fail case with out buffer too small */
    ExpectIntLT(wc_PKCS7_DecodeCompressedData(pkcs7, out, outSz,
        decompressed, outSz), 0);

    /* success case */
    ExpectIntEQ(wc_PKCS7_DecodeCompressedData(pkcs7, out, outSz,
        decompressed, decompressedSz), cert_sz);
    ExpectIntEQ(pkcs7->contentOID, DATA);
    ExpectIntEQ(XMEMCMP(decompressed, cert_buf, cert_sz), 0);
    XFREE(decompressed, heap, DYNAMIC_TYPE_TMP_BUFFER);
    decompressed = NULL;

    /* test decompression function with different 'max' inputs */
    outSz = sizeof(out);
    ExpectIntGT((outSz = wc_Compress(out, outSz, cert_buf, (word32)cert_sz, 0)),
        0);
    ExpectIntLT(wc_DeCompressDynamic(&decompressed, 1, DYNAMIC_TYPE_TMP_BUFFER,
        out, outSz, 0, heap), 0);
    ExpectNull(decompressed);
    ExpectIntGT(wc_DeCompressDynamic(&decompressed, -1, DYNAMIC_TYPE_TMP_BUFFER,
        out, outSz, 0, heap), 0);
    ExpectNotNull(decompressed);
    ExpectIntEQ(XMEMCMP(decompressed, cert_buf, cert_sz), 0);
    XFREE(decompressed, heap, DYNAMIC_TYPE_TMP_BUFFER);
    decompressed = NULL;

    ExpectIntGT(wc_DeCompressDynamic(&decompressed, DYNAMIC_TYPE_TMP_BUFFER, 5,
        out, outSz, 0, heap), 0);
    ExpectNotNull(decompressed);
    ExpectIntEQ(XMEMCMP(decompressed, cert_buf, cert_sz), 0);
    XFREE(decompressed, heap, DYNAMIC_TYPE_TMP_BUFFER);

    if (cert_buf != NULL)
        free(cert_buf);
    wc_PKCS7_Free(pkcs7);
#endif
    return EXPECT_RESULT();
}

static int test_wc_i2d_PKCS12(void)
{
    EXPECT_DECLS;
#if !defined(NO_ASN) && !defined(NO_PWDBASED) && defined(HAVE_PKCS12) \
    && !defined(NO_FILESYSTEM) && !defined(NO_RSA) \
    && !defined(NO_AES) && !defined(NO_SHA)
    WC_PKCS12* pkcs12 = NULL;
    unsigned char der[FOURK_BUF * 2];
    unsigned char* pt;
    int derSz = 0;
    unsigned char out[FOURK_BUF * 2];
    int outSz = FOURK_BUF * 2;
    const char p12_f[] = "./certs/test-servercert.p12";
    XFILE f = XBADFILE;

    ExpectTrue((f =  XFOPEN(p12_f, "rb")) != XBADFILE);
    ExpectIntGT(derSz = (int)XFREAD(der, 1, sizeof(der), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);

    ExpectNotNull(pkcs12 = wc_PKCS12_new());
    ExpectIntEQ(wc_d2i_PKCS12(der, (word32)derSz, pkcs12), 0);
    ExpectIntEQ(wc_i2d_PKCS12(pkcs12, NULL, &outSz), WC_NO_ERR_TRACE(LENGTH_ONLY_E));
    ExpectIntEQ(outSz, derSz);

    outSz = derSz - 1;
    pt = out;
    ExpectIntLE(wc_i2d_PKCS12(pkcs12, &pt, &outSz), 0);

    outSz = derSz;
    ExpectIntEQ(wc_i2d_PKCS12(pkcs12, &pt, &outSz), derSz);
    ExpectIntEQ((pt == out), 0);

    pt = NULL;
    ExpectIntEQ(wc_i2d_PKCS12(pkcs12, &pt, NULL), derSz);
    XFREE(pt, NULL, DYNAMIC_TYPE_PKCS);
    wc_PKCS12_free(pkcs12);
    pkcs12 = NULL;

    /* Run the same test but use wc_d2i_PKCS12_fp. */
    ExpectNotNull(pkcs12 = wc_PKCS12_new());
    ExpectIntEQ(wc_d2i_PKCS12_fp("./certs/test-servercert.p12", &pkcs12), 0);
    ExpectIntEQ(wc_i2d_PKCS12(pkcs12, NULL, &outSz), WC_NO_ERR_TRACE(LENGTH_ONLY_E));
    ExpectIntEQ(outSz, derSz);
    wc_PKCS12_free(pkcs12);
    pkcs12 = NULL;

    /* wc_d2i_PKCS12_fp can also allocate the PKCS12 object for the caller. */
    ExpectIntEQ(wc_d2i_PKCS12_fp("./certs/test-servercert.p12", &pkcs12), 0);
    ExpectIntEQ(wc_i2d_PKCS12(pkcs12, NULL, &outSz), WC_NO_ERR_TRACE(LENGTH_ONLY_E));
    ExpectIntEQ(outSz, derSz);
    wc_PKCS12_free(pkcs12);
    pkcs12 = NULL;
#endif
    return EXPECT_RESULT();
}


/*----------------------------------------------------------------------------*
 | ASN.1 Tests
 *----------------------------------------------------------------------------*/

static int test_wolfSSL_ASN1_BIT_STRING(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && defined(OPENSSL_ALL)
    ASN1_BIT_STRING* str = NULL;
    ASN1_BIT_STRING* str2 = NULL;
    unsigned char* der = NULL;

    ExpectNotNull(str = ASN1_BIT_STRING_new());
    /* Empty data testing. */
    ExpectIntEQ(ASN1_BIT_STRING_get_bit(str, 1), 0);
    ASN1_BIT_STRING_free(str);
    str = NULL;

    ExpectNotNull(str = ASN1_BIT_STRING_new());

    /* Invalid parameter testing. */
    ExpectIntEQ(ASN1_BIT_STRING_set_bit(NULL, 42, 1), 0);
    ExpectIntEQ(ASN1_BIT_STRING_set_bit(str, -1, 1), 0);
    ExpectIntEQ(ASN1_BIT_STRING_set_bit(str, 42, 2), 0);
    ExpectIntEQ(ASN1_BIT_STRING_set_bit(str, 42, -1), 0);

    /* No bit string - bit is always 0. */
    ExpectIntEQ(ASN1_BIT_STRING_get_bit(NULL, 42), 0);
    ExpectIntEQ(ASN1_BIT_STRING_get_bit(NULL, -1), 0);
    ExpectIntEQ(ASN1_BIT_STRING_get_bit(str, -1), 0);
    ExpectIntEQ(ASN1_BIT_STRING_get_bit(str, 0), 0);

    ExpectIntEQ(ASN1_BIT_STRING_set_bit(str, 42, 1), 1);
    ExpectIntEQ(ASN1_BIT_STRING_get_bit(str, 42), 1);
    ExpectIntEQ(ASN1_BIT_STRING_get_bit(str, 41), 0);
    ExpectIntEQ(ASN1_BIT_STRING_get_bit(str, -1), 0);
    ExpectIntEQ(ASN1_BIT_STRING_set_bit(str, 84, 1), 1);
    ExpectIntEQ(ASN1_BIT_STRING_get_bit(str, 84), 1);
    ExpectIntEQ(ASN1_BIT_STRING_get_bit(str, 83), 0);
    ExpectIntEQ(ASN1_BIT_STRING_set_bit(str, 91, 0), 1);
    ExpectIntEQ(ASN1_BIT_STRING_get_bit(str, 91), 0);
    ExpectIntEQ(ASN1_BIT_STRING_set_bit(str, 89, 0), 1);
    ExpectIntEQ(ASN1_BIT_STRING_get_bit(str, 89), 0);
    ExpectIntEQ(ASN1_BIT_STRING_set_bit(str, 42, 0), 1);
    ExpectIntEQ(ASN1_BIT_STRING_get_bit(str, 42), 0);

    ExpectIntEQ(i2d_ASN1_BIT_STRING(str, NULL), 14);
    ExpectIntEQ(i2d_ASN1_BIT_STRING(str, &der), 14);
#ifdef WOLFSSL_ASN_TEMPLATE
    {
        const unsigned char* tmp = der;
        ExpectNotNull(d2i_ASN1_BIT_STRING(&str2, &tmp, 14));
    }
#endif

    ASN1_BIT_STRING_free(str);
    ASN1_BIT_STRING_free(str2);
    ASN1_BIT_STRING_free(NULL);
    XFREE(der, NULL, DYNAMIC_TYPE_ASN1);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_ASN1_INTEGER(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_ASN)
    ASN1_INTEGER* a = NULL;
    ASN1_INTEGER* dup = NULL;
    const unsigned char invalidLenDer[] = {
        0x02, 0x20, 0x00
    };
    const unsigned char longDer[] = {
        0x02, 0x20,
            0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
            0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
            0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
            0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08
    };
    const unsigned char* p;

    /* Invalid parameter testing. */
    ASN1_INTEGER_free(NULL);
    ExpectNull(wolfSSL_ASN1_INTEGER_dup(NULL));

    ExpectNotNull(a = ASN1_INTEGER_new());
    ExpectNotNull(dup = wolfSSL_ASN1_INTEGER_dup(a));
    ASN1_INTEGER_free(dup);
    dup = NULL;
    ASN1_INTEGER_free(a);
    a = NULL;

    p = invalidLenDer;
    ExpectNull(d2i_ASN1_INTEGER(NULL, &p, sizeof(invalidLenDer)));

    p = longDer;
    ExpectNotNull(a = d2i_ASN1_INTEGER(NULL, &p, sizeof(longDer)));
    ExpectPtrNE(p, longDer);
    ExpectNotNull(dup = wolfSSL_ASN1_INTEGER_dup(a));
    ASN1_INTEGER_free(dup);
    ASN1_INTEGER_free(a);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_ASN1_INTEGER_cmp(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_ASN)
    ASN1_INTEGER* a = NULL;
    ASN1_INTEGER* b = NULL;

    ExpectNotNull(a = ASN1_INTEGER_new());
    ExpectNotNull(b = ASN1_INTEGER_new());
    ExpectIntEQ(ASN1_INTEGER_set(a, 1), 1);
    ExpectIntEQ(ASN1_INTEGER_set(b, 1), 1);

    /* Invalid parameter testing. */
    ExpectIntEQ(wolfSSL_ASN1_INTEGER_cmp(NULL, NULL), -1);
    ExpectIntEQ(wolfSSL_ASN1_INTEGER_cmp(a, NULL), -1);
    ExpectIntEQ(wolfSSL_ASN1_INTEGER_cmp(NULL, b), -1);

    ExpectIntEQ(wolfSSL_ASN1_INTEGER_cmp(a, b), 0);
    ExpectIntEQ(ASN1_INTEGER_set(b, -1), 1);
    ExpectIntGT(wolfSSL_ASN1_INTEGER_cmp(a, b), 0);
    ExpectIntEQ(ASN1_INTEGER_set(a, -2), 1);
    ExpectIntLT(wolfSSL_ASN1_INTEGER_cmp(a, b), 0);
    ExpectIntEQ(ASN1_INTEGER_set(b, 1), 1);
    ExpectIntLT(wolfSSL_ASN1_INTEGER_cmp(a, b), 0);
    ExpectIntEQ(ASN1_INTEGER_set(a, 0x01), 1);
    ExpectIntEQ(ASN1_INTEGER_set(b, 0x1000), 1);
    ExpectIntLT(wolfSSL_ASN1_INTEGER_cmp(a, b), 0);
    ExpectIntGT(wolfSSL_ASN1_INTEGER_cmp(b, a), 0);

    ASN1_INTEGER_free(b);
    ASN1_INTEGER_free(a);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_ASN1_INTEGER_BN(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_ASN)
    ASN1_INTEGER* ai = NULL;
    ASN1_INTEGER* ai2 = NULL;
    BIGNUM* bn = NULL;
    BIGNUM* bn2 = NULL;

    ExpectNotNull(ai = ASN1_INTEGER_new());
    ExpectNotNull(bn2 = BN_new());

    /* Invalid parameter testing. */
    ExpectNull(bn = ASN1_INTEGER_to_BN(NULL, NULL));
    ExpectNull(ai2 = BN_to_ASN1_INTEGER(NULL, NULL));

    /* at the moment hard setting since no set function */
    if (ai != NULL) {
        ai->data[0] = 0xff; /* No DER encoding. */
        ai->length = 1;
    }
#if defined(WOLFSSL_QT) || defined(WOLFSSL_HAPROXY)
    ExpectNotNull(bn = ASN1_INTEGER_to_BN(ai, NULL));
    BN_free(bn);
    bn = NULL;
#else
    ExpectNull(ASN1_INTEGER_to_BN(ai, NULL));
#endif

    if (ai != NULL) {
        ai->data[0] = 0x02; /* tag for ASN_INTEGER */
        ai->data[1] = 0x04; /* bad length of integer */
        ai->data[2] = 0x03;
        ai->length = 3;
    }
#if defined(WOLFSSL_QT) || defined(WOLFSSL_HAPROXY)
    /* Interpreted as a number 0x020403. */
    ExpectNotNull(bn = ASN1_INTEGER_to_BN(ai, NULL));
    BN_free(bn);
    bn = NULL;
#else
    ExpectNull(ASN1_INTEGER_to_BN(ai, NULL));
#endif

    if (ai != NULL) {
        ai->data[0] = 0x02; /* tag for ASN_INTEGER */
        ai->data[1] = 0x01; /* length of integer */
        ai->data[2] = 0x03;
        ai->length = 3;
    }
    ExpectNotNull(bn = ASN1_INTEGER_to_BN(ai, NULL));
    ExpectNotNull(ai2 = BN_to_ASN1_INTEGER(bn, NULL));
    ExpectIntEQ(ASN1_INTEGER_cmp(ai, ai2), 0);
    ExpectNotNull(bn2 = ASN1_INTEGER_to_BN(ai2, bn2));
    ExpectIntEQ(BN_cmp(bn, bn2), 0);

    if (ai != NULL) {
        ai->data[0] = 0x02; /* tag for ASN_INTEGER */
        ai->data[1] = 0x02; /* length of integer */
        ai->data[2] = 0x00; /* padding byte to ensure positive */
        ai->data[3] = 0xff;
        ai->length = 4;
    }
    ExpectNotNull(bn = ASN1_INTEGER_to_BN(ai, bn));
    ExpectNotNull(ai2 = BN_to_ASN1_INTEGER(bn, ai2));
    ExpectIntEQ(ASN1_INTEGER_cmp(ai, ai2), 0);
    ExpectNotNull(bn2 = ASN1_INTEGER_to_BN(ai2, bn2));
    ExpectIntEQ(BN_cmp(bn, bn2), 0);

    if (ai != NULL) {
        ai->data[0] = 0x02; /* tag for ASN_INTEGER */
        ai->data[1] = 0x01; /* length of integer */
        ai->data[2] = 0x00;
        ai->length = 3;
    }
    ExpectNotNull(bn = ASN1_INTEGER_to_BN(ai, bn));
    ExpectNotNull(ai2 = BN_to_ASN1_INTEGER(bn, ai2));
    ExpectIntEQ(ASN1_INTEGER_cmp(ai, ai2), 0);
    ExpectNotNull(bn2 = ASN1_INTEGER_to_BN(ai2, bn2));
    ExpectIntEQ(BN_cmp(bn, bn2), 0);

    if (ai != NULL) {
        ai->data[0] = 0x02; /* tag for ASN_INTEGER */
        ai->data[1] = 0x01; /* length of integer */
        ai->data[2] = 0x01;
        ai->length = 3;
        ai->negative = 1;
    }
    ExpectNotNull(bn = ASN1_INTEGER_to_BN(ai, bn));
    ExpectNotNull(ai2 = BN_to_ASN1_INTEGER(bn, ai2));
    ExpectIntEQ(ASN1_INTEGER_cmp(ai, ai2), 0);
    ExpectNotNull(bn2 = ASN1_INTEGER_to_BN(ai2, bn2));
    ExpectIntEQ(BN_cmp(bn, bn2), 0);

    BN_free(bn2);
    BN_free(bn);
    ASN1_INTEGER_free(ai2);
    ASN1_INTEGER_free(ai);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_ASN1_INTEGER_get_set(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_ASN)
    ASN1_INTEGER *a = NULL;
    long val;

    ExpectNotNull(a = ASN1_INTEGER_new());
    /* Invalid parameter testing. */
    ExpectIntEQ(ASN1_INTEGER_get(NULL), 0);
#if defined(WOLFSSL_QT) || defined(WOLFSSL_HAPROXY)
    ExpectIntEQ(ASN1_INTEGER_get(a), 0);
#else
    ExpectIntEQ(ASN1_INTEGER_get(a), -1);
#endif
    ASN1_INTEGER_free(a);
    a = NULL;

    ExpectNotNull(a = ASN1_INTEGER_new());
    val = 0;
    ExpectIntEQ(ASN1_INTEGER_set(NULL, val), 0);
    ASN1_INTEGER_free(a);
    a = NULL;

    /* 0 */
    ExpectNotNull(a = ASN1_INTEGER_new());
    val = 0;
    ExpectIntEQ(ASN1_INTEGER_set(a, val), 1);
    ExpectTrue(ASN1_INTEGER_get(a) == val);
    ASN1_INTEGER_free(a);
    a = NULL;

    /* 40 */
    ExpectNotNull(a = ASN1_INTEGER_new());
    val = 40;
    ExpectIntEQ(ASN1_INTEGER_set(a, val), 1);
    ExpectTrue(ASN1_INTEGER_get(a) == val);
    ASN1_INTEGER_free(a);
    a = NULL;

    /* -40 */
    ExpectNotNull(a = ASN1_INTEGER_new());
    val = -40;
    ExpectIntEQ(ASN1_INTEGER_set(a, val), 1);
    ExpectTrue(ASN1_INTEGER_get(a) == val);
    ASN1_INTEGER_free(a);
    a = NULL;

    /* 128 */
    ExpectNotNull(a = ASN1_INTEGER_new());
    val = 128;
    ExpectIntEQ(ASN1_INTEGER_set(a, val), 1);
    ExpectTrue(ASN1_INTEGER_get(a) == val);
    ASN1_INTEGER_free(a);
    a = NULL;

    /* -128 */
    ExpectNotNull(a = ASN1_INTEGER_new());
    val = -128;
    ExpectIntEQ(ASN1_INTEGER_set(a, val), 1);
    ExpectTrue(ASN1_INTEGER_get(a) == val);
    ASN1_INTEGER_free(a);
    a = NULL;

    /* 200 */
    ExpectNotNull(a = ASN1_INTEGER_new());
    val = 200;
    ExpectIntEQ(ASN1_INTEGER_set(a, val), 1);
    ExpectTrue(ASN1_INTEGER_get(a) == val);
    ASN1_INTEGER_free(a);
    a = NULL;

    /* int max (2147483647) */
    ExpectNotNull(a = ASN1_INTEGER_new());
    val = 2147483647;
    ExpectIntEQ(ASN1_INTEGER_set(a, val), 1);
    ExpectTrue(ASN1_INTEGER_get(a) == val);
    ASN1_INTEGER_free(a);
    a = NULL;

    /* int min (-2147483648) */
    ExpectNotNull(a = ASN1_INTEGER_new());
    val = -2147483647 - 1;
    ExpectIntEQ(ASN1_INTEGER_set(a, val), 1);
    ExpectTrue(ASN1_INTEGER_get(a) == val);
    ASN1_INTEGER_free(a);
    a = NULL;

    /* long max positive */
    ExpectNotNull(a = ASN1_INTEGER_new());
    val = (long)(((unsigned long)-1) >> 1);
    ExpectIntEQ(ASN1_INTEGER_set(a, val), 1);
    ExpectTrue(ASN1_INTEGER_get(a) == val);
    ASN1_INTEGER_free(a);
#endif
    return EXPECT_RESULT();
}

#if defined(OPENSSL_EXTRA)
typedef struct ASN1IntTestVector {
    const byte* der;
    const size_t derSz;
    const long value;
} ASN1IntTestVector;
#endif
static int test_wolfSSL_d2i_ASN1_INTEGER(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA)
    size_t i;
    WOLFSSL_ASN1_INTEGER* a = NULL;
    WOLFSSL_ASN1_INTEGER* b = NULL;
    WOLFSSL_ASN1_INTEGER* c = NULL;
    const byte* p = NULL;
    byte* p2 = NULL;
    byte* reEncoded = NULL;
    int reEncodedSz = 0;

    static const byte zeroDer[] = {
        0x02, 0x01, 0x00
    };
    static const byte oneDer[] = {
        0x02, 0x01, 0x01
    };
    static const byte negativeDer[] = {
        0x02, 0x03, 0xC1, 0x16, 0x0D
    };
    static const byte positiveDer[] = {
        0x02, 0x03, 0x01, 0x00, 0x01
    };
    static const byte primeDer[] = {
        0x02, 0x82, 0x01, 0x01, 0x00, 0xc0, 0x95, 0x08, 0xe1, 0x57, 0x41,
        0xf2, 0x71, 0x6d, 0xb7, 0xd2, 0x45, 0x41, 0x27, 0x01, 0x65, 0xc6,
        0x45, 0xae, 0xf2, 0xbc, 0x24, 0x30, 0xb8, 0x95, 0xce, 0x2f, 0x4e,
        0xd6, 0xf6, 0x1c, 0x88, 0xbc, 0x7c, 0x9f, 0xfb, 0xa8, 0x67, 0x7f,
        0xfe, 0x5c, 0x9c, 0x51, 0x75, 0xf7, 0x8a, 0xca, 0x07, 0xe7, 0x35,
        0x2f, 0x8f, 0xe1, 0xbd, 0x7b, 0xc0, 0x2f, 0x7c, 0xab, 0x64, 0xa8,
        0x17, 0xfc, 0xca, 0x5d, 0x7b, 0xba, 0xe0, 0x21, 0xe5, 0x72, 0x2e,
        0x6f, 0x2e, 0x86, 0xd8, 0x95, 0x73, 0xda, 0xac, 0x1b, 0x53, 0xb9,
        0x5f, 0x3f, 0xd7, 0x19, 0x0d, 0x25, 0x4f, 0xe1, 0x63, 0x63, 0x51,
        0x8b, 0x0b, 0x64, 0x3f, 0xad, 0x43, 0xb8, 0xa5, 0x1c, 0x5c, 0x34,
        0xb3, 0xae, 0x00, 0xa0, 0x63, 0xc5, 0xf6, 0x7f, 0x0b, 0x59, 0x68,
        0x78, 0x73, 0xa6, 0x8c, 0x18, 0xa9, 0x02, 0x6d, 0xaf, 0xc3, 0x19,
        0x01, 0x2e, 0xb8, 0x10, 0xe3, 0xc6, 0xcc, 0x40, 0xb4, 0x69, 0xa3,
        0x46, 0x33, 0x69, 0x87, 0x6e, 0xc4, 0xbb, 0x17, 0xa6, 0xf3, 0xe8,
        0xdd, 0xad, 0x73, 0xbc, 0x7b, 0x2f, 0x21, 0xb5, 0xfd, 0x66, 0x51,
        0x0c, 0xbd, 0x54, 0xb3, 0xe1, 0x6d, 0x5f, 0x1c, 0xbc, 0x23, 0x73,
        0xd1, 0x09, 0x03, 0x89, 0x14, 0xd2, 0x10, 0xb9, 0x64, 0xc3, 0x2a,
        0xd0, 0xa1, 0x96, 0x4a, 0xbc, 0xe1, 0xd4, 0x1a, 0x5b, 0xc7, 0xa0,
        0xc0, 0xc1, 0x63, 0x78, 0x0f, 0x44, 0x37, 0x30, 0x32, 0x96, 0x80,
        0x32, 0x23, 0x95, 0xa1, 0x77, 0xba, 0x13, 0xd2, 0x97, 0x73, 0xe2,
        0x5d, 0x25, 0xc9, 0x6a, 0x0d, 0xc3, 0x39, 0x60, 0xa4, 0xb4, 0xb0,
        0x69, 0x42, 0x42, 0x09, 0xe9, 0xd8, 0x08, 0xbc, 0x33, 0x20, 0xb3,
        0x58, 0x22, 0xa7, 0xaa, 0xeb, 0xc4, 0xe1, 0xe6, 0x61, 0x83, 0xc5,
        0xd2, 0x96, 0xdf, 0xd9, 0xd0, 0x4f, 0xad, 0xd7
    };
    static const byte garbageDer[] = {0xDE, 0xAD, 0xBE, 0xEF};

    static const ASN1IntTestVector testVectors[] = {
        {zeroDer, sizeof(zeroDer), 0},
        {oneDer, sizeof(oneDer), 1},
        {negativeDer, sizeof(negativeDer), -4123123},
        {positiveDer, sizeof(positiveDer), 65537},
        {primeDer, sizeof(primeDer), 0}
    };
    static const size_t NUM_TEST_VECTORS =
        sizeof(testVectors)/sizeof(testVectors[0]);

    /* Check d2i error conditions */
    /* NULL pointer to input. */
    ExpectNull((a = wolfSSL_d2i_ASN1_INTEGER(&b, NULL, 1)));
    ExpectNull(b);
    /* NULL input. */
    ExpectNull((a = wolfSSL_d2i_ASN1_INTEGER(&b, &p, 1)));
    ExpectNull(b);
    /* 0 length. */
    p = testVectors[0].der;
    ExpectNull((a = wolfSSL_d2i_ASN1_INTEGER(&b, &p, 0)));
    ExpectNull(b);
    /* Negative length. */
    p = testVectors[0].der;
    ExpectNull((a = wolfSSL_d2i_ASN1_INTEGER(&b, &p, -1)));
    ExpectNull(b);
    /* Garbage DER input. */
    p = garbageDer;
    ExpectNull((a = wolfSSL_d2i_ASN1_INTEGER(&b, &p, sizeof(garbageDer))));
    ExpectNull(b);

    /* Check i2d error conditions */
    /* NULL input. */
    ExpectIntLT(wolfSSL_i2d_ASN1_INTEGER(NULL, &p2), 0);
    /* 0 length input data buffer (a->length == 0). */
    ExpectNotNull((a = wolfSSL_ASN1_INTEGER_new()));
    ExpectIntLT(wolfSSL_i2d_ASN1_INTEGER(a, &p2), 0);
    if (a != NULL)
        a->data = NULL;
    /* NULL input data buffer. */
    ExpectIntLT(wolfSSL_i2d_ASN1_INTEGER(a, &p2), 0);
    if (a != NULL) {
        /* Reset a->data. */
        a->data = a->intData;
    }
    /* Set a to valid value. */
    ExpectIntEQ(wolfSSL_ASN1_INTEGER_set(a, 1), WOLFSSL_SUCCESS);
    /* NULL output buffer. */
    ExpectIntEQ(wolfSSL_i2d_ASN1_INTEGER(a, NULL), 3);
    wolfSSL_ASN1_INTEGER_free(a);
    a = NULL;

    for (i = 0; i < NUM_TEST_VECTORS; ++i) {
        p = testVectors[i].der;
        ExpectNotNull(a = wolfSSL_d2i_ASN1_INTEGER(&b, &p,
            testVectors[i].derSz));
        ExpectIntEQ(wolfSSL_ASN1_INTEGER_cmp(a, b), 0);

        if (testVectors[i].derSz <= sizeof(long)) {
            ExpectNotNull(c = wolfSSL_ASN1_INTEGER_new());
            ExpectIntEQ(wolfSSL_ASN1_INTEGER_set(c, testVectors[i].value), 1);
            ExpectIntEQ(wolfSSL_ASN1_INTEGER_cmp(a, c), 0);
            wolfSSL_ASN1_INTEGER_free(c);
            c = NULL;
        }

        /* Convert to DER without a pre-allocated output buffer. */
        ExpectIntGT((reEncodedSz = wolfSSL_i2d_ASN1_INTEGER(a, &reEncoded)), 0);
        ExpectIntEQ(reEncodedSz, testVectors[i].derSz);
        ExpectIntEQ(XMEMCMP(reEncoded, testVectors[i].der, reEncodedSz), 0);

        /* Convert to DER with a pre-allocated output buffer. In this case, the
         * output buffer pointer should be incremented just past the end of the
         * encoded data. */
        p2 = reEncoded;
        ExpectIntGT((reEncodedSz = wolfSSL_i2d_ASN1_INTEGER(a, &p2)), 0);
        ExpectIntEQ(reEncodedSz, testVectors[i].derSz);
        ExpectPtrEq(reEncoded, p2 - reEncodedSz);
        ExpectIntEQ(XMEMCMP(reEncoded, testVectors[i].der, reEncodedSz), 0);

        XFREE(reEncoded, NULL, DYNAMIC_TYPE_ASN1);
        reEncoded = NULL;
        wolfSSL_ASN1_INTEGER_free(a);
        a = NULL;
    }
#endif /* OPENSSL_EXTRA */
    return EXPECT_RESULT();
}

static int test_wolfSSL_a2i_ASN1_INTEGER(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_BIO)
    BIO* bio = NULL;
    BIO* out = NULL;
    BIO* fixed = NULL;
    ASN1_INTEGER* ai = NULL;
    char buf[] = "123456\n12345\n1123456789123456\\\n78901234567890 \r\n\n";
    char tmp[1024];
    int  tmpSz;

    const char expected1[] = "123456";
    const char expected2[] = "112345678912345678901234567890";
    char longStr[] = "123456781234567812345678123456781234567812345678\n"
        "123456781234567812345678123456781234567812345678\\\n12345678\n";

    ExpectNotNull(out = BIO_new(BIO_s_mem()));
    ExpectNotNull(ai = ASN1_INTEGER_new());

    ExpectNotNull(bio = BIO_new_mem_buf(buf, -1));

    /* Invalid parameter testing. */
    ExpectIntEQ(a2i_ASN1_INTEGER(NULL, NULL, NULL, -1), 0);
    ExpectIntEQ(a2i_ASN1_INTEGER(bio, NULL, NULL, -1), 0);
    ExpectIntEQ(a2i_ASN1_INTEGER(NULL, ai, NULL, -1), 0);
    ExpectIntEQ(a2i_ASN1_INTEGER(NULL, NULL, tmp, -1), 0);
    ExpectIntEQ(a2i_ASN1_INTEGER(NULL, NULL, NULL, sizeof(tmp)), 0);
    ExpectIntEQ(a2i_ASN1_INTEGER(NULL, ai, tmp, sizeof(tmp)), 0);
    ExpectIntEQ(a2i_ASN1_INTEGER(bio, NULL, tmp, sizeof(tmp)), 0);
    ExpectIntEQ(a2i_ASN1_INTEGER(bio, ai, NULL, sizeof(tmp)), 0);
    ExpectIntEQ(a2i_ASN1_INTEGER(bio, ai, tmp, -1), 0);
    ExpectIntEQ(i2a_ASN1_INTEGER(NULL, NULL), 0);
    ExpectIntEQ(i2a_ASN1_INTEGER(bio, NULL), 0);
    ExpectIntEQ(i2a_ASN1_INTEGER(NULL, ai), 0);

    /* No data to read from BIO. */
    ExpectIntEQ(a2i_ASN1_INTEGER(out, ai, tmp, sizeof(tmp)), 0);

    /* read first line */
    ExpectIntEQ(a2i_ASN1_INTEGER(bio, ai, tmp, sizeof(tmp)), 1);
    ExpectIntEQ(i2a_ASN1_INTEGER(out, ai), 6);
    XMEMSET(tmp, 0, sizeof(tmp));
    tmpSz = BIO_read(out, tmp, sizeof(tmp));
    ExpectIntEQ(tmpSz, 6);
    ExpectIntEQ(XMEMCMP(tmp, expected1, tmpSz), 0);

    /* fail on second line (not % 2) */
    ExpectIntEQ(a2i_ASN1_INTEGER(bio, ai, tmp, sizeof(tmp)), 0);

    /* read 3rd long line */
    ExpectIntEQ(a2i_ASN1_INTEGER(bio, ai, tmp, sizeof(tmp)), 1);
    ExpectIntEQ(i2a_ASN1_INTEGER(out, ai), 30);
    XMEMSET(tmp, 0, sizeof(tmp));
    tmpSz = BIO_read(out, tmp, sizeof(tmp));
    ExpectIntEQ(tmpSz, 30);
    ExpectIntEQ(XMEMCMP(tmp, expected2, tmpSz), 0);

    /* fail on empty line */
    ExpectIntEQ(a2i_ASN1_INTEGER(bio, ai, tmp, sizeof(tmp)), 0);

    BIO_free(bio);
    bio = NULL;

    /* Make long integer, requiring dynamic memory, even longer. */
    ExpectNotNull(bio = BIO_new_mem_buf(longStr, -1));
    ExpectIntEQ(a2i_ASN1_INTEGER(bio, ai, tmp, sizeof(tmp)), 1);
    ExpectIntEQ(i2a_ASN1_INTEGER(out, ai), 48);
    XMEMSET(tmp, 0, sizeof(tmp));
    tmpSz = BIO_read(out, tmp, sizeof(tmp));
    ExpectIntEQ(tmpSz, 48);
    ExpectIntEQ(a2i_ASN1_INTEGER(bio, ai, tmp, sizeof(tmp)), 1);
    ExpectIntEQ(i2a_ASN1_INTEGER(out, ai), 56);
    XMEMSET(tmp, 0, sizeof(tmp));
    tmpSz = BIO_read(out, tmp, sizeof(tmp));
    ExpectIntEQ(tmpSz, 56);
    ExpectIntEQ(wolfSSL_ASN1_INTEGER_set(ai, 1), 1);
    BIO_free(bio);
    BIO_free(out);

    ExpectNotNull(fixed = BIO_new(wolfSSL_BIO_s_fixed_mem()));
    ExpectIntEQ(BIO_set_write_buf_size(fixed, 1), 1);
    /* Ensure there is 0 bytes available to write into. */
    ExpectIntEQ(BIO_write(fixed, tmp, 1), 1);
    ExpectIntEQ(i2a_ASN1_INTEGER(fixed, ai), 0);
    ExpectIntEQ(BIO_set_write_buf_size(fixed, 1), 1);
    ExpectIntEQ(i2a_ASN1_INTEGER(fixed, ai), 0);
    BIO_free(fixed);

    ASN1_INTEGER_free(ai);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_i2c_ASN1_INTEGER(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_ASN)
    ASN1_INTEGER *a = NULL;
    unsigned char *pp = NULL,*tpp = NULL;
    int ret = 0;

    ExpectNotNull(a = wolfSSL_ASN1_INTEGER_new());

    /* Invalid parameter testing. */
    /* Set pp to an invalid value. */
    pp = NULL;
    ExpectIntEQ(i2c_ASN1_INTEGER(NULL, &pp), 0);
    ExpectIntEQ(i2c_ASN1_INTEGER(a, &pp), 0);
    ExpectIntEQ(i2c_ASN1_INTEGER(NULL, NULL), 0);

    /* 40 */
    if (a != NULL) {
        a->intData[0] = ASN_INTEGER;
        a->intData[1] = 1;
        a->intData[2] = 40;
    }
    ExpectIntEQ(ret = i2c_ASN1_INTEGER(a, NULL), 1);
    ExpectNotNull(pp = (unsigned char*)XMALLOC(ret + 1, NULL,
                DYNAMIC_TYPE_TMP_BUFFER));
    tpp = pp;
    ExpectNotNull(XMEMSET(tpp, 0, ret + 1));
    ExpectIntEQ(i2c_ASN1_INTEGER(a, &tpp), 1);
    tpp--;
    ExpectIntEQ(*tpp, 40);
    XFREE(pp, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    pp = NULL;

    /* 128 */
    if (a != NULL) {
        a->intData[0] = ASN_INTEGER;
        a->intData[1] = 1;
        a->intData[2] = 128;
    }
    ExpectIntEQ(ret = i2c_ASN1_INTEGER(a, NULL), 2);
    ExpectNotNull(pp = (unsigned char*)XMALLOC(ret + 1, NULL,
                DYNAMIC_TYPE_TMP_BUFFER));
    tpp = pp;
    if (tpp != NULL) {
        ExpectNotNull(XMEMSET(tpp, 0, ret + 1));
        ExpectIntEQ(i2c_ASN1_INTEGER(a, &tpp), 2);
        tpp--;
        ExpectIntEQ(*(tpp--), 128);
        ExpectIntEQ(*tpp, 0);
    }
    XFREE(pp, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    pp = NULL;

    /* -40 */
    if (a != NULL) {
        a->intData[0] = ASN_INTEGER;
        a->intData[1] = 1;
        a->intData[2] = 40;
        a->negative = 1;
    }
    ExpectIntEQ(ret = i2c_ASN1_INTEGER(a, NULL), 1);
    ExpectNotNull(pp = (unsigned char*)XMALLOC(ret + 1, NULL,
                DYNAMIC_TYPE_TMP_BUFFER));
    tpp = pp;
    if (tpp != NULL) {
        ExpectNotNull(XMEMSET(tpp, 0, ret + 1));
        ExpectIntEQ(i2c_ASN1_INTEGER(a, &tpp), 1);
        tpp--;
        ExpectIntEQ(*tpp, 216);
    }
    XFREE(pp, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    pp = NULL;

    /* -128 */
    if (a != NULL) {
        a->intData[0] = ASN_INTEGER;
        a->intData[1] = 1;
        a->intData[2] = 128;
        a->negative = 1;
    }
    ExpectIntEQ(ret = i2c_ASN1_INTEGER(a, NULL), 1);
    ExpectNotNull(pp = (unsigned char*)XMALLOC(ret + 1, NULL,
                DYNAMIC_TYPE_TMP_BUFFER));
    tpp = pp;
    if (tpp != NULL) {
        ExpectNotNull(XMEMSET(tpp, 0, ret + 1));
        ExpectIntEQ(i2c_ASN1_INTEGER(a, &tpp), 1);
        tpp--;
        ExpectIntEQ(*tpp, 128);
    }
    XFREE(pp, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    pp = NULL;

    /* -200 */
    if (a != NULL) {
        a->intData[0] = ASN_INTEGER;
        a->intData[1] = 1;
        a->intData[2] = 200;
        a->negative = 1;
    }
    ExpectIntEQ(ret = i2c_ASN1_INTEGER(a, NULL), 2);
    ExpectNotNull(pp = (unsigned char*)XMALLOC(ret + 1, NULL,
            DYNAMIC_TYPE_TMP_BUFFER));
    tpp = pp;
    if (tpp != NULL) {
        ExpectNotNull(XMEMSET(tpp, 0, ret + 1));
        ExpectIntEQ(i2c_ASN1_INTEGER(a, &tpp), 2);
        tpp--;
        ExpectIntEQ(*(tpp--), 56);
        ExpectIntEQ(*tpp, 255);
    }
    XFREE(pp, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    pp = NULL;

    /* Empty */
    if (a != NULL) {
        a->intData[0] = ASN_INTEGER;
        a->intData[1] = 0;
        a->negative = 0;
    }
    ExpectIntEQ(ret = i2c_ASN1_INTEGER(a, NULL), 1);
    ExpectNotNull(pp = (unsigned char*)XMALLOC(ret + 1, NULL,
                DYNAMIC_TYPE_TMP_BUFFER));
    tpp = pp;
    if (tpp != NULL) {
        ExpectNotNull(XMEMSET(tpp, 0, ret + 1));
        ExpectIntEQ(i2c_ASN1_INTEGER(a, &tpp), 1);
        tpp--;
        ExpectIntEQ(*tpp, 0);
    }
    XFREE(pp, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    pp = NULL;

    /* 0 */
    if (a != NULL) {
        a->intData[0] = ASN_INTEGER;
        a->intData[1] = 1;
        a->intData[2] = 0;
        a->negative = 1;
    }
    ExpectIntEQ(ret = i2c_ASN1_INTEGER(a, NULL), 1);
    ExpectNotNull(pp = (unsigned char*)XMALLOC(ret + 1, NULL,
                DYNAMIC_TYPE_TMP_BUFFER));
    if (tpp != NULL) {
        tpp = pp;
        ExpectNotNull(XMEMSET(tpp, 0, ret + 1));
        ExpectIntEQ(i2c_ASN1_INTEGER(a, &tpp), 1);
        tpp--;
        ExpectIntEQ(*tpp, 0);
    }
    XFREE(pp, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    pp = NULL;

    /* 0x100 */
    if (a != NULL) {
        a->intData[0] = ASN_INTEGER;
        a->intData[1] = 2;
        a->intData[2] = 0x01;
        a->intData[3] = 0x00;
        a->negative = 0;
    }
    ExpectIntEQ(ret = i2c_ASN1_INTEGER(a, NULL), 2);
    ExpectNotNull(pp = (unsigned char*)XMALLOC(ret + 1, NULL,
                DYNAMIC_TYPE_TMP_BUFFER));
    if (tpp != NULL) {
        tpp = pp;
        ExpectNotNull(XMEMSET(tpp, 0, ret + 1));
        ExpectIntEQ(i2c_ASN1_INTEGER(a, &tpp), 2);
        tpp -= 2;
        ExpectIntEQ(tpp[0], 0x01);
        ExpectIntEQ(tpp[1], 0x00);
    }
    XFREE(pp, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    pp = NULL;

    /* -0x8000 => 0x8000 */
    if (a != NULL) {
        a->intData[0] = ASN_INTEGER;
        a->intData[1] = 2;
        a->intData[2] = 0x80;
        a->intData[3] = 0x00;
        a->negative = 1;
    }
    ExpectIntEQ(ret = i2c_ASN1_INTEGER(a, NULL), 2);
    ExpectNotNull(pp = (unsigned char*)XMALLOC(ret + 1, NULL,
                DYNAMIC_TYPE_TMP_BUFFER));
    tpp = pp;
    if (tpp != NULL) {
        ExpectNotNull(XMEMSET(tpp, 0, ret + 1));
        ExpectIntEQ(i2c_ASN1_INTEGER(a, &tpp), 2);
        tpp -= 2;
        ExpectIntEQ(tpp[0], 0x80);
        ExpectIntEQ(tpp[1], 0x00);
    }
    XFREE(pp, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    pp = NULL;

    /* -0x8001 => 0xFF7FFF */
    if (a != NULL) {
        a->intData[0] = ASN_INTEGER;
        a->intData[1] = 2;
        a->intData[2] = 0x80;
        a->intData[3] = 0x01;
        a->negative = 1;
    }
    ExpectIntEQ(ret = i2c_ASN1_INTEGER(a, NULL), 3);
    ExpectNotNull(pp = (unsigned char*)XMALLOC(ret + 1, NULL,
                DYNAMIC_TYPE_TMP_BUFFER));
    tpp = pp;
    if (tpp != NULL) {
        ExpectNotNull(XMEMSET(tpp, 0, ret + 1));
        ExpectIntEQ(i2c_ASN1_INTEGER(a, &tpp), 3);
        tpp -= 3;
        ExpectIntEQ(tpp[0], 0xFF);
        ExpectIntEQ(tpp[1], 0x7F);
        ExpectIntEQ(tpp[2], 0xFF);
    }
    XFREE(pp, NULL, DYNAMIC_TYPE_TMP_BUFFER);

    wolfSSL_ASN1_INTEGER_free(a);
#endif /* OPENSSL_EXTRA && !NO_ASN */
    return EXPECT_RESULT();
}

static int test_wolfSSL_ASN1_OBJECT(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA)
    ASN1_OBJECT* a = NULL;
    ASN1_OBJECT s;
    const unsigned char der[] = { 0x06, 0x01, 0x00 };

    /* Invalid parameter testing. */
    ASN1_OBJECT_free(NULL);
    ExpectNull(wolfSSL_ASN1_OBJECT_dup(NULL));

    /* Test that a static ASN1_OBJECT can be freed. */
    XMEMSET(&s, 0, sizeof(ASN1_OBJECT));
    ASN1_OBJECT_free(&s);
    ExpectNotNull(a = wolfSSL_ASN1_OBJECT_dup(&s));
    ASN1_OBJECT_free(a);
    a = NULL;
    s.obj = der;
    s.objSz = sizeof(der);
    ExpectNotNull(a = wolfSSL_ASN1_OBJECT_dup(&s));
    ASN1_OBJECT_free(a);
    ASN1_OBJECT_free(&s);
#endif /* OPENSSL_EXTRA */
    return EXPECT_RESULT();
}

static int test_wolfSSL_ASN1_get_object(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(HAVE_ECC) && defined(USE_CERT_BUFFERS_256)
    const unsigned char* derBuf = cliecc_cert_der_256;
    const unsigned char* nullPtr = NULL;
    const unsigned char objDerInvalidLen[] = { 0x30, 0x81 };
    const unsigned char objDerBadLen[] = { 0x30, 0x04 };
    const unsigned char objDerNotObj[] = { 0x02, 0x01, 0x00 };
    const unsigned char objDerNoData[] = { 0x06, 0x00 };
    const unsigned char* p;
    unsigned char objDer[10];
    unsigned char* der;
    unsigned char* derPtr;
    int len = sizeof_cliecc_cert_der_256;
    long asnLen = 0;
    int tag = 0;
    int cls = 0;
    ASN1_OBJECT* a = NULL;
    ASN1_OBJECT s;

    XMEMSET(&s, 0, sizeof(ASN1_OBJECT));

    /* Invalid encoding at length. */
    p = objDerInvalidLen;
    ExpectIntEQ(ASN1_get_object(&p, &asnLen, &tag, &cls, sizeof(objDerBadLen)),
        0x80);
    p = objDerBadLen;
    /* Error = 0x80, Constructed = 0x20 */
    ExpectIntEQ(ASN1_get_object(&p, &asnLen, &tag, &cls, sizeof(objDerBadLen)),
        0x80 | 0x20);

    /* Read a couple TLV triplets and make sure they match the expected values
     */

    /* SEQUENCE */
    ExpectIntEQ(ASN1_get_object(&derBuf, &asnLen, &tag, &cls, len) & 0x80, 0);
    ExpectIntEQ(asnLen, 862);
    ExpectIntEQ(tag, 0x10);
    ExpectIntEQ(cls, 0);

    /* SEQUENCE */
    ExpectIntEQ(ASN1_get_object(&derBuf, &asnLen, &tag, &cls,
            len - (derBuf - cliecc_cert_der_256)) & 0x80, 0);
    ExpectIntEQ(asnLen, 772);
    ExpectIntEQ(tag, 0x10);
    ExpectIntEQ(cls, 0);

    /* [0] */
    ExpectIntEQ(ASN1_get_object(&derBuf, &asnLen, &tag, &cls,
            len - (derBuf - cliecc_cert_der_256)) & 0x80, 0);
    ExpectIntEQ(asnLen, 3);
    ExpectIntEQ(tag, 0);
    ExpectIntEQ(cls, 0x80);

    /* INTEGER */
    ExpectIntEQ(ASN1_get_object(&derBuf, &asnLen, &tag, &cls,
            len - (derBuf - cliecc_cert_der_256)) & 0x80, 0);
    ExpectIntEQ(asnLen, 1);
    ExpectIntEQ(tag, 0x2);
    ExpectIntEQ(cls, 0);
    derBuf += asnLen;

    /* INTEGER */
    ExpectIntEQ(ASN1_get_object(&derBuf, &asnLen, &tag, &cls,
            len - (derBuf - cliecc_cert_der_256)) & 0x80, 0);
    ExpectIntEQ(asnLen, 20);
    ExpectIntEQ(tag, 0x2);
    ExpectIntEQ(cls, 0);
    derBuf += asnLen;

    /* SEQUENCE */
    ExpectIntEQ(ASN1_get_object(&derBuf, &asnLen, &tag, &cls,
            len - (derBuf - cliecc_cert_der_256)) & 0x80, 0);
    ExpectIntEQ(asnLen, 10);
    ExpectIntEQ(tag, 0x10);
    ExpectIntEQ(cls, 0);

    /* Found OBJECT_ID. */

    /* Invalid parameter testing. */
    ExpectIntEQ(ASN1_get_object(NULL, NULL, NULL, NULL, 0), 0x80);
    ExpectIntEQ(ASN1_get_object(&nullPtr, NULL, NULL, NULL, 0), 0x80);
    ExpectIntEQ(ASN1_get_object(NULL, &asnLen, &tag, &cls, len), 0x80);
    ExpectIntEQ(ASN1_get_object(&nullPtr, &asnLen, &tag, &cls, len), 0x80);
    ExpectIntEQ(ASN1_get_object(&derBuf, NULL, &tag, &cls, len), 0x80);
    ExpectIntEQ(ASN1_get_object(&derBuf, &asnLen, NULL, &cls, len), 0x80);
    ExpectIntEQ(ASN1_get_object(&derBuf, &asnLen, &tag, NULL, len), 0x80);
    ExpectIntEQ(ASN1_get_object(&derBuf, &asnLen, &tag, &cls, 0), 0x80);
    ExpectIntEQ(ASN1_get_object(&derBuf, &asnLen, &tag, &cls, -1), 0x80);
    ExpectNull(d2i_ASN1_OBJECT(NULL, NULL, -1));
    ExpectNull(d2i_ASN1_OBJECT(NULL, &nullPtr, -1));
    ExpectNull(d2i_ASN1_OBJECT(NULL, &derBuf, -1));
    ExpectNull(d2i_ASN1_OBJECT(NULL, NULL, 0));
    ExpectNull(d2i_ASN1_OBJECT(&a, NULL, len));
    ExpectNull(d2i_ASN1_OBJECT(&a, &nullPtr, len));
    ExpectNull(d2i_ASN1_OBJECT(&a, &derBuf, -1));
    ExpectNull(c2i_ASN1_OBJECT(NULL, NULL, -1));
    ExpectNull(c2i_ASN1_OBJECT(NULL, &nullPtr, -1));
    ExpectNull(c2i_ASN1_OBJECT(NULL, &derBuf, -1));
    ExpectNull(c2i_ASN1_OBJECT(NULL, NULL, 1));
    ExpectNull(c2i_ASN1_OBJECT(NULL, &nullPtr, 1));

    /* Invalid encoding at length. */
    p = objDerInvalidLen;
    ExpectNull(d2i_ASN1_OBJECT(&a, &p, sizeof(objDerInvalidLen)));
    p = objDerBadLen;
    ExpectNull(d2i_ASN1_OBJECT(&a, &p, sizeof(objDerBadLen)));
    p = objDerNotObj;
    ExpectNull(d2i_ASN1_OBJECT(&a, &p, sizeof(objDerNotObj)));
    p = objDerNoData;
    ExpectNull(d2i_ASN1_OBJECT(&a, &p, sizeof(objDerNoData)));

    /* Create an ASN OBJECT from content */
    p = derBuf + 2;
    ExpectNotNull(a = c2i_ASN1_OBJECT(NULL, &p, 8));
    ASN1_OBJECT_free(a);
    a = NULL;
    /* Create an ASN OBJECT from DER */
    ExpectNotNull(d2i_ASN1_OBJECT(&a, &derBuf, len));

    /* Invalid parameter testing. */
    ExpectIntEQ(i2d_ASN1_OBJECT(NULL, NULL), 0);
    ExpectIntEQ(i2d_ASN1_OBJECT(&s, NULL), 0);

    ExpectIntEQ(i2d_ASN1_OBJECT(a, NULL), 10);
    der = NULL;
    ExpectIntEQ(i2d_ASN1_OBJECT(a, &der), 10);
    derPtr = objDer;
    ExpectIntEQ(i2d_ASN1_OBJECT(a, &derPtr), 10);
    ExpectPtrNE(derPtr, objDer);
    ExpectIntEQ(XMEMCMP(der, objDer, 10), 0);
    XFREE(der, NULL, DYNAMIC_TYPE_OPENSSL);

    ASN1_OBJECT_free(a);
#endif /* OPENSSL_EXTRA && HAVE_ECC && USE_CERT_BUFFERS_256 */
    return EXPECT_RESULT();
}

static int test_wolfSSL_i2a_ASN1_OBJECT(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_ASN) && !defined(NO_BIO)
    ASN1_OBJECT* obj = NULL;
    ASN1_OBJECT* a = NULL;
    BIO *bio = NULL;
    const unsigned char notObjDer[] = { 0x04, 0x01, 0xff };
    const unsigned char* p;

    ExpectNotNull(obj = OBJ_nid2obj(NID_sha256));
    ExpectTrue((bio = BIO_new(BIO_s_mem())) != NULL);

    ExpectIntGT(wolfSSL_i2a_ASN1_OBJECT(bio, obj), 0);
    ExpectIntGT(wolfSSL_i2a_ASN1_OBJECT(bio, NULL), 0);

    ExpectIntEQ(wolfSSL_i2a_ASN1_OBJECT(NULL, obj), 0);

    /* No DER encoding in ASN1_OBJECT. */
    ExpectNotNull(a = wolfSSL_ASN1_OBJECT_new());
    ExpectIntEQ(wolfSSL_i2a_ASN1_OBJECT(bio, a), 0);
    ASN1_OBJECT_free(a);
    a = NULL;
    /* DER encoding */
    p = notObjDer;
    ExpectNotNull(a = c2i_ASN1_OBJECT(NULL, &p, 3));
    ExpectIntEQ(wolfSSL_i2a_ASN1_OBJECT(bio, a), 5);
    ASN1_OBJECT_free(a);

    BIO_free(bio);
    ASN1_OBJECT_free(obj);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_i2t_ASN1_OBJECT(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN)
    char buf[50] = {0};
    ASN1_OBJECT* obj;
    const char* oid = "2.5.29.19";
    const char* ln  = "X509v3 Basic Constraints";

    obj = NULL;
    ExpectIntEQ(i2t_ASN1_OBJECT(NULL, sizeof(buf), obj), 0);
    ExpectIntEQ(i2t_ASN1_OBJECT(buf, sizeof(buf), NULL), 0);
    ExpectIntEQ(i2t_ASN1_OBJECT(buf, 0, NULL), 0);

    ExpectNotNull(obj = OBJ_txt2obj(oid, 0));
    XMEMSET(buf, 0, sizeof(buf));
    ExpectIntEQ(i2t_ASN1_OBJECT(buf, sizeof(buf), obj), XSTRLEN(ln));
    ExpectIntEQ(XSTRNCMP(buf, ln, XSTRLEN(ln)), 0);
    ASN1_OBJECT_free(obj);
#endif /* OPENSSL_EXTRA && WOLFSSL_CERT_EXT && WOLFSSL_CERT_GEN */
    return EXPECT_RESULT();
}

static int test_wolfSSL_sk_ASN1_OBJECT(void)
{
    EXPECT_DECLS;
#if !defined(NO_ASN) && (defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL))
    WOLFSSL_STACK* sk = NULL;
    WOLFSSL_ASN1_OBJECT* obj;

    ExpectNotNull(obj = wolfSSL_ASN1_OBJECT_new());

    ExpectNotNull(sk = wolfSSL_sk_new_asn1_obj());
    wolfSSL_sk_ASN1_OBJECT_free(sk);
    sk = NULL;

    ExpectNotNull(sk = wolfSSL_sk_new_asn1_obj());
    ExpectIntEQ(wolfSSL_sk_ASN1_OBJECT_push(NULL, NULL), -1);
    ExpectIntEQ(wolfSSL_sk_ASN1_OBJECT_push(sk, NULL), 0);
    ExpectIntEQ(wolfSSL_sk_ASN1_OBJECT_push(NULL, obj), -1);
    ExpectIntEQ(wolfSSL_sk_ASN1_OBJECT_push(sk, obj), 1);
    wolfSSL_sk_ASN1_OBJECT_pop_free(sk, NULL);
    sk = NULL;
    /* obj freed in pop_free call. */

    ExpectNotNull(obj = wolfSSL_ASN1_OBJECT_new());
    ExpectNotNull(sk = wolfSSL_sk_new_asn1_obj());
    ExpectIntEQ(wolfSSL_sk_ASN1_OBJECT_push(sk, obj), 1);
    ExpectPtrEq(obj, wolfSSL_sk_ASN1_OBJECT_pop(sk));
    wolfSSL_sk_ASN1_OBJECT_free(sk);
    wolfSSL_ASN1_OBJECT_free(obj);
#endif /* !NO_ASN && (OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL) */
    return EXPECT_RESULT();
}

static int test_wolfSSL_ASN1_STRING(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA)
    ASN1_STRING* str = NULL;
    ASN1_STRING* c = NULL;
    const char data[]  = "hello wolfSSL";
    const char data2[] = "Same len data";
    const char longData[] =
        "This string must be longer than CTC_NAME_SIZE that is defined as 64.";

    ExpectNotNull(str = ASN1_STRING_type_new(V_ASN1_OCTET_STRING));
    ASN1_STRING_free(str);
    str = NULL;

    ExpectNotNull(str = ASN1_STRING_type_new(V_ASN1_OCTET_STRING));
    ExpectIntEQ(ASN1_STRING_type(str), V_ASN1_OCTET_STRING);
    ExpectIntEQ(ASN1_STRING_type(NULL), 0);
    /* Check setting to NULL works. */
    ExpectIntEQ(ASN1_STRING_set(str, NULL, 0), 1);
    ExpectIntEQ(ASN1_STRING_set(str, (const void*)data, sizeof(data)), 1);
    ExpectIntEQ(ASN1_STRING_set(str, (const void*)data, -1), 1);
    ExpectIntEQ(ASN1_STRING_set(str, NULL, -1), 0);
    ExpectIntEQ(ASN1_STRING_set(NULL, NULL, 0), 0);

    ExpectIntEQ(wolfSSL_ASN1_STRING_copy(NULL, NULL), 0);
    ExpectIntEQ(wolfSSL_ASN1_STRING_copy(str, NULL), 0);
    ExpectIntEQ(wolfSSL_ASN1_STRING_copy(NULL, str), 0);
    ExpectNull(wolfSSL_ASN1_STRING_dup(NULL));

    ExpectNotNull(c = wolfSSL_ASN1_STRING_dup(str));
    ExpectIntEQ(ASN1_STRING_cmp(NULL, NULL), -1);
    ExpectIntEQ(ASN1_STRING_cmp(str, NULL), -1);
    ExpectIntEQ(ASN1_STRING_cmp(NULL, c), -1);
    ExpectIntEQ(ASN1_STRING_cmp(str, c), 0);
    ExpectIntEQ(ASN1_STRING_set(c, (const void*)data2, -1), 1);
    ExpectIntGT(ASN1_STRING_cmp(str, c), 0);
    ExpectIntEQ(ASN1_STRING_set(str, (const void*)longData, -1), 1);
    ExpectIntEQ(wolfSSL_ASN1_STRING_copy(c, str), 1);
    ExpectIntEQ(ASN1_STRING_cmp(str, c), 0);
    /* Check setting back to smaller size frees dynamic data. */
    ExpectIntEQ(ASN1_STRING_set(str, (const void*)data, -1), 1);
    ExpectIntLT(ASN1_STRING_cmp(str, c), 0);
    ExpectIntGT(ASN1_STRING_cmp(c, str), 0);

    ExpectNull(ASN1_STRING_get0_data(NULL));
    ExpectNotNull(ASN1_STRING_get0_data(str));
    ExpectNull(ASN1_STRING_data(NULL));
    ExpectNotNull(ASN1_STRING_data(str));
    ExpectIntEQ(ASN1_STRING_length(NULL), 0);
    ExpectIntGT(ASN1_STRING_length(str), 0);

    ASN1_STRING_free(c);
    ASN1_STRING_free(str);
    ASN1_STRING_free(NULL);

#ifndef NO_WOLFSSL_STUB
    ExpectNull(d2i_DISPLAYTEXT(NULL, NULL, 0));
#endif
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_ASN1_STRING_to_UTF8(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(NO_ASN) && !defined(NO_RSA) && \
    !defined(NO_FILESYSTEM)
    WOLFSSL_X509* x509 = NULL;
    WOLFSSL_X509_NAME* subject = NULL;
    WOLFSSL_X509_NAME_ENTRY* e = NULL;
    WOLFSSL_ASN1_STRING* a = NULL;
    FILE* file = XBADFILE;
    int idx = 0;
    char targetOutput[16] = "www.wolfssl.com";
    unsigned char* actual_output = NULL;
    int len = 0;

    ExpectNotNull(file = fopen("./certs/server-cert.pem", "rb"));
    ExpectNotNull(x509 = wolfSSL_PEM_read_X509(file, NULL, NULL, NULL));
    if (file != XBADFILE)
        fclose(file);

    /* wolfSSL_ASN1_STRING_to_UTF8(): NID_commonName */
    ExpectNotNull(subject = wolfSSL_X509_get_subject_name(x509));
    ExpectIntEQ((idx = wolfSSL_X509_NAME_get_index_by_NID(subject,
                    NID_commonName, -1)), 5);
    ExpectNotNull(e = wolfSSL_X509_NAME_get_entry(subject, idx));
    ExpectNotNull(a = wolfSSL_X509_NAME_ENTRY_get_data(e));
    ExpectIntEQ((len = wolfSSL_ASN1_STRING_to_UTF8(&actual_output, a)), 15);
    ExpectIntEQ(strncmp((const char*)actual_output, targetOutput, (size_t)len), 0);
    a = NULL;

    /* wolfSSL_ASN1_STRING_to_UTF8(NULL, valid) */
    ExpectIntEQ((len = wolfSSL_ASN1_STRING_to_UTF8(NULL, a)), -1);

    /* wolfSSL_ASN1_STRING_to_UTF8(valid, NULL) */
    ExpectIntEQ((len = wolfSSL_ASN1_STRING_to_UTF8(&actual_output, NULL)), -1);

    /* wolfSSL_ASN1_STRING_to_UTF8(NULL, NULL) */
    ExpectIntEQ((len = wolfSSL_ASN1_STRING_to_UTF8(NULL, NULL)), -1);

    wolfSSL_X509_free(x509);
    XFREE(actual_output, NULL, DYNAMIC_TYPE_TMP_BUFFER);

    ExpectNotNull(a = ASN1_STRING_new());
    ExpectIntEQ(wolfSSL_ASN1_STRING_to_UTF8(&actual_output, a), -1);
    ASN1_STRING_free(a);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_i2s_ASN1_STRING(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_ASN)
    WOLFSSL_ASN1_STRING* str = NULL;
    const char* data = "test_wolfSSL_i2s_ASN1_STRING";
    char* ret = NULL;

    ExpectNotNull(str = ASN1_STRING_new());

    ExpectNull(ret = wolfSSL_i2s_ASN1_STRING(NULL, NULL));
    XFREE(ret, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    ret = NULL;
    /* No data. */
    ExpectNull(ret = wolfSSL_i2s_ASN1_STRING(NULL, str));
    XFREE(ret, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    ret = NULL;

    ExpectIntEQ(ASN1_STRING_set(str, data, 0), 1);
    ExpectNotNull(ret = wolfSSL_i2s_ASN1_STRING(NULL, str));
    XFREE(ret, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    ret = NULL;

    ExpectIntEQ(ASN1_STRING_set(str, data, -1), 1);
    /* No type. */
    ExpectNotNull(ret = wolfSSL_i2s_ASN1_STRING(NULL, str));
    XFREE(ret, NULL, DYNAMIC_TYPE_TMP_BUFFER);

    ASN1_STRING_free(str);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_ASN1_STRING_canon(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_TEST_STATIC_BUILD)
#if !defined(NO_CERTS) && (defined(OPENSSL_ALL) || defined(OPENSSL_EXTRA) || \
    defined(OPENSSL_EXTRA_X509_SMALL))
    WOLFSSL_ASN1_STRING* orig = NULL;
    WOLFSSL_ASN1_STRING* canon = NULL;
    const char* data = "test_wolfSSL_ASN1_STRING_canon";
    const char* whitespaceOnly = "\t\r\n";
    const char* modData = "  \x01\f\t\x02\r\n\v\xff\nTt \n";
    const char* canonData = "\x01 \x02 \xff tt";
    const char longData[] =
        "This string must be longer than CTC_NAME_SIZE that is defined as 64.";

    ExpectNotNull(orig = ASN1_STRING_new());
    ExpectNotNull(canon = ASN1_STRING_new());

    /* Invalid parameter testing. */
    ExpectIntEQ(wolfSSL_ASN1_STRING_canon(NULL, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_ASN1_STRING_canon(canon, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_ASN1_STRING_canon(NULL, orig), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    ExpectIntEQ(wolfSSL_ASN1_STRING_canon(canon, orig), 1);
    ExpectIntEQ(ASN1_STRING_cmp(orig, canon), 0);

    ExpectIntEQ(ASN1_STRING_set(orig, longData, (int)XSTRLEN(data)), 1);
    ExpectIntEQ(wolfSSL_ASN1_STRING_canon(canon, orig), 1);
    ExpectIntEQ(ASN1_STRING_cmp(orig, canon), 0);

    ExpectIntEQ(ASN1_STRING_set(orig, data, (int)XSTRLEN(data)), 1);
    ExpectIntEQ(wolfSSL_ASN1_STRING_canon(canon, orig), 1);
    ExpectIntEQ(ASN1_STRING_cmp(orig, canon), 0);

    ASN1_STRING_free(orig);
    orig = NULL;

    ExpectNotNull(orig = ASN1_STRING_type_new(MBSTRING_UTF8));
    ExpectIntEQ(ASN1_STRING_set(orig, modData, 15), 1);
    ExpectIntEQ(wolfSSL_ASN1_STRING_canon(canon, orig), 1);
    ExpectIntEQ(ASN1_STRING_set(orig, canonData, 8), 1);
    ExpectIntEQ(ASN1_STRING_cmp(orig, canon), 0);
    ASN1_STRING_free(orig);
    orig = NULL;

    ExpectNotNull(orig = ASN1_STRING_type_new(V_ASN1_PRINTABLESTRING));
    ExpectIntEQ(ASN1_STRING_set(orig, whitespaceOnly, 3), 1);
    ExpectIntEQ(wolfSSL_ASN1_STRING_canon(canon, orig), 1);
    ASN1_STRING_free(orig);
    orig = NULL;
    ExpectNotNull(orig = ASN1_STRING_type_new(MBSTRING_UTF8));
    ExpectIntEQ(ASN1_STRING_cmp(orig, canon), 0);

    ASN1_STRING_free(orig);
    ASN1_STRING_free(canon);
#endif
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_ASN1_STRING_print(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(NO_ASN) && !defined(NO_CERTS) && \
    !defined(NO_BIO)
    ASN1_STRING* asnStr = NULL;
    const char HELLO_DATA[]= \
                      {'H','e','l','l','o',' ','w','o','l','f','S','S','L','!'};
    #define MAX_UNPRINTABLE_CHAR 32
    #define MAX_BUF 255
    unsigned char unprintableData[MAX_UNPRINTABLE_CHAR + sizeof(HELLO_DATA)];
    unsigned char expected[sizeof(unprintableData)+1];
    unsigned char rbuf[MAX_BUF];
    BIO *bio = NULL;
    int p_len;
    int i;

    /* setup */

    for (i = 0; i < (int)sizeof(HELLO_DATA); i++) {
        unprintableData[i]  = (unsigned char)HELLO_DATA[i];
        expected[i]         = (unsigned char)HELLO_DATA[i];
    }

    for (i = 0; i < (int)MAX_UNPRINTABLE_CHAR; i++) {
        unprintableData[sizeof(HELLO_DATA)+i] = i;

        if (i == (int)'\n' || i == (int)'\r')
            expected[sizeof(HELLO_DATA)+i] = i;
        else
            expected[sizeof(HELLO_DATA)+i] = '.';
    }

    unprintableData[sizeof(unprintableData)-1] = '\0';
    expected[sizeof(expected)-1] = '\0';

    XMEMSET(rbuf, 0, MAX_BUF);
    ExpectNotNull(bio = BIO_new(BIO_s_mem()));
    ExpectIntEQ(BIO_set_write_buf_size(bio, MAX_BUF), 0);

    ExpectNotNull(asnStr = ASN1_STRING_type_new(V_ASN1_OCTET_STRING));
    ExpectIntEQ(ASN1_STRING_set(asnStr,(const void*)unprintableData,
            (int)sizeof(unprintableData)), 1);
    /* test */
    ExpectIntEQ(wolfSSL_ASN1_STRING_print(NULL, NULL), 0);
    ExpectIntEQ(wolfSSL_ASN1_STRING_print(bio, NULL), 0);
    ExpectIntEQ(wolfSSL_ASN1_STRING_print(NULL, asnStr), 0);
    ExpectIntEQ(p_len = wolfSSL_ASN1_STRING_print(bio, asnStr), 46);
    ExpectIntEQ(BIO_read(bio, (void*)rbuf, 46), 46);

    ExpectStrEQ((char*)rbuf, (const char*)expected);

    BIO_free(bio);
    bio = NULL;

    ExpectNotNull(bio = BIO_new(wolfSSL_BIO_s_fixed_mem()));
    ExpectIntEQ(BIO_set_write_buf_size(bio, 1), 1);
    /* Ensure there is 0 bytes available to write into. */
    ExpectIntEQ(BIO_write(bio, rbuf, 1), 1);
    ExpectIntEQ(wolfSSL_ASN1_STRING_print(bio, asnStr), 0);
    ExpectIntEQ(BIO_set_write_buf_size(bio, 1), 1);
    ExpectIntEQ(wolfSSL_ASN1_STRING_print(bio, asnStr), 0);
    ExpectIntEQ(BIO_set_write_buf_size(bio, 45), 1);
    ExpectIntEQ(wolfSSL_ASN1_STRING_print(bio, asnStr), 0);
    BIO_free(bio);

    ASN1_STRING_free(asnStr);
#endif /* OPENSSL_EXTRA && !NO_ASN && !NO_CERTS && !NO_BIO */
    return EXPECT_RESULT();
}

static int test_wolfSSL_ASN1_STRING_print_ex(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_ASN) && !defined(NO_BIO)
    ASN1_STRING* asn_str = NULL;
    const char data[] = "Hello wolfSSL!";
    ASN1_STRING* esc_str = NULL;
    const char esc_data[] = "a+;<>";
    ASN1_STRING* neg_int = NULL;
    const char neg_int_data[] = "\xff";
    ASN1_STRING* neg_enum = NULL;
    const char neg_enum_data[] = "\xff";
    BIO *bio = NULL;
    BIO *fixed = NULL;
    unsigned long flags;
    int p_len;
    unsigned char rbuf[255];

    /* setup */
    XMEMSET(rbuf, 0, 255);
    ExpectNotNull(bio = BIO_new(BIO_s_mem()));
    ExpectIntEQ(BIO_set_write_buf_size(bio, 255), 0);
    ExpectNotNull(fixed = BIO_new(wolfSSL_BIO_s_fixed_mem()));

    ExpectNotNull(asn_str = ASN1_STRING_type_new(V_ASN1_OCTET_STRING));
    ExpectIntEQ(ASN1_STRING_set(asn_str, (const void*)data, sizeof(data)), 1);
    ExpectNotNull(esc_str = ASN1_STRING_type_new(V_ASN1_OCTET_STRING));
    ExpectIntEQ(ASN1_STRING_set(esc_str, (const void*)esc_data,
        sizeof(esc_data)), 1);
    ExpectNotNull(neg_int = ASN1_STRING_type_new(V_ASN1_NEG_INTEGER));
    ExpectIntEQ(ASN1_STRING_set(neg_int, (const void*)neg_int_data,
        sizeof(neg_int_data) - 1), 1);
    ExpectNotNull(neg_enum = ASN1_STRING_type_new(V_ASN1_NEG_ENUMERATED));
    ExpectIntEQ(ASN1_STRING_set(neg_enum, (const void*)neg_enum_data,
        sizeof(neg_enum_data) - 1), 1);

    /* Invalid parameter testing. */
    ExpectIntEQ(wolfSSL_ASN1_STRING_print_ex(NULL, NULL, 0), 0);
    ExpectIntEQ(wolfSSL_ASN1_STRING_print_ex(bio, NULL, 0), 0);
    ExpectIntEQ(wolfSSL_ASN1_STRING_print_ex(NULL, asn_str, 0), 0);

    /* no flags */
    XMEMSET(rbuf, 0, 255);
    flags = 0;
    ExpectIntEQ(p_len = wolfSSL_ASN1_STRING_print_ex(bio, asn_str, flags), 15);
    ExpectIntEQ(BIO_read(bio, (void*)rbuf, 15), 15);
    ExpectStrEQ((char*)rbuf, "Hello wolfSSL!");
    ExpectIntEQ(BIO_set_write_buf_size(fixed, 1), 1);
    /* Ensure there is 0 bytes available to write into. */
    ExpectIntEQ(BIO_write(fixed, rbuf, 1), 1);
    ExpectIntEQ(wolfSSL_ASN1_STRING_print_ex(fixed, asn_str, flags), 0);
    ExpectIntEQ(BIO_set_write_buf_size(fixed, 1), 1);
    ExpectIntEQ(wolfSSL_ASN1_STRING_print_ex(fixed, asn_str, flags), 0);
    ExpectIntEQ(BIO_set_write_buf_size(fixed, 14), 1);
    ExpectIntEQ(wolfSSL_ASN1_STRING_print_ex(fixed, asn_str, flags), 0);

    /* RFC2253 Escape */
    XMEMSET(rbuf, 0, 255);
    flags = ASN1_STRFLGS_ESC_2253;
    ExpectIntEQ(p_len = wolfSSL_ASN1_STRING_print_ex(bio, esc_str, flags), 9);
    ExpectIntEQ(BIO_read(bio, (void*)rbuf, 9), 9);
    ExpectStrEQ((char*)rbuf, "a\\+\\;\\<\\>");
    ExpectIntEQ(BIO_set_write_buf_size(fixed, 1), 1);
    /* Ensure there is 0 bytes available to write into. */
    ExpectIntEQ(BIO_write(fixed, rbuf, 1), 1);
    ExpectIntEQ(wolfSSL_ASN1_STRING_print_ex(fixed, esc_str, flags), 0);
    ExpectIntEQ(BIO_set_write_buf_size(fixed, 1), 1);
    ExpectIntEQ(wolfSSL_ASN1_STRING_print_ex(fixed, esc_str, flags), 0);
    ExpectIntEQ(BIO_set_write_buf_size(fixed, 8), 1);
    ExpectIntEQ(wolfSSL_ASN1_STRING_print_ex(fixed, esc_str, flags), 0);

    /* Show type */
    XMEMSET(rbuf, 0, 255);
    flags = ASN1_STRFLGS_SHOW_TYPE;
    ExpectIntEQ(p_len = wolfSSL_ASN1_STRING_print_ex(bio, asn_str, flags), 28);
    ExpectIntEQ(BIO_read(bio, (void*)rbuf, 28), 28);
    ExpectStrEQ((char*)rbuf, "OCTET STRING:Hello wolfSSL!");
    ExpectIntEQ(BIO_set_write_buf_size(fixed, 1), 1);
    /* Ensure there is 0 bytes available to write into. */
    ExpectIntEQ(BIO_write(fixed, rbuf, 1), 1);
    ExpectIntEQ(wolfSSL_ASN1_STRING_print_ex(fixed, asn_str, flags), 0);
    ExpectIntEQ(BIO_set_write_buf_size(fixed, 1), 1);
    ExpectIntEQ(wolfSSL_ASN1_STRING_print_ex(fixed, asn_str, flags), 0);
    ExpectIntEQ(BIO_set_write_buf_size(fixed, 12), 1);
    ExpectIntEQ(wolfSSL_ASN1_STRING_print_ex(fixed, asn_str, flags), 0);
    ExpectIntEQ(BIO_set_write_buf_size(fixed, 27), 1);
    ExpectIntEQ(wolfSSL_ASN1_STRING_print_ex(fixed, asn_str, flags), 0);

    /* Dump All */
    XMEMSET(rbuf, 0, 255);
    flags = ASN1_STRFLGS_DUMP_ALL;
    ExpectIntEQ(p_len = wolfSSL_ASN1_STRING_print_ex(bio, asn_str, flags), 31);
    ExpectIntEQ(BIO_read(bio, (void*)rbuf, 31), 31);
    ExpectStrEQ((char*)rbuf, "#48656C6C6F20776F6C6653534C2100");
    ExpectIntEQ(BIO_set_write_buf_size(fixed, 1), 1);
    /* Ensure there is 0 bytes available to write into. */
    ExpectIntEQ(BIO_write(fixed, rbuf, 1), 1);
    ExpectIntEQ(wolfSSL_ASN1_STRING_print_ex(fixed, asn_str, flags), 0);
    ExpectIntEQ(BIO_set_write_buf_size(fixed, 1), 1);
    ExpectIntEQ(wolfSSL_ASN1_STRING_print_ex(fixed, asn_str, flags), 0);
    ExpectIntEQ(BIO_set_write_buf_size(fixed, 30), 1);
    ExpectIntEQ(wolfSSL_ASN1_STRING_print_ex(fixed, asn_str, flags), 0);

    /* Dump Der */
    XMEMSET(rbuf, 0, 255);
    flags = ASN1_STRFLGS_DUMP_ALL | ASN1_STRFLGS_DUMP_DER;
    ExpectIntEQ(p_len = wolfSSL_ASN1_STRING_print_ex(bio, asn_str, flags), 35);
    ExpectIntEQ(BIO_read(bio, (void*)rbuf, 35), 35);
    ExpectStrEQ((char*)rbuf, "#040F48656C6C6F20776F6C6653534C2100");
    ExpectIntEQ(BIO_set_write_buf_size(fixed, 1), 1);
    /* Ensure there is 0 bytes available to write into. */
    ExpectIntEQ(BIO_write(fixed, rbuf, 1), 1);
    ExpectIntEQ(wolfSSL_ASN1_STRING_print_ex(fixed, asn_str, flags), 0);
    ExpectIntEQ(BIO_set_write_buf_size(fixed, 1), 1);
    ExpectIntEQ(wolfSSL_ASN1_STRING_print_ex(fixed, asn_str, flags), 0);
    ExpectIntEQ(BIO_set_write_buf_size(fixed, 2), 1);
    ExpectIntEQ(wolfSSL_ASN1_STRING_print_ex(fixed, asn_str, flags), 0);
    ExpectIntEQ(BIO_set_write_buf_size(fixed, 30), 1);
    ExpectIntEQ(wolfSSL_ASN1_STRING_print_ex(fixed, asn_str, flags), 0);

    /* Dump All + Show type */
    XMEMSET(rbuf, 0, 255);
    flags = ASN1_STRFLGS_DUMP_ALL | ASN1_STRFLGS_SHOW_TYPE;
    ExpectIntEQ(p_len = wolfSSL_ASN1_STRING_print_ex(bio, asn_str, flags), 44);
    ExpectIntEQ(BIO_read(bio, (void*)rbuf, 44), 44);
    ExpectStrEQ((char*)rbuf, "OCTET STRING:#48656C6C6F20776F6C6653534C2100");

    /* Dump All + Show type - Negative Integer. */
    XMEMSET(rbuf, 0, 255);
    flags = ASN1_STRFLGS_DUMP_ALL | ASN1_STRFLGS_SHOW_TYPE;
    ExpectIntEQ(p_len = wolfSSL_ASN1_STRING_print_ex(bio, neg_int, flags), 11);
    ExpectIntEQ(BIO_read(bio, (void*)rbuf, 11), 11);
    ExpectStrEQ((char*)rbuf, "INTEGER:#FF");

    /* Dump All + Show type - Negative Enumerated. */
    XMEMSET(rbuf, 0, 255);
    flags = ASN1_STRFLGS_DUMP_ALL | ASN1_STRFLGS_SHOW_TYPE;
    ExpectIntEQ(p_len = wolfSSL_ASN1_STRING_print_ex(bio, neg_enum, flags), 14);
    ExpectIntEQ(BIO_read(bio, (void*)rbuf, 14), 14);
    ExpectStrEQ((char*)rbuf, "ENUMERATED:#FF");

    BIO_free(fixed);
    BIO_free(bio);
    ASN1_STRING_free(asn_str);
    ASN1_STRING_free(esc_str);
    ASN1_STRING_free(neg_int);
    ASN1_STRING_free(neg_enum);

    ExpectStrEQ(wolfSSL_ASN1_tag2str(-1), "(unknown)");
    ExpectStrEQ(wolfSSL_ASN1_tag2str(31), "(unknown)");
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_ASN1_UNIVERSALSTRING_to_string(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(NO_ASN)
    ASN1_STRING* asn1str_test = NULL;
    ASN1_STRING* asn1str_answer = NULL;
    /* Each character is encoded using 4 bytes */
    char input[] = {
            0, 0, 0, 'T',
            0, 0, 0, 'e',
            0, 0, 0, 's',
            0, 0, 0, 't',
    };
    char output[] = "Test";
    char badInput[] = {
            1, 0, 0, 'T',
            0, 1, 0, 'e',
            0, 0, 1, 's',
    };

    ExpectIntEQ(ASN1_UNIVERSALSTRING_to_string(NULL), 0);
    /* Test wrong type. */
    ExpectNotNull(asn1str_test = ASN1_STRING_type_new(V_ASN1_OCTET_STRING));
    ExpectIntEQ(ASN1_UNIVERSALSTRING_to_string(asn1str_test), 0);
    ASN1_STRING_free(asn1str_test);
    asn1str_test = NULL;

    ExpectNotNull(asn1str_test = ASN1_STRING_type_new(V_ASN1_UNIVERSALSTRING));

    /* Test bad length. */
    ExpectIntEQ(ASN1_STRING_set(asn1str_test, input, sizeof(input) - 1), 1);
    ExpectIntEQ(ASN1_UNIVERSALSTRING_to_string(asn1str_test), 0);
    /* Test bad input. */
    ExpectIntEQ(ASN1_STRING_set(asn1str_test, badInput + 0, 4), 1);
    ExpectIntEQ(ASN1_UNIVERSALSTRING_to_string(asn1str_test), 0);
    ExpectIntEQ(ASN1_STRING_set(asn1str_test, badInput + 4, 4), 1);
    ExpectIntEQ(ASN1_UNIVERSALSTRING_to_string(asn1str_test), 0);
    ExpectIntEQ(ASN1_STRING_set(asn1str_test, badInput + 8, 4), 1);
    ExpectIntEQ(ASN1_UNIVERSALSTRING_to_string(asn1str_test), 0);

    ExpectIntEQ(ASN1_STRING_set(asn1str_test, input, sizeof(input)), 1);
    ExpectIntEQ(ASN1_UNIVERSALSTRING_to_string(asn1str_test), 1);

    ExpectNotNull(
        asn1str_answer = ASN1_STRING_type_new(V_ASN1_PRINTABLESTRING));
    ExpectIntEQ(ASN1_STRING_set(asn1str_answer, output, sizeof(output)-1), 1);

    ExpectIntEQ(ASN1_STRING_cmp(asn1str_test, asn1str_answer), 0);

    ASN1_STRING_free(asn1str_test);
    ASN1_STRING_free(asn1str_answer);
#endif /* OPENSSL_ALL && !NO_ASN */
    return EXPECT_RESULT();
}

static int test_wolfSSL_ASN1_GENERALIZEDTIME_free(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_ASN_TIME)
    WOLFSSL_ASN1_GENERALIZEDTIME* asn1_gtime = NULL;

    ExpectNotNull(asn1_gtime = ASN1_GENERALIZEDTIME_new());
    if (asn1_gtime != NULL)
        XMEMCPY(asn1_gtime->data, "20180504123500Z", ASN_GENERALIZED_TIME_SIZE);
    ASN1_GENERALIZEDTIME_free(asn1_gtime);
#endif /* OPENSSL_EXTRA && !NO_ASN_TIME */
    return EXPECT_RESULT();
}

static int test_wolfSSL_ASN1_GENERALIZEDTIME_print(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_ASN_TIME) && !defined(NO_BIO)
    WOLFSSL_ASN1_GENERALIZEDTIME* gtime = NULL;
    BIO* bio = NULL;
    unsigned char buf[24];
    int i;

    ExpectNotNull(bio = BIO_new(BIO_s_mem()));
    BIO_set_write_buf_size(bio, 24);

    ExpectNotNull(gtime = ASN1_GENERALIZEDTIME_new());
    /* Type not set. */
    ExpectIntEQ(wolfSSL_ASN1_GENERALIZEDTIME_print(bio, gtime), 0);
    ExpectIntEQ(wolfSSL_ASN1_TIME_set_string(gtime, "20180504123500Z"), 1);

    /* Invalid parameters testing. */
    ExpectIntEQ(wolfSSL_ASN1_GENERALIZEDTIME_print(NULL, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_ASN1_GENERALIZEDTIME_print(bio, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wolfSSL_ASN1_GENERALIZEDTIME_print(NULL, gtime), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    ExpectIntEQ(wolfSSL_ASN1_GENERALIZEDTIME_print(bio, gtime), 1);
    ExpectIntEQ(BIO_read(bio, buf, sizeof(buf)), 20);
    ExpectIntEQ(XMEMCMP(buf, "May 04 12:35:00 2018", 20), 0);

    BIO_free(bio);
    bio = NULL;

    ExpectNotNull(bio = BIO_new(wolfSSL_BIO_s_fixed_mem()));
    ExpectIntEQ(BIO_set_write_buf_size(bio, 1), 1);
    /* Ensure there is 0 bytes available to write into. */
    ExpectIntEQ(BIO_write(bio, buf, 1), 1);
    ExpectIntEQ(wolfSSL_ASN1_GENERALIZEDTIME_print(bio, gtime), 0);
    for (i = 1; i < 20; i++) {
        ExpectIntEQ(BIO_set_write_buf_size(bio, i), 1);
        ExpectIntEQ(wolfSSL_ASN1_GENERALIZEDTIME_print(bio, gtime), 0);
    }
    BIO_free(bio);

    wolfSSL_ASN1_GENERALIZEDTIME_free(gtime);
#endif /* OPENSSL_EXTRA && !NO_ASN_TIME && !NO_BIO */
    return EXPECT_RESULT();
}

static int test_wolfSSL_ASN1_TIME(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_ASN_TIME)
    WOLFSSL_ASN1_TIME* asn_time = NULL;
    unsigned char *data = NULL;

    ExpectNotNull(asn_time = ASN1_TIME_new());

#ifndef NO_WOLFSSL_STUB
    ExpectNotNull(ASN1_TIME_set(asn_time, 1));
#endif
    ExpectIntEQ(ASN1_TIME_set_string(NULL, NULL), 0);
    ExpectIntEQ(ASN1_TIME_set_string(asn_time, NULL), 0);
    ExpectIntEQ(ASN1_TIME_set_string(NULL,
        "String longer than CTC_DATA_SIZE that is 32 bytes"), 0);
    ExpectIntEQ(ASN1_TIME_set_string(NULL, "101219181011Z"), 1);
    ExpectIntEQ(ASN1_TIME_set_string(asn_time, "101219181011Z"), 1);

    ExpectIntEQ(wolfSSL_ASN1_TIME_get_length(NULL), 0);
    ExpectIntEQ(wolfSSL_ASN1_TIME_get_length(asn_time), ASN_UTC_TIME_SIZE - 1);
    ExpectNull(wolfSSL_ASN1_TIME_get_data(NULL));
    ExpectNotNull(data = wolfSSL_ASN1_TIME_get_data(asn_time));
    ExpectIntEQ(XMEMCMP(data, "101219181011Z", 14), 0);

    ExpectIntEQ(ASN1_TIME_check(NULL), 0);
    ExpectIntEQ(ASN1_TIME_check(asn_time), 1);

    ExpectIntEQ(ASN1_TIME_set_string_X509(asn_time, "101219181011Z"), 1);
    ExpectIntEQ(ASN1_TIME_set_string_X509(asn_time, "101219181011Za"), 0);

    ASN1_TIME_free(asn_time);
    ASN1_TIME_free(NULL);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_ASN1_TIME_to_string(void)
{
    EXPECT_DECLS;
#ifndef NO_ASN_TIME
#if defined(WOLFSSL_MYSQL_COMPATIBLE) || defined(WOLFSSL_NGINX) || \
    defined(WOLFSSL_HAPROXY) || defined(OPENSSL_EXTRA) || defined(OPENSSL_ALL)
    WOLFSSL_ASN1_TIME* t = NULL;
    char buf[ASN_GENERALIZED_TIME_SIZE];

    ExpectNotNull((t = ASN1_TIME_new()));
    ExpectIntEQ(ASN1_TIME_set_string(t, "030222211515Z"), 1);

    /* Invalid parameter testing. */
    ExpectNull(ASN1_TIME_to_string(NULL, NULL, 4));
    ExpectNull(ASN1_TIME_to_string(t, NULL, 4));
    ExpectNull(ASN1_TIME_to_string(NULL, buf, 4));
    ExpectNull(ASN1_TIME_to_string(NULL, NULL, 5));
    ExpectNull(ASN1_TIME_to_string(NULL, buf, 5));
    ExpectNull(ASN1_TIME_to_string(t, NULL, 5));
    ExpectNull(ASN1_TIME_to_string(t, buf, 4));
    /* Buffer needs to be longer than minimum of 5 characters. */
    ExpectNull(ASN1_TIME_to_string(t, buf, 5));

    ASN1_TIME_free(t);
#endif
#endif /* NO_ASN_TIME */
    return EXPECT_RESULT();
}

static int test_wolfSSL_ASN1_TIME_diff_compare(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_ASN_TIME)
    ASN1_TIME* fromTime = NULL;
    ASN1_TIME* closeToTime = NULL;
    ASN1_TIME* toTime = NULL;
    ASN1_TIME* invalidTime = NULL;
    int daysDiff = 0;
    int secsDiff = 0;

    ExpectNotNull((fromTime = ASN1_TIME_new()));
    /* Feb 22, 2003, 21:15:15 */
    ExpectIntEQ(ASN1_TIME_set_string(fromTime, "030222211515Z"), 1);
    ExpectNotNull((closeToTime = ASN1_TIME_new()));
    /* Feb 22, 2003, 21:16:15 */
    ExpectIntEQ(ASN1_TIME_set_string(closeToTime, "030222211615Z"), 1);
    ExpectNotNull((toTime = ASN1_TIME_new()));
    /* Dec 19, 2010, 18:10:11 */
    ExpectIntEQ(ASN1_TIME_set_string(toTime, "101219181011Z"), 1);
    ExpectNotNull((invalidTime = ASN1_TIME_new()));
    /* Dec 19, 2010, 18:10:11 but 'U' instead of 'Z' which is invalid. */
    ExpectIntEQ(ASN1_TIME_set_string(invalidTime, "102519181011U"), 1);

    ExpectIntEQ(ASN1_TIME_diff(&daysDiff, &secsDiff, fromTime, invalidTime), 0);
    ExpectIntEQ(ASN1_TIME_diff(&daysDiff, &secsDiff, invalidTime, toTime), 0);

    ExpectIntEQ(ASN1_TIME_diff(&daysDiff, &secsDiff, fromTime, toTime), 1);

    /* Test when secsDiff or daysDiff is NULL. */
    ExpectIntEQ(ASN1_TIME_diff(NULL, &secsDiff, fromTime, toTime), 1);
    ExpectIntEQ(ASN1_TIME_diff(&daysDiff, NULL, fromTime, toTime), 1);
    ExpectIntEQ(ASN1_TIME_diff(NULL, NULL, fromTime, toTime), 1);

    /* If both times are NULL, difference is 0. */
    ExpectIntEQ(ASN1_TIME_diff(&daysDiff, &secsDiff, NULL, NULL), 1);
    ExpectIntEQ(daysDiff, 0);
    ExpectIntEQ(secsDiff, 0);

    /* If one time is NULL, it defaults to the current time. */
    ExpectIntEQ(ASN1_TIME_diff(&daysDiff, &secsDiff, NULL, toTime), 1);
    ExpectIntEQ(ASN1_TIME_diff(&daysDiff, &secsDiff, fromTime, NULL), 1);

    /* Normal operation. Both times non-NULL. */
    ExpectIntEQ(ASN1_TIME_diff(&daysDiff, &secsDiff, fromTime, toTime), 1);
    ExpectIntEQ(daysDiff, 2856);
    ExpectIntEQ(secsDiff, 75296);
    /* Swapping the times should return negative values. */
    ExpectIntEQ(ASN1_TIME_diff(&daysDiff, &secsDiff, toTime, fromTime), 1);
    ExpectIntEQ(daysDiff, -2856);
    ExpectIntEQ(secsDiff, -75296);

    /* Compare with invalid time string. */
    ExpectIntEQ(ASN1_TIME_compare(fromTime, invalidTime), -2);
    ExpectIntEQ(ASN1_TIME_compare(invalidTime, toTime), -2);
    /* Compare with days difference of 0. */
    ExpectIntEQ(ASN1_TIME_compare(fromTime, closeToTime), -1);
    ExpectIntEQ(ASN1_TIME_compare(closeToTime, fromTime), 1);
    /* Days and seconds differences not 0. */
    ExpectIntEQ(ASN1_TIME_compare(fromTime, toTime), -1);
    ExpectIntEQ(ASN1_TIME_compare(toTime, fromTime), 1);
    /* Same time. */
    ExpectIntEQ(ASN1_TIME_compare(fromTime, fromTime), 0);

    /* Compare regression test: No seconds difference, just difference in days.
     */
    ASN1_TIME_set_string(fromTime, "19700101000000Z");
    ASN1_TIME_set_string(toTime, "19800101000000Z");
    ExpectIntEQ(ASN1_TIME_compare(fromTime, toTime), -1);
    ExpectIntEQ(ASN1_TIME_compare(toTime, fromTime), 1);
    ExpectIntEQ(ASN1_TIME_compare(fromTime, fromTime), 0);

    /* Edge case with Unix epoch. */
    ExpectNotNull(ASN1_TIME_set_string(fromTime, "19700101000000Z"));
    ExpectNotNull(ASN1_TIME_set_string(toTime, "19800101000000Z"));
    ExpectIntEQ(ASN1_TIME_diff(&daysDiff, &secsDiff, fromTime, toTime), 1);
    ExpectIntEQ(daysDiff, 3652);
    ExpectIntEQ(secsDiff, 0);

    /* Edge case with year > 2038 (year 2038 problem). */
    ExpectNotNull(ASN1_TIME_set_string(toTime, "99991231235959Z"));
    ExpectIntEQ(ASN1_TIME_diff(&daysDiff, &secsDiff, fromTime, toTime), 1);
    ExpectIntEQ(daysDiff, 2932896);
    ExpectIntEQ(secsDiff, 86399);

    ASN1_TIME_free(fromTime);
    ASN1_TIME_free(closeToTime);
    ASN1_TIME_free(toTime);
    ASN1_TIME_free(invalidTime);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_ASN1_TIME_adj(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_ASN_TIME) && \
    !defined(USER_TIME) && !defined(TIME_OVERRIDES)
    const int year = 365*24*60*60;
    const int day  = 24*60*60;
    const int hour = 60*60;
    const int mini = 60;
    const byte asn_utc_time = ASN_UTC_TIME;
#if !defined(TIME_T_NOT_64BIT) && !defined(NO_64BIT)
    const byte asn_gen_time = ASN_GENERALIZED_TIME;
#endif
    WOLFSSL_ASN1_TIME* asn_time = NULL;
    WOLFSSL_ASN1_TIME* s = NULL;
    int offset_day;
    long offset_sec;
    char date_str[CTC_DATE_SIZE + 1];
    time_t t;

    ExpectNotNull(s = wolfSSL_ASN1_TIME_new());
    /* UTC notation test */
    /* 2000/2/15 20:30:00 */
    t = (time_t)30 * year + 45 * day + 20 * hour + 30 * mini + 7 * day;
    offset_day = 7;
    offset_sec = 45 * mini;
    /* offset_sec = -45 * min;*/
    ExpectNotNull(asn_time =
            wolfSSL_ASN1_TIME_adj(s, t, offset_day, offset_sec));
    ExpectTrue(asn_time->type == asn_utc_time);
    ExpectNotNull(XSTRNCPY(date_str, (const char*)&asn_time->data,
        CTC_DATE_SIZE));
    date_str[CTC_DATE_SIZE] = '\0';
    ExpectIntEQ(0, XMEMCMP(date_str, "000222211500Z", 13));

    /* negative offset */
    offset_sec = -45 * mini;
    asn_time = wolfSSL_ASN1_TIME_adj(s, t, offset_day, offset_sec);
    ExpectNotNull(asn_time);
    ExpectTrue(asn_time->type == asn_utc_time);
    ExpectNotNull(XSTRNCPY(date_str, (const char*)&asn_time->data,
        CTC_DATE_SIZE));
    date_str[CTC_DATE_SIZE] = '\0';
    ExpectIntEQ(0, XMEMCMP(date_str, "000222194500Z", 13));

    XFREE(s, NULL, DYNAMIC_TYPE_OPENSSL);
    s = NULL;
    XMEMSET(date_str, 0, sizeof(date_str));

    /* Generalized time will overflow time_t if not long */
#if !defined(TIME_T_NOT_64BIT) && !defined(NO_64BIT)
    s = (WOLFSSL_ASN1_TIME*)XMALLOC(sizeof(WOLFSSL_ASN1_TIME), NULL,
                                    DYNAMIC_TYPE_OPENSSL);
    /* GeneralizedTime notation test */
    /* 2055/03/01 09:00:00 */
    t = (time_t)85 * year + 59 * day + 9 * hour + 21 * day;
        offset_day = 12;
        offset_sec = 10 * mini;
    ExpectNotNull(asn_time = wolfSSL_ASN1_TIME_adj(s, t, offset_day,
        offset_sec));
    ExpectTrue(asn_time->type == asn_gen_time);
    ExpectNotNull(XSTRNCPY(date_str, (const char*)&asn_time->data,
        CTC_DATE_SIZE));
    date_str[CTC_DATE_SIZE] = '\0';
    ExpectIntEQ(0, XMEMCMP(date_str, "20550313091000Z", 15));

    XFREE(s, NULL, DYNAMIC_TYPE_OPENSSL);
    s = NULL;
    XMEMSET(date_str, 0, sizeof(date_str));
#endif /* !TIME_T_NOT_64BIT && !NO_64BIT */

    /* if WOLFSSL_ASN1_TIME struct is not allocated */
    s = NULL;

    t = (time_t)30 * year + 45 * day + 20 * hour + 30 * mini + 15 + 7 * day;
    offset_day = 7;
    offset_sec = 45 * mini;
    ExpectNotNull(asn_time = wolfSSL_ASN1_TIME_adj(s, t, offset_day,
        offset_sec));
    ExpectTrue(asn_time->type == asn_utc_time);
    ExpectNotNull(XSTRNCPY(date_str, (const char*)&asn_time->data,
        CTC_DATE_SIZE));
    date_str[CTC_DATE_SIZE] = '\0';
    ExpectIntEQ(0, XMEMCMP(date_str, "000222211515Z", 13));
    XFREE(asn_time, NULL, DYNAMIC_TYPE_OPENSSL);
    asn_time = NULL;

    ExpectNotNull(asn_time = wolfSSL_ASN1_TIME_adj(NULL, t, offset_day,
        offset_sec));
    ExpectTrue(asn_time->type == asn_utc_time);
    ExpectNotNull(XSTRNCPY(date_str, (const char*)&asn_time->data,
        CTC_DATE_SIZE));
    date_str[CTC_DATE_SIZE] = '\0';
    ExpectIntEQ(0, XMEMCMP(date_str, "000222211515Z", 13));
    XFREE(asn_time, NULL, DYNAMIC_TYPE_OPENSSL);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_ASN1_TIME_to_tm(void)
{
    EXPECT_DECLS;
#if (defined(WOLFSSL_MYSQL_COMPATIBLE) || defined(WOLFSSL_NGINX) || \
      defined(WOLFSSL_HAPROXY) || defined(OPENSSL_EXTRA) || \
      defined(OPENSSL_ALL)) && !defined(NO_ASN_TIME)
    ASN1_TIME asnTime;
    struct tm tm;
    time_t testTime = 1683926567; /* Fri May 12 09:22:47 PM UTC 2023 */

    XMEMSET(&tm, 0, sizeof(struct tm));

    XMEMSET(&asnTime, 0, sizeof(ASN1_TIME));
    ExpectIntEQ(ASN1_TIME_set_string(&asnTime, "000222211515Z"), 1);
    ExpectIntEQ(ASN1_TIME_to_tm(&asnTime, NULL), 1);
    ExpectIntEQ(ASN1_TIME_to_tm(&asnTime, &tm), 1);

    ExpectIntEQ(tm.tm_sec, 15);
    ExpectIntEQ(tm.tm_min, 15);
    ExpectIntEQ(tm.tm_hour, 21);
    ExpectIntEQ(tm.tm_mday, 22);
    ExpectIntEQ(tm.tm_mon, 1);
    ExpectIntEQ(tm.tm_year, 100);
    ExpectIntEQ(tm.tm_isdst, 0);
#ifdef XMKTIME
    ExpectIntEQ(tm.tm_wday, 2);
    ExpectIntEQ(tm.tm_yday, 52);
#endif

    ExpectIntEQ(ASN1_TIME_set_string(&asnTime, "500222211515Z"), 1);
    ExpectIntEQ(ASN1_TIME_to_tm(&asnTime, &tm), 1);
    ExpectIntEQ(tm.tm_year, 50);

    /* Get current time. */
    ExpectIntEQ(ASN1_TIME_to_tm(NULL, NULL), 0);
    ExpectIntEQ(ASN1_TIME_to_tm(NULL, &tm), 1);

    XMEMSET(&asnTime, 0, sizeof(ASN1_TIME));
    /* 0 length. */
    ExpectIntEQ(ASN1_TIME_to_tm(&asnTime, &tm), 0);
    /* No type. */
    asnTime.length = 1;
    ExpectIntEQ(ASN1_TIME_to_tm(&asnTime, &tm), 0);
    /* Not UTCTIME length. */
    asnTime.type = V_ASN1_UTCTIME;
    ExpectIntEQ(ASN1_TIME_to_tm(&asnTime, &tm), 0);
    /* Not GENERALIZEDTIME length. */
    asnTime.type = V_ASN1_GENERALIZEDTIME;
    ExpectIntEQ(ASN1_TIME_to_tm(&asnTime, &tm), 0);

    /* Not Zulu timezone. */
    ExpectIntEQ(ASN1_TIME_set_string(&asnTime, "000222211515U"), 1);
    ExpectIntEQ(ASN1_TIME_to_tm(&asnTime, &tm), 0);
    ExpectIntEQ(ASN1_TIME_set_string(&asnTime, "20000222211515U"), 1);
    ExpectIntEQ(ASN1_TIME_to_tm(&asnTime, &tm), 0);

#ifdef XMKTIME
    ExpectNotNull(ASN1_TIME_adj(&asnTime, testTime, 0, 0));
    ExpectIntEQ(ASN1_TIME_to_tm(&asnTime, &tm), 1);
    ExpectIntEQ(tm.tm_sec, 47);
    ExpectIntEQ(tm.tm_min, 22);
    ExpectIntEQ(tm.tm_hour, 21);
    ExpectIntEQ(tm.tm_mday, 12);
    ExpectIntEQ(tm.tm_mon, 4);
    ExpectIntEQ(tm.tm_year, 123);
    ExpectIntEQ(tm.tm_wday, 5);
    ExpectIntEQ(tm.tm_yday, 131);
    /* Confirm that when used with a tm struct from ASN1_TIME_adj, all other
       fields are zeroed out as expected. */
    ExpectIntEQ(tm.tm_isdst, 0);
#endif
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_ASN1_TIME_to_generalizedtime(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_ASN_TIME)
    WOLFSSL_ASN1_TIME *t = NULL;
    WOLFSSL_ASN1_TIME *out = NULL;
    WOLFSSL_ASN1_TIME *gtime = NULL;
    int tlen = 0;
    unsigned char *data = NULL;

    ExpectNotNull(t = wolfSSL_ASN1_TIME_new());
    ExpectNull(wolfSSL_ASN1_TIME_to_generalizedtime(NULL, &out));
    /* type not set. */
    ExpectNull(wolfSSL_ASN1_TIME_to_generalizedtime(t, &out));
    XFREE(t, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    t = NULL;

    /* UTC Time test */
    ExpectNotNull(t = wolfSSL_ASN1_TIME_new());
    if (t != NULL) {
        XMEMSET(t->data, 0, ASN_GENERALIZED_TIME_SIZE);
        t->type = ASN_UTC_TIME;
        t->length = ASN_UTC_TIME_SIZE;
        XMEMCPY(t->data, "050727123456Z", ASN_UTC_TIME_SIZE);
    }

    ExpectIntEQ(tlen = wolfSSL_ASN1_TIME_get_length(t), ASN_UTC_TIME_SIZE);
    ExpectStrEQ((char*)(data = wolfSSL_ASN1_TIME_get_data(t)), "050727123456Z");

    out = NULL;
    ExpectNotNull(gtime = wolfSSL_ASN1_TIME_to_generalizedtime(t, &out));
    wolfSSL_ASN1_TIME_free(gtime);
    gtime = NULL;
    ExpectNotNull(out = wolfSSL_ASN1_TIME_new());
    ExpectNotNull(gtime = wolfSSL_ASN1_TIME_to_generalizedtime(t, &out));
    ExpectPtrEq(gtime, out);
    ExpectIntEQ(gtime->type, ASN_GENERALIZED_TIME);
    ExpectIntEQ(gtime->length, ASN_GENERALIZED_TIME_SIZE);
    ExpectStrEQ((char*)gtime->data, "20050727123456Z");

    /* Generalized Time test */
    ExpectNotNull(XMEMSET(t, 0, ASN_GENERALIZED_TIME_SIZE));
    ExpectNotNull(XMEMSET(out, 0, ASN_GENERALIZED_TIME_SIZE));
    ExpectNotNull(XMEMSET(data, 0, ASN_GENERALIZED_TIME_SIZE));
    if (t != NULL) {
        t->type = ASN_GENERALIZED_TIME;
        t->length = ASN_GENERALIZED_TIME_SIZE;
        XMEMCPY(t->data, "20050727123456Z", ASN_GENERALIZED_TIME_SIZE);
    }

    ExpectIntEQ(tlen = wolfSSL_ASN1_TIME_get_length(t),
        ASN_GENERALIZED_TIME_SIZE);
    ExpectStrEQ((char*)(data = wolfSSL_ASN1_TIME_get_data(t)),
        "20050727123456Z");
    ExpectNotNull(gtime = wolfSSL_ASN1_TIME_to_generalizedtime(t, &out));
    ExpectIntEQ(gtime->type, ASN_GENERALIZED_TIME);
    ExpectIntEQ(gtime->length, ASN_GENERALIZED_TIME_SIZE);
    ExpectStrEQ((char*)gtime->data, "20050727123456Z");

    /* UTC Time to Generalized Time 1900's test */
    ExpectNotNull(XMEMSET(t, 0, ASN_GENERALIZED_TIME_SIZE));
    ExpectNotNull(XMEMSET(out, 0, ASN_GENERALIZED_TIME_SIZE));
    ExpectNotNull(XMEMSET(data, 0, ASN_GENERALIZED_TIME_SIZE));
    if (t != NULL) {
        t->type = ASN_UTC_TIME;
        t->length = ASN_UTC_TIME_SIZE;
        XMEMCPY(t->data, "500727123456Z", ASN_UTC_TIME_SIZE);
    }

    ExpectNotNull(gtime = wolfSSL_ASN1_TIME_to_generalizedtime(t, &out));
    ExpectIntEQ(gtime->type, ASN_GENERALIZED_TIME);
    ExpectIntEQ(gtime->length, ASN_GENERALIZED_TIME_SIZE);
    ExpectStrEQ((char*)gtime->data, "19500727123456Z");
    XFREE(out, NULL, DYNAMIC_TYPE_TMP_BUFFER);

    /* Null parameter test */
    ExpectNotNull(XMEMSET(t, 0, ASN_GENERALIZED_TIME_SIZE));
    gtime = NULL;
    out = NULL;
    if (t != NULL) {
        t->type = ASN_UTC_TIME;
        t->length = ASN_UTC_TIME_SIZE;
        XMEMCPY(t->data, "050727123456Z", ASN_UTC_TIME_SIZE);
    }
    ExpectNotNull(gtime = wolfSSL_ASN1_TIME_to_generalizedtime(t, NULL));
    ExpectIntEQ(gtime->type, ASN_GENERALIZED_TIME);
    ExpectIntEQ(gtime->length, ASN_GENERALIZED_TIME_SIZE);
    ExpectStrEQ((char*)gtime->data, "20050727123456Z");

    XFREE(gtime, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(t, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_ASN1_TIME_print(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && !defined(NO_RSA) && !defined(NO_BIO) && \
    (defined(WOLFSSL_MYSQL_COMPATIBLE) || defined(WOLFSSL_NGINX) || \
     defined(WOLFSSL_HAPROXY) || defined(OPENSSL_EXTRA) || \
     defined(OPENSSL_ALL)) && defined(USE_CERT_BUFFERS_2048) && \
    !defined(NO_ASN_TIME)
    BIO*  bio = NULL;
    BIO*  fixed = NULL;
    X509*  x509 = NULL;
    const unsigned char* der = client_cert_der_2048;
    ASN1_TIME* notAfter = NULL;
    ASN1_TIME* notBefore = NULL;
    unsigned char buf[25];

    ExpectNotNull(bio = BIO_new(BIO_s_mem()));
    ExpectNotNull(fixed = BIO_new(wolfSSL_BIO_s_fixed_mem()));
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_buffer(der,
                sizeof_client_cert_der_2048, WOLFSSL_FILETYPE_ASN1));
    ExpectNotNull(notBefore = X509_get_notBefore(x509));

    ExpectIntEQ(ASN1_TIME_print(NULL, NULL), 0);
    ExpectIntEQ(ASN1_TIME_print(bio, NULL), 0);
    ExpectIntEQ(ASN1_TIME_print(NULL, notBefore), 0);

    ExpectIntEQ(ASN1_TIME_print(bio, notBefore), 1);
    ExpectIntEQ(BIO_read(bio, buf, sizeof(buf)), 24);
    ExpectIntEQ(XMEMCMP(buf, "Dec 18 21:25:29 2024 GMT", sizeof(buf) - 1), 0);

    /* Test BIO_write fails. */
    ExpectIntEQ(BIO_set_write_buf_size(fixed, 1), 1);
    /* Ensure there is 0 bytes available to write into. */
    ExpectIntEQ(BIO_write(fixed, buf, 1), 1);
    ExpectIntEQ(ASN1_TIME_print(fixed, notBefore), 0);
    ExpectIntEQ(BIO_set_write_buf_size(fixed, 1), 1);
    ExpectIntEQ(ASN1_TIME_print(fixed, notBefore), 0);
    ExpectIntEQ(BIO_set_write_buf_size(fixed, 23), 1);
    ExpectIntEQ(ASN1_TIME_print(fixed, notBefore), 0);

    /* create a bad time and test results */
    ExpectNotNull(notAfter = X509_get_notAfter(x509));
    ExpectIntEQ(ASN1_TIME_check(notAfter), 1);
    if (EXPECT_SUCCESS()) {
        notAfter->data[8] = 0;
        notAfter->data[3] = 0;
    }
    ExpectIntNE(ASN1_TIME_print(bio, notAfter), 1);
    ExpectIntEQ(BIO_read(bio, buf, sizeof(buf)), 14);
    ExpectIntEQ(XMEMCMP(buf, "Bad time value", 14), 0);
    ExpectIntEQ(ASN1_TIME_check(notAfter), 0);

    BIO_free(bio);
    BIO_free(fixed);
    X509_free(x509);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_ASN1_UTCTIME_print(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_ASN_TIME) && !defined(NO_BIO)
    BIO*  bio = NULL;
    ASN1_UTCTIME* utc = NULL;
    unsigned char buf[25];
    const char* validDate   = "190424111501Z";   /* UTC =   YYMMDDHHMMSSZ */
    const char* invalidDate = "190424111501X";   /* UTC =   YYMMDDHHMMSSZ */
    const char* genDate     = "20190424111501Z"; /* GEN = YYYYMMDDHHMMSSZ */

    /* Valid date */
    ExpectNotNull(bio = BIO_new(BIO_s_mem()));
    ExpectNotNull(utc = (ASN1_UTCTIME*)XMALLOC(sizeof(ASN1_UTCTIME), NULL,
                                                           DYNAMIC_TYPE_ASN1));
    if (utc != NULL) {
        utc->type = ASN_UTC_TIME;
        utc->length = ASN_UTC_TIME_SIZE;
        XMEMCPY(utc->data, (byte*)validDate, ASN_UTC_TIME_SIZE);
    }

    ExpectIntEQ(ASN1_UTCTIME_print(NULL, NULL), 0);
    ExpectIntEQ(ASN1_UTCTIME_print(bio, NULL), 0);
    ExpectIntEQ(ASN1_UTCTIME_print(NULL, utc), 0);

    ExpectIntEQ(ASN1_UTCTIME_print(bio, utc), 1);
    ExpectIntEQ(BIO_read(bio, buf, sizeof(buf)), 24);
    ExpectIntEQ(XMEMCMP(buf, "Apr 24 11:15:01 2019 GMT", sizeof(buf)-1), 0);

    XMEMSET(buf, 0, sizeof(buf));
    BIO_free(bio);
    bio = NULL;

    /* Invalid format */
    ExpectNotNull(bio = BIO_new(BIO_s_mem()));
    if (utc != NULL) {
        utc->type = ASN_UTC_TIME;
        utc->length = ASN_UTC_TIME_SIZE;
        XMEMCPY(utc->data, (byte*)invalidDate, ASN_UTC_TIME_SIZE);
    }
    ExpectIntEQ(ASN1_UTCTIME_print(bio, utc), 0);
    ExpectIntEQ(BIO_read(bio, buf, sizeof(buf)), 14);
    ExpectIntEQ(XMEMCMP(buf, "Bad time value", 14), 0);

    /* Invalid type */
    if (utc != NULL) {
        utc->type = ASN_GENERALIZED_TIME;
        utc->length = ASN_GENERALIZED_TIME_SIZE;
        XMEMCPY(utc->data, (byte*)genDate, ASN_GENERALIZED_TIME_SIZE);
    }
    ExpectIntEQ(ASN1_UTCTIME_print(bio, utc), 0);

    XFREE(utc, NULL, DYNAMIC_TYPE_ASN1);
    BIO_free(bio);
#endif /* OPENSSL_EXTRA && !NO_ASN_TIME && !NO_BIO */
    return EXPECT_RESULT();
}

static int test_wolfSSL_ASN1_TYPE(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) || defined(WOLFSSL_APACHE_HTTPD) || \
    defined(WOLFSSL_HAPROXY) || defined(WOLFSSL_WPAS)
    WOLFSSL_ASN1_TYPE* t = NULL;
    WOLFSSL_ASN1_OBJECT* obj = NULL;
#ifndef NO_ASN_TIME
    WOLFSSL_ASN1_TIME* time = NULL;
#endif
    WOLFSSL_ASN1_STRING* str = NULL;
    unsigned char data[] = { 0x00 };

    ASN1_TYPE_set(NULL, V_ASN1_NULL, NULL);

    ExpectNotNull(t = wolfSSL_ASN1_TYPE_new());
    ASN1_TYPE_set(t, V_ASN1_EOC, NULL);
    wolfSSL_ASN1_TYPE_free(t);
    t = NULL;

    ExpectNotNull(t = wolfSSL_ASN1_TYPE_new());
    ASN1_TYPE_set(t, V_ASN1_NULL, NULL);
    ASN1_TYPE_set(t, V_ASN1_NULL, data);
    wolfSSL_ASN1_TYPE_free(t);
    t = NULL;

    ExpectNotNull(t = wolfSSL_ASN1_TYPE_new());
    ExpectNotNull(obj = wolfSSL_ASN1_OBJECT_new());
    ASN1_TYPE_set(t, V_ASN1_OBJECT, obj);
    wolfSSL_ASN1_TYPE_free(t);
    t = NULL;

#ifndef NO_ASN_TIME
    ExpectNotNull(t = wolfSSL_ASN1_TYPE_new());
    ExpectNotNull(time = wolfSSL_ASN1_TIME_new());
    ASN1_TYPE_set(t, V_ASN1_UTCTIME, time);
    wolfSSL_ASN1_TYPE_free(t);
    t = NULL;

    ExpectNotNull(t = wolfSSL_ASN1_TYPE_new());
    ExpectNotNull(time = wolfSSL_ASN1_TIME_new());
    ASN1_TYPE_set(t, V_ASN1_GENERALIZEDTIME, time);
    wolfSSL_ASN1_TYPE_free(t);
    t = NULL;
#endif

    ExpectNotNull(t = wolfSSL_ASN1_TYPE_new());
    ExpectNotNull(str = wolfSSL_ASN1_STRING_new());
    ASN1_TYPE_set(t, V_ASN1_UTF8STRING, str);
    wolfSSL_ASN1_TYPE_free(t);
    t = NULL;

    ExpectNotNull(t = wolfSSL_ASN1_TYPE_new());
    ExpectNotNull(str = wolfSSL_ASN1_STRING_new());
    ASN1_TYPE_set(t, V_ASN1_PRINTABLESTRING, str);
    wolfSSL_ASN1_TYPE_free(t);
    t = NULL;

    ExpectNotNull(t = wolfSSL_ASN1_TYPE_new());
    ExpectNotNull(str = wolfSSL_ASN1_STRING_new());
    ASN1_TYPE_set(t, V_ASN1_T61STRING, str);
    wolfSSL_ASN1_TYPE_free(t);
    t = NULL;

    ExpectNotNull(t = wolfSSL_ASN1_TYPE_new());
    ExpectNotNull(str = wolfSSL_ASN1_STRING_new());
    ASN1_TYPE_set(t, V_ASN1_IA5STRING, str);
    wolfSSL_ASN1_TYPE_free(t);
    t = NULL;

    ExpectNotNull(t = wolfSSL_ASN1_TYPE_new());
    ExpectNotNull(str = wolfSSL_ASN1_STRING_new());
    ASN1_TYPE_set(t, V_ASN1_UNIVERSALSTRING, str);
    wolfSSL_ASN1_TYPE_free(t);
    t = NULL;

    ExpectNotNull(t = wolfSSL_ASN1_TYPE_new());
    ExpectNotNull(str = wolfSSL_ASN1_STRING_new());
    ASN1_TYPE_set(t, V_ASN1_SEQUENCE, str);
    wolfSSL_ASN1_TYPE_free(t);
    t = NULL;
#endif
    return EXPECT_RESULT();
}

/* Testing code used in old dpp.c in hostap */
#if defined(OPENSSL_ALL) && defined(HAVE_ECC) && defined(USE_CERT_BUFFERS_256)
typedef struct {
    /* AlgorithmIdentifier ecPublicKey with optional parameters present
     * as an OID identifying the curve */
    X509_ALGOR *alg;
    /* Compressed format public key per ANSI X9.63 */
    ASN1_BIT_STRING *pub_key;
} DPP_BOOTSTRAPPING_KEY;

ASN1_SEQUENCE(DPP_BOOTSTRAPPING_KEY) = {
    ASN1_SIMPLE(DPP_BOOTSTRAPPING_KEY, alg, X509_ALGOR),
    ASN1_SIMPLE(DPP_BOOTSTRAPPING_KEY, pub_key, ASN1_BIT_STRING)
} ASN1_SEQUENCE_END(DPP_BOOTSTRAPPING_KEY)

IMPLEMENT_ASN1_FUNCTIONS(DPP_BOOTSTRAPPING_KEY)

typedef struct {
    int type;
    union {
        ASN1_BIT_STRING *str1;
        ASN1_BIT_STRING *str2;
        ASN1_BIT_STRING *str3;
    } d;
} ASN1_CHOICE_TEST;

ASN1_CHOICE(ASN1_CHOICE_TEST) = {
    ASN1_IMP(ASN1_CHOICE_TEST, d.str1, ASN1_BIT_STRING, 1),
    ASN1_IMP(ASN1_CHOICE_TEST, d.str2, ASN1_BIT_STRING, 2),
    ASN1_IMP(ASN1_CHOICE_TEST, d.str3, ASN1_BIT_STRING, 3)
} ASN1_CHOICE_END(ASN1_CHOICE_TEST)

IMPLEMENT_ASN1_FUNCTIONS(ASN1_CHOICE_TEST)

/* Test nested objects */
typedef struct {
    DPP_BOOTSTRAPPING_KEY* key;
    ASN1_INTEGER* asnNum;
    ASN1_INTEGER* expNum;
    STACK_OF(ASN1_GENERALSTRING) *strList;
    ASN1_CHOICE_TEST* str;
} TEST_ASN1_NEST1;

ASN1_SEQUENCE(TEST_ASN1_NEST1) = {
    ASN1_SIMPLE(TEST_ASN1_NEST1, key, DPP_BOOTSTRAPPING_KEY),
    ASN1_SIMPLE(TEST_ASN1_NEST1, asnNum, ASN1_INTEGER),
    ASN1_EXP(TEST_ASN1_NEST1, expNum, ASN1_INTEGER, 0),
    ASN1_EXP_SEQUENCE_OF(TEST_ASN1_NEST1, strList, ASN1_GENERALSTRING, 1),
    ASN1_SIMPLE(TEST_ASN1_NEST1, str, ASN1_CHOICE_TEST)
} ASN1_SEQUENCE_END(TEST_ASN1_NEST1)

IMPLEMENT_ASN1_FUNCTIONS(TEST_ASN1_NEST1)

typedef struct {
    ASN1_INTEGER* num;
    DPP_BOOTSTRAPPING_KEY* key;
    TEST_ASN1_NEST1* asn1_obj;
} TEST_ASN1_NEST2;

ASN1_SEQUENCE(TEST_ASN1_NEST2) = {
    ASN1_SIMPLE(TEST_ASN1_NEST2, num, ASN1_INTEGER),
    ASN1_SIMPLE(TEST_ASN1_NEST2, key, DPP_BOOTSTRAPPING_KEY),
    ASN1_SIMPLE(TEST_ASN1_NEST2, asn1_obj, TEST_ASN1_NEST1)
} ASN1_SEQUENCE_END(TEST_ASN1_NEST2)

IMPLEMENT_ASN1_FUNCTIONS(TEST_ASN1_NEST2)
/* End nested objects */

typedef struct {
    ASN1_INTEGER *integer;
} TEST_ASN1;

ASN1_SEQUENCE(TEST_ASN1) = {
    ASN1_SIMPLE(TEST_ASN1, integer, ASN1_INTEGER),
} ASN1_SEQUENCE_END(TEST_ASN1)

IMPLEMENT_ASN1_FUNCTIONS(TEST_ASN1)

typedef STACK_OF(ASN1_INTEGER) TEST_ASN1_ITEM;

ASN1_ITEM_TEMPLATE(TEST_ASN1_ITEM) =
        ASN1_EX_TEMPLATE_TYPE(ASN1_TFLG_SEQUENCE_OF, 0, MemName, ASN1_INTEGER)
ASN1_ITEM_TEMPLATE_END(TEST_ASN1_ITEM)

IMPLEMENT_ASN1_FUNCTIONS(TEST_ASN1_ITEM)
#endif

static int test_wolfSSL_IMPLEMENT_ASN1_FUNCTIONS(void)
{
    EXPECT_DECLS;
    /* Testing code used in dpp.c in hostap */
#if defined(OPENSSL_ALL) && defined(HAVE_ECC) && defined(USE_CERT_BUFFERS_256)
#if !defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION>2))
    EC_KEY *eckey = NULL;
    EVP_PKEY *key = NULL;
    size_t len = 0;
    unsigned char *der = NULL;
    unsigned char *der2 = NULL;
    const unsigned char *tmp = NULL;
    DPP_BOOTSTRAPPING_KEY *bootstrap = NULL, *bootstrap2 = NULL;
    const unsigned char *in = ecc_clikey_der_256;
    WOLFSSL_ASN1_OBJECT* ec_obj = NULL;
    WOLFSSL_ASN1_OBJECT* group_obj = NULL;
    const EC_GROUP *group = NULL;
    const EC_POINT *point = NULL;
    int nid;
    TEST_ASN1 *test_asn1 = NULL;
    TEST_ASN1 *test_asn1_2 = NULL;

    const unsigned char badObjDer[] = { 0x06, 0x00 };
    const unsigned char goodObjDer[] = {
        0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01
    };
    WOLFSSL_ASN1_ITEM emptyTemplate;

    XMEMSET(&emptyTemplate, 0, sizeof(WOLFSSL_ASN1_ITEM));

    ExpectNotNull(bootstrap = DPP_BOOTSTRAPPING_KEY_new());

    der = NULL;
    ExpectIntEQ(i2d_DPP_BOOTSTRAPPING_KEY(NULL, &der), -1);
    ExpectIntEQ(wolfSSL_ASN1_item_i2d(bootstrap, &der, NULL), -1);
    ExpectIntEQ(i2d_DPP_BOOTSTRAPPING_KEY(bootstrap, &der), -1);

    ExpectNotNull(key = d2i_PrivateKey(EVP_PKEY_EC, NULL, &in,
                                       (long)sizeof_ecc_clikey_der_256));
    ExpectNotNull(eckey = EVP_PKEY_get1_EC_KEY(key));
    ExpectNotNull(group = EC_KEY_get0_group(eckey));
    ExpectNotNull(point = EC_KEY_get0_public_key(eckey));
    nid = EC_GROUP_get_curve_name(group);

    ec_obj = OBJ_nid2obj(EVP_PKEY_EC);
    group_obj = OBJ_nid2obj(nid);
    if ((ec_obj != NULL) && (group_obj != NULL)) {
        ExpectIntEQ(X509_ALGOR_set0(NULL, ec_obj, V_ASN1_OBJECT,
            group_obj), 0);
        ExpectIntEQ(X509_ALGOR_set0(bootstrap->alg, NULL, V_ASN1_OBJECT,
            NULL), 1);
        ExpectIntEQ(X509_ALGOR_set0(bootstrap->alg, ec_obj, V_ASN1_OBJECT,
            group_obj), 1);
        if (EXPECT_SUCCESS()) {
            ec_obj = NULL;
            group_obj = NULL;
        }
    }
    wolfSSL_ASN1_OBJECT_free(group_obj);
    wolfSSL_ASN1_OBJECT_free(ec_obj);
    ExpectIntEQ(EC_POINT_point2oct(group, point, 0, NULL, 0, NULL), 0);
#ifdef HAVE_COMP_KEY
    ExpectIntGT((len = EC_POINT_point2oct(
                                   group, point, POINT_CONVERSION_COMPRESSED,
                                   NULL, 0, NULL)), 0);
#else
    ExpectIntGT((len = EC_POINT_point2oct(
                                   group, point, POINT_CONVERSION_UNCOMPRESSED,
                                   NULL, 0, NULL)), 0);
#endif
    ExpectNotNull(der = (unsigned char*)XMALLOC(len, NULL, DYNAMIC_TYPE_ASN1));
#ifdef HAVE_COMP_KEY
    ExpectIntEQ(EC_POINT_point2oct(group, point, POINT_CONVERSION_COMPRESSED,
                                   der, len-1, NULL), 0);
    ExpectIntEQ(EC_POINT_point2oct(group, point, POINT_CONVERSION_COMPRESSED,
                                   der, len, NULL), len);
#else
    ExpectIntEQ(EC_POINT_point2oct(group, point, POINT_CONVERSION_UNCOMPRESSED,
                                   der, len-1, NULL), 0);
    ExpectIntEQ(EC_POINT_point2oct(group, point, POINT_CONVERSION_UNCOMPRESSED,
                                   der, len, NULL), len);
#endif
    if (EXPECT_SUCCESS()) {
        bootstrap->pub_key->data = der;
        bootstrap->pub_key->length = (int)len;
        /* Not actually used */
        bootstrap->pub_key->flags &= ~(ASN1_STRING_FLAG_BITS_LEFT | 0x07);
        bootstrap->pub_key->flags |= ASN1_STRING_FLAG_BITS_LEFT;
    }

    ExpectIntEQ(i2d_DPP_BOOTSTRAPPING_KEY(bootstrap, NULL), 16+len);
    der = NULL;
    ExpectIntEQ(i2d_DPP_BOOTSTRAPPING_KEY(bootstrap, &der), 16+len);
    der2 = NULL;
#ifdef WOLFSSL_ASN_TEMPLATE
    tmp = der;
    ExpectNotNull(d2i_DPP_BOOTSTRAPPING_KEY(&bootstrap2, &tmp, 16+len));
    ExpectIntEQ(i2d_DPP_BOOTSTRAPPING_KEY(bootstrap2, &der2), 16+len);
    ExpectBufEQ(der, der2, 49);
#endif

    XFREE(der, NULL, DYNAMIC_TYPE_ASN1);
    XFREE(der2, NULL, DYNAMIC_TYPE_ASN1);
    EVP_PKEY_free(key);
    EC_KEY_free(eckey);
    DPP_BOOTSTRAPPING_KEY_free(bootstrap);
    DPP_BOOTSTRAPPING_KEY_free(bootstrap2);
    bootstrap = NULL;
    DPP_BOOTSTRAPPING_KEY_free(NULL);

    /* Create bootstrap key with bad OBJECT_ID DER data, parameter that is
     * a NULL and an empty BIT_STRING. */
    ExpectNotNull(bootstrap = DPP_BOOTSTRAPPING_KEY_new());
    ExpectNotNull(bootstrap->alg->algorithm = wolfSSL_ASN1_OBJECT_new());
    if (EXPECT_SUCCESS()) {
        bootstrap->alg->algorithm->obj = badObjDer;
        bootstrap->alg->algorithm->objSz = (unsigned int)sizeof(badObjDer);
    }
    ExpectNotNull(bootstrap->alg->parameter = wolfSSL_ASN1_TYPE_new());
    if (EXPECT_SUCCESS()) {
        bootstrap->alg->parameter->type = V_ASN1_NULL;
        bootstrap->alg->parameter->value.ptr = NULL;
        bootstrap->pub_key->data = NULL;
        bootstrap->pub_key->length = 0;
        /* Not actually used */
        bootstrap->pub_key->flags &= ~(ASN1_STRING_FLAG_BITS_LEFT | 0x07);
        bootstrap->pub_key->flags |= ASN1_STRING_FLAG_BITS_LEFT;
    }
    /* Encode with bad OBJECT_ID. */
    der = NULL;
    ExpectIntEQ(i2d_DPP_BOOTSTRAPPING_KEY(bootstrap, &der), -1);

    /* Fix OBJECT_ID and encode with empty BIT_STRING. */
    if (EXPECT_SUCCESS()) {
        bootstrap->alg->algorithm->obj = goodObjDer;
        bootstrap->alg->algorithm->objSz = (unsigned int)sizeof(goodObjDer);
        bootstrap->alg->algorithm->grp = 2;
    }
    der = NULL;
    ExpectIntEQ(i2d_DPP_BOOTSTRAPPING_KEY(bootstrap, &der), 16);
    ExpectIntEQ(wolfSSL_ASN1_item_i2d(bootstrap, &der, &emptyTemplate), -1);
    XFREE(der, NULL, DYNAMIC_TYPE_ASN1);
    DPP_BOOTSTRAPPING_KEY_free(bootstrap);

    /* Test integer */
    ExpectNotNull(test_asn1 = TEST_ASN1_new());
    der = NULL;
    ExpectIntEQ(ASN1_INTEGER_set(test_asn1->integer, 100), 1);
    ExpectIntEQ(i2d_TEST_ASN1(test_asn1, &der), 5);
    tmp = der;
    ExpectNotNull(d2i_TEST_ASN1(&test_asn1_2, &tmp, 5));
    der2 = NULL;
    ExpectIntEQ(i2d_TEST_ASN1(test_asn1_2, &der2), 5);
    ExpectBufEQ(der, der2, 5);
    XFREE(der, NULL, DYNAMIC_TYPE_ASN1);
    XFREE(der2, NULL, DYNAMIC_TYPE_ASN1);
    TEST_ASN1_free(test_asn1);
    TEST_ASN1_free(test_asn1_2);

    /* Test integer cases. */
    ExpectNull(wolfSSL_ASN1_item_new(NULL));
    TEST_ASN1_free(NULL);

    /* Test nested asn1 objects */
    {
        TEST_ASN1_NEST2 *nested_asn1 = NULL;
        TEST_ASN1_NEST2 *nested_asn1_2 = NULL;
        int i;

        ExpectNotNull(nested_asn1 = TEST_ASN1_NEST2_new());
        /* Populate nested_asn1 with some random data */
        /* nested_asn1->num */
        ExpectIntEQ(ASN1_INTEGER_set(nested_asn1->num, 30003), 1);
        /* nested_asn1->key */
        ec_obj = OBJ_nid2obj(EVP_PKEY_EC);
        group_obj = OBJ_nid2obj(NID_secp256k1);
        ExpectIntEQ(X509_ALGOR_set0(nested_asn1->key->alg, ec_obj,
                V_ASN1_OBJECT, group_obj), 1);
        if (EXPECT_SUCCESS()) {
            ec_obj = NULL;
            group_obj = NULL;
        }
        else {
            wolfSSL_ASN1_OBJECT_free(ec_obj);
            wolfSSL_ASN1_OBJECT_free(group_obj);
        }
        ExpectIntEQ(ASN1_BIT_STRING_set_bit(nested_asn1->key->pub_key, 50, 1),
                1);
        /* nested_asn1->asn1_obj->key */
        ec_obj = OBJ_nid2obj(EVP_PKEY_EC);
        group_obj = OBJ_nid2obj(NID_secp256k1);
        ExpectIntEQ(X509_ALGOR_set0(nested_asn1->asn1_obj->key->alg, ec_obj,
                V_ASN1_OBJECT, group_obj), 1);
        if (EXPECT_SUCCESS()) {
            ec_obj = NULL;
            group_obj = NULL;
        }
        else {
            wolfSSL_ASN1_OBJECT_free(ec_obj);
            wolfSSL_ASN1_OBJECT_free(group_obj);
        }
        ExpectIntEQ(ASN1_BIT_STRING_set_bit(nested_asn1->asn1_obj->key->pub_key,
                500, 1), 1);
        /* nested_asn1->asn1_obj->asnNum */
        ExpectIntEQ(ASN1_INTEGER_set(nested_asn1->asn1_obj->asnNum, 666666), 1);
        /* nested_asn1->asn1_obj->expNum */
        ExpectIntEQ(ASN1_INTEGER_set(nested_asn1->asn1_obj->expNum, 22222), 1);
        /* nested_asn1->asn1_obj->strList */
        for (i = 10; i >= 0; i--) {
            ASN1_GENERALSTRING* genStr = NULL;
            char fmtStr[20];

            ExpectIntGT(snprintf(fmtStr, sizeof(fmtStr), "Bonjour #%d", i), 0);
            ExpectNotNull(genStr = ASN1_GENERALSTRING_new());
            ExpectIntEQ(ASN1_GENERALSTRING_set(genStr, fmtStr, -1), 1);
            ExpectIntGT(
                    sk_ASN1_GENERALSTRING_push(nested_asn1->asn1_obj->strList,
                            genStr), 0);
            if (EXPECT_FAIL()) {
                ASN1_GENERALSTRING_free(genStr);
            }
        }
        /* nested_asn1->asn1_obj->str */
        ExpectNotNull(nested_asn1->asn1_obj->str->d.str2
                = ASN1_BIT_STRING_new());
        ExpectIntEQ(ASN1_BIT_STRING_set_bit(nested_asn1->asn1_obj->str->d.str2,
                150, 1), 1);
        if (nested_asn1 != NULL) {
            nested_asn1->asn1_obj->str->type = 2;
        }

        der = NULL;
        ExpectIntEQ(i2d_TEST_ASN1_NEST2(nested_asn1, &der), 285);
#ifdef WOLFSSL_ASN_TEMPLATE
        tmp = der;
        ExpectNotNull(d2i_TEST_ASN1_NEST2(&nested_asn1_2, &tmp, 285));
        der2 = NULL;
        ExpectIntEQ(i2d_TEST_ASN1_NEST2(nested_asn1_2, &der2), 285);
        ExpectBufEQ(der, der2, 285);
        XFREE(der2, NULL, DYNAMIC_TYPE_ASN1);
#endif
        XFREE(der, NULL, DYNAMIC_TYPE_ASN1);

        TEST_ASN1_NEST2_free(nested_asn1);
        TEST_ASN1_NEST2_free(nested_asn1_2);
    }

    /* Test ASN1_ITEM_TEMPLATE */
    {
        TEST_ASN1_ITEM* asn1_item = NULL;
        TEST_ASN1_ITEM* asn1_item2 = NULL;
        int i;

        ExpectNotNull(asn1_item = TEST_ASN1_ITEM_new());
        for (i = 0; i < 11; i++) {
            ASN1_INTEGER* asn1_num = NULL;

            ExpectNotNull(asn1_num = ASN1_INTEGER_new());
            ExpectIntEQ(ASN1_INTEGER_set(asn1_num, i), 1);
            ExpectIntGT(wolfSSL_sk_insert(asn1_item, asn1_num, -1), 0);
            if (EXPECT_FAIL()) {
                ASN1_INTEGER_free(asn1_num);
            }
        }

        der = NULL;
        ExpectIntEQ(i2d_TEST_ASN1_ITEM(asn1_item, &der), 35);
        tmp = der;
        ExpectNotNull(d2i_TEST_ASN1_ITEM(&asn1_item2, &tmp, 35));
        der2 = NULL;
        ExpectIntEQ(i2d_TEST_ASN1_ITEM(asn1_item2, &der2), 35);
        ExpectBufEQ(der, der2, 35);
        XFREE(der, NULL, DYNAMIC_TYPE_ASN1);
        XFREE(der2, NULL, DYNAMIC_TYPE_ASN1);

        TEST_ASN1_ITEM_free(asn1_item);
        TEST_ASN1_ITEM_free(asn1_item2);
    }

#endif /* !HAVE_FIPS || HAVE_FIPS_VERSION > 2 */
#endif /* OPENSSL_ALL && HAVE_ECC && USE_CERT_BUFFERS_256 */
    return EXPECT_RESULT();
}

static int test_wolfSSL_i2d_ASN1_TYPE(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA)
    /* Taken from one of sssd's certs othernames */
    unsigned char str_bin[] = {
        0x04, 0x10, 0xa4, 0x9b, 0xc8, 0xf4, 0x85, 0x8e, 0x89, 0x4d, 0x85, 0x8d,
        0x27, 0xbd, 0x63, 0xaa, 0x93, 0x93
    };
    ASN1_TYPE* asn1type = NULL;
    unsigned char* der = NULL;

    /* Create ASN1_TYPE manually as we don't have a d2i version yet */
    {
        ASN1_STRING* str = NULL;
        ExpectNotNull(str = ASN1_STRING_type_new(V_ASN1_SEQUENCE));
        ExpectIntEQ(ASN1_STRING_set(str, str_bin, sizeof(str_bin)), 1);
        ExpectNotNull(asn1type = ASN1_TYPE_new());
        if (asn1type != NULL) {
            ASN1_TYPE_set(asn1type, V_ASN1_SEQUENCE, str);
        }
        else {
            ASN1_STRING_free(str);
        }
    }

    ExpectIntEQ(i2d_ASN1_TYPE(asn1type, NULL), sizeof(str_bin));
    ExpectIntEQ(i2d_ASN1_TYPE(asn1type, &der), sizeof(str_bin));
    ExpectBufEQ(der, str_bin, sizeof(str_bin));

    ASN1_TYPE_free(asn1type);
    XFREE(der, NULL, DYNAMIC_TYPE_ASN1);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_i2d_ASN1_SEQUENCE(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA)
    /* Taken from one of sssd's certs othernames */
    unsigned char str_bin[] = {
      0x04, 0x10, 0xa4, 0x9b, 0xc8, 0xf4, 0x85, 0x8e, 0x89, 0x4d, 0x85, 0x8d,
      0x27, 0xbd, 0x63, 0xaa, 0x93, 0x93
    };
    ASN1_STRING* str = NULL;
    unsigned char* der = NULL;

    ExpectNotNull(str = ASN1_STRING_type_new(V_ASN1_SEQUENCE));
    ExpectIntEQ(ASN1_STRING_set(str, str_bin, sizeof(str_bin)), 1);
    ExpectIntEQ(i2d_ASN1_SEQUENCE(str, NULL), sizeof(str_bin));
    ExpectIntEQ(i2d_ASN1_SEQUENCE(str, &der), sizeof(str_bin));

    ASN1_STRING_free(str);
    XFREE(der, NULL, DYNAMIC_TYPE_ASN1);
#endif
    return EXPECT_RESULT();
}

static int test_ASN1_strings(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA)
    char text[] = "\0\0test string";
    unsigned char* der = NULL;
    ASN1_STRING* str = NULL;

    /* Set the length byte */
    text[1] = XSTRLEN(text + 2);

    /* GENERALSTRING */
    {
        const unsigned char* p = (const unsigned char*)text;
        text[0] = ASN_GENERALSTRING;
        ExpectNotNull(d2i_ASN1_GENERALSTRING(&str, &p, sizeof(text)));
        ExpectIntEQ(i2d_ASN1_GENERALSTRING(str, &der), 13);
        ASN1_STRING_free(str);
        str = NULL;
        XFREE(der, NULL, DYNAMIC_TYPE_ASN1);
        der = NULL;
    }

    /* OCTET_STRING */
    {
        const unsigned char* p = (const unsigned char*)text;
        text[0] = ASN_OCTET_STRING;
        ExpectNotNull(d2i_ASN1_OCTET_STRING(&str, &p, sizeof(text)));
        ExpectIntEQ(i2d_ASN1_OCTET_STRING(str, &der), 13);
        ASN1_STRING_free(str);
        str = NULL;
        XFREE(der, NULL, DYNAMIC_TYPE_ASN1);
        der = NULL;
    }

    /* UTF8STRING */
    {
        const unsigned char* p = (const unsigned char*)text;
        text[0] = ASN_UTF8STRING;
        ExpectNotNull(d2i_ASN1_UTF8STRING(&str, &p, sizeof(text)));
        ExpectIntEQ(i2d_ASN1_UTF8STRING(str, &der), 13);
        ASN1_STRING_free(str);
        str = NULL;
        XFREE(der, NULL, DYNAMIC_TYPE_ASN1);
        der = NULL;
    }

#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_lhash(void)
{
    EXPECT_DECLS;
#ifdef OPENSSL_ALL
    const char testStr[] = "Like a true nature's child\n"
                           "We were born\n"
                           "Born to be wild";

#ifdef NO_SHA
    ExpectIntEQ(lh_strhash(testStr), 0xf9dc8a43);
#else
    ExpectIntEQ(lh_strhash(testStr), 0x5b7541dc);
#endif
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_NAME(void)
{
    EXPECT_DECLS;
#if (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)) && \
    !defined(NO_CERTS) && !defined(NO_FILESYSTEM) && \
    !defined(NO_RSA) && defined(WOLFSSL_CERT_GEN) && \
    (defined(WOLFSSL_CERT_REQ) || defined(WOLFSSL_CERT_EXT) || \
     defined(OPENSSL_EXTRA))
    X509* x509 = NULL;
#ifndef OPENSSL_EXTRA
    const unsigned char* c = NULL;
    int bytes = 0;
#endif
    unsigned char buf[4096];
    XFILE f = XBADFILE;
    const X509_NAME* a = NULL;
    const X509_NAME* b = NULL;
    X509_NAME* d2i_name = NULL;
    int sz = 0;
    unsigned char* tmp = NULL;
    char file[] = "./certs/ca-cert.der";
#ifndef OPENSSL_EXTRA_X509_SMALL
    byte empty[] = { /* CN=empty emailAddress= */
        0x30, 0x21, 0x31, 0x0E, 0x30, 0x0C, 0x06, 0x03,
        0x55, 0x04, 0x03, 0x0C, 0x05, 0x65, 0x6D, 0x70,
        0x74, 0x79, 0x31, 0x0F, 0x30, 0x0D, 0x06, 0x09,
        0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09,
        0x01, 0x16, 0x00
    };
#endif
#if defined(OPENSSL_EXTRA) && !defined(NO_PWDBASED)
    byte   digest[64]; /* max digest size */
    word32 digestSz;
#endif

#ifndef OPENSSL_EXTRA_X509_SMALL
    /* test compile of deprecated function, returns 0 */
    ExpectIntEQ(CRYPTO_thread_id(), 0);
#endif

    ExpectNotNull(a = X509_NAME_new());
    ExpectNotNull(b = X509_NAME_new());
#ifndef OPENSSL_EXTRA_X509_SMALL
    ExpectIntEQ(X509_NAME_cmp(a, b), 0);
#endif
    X509_NAME_free((X509_NAME*)b);
    X509_NAME_free((X509_NAME*)a);
    a = NULL;

    ExpectTrue((f = XFOPEN(file, "rb")) != XBADFILE);
#ifndef OPENSSL_EXTRA
    ExpectIntGT(bytes = (int)XFREAD(buf, 1, sizeof(buf), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);

    c = buf;
    ExpectNotNull(x509 = wolfSSL_X509_d2i_ex(NULL, c, bytes, HEAP_HINT));
#else
    ExpectNull(wolfSSL_X509_d2i_fp(NULL, XBADFILE));
    ExpectNotNull(wolfSSL_X509_d2i_fp(&x509, f));
    if (f != XBADFILE)
        XFCLOSE(f);
#endif

    /* test cmp function */
    ExpectNull(X509_get_issuer_name(NULL));
    ExpectNotNull(a = X509_get_issuer_name(x509));
    ExpectNull(X509_get_subject_name(NULL));
    ExpectNotNull(b = X509_get_subject_name(x509));
#ifdef KEEP_PEER_CERT
    ExpectNull(wolfSSL_X509_get_subjectCN(NULL));
    ExpectNotNull(wolfSSL_X509_get_subjectCN(x509));
#endif

#if defined(OPENSSL_EXTRA)
    ExpectIntEQ(X509_check_issued(NULL, NULL),
        WOLFSSL_X509_V_ERR_SUBJECT_ISSUER_MISMATCH);
    ExpectIntEQ(X509_check_issued(x509, NULL),
        WOLFSSL_X509_V_ERR_SUBJECT_ISSUER_MISMATCH);
    ExpectIntEQ(X509_check_issued(NULL, x509),
        WOLFSSL_X509_V_ERR_SUBJECT_ISSUER_MISMATCH);
    ExpectIntEQ(X509_check_issued(x509, x509), WOLFSSL_X509_V_OK);

    ExpectIntEQ(X509_NAME_cmp(NULL, NULL), -2);
    ExpectIntEQ(X509_NAME_cmp(NULL, b), -2);
    ExpectIntEQ(X509_NAME_cmp(a, NULL), -2);
    ExpectIntEQ(X509_NAME_cmp(a, b), 0); /* self signed should be 0 */

#if !defined(NO_PWDBASED)
    ExpectIntEQ(wolfSSL_X509_NAME_digest(NULL, NULL, NULL, NULL), 0);
    ExpectIntEQ(wolfSSL_X509_NAME_digest(a, NULL, NULL, NULL), 0);
#ifndef NO_SHA256
    ExpectIntEQ(wolfSSL_X509_NAME_digest(NULL, wolfSSL_EVP_sha256(), NULL,
        NULL), 0);
#endif
    ExpectIntEQ(wolfSSL_X509_NAME_digest(NULL, NULL, digest, NULL), 0);
    ExpectIntEQ(wolfSSL_X509_NAME_digest(NULL, NULL, NULL, &digestSz), 0);
    ExpectIntEQ(wolfSSL_X509_NAME_digest(a, NULL, digest,
        &digestSz), 0);
#ifndef NO_SHA256
    ExpectIntEQ(wolfSSL_X509_NAME_digest(NULL, wolfSSL_EVP_sha256(), digest,
        &digestSz), 0);
    ExpectIntEQ(wolfSSL_X509_NAME_digest(a, wolfSSL_EVP_sha256(), NULL,
        &digestSz), 0);
    ExpectIntEQ(wolfSSL_X509_NAME_digest(a, wolfSSL_EVP_sha256(), digest,
        NULL), 1);
    ExpectIntEQ(wolfSSL_X509_NAME_digest(a, wolfSSL_EVP_sha256(), digest,
        &digestSz), 1);
    ExpectTrue(digestSz == 32);
#endif
#else
    ExpectIntEQ(wolfSSL_X509_NAME_digest(NULL, NULL, NULL, NULL),
        NOT_COMPILED_IN);
#endif
#endif /* OPENSSL_EXTRA */

    tmp = buf;
    ExpectIntGT((sz = i2d_X509_NAME((X509_NAME*)a, &tmp)), 0);
    if (sz > 0 && tmp == buf) {
        fprintf(stderr, "\nERROR - %s line %d failed with:", __FILE__,
            __LINE__);
        fprintf(stderr, " Expected pointer to be incremented\n");
        abort();
    }

#ifndef OPENSSL_EXTRA_X509_SMALL
    tmp = buf;
    ExpectNotNull(d2i_name = d2i_X509_NAME(NULL, &tmp, sz));
#endif

    /* if output parameter is NULL, should still return required size. */
    ExpectIntGT((sz = i2d_X509_NAME((X509_NAME*)b, NULL)), 0);
    /* retry but with the function creating a buffer */
    tmp = NULL;
    ExpectIntGT((sz = i2d_X509_NAME((X509_NAME*)b, &tmp)), 0);
    XFREE(tmp, NULL, DYNAMIC_TYPE_OPENSSL);
    tmp = NULL;

#ifdef WOLFSSL_CERT_NAME_ALL
    /* test for givenName and name */
    {
        WOLFSSL_X509_NAME_ENTRY* entry = NULL;
        WOLFSSL_X509_NAME_ENTRY empty;
        const byte gName[] = "test-given-name";
        const byte name[] = "test-name";

        XMEMSET(&empty, 0, sizeof(empty));

        ExpectNull(wolfSSL_X509_NAME_ENTRY_create_by_NID(NULL,
            NID_givenName, ASN_UTF8STRING, NULL, sizeof(gName)));
        ExpectNotNull(entry = wolfSSL_X509_NAME_ENTRY_create_by_NID(NULL,
            NID_givenName, ASN_UTF8STRING, gName, sizeof(gName)));
        ExpectNotNull(wolfSSL_X509_NAME_ENTRY_create_by_NID(&entry,
            NID_givenName, ASN_UTF8STRING, gName, sizeof(gName)));
        ExpectIntEQ(wolfSSL_X509_NAME_add_entry(NULL         , NULL  , -1, 0),
            0);
        ExpectIntEQ(wolfSSL_X509_NAME_add_entry((X509_NAME*)b, NULL  , -1, 0),
            0);
        ExpectIntEQ(wolfSSL_X509_NAME_add_entry(NULL         , entry , -1, 0),
            0);
        ExpectIntEQ(wolfSSL_X509_NAME_add_entry((X509_NAME*)b, &empty, -1, 0),
            0);
        ExpectIntEQ(wolfSSL_X509_NAME_add_entry((X509_NAME*)b, entry , 99, 0),
            0);
        ExpectIntEQ(wolfSSL_X509_NAME_add_entry((X509_NAME*)b, entry , -1, 0),
            1);
        wolfSSL_X509_NAME_ENTRY_free(entry);
        entry = NULL;

        ExpectNotNull(wolfSSL_X509_NAME_ENTRY_create_by_NID(&entry,
            NID_name, ASN_UTF8STRING, name, sizeof(name)));
        ExpectIntEQ(wolfSSL_X509_NAME_add_entry((X509_NAME*)b, entry, -1, 0),
            1);
        wolfSSL_X509_NAME_ENTRY_free(entry);

        tmp = NULL;
        ExpectIntGT((sz = i2d_X509_NAME((X509_NAME*)b, &tmp)), 0);
        XFREE(tmp, NULL, DYNAMIC_TYPE_OPENSSL);
    }
#endif

    b = NULL;
    ExpectNull(X509_NAME_dup(NULL));
    ExpectNotNull(b = X509_NAME_dup((X509_NAME*)a));
#ifndef OPENSSL_EXTRA_X509_SMALL
    ExpectIntEQ(X509_NAME_cmp(a, b), 0);
#endif
    ExpectIntEQ(X509_NAME_entry_count(NULL), 0);
    ExpectIntEQ(X509_NAME_entry_count((X509_NAME*)b), 7);
    X509_NAME_free((X509_NAME*)b);
    ExpectNotNull(b = wolfSSL_X509_NAME_new());
    ExpectIntEQ(X509_NAME_entry_count((X509_NAME*)b), 0);
    ExpectIntEQ(wolfSSL_X509_NAME_copy(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_X509_NAME_copy((X509_NAME*)a, NULL), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_X509_NAME_copy(NULL, (X509_NAME*)b), BAD_FUNC_ARG);
    ExpectIntEQ(wolfSSL_X509_NAME_copy((X509_NAME*)a, (X509_NAME*)b), 1);
    ExpectIntEQ(X509_NAME_entry_count((X509_NAME*)b), 7);
    X509_NAME_free((X509_NAME*)b);
    X509_NAME_free(d2i_name);
    d2i_name = NULL;
    X509_free(x509);

#ifndef OPENSSL_EXTRA_X509_SMALL
    /* test with an empty domain component */
    tmp = empty;
    sz  = sizeof(empty);
    ExpectNotNull(d2i_name = d2i_X509_NAME(NULL, &tmp, sz));
    ExpectIntEQ(X509_NAME_entry_count(d2i_name), 2);

    /* size of empty emailAddress will be 0 */
    tmp = buf;
    ExpectIntEQ(X509_NAME_get_text_by_NID(d2i_name, NID_emailAddress,
                (char*)tmp, sizeof(buf)), 0);

    /* should contain no organization name */
    tmp = buf;
    ExpectIntEQ(X509_NAME_get_text_by_NID(d2i_name, NID_organizationName,
                (char*)tmp, sizeof(buf)), -1);
    X509_NAME_free(d2i_name);
#endif
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_NAME_hash(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_FILESYSTEM) && \
    !defined(NO_RSA) && !defined(NO_SHA) && !defined(NO_BIO)
    BIO* bio = NULL;
    X509* x509 = NULL;
    X509_NAME* name = NULL;

    ExpectIntEQ(X509_NAME_hash(NULL), 0);
    ExpectNotNull(name = wolfSSL_X509_NAME_new_ex(NULL));
    ExpectIntEQ(X509_NAME_hash(name), 0);
    X509_NAME_free(name);

    ExpectNotNull(bio = BIO_new(BIO_s_file()));
    ExpectIntGT(BIO_read_filename(bio, svrCertFile), 0);
    ExpectNotNull(PEM_read_bio_X509(bio, &x509, NULL, NULL));
    ExpectIntEQ(X509_NAME_hash(X509_get_subject_name(x509)), 0x137DC03F);
    ExpectIntEQ(X509_NAME_hash(X509_get_issuer_name(x509)), 0xFDB2DA4);
    X509_free(x509);
    BIO_free(bio);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_NAME_print_ex(void)
{
    EXPECT_DECLS;
#if (defined(OPENSSL_ALL) || (defined(OPENSSL_EXTRA) && \
     (defined(HAVE_STUNNEL) || defined(WOLFSSL_NGINX) || \
     defined(HAVE_LIGHTY) || defined(WOLFSSL_HAPROXY) || \
     defined(WOLFSSL_OPENSSH) || defined(HAVE_SBLIM_SFCB)))) && \
    !defined(NO_BIO) && !defined(NO_RSA)
    int memSz = 0;
    byte* mem = NULL;
    BIO* bio = NULL;
    BIO* membio = NULL;
    X509* x509 = NULL;
    X509_NAME* name = NULL;
    X509_NAME* empty = NULL;

    const char* expNormal  = "C=US, CN=wolfssl.com";
    const char* expEqSpace = "C = US, CN = wolfssl.com";
    const char* expReverse = "CN=wolfssl.com, C=US";

    const char* expNotEscaped = "C= US,+\"\\ , CN=#wolfssl.com<>;";
    const char* expNotEscapedRev = "CN=#wolfssl.com<>;, C= US,+\"\\ ";
    const char* expRFC5523 =
        "CN=\\#wolfssl.com\\<\\>\\;, C=\\ US\\,\\+\\\"\\\\\\ ";

    /* Test with real cert (svrCertFile) first */
    ExpectNotNull(bio = BIO_new(BIO_s_file()));
    ExpectIntGT(BIO_read_filename(bio, svrCertFile), 0);
    ExpectNotNull(PEM_read_bio_X509(bio, &x509, NULL, NULL));
    ExpectNotNull(name = X509_get_subject_name(x509));

    /* Test without flags */
    ExpectNotNull(membio = BIO_new(BIO_s_mem()));
    ExpectNotNull(empty = wolfSSL_X509_NAME_new());
    ExpectIntEQ(X509_NAME_print_ex(NULL, NULL, 0, 0), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_NAME_print_ex(membio, NULL, 0, 0), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_NAME_print_ex(NULL, name, 0, 0), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_NAME_print_ex(membio, empty, 0, 0), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_NAME_print_ex(membio, name, 0, 0), WOLFSSL_SUCCESS);
    wolfSSL_X509_NAME_free(empty);
    BIO_free(membio);
    membio = NULL;

    /* Test flag: XN_FLAG_RFC2253 */
    ExpectNotNull(membio = BIO_new(BIO_s_mem()));
    ExpectIntEQ(X509_NAME_print_ex(membio, name, 0,
                XN_FLAG_RFC2253), WOLFSSL_SUCCESS);
    BIO_free(membio);
    membio = NULL;

    /* Test flag: XN_FLAG_RFC2253 | XN_FLAG_DN_REV */
    ExpectNotNull(membio = BIO_new(BIO_s_mem()));
    ExpectIntEQ(X509_NAME_print_ex(membio, name, 0,
                XN_FLAG_RFC2253 | XN_FLAG_DN_REV), WOLFSSL_SUCCESS);
    BIO_free(membio);
    membio = NULL;

    X509_free(x509);
    BIO_free(bio);
    name = NULL;

    /* Test normal case without escaped characters */
    {
        /* Create name: "/C=US/CN=wolfssl.com" */
        ExpectNotNull(name = X509_NAME_new());
        ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName",
                    MBSTRING_UTF8, (byte*)"US", 2, -1, 0),
                    WOLFSSL_SUCCESS);
        ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName",
                    MBSTRING_UTF8, (byte*)"wolfssl.com", 11, -1, 0),
                    WOLFSSL_SUCCESS);

        /* Test without flags */
        ExpectNotNull(membio = BIO_new(BIO_s_mem()));
        ExpectIntEQ(X509_NAME_print_ex(membio, name, 0, 0), WOLFSSL_SUCCESS);
        ExpectIntGE((memSz = BIO_get_mem_data(membio, &mem)), 0);
        ExpectIntEQ(memSz, XSTRLEN(expNormal));
        ExpectIntEQ(XSTRNCMP((char*)mem, expNormal, XSTRLEN(expNormal)), 0);
        BIO_free(membio);
        membio = NULL;

        /* Test with XN_FLAG_ONELINE which should enable XN_FLAG_SPC_EQ for
           spaces around '=' */
        ExpectNotNull(membio = BIO_new(BIO_s_mem()));
        ExpectIntEQ(X509_NAME_print_ex(membio, name, 0, XN_FLAG_ONELINE),
            WOLFSSL_SUCCESS);
        ExpectIntGE((memSz = BIO_get_mem_data(membio, &mem)), 0);
        ExpectIntEQ(memSz, XSTRLEN(expEqSpace));
        ExpectIntEQ(XSTRNCMP((char*)mem, expEqSpace, XSTRLEN(expEqSpace)), 0);
        BIO_free(membio);
        membio = NULL;

        /* Test flags: XN_FLAG_RFC2253 - should be reversed */
        ExpectNotNull(membio = BIO_new(BIO_s_mem()));
        ExpectIntEQ(X509_NAME_print_ex(membio, name, 0,
                    XN_FLAG_RFC2253), WOLFSSL_SUCCESS);
        ExpectIntGE((memSz = BIO_get_mem_data(membio, &mem)), 0);
        ExpectIntEQ(memSz, XSTRLEN(expReverse));
        BIO_free(membio);
        membio = NULL;

        /* Test flags: XN_FLAG_DN_REV - reversed */
        ExpectNotNull(membio = BIO_new(BIO_s_mem()));
        ExpectIntEQ(X509_NAME_print_ex(membio, name, 0,
                    XN_FLAG_DN_REV), WOLFSSL_SUCCESS);
        ExpectIntGE((memSz = BIO_get_mem_data(membio, &mem)), 0);
        ExpectIntEQ(memSz, XSTRLEN(expReverse));
        ExpectIntEQ(XSTRNCMP((char*)mem, expReverse, XSTRLEN(expReverse)), 0);
        BIO_free(membio);
        membio = NULL;

        X509_NAME_free(name);
        name = NULL;
    }

    /* Test RFC2253 characters are escaped with backslashes */
    {
        ExpectNotNull(name = X509_NAME_new());
        ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "countryName",
                    /* space at beginning and end, and: ,+"\ */
                    MBSTRING_UTF8, (byte*)" US,+\"\\ ", 8, -1, 0),
                    WOLFSSL_SUCCESS);
        ExpectIntEQ(X509_NAME_add_entry_by_txt(name, "commonName",
                    /* # at beginning, and: <>;*/
                    MBSTRING_UTF8, (byte*)"#wolfssl.com<>;", 15, -1, 0),
                    WOLFSSL_SUCCESS);

        /* Test without flags */
        ExpectNotNull(membio = BIO_new(BIO_s_mem()));
        ExpectIntEQ(X509_NAME_print_ex(membio, name, 0, 0), WOLFSSL_SUCCESS);
        ExpectIntGE((memSz = BIO_get_mem_data(membio, &mem)), 0);
        ExpectIntEQ(memSz, XSTRLEN(expNotEscaped));
        ExpectIntEQ(XSTRNCMP((char*)mem, expNotEscaped,
                    XSTRLEN(expNotEscaped)), 0);
        BIO_free(membio);
        membio = NULL;

        /* Test flags: XN_FLAG_RFC5523 - should be reversed and escaped */
        ExpectNotNull(membio = BIO_new(BIO_s_mem()));
        ExpectIntEQ(X509_NAME_print_ex(membio, name, 0,
                    XN_FLAG_RFC2253), WOLFSSL_SUCCESS);
        ExpectIntGE((memSz = BIO_get_mem_data(membio, &mem)), 0);
        ExpectIntEQ(memSz, XSTRLEN(expRFC5523));
        ExpectIntEQ(XSTRNCMP((char*)mem, expRFC5523, XSTRLEN(expRFC5523)), 0);
        BIO_free(membio);
        membio = NULL;

        /* Test flags: XN_FLAG_DN_REV - reversed but not escaped */
        ExpectNotNull(membio = BIO_new(BIO_s_mem()));
        ExpectIntEQ(X509_NAME_print_ex(membio, name, 0,
                    XN_FLAG_DN_REV), WOLFSSL_SUCCESS);
        ExpectIntGE((memSz = BIO_get_mem_data(membio, &mem)), 0);
        ExpectIntEQ(memSz, XSTRLEN(expNotEscapedRev));
        ExpectIntEQ(XSTRNCMP((char*)mem, expNotEscapedRev,
                    XSTRLEN(expNotEscapedRev)), 0);
        BIO_free(membio);

        X509_NAME_free(name);
    }
#endif
    return EXPECT_RESULT();
}

#ifndef NO_BIO
static int test_wolfSSL_X509_INFO_multiple_info(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(NO_RSA)
    STACK_OF(X509_INFO) *info_stack = NULL;
    X509_INFO *info = NULL;
    int len;
    int i;
    const char* files[] = {
        cliCertFile,
        cliKeyFile,
        /* This needs to be the order as svrCertFile contains the
         * intermediate cert as well. */
        svrKeyFile,
        svrCertFile,
        NULL,
    };
    const char** curFile;
    BIO *fileBIO = NULL;
    BIO *concatBIO = NULL;
    byte tmp[FOURK_BUF];

    /* concatenate the cert and the key file to force PEM_X509_INFO_read_bio
     * to group objects together. */
    ExpectNotNull(concatBIO = BIO_new(BIO_s_mem()));
    for (curFile = files; EXPECT_SUCCESS() && *curFile != NULL; curFile++) {
        int fileLen = 0;
        ExpectNotNull(fileBIO = BIO_new_file(*curFile, "rb"));
        ExpectIntGT(fileLen = wolfSSL_BIO_get_len(fileBIO), 0);
        if (EXPECT_SUCCESS()) {
            while ((len = BIO_read(fileBIO, tmp, sizeof(tmp))) > 0) {
                ExpectIntEQ(BIO_write(concatBIO, tmp, len), len);
                fileLen -= len;
                if (EXPECT_FAIL())
                    break;
            }
            /* Make sure we read the entire file */
            ExpectIntEQ(fileLen, 0);
        }
        BIO_free(fileBIO);
        fileBIO = NULL;
    }

    ExpectNotNull(info_stack = PEM_X509_INFO_read_bio(concatBIO, NULL, NULL,
        NULL));
    ExpectIntEQ(sk_X509_INFO_num(info_stack), 3);
    for (i = 0; i < sk_X509_INFO_num(info_stack); i++) {
        ExpectNotNull(info = sk_X509_INFO_value(info_stack, i));
        ExpectNotNull(info->x509);
        ExpectNull(info->crl);
        if (i != 2) {
            ExpectNotNull(info->x_pkey);
            ExpectIntEQ(X509_check_private_key(info->x509,
                                               info->x_pkey->dec_pkey), 1);
        }
        else {
            ExpectNull(info->x_pkey);
        }
    }

    sk_X509_INFO_pop_free(info_stack, X509_INFO_free);
    BIO_free(concatBIO);
#endif
    return EXPECT_RESULT();
}
#endif

#ifndef NO_BIO
static int test_wolfSSL_X509_INFO(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(NO_RSA)
    STACK_OF(X509_INFO) *info_stack = NULL;
    X509_INFO *info = NULL;
    BIO *cert = NULL;
    int i;
    /* PEM in hex format to avoid null terminator */
    byte data[] = {
        0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47,
        0x49, 0x4e, 0x20, 0x43, 0x45, 0x52, 0x54, 0x63, 0x2d, 0x2d, 0x2d, 0x2d,
        0x2d, 0x0a, 0x4d, 0x49, 0x49, 0x44, 0x4d, 0x54, 0x42, 0x75, 0x51, 0x3d,
        0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20, 0x2d, 0x2d,
        0x2d, 0x2d, 0x2d
    };
    /* PEM in hex format to avoid null terminator */
    byte data2[] = {
        0x41, 0x53, 0x4e, 0x31, 0x20, 0x4f, 0x49, 0x44, 0x3a, 0x20, 0x70, 0x72,
        0x69, 0x6d, 0x65, 0x32, 0x35, 0x36, 0x76, 0x31, 0x0a, 0x2d, 0x2d, 0x2d,
        0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x45, 0x43, 0x20, 0x50,
        0x41, 0x52, 0x41, 0x4d, 0x45, 0x54, 0x45, 0x52, 0x53, 0x2d, 0x2d, 0x2d,
        0x2d, 0x43, 0x65, 0x72, 0x74, 0x69, 0x2d, 0x0a, 0x42, 0x67, 0x67, 0x71,
        0x68, 0x6b, 0x6a, 0x4f, 0x50, 0x51, 0x4d, 0x42, 0x42, 0x77, 0x3d, 0x3d,
        0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d
    };

    ExpectNotNull(cert = BIO_new_file(cliCertFileExt, "rb"));
    ExpectNotNull(info_stack = PEM_X509_INFO_read_bio(cert, NULL, NULL, NULL));
    for (i = 0; i < sk_X509_INFO_num(info_stack); i++) {
        ExpectNotNull(info = sk_X509_INFO_value(info_stack, i));
        ExpectNotNull(info->x509);
        ExpectNull(info->crl);
        ExpectNull(info->x_pkey);
    }
    sk_X509_INFO_pop_free(info_stack, X509_INFO_free);
    info_stack = NULL;
    BIO_free(cert);
    cert = NULL;

    ExpectNotNull(cert = BIO_new_file(cliCertFileExt, "rb"));
    ExpectNotNull(info_stack = PEM_X509_INFO_read_bio(cert, NULL, NULL, NULL));
    sk_X509_INFO_pop_free(info_stack, X509_INFO_free);
    info_stack = NULL;
    BIO_free(cert);
    cert = NULL;

    /* This case should fail due to invalid input. */
    ExpectNotNull(cert = BIO_new(BIO_s_mem()));
    ExpectIntEQ(BIO_write(cert, data, sizeof(data)), sizeof(data));
    ExpectNull(info_stack = PEM_X509_INFO_read_bio(cert, NULL, NULL, NULL));
    sk_X509_INFO_pop_free(info_stack, X509_INFO_free);
    info_stack = NULL;
    BIO_free(cert);
    cert = NULL;
    ExpectNotNull(cert = BIO_new(BIO_s_mem()));
    ExpectIntEQ(BIO_write(cert, data2, sizeof(data2)), sizeof(data2));
    ExpectNull(info_stack = PEM_X509_INFO_read_bio(cert, NULL, NULL, NULL));
    sk_X509_INFO_pop_free(info_stack, X509_INFO_free);
    BIO_free(cert);
#endif
    return EXPECT_RESULT();
}
#endif

static int test_wolfSSL_X509_subject_name_hash(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_FILESYSTEM) && \
    !defined(NO_RSA) && (!defined(NO_SHA) || !defined(NO_SHA256))
    X509* x509 = NULL;
    X509_NAME* subjectName = NULL;
    unsigned long ret1 = 0;
    unsigned long ret2 = 0;

    ExpectNotNull(x509 = X509_new());
    ExpectIntEQ(X509_subject_name_hash(NULL), 0);
    ExpectIntEQ(X509_subject_name_hash(x509), 0);
    X509_free(x509);
    x509 = NULL;

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
                SSL_FILETYPE_PEM));
    ExpectNotNull(subjectName = wolfSSL_X509_get_subject_name(x509));

    /* These two
     *   - X509_subject_name_hash(x509)
     *   - X509_NAME_hash(X509_get_subject_name(x509))
     *  should give the same hash, if !defined(NO_SHA) is true. */

    ret1 = X509_subject_name_hash(x509);
    ExpectIntNE(ret1, 0);

#if !defined(NO_SHA)
    ret2 = X509_NAME_hash(X509_get_subject_name(x509));
    ExpectIntNE(ret2, 0);

    ExpectIntEQ(ret1, ret2);
#else
    (void) ret2;
#endif

    X509_free(x509);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_issuer_name_hash(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_FILESYSTEM) \
    && !defined(NO_RSA) && (!defined(NO_SHA) || !defined(NO_SHA256))
    X509* x509 = NULL;
    X509_NAME* issuertName = NULL;
    unsigned long ret1 = 0;
    unsigned long ret2 = 0;

    ExpectNotNull(x509 = X509_new());
    ExpectIntEQ(X509_issuer_name_hash(NULL), 0);
    ExpectIntEQ(X509_issuer_name_hash(x509), 0);
    X509_free(x509);
    x509 = NULL;

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
                SSL_FILETYPE_PEM));
    ExpectNotNull(issuertName = wolfSSL_X509_get_issuer_name(x509));

    /* These two
     *   - X509_issuer_name_hash(x509)
     *   - X509_NAME_hash(X509_get_issuer_name(x509))
     *  should give the same hash, if !defined(NO_SHA) is true. */

    ret1 = X509_issuer_name_hash(x509);
    ExpectIntNE(ret1, 0);

#if !defined(NO_SHA)
    ret2 = X509_NAME_hash(X509_get_issuer_name(x509));
    ExpectIntNE(ret2, 0);

    ExpectIntEQ(ret1, ret2);
#else
    (void) ret2;
#endif

    X509_free(x509);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_check_host(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_FILESYSTEM) \
    && !defined(NO_SHA) && !defined(NO_RSA)
    X509* x509 = NULL;
    const char altName[] = "example.com";
    const char badAltName[] = "a.example.com";

    ExpectIntEQ(X509_check_host(NULL, NULL, XSTRLEN(altName), 0, NULL),
            WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    /* cliCertFile has subjectAltName set to 'example.com', '127.0.0.1' */
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
                SSL_FILETYPE_PEM));

    ExpectIntEQ(X509_check_host(x509, altName, XSTRLEN(altName), 0, NULL),
            WOLFSSL_SUCCESS);

    ExpectIntEQ(X509_check_host(x509, badAltName, XSTRLEN(badAltName), 0, NULL),
            WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    ExpectIntEQ(X509_check_host(x509, NULL, 0, 0, NULL),
            WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    /* Check WOLFSSL_LEFT_MOST_WILDCARD_ONLY flag set */
    ExpectIntEQ(X509_check_host(x509, altName, XSTRLEN(altName),
            WOLFSSL_LEFT_MOST_WILDCARD_ONLY, NULL), WOLFSSL_SUCCESS);

    ExpectIntEQ(X509_check_host(x509, NULL, 0,
            WOLFSSL_LEFT_MOST_WILDCARD_ONLY, NULL),
            WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    ExpectIntEQ(X509_check_host(x509, badAltName, XSTRLEN(badAltName),
            WOLFSSL_LEFT_MOST_WILDCARD_ONLY, NULL),
            WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    ExpectIntEQ(wolfSSL_X509_check_host(x509, altName, XSTRLEN(altName),
        WOLFSSL_NO_WILDCARDS, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_X509_check_host(x509, altName, XSTRLEN(altName),
        WOLFSSL_NO_PARTIAL_WILDCARDS, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(wolfSSL_X509_check_host(x509, altName, XSTRLEN(altName),
        WOLFSSL_MULTI_LABEL_WILDCARDS, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    X509_free(x509);

    ExpectIntEQ(X509_check_host(NULL, altName, XSTRLEN(altName), 0, NULL),
            WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    /* Check again with WOLFSSL_LEFT_MOST_WILDCARD_ONLY flag set */
    ExpectIntEQ(X509_check_host(NULL, altName, XSTRLEN(altName),
            WOLFSSL_LEFT_MOST_WILDCARD_ONLY, NULL),
            WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_check_email(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_CERT_GEN) && !defined(NO_RSA)
    X509* x509 = NULL;
    X509* empty = NULL;
    const char goodEmail[] = "info@wolfssl.com";
    const char badEmail[] = "disinfo@wolfssl.com";

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
                SSL_FILETYPE_PEM));
    ExpectNotNull(empty = wolfSSL_X509_new());

    ExpectIntEQ(wolfSSL_X509_check_email(NULL, NULL, 0, 0), 0);
    ExpectIntEQ(wolfSSL_X509_check_email(x509, NULL, 0, 0), 0);
    ExpectIntEQ(wolfSSL_X509_check_email(NULL, goodEmail, XSTRLEN(goodEmail),
        0), 0);
    ExpectIntEQ(wolfSSL_X509_check_email(empty, goodEmail, XSTRLEN(goodEmail),
        0), 0);

    /* Should fail on non-matching email address */
    ExpectIntEQ(wolfSSL_X509_check_email(x509, badEmail, XSTRLEN(badEmail), 0),
            WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    /* Should succeed on matching email address */
    ExpectIntEQ(wolfSSL_X509_check_email(x509, goodEmail, XSTRLEN(goodEmail), 0),
            WOLFSSL_SUCCESS);
    /* Should compute length internally when not provided */
    ExpectIntEQ(wolfSSL_X509_check_email(x509, goodEmail, 0, 0),
            WOLFSSL_SUCCESS);
    /* Should fail when email address is NULL */
    ExpectIntEQ(wolfSSL_X509_check_email(x509, NULL, 0, 0),
            WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    X509_free(empty);
    X509_free(x509);

    /* Should fail when x509 is NULL */
    ExpectIntEQ(wolfSSL_X509_check_email(NULL, goodEmail, 0, 0),
            WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
#endif /* OPENSSL_EXTRA && WOLFSSL_CERT_GEN */
    return EXPECT_RESULT();
}

static int test_wc_PemToDer(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && defined(WOLFSSL_PEM_TO_DER) && !defined(NO_FILESYSTEM)
    int ret;
    DerBuffer* pDer = NULL;
    const char* ca_cert = "./certs/server-cert.pem";
    const char* trusted_cert = "./certs/test/ossl-trusted-cert.pem";
    byte* cert_buf = NULL;
    size_t cert_sz = 0;
    int eccKey = 0;
    EncryptedInfo info;

    XMEMSET(&info, 0, sizeof(info));

    ExpectIntEQ(ret = load_file(ca_cert, &cert_buf, &cert_sz), 0);
    ExpectIntEQ(ret = wc_PemToDer(cert_buf, (long int)cert_sz, CERT_TYPE, &pDer, NULL,
        &info, &eccKey), 0);
    wc_FreeDer(&pDer);
    pDer = NULL;

    if (cert_buf != NULL) {
        free(cert_buf);
        cert_buf = NULL;
    }

    /* Test that -----BEGIN TRUSTED CERTIFICATE----- banner parses OK */
    ExpectIntEQ(ret = load_file(trusted_cert, &cert_buf, &cert_sz), 0);
    ExpectIntEQ(ret = wc_PemToDer(cert_buf, (long int)cert_sz, TRUSTED_CERT_TYPE, &pDer, NULL,
        &info, &eccKey), 0);
    wc_FreeDer(&pDer);
    pDer = NULL;

    if (cert_buf != NULL) {
        free(cert_buf);
        cert_buf = NULL;
    }

#ifdef HAVE_ECC
    {
        const char* ecc_private_key = "./certs/ecc-privOnlyKey.pem";
        byte key_buf[256] = {0};

        /* Test fail of loading a key with cert type */
        ExpectIntEQ(load_file(ecc_private_key, &cert_buf, &cert_sz), 0);
        key_buf[0] = '\n';
        ExpectNotNull(XMEMCPY(key_buf + 1, cert_buf, cert_sz));
        ExpectIntNE((ret = wc_PemToDer(key_buf, (long int)cert_sz + 1, CERT_TYPE,
            &pDer, NULL, &info, &eccKey)), 0);

    #ifdef OPENSSL_EXTRA
        ExpectIntEQ((ret = wc_PemToDer(key_buf, cert_sz + 1, PRIVATEKEY_TYPE,
            &pDer, NULL, &info, &eccKey)), 0);
    #endif
        wc_FreeDer(&pDer);
        if (cert_buf != NULL)
            free(cert_buf);
    }
#endif
#endif
    return EXPECT_RESULT();
}

static int test_wc_AllocDer(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS)
    DerBuffer* pDer = NULL;
    word32 testSize = 1024;

    ExpectIntEQ(wc_AllocDer(NULL, testSize, CERT_TYPE, HEAP_HINT),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_AllocDer(&pDer, testSize, CERT_TYPE, HEAP_HINT), 0);
    ExpectNotNull(pDer);
    wc_FreeDer(&pDer);
#endif
    return EXPECT_RESULT();
}

static int test_wc_CertPemToDer(void)
{
    EXPECT_DECLS;
#if !defined(NO_CERTS) && defined(WOLFSSL_PEM_TO_DER) && !defined(NO_FILESYSTEM)
    const char* ca_cert = "./certs/ca-cert.pem";
    byte* cert_buf = NULL;
    size_t cert_sz = 0;
    size_t cert_dersz = 0;
    byte* cert_der = NULL;

    ExpectIntEQ(load_file(ca_cert, &cert_buf, &cert_sz), 0);
    cert_dersz = cert_sz; /* DER will be smaller than PEM */
    ExpectNotNull(cert_der = (byte*)malloc(cert_dersz));
    ExpectIntGE(wc_CertPemToDer(cert_buf, (int)cert_sz, cert_der,
        (int)cert_dersz, CERT_TYPE), 0);

    ExpectIntEQ(wc_CertPemToDer(NULL, (int)cert_sz, NULL, -1, CERT_TYPE),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_CertPemToDer(cert_buf, (int)cert_sz, NULL, -1, CERT_TYPE),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_CertPemToDer(NULL, (int)cert_sz, cert_der, -1, CERT_TYPE),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_CertPemToDer(NULL, (int)cert_sz, NULL, (int)cert_dersz,
        CERT_TYPE), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_CertPemToDer(NULL, (int)cert_sz, cert_der,
        (int)cert_dersz, CERT_TYPE), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_CertPemToDer(cert_buf, (int)cert_sz, NULL,
        (int)cert_dersz, CERT_TYPE), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(wc_CertPemToDer(cert_buf, (int)cert_sz, cert_der, -1,
        CERT_TYPE), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    if (cert_der != NULL)
        free(cert_der);
    if (cert_buf != NULL)
        free(cert_buf);
#endif
    return EXPECT_RESULT();
}

static int test_wc_KeyPemToDer(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_PEM_TO_DER) && !defined(NO_FILESYSTEM) && !defined(NO_RSA)
    int ret = 0;
    const byte cert_buf[] = \
    "-----BEGIN PRIVATE KEY-----\n"
    "MIIEvgIBADANBgkqhkiG9w0BAQEFAASCBKgwggSkAgEAAoIBAQDMG5KgWxP002pA\n"
    "QJIdA4H5N0oM1Wf0LrHcos5RYUlrHDkC2b5p2BUpVRPmgDAFD2+8leim98x0BvcB\n"
    "k48TNzrVynuwyVEY664+iQyzEBO5v27HPRydOddprbLCvRO036XINGIjauy1jHFi\n"
    "HaDVx3bexSwgp9aefUGAszFXi4q1J4GacV7Cr2b/wBqUHqWv4ZXPu6R9/UYngTkD\n"
    "UDJL5gLlLfcLzNyyodKPHPCIAKdWn6mSVdcHk8XVpK4y9lgz4E7YDWA6ohKZgWgG\n"
    "2RDha8CMilFMDgYa0G0SiS9g3PQx0qh3AMXJJsKSVhScFCZufAE0kV6KvjP7jAqP\n"
    "XBiSkRGPAgMBAAECggEAW7hmRyY2jRX2UMJThrM9VIs6fRLnYI0dQ0tsEJj536ay\n"
    "nevQjArc05KWW0Yujg+WRDZPcry3RUqd9Djlmhp/F3Si6dpF1b+PMS3wJYVrf9Sd\n"
    "SO5W7faArU4vnyBNe0HnY1Ta5xSVI65lg1RSIs88RTZwsooJwXYDGf0shq0/21CE\n"
    "V8HOb27DDYNcEnm35lzaONjFnMqQQT2Vs9anRrPiSEXNleEvTgLVXZtGTyCGTz6v\n"
    "x86Y8eSWL9YNHvPE1I+mDPuocfSR7eRNgRu7SK3mn94W5mqd7Ns072YKX/2XN1mO\n"
    "66+ZFHO6v4dK1u7cSjuwrU1EhLHpUsgDz6Bna5InyQKBgQDv5l8RPy8UneKSADaf\n"
    "M5L/5675I/5t4nqVjvbnQje00YveLTAEjlJBNR93Biln3sYgnvNamYDCxyEuUZ/I\n"
    "S/vmBL9PoxfGZow4FcsIBOEbIn3E0SYJgCBNWthquUvGpKsYDnThJuhO+1cVmxAJ\n"
    "BUOjLFnJYHM0a+Vmk9GexT2OBwKBgQDZzkUBOK7Im3eiYytFocUJyhqMH30d49X9\n"
    "ujC7kGw4UWAqVe7YCSvlBa8nzWpRWK2kRpu3M0272RU0V4geyWqT+nr/SvRRPtNP\n"
    "F5dY8l3yR7hjtSejqqjOfBcZT6ETJxI4tiG0+Nl5BlfM5M+0nxnkWpRcHuOR3j79\n"
    "YUFERyN+OQKBgQCjlOKeUAc6d65W/+4/AFvsQ378Q57qLtSHxsR1TKHPmlNVXFqx\n"
    "wJo1/JNIBduWCEHxXHF0BdfW+RGXE/FwEt/hKLuLAhrkHmjelX2sKieU6R/5ZOQa\n"
    "9lMQbDHGFDOncAF6leD85hriQGBRSzrT69MDIOrYdfwYcroqCAGX0cb3YQKBgQC8\n"
    "iIFQylj5SyHmjcMSNjKSA8CxFDzAV8yPIdE3Oo+CvGXqn5HsrRuy1hXE9VmXapR8\n"
    "A6ackSszdHiXY0FvrNe1mfdH7wDHJwPQjdIzazCJHS3uGQxj7sDKY7226ie6pXJv\n"
    "ZrCMr2/IBAaSVGm6ppHKCeIsT4ybYm7R85KEYLPHeQKBgBeJOMBinXQfWN/1jT9b\n"
    "6Ywrutvp2zP8hVxQGSZJ0WG4iewZyFLsPUlbWRXOSYNPElHmdD0ZomdLVm+lSpAA\n"
    "XSH5FJ/IFCwqq7Eft6Gf8NFRV+NjPMUny+PnjHe4oFP8YK/Ek22K3ttNG8Hw69Aw\n"
    "AQue5o6oVfhgLiJzMdo/77gw\n"
    "-----END PRIVATE KEY-----\n";
    const int cert_sz = sizeof(cert_buf);
    const char cert_pw[] = "password";
    int cert_dersz = 0;
    byte* cert_der = NULL;

    /* Bad arg: Cert buffer is NULL */
    ExpectIntEQ(wc_KeyPemToDer(NULL, cert_sz, cert_der, cert_dersz, ""),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    /* Bad arg: Cert DER buffer non-NULL but size zero (or less) */
    ExpectIntEQ(wc_KeyPemToDer(cert_buf, cert_sz, (byte*)&cert_der, 0, ""),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    /* Test normal operation */
    cert_dersz = cert_sz; /* DER will be smaller than PEM */
    ExpectNotNull(cert_der = (byte*)malloc((size_t)cert_dersz));
    ExpectIntGE(ret = wc_KeyPemToDer(cert_buf, cert_sz, cert_der, cert_dersz,
        cert_pw), 0);
    ExpectIntLE(ret, cert_sz);
    if (cert_der != NULL) {
        free(cert_der);
        cert_der = NULL;
    }

    /* Test NULL for DER buffer to return needed DER buffer size */
    ExpectIntGT(ret = wc_KeyPemToDer(cert_buf, cert_sz, NULL, 0, ""), 0);
    ExpectIntLE(ret, cert_sz);
    if (EXPECT_SUCCESS())
        cert_dersz = ret;
    ExpectNotNull(cert_der = (byte*)malloc((size_t)cert_dersz));
    ExpectIntGE(ret = wc_KeyPemToDer(cert_buf, cert_sz, cert_der, cert_dersz,
        cert_pw), 0);
    ExpectIntLE(ret, cert_sz);
    if (cert_der != NULL)
        free(cert_der);
#endif
    return EXPECT_RESULT();
}

static int test_wc_PubKeyPemToDer(void)
{
    EXPECT_DECLS;
#if defined(WOLFSSL_PEM_TO_DER) && !defined(NO_FILESYSTEM) && \
   (defined(WOLFSSL_CERT_EXT) || defined(WOLFSSL_PUB_PEM_TO_DER))
    int ret = 0;
    const char* key = "./certs/ecc-client-keyPub.pem";
    byte* cert_buf = NULL;
    size_t cert_sz = 0, cert_dersz = 0;
    byte* cert_der = NULL;

    ExpectIntEQ(wc_PubKeyPemToDer(cert_buf, (int)cert_sz,
        cert_der, (int)cert_dersz), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    ExpectIntEQ(load_file(key, &cert_buf, &cert_sz), 0);
    cert_dersz = cert_sz; /* DER will be smaller than PEM */
    ExpectNotNull(cert_der = (byte*)malloc(cert_dersz));
    ExpectIntGE(wc_PubKeyPemToDer(cert_buf, (int)cert_sz, cert_der,
        (int)cert_dersz), 0);
    if (cert_der != NULL) {
        free(cert_der);
        cert_der = NULL;
    }

    /* Test NULL for DER buffer to return needed DER buffer size */
    ExpectIntGT(ret = wc_PubKeyPemToDer(cert_buf, (int)cert_sz, NULL, 0), 0);
    ExpectIntLE(ret, cert_sz);
    cert_dersz = (size_t)ret;
    ExpectNotNull(cert_der = (byte*)malloc(cert_dersz));
    ExpectIntGE(wc_PubKeyPemToDer(cert_buf, (int)cert_sz, cert_der,
        (int)cert_dersz), 0);
    if (cert_der != NULL) {
        free(cert_der);
    }

    if (cert_buf != NULL) {
        free(cert_buf);
    }
#endif
    return EXPECT_RESULT();
}

static int test_wc_PemPubKeyToDer(void)
{
    EXPECT_DECLS;
#if !defined(NO_FILESYSTEM) && \
    (defined(WOLFSSL_CERT_EXT) || defined(WOLFSSL_PUB_PEM_TO_DER))
    const char* key = "./certs/ecc-client-keyPub.pem";
    size_t cert_dersz = 1024;
    byte* cert_der = NULL;

    ExpectIntGE(wc_PemPubKeyToDer(NULL, cert_der, (int)cert_dersz),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    ExpectNotNull(cert_der = (byte*)malloc(cert_dersz));
    ExpectIntGE(wc_PemPubKeyToDer(key, cert_der, (int)cert_dersz), 0);
    if (cert_der != NULL) {
        free(cert_der);
    }
#endif
    return EXPECT_RESULT();
}

static int test_wc_GetPubKeyDerFromCert(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA) || defined(HAVE_ECC)
    int ret;
    word32 idx = 0;
    byte keyDer[TWOK_BUF];  /* large enough for up to RSA 2048 */
    word32 keyDerSz = (word32)sizeof(keyDer);
    DecodedCert decoded;
#if !defined(NO_RSA) && defined(WOLFSSL_CERT_REQ) && !defined(NO_FILESYSTEM)
    byte certBuf[6000]; /* for PEM and CSR, client-cert.pem is 5-6kB */
    word32 certBufSz = sizeof(certBuf);
#endif
#if ((!defined(USE_CERT_BUFFERS_2048) && !defined(USE_CERT_BUFFERS_1024)) || \
     defined(WOLFSSL_CERT_REQ)) && !defined(NO_RSA) && !defined(NO_FILESYSTEM)
    XFILE fp = XBADFILE;
#endif
#ifndef NO_RSA
    RsaKey rsaKey;
    #if defined(USE_CERT_BUFFERS_2048)
        byte* rsaCertDer = (byte*)client_cert_der_2048;
        word32 rsaCertDerSz = sizeof_client_cert_der_2048;
    #elif defined(USE_CERT_BUFFERS_1024)
        byte* rsaCertDer = (byte*)client_cert_der_1024;
        word32 rsaCertDerSz = sizeof_client_cert_der_1024;
    #else
        unsigned char rsaCertDer[TWOK_BUF];
        word32 rsaCertDerSz;
    #endif
#endif
#ifdef HAVE_ECC
    ecc_key eccKey;
    #if defined(USE_CERT_BUFFERS_256)
        byte* eccCert = (byte*)cliecc_cert_der_256;
        word32 eccCertSz = sizeof_cliecc_cert_der_256;
    #else
        unsigned char eccCert[ONEK_BUF];
        word32 eccCertSz;
        XFILE fp2 = XBADFILE;
    #endif
#endif

#ifndef NO_RSA

#if !defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048)
    ExpectTrue((fp = XFOPEN("./certs/1024/client-cert.der", "rb")) != XBADFILE);
    ExpectIntGT(rsaCertDerSz = (word32)XFREAD(rsaCertDer, 1, sizeof(rsaCertDer),
        fp), 0);
    if (fp != XBADFILE) {
        XFCLOSE(fp);
        fp = XBADFILE;
    }
#endif

    /* good test case - RSA DER cert */
    wc_InitDecodedCert(&decoded, rsaCertDer, rsaCertDerSz, NULL);
    ExpectIntEQ(wc_ParseCert(&decoded, CERT_TYPE, NO_VERIFY, NULL), 0);

    ExpectIntEQ(wc_GetPubKeyDerFromCert(&decoded, keyDer, &keyDerSz), 0);
    ExpectIntGT(keyDerSz, 0);

    /* sanity check, verify we can import DER public key */
    ret = wc_InitRsaKey(&rsaKey, HEAP_HINT);
    ExpectIntEQ(ret, 0);
    ExpectIntEQ(wc_RsaPublicKeyDecode(keyDer, &idx, &rsaKey, keyDerSz), 0);
    if (ret == 0) {
        wc_FreeRsaKey(&rsaKey);
    }

    /* test LENGTH_ONLY_E case */
    keyDerSz = 0;
    ExpectIntEQ(wc_GetPubKeyDerFromCert(&decoded, NULL, &keyDerSz),
        WC_NO_ERR_TRACE(LENGTH_ONLY_E));
    ExpectIntGT(keyDerSz, 0);

    /* bad args: DecodedCert NULL */
    ExpectIntEQ(wc_GetPubKeyDerFromCert(NULL, keyDer, &keyDerSz), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    /* bad args: output key buff size */
    ExpectIntEQ(wc_GetPubKeyDerFromCert(&decoded, keyDer, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    /* bad args: zero size output key buffer */
    keyDerSz = 0;
    ExpectIntEQ(ret = wc_GetPubKeyDerFromCert(&decoded, keyDer, &keyDerSz),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    wc_FreeDecodedCert(&decoded);

    /* Certificate Request Tests */
    #if defined(WOLFSSL_CERT_REQ) && !defined(NO_FILESYSTEM)
    {
        XMEMSET(certBuf, 0, sizeof(certBuf));
        ExpectTrue((fp = XFOPEN("./certs/csr.signed.der", "rb")) != XBADFILE);
        ExpectIntGT(certBufSz = (word32)XFREAD(certBuf, 1, certBufSz, fp), 0);
        if (fp != XBADFILE) {
            XFCLOSE(fp);
        }

        wc_InitDecodedCert(&decoded, certBuf, certBufSz, NULL);
        ExpectIntEQ(wc_ParseCert(&decoded, CERTREQ_TYPE, VERIFY, NULL), 0);

        /* good test case - RSA DER certificate request */
        keyDerSz = sizeof(keyDer);
        ExpectIntEQ(ret = wc_GetPubKeyDerFromCert(&decoded, keyDer, &keyDerSz),
            0);
        ExpectIntGT(keyDerSz, 0);

        /* sanity check, verify we can import DER public key */
        ret = wc_InitRsaKey(&rsaKey, HEAP_HINT);
        ExpectIntEQ(ret, 0);
        idx = 0;
        ExpectIntEQ(wc_RsaPublicKeyDecode(keyDer, &idx, &rsaKey, keyDerSz), 0);
        if (ret == 0) {
            wc_FreeRsaKey(&rsaKey);
        }

        wc_FreeDecodedCert(&decoded);
    }
    #endif /* WOLFSSL_CERT_REQ */
#endif /* NO_RSA */

#ifdef HAVE_ECC
    #ifndef USE_CERT_BUFFERS_256
        ExpectTrue((fp2 = XFOPEN("./certs/client-ecc-cert.der", "rb")) !=
            XBADFILE);
        ExpectIntGT(eccCertSz = (word32)XFREAD(eccCert, 1, ONEK_BUF, fp2), 0);
        if (fp2 != XBADFILE) {
            XFCLOSE(fp2);
        }
    #endif

    wc_InitDecodedCert(&decoded, eccCert, eccCertSz, NULL);
    ExpectIntEQ(wc_ParseCert(&decoded, CERT_TYPE, NO_VERIFY, NULL), 0);

    /* good test case - ECC */
    XMEMSET(keyDer, 0, sizeof(keyDer));
    keyDerSz = sizeof(keyDer);
    ExpectIntEQ(wc_GetPubKeyDerFromCert(&decoded, keyDer, &keyDerSz), 0);
    ExpectIntGT(keyDerSz, 0);

    /* sanity check, verify we can import DER public key */
    ret = wc_ecc_init(&eccKey);
    ExpectIntEQ(ret, 0);
    idx = 0; /* reset idx to 0, used above in RSA case */
    ExpectIntEQ(wc_EccPublicKeyDecode(keyDer, &idx, &eccKey, keyDerSz), 0);
    if (ret == 0) {
        wc_ecc_free(&eccKey);
    }

    /* test LENGTH_ONLY_E case */
    keyDerSz = 0;
    ExpectIntEQ(wc_GetPubKeyDerFromCert(&decoded, NULL, &keyDerSz),
        WC_NO_ERR_TRACE(LENGTH_ONLY_E));
    ExpectIntGT(keyDerSz, 0);

    wc_FreeDecodedCert(&decoded);
#endif
#endif /* !NO_RSA || HAVE_ECC */
    return EXPECT_RESULT();
}

static int test_wc_CheckCertSigPubKey(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_FILESYSTEM) && \
    !defined(NO_RSA) && defined(WOLFSSL_PEM_TO_DER) && defined(HAVE_ECC)
    int ret = 0;
    const char* ca_cert = "./certs/ca-cert.pem";
    byte* cert_buf = NULL;
    size_t cert_sz = 0;
    byte* cert_der = NULL;
    word32 cert_dersz = 0;
    byte keyDer[TWOK_BUF];  /* large enough for up to RSA 2048 */
    word32 keyDerSz = (word32)sizeof(keyDer);
    DecodedCert decoded;

    ExpectIntEQ(load_file(ca_cert, &cert_buf, &cert_sz), 0);
    cert_dersz = (word32)cert_sz; /* DER will be smaller than PEM */
    ExpectNotNull(cert_der = (byte*)malloc(cert_dersz));
    ExpectIntGE(ret = wc_CertPemToDer(cert_buf, (int)cert_sz, cert_der,
        (int)cert_dersz, CERT_TYPE), 0);

    wc_InitDecodedCert(&decoded, cert_der, cert_dersz, NULL);
    ExpectIntEQ(wc_ParseCert(&decoded, CERT_TYPE, NO_VERIFY, NULL), 0);

    ExpectIntEQ(wc_GetPubKeyDerFromCert(&decoded, keyDer, &keyDerSz), 0);
    ExpectIntGT(keyDerSz, 0);

    /* Good test case. */
    ExpectIntEQ(wc_CheckCertSigPubKey(cert_der, cert_dersz, NULL, keyDer,
        keyDerSz, RSAk), 0);

    /* No certificate. */
    ExpectIntEQ(wc_CheckCertSigPubKey(NULL, cert_dersz, NULL, keyDer, keyDerSz,
        ECDSAk), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    /* Bad cert size. */
    ExpectIntNE(ret = wc_CheckCertSigPubKey(cert_der, 0, NULL, keyDer, keyDerSz,
        RSAk), 0);
    ExpectTrue(ret == WC_NO_ERR_TRACE(ASN_PARSE_E) || ret == WC_NO_ERR_TRACE(BUFFER_E));

    /* No public key. */
    ExpectIntEQ(wc_CheckCertSigPubKey(cert_der, cert_dersz, NULL, NULL,
        keyDerSz, RSAk), WC_NO_ERR_TRACE(ASN_NO_SIGNER_E));

    /* Bad public key size. */
    ExpectIntEQ(wc_CheckCertSigPubKey(cert_der, cert_dersz, NULL, keyDer, 0,
        RSAk), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    /* Wrong aglo. */
    ExpectIntEQ(wc_CheckCertSigPubKey(cert_der, cert_dersz, NULL, keyDer,
        keyDerSz, ECDSAk), WC_NO_ERR_TRACE(ASN_PARSE_E));

    wc_FreeDecodedCert(&decoded);
    if (cert_der != NULL)
        free(cert_der);
    if (cert_buf != NULL)
        free(cert_buf);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_ext_d2i(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS)
    X509* x509 = NULL;

    ExpectNotNull(x509 = wolfSSL_X509_new());

    ExpectNull(wolfSSL_X509_get_ext_d2i(x509, NID_basic_constraints,
        NULL, NULL));
    ExpectNull(wolfSSL_X509_get_ext_d2i(x509, NID_subject_alt_name,
        NULL, NULL));
    ExpectNull(wolfSSL_X509_get_ext_d2i(x509, NID_authority_key_identifier,
        NULL, NULL));
    ExpectNull(wolfSSL_X509_get_ext_d2i(x509, NID_subject_key_identifier,
        NULL, NULL));
    ExpectNull(wolfSSL_X509_get_ext_d2i(x509, NID_key_usage,
        NULL, NULL));
    ExpectNull(wolfSSL_X509_get_ext_d2i(x509, NID_crl_distribution_points,
        NULL, NULL));
    ExpectNull(wolfSSL_X509_get_ext_d2i(x509, NID_ext_key_usage,
        NULL, NULL));
    ExpectNull(wolfSSL_X509_get_ext_d2i(x509, NID_info_access,
        NULL, NULL));
    ExpectNull(wolfSSL_X509_get_ext_d2i(x509, NID_certificate_policies,
        NULL, NULL));
    /* Invalid NID for an extension. */
    ExpectNull(wolfSSL_X509_get_ext_d2i(x509, NID_description,
        NULL, NULL));

    wolfSSL_X509_free(x509);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_certs(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_FILESYSTEM) && \
    !defined(NO_TLS) && !defined(NO_RSA)
    X509*  x509ext = NULL;
    X509*  x509 = NULL;
#ifdef OPENSSL_ALL
    WOLFSSL_X509_EXTENSION* ext = NULL;
    ASN1_OBJECT* obj = NULL;
#endif
    WOLFSSL*     ssl = NULL;
    WOLFSSL_CTX* ctx = NULL;
    STACK_OF(ASN1_OBJECT)* sk = NULL;
    ASN1_STRING* asn1_str = NULL;
    AUTHORITY_KEYID* akey = NULL;
    WOLFSSL_STACK* skid = NULL;
    BASIC_CONSTRAINTS* bc = NULL;
    int crit = 0;

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(SSLv23_server_method()));
#else
    ExpectNotNull(ctx = SSL_CTX_new(SSLv23_client_method()));
#endif
    ExpectTrue(SSL_CTX_use_certificate_file(ctx, svrCertFile, SSL_FILETYPE_PEM));
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, SSL_FILETYPE_PEM));
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, cliKeyFile, SSL_FILETYPE_PEM));
    #if !defined(NO_CHECK_PRIVATE_KEY)
    ExpectIntEQ(SSL_CTX_check_private_key(ctx), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    #endif
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, SSL_FILETYPE_PEM));
    #if !defined(NO_CHECK_PRIVATE_KEY)
    ExpectIntEQ(SSL_CTX_check_private_key(ctx), SSL_SUCCESS);
    #endif
    ExpectNotNull(ssl = SSL_new(ctx));

    /* Invalid parameters. */
    ExpectIntEQ(SSL_use_certificate_file(NULL, NULL, WOLFSSL_FILETYPE_PEM),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(SSL_use_certificate_file(ssl, NULL, WOLFSSL_FILETYPE_PEM),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_use_certificate_file(NULL, "./certs/server-cert.pem",
        WOLFSSL_FILETYPE_PEM), WC_NO_ERR_TRACE(BAD_FUNC_ARG));

    #if !defined(NO_CHECK_PRIVATE_KEY)
    ExpectIntEQ(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif

    #ifdef HAVE_PK_CALLBACKS
    ExpectIntEQ((int)SSL_set_tlsext_debug_arg(ssl, NULL), WOLFSSL_SUCCESS);
    #endif /* HAVE_PK_CALLBACKS */

    /* Invalid parameters. */
    ExpectNotNull(x509 = wolfSSL_X509_new());
    ExpectIntEQ(SSL_use_certificate(NULL, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_use_certificate(ssl, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_use_certificate(NULL, x509), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    /* No data in certificate. */
    ExpectIntEQ(SSL_use_certificate(ssl, x509), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    wolfSSL_X509_free(x509);
    x509 = NULL;

    /* create and use x509 */
    ExpectNull(wolfSSL_X509_load_certificate_file(cliCertFileExt, -1));
    ExpectNull(wolfSSL_X509_load_certificate_file("/tmp/badfile",
        WOLFSSL_FILETYPE_PEM));
    ExpectNull(wolfSSL_X509_load_certificate_file(NULL, WOLFSSL_FILETYPE_PEM));
    ExpectNull(wolfSSL_X509_load_certificate_file(cliCertFileExt,
        WOLFSSL_FILETYPE_ASN1));
#ifdef OPENSSL_ALL
    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(cliCertFile,
        WOLFSSL_FILETYPE_PEM));
#endif
    ExpectNotNull(x509ext = wolfSSL_X509_load_certificate_file(cliCertFileExt,
        WOLFSSL_FILETYPE_PEM));
    ExpectIntEQ(SSL_use_certificate(ssl, x509ext), WOLFSSL_SUCCESS);

    #if !defined(NO_CHECK_PRIVATE_KEY)
    /* with loading in a new cert the check on private key should now fail */
    ExpectIntNE(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif


    #if defined(USE_CERT_BUFFERS_2048)
        /* Invalid parameters. */
        ExpectIntEQ(SSL_use_certificate_ASN1(NULL, NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
        ExpectIntEQ(SSL_use_certificate_ASN1(ssl, NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
        ExpectIntEQ(SSL_use_certificate_ASN1(NULL,
                (unsigned char*)server_cert_der_2048, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
        /* No data. */
        ExpectIntEQ(SSL_use_certificate_ASN1(ssl,
                (unsigned char*)server_cert_der_2048, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

        ExpectIntEQ(SSL_use_certificate_ASN1(ssl,
                                  (unsigned char*)server_cert_der_2048,
                                  sizeof_server_cert_der_2048), WOLFSSL_SUCCESS);
    #endif

    #if !defined(NO_SHA) && !defined(NO_SHA256) && !defined(NO_PWDBASED)
    /************* Get Digest of Certificate ******************/
    {
        byte   digest[64]; /* max digest size */
        word32 digestSz;
        X509*  x509Empty = NULL;

        XMEMSET(digest, 0, sizeof(digest));
        ExpectIntEQ(X509_digest(NULL, wolfSSL_EVP_sha1(), digest, &digestSz),
                    WOLFSSL_FAILURE);
        ExpectIntEQ(X509_digest(x509ext, NULL, digest, &digestSz),
                    WOLFSSL_FAILURE);
        ExpectIntEQ(X509_digest(x509ext, wolfSSL_EVP_sha1(), NULL, &digestSz),
                    WOLFSSL_FAILURE);
        ExpectIntEQ(X509_digest(x509ext, wolfSSL_EVP_sha1(), digest, NULL),
                    WOLFSSL_SUCCESS);
        ExpectIntEQ(X509_digest(x509ext, wolfSSL_EVP_sha1(), digest, &digestSz),
                    WOLFSSL_SUCCESS);
        ExpectIntEQ(X509_digest(x509ext, wolfSSL_EVP_sha256(), digest,
                    &digestSz), WOLFSSL_SUCCESS);

        ExpectNotNull(x509Empty = wolfSSL_X509_new());
        ExpectIntEQ(X509_digest(x509Empty, wolfSSL_EVP_sha256(), digest,
                    &digestSz), WOLFSSL_FAILURE);
        wolfSSL_X509_free(x509Empty);
    }
    #endif /* !NO_SHA && !NO_SHA256 && !NO_PWDBASED */

    #if !defined(NO_SHA) && !defined(NO_SHA256) && !defined(NO_PWDBASED)
    /************* Get Digest of Certificate ******************/
    {
        byte   digest[64]; /* max digest size */
        word32 digestSz;
        X509*  x509Empty = NULL;

        XMEMSET(digest, 0, sizeof(digest));
        ExpectIntEQ(X509_pubkey_digest(NULL, wolfSSL_EVP_sha1(), digest,
                    &digestSz), WOLFSSL_FAILURE);
        ExpectIntEQ(X509_pubkey_digest(x509ext, NULL, digest, &digestSz),
                    WOLFSSL_FAILURE);
        ExpectIntEQ(X509_pubkey_digest(x509ext, wolfSSL_EVP_sha1(), NULL,
                    &digestSz), WOLFSSL_FAILURE);
        ExpectIntEQ(X509_pubkey_digest(x509ext, wolfSSL_EVP_sha1(), digest,
                    NULL), WOLFSSL_SUCCESS);
        ExpectIntEQ(X509_pubkey_digest(x509ext, wolfSSL_EVP_sha1(), digest,
                    &digestSz), WOLFSSL_SUCCESS);
        ExpectIntEQ(X509_pubkey_digest(x509ext, wolfSSL_EVP_sha256(), digest,
                    &digestSz), WOLFSSL_SUCCESS);

        ExpectNotNull(x509Empty = wolfSSL_X509_new());
        ExpectIntEQ(X509_pubkey_digest(x509Empty, wolfSSL_EVP_sha256(), digest,
                    &digestSz), WOLFSSL_FAILURE);
        wolfSSL_X509_free(x509Empty);
    }
    #endif /* !NO_SHA && !NO_SHA256 && !NO_PWDBASED */

    /* test and checkout X509 extensions */
    ExpectNotNull(bc = (BASIC_CONSTRAINTS*)X509_get_ext_d2i(x509ext,
        NID_basic_constraints, NULL, NULL));
    BASIC_CONSTRAINTS_free(bc);
    bc = NULL;
    ExpectNotNull(bc = (BASIC_CONSTRAINTS*)X509_get_ext_d2i(x509ext,
        NID_basic_constraints, &crit, NULL));
    ExpectIntEQ(crit, 0);

#ifdef OPENSSL_ALL
    ExpectNull(X509V3_EXT_i2d(NID_basic_constraints, crit, NULL));
    {
        int i;
        int unsupportedNid[] = {
            0,
            NID_inhibit_any_policy,
            NID_certificate_policies,
            NID_policy_mappings,
            NID_name_constraints,
            NID_policy_constraints,
            NID_crl_distribution_points
        };
        int unsupportedNidCnt = (int)(sizeof(unsupportedNid) /
                                      sizeof(*unsupportedNid));

        for (i = 0; i < unsupportedNidCnt; i++) {
            ExpectNotNull(ext = X509V3_EXT_i2d(unsupportedNid[i], crit, bc));
            X509_EXTENSION_free(ext);
            ext = NULL;
        }
    }
    ExpectNotNull(ext = X509V3_EXT_i2d(NID_basic_constraints, crit, bc));
    X509_EXTENSION_free(ext);
    ext = NULL;

    ExpectNotNull(ext = X509_EXTENSION_new());
    ExpectIntEQ(X509_EXTENSION_set_critical(NULL, 1), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_EXTENSION_set_critical(ext, 1), WOLFSSL_SUCCESS);
    ExpectNotNull(obj = OBJ_nid2obj(NID_basic_constraints));
    ExpectIntEQ(X509_EXTENSION_set_object(NULL, NULL), SSL_FAILURE);
    ExpectIntEQ(X509_EXTENSION_set_object(NULL, obj), SSL_FAILURE);
    ExpectIntEQ(X509_EXTENSION_set_object(ext, NULL), SSL_SUCCESS);
    ExpectIntEQ(X509_EXTENSION_set_object(ext, obj), SSL_SUCCESS);
    /* Check old object is being freed. */
    ExpectIntEQ(X509_EXTENSION_set_object(ext, obj), SSL_SUCCESS);
    ASN1_OBJECT_free(obj);
    obj = NULL;
    X509_EXTENSION_free(ext);
    ext = NULL;

    ExpectNotNull(ext = X509_EXTENSION_new());
    ExpectIntEQ(X509_EXTENSION_set_critical(ext, 0), WOLFSSL_SUCCESS);
    ExpectIntEQ(X509_EXTENSION_set_data(ext, NULL),
            WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectNotNull(asn1_str = (ASN1_STRING*)X509_get_ext_d2i(x509ext,
            NID_key_usage, NULL, NULL));
    ASN1_STRING_free(asn1_str);
    asn1_str = NULL;
    ExpectNotNull(asn1_str = (ASN1_STRING*)X509_get_ext_d2i(x509ext,
            NID_key_usage, &crit, NULL));
    ExpectIntEQ(X509_EXTENSION_set_data(ext, asn1_str), SSL_SUCCESS);
    ExpectIntEQ(X509_EXTENSION_set_data(ext, asn1_str), SSL_SUCCESS);
    ASN1_STRING_free(asn1_str); /* X509_EXTENSION_set_data has made a copy
                                 * and X509_get_ext_d2i has created new */
    asn1_str = NULL;
    X509_EXTENSION_free(ext);
    ext = NULL;

#endif
    BASIC_CONSTRAINTS_free(NULL);
    BASIC_CONSTRAINTS_free(bc);
    bc = NULL;

    ExpectNotNull(asn1_str = (ASN1_STRING*)X509_get_ext_d2i(x509ext,
        NID_key_usage, NULL, NULL));
    ASN1_STRING_free(asn1_str);
    asn1_str = NULL;
    ExpectNotNull(asn1_str = (ASN1_STRING*)X509_get_ext_d2i(x509ext,
        NID_key_usage, &crit, NULL));
    ExpectIntEQ(crit, 1);
    ExpectIntEQ(asn1_str->type, NID_key_usage);
#ifdef OPENSSL_ALL
    ExpectNotNull(ext = X509V3_EXT_i2d(NID_key_usage, crit, asn1_str));
    X509_EXTENSION_free(ext);
    ext = NULL;
#endif
    ASN1_STRING_free(asn1_str);
    asn1_str = NULL;

#ifdef OPENSSL_ALL
    ExpectNotNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509,
        NID_ext_key_usage, NULL, NULL));
    EXTENDED_KEY_USAGE_free(NULL);
    EXTENDED_KEY_USAGE_free(sk);
    sk = NULL;
    ExpectNotNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509,
        NID_ext_key_usage, &crit, NULL));
    ExpectNotNull(ext = X509V3_EXT_i2d(NID_ext_key_usage, crit, sk));
    X509_EXTENSION_free(ext);
    ext = NULL;
    EXTENDED_KEY_USAGE_free(sk);
    sk = NULL;
#else
    sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext, NID_ext_key_usage,
            &crit, NULL);
    ExpectNull(sk);
#endif

    ExpectNotNull(akey = (AUTHORITY_KEYID*)X509_get_ext_d2i(x509ext,
        NID_authority_key_identifier, NULL, NULL));
    wolfSSL_AUTHORITY_KEYID_free(NULL);
    wolfSSL_AUTHORITY_KEYID_free(akey);
    akey = NULL;
    ExpectNotNull(akey = (AUTHORITY_KEYID*)X509_get_ext_d2i(x509ext,
        NID_authority_key_identifier, &crit, NULL));
#ifdef OPENSSL_ALL
    ExpectNotNull(ext = X509V3_EXT_i2d(NID_authority_key_identifier, crit,
        akey));
    X509_EXTENSION_free(ext);
    ext = NULL;
#endif
    wolfSSL_AUTHORITY_KEYID_free(akey);
    akey = NULL;

    ExpectNotNull(skid = (WOLFSSL_STACK*)X509_get_ext_d2i(x509ext,
        NID_subject_key_identifier, NULL, NULL));
    wolfSSL_sk_ASN1_OBJECT_pop_free(skid, wolfSSL_ASN1_OBJECT_free);
    skid = NULL;
    ExpectNotNull(skid = (WOLFSSL_STACK*)X509_get_ext_d2i(x509ext,
        NID_subject_key_identifier, &crit, NULL));
#ifdef OPENSSL_ALL
    ExpectNotNull(ext = X509V3_EXT_i2d(NID_subject_key_identifier, crit,
        skid));
    X509_EXTENSION_free(ext);
    ext = NULL;
#endif
    wolfSSL_sk_ASN1_OBJECT_pop_free(skid, wolfSSL_ASN1_OBJECT_free);
    skid = NULL;

    /* NID not yet supported */
    ExpectNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext,
        NID_private_key_usage_period, &crit, NULL));
    ExpectIntEQ(crit, -1);
    sk_ASN1_OBJECT_free(sk);
    sk = NULL;

    ExpectNotNull(sk = (STACK_OF(GENERAL_NAME)*)X509_get_ext_d2i(x509ext,
        NID_subject_alt_name, NULL, NULL));
    sk_GENERAL_NAME_free(sk);
    sk = NULL;
    ExpectNotNull(sk = (STACK_OF(GENERAL_NAME)*)X509_get_ext_d2i(x509ext,
        NID_subject_alt_name, &crit, NULL));
    {
        int i;
        for (i = 0; i < sk_GENERAL_NAME_num(sk); i++) {
            GENERAL_NAME* gen = sk_GENERAL_NAME_value(sk, i);
            ExpectIntEQ(gen->type, GEN_DNS);
            ExpectIntEQ(gen->d.dNSName->type, V_ASN1_IA5STRING);
        }
    }
    sk_GENERAL_NAME_free(sk);
    sk = NULL;

    /* NID not yet supported */
    ExpectNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext,
        NID_issuer_alt_name, &crit, NULL));
    ExpectIntEQ(crit, -1);
    sk_ASN1_OBJECT_free(sk);
    sk = NULL;

    /* NID not yet supported */
    ExpectNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext,
        NID_info_access, &crit, NULL));
    sk_ASN1_OBJECT_free(sk);
    sk = NULL;

    /* NID not yet supported */
    ExpectNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext,
        NID_sinfo_access, &crit, NULL));
    ExpectIntEQ(crit, -1);
    sk_ASN1_OBJECT_free(sk);
    sk = NULL;

    /* NID not yet supported */
    ExpectNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext,
        NID_name_constraints, &crit, NULL));
    ExpectIntEQ(crit, -1);
    sk_ASN1_OBJECT_free(sk);
    sk = NULL;

    /* no cert policy set */
    ExpectNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext,
            NID_certificate_policies, &crit, NULL));
    sk_ASN1_OBJECT_free(sk);
    sk = NULL;

    /* NID not yet supported */
    ExpectNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext,
        NID_policy_mappings, &crit, NULL));
    ExpectIntEQ(crit, -1);
    sk_ASN1_OBJECT_free(sk);
    sk = NULL;

    /* NID not yet supported */
    ExpectNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext,
        NID_policy_constraints, &crit, NULL));
    ExpectIntEQ(crit, -1);
    sk_ASN1_OBJECT_free(sk);
    sk = NULL;

    /* NID not yet supported */
    ExpectNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext,
        NID_inhibit_any_policy, &crit, NULL));
    ExpectIntEQ(crit, -1);
    sk_ASN1_OBJECT_free(sk);
    sk = NULL;

    /* NID not yet supported */
    ExpectNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext,
        NID_tlsfeature, &crit, NULL));
    ExpectIntEQ(crit, -1);
    sk_ASN1_OBJECT_free(sk);
    sk = NULL;

    /* test invalid cases */
    crit = 0;
    ExpectNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(x509ext, -1, &crit,
        NULL));
    ExpectIntEQ(crit, -1);
    /* NULL passed for criticality. */
    ExpectNull(sk = (STACK_OF(ASN1_OBJECT)*)X509_get_ext_d2i(NULL,
        NID_tlsfeature, NULL, NULL));

    ExpectIntEQ(SSL_get_hit(ssl), 0);
#ifdef OPENSSL_ALL
    X509_free(x509);
#endif
    X509_free(x509ext);
    SSL_free(ssl);
    SSL_CTX_free(ctx);
#endif /* OPENSSL_EXTRA && !NO_CERTS */
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_check_private_key(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_RSA) && \
        defined(USE_CERT_BUFFERS_2048) && !defined(NO_CHECK_PRIVATE_KEY)
    X509*  x509 = NULL;
    EVP_PKEY* pkey = NULL;
    const byte* key;

    /* Check with correct key */
    ExpectNotNull((x509 = X509_load_certificate_file(cliCertFile,
        SSL_FILETYPE_PEM)));
    key = client_key_der_2048;
    ExpectNotNull(d2i_PrivateKey(EVP_PKEY_RSA, &pkey, &key,
        (long)sizeof_client_key_der_2048));
    ExpectIntEQ(X509_check_private_key(x509, pkey), 1);
    EVP_PKEY_free(pkey);
    pkey = NULL;

    /* Check with wrong key */
    key = server_key_der_2048;
    ExpectNotNull(d2i_PrivateKey(EVP_PKEY_RSA, &pkey, &key,
        (long)sizeof_server_key_der_2048));
    ExpectIntEQ(X509_check_private_key(x509, pkey), 0);

    /* test for incorrect parameter */
    ExpectIntEQ(X509_check_private_key(NULL, pkey), 0);
    ExpectIntEQ(X509_check_private_key(x509, NULL), 0);
    ExpectIntEQ(X509_check_private_key(NULL, NULL), 0);

    EVP_PKEY_free(pkey);
    X509_free(x509);
#endif
    return EXPECT_RESULT();
}


static int test_wolfSSL_private_keys(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_TLS) && \
   !defined(NO_FILESYSTEM)
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
    WOLFSSL*     ssl = NULL;
    WOLFSSL_CTX* ctx = NULL;
    EVP_PKEY* pkey = NULL;

    OpenSSL_add_all_digests();
    OpenSSL_add_all_algorithms();

#ifndef NO_RSA
    #ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_server_method()));
    #else
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
    #endif
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
        WOLFSSL_FILETYPE_PEM));
    /* Have to load a cert before you can check the private key against that
     * certificates public key! */
    #if !defined(NO_CHECK_PRIVATE_KEY)
    ExpectIntEQ(wolfSSL_CTX_check_private_key(ctx), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    #endif
    ExpectTrue(SSL_CTX_use_certificate_file(ctx, svrCertFile,
        WOLFSSL_FILETYPE_PEM));
    #if !defined(NO_CHECK_PRIVATE_KEY)
    ExpectIntEQ(wolfSSL_CTX_check_private_key(ctx), WOLFSSL_SUCCESS);
    #endif
    ExpectNotNull(ssl = SSL_new(ctx));

    #if !defined(NO_CHECK_PRIVATE_KEY)
    ExpectIntEQ(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif

    /* Invalid parameters. */
    ExpectIntEQ(SSL_use_PrivateKey_file(NULL, NULL, WOLFSSL_FILETYPE_PEM),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(SSL_use_PrivateKey_file(NULL, svrKeyFile, WOLFSSL_FILETYPE_PEM),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ(SSL_use_PrivateKey_file(ssl, NULL, WOLFSSL_FILETYPE_PEM),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

#ifdef USE_CERT_BUFFERS_2048
    {
    const unsigned char* server_key = (const unsigned char*)server_key_der_2048;
    unsigned char buf[FOURK_BUF];
    word32 bufSz;

    /* Invalid parameters. */
    ExpectIntEQ(SSL_use_RSAPrivateKey_ASN1(NULL, NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_use_RSAPrivateKey_ASN1(ssl, NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_use_RSAPrivateKey_ASN1(NULL,
        (unsigned char*)client_key_der_2048, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_use_PrivateKey_ASN1(0, NULL, NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_use_PrivateKey_ASN1(0, ssl, NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_use_PrivateKey_ASN1(0, NULL, (unsigned char*)server_key, 0),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_CTX_use_PrivateKey_ASN1(0, NULL, NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_CTX_use_PrivateKey_ASN1(0, ctx, NULL, 0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_CTX_use_PrivateKey_ASN1(0, NULL, (unsigned char*)server_key,
        0), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    ExpectIntEQ(SSL_use_RSAPrivateKey_ASN1(ssl,
                (unsigned char*)client_key_der_2048,
                sizeof_client_key_der_2048), WOLFSSL_SUCCESS);
    #if !defined(NO_CHECK_PRIVATE_KEY)
    /* Should mismatch now that a different private key loaded */
    ExpectIntNE(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif

    ExpectIntEQ(SSL_use_PrivateKey_ASN1(0, ssl,
                (unsigned char*)server_key,
                sizeof_server_key_der_2048), WOLFSSL_SUCCESS);
    #if !defined(NO_CHECK_PRIVATE_KEY)
    /* After loading back in DER format of original key, should match */
    ExpectIntEQ(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif

    /* test loading private key to the WOLFSSL_CTX */
    ExpectIntEQ(SSL_CTX_use_PrivateKey_ASN1(0, ctx,
                (unsigned char*)client_key_der_2048,
                sizeof_client_key_der_2048), WOLFSSL_SUCCESS);

    #if !defined(NO_CHECK_PRIVATE_KEY)
    /* Should mismatch now that a different private key loaded */
    ExpectIntNE(wolfSSL_CTX_check_private_key(ctx), WOLFSSL_SUCCESS);
    #endif

    ExpectIntEQ(SSL_CTX_use_PrivateKey_ASN1(0, ctx,
                (unsigned char*)server_key,
                sizeof_server_key_der_2048), WOLFSSL_SUCCESS);
    #if !defined(NO_CHECK_PRIVATE_KEY)
    /* After loading back in DER format of original key, should match */
    ExpectIntEQ(wolfSSL_CTX_check_private_key(ctx), WOLFSSL_SUCCESS);
    #endif

    /* Invalid parameters. */
    ExpectNotNull(pkey = wolfSSL_EVP_PKEY_new());
    ExpectIntEQ(SSL_use_PrivateKey(NULL, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_use_PrivateKey(ssl, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(SSL_use_PrivateKey(NULL, pkey), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    /* pkey is empty - no key data to use. */
    ExpectIntEQ(SSL_use_PrivateKey(ssl, pkey), WC_NO_ERR_TRACE(ASN_PARSE_E));
    wolfSSL_EVP_PKEY_free(pkey);
    pkey = NULL;

    /* set PKEY and test again */
    ExpectNotNull(wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, &pkey,
                &server_key, (long)sizeof_server_key_der_2048));
    ExpectIntEQ(SSL_use_PrivateKey(ssl, pkey), WOLFSSL_SUCCESS);

    /* reuse PKEY structure and test
     * this should be checked with a memory management sanity checker */
    ExpectFalse(server_key == (const unsigned char*)server_key_der_2048);
    server_key = (const unsigned char*)server_key_der_2048;
    ExpectNotNull(wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, &pkey,
                &server_key, (long)sizeof_server_key_der_2048));
    ExpectIntEQ(SSL_use_PrivateKey(ssl, pkey), WOLFSSL_SUCCESS);

    /* check striping PKCS8 header with wolfSSL_d2i_PrivateKey */
    bufSz = FOURK_BUF;
    ExpectIntGT((bufSz = (word32)wc_CreatePKCS8Key(buf, &bufSz,
                    (byte*)server_key_der_2048, sizeof_server_key_der_2048,
                    RSAk, NULL, 0)), 0);
    server_key = (const unsigned char*)buf;
    ExpectNotNull(wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, &pkey, &server_key,
                (long)bufSz));
    }
#endif


    EVP_PKEY_free(pkey);
    pkey = NULL;
    SSL_free(ssl); /* frees x509 also since loaded into ssl */
    ssl = NULL;
    SSL_CTX_free(ctx);
    ctx = NULL;
#endif /* end of RSA private key match tests */


#ifdef HAVE_ECC
    #ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_server_method()));
    #else
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
    #endif
    ExpectTrue(SSL_CTX_use_certificate_file(ctx, eccCertFile,
                                                         WOLFSSL_FILETYPE_PEM));
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, eccKeyFile,
                                                         WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(ssl = SSL_new(ctx));

    #if !defined(NO_CHECK_PRIVATE_KEY)
    ExpectIntEQ(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif
    SSL_free(ssl);
    ssl = NULL;


    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, cliEccKeyFile,
                                                         WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(ssl = SSL_new(ctx));

    #ifdef WOLFSSL_VALIDATE_ECC_IMPORT
    ExpectIntNE(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif

    SSL_free(ssl);
    ssl = NULL;
    SSL_CTX_free(ctx);
    ctx = NULL;
#endif /* end of ECC private key match tests */

#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_IMPORT)
    #ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_server_method()));
    #else
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
    #endif
    ExpectTrue(SSL_CTX_use_certificate_file(ctx, edCertFile,
                                                         WOLFSSL_FILETYPE_PEM));
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, edKeyFile,
                                                         WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(ssl = SSL_new(ctx));

    #if !defined(NO_CHECK_PRIVATE_KEY)
    ExpectIntEQ(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif
    SSL_free(ssl);
    ssl = NULL;


    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, cliEdKeyFile,
                                                         WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(ssl = SSL_new(ctx));

    #if !defined(NO_CHECK_PRIVATE_KEY)
    #ifdef HAVE_ED25519_MAKE_KEY
    ExpectIntNE(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #else
    ExpectIntEQ(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif
    #endif

    SSL_free(ssl);
    ssl = NULL;
    SSL_CTX_free(ctx);
    ctx = NULL;
#endif /* end of Ed25519 private key match tests */

#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_IMPORT)
    #ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_server_method()));
    #else
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
    #endif
    ExpectTrue(SSL_CTX_use_certificate_file(ctx, ed448CertFile,
                                                         WOLFSSL_FILETYPE_PEM));
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, ed448KeyFile,
                                                         WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(ssl = SSL_new(ctx));

    #if !defined(NO_CHECK_PRIVATE_KEY)
    ExpectIntEQ(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif
    SSL_free(ssl);
    ssl = NULL;


    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, cliEd448KeyFile,
                                                         WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(ssl = SSL_new(ctx));

    #if !defined(NO_CHECK_PRIVATE_KEY)
    ExpectIntNE(wolfSSL_check_private_key(ssl), WOLFSSL_SUCCESS);
    #endif

    SSL_free(ssl);
    ssl = NULL;
    SSL_CTX_free(ctx);
    ctx = NULL;
#endif /* end of Ed448 private key match tests */

    EVP_cleanup();

    /* test existence of no-op macros in wolfssl/openssl/ssl.h */
    CONF_modules_free();
    ENGINE_cleanup();
    CONF_modules_unload();

    (void)ssl;
    (void)ctx;
    (void)pkey;
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_CERTS) */
    return EXPECT_RESULT();
}

static int test_wolfSSL_PEM_def_callback(void)
{
    EXPECT_DECLS;
#ifdef OPENSSL_EXTRA
    char buf[10];
    const char* defpwd = "DEF PWD";
    int defpwdLen = (int)XSTRLEN(defpwd);
    int smallLen = 1;

    /* Bad parameters. */
    ExpectIntEQ(wolfSSL_PEM_def_callback(NULL, sizeof(buf), 0, NULL), 0);
    ExpectIntEQ(wolfSSL_PEM_def_callback(NULL, sizeof(buf), 0, (void*)defpwd),
        0);
    ExpectIntEQ(wolfSSL_PEM_def_callback(buf, sizeof(buf), 0, NULL), 0);

    XMEMSET(buf, 0, sizeof(buf));
    ExpectIntEQ(wolfSSL_PEM_def_callback(buf, sizeof(buf), 0, (void*)defpwd),
        defpwdLen);
    ExpectIntEQ(XMEMCMP(buf, defpwd, defpwdLen), 0);
    ExpectIntEQ(buf[defpwdLen], 0);
    /* Size of buffer is smaller than default password. */
    XMEMSET(buf, 0, sizeof(buf));
    ExpectIntEQ(wolfSSL_PEM_def_callback(buf, smallLen, 0, (void*)defpwd),
        smallLen);
    ExpectIntEQ(XMEMCMP(buf, defpwd, smallLen), 0);
    ExpectIntEQ(buf[smallLen], 0);
#endif /* OPENSSL_EXTRA */
    return EXPECT_RESULT();
}

static int test_wolfSSL_PEM_read_PrivateKey(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_FILESYSTEM) && (!defined(NO_RSA) || \
    !defined(NO_DSA) || defined(HAVE_ECC) || !defined(NO_DH))
    XFILE file = XBADFILE;
#if !defined(NO_RSA)
    const char* fname_rsa = "./certs/server-key.pem";
    RSA* rsa = NULL;
    WOLFSSL_EVP_PKEY_CTX* ctx = NULL;
    unsigned char* sig = NULL;
    size_t sigLen = 0;
    const unsigned char tbs[] = {0, 1, 2, 3, 4, 5, 6, 7};
    size_t tbsLen = sizeof(tbs);
#endif
#if !defined(NO_DSA)
    const char* fname_dsa = "./certs/dsa2048.pem";
#endif
#if defined(HAVE_ECC)
    const char* fname_ec = "./certs/ecc-key.pem";
#endif
#if !defined(NO_DH)
    const char* fname_dh = "./certs/dh-priv-2048.pem";
#endif
    EVP_PKEY* pkey = NULL;

    /* Check error case. */
    ExpectNull(pkey = PEM_read_PrivateKey(NULL, NULL, NULL, NULL));

    /* not a PEM key. */
    ExpectTrue((file = XFOPEN("./certs/ecc-key.der", "rb")) != XBADFILE);
    ExpectNull(PEM_read_PrivateKey(file, NULL, NULL, NULL));
    if (file != XBADFILE)
        XFCLOSE(file);
    file = XBADFILE;

#ifndef NO_RSA
    /* Read in an RSA key. */
    ExpectTrue((file = XFOPEN(fname_rsa, "rb")) != XBADFILE);
    ExpectNotNull(pkey = PEM_read_PrivateKey(file, NULL, NULL, NULL));
    if (file != XBADFILE)
        XFCLOSE(file);
    file = XBADFILE;

    /* Make sure the key is usable by signing some data with it. */
    ExpectNotNull(rsa = EVP_PKEY_get0_RSA(pkey));
    ExpectIntGT((sigLen = RSA_size(rsa)), 0);
    ExpectNotNull(sig = (unsigned char*)XMALLOC(sigLen, HEAP_HINT,
        DYNAMIC_TYPE_TMP_BUFFER));
    ExpectNotNull(ctx = EVP_PKEY_CTX_new(pkey, NULL));
    ExpectIntEQ(EVP_PKEY_sign_init(ctx), WOLFSSL_SUCCESS);
    ExpectIntEQ(EVP_PKEY_sign(ctx, sig, &sigLen, tbs, tbsLen),
        WOLFSSL_SUCCESS);

    XFREE(sig, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    EVP_PKEY_CTX_free(ctx);
    EVP_PKEY_free(pkey);
    pkey = NULL;
#endif

#ifndef NO_DSA
    /* Read in a DSA key. */
    ExpectTrue((file = XFOPEN(fname_dsa, "rb")) != XBADFILE);
#if defined(WOLFSSL_QT) || defined(OPENSSL_ALL) || defined(WOLFSSL_OPENSSH)
    ExpectNotNull(pkey = PEM_read_PrivateKey(file, NULL, NULL, NULL));
    EVP_PKEY_free(pkey);
    pkey = NULL;
#else
    ExpectNull(PEM_read_PrivateKey(file, NULL, NULL, NULL));
#endif
    if (file != XBADFILE)
        XFCLOSE(file);
    file = XBADFILE;
#endif

#ifdef HAVE_ECC
    /* Read in an EC key. */
    ExpectTrue((file = XFOPEN(fname_ec, "rb")) != XBADFILE);
    ExpectNotNull(pkey = EVP_PKEY_new());
    ExpectPtrEq(PEM_read_PrivateKey(file, &pkey, NULL, NULL), pkey);
    if (file != XBADFILE)
        XFCLOSE(file);
    file = XBADFILE;
    EVP_PKEY_free(pkey);
    pkey = NULL;
#endif

#ifndef NO_DH
    /* Read in a DH key. */
    ExpectTrue((file = XFOPEN(fname_dh, "rb")) != XBADFILE);
#if (defined(WOLFSSL_QT) || defined(OPENSSL_ALL) || \
     defined(WOLFSSL_OPENSSH)) && (!defined(HAVE_FIPS) || \
     (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2)))
    ExpectNotNull(pkey = PEM_read_PrivateKey(file, NULL, NULL, NULL));
    EVP_PKEY_free(pkey);
    pkey = NULL;
#else
    ExpectNull(PEM_read_PrivateKey(file, NULL, NULL, NULL));
#endif
    if (file != XBADFILE)
        XFCLOSE(file);
    file = XBADFILE;
#endif
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_PEM_read_PUBKEY(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_RSA) \
    && !defined(NO_FILESYSTEM)
    XFILE file = XBADFILE;
    const char* fname = "./certs/client-keyPub.pem";
    EVP_PKEY* pkey = NULL;

    /* Check error case. */
    ExpectNull(pkey = PEM_read_PUBKEY(NULL, NULL, NULL, NULL));

    /* Read in an RSA key. */
    ExpectTrue((file = XFOPEN(fname, "rb")) != XBADFILE);
    ExpectNotNull(pkey = PEM_read_PUBKEY(file, NULL, NULL, NULL));
    EVP_PKEY_free(pkey);
    pkey = NULL;
    if (file != XBADFILE)
        XFCLOSE(file);
    file = XBADFILE;
    ExpectTrue((file = XFOPEN(fname, "rb")) != XBADFILE);
    ExpectNotNull(pkey = EVP_PKEY_new());
    ExpectPtrEq(PEM_read_PUBKEY(file, &pkey, NULL, NULL), pkey);
    EVP_PKEY_free(pkey);
    if (file != XBADFILE)
        XFCLOSE(file);
#endif
    return EXPECT_RESULT();
}

/* test loading RSA key using BIO */
static int test_wolfSSL_PEM_PrivateKey_rsa(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_RSA) && \
    defined(USE_CERT_BUFFERS_2048) && !defined(NO_FILESYSTEM) && \
    !defined(NO_BIO)
    BIO*      bio = NULL;
    XFILE file = XBADFILE;
    const char* fname = "./certs/server-key.pem";
    const char* fname_rsa_p8 = "./certs/server-keyPkcs8.pem";
    EVP_PKEY* pkey  = NULL;
    size_t sz = 0;
    byte* buf = NULL;
    EVP_PKEY* pkey2 = NULL;
    EVP_PKEY* pkey3 = NULL;
    RSA* rsa_key = NULL;
#if defined(WOLFSSL_KEY_GEN) || defined(WOLFSSL_CERT_GEN)
    unsigned char extra[10];
    int i;
    BIO* pub_bio = NULL;
    const unsigned char* server_key = (const unsigned char*)server_key_der_2048;
#endif

    ExpectTrue((file = XFOPEN(fname, "rb")) != XBADFILE);
    ExpectTrue(XFSEEK(file, 0, XSEEK_END) == 0);
    ExpectIntGT(sz = XFTELL(file), 0);
    ExpectTrue(XFSEEK(file, 0, XSEEK_SET) == 0);
    ExpectNotNull(buf = (byte*)XMALLOC(sz, NULL, DYNAMIC_TYPE_FILE));
    if (buf != NULL) {
        ExpectIntEQ(XFREAD(buf, 1, sz, file), sz);
    }
    if (file != XBADFILE) {
        XFCLOSE(file);
        file = XBADFILE;
    }

    /* Test using BIO new mem and loading PEM private key */
    ExpectNotNull(bio = BIO_new_mem_buf(buf, (int)sz));
    ExpectNotNull((pkey = PEM_read_bio_PrivateKey(bio, NULL, NULL, NULL)));
    XFREE(buf, NULL, DYNAMIC_TYPE_FILE);
    buf = NULL;
    BIO_free(bio);
    bio = NULL;

    /* New empty EVP_PKEY */
    ExpectNotNull(pkey2 = EVP_PKEY_new());
    if (pkey2 != NULL) {
        pkey2->type = EVP_PKEY_RSA;
    }
    /* Test parameter copy */
    ExpectIntEQ(EVP_PKEY_copy_parameters(pkey2, pkey), 0);
    EVP_PKEY_free(pkey2);
    EVP_PKEY_free(pkey);
    pkey  = NULL;

    /* Qt unit test case : rsa pkcs8 key */
    ExpectTrue((file = XFOPEN(fname_rsa_p8, "rb")) != XBADFILE);
    ExpectTrue(XFSEEK(file, 0, XSEEK_END) == 0);
    ExpectIntGT(sz = XFTELL(file), 0);
    ExpectTrue(XFSEEK(file, 0, XSEEK_SET) == 0);
    ExpectNotNull(buf = (byte*)XMALLOC(sz, NULL, DYNAMIC_TYPE_FILE));
    if (buf) {
        ExpectIntEQ(XFREAD(buf, 1, sz, file), sz);
    }
    if (file != XBADFILE) {
        XFCLOSE(file);
        file = XBADFILE;
    }

    ExpectNotNull(bio = BIO_new_mem_buf(buf, (int)sz));
    ExpectNotNull((pkey = PEM_read_bio_PrivateKey(bio, NULL, NULL, NULL)));
    XFREE(buf, NULL, DYNAMIC_TYPE_FILE);
    buf = NULL;
    BIO_free(bio);
    bio = NULL;
    ExpectNotNull(pkey3 = EVP_PKEY_new());

    ExpectNotNull(rsa_key = EVP_PKEY_get1_RSA(pkey));
    ExpectIntEQ(EVP_PKEY_set1_RSA(pkey3, rsa_key), WOLFSSL_SUCCESS);

#ifdef WOLFSSL_ERROR_CODE_OPENSSL
    ExpectIntEQ(EVP_PKEY_cmp(pkey, pkey3), 1/* match */);
#else
    ExpectIntEQ(EVP_PKEY_cmp(pkey, pkey3), 0);
#endif

    RSA_free(rsa_key);
    EVP_PKEY_free(pkey3);
    EVP_PKEY_free(pkey);
    pkey  = NULL;
    pkey2 = NULL;

#if defined(WOLFSSL_KEY_GEN) || defined(WOLFSSL_CERT_GEN)
    #define BIO_PEM_TEST_CHAR 'a'
    XMEMSET(extra, BIO_PEM_TEST_CHAR, sizeof(extra));

    ExpectNotNull(bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));
    ExpectIntEQ(BIO_set_write_buf_size(bio, 4096), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectNotNull(pub_bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));
    ExpectIntEQ(BIO_set_write_buf_size(pub_bio, 4096), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    ExpectNull(d2i_PrivateKey(EVP_PKEY_EC, &pkey, &server_key,
        (long)sizeof_server_key_der_2048));
    ExpectNull(pkey);

    ExpectNotNull(wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, &pkey, &server_key,
        (long)sizeof_server_key_der_2048));
    ExpectIntEQ(PEM_write_bio_PrivateKey(NULL, pkey, NULL, NULL, 0, NULL, NULL),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_bio_PrivateKey(bio,  NULL, NULL, NULL, 0, NULL, NULL),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_bio_PrivateKey(bio,  pkey, NULL, NULL, 0, NULL, NULL),
        WOLFSSL_SUCCESS);
    ExpectIntGT(BIO_pending(bio), 0);
    ExpectIntEQ(BIO_pending(bio), 1679);
    /* Check if the pubkey API writes only the public key */
#ifdef WOLFSSL_KEY_GEN
    ExpectIntEQ(PEM_write_bio_PUBKEY(NULL, pkey), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_bio_PUBKEY(pub_bio, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_bio_PUBKEY(pub_bio, pkey), WOLFSSL_SUCCESS);
    ExpectIntGT(BIO_pending(pub_bio), 0);
    /* Previously both the private key and the pubkey calls would write
     * out the private key and the PEM header was the only difference.
     * The public PEM should be significantly shorter than the
     * private key versison. */
    ExpectIntEQ(BIO_pending(pub_bio), 451);
#else
    /* Not supported. */
    ExpectIntEQ(PEM_write_bio_PUBKEY(pub_bio, pkey), 0);
#endif

    /* test creating new EVP_PKEY with good args */
    ExpectNotNull((pkey2 = PEM_read_bio_PrivateKey(bio, NULL, NULL, NULL)));
    if (pkey && pkey->pkey.ptr && pkey2 && pkey2->pkey.ptr) {
        ExpectIntEQ((int)XMEMCMP(pkey->pkey.ptr, pkey2->pkey.ptr,
            pkey->pkey_sz), 0);
    }

    /* test of reuse of EVP_PKEY */
    ExpectNull(PEM_read_bio_PrivateKey(bio, &pkey, NULL, NULL));
    ExpectIntEQ(BIO_pending(bio), 0);
    ExpectIntEQ(PEM_write_bio_PrivateKey(bio, pkey, NULL, NULL, 0, NULL, NULL),
            SSL_SUCCESS);
    /* add 10 extra bytes after PEM */
    ExpectIntEQ(BIO_write(bio, extra, 10), 10);
    ExpectNotNull(PEM_read_bio_PrivateKey(bio, &pkey, NULL, NULL));
    ExpectNotNull(pkey);
    if (pkey && pkey->pkey.ptr && pkey2 && pkey2->pkey.ptr) {
        ExpectIntEQ((int)XMEMCMP(pkey->pkey.ptr, pkey2->pkey.ptr,
            pkey->pkey_sz), 0);
    }
    /* check 10 extra bytes still there */
    ExpectIntEQ(BIO_pending(bio), 10);
    ExpectIntEQ(BIO_read(bio, extra, 10), 10);
    for (i = 0; i < 10; i++) {
        ExpectIntEQ(extra[i], BIO_PEM_TEST_CHAR);
    }

    BIO_free(pub_bio);
    BIO_free(bio);
    bio = NULL;
    EVP_PKEY_free(pkey);
    pkey  = NULL;
    EVP_PKEY_free(pkey2);
#endif /* WOLFSSL_KEY_GEN || WOLFSSL_CERT_GEN */
#endif /* OPENSSL_EXTRA && !NO_CERTS && !NO_RSA && USE_CERT_BUFFERS_2048 &&
        * !NO_FILESYSTEM && !NO_BIO */
    return EXPECT_RESULT();
}

/* test loading ECC key using BIO */
static int test_wolfSSL_PEM_PrivateKey_ecc(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && defined(HAVE_ECC) && \
    !defined(NO_FILESYSTEM) && !defined(NO_BIO)
    BIO*      bio = NULL;
    EVP_PKEY* pkey  = NULL;
    XFILE file = XBADFILE;
    const char* fname = "./certs/ecc-key.pem";
    const char* fname_ecc_p8  = "./certs/ecc-keyPkcs8.pem";

    size_t sz = 0;
    byte* buf = NULL;
    EVP_PKEY* pkey2 = NULL;
    EVP_PKEY* pkey3 = NULL;
    EC_KEY*   ec_key = NULL;
    int nid = 0;
    BIO* pub_bio = NULL;

    ExpectTrue((file = XFOPEN(fname, "rb")) != XBADFILE);
    ExpectTrue(XFSEEK(file, 0, XSEEK_END) == 0);
    ExpectIntGT(sz = XFTELL(file), 0);
    ExpectTrue(XFSEEK(file, 0, XSEEK_SET) == 0);
    ExpectNotNull(buf = (byte*)XMALLOC(sz, NULL, DYNAMIC_TYPE_FILE));
    if (buf) {
        ExpectIntEQ(XFREAD(buf, 1, sz, file), sz);
    }
    if (file != XBADFILE) {
        XFCLOSE(file);
        file = XBADFILE;
    }

    /* Test using BIO new mem and loading PEM private key */
    ExpectNotNull(bio = BIO_new_mem_buf(buf, (int)sz));
    ExpectNotNull((pkey = PEM_read_bio_PrivateKey(bio, NULL, NULL, NULL)));
    BIO_free(bio);
    bio = NULL;
    XFREE(buf, NULL, DYNAMIC_TYPE_FILE);
    buf = NULL;
    ExpectNotNull(bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));
    ExpectNotNull(pub_bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));
    ExpectIntEQ(PEM_write_bio_PrivateKey(bio, pkey, NULL, NULL, 0, NULL, NULL),
        WOLFSSL_SUCCESS);
    ExpectIntGT(BIO_pending(bio), 0);
    /* No parameters. */
    ExpectIntEQ(BIO_pending(bio), 227);
    /* Check if the pubkey API writes only the public key */
#ifdef WOLFSSL_KEY_GEN
    ExpectIntEQ(PEM_write_bio_PUBKEY(pub_bio, pkey), WOLFSSL_SUCCESS);
    ExpectIntGT(BIO_pending(pub_bio), 0);
    /* Previously both the private key and the pubkey calls would write
     * out the private key and the PEM header was the only difference.
     * The public PEM should be significantly shorter than the
     * private key versison. */
    ExpectIntEQ(BIO_pending(pub_bio), 178);
#endif
    BIO_free(pub_bio);
    BIO_free(bio);
    bio = NULL;
    ExpectNotNull(pkey2 = EVP_PKEY_new());
    ExpectNotNull(pkey3 = EVP_PKEY_new());
    if (pkey2 != NULL) {
         pkey2->type = EVP_PKEY_EC;
    }
    /* Test parameter copy */
    ExpectIntEQ(EVP_PKEY_copy_parameters(pkey2, pkey), 1);


    /* Qt unit test case 1*/
    ExpectNotNull(ec_key = EVP_PKEY_get1_EC_KEY(pkey));
    ExpectIntEQ(EVP_PKEY_set1_EC_KEY(pkey3, ec_key), WOLFSSL_SUCCESS);
    #ifdef WOLFSSL_ERROR_CODE_OPENSSL
    ExpectIntEQ(EVP_PKEY_cmp(pkey, pkey3), 1/* match */);
    #else
    ExpectIntEQ(EVP_PKEY_cmp(pkey, pkey3), 0);
    #endif
    /* Test default digest */
    ExpectIntEQ(EVP_PKEY_get_default_digest_nid(pkey, &nid), 1);
    ExpectIntEQ(nid, NID_sha256);
    EC_KEY_free(ec_key);
    ec_key = NULL;
    EVP_PKEY_free(pkey3);
    pkey3 = NULL;
    EVP_PKEY_free(pkey2);
    pkey2 = NULL;
    EVP_PKEY_free(pkey);
    pkey  = NULL;

    /* Qt unit test case ec pkcs8 key */
    ExpectTrue((file = XFOPEN(fname_ecc_p8, "rb")) != XBADFILE);
    ExpectTrue(XFSEEK(file, 0, XSEEK_END) == 0);
    ExpectIntGT(sz = XFTELL(file), 0);
    ExpectTrue(XFSEEK(file, 0, XSEEK_SET) == 0);
    ExpectNotNull(buf = (byte*)XMALLOC(sz, NULL, DYNAMIC_TYPE_FILE));
    if (buf) {
        ExpectIntEQ(XFREAD(buf, 1, sz, file), sz);
    }
    if (file != XBADFILE) {
        XFCLOSE(file);
        file = XBADFILE;
    }

    ExpectNotNull(bio = BIO_new_mem_buf(buf, (int)sz));
    ExpectNotNull((pkey = PEM_read_bio_PrivateKey(bio, NULL, NULL, NULL)));
    XFREE(buf, NULL, DYNAMIC_TYPE_FILE);
    buf = NULL;
    BIO_free(bio);
    bio = NULL;
    ExpectNotNull(pkey3 = EVP_PKEY_new());
    /* Qt unit test case */
    ExpectNotNull(ec_key = EVP_PKEY_get1_EC_KEY(pkey));
    ExpectIntEQ(EVP_PKEY_set1_EC_KEY(pkey3, ec_key), WOLFSSL_SUCCESS);
#ifdef WOLFSSL_ERROR_CODE_OPENSSL
    ExpectIntEQ(EVP_PKEY_cmp(pkey, pkey3), 1/* match */);
#else
    ExpectIntEQ(EVP_PKEY_cmp(pkey, pkey3), 0);
#endif
    EC_KEY_free(ec_key);
    EVP_PKEY_free(pkey3);
    EVP_PKEY_free(pkey);
    pkey  = NULL;
#endif
    return EXPECT_RESULT();
}

/* test loading DSA key using BIO */
static int test_wolfSSL_PEM_PrivateKey_dsa(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_DSA) && \
    !defined(NO_FILESYSTEM) && !defined(NO_BIO)
#if defined(WOLFSSL_QT) || defined(OPENSSL_ALL)
    BIO*      bio = NULL;
    EVP_PKEY* pkey  = NULL;

    ExpectNotNull(bio = BIO_new_file("./certs/dsa2048.pem", "rb"));
    /* Private DSA EVP_PKEY */
    ExpectNotNull(pkey = wolfSSL_PEM_read_bio_PrivateKey(bio, NULL, NULL,
        NULL));
    BIO_free(bio);
    bio = NULL;

    ExpectNotNull(bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));
#if defined(OPENSSL_ALL) && !defined(NO_PWDBASED) && defined(HAVE_PKCS8)
#ifdef WOLFSSL_ASN_TEMPLATE
    ExpectIntEQ(PEM_write_bio_PKCS8PrivateKey(bio, pkey, NULL, NULL, 0, NULL,
        NULL), 1216);
#else
    ExpectIntEQ(PEM_write_bio_PKCS8PrivateKey(bio, pkey, NULL, NULL, 0, NULL,
        NULL), 1212);
#endif
#endif

#ifdef WOLFSSL_KEY_GEN
    ExpectIntEQ(PEM_write_bio_PUBKEY(bio, pkey), 1);
#ifdef WOLFSSL_ASN_TEMPLATE
    ExpectIntEQ(BIO_pending(bio), 2394);
#else
    ExpectIntEQ(BIO_pending(bio), 2390);
#endif
    BIO_reset(bio);
#endif

    ExpectIntEQ(PEM_write_bio_PrivateKey(bio, pkey, NULL, NULL, 0, NULL, NULL),
        1);
    ExpectIntEQ(BIO_pending(bio), 1196);

    BIO_free(bio);
    bio = NULL;

    EVP_PKEY_free(pkey);
    pkey  = NULL;
#endif
#endif
    return EXPECT_RESULT();
}

/* test loading DH key using BIO */
static int test_wolfSSL_PEM_PrivateKey_dh(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_DH) && \
    !defined(NO_FILESYSTEM) && !defined(NO_BIO)
#if (defined(WOLFSSL_QT) || defined(OPENSSL_ALL) || \
     defined(WOLFSSL_OPENSSH)) && (!defined(HAVE_FIPS) || \
     (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2)))
    BIO*      bio = NULL;
    EVP_PKEY* pkey  = NULL;
    int       expectedBytes = 0;

    ExpectNotNull(bio = BIO_new_file("./certs/dh-priv-2048.pem", "rb"));
    /* Private DH EVP_PKEY */
    ExpectNotNull(pkey = wolfSSL_PEM_read_bio_PrivateKey(bio, NULL, NULL,
        NULL));
    BIO_free(bio);
    bio = NULL;

    ExpectNotNull(bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));

#if defined(OPENSSL_ALL) && !defined(NO_PWDBASED) && defined(HAVE_PKCS8)
    expectedBytes += 806;
    ExpectIntEQ(PEM_write_bio_PKCS8PrivateKey(bio, pkey, NULL, NULL, 0, NULL,
        NULL), expectedBytes);
#endif
#ifdef WOLFSSL_KEY_GEN
    ExpectIntEQ(PEM_write_bio_PUBKEY(bio, pkey), 0);
#endif

    ExpectIntEQ(PEM_write_bio_PrivateKey(bio, pkey, NULL, NULL, 0, NULL, NULL),
        1);
    expectedBytes += 806;
    ExpectIntEQ(BIO_pending(bio), expectedBytes);

    BIO_free(bio);
    bio = NULL;

    EVP_PKEY_free(pkey);
    pkey  = NULL;
#endif
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_PEM_PrivateKey(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_TLS) && \
    (!defined(NO_RSA) || defined(HAVE_ECC)) && defined(USE_CERT_BUFFERS_2048)
#ifndef NO_BIO
    BIO*      bio = NULL;
#endif
    EVP_PKEY* pkey  = NULL;
    const unsigned char* server_key = (const unsigned char*)server_key_der_2048;

#ifndef NO_BIO

    /* test creating new EVP_PKEY with bad arg */
    ExpectNull((pkey = PEM_read_bio_PrivateKey(NULL, NULL, NULL, NULL)));

    /* Test bad EVP_PKEY type. */
    /* New HMAC EVP_PKEY */
    ExpectNotNull(bio = BIO_new_mem_buf("", 1));
    ExpectNotNull(pkey = EVP_PKEY_new());
    if (pkey != NULL) {
        pkey->type = EVP_PKEY_HMAC;
    }
    ExpectIntEQ(PEM_write_bio_PrivateKey(bio, pkey, NULL, NULL, 0, NULL, NULL),
        0);
#if defined(OPENSSL_ALL) && !defined(NO_PWDBASED) && defined(HAVE_PKCS8)
    ExpectIntEQ(PEM_write_bio_PKCS8PrivateKey(bio, pkey, NULL, NULL, 0, NULL,
        NULL), 0);
#endif
#ifdef WOLFSSL_KEY_GEN
    ExpectIntEQ(PEM_write_bio_PUBKEY(bio, pkey), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
#endif
    EVP_PKEY_free(pkey);
    pkey = NULL;
    BIO_free(bio);
    bio = NULL;


    /* key is DES encrypted */
    #if !defined(NO_DES3) && defined(WOLFSSL_ENCRYPTED_KEYS) && \
        !defined(NO_RSA) && !defined(NO_BIO) && !defined(NO_FILESYSTEM) && \
        !defined(NO_MD5) && defined(WOLFSSL_KEY_GEN) && !defined(NO_RSA)
    {
        XFILE f = XBADFILE;
        wc_pem_password_cb* passwd_cb = NULL;
        void* passwd_cb_userdata;
        SSL_CTX* ctx = NULL;
        char passwd[] = "bad password";

    #ifndef WOLFSSL_NO_TLS12
        #ifndef NO_WOLFSSL_SERVER
        ExpectNotNull(ctx = SSL_CTX_new(TLSv1_2_server_method()));
        #else
        ExpectNotNull(ctx = SSL_CTX_new(TLSv1_2_client_method()));
        #endif
    #else
        #ifndef NO_WOLFSSL_SERVER
        ExpectNotNull(ctx = SSL_CTX_new(TLSv1_3_server_method()));
        #else
        ExpectNotNull(ctx = SSL_CTX_new(TLSv1_3_client_method()));
        #endif
    #endif

        ExpectNotNull(bio = BIO_new_file("./certs/server-keyEnc.pem", "rb"));
        SSL_CTX_set_default_passwd_cb(ctx, PasswordCallBack);
        ExpectNotNull(passwd_cb = SSL_CTX_get_default_passwd_cb(ctx));
        ExpectNull(passwd_cb_userdata =
            SSL_CTX_get_default_passwd_cb_userdata(ctx));

        /* fail case with password call back */
        ExpectNull(pkey = PEM_read_bio_PrivateKey(bio, NULL, NULL,
                    (void*)passwd));
        BIO_free(bio);
        ExpectNotNull(bio = BIO_new_file("./certs/server-keyEnc.pem", "rb"));
        ExpectNull(pkey = PEM_read_bio_PrivateKey(bio, NULL, passwd_cb,
                    (void*)passwd));
        BIO_free(bio);

        ExpectTrue((f = XFOPEN("./certs/server-keyEnc.pem", "rb")) != XBADFILE);
        ExpectNotNull(bio = BIO_new_fp(f, BIO_CLOSE));
        if ((bio == NULL) && (f != XBADFILE)) {
            XFCLOSE(f);
        }

        /* use callback that works */
        ExpectNotNull(pkey = PEM_read_bio_PrivateKey(bio, NULL, passwd_cb,
                (void*)"yassl123"));

        ExpectIntEQ(SSL_CTX_use_PrivateKey(ctx, pkey), SSL_SUCCESS);

        EVP_PKEY_free(pkey);
        pkey  = NULL;
        BIO_free(bio);
        bio = NULL;
        SSL_CTX_free(ctx);
    }
    #endif /* !defined(NO_DES3) */

#endif /* !NO_BIO */

    #if defined(HAVE_ECC) && !defined(NO_FILESYSTEM)
    {
        unsigned char buf[2048];
        size_t bytes = 0;
        XFILE f = XBADFILE;
        SSL_CTX* ctx = NULL;

    #ifndef WOLFSSL_NO_TLS12
        #ifndef NO_WOLFSSL_SERVER
        ExpectNotNull(ctx = SSL_CTX_new(TLSv1_2_server_method()));
        #else
        ExpectNotNull(ctx = SSL_CTX_new(TLSv1_2_client_method()));
        #endif
    #else
        #ifndef NO_WOLFSSL_SERVER
        ExpectNotNull(ctx = SSL_CTX_new(wolfTLSv1_3_server_method()));
        #else
        ExpectNotNull(ctx = SSL_CTX_new(wolfTLSv1_3_client_method()));
        #endif
    #endif

        ExpectTrue((f = XFOPEN("./certs/ecc-key.der", "rb")) != XBADFILE);
        ExpectIntGT(bytes = (size_t)XFREAD(buf, 1, sizeof(buf), f), 0);
        if (f != XBADFILE)
            XFCLOSE(f);

        server_key = buf;
        pkey = NULL;
        ExpectNull(d2i_PrivateKey(EVP_PKEY_RSA, &pkey, &server_key, (long int)bytes));
        ExpectNull(pkey);
        ExpectNotNull(d2i_PrivateKey(EVP_PKEY_EC, &pkey, &server_key, (long int)bytes));
        ExpectIntEQ(SSL_CTX_use_PrivateKey(ctx, pkey), SSL_SUCCESS);

        EVP_PKEY_free(pkey);
        pkey = NULL;
        SSL_CTX_free(ctx);
        server_key = NULL;
    }
    #endif

#ifndef NO_BIO
    (void)bio;
#endif
    (void)pkey;
    (void)server_key;
#endif /* OPENSSL_EXTRA && !NO_CERTS && !NO_RSA && USE_CERT_BUFFERS_2048 */
    return EXPECT_RESULT();
}

static int test_wolfSSL_PEM_file_RSAKey(void)
{
    EXPECT_DECLS;
#if (defined(OPENSSL_EXTRA) || defined(OPENSSL_ALL)) && \
    defined(WOLFSSL_KEY_GEN) && !defined(NO_RSA) && \
    !defined(NO_FILESYSTEM) && !defined(NO_CERTS)
    RSA* rsa = NULL;
    XFILE fp = XBADFILE;

    ExpectTrue((fp = XFOPEN("./certs/rsa-pub-2048.pem", "rb")) != XBADFILE);
    ExpectNotNull((rsa = PEM_read_RSA_PUBKEY(fp, NULL, NULL, NULL)));
    if (fp != XBADFILE)
        XFCLOSE(fp);
    ExpectIntEQ(RSA_size(rsa), 256);

    ExpectIntEQ(PEM_write_RSAPublicKey(XBADFILE, rsa), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_RSAPublicKey(stderr, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_RSAPublicKey(stderr, rsa), WOLFSSL_SUCCESS);

    ExpectIntEQ(PEM_write_RSA_PUBKEY(XBADFILE, rsa), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_RSA_PUBKEY(stderr, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_RSA_PUBKEY(stderr, rsa), WOLFSSL_SUCCESS);

    RSA_free(rsa);
#endif /* defined(OPENSSL_EXTRA) || defined(OPENSSL_ALL)) && \
         (defined(WOLFSSL_KEY_GEN) || WOLFSSL_CERT_GEN) && \
         !defined(NO_FILESYSTEM) && !defined(NO_RSA) && !defined(NO_CERTS) */
    return EXPECT_RESULT();
}

static int test_wolfSSL_PEM_file_RSAPrivateKey(void)
{
    EXPECT_DECLS;
#if !defined(NO_RSA) && defined(OPENSSL_EXTRA) && defined(WOLFSSL_KEY_GEN) && \
    !defined(NO_FILESYSTEM) && \
    (defined(WOLFSSL_PEM_TO_DER) || defined(WOLFSSL_DER_TO_PEM))
    RSA* rsa = NULL;
    XFILE f = NULL;

    ExpectTrue((f = XFOPEN(svrKeyFile, "rb")) != XBADFILE);
    ExpectNotNull((rsa = PEM_read_RSAPrivateKey(f, NULL, NULL, NULL)));
    ExpectIntEQ(RSA_size(rsa), 256);
    if (f != XBADFILE) {
        XFCLOSE(f);
        f = XBADFILE;
    }

    ExpectIntEQ(PEM_write_RSAPrivateKey(XBADFILE, rsa, NULL, NULL, 0, NULL,
        NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_RSAPrivateKey(stderr, NULL, NULL, NULL, 0, NULL,
        NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_RSAPrivateKey(stderr, rsa, NULL, NULL, 0, NULL, NULL),
        WOLFSSL_SUCCESS);

    RSA_free(rsa);

#ifdef HAVE_ECC
    ExpectTrue((f = XFOPEN(eccKeyFile, "rb")) != XBADFILE);
    ExpectNull((rsa = PEM_read_RSAPrivateKey(f, NULL, NULL, NULL)));
    if (f != XBADFILE)
        XFCLOSE(f);
#endif /* HAVE_ECC */
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_CERTS) */
    return EXPECT_RESULT();
}

static int test_wolfSSL_PEM_read_RSA_PUBKEY(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
       !defined(NO_FILESYSTEM) && !defined(NO_RSA)
    XFILE file = XBADFILE;
    const char* fname = "./certs/client-keyPub.pem";
    RSA *rsa = NULL;

    ExpectNull(wolfSSL_PEM_read_RSA_PUBKEY(XBADFILE, NULL, NULL, NULL));

    ExpectTrue((file = XFOPEN(fname, "rb")) != XBADFILE);
    ExpectNotNull((rsa = PEM_read_RSA_PUBKEY(file, NULL, NULL, NULL)));
    ExpectIntEQ(RSA_size(rsa), 256);
    RSA_free(rsa);
    if (file != XBADFILE)
       XFCLOSE(file);
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_CERTS) */
    return EXPECT_RESULT();
}

#ifndef NO_BIO
static int test_wolfSSL_PEM_bio_RSAKey(void)
{
    EXPECT_DECLS;
#if (defined(OPENSSL_EXTRA) || defined(OPENSSL_ALL)) && \
    defined(WOLFSSL_KEY_GEN) && !defined(NO_RSA) && \
    !defined(NO_FILESYSTEM) && !defined(NO_CERTS)
    RSA* rsa = NULL;
    BIO* bio = NULL;

    /* PrivateKey */
    ExpectNotNull(bio = BIO_new_file(svrKeyFile, "rb"));
    ExpectNull((rsa = PEM_read_bio_RSAPrivateKey(NULL, NULL, NULL, NULL)));
    ExpectNotNull(PEM_read_bio_RSAPrivateKey(bio, &rsa, NULL, NULL));
    ExpectNotNull(rsa);
    ExpectIntEQ(RSA_size(rsa), 256);
    ExpectIntEQ(PEM_write_bio_RSAPrivateKey(NULL, NULL, NULL, NULL, 0, NULL, \
                                            NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    BIO_free(bio);
    bio = NULL;
    ExpectNotNull(bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));
    ExpectIntEQ(PEM_write_bio_RSAPrivateKey(bio, rsa, NULL, NULL, 0, NULL, \
                                            NULL), WOLFSSL_SUCCESS);
    BIO_free(bio);
    bio = NULL;
    RSA_free(rsa);
    rsa = NULL;

    /* PUBKEY */
    ExpectNotNull(bio = BIO_new_file("./certs/rsa-pub-2048.pem", "rb"));
    ExpectNull((rsa = PEM_read_bio_RSA_PUBKEY(NULL, NULL, NULL, NULL)));
    ExpectNotNull((rsa = PEM_read_bio_RSA_PUBKEY(bio, NULL, NULL, NULL)));
    ExpectIntEQ(RSA_size(rsa), 256);
    ExpectIntEQ(PEM_write_bio_RSA_PUBKEY(NULL, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    BIO_free(bio);
    bio = NULL;
    ExpectNotNull(bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));
    ExpectIntEQ(PEM_write_bio_RSA_PUBKEY(bio, rsa), WOLFSSL_SUCCESS);
    BIO_free(bio);
    bio = NULL;

    RSA_free(rsa);
    rsa = NULL;

    /* Ensure that keys beginning with BEGIN RSA PUBLIC KEY can be read, too. */
    ExpectNotNull(bio = BIO_new_file("./certs/server-keyPub.pem", "rb"));
    ExpectNotNull((rsa = PEM_read_bio_RSA_PUBKEY(bio, NULL, NULL, NULL)));
    BIO_free(bio);
    bio = NULL;
    RSA_free(rsa);
    rsa = NULL;

    #ifdef HAVE_ECC
    /* ensure that non-rsa keys do not work */
    ExpectNotNull(bio = BIO_new_file(eccKeyFile, "rb")); /* ecc key */
    ExpectNull((rsa = PEM_read_bio_RSAPrivateKey(bio, NULL, NULL, NULL)));
    ExpectNull((rsa = PEM_read_bio_RSA_PUBKEY(bio, NULL, NULL, NULL)));
    BIO_free(bio);
    bio = NULL;
    RSA_free(rsa);
    rsa = NULL;
    #endif /* HAVE_ECC */
#endif /* defined(OPENSSL_EXTRA) || defined(OPENSSL_ALL)) && \
         (defined(WOLFSSL_KEY_GEN) || WOLFSSL_CERT_GEN) && \
         !defined(NO_FILESYSTEM) && !defined(NO_RSA) && !defined(NO_CERTS) */
    return EXPECT_RESULT();
}

static int test_wolfSSL_PEM_bio_RSAPrivateKey(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
   !defined(NO_FILESYSTEM) && !defined(NO_RSA)
    RSA* rsa = NULL;
    RSA* rsa_dup = NULL;
    BIO* bio = NULL;

    ExpectNotNull(bio = BIO_new_file(svrKeyFile, "rb"));
    ExpectNotNull((rsa = PEM_read_bio_RSAPrivateKey(bio, NULL, NULL, NULL)));
    ExpectIntEQ(RSA_size(rsa), 256);

#if defined(WOLFSSL_KEY_GEN) && !defined(NO_RSA)
    ExpectNull(rsa_dup = RSAPublicKey_dup(NULL));
    /* Test duplicating empty key. */
    ExpectNotNull(rsa_dup = RSA_new());
    ExpectNull(RSAPublicKey_dup(rsa_dup));
    RSA_free(rsa_dup);
    rsa_dup = NULL;
    ExpectNotNull(rsa_dup = RSAPublicKey_dup(rsa));
    ExpectPtrNE(rsa_dup, rsa);
#endif

    /* test if valgrind complains about unreleased memory */
    RSA_up_ref(rsa);
    RSA_free(rsa);

    BIO_free(bio);
    bio = NULL;
    RSA_free(rsa);
    rsa = NULL;
    RSA_free(rsa_dup);
    rsa_dup = NULL;

#ifdef HAVE_ECC
    ExpectNotNull(bio = BIO_new_file(eccKeyFile, "rb"));
    ExpectNull((rsa = PEM_read_bio_RSAPrivateKey(bio, NULL, NULL, NULL)));

    BIO_free(bio);
#endif /* HAVE_ECC */
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_CERTS) */
    return EXPECT_RESULT();
}

static int test_wolfSSL_PEM_bio_DSAKey(void)
{
    EXPECT_DECLS;
#ifndef HAVE_SELFTEST
#if (defined(WOLFSSL_QT) || defined(OPENSSL_ALL)) && !defined(NO_CERTS) && \
    defined(WOLFSSL_KEY_GEN) && !defined(NO_FILESYSTEM) && !defined(NO_DSA)
    DSA* dsa = NULL;
    BIO* bio = NULL;

    /* PrivateKey */
    ExpectNotNull(bio = BIO_new_file("./certs/1024/dsa1024.pem", "rb"));
    ExpectNull((dsa = PEM_read_bio_DSAPrivateKey(NULL, NULL, NULL, NULL)));
    ExpectNotNull((dsa = PEM_read_bio_DSAPrivateKey(bio, NULL, NULL, NULL)));
    ExpectIntEQ(BN_num_bytes(dsa->g), 128);
    ExpectIntEQ(PEM_write_bio_DSAPrivateKey(NULL, NULL, NULL, NULL, 0, NULL,
        NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    BIO_free(bio);
    bio = NULL;
    ExpectNotNull(bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));
    ExpectIntEQ(PEM_write_bio_DSAPrivateKey(bio, dsa, NULL, NULL, 0, NULL,
        NULL), WOLFSSL_SUCCESS);
    BIO_free(bio);
    bio = NULL;
    DSA_free(dsa);
    dsa = NULL;

    /* PUBKEY */
    ExpectNotNull(bio = BIO_new_file("./certs/1024/dsa-pub-1024.pem", "rb"));
    ExpectNull((dsa = PEM_read_bio_DSA_PUBKEY(NULL, NULL, NULL, NULL)));
    ExpectNotNull((dsa = PEM_read_bio_DSA_PUBKEY(bio, NULL, NULL, NULL)));
    ExpectIntEQ(BN_num_bytes(dsa->g), 128);
    ExpectIntEQ(PEM_write_bio_DSA_PUBKEY(NULL, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    BIO_free(bio);
    bio = NULL;
    ExpectNotNull(bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));
    ExpectIntEQ(PEM_write_bio_DSA_PUBKEY(bio, dsa), WOLFSSL_SUCCESS);
    BIO_free(bio);
    bio = NULL;
    DSA_free(dsa);
    dsa = NULL;

    #ifdef HAVE_ECC
    /* ensure that non-dsa keys do not work */
    ExpectNotNull(bio = BIO_new_file(eccKeyFile, "rb")); /* ecc key */
    ExpectNull((dsa = PEM_read_bio_DSAPrivateKey(bio, NULL, NULL, NULL)));
    ExpectNull((dsa = PEM_read_bio_DSA_PUBKEY(bio, NULL, NULL, NULL)));
    BIO_free(bio);
    bio = NULL;
    DSA_free(dsa);
    dsa = NULL;
    #endif /* HAVE_ECC */
#endif /* defined(WOLFSSL_QT) || defined(OPENSSL_ALL)) && \
         !defined(NO_CERTS) && defined(WOLFSSL_KEY_GEN) && \
         !defined(NO_FILESYSTEM) && !defined(NO_DSA) */
#endif /* HAVE_SELFTEST */
    return EXPECT_RESULT();
}

static int test_wolfSSL_PEM_bio_ECKey(void)
{
    EXPECT_DECLS;
#if (defined(OPENSSL_EXTRA) || defined(OPENSSL_ALL)) && \
    defined(WOLFSSL_KEY_GEN) && !defined(NO_FILESYSTEM) && defined(HAVE_ECC)
    EC_KEY* ec = NULL;
    EC_KEY* ec2;
    BIO* bio = NULL;
#if defined(WOLFSSL_PEM_TO_DER) || defined(WOLFSSL_DER_TO_PEM)
    unsigned char* pem = NULL;
    int pLen;
#endif
    static char ec_key_bad_1[] = "-----BEGIN PUBLIC KEY-----\n"
                                 "MAA=\n"
                                 "-----END PUBLIC KEY-----";
    static char ec_priv_key_bad_1[] = "-----BEGIN EC PRIVATE KEY-----\n"
                                      "MAA=\n"
                                      "-----END EC PRIVATE KEY-----";

    /* PrivateKey */
    ExpectNotNull(bio = BIO_new_file("./certs/ecc-key.pem", "rb"));
    ExpectNull((ec = PEM_read_bio_ECPrivateKey(NULL, NULL, NULL, NULL)));
    ec2 = NULL;
    ExpectNotNull((ec = PEM_read_bio_ECPrivateKey(bio, &ec2, NULL, NULL)));
    ExpectIntEQ(ec == ec2, 1);
    ExpectIntEQ(wc_ecc_size((ecc_key*)ec->internal), 32);
    ExpectIntEQ(PEM_write_bio_ECPrivateKey(NULL, NULL, NULL, NULL, 0, NULL,
        NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_bio_ECPrivateKey(bio, NULL, NULL, NULL, 0, NULL,
        NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_bio_ECPrivateKey(NULL, ec, NULL, NULL, 0, NULL, NULL),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    BIO_free(bio);
    bio = NULL;
    /* Public key data - fail. */
    ExpectNotNull(bio = BIO_new_file("./certs/ecc-client-keyPub.pem", "rb"));
    ExpectNull(PEM_read_bio_ECPrivateKey(bio, NULL, NULL, NULL));
    BIO_free(bio);
    bio = NULL;
    ExpectNotNull(bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));
    ExpectIntEQ(PEM_write_bio_ECPrivateKey(bio, ec, NULL, NULL, 0, NULL, \
                                           NULL), WOLFSSL_SUCCESS);
    BIO_free(bio);
    bio = NULL;

    ExpectIntEQ(PEM_write_ECPrivateKey(XBADFILE, NULL, NULL, NULL, 0, NULL,
        NULL),WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_ECPrivateKey(stderr, NULL, NULL, NULL, 0, NULL, NULL),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_ECPrivateKey(XBADFILE, ec, NULL, NULL, 0, NULL, NULL),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_ECPrivateKey(stderr, ec, NULL, NULL, 0, NULL, NULL),
        WOLFSSL_SUCCESS);

    ExpectIntEQ(wolfSSL_PEM_write_mem_ECPrivateKey(NULL, NULL, NULL, 0, NULL,
        NULL), 0);
#if defined(WOLFSSL_PEM_TO_DER) || defined(WOLFSSL_DER_TO_PEM)
    ExpectIntEQ(wolfSSL_PEM_write_mem_ECPrivateKey(ec, NULL, NULL, 0, NULL,
        NULL), 0);
    ExpectIntEQ(wolfSSL_PEM_write_mem_ECPrivateKey(NULL, NULL, NULL, 0, &pem,
        NULL), 0);
    ExpectIntEQ(wolfSSL_PEM_write_mem_ECPrivateKey(NULL, NULL, NULL, 0, NULL,
        &pLen), 0);
    ExpectIntEQ(wolfSSL_PEM_write_mem_ECPrivateKey(NULL, NULL, NULL, 0, &pem,
        &pLen), 0);
    ExpectIntEQ(wolfSSL_PEM_write_mem_ECPrivateKey(ec, NULL, NULL, 0, NULL,
        &pLen), 0);
    ExpectIntEQ(wolfSSL_PEM_write_mem_ECPrivateKey(ec, NULL, NULL, 0, &pem,
        NULL), 0);
    ExpectIntEQ(wolfSSL_PEM_write_mem_ECPrivateKey(ec, NULL, NULL, 0, &pem,
        &pLen), 1);
    ExpectIntGT(pLen, 0);
    XFREE(pem, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    EC_KEY_free(ec);
    ec = NULL;

    /* PUBKEY */
    ExpectNotNull(bio = BIO_new_file("./certs/ecc-client-keyPub.pem", "rb"));
    ExpectNull((ec = PEM_read_bio_EC_PUBKEY(NULL, NULL, NULL, NULL)));
    ec2 = NULL;
    ExpectNotNull((ec = PEM_read_bio_EC_PUBKEY(bio, &ec2, NULL, NULL)));
    ExpectIntEQ(ec == ec2, 1);
    ExpectIntEQ(wc_ecc_size((ecc_key*)ec->internal), 32);
    ExpectIntEQ(PEM_write_bio_EC_PUBKEY(NULL, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    BIO_free(bio);
    bio = NULL;
    /* Test 0x30, 0x00 fails. */
    ExpectNotNull(bio = BIO_new_mem_buf((unsigned char*)ec_key_bad_1,
        sizeof(ec_key_bad_1)));
    ExpectNull(PEM_read_bio_EC_PUBKEY(bio, NULL, NULL, NULL));
    BIO_free(bio);
    bio = NULL;

    /* Private key data - fail. */
    ExpectNotNull(bio = BIO_new_file("./certs/ecc-key.pem", "rb"));
    ExpectNull(PEM_read_bio_EC_PUBKEY(bio, NULL, NULL, NULL));
    BIO_free(bio);
    bio = NULL;
    ExpectNotNull(bio = wolfSSL_BIO_new(wolfSSL_BIO_s_mem()));
    ExpectIntEQ(PEM_write_bio_EC_PUBKEY(bio, ec), WOLFSSL_SUCCESS);
    BIO_free(bio);
    bio = NULL;

    /* Same test as above, but with a file pointer rather than a BIO. */
    ExpectIntEQ(PEM_write_EC_PUBKEY(NULL, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_EC_PUBKEY(NULL, ec), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_EC_PUBKEY(stderr, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ(PEM_write_EC_PUBKEY(stderr, ec), WOLFSSL_SUCCESS);

    EC_KEY_free(ec);
    ec = NULL;

    #ifndef NO_RSA
    /* ensure that non-ec keys do not work */
    ExpectNotNull(bio = BIO_new_file(svrKeyFile, "rb")); /* rsa key */
    ExpectNull((ec = PEM_read_bio_ECPrivateKey(bio, NULL, NULL, NULL)));
    ExpectNull((ec = PEM_read_bio_EC_PUBKEY(bio, NULL, NULL, NULL)));
    BIO_free(bio);
    bio = NULL;
    EC_KEY_free(ec);
    ec = NULL;
    #endif /* !NO_RSA */
    /* Test 0x30, 0x00 fails. */
    ExpectNotNull(bio = BIO_new_mem_buf((unsigned char*)ec_priv_key_bad_1,
        sizeof(ec_priv_key_bad_1)));
    ExpectNull(PEM_read_bio_ECPrivateKey(bio, NULL, NULL, NULL));
    BIO_free(bio);
    bio = NULL;
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_CERTS) */
    return EXPECT_RESULT();
}

static int test_wolfSSL_PEM_PUBKEY(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(HAVE_ECC)
    BIO*      bio = NULL;
    EVP_PKEY* pkey  = NULL;

    /* test creating new EVP_PKEY with bad arg */
    ExpectNull((pkey = PEM_read_bio_PUBKEY(NULL, NULL, NULL, NULL)));

    /* test loading ECC key using BIO */
#if defined(HAVE_ECC) && !defined(NO_FILESYSTEM)
    {
        XFILE file = XBADFILE;
        const char* fname = "./certs/ecc-client-keyPub.pem";
        size_t sz = 0;
        byte* buf = NULL;

        EVP_PKEY* pkey2 = NULL;
        EC_KEY*   ec_key = NULL;

        ExpectTrue((file = XFOPEN(fname, "rb")) != XBADFILE);
        ExpectIntEQ(XFSEEK(file, 0, XSEEK_END), 0);
        ExpectIntGT(sz = XFTELL(file), 0);
        ExpectIntEQ(XFSEEK(file, 0, XSEEK_SET), 0);
        ExpectNotNull(buf = (byte*)XMALLOC(sz, NULL, DYNAMIC_TYPE_FILE));
        if (buf != NULL) {
            ExpectIntEQ(XFREAD(buf, 1, sz, file), sz);
        }
        if (file != XBADFILE) {
            XFCLOSE(file);
        }

        /* Test using BIO new mem and loading PEM private key */
        ExpectNotNull(bio = BIO_new_mem_buf(buf, (int)sz));
        ExpectNotNull((pkey = PEM_read_bio_PUBKEY(bio, NULL, NULL, NULL)));
        BIO_free(bio);
        bio = NULL;
        EVP_PKEY_free(pkey);
        pkey = NULL;
        ExpectNotNull(bio = BIO_new_mem_buf(buf, (int)sz));
        ExpectNotNull(pkey = EVP_PKEY_new());
        ExpectPtrEq(PEM_read_bio_PUBKEY(bio, &pkey, NULL, NULL), pkey);
        XFREE(buf, NULL, DYNAMIC_TYPE_FILE);
        BIO_free(bio);
        bio = NULL;

        /* Qt unit test case*/
        ExpectNotNull(pkey2 = EVP_PKEY_new());
        ExpectNotNull(ec_key = EVP_PKEY_get1_EC_KEY(pkey));
        ExpectIntEQ(EVP_PKEY_set1_EC_KEY(pkey2, ec_key), WOLFSSL_SUCCESS);
    #ifdef WOLFSSL_ERROR_CODE_OPENSSL
        ExpectIntEQ(EVP_PKEY_cmp(pkey, pkey2), 1/* match */);
    #else
        ExpectIntEQ(EVP_PKEY_cmp(pkey, pkey2), 0);
    #endif

        EC_KEY_free(ec_key);
        EVP_PKEY_free(pkey2);
        EVP_PKEY_free(pkey);
        pkey = NULL;
    }
#endif

    (void)bio;
    (void)pkey;
#endif
    return EXPECT_RESULT();
}

#endif /* !NO_BIO */

static int test_DSA_do_sign_verify(void)
{
    EXPECT_DECLS;
#if !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
#if defined(OPENSSL_EXTRA) && !defined(NO_FILESYSTEM) && \
    !defined(NO_DSA)
    unsigned char digest[WC_SHA_DIGEST_SIZE];
    DSA_SIG* sig = NULL;
    DSA* dsa = NULL;
    word32  bytes;
    byte sigBin[DSA_SIG_SIZE];
    int dsacheck;

#ifdef USE_CERT_BUFFERS_1024
    byte    tmp[ONEK_BUF];

    XMEMSET(tmp, 0, sizeof(tmp));
    XMEMCPY(tmp, dsa_key_der_1024, sizeof_dsa_key_der_1024);
    bytes = sizeof_dsa_key_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    byte    tmp[TWOK_BUF];

    XMEMSET(tmp, 0, sizeof(tmp));
    XMEMCPY(tmp, dsa_key_der_2048, sizeof_dsa_key_der_2048);
    bytes = sizeof_dsa_key_der_2048;
#else
    byte    tmp[TWOK_BUF];
    XFILE   fp = XBADFILE;

    XMEMSET(tmp, 0, sizeof(tmp));
    ExpectTrue((fp = XFOPEN("./certs/dsa2048.der", "rb") != XBADFILE);
    ExpectIntGT(bytes = (word32) XFREAD(tmp, 1, sizeof(tmp), fp), 0);
    if (fp != XBADFILE)
        XFCLOSE(fp);
#endif /* END USE_CERT_BUFFERS_1024 */

    XMEMSET(digest, 202, sizeof(digest));

    ExpectNotNull(dsa = DSA_new());
    ExpectIntEQ(DSA_LoadDer(dsa, tmp, (int)bytes), 1);

    ExpectIntEQ(wolfSSL_DSA_do_sign(digest, sigBin, dsa), 1);
    ExpectIntEQ(wolfSSL_DSA_do_verify(digest, sigBin, dsa, &dsacheck), 1);

    ExpectNotNull(sig = DSA_do_sign(digest, WC_SHA_DIGEST_SIZE, dsa));
    ExpectIntEQ(DSA_do_verify(digest, WC_SHA_DIGEST_SIZE, sig, dsa), 1);

    DSA_SIG_free(sig);
    DSA_free(dsa);
#endif
#endif /* !HAVE_SELFTEST && !HAVE_FIPS */
    return EXPECT_RESULT();
}

static int test_wolfSSL_tmp_dh(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_FILESYSTEM) && \
    !defined(NO_RSA) && !defined(NO_DH) && !defined(NO_BIO) && \
    !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    byte buff[6000];
    static const unsigned char p[] = {
        0xb0, 0xa1, 0x08, 0x06, 0x9c, 0x08, 0x13, 0xba,
        0x59, 0x06, 0x3c, 0xbc, 0x30, 0xd5, 0xf5, 0x00,
        0xc1, 0x4f, 0x44, 0xa7, 0xd6, 0xef, 0x4a, 0xc6,
        0x25, 0x27, 0x1c, 0xe8, 0xd2, 0x96, 0x53, 0x0a,
        0x5c, 0x91, 0xdd, 0xa2, 0xc2, 0x94, 0x84, 0xbf,
        0x7d, 0xb2, 0x44, 0x9f, 0x9b, 0xd2, 0xc1, 0x8a,
        0xc5, 0xbe, 0x72, 0x5c, 0xa7, 0xe7, 0x91, 0xe6,
        0xd4, 0x9f, 0x73, 0x07, 0x85, 0x5b, 0x66, 0x48,
        0xc7, 0x70, 0xfa, 0xb4, 0xee, 0x02, 0xc9, 0x3d,
        0x9a, 0x4a, 0xda, 0x3d, 0xc1, 0x46, 0x3e, 0x19,
        0x69, 0xd1, 0x17, 0x46, 0x07, 0xa3, 0x4d, 0x9f,
        0x2b, 0x96, 0x17, 0x39, 0x6d, 0x30, 0x8d, 0x2a,
        0xf3, 0x94, 0xd3, 0x75, 0xcf, 0xa0, 0x75, 0xe6,
        0xf2, 0x92, 0x1f, 0x1a, 0x70, 0x05, 0xaa, 0x04,
        0x83, 0x57, 0x30, 0xfb, 0xda, 0x76, 0x93, 0x38,
        0x50, 0xe8, 0x27, 0xfd, 0x63, 0xee, 0x3c, 0xe5,
        0xb7, 0xc8, 0x09, 0xae, 0x6f, 0x50, 0x35, 0x8e,
        0x84, 0xce, 0x4a, 0x00, 0xe9, 0x12, 0x7e, 0x5a,
        0x31, 0xd7, 0x33, 0xfc, 0x21, 0x13, 0x76, 0xcc,
        0x16, 0x30, 0xdb, 0x0c, 0xfc, 0xc5, 0x62, 0xa7,
        0x35, 0xb8, 0xef, 0xb7, 0xb0, 0xac, 0xc0, 0x36,
        0xf6, 0xd9, 0xc9, 0x46, 0x48, 0xf9, 0x40, 0x90,
        0x00, 0x2b, 0x1b, 0xaa, 0x6c, 0xe3, 0x1a, 0xc3,
        0x0b, 0x03, 0x9e, 0x1b, 0xc2, 0x46, 0xe4, 0x48,
        0x4e, 0x22, 0x73, 0x6f, 0xc3, 0x5f, 0xd4, 0x9a,
        0xd6, 0x30, 0x07, 0x48, 0xd6, 0x8c, 0x90, 0xab,
        0xd4, 0xf6, 0xf1, 0xe3, 0x48, 0xd3, 0x58, 0x4b,
        0xa6, 0xb9, 0xcd, 0x29, 0xbf, 0x68, 0x1f, 0x08,
        0x4b, 0x63, 0x86, 0x2f, 0x5c, 0x6b, 0xd6, 0xb6,
        0x06, 0x65, 0xf7, 0xa6, 0xdc, 0x00, 0x67, 0x6b,
        0xbb, 0xc3, 0xa9, 0x41, 0x83, 0xfb, 0xc7, 0xfa,
        0xc8, 0xe2, 0x1e, 0x7e, 0xaf, 0x00, 0x3f, 0x93
    };
    int pSz = (int)sizeof(p);
#if !defined(WOLFSSL_OLD_PRIME_CHECK) && !defined(HAVE_FIPS) && \
    !defined(HAVE_SELFTEST)
    static const unsigned char bad_p[] = {
        0xb0, 0xa1, 0x08, 0x06, 0x9c, 0x08, 0x13, 0xba,
        0x59, 0x06, 0x3c, 0xbc, 0x30, 0xd5, 0xf5, 0x00,
        0xc1, 0x4f, 0x44, 0xa7, 0xd6, 0xef, 0x4a, 0xc6,
        0x25, 0x27, 0x1c, 0xe8, 0xd2, 0x96, 0x53, 0x0a,
        0x5c, 0x91, 0xdd, 0xa2, 0xc2, 0x94, 0x84, 0xbf,
        0x7d, 0xb2, 0x44, 0x9f, 0x9b, 0xd2, 0xc1, 0x8a,
        0xc5, 0xbe, 0x72, 0x5c, 0xa7, 0xe7, 0x91, 0xe6,
        0xd4, 0x9f, 0x73, 0x07, 0x85, 0x5b, 0x66, 0x48,
        0xc7, 0x70, 0xfa, 0xb4, 0xee, 0x02, 0xc9, 0x3d,
        0x9a, 0x4a, 0xda, 0x3d, 0xc1, 0x46, 0x3e, 0x19,
        0x69, 0xd1, 0x17, 0x46, 0x07, 0xa3, 0x4d, 0x9f,
        0x2b, 0x96, 0x17, 0x39, 0x6d, 0x30, 0x8d, 0x2a,
        0xf3, 0x94, 0xd3, 0x75, 0xcf, 0xa0, 0x75, 0xe6,
        0xf2, 0x92, 0x1f, 0x1a, 0x70, 0x05, 0xaa, 0x04,
        0x83, 0x57, 0x30, 0xfb, 0xda, 0x76, 0x93, 0x38,
        0x50, 0xe8, 0x27, 0xfd, 0x63, 0xee, 0x3c, 0xe5,
        0xb7, 0xc8, 0x09, 0xae, 0x6f, 0x50, 0x35, 0x8e,
        0x84, 0xce, 0x4a, 0x00, 0xe9, 0x12, 0x7e, 0x5a,
        0x31, 0xd7, 0x33, 0xfc, 0x21, 0x13, 0x76, 0xcc,
        0x16, 0x30, 0xdb, 0x0c, 0xfc, 0xc5, 0x62, 0xa7,
        0x35, 0xb8, 0xef, 0xb7, 0xb0, 0xac, 0xc0, 0x36,
        0xf6, 0xd9, 0xc9, 0x46, 0x48, 0xf9, 0x40, 0x90,
        0x00, 0x2b, 0x1b, 0xaa, 0x6c, 0xe3, 0x1a, 0xc3,
        0x0b, 0x03, 0x9e, 0x1b, 0xc2, 0x46, 0xe4, 0x48,
        0x4e, 0x22, 0x73, 0x6f, 0xc3, 0x5f, 0xd4, 0x9a,
        0xd6, 0x30, 0x07, 0x48, 0xd6, 0x8c, 0x90, 0xab,
        0xd4, 0xf6, 0xf1, 0xe3, 0x48, 0xd3, 0x58, 0x4b,
        0xa6, 0xb9, 0xcd, 0x29, 0xbf, 0x68, 0x1f, 0x08,
        0x4b, 0x63, 0x86, 0x2f, 0x5c, 0x6b, 0xd6, 0xb6,
        0x06, 0x65, 0xf7, 0xa6, 0xdc, 0x00, 0x67, 0x6b,
        0xbb, 0xc3, 0xa9, 0x41, 0x83, 0xfb, 0xc7, 0xfa,
        0xc8, 0xe2, 0x1e, 0x7e, 0xaf, 0x00, 0x3f, 0x91
    };
#endif
    static const unsigned char g[] = { 0x02 };
    int gSz = (int)sizeof(g);
#if !defined(NO_DSA)
    char file[] = "./certs/dsaparams.pem";
    DSA* dsa = NULL;
#else
    char file[] = "./certs/dh2048.pem";
#endif
    XFILE f = XBADFILE;
    int  bytes = 0;
    DH*  dh = NULL;
    DH*  dh2 = NULL;
    BIO*     bio = NULL;
    SSL*     ssl = NULL;
    SSL_CTX* ctx = NULL;
#ifndef NO_WOLFSSL_CLIENT
    SSL*     ssl_c = NULL;
    SSL_CTX* ctx_c = NULL;
#endif

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_server_method()));
    ExpectTrue(SSL_CTX_use_certificate_file(ctx, svrCertFile,
        WOLFSSL_FILETYPE_PEM));
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
        WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(ssl = SSL_new(ctx));
#endif
#ifndef NO_WOLFSSL_CLIENT
    ExpectNotNull(ctx_c = SSL_CTX_new(wolfSSLv23_client_method()));
    ExpectTrue(SSL_CTX_use_certificate_file(ctx_c, svrCertFile,
        WOLFSSL_FILETYPE_PEM));
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx_c, svrKeyFile,
        WOLFSSL_FILETYPE_PEM));
    ExpectNotNull(ssl_c = SSL_new(ctx_c));
#ifdef NO_WOLFSSL_SERVER
    ctx = ctx_c;
    ssl = ssl_c;
#endif
#endif

    XMEMSET(buff, 0, sizeof(buff));
    ExpectTrue((f = XFOPEN(file, "rb")) != XBADFILE);
    ExpectIntGT(bytes = (int)XFREAD(buff, 1, sizeof(buff), f), 0);
    if (f != XBADFILE)
        XFCLOSE(f);

    ExpectNotNull(bio = BIO_new_mem_buf((void*)buff, bytes));

#if !defined(NO_DSA)
    dsa = wolfSSL_PEM_read_bio_DSAparams(bio, NULL, NULL, NULL);
    ExpectNotNull(dsa);

    dh = wolfSSL_DSA_dup_DH(dsa);
#else
    dh = wolfSSL_PEM_read_bio_DHparams(bio, NULL, NULL, NULL);
#endif
    ExpectNotNull(dh);
#if defined(WOLFSSL_DH_EXTRA) && \
    (defined(WOLFSSL_QT) || defined(OPENSSL_ALL) || defined(WOLFSSL_OPENSSH))
    ExpectNotNull(dh2 = wolfSSL_DH_dup(dh));
    DH_free(dh2);
    dh2 = NULL;
#endif

    /* Failure cases */
    ExpectIntEQ((int)wolfSSL_CTX_SetTmpDH(NULL, NULL, 0, NULL, 0),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ((int)wolfSSL_CTX_SetTmpDH(ctx , NULL, 0, NULL, 0),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ((int)wolfSSL_CTX_SetTmpDH(NULL, p   , 0, NULL, 0),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ((int)wolfSSL_CTX_SetTmpDH(NULL, NULL, 0, g   , 0),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ((int)wolfSSL_CTX_SetTmpDH(ctx , p  , 0, NULL, 0),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ((int)wolfSSL_CTX_SetTmpDH(ctx , NULL, 0, g   , 0),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ((int)wolfSSL_CTX_SetTmpDH(NULL, p   , 0, g   , 0),
        WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ((int)wolfSSL_CTX_SetTmpDH(ctx , p   , 1, g   , 1),
        WC_NO_ERR_TRACE(DH_KEY_SIZE_E));
    ExpectIntEQ((int)wolfSSL_CTX_SetTmpDH(ctx , buff, 6000, g   , 1),
        WC_NO_ERR_TRACE(DH_KEY_SIZE_E));
#if !defined(WOLFSSL_OLD_PRIME_CHECK) && !defined(HAVE_FIPS) && \
    !defined(HAVE_SELFTEST)
    ExpectIntEQ((int)wolfSSL_CTX_SetTmpDH(ctx, bad_p, pSz, g, gSz),
        WC_NO_ERR_TRACE(DH_CHECK_PUB_E));
#endif
    ExpectIntEQ((int)wolfSSL_SetTmpDH(NULL, NULL, 0, NULL, 0),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ((int)wolfSSL_SetTmpDH(ssl , NULL, 0, NULL, 0),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ((int)wolfSSL_SetTmpDH(NULL, p   , 0, NULL, 0),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ((int)wolfSSL_SetTmpDH(NULL, NULL, 0, g   , 0),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ((int)wolfSSL_SetTmpDH(ssl , p  , 0, NULL, 0),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ((int)wolfSSL_SetTmpDH(ssl , NULL, 0, g   , 0),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ((int)wolfSSL_SetTmpDH(NULL, p   , 0, g   , 0),
        WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ((int)wolfSSL_SetTmpDH(ssl , p   , 1, g   , 1),
        WC_NO_ERR_TRACE(DH_KEY_SIZE_E));
    ExpectIntEQ((int)wolfSSL_SetTmpDH(ssl , buff, 6000, g   , 1),
        WC_NO_ERR_TRACE(DH_KEY_SIZE_E));
#if !defined(WOLFSSL_OLD_PRIME_CHECK) && !defined(HAVE_FIPS) && \
    !defined(HAVE_SELFTEST)
#ifndef NO_WOLFSSL_SERVER
    /* Parameters will be tested later so it passes now. */
    ExpectIntEQ((int)wolfSSL_SetTmpDH(ssl, bad_p, pSz, g, gSz),
        WOLFSSL_SUCCESS);
#endif
#endif
#ifndef NO_WOLFSSL_CLIENT
    ExpectIntEQ((int)wolfSSL_SetTmpDH(ssl_c, p, pSz, g, gSz),
        WC_NO_ERR_TRACE(SIDE_ERROR));
#endif
    ExpectIntEQ((int)SSL_CTX_set_tmp_dh(NULL, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ((int)SSL_CTX_set_tmp_dh(ctx , NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ((int)SSL_CTX_set_tmp_dh(NULL, dh  ), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ((int)SSL_set_tmp_dh(NULL, NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ((int)SSL_set_tmp_dh(ssl , NULL), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    ExpectIntEQ((int)SSL_set_tmp_dh(NULL, dh  ), WC_NO_ERR_TRACE(BAD_FUNC_ARG));
    /* No p/g to use. */
    dh2 = wolfSSL_DH_new();
    ExpectIntEQ((int)SSL_CTX_set_tmp_dh(ctx , dh2 ), WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
    ExpectIntEQ((int)SSL_set_tmp_dh(ssl , dh2 ), WC_NO_ERR_TRACE(WOLFSSL_FATAL_ERROR));
    DH_free(dh2);
    dh2 = NULL;

    ExpectIntEQ((int)wolfSSL_CTX_SetTmpDH(ctx, p, pSz, g, gSz),
        WOLFSSL_SUCCESS);
    ExpectIntEQ((int)SSL_CTX_set_tmp_dh(ctx, dh), WOLFSSL_SUCCESS);
#ifndef NO_WOLFSSL_SERVER
    ExpectIntEQ((int)SSL_set_tmp_dh(ssl, dh), WOLFSSL_SUCCESS);
#else
    ExpectIntEQ((int)SSL_set_tmp_dh(ssl, dh), WC_NO_ERR_TRACE(SIDE_ERROR));
#endif

    BIO_free(bio);
#if !defined(NO_DSA)
    DSA_free(dsa);
#endif
    DH_free(dh);
    dh = NULL;
#ifndef NO_WOLFSSL_CLIENT
    if (ssl != ssl_c) {
        SSL_free(ssl_c);
    }
#endif
    SSL_free(ssl);
#ifndef NO_WOLFSSL_CLIENT
    if (ctx != ctx_c) {
        SSL_CTX_free(ctx_c);
    }
#endif
    SSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_ctrl(void)
{
    EXPECT_DECLS;
#if defined (OPENSSL_EXTRA) && !defined(NO_BIO)
    byte buff[6000];
    BIO* bio = NULL;
    int  bytes;
    BUF_MEM* ptr = NULL;

    XMEMSET(buff, 0, sizeof(buff));

    bytes = sizeof(buff);
    ExpectNotNull(bio = BIO_new_mem_buf((void*)buff, bytes));
    ExpectNotNull(BIO_s_socket());

    ExpectIntEQ((int)wolfSSL_BIO_get_mem_ptr(bio, &ptr), WOLFSSL_SUCCESS);

    /* needs tested after stubs filled out @TODO
        SSL_ctrl
        SSL_CTX_ctrl
    */

    BIO_free(bio);
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_BIO) */
    return EXPECT_RESULT();
}


static int test_wolfSSL_EVP_PKEY_new_mac_key(void)
{
    EXPECT_DECLS;
#ifdef OPENSSL_EXTRA
    static const unsigned char pw[] = "password";
    static const int pwSz = sizeof(pw) - 1;
    size_t checkPwSz = 0;
    const unsigned char* checkPw = NULL;
    WOLFSSL_EVP_PKEY* key = NULL;

    ExpectNull(key = wolfSSL_EVP_PKEY_new_mac_key(0, NULL, pw, pwSz));
    ExpectNull(key = wolfSSL_EVP_PKEY_new_mac_key(0, NULL, NULL, pwSz));

    ExpectNotNull(key = wolfSSL_EVP_PKEY_new_mac_key(EVP_PKEY_HMAC, NULL, pw,
        pwSz));
    if (key != NULL) {
        ExpectIntEQ(key->type, EVP_PKEY_HMAC);
        ExpectIntEQ(key->save_type, EVP_PKEY_HMAC);
        ExpectIntEQ(key->pkey_sz, pwSz);
        ExpectIntEQ(XMEMCMP(key->pkey.ptr, pw, pwSz), 0);
    }
    ExpectNotNull(checkPw = wolfSSL_EVP_PKEY_get0_hmac(key, &checkPwSz));
    ExpectIntEQ((int)checkPwSz, pwSz);
    ExpectIntEQ(XMEMCMP(checkPw, pw, pwSz), 0);
    wolfSSL_EVP_PKEY_free(key);
    key = NULL;

    ExpectNotNull(key = wolfSSL_EVP_PKEY_new_mac_key(EVP_PKEY_HMAC, NULL, pw,
        0));
    ExpectIntEQ(key->pkey_sz, 0);
    if (EXPECT_SUCCESS()) {
        /* Allocation for key->pkey.ptr may fail - OK key len is 0 */
        checkPw = wolfSSL_EVP_PKEY_get0_hmac(key, &checkPwSz);
    }
    ExpectTrue((checkPwSz == 0) || (checkPw != NULL));
    ExpectIntEQ((int)checkPwSz, 0);
    wolfSSL_EVP_PKEY_free(key);
    key = NULL;

    ExpectNotNull(key = wolfSSL_EVP_PKEY_new_mac_key(EVP_PKEY_HMAC, NULL, NULL,
        0));
    ExpectIntEQ(key->pkey_sz, 0);
    if (EXPECT_SUCCESS()) {
        /* Allocation for key->pkey.ptr may fail - OK key len is 0 */
        checkPw = wolfSSL_EVP_PKEY_get0_hmac(key, &checkPwSz);
    }
    ExpectTrue((checkPwSz == 0) || (checkPw != NULL));
    ExpectIntEQ((int)checkPwSz, 0);
    wolfSSL_EVP_PKEY_free(key);
    key = NULL;
#endif /* OPENSSL_EXTRA */
    return EXPECT_RESULT();
}


static int test_wolfSSL_EVP_PKEY_new_CMAC_key(void)
{
    EXPECT_DECLS;
#ifdef OPENSSL_EXTRA
#if defined(WOLFSSL_CMAC) && !defined(NO_AES) && defined(WOLFSSL_AES_DIRECT)
    const char *priv = "ABCDEFGHIJKLMNOP";
    const WOLFSSL_EVP_CIPHER* cipher = EVP_aes_128_cbc();
    WOLFSSL_EVP_PKEY* key = NULL;

    ExpectNull(key = wolfSSL_EVP_PKEY_new_CMAC_key(
        NULL, NULL, AES_128_KEY_SIZE, cipher));
    ExpectNull(key = wolfSSL_EVP_PKEY_new_CMAC_key(
        NULL, (const unsigned char *)priv, 0, cipher));
    ExpectNull(key = wolfSSL_EVP_PKEY_new_CMAC_key(
        NULL, (const unsigned char *)priv, AES_128_KEY_SIZE, NULL));

    ExpectNotNull(key = wolfSSL_EVP_PKEY_new_CMAC_key(
        NULL, (const unsigned char *)priv, AES_128_KEY_SIZE, cipher));
    wolfSSL_EVP_PKEY_free(key);
#endif /* WOLFSSL_CMAC && !NO_AES && WOLFSSL_AES_DIRECT */
#endif /* OPENSSL_EXTRA */
    return EXPECT_RESULT();
}

static int test_wolfSSL_EVP_Digest(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_SHA256) && !defined(NO_PWDBASED)
    const char* in = "abc";
    int   inLen = (int)XSTRLEN(in);
    byte  out[WC_SHA256_DIGEST_SIZE];
    unsigned int outLen;
    const char* expOut =
        "\xBA\x78\x16\xBF\x8F\x01\xCF\xEA\x41\x41\x40\xDE\x5D\xAE\x22"
        "\x23\xB0\x03\x61\xA3\x96\x17\x7A\x9C\xB4\x10\xFF\x61\xF2\x00"
        "\x15\xAD";

    ExpectIntEQ(wolfSSL_EVP_Digest((unsigned char*)in, inLen, out, &outLen,
        "SHA256", NULL), 1);
    ExpectIntEQ(outLen, WC_SHA256_DIGEST_SIZE);
    ExpectIntEQ(XMEMCMP(out, expOut, WC_SHA256_DIGEST_SIZE), 0);
#endif /* OPEN_EXTRA && ! NO_SHA256 */
    return EXPECT_RESULT();
}

static int test_wolfSSL_EVP_Digest_all(void)
{
    EXPECT_DECLS;
#ifdef OPENSSL_EXTRA
    const char* digests[] = {
#ifndef NO_MD5
        "MD5",
#endif
#ifndef NO_SHA
        "SHA",
#endif
#ifdef WOLFSSL_SHA224
        "SHA224",
#endif
#ifndef NO_SHA256
        "SHA256",
#endif
#ifdef WOLFSSL_SHA384
        "SHA384",
#endif
#ifdef WOLFSSL_SHA512
        "SHA512",
#endif
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_224)
        "SHA512-224",
#endif
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_256)
        "SHA512-256",
#endif
#ifdef WOLFSSL_SHA3
#ifndef WOLFSSL_NOSHA3_224
        "SHA3-224",
#endif
#ifndef WOLFSSL_NOSHA3_256
        "SHA3-256",
#endif
        "SHA3-384",
#ifndef WOLFSSL_NOSHA3_512
        "SHA3-512",
#endif
#endif /* WOLFSSL_SHA3 */
        NULL
    };
    const char** d;
    const unsigned char in[] = "abc";
    int   inLen = XSTR_SIZEOF(in);
    byte  out[WC_MAX_DIGEST_SIZE];
    unsigned int outLen;

    for (d = digests; *d != NULL; d++) {
        ExpectIntEQ(EVP_Digest(in, inLen, out, &outLen, *d, NULL), 1);
        ExpectIntGT(outLen, 0);
        ExpectIntEQ(EVP_MD_size(*d), outLen);
    }
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_EVP_MD_size(void)
{
    EXPECT_DECLS;
#ifdef OPENSSL_EXTRA
    WOLFSSL_EVP_MD_CTX mdCtx;

#ifdef WOLFSSL_SHA3
#ifndef WOLFSSL_NOSHA3_224
    wolfSSL_EVP_MD_CTX_init(&mdCtx);

    ExpectIntEQ(wolfSSL_EVP_DigestInit(&mdCtx, "SHA3-224"), 1);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_size(&mdCtx), WC_SHA3_224_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_block_size(&mdCtx), WC_SHA3_224_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);
#endif
#ifndef WOLFSSL_NOSHA3_256
    wolfSSL_EVP_MD_CTX_init(&mdCtx);

    ExpectIntEQ(wolfSSL_EVP_DigestInit(&mdCtx, "SHA3-256"), 1);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_size(&mdCtx), WC_SHA3_256_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_block_size(&mdCtx), WC_SHA3_256_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);
#endif
    wolfSSL_EVP_MD_CTX_init(&mdCtx);

    ExpectIntEQ(wolfSSL_EVP_DigestInit(&mdCtx, "SHA3-384"), 1);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_size(&mdCtx), WC_SHA3_384_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_block_size(&mdCtx), WC_SHA3_384_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);
#ifndef WOLFSSL_NOSHA3_512
    wolfSSL_EVP_MD_CTX_init(&mdCtx);

    ExpectIntEQ(wolfSSL_EVP_DigestInit(&mdCtx, "SHA3-512"), 1);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_size(&mdCtx), WC_SHA3_512_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_block_size(&mdCtx), WC_SHA3_512_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);
#endif
#endif /* WOLFSSL_SHA3 */

#ifndef NO_SHA256
    wolfSSL_EVP_MD_CTX_init(&mdCtx);

    ExpectIntEQ(wolfSSL_EVP_DigestInit(&mdCtx, "SHA256"), 1);
    ExpectIntEQ(wolfSSL_EVP_MD_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_SHA256_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_block_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_SHA256_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_size(&mdCtx), WC_SHA256_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_block_size(&mdCtx), WC_SHA256_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);

#endif

#ifndef NO_MD5
    wolfSSL_EVP_MD_CTX_init(&mdCtx);

    ExpectIntEQ(wolfSSL_EVP_DigestInit(&mdCtx, "MD5"), 1);
    ExpectIntEQ(wolfSSL_EVP_MD_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_MD5_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_block_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_MD5_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_size(&mdCtx), WC_MD5_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_block_size(&mdCtx), WC_MD5_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);

#endif

#ifdef WOLFSSL_SHA224
    wolfSSL_EVP_MD_CTX_init(&mdCtx);

    ExpectIntEQ(wolfSSL_EVP_DigestInit(&mdCtx, "SHA224"), 1);
    ExpectIntEQ(wolfSSL_EVP_MD_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_SHA224_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_block_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_SHA224_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_size(&mdCtx), WC_SHA224_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_block_size(&mdCtx), WC_SHA224_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);

#endif

#ifdef WOLFSSL_SHA384
    wolfSSL_EVP_MD_CTX_init(&mdCtx);

    ExpectIntEQ(wolfSSL_EVP_DigestInit(&mdCtx, "SHA384"), 1);
    ExpectIntEQ(wolfSSL_EVP_MD_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_SHA384_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_block_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_SHA384_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_size(&mdCtx), WC_SHA384_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_block_size(&mdCtx), WC_SHA384_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);

#endif

#ifdef WOLFSSL_SHA512
    wolfSSL_EVP_MD_CTX_init(&mdCtx);

    ExpectIntEQ(wolfSSL_EVP_DigestInit(&mdCtx, "SHA512"), 1);
    ExpectIntEQ(wolfSSL_EVP_MD_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_SHA512_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_block_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_SHA512_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_size(&mdCtx), WC_SHA512_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_block_size(&mdCtx), WC_SHA512_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);

#endif

#ifndef NO_SHA
    wolfSSL_EVP_MD_CTX_init(&mdCtx);

    ExpectIntEQ(wolfSSL_EVP_DigestInit(&mdCtx, "SHA"), 1);
    ExpectIntEQ(wolfSSL_EVP_MD_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_SHA_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_block_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_SHA_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_size(&mdCtx), WC_SHA_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_block_size(&mdCtx), WC_SHA_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);

    wolfSSL_EVP_MD_CTX_init(&mdCtx);

    ExpectIntEQ(wolfSSL_EVP_DigestInit(&mdCtx, "SHA1"), 1);
    ExpectIntEQ(wolfSSL_EVP_MD_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_SHA_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_block_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)),
        WC_SHA_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_size(&mdCtx), WC_SHA_DIGEST_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_block_size(&mdCtx), WC_SHA_BLOCK_SIZE);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);
#endif
    /* error case */
    wolfSSL_EVP_MD_CTX_init(&mdCtx);

    ExpectIntEQ(wolfSSL_EVP_DigestInit(&mdCtx, ""), 0);
    ExpectIntEQ(wolfSSL_EVP_MD_size(wolfSSL_EVP_MD_CTX_md(&mdCtx)), 0);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_block_size(&mdCtx), 0);
    /* Cleanup is valid on uninit'ed struct */
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);
#endif /* OPENSSL_EXTRA */
    return EXPECT_RESULT();
}

static int test_wolfSSL_EVP_MD_pkey_type(void)
{
    EXPECT_DECLS;
#ifdef OPENSSL_EXTRA
    const WOLFSSL_EVP_MD* md;

#ifndef NO_MD5
    ExpectNotNull(md = EVP_md5());
    ExpectIntEQ(EVP_MD_pkey_type(md), NID_md5WithRSAEncryption);
#endif
#ifndef NO_SHA
    ExpectNotNull(md = EVP_sha1());
    ExpectIntEQ(EVP_MD_pkey_type(md), NID_sha1WithRSAEncryption);
#endif
#ifdef WOLFSSL_SHA224
    ExpectNotNull(md = EVP_sha224());
    ExpectIntEQ(EVP_MD_pkey_type(md), NID_sha224WithRSAEncryption);
#endif
    ExpectNotNull(md = EVP_sha256());
    ExpectIntEQ(EVP_MD_pkey_type(md), NID_sha256WithRSAEncryption);
#ifdef WOLFSSL_SHA384
    ExpectNotNull(md = EVP_sha384());
    ExpectIntEQ(EVP_MD_pkey_type(md), NID_sha384WithRSAEncryption);
#endif
#ifdef WOLFSSL_SHA512
    ExpectNotNull(md = EVP_sha512());
    ExpectIntEQ(EVP_MD_pkey_type(md), NID_sha512WithRSAEncryption);
#endif
#endif
    return EXPECT_RESULT();
}

#ifdef OPENSSL_EXTRA
static int test_hmac_signing(const WOLFSSL_EVP_MD *type, const byte* testKey,
        size_t testKeySz, const char* testData, size_t testDataSz,
        const byte* testResult, size_t testResultSz)
{
    EXPECT_DECLS;
    unsigned char check[WC_MAX_DIGEST_SIZE];
    size_t checkSz = 0;
    WOLFSSL_EVP_PKEY* key = NULL;
    WOLFSSL_EVP_MD_CTX mdCtx;

    ExpectNotNull(key = wolfSSL_EVP_PKEY_new_mac_key(EVP_PKEY_HMAC, NULL,
                                                      testKey, (int)testKeySz));
    wolfSSL_EVP_MD_CTX_init(&mdCtx);
    ExpectIntEQ(wolfSSL_EVP_DigestSignInit(&mdCtx, NULL, type, NULL, key), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestSignUpdate(&mdCtx, testData,
                                                  (unsigned int)testDataSz), 1);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, NULL, &checkSz), 1);
    ExpectIntEQ((int)checkSz, (int)testResultSz);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, check, &checkSz), 1);
    ExpectIntEQ((int)checkSz,(int)testResultSz);
    ExpectIntEQ(XMEMCMP(testResult, check, testResultSz), 0);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);

    ExpectIntEQ(wolfSSL_EVP_DigestVerifyInit(&mdCtx, NULL, type, NULL, key), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyUpdate(&mdCtx, testData,
                                                  (unsigned int)testDataSz), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyFinal(&mdCtx, testResult, checkSz), 1);

    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);
    wolfSSL_EVP_MD_CTX_init(&mdCtx);
    ExpectIntEQ(wolfSSL_EVP_DigestSignInit(&mdCtx, NULL, type, NULL, key), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestSignUpdate(&mdCtx, testData, 4), 1);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, NULL, &checkSz), 1);
    ExpectIntEQ((int)checkSz, (int)testResultSz);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, check, &checkSz), 1);
    ExpectIntEQ((int)checkSz,(int)testResultSz);
    ExpectIntEQ(wolfSSL_EVP_DigestSignUpdate(&mdCtx, testData + 4,
                                              (unsigned int)testDataSz - 4), 1);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, check, &checkSz), 1);
    ExpectIntEQ((int)checkSz,(int)testResultSz);
    ExpectIntEQ(XMEMCMP(testResult, check, testResultSz), 0);

    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyInit(&mdCtx, NULL, type, NULL, key), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyUpdate(&mdCtx, testData, 4), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyUpdate(&mdCtx, testData + 4,
                                              (unsigned int)testDataSz - 4), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyFinal(&mdCtx, testResult, checkSz), 1);

    ExpectIntEQ(wolfSSL_EVP_MD_CTX_cleanup(&mdCtx), 1);

    wolfSSL_EVP_PKEY_free(key);

    return EXPECT_RESULT();
}
#endif

static int test_wolfSSL_EVP_MD_hmac_signing(void)
{
    EXPECT_DECLS;
#ifdef OPENSSL_EXTRA
    static const unsigned char testKey[] =
    {
        0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
        0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
        0x0b, 0x0b, 0x0b, 0x0b
    };
    static const char testData[] = "Hi There";
#ifdef WOLFSSL_SHA224
    static const unsigned char testResultSha224[] =
    {
        0x89, 0x6f, 0xb1, 0x12, 0x8a, 0xbb, 0xdf, 0x19,
        0x68, 0x32, 0x10, 0x7c, 0xd4, 0x9d, 0xf3, 0x3f,
        0x47, 0xb4, 0xb1, 0x16, 0x99, 0x12, 0xba, 0x4f,
        0x53, 0x68, 0x4b, 0x22
    };
#endif
#ifndef NO_SHA256
    static const unsigned char testResultSha256[] =
    {
        0xb0, 0x34, 0x4c, 0x61, 0xd8, 0xdb, 0x38, 0x53,
        0x5c, 0xa8, 0xaf, 0xce, 0xaf, 0x0b, 0xf1, 0x2b,
        0x88, 0x1d, 0xc2, 0x00, 0xc9, 0x83, 0x3d, 0xa7,
        0x26, 0xe9, 0x37, 0x6c, 0x2e, 0x32, 0xcf, 0xf7
    };
#endif
#ifdef WOLFSSL_SHA384
    static const unsigned char testResultSha384[] =
    {
        0xaf, 0xd0, 0x39, 0x44, 0xd8, 0x48, 0x95, 0x62,
        0x6b, 0x08, 0x25, 0xf4, 0xab, 0x46, 0x90, 0x7f,
        0x15, 0xf9, 0xda, 0xdb, 0xe4, 0x10, 0x1e, 0xc6,
        0x82, 0xaa, 0x03, 0x4c, 0x7c, 0xeb, 0xc5, 0x9c,
        0xfa, 0xea, 0x9e, 0xa9, 0x07, 0x6e, 0xde, 0x7f,
        0x4a, 0xf1, 0x52, 0xe8, 0xb2, 0xfa, 0x9c, 0xb6
    };
#endif
#ifdef WOLFSSL_SHA512
    static const unsigned char testResultSha512[] =
    {
        0x87, 0xaa, 0x7c, 0xde, 0xa5, 0xef, 0x61, 0x9d,
        0x4f, 0xf0, 0xb4, 0x24, 0x1a, 0x1d, 0x6c, 0xb0,
        0x23, 0x79, 0xf4, 0xe2, 0xce, 0x4e, 0xc2, 0x78,
        0x7a, 0xd0, 0xb3, 0x05, 0x45, 0xe1, 0x7c, 0xde,
        0xda, 0xa8, 0x33, 0xb7, 0xd6, 0xb8, 0xa7, 0x02,
        0x03, 0x8b, 0x27, 0x4e, 0xae, 0xa3, 0xf4, 0xe4,
        0xbe, 0x9d, 0x91, 0x4e, 0xeb, 0x61, 0xf1, 0x70,
        0x2e, 0x69, 0x6c, 0x20, 0x3a, 0x12, 0x68, 0x54
    };
#endif
#ifdef WOLFSSL_SHA3
    #ifndef WOLFSSL_NOSHA3_224
    static const unsigned char testResultSha3_224[] =
    {
        0x3b, 0x16, 0x54, 0x6b, 0xbc, 0x7b, 0xe2, 0x70,
        0x6a, 0x03, 0x1d, 0xca, 0xfd, 0x56, 0x37, 0x3d,
        0x98, 0x84, 0x36, 0x76, 0x41, 0xd8, 0xc5, 0x9a,
        0xf3, 0xc8, 0x60, 0xf7
    };
    #endif
    #ifndef WOLFSSL_NOSHA3_256
    static const unsigned char testResultSha3_256[] =
    {
        0xba, 0x85, 0x19, 0x23, 0x10, 0xdf, 0xfa, 0x96,
        0xe2, 0xa3, 0xa4, 0x0e, 0x69, 0x77, 0x43, 0x51,
        0x14, 0x0b, 0xb7, 0x18, 0x5e, 0x12, 0x02, 0xcd,
        0xcc, 0x91, 0x75, 0x89, 0xf9, 0x5e, 0x16, 0xbb
    };
    #endif
    #ifndef WOLFSSL_NOSHA3_384
    static const unsigned char testResultSha3_384[] =
    {
        0x68, 0xd2, 0xdc, 0xf7, 0xfd, 0x4d, 0xdd, 0x0a,
        0x22, 0x40, 0xc8, 0xa4, 0x37, 0x30, 0x5f, 0x61,
        0xfb, 0x73, 0x34, 0xcf, 0xb5, 0xd0, 0x22, 0x6e,
        0x1b, 0xc2, 0x7d, 0xc1, 0x0a, 0x2e, 0x72, 0x3a,
        0x20, 0xd3, 0x70, 0xb4, 0x77, 0x43, 0x13, 0x0e,
        0x26, 0xac, 0x7e, 0x3d, 0x53, 0x28, 0x86, 0xbd
    };
    #endif
    #ifndef WOLFSSL_NOSHA3_512
    static const unsigned char testResultSha3_512[] =
    {
        0xeb, 0x3f, 0xbd, 0x4b, 0x2e, 0xaa, 0xb8, 0xf5,
        0xc5, 0x04, 0xbd, 0x3a, 0x41, 0x46, 0x5a, 0xac,
        0xec, 0x15, 0x77, 0x0a, 0x7c, 0xab, 0xac, 0x53,
        0x1e, 0x48, 0x2f, 0x86, 0x0b, 0x5e, 0xc7, 0xba,
        0x47, 0xcc, 0xb2, 0xc6, 0xf2, 0xaf, 0xce, 0x8f,
        0x88, 0xd2, 0x2b, 0x6d, 0xc6, 0x13, 0x80, 0xf2,
        0x3a, 0x66, 0x8f, 0xd3, 0x88, 0x8b, 0xb8, 0x05,
        0x37, 0xc0, 0xa0, 0xb8, 0x64, 0x07, 0x68, 0x9e
    };
    #endif
#endif

#ifndef NO_SHA256
    ExpectIntEQ(test_hmac_signing(wolfSSL_EVP_sha256(), testKey,
        sizeof(testKey), testData, XSTRLEN(testData), testResultSha256,
        sizeof(testResultSha256)), TEST_SUCCESS);
#endif
#ifdef WOLFSSL_SHA224
    ExpectIntEQ(test_hmac_signing(wolfSSL_EVP_sha224(), testKey,
        sizeof(testKey), testData, XSTRLEN(testData), testResultSha224,
        sizeof(testResultSha224)), TEST_SUCCESS);
#endif
#ifdef WOLFSSL_SHA384
    ExpectIntEQ(test_hmac_signing(wolfSSL_EVP_sha384(), testKey,
        sizeof(testKey), testData, XSTRLEN(testData), testResultSha384,
        sizeof(testResultSha384)), TEST_SUCCESS);
#endif
#ifdef WOLFSSL_SHA512
    ExpectIntEQ(test_hmac_signing(wolfSSL_EVP_sha512(), testKey,
        sizeof(testKey), testData, XSTRLEN(testData), testResultSha512,
        sizeof(testResultSha512)), TEST_SUCCESS);
#endif
#ifdef WOLFSSL_SHA3
    #ifndef WOLFSSL_NOSHA3_224
    ExpectIntEQ(test_hmac_signing(wolfSSL_EVP_sha3_224(), testKey,
        sizeof(testKey), testData, XSTRLEN(testData), testResultSha3_224,
        sizeof(testResultSha3_224)), TEST_SUCCESS);
    #endif
    #ifndef WOLFSSL_NOSHA3_256
    ExpectIntEQ(test_hmac_signing(wolfSSL_EVP_sha3_256(), testKey,
        sizeof(testKey), testData, XSTRLEN(testData), testResultSha3_256,
        sizeof(testResultSha3_256)), TEST_SUCCESS);
    #endif
    #ifndef WOLFSSL_NOSHA3_384
    ExpectIntEQ(test_hmac_signing(wolfSSL_EVP_sha3_384(), testKey,
        sizeof(testKey), testData, XSTRLEN(testData), testResultSha3_384,
        sizeof(testResultSha3_384)), TEST_SUCCESS);
    #endif
    #ifndef WOLFSSL_NOSHA3_512
    ExpectIntEQ(test_hmac_signing(wolfSSL_EVP_sha3_512(), testKey,
        sizeof(testKey), testData, XSTRLEN(testData), testResultSha3_512,
        sizeof(testResultSha3_512)), TEST_SUCCESS);
    #endif
#endif
#endif /* OPENSSL_EXTRA */
    return EXPECT_RESULT();
}


static int test_wolfSSL_EVP_MD_rsa_signing(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_RSA) && defined(USE_CERT_BUFFERS_2048)
    WOLFSSL_EVP_PKEY* privKey = NULL;
    WOLFSSL_EVP_PKEY* pubKey = NULL;
    WOLFSSL_EVP_PKEY_CTX* keyCtx = NULL;
    const char testData[] = "Hi There";
    WOLFSSL_EVP_MD_CTX mdCtx;
    WOLFSSL_EVP_MD_CTX mdCtxCopy;
    int ret;
    size_t checkSz = -1;
    int sz = 2048 / 8;
    const unsigned char* cp;
    const unsigned char* p;
    unsigned char check[2048/8];
    size_t i;
    int paddings[] = {
            RSA_PKCS1_PADDING,
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST) && defined(WC_RSA_PSS)
            RSA_PKCS1_PSS_PADDING,
#endif
    };


    cp = client_key_der_2048;
    ExpectNotNull((privKey = wolfSSL_d2i_PrivateKey(EVP_PKEY_RSA, NULL, &cp,
                                                  sizeof_client_key_der_2048)));
    p = client_keypub_der_2048;
    ExpectNotNull((pubKey = wolfSSL_d2i_PUBKEY(NULL, &p,
                                               sizeof_client_keypub_der_2048)));

    wolfSSL_EVP_MD_CTX_init(&mdCtx);
    wolfSSL_EVP_MD_CTX_init(&mdCtxCopy);
    ExpectIntEQ(wolfSSL_EVP_DigestSignInit(&mdCtx, NULL, wolfSSL_EVP_sha256(),
                                                             NULL, privKey), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestSignUpdate(&mdCtx, testData,
                                          (unsigned int)XSTRLEN(testData)), 1);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, NULL, &checkSz), 1);
    ExpectIntEQ((int)checkSz, sz);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, check, &checkSz), 1);
    ExpectIntEQ((int)checkSz,sz);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_copy_ex(&mdCtxCopy, &mdCtx), 1);
    ExpectIntEQ(wolfSSL_EVP_MD_CTX_copy_ex(&mdCtxCopy, &mdCtx), 1);
    ret = wolfSSL_EVP_MD_CTX_cleanup(&mdCtxCopy);
    ExpectIntEQ(ret, 1);
    ret = wolfSSL_EVP_MD_CTX_cleanup(&mdCtx);
    ExpectIntEQ(ret, 1);

    wolfSSL_EVP_MD_CTX_init(&mdCtx);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyInit(&mdCtx, NULL, wolfSSL_EVP_sha256(),
                                                              NULL, pubKey), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyUpdate(&mdCtx, testData,
                                               (unsigned int)XSTRLEN(testData)),
                1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyFinal(&mdCtx, check, checkSz), 1);
    ret = wolfSSL_EVP_MD_CTX_cleanup(&mdCtx);
    ExpectIntEQ(ret, 1);

    wolfSSL_EVP_MD_CTX_init(&mdCtx);
    ExpectIntEQ(wolfSSL_EVP_DigestSignInit(&mdCtx, NULL, wolfSSL_EVP_sha256(),
                                                             NULL, privKey), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestSignUpdate(&mdCtx, testData, 4), 1);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, NULL, &checkSz), 1);
    ExpectIntEQ((int)checkSz, sz);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, check, &checkSz), 1);
    ExpectIntEQ((int)checkSz, sz);
    ExpectIntEQ(wolfSSL_EVP_DigestSignUpdate(&mdCtx, testData + 4,
                                      (unsigned int)XSTRLEN(testData) - 4), 1);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, check, &checkSz), 1);
    ExpectIntEQ((int)checkSz, sz);
    ret = wolfSSL_EVP_MD_CTX_cleanup(&mdCtx);
    ExpectIntEQ(ret, 1);

    wolfSSL_EVP_MD_CTX_init(&mdCtx);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyInit(&mdCtx, NULL, wolfSSL_EVP_sha256(),
                                                              NULL, pubKey), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyUpdate(&mdCtx, testData, 4), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyUpdate(&mdCtx, testData + 4,
                                           (unsigned int)XSTRLEN(testData) - 4),
                1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyFinal(&mdCtx, check, checkSz), 1);
    ret = wolfSSL_EVP_MD_CTX_cleanup(&mdCtx);
    ExpectIntEQ(ret, 1);

    /* Check all signing padding types */
    for (i = 0; i < sizeof(paddings)/sizeof(int); i++) {
        wolfSSL_EVP_MD_CTX_init(&mdCtx);
        ExpectIntEQ(wolfSSL_EVP_DigestSignInit(&mdCtx, &keyCtx,
                wolfSSL_EVP_sha256(), NULL, privKey), 1);
        ExpectIntEQ(wolfSSL_EVP_PKEY_CTX_set_rsa_padding(keyCtx,
                paddings[i]), 1);
        ExpectIntEQ(wolfSSL_EVP_DigestSignUpdate(&mdCtx, testData,
                (unsigned int)XSTRLEN(testData)), 1);
        checkSz = sizeof(check);
        ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, NULL, &checkSz), 1);
        ExpectIntEQ((int)checkSz, sz);
        checkSz = sizeof(check);
        ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, check, &checkSz), 1);
        ExpectIntEQ((int)checkSz,sz);
        ret = wolfSSL_EVP_MD_CTX_cleanup(&mdCtx);
        ExpectIntEQ(ret, 1);

        wolfSSL_EVP_MD_CTX_init(&mdCtx);
        ExpectIntEQ(wolfSSL_EVP_DigestVerifyInit(&mdCtx, &keyCtx,
                wolfSSL_EVP_sha256(), NULL, pubKey), 1);
        ExpectIntEQ(wolfSSL_EVP_PKEY_CTX_set_rsa_padding(keyCtx,
                paddings[i]), 1);
        ExpectIntEQ(wolfSSL_EVP_DigestVerifyUpdate(&mdCtx, testData,
                (unsigned int)XSTRLEN(testData)), 1);
        ExpectIntEQ(wolfSSL_EVP_DigestVerifyFinal(&mdCtx, check, checkSz), 1);
        ret = wolfSSL_EVP_MD_CTX_cleanup(&mdCtx);
        ExpectIntEQ(ret, 1);
    }

    wolfSSL_EVP_PKEY_free(pubKey);
    wolfSSL_EVP_PKEY_free(privKey);
#endif
    return EXPECT_RESULT();
}


static int test_wolfSSL_EVP_MD_ecc_signing(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(HAVE_ECC) && defined(USE_CERT_BUFFERS_256)
    WOLFSSL_EVP_PKEY* privKey = NULL;
    WOLFSSL_EVP_PKEY* pubKey = NULL;
    const char testData[] = "Hi There";
    WOLFSSL_EVP_MD_CTX mdCtx;
    int ret;
    const unsigned char* cp;
    const unsigned char* p;
    unsigned char check[2048/8];
    size_t checkSz = sizeof(check);

    XMEMSET(check, 0, sizeof(check));

    cp = ecc_clikey_der_256;
    ExpectNotNull(privKey = wolfSSL_d2i_PrivateKey(EVP_PKEY_EC, NULL, &cp,
                                                   sizeof_ecc_clikey_der_256));
    p = ecc_clikeypub_der_256;
    ExpectNotNull((pubKey = wolfSSL_d2i_PUBKEY(NULL, &p,
                                                sizeof_ecc_clikeypub_der_256)));

    wolfSSL_EVP_MD_CTX_init(&mdCtx);
    ExpectIntEQ(wolfSSL_EVP_DigestSignInit(&mdCtx, NULL, wolfSSL_EVP_sha256(),
                                                             NULL, privKey), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestSignUpdate(&mdCtx, testData,
                                          (unsigned int)XSTRLEN(testData)), 1);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, NULL, &checkSz), 1);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, check, &checkSz), 1);
    ret = wolfSSL_EVP_MD_CTX_cleanup(&mdCtx);
    ExpectIntEQ(ret, 1);

    wolfSSL_EVP_MD_CTX_init(&mdCtx);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyInit(&mdCtx, NULL, wolfSSL_EVP_sha256(),
                                                              NULL, pubKey), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyUpdate(&mdCtx, testData,
                                               (unsigned int)XSTRLEN(testData)),
                1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyFinal(&mdCtx, check, checkSz), 1);
    ret = wolfSSL_EVP_MD_CTX_cleanup(&mdCtx);
    ExpectIntEQ(ret, 1);

    wolfSSL_EVP_MD_CTX_init(&mdCtx);
    ExpectIntEQ(wolfSSL_EVP_DigestSignInit(&mdCtx, NULL, wolfSSL_EVP_sha256(),
                                                             NULL, privKey), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestSignUpdate(&mdCtx, testData, 4), 1);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, NULL, &checkSz), 1);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, check, &checkSz), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestSignUpdate(&mdCtx, testData + 4,
                                      (unsigned int)XSTRLEN(testData) - 4), 1);
    checkSz = sizeof(check);
    ExpectIntEQ(wolfSSL_EVP_DigestSignFinal(&mdCtx, check, &checkSz), 1);
    ret = wolfSSL_EVP_MD_CTX_cleanup(&mdCtx);
    ExpectIntEQ(ret, 1);

    wolfSSL_EVP_MD_CTX_init(&mdCtx);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyInit(&mdCtx, NULL, wolfSSL_EVP_sha256(),
                                                              NULL, pubKey), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyUpdate(&mdCtx, testData, 4), 1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyUpdate(&mdCtx, testData + 4,
                                           (unsigned int)XSTRLEN(testData) - 4),
                1);
    ExpectIntEQ(wolfSSL_EVP_DigestVerifyFinal(&mdCtx, check, checkSz), 1);
    ret = wolfSSL_EVP_MD_CTX_cleanup(&mdCtx);
    ExpectIntEQ(ret, 1);

    wolfSSL_EVP_PKEY_free(pubKey);
    wolfSSL_EVP_PKEY_free(privKey);
#endif
    return EXPECT_RESULT();
}


static int test_wolfSSL_CTX_add_extra_chain_cert(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && !defined(NO_TLS) && \
   !defined(NO_FILESYSTEM) && !defined(NO_RSA) && !defined(NO_BIO)
#if !defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER)
    char caFile[] = "./certs/client-ca.pem";
    char clientFile[] = "./certs/client-cert.pem";
    SSL_CTX* ctx = NULL;
    X509* x509 = NULL;
    BIO *bio = NULL;
    X509 *cert = NULL;
    X509 *ca = NULL;
    STACK_OF(X509) *chain = NULL;
    STACK_OF(X509) *chain2 = NULL;

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
#endif

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(caFile,
        WOLFSSL_FILETYPE_PEM));

    /* Negative tests. */
    ExpectIntEQ((int)SSL_CTX_add_extra_chain_cert(NULL, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ((int)SSL_CTX_add_extra_chain_cert(ctx, NULL), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    ExpectIntEQ((int)SSL_CTX_add_extra_chain_cert(NULL, x509), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));

    ExpectIntEQ((int)SSL_CTX_add_extra_chain_cert(ctx, x509), WOLFSSL_SUCCESS);

    ExpectNotNull(x509 = wolfSSL_X509_new());
    /* Empty certificate. */
    ExpectIntEQ((int)SSL_CTX_add_extra_chain_cert(ctx, x509), WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
    wolfSSL_X509_free(x509);
    x509 = NULL;

    ExpectNotNull(x509 = wolfSSL_X509_load_certificate_file(clientFile,
        WOLFSSL_FILETYPE_PEM));

    /* additional test of getting EVP_PKEY key size from X509
     * Do not run with user RSA because wolfSSL_RSA_size is not currently
     * allowed with user RSA */
    {
        EVP_PKEY* pkey = NULL;
    #if defined(HAVE_ECC)
        X509* ecX509 = NULL;
    #endif /* HAVE_ECC */

        ExpectNotNull(pkey = X509_get_pubkey(x509));
        /* current RSA key is 2048 bit (256 bytes) */
        ExpectIntEQ(EVP_PKEY_size(pkey), 256);

        EVP_PKEY_free(pkey);
        pkey = NULL;

#if defined(HAVE_ECC)
    #if defined(USE_CERT_BUFFERS_256)
        ExpectNotNull(ecX509 = wolfSSL_X509_load_certificate_buffer(
                    cliecc_cert_der_256, sizeof_cliecc_cert_der_256,
                    SSL_FILETYPE_ASN1));
    #else
        ExpectNotNull(ecX509 = wolfSSL_X509_load_certificate_file(
            cliEccCertFile, SSL_FILETYPE_PEM));
    #endif
        pkey = X509_get_pubkey(ecX509);
        ExpectNotNull(pkey);
        /* current ECC key is 256 bit (32 bytes) */
        ExpectIntGE(EVP_PKEY_size(pkey), 72);

        X509_free(ecX509);
        ecX509 = NULL;
        EVP_PKEY_free(pkey);
        pkey = NULL;
#endif /* HAVE_ECC */
    }

    ExpectIntEQ((int)SSL_CTX_add_extra_chain_cert(ctx, x509), SSL_SUCCESS);
    if (EXPECT_SUCCESS()) {
        x509 = NULL;
    }

#ifdef WOLFSSL_ENCRYPTED_KEYS
    ExpectNull(SSL_CTX_get_default_passwd_cb(ctx));
    ExpectNull(SSL_CTX_get_default_passwd_cb_userdata(ctx));
#endif
    SSL_CTX_free(ctx);
    ctx = NULL;

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
#endif
    /* Test haproxy use case */
    ExpectNotNull(bio = BIO_new_file(svrCertFile, "r"));
    /* Read Certificate */
    ExpectNotNull(cert = PEM_read_bio_X509_AUX(bio, NULL, NULL, NULL));
    ExpectNotNull(ca = PEM_read_bio_X509(bio, NULL, NULL, NULL));
    ExpectNotNull(chain = sk_X509_new_null());
    ExpectIntEQ(sk_X509_push(chain, ca), 1);
    if (EXPECT_SUCCESS()) {
        ca = NULL;
    }
    ExpectNotNull(chain2 = X509_chain_up_ref(chain));
    ExpectNotNull(ca = sk_X509_shift(chain2));
    ExpectIntEQ(SSL_CTX_use_certificate(ctx, cert), 1);
    ExpectIntEQ(SSL_CTX_add_extra_chain_cert(ctx, ca), 1);
    if (EXPECT_SUCCESS()) {
        ca = NULL;
    }

    BIO_free(bio);
    X509_free(cert);
    X509_free(ca);
    X509_free(x509);
    sk_X509_pop_free(chain, X509_free);
    sk_X509_pop_free(chain2, X509_free);
    SSL_CTX_free(ctx);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
         !defined(NO_FILESYSTEM) && !defined(NO_RSA) && !defined (NO_BIO) */
    return EXPECT_RESULT();
}

#if !defined(NO_WOLFSSL_CLIENT) && !defined(NO_WOLFSSL_SERVER)
static int test_wolfSSL_ERR_peek_last_error_line(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
    !defined(NO_FILESYSTEM) && defined(DEBUG_WOLFSSL) && \
    !defined(NO_OLD_TLS) && !defined(WOLFSSL_NO_TLS12) && \
    defined(HAVE_IO_TESTS_DEPENDENCIES) && !defined(NO_ERROR_QUEUE)
    callback_functions client_cb;
    callback_functions server_cb;
    int         line = 0;
    int         flag = ERR_TXT_STRING;
    const char* file = NULL;
    const char* data = NULL;

    /* create a failed connection and inspect the error */
    XMEMSET(&client_cb, 0, sizeof(callback_functions));
    XMEMSET(&server_cb, 0, sizeof(callback_functions));
    client_cb.method  = wolfTLSv1_1_client_method;
    server_cb.method  = wolfTLSv1_2_server_method;

    test_wolfSSL_client_server_nofail(&client_cb, &server_cb);

    ExpectIntGT(ERR_get_error_line_data(NULL, NULL, &data, &flag), 0);
    ExpectNotNull(data);

    /* check clearing error state */
    ERR_remove_state(0);
    ExpectIntEQ((int)ERR_peek_last_error_line(NULL, NULL), 0);
    ERR_peek_last_error_line(NULL, &line);
    ExpectIntEQ(line, 0);
    ERR_peek_last_error_line(&file, NULL);
    ExpectNull(file);

    /* retry connection to fill error queue */
    XMEMSET(&client_cb, 0, sizeof(callback_functions));
    XMEMSET(&server_cb, 0, sizeof(callback_functions));
    client_cb.method  = wolfTLSv1_1_client_method;
    server_cb.method  = wolfTLSv1_2_server_method;

    test_wolfSSL_client_server_nofail(&client_cb, &server_cb);

    /* check that error code was stored */
    ExpectIntNE((int)ERR_peek_last_error_line(NULL, NULL), 0);
    ERR_peek_last_error_line(NULL, &line);
    ExpectIntNE(line, 0);
    ERR_peek_last_error_line(&file, NULL);
    ExpectNotNull(file);

    fprintf(stderr, "\nTesting error print out\n");
    ERR_print_errors_fp(stderr);
    fprintf(stderr, "Done testing print out\n\n");
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_CERTS) &&
        * !defined(NO_FILESYSTEM) && !defined(DEBUG_WOLFSSL) */
    return EXPECT_RESULT();
}
#endif /* !NO_WOLFSSL_CLIENT && !NO_WOLFSSL_SERVER */

#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
       !defined(NO_FILESYSTEM) && !defined(NO_RSA)
static int verify_cb(int ok, X509_STORE_CTX *ctx)
{
    (void) ok;
    (void) ctx;
    fprintf(stderr, "ENTER verify_cb\n");
    return SSL_SUCCESS;
}
#endif

static int test_wolfSSL_X509_Name_canon(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(NO_CERTS) && \
    !defined(NO_FILESYSTEM) && !defined(NO_SHA) && \
     defined(WOLFSSL_CERT_GEN) && \
    (defined(WOLFSSL_CERT_REQ) || defined(WOLFSSL_CERT_EXT)) && !defined(NO_RSA)
    const long ex_hash1 = 0x0fdb2da4;
    const long ex_hash2 = 0x9f3e8c9e;
    X509_NAME *name = NULL;
    X509 *x509 = NULL;
    XFILE file = XBADFILE;
    unsigned long hash = 0;
    byte digest[WC_MAX_DIGEST_SIZE] = {0};
    byte  *pbuf = NULL;
    word32 len = 0;
    (void) ex_hash2;

    ExpectTrue((file = XFOPEN(caCertFile, "rb")) != XBADFILE);
    ExpectNotNull(x509 = PEM_read_X509(file, NULL, NULL, NULL));
    ExpectNotNull(name = X509_get_issuer_name(x509));

    /* When output buffer is NULL, should return necessary output buffer
     * length.*/
    ExpectIntEQ(wolfSSL_i2d_X509_NAME_canon(NULL, NULL), BAD_FUNC_ARG);
    ExpectIntGT(wolfSSL_i2d_X509_NAME_canon(name, NULL), 0);
    ExpectIntGT((len = (word32)wolfSSL_i2d_X509_NAME_canon(name, &pbuf)), 0);
    ExpectIntEQ(wc_ShaHash((const byte*)pbuf, (word32)len, digest), 0);

    hash = (((unsigned long)digest[3] << 24) |
            ((unsigned long)digest[2] << 16) |
            ((unsigned long)digest[1] <<  8) |
            ((unsigned long)digest[0]));
    ExpectIntEQ(hash, ex_hash1);

    if (file != XBADFILE) {
        XFCLOSE(file);
        file = XBADFILE;
    }
    X509_free(x509);
    x509 = NULL;
    XFREE(pbuf, NULL, DYNAMIC_TYPE_OPENSSL);
    pbuf = NULL;

    ExpectTrue((file = XFOPEN(cliCertFile, "rb")) != XBADFILE);
    ExpectNotNull(x509 = PEM_read_X509(file, NULL, NULL, NULL));
    ExpectNotNull(name = X509_get_issuer_name(x509));

    ExpectIntGT((len = (word32)wolfSSL_i2d_X509_NAME_canon(name, &pbuf)), 0);
    ExpectIntEQ(wc_ShaHash((const byte*)pbuf, (word32)len, digest), 0);

    hash = (((unsigned long)digest[3] << 24) |
            ((unsigned long)digest[2] << 16) |
            ((unsigned long)digest[1] <<  8) |
            ((unsigned long)digest[0]));

    ExpectIntEQ(hash, ex_hash2);

    if (file != XBADFILE)
        XFCLOSE(file);
    X509_free(x509);
    XFREE(pbuf, NULL, DYNAMIC_TYPE_OPENSSL);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_LOOKUP_ctrl_hash_dir(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(NO_FILESYSTEM) && !defined(NO_WOLFSSL_DIR)
    const int  MAX_DIR = 4;
    const char paths[][32] = {
                             "./certs/ed25519",
                             "./certs/ecc",
                             "./certs/crl",
                             "./certs/",
                            };

    char CertCrl_path[MAX_FILENAME_SZ];
    char *p;
    X509_STORE* str = NULL;
    X509_LOOKUP* lookup = NULL;
    WOLFSSL_STACK* sk = NULL;
    int len, total_len, i;

    (void)sk;

    XMEMSET(CertCrl_path, 0, MAX_FILENAME_SZ);

    /* illegal string */
    ExpectNotNull((str = wolfSSL_X509_STORE_new()));
    ExpectNotNull(lookup = X509_STORE_add_lookup(str, X509_LOOKUP_file()));
    ExpectIntEQ(X509_LOOKUP_ctrl(lookup, X509_L_ADD_DIR, "",
                                    SSL_FILETYPE_PEM, NULL), 0);
    ExpectIntEQ(X509_LOOKUP_ctrl(lookup, X509_L_ADD_STORE, "",
        SSL_FILETYPE_PEM, NULL), WOLFSSL_NOT_IMPLEMENTED);
    ExpectIntEQ(X509_LOOKUP_ctrl(lookup, X509_L_LOAD_STORE, "",
        SSL_FILETYPE_PEM, NULL), WOLFSSL_NOT_IMPLEMENTED);
    ExpectIntEQ(X509_LOOKUP_ctrl(lookup, 0, "",
        SSL_FILETYPE_PEM, NULL), WOLFSSL_FAILURE);

    /* free store */
    X509_STORE_free(str);
    str = NULL;

    /* short folder string */
    ExpectNotNull((str = wolfSSL_X509_STORE_new()));
    ExpectNotNull(lookup = X509_STORE_add_lookup(str, X509_LOOKUP_file()));
    ExpectIntEQ(X509_LOOKUP_ctrl(lookup, X509_L_ADD_DIR, "./",
                                    SSL_FILETYPE_PEM,NULL), 1);
    #if defined(WOLFSSL_INT_H)
    /* only available when including internal.h */
    ExpectNotNull(sk = lookup->dirs->dir_entry);
    #endif
    /* free store */
    X509_STORE_free(str);
    str = NULL;

    /* typical function check */
    p = &CertCrl_path[0];
    total_len = 0;

    for (i = MAX_DIR - 1; i>=0 && total_len < MAX_FILENAME_SZ; i--) {
        len = (int)XSTRLEN((const char*)&paths[i]);
        total_len += len;
        XSTRNCPY(p, paths[i], MAX_FILENAME_SZ - total_len);
        p += len;
        if (i != 0) *(p++) = SEPARATOR_CHAR;
    }

    ExpectNotNull((str = wolfSSL_X509_STORE_new()));
    ExpectNotNull(lookup = X509_STORE_add_lookup(str, X509_LOOKUP_file()));
    ExpectIntEQ(X509_LOOKUP_ctrl(lookup, X509_L_ADD_DIR, CertCrl_path,
                                    SSL_FILETYPE_PEM,NULL), 1);
    #if defined(WOLFSSL_INT_H)
    /* only available when including internal.h */
    ExpectNotNull(sk = lookup->dirs->dir_entry);
    #endif

    X509_STORE_free(str);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_LOOKUP_ctrl_file(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(NO_CERTS) && \
    !defined(NO_FILESYSTEM) && !defined(NO_RSA) && \
    defined(WOLFSSL_SIGNER_DER_CERT)
    X509_STORE_CTX* ctx = NULL;
    X509_STORE* str = NULL;
    X509_LOOKUP* lookup = NULL;

    X509* cert1 = NULL;
    X509* x509Ca = NULL;
    X509* x509Svr = NULL;
    X509* issuer = NULL;

    WOLFSSL_STACK* sk = NULL;
    X509_NAME* caName = NULL;
    X509_NAME* issuerName = NULL;

    XFILE file1 = XBADFILE;
    int i;
    int cert_count = 0;
    int cmp;

    char der[] = "certs/ca-cert.der";

#ifdef HAVE_CRL
    char pem[][100] = {
        "./certs/crl/crl.pem",
        "./certs/crl/crl2.pem",
        "./certs/crl/caEccCrl.pem",
        "./certs/crl/eccCliCRL.pem",
        "./certs/crl/eccSrvCRL.pem",
        ""
    };
#endif
    ExpectTrue((file1 = XFOPEN("./certs/ca-cert.pem", "rb")) != XBADFILE);
    ExpectNotNull(cert1 = wolfSSL_PEM_read_X509(file1, NULL, NULL, NULL));
    if (file1 != XBADFILE)
        XFCLOSE(file1);

    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectNotNull((str = wolfSSL_X509_STORE_new()));
    ExpectNotNull(lookup = X509_STORE_add_lookup(str, X509_LOOKUP_file()));
    ExpectIntEQ(wolfSSL_X509_load_cert_crl_file(NULL, NULL,
        WOLFSSL_FILETYPE_PEM), 0);
    ExpectIntEQ(wolfSSL_X509_load_cert_crl_file(lookup, NULL,
        WOLFSSL_FILETYPE_PEM), 0);
    ExpectIntEQ(wolfSSL_X509_load_cert_crl_file(NULL, caCertFile,
        WOLFSSL_FILETYPE_PEM), 0);
    ExpectIntEQ(wolfSSL_X509_load_cert_crl_file(NULL, der       ,
        WOLFSSL_FILETYPE_PEM), 0);
    ExpectIntEQ(X509_LOOKUP_ctrl(lookup, X509_L_FILE_LOAD, caCertFile,
                                    SSL_FILETYPE_PEM,NULL), 1);
    ExpectNotNull(sk = wolfSSL_CertManagerGetCerts(str->cm));
    ExpectIntEQ((cert_count = sk_X509_num(sk)), 1);

    /* check if CA cert is loaded into the store */
    for (i = 0; i < cert_count; i++) {
        x509Ca = sk_X509_value(sk, i);
        ExpectIntEQ(0, wolfSSL_X509_cmp(x509Ca, cert1));
    }

    ExpectNotNull((x509Svr =
            wolfSSL_X509_load_certificate_file(svrCertFile, SSL_FILETYPE_PEM)));

    ExpectIntEQ(X509_STORE_CTX_init(ctx, str, x509Svr, NULL), SSL_SUCCESS);

    ExpectNull(X509_STORE_CTX_get0_current_issuer(NULL));
    issuer = X509_STORE_CTX_get0_current_issuer(ctx);
    ExpectNull(issuer);

    ExpectIntEQ(X509_verify_cert(ctx), 1);

    issuer = X509_STORE_CTX_get0_current_issuer(ctx);
    ExpectNotNull(issuer);
    caName = X509_get_subject_name(x509Ca);
    ExpectNotNull(caName);
    issuerName = X509_get_subject_name(issuer);
    ExpectNotNull(issuerName);
    cmp = X509_NAME_cmp(caName, issuerName);
    ExpectIntEQ(cmp, 0);

    /* load der format */
    issuer = NULL;
    X509_STORE_CTX_free(ctx);
    ctx = NULL;
    X509_STORE_free(str);
    str = NULL;
    sk_X509_pop_free(sk, NULL);
    sk = NULL;
    X509_free(x509Svr);
    x509Svr = NULL;

    ExpectNotNull((str = wolfSSL_X509_STORE_new()));
    ExpectNotNull(lookup = X509_STORE_add_lookup(str, X509_LOOKUP_file()));
    ExpectIntEQ(X509_LOOKUP_ctrl(lookup, X509_L_FILE_LOAD, der,
                                    SSL_FILETYPE_ASN1,NULL), 1);
    ExpectNotNull(sk = wolfSSL_CertManagerGetCerts(str->cm));
    ExpectIntEQ((cert_count = sk_X509_num(sk)), 1);
    /* check if CA cert is loaded into the store */
    for (i = 0; i < cert_count; i++) {
        x509Ca = sk_X509_value(sk, i);
        ExpectIntEQ(0, wolfSSL_X509_cmp(x509Ca, cert1));
    }

    X509_STORE_free(str);
    str = NULL;
    sk_X509_pop_free(sk, NULL);
    sk = NULL;
    X509_free(cert1);
    cert1 = NULL;

#ifdef HAVE_CRL
    ExpectNotNull(str = wolfSSL_X509_STORE_new());
    ExpectNotNull(lookup = X509_STORE_add_lookup(str, X509_LOOKUP_file()));
    ExpectIntEQ(X509_LOOKUP_ctrl(lookup, X509_L_FILE_LOAD, caCertFile,
                                                    SSL_FILETYPE_PEM,NULL), 1);
    ExpectIntEQ(X509_LOOKUP_ctrl(lookup, X509_L_FILE_LOAD,
                                "certs/server-revoked-cert.pem",
                                 SSL_FILETYPE_PEM,NULL), 1);
    if (str) {
        ExpectIntEQ(wolfSSL_CertManagerVerify(str->cm, svrCertFile,
                    WOLFSSL_FILETYPE_PEM), 1);
        /* since store hasn't yet known the revoked cert*/
        ExpectIntEQ(wolfSSL_CertManagerVerify(str->cm,
                    "certs/server-revoked-cert.pem",
                    WOLFSSL_FILETYPE_PEM), 1);
    }
    for (i = 0; pem[i][0] != '\0'; i++)
    {
        ExpectIntEQ(X509_LOOKUP_ctrl(lookup, X509_L_FILE_LOAD, pem[i],
                                        SSL_FILETYPE_PEM, NULL), 1);
    }

    if (str) {
        /* since store knows crl list */
        ExpectIntEQ(wolfSSL_CertManagerVerify(str->cm,
                    "certs/server-revoked-cert.pem",
                    WOLFSSL_FILETYPE_PEM ), WC_NO_ERR_TRACE(CRL_CERT_REVOKED));
    }

    ExpectIntEQ(X509_LOOKUP_ctrl(NULL, 0, NULL, 0, NULL), 0);
    X509_STORE_free(str);
#endif
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_trusted_stack_cleanup(void)
{
    int res = TEST_SKIPPED;
#if defined(OPENSSL_EXTRA)
    X509_STORE_CTX_cleanup(NULL);
    X509_STORE_CTX_trusted_stack(NULL, NULL);

    res = TEST_SUCCESS;
#endif
    return res;
}

static int test_wolfSSL_X509_STORE_CTX_get_issuer(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_RSA)
    X509_STORE_CTX* ctx = NULL;
    X509_STORE* str = NULL;
    X509* x509Ca = NULL;
    X509* x509Svr = NULL;
    X509* issuer = NULL;
    X509_NAME* caName = NULL;
    X509_NAME* issuerName = NULL;

    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectNotNull((str = wolfSSL_X509_STORE_new()));
    ExpectNotNull((x509Ca =
            wolfSSL_X509_load_certificate_file(caCertFile, SSL_FILETYPE_PEM)));
    ExpectIntEQ(X509_STORE_add_cert(str, x509Ca), SSL_SUCCESS);
    ExpectNotNull((x509Svr =
            wolfSSL_X509_load_certificate_file(svrCertFile, SSL_FILETYPE_PEM)));

    ExpectIntEQ(X509_STORE_CTX_init(ctx, str, x509Svr, NULL), SSL_SUCCESS);

    /* Issuer0 is not set until chain is built for verification */
    ExpectNull(X509_STORE_CTX_get0_current_issuer(NULL));
    ExpectNull(issuer = X509_STORE_CTX_get0_current_issuer(ctx));

    /* Issuer1 will use the store to make a new issuer */
    ExpectIntEQ(X509_STORE_CTX_get1_issuer(&issuer, ctx, x509Svr), 1);
    ExpectNotNull(issuer);
    X509_free(issuer);

    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectNotNull(issuer = X509_STORE_CTX_get0_current_issuer(ctx));
    ExpectNotNull(caName = X509_get_subject_name(x509Ca));
    ExpectNotNull(issuerName = X509_get_subject_name(issuer));
#ifdef WOLFSSL_SIGNER_DER_CERT
    ExpectIntEQ(X509_NAME_cmp(caName, issuerName), 0);
#endif

    X509_STORE_CTX_free(ctx);
    X509_free(x509Svr);
    X509_STORE_free(str);
    X509_free(x509Ca);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_PKCS7_certs(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(NO_CERTS) && !defined(NO_BIO) && \
   !defined(NO_FILESYSTEM) && !defined(NO_RSA) && defined(HAVE_PKCS7)
    STACK_OF(X509)* sk = NULL;
    STACK_OF(X509_INFO)* info_sk = NULL;
    PKCS7 *p7 = NULL;
    BIO* bio = NULL;
    const byte* p = NULL;
    int buflen = 0;
    int i;

    /* Test twice. Once with d2i and once without to test
     * that everything is free'd correctly. */
    for (i = 0; i < 2; i++) {
        ExpectNotNull(p7 = PKCS7_new());
        if (p7 != NULL) {
            p7->version = 1;
        #ifdef NO_SHA
            p7->hashOID = SHA256h;
        #else
            p7->hashOID = SHAh;
        #endif
        }
        ExpectNotNull(bio = BIO_new(BIO_s_file()));
        ExpectIntGT(BIO_read_filename(bio, svrCertFile), 0);
        ExpectNotNull(info_sk = PEM_X509_INFO_read_bio(bio, NULL, NULL, NULL));
        ExpectIntEQ(sk_X509_INFO_num(info_sk), 2);
        ExpectNotNull(sk = sk_X509_new_null());
        while (EXPECT_SUCCESS() && (sk_X509_INFO_num(info_sk) > 0)) {
            X509_INFO* info = NULL;
            ExpectNotNull(info = sk_X509_INFO_shift(info_sk));
            ExpectIntGT(sk_X509_push(sk, info->x509), 0);
            if (EXPECT_SUCCESS() && (info != NULL)) {
                info->x509 = NULL;
            }
            X509_INFO_free(info);
        }
        sk_X509_INFO_pop_free(info_sk, X509_INFO_free);
        info_sk = NULL;
        BIO_free(bio);
        bio = NULL;
        ExpectNotNull(bio = BIO_new(BIO_s_mem()));
        ExpectIntEQ(wolfSSL_PKCS7_encode_certs(p7, sk, bio), 1);
        if ((sk != NULL) && ((p7 == NULL) || (bio == NULL))) {
            sk_X509_pop_free(sk, X509_free);
        }
        sk = NULL;
        ExpectIntGT((buflen = BIO_get_mem_data(bio, &p)), 0);

        if (i == 0) {
            PKCS7_free(p7);
            p7 = NULL;
            ExpectNotNull(d2i_PKCS7(&p7, &p, buflen));
            if (p7 != NULL) {
                /* Reset certs to force wolfSSL_PKCS7_to_stack to regenerate
                 * them */
                ((WOLFSSL_PKCS7*)p7)->certs = NULL;
            }
            /* PKCS7_free free's the certs */
            ExpectNotNull(wolfSSL_PKCS7_to_stack(p7));
        }

        BIO_free(bio);
        bio = NULL;
        PKCS7_free(p7);
        p7 = NULL;
    }
#endif /* defined(OPENSSL_ALL) && !defined(NO_CERTS) && \
         !defined(NO_FILESYSTEM) && !defined(NO_RSA) && defined(HAVE_PKCS7) */
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
   !defined(NO_FILESYSTEM) && !defined(NO_RSA)
    X509_STORE_CTX* ctx = NULL;
    X509_STORE* str = NULL;
    X509* x509 = NULL;
#ifdef OPENSSL_ALL
    X509* x5092 = NULL;
    STACK_OF(X509) *sk = NULL;
    STACK_OF(X509) *sk2 = NULL;
    STACK_OF(X509) *sk3 = NULL;
#endif

    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectNotNull((str = wolfSSL_X509_STORE_new()));
    ExpectNotNull((x509 =
                wolfSSL_X509_load_certificate_file(svrCertFile, SSL_FILETYPE_PEM)));
    ExpectIntEQ(X509_STORE_add_cert(str, x509), SSL_SUCCESS);
#ifdef OPENSSL_ALL
    /* sk_X509_new only in OPENSSL_ALL */
    sk = sk_X509_new_null();
    ExpectNotNull(sk);
    ExpectIntEQ(X509_STORE_CTX_init(ctx, str, x509, sk), SSL_SUCCESS);
#else
    ExpectIntEQ(X509_STORE_CTX_init(ctx, str, x509, NULL), SSL_SUCCESS);
#endif
    ExpectIntEQ(SSL_get_ex_data_X509_STORE_CTX_idx(), 0);
    X509_STORE_CTX_set_error(ctx, -5);
    X509_STORE_CTX_set_error(NULL, -5);

    X509_STORE_CTX_free(ctx);
    ctx = NULL;
#ifdef OPENSSL_ALL
    sk_X509_pop_free(sk, NULL);
    sk = NULL;
#endif
    X509_STORE_free(str);
    str = NULL;
    X509_free(x509);
    x509 = NULL;

    ExpectNotNull(ctx = X509_STORE_CTX_new());
    X509_STORE_CTX_set_verify_cb(ctx, verify_cb);
    X509_STORE_CTX_free(ctx);
    ctx = NULL;

#ifdef OPENSSL_ALL
    /* test X509_STORE_CTX_get(1)_chain */
    ExpectNotNull((x509 = X509_load_certificate_file(svrCertFile,
                                                     SSL_FILETYPE_PEM)));
    ExpectNotNull((x5092 = X509_load_certificate_file(cliCertFile,
                                                     SSL_FILETYPE_PEM)));
    ExpectNotNull((sk = sk_X509_new_null()));
    ExpectIntEQ(sk_X509_push(sk, x509), 1);
    if (EXPECT_FAIL()) {
        X509_free(x509);
        x509 = NULL;
    }
    ExpectNotNull((str = X509_STORE_new()));
    ExpectNotNull((ctx = X509_STORE_CTX_new()));
    ExpectIntEQ(X509_STORE_CTX_init(ctx, str, x5092, sk), 1);
    ExpectNull((sk2 = X509_STORE_CTX_get_chain(NULL)));
    ExpectNull((sk2 = X509_STORE_CTX_get_chain(ctx)));
    ExpectNull((sk3 = X509_STORE_CTX_get1_chain(NULL)));
    ExpectNull((sk3 = X509_STORE_CTX_get1_chain(ctx)));
    X509_STORE_CTX_free(ctx);
    ctx = NULL;
    X509_STORE_free(str);
    str = NULL;
    /* CTX certs not freed yet */
    X509_free(x5092);
    x5092 = NULL;
    sk_X509_pop_free(sk, NULL);
    sk = NULL;
    /* sk3 is dup so free here */
    sk_X509_pop_free(sk3, NULL);
    sk3 = NULL;
#endif

    /* test X509_STORE_CTX_get/set_ex_data */
    {
        int i = 0, tmpData = 5;
        void* tmpDataRet;
        ExpectNotNull(ctx = X509_STORE_CTX_new());
    #ifdef HAVE_EX_DATA
        for (i = 0; i < MAX_EX_DATA; i++) {
            ExpectIntEQ(X509_STORE_CTX_set_ex_data(ctx, i, &tmpData),
                        WOLFSSL_SUCCESS);
            tmpDataRet = (int*)X509_STORE_CTX_get_ex_data(ctx, i);
            ExpectNotNull(tmpDataRet);
            ExpectIntEQ(tmpData, *(int*)tmpDataRet);
        }
    #else
        ExpectIntEQ(X509_STORE_CTX_set_ex_data(ctx, i, &tmpData),
                    WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
        tmpDataRet = (int*)X509_STORE_CTX_get_ex_data(ctx, i);
        ExpectNull(tmpDataRet);
    #endif
        X509_STORE_CTX_free(ctx);
        ctx = NULL;
    }

    /* test X509_STORE_get/set_ex_data */
    {
        int i = 0, tmpData = 99;
        void* tmpDataRet;
        ExpectNotNull(str = X509_STORE_new());
    #ifdef HAVE_EX_DATA
        for (i = 0; i < MAX_EX_DATA; i++) {
            ExpectIntEQ(X509_STORE_set_ex_data(str, i, &tmpData),
                        WOLFSSL_SUCCESS);
            tmpDataRet = (int*)X509_STORE_get_ex_data(str, i);
            ExpectNotNull(tmpDataRet);
            ExpectIntEQ(tmpData, *(int*)tmpDataRet);
        }
    #else
        ExpectIntEQ(X509_STORE_set_ex_data(str, i, &tmpData),
                    WC_NO_ERR_TRACE(WOLFSSL_FAILURE));
        tmpDataRet = (int*)X509_STORE_get_ex_data(str, i);
        ExpectNull(tmpDataRet);
    #endif
        X509_STORE_free(str);
        str = NULL;
    }

#endif /* defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
        * !defined(NO_FILESYSTEM) && !defined(NO_RSA) */

    return EXPECT_RESULT();
}

#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
   !defined(NO_FILESYSTEM) && !defined(NO_RSA)

typedef struct {
    const char *caFile;
    const char *caIntFile;
    const char *caInt2File;
    const char *leafFile;
    X509 *x509Ca;
    X509 *x509CaInt;
    X509 *x509CaInt2;
    X509 *x509Leaf;
    STACK_OF(X509)* expectedChain;
} X509_STORE_test_data;

static X509 * test_wolfSSL_X509_STORE_CTX_ex_helper(const char *file)
{
    XFILE fp = XBADFILE;
    X509 *x = NULL;

    fp = XFOPEN(file, "rb");
    if (fp == NULL) {
        return NULL;
    }
    x = PEM_read_X509(fp, 0, 0, 0);
    XFCLOSE(fp);

    return x;
}

static int test_wolfSSL_X509_STORE_CTX_ex1(X509_STORE_test_data *testData)
{
    EXPECT_DECLS;
    X509_STORE* store = NULL;
    X509_STORE_CTX* ctx = NULL;
    STACK_OF(X509)* chain = NULL;
    int i = 0;

    /* Test case 1, add X509 certs to store and verify */
    ExpectNotNull(store = X509_STORE_new());
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509Ca), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt2), 1);
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, testData->x509Leaf, NULL), 1);
    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectNotNull(chain = X509_STORE_CTX_get_chain(ctx));
    ExpectIntEQ(sk_X509_num(chain), sk_X509_num(testData->expectedChain));
    for (i = 0; i < sk_X509_num(chain); i++) {
        ExpectIntEQ(X509_cmp(sk_X509_value(chain, i),
                             sk_X509_value(testData->expectedChain, i)), 0);
    }
    X509_STORE_CTX_free(ctx);
    X509_STORE_free(store);
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_ex2(X509_STORE_test_data *testData)
{
    EXPECT_DECLS;
    X509_STORE* store = NULL;
    X509_STORE_CTX* ctx = NULL;
    STACK_OF(X509)* chain = NULL;
    int i = 0;

    /* Test case 2, add certs by filename to store and verify */
    ExpectNotNull(store = X509_STORE_new());
    ExpectIntEQ(X509_STORE_load_locations(
        store, testData->caFile, NULL), 1);
    ExpectIntEQ(X509_STORE_load_locations(
        store, testData->caIntFile, NULL), 1);
    ExpectIntEQ(X509_STORE_load_locations(
        store, testData->caInt2File, NULL), 1);
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, testData->x509Leaf, NULL), 1);
    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectNotNull(chain = X509_STORE_CTX_get_chain(ctx));
    ExpectIntEQ(sk_X509_num(chain), sk_X509_num(testData->expectedChain));
    for (i = 0; i < sk_X509_num(chain); i++) {
        ExpectIntEQ(X509_cmp(sk_X509_value(chain, i),
                             sk_X509_value(testData->expectedChain, i)), 0);
    }
    X509_STORE_CTX_free(ctx);
    X509_STORE_free(store);
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_ex3(X509_STORE_test_data *testData)
{
    EXPECT_DECLS;
    X509_STORE* store = NULL;
    X509_STORE_CTX* ctx = NULL;
    STACK_OF(X509)* chain = NULL;
    int i = 0;

    /* Test case 3, mix and match X509 with files */
    ExpectNotNull(store = X509_STORE_new());
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt2), 1);
    ExpectIntEQ(X509_STORE_load_locations(
        store, testData->caFile, NULL), 1);
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, testData->x509Leaf, NULL), 1);
    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectNotNull(chain = X509_STORE_CTX_get_chain(ctx));
    ExpectIntEQ(sk_X509_num(chain), sk_X509_num(testData->expectedChain));
    for (i = 0; i < sk_X509_num(chain); i++) {
        ExpectIntEQ(X509_cmp(sk_X509_value(chain, i),
                             sk_X509_value(testData->expectedChain, i)), 0);
    }
    X509_STORE_CTX_free(ctx);
    X509_STORE_free(store);
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_ex4(X509_STORE_test_data *testData)
{
    EXPECT_DECLS;
    X509_STORE* store = NULL;
    X509_STORE_CTX* ctx = NULL;
    STACK_OF(X509)* chain = NULL;
    STACK_OF(X509)* inter = NULL;
    int i = 0;

    /* Test case 4, CA loaded by file, intermediates passed on init */
    ExpectNotNull(store = X509_STORE_new());
    ExpectIntEQ(X509_STORE_load_locations(
        store, testData->caFile, NULL), 1);
    ExpectNotNull(inter = sk_X509_new_null());
    ExpectIntGE(sk_X509_push(inter, testData->x509CaInt), 1);
    ExpectIntGE(sk_X509_push(inter, testData->x509CaInt2), 1);
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, testData->x509Leaf, inter), 1);
    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectNotNull(chain = X509_STORE_CTX_get_chain(ctx));
    ExpectIntEQ(sk_X509_num(chain), sk_X509_num(testData->expectedChain));
    for (i = 0; i < sk_X509_num(chain); i++) {
        ExpectIntEQ(X509_cmp(sk_X509_value(chain, i),
                             sk_X509_value(testData->expectedChain, i)), 0);
    }
    X509_STORE_CTX_free(ctx);
    X509_STORE_free(store);
    sk_X509_free(inter);
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_ex5(X509_STORE_test_data *testData)
{
    EXPECT_DECLS;
    X509_STORE* store = NULL;
    X509_STORE_CTX* ctx = NULL;
    STACK_OF(X509)* chain = NULL;
    STACK_OF(X509)* trusted = NULL;
    int i = 0;

    /* Test case 5, manually set trusted stack */
    ExpectNotNull(store = X509_STORE_new());
    ExpectNotNull(trusted = sk_X509_new_null());
    ExpectIntGE(sk_X509_push(trusted, testData->x509Ca), 1);
    ExpectIntGE(sk_X509_push(trusted, testData->x509CaInt), 1);
    ExpectIntGE(sk_X509_push(trusted, testData->x509CaInt2), 1);
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, testData->x509Leaf, NULL), 1);
    X509_STORE_CTX_trusted_stack(ctx, trusted);
    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectNotNull(chain = X509_STORE_CTX_get_chain(ctx));
    ExpectIntEQ(sk_X509_num(chain), sk_X509_num(testData->expectedChain));
    for (i = 0; i < sk_X509_num(chain); i++) {
        ExpectIntEQ(X509_cmp(sk_X509_value(chain, i),
                             sk_X509_value(testData->expectedChain, i)), 0);
    }
    X509_STORE_CTX_free(ctx);
    X509_STORE_free(store);
    sk_X509_free(trusted);
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_ex6(X509_STORE_test_data *testData)
{
    EXPECT_DECLS;
    X509_STORE* store = NULL;
    X509_STORE_CTX* ctx = NULL;
    STACK_OF(X509)* chain = NULL;
    STACK_OF(X509)* trusted = NULL;
    STACK_OF(X509)* inter = NULL;
    int i = 0;

    /* Test case 6, manually set trusted stack will be unified with
     * any intermediates provided on init */
    ExpectNotNull(store = X509_STORE_new());
    ExpectNotNull(trusted = sk_X509_new_null());
    ExpectNotNull(inter = sk_X509_new_null());
    ExpectIntGE(sk_X509_push(trusted, testData->x509Ca), 1);
    ExpectIntGE(sk_X509_push(inter, testData->x509CaInt), 1);
    ExpectIntGE(sk_X509_push(inter, testData->x509CaInt2), 1);
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, testData->x509Leaf, inter), 1);
    X509_STORE_CTX_trusted_stack(ctx, trusted);
    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectNotNull(chain = X509_STORE_CTX_get_chain(ctx));
    ExpectIntEQ(sk_X509_num(chain), sk_X509_num(testData->expectedChain));
    for (i = 0; i < sk_X509_num(chain); i++) {
        ExpectIntEQ(X509_cmp(sk_X509_value(chain, i),
                             sk_X509_value(testData->expectedChain, i)), 0);
    }
    X509_STORE_CTX_free(ctx);
    X509_STORE_free(store);
    sk_X509_free(trusted);
    sk_X509_free(inter);
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_ex7(X509_STORE_test_data *testData)
{
    EXPECT_DECLS;
    X509_STORE* store = NULL;
    X509_STORE_CTX* ctx = NULL;
    STACK_OF(X509)* chain = NULL;
    int i = 0;

    /* Test case 7, certs added to store after ctx init are still used */
    ExpectNotNull(store = X509_STORE_new());
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, testData->x509Leaf, NULL), 1);
    ExpectIntNE(X509_verify_cert(ctx), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt2), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509Ca), 1);
    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectNotNull(chain = X509_STORE_CTX_get_chain(ctx));
    ExpectIntEQ(sk_X509_num(chain), sk_X509_num(testData->expectedChain));
    for (i = 0; i < sk_X509_num(chain); i++) {
        ExpectIntEQ(X509_cmp(sk_X509_value(chain, i),
                             sk_X509_value(testData->expectedChain, i)), 0);
    }
    X509_STORE_CTX_free(ctx);
    X509_STORE_free(store);
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_ex8(X509_STORE_test_data *testData)
{
    EXPECT_DECLS;
    X509_STORE* store = NULL;
    X509_STORE_CTX* ctx = NULL;
    STACK_OF(X509)* chain = NULL;
    int i = 0;

    /* Test case 8, Only full chain verifies */
    ExpectNotNull(store = X509_STORE_new());
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, testData->x509Leaf, NULL), 1);
    ExpectIntNE(X509_verify_cert(ctx), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt2), 1);
    ExpectIntNE(X509_verify_cert(ctx), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt), 1);
    ExpectIntNE(X509_verify_cert(ctx), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509Ca), 1);
    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectNotNull(chain = X509_STORE_CTX_get_chain(ctx));
    ExpectIntEQ(sk_X509_num(chain), sk_X509_num(testData->expectedChain));
    for (i = 0; i < sk_X509_num(chain); i++) {
        ExpectIntEQ(X509_cmp(sk_X509_value(chain, i),
                             sk_X509_value(testData->expectedChain, i)), 0);
    }
    X509_STORE_CTX_free(ctx);
    X509_STORE_free(store);
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_ex9(X509_STORE_test_data *testData)
{
    EXPECT_DECLS;
    X509_STORE* store = NULL;
    X509_STORE_CTX* ctx = NULL;
    X509_STORE_CTX* ctx2 = NULL;
    STACK_OF(X509)* trusted = NULL;

    /* Test case 9, certs added to store should not be reflected in ctx that
     * has been manually set with a trusted stack, but are reflected in ctx
     * that has not set trusted stack */
    ExpectNotNull(store = X509_STORE_new());
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectNotNull(ctx2 = X509_STORE_CTX_new());
    ExpectNotNull(trusted = sk_X509_new_null());
    ExpectIntGE(sk_X509_push(trusted, testData->x509Ca), 1);
    ExpectIntGE(sk_X509_push(trusted, testData->x509CaInt), 1);
    ExpectIntGE(sk_X509_push(trusted, testData->x509CaInt2), 1);
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, testData->x509Leaf, NULL), 1);
    ExpectIntEQ(X509_STORE_CTX_init(ctx2, store, testData->x509Leaf, NULL), 1);
    ExpectIntNE(X509_verify_cert(ctx), 1);
    ExpectIntNE(X509_verify_cert(ctx2), 1);
    X509_STORE_CTX_trusted_stack(ctx, trusted);
    /* CTX1 should now verify */
    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectIntNE(X509_verify_cert(ctx2), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509Ca), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt2), 1);
    /* CTX2 should now verify */
    ExpectIntEQ(X509_verify_cert(ctx2), 1);
    X509_STORE_CTX_free(ctx);
    X509_STORE_CTX_free(ctx2);
    X509_STORE_free(store);
    sk_X509_free(trusted);
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_ex10(X509_STORE_test_data *testData)
{
    EXPECT_DECLS;
    X509_STORE* store = NULL;
    X509_STORE_CTX* ctx = NULL;
    STACK_OF(X509)* chain = NULL;

    /* Test case 10, ensure partial chain flag works */
    ExpectNotNull(store = X509_STORE_new());
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt2), 1);
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, testData->x509Leaf, NULL), 1);
    /* Fails because chain is incomplete */
    ExpectIntNE(X509_verify_cert(ctx), 1);
    ExpectIntEQ(X509_STORE_set_flags(store, X509_V_FLAG_PARTIAL_CHAIN), 1);
    /* Partial chain now OK */
    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectNotNull(chain = X509_STORE_CTX_get_chain(ctx));
    X509_STORE_CTX_free(ctx);
    X509_STORE_free(store);
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_ex11(X509_STORE_test_data *testData)
{
    EXPECT_DECLS;
    X509_STORE* store = NULL;
    X509_STORE_CTX* ctx = NULL;
    STACK_OF(X509)* chain = NULL;

    /* Test case 11, test partial chain flag on ctx itself */
    ExpectNotNull(store = X509_STORE_new());
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt), 1);
    ExpectIntEQ(X509_STORE_add_cert(store, testData->x509CaInt2), 1);
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, testData->x509Leaf, NULL), 1);
    /* Fails because chain is incomplete */
    ExpectIntNE(X509_verify_cert(ctx), 1);
    X509_STORE_CTX_set_flags(ctx, X509_V_FLAG_PARTIAL_CHAIN);
    /* Partial chain now OK */
    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectNotNull(chain = X509_STORE_CTX_get_chain(ctx));
    X509_STORE_CTX_free(ctx);
    X509_STORE_free(store);
    return EXPECT_RESULT();
}
#endif

static int test_wolfSSL_X509_STORE_CTX_ex(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
   !defined(NO_FILESYSTEM) && !defined(NO_RSA)
    X509_STORE_test_data testData;
    XMEMSET((void *)&testData, 0, sizeof(X509_STORE_test_data));
    testData.caFile =     "./certs/ca-cert.pem";
    testData.caIntFile =  "./certs/intermediate/ca-int-cert.pem";
    testData.caInt2File = "./certs/intermediate/ca-int2-cert.pem";
    testData.leafFile =   "./certs/intermediate/server-chain.pem";

    ExpectNotNull(testData.x509Ca = \
                  test_wolfSSL_X509_STORE_CTX_ex_helper(testData.caFile));
    ExpectNotNull(testData.x509CaInt = \
                  test_wolfSSL_X509_STORE_CTX_ex_helper(testData.caIntFile));
    ExpectNotNull(testData.x509CaInt2 = \
                  test_wolfSSL_X509_STORE_CTX_ex_helper(testData.caInt2File));
    ExpectNotNull(testData.x509Leaf = \
                  test_wolfSSL_X509_STORE_CTX_ex_helper(testData.leafFile));
    ExpectNotNull(testData.expectedChain = sk_X509_new_null());
    ExpectIntGE(sk_X509_push(testData.expectedChain, testData.x509Leaf), 1);
    ExpectIntGE(sk_X509_push(testData.expectedChain, testData.x509CaInt2), 1);
    ExpectIntGE(sk_X509_push(testData.expectedChain, testData.x509CaInt), 1);
    ExpectIntGE(sk_X509_push(testData.expectedChain, testData.x509Ca), 1);

    ExpectIntEQ(test_wolfSSL_X509_STORE_CTX_ex1(&testData), 1);
    ExpectIntEQ(test_wolfSSL_X509_STORE_CTX_ex2(&testData), 1);
    ExpectIntEQ(test_wolfSSL_X509_STORE_CTX_ex3(&testData), 1);
    ExpectIntEQ(test_wolfSSL_X509_STORE_CTX_ex4(&testData), 1);
    ExpectIntEQ(test_wolfSSL_X509_STORE_CTX_ex5(&testData), 1);
    ExpectIntEQ(test_wolfSSL_X509_STORE_CTX_ex6(&testData), 1);
    ExpectIntEQ(test_wolfSSL_X509_STORE_CTX_ex7(&testData), 1);
    ExpectIntEQ(test_wolfSSL_X509_STORE_CTX_ex8(&testData), 1);
    ExpectIntEQ(test_wolfSSL_X509_STORE_CTX_ex9(&testData), 1);
    ExpectIntEQ(test_wolfSSL_X509_STORE_CTX_ex10(&testData), 1);
    ExpectIntEQ(test_wolfSSL_X509_STORE_CTX_ex11(&testData), 1);

    if(testData.x509Ca) {
        X509_free(testData.x509Ca);
    }
    if(testData.x509CaInt) {
        X509_free(testData.x509CaInt);
    }
    if(testData.x509CaInt2) {
        X509_free(testData.x509CaInt2);
    }
    if(testData.x509Leaf) {
        X509_free(testData.x509Leaf);
    }
    if (testData.expectedChain) {
        sk_X509_free(testData.expectedChain);
    }

#endif /* defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
        * !defined(NO_FILESYSTEM) && !defined(NO_RSA) */

    return EXPECT_RESULT();
}


#if defined(OPENSSL_EXTRA) && !defined(NO_RSA)
static int test_X509_STORE_untrusted_load_cert_to_stack(const char* filename,
        STACK_OF(X509)* chain)
{
    EXPECT_DECLS;
    XFILE fp = XBADFILE;
    X509* cert = NULL;

    ExpectTrue((fp = XFOPEN(filename, "rb"))
            != XBADFILE);
    ExpectNotNull(cert = PEM_read_X509(fp, 0, 0, 0 ));
    if (fp != XBADFILE) {
        XFCLOSE(fp);
        fp = XBADFILE;
    }
    ExpectIntGT(sk_X509_push(chain, cert), 0);
    if (EXPECT_FAIL())
        X509_free(cert);

    return EXPECT_RESULT();
}

#if defined(OPENSSL_ALL)

static int last_errcode;
static int last_errdepth;

static int X509Callback(int ok, X509_STORE_CTX *ctx)
{

    if (!ok) {
        last_errcode  = X509_STORE_CTX_get_error(ctx);
        last_errdepth = X509_STORE_CTX_get_error_depth(ctx);
    }
    /* Always return OK to allow verification to continue.*/
    return 1;
}

static int test_X509_STORE_InvalidCa(void)
{
    EXPECT_DECLS;
    const char* filename = "./certs/intermediate/ca_false_intermediate/"
                                                    "test_int_not_cacert.pem";
    const char* srvfile = "./certs/intermediate/ca_false_intermediate/"
                                            "test_sign_bynoca_srv.pem";
    X509_STORE_CTX* ctx = NULL;
    X509_STORE* str = NULL;
    XFILE fp = XBADFILE;
    X509* cert = NULL;
    STACK_OF(X509)* untrusted = NULL;

    last_errcode = 0;
    last_errdepth = 0;

    ExpectTrue((fp = XFOPEN(srvfile, "rb"))
            != XBADFILE);
    ExpectNotNull(cert = PEM_read_X509(fp, 0, 0, 0 ));
    if (fp != XBADFILE) {
        XFCLOSE(fp);
        fp = XBADFILE;
    }

    ExpectNotNull(str = X509_STORE_new());
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectNotNull(untrusted = sk_X509_new_null());

    /* create cert chain stack */
    ExpectIntEQ(test_X509_STORE_untrusted_load_cert_to_stack(filename,
                untrusted), TEST_SUCCESS);

    X509_STORE_set_verify_cb(str, X509Callback);

    ExpectIntEQ(X509_STORE_load_locations(str,
                "./certs/intermediate/ca_false_intermediate/test_ca.pem",
                                                                    NULL), 1);

    ExpectIntEQ(X509_STORE_CTX_init(ctx, str, cert, untrusted), 1);
    ExpectIntEQ(X509_verify_cert(ctx), 1);
    ExpectIntEQ(last_errcode, X509_V_ERR_INVALID_CA);

    X509_free(cert);
    X509_STORE_free(str);
    X509_STORE_CTX_free(ctx);
    sk_X509_pop_free(untrusted, NULL);

    return EXPECT_RESULT();
}
#endif /* OPENSSL_ALL */



static int test_X509_STORE_untrusted_certs(const char** filenames, int ret,
        int err, int loadCA)
{
    EXPECT_DECLS;
    X509_STORE_CTX* ctx = NULL;
    X509_STORE* str = NULL;
    XFILE fp = XBADFILE;
    X509* cert = NULL;
    STACK_OF(X509)* untrusted = NULL;

    ExpectTrue((fp = XFOPEN("./certs/intermediate/server-int-cert.pem", "rb"))
            != XBADFILE);
    ExpectNotNull(cert = PEM_read_X509(fp, 0, 0, 0 ));
    if (fp != XBADFILE) {
        XFCLOSE(fp);
        fp = XBADFILE;
    }

    ExpectNotNull(str = X509_STORE_new());
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectNotNull(untrusted = sk_X509_new_null());

    ExpectIntEQ(X509_STORE_set_flags(str, 0), 1);
    if (loadCA) {
        ExpectIntEQ(X509_STORE_load_locations(str, "./certs/ca-cert.pem", NULL),
                1);
    }
    for (; *filenames; filenames++) {
        ExpectIntEQ(test_X509_STORE_untrusted_load_cert_to_stack(*filenames,
                untrusted), TEST_SUCCESS);
    }

    ExpectIntEQ(X509_STORE_CTX_init(ctx, str, cert, untrusted), 1);
    ExpectIntEQ(X509_verify_cert(ctx), ret);
    ExpectIntEQ(X509_STORE_CTX_get_error(ctx), err);

    X509_free(cert);
    X509_STORE_free(str);
    X509_STORE_CTX_free(ctx);
    sk_X509_pop_free(untrusted, NULL);

    return EXPECT_RESULT();
}
#endif

static int test_X509_STORE_untrusted(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_RSA)
    const char* untrusted1[] = {
        "./certs/intermediate/ca-int2-cert.pem",
        NULL
    };
    const char* untrusted2[] = {
        "./certs/intermediate/ca-int-cert.pem",
        "./certs/intermediate/ca-int2-cert.pem",
        NULL
    };
    const char* untrusted3[] = {
        "./certs/intermediate/ca-int-cert.pem",
        "./certs/intermediate/ca-int2-cert.pem",
        "./certs/ca-cert.pem",
        NULL
    };
    /* Adding unrelated certs that should be ignored */
    const char* untrusted4[] = {
        "./certs/client-ca.pem",
        "./certs/intermediate/ca-int-cert.pem",
        "./certs/server-cert.pem",
        "./certs/intermediate/ca-int2-cert.pem",
        NULL
    };

    /* Only immediate issuer in untrusted chain. Fails since can't build chain
     * to loaded CA. */
    ExpectIntEQ(test_X509_STORE_untrusted_certs(untrusted1, 0,
            X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY, 1), TEST_SUCCESS);
    /* Succeeds because path to loaded CA is available. */
    ExpectIntEQ(test_X509_STORE_untrusted_certs(untrusted2, 1, 0, 1),
            TEST_SUCCESS);
    /* Root CA in untrusted chain is OK so long as CA has been loaded
     * properly */
    ExpectIntEQ(test_X509_STORE_untrusted_certs(untrusted3, 1, 0, 1),
            TEST_SUCCESS);
    /* Still needs properly loaded CA, while including it in untrusted
     * list is not an error, it also doesn't count for verify */
    ExpectIntEQ(test_X509_STORE_untrusted_certs(untrusted3, 0,
                X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY, 0),
            TEST_SUCCESS);
    /* Succeeds because path to loaded CA is available. */
    ExpectIntEQ(test_X509_STORE_untrusted_certs(untrusted4, 1, 0, 1),
            TEST_SUCCESS);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_set_flags(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
   !defined(NO_FILESYSTEM) && !defined(NO_RSA)
    X509_STORE* store = NULL;
    X509* x509 = NULL;

    ExpectNotNull((store = wolfSSL_X509_STORE_new()));
    ExpectNotNull((x509 = wolfSSL_X509_load_certificate_file(svrCertFile,
        WOLFSSL_FILETYPE_PEM)));
    ExpectIntEQ(X509_STORE_add_cert(store, x509), WOLFSSL_SUCCESS);

#ifdef HAVE_CRL
    ExpectIntEQ(X509_STORE_set_flags(store, WOLFSSL_CRL_CHECKALL),
        WOLFSSL_SUCCESS);
#else
    ExpectIntEQ(X509_STORE_set_flags(store, WOLFSSL_CRL_CHECKALL),
        WC_NO_ERR_TRACE(NOT_COMPILED_IN));
#endif

    wolfSSL_X509_free(x509);
    wolfSSL_X509_STORE_free(store);
#endif /* defined(OPENSSL_EXTRA) && !defined(NO_CERTS) &&
        * !defined(NO_FILESYSTEM) && !defined(NO_RSA) */
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_LOOKUP_load_file(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(HAVE_CRL) && \
   !defined(NO_FILESYSTEM) && !defined(NO_RSA) && \
   (!defined(NO_WOLFSSL_CLIENT) || !defined(WOLFSSL_NO_CLIENT_AUTH))
    WOLFSSL_X509_STORE*  store = NULL;
    WOLFSSL_X509_LOOKUP* lookup = NULL;

    ExpectNotNull(store = wolfSSL_X509_STORE_new());
    ExpectNotNull(lookup = X509_STORE_add_lookup(store, X509_LOOKUP_file()));
    ExpectIntEQ(wolfSSL_X509_LOOKUP_load_file(lookup, "certs/client-ca.pem",
        X509_FILETYPE_PEM), 1);
    ExpectIntEQ(wolfSSL_X509_LOOKUP_load_file(lookup, "certs/crl/crl2.pem",
        X509_FILETYPE_PEM), 1);

    if (store != NULL) {
        ExpectIntEQ(wolfSSL_CertManagerVerify(store->cm, cliCertFile,
            WOLFSSL_FILETYPE_PEM), 1);
        ExpectIntEQ(wolfSSL_CertManagerVerify(store->cm, svrCertFile,
            WOLFSSL_FILETYPE_PEM), WC_NO_ERR_TRACE(ASN_NO_SIGNER_E));
    }
    ExpectIntEQ(wolfSSL_X509_LOOKUP_load_file(lookup, "certs/ca-cert.pem",
        X509_FILETYPE_PEM), 1);
    if (store != NULL) {
        ExpectIntEQ(wolfSSL_CertManagerVerify(store->cm, svrCertFile,
            WOLFSSL_FILETYPE_PEM), 1);
    }

    wolfSSL_X509_STORE_free(store);
#endif /* defined(OPENSSL_EXTRA) && defined(HAVE_CRL) &&
        * !defined(NO_FILESYSTEM) && !defined(NO_RSA) */
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_set_time(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA)
    WOLFSSL_X509_STORE_CTX* ctx = NULL;
    time_t c_time;

    ExpectNotNull(ctx = wolfSSL_X509_STORE_CTX_new());
    c_time = 365*24*60*60;
    wolfSSL_X509_STORE_CTX_set_time(ctx, 0, c_time);
    ExpectTrue((ctx->param->flags & WOLFSSL_USE_CHECK_TIME) ==
        WOLFSSL_USE_CHECK_TIME);
    ExpectTrue(ctx->param->check_time == c_time);
    wolfSSL_X509_STORE_CTX_free(ctx);
#endif /* OPENSSL_EXTRA */
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_get0_set1_param(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA)
#if !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    SSL_CTX* ctx = NULL;
    WOLFSSL_X509_VERIFY_PARAM* pParam = NULL;
    WOLFSSL_X509_VERIFY_PARAM* pvpm = NULL;
    char testIPv4[] = "127.0.0.1";
    char testhostName[] = "foo.hoge.com";

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
#endif

    ExpectNull(SSL_CTX_get0_param(NULL));
    ExpectNotNull(pParam = SSL_CTX_get0_param(ctx));

    ExpectNotNull(pvpm = (WOLFSSL_X509_VERIFY_PARAM *)XMALLOC(
        sizeof(WOLFSSL_X509_VERIFY_PARAM), NULL, DYNAMIC_TYPE_OPENSSL));
    ExpectNotNull(XMEMSET(pvpm, 0, sizeof(WOLFSSL_X509_VERIFY_PARAM)));

    ExpectIntEQ(wolfSSL_X509_VERIFY_PARAM_set1_host(pvpm, testhostName,
        (int)XSTRLEN(testhostName)), WOLFSSL_SUCCESS);
    ExpectIntEQ(wolfSSL_X509_VERIFY_PARAM_set1_ip_asc(pvpm, testIPv4),
        WOLFSSL_SUCCESS);
    wolfSSL_X509_VERIFY_PARAM_set_hostflags(pvpm, 0x01);

    ExpectIntEQ(SSL_CTX_set1_param(ctx, pvpm), 1);
    ExpectIntEQ(0, XSTRNCMP(pParam->hostName, testhostName,
        (int)XSTRLEN(testhostName)));
    ExpectIntEQ(0x01, pParam->hostFlags);
    ExpectIntEQ(0, XSTRNCMP(pParam->ipasc, testIPv4, WOLFSSL_MAX_IPSTR));

    /* test for incorrect parameter */
    ExpectIntEQ(1,SSL_CTX_set1_param(ctx, NULL));
    ExpectIntEQ(1,SSL_CTX_set1_param(NULL, pvpm));
    ExpectIntEQ(1,SSL_CTX_set1_param(NULL, NULL));

    SSL_CTX_free(ctx);

    XFREE(pvpm, NULL, DYNAMIC_TYPE_OPENSSL);
#endif /* !NO_WOLFSSL_CLIENT || !NO_WOLFSSL_SERVER */
#endif /* OPENSSL_EXTRA && !defined(NO_RSA)*/
    return EXPECT_RESULT();
}

static int test_wolfSSL_get0_param(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_RSA) && !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    SSL_CTX* ctx = NULL;
    SSL*     ssl = NULL;

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
#endif
    ExpectTrue(SSL_CTX_use_certificate_file(ctx, svrCertFile,
        SSL_FILETYPE_PEM));
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, SSL_FILETYPE_PEM));
    ExpectNotNull(ssl = SSL_new(ctx));

    ExpectNotNull(SSL_get0_param(ssl));

    SSL_free(ssl);
    SSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_VERIFY_PARAM_set1_host(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA)
    const char host[] = "www.example.com";
    WOLFSSL_X509_VERIFY_PARAM* pParam = NULL;

    ExpectNotNull(pParam = (WOLFSSL_X509_VERIFY_PARAM*)XMALLOC(
        sizeof(WOLFSSL_X509_VERIFY_PARAM), HEAP_HINT, DYNAMIC_TYPE_OPENSSL));
    if (pParam != NULL) {
        XMEMSET(pParam, 0, sizeof(WOLFSSL_X509_VERIFY_PARAM));

        ExpectIntEQ(X509_VERIFY_PARAM_set1_host(NULL, host, sizeof(host)),
            WOLFSSL_FAILURE);

        X509_VERIFY_PARAM_set1_host(pParam, host, sizeof(host));

        ExpectIntEQ(XMEMCMP(pParam->hostName, host, sizeof(host)), 0);

        XMEMSET(pParam, 0, sizeof(WOLFSSL_X509_VERIFY_PARAM));

        ExpectIntNE(XMEMCMP(pParam->hostName, host, sizeof(host)), 0);

        XFREE(pParam, HEAP_HINT, DYNAMIC_TYPE_OPENSSL);
    }
#endif /* OPENSSL_EXTRA */
    return EXPECT_RESULT();
}

static int test_wolfSSL_set1_host(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_RSA) && !defined(NO_TLS) && \
    (!defined(NO_WOLFSSL_CLIENT) || !defined(NO_WOLFSSL_SERVER))
    const char host[] = "www.test_wolfSSL_set1_host.com";
    const char emptyStr[] = "";
    SSL_CTX*   ctx = NULL;
    SSL*       ssl = NULL;
    WOLFSSL_X509_VERIFY_PARAM* pParam = NULL;

#ifndef NO_WOLFSSL_SERVER
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_server_method()));
#else
    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
#endif
    ExpectTrue(SSL_CTX_use_certificate_file(ctx, svrCertFile,
        SSL_FILETYPE_PEM));
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, SSL_FILETYPE_PEM));
    ExpectNotNull(ssl = SSL_new(ctx));

    pParam = SSL_get0_param(ssl);

    /* we should get back host string */
    ExpectIntEQ(SSL_set1_host(ssl, host), WOLFSSL_SUCCESS);
    ExpectIntEQ(XMEMCMP(pParam->hostName, host, sizeof(host)), 0);

    /* we should get back empty string */
    ExpectIntEQ(SSL_set1_host(ssl, emptyStr), WOLFSSL_SUCCESS);
    ExpectIntEQ(XMEMCMP(pParam->hostName, emptyStr, sizeof(emptyStr)), 0);

    /* we should get back host string */
    ExpectIntEQ(SSL_set1_host(ssl, host), WOLFSSL_SUCCESS);
    ExpectIntEQ(XMEMCMP(pParam->hostName, host, sizeof(host)), 0);

    /* we should get back empty string */
    ExpectIntEQ(SSL_set1_host(ssl, NULL), WOLFSSL_SUCCESS);
    ExpectIntEQ(XMEMCMP(pParam->hostName, emptyStr, sizeof(emptyStr)), 0);

    SSL_free(ssl);
    SSL_CTX_free(ctx);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_VERIFY_PARAM_set1_ip(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA)
    unsigned char buf[16] = {0};
    WOLFSSL_X509_VERIFY_PARAM* param = NULL;

    ExpectNotNull(param = X509_VERIFY_PARAM_new());

    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(NULL, NULL, 1), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(param, NULL, 1), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(NULL, buf, 1), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(NULL, NULL, 16), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(NULL, NULL, 4), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(NULL, NULL, 0), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(param, buf, 1), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(param, NULL, 16), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(param, NULL, 4), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(NULL, buf, 16), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(NULL, buf, 4), WOLFSSL_FAILURE);
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(NULL, buf, 0), WOLFSSL_FAILURE);

    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(param, NULL, 0), WOLFSSL_SUCCESS);

    /* test 127.0.0.1 */
    buf[0] =0x7f; buf[1] = 0; buf[2] = 0; buf[3] = 1;
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(param, &buf[0], 4), SSL_SUCCESS);
    ExpectIntEQ(XSTRNCMP(param->ipasc, "127.0.0.1", sizeof(param->ipasc)), 0);

    /* test 2001:db8:3333:4444:5555:6666:7777:8888 */
    buf[0]=32;buf[1]=1;buf[2]=13;buf[3]=184;
    buf[4]=51;buf[5]=51;buf[6]=68;buf[7]=68;
    buf[8]=85;buf[9]=85;buf[10]=102;buf[11]=102;
    buf[12]=119;buf[13]=119;buf[14]=136;buf[15]=136;
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(param, &buf[0], 16), SSL_SUCCESS);
    ExpectIntEQ(XSTRNCMP(param->ipasc,
        "2001:db8:3333:4444:5555:6666:7777:8888", sizeof(param->ipasc)), 0);

    /* test 2001:db8:: */
    buf[0]=32;buf[1]=1;buf[2]=13;buf[3]=184;
    buf[4]=0;buf[5]=0;buf[6]=0;buf[7]=0;
    buf[8]=0;buf[9]=0;buf[10]=0;buf[11]=0;
    buf[12]=0;buf[13]=0;buf[14]=0;buf[15]=0;
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(param, &buf[0], 16), SSL_SUCCESS);
    ExpectIntEQ(XSTRNCMP(param->ipasc, "2001:db8::", sizeof(param->ipasc)), 0);

    /* test ::1234:5678 */
    buf[0]=0;buf[1]=0;buf[2]=0;buf[3]=0;
    buf[4]=0;buf[5]=0;buf[6]=0;buf[7]=0;
    buf[8]=0;buf[9]=0;buf[10]=0;buf[11]=0;
    buf[12]=18;buf[13]=52;buf[14]=86;buf[15]=120;
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(param, &buf[0], 16), SSL_SUCCESS);
    ExpectIntEQ(XSTRNCMP(param->ipasc, "::1234:5678", sizeof(param->ipasc)), 0);


    /* test 2001:db8::1234:5678 */
    buf[0]=32;buf[1]=1;buf[2]=13;buf[3]=184;
    buf[4]=0;buf[5]=0;buf[6]=0;buf[7]=0;
    buf[8]=0;buf[9]=0;buf[10]=0;buf[11]=0;
    buf[12]=18;buf[13]=52;buf[14]=86;buf[15]=120;
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(param, &buf[0], 16), SSL_SUCCESS);
    ExpectIntEQ(XSTRNCMP(param->ipasc, "2001:db8::1234:5678",
                                                sizeof(param->ipasc)), 0);

    /* test 2001:0db8:0001:0000:0000:0ab9:c0a8:0102*/
    /*      2001:db8:1::ab9:c0a8:102 */
    buf[0]=32;buf[1]=1;buf[2]=13;buf[3]=184;
    buf[4]=0;buf[5]=1;buf[6]=0;buf[7]=0;
    buf[8]=0;buf[9]=0;buf[10]=10;buf[11]=185;
    buf[12]=192;buf[13]=168;buf[14]=1;buf[15]=2;
    ExpectIntEQ(X509_VERIFY_PARAM_set1_ip(param, &buf[0], 16), SSL_SUCCESS);
    ExpectIntEQ(XSTRNCMP(param->ipasc, "2001:db8:1::ab9:c0a8:102",
                                                sizeof(param->ipasc)), 0);

    XFREE(param, HEAP_HINT, DYNAMIC_TYPE_OPENSSL);
#endif /* OPENSSL_EXTRA */
    return EXPECT_RESULT();
}

static int test_wolfSSL_X509_STORE_CTX_get0_store(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA)
    X509_STORE* store = NULL;
    X509_STORE_CTX* ctx = NULL;
    X509_STORE_CTX* ctx_no_init = NULL;

    ExpectNotNull((store = X509_STORE_new()));
    ExpectNotNull(ctx = X509_STORE_CTX_new());
    ExpectNotNull(ctx_no_init = X509_STORE_CTX_new());
    ExpectIntEQ(X509_STORE_CTX_init(ctx, store, NULL, NULL), SSL_SUCCESS);

    ExpectNull(X509_STORE_CTX_get0_store(NULL));
    /* should return NULL if ctx has not bee initialized */
    ExpectNull(X509_STORE_CTX_get0_store(ctx_no_init));
    ExpectNotNull(X509_STORE_CTX_get0_store(ctx));

    wolfSSL_X509_STORE_CTX_free(ctx);
    wolfSSL_X509_STORE_CTX_free(ctx_no_init);
    X509_STORE_free(store);
#endif /* OPENSSL_EXTRA */
    return EXPECT_RESULT();
}

#if defined(OPENSSL_ALL) && !defined(NO_RSA) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CLIENT) && defined(HAVE_ECC) && !defined(NO_TLS) && \
    defined(HAVE_AESGCM)
static int test_wolfSSL_get_client_ciphers_ctx_ready(WOLFSSL_CTX* ctx)
{
    EXPECT_DECLS;
    ExpectTrue(wolfSSL_CTX_set_cipher_list(ctx, "ECDHE-RSA-AES128-GCM-SHA256"));
    return EXPECT_RESULT();
}


static int test_wolfSSL_get_client_ciphers_on_result(WOLFSSL* ssl) {
    EXPECT_DECLS;
    WOLF_STACK_OF(WOLFSSL_CIPHER)* ciphers;

    ciphers = SSL_get_client_ciphers(ssl);
    if (wolfSSL_is_server(ssl) == 0) {
        ExpectNull(ciphers);
    }
    else {
        WOLFSSL_CIPHER* current;

        /* client should have only sent over one cipher suite */
        ExpectNotNull(ciphers);
        ExpectIntEQ(sk_SSL_CIPHER_num(ciphers), 1);
        current = sk_SSL_CIPHER_value(ciphers, 0);
        ExpectNotNull(current);
    #if !defined(WOLFSSL_CIPHER_INTERNALNAME) && !defined(NO_ERROR_STRINGS) && \
        !defined(WOLFSSL_QT)
        ExpectStrEQ("TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256",
            SSL_CIPHER_get_name(current));
    #else
        ExpectStrEQ("ECDHE-RSA-AES128-GCM-SHA256",
            SSL_CIPHER_get_name(current));
    #endif
    }
    return EXPECT_RESULT();
}
#endif

static int test_wolfSSL_get_client_ciphers(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(NO_RSA) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CLIENT) && defined(HAVE_ECC) && !defined(NO_TLS) && \
    defined(HAVE_AESGCM)
    test_ssl_cbf server_cb;
    test_ssl_cbf client_cb;

    XMEMSET(&client_cb, 0, sizeof(test_ssl_cbf));
    XMEMSET(&server_cb, 0, sizeof(test_ssl_cbf));
    client_cb.method = wolfTLSv1_2_client_method;
    server_cb.method = wolfTLSv1_2_server_method;
    client_cb.devId = testDevId;
    server_cb.devId = testDevId;
    client_cb.ctx_ready = test_wolfSSL_get_client_ciphers_ctx_ready;
    client_cb.on_result = test_wolfSSL_get_client_ciphers_on_result;
    server_cb.on_result = test_wolfSSL_get_client_ciphers_on_result;

    ExpectIntEQ(test_wolfSSL_client_server_nofail_memio(&client_cb,
        &server_cb, NULL), TEST_SUCCESS);
#endif
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_set_client_CA_list(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_ALL) && !defined(NO_RSA) && !defined(NO_CERTS) && \
    !defined(NO_WOLFSSL_CLIENT) && !defined(NO_BIO) && !defined(NO_TLS)
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL* ssl = NULL;
    X509_NAME* name = NULL;
    STACK_OF(X509_NAME)* names = NULL;
    STACK_OF(X509_NAME)* ca_list = NULL;
    int names_len = 0;
    int i;

    ExpectNotNull(ctx = wolfSSL_CTX_new(wolfSSLv23_server_method()));
    /* Send two X501 names in cert request */
    names = SSL_load_client_CA_file(cliCertFile);
    ExpectNotNull(names);
    ca_list = SSL_load_client_CA_file(caCertFile);
    ExpectNotNull(ca_list);
    ExpectNotNull(name = sk_X509_NAME_value(ca_list, 0));
    ExpectIntEQ(sk_X509_NAME_push(names, name), 2);
    if (EXPECT_FAIL()) {
        wolfSSL_X509_NAME_free(name);
        name = NULL;
    }
    SSL_CTX_set_client_CA_list(ctx, names);
    /* This should only free the stack structure */
    sk_X509_NAME_free(ca_list);
    ca_list = NULL;
    ExpectNotNull(ca_list = SSL_CTX_get_client_CA_list(ctx));
    ExpectIntEQ(sk_X509_NAME_num(ca_list), sk_X509_NAME_num(names));

    ExpectIntEQ(sk_X509_NAME_find(NULL, name), BAD_FUNC_ARG);
    ExpectIntEQ(sk_X509_NAME_find(names, NULL), WOLFSSL_FATAL_ERROR);
    ExpectIntGT((names_len = sk_X509_NAME_num(names)), 0);
    for (i = 0; i < names_len; i++) {
        ExpectNotNull(name = sk_X509_NAME_value(names, i));
        ExpectIntEQ(sk_X509_NAME_find(names, name), i);
    }

    /* Needed to be able to create ssl object */
    ExpectTrue(SSL_CTX_use_certificate_file(ctx, svrCertFile,
        SSL_FILETYPE_PEM));
    ExpectTrue(SSL_CTX_use_PrivateKey_file(ctx, svrKeyFile, SSL_FILETYPE_PEM));
    ExpectNotNull(ssl = wolfSSL_new(ctx));
    /* load again as old names are responsibility of ctx to free*/
    names = SSL_load_client_CA_file(cliCertFile);
    ExpectNotNull(names);
    SSL_set_client_CA_list(ssl, names);
    ExpectNotNull(ca_list = SSL_get_client_CA_list(ssl));
    ExpectIntEQ(sk_X509_NAME_num(ca_list), sk_X509_NAME_num(names));

    ExpectIntGT((names_len = sk_X509_NAME_num(names)), 0);
    for (i = 0; i < names_len; i++) {
        ExpectNotNull(name = sk_X509_NAME_value(names, i));
        ExpectIntEQ(sk_X509_NAME_find(names, name), i);
    }

#if !defined(SINGLE_THREADED) && defined(SESSION_CERTS)
    {
        tcp_ready ready;
        func_args server_args;
        callback_functions server_cb;
        THREAD_TYPE serverThread;
        WOLFSSL* ssl_client = NULL;
        WOLFSSL_CTX* ctx_client = NULL;
        SOCKET_T sockfd = 0;

        /* wolfSSL_get_client_CA_list() with handshake */

        StartTCP();
        InitTcpReady(&ready);

        XMEMSET(&server_args, 0, sizeof(func_args));
        XMEMSET(&server_cb, 0, sizeof(callback_functions));

        server_args.signal    = &ready;
        server_args.callbacks = &server_cb;

        /* we are responsible for free'ing WOLFSSL_CTX */
        server_cb.ctx = ctx;
        server_cb.isSharedCtx = 1;

        ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_load_verify_locations(ctx,
            cliCertFile, 0));

        start_thread(test_server_nofail, &server_args, &serverThread);
        wait_tcp_ready(&server_args);

        tcp_connect(&sockfd, wolfSSLIP, server_args.signal->port, 0, 0, NULL);
        ExpectNotNull(ctx_client =
            wolfSSL_CTX_new(wolfTLSv1_2_client_method()));
        ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_load_verify_locations(
            ctx_client, caCertFile, 0));
        ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_use_certificate_file(
            ctx_client, cliCertFile, SSL_FILETYPE_PEM));
        ExpectIntEQ(WOLFSSL_SUCCESS, wolfSSL_CTX_use_PrivateKey_file(
            ctx_client, cliKeyFile, SSL_FILETYPE_PEM));

        ExpectNotNull(ssl_client = wolfSSL_new(ctx_client));
        ExpectIntEQ(wolfSSL_set_fd(ssl_client, sockfd), WOLFSSL_SUCCESS);
        ExpectIntEQ(wolfSSL_connect(ssl_client), WOLFSSL_SUCCESS);

        ExpectNotNull(ca_list = SSL_get_client_CA_list(ssl_client));
        /* We are expecting two cert names to be sent */
        ExpectIntEQ(sk_X509_NAME_num(ca_list), 2);

        ExpectNotNull(names = SSL_CTX_get_client_CA_list(ctx));
        for (i=0; i<sk_X509_NAME_num(ca_list); i++) {
            ExpectNotNull(name = sk_X509_NAME_value(ca_list, i));
            ExpectIntGE(sk_X509_NAME_find(names, name), 0);
        }

        wolfSSL_shutdown(ssl_client);
        wolfSSL_free(ssl_client);
        wolfSSL_CTX_free(ctx_client);

        CloseSocket(sockfd);

        join_thread(serverThread);
        FreeTcpReady(&ready);
    }
#endif

    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
#endif /* OPENSSL_EXTRA && !NO_RSA && !NO_CERTS && !NO_WOLFSSL_CLIENT &&
        * !NO_BIO */
    return EXPECT_RESULT();
}

static int test_wolfSSL_CTX_add_client_CA(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && !defined(NO_RSA) && !defined(NO_CERTS) && \
    !defined(NO_TLS) && !defined(NO_WOLFSSL_CLIENT)
    WOLFSSL_CTX* ctx = NULL;
    WOLFSSL_X509* x509 = NULL;
    WOLFSSL_X509* x509_a = NULL;
    STACK_OF(X509_NAME)* ca_list = NULL;

    ExpectNotNull(ctx = SSL_CTX_new(wolfSSLv23_client_method()));
    /* Add client cert */
    ExpectNotNull(x509 = X509_load_certificate_file(cliCertFile,
        SSL_FILETYPE_PEM));
    ExpectIntEQ(SSL_CTX_add_client_CA(ctx, x509), SSL_SUCCESS);
    ExpectNotNull(ca_list = SSL_CTX_get_client_CA_list(ctx));
    /* Add another client cert */
    ExpectNotNull(x509_a = X509_load_certificate_file(cliCertFile,
        SSL_FILETYPE_PEM));
    ExpectIntEQ(SSL_CTX_add_client_CA(ctx, x509_a), SSL_SUCCESS);

    /* test for incorrect parameter */
    ExpectIntEQ(SSL_CTX_add_client_CA(NULL, x509), 0);
    ExpectIntEQ(SSL_CTX_add_client_CA(ctx, NULL), 0);
    ExpectIntEQ(SSL_CTX_add_client_CA(NULL, NULL), 0);

    X509_free(x509);
    X509_free(x509_a);
    SSL_CTX_free(ctx);
#endif /* OPENSSL_EXTRA  && !NO_RSA && !NO_CERTS && !NO_WOLFSSL_CLIENT */
    return EXPECT_RESULT();
}
#if defined(WOLFSSL_TLS13) && defined(HAVE_ECH) && \
    defined(HAVE_IO_TESTS_DEPENDENCIES)
static THREAD_RETURN WOLFSSL_THREAD server_task_ech(void* args)
{
    callback_functions* callbacks = ((func_args*)args)->callbacks;
    WOLFSSL_CTX* ctx       = callbacks->ctx;
    WOLFSSL*  ssl   = NULL;
    SOCKET_T  sfd   = 0;
    SOCKET_T  cfd   = 0;
    word16    port;
    char      input[1024];
    int       idx;
    int       ret, err = 0;
    const char* privateName = "ech-private-name.com";
    int         privateNameLen = (int)XSTRLEN(privateName);

    ((func_args*)args)->return_code = TEST_FAIL;
    port = ((func_args*)args)->signal->port;

    AssertIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CTX_load_verify_locations(ctx, cliCertFile, 0));

    AssertIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CTX_use_certificate_file(ctx, svrCertFile,
            WOLFSSL_FILETYPE_PEM));

    AssertIntEQ(WOLFSSL_SUCCESS,
        wolfSSL_CTX_use_PrivateKey_file(ctx, svrKeyFile,
                 WOLFSSL_FILETYPE_PEM));

    if (callbacks->ctx_ready)
        callbacks->ctx_ready(ctx);

    ssl = wolfSSL_new(ctx);

    /* set the sni for the server */
    wolfSSL_UseSNI(ssl, WOLFSSL_SNI_HOST_NAME, privateName, privateNameLen);

    tcp_accept(&sfd, &cfd, (func_args*)args, port, 0, 0, 0, 0, 1, NULL, NULL);
    CloseSocket(sfd);
    AssertIntEQ(WOLFSSL_SUCCESS, wolfSSL_set_fd(ssl, cfd));

    if (callbacks->ssl_ready)
        callbacks->ssl_ready(ssl);

    do {
        err = 0; /* Reset error */
        ret = wolfSSL_accept(ssl);
        if (ret != WOLFSSL_SUCCESS) {
            err = wolfSSL_get_error(ssl, 0);
        }
    } while (ret != WOLFSSL_SUCCESS && err == WC_NO_ERR_TRACE(WC_PENDING_E));

    if (ret != WOLFSSL_SUCCESS) {
        char buff[WOLFSSL_MAX_ERROR_SZ];
        fprintf(stderr, "error = %d, %s\n", err, wolfSSL_ERR_error_string(err, buff));
    }
    else {
        if (0 < (idx = wolfSSL_read(ssl, input, sizeof(input)-1))) {
            input[idx] = 0;
           fprintf(stderr, "Client message: %s\n", input);
        }

        AssertIntEQ(privateNameLen, wolfSSL_write(ssl, privateName,
            privateNameLen));
        ((func_args*)args)->return_code = TEST_SUCCESS;
    }

    if (callbacks->on_result)
        callbacks->on_result(ssl);

    wolfSSL_shutdown(ssl);
    wolfSSL_free(ssl);
    wolfSSL_CTX_free(ctx);
    CloseSocket(cfd);

#ifdef FP_ECC
    wc_ecc_fp_free();
#endif

    WOLFSSL_RETURN_FROM_THREAD(0);
}
#endif /* HAVE_ECH && WOLFSSL_TLS13 */

#if defined(OPENSSL_EXTRA) && defined(HAVE_SECRET_CALLBACK)
static void keyLog_callback(const WOLFSSL* ssl, const char* line)
{
    XFILE fp;
    const byte lf = '\n';

    AssertNotNull(ssl);
    AssertNotNull(line);

    fp = XFOPEN("./MyKeyLog.txt", "a");
    XFWRITE(line, 1, XSTRLEN(line), fp);
    XFWRITE((void*)&lf, 1, 1, fp);
    XFFLUSH(fp);
    XFCLOSE(fp);
}
#endif /* OPENSSL_EXTRA && HAVE_SECRET_CALLBACK */
static int test_wolfSSL_CTX_set_keylog_callback(void)
{
    EXPECT_DECLS;
#if defined(OPENSSL_EXTRA) && defined(HAVE_SECRET_CALLBACK) && \
                              !defined(NO_WOLFSSL_CLIENT)
    SSL_CTX* ctx = NULL;