/*
 *	Copyright 1996, University Corporation for Atmospheric Research
 *      See netcdf/COPYRIGHT file for copying and redistribution conditions.
 */
/* $Id: v1hpg.c,v 1.41 1997/02/22 04:08:39 davis Exp $ */

#include "nc.h"
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <assert.h>
#include "rnd.h"
#include "ncx.h"

/*
 * This module defines the external representation
 * of the "header" of a netcdf version one file.
 * For each of the components of the NC structure,
 * There are (static) ncx_len_XXX(), ncx_put_XXX()
 * and v1h_get_XXX() functions. These define the
 * external representation of the components.
 * The exported entry points for the whole NC structure
 * are built up from these.
 */


/*
 * "magic number" at beginning of file: 0x43444601 (big endian)
 * assert(sizeof(ncmagic) % X_ALIGN == 0);
 */
static const schar ncmagic[] = {'C', 'D', 'F', 0x01};


/*
 * v1hgs == "Version 1 Header Get Stream"
 *
 * The netcdf file version 1 header is
 * of unknown and potentially unlimited size.
 * So, we don't know how much to get() on
 * the initial read. We build a stream, 'v1hgs'
 * on top of ncio to do the header get.
 */
typedef struct v1hgs {
	ncio *nciop;
	off_t offset;	/* argument to nciop->get() */
	size_t extent;	/* argument to nciop->get() */
	void *base;	/* beginning of current buffer */
	const void *pos;	/* current position in buffer */
	void *end;	/* end of current buffer = base + extent */
} v1hgs;


/*
 * Release the stream, invalidate buffer
 */
static int
rel_v1hgs(v1hgs *gsp)
{
	int status;
	if(gsp->offset == OFF_NONE || gsp->base == NULL)
		return ENOERR;
	status = gsp->nciop->rel(gsp->nciop, gsp->offset, 0);
	gsp->end = NULL;
	gsp->pos = NULL;
	gsp->base = NULL;
	return status;
}


/*
 * Release the current chunk and get the next one.
 * Also used for initialization when gsp->base == NULL.
 */
static int
fault_v1hgs(v1hgs *gsp, size_t extent)
{
	int status;

	if(gsp->base != NULL)
	{
		const size_t incr = (char *)gsp->pos - (char *)gsp->base;
		status = rel_v1hgs(gsp);
		if(status)
			return status;
		gsp->offset += incr;
	}
	
	if(extent > gsp->extent)
		gsp->extent = extent;	

	status = gsp->nciop->get(gsp->nciop,
		 	gsp->offset, gsp->extent,
			0, &gsp->base);
	if(status)
		return status;

	gsp->pos = gsp->base;
	gsp->end = (char *)gsp->base + gsp->extent;

	return ENOERR;
}


/*
 * Ensure that 'nextread' bytes are available.
 */
static int
check_v1hgs(v1hgs *gsp, size_t nextread)
{

#if 0 /* DEBUG */
fprintf(stderr, "nextread %lu, remaining %lu\n",
	(unsigned long)nextread,
	(unsigned long)((char *)gsp->end - (char *)gsp->pos));
#endif

	if((char *)gsp->pos + nextread < (char *)gsp->end)
		return ENOERR;
	return fault_v1hgs(gsp, nextread);
}

/* End v1hgs */


static int
v1h_get_size_t(v1hgs *gsp, size_t *sp)
{
	int status = check_v1hgs(gsp, X_SIZEOF_SIZE_T);
	if(status != ENOERR)
		return status;
	return ncx_get_size_t(&gsp->pos, sp);
}


/* Begin nc_type */

#define X_SIZEOF_NC_TYPE X_SIZEOF_INT

static int
ncx_put_nc_type(void **xpp, const nc_type *typep)
{
	const int itype = (int) *typep;
	const int status =  ncx_put_int_int(*xpp, &itype);
	*xpp = (void *)((char *)(*xpp) + X_SIZEOF_INT);
	return status;
}


static int
v1h_get_nc_type(v1hgs *gsp, nc_type *typep)
{
	int type = 0;
	int status = check_v1hgs(gsp, X_SIZEOF_INT);
	if(status != ENOERR)
		return status;
	status =  ncx_get_int_int(gsp->pos, &type);
	gsp->pos = (void *)((char *)gsp->pos + X_SIZEOF_INT);
	if(status != ENOERR)
		return status;

	assert(type == NC_BYTE
		|| type == NC_CHAR
		|| type == NC_SHORT
		|| type == NC_INT
		|| type == NC_FLOAT
		|| type == NC_DOUBLE);

	/* else */
	*typep = (nc_type) type;

	return ENOERR;
}

/* End nc_type */
/* Begin NCtype (internal tags) */

#define X_SIZEOF_NCTYPE X_SIZEOF_INT

static int
ncx_put_NCtype(void **xpp, NCtype type)
{
	const int itype = (int) type;
	const int status = ncx_put_int_int(*xpp, &itype);
	*xpp = (void *)((char *)(*xpp) + X_SIZEOF_INT);
	return status;
}

static int
v1h_get_NCtype(v1hgs *gsp, NCtype *typep)
{
	int type = 0;
	int status = check_v1hgs(gsp, X_SIZEOF_INT);
	if(status != ENOERR)
		return status;
	status =  ncx_get_int_int(gsp->pos, &type);
	gsp->pos = (void *)((char *)gsp->pos + X_SIZEOF_INT);
	if(status != ENOERR)
		return status;
	/* else */
	*typep = (NCtype) type;
	return ENOERR;
}

/* End NCtype */
/* Begin NC_string */

/*
 * How much space will the xdr'd string take.
 * Formerly
NC_xlen_string(cdfstr)
 */
static size_t
ncx_len_NC_string(const NC_string *ncstrp)
{
	size_t sz = X_SIZEOF_SIZE_T; /* nchars */

	assert(ncstrp != NULL);

	if(ncstrp->nchars != 0) 
	{
#if 0
		assert(ncstrp->nchars % X_ALIGN == 0);
		sz += ncstrp->nchars;
#else
		sz += _RNDUP(ncstrp->nchars, X_ALIGN);
#endif
	}
	return sz;
}


static int
ncx_put_NC_string(void **xpp, const NC_string *ncstrp)
{
	int status;

#if 0
	assert(ncstrp->nchars % X_ALIGN == 0);
#endif

	assert(ncstrp->nchars <= NC_MAX_NAME);

	status = ncx_put_size_t(xpp, &ncstrp->nchars);
	if(status != ENOERR)
		return status;
	status = ncx_pad_putn_text(xpp, ncstrp->nchars, ncstrp->cp);
	if(status != ENOERR)
		return status;

	return ENOERR;
}


static int
v1h_get_NC_string(v1hgs *gsp, NC_string **ncstrpp)
{
	int status;
	size_t nchars = 0;
	NC_string *ncstrp;

	status = v1h_get_size_t(gsp, &nchars);
	if(status != ENOERR)
		return status;

	assert(nchars <= NC_MAX_NAME);

	ncstrp = new_NC_string(nchars, NULL);
	if(ncstrp == NULL)
	{
		return errno;
	}


#if 0
/* assert(ncstrp->nchars == nchars || ncstrp->nchars - nchars < X_ALIGN); */
	assert(ncstrp->nchars % X_ALIGN == 0);
	status = check_v1hgs(gsp, ncstrp->nchars);
#else
	
	status = check_v1hgs(gsp, _RNDUP(ncstrp->nchars, X_ALIGN));
#endif
	if(status != ENOERR)
		goto unwind_alloc;

	status = ncx_pad_getn_text(&gsp->pos, nchars, ncstrp->cp);
	if(status != ENOERR)
		goto unwind_alloc;

	*ncstrpp = ncstrp;

	return ENOERR;

unwind_alloc:
	free_NC_string(ncstrp);
	return status;
	
}

/* End NC_string */
/* Begin NC_dim */

/*
 * How much space will the xdr'd dim take.
 * Formerly
NC_xlen_dim(dpp)
 */
static size_t
ncx_len_NC_dim(const NC_dim *dimp)
{
	size_t sz;

	assert(dimp != NULL);

	sz = ncx_len_NC_string(dimp->name);
	sz += X_SIZEOF_SIZE_T;

	return(sz);
}


static int
ncx_put_NC_dim(void **xpp, const NC_dim *dimp)
{
	int status;

	status = ncx_put_NC_string(xpp, dimp->name);
	if(status != ENOERR)
		return status;

	status = ncx_put_size_t(xpp, &dimp->size);
	if(status != ENOERR)
		return status;

	return ENOERR;
}

static int
v1h_get_NC_dim(v1hgs *gsp, NC_dim **dimpp)
{
	int status;
	NC_string *ncstrp;
	NC_dim *dimp;

	status = v1h_get_NC_string(gsp, &ncstrp);
	if(status != ENOERR)
		return status;

	dimp = new_x_NC_dim(ncstrp);
	if(dimp == NULL)
	{
		status = errno;
		goto unwind_name;
	}

	status = v1h_get_size_t(gsp, &dimp->size);
	if(status != ENOERR)
	{
		free_NC_dim(dimp); /* frees name */
		return status;
	}

	*dimpp = dimp;

	return ENOERR;

unwind_name:
	free_NC_string(ncstrp);
	return status;
}


static size_t
ncx_len_NC_dimarray(const NC_dimarray *ncap)
{
	size_t xlen = X_SIZEOF_NCTYPE;	/* type */
	xlen += X_SIZEOF_SIZE_T;	/* count */
	if(ncap == NULL)
		return xlen;
	/* else */
	{
		const NC_dim **dpp = (const NC_dim **)ncap->value;
		const NC_dim *const *const end = &dpp[ncap->nelems];
		for(  /*NADA*/; dpp < end; dpp++)
		{
			xlen += ncx_len_NC_dim(*dpp);
		}
	}
	return xlen;
}


static int
ncx_put_NC_dimarray(void **xpp, const NC_dimarray *ncap)
{
	int status;

	assert(*xpp != NULL);

	if(ncap == NULL
#if 1
		/* Backward:
		 * This clause is for 'byte for byte'
		 * backward compatibility.
		 * Strickly speaking, it is 'bug for bug'.
		 */
		|| ncap->nelems == 0
#endif
		)
	{
		/*
		 * Handle empty netcdf
		 */
		const size_t nosz = 0;

		status = ncx_put_NCtype(xpp, NC_UNSPECIFIED);
		if(status != ENOERR)
			return status;
		status = ncx_put_size_t(xpp, &nosz);
		if(status != ENOERR)
			return status;
		return ENOERR;
	}
	/* else */

	status = ncx_put_NCtype(xpp, NC_DIMENSION);
	if(status != ENOERR)
		return status;
	status = ncx_put_size_t(xpp, &ncap->nelems);
	if(status != ENOERR)
		return status;

	{
		const NC_dim **dpp = (const NC_dim **)ncap->value;
		const NC_dim *const *const end = &dpp[ncap->nelems];
		for( /*NADA*/; dpp < end; dpp++)
		{
			status = ncx_put_NC_dim(xpp, *dpp);
			if(status)
				return status;
		}
	}
	return ENOERR;
}


static int
v1h_get_NC_dimarray(v1hgs *gsp, NC_dimarray *ncap)
{
	int status;
	NCtype type = NC_UNSPECIFIED;

	assert(gsp != NULL && gsp->pos != NULL);
	assert(ncap != NULL);
	assert(ncap->value == NULL);

	status = v1h_get_NCtype(gsp, &type);
	if(status != ENOERR)
		return status;

	status = v1h_get_size_t(gsp, &ncap->nelems);
	if(status != ENOERR)
		return status;
	
	if(ncap->nelems == 0)
		return ENOERR;
	/* else */
	if(type != NC_DIMENSION)
		return EINVAL;

	ncap->value = (NC_dim **) malloc(ncap->nelems * sizeof(NC_dim *));
	if(ncap->value == NULL)
		return errno;
	ncap->nalloc = ncap->nelems;

	{
		NC_dim **dpp = ncap->value;
		NC_dim *const *const end = &dpp[ncap->nelems];
		for( /*NADA*/; dpp < end; dpp++)
		{
			status = v1h_get_NC_dim(gsp, dpp);
			if(status)
			{
				ncap->nelems = dpp - ncap->value;
				free_NC_dimarrayV(ncap);
				return status;
			}
		}
	}

	return ENOERR;
}


/* End NC_dim */
/* Begin NC_attr */


/*
 * How much space will 'attrp' take in external representation?
 * Formerly
NC_xlen_attr(app)
 */
static size_t
ncx_len_NC_attr(const NC_attr *attrp)
{
	size_t sz;

	assert(attrp != NULL);

	sz = ncx_len_NC_string(attrp->name);
	sz += X_SIZEOF_NC_TYPE; /* type */
	sz += X_SIZEOF_SIZE_T; /* nelems */
	sz += attrp->xsz;

	return(sz);
}


static int
ncx_put_NC_attr(void **xpp, const NC_attr *attrp)
{
	int status;

	status = ncx_put_NC_string(xpp, attrp->name);
	if(status != ENOERR)
		return status;

	status = ncx_put_nc_type(xpp, &attrp->type);
	if(status != ENOERR)
		return status;

	status = ncx_put_size_t(xpp, &attrp->nelems);
	if(status != ENOERR)
		return status;

	(void) memcpy(*xpp, attrp->xvalue, attrp->xsz);
	*xpp = (void *)((char *)(*xpp) + attrp->xsz);

	return ENOERR;
}


#undef MIN
#define MIN(mm,nn) (((mm) < (nn)) ? (mm) : (nn))

/*
 * Get the values of an attribute
 * The loop is necessary since attrp->nelems
 * could potentially be quite large.
 */
static int
v1h_get_NC_attrV(v1hgs *gsp, NC_attr *attrp)
{
	int status;
	const size_t perchunk =  gsp->extent;
	size_t remaining = attrp->xsz;
	void *value = attrp->xvalue;
	size_t nget; 

	assert(gsp->extent % X_ALIGN == 0);
	
	do {
		nget = MIN(perchunk, remaining);
	
		status = check_v1hgs(gsp, nget);
		if(status != ENOERR)
			return status;
	
		(void) memcpy(value, gsp->pos, nget);

		gsp->pos = (void *)((char *)gsp->pos + nget);
		value = (void *)((char *)value + nget);
		remaining -= nget;

	} while(remaining != 0); 

	return ENOERR;
}


static int
v1h_get_NC_attr(v1hgs *gsp, NC_attr **attrpp)
{
	NC_string *strp;
	int status;
	nc_type type;
	size_t nelems;
	NC_attr *attrp;

	status = v1h_get_NC_string(gsp, &strp);
	if(status != ENOERR)
		return status;

	status = v1h_get_nc_type(gsp, &type);
	if(status != ENOERR)
		goto unwind_name;

	status = v1h_get_size_t(gsp, &nelems);
	if(status != ENOERR)
		goto unwind_name;

	attrp = new_x_NC_attr(strp, type, nelems);
	if(attrp == NULL)
	{
		status = errno;
		goto unwind_name;
	}
	
	status = v1h_get_NC_attrV(gsp, attrp);
	if(status != ENOERR)
	{
		free_NC_attr(attrp); /* frees strp */
		return status;
	}

	*attrpp = attrp;

	return ENOERR;

unwind_name:
	free_NC_string(strp);
	return status;
}


static size_t
ncx_len_NC_attrarray(const NC_attrarray *ncap)
{
	size_t xlen = X_SIZEOF_NCTYPE;	/* type */
	xlen += X_SIZEOF_SIZE_T;	/* count */
	if(ncap == NULL)
		return xlen;
	/* else */
	{
		const NC_attr **app = (const NC_attr **)ncap->value;
		const NC_attr *const *const end = &app[ncap->nelems];
		for( /*NADA*/; app < end; app++)
		{
			xlen += ncx_len_NC_attr(*app);
		}
	}
	return xlen;
}


static int
ncx_put_NC_attrarray(void **xpp, const NC_attrarray *ncap)
{
	int status;

	assert(*xpp != NULL);

	if(ncap == NULL
#if 1
		/* Backward:
		 * This clause is for 'byte for byte'
		 * backward compatibility.
		 * Strickly speaking, it is 'bug for bug'.
		 */
		|| ncap->nelems == 0
#endif
		)
	{
		/*
		 * Handle empty netcdf
		 */
		const size_t nosz = 0;

		status = ncx_put_NCtype(xpp, NC_UNSPECIFIED);
		if(status != ENOERR)
			return status;
		status = ncx_put_size_t(xpp, &nosz);
		if(status != ENOERR)
			return status;
		return ENOERR;
	}
	/* else */

	status = ncx_put_NCtype(xpp, NC_ATTRIBUTE);
	if(status != ENOERR)
		return status;
	status = ncx_put_size_t(xpp, &ncap->nelems);
	if(status != ENOERR)
		return status;

	{
		const NC_attr **app = (const NC_attr **)ncap->value;
		const NC_attr *const *const end = &app[ncap->nelems];
		for( /*NADA*/; app < end; app++)
		{
			status = ncx_put_NC_attr(xpp, *app);
			if(status)
				return status;
		}
	}
	return ENOERR;
}


static int
v1h_get_NC_attrarray(v1hgs *gsp, NC_attrarray *ncap)
{
	int status;
	NCtype type = NC_UNSPECIFIED;

	assert(gsp != NULL && gsp->pos != NULL);
	assert(ncap != NULL);
	assert(ncap->value == NULL);

	status = v1h_get_NCtype(gsp, &type);
	if(status != ENOERR)
		return status;
	status = v1h_get_size_t(gsp, &ncap->nelems);
	if(status != ENOERR)
		return status;
	
	if(ncap->nelems == 0)
		return ENOERR;
	/* else */
	if(type != NC_ATTRIBUTE)
		return EINVAL;

	ncap->value = (NC_attr **) malloc(ncap->nelems * sizeof(NC_attr *));
	if(ncap->value == NULL)
		return errno;
	ncap->nalloc = ncap->nelems;

	{
		NC_attr **app = ncap->value;
		NC_attr *const *const end = &app[ncap->nelems];
		for( /*NADA*/; app < end; app++)
		{
			status = v1h_get_NC_attr(gsp, app);
			if(status)
			{
				ncap->nelems = app - ncap->value;
				free_NC_attrarrayV(ncap);
				return status;
			}
		}
	}

	return ENOERR;
}

/* End NC_attr */
/* Begin NC_var */

/*
 * How much space will the xdr'd var take.
 * Formerly
NC_xlen_var(vpp)
 */
static size_t
ncx_len_NC_var(const NC_var *varp)
{
	size_t sz;

	assert(varp != NULL);

	sz = ncx_len_NC_string(varp->name);
	sz += X_SIZEOF_SIZE_T; /* ndims */
	sz += ncx_len_int(varp->ndims); /* dimids */
	sz += ncx_len_NC_attrarray(&varp->attrs);
	sz += X_SIZEOF_NC_TYPE; /* type */
	sz += X_SIZEOF_SIZE_T; /* len */
	sz += X_SIZEOF_OFF_T; /* begin */

	return(sz);
}


static int
ncx_put_NC_var(void **xpp, const NC_var *varp)
{
	int status;

	status = ncx_put_NC_string(xpp, varp->name);
	if(status != ENOERR)
		return status;

	status = ncx_put_size_t(xpp, &varp->ndims);
	if(status != ENOERR)
		return status;

	status = ncx_putn_int_int(xpp, varp->ndims, varp->dimids);
	if(status != ENOERR)
		return status;

	status = ncx_put_NC_attrarray(xpp, &varp->attrs);
	if(status != ENOERR)
		return status;

	status = ncx_put_nc_type(xpp, &varp->type);
	if(status != ENOERR)
		return status;

	status = ncx_put_size_t(xpp, &varp->len);
	if(status != ENOERR)
		return status;

	status = ncx_put_off_t(xpp, &varp->begin);
	if(status != ENOERR)
		return status;

	return ENOERR;
}


static int
v1h_get_NC_var(v1hgs *gsp, NC_var **varpp)
{
	NC_string *strp;
	int status;
	size_t ndims;
	NC_var *varp;

	status = v1h_get_NC_string(gsp, &strp);
	if(status != ENOERR)
		return status;

	status = v1h_get_size_t(gsp, &ndims);
	if(status != ENOERR)
		goto unwind_name;

	varp = new_x_NC_var(strp, ndims);
	if(varp == NULL)
	{
		status = errno;
		goto unwind_name;
	}

	status = check_v1hgs(gsp, ncx_len_int(ndims));
	if(status != ENOERR)
		goto unwind_alloc;
	status = ncx_getn_int_int(&gsp->pos, ndims, varp->dimids);
	if(status != ENOERR)
		goto unwind_alloc;

	status = v1h_get_NC_attrarray(gsp, &varp->attrs);
	if(status != ENOERR)
		goto unwind_alloc;

	status = v1h_get_nc_type(gsp, &varp->type);
	if(status != ENOERR)
		 goto unwind_alloc;

	status = v1h_get_size_t(gsp, &varp->len);
	if(status != ENOERR)
		 goto unwind_alloc;

	status = check_v1hgs(gsp, X_SIZEOF_OFF_T);
	if(status != ENOERR)
		 goto unwind_alloc;
	status = ncx_get_off_t(&gsp->pos, &varp->begin);
	if(status != ENOERR)
		 goto unwind_alloc;
	
	*varpp = varp;
	return ENOERR;

unwind_alloc:
	free_NC_var(varp); /* frees name */
	return status;

unwind_name:
	free_NC_string(strp);
	return status;
}


static size_t
ncx_len_NC_vararray(const NC_vararray *ncap)
{
	size_t xlen = X_SIZEOF_NCTYPE;	/* type */
	xlen += X_SIZEOF_SIZE_T;	/* count */
	if(ncap == NULL)
		return xlen;
	/* else */
	{
		const NC_var **vpp = (const NC_var **)ncap->value;
		const NC_var *const *const end = &vpp[ncap->nelems];
		for( /*NADA*/; vpp < end; vpp++)
		{
			xlen += ncx_len_NC_var(*vpp);
		}
	}
	return xlen;
}


static int
ncx_put_NC_vararray(void **xpp, const NC_vararray *ncap)
{
	int status;

	assert(*xpp != NULL);

	if(ncap == NULL
#if 1
		/* Backward:
		 * This clause is for 'byte for byte'
		 * backward compatibility.
		 * Strickly speaking, it is 'bug for bug'.
		 */
		|| ncap->nelems == 0
#endif
		)
	{
		/*
		 * Handle empty netcdf
		 */
		const size_t nosz = 0;

		status = ncx_put_NCtype(xpp, NC_UNSPECIFIED);
		if(status != ENOERR)
			return status;
		status = ncx_put_size_t(xpp, &nosz);
		if(status != ENOERR)
			return status;
		return ENOERR;
	}
	/* else */

	status = ncx_put_NCtype(xpp, NC_VARIABLE);
	if(status != ENOERR)
		return status;
	status = ncx_put_size_t(xpp, &ncap->nelems);
	if(status != ENOERR)
		return status;

	{
		const NC_var **vpp = (const NC_var **)ncap->value;
		const NC_var *const *const end = &vpp[ncap->nelems];
		for( /*NADA*/; vpp < end; vpp++)
		{
			status = ncx_put_NC_var(xpp, *vpp);
			if(status)
				return status;
		}
	}
	return ENOERR;
}


static int
v1h_get_NC_vararray(v1hgs *gsp, NC_vararray *ncap)
{
	int status;
	NCtype type = NC_UNSPECIFIED;

	assert(gsp != NULL && gsp->pos != NULL);
	assert(ncap != NULL);
	assert(ncap->value == NULL);

	status = v1h_get_NCtype(gsp, &type);
	if(status != ENOERR)
		return status;
	
	status = v1h_get_size_t(gsp, &ncap->nelems);
	if(status != ENOERR)
		return status;
	
	if(ncap->nelems == 0)
		return ENOERR;
	/* else */
	if(type != NC_VARIABLE)
		return EINVAL;

	ncap->value = (NC_var **) malloc(ncap->nelems * sizeof(NC_var *));
	if(ncap->value == NULL)
		return errno;
	ncap->nalloc = ncap->nelems;

	{
		NC_var **vpp = ncap->value;
		NC_var *const *const end = &vpp[ncap->nelems];
		for( /*NADA*/; vpp < end; vpp++)
		{
			status = v1h_get_NC_var(gsp, vpp);
			if(status)
			{
				ncap->nelems = vpp - ncap->value;
				free_NC_vararrayV(ncap);
				return status;
			}
		}
	}

	return ENOERR;
}


/* End NC_var */
/* Begin NC */


/*
 * Recompute the shapes of all variables
 * Sets ncp->begin_var to start of first variable.
 * Sets ncp->begin_rec to start of first record variable.
 * Returns -1 on error. The only possible error is an reference
 * to a non existent dimension, which would occur for a corrupt
 * netcdf file.
 */
static int
NC_computeshapes(NC *ncp)
{
	NC_var **vpp = (NC_var **)ncp->vars.value;
	NC_var *const *const end = &vpp[ncp->vars.nelems];
	NC_var *first_var = NULL;	/* first "non-record" var */
	NC_var *first_rec = NULL;	/* first "record" var */
	int status;

	ncp->begin_var = (off_t) ncp->xsz;
	ncp->begin_rec = (off_t) ncp->xsz;
	ncp->recsize = 0;

	if(ncp->vars.nelems == 0)
		return(0);
	
	for( /*NADA*/; vpp < end; vpp++)
	{
		status = NC_var_shape(*vpp, &ncp->dims);
		if(status != ENOERR)
			return(status);

	  	if(IS_RECVAR(*vpp))	
		{
	  		if(first_rec == NULL)	
				first_rec = *vpp;
			ncp->recsize += (*vpp)->len;
		}
		else if(first_var == NULL)
		{
			first_var = *vpp;
			/*
			 * Overwritten each time thru.
			 * Usually overwritten in first_rec != NULL clause.
			 */
			ncp->begin_rec = (*vpp)->begin + (off_t)(*vpp)->len;
		}
	}

	if(first_rec != NULL)
	{
		assert(ncp->begin_rec <= first_rec->begin);
		ncp->begin_rec = first_rec->begin;
		/*
	 	 * for special case of exactly one record variable, pack value
	 	 */
		if(ncp->recsize == first_rec->len)
			ncp->recsize = *first_rec->dsizes * first_rec->xsz;
	}

	if(first_var != NULL)
	{
		ncp->begin_var = first_var->begin;
	}
	else
	{
		ncp->begin_var = ncp->begin_rec;
	}

	assert(ncp->begin_var > 0);
	assert(ncp->xsz <= (size_t)ncp->begin_var);
	assert(ncp->begin_rec > 0);
	assert(ncp->begin_var <= (size_t)ncp->begin_rec);
	
	return(ENOERR);
}


size_t
ncx_len_NC(const NC *ncp)
{
	size_t xlen = sizeof(ncmagic);

	assert(ncp != NULL);
	
	xlen += X_SIZEOF_SIZE_T; /* numrecs */
	xlen += ncx_len_NC_dimarray(&ncp->dims);
	xlen += ncx_len_NC_attrarray(&ncp->attrs);
	xlen += ncx_len_NC_vararray(&ncp->vars);

	return xlen;
}


int
ncx_put_NC(void **xpp, const NC *ncp)
{
	int status;

	assert(*xpp != NULL);
	assert(ncp != NULL);

	status = ncx_putn_schar_schar(xpp, sizeof(ncmagic), ncmagic);
	if(status != ENOERR)
		return status;

	status = ncx_put_size_t(xpp, &ncp->numrecs);
	if(status != ENOERR)
		return status;

	status = ncx_put_NC_dimarray(xpp, &ncp->dims);
	if(status != ENOERR)
		return status;

	status = ncx_put_NC_attrarray(xpp, &ncp->attrs);
	if(status != ENOERR)
		return status;

	status = ncx_put_NC_vararray(xpp, &ncp->vars);
	if(status != ENOERR)
		return status;

	return ENOERR;
}


int
nc_get_NC(NC *ncp)
{
	int status;
	v1hgs gs;

	assert(ncp != NULL);

	/* Initialize stream gs */

	gs.nciop = ncp->nciop;
	gs.offset = 0; /* beginning of file */
	gs.extent = 0;
	gs.base = NULL;

	{
		/*
		 * Come up with a reasonable stream read size.
		 */
		size_t extent = ncp->xsz;
		if(extent <= MIN_NC_XSZ)
		{
			/* first time read */
			extent = ncp->chunk;
			/* Protection for when ncp->chunk is huge;
			 * no need to read hugely. */
	      		if(extent > 8192)
				extent = 8192;
		}
		else if(extent > ncp->chunk)
		{
			extent = ncp->chunk;
		}
		
		status = fault_v1hgs(&gs, extent);
		if(status)
			return status;
	}

	/* get the header from the stream gs */

	{
		/* Get & check magic number */
		schar magic[sizeof(ncmagic)];
		(void) memset(magic, 0, sizeof(magic));

		status = ncx_getn_schar_schar(&gs.pos, sizeof(magic), magic);
		if(status != ENOERR)
			goto unwind_get;
	
		if(memcmp(magic, ncmagic, sizeof(ncmagic)) != 0)
		{
			status = NC_ENOTNC;
			goto unwind_get;
		}
	}
	
	status = ncx_get_size_t(&gs.pos, &ncp->numrecs);
	if(status != ENOERR)
		goto unwind_get;

	assert((char *)gs.pos < (char *)gs.end);

	status = v1h_get_NC_dimarray(&gs, &ncp->dims);
	if(status != ENOERR)
		goto unwind_get;

	status = v1h_get_NC_attrarray(&gs, &ncp->attrs);
	if(status != ENOERR)
		goto unwind_get;

	status = v1h_get_NC_vararray(&gs, &ncp->vars);
	if(status != ENOERR)
		goto unwind_get;
		
	ncp->xsz = ncx_len_NC(ncp);

	status = NC_computeshapes(ncp);

unwind_get:
	(void) rel_v1hgs(&gs);
	return status;
}
