/*
 *	Copyright 1993, University Corporation for Atmospheric Research
 *      See netcdf/COPYRIGHT file for copying and redistribution conditions.
 */
/* $Id: error.c,v 1.14 90/02/23 16:08:55 davis Exp */

/*LINTLIBRARY*/

#include "ncconfig.h"
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <errno.h>
#include "error.h"
#include "netcdf.h"


#ifndef NO_STRERROR
#include <string.h> /* contains prototype for ansi libc function strerror() */
#else
/* provide a strerror function for older unix systems */
static char *
strerror(int errnum)
{
    extern int sys_nerr;
    extern char *sys_errlist[];

    if(errnum < 0 || errnum >= sys_nerr) return NULL;
    /* else */
    return sys_errlist[errnum];
}
#endif /* NO_STRERROR */


#ifdef vms
/*
 * On the vms system, when a system error occurs which is not
 * mapped into the unix styled errno values, errno is set EVMSERR
 * and a VMS error code is set in vaxc$errno.
 * This routine prints the systems message associated with status return
 * from a system services call.
 */

#include <descrip.h>
#include <ssdef.h>

static int
vmserror1( int status )
{
	short msglen;
	char msgbuf[256];
	$DESCRIPTOR(message, msgbuf);
	register ret;

	ret = SYS$GETMSG(status, &msglen, &message, 15, 0);
	
	switch(ret) {
	case SS$_BUFFEROVF :
	case SS$_NORMAL :
		(void) fprintf(stderr, "%.*s\n", msglen, msgbuf);
		break;
	default :
		break;
	}
	return(ret);
}
#endif /* vms */


static char unknown[] = "Unknown Error";

#if 0
/*
 * Log SYSTEM errors
 * Use where you would want to call perror(3).
 * Calling sequence is
 *	nc_serror(format, arg1, arg2,...)
 * with zero or more args of types compatible with the associated format
 * specifiers.  For example:
 *         nc_serror("shutting down");
 *	   nc_serror("can't open %s", file_name);
 *         nc_serror("process %d in state %s",pid,state);
 */
void
nc_serror(char *fmt, ...)
{

    if( ncopts & NC_VERBOSE )
	{
	    va_list args;
	    int errnum = errno;		/* save real errno in case we wipe it out */
	    char * cp;
	    va_start(args, fmt);
	    (void) fprintf(stderr,"%s: ", cdf_routine_name);
	    (void) vfprintf(stderr,fmt,args);
	    va_end(args);

	    switch(errnum) {
	    case 0 :
		(void) fputc('\n',stderr);
		break;
#ifdef vms
	    case EVMSERR :
		(void) fputc(': ',stderr);
		(void) vmserror1(vaxc$errno);
		break;
#endif /* vms */
	    default :
		(void) fprintf(stderr,": %s\n",
			       (cp = strerror(errnum)) == NULL ? unknown : cp );
		break;
	    }
	    (void) fflush(stderr);	/* to ensure log files are current */
	    errno = errnum;
	} /* NC_VERBOSE */

    if( errno == 0 )
	{
	    ncerr = NC_NOERR;
	}
    else
	{
	ncerr = NC_SYSERR;
	}
    
    if( ncopts & NC_FATAL )
	{
	    exit(ncopts);
	}
}


/*
 * Like nc_serror above, but doesn't check for system error.
 * Use for logging error conditions which are not system errors.
 * Calling sequence is
 *	NCadvise(ncerr, format, arg1, arg2,...)
 * with zero or more args of types compatible with the associated format
 * specifiers.  For example:
 *         NCadvise(NC_NOERR, "just advice");
 *         NCadvise(NC_EBADID, "%d is not a valid cdf id", cdfid);
 */
void
NCadvise(int err, char *fmt,...)
{
	va_list args;

	ncerr = err;

	if( ncopts & NC_VERBOSE )
	{
		(void) fprintf(stderr,"%s: ", cdf_routine_name);
		va_start(args ,fmt);
		(void) vfprintf(stderr,fmt,args);
		va_end(args);
		(void) fputc('\n',stderr);
		(void) fflush(stderr);	/* to ensure log files are current */
	}

	if( (ncopts & NC_FATAL) && ncerr != NC_NOERR )
	{
		exit(ncopts);
	}
}


/*
 * Like nc_serror above, but you pass in the system errno.
 * Calling sequence is
 *	NCserror(errno, format, arg1, arg2,...)
 * with zero or more args of types compatible with the associated format
 * specifiers.  For example:
 *         NCserror(EACCESS, "Couldn't open %s", path);
 */
void
NCserror(int syserr, char *fmt,...)
{
	va_list args;

	ncerr = NC_SYSERR;

	if( ncopts & NC_VERBOSE )
	{
		char * cp;
		(void) fprintf(stderr,"%s: ", cdf_routine_name);
		va_start(args ,fmt);
		(void) vfprintf(stderr,fmt,args);
		va_end(args);
		switch(syserr) {
		case 0 :
			(void) fputc('\n',stderr);
			break;
#ifdef vms
		case EVMSERR :
			(void) fputc(': ',stderr);
			(void) vmserror1(vaxc$errno);
			break;
#endif /* vms */
		default :
			(void) fprintf(stderr,": %s\n",
				(cp = strerror(syserr)) == NULL ?
					unknown : cp );
			break;
		}
		(void) fflush(stderr);	/* to ensure log files are current */
	}

	if( (ncopts & NC_FATAL) && syserr != 0 )
	{
		exit(ncopts);
	}
}
#endif /* old */


const char *
nc_strerror(int err)
{

#if 0 /* vms */
	if(err == EVMSERR)
	{
		/* TODO */
	}	
#endif /* vms */

	if(err > 0)
	{
		const char *cp = (const char *) strerror(err);
		if(cp == NULL)
			return unknown;
		/* else */
		return cp;
	}
	/* else */
	switch (err) {
	case NC_NOERR:
	    return "No error";
	case NC_EBADID:
	    return "Not a netCDF id";
	case NC_ENFILE:
	    return "Too many netCDF files open";
	case NC_EEXIST:
	    return "netCDF file exists && NC_NOCLOBBER";
	case NC_EINVAL:
	    return "Invalid argument";
	case NC_EPERM:
	    return "Write to read only";
	case NC_ENOTINDEFINE:
	    return "Operation not allowed in data mode";
	case NC_EINDEFINE:
	    return "Operation not allowed in define mode";
	case NC_EINVALCOORDS:
	    return "Index exceeds dimension bound";
	case NC_EMAXDIMS:
	    return "NC_MAX_DIMS exceeded";
	case NC_ENAMEINUSE:
	    return "String match to name in use";
	case NC_ENOTATT:
	    return "Attribute not found";
	case NC_EMAXATTS:
	    return "NC_MAX_ATTRS exceeded";
	case NC_EBADTYPE:
	    return "Not a netCDF data type";
	case NC_EBADDIM:
	    return "Invalid dimension id or name";
	case NC_EUNLIMPOS:
	    return "NC_UNLIMITED in the wrong index";
	case NC_EMAXVARS:
	    return "NC_MAX_VARS exceeded";
	case NC_ENOTVAR:
	    return "Variable not found";
	case NC_EGLOBAL:
	    return "Action prohibited on NC_GLOBAL varid";
	case NC_ENOTNC:
	    return "Not a netCDF file";
	case NC_ESTS:
	    return "In Fortran, string too short";
	case NC_EMAXNAME:
	    return "NC_MAX_NAME exceeded";
	case NC_EUNLIMIT:
	    return "NC_UNLIMITED size already in use";
	case NC_ENORECVARS:
	    return "nc_rec op when there are no record vars";
	case NC_ECHAR:
	    return "Attempt to convert between text & numbers";
	case NC_EEDGE:
	    return "Edge+start exceeds dimension bound";
	case NC_ESTRIDE:
	    return "Illegal stride";
	case NC_EBADNAME:
	    return "Attribute or variable name contains illegal characters";
	case NC_ERANGE:
	    return "Numeric conversion not representable";
	}
	/* default */
	return unknown;
}
