!---------------------------------------------------------------
! Copyright (C) 2009-2015 GFD Dennou Club. All rights reserved.
!---------------------------------------------------------------

module Ellip_Slv   ! 代数演算を用いて楕円型偏微分方程式を解くモジュール

  use math_const
  use derivation

  public  :: Ellip_GauSei_2d,  &
  &          Ellip_Gausei_2df,  &
  &          Ellip_Gausei_2dd

  public  :: Ellip_GauSei_3d,  &
  &          Ellip_Gausei_3df,  &
  &          Ellip_Gausei_3dd

  public  :: Ellip_Jacobi_2d,  &
  &          Ellip_Jacobi_2df,  &
  &          Ellip_Jacobi_2dd

  public  :: Ellip_Jacobi_3d,  &
  &          Ellip_Jacobi_3df,  &
  &          Ellip_Jacobi_3dd

  public  :: Full_Multi_Grid_2d,  &
  &          Full_Multi_Grid_2df,  &
  &          Full_Multi_Grid_2dd

  public  :: Full_Multi_Grid_3d,  &
  &          Full_Multi_Grid_3df,  &
  &          Full_Multi_Grid_3dd

  public  :: GauSei_Vcycle_2d,  &
  &          GauSei_Vcycle_2df,  &
  &          GauSei_Vcycle_2dd

  public  :: GauSei_Vcycle_3d,  &
  &          GauSei_Vcycle_3df,  &
  &          GauSei_Vcycle_3dd

  private :: set_bound,  &
  &          setval_boundf,  &
  &          setval_boundd,  &
  &          set_coef,  &
  &          set_coed,  &
  &          calculate_boundf,  &
  &          calculate_boundd

  private :: set_bound_3d,  &
  &          setval_bound_3df,  &
  &          setval_bound_3dd,  &
  &          set_coe_3df,  &
  &          set_coe_3dd,  &
  &          calculate_bound_3df,  &
  &          calculate_bound_3dd

  private :: forward_interpo_2d,  &
  &          forward_interpo_2df,  &
  &          forward_interpo_2dd,  &
  &          forward_interpo_3d,  &
  &          forward_interpo_3df,  &
  &          forward_interpo_3dd,  &
  &          backward_interpo_2d,  &
  &          backward_interpo_2df,  &
  &          backward_interpo_2dd,  &
  &          backward_interpo_3d,  &
  &          backward_interpo_3df,  &
  &          backward_interpo_3dd,  &
  &          calc_error_2d,  &
  &          calc_error_2df,  &
  &          calc_error_2dd,  &
  &          calc_error_3d,  &
  &          calc_error_3df,  &
  &          calc_error_3dd

!  private :: check_bound

  interface Ellip_GauSei_2d
     module procedure Ellip_Gausei_2df,  &
  &                   Ellip_Gausei_2dd
  end interface Ellip_GauSei_2d

  interface Ellip_GauSei_3d
     module procedure Ellip_Gausei_3df,  &
  &                   Ellip_Gausei_3dd
  end interface Ellip_GauSei_3d

  interface Ellip_Jacobi_2d
     module procedure Ellip_Jacobi_2df,  &
  &                   Ellip_Jacobi_2dd
  end interface Ellip_Jacobi_2d

  interface Ellip_Jacobi_3d
     module procedure Ellip_Jacobi_3df,  &
  &                   Ellip_Jacobi_3dd
  end interface Ellip_Jacobi_3d

  interface Full_Multi_Grid_2d
     module procedure Full_Multi_Grid_2df,  &
  &                   Full_Multi_Grid_2dd
  end interface Full_Multi_Grid_2d

  interface Full_Multi_Grid_3d
     module procedure Full_Multi_Grid_3df,  &
  &                   Full_Multi_Grid_3dd
  end interface Full_Multi_Grid_3d

  interface GauSei_Vcycle_2d
     module procedure GauSei_Vcycle_2df,  &
  &                   GauSei_Vcycle_2dd
  end interface GauSei_Vcycle_2d

  interface GauSei_Vcycle_3d
     module procedure GauSei_Vcycle_3df,  &
  &                   GauSei_Vcycle_3dd
  end interface GauSei_Vcycle_3d

  interface calc_error_2d
     module procedure calc_error_2df,  &
  &                   calc_error_2dd
  end interface calc_error_2d

  interface calc_error_3d
     module procedure calc_error_3df,  &
  &                   calc_error_3dd
  end interface calc_error_3d

  interface forward_interpo_2d
     module procedure forward_interpo_2df,  &
  &                   forward_interpo_2dd
  end interface forward_interpo_2d

  interface forward_interpo_3d
     module procedure forward_interpo_3df,  &
  &                   forward_interpo_3dd
  end interface forward_interpo_3d

  interface backward_interpo_2d
     module procedure backward_interpo_2df,  &
  &                   backward_interpo_2dd
  end interface backward_interpo_2d

  interface backward_interpo_3d
     module procedure backward_interpo_3df,  &
  &                   backward_interpo_3dd
  end interface backward_interpo_3d

contains

subroutine Ellip_GauSei_2df( x, y, rho, eps, boundary, psi, bound_opt,  &
  &                          a, b, c, d, e, f, undef, inner_bound, init_flag,  &
  &                          accel, ln, check_flag, helmod_flag, helmod_dl )
! ガウス=ザイデル法による楕円型方程式の求積
! 各オプション配列は, ポアソン系の各微分項の値. デフォルトはゼロで設定される.
! $$a\dfrac{\partial ^2\psi}{\partial x^2} +b\dfrac{\partial ^2\psi}{\partial x\partial y} +c\dfrac{\partial ^2\psi}{\partial y^2} +d\dfrac{\partial \psi}{\partial x} +e\dfrac{\partial \psi}{\partial y} +f\psi =\rho $$
! の各係数に対応している.
  implicit none
  real, intent(in) :: x(:)  ! 領域の横座標
  real, intent(in) :: y(:)  ! 領域の縦座標
  real, intent(in) :: rho(size(x),size(y))  ! ポアソン方程式の強制項
                   ! rho =0 でラプラス方程式も求積可能
  real, intent(in) :: eps  ! 収束条件
  character(4), intent(in) :: boundary  ! 境界条件
                ! 4 文字で各辺の境界条件を与える.
                ! 1 文字目 : x 下端, 2 文字目 : y 左端, 3 文字目 : x 上端,
                ! 4 文字目 : y 右端
                ! boundary は 1 : 固定端境界, 2 : 自由端境界, 3 : 周期境界
  real, intent(in), optional :: bound_opt(size(x),size(y))  ! 境界での強制
                             ! ノイマン境界の場合 : フラックス値
  real, intent(in), optional :: a(size(x),size(y))  ! 各微分項の係数
  real, intent(in), optional :: b(size(x),size(y))  ! 各微分項の係数
  real, intent(in), optional :: c(size(x),size(y))  ! 各微分項の係数
  real, intent(in), optional :: d(size(x),size(y))  ! 各微分項の係数
  real, intent(in), optional :: e(size(x),size(y))  ! 各微分項の係数
  real, intent(in), optional :: f(size(x),size(y))  ! 各微分項の係数
  real, intent(inout) :: psi(size(x),size(y))  ! ポアソン方程式の解
  real, intent(in), optional :: undef  ! 未定義値
  integer, intent(in), optional :: inner_bound(size(x),size(y))
                             ! 内部領域の境界. 値に応じてその格子点で境界値計算
                             ! 1 = 固定端境界, 10 = 境界の内側.
                             ! 2 = y 方向自由端境界 (フラックスは上向き)
                             ! -2 = y 方向自由端境界 (フラックスは下向き)
                             ! 4 = x 方向自由端境界 (フラックスは右向き)
                             ! -4 = x 方向自由端境界 (フラックスは左向き)
                             ! 3 = 周期境界
                             ! 8 = |_, ~| で両方とも自由境界条件
                             ! -8 = |~, _| で両方とも自由境界条件
                             ! この引数が与えられなければ全領域を計算する.
                             ! 境界の内側格子点 (10) は反復計算を行わず,
                             ! undef で設定された値もしくはゼロが入る.
                             ! このときの境界値は bound_opt の値が用いられる.
  logical, intent(in), optional :: init_flag  ! psi の値をゼロで初期化するか.
                             ! .true. = 初期化する. .false. = 初期化しない.
                             ! デフォルトでは初期化する.
  real, intent(in), optional :: accel  ! SOR の加速係数 (0 < accel < 2)
                             ! デフォルト = 1
  integer, intent(in), optional :: ln  ! 反復回数
                             ! この値が与えられるとき, eps の値に無関係に
                             ! ln 回ループさせる.
  logical, intent(in), optional :: check_flag  ! true の場合, 最高階数の
                             ! 係数が負号でも stop する.
                             ! default = .false. (係数が 0 になる場合のみ stop)
  logical, intent(in), optional :: helmod_flag  ! g が与えられたときのヘルムホルツ
                             ! 方程式の収束判定を満たすために, g の値を調整する.
                             ! .true.: 調整してメッセージを出力する.
                             ! .false.: 調整せず, 収束判定を満たさなければメッセージを出して終了する
                             ! default = .false.
  logical, intent(in), optional :: helmod_dl  ! helmod_flag = .true. のときのデバッグ出力
                             ! .true.: g が調整されたときに逐次メッセージを出力する.
                             ! .false.: ルーチンが起動したときに一度だけメッセージを出力する.
                             ! default = .false.

  integer :: i, j, ix, jy, nl, counter
  integer :: nx  ! x 方向の配列要素
  integer :: ny  ! y 方向の配列要素
  integer :: nmax  ! nx, ny の最大値
  integer :: signb  ! 各係数を計算するかどうか
  integer, dimension(size(x),size(y)) :: ib

  real :: defun
  real :: tmp, err, err_max
  real :: tmp_b, accc
  real :: dcosinv_nmax, hel_fact, h2, tmpf, acccinv
  real :: bnd(size(x),size(y))
  real :: dx(size(x)), dy(size(y)), dx2(size(x)), dy2(size(y))
  real, dimension(size(x),size(y)) :: dxdy
  real, dimension(size(x),size(y)) :: at, bt, ct, dt, et, ft
  real, dimension(size(x),size(y)) :: adp, adm, cep, cem, ac, divi

  character(4) :: bound
  logical :: sor_flag, hel_flag, hel_dl
  logical, dimension(size(x),size(y)) :: inner_flag

  bound(1:4)=boundary(1:4)

  nx=size(x)
  ny=size(y)

!-- 応答関数の初期化

  if(present(init_flag))then
     if(init_flag.eqv..true.)then
        psi = 0.0
     end if
  else
     psi = 0.0
  end if

!-- 内部境界の判別フラグの設定

  if(present(inner_bound))then
     call set_bound( bound, ib, inner_flag, inner_bound )
  else
     call set_bound( bound, ib, inner_flag )
  end if

!-- 領域・内部境界における境界値の設定

  if(present(bound_opt))then
     call setval_boundf( ib, bnd, psi, bound_opt )
  else
     call setval_boundf( ib, bnd, psi )
  end if

!-- 未定義値の設定

  if(present(undef))then
     defun=undef
  else
     defun=0.0
  end if

!-- 係数の代入
!-- a, c については, 値が入れられていなければ, 全配列に 1 を代入する.
  if(present(a))then
     call set_coef( at, ext=a )
  else
     call set_coef( at, def=1.0 )
  end if

  if(present(c))then
     call set_coef( ct, ext=c )
  else
     call set_coef( ct, def=1.0 )
  end if

  if(present(b))then
     call set_coef( bt, ext=b )
     signb=1
  else
     call set_coef( bt, def=0.0 )
     signb=0
  end if

  if(present(d))then
     call set_coef( dt, ext=d )
  else
     call set_coef( dt, def=0.0 )
  end if

  if(present(e))then
     call set_coef( et, ext=e )
  else
     call set_coef( et, def=0.0 )
  end if

!-- 最高階数における係数チェック. (係数がゼロでないか調べる.)

  if(present(check_flag))then
     if(check_flag.eqv..true.)then
        call check_le_coef( at(2:nx-1,2:ny-1), 0.0, undeff=defun )
        call check_le_coef( ct(2:nx-1,2:ny-1), 0.0, undeff=defun )
     else
        call check_coef( at(2:nx-1,2:ny-1), 0.0, undeff=defun )
        call check_coef( ct(2:nx-1,2:ny-1), 0.0, undeff=defun )
     end if
  else
     call check_coef( at(2:nx-1,2:ny-1), 0.0, undeff=defun )
     call check_coef( ct(2:nx-1,2:ny-1), 0.0, undeff=defun )
  end if

!-- 加速係数の判定

  if(present(accel))then
     accc=accel
     sor_flag=.true.
  else
     accc=1.0
     sor_flag=.false.
  end if

!-- ln の処理
  if(present(ln))then
     nl=ln
  else
     nl=0
  end if

!-- ヘルムホルツ方程式収束判定

  hel_flag=.false.
  hel_dl=.false.
  if(present(f))then
     call set_coef( ft, ext=f )
     nmax=max(nx,ny)
     dcosinv_nmax=1.0/cos(pi/real(nmax))
     if(present(helmod_flag))then
        hel_flag=helmod_flag
        if((hel_flag.eqv..true.).and.present(helmod_dl))then
           hel_dl=helmod_dl
           if(hel_dl.eqv..false.)then
              write(*,*) "*** MESSAGE (Ellip_GauSei_3dd) ***: helmod_flag is active"
           end if
        end if
     end if
     acccinv=1.0/accc
  else
     call set_coef( ft, def=0.0 )
  end if

!-- 以下で先に格子間隔等の 1 回計算でよいものを求めておく.
!-- これらは 1 方向のみで変化すればよい.
!-- 格子点間隔の計算
  do i=2,nx-1
     dx(i)=(x(i+1)-x(i-1))*0.5
     dx2(i)=dx(i)**2
  end do
  do j=2,ny-1
     dy(j)=(y(j+1)-y(j-1))*0.5
     dy2(j)=dy(j)**2
  end do

  dx(1)=(x(2)-x(1))
  dx(nx)=(x(nx)-x(nx-1))
  dy(1)=(y(2)-y(1))
  dy(ny)=(y(ny)-y(ny-1))

  do j=1,ny
     do i=1,nx
        dxdy(i,j)=dx(i)*dy(j)
     end do
  end do

!-- ポアソン係数の計算
!-- 実際にポアソン係数が使用されるのは, 境界を除いた 1 回り内側なので,
!-- 計算量削減のため, ループをそのようにしておく.

  ac=0.0
  adp=0.0
  adm=0.0
  cep=0.0
  cem=0.0
  bt=0.0

!-- 最高次数係数 ac の計算
!$omp parallel default(shared)

  if(present(f))then
  ! f が指定された場合, ヘルムホルツ方程式の収束判定も実施 (David et al. 2014)

!$omp do schedule(dynamic) private(i,j,hel_fact,h2,tmpf)

     do j=2,ny-1
        do i=2,nx-1
           tmpf=ft(i,j)
           if(ft(i,j)>0.0)then
              h2=1.0/(2.0*(at(i,j)/dx2(i)+ct(i,j)/dy2(j)))
              hel_fact=acccinv*dcosinv_nmax*(1.0-ft(i,j)*h2)
              if(abs(hel_fact)<=1.0)then
                 if(hel_flag.eqv..true.)then
                    tmpf=(1.0-(10.0/9.0)/(acccinv*dcosinv_nmax))/h2
                    if(hel_dl.eqv..true.)then
                       write(*,*) "*** MESSAGE (Ellip_GauSei_3d) ***: Detect the critical value for the Helmholtz solver."
                       write(*,*) "Forced the grid value to the smaller than the critical value."
                       write(*,'(a12,1P2E16.8,3i5)') "Original vs Modified: ", ft(i,j), tmpf, i, j
                    end if
                 else
                    write(*,*) "*** ERROR (Ellip_GauSei_3d) ***: Detect critical value for the Helmholtz solver."
                    write(*,*) "Critical value (1-f/c), i, j:", hel_fact, i, j
                    !write(*,*) "Critical value (1-f/c), i, j, k:", hel_fact, i, j, k
                    stop
                 end if
                 stop
              end if
           end if
           ac(i,j)=1.0/(2.0*(at(i,j)/dx2(i)+ct(i,j)/dy2(j))-tmpf)
        end do
     end do

!$omp end do

  else

!$omp do schedule(dynamic) private(i,j)

     do j=2,ny-1
        do i=2,nx-1
           ac(i,j)=1.0/(2.0*(at(i,j)/dx2(i)+ct(i,j)/dy2(j))-ft(i,j))
        end do
     end do

!$omp end do

  end if

!$omp barrier
!$omp do schedule(dynamic) private(i,j)

  do j=2,ny-1
     do i=2,nx-1
        adp(i,j)=(at(i,j)/(dx2(i))+0.5*dt(i,j)/dx(i))*ac(i,j)
        adm(i,j)=(at(i,j)/(dx2(i))-0.5*dt(i,j)/dx(i))*ac(i,j)
        cep(i,j)=(ct(i,j)/(dy2(j))+0.5*et(i,j)/dy(j))*ac(i,j)
        cem(i,j)=(ct(i,j)/(dy2(j))-0.5*et(i,j)/dy(j))*ac(i,j)
        bt(i,j)=0.25*bt(i,j)/(dxdy(i,j))*ac(i,j)
     end do
  end do

!$omp end do
!$omp end parallel

  err_max=eps  ! while に入るための便宜的措置
  counter=0

!-- 実際のソルバ ---
  do while(err_max>=eps)
     err_max=0.0

     do j=1,ny
        do i=1,nx

!-- 以降, 反復計算に必要な点の周囲 8 点についてそれぞれ
!-- inner_flag のチェックと同時に適切な値をそれぞれ逐次計算する,
!-- 下で, 各方向の格子計算を行っているが, ib=8,-8 は 4 隅格子のときにしか
!-- case select しない. なぜなら, 真横や上下に隅格子がくることはありえない.
           if(inner_flag(i,j).eqv..false.)then  ! .false. なら領域計算開始
              tmp=-rho(i,j)*ac(i,j)
              tmp=tmp+adp(i,j)*psi(i+1,j)  &
 &                   +adm(i,j)*psi(i-1,j)  &
 &                   +cep(i,j)*psi(i,j+1)  &
 &                   +cem(i,j)*psi(i,j-1)

              if(signb==0)then  ! そもそも bt = 0 なら計算しない.
                 tmp_b=0.0
              else
                 tmp_b=bt(i,j)*(psi(i+1,j+1)+psi(i-1,j-1)  &
  &                            -psi(i+1,j-1)-psi(i-1,j+1))
              end if

              tmp=tmp+tmp_b

           else  ! .true. なら境界計算に移行.

              select case (ib(i,j))
              case (1)
                 tmp=bnd(i,j)

              case (2)  ! x 方向にフラックス一定, 上側が参照値
                 tmp=psi(i+1,j)-bnd(i,j)*dx(i)

              case (-2)  ! x 方向にフラックス一定, 下側が参照値
                 tmp=psi(i-1,j)+bnd(i,j)*dx(i)

              case (4)  ! y 方向にフラックス一定, 右側が参照値
                 tmp=psi(i,j+1)-bnd(i,j)*dy(j)

              case (-4)  ! y 方向にフラックス一定, 左側が参照値
                 tmp=psi(i,j-1)+bnd(i,j)*dy(j)

              case (3)  ! 周期境界
                 if(i==1)then
                    ix=nx-1
                 else if(i==nx)then
                    ix=2
                 else
                    ix=i
                 end if
                 if(j==1)then
                    jy=ny-1
                 else if(j==ny)then
                    jy=2
                 else
                    jy=j
                 end if

                 tmp=psi(ix,jy)

              case (7)  ! 両方フラックス一定で内部境界限定.
                 if((ib(i+1,j+1)==10).and.(ib(i+1,j)/=10).and.  &
  &                 (ib(i,j+1)/=10))then
                    tmp=0.5*(psi(i-1,j+1)+psi(i+1,j-1)  &
  &                         +bnd(i,j+1)*dx(i)+bnd(i+1,j)*dy(j))

                 else if((ib(i-1,j-1)==10).and.(ib(i-1,j)/=10).and.  &
  &                      (ib(i,j-1)/=10))then
                    tmp=0.5*(psi(i+1,j-1)+psi(i-1,j+1)  &
  &                         -bnd(i,j-1)*dx(i)-bnd(i-1,j)*dy(j))

                 end if

              case (-7)  ! 両方フラックス一定で内部境界限定.
                 if((ib(i-1,j+1)==10).and.(ib(i-1,j)/=10).and.  &
  &                 (ib(i,j+1)/=10))then
                    tmp=0.5*(psi(i+1,j+1)+psi(i-1,j-1)  &
  &                         -bnd(i,j+1)*dx(i)+bnd(i-1,j)*dy(j))

                 else if((ib(i+1,j-1)==10).and.(ib(i+1,j)/=10).and.  &
  &                 (ib(i,j-1)/=10))then
                    tmp=0.5*(psi(i-1,j-1)+psi(i+1,j+1)  &
  &                         +bnd(i,j-1)*dx(i)-bnd(i+1,j)*dy(j))

                 end if

              case (8)  ! 両方フラックス一定で左下角か右上角, もしくは内部境界.
                 if(i==1.and.j==1)then  ! -- 評価 1
                    tmp=psi(i+1,j+1)-0.5*(bnd(i+1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 else if(i==nx.and.j==ny)then  ! -- 評価 2
                    tmp=psi(i-1,j-1)+0.5*(bnd(i-1,j)*dy(j)+bnd(i,j-1)*dx(i))

                 else if(ib(i-1,j)==10.and.ib(i,j-1)==10)then
                    ! -- 評価 1 と同じ
                    tmp=psi(i+1,j+1)-0.5*(bnd(i+1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 else if(ib(i+1,j)==10.and.ib(i,j+1)==10)then
                    ! -- 評価 2 と同じ
                    tmp=psi(i-1,j-1)+0.5*(bnd(i-1,j)*dy(j)+bnd(i,j-1)*dx(i))

                 end if

              case (-8)  ! 両方フラックス一定で右下角か左上角
                 if(i==1.and.j==ny)then  ! -- 評価 1
                    tmp=psi(i+1,j-1)+0.5*(bnd(i+1,j)*dy(j)-bnd(i,j-1)*dx(i))

                 else if(i==nx.and.j==1)then  ! -- 評価 2
                    tmp=psi(i-1,j+1)+0.5*(-bnd(i-1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 else if(ib(i-1,j)==10.and.ib(i,j+1)==10)then
                    ! -- 評価 1 と同じ
                    tmp=psi(i+1,j-1)+0.5*(bnd(i+1,j)*dy(j)-bnd(i,j-1)*dx(i))

                 else if(ib(i+1,j)==10.and.ib(i,j-1)==10)then
                    ! -- 評価 2 と同じ
                    tmp=psi(i-1,j+1)+0.5*(-bnd(i-1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 end if
              end select

           end if

           if(sor_flag.eqv..true.)then
              tmp=(1.0-accc)*psi(i,j)+tmp*accc
           end if

           err=abs(tmp-psi(i,j))

!-- 最大誤差の更新
           if(err_max<=err)then
              err_max=err
           end if

           psi(i,j)=tmp

        end do
     end do

     if(nl/=0)then
        err_max=eps
        counter=counter+1
        if(counter==nl)then
           exit
        end if
     end if

  end do

!-- 境界の設定

  call calculate_boundf( ib, dx, dy, bnd, psi )

!-- 未定義領域には undef を代入する.

  do j=1,ny
     do i=1,nx
        if(ib(i,j)==10)then
           psi(i,j)=defun
        end if
     end do
  end do

end subroutine Ellip_GauSei_2df

!----------------------------------
!----------------------------------

subroutine Ellip_GauSei_2dd( x, y, rho, eps, boundary, psi, bound_opt,  &
  &                          a, b, c, d, e, f, undef, inner_bound, init_flag,  &
  &                          accel, ln, check_flag, helmod_flag, helmod_dl )
! ガウス=ザイデル法による楕円型方程式の求積
! 各オプション配列は, ポアソン系の各微分項の値. デフォルトはゼロで設定される.
! $$a\dfrac{\partial ^2\psi}{\partial x^2} +b\dfrac{\partial ^2\psi}{\partial x\partial y} +c\dfrac{\partial ^2\psi}{\partial y^2} +d\dfrac{\partial \psi}{\partial x} +e\dfrac{\partial \psi}{\partial y} +f\psi =\rho $$
! の各係数に対応している.
  implicit none
  double precision, intent(in) :: x(:)  ! 領域の横座標
  double precision, intent(in) :: y(:)  ! 領域の縦座標
  double precision, intent(in) :: rho(size(x),size(y))  ! ポアソン方程式の強制項
                   ! rho =0 でラプラス方程式も求積可能
  double precision, intent(in) :: eps  ! 収束条件
  character(4), intent(in) :: boundary  ! 境界条件
                ! 4 文字で各辺の境界条件を与える.
                ! 1 文字目 : x 下端, 2 文字目 : y 左端, 3 文字目 : x 上端,
                ! 4 文字目 : y 右端
                ! boundary は 1 : 固定端境界, 2 : 自由端境界, 3 : 周期境界
  double precision, intent(in), optional :: bound_opt(size(x),size(y))  ! 境界での強制
                             ! ノイマン境界の場合 : フラックス値
  double precision, intent(in), optional :: a(size(x),size(y))  ! 各微分項の係数
  double precision, intent(in), optional :: b(size(x),size(y))  ! 各微分項の係数
  double precision, intent(in), optional :: c(size(x),size(y))  ! 各微分項の係数
  double precision, intent(in), optional :: d(size(x),size(y))  ! 各微分項の係数
  double precision, intent(in), optional :: e(size(x),size(y))  ! 各微分項の係数
  double precision, intent(in), optional :: f(size(x),size(y))  ! 各微分項の係数
  double precision, intent(inout) :: psi(size(x),size(y))  ! ポアソン方程式の解
  double precision, intent(in), optional :: undef  ! 未定義値
  integer, intent(in), optional :: inner_bound(size(x),size(y))
                             ! 内部領域の境界. 値に応じてその格子点で境界値計算
                             ! 1 = 固定端境界, 10 = 境界の内側.
                             ! 2 = y 方向自由端境界 (フラックスは上向き)
                             ! -2 = y 方向自由端境界 (フラックスは下向き)
                             ! 4 = x 方向自由端境界 (フラックスは右向き)
                             ! -4 = x 方向自由端境界 (フラックスは左向き)
                             ! 3 = 周期境界
                             ! 8 = |_, ~| で両方とも自由境界条件
                             ! -8 = |~, _| で両方とも自由境界条件
                             ! この引数が与えられなければ全領域を計算する.
                             ! 境界の内側格子点 (10) は反復計算を行わず,
                             ! undef で設定された値もしくはゼロが入る.
                             ! このときの境界値は bound_opt の値が用いられる.
  logical, intent(in), optional :: init_flag  ! psi の値をゼロで初期化するか.
                             ! .true. = 初期化する. .false. = 初期化しない.
                             ! デフォルトでは初期化する.
  double precision, intent(in), optional :: accel  ! SOR の加速係数 (0 < accel < 2)
                             ! デフォルト = 1
  integer, intent(in), optional :: ln  ! 反復回数
                             ! この値が与えられるとき, eps の値に無関係に
                             ! ln 回ループさせる.
  logical, intent(in), optional :: check_flag  ! true の場合, 最高階数の
                             ! 係数が負号でも stop する.
                             ! default = .false. (係数が 0 になる場合のみ stop)
  logical, intent(in), optional :: helmod_flag  ! g が与えられたときのヘルムホルツ
                             ! 方程式の収束判定を満たすために, g の値を調整する.
                             ! .true.: 調整してメッセージを出力する.
                             ! .false.: 調整せず, 収束判定を満たさなければメッセージを出して終了する
                             ! default = .false.
  logical, intent(in), optional :: helmod_dl  ! helmod_flag = .true. のときのデバッグ出力
                             ! .true.: g が調整されたときに逐次メッセージを出力する.
                             ! .false.: ルーチンが起動したときに一度だけメッセージを出力する.
                             ! default = .false.

  integer :: i, j, ix, jy, nl, counter
  integer :: nx  ! x 方向の配列要素
  integer :: ny  ! y 方向の配列要素
  integer :: nmax  ! nx, ny の最大値
  integer :: signb  ! 各係数を計算するかどうか
  integer, dimension(size(x),size(y)) :: ib

  double precision :: defun
  double precision :: tmp, err, err_max
  double precision :: tmp_b, accc
  double precision :: dcosinv_nmax, hel_fact, h2, tmpf, acccinv
  double precision :: bnd(size(x),size(y))
  double precision :: dx(size(x)), dy(size(y)), dx2(size(x)), dy2(size(y))
  double precision, dimension(size(x),size(y)) :: dxdy
  double precision, dimension(size(x),size(y)) :: at, bt, ct, dt, et, ft
  double precision, dimension(size(x),size(y)) :: adp, adm, cep, cem, ac, divi

  character(4) :: bound
  logical :: sor_flag, hel_flag, hel_dl
  logical, dimension(size(x),size(y)) :: inner_flag

  bound(1:4)=boundary(1:4)

  nx=size(x)
  ny=size(y)

!-- 応答関数の初期化

  if(present(init_flag))then
     if(init_flag.eqv..true.)then
        psi = 0.0d0
     end if
  else
     psi = 0.0d0
  end if

!-- 内部境界の判別フラグの設定

  if(present(inner_bound))then
     call set_bound( bound, ib, inner_flag, inner_bound )
  else
     call set_bound( bound, ib, inner_flag )
  end if

!-- 領域・内部境界における境界値の設定

  if(present(bound_opt))then
     call setval_boundd( ib, bnd, psi, bound_opt )
  else
     call setval_boundd( ib, bnd, psi )
  end if

!-- 未定義値の設定

  if(present(undef))then
     defun=undef
  else
     defun=0.0d0
  end if

!-- 係数の代入
!-- a, c については, 値が入れられていなければ, 全配列に 1 を代入する.
  if(present(a))then
     call set_coed( at, ext=a )
  else
     call set_coed( at, def=1.0d0 )
  end if

  if(present(c))then
     call set_coed( ct, ext=c )
  else
     call set_coed( ct, def=1.0d0 )
  end if

  if(present(b))then
     call set_coed( bt, ext=b )
     signb=1
  else
     call set_coed( bt, def=0.0d0 )
     signb=0
  end if

  if(present(d))then
     call set_coed( dt, ext=d )
  else
     call set_coed( dt, def=0.0d0 )
  end if

  if(present(e))then
     call set_coed( et, ext=e )
  else
     call set_coed( et, def=0.0d0 )
  end if

!-- 最高階数における係数チェック. (係数がゼロでないか調べる.)

  if(present(check_flag))then
     if(check_flag.eqv..true.)then
        call check_le_coed( at(2:nx-1,2:ny-1), 0.0d0, undeff=defun )
        call check_le_coed( ct(2:nx-1,2:ny-1), 0.0d0, undeff=defun )
     else
        call check_coed( at(2:nx-1,2:ny-1), 0.0d0, undeff=defun )
        call check_coed( ct(2:nx-1,2:ny-1), 0.0d0, undeff=defun )
     end if
  else
     call check_coed( at(2:nx-1,2:ny-1), 0.0d0, undeff=defun )
     call check_coed( ct(2:nx-1,2:ny-1), 0.0d0, undeff=defun )
  end if

!-- 加速係数の判定

  if(present(accel))then
     accc=accel
     sor_flag=.true.
  else
     accc=1.0d0
     sor_flag=.false.
  end if

!-- ln の処理
  if(present(ln))then
     nl=ln
  else
     nl=0
  end if

!-- ヘルムホルツ方程式収束判定

  hel_flag=.false.
  hel_dl=.false.
  if(present(f))then
     call set_coed( ft, ext=f )
     nmax=max(nx,ny)
     dcosinv_nmax=1.0d0/dcos(pi_dp/dble(nmax))
     if(present(helmod_flag))then
        hel_flag=helmod_flag
        if((hel_flag.eqv..true.).and.present(helmod_dl))then
           hel_dl=helmod_dl
           if(hel_dl.eqv..false.)then
              write(*,*) "*** MESSAGE (Ellip_GauSei_3dd) ***: helmod_flag is active"
           end if
        end if
     end if
     acccinv=1.0d0/accc
  else
     call set_coed( ft, def=0.0d0 )
  end if

!-- 以下で先に格子間隔等の 1 回計算でよいものを求めておく.
!-- これらは 1 方向のみで変化すればよい.
!-- 格子点間隔の計算
  do i=2,nx-1
     dx(i)=(x(i+1)-x(i-1))*0.5d0
     dx2(i)=dx(i)**2
  end do
  do j=2,ny-1
     dy(j)=(y(j+1)-y(j-1))*0.5d0
     dy2(j)=dy(j)**2
  end do

  dx(1)=(x(2)-x(1))
  dx(nx)=(x(nx)-x(nx-1))
  dy(1)=(y(2)-y(1))
  dy(ny)=(y(ny)-y(ny-1))

  do j=1,ny
     do i=1,nx
        dxdy(i,j)=dx(i)*dy(j)
     end do
  end do

!-- ポアソン係数の計算
!-- 実際にポアソン係数が使用されるのは, 境界を除いた 1 回り内側なので,
!-- 計算量削減のため, ループをそのようにしておく.

  ac=0.0d0
  adp=0.0d0
  adm=0.0d0
  cep=0.0d0
  cem=0.0d0
  bt=0.0d0

!-- 最高次数係数 ac の計算
!$omp parallel default(shared)

  if(present(f))then
  ! f が指定された場合, ヘルムホルツ方程式の収束判定も実施 (David et al. 2014)

!$omp do schedule(dynamic) private(i,j,hel_fact,h2,tmpf)

     do j=2,ny-1
        do i=2,nx-1
           tmpf=ft(i,j)
           if(ft(i,j)>0.0d0)then
              h2=1.0d0/(2.0d0*(at(i,j)/dx2(i)+ct(i,j)/dy2(j)))
              hel_fact=acccinv*dcosinv_nmax*(1.0d0-ft(i,j)*h2)
              if(dabs(hel_fact)<=1.0d0)then
                 if(hel_flag.eqv..true.)then
                    tmpf=(1.0d0-(10.0d0/9.0d0)/(acccinv*dcosinv_nmax))/h2
                    if(hel_dl.eqv..true.)then
                       write(*,*) "*** MESSAGE (Ellip_GauSei_3d) ***: Detect the critical value for the Helmholtz solver."
                       write(*,*) "Forced the grid value to the smaller than the critical value."
                       write(*,'(a12,1P2E16.8,3i5)') "Original vs Modified: ", ft(i,j), tmpf, i, j
                    end if
                 else
                    write(*,*) "*** ERROR (Ellip_GauSei_3d) ***: Detect critical value for the Helmholtz solver."
                    write(*,*) "Critical value (1-f/c), i, j:", hel_fact, i, j
                    !write(*,*) "Critical value (1-f/c), i, j, k:", hel_fact, i, j, k
                    stop
                 end if
                 stop
              end if
           end if
           ac(i,j)=1.0d0/(2.0d0*(at(i,j)/dx2(i)+ct(i,j)/dy2(j))-tmpf)
        end do
     end do

!$omp end do

  else

!$omp do schedule(dynamic) private(i,j)

     do j=2,ny-1
        do i=2,nx-1
           ac(i,j)=1.0d0/(2.0d0*(at(i,j)/dx2(i)+ct(i,j)/dy2(j))-ft(i,j))
        end do
     end do

!$omp end do

  end if

!$omp barrier
!$omp do schedule(dynamic) private(i,j)

  do j=2,ny-1
     do i=2,nx-1
        adp(i,j)=(at(i,j)/(dx2(i))+0.5d0*dt(i,j)/dx(i))*ac(i,j)
        adm(i,j)=(at(i,j)/(dx2(i))-0.5d0*dt(i,j)/dx(i))*ac(i,j)
        cep(i,j)=(ct(i,j)/(dy2(j))+0.5d0*et(i,j)/dy(j))*ac(i,j)
        cem(i,j)=(ct(i,j)/(dy2(j))-0.5d0*et(i,j)/dy(j))*ac(i,j)
        bt(i,j)=0.25d0*bt(i,j)/(dxdy(i,j))*ac(i,j)
     end do
  end do

!$omp end do
!$omp end parallel

  err_max=eps  ! while に入るための便宜的措置
  counter=0

!-- 実際のソルバ ---
  do while(err_max>=eps)
     err_max=0.0d0

     do j=1,ny
        do i=1,nx

!-- 以降, 反復計算に必要な点の周囲 8 点についてそれぞれ
!-- inner_flag のチェックと同時に適切な値をそれぞれ逐次計算する,
!-- 下で, 各方向の格子計算を行っているが, ib=8,-8 は 4 隅格子のときにしか
!-- case select しない. なぜなら, 真横や上下に隅格子がくることはありえない.
           if(inner_flag(i,j).eqv..false.)then  ! .false. なら領域計算開始
              tmp=-rho(i,j)*ac(i,j)
              tmp=tmp+adp(i,j)*psi(i+1,j)  &
 &                   +adm(i,j)*psi(i-1,j)  &
 &                   +cep(i,j)*psi(i,j+1)  &
 &                   +cem(i,j)*psi(i,j-1)

              if(signb==0)then  ! そもそも bt = 0 なら計算しない.
                 tmp_b=0.0d0
              else
                 tmp_b=bt(i,j)*(psi(i+1,j+1)+psi(i-1,j-1)  &
  &                            -psi(i+1,j-1)-psi(i-1,j+1))
              end if

              tmp=tmp+tmp_b

           else  ! .true. なら境界計算に移行.

              select case (ib(i,j))
              case (1)
                 tmp=bnd(i,j)

              case (2)  ! x 方向にフラックス一定, 上側が参照値
                 tmp=psi(i+1,j)-bnd(i,j)*dx(i)

              case (-2)  ! x 方向にフラックス一定, 下側が参照値
                 tmp=psi(i-1,j)+bnd(i,j)*dx(i)

              case (4)  ! y 方向にフラックス一定, 右側が参照値
                 tmp=psi(i,j+1)-bnd(i,j)*dy(j)

              case (-4)  ! y 方向にフラックス一定, 左側が参照値
                 tmp=psi(i,j-1)+bnd(i,j)*dy(j)

              case (3)  ! 周期境界
                 if(i==1)then
                    ix=nx-1
                 else if(i==nx)then
                    ix=2
                 else
                    ix=i
                 end if
                 if(j==1)then
                    jy=ny-1
                 else if(j==ny)then
                    jy=2
                 else
                    jy=j
                 end if

                 tmp=psi(ix,jy)

              case (7)  ! 両方フラックス一定で内部境界限定.
                 if((ib(i+1,j+1)==10).and.(ib(i+1,j)/=10).and.  &
  &                 (ib(i,j+1)/=10))then
                    tmp=0.5d0*(psi(i-1,j+1)+psi(i+1,j-1)  &
  &                         +bnd(i,j+1)*dx(i)+bnd(i+1,j)*dy(j))

                 else if((ib(i-1,j-1)==10).and.(ib(i-1,j)/=10).and.  &
  &                      (ib(i,j-1)/=10))then
                    tmp=0.5d0*(psi(i+1,j-1)+psi(i-1,j+1)  &
  &                         -bnd(i,j-1)*dx(i)-bnd(i-1,j)*dy(j))

                 end if

              case (-7)  ! 両方フラックス一定で内部境界限定.
                 if((ib(i-1,j+1)==10).and.(ib(i-1,j)/=10).and.  &
  &                 (ib(i,j+1)/=10))then
                    tmp=0.5d0*(psi(i+1,j+1)+psi(i-1,j-1)  &
  &                         -bnd(i,j+1)*dx(i)+bnd(i-1,j)*dy(j))

                 else if((ib(i+1,j-1)==10).and.(ib(i+1,j)/=10).and.  &
  &                 (ib(i,j-1)/=10))then
                    tmp=0.5d0*(psi(i-1,j-1)+psi(i+1,j+1)  &
  &                         +bnd(i,j-1)*dx(i)-bnd(i+1,j)*dy(j))

                 end if

              case (8)  ! 両方フラックス一定で左下角か右上角, もしくは内部境界.
                 if(i==1.and.j==1)then  ! -- 評価 1
                    tmp=psi(i+1,j+1)-0.5d0*(bnd(i+1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 else if(i==nx.and.j==ny)then  ! -- 評価 2
                    tmp=psi(i-1,j-1)+0.5d0*(bnd(i-1,j)*dy(j)+bnd(i,j-1)*dx(i))

                 else if(ib(i-1,j)==10.and.ib(i,j-1)==10)then
                    ! -- 評価 1 と同じ
                    tmp=psi(i+1,j+1)-0.5d0*(bnd(i+1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 else if(ib(i+1,j)==10.and.ib(i,j+1)==10)then
                    ! -- 評価 2 と同じ
                    tmp=psi(i-1,j-1)+0.5d0*(bnd(i-1,j)*dy(j)+bnd(i,j-1)*dx(i))

                 end if

              case (-8)  ! 両方フラックス一定で右下角か左上角
                 if(i==1.and.j==ny)then  ! -- 評価 1
                    tmp=psi(i+1,j-1)+0.5d0*(bnd(i+1,j)*dy(j)-bnd(i,j-1)*dx(i))

                 else if(i==nx.and.j==1)then  ! -- 評価 2
                    tmp=psi(i-1,j+1)+0.5d0*(-bnd(i-1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 else if(ib(i-1,j)==10.and.ib(i,j+1)==10)then
                    ! -- 評価 1 と同じ
                    tmp=psi(i+1,j-1)+0.5d0*(bnd(i+1,j)*dy(j)-bnd(i,j-1)*dx(i))

                 else if(ib(i+1,j)==10.and.ib(i,j-1)==10)then
                    ! -- 評価 2 と同じ
                    tmp=psi(i-1,j+1)+0.5d0*(-bnd(i-1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 end if
              end select

           end if

           if(sor_flag.eqv..true.)then
              tmp=(1.0d0-accc)*psi(i,j)+tmp*accc
           end if

           err=abs(tmp-psi(i,j))

!-- 最大誤差の更新
           if(err_max<=err)then
              err_max=err
           end if

           psi(i,j)=tmp

        end do
     end do

     if(nl/=0)then
        err_max=eps
        counter=counter+1
        if(counter==nl)then
           exit
        end if
     end if

  end do

!-- 境界の設定

  call calculate_boundd( ib, dx, dy, bnd, psi )

!-- 未定義領域には undef を代入する.

  do j=1,ny
     do i=1,nx
        if(ib(i,j)==10)then
           psi(i,j)=defun
        end if
     end do
  end do

end subroutine Ellip_GauSei_2dd

!----------------------------------
!----------------------------------

subroutine Ellip_Jacobi_2df( x, y, rho, eps, boundary, psi, bound_opt,  &
  &                          a, b, c, d, e, f, undef, inner_bound, init_flag,  &
  &                          accel, ln, check_flag, helmod_flag, helmod_dl )
  ! openmp によるスレッド並列が可能.
  ! ガウス・ザイデル法ではアルゴリズムの点から並列化が困難と思われたので,
  ! 並列計算によるポアソン方程式の求積が必要となるなら,
  ! ヤコビ法のものを使用されたい.
  implicit none
  real, intent(in) :: x(:)  ! 領域の横座標
  real, intent(in) :: y(:)  ! 領域の縦座標
  real, intent(in) :: rho(size(x),size(y))  ! ポアソン方程式の強制項
                   ! rho =0 でラプラス方程式も求積可能
  real, intent(in) :: eps  ! 収束条件
  character(4), intent(in) :: boundary  ! 境界条件
                ! 4 文字で各辺の境界条件を与える.
                ! 1 文字目 : x 下端, 2 文字目 : y 左端, 3 文字目 : x 上端,
                ! 4 文字目 : y 右端
                ! boundary は 1 : 固定端境界, 2 : 自由端境界, 3 : 周期境界
  real, intent(in), optional :: bound_opt(size(x),size(y))  ! 境界での強制
                             ! ノイマン境界の場合 : フラックス値
  real, intent(in), optional :: a(size(x),size(y))  ! 各微分項の係数
  real, intent(in), optional :: b(size(x),size(y))  ! 各微分項の係数
  real, intent(in), optional :: c(size(x),size(y))  ! 各微分項の係数
  real, intent(in), optional :: d(size(x),size(y))  ! 各微分項の係数
  real, intent(in), optional :: e(size(x),size(y))  ! 各微分項の係数
  real, intent(in), optional :: f(size(x),size(y))  ! 各微分項の係数
  real, intent(in), optional :: undef  ! 未定義値
  integer, intent(in), optional :: inner_bound(size(x),size(y))
                             ! 内部領域の境界. 値に応じてその格子点で境界値計算
                             ! 1 = 固定端境界, 10 = 境界の内側.
                             ! 2 = y 方向自由端境界 (フラックスは上向き)
                             ! -2 = y 方向自由端境界 (フラックスは下向き)
                             ! 4 = x 方向自由端境界 (フラックスは右向き)
                             ! -4 = x 方向自由端境界 (フラックスは左向き)
                             ! 3 = 周期境界, -3 = 隅領域で両方とも周期境界
                             ! 8 = |_, ~| で両方とも自由境界条件
                             ! -8 = |~, _| で両方とも自由境界条件
                             ! この引数が与えられなければ全領域を計算する.
                             ! 境界の内側格子点 (10) は反復計算を行わず,
                             ! undef で設定された値もしくはゼロが入る.
                             ! このときの境界値は bound_opt の値が用いられる.
  real, intent(inout) :: psi(size(x),size(y))  ! ポアソン方程式の解
  logical, intent(in), optional :: init_flag  ! psi の値をゼロで初期化するか.
                             ! .true. = 初期化する. .false. = 初期化しない.
                             ! デフォルトでは初期化する.
  real, intent(in), optional :: accel  ! SOR の加速係数 (0 < accel < 2)
                             ! デフォルト = 1
  integer, intent(in), optional :: ln  ! 反復回数
                             ! この値が与えられるとき, eps の値に無関係に
                             ! ln 回ループさせる.
  logical, intent(in), optional :: check_flag  ! true の場合, 最高階数の
                             ! 係数が負号でも stop する.
                             ! default = .false. (係数が 0 になる場合のみ stop)
  logical, intent(in), optional :: helmod_flag  ! g が与えられたときのヘルムホルツ
                             ! 方程式の収束判定を満たすために, g の値を調整する.
                             ! .true.: 調整してメッセージを出力する.
                             ! .false.: 調整せず, 収束判定を満たさなければメッセージを出して終了する
                             ! default = .false.
  logical, intent(in), optional :: helmod_dl  ! helmod_flag = .true. のときのデバッグ出力
                             ! .true.: g が調整されたときに逐次メッセージを出力する.
                             ! .false.: ルーチンが起動したときに一度だけメッセージを出力する.
                             ! default = .false.

  integer :: i, j, ix, jy, nl, counter
  integer :: nx  ! x 方向の配列要素
  integer :: ny  ! y 方向の配列要素
  integer :: nmax  ! nx, ny の最大値
  integer :: signb  ! クロスターム b が存在するか
  integer, dimension(size(x),size(y)) :: ib

  real :: defun
  real :: err, err_max, accc, emax
  real :: dcosinv_nmax, hel_fact, h2, tmpf, acccinv
  real :: bnd(size(x),size(y))
  real :: dx(size(x)), dy(size(y)), dx2(size(x)), dy2(size(y))
  real, dimension(size(x),size(y)) :: dxdy
  real, dimension(size(x),size(y)) :: at, bt, ct, dt, et, ft
  real, dimension(size(x),size(y)) :: adp, adm, cep, cem, ac
  real, dimension(size(x),size(y)) :: tmp, tmp_b, divi

  character(4) :: bound
  logical :: sor_flag, hel_flag, hel_dl
  logical, dimension(size(x),size(y)) :: inner_flag

  bound(1:4)=boundary(1:4)

  nx=size(x)
  ny=size(y)

!-- 応答関数の初期化

  if(present(init_flag))then
     if(init_flag.eqv..true.)then
        psi = 0.0
     end if
  else
     psi = 0.0
  end if

!-- 内部境界の判別フラグの設定

  if(present(inner_bound))then
     call set_bound( bound, ib, inner_flag, inner_bound )
  else
     call set_bound( bound, ib, inner_flag )
  end if

!-- 領域・内部境界における境界値の設定

  if(present(bound_opt))then
     call setval_boundf( ib, bnd, psi, bound_opt )
  else
     call setval_boundf( ib, bnd, psi )
  end if

!-- 未定義値の設定

  if(present(undef))then
     defun=undef
  else
     defun=0.0
  end if

!-- 係数の代入
!-- a, c については, 値が入れられていなければ, 全配列に 1 を代入する.

  if(present(a))then
     call set_coef( at, ext=a )
  else
     call set_coef( at, def=1.0 )
  end if

  if(present(c))then
     call set_coef( ct, ext=c )
  else
     call set_coef( ct, def=1.0 )
  end if

  if(present(b))then
     call set_coef( bt, ext=b )
     signb=1
  else
     call set_coef( bt, def=0.0 )
     signb=0
  end if

  if(present(d))then
     call set_coef( dt, ext=d )
  else
     call set_coef( dt, def=0.0 )
  end if

  if(present(e))then
     call set_coef( et, ext=e )
  else
     call set_coef( et, def=0.0 )
  end if

!-- 最高階数における係数チェック. (係数がゼロでないか調べる.)

  if(present(check_flag))then
     if(check_flag.eqv..true.)then
        call check_le_coef( at(2:nx-1,2:ny-1), 0.0, undeff=defun )
        call check_le_coef( ct(2:nx-1,2:ny-1), 0.0, undeff=defun )
     else
        call check_coef( at(2:nx-1,2:ny-1), 0.0, undeff=defun )
        call check_coef( ct(2:nx-1,2:ny-1), 0.0, undeff=defun )
     end if
  else
     call check_coef( at(2:nx-1,2:ny-1), 0.0, undeff=defun )
     call check_coef( ct(2:nx-1,2:ny-1), 0.0, undeff=defun )
  end if

!-- 加速係数の判定

  if(present(accel))then
     accc=accel
     sor_flag=.true.
  else
     accc=1.0
     sor_flag=.false.
  end if

!-- ln の処理
  if(present(ln))then
     nl=ln
  else
     nl=0
  end if

!-- ヘルムホルツ方程式収束判定

  hel_flag=.false.
  hel_dl=.false.
  if(present(f))then
     call set_coef( ft, ext=f )
     nmax=max(nx,ny)
     dcosinv_nmax=1.0/cos(pi/real(nmax))
     if(present(helmod_flag))then
        hel_flag=helmod_flag
        if((hel_flag.eqv..true.).and.present(helmod_dl))then
           hel_dl=helmod_dl
           if(hel_dl.eqv..false.)then
              write(*,*) "*** MESSAGE (Ellip_GauSei_3dd) ***: helmod_flag is active"
           end if
        end if
     end if
     acccinv=1.0/accc
  else
     call set_coef( ft, def=0.0 )
  end if

!-- 以下で先に格子間隔等の 1 回計算でよいものを求めておく.
!-- これらは 1 方向のみで変化すればよい.
!-- 格子点間隔の計算
  do i=2,nx-1
     dx(i)=(x(i+1)-x(i-1))*0.5
     dx2(i)=dx(i)**2
  end do
  do j=2,ny-1
     dy(j)=(y(j+1)-y(j-1))*0.5
     dy2(j)=dy(j)**2
  end do

  dx(1)=(x(2)-x(1))
  dx(nx)=(x(nx)-x(nx-1))
  dy(1)=(y(2)-y(1))
  dy(ny)=(y(ny)-y(ny-1))

  do j=1,ny
     do i=1,nx
        dxdy(i,j)=dx(i)*dy(j)
     end do
  end do

!-- ポアソン係数の計算
!-- 実際にポアソン係数が使用されるのは, 境界を除いた 1 回り内側なので,
!-- 計算量削減のため, ループをそのようにしておく.

  ac=0.0
  adp=0.0
  adm=0.0
  cep=0.0
  cem=0.0
  bt=0.0

!-- 最高次数係数 ac の計算
!$omp parallel default(shared)

  if(present(f))then
  ! f が指定された場合, ヘルムホルツ方程式の収束判定も実施 (David et al. 2014)

!$omp do schedule(dynamic) private(i,j,hel_fact,h2,tmpf)

     do j=2,ny-1
        do i=2,nx-1
           tmpf=ft(i,j)
           if(ft(i,j)>0.0)then
              h2=1.0/(2.0*(at(i,j)/dx2(i)+ct(i,j)/dy2(j)))
              hel_fact=acccinv*dcosinv_nmax*(1.0-ft(i,j)*h2)
              if(abs(hel_fact)<=1.0)then
                 if(hel_flag.eqv..true.)then
                    tmpf=(1.0-(10.0/9.0)/(acccinv*dcosinv_nmax))/h2
                    if(hel_dl.eqv..true.)then
                       write(*,*) "*** MESSAGE (Ellip_GauSei_3d) ***: Detect the critical value for the Helmholtz solver."
                       write(*,*) "Forced the grid value to the smaller than the critical value."
                       write(*,'(a12,1P2E16.8,3i5)') "Original vs Modified: ", ft(i,j), tmpf, i, j
                    end if
                 else
                    write(*,*) "*** ERROR (Ellip_GauSei_3d) ***: Detect critical value for the Helmholtz solver."
                    write(*,*) "Critical value (1-f/c), i, j:", hel_fact, i, j
                    !write(*,*) "Critical value (1-f/c), i, j, k:", hel_fact, i, j, k
                    stop
                 end if
                 stop
              end if
           end if
           ac(i,j)=1.0/(2.0*(at(i,j)/dx2(i)+ct(i,j)/dy2(j))-tmpf)
        end do
     end do

!$omp end do

  else

!$omp do schedule(dynamic) private(i,j)

     do j=2,ny-1
        do i=2,nx-1
           ac(i,j)=1.0/(2.0*(at(i,j)/dx2(i)+ct(i,j)/dy2(j))-ft(i,j))
        end do
     end do

!$omp end do

  end if

!$omp barrier
!$omp do schedule(dynamic) private(i,j)

  do j=2,ny-1
     do i=2,nx-1
        adp(i,j)=(at(i,j)/(dx2(i))+0.5*dt(i,j)/dx(i))*ac(i,j)
        adm(i,j)=(at(i,j)/(dx2(i))-0.5*dt(i,j)/dx(i))*ac(i,j)
        cep(i,j)=(ct(i,j)/(dy2(j))+0.5*et(i,j)/dy(j))*ac(i,j)
        cem(i,j)=(ct(i,j)/(dy2(j))-0.5*et(i,j)/dy(j))*ac(i,j)
        bt(i,j)=0.25*bt(i,j)/(dxdy(i,j))*ac(i,j)
     end do
  end do

!$omp end do
!$omp end parallel

  err_max=eps  ! while に入るための便宜的措置
  counter=0

!-- 実際のソルバ ---
  do while(err_max>=eps)
     err_max=0.0
!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,ix,jy)
     do j=1,ny
        do i=1,nx

!-- 以降, 反復計算に必要な点の周囲 8 点についてそれぞれ
!-- inner_flag のチェックと同時に適切な値をそれぞれ逐次計算する,
!-- 下で, 各方向の格子計算を行っているが, ib=8,-8 は 4 隅格子のときにしか
!-- case select しない. なぜなら, 真横や上下に隅格子がくることはありえない.
           if(inner_flag(i,j).eqv..false.)then  ! .false. なら領域計算開始
              tmp(i,j)=-rho(i,j)*ac(i,j)
              tmp(i,j)=tmp(i,j)+adp(i,j)*psi(i+1,j)  &
 &                             +adm(i,j)*psi(i-1,j)  &
 &                             +cep(i,j)*psi(i,j+1)  &
 &                             +cem(i,j)*psi(i,j-1)
              if(signb==0)then  ! そもそも bt = 0 なら計算しない.
                  tmp_b(i,j)=0.0
              else
                  tmp_b(i,j)=bt(i,j)*(psi(i+1,j+1)+psi(i-1,j-1)  &
  &                             -psi(i+1,j-1)-psi(i-1,j+1))
              end if

              tmp(i,j)=tmp(i,j)+tmp_b(i,j)

           else  ! .true. なら境界計算に移行.

              select case (ib(i,j))
              case (1)
                 tmp(i,j)=bnd(i,j)

              case (2)  ! x 方向にフラックス一定, 上側が参照値
                 tmp(i,j)=psi(i+1,j)-bnd(i,j)*dx(i)

              case (-2)  ! x 方向にフラックス一定, 下側が参照値
                 tmp(i,j)=psi(i-1,j)+bnd(i,j)*dx(i)

              case (4)  ! y 方向にフラックス一定, 右側が参照値
                 tmp(i,j)=psi(i,j+1)-bnd(i,j)*dy(j)

              case (-4)  ! y 方向にフラックス一定, 左側が参照値
                 tmp(i,j)=psi(i,j-1)+bnd(i,j)*dy(j)

              case (3)  ! 周期境界
                 if(i==1)then
                    ix=nx-1
                 else if(i==nx)then
                    ix=2
                 else
                    ix=i
                 end if
                 if(j==1)then
                    jy=ny-1
                 else if(j==ny)then
                    jy=2
                 else
                    jy=j
                 end if

                 tmp(i,j)=psi(ix,jy)

              case (7)  ! 両方フラックス一定で内部境界限定.
                 if((ib(i+1,j+1)==10).and.(ib(i+1,j)/=10).and.  &
  &                 (ib(i,j+1)/=10))then
                    tmp(i,j)=0.5*(psi(i-1,j)+psi(i,j-1))  &
  &                          +0.5*bnd(i,j)*(dy(j)+dx(i))

                 else if((ib(i-1,j-1)==10).and.(ib(i-1,j)/=10).and.  &
  &                      (ib(i,j-1)/=10))then
                    tmp(i,j)=0.5*(psi(i+1,j)+psi(i,j+1))  &
  &                          -0.5*bnd(i,j)*(dy(j)+dx(i))

                 end if

              case (-7)  ! 両方フラックス一定で内部境界限定.
                 if((ib(i-1,j+1)==10).and.(ib(i-1,j)/=10).and.  &
  &                 (ib(i,j+1)/=10))then
                    tmp(i,j)=0.5*(psi(i+1,j)+psi(i,j-1))  &
  &                          +0.5*bnd(i,j)*(dy(j)-dx(i))

                 else if((ib(i+1,j-1)==10).and.(ib(i+1,j)/=10).and.  &
  &                 (ib(i,j-1)/=10))then
                    tmp(i,j)=0.5*(psi(i-1,j)+psi(i,j+1))  &
  &                          +0.5*bnd(i,j)*(-dy(j)+dx(i))

                 end if

              case (8)  ! 両方フラックス一定で左下角か右上角, もしくは内部境界.
                 if(i==1.and.j==1)then  ! -- 評価 1
                    tmp(i,j)=psi(i+1,j+1)-0.5*(bnd(i+1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 else if(i==nx.and.j==ny)then  ! -- 評価 2
                    tmp(i,j)=psi(i-1,j-1)+0.5*(bnd(i-1,j)*dy(j)+bnd(i,j-1)*dx(i))

                 else if(ib(i-1,j)==10.and.ib(i,j-1)==10)then
                    ! -- 評価 1 と同じ
                    tmp(i,j)=psi(i+1,j+1)-0.5*(bnd(i+1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 else if(ib(i+1,j)==10.and.ib(i,j+1)==10)then
                    ! -- 評価 2 と同じ
                    tmp(i,j)=psi(i-1,j-1)+0.5*(bnd(i-1,j)*dy(j)+bnd(i,j-1)*dx(i))

                 end if

              case (-8)  ! 両方フラックス一定で右下角か左上角
                 if(i==1.and.j==ny)then  ! -- 評価 1
                    tmp(i,j)=psi(i+1,j-1)+0.5*(bnd(i+1,j)*dy(j)-bnd(i,j-1)*dx(i))

                 else if(i==nx.and.j==1)then  ! -- 評価 2
                    tmp(i,j)=psi(i-1,j+1)+0.5*(-bnd(i-1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 else if(ib(i-1,j)==10.and.ib(i,j+1)==10)then
                    ! -- 評価 1 と同じ
                    tmp(i,j)=psi(i+1,j-1)+0.5*(bnd(i+1,j)*dy(j)-bnd(i,j-1)*dx(i))

                 else if(ib(i+1,j)==10.and.ib(i,j-1)==10)then
                    ! -- 評価 2 と同じ
                    tmp(i,j)=psi(i-1,j+1)+0.5*(-bnd(i-1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 end if
              end select

           end if

           if(sor_flag.eqv..true.)then
              tmp(i,j)=(1.0-accc)*psi(i,j)+tmp(i,j)*accc
           end if

        end do
     end do
!$omp end do
!$omp end parallel

!!-- ここまでは if 文のインデントを調節しない.
!        end if
!
!        tmp=tmp/ac(i,j)
!-- 誤差の計算 ---
     do j=2,ny-1
        do i=2,nx-1
           err=abs(tmp(i,j)-psi(i,j))

!-- 最大誤差の更新
           if(err_max<=err)then
              err_max=err
           end if
        end do
     end do

!     write(*,*) "*** MESSAGE (Ellip_Slv) *** : error max is", err, err_max

!-- 一斉更新

     do j=1,ny
        do i=1,nx
           psi(i,j)=tmp(i,j)
        end do
     end do

     if(nl/=0)then
        counter=counter+1
        if(counter==nl)then
           exit
        else
           err_max=eps
        end if
     end if

!!-- 特異摂動回避処理
!     if(present(epsm))then
!        emax=check_diff_error_2d( x, y, psi, rho, at, bt, ct, dt, et, ft, ib )
!        if(emax<=epsm)then
!           exit
!        end if
!     end if

  end do
  
!-- 境界の設定

  call calculate_boundf( ib, dx, dy, bnd, psi )

!-- 未定義領域には undef を代入する.

  do j=1,ny
     do i=1,nx
        if(ib(i,j)==10)then
           psi(i,j)=defun
        end if
     end do
  end do

end subroutine Ellip_Jacobi_2df

!----------------------------------
!----------------------------------

subroutine Ellip_Jacobi_2dd( x, y, rho, eps, boundary, psi, bound_opt,  &
  &                          a, b, c, d, e, f, undef, inner_bound, init_flag,  &
  &                          accel, ln, check_flag, helmod_flag, helmod_dl )
  ! openmp によるスレッド並列が可能.
  ! ガウス・ザイデル法ではアルゴリズムの点から並列化が困難と思われたので,
  ! 並列計算によるポアソン方程式の求積が必要となるなら,
  ! ヤコビ法のものを使用されたい.
  implicit none
  double precision, intent(in) :: x(:)  ! 領域の横座標
  double precision, intent(in) :: y(:)  ! 領域の縦座標
  double precision, intent(in) :: rho(size(x),size(y))  ! ポアソン方程式の強制項
                   ! rho =0 でラプラス方程式も求積可能
  double precision, intent(in) :: eps  ! 収束条件
  character(4), intent(in) :: boundary  ! 境界条件
                ! 4 文字で各辺の境界条件を与える.
                ! 1 文字目 : x 下端, 2 文字目 : y 左端, 3 文字目 : x 上端,
                ! 4 文字目 : y 右端
                ! boundary は 1 : 固定端境界, 2 : 自由端境界, 3 : 周期境界
  double precision, intent(in), optional :: bound_opt(size(x),size(y))  ! 境界での強制
                             ! ノイマン境界の場合 : フラックス値
  double precision, intent(in), optional :: a(size(x),size(y))  ! 各微分項の係数
  double precision, intent(in), optional :: b(size(x),size(y))  ! 各微分項の係数
  double precision, intent(in), optional :: c(size(x),size(y))  ! 各微分項の係数
  double precision, intent(in), optional :: d(size(x),size(y))  ! 各微分項の係数
  double precision, intent(in), optional :: e(size(x),size(y))  ! 各微分項の係数
  double precision, intent(in), optional :: f(size(x),size(y))  ! 各微分項の係数
  double precision, intent(in), optional :: undef  ! 未定義値
  integer, intent(in), optional :: inner_bound(size(x),size(y))
                             ! 内部領域の境界. 値に応じてその格子点で境界値計算
                             ! 1 = 固定端境界, 10 = 境界の内側.
                             ! 2 = y 方向自由端境界 (フラックスは上向き)
                             ! -2 = y 方向自由端境界 (フラックスは下向き)
                             ! 4 = x 方向自由端境界 (フラックスは右向き)
                             ! -4 = x 方向自由端境界 (フラックスは左向き)
                             ! 3 = 周期境界, -3 = 隅領域で両方とも周期境界
                             ! 8 = |_, ~| で両方とも自由境界条件
                             ! -8 = |~, _| で両方とも自由境界条件
                             ! この引数が与えられなければ全領域を計算する.
                             ! 境界の内側格子点 (10) は反復計算を行わず,
                             ! undef で設定された値もしくはゼロが入る.
                             ! このときの境界値は bound_opt の値が用いられる.
  double precision, intent(inout) :: psi(size(x),size(y))  ! ポアソン方程式の解
  logical, intent(in), optional :: init_flag  ! psi の値をゼロで初期化するか.
                             ! .true. = 初期化する. .false. = 初期化しない.
                             ! デフォルトでは初期化する.
  double precision, intent(in), optional :: accel  ! SOR の加速係数 (0 < accel < 2)
                             ! デフォルト = 1
  integer, intent(in), optional :: ln  ! 反復回数
                             ! この値が与えられるとき, eps の値に無関係に
                             ! ln 回ループさせる.
  logical, intent(in), optional :: check_flag  ! true の場合, 最高階数の
                             ! 係数が負号でも stop する.
                             ! default = .false. (係数が 0 になる場合のみ stop)
  logical, intent(in), optional :: helmod_flag  ! g が与えられたときのヘルムホルツ
                             ! 方程式の収束判定を満たすために, g の値を調整する.
                             ! .true.: 調整してメッセージを出力する.
                             ! .false.: 調整せず, 収束判定を満たさなければメッセージを出して終了する
                             ! default = .false.
  logical, intent(in), optional :: helmod_dl  ! helmod_flag = .true. のときのデバッグ出力
                             ! .true.: g が調整されたときに逐次メッセージを出力する.
                             ! .false.: ルーチンが起動したときに一度だけメッセージを出力する.
                             ! default = .false.

  integer :: i, j, ix, jy, nl, counter
  integer :: nx  ! x 方向の配列要素
  integer :: ny  ! y 方向の配列要素
  integer :: nmax  ! nx, ny の最大値
  integer :: signb  ! クロスターム b が存在するか
  integer, dimension(size(x),size(y)) :: ib

  double precision :: defun
  double precision :: err, err_max, accc, emax
  double precision :: dcosinv_nmax, hel_fact, h2, tmpf, acccinv
  double precision :: bnd(size(x),size(y))
  double precision :: dx(size(x)), dy(size(y)), dx2(size(x)), dy2(size(y))
  double precision, dimension(size(x),size(y)) :: dxdy
  double precision, dimension(size(x),size(y)) :: at, bt, ct, dt, et, ft
  double precision, dimension(size(x),size(y)) :: adp, adm, cep, cem, ac
  double precision, dimension(size(x),size(y)) :: tmp, tmp_b, divi

  character(4) :: bound
  logical :: sor_flag, hel_flag, hel_dl
  logical, dimension(size(x),size(y)) :: inner_flag

  bound(1:4)=boundary(1:4)

  nx=size(x)
  ny=size(y)

!-- 応答関数の初期化

  if(present(init_flag))then
     if(init_flag.eqv..true.)then
        psi = 0.0d0
     end if
  else
     psi = 0.0d0
  end if

!-- 内部境界の判別フラグの設定

  if(present(inner_bound))then
     call set_bound( bound, ib, inner_flag, inner_bound )
  else
     call set_bound( bound, ib, inner_flag )
  end if

!-- 領域・内部境界における境界値の設定

  if(present(bound_opt))then
     call setval_boundd( ib, bnd, psi, bound_opt )
  else
     call setval_boundd( ib, bnd, psi )
  end if

!-- 未定義値の設定

  if(present(undef))then
     defun=undef
  else
     defun=0.0d0
  end if

!-- 係数の代入
!-- a, c については, 値が入れられていなければ, 全配列に 1 を代入する.

  if(present(a))then
     call set_coed( at, ext=a )
  else
     call set_coed( at, def=1.0d0 )
  end if

  if(present(c))then
     call set_coed( ct, ext=c )
  else
     call set_coed( ct, def=1.0d0 )
  end if

  if(present(b))then
     call set_coed( bt, ext=b )
     signb=1
  else
     call set_coed( bt, def=0.0d0 )
     signb=0
  end if

  if(present(d))then
     call set_coed( dt, ext=d )
  else
     call set_coed( dt, def=0.0d0 )
  end if

  if(present(e))then
     call set_coed( et, ext=e )
  else
     call set_coed( et, def=0.0d0 )
  end if

!-- 最高階数における係数チェック. (係数がゼロでないか調べる.)

  if(present(check_flag))then
     if(check_flag.eqv..true.)then
        call check_le_coed( at(2:nx-1,2:ny-1), 0.0d0, undeff=defun )
        call check_le_coed( ct(2:nx-1,2:ny-1), 0.0d0, undeff=defun )
     else
        call check_coed( at(2:nx-1,2:ny-1), 0.0d0, undeff=defun )
        call check_coed( ct(2:nx-1,2:ny-1), 0.0d0, undeff=defun )
     end if
  else
     call check_coed( at(2:nx-1,2:ny-1), 0.0d0, undeff=defun )
     call check_coed( ct(2:nx-1,2:ny-1), 0.0d0, undeff=defun )
  end if

!-- 加速係数の判定

  if(present(accel))then
     accc=accel
     sor_flag=.true.
  else
     accc=1.0d0
     sor_flag=.false.
  end if

!-- ln の処理
  if(present(ln))then
     nl=ln
  else
     nl=0
  end if

!-- ヘルムホルツ方程式収束判定

  hel_flag=.false.
  hel_dl=.false.
  if(present(f))then
     call set_coed( ft, ext=f )
     nmax=max(nx,ny)
     dcosinv_nmax=1.0d0/dcos(pi_dp/dble(nmax))
     if(present(helmod_flag))then
        hel_flag=helmod_flag
        if((hel_flag.eqv..true.).and.present(helmod_dl))then
           hel_dl=helmod_dl
           if(hel_dl.eqv..false.)then
              write(*,*) "*** MESSAGE (Ellip_GauSei_3dd) ***: helmod_flag is active"
           end if
        end if
     end if
     acccinv=1.0d0/accc
  else
     call set_coed( ft, def=0.0d0 )
  end if

!-- 以下で先に格子間隔等の 1 回計算でよいものを求めておく.
!-- これらは 1 方向のみで変化すればよい.
!-- 格子点間隔の計算
  do i=2,nx-1
     dx(i)=(x(i+1)-x(i-1))*0.5d0
     dx2(i)=dx(i)**2
  end do
  do j=2,ny-1
     dy(j)=(y(j+1)-y(j-1))*0.5d0
     dy2(j)=dy(j)**2
  end do

  dx(1)=(x(2)-x(1))
  dx(nx)=(x(nx)-x(nx-1))
  dy(1)=(y(2)-y(1))
  dy(ny)=(y(ny)-y(ny-1))

  do j=1,ny
     do i=1,nx
        dxdy(i,j)=dx(i)*dy(j)
     end do
  end do

!-- ポアソン係数の計算
!-- 実際にポアソン係数が使用されるのは, 境界を除いた 1 回り内側なので,
!-- 計算量削減のため, ループをそのようにしておく.

  ac=0.0d0
  adp=0.0d0
  adm=0.0d0
  cep=0.0d0
  cem=0.0d0
  bt=0.0d0

!-- 最高次数係数 ac の計算
!$omp parallel default(shared)

  if(present(f))then
  ! f が指定された場合, ヘルムホルツ方程式の収束判定も実施 (David et al. 2014)

!$omp do schedule(dynamic) private(i,j,hel_fact,h2,tmpf)

     do j=2,ny-1
        do i=2,nx-1
           tmpf=ft(i,j)
           if(ft(i,j)>0.0d0)then
              h2=1.0d0/(2.0d0*(at(i,j)/dx2(i)+ct(i,j)/dy2(j)))
              hel_fact=acccinv*dcosinv_nmax*(1.0d0-ft(i,j)*h2)
              if(dabs(hel_fact)<=1.0d0)then
                 if(hel_flag.eqv..true.)then
                    tmpf=(1.0d0-(10.0d0/9.0d0)/(acccinv*dcosinv_nmax))/h2
                    if(hel_dl.eqv..true.)then
                       write(*,*) "*** MESSAGE (Ellip_GauSei_3d) ***: Detect the critical value for the Helmholtz solver."
                       write(*,*) "Forced the grid value to the smaller than the critical value."
                       write(*,'(a12,1P2E16.8,3i5)') "Original vs Modified: ", ft(i,j), tmpf, i, j
                    end if
                 else
                    write(*,*) "*** ERROR (Ellip_GauSei_3d) ***: Detect critical value for the Helmholtz solver."
                    write(*,*) "Critical value (1-f/c), i, j:", hel_fact, i, j
                    !write(*,*) "Critical value (1-f/c), i, j, k:", hel_fact, i, j, k
                    stop
                 end if
                 stop
              end if
           end if
           ac(i,j)=1.0d0/(2.0d0*(at(i,j)/dx2(i)+ct(i,j)/dy2(j))-tmpf)
        end do
     end do

!$omp end do

  else

!$omp do schedule(dynamic) private(i,j)

     do j=2,ny-1
        do i=2,nx-1
           ac(i,j)=1.0d0/(2.0d0*(at(i,j)/dx2(i)+ct(i,j)/dy2(j))-ft(i,j))
        end do
     end do

!$omp end do

  end if

!$omp barrier
!$omp do schedule(dynamic) private(i,j)

  do j=2,ny-1
     do i=2,nx-1
        adp(i,j)=(at(i,j)/(dx2(i))+0.5d0*dt(i,j)/dx(i))*ac(i,j)
        adm(i,j)=(at(i,j)/(dx2(i))-0.5d0*dt(i,j)/dx(i))*ac(i,j)
        cep(i,j)=(ct(i,j)/(dy2(j))+0.5d0*et(i,j)/dy(j))*ac(i,j)
        cem(i,j)=(ct(i,j)/(dy2(j))-0.5d0*et(i,j)/dy(j))*ac(i,j)
        bt(i,j)=0.25d0*bt(i,j)/(dxdy(i,j))*ac(i,j)
     end do
  end do

!$omp end do
!$omp end parallel

  err_max=eps  ! while に入るための便宜的措置
  counter=0

!-- 実際のソルバ ---
  do while(err_max>=eps)
     err_max=0.0d0
!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,ix,jy)
     do j=1,ny
        do i=1,nx

!-- 以降, 反復計算に必要な点の周囲 8 点についてそれぞれ
!-- inner_flag のチェックと同時に適切な値をそれぞれ逐次計算する,
!-- 下で, 各方向の格子計算を行っているが, ib=8,-8 は 4 隅格子のときにしか
!-- case select しない. なぜなら, 真横や上下に隅格子がくることはありえない.
           if(inner_flag(i,j).eqv..false.)then  ! .false. なら領域計算開始
              tmp(i,j)=-rho(i,j)*ac(i,j)
              tmp(i,j)=tmp(i,j)+adp(i,j)*psi(i+1,j)  &
 &                             +adm(i,j)*psi(i-1,j)  &
 &                             +cep(i,j)*psi(i,j+1)  &
 &                             +cem(i,j)*psi(i,j-1)
              if(signb==0)then  ! そもそも bt = 0 なら計算しない.
                  tmp_b(i,j)=0.0d0
              else
                  tmp_b(i,j)=bt(i,j)*(psi(i+1,j+1)+psi(i-1,j-1)  &
  &                             -psi(i+1,j-1)-psi(i-1,j+1))
              end if

              tmp(i,j)=tmp(i,j)+tmp_b(i,j)

           else  ! .true. なら境界計算に移行.

              select case (ib(i,j))
              case (1)
                 tmp(i,j)=bnd(i,j)

              case (2)  ! x 方向にフラックス一定, 上側が参照値
                 tmp(i,j)=psi(i+1,j)-bnd(i,j)*dx(i)

              case (-2)  ! x 方向にフラックス一定, 下側が参照値
                 tmp(i,j)=psi(i-1,j)+bnd(i,j)*dx(i)

              case (4)  ! y 方向にフラックス一定, 右側が参照値
                 tmp(i,j)=psi(i,j+1)-bnd(i,j)*dy(j)

              case (-4)  ! y 方向にフラックス一定, 左側が参照値
                 tmp(i,j)=psi(i,j-1)+bnd(i,j)*dy(j)

              case (3)  ! 周期境界
                 if(i==1)then
                    ix=nx-1
                 else if(i==nx)then
                    ix=2
                 else
                    ix=i
                 end if
                 if(j==1)then
                    jy=ny-1
                 else if(j==ny)then
                    jy=2
                 else
                    jy=j
                 end if

                 tmp(i,j)=psi(ix,jy)

              case (7)  ! 両方フラックス一定で内部境界限定.
                 if((ib(i+1,j+1)==10).and.(ib(i+1,j)/=10).and.  &
  &                 (ib(i,j+1)/=10))then
                    tmp(i,j)=0.5d0*(psi(i-1,j)+psi(i,j-1))  &
  &                          +0.5d0*bnd(i,j)*(dy(j)+dx(i))

                 else if((ib(i-1,j-1)==10).and.(ib(i-1,j)/=10).and.  &
  &                      (ib(i,j-1)/=10))then
                    tmp(i,j)=0.5d0*(psi(i+1,j)+psi(i,j+1))  &
  &                          -0.5d0*bnd(i,j)*(dy(j)+dx(i))

                 end if

              case (-7)  ! 両方フラックス一定で内部境界限定.
                 if((ib(i-1,j+1)==10).and.(ib(i-1,j)/=10).and.  &
  &                 (ib(i,j+1)/=10))then
                    tmp(i,j)=0.5d0*(psi(i+1,j)+psi(i,j-1))  &
  &                          +0.5d0*bnd(i,j)*(dy(j)-dx(i))

                 else if((ib(i+1,j-1)==10).and.(ib(i+1,j)/=10).and.  &
  &                 (ib(i,j-1)/=10))then
                    tmp(i,j)=0.5d0*(psi(i-1,j)+psi(i,j+1))  &
  &                          +0.5d0*bnd(i,j)*(-dy(j)+dx(i))

                 end if

              case (8)  ! 両方フラックス一定で左下角か右上角, もしくは内部境界.
                 if(i==1.and.j==1)then  ! -- 評価 1
                    tmp(i,j)=psi(i+1,j+1)-0.5d0*(bnd(i+1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 else if(i==nx.and.j==ny)then  ! -- 評価 2
                    tmp(i,j)=psi(i-1,j-1)+0.5d0*(bnd(i-1,j)*dy(j)+bnd(i,j-1)*dx(i))

                 else if(ib(i-1,j)==10.and.ib(i,j-1)==10)then
                    ! -- 評価 1 と同じ
                    tmp(i,j)=psi(i+1,j+1)-0.5d0*(bnd(i+1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 else if(ib(i+1,j)==10.and.ib(i,j+1)==10)then
                    ! -- 評価 2 と同じ
                    tmp(i,j)=psi(i-1,j-1)+0.5d0*(bnd(i-1,j)*dy(j)+bnd(i,j-1)*dx(i))

                 end if

              case (-8)  ! 両方フラックス一定で右下角か左上角
                 if(i==1.and.j==ny)then  ! -- 評価 1
                    tmp(i,j)=psi(i+1,j-1)+0.5d0*(bnd(i+1,j)*dy(j)-bnd(i,j-1)*dx(i))

                 else if(i==nx.and.j==1)then  ! -- 評価 2
                    tmp(i,j)=psi(i-1,j+1)+0.5d0*(-bnd(i-1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 else if(ib(i-1,j)==10.and.ib(i,j+1)==10)then
                    ! -- 評価 1 と同じ
                    tmp(i,j)=psi(i+1,j-1)+0.5d0*(bnd(i+1,j)*dy(j)-bnd(i,j-1)*dx(i))

                 else if(ib(i+1,j)==10.and.ib(i,j-1)==10)then
                    ! -- 評価 2 と同じ
                    tmp(i,j)=psi(i-1,j+1)+0.5d0*(-bnd(i-1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 end if
              end select

           end if

           if(sor_flag.eqv..true.)then
              tmp(i,j)=(1.0d0-accc)*psi(i,j)+tmp(i,j)*accc
           end if

        end do
     end do
!$omp end do
!$omp end parallel

!!-- ここまでは if 文のインデントを調節しない.
!        end if
!
!        tmp=tmp/ac(i,j)
!-- 誤差の計算 ---
     do j=2,ny-1
        do i=2,nx-1
           err=abs(tmp(i,j)-psi(i,j))

!-- 最大誤差の更新
           if(err_max<=err)then
              err_max=err
           end if
        end do
     end do

!     write(*,*) "*** MESSAGE (Ellip_Slv) *** : error max is", err, err_max

!-- 一斉更新

     do j=1,ny
        do i=1,nx
           psi(i,j)=tmp(i,j)
        end do
     end do

     if(nl/=0)then
        counter=counter+1
        if(counter==nl)then
           exit
        else
           err_max=eps
        end if
     end if

!!-- 特異摂動回避処理
!     if(present(epsm))then
!        emax=check_diff_error_2d( x, y, psi, rho, at, bt, ct, dt, et, ft, ib )
!        if(emax<=epsm)then
!           exit
!        end if
!     end if

  end do
  
!-- 境界の設定

  call calculate_boundd( ib, dx, dy, bnd, psi )

!-- 未定義領域には undef を代入する.

  do j=1,ny
     do i=1,nx
        if(ib(i,j)==10)then
           psi(i,j)=defun
        end if
     end do
  end do

end subroutine Ellip_Jacobi_2dd

!----------------------------------
!----------------------------------

subroutine Ellip_GauSei_3df(x, y, z, rho, eps, boundary, psi, bound_opt,  &
  &                         xa, ya, za, a, b, c, d, e, f, g,  &
  &                         undef, inner_bound, init_flag, accel, ln, check_flag,  &
  &                         helmod_flag, helmod_dl )
! ガウス=ザイデル法による楕円型方程式の求積
! 各オプション配列は, ポアソン系の各微分項の値. デフォルトはゼロで設定される.
! $$xa\dfrac{\partial ^2\psi}{\partial x^2} +ya\dfrac{\partial ^2\psi}{\partial y^2} +za\dfrac{\partial ^2\psi}{\partial z^2} +a\dfrac{\partial ^2\psi}{\partial x\partial y} +b\dfrac{\partial ^2\psi}{\partial y\partial z} +c\dfrac{\partial ^2\psi}{\partial z\partial x} +d\dfrac{\partial \psi}{\partial x} +e\dfrac{\partial \psi}{\partial y} +f\dfrac{\partial \psi}{\partial z} +g\psi =\rho $$
! の各係数に対応している.
  implicit none
  real, intent(in) :: x(:)  ! 領域の x 座標
  real, intent(in) :: y(:)  ! 領域の y 座標
  real, intent(in) :: z(:)  ! 領域の z 座標
  real, intent(in) :: rho(size(x),size(y),size(z))  ! ポアソン方程式の強制項
                   ! rho =0 でラプラス方程式も求積可能
  real, intent(in) :: eps  ! 収束条件
  character(6), intent(in) :: boundary  ! 境界条件
                ! 4 文字で各辺の境界条件を与える.
                ! 1 文字目 : x 下端, 2 文字目 : y 左端, 3 文字目 : x 上端,
                ! 4 文字目 : y 右端
                ! boundary は 1 : 固定端境界, 2 : 自由端境界, 3 : 周期境界
  real, intent(in), optional :: bound_opt(size(x),size(y),size(z))
                             ! 境界での強制
                             ! ノイマン境界の場合 : フラックス値
  real, intent(in), optional :: xa(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: ya(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: za(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: a(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: b(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: c(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: d(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: e(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: f(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: g(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(inout) :: psi(size(x),size(y),size(z))  ! ポアソン方程式の解
  real, intent(in), optional :: undef  ! 未定義値
  integer, intent(in), optional :: inner_bound(size(x),size(y),size(z))
                             ! 内部領域の境界. 値に応じてその格子点で境界値計算
                             ! 1 = 固定端境界, 10 = 境界の内側.
                             ! 2 = y 方向自由端境界 (フラックスは上向き)
                             ! -2 = y 方向自由端境界 (フラックスは下向き)
                             ! 4 = x 方向自由端境界 (フラックスは右向き)
                             ! -4 = x 方向自由端境界 (フラックスは左向き)
                             ! 3 = 周期境界
                             ! 8 = |_, ~| で両方とも自由境界条件
                             ! -8 = |~, _| で両方とも自由境界条件
                             ! この引数が与えられなければ全領域を計算する.
                             ! 境界の内側格子点 (10) は反復計算を行わず,
                             ! undef で設定された値もしくはゼロが入る.
                             ! このときの境界値は bound_opt の値が用いられる.
  logical, intent(in), optional :: init_flag  ! psi の値をゼロで初期化するか.
                             ! .true. = 初期化する. .false. = 初期化しない.
                             ! デフォルトでは初期化する.
  real, intent(in), optional :: accel  ! SOR の加速係数 (0 < accel < 2)
                             ! デフォルト = 1
  integer, intent(in), optional :: ln  ! 反復回数
                             ! この値が与えられるとき, eps の値に無関係に
                             ! ln 回ループさせる.
  logical, intent(in), optional :: check_flag  ! true の場合, 最高階数の
                             ! 係数が負号でも stop する.
                             ! default = .false. (係数が 0 になる場合のみ stop)
  logical, intent(in), optional :: helmod_flag  ! g が与えられたときのヘルムホルツ
                             ! 方程式の収束判定を満たすために, g の値を調整する.
                             ! .true.: 調整してメッセージを出力する.
                             ! .false.: 調整せず, 収束判定を満たさなければメッセージを出して終了する
                             ! default = .false.
  logical, intent(in), optional :: helmod_dl  ! helmod_flag = .true. のときのデバッグ出力
                             ! .true.: g が調整されたときに逐次メッセージを出力する.
                             ! .false.: ルーチンが起動したときに一度だけメッセージを出力する.
                             ! default = .false.

  integer :: i, j, k, ix, jy, kz, nl, counter
  integer :: nx  ! x 方向の配列要素
  integer :: ny  ! y 方向の配列要素
  integer :: nz  ! z 方向の配列要素
  integer :: nmax  ! nx, ny, nz の最大値
  integer :: signa, signb, signc  ! 各係数を計算するかどうか
  integer, dimension(size(x),size(y),size(z)) :: ib

  real :: defun
  real :: tmp, err, err_max
  real :: tmp_b, accc
  real :: dcosinv_nmax, hel_fact, h2, tmpg, acccinv
  real :: bnd(size(x),size(y),size(z))
  real :: dx(size(x)), dy(size(y)), dx2(size(x)), dy2(size(y))
  real :: dz(size(z)), dz2(size(z))
  real, dimension(size(x),size(y)) :: dxdy
  real, dimension(size(y),size(z)) :: dydz
  real, dimension(size(x),size(z)) :: dxdz
  real, dimension(size(x),size(y),size(z)) :: xt, yt, zt, at, bt, ct, dt, et, ft, gt
  real, dimension(size(x),size(y),size(z)) :: xdp, xdm, yep, yem, zfp, zfm, xyz, divi

  character(6) :: bound
  logical :: sor_flag, hel_flag, hel_dl
  logical, dimension(size(x),size(y),size(z)) :: inner_flag

  bound(1:6)=boundary(1:6)

  nx=size(x)
  ny=size(y)
  nz=size(z)

!-- 応答関数の初期化

  if(present(init_flag))then
     if(init_flag.eqv..true.)then
        psi = 0.0
     end if
  else
     psi = 0.0
  end if

!-- 内部境界の判別フラグの設定

  if(present(inner_bound))then
     call set_bound_3d( bound, ib, inner_flag, inner_bound )
  else
     call set_bound_3d( bound, ib, inner_flag )
  end if

!-- 領域・内部境界における境界値の設定

  if(present(bound_opt))then
     call setval_bound_3df( ib, bnd, psi, bound_opt )
  else
     call setval_bound_3df( ib, bnd, psi )
  end if

!-- 未定義値の設定

  if(present(undef))then
     defun=undef
  else
     defun=0.0
  end if

!-- 係数の代入
!-- xa, ya, za については, 値が入れられていなければ, 全配列に 1 を代入する.
  if(present(xa))then
     call set_coe_3df( xt, ext=xa )
  else
     call set_coe_3df( xt, def=1.0 )
  end if

  if(present(ya))then
     call set_coe_3df( yt, ext=ya )
  else
     call set_coe_3df( yt, def=1.0 )
  end if

  if(present(za))then
     call set_coe_3df( zt, ext=za )
  else
     call set_coe_3df( zt, def=1.0 )
  end if

  if(present(a))then
     call set_coe_3df( at, ext=a )
     signa=1
  else
     call set_coe_3df( at, def=0.0 )
     signa=0
  end if

  if(present(b))then
     call set_coe_3df( bt, ext=b )
     signb=1
  else
     call set_coe_3df( bt, def=0.0 )
     signb=0
  end if

  if(present(c))then
     signc=1
     call set_coe_3df( ct, ext=c )
  else
     call set_coe_3df( ct, def=0.0 )
     signc=0
  end if

  if(present(d))then
     call set_coe_3df( dt, ext=d )
  else
     call set_coe_3df( dt, def=0.0 )
  end if

  if(present(e))then
     call set_coe_3df( et, ext=e )
  else
     call set_coe_3df( et, def=0.0 )
  end if

  if(present(f))then
     call set_coe_3df( ft, ext=f )
  else
     call set_coe_3df( ft, def=0.0 )
  end if

!-- 最高階数における係数チェック. (係数がゼロでないか調べる.)

  if(present(check_flag))then
     if(check_flag.eqv..true.)then
        call check_le_coe_3df( xt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
        call check_le_coe_3df( yt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
        call check_le_coe_3df( zt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
     else
        call check_coe_3df( xt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
        call check_coe_3df( yt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
        call check_coe_3df( zt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
     end if
  else
     call check_coe_3df( xt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
     call check_coe_3df( yt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
     call check_coe_3df( zt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
  end if

!-- 加速係数の判定

  if(present(accel))then
     accc=accel
     sor_flag=.true.
  else
     accc=1.0
     sor_flag=.false.
  end if

!-- ln の処理
  if(present(ln))then
     nl=ln
  else
     nl=0
  end if

!-- ヘルムホルツ方程式収束判定

  hel_flag=.false.
  hel_dl=.false.
  if(present(g))then
     call set_coe_3df( gt, ext=g )
     nmax=max(max(nx,ny),nz)
     dcosinv_nmax=1.0/cos(pi/real(nmax))
     if(present(helmod_flag))then
        hel_flag=helmod_flag
        if((hel_flag.eqv..true.).and.present(helmod_dl))then
           hel_dl=helmod_dl
           if(hel_dl.eqv..false.)then
              write(*,*) "*** MESSAGE (Ellip_GauSei_3dd) ***: helmod_flag is active"
           end if
        end if
     end if
     acccinv=1.0/accc
  else
     call set_coe_3df( gt, def=0.0 )
  end if

!-- 以下で先に格子間隔等の 1 回計算でよいものを求めておく.
!-- これらは 1 方向のみで変化すればよい.
!-- 格子点間隔の計算
  do i=2,nx-1
     dx(i)=(x(i+1)-x(i-1))*0.5
     dx2(i)=dx(i)**2
  end do
  do j=2,ny-1
     dy(j)=(y(j+1)-y(j-1))*0.5
     dy2(j)=dy(j)**2
  end do
  do k=2,nz-1
     dz(k)=(z(k+1)-z(k-1))*0.5
     dz2(k)=dz(k)**2
  end do

  dx(1)=(x(2)-x(1))
  dx(nx)=(x(nx)-x(nx-1))
  dy(1)=(y(2)-y(1))
  dy(ny)=(y(ny)-y(ny-1))
  dz(1)=(z(2)-z(1))
  dz(nz)=(z(nz)-z(nz-1))

  do j=1,ny
     do i=1,nx
        dxdy(i,j)=dx(i)*dy(j)
     end do
  end do
  do k=1,nz
     do j=1,ny
        dydz(j,k)=dy(j)*dz(k)
     end do
  end do
  do k=1,nz
     do i=1,nx
        dxdz(i,k)=dx(i)*dz(k)
     end do
  end do

!-- ポアソン係数の計算
!-- 実際にポアソン係数が使用されるのは, 境界を除いた 1 回り内側なので,
!-- 計算量削減のため, ループをそのようにしておく.

  xyz=0.0
  xdp=0.0
  xdm=0.0
  yep=0.0
  yem=0.0
  zfp=0.0
  zfm=0.0

!-- 最高次数係数 ac の計算
!$omp parallel default(shared)

  if(present(g))then
  ! g が指定された場合, ヘルムホルツ方程式の収束判定も実施 (David et al. 2014)

!$omp do schedule(dynamic) private(i,j,k,hel_fact,h2,tmpg)

     do k=2,nz-1
        do j=2,ny-1
           do i=2,nx-1
              tmpg=gt(i,j,k)
              if(gt(i,j,k)>0.0)then
                 h2=1.0/(2.0*(xt(i,j,k)/dx2(i)+yt(i,j,k)/dy2(j)+zt(i,j,k)/dz2(k)))
                 hel_fact=acccinv*dcosinv_nmax*(1.0-gt(i,j,k)*h2)
                 if(abs(hel_fact)<=1.0)then
                    if(hel_flag.eqv..true.)then
                       tmpg=(1.0-(10.0/9.0)/(acccinv*dcosinv_nmax))/h2
                       if(hel_dl.eqv..true.)then
                          write(*,*) "*** MESSAGE (Ellip_GauSei_3d) ***: Detect the critical value for the Helmholtz solver."
                          write(*,*) "Forced the grid value to the smaller than the critical value."
                          write(*,'(a12,1P2E16.8,3i5)') "Original vs Modified: ", gt(i,j,k), tmpg, i, j, k
                       end if
                    else
                       write(*,*) "*** ERROR (Ellip_GauSei_3d) ***: Detect critical value for the Helmholtz solver."
                       write(*,*) "Critical value (1-f/c), i, j, k:", hel_fact, gt(i,j,k), i, j, k
                       !write(*,*) "Critical value (1-f/c), i, j, k:", hel_fact, i, j, k
                       stop
                    end if
                 end if
              end if
              xyz(i,j,k)=1.0/(2.0*(xt(i,j,k)/dx2(i)+yt(i,j,k)/dy2(j)+zt(i,j,k)/dz2(k))-tmpg)
           end do
        end do
     end do

!$omp end do

  else

!$omp do schedule(runtime) private(i,j,k)

     do k=2,nz-1
        do j=2,ny-1
           do i=2,nx-1
              xyz(i,j,k)=1.0/(2.0*(xt(i,j,k)/dx2(i)+yt(i,j,k)/dy2(j)+zt(i,j,k)/dz2(k))-gt(i,j,k))
           end do
        end do
     end do

!$omp end do

  end if

!$omp barrier
!$omp do schedule(dynamic) private(i,j,k)

  do k=2,nz-1
     do j=2,ny-1
        do i=2,nx-1
           xdp(i,j,k)=(xt(i,j,k)/(dx2(i))+0.5*dt(i,j,k)/dx(i))*xyz(i,j,k)
           xdm(i,j,k)=(xt(i,j,k)/(dx2(i))-0.5*dt(i,j,k)/dx(i))*xyz(i,j,k)
           yep(i,j,k)=(yt(i,j,k)/(dy2(j))+0.5*et(i,j,k)/dy(j))*xyz(i,j,k)
           yem(i,j,k)=(yt(i,j,k)/(dy2(j))-0.5*et(i,j,k)/dy(j))*xyz(i,j,k)
           zfp(i,j,k)=(zt(i,j,k)/(dz2(k))+0.5*ft(i,j,k)/dz(k))*xyz(i,j,k)
           zfm(i,j,k)=(zt(i,j,k)/(dz2(k))-0.5*ft(i,j,k)/dz(k))*xyz(i,j,k)
           if(signa==1)then
              at(i,j,k)=0.25*at(i,j,k)/(dxdy(i,j))*xyz(i,j,k)
           else
              at(i,j,k)=0.0
           end if
           if(signb==1)then
              bt(i,j,k)=0.25*bt(i,j,k)/(dydz(j,k))*xyz(i,j,k)
           else
              bt(i,j,k)=0.0
           end if
           if(signc==1)then
              ct(i,j,k)=0.25*ct(i,j,k)/(dxdz(i,k))*xyz(i,j,k)
           else
              ct(i,j,k)=0.0
           end if
        end do
     end do
  end do

!$omp end do
!$omp end parallel

  err_max=eps  ! while に入るための便宜的措置
  counter=0

!-- 実際のソルバ ---
  do while(err_max>=eps)
     err_max=0.0

     do k=1,nz
        do j=1,ny
           do i=1,nx

!-- 以降, 反復計算に必要な点の周囲 8 点についてそれぞれ
!-- inner_flag のチェックと同時に適切な値をそれぞれ逐次計算する,
!-- 下で, 各方向の格子計算を行っているが, ib=8,-8 は 4 隅格子のときにしか
!-- case select しない. なぜなら, 真横や上下に隅格子がくることはありえない.
              if(inner_flag(i,j,k).eqv..false.)then  ! .false. なら領域計算開始

                 tmp=-rho(i,j,k)*xyz(i,j,k)
                 tmp=tmp+xdp(i,j,k)*psi(i+1,j,k)  &
  &                     +xdm(i,j,k)*psi(i-1,j,k)  &
  &                     +yep(i,j,k)*psi(i,j+1,k)  &
  &                     +yem(i,j,k)*psi(i,j-1,k)  &
  &                     +zfp(i,j,k)*psi(i,j,k+1)  &
  &                     +zfm(i,j,k)*psi(i,j,k-1)

                 if(signa==1)then
                    tmp_b=at(i,j,k)*(psi(i+1,j+1,k)+psi(i-1,j-1,k)  &
  &                                 -psi(i+1,j-1,k)-psi(i-1,j+1,k))
                 else
                    tmp_b=0.0
                 end if
                 if(signb==1)then  ! そもそも bt = 0 なら計算しない.
                    tmp_b=tmp_b+bt(i,j,k)*(psi(i,j+1,k+1)+psi(i,j-1,k-1)  &
  &                                       -psi(i,j-1,k+1)-psi(i,j+1,k-1))
                 end if
                 if(signc==1)then  ! そもそも bt = 0 なら計算しない.
                    tmp_b=tmp_b+ct(i,j,k)*(psi(i+1,j,k+1)+psi(i-1,j,k-1)  &
  &                                       -psi(i-1,j,k+1)-psi(i+1,j,k-1))
                 end if
              
                 tmp=tmp+tmp_b

              else   ! .true. なら境界計算開始.

                 select case (ib(i,j,k))
                 case (1)
                    tmp=bnd(i,j,k)

                 case (2)  ! x 方向にフラックス一定, 上側が参照値
                    tmp=psi(i+1,j,k)-bnd(i,j,k)*dx(i)

                 case (-2)  ! x 方向にフラックス一定, 下側が参照値
                    tmp=psi(i-1,j,k)+bnd(i,j,k)*dx(i)

                 case (4)  ! y 方向にフラックス一定, 右側が参照値
                    tmp=psi(i,j+1,k)-bnd(i,j,k)*dy(j)

                 case (-4)  ! y 方向にフラックス一定, 左側が参照値
                    tmp=psi(i,j-1,k)+bnd(i,j,k)*dy(j)

                 case (6)  ! y 方向にフラックス一定, 右側が参照値
                    tmp=psi(i,j,k+1)-bnd(i,j,k)*dz(k)

                 case (-6)  ! y 方向にフラックス一定, 左側が参照値
                    tmp=psi(i,j,k-1)+bnd(i,j,k)*dz(k)

                 case (3)  ! 12 辺, もしくは 8 点で周期境界を判断
                    if(i==1)then
                       ix=nx-1
                    else if(i==nx)then
                       ix=2
                    else
                       ix=i
                    end if
                    if(j==1)then
                       jy=ny-1
                    else if(j==ny)then
                       jy=2
                    else
                       jy=j
                    end if
                    if(k==1)then
                       kz=nz-1
                    else if(k==nz)then
                       kz=2
                    else
                       kz=k
                    end if
                    tmp=psi(ix,jy,kz)

                 case (8)  ! 両方フラックス一定で z 面の x, y 右上か左下角.
                    if(i==1.and.j==1)then  ! -- 評価 1
                       tmp=psi(i+1,j+1,k)  &
  &                        -0.5*(bnd(i+1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                    else if(i==nx.and.j==ny)then  ! -- 評価 2
                       tmp=psi(i-1,j-1,k)  &
  &                        +0.5*(bnd(i-1,j,k)*dy(j)+bnd(i,j-1,k)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j-1,k)==10)then
                       ! -- 評価 1 と同じ
                       tmp=psi(i+1,j+1,k)  &
  &                        -0.5*(bnd(i+1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j+1,k)==10)then
                       ! -- 評価 2 と同じ
                       tmp=psi(i-1,j-1,k)  &
  &                        +0.5*(bnd(i-1,j,k)*dy(j)+bnd(i,j-1,k)*dx(i))

                    end if

                 case (-8)  ! 両方フラックス一定で z 面の x, y 右下か左上角.
                    if(i==1.and.j==ny)then  ! -- 評価 1
                       tmp=psi(i+1,j-1,k)  &
  &                        +0.5*(bnd(i+1,j,k)*dy(j)-bnd(i,j-1,k)*dx(i))

                    else if(i==nx.and.j==1)then  ! -- 評価 2
                       tmp=psi(i-1,j+1,k)  &
  &                        +0.5*(-bnd(i-1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j+1,k)==10)then
                       ! -- 評価 1 と同じ
                       tmp=psi(i+1,j-1,k)  &
  &                        +0.5*(bnd(i+1,j,k)*dy(j)-bnd(i,j-1,k)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j-1,k)==10)then
                       ! -- 評価 2 と同じ
                       tmp=psi(i-1,j+1,k)  &
  &                        +0.5*(-bnd(i-1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))
                    end if

                 case (12)  ! 両方フラックス一定で y 面の x, z 右上か左下角.
                    if(i==1.and.k==1)then  ! -- 評価 1
                       tmp=psi(i+1,j,k+1)  &
  &                        -0.5*(bnd(i+1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    else if(i==nx.and.k==nz)then  ! -- 評価 2
                       tmp=psi(i-1,j,k-1)  &
  &                        +0.5*(bnd(i-1,j,k)*dz(k)+bnd(i,j,k-1)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 1 と同じ
                       tmp=psi(i+1,j,k+1)  &
  &                        -0.5*(bnd(i+1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 2 と同じ
                       tmp=psi(i-1,j,k-1)  &
  &                        +0.5*(bnd(i-1,j,k)*dz(k)+bnd(i,j,k-1)*dx(i))
                    end if

                 case (-12)  ! 両方フラックス一定で y 面の x, z 右下か左上角.
                    if(i==1.and.k==nz)then  ! -- 評価 1
                       tmp=psi(i+1,j,k-1)  &
  &                        +0.5*(bnd(i+1,j,k)*dz(k)-bnd(i,j,k-1)*dx(i))

                    else if(i==nx.and.k==1)then  ! -- 評価 2
                       tmp=psi(i-1,j,k+1)  &
  &                        +0.5*(-bnd(i-1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 1 と同じ
                       tmp=psi(i+1,j,k-1)  &
  &                        +0.5*(bnd(i+1,j,k)*dz(k)-bnd(i,j,k-1)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 2 と同じ
                       tmp=psi(i-1,j,k+1)  &
  &                        +0.5*(-bnd(i-1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))
                    end if

                 case (24)  ! 両方フラックス一定で x 面の y, z 右上か左下角.
                    if(j==1.and.k==1)then  ! -- 評価 1
                       tmp=psi(i,j+1,k+1)  &
  &                        -0.5*(bnd(i,j+1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    else if(j==ny.and.k==nz)then  ! -- 評価 2
                       tmp=psi(i,j-1,k-1)  &
  &                        +0.5*(bnd(i,j-1,k)*dz(k)+bnd(i,j,k-1)*dy(j))

                    else if(ib(i,j-1,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 1 と同じ
                       tmp=psi(i,j+1,k+1)  &
  &                        -0.5*(bnd(i,j+1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    else if(ib(i,j+1,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 2 と同じ
                       tmp=psi(i,j-1,k-1)  &
  &                        +0.5*(bnd(i,j-1,k)*dz(k)+bnd(i,j,k-1)*dy(j))
                    end if

                 case (-24)  ! 両方フラックス一定で x 面の y, z 右下か左上角.
                    if(j==1.and.k==nz)then  ! -- 評価 1
                       tmp=psi(i,j+1,k-1)  &
  &                        +0.5*(bnd(i,j+1,k)*dz(k)-bnd(i,j,k-1)*dy(j))

                    else if(j==ny.and.k==1)then  ! -- 評価 2
                       tmp=psi(i,j-1,k+1)  &
  &                        +0.5*(-bnd(i,j-1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    else if(ib(i,j-1,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 1 と同じ
                       tmp=psi(i,j+1,k-1)  &
  &                        +0.5*(bnd(i,j+1,k)*dz(k)-bnd(i,j,k-1)*dy(j))

                    else if(ib(i,j+1,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 2 と同じ
                       tmp=psi(i,j-1,k+1)  &
  &                        +0.5*(-bnd(i,j-1,k)*dz(k)+bnd(i,j,k+1)*dy(j))
                    end if

                 !-- 以降, 隅領域なので, 個別に設定.
                 case (11)  ! 両方フラックス一定で (1,1,1) 点, もしくは内部領域.
                    tmp=psi(i+1,j+1,k+1)  &
  &                           -(bnd(i,j+1,k+1)*dx(i)  &
  &                            +bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j+1,k)*dz(k))/3.0

                 case (13)  ! 両方フラックス一定で (nx,1,1) 点, もしくは内部領域.
                    tmp=psi(i-1,j+1,k+1)  &
  &                           -(-bnd(i,j+1,k+1)*dx(i)  &
  &                             +bnd(i-1,j,k+1)*dy(j)  &
  &                             +bnd(i-1,j+1,k)*dz(k))/3.0

                 case (17)  ! 両方フラックス一定で (1,ny,1) 点, もしくは内部領域.
                    tmp=psi(i+1,j-1,k+1)  &
  &                           -(bnd(i,j-1,k+1)*dx(i)  &
  &                            -bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j-1,k)*dz(k))/3.0

                 case (19)  ! 両方フラックス一定で (nx,ny,1) 点, もしくは内部領域.
                    tmp=psi(i-1,j-1,k+1)  &
  &                           -(-bnd(i,j-1,k+1)*dx(i)  &
  &                             -bnd(i-1,j,k+1)*dy(j)  &
  &                             +bnd(i-1,j-1,k)*dz(k))/3.0

                 case (23)  ! 両方フラックス一定で (1,1,nz) 点, もしくは内部領域.
                    tmp=psi(i+1,j+1,k-1)  &
  &                           -(bnd(i,j+1,k-1)*dx(i)  &
  &                            +bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j+1,k)*dz(k))/3.0

                 case (29)  ! 両方フラックス一定で (nx,1,nz) 点, もしくは内部領域.
                    tmp=psi(i-1,j+1,k-1)  &
  &                           -(-bnd(i,j+1,k-1)*dx(i)  &
  &                             +bnd(i-1,j,k-1)*dy(j)  &
  &                             -bnd(i-1,j+1,k)*dz(k))/3.0

                 case (31)  ! 両方フラックス一定で (1,ny,nz) 点, もしくは内部領域.
                    tmp=psi(i+1,j-1,k-1)  &
  &                           -(bnd(i,j-1,k-1)*dx(i)  &
  &                            -bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j-1,k)*dz(k))/3.0

                 case (37)  ! 両方フラックス一定で (nx,ny,nz) 点, もしくは内部領域.
                    tmp=psi(i-1,j-1,k-1)  &
  &                           -(-bnd(i,j-1,k-1)*dx(i)  &
  &                             -bnd(i-1,j,k-1)*dy(j)  &
  &                             -bnd(i-1,j-1,k)*dz(k))/3.0

                 case (-11)  ! 両方フラックス一定で (i+1,j+1,k+1) に 10 が設定.
                    tmp=(psi(i-1,j+1,k+1)+psi(i+1,j-1,k+1)+psi(i+1,j+1,k-1)  &
  &                            +bnd(i,j+1,k+1)*dx(i)+bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j+1,k)*dz(k))/3.0

                 case (-13)  ! 両方フラックス一定で (i-1,j+1,k+1) に 10 が設定.
                    tmp=(psi(i+1,j+1,k+1)+psi(i-1,j-1,k+1)+psi(i-1,j+1,k-1)  &
  &                            -bnd(i,j+1,k+1)*dx(i)+bnd(i-1,j,k+1)*dy(j)  &
  &                            +bnd(i-1,j+1,k)*dz(k))/3.0

                 case (-17)  ! 両方フラックス一定で (i+1,j-1,k+1) に 10 が設定.
                    tmp=(psi(i-1,j-1,k+1)+psi(i+1,j+1,k+1)+psi(i+1,j-1,k-1)  &
  &                            +bnd(i,j-1,k+1)*dx(i)-bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j-1,k)*dz(k))/3.0

                 case (-19)  ! 両方フラックス一定で (i-1,j-1,k+1) に 10 が設定.
                    tmp=(psi(i+1,j-1,k+1)+psi(i-1,j+1,k+1)+psi(i-1,j-1,k-1)  &
  &                            -bnd(i,j-1,k+1)*dx(i)-bnd(i-1,j,k+1)*dy(j)  &
  &                            +bnd(i-1,j-1,k)*dz(k))/3.0

                 case (-23)  ! 両方フラックス一定で (i+1,j+1,k-1) に 10 が設定.
                    tmp=(psi(i-1,j+1,k-1)+psi(i+1,j-1,k-1)+psi(i+1,j+1,k+1)  &
  &                            +bnd(i,j+1,k-1)*dx(i)+bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j+1,k)*dz(k))/3.0

                 case (-29)  ! 両方フラックス一定で (i-1,j+1,k-1) に 10 が設定.
                    tmp=(psi(i+1,j+1,k-1)+psi(i-1,j-1,k-1)+psi(i-1,j+1,k+1)  &
  &                            -bnd(i,j+1,k-1)*dx(i)+bnd(i-1,j,k-1)*dy(j)  &
  &                            -bnd(i-1,j+1,k)*dz(k))/3.0

                 case (-31)  ! 両方フラックス一定で (i+1,j-1,k-1) に 10 が設定.
                    tmp=(psi(i-1,j-1,k-1)+psi(i+1,j+1,k-1)+psi(i+1,j-1,k+1)  &
  &                            +bnd(i,j-1,k-1)*dx(i)-bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j-1,k)*dz(k))/3.0

                 case (-37)  ! 両方フラックス一定で (i-1,j-1,k-1) に 10 が設定.
                    tmp=(psi(i+1,j-1,k-1)+psi(i-1,j+1,k-1)+psi(i-1,j-1,k+1)  &
  &                            -bnd(i,j-1,k-1)*dx(i)-bnd(i-1,j,k-1)*dy(j)  &
  &                            -bnd(i-1,j-1,k)*dz(k))/3.0

                 end select

              end if

              if(sor_flag.eqv..true.)then
                 tmp=(1.0-accc)*psi(i,j,k)+tmp*accc
              end if

              err=abs(tmp-psi(i,j,k))

!-- 最大誤差の更新
              if(err_max<=err)then
                 err_max=err
              end if

              psi(i,j,k)=tmp

           end do
        end do
     end do

     if(nl/=0)then
        counter=counter+1
        if(counter==nl)then
           exit
        else
           err_max=eps
        end if
     end if

  end do

!-- 境界の設定

  call calculate_bound_3df( ib, dx, dy, dz, bnd, psi )

!-- 未定義領域には undef を代入する.

  do k=1,nz
     do j=1,ny
        do i=1,nx
           if(ib(i,j,k)==10)then
              psi(i,j,k)=defun
           end if
        end do
     end do
  end do

end subroutine Ellip_GauSei_3df

!----------------------------------
!----------------------------------

subroutine Ellip_GauSei_3dd(x, y, z, rho, eps, boundary, psi, bound_opt,  &
  &                         xa, ya, za, a, b, c, d, e, f, g,  &
  &                         undef, inner_bound, init_flag, accel, ln, check_flag,  &
  &                         helmod_flag, helmod_dl )
! ガウス=ザイデル法による楕円型方程式の求積
! 各オプション配列は, ポアソン系の各微分項の値. デフォルトはゼロで設定される.
! $$xa\dfrac{\partial ^2\psi}{\partial x^2} +ya\dfrac{\partial ^2\psi}{\partial y^2} +za\dfrac{\partial ^2\psi}{\partial z^2} +a\dfrac{\partial ^2\psi}{\partial x\partial y} +b\dfrac{\partial ^2\psi}{\partial y\partial z} +c\dfrac{\partial ^2\psi}{\partial z\partial x} +d\dfrac{\partial \psi}{\partial x} +e\dfrac{\partial \psi}{\partial y} +f\dfrac{\partial \psi}{\partial z} +g\psi =\rho $$
! の各係数に対応している.
  implicit none
  double precision, intent(in) :: x(:)  ! 領域の x 座標
  double precision, intent(in) :: y(:)  ! 領域の y 座標
  double precision, intent(in) :: z(:)  ! 領域の z 座標
  double precision, intent(in) :: rho(size(x),size(y),size(z))  ! ポアソン方程式の強制項
                   ! rho =0 でラプラス方程式も求積可能
  double precision, intent(in) :: eps  ! 収束条件
  character(6), intent(in) :: boundary  ! 境界条件
                ! 4 文字で各辺の境界条件を与える.
                ! 1 文字目 : x 下端, 2 文字目 : y 左端, 3 文字目 : x 上端,
                ! 4 文字目 : y 右端
                ! boundary は 1 : 固定端境界, 2 : 自由端境界, 3 : 周期境界
  double precision, intent(in), optional :: bound_opt(size(x),size(y),size(z))
                             ! 境界での強制
                             ! ノイマン境界の場合 : フラックス値
  double precision, intent(in), optional :: xa(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: ya(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: za(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: a(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: b(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: c(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: d(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: e(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: f(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: g(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(inout) :: psi(size(x),size(y),size(z))  ! ポアソン方程式の解
  double precision, intent(in), optional :: undef  ! 未定義値
  integer, intent(in), optional :: inner_bound(size(x),size(y),size(z))
                             ! 内部領域の境界. 値に応じてその格子点で境界値計算
                             ! 1 = 固定端境界, 10 = 境界の内側.
                             ! 2 = y 方向自由端境界 (フラックスは上向き)
                             ! -2 = y 方向自由端境界 (フラックスは下向き)
                             ! 4 = x 方向自由端境界 (フラックスは右向き)
                             ! -4 = x 方向自由端境界 (フラックスは左向き)
                             ! 3 = 周期境界
                             ! 8 = |_, ~| で両方とも自由境界条件
                             ! -8 = |~, _| で両方とも自由境界条件
                             ! この引数が与えられなければ全領域を計算する.
                             ! 境界の内側格子点 (10) は反復計算を行わず,
                             ! undef で設定された値もしくはゼロが入る.
                             ! このときの境界値は bound_opt の値が用いられる.
  logical, intent(in), optional :: init_flag  ! psi の値をゼロで初期化するか.
                             ! .true. = 初期化する. .false. = 初期化しない.
                             ! デフォルトでは初期化する.
  double precision, intent(in), optional :: accel  ! SOR の加速係数 (0 < accel < 2)
                             ! デフォルト = 1
  integer, intent(in), optional :: ln  ! 反復回数
                             ! この値が与えられるとき, eps の値に無関係に
                             ! ln 回ループさせる.
  logical, intent(in), optional :: check_flag  ! true の場合, 最高階数の
                             ! 係数が負号でも stop する.
                             ! default = .false. (係数が 0 になる場合のみ stop)
  logical, intent(in), optional :: helmod_flag  ! g が与えられたときのヘルムホルツ
                             ! 方程式の収束判定を満たすために, g の値を調整する.
                             ! .true.: 調整してメッセージを出力する.
                             ! .false.: 調整せず, 収束判定を満たさなければメッセージを出して終了する
                             ! default = .false.
  logical, intent(in), optional :: helmod_dl  ! helmod_flag = .true. のときのデバッグ出力
                             ! .true.: g が調整されたときに逐次メッセージを出力する.
                             ! .false.: ルーチンが起動したときに一度だけメッセージを出力する.
                             ! default = .false.

  integer :: i, j, k, ix, jy, kz, nl, counter
  integer :: nx  ! x 方向の配列要素
  integer :: ny  ! y 方向の配列要素
  integer :: nz  ! z 方向の配列要素
  integer :: nmax  ! nx, ny, nz の最大値
  integer :: signa, signb, signc  ! 各係数を計算するかどうか
  integer, dimension(size(x),size(y),size(z)) :: ib

  double precision, parameter :: eps_helmod=1.0d-5  ! helmod_fact における微小係数
  double precision :: defun
  double precision :: tmp, err, err_max
  double precision :: tmp_b, accc
  double precision :: dcosinv_nmax, hel_fact, h2, tmpg, acccinv
  double precision :: bnd(size(x),size(y),size(z))
  double precision :: dx(size(x)), dy(size(y)), dx2(size(x)), dy2(size(y))
  double precision :: dz(size(z)), dz2(size(z))
  double precision, dimension(size(x),size(y)) :: dxdy
  double precision, dimension(size(y),size(z)) :: dydz
  double precision, dimension(size(x),size(z)) :: dxdz
  double precision, dimension(size(x),size(y),size(z)) :: xt, yt, zt, at, bt, ct, dt, et, ft, gt
  double precision, dimension(size(x),size(y),size(z)) :: xdp, xdm, yep, yem, zfp, zfm, xyz, divi

  character(6) :: bound
  logical :: hel_flag, hel_dl
  logical, dimension(size(x),size(y),size(z)) :: inner_flag

  bound(1:6)=boundary(1:6)

  nx=size(x)
  ny=size(y)
  nz=size(z)

!-- 応答関数の初期化

  if(present(init_flag))then
     if(init_flag.eqv..true.)then
        psi = 0.0d0
     end if
  else
     psi = 0.0d0
  end if

!-- 内部境界の判別フラグの設定

  if(present(inner_bound))then
     call set_bound_3d( bound, ib, inner_flag, inner_bound )
  else
     call set_bound_3d( bound, ib, inner_flag )
  end if

!-- 領域・内部境界における境界値の設定

  if(present(bound_opt))then
     call setval_bound_3dd( ib, bnd, psi, bound_opt )
  else
     call setval_bound_3dd( ib, bnd, psi )
  end if

!-- 未定義値の設定

  if(present(undef))then
     defun=undef
  else
     defun=0.0d0
  end if

!-- 係数の代入
!-- xa, ya, za については, 値が入れられていなければ, 全配列に 1 を代入する.
  if(present(xa))then
     call set_coe_3dd( xt, ext=xa )
  else
     call set_coe_3dd( xt, def=1.0d0 )
  end if

  if(present(ya))then
     call set_coe_3dd( yt, ext=ya )
  else
     call set_coe_3dd( yt, def=1.0d0 )
  end if

  if(present(za))then
     call set_coe_3dd( zt, ext=za )
  else
     call set_coe_3dd( zt, def=1.0d0 )
  end if

  if(present(a))then
     call set_coe_3dd( at, ext=a )
     signa=1
  else
     call set_coe_3dd( at, def=0.0d0 )
     signa=0
  end if

  if(present(b))then
     call set_coe_3dd( bt, ext=b )
     signb=1
  else
     call set_coe_3dd( bt, def=0.0d0 )
     signb=0
  end if

  if(present(c))then
     signc=1
     call set_coe_3dd( ct, ext=c )
  else
     call set_coe_3dd( ct, def=0.0d0 )
     signc=0
  end if

  if(present(d))then
     call set_coe_3dd( dt, ext=d )
  else
     call set_coe_3dd( dt, def=0.0d0 )
  end if

  if(present(e))then
     call set_coe_3dd( et, ext=e )
  else
     call set_coe_3dd( et, def=0.0d0 )
  end if

  if(present(f))then
     call set_coe_3dd( ft, ext=f )
  else
     call set_coe_3dd( ft, def=0.0d0 )
  end if

!-- 最高階数における係数チェック. (係数がゼロでないか調べる.)

  if(present(check_flag))then
     if(check_flag.eqv..true.)then
        call check_le_coe_3dd( xt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
        call check_le_coe_3dd( yt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
        call check_le_coe_3dd( zt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
     else
        call check_coe_3dd( xt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
        call check_coe_3dd( yt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
        call check_coe_3dd( zt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
     end if
  else
     call check_coe_3dd( xt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
     call check_coe_3dd( yt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
     call check_coe_3dd( zt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
  end if

!-- 加速係数の判定

  if(present(accel))then
     accc=accel
  else
     accc=1.0d0
  end if

!-- ln の処理
  if(present(ln))then
     nl=ln
  else
     nl=0
  end if

!-- ヘルムホルツ方程式収束判定

  hel_flag=.false.
  hel_dl=.false.
  if(present(g))then
     call set_coe_3dd( gt, ext=g )
     nmax=max(max(nx,ny),nz)
     dcosinv_nmax=1.0d0/dcos(pi_dp/dble(nmax))
     if(present(helmod_flag))then
        hel_flag=helmod_flag
        if((hel_flag.eqv..true.).and.present(helmod_dl))then
           hel_dl=helmod_dl
           if(hel_dl.eqv..false.)then
              write(*,*) "*** MESSAGE (Ellip_GauSei_3dd) ***: helmod_flag is active"
           end if
        end if
     end if
     acccinv=1.0d0/accc
  else
     call set_coe_3dd( gt, def=0.0d0 )
  end if

!-- 以下で先に格子間隔等の 1 回計算でよいものを求めておく.
!-- これらは 1 方向のみで変化すればよい.
!-- 格子点間隔の計算
  do i=2,nx-1
     dx(i)=(x(i+1)-x(i-1))*0.5d0
     dx2(i)=dx(i)**2
  end do
  do j=2,ny-1
     dy(j)=(y(j+1)-y(j-1))*0.5d0
     dy2(j)=dy(j)**2
  end do
  do k=2,nz-1
     dz(k)=(z(k+1)-z(k-1))*0.5d0
     dz2(k)=dz(k)**2
  end do

  dx(1)=(x(2)-x(1))
  dx(nx)=(x(nx)-x(nx-1))
  dy(1)=(y(2)-y(1))
  dy(ny)=(y(ny)-y(ny-1))
  dz(1)=(z(2)-z(1))
  dz(nz)=(z(nz)-z(nz-1))

  do j=1,ny
     do i=1,nx
        dxdy(i,j)=dx(i)*dy(j)
     end do
  end do
  do k=1,nz
     do j=1,ny
        dydz(j,k)=dy(j)*dz(k)
     end do
  end do
  do k=1,nz
     do i=1,nx
        dxdz(i,k)=dx(i)*dz(k)
     end do
  end do

!-- ポアソン係数の計算
!-- 実際にポアソン係数が使用されるのは, 境界を除いた 1 回り内側なので,
!-- 計算量削減のため, ループをそのようにしておく.

  xyz=0.0d0
  xdp=0.0d0
  xdm=0.0d0
  yep=0.0d0
  yem=0.0d0
  zfp=0.0d0
  zfm=0.0d0

!-- 最高次数係数 ac の計算
!$omp parallel default(shared)

  if(present(g))then
  ! g が指定された場合, ヘルムホルツ方程式の収束判定も実施 (David et al. 2014)

!$omp do schedule(dynamic) private(i,j,k,hel_fact,h2,tmpg)

     do k=2,nz-1
        do j=2,ny-1
           do i=2,nx-1
              tmpg=gt(i,j,k)
              if(gt(i,j,k)>0.0d0)then
                 h2=1.0d0/(2.0d0*(xt(i,j,k)/dx2(i)+yt(i,j,k)/dy2(j)+zt(i,j,k)/dz2(k)))
                 hel_fact=acccinv*dcosinv_nmax*(1.0d0-gt(i,j,k)*h2)
                 if(dabs(hel_fact)<=1.0d0)then
                    if(hel_flag.eqv..true.)then
                       if(hel_fact>0.0d0)then
                          tmpg=(1.0d0-(1.0d0/(1.0d0-eps_helmod))/(acccinv*dcosinv_nmax))/h2
                       else
                          tmpg=(1.0d0+(1.0d0/(1.0d0-eps_helmod))/(acccinv*dcosinv_nmax))/h2
                       end if
                       if(hel_dl.eqv..true.)then
                          write(*,*) "*** MESSAGE (Ellip_GauSei_3d) ***: Detect the critical value for the Helmholtz solver."
                          write(*,*) "Forced the grid value to the smaller than the critical value."
                          write(*,'(a12,1P2E16.8,3i5)') "Original vs Modified: ", gt(i,j,k), tmpg, i, j, k
                       end if
                    else
                       write(*,*) "*** ERROR (Ellip_GauSei_3d) ***: Detect critical value for the Helmholtz solver."
                       write(*,*) "Critical value (1-f/c), i, j, k:", hel_fact, gt(i,j,k), i, j, k
                       !write(*,*) "Critical value (1-f/c), i, j, k:", hel_fact, i, j, k
                       stop
                    end if
                 end if
              end if
              xyz(i,j,k)=accc/(2.0d0*(xt(i,j,k)/dx2(i)+yt(i,j,k)/dy2(j)+zt(i,j,k)/dz2(k))-tmpg)
           end do
        end do
     end do

!$omp end do

  else

!$omp do schedule(runtime) private(i,j,k)

     do k=2,nz-1
        do j=2,ny-1
           do i=2,nx-1
              xyz(i,j,k)=accc/(2.0d0*(xt(i,j,k)/dx2(i)+yt(i,j,k)/dy2(j)+zt(i,j,k)/dz2(k))-gt(i,j,k))
           end do
        end do
     end do

!$omp end do

  end if

!$omp barrier
!$omp do schedule(dynamic) private(i,j,k)

  do k=2,nz-1
     do j=2,ny-1
        do i=2,nx-1
           xdp(i,j,k)=(xt(i,j,k)/(dx2(i))+0.5d0*dt(i,j,k)/dx(i))*xyz(i,j,k)
           xdm(i,j,k)=(xt(i,j,k)/(dx2(i))-0.5d0*dt(i,j,k)/dx(i))*xyz(i,j,k)
           yep(i,j,k)=(yt(i,j,k)/(dy2(j))+0.5d0*et(i,j,k)/dy(j))*xyz(i,j,k)
           yem(i,j,k)=(yt(i,j,k)/(dy2(j))-0.5d0*et(i,j,k)/dy(j))*xyz(i,j,k)
           zfp(i,j,k)=(zt(i,j,k)/(dz2(k))+0.5d0*ft(i,j,k)/dz(k))*xyz(i,j,k)
           zfm(i,j,k)=(zt(i,j,k)/(dz2(k))-0.5d0*ft(i,j,k)/dz(k))*xyz(i,j,k)
           if(signa==1)then
              at(i,j,k)=0.25d0*at(i,j,k)/(dxdy(i,j))*xyz(i,j,k)
           else
              at(i,j,k)=0.0d0
           end if
           if(signb==1)then
              bt(i,j,k)=0.25d0*bt(i,j,k)/(dydz(j,k))*xyz(i,j,k)
           else
              bt(i,j,k)=0.0d0
           end if
           if(signc==1)then
              ct(i,j,k)=0.25d0*ct(i,j,k)/(dxdz(i,k))*xyz(i,j,k)
           else
              ct(i,j,k)=0.0d0
           end if
        end do
     end do
  end do

!$omp end do
!$omp end parallel

  err_max=eps  ! while に入るための便宜的措置
  counter=0

!-- 実際のソルバ ---
  do while(err_max>=eps)
     err_max=0.0d0

     do k=1,nz
        do j=1,ny
           do i=1,nx

!-- 以降, 反復計算に必要な点の周囲 8 点についてそれぞれ
!-- inner_flag のチェックと同時に適切な値をそれぞれ逐次計算する,
!-- 下で, 各方向の格子計算を行っているが, ib=8,-8 は 4 隅格子のときにしか
!-- case select しない. なぜなら, 真横や上下に隅格子がくることはありえない.
              if(inner_flag(i,j,k).eqv..false.)then  ! .false. なら領域計算開始

                 tmp=-rho(i,j,k)*xyz(i,j,k)
                 tmp=tmp+xdp(i,j,k)*psi(i+1,j,k)  &
  &                     +xdm(i,j,k)*psi(i-1,j,k)  &
  &                     +yep(i,j,k)*psi(i,j+1,k)  &
  &                     +yem(i,j,k)*psi(i,j-1,k)  &
  &                     +zfp(i,j,k)*psi(i,j,k+1)  &
  &                     +zfm(i,j,k)*psi(i,j,k-1)

                 if(signa==1)then
                    tmp_b=at(i,j,k)*(psi(i+1,j+1,k)+psi(i-1,j-1,k)  &
  &                                 -psi(i+1,j-1,k)-psi(i-1,j+1,k))
                 else
                    tmp_b=0.0d0
                 end if
                 if(signb==1)then  ! そもそも bt = 0 なら計算しない.
                    tmp_b=tmp_b+bt(i,j,k)*(psi(i,j+1,k+1)+psi(i,j-1,k-1)  &
  &                                       -psi(i,j-1,k+1)-psi(i,j+1,k-1))
                 end if
                 if(signc==1)then  ! そもそも bt = 0 なら計算しない.
                    tmp_b=tmp_b+ct(i,j,k)*(psi(i+1,j,k+1)+psi(i-1,j,k-1)  &
  &                                       -psi(i-1,j,k+1)-psi(i+1,j,k-1))
                 end if
              
                 tmp=tmp+tmp_b

              else   ! .true. なら境界計算開始.

                 select case (ib(i,j,k))
                 case (1)
                    tmp=bnd(i,j,k)

                 case (2)  ! x 方向にフラックス一定, 上側が参照値
                    tmp=psi(i+1,j,k)-bnd(i,j,k)*dx(i)

                 case (-2)  ! x 方向にフラックス一定, 下側が参照値
                    tmp=psi(i-1,j,k)+bnd(i,j,k)*dx(i)

                 case (4)  ! y 方向にフラックス一定, 右側が参照値
                    tmp=psi(i,j+1,k)-bnd(i,j,k)*dy(j)

                 case (-4)  ! y 方向にフラックス一定, 左側が参照値
                    tmp=psi(i,j-1,k)+bnd(i,j,k)*dy(j)

                 case (6)  ! y 方向にフラックス一定, 右側が参照値
                    tmp=psi(i,j,k+1)-bnd(i,j,k)*dz(k)

                 case (-6)  ! y 方向にフラックス一定, 左側が参照値
                    tmp=psi(i,j,k-1)+bnd(i,j,k)*dz(k)

                 case (3)  ! 12 辺, もしくは 8 点で周期境界を判断
                    if(i==1)then
                       ix=nx-1
                    else if(i==nx)then
                       ix=2
                    else
                       ix=i
                    end if
                    if(j==1)then
                       jy=ny-1
                    else if(j==ny)then
                       jy=2
                    else
                       jy=j
                    end if
                    if(k==1)then
                       kz=nz-1
                    else if(k==nz)then
                       kz=2
                    else
                       kz=k
                    end if
                    tmp=psi(ix,jy,kz)

                 case (8)  ! 両方フラックス一定で z 面の x, y 右上か左下角.
                    if(i==1.and.j==1)then  ! -- 評価 1
                       tmp=psi(i+1,j+1,k)  &
  &                        -0.5d0*(bnd(i+1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                    else if(i==nx.and.j==ny)then  ! -- 評価 2
                       tmp=psi(i-1,j-1,k)  &
  &                        +0.5d0*(bnd(i-1,j,k)*dy(j)+bnd(i,j-1,k)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j-1,k)==10)then
                       ! -- 評価 1 と同じ
                       tmp=psi(i+1,j+1,k)  &
  &                        -0.5d0*(bnd(i+1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j+1,k)==10)then
                       ! -- 評価 2 と同じ
                       tmp=psi(i-1,j-1,k)  &
  &                        +0.5d0*(bnd(i-1,j,k)*dy(j)+bnd(i,j-1,k)*dx(i))

                    end if

                 case (-8)  ! 両方フラックス一定で z 面の x, y 右下か左上角.
                    if(i==1.and.j==ny)then  ! -- 評価 1
                       tmp=psi(i+1,j-1,k)  &
  &                        +0.5d0*(bnd(i+1,j,k)*dy(j)-bnd(i,j-1,k)*dx(i))

                    else if(i==nx.and.j==1)then  ! -- 評価 2
                       tmp=psi(i-1,j+1,k)  &
  &                        +0.5d0*(-bnd(i-1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j+1,k)==10)then
                       ! -- 評価 1 と同じ
                       tmp=psi(i+1,j-1,k)  &
  &                        +0.5d0*(bnd(i+1,j,k)*dy(j)-bnd(i,j-1,k)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j-1,k)==10)then
                       ! -- 評価 2 と同じ
                       tmp=psi(i-1,j+1,k)  &
  &                        +0.5d0*(-bnd(i-1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))
                    end if

                 case (12)  ! 両方フラックス一定で y 面の x, z 右上か左下角.
                    if(i==1.and.k==1)then  ! -- 評価 1
                       tmp=psi(i+1,j,k+1)  &
  &                        -0.5d0*(bnd(i+1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    else if(i==nx.and.k==nz)then  ! -- 評価 2
                       tmp=psi(i-1,j,k-1)  &
  &                        +0.5d0*(bnd(i-1,j,k)*dz(k)+bnd(i,j,k-1)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 1 と同じ
                       tmp=psi(i+1,j,k+1)  &
  &                        -0.5d0*(bnd(i+1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 2 と同じ
                       tmp=psi(i-1,j,k-1)  &
  &                        +0.5d0*(bnd(i-1,j,k)*dz(k)+bnd(i,j,k-1)*dx(i))
                    end if

                 case (-12)  ! 両方フラックス一定で y 面の x, z 右下か左上角.
                    if(i==1.and.k==nz)then  ! -- 評価 1
                       tmp=psi(i+1,j,k-1)  &
  &                        +0.5d0*(bnd(i+1,j,k)*dz(k)-bnd(i,j,k-1)*dx(i))

                    else if(i==nx.and.k==1)then  ! -- 評価 2
                       tmp=psi(i-1,j,k+1)  &
  &                        +0.5d0*(-bnd(i-1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 1 と同じ
                       tmp=psi(i+1,j,k-1)  &
  &                        +0.5d0*(bnd(i+1,j,k)*dz(k)-bnd(i,j,k-1)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 2 と同じ
                       tmp=psi(i-1,j,k+1)  &
  &                        +0.5d0*(-bnd(i-1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))
                    end if

                 case (24)  ! 両方フラックス一定で x 面の y, z 右上か左下角.
                    if(j==1.and.k==1)then  ! -- 評価 1
                       tmp=psi(i,j+1,k+1)  &
  &                        -0.5d0*(bnd(i,j+1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    else if(j==ny.and.k==nz)then  ! -- 評価 2
                       tmp=psi(i,j-1,k-1)  &
  &                        +0.5d0*(bnd(i,j-1,k)*dz(k)+bnd(i,j,k-1)*dy(j))

                    else if(ib(i,j-1,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 1 と同じ
                       tmp=psi(i,j+1,k+1)  &
  &                        -0.5d0*(bnd(i,j+1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    else if(ib(i,j+1,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 2 と同じ
                       tmp=psi(i,j-1,k-1)  &
  &                        +0.5d0*(bnd(i,j-1,k)*dz(k)+bnd(i,j,k-1)*dy(j))
                    end if

                 case (-24)  ! 両方フラックス一定で x 面の y, z 右下か左上角.
                    if(j==1.and.k==nz)then  ! -- 評価 1
                       tmp=psi(i,j+1,k-1)  &
  &                        +0.5d0*(bnd(i,j+1,k)*dz(k)-bnd(i,j,k-1)*dy(j))

                    else if(j==ny.and.k==1)then  ! -- 評価 2
                       tmp=psi(i,j-1,k+1)  &
  &                        +0.5d0*(-bnd(i,j-1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    else if(ib(i,j-1,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 1 と同じ
                       tmp=psi(i,j+1,k-1)  &
  &                        +0.5d0*(bnd(i,j+1,k)*dz(k)-bnd(i,j,k-1)*dy(j))

                    else if(ib(i,j+1,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 2 と同じ
                       tmp=psi(i,j-1,k+1)  &
  &                        +0.5d0*(-bnd(i,j-1,k)*dz(k)+bnd(i,j,k+1)*dy(j))
                    end if

                 !-- 以降, 隅領域なので, 個別に設定.
                 case (11)  ! 両方フラックス一定で (1,1,1) 点, もしくは内部領域.
                    tmp=psi(i+1,j+1,k+1)  &
  &                           -(bnd(i,j+1,k+1)*dx(i)  &
  &                            +bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j+1,k)*dz(k))/3.0d0

                 case (13)  ! 両方フラックス一定で (nx,1,1) 点, もしくは内部領域.
                    tmp=psi(i-1,j+1,k+1)  &
  &                           -(-bnd(i,j+1,k+1)*dx(i)  &
  &                             +bnd(i-1,j,k+1)*dy(j)  &
  &                             +bnd(i-1,j+1,k)*dz(k))/3.0d0

                 case (17)  ! 両方フラックス一定で (1,ny,1) 点, もしくは内部領域.
                    tmp=psi(i+1,j-1,k+1)  &
  &                           -(bnd(i,j-1,k+1)*dx(i)  &
  &                            -bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j-1,k)*dz(k))/3.0d0

                 case (19)  ! 両方フラックス一定で (nx,ny,1) 点, もしくは内部領域.
                    tmp=psi(i-1,j-1,k+1)  &
  &                           -(-bnd(i,j-1,k+1)*dx(i)  &
  &                             -bnd(i-1,j,k+1)*dy(j)  &
  &                             +bnd(i-1,j-1,k)*dz(k))/3.0d0

                 case (23)  ! 両方フラックス一定で (1,1,nz) 点, もしくは内部領域.
                    tmp=psi(i+1,j+1,k-1)  &
  &                           -(bnd(i,j+1,k-1)*dx(i)  &
  &                            +bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j+1,k)*dz(k))/3.0d0

                 case (29)  ! 両方フラックス一定で (nx,1,nz) 点, もしくは内部領域.
                    tmp=psi(i-1,j+1,k-1)  &
  &                           -(-bnd(i,j+1,k-1)*dx(i)  &
  &                             +bnd(i-1,j,k-1)*dy(j)  &
  &                             -bnd(i-1,j+1,k)*dz(k))/3.0d0

                 case (31)  ! 両方フラックス一定で (1,ny,nz) 点, もしくは内部領域.
                    tmp=psi(i+1,j-1,k-1)  &
  &                           -(bnd(i,j-1,k-1)*dx(i)  &
  &                            -bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j-1,k)*dz(k))/3.0d0

                 case (37)  ! 両方フラックス一定で (nx,ny,nz) 点, もしくは内部領域.
                    tmp=psi(i-1,j-1,k-1)  &
  &                           -(-bnd(i,j-1,k-1)*dx(i)  &
  &                             -bnd(i-1,j,k-1)*dy(j)  &
  &                             -bnd(i-1,j-1,k)*dz(k))/3.0d0

                 case (-11)  ! 両方フラックス一定で (i+1,j+1,k+1) に 10 が設定.
                    tmp=(psi(i-1,j+1,k+1)+psi(i+1,j-1,k+1)+psi(i+1,j+1,k-1)  &
  &                            +bnd(i,j+1,k+1)*dx(i)+bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j+1,k)*dz(k))/3.0d0

                 case (-13)  ! 両方フラックス一定で (i-1,j+1,k+1) に 10 が設定.
                    tmp=(psi(i+1,j+1,k+1)+psi(i-1,j-1,k+1)+psi(i-1,j+1,k-1)  &
  &                            -bnd(i,j+1,k+1)*dx(i)+bnd(i-1,j,k+1)*dy(j)  &
  &                            +bnd(i-1,j+1,k)*dz(k))/3.0d0

                 case (-17)  ! 両方フラックス一定で (i+1,j-1,k+1) に 10 が設定.
                    tmp=(psi(i-1,j-1,k+1)+psi(i+1,j+1,k+1)+psi(i+1,j-1,k-1)  &
  &                            +bnd(i,j-1,k+1)*dx(i)-bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j-1,k)*dz(k))/3.0d0

                 case (-19)  ! 両方フラックス一定で (i-1,j-1,k+1) に 10 が設定.
                    tmp=(psi(i+1,j-1,k+1)+psi(i-1,j+1,k+1)+psi(i-1,j-1,k-1)  &
  &                            -bnd(i,j-1,k+1)*dx(i)-bnd(i-1,j,k+1)*dy(j)  &
  &                            +bnd(i-1,j-1,k)*dz(k))/3.0d0

                 case (-23)  ! 両方フラックス一定で (i+1,j+1,k-1) に 10 が設定.
                    tmp=(psi(i-1,j+1,k-1)+psi(i+1,j-1,k-1)+psi(i+1,j+1,k+1)  &
  &                            +bnd(i,j+1,k-1)*dx(i)+bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j+1,k)*dz(k))/3.0d0

                 case (-29)  ! 両方フラックス一定で (i-1,j+1,k-1) に 10 が設定.
                    tmp=(psi(i+1,j+1,k-1)+psi(i-1,j-1,k-1)+psi(i-1,j+1,k+1)  &
  &                            -bnd(i,j+1,k-1)*dx(i)+bnd(i-1,j,k-1)*dy(j)  &
  &                            -bnd(i-1,j+1,k)*dz(k))/3.0d0

                 case (-31)  ! 両方フラックス一定で (i+1,j-1,k-1) に 10 が設定.
                    tmp=(psi(i-1,j-1,k-1)+psi(i+1,j+1,k-1)+psi(i+1,j-1,k+1)  &
  &                            +bnd(i,j-1,k-1)*dx(i)-bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j-1,k)*dz(k))/3.0d0

                 case (-37)  ! 両方フラックス一定で (i-1,j-1,k-1) に 10 が設定.
                    tmp=(psi(i+1,j-1,k-1)+psi(i-1,j+1,k-1)+psi(i-1,j-1,k+1)  &
  &                            -bnd(i,j-1,k-1)*dx(i)-bnd(i-1,j,k-1)*dy(j)  &
  &                            -bnd(i-1,j-1,k)*dz(k))/3.0d0

                 end select

                 tmp=tmp*accc   ! 内部領域には xyz を通して accc がかかっているが, 境界計算ではかかっていないのでここでかける.

              end if

              ! 修正分 tmp には, accc がかかっているので psi_old * accc となっている状態
              tmp=(1.0d0-accc)*psi(i,j,k)+tmp
              !tmp=(1.0d0-accc)*psi(i,j,k)+tmp*accc

              err=abs(tmp-psi(i,j,k))

!-- 最大誤差の更新
              if(err_max<=err)then
                 err_max=err
              end if

              psi(i,j,k)=tmp

           end do
        end do
     end do

     if(nl/=0)then
        counter=counter+1
        if(counter==nl)then
           exit
        else
           err_max=eps
        end if
     end if

  end do

!-- 境界の設定

  call calculate_bound_3dd( ib, dx, dy, dz, bnd, psi )

!-- 未定義領域には undef を代入する.

  do k=1,nz
     do j=1,ny
        do i=1,nx
           if(ib(i,j,k)==10)then
              psi(i,j,k)=defun
           end if
        end do
     end do
  end do

end subroutine Ellip_GauSei_3dd

!----------------------------------
!----------------------------------

subroutine Ellip_Jacobi_3df(x, y, z, rho, eps, boundary, psi, bound_opt,  &
  &                         xa, ya, za, a, b, c, d, e, f, g,  &
  &                         undef, inner_bound, init_flag, accel, ln, check_flag,  &
  &                         helmod_flag, helmod_dl )
use omp_lib
! openmp によるスレッド並列が可能.
! ガウス・ザイデル法ではアルゴリズムの点から並列化が困難と思われたので,
! 並列計算によるポアソン方程式の求積が必要となるなら,
! ヤコビ法のものを使用されたい.
! 各オプション配列は, ポアソン系の各微分項の値. デフォルトはゼロで設定される.
! $$xa\dfrac{\partial ^2\psi}{\partial x^2} +ya\dfrac{\partial ^2\psi}{\partial y^2} +za\dfrac{\partial ^2\psi}{\partial z^2} +a\dfrac{\partial ^2\psi}{\partial x\partial y} +b\dfrac{\partial ^2\psi}{\partial y\partial z} +c\dfrac{\partial ^2\psi}{\partial z\partial x} +d\dfrac{\partial \psi}{\partial x} +e\dfrac{\partial \psi}{\partial y} +f\dfrac{\partial \psi}{\partial z} +g\psi =\rho $$
! の各係数に対応している.
  implicit none
  real, intent(in) :: x(:)  ! 領域の x 座標
  real, intent(in) :: y(:)  ! 領域の y 座標
  real, intent(in) :: z(:)  ! 領域の z 座標
  real, intent(in) :: rho(size(x),size(y),size(z))  ! ポアソン方程式の強制項
                   ! rho =0 でラプラス方程式も求積可能
  real, intent(in) :: eps  ! 収束条件
  character(6), intent(in) :: boundary  ! 境界条件
                ! 4 文字で各辺の境界条件を与える.
                ! 1 文字目 : x 下端, 2 文字目 : y 左端, 3 文字目 : x 上端,
                ! 4 文字目 : y 右端
                ! boundary は 1 : 固定端境界, 2 : 自由端境界, 3 : 周期境界
  real, intent(in), optional :: bound_opt(size(x),size(y),size(z))
                             ! 境界での強制
                             ! ノイマン境界の場合 : フラックス値
  real, intent(in), optional :: xa(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: ya(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: za(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: a(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: b(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: c(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: d(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: e(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: f(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: g(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(inout) :: psi(size(x),size(y),size(z))  ! ポアソン方程式の解
  real, intent(in), optional :: undef  ! 未定義値
  integer, intent(in), optional :: inner_bound(size(x),size(y),size(z))
                             ! 内部領域の境界. 値に応じてその格子点で境界値計算
                             ! 1 = 固定端境界, 10 = 境界の内側.
                             ! 2 = y 方向自由端境界 (フラックスは上向き)
                             ! -2 = y 方向自由端境界 (フラックスは下向き)
                             ! 4 = x 方向自由端境界 (フラックスは右向き)
                             ! -4 = x 方向自由端境界 (フラックスは左向き)
                             ! 3 = 周期境界
                             ! 8 = |_, ~| で両方とも自由境界条件
                             ! -8 = |~, _| で両方とも自由境界条件
                             ! この引数が与えられなければ全領域を計算する.
                             ! 境界の内側格子点 (10) は反復計算を行わず,
                             ! undef で設定された値もしくはゼロが入る.
                             ! このときの境界値は bound_opt の値が用いられる.
  logical, intent(in), optional :: init_flag  ! psi の値をゼロで初期化するか.
                             ! .true. = 初期化する. .false. = 初期化しない.
                             ! デフォルトでは初期化する.
  real, intent(in), optional :: accel  ! SOR の加速係数 (0 < accel < 2)
                             ! デフォルト = 1
  integer, intent(in), optional :: ln  ! 反復回数
                             ! この値が与えられるとき, eps の値に無関係に
                             ! ln 回ループさせる.
  logical, intent(in), optional :: check_flag  ! true の場合, 最高階数の
                             ! 係数が負号でも stop する.
                             ! default = .false. (係数が 0 になる場合のみ stop)
  logical, intent(in), optional :: helmod_flag  ! g が与えられたときのヘルムホルツ
                             ! 方程式の収束判定を満たすために, g の値を調整する.
                             ! .true.: 調整してメッセージを出力する.
                             ! .false.: 調整せず, 収束判定を満たさなければメッセージを出して終了する
                             ! default = .false.
  logical, intent(in), optional :: helmod_dl  ! helmod_flag = .true. のときのデバッグ出力
                             ! .true.: g が調整されたときに逐次メッセージを出力する.
                             ! .false.: ルーチンが起動したときに一度だけメッセージを出力する.
                             ! default = .false.

  integer :: i, j, k, ix, jy, kz, nl, counter
  integer :: nx  ! x 方向の配列要素
  integer :: ny  ! y 方向の配列要素
  integer :: nz  ! z 方向の配列要素
  integer :: nmax  ! nx, ny, nz の最大値
  integer :: signa, signb, signc  ! 各係数を計算するかどうか
  integer, dimension(size(x),size(y),size(z)) :: ib

  real :: defun
  real :: err, err_max, accc
  real :: dcosinv_nmax, hel_fact, h2, tmpg, acccinv
  real :: bnd(size(x),size(y),size(z))
  real :: dx(size(x)), dy(size(y)), dx2(size(x)), dy2(size(y))
  real :: dz(size(z)), dz2(size(z))
  real, dimension(size(x),size(y)) :: dxdy
  real, dimension(size(y),size(z)) :: dydz
  real, dimension(size(x),size(z)) :: dxdz
  real, dimension(size(x),size(y),size(z)) :: xt, yt, zt, at, bt, ct, dt, et, ft, gt
  real, dimension(size(x),size(y),size(z)) :: xdp, xdm, yep, yem, zfp, zfm, xyz, divi
  real, dimension(size(x),size(y),size(z)) :: tmp, tmp_b

  character(6) :: bound
  logical :: sor_flag, hel_flag, hel_dl
  logical, dimension(size(x),size(y),size(z)) :: inner_flag

  bound(1:6)=boundary(1:6)

  nx=size(x)
  ny=size(y)
  nz=size(z)

!-- 応答関数の初期化

  if(present(init_flag))then
     if(init_flag.eqv..true.)then
        psi = 0.0
     end if
  else
     psi = 0.0
  end if

!-- 内部境界の判別フラグの設定

  if(present(inner_bound))then
     call set_bound_3d( bound, ib, inner_flag, inner_bound )
  else
     call set_bound_3d( bound, ib, inner_flag )
  end if

!-- 領域・内部境界における境界値の設定

  if(present(bound_opt))then
     call setval_bound_3df( ib, bnd, psi, bound_opt )
  else
     call setval_bound_3df( ib, bnd, psi )
  end if

!-- 未定義値の設定

  if(present(undef))then
     defun=undef
  else
     defun=0.0
  end if

!-- 係数の代入
!-- xa, ya, za については, 値が入れられていなければ, 全配列に 1 を代入する.
  if(present(xa))then
     call set_coe_3df( xt, ext=xa )
  else
     call set_coe_3df( xt, def=1.0 )
  end if

  if(present(ya))then
     call set_coe_3df( yt, ext=ya )
  else
     call set_coe_3df( yt, def=1.0 )
  end if

  if(present(za))then
     call set_coe_3df( zt, ext=za )
  else
     call set_coe_3df( zt, def=1.0 )
  end if

  if(present(a))then
     call set_coe_3df( at, ext=a )
     signa=1
  else
     call set_coe_3df( at, def=0.0 )
     signa=0
  end if

  if(present(b))then
     call set_coe_3df( bt, ext=b )
     signb=1
  else
     call set_coe_3df( bt, def=0.0 )
     signb=0
  end if

  if(present(c))then
     signc=1
     call set_coe_3df( ct, ext=c )
  else
     call set_coe_3df( ct, def=0.0 )
     signc=0
  end if

  if(present(d))then
     call set_coe_3df( dt, ext=d )
  else
     call set_coe_3df( dt, def=0.0 )
  end if

  if(present(e))then
     call set_coe_3df( et, ext=e )
  else
     call set_coe_3df( et, def=0.0 )
  end if

  if(present(f))then
     call set_coe_3df( ft, ext=f )
  else
     call set_coe_3df( ft, def=0.0 )
  end if

!-- 最高階数における係数チェック. (係数がゼロでないか調べる.)

  if(present(check_flag))then
     if(check_flag.eqv..true.)then
        call check_le_coe_3df( xt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
        call check_le_coe_3df( yt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
        call check_le_coe_3df( zt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
     else
        call check_coe_3df( xt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
        call check_coe_3df( yt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
        call check_coe_3df( zt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
     end if
  else
     call check_coe_3df( xt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
     call check_coe_3df( yt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
     call check_coe_3df( zt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
  end if

!-- 加速係数の判定

  if(present(accel))then
     accc=accel
     sor_flag=.true.
  else
     accc=1.0
     sor_flag=.false.
  end if

!-- ln の処理
  if(present(ln))then
     nl=ln
  else
     nl=0
  end if

!-- ヘルムホルツ方程式収束判定

  hel_flag=.false.
  hel_dl=.false.
  if(present(g))then
     call set_coe_3df( gt, ext=g )
     nmax=max(max(nx,ny),nz)
     dcosinv_nmax=1.0/cos(pi/real(nmax))
     if(present(helmod_flag))then
        hel_flag=helmod_flag
        if((hel_flag.eqv..true.).and.present(helmod_dl))then
           hel_dl=helmod_dl
           if(hel_dl.eqv..false.)then
              write(*,*) "*** MESSAGE (Ellip_GauSei_3dd) ***: helmod_flag is active"
           end if
        end if
     end if
     acccinv=1.0/accc
  else
     call set_coe_3df( gt, def=0.0 )
  end if

!-- 以下で先に格子間隔等の 1 回計算でよいものを求めておく.
!-- これらは 1 方向のみで変化すればよい.
!-- 格子点間隔の計算
  do i=2,nx-1
     dx(i)=(x(i+1)-x(i-1))*0.5
     dx2(i)=dx(i)**2
  end do
  do j=2,ny-1
     dy(j)=(y(j+1)-y(j-1))*0.5
     dy2(j)=dy(j)**2
  end do
  do k=2,nz-1
     dz(k)=(z(k+1)-z(k-1))*0.5
     dz2(k)=dz(k)**2
  end do

  dx(1)=(x(2)-x(1))
  dx(nx)=(x(nx)-x(nx-1))
  dy(1)=(y(2)-y(1))
  dy(ny)=(y(ny)-y(ny-1))
  dz(1)=(z(2)-z(1))
  dz(nz)=(z(nz)-z(nz-1))

  do j=1,ny
     do i=1,nx
        dxdy(i,j)=dx(i)*dy(j)
     end do
  end do
  do k=1,nz
     do j=1,ny
        dydz(j,k)=dy(j)*dz(k)
     end do
  end do
  do k=1,nz
     do i=1,nx
        dxdz(i,k)=dx(i)*dz(k)
     end do
  end do

!-- ポアソン係数の計算
!-- 実際にポアソン係数が使用されるのは, 境界を除いた 1 回り内側なので,
!-- 計算量削減のため, ループをそのようにしておく.

  xyz=0.0
  xdp=0.0
  xdm=0.0
  yep=0.0
  yem=0.0
  zfp=0.0
  zfm=0.0

!-- 最高次数係数 ac の計算
!$omp parallel default(shared)

  if(present(g))then
  ! g が指定された場合, ヘルムホルツ方程式の収束判定も実施 (David et al. 2014)

!$omp do schedule(dynamic) private(i,j,k,hel_fact,h2,tmpg)

     do k=2,nz-1
        do j=2,ny-1
           do i=2,nx-1
              tmpg=gt(i,j,k)
              if(gt(i,j,k)>0.0)then
                 h2=1.0/(2.0*(xt(i,j,k)/dx2(i)+yt(i,j,k)/dy2(j)+zt(i,j,k)/dz2(k)))
                 hel_fact=acccinv*dcosinv_nmax*(1.0-gt(i,j,k)*h2)
                 if(abs(hel_fact)<=1.0)then
                    if(hel_flag.eqv..true.)then
                       tmpg=(1.0-(10.0/9.0)/(acccinv*dcosinv_nmax))/h2
                       if(hel_dl.eqv..true.)then
                          write(*,*) "*** MESSAGE (Ellip_GauSei_3d) ***: Detect the critical value for the Helmholtz solver."
                          write(*,*) "Forced the grid value to the smaller than the critical value."
                          write(*,'(a12,1P2E16.8,3i5)') "Original vs Modified: ", gt(i,j,k), tmpg, i, j, k
                       end if
                    else
                       write(*,*) "*** ERROR (Ellip_GauSei_3d) ***: Detect critical value for the Helmholtz solver."
                       write(*,*) "Critical value (1-f/c), i, j, k:", hel_fact, gt(i,j,k), i, j, k
                       !write(*,*) "Critical value (1-f/c), i, j, k:", hel_fact, i, j, k
                       stop
                    end if
                 end if
              end if
              xyz(i,j,k)=1.0/(2.0*(xt(i,j,k)/dx2(i)+yt(i,j,k)/dy2(j)+zt(i,j,k)/dz2(k))-tmpg)
           end do
        end do
     end do

!$omp end do

  else

!$omp do schedule(runtime) private(i,j,k)

     do k=2,nz-1
        do j=2,ny-1
           do i=2,nx-1
              xyz(i,j,k)=1.0/(2.0*(xt(i,j,k)/dx2(i)+yt(i,j,k)/dy2(j)+zt(i,j,k)/dz2(k))-gt(i,j,k))
           end do
        end do
     end do

!$omp end do

  end if

!$omp barrier
!$omp do schedule(dynamic) private(i,j,k)

  do k=2,nz-1
     do j=2,ny-1
        do i=2,nx-1
           xdp(i,j,k)=(xt(i,j,k)/(dx2(i))+0.5*dt(i,j,k)/dx(i))*xyz(i,j,k)
           xdm(i,j,k)=(xt(i,j,k)/(dx2(i))-0.5*dt(i,j,k)/dx(i))*xyz(i,j,k)
           yep(i,j,k)=(yt(i,j,k)/(dy2(j))+0.5*et(i,j,k)/dy(j))*xyz(i,j,k)
           yem(i,j,k)=(yt(i,j,k)/(dy2(j))-0.5*et(i,j,k)/dy(j))*xyz(i,j,k)
           zfp(i,j,k)=(zt(i,j,k)/(dz2(k))+0.5*ft(i,j,k)/dz(k))*xyz(i,j,k)
           zfm(i,j,k)=(zt(i,j,k)/(dz2(k))-0.5*ft(i,j,k)/dz(k))*xyz(i,j,k)
           if(signa==1)then
              at(i,j,k)=0.25*at(i,j,k)/(dxdy(i,j))*xyz(i,j,k)
           else
              at(i,j,k)=0.0
           end if
           if(signb==1)then
              bt(i,j,k)=0.25*bt(i,j,k)/(dydz(j,k))*xyz(i,j,k)
           else
              bt(i,j,k)=0.0
           end if
           if(signc==1)then
              ct(i,j,k)=0.25*ct(i,j,k)/(dxdz(i,k))*xyz(i,j,k)
           else
              ct(i,j,k)=0.0
           end if
        end do
     end do
  end do

!$omp end do
!$omp end parallel

  err_max=eps  ! while に入るための便宜的措置
  counter=0

!-- 実際のソルバ ---
  do while(err_max>=eps)
     err_max=0.0

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k,ix,jy,kz)
     do k=1,nz
        do j=1,ny
           do i=1,nx

!-- 以降, 反復計算に必要な点の周囲 8 点についてそれぞれ
!-- inner_flag のチェックと同時に適切な値をそれぞれ逐次計算する,
!-- 下で, 各方向の格子計算を行っているが, ib=8,-8 は 4 隅格子のときにしか
!-- case select しない. なぜなら, 真横や上下に隅格子がくることはありえない.
              if(inner_flag(i,j,k).eqv..false.)then  ! .false. なら領域計算開始

                 tmp(i,j,k)=-rho(i,j,k)*xyz(i,j,k)
                 tmp(i,j,k)=tmp(i,j,k)+xdp(i,j,k)*psi(i+1,j,k)  &
  &                                   +xdm(i,j,k)*psi(i-1,j,k)  &
  &                                   +yep(i,j,k)*psi(i,j+1,k)  &
  &                                   +yem(i,j,k)*psi(i,j-1,k)  &
  &                                   +zfp(i,j,k)*psi(i,j,k+1)  &
  &                                   +zfm(i,j,k)*psi(i,j,k-1)


                 if(signa==1)then
                    tmp_b(i,j,k)=at(i,j,k)*(psi(i+1,j+1,k)+psi(i-1,j-1,k)  &
  &                                        -psi(i+1,j-1,k)-psi(i-1,j+1,k))
                 else
                    tmp_b(i,j,k)=0.0
                 end if
                 if(signb==1)then  ! そもそも bt = 0 なら計算しない.
                    tmp_b(i,j,k)=tmp_b(i,j,k)  &
  &                             +bt(i,j,k)*(psi(i,j+1,k+1)+psi(i,j-1,k-1)  &
  &                                        -psi(i,j-1,k+1)-psi(i,j+1,k-1))
                 end if
                 if(signc==1)then  ! そもそも bt = 0 なら計算しない.
                    tmp_b(i,j,k)=tmp_b(i,j,k)  &
  &                             +ct(i,j,k)*(psi(i+1,j,k+1)+psi(i-1,j,k-1)  &
  &                                        -psi(i-1,j,k+1)-psi(i+1,j,k-1))
                 end if
              
                 tmp(i,j,k)=tmp(i,j,k)+tmp_b(i,j,k)

              else   ! .true. なら境界計算開始.

                 select case (ib(i,j,k))
                 case (1)
                    tmp(i,j,k)=bnd(i,j,k)

                 case (2)  ! x 方向にフラックス一定, 上側が参照値
                    tmp(i,j,k)=psi(i+1,j,k)-bnd(i,j,k)*dx(i)

                 case (-2)  ! x 方向にフラックス一定, 下側が参照値
                    tmp(i,j,k)=psi(i-1,j,k)+bnd(i,j,k)*dx(i)

                 case (4)  ! y 方向にフラックス一定, 右側が参照値
                    tmp(i,j,k)=psi(i,j+1,k)-bnd(i,j,k)*dy(j)

                 case (-4)  ! y 方向にフラックス一定, 左側が参照値
                    tmp(i,j,k)=psi(i,j-1,k)+bnd(i,j,k)*dy(j)

                 case (6)  ! y 方向にフラックス一定, 右側が参照値
                    tmp(i,j,k)=psi(i,j,k+1)-bnd(i,j,k)*dz(k)

                 case (-6)  ! y 方向にフラックス一定, 左側が参照値
                    tmp(i,j,k)=psi(i,j,k-1)+bnd(i,j,k)*dz(k)

                 case (3)  ! 12 辺, もしくは 8 点で周期境界を判断
                    if(i==1)then
                       ix=nx-1
                    else if(i==nx)then
                       ix=2
                    else
                       ix=i
                    end if
                    if(j==1)then
                       jy=ny-1
                    else if(j==ny)then
                       jy=2
                    else
                       jy=j
                    end if
                    if(k==1)then
                       kz=nz-1
                    else if(k==nz)then
                       kz=2
                    else
                       kz=k
                    end if
                    tmp(i,j,k)=psi(ix,jy,kz)

                 case (8)  ! 両方フラックス一定で z 面の x, y 右上か左下角.
                    if(i==1.and.j==1)then  ! -- 評価 1
                       tmp(i,j,k)=psi(i+1,j+1,k)  &
  &                        -0.5*(bnd(i+1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                    else if(i==nx.and.j==ny)then  ! -- 評価 2
                       tmp(i,j,k)=psi(i-1,j-1,k)  &
  &                        +0.5*(bnd(i-1,j,k)*dy(j)+bnd(i,j-1,k)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j-1,k)==10)then
                       ! -- 評価 1 と同じ
                       tmp(i,j,k)=psi(i+1,j+1,k)  &
  &                        -0.5*(bnd(i+1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j+1,k)==10)then
                       ! -- 評価 2 と同じ
                       tmp(i,j,k)=psi(i-1,j-1,k)  &
  &                        +0.5*(bnd(i-1,j,k)*dy(j)+bnd(i,j-1,k)*dx(i))

                    end if

                 case (-8)  ! 両方フラックス一定で z 面の x, y 右下か左上角.
                    if(i==1.and.j==ny)then  ! -- 評価 1
                       tmp(i,j,k)=psi(i+1,j-1,k)  &
  &                        +0.5*(bnd(i+1,j,k)*dy(j)-bnd(i,j-1,k)*dx(i))

                    else if(i==nx.and.j==1)then  ! -- 評価 2
                       tmp(i,j,k)=psi(i-1,j+1,k)  &
  &                        +0.5*(-bnd(i-1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j+1,k)==10)then
                       ! -- 評価 1 と同じ
                       tmp(i,j,k)=psi(i+1,j-1,k)  &
  &                        +0.5*(bnd(i+1,j,k)*dy(j)-bnd(i,j-1,k)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j-1,k)==10)then
                       ! -- 評価 2 と同じ
                       tmp(i,j,k)=psi(i-1,j+1,k)  &
  &                        +0.5*(-bnd(i-1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))
                    end if

                 case (12)  ! 両方フラックス一定で y 面の x, z 右上か左下角.
                    if(i==1.and.k==1)then  ! -- 評価 1
                       tmp(i,j,k)=psi(i+1,j,k+1)  &
  &                        -0.5*(bnd(i+1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    else if(i==nx.and.k==nz)then  ! -- 評価 2
                       tmp(i,j,k)=psi(i-1,j,k-1)  &
  &                        +0.5*(bnd(i-1,j,k)*dz(k)+bnd(i,j,k-1)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 1 と同じ
                       tmp(i,j,k)=psi(i+1,j,k+1)  &
  &                        -0.5*(bnd(i+1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 2 と同じ
                       tmp(i,j,k)=psi(i-1,j,k-1)  &
  &                        +0.5*(bnd(i-1,j,k)*dz(k)+bnd(i,j,k-1)*dx(i))

                    end if

                 case (-12)  ! 両方フラックス一定で y 面の x, z 右下か左上角.
                    if(i==1.and.k==nz)then  ! -- 評価 1
                       tmp(i,j,k)=psi(i+1,j,k-1)  &
  &                        +0.5*(bnd(i+1,j,k)*dz(k)-bnd(i,j,k-1)*dx(i))

                    else if(i==nx.and.k==1)then  ! -- 評価 2
                       tmp(i,j,k)=psi(i-1,j,k+1)  &
  &                        +0.5*(-bnd(i-1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 1 と同じ
                       tmp(i,j,k)=psi(i+1,j,k-1)  &
  &                        +0.5*(bnd(i+1,j,k)*dz(k)-bnd(i,j,k-1)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 2 と同じ
                       tmp(i,j,k)=psi(i-1,j,k+1)  &
  &                        +0.5*(-bnd(i-1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    end if

                 case (24)  ! 両方フラックス一定で x 面の y, z 右上か左下角.
                    if(j==1.and.k==1)then  ! -- 評価 1
                       tmp(i,j,k)=psi(i,j+1,k+1)  &
  &                        -0.5*(bnd(i,j+1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    else if(j==ny.and.k==nz)then  ! -- 評価 2
                       tmp(i,j,k)=psi(i,j-1,k-1)  &
  &                        +0.5*(bnd(i,j-1,k)*dz(k)+bnd(i,j,k-1)*dy(j))

                    else if(ib(i,j-1,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 1 と同じ
                       tmp(i,j,k)=psi(i,j+1,k+1)  &
  &                        -0.5*(bnd(i,j+1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    else if(ib(i,j+1,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 2 と同じ
                       tmp(i,j,k)=psi(i,j-1,k-1)  &
  &                        +0.5*(bnd(i,j-1,k)*dz(k)+bnd(i,j,k-1)*dy(j))

                    end if

                 case (-24)  ! 両方フラックス一定で x 面の y, z 右下か左上角.
                    if(j==1.and.k==nz)then  ! -- 評価 1
                       tmp(i,j,k)=psi(i,j+1,k-1)  &
  &                        +0.5*(bnd(i,j+1,k)*dz(k)-bnd(i,j,k-1)*dy(j))

                    else if(j==ny.and.k==1)then  ! -- 評価 2
                       tmp(i,j,k)=psi(i,j-1,k+1)  &
  &                        +0.5*(-bnd(i,j-1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    else if(ib(i,j-1,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 1 と同じ
                       tmp(i,j,k)=psi(i,j+1,k-1)  &
  &                        +0.5*(bnd(i,j+1,k)*dz(k)-bnd(i,j,k-1)*dy(j))

                    else if(ib(i,j+1,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 2 と同じ
                       tmp(i,j,k)=psi(i,j-1,k+1)  &
  &                        +0.5*(-bnd(i,j-1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    end if

                 !-- 以降, 隅領域なので, 個別に設定.
                 case (11)  ! 両方フラックス一定で (1,1,1) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i+1,j+1,k+1)  &
  &                           -(bnd(i,j+1,k+1)*dx(i)  &
  &                            +bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j+1,k)*dz(k))/3.0

                 case (13)  ! 両方フラックス一定で (nx,1,1) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i-1,j+1,k+1)  &
  &                           -(-bnd(i,j+1,k+1)*dx(i)  &
  &                             +bnd(i-1,j,k+1)*dy(j)  &
  &                             +bnd(i-1,j+1,k)*dz(k))/3.0

                 case (17)  ! 両方フラックス一定で (1,ny,1) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i+1,j-1,k+1)  &
  &                           -(bnd(i,j-1,k+1)*dx(i)  &
  &                            -bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j-1,k)*dz(k))/3.0

                 case (19)  ! 両方フラックス一定で (nx,ny,1) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i-1,j-1,k+1)  &
  &                           -(-bnd(i,j-1,k+1)*dx(i)  &
  &                             -bnd(i-1,j,k+1)*dy(j)  &
  &                             +bnd(i-1,j-1,k)*dz(k))/3.0

                 case (23)  ! 両方フラックス一定で (1,1,nz) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i+1,j+1,k-1)  &
  &                           -(bnd(i,j+1,k-1)*dx(i)  &
  &                            +bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j+1,k)*dz(k))/3.0

                 case (29)  ! 両方フラックス一定で (nx,1,nz) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i-1,j+1,k-1)  &
  &                           -(-bnd(i,j+1,k-1)*dx(i)  &
  &                             +bnd(i-1,j,k-1)*dy(j)  &
  &                             -bnd(i-1,j+1,k)*dz(k))/3.0

                 case (31)  ! 両方フラックス一定で (1,ny,nz) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i+1,j-1,k-1)  &
  &                           -(bnd(i,j-1,k-1)*dx(i)  &
  &                            -bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j-1,k)*dz(k))/3.0

                 case (37)  ! 両方フラックス一定で (nx,ny,nz) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i-1,j-1,k-1)  &
  &                           -(-bnd(i,j-1,k-1)*dx(i)  &
  &                             -bnd(i-1,j,k-1)*dy(j)  &
  &                             -bnd(i-1,j-1,k)*dz(k))/3.0

                 case (-11)  ! 両方フラックス一定で (i+1,j+1,k+1) に 10 が設定.
                    tmp(i,j,k)=(psi(i-1,j+1,k+1)+psi(i+1,j-1,k+1)+psi(i+1,j+1,k-1)  &
  &                            +bnd(i,j+1,k+1)*dx(i)+bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j+1,k)*dz(k))/3.0

                 case (-13)  ! 両方フラックス一定で (i-1,j+1,k+1) に 10 が設定.
                    tmp(i,j,k)=(psi(i+1,j+1,k+1)+psi(i-1,j-1,k+1)+psi(i-1,j+1,k-1)  &
  &                            -bnd(i,j+1,k+1)*dx(i)+bnd(i-1,j,k+1)*dy(j)  &
  &                            +bnd(i-1,j+1,k)*dz(k))/3.0

                 case (-17)  ! 両方フラックス一定で (i+1,j-1,k+1) に 10 が設定.
                    tmp(i,j,k)=(psi(i-1,j-1,k+1)+psi(i+1,j+1,k+1)+psi(i+1,j-1,k-1)  &
  &                            +bnd(i,j-1,k+1)*dx(i)-bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j-1,k)*dz(k))/3.0

                 case (-19)  ! 両方フラックス一定で (i-1,j-1,k+1) に 10 が設定.
                    tmp(i,j,k)=(psi(i+1,j-1,k+1)+psi(i-1,j+1,k+1)+psi(i-1,j-1,k-1)  &
  &                            -bnd(i,j-1,k+1)*dx(i)-bnd(i-1,j,k+1)*dy(j)  &
  &                            +bnd(i-1,j-1,k)*dz(k))/3.0

                 case (-23)  ! 両方フラックス一定で (i+1,j+1,k-1) に 10 が設定.
                    tmp(i,j,k)=(psi(i-1,j+1,k-1)+psi(i+1,j-1,k-1)+psi(i+1,j+1,k+1)  &
  &                            +bnd(i,j+1,k-1)*dx(i)+bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j+1,k)*dz(k))/3.0

                 case (-29)  ! 両方フラックス一定で (i-1,j+1,k-1) に 10 が設定.
                    tmp(i,j,k)=(psi(i+1,j+1,k-1)+psi(i-1,j-1,k-1)+psi(i-1,j+1,k+1)  &
  &                            -bnd(i,j+1,k-1)*dx(i)+bnd(i-1,j,k-1)*dy(j)  &
  &                            -bnd(i-1,j+1,k)*dz(k))/3.0

                 case (-31)  ! 両方フラックス一定で (i+1,j-1,k-1) に 10 が設定.
                    tmp(i,j,k)=(psi(i-1,j-1,k-1)+psi(i+1,j+1,k-1)+psi(i+1,j-1,k+1)  &
  &                            +bnd(i,j-1,k-1)*dx(i)-bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j-1,k)*dz(k))/3.0

                 case (-37)  ! 両方フラックス一定で (i-1,j-1,k-1) に 10 が設定.
                    tmp(i,j,k)=(psi(i+1,j-1,k-1)+psi(i-1,j+1,k-1)+psi(i-1,j-1,k+1)  &
  &                            -bnd(i,j-1,k-1)*dx(i)-bnd(i-1,j,k-1)*dy(j)  &
  &                            -bnd(i-1,j-1,k)*dz(k))/3.0

                 end select

              end if

              if(sor_flag.eqv..true.)then
                 tmp(i,j,k)=(1.0-accc)*psi(i,j,k)+tmp(i,j,k)*accc
              end if

           end do
        end do
     end do
!$omp end do
!$omp end parallel

!-- 最大誤差の更新
     do k=2,nz-1
        do j=2,ny-1
           do i=2,nx-1
              err=abs(tmp(i,j,k)-psi(i,j,k))
              if(err_max<=err)then
                 err_max=err
              end if
           end do
        end do
     end do

!-- 一斉更新

     do k=1,nz
        do j=1,ny
           do i=1,nx
              psi(i,j,k)=tmp(i,j,k)
           end do
        end do
     end do

     if(nl/=0)then
        counter=counter+1
        if(counter==nl)then
           exit
        else
           err_max=eps
        end if
     end if

write(*,*) "err_max", err_max
  end do

!-- 境界の設定

  call calculate_bound_3df( ib, dx, dy, dz, bnd, psi )

!-- 未定義領域には undef を代入する.

  do k=1,nz
     do j=1,ny
        do i=1,nx
           if(ib(i,j,k)==10)then
              psi(i,j,k)=defun
           end if
        end do
     end do
  end do

end subroutine Ellip_Jacobi_3df

!----------------------------------
!----------------------------------

subroutine Ellip_Jacobi_3dd(x, y, z, rho, eps, boundary, psi, bound_opt,  &
  &                         xa, ya, za, a, b, c, d, e, f, g,  &
  &                         undef, inner_bound, init_flag, accel, ln, check_flag,  &
  &                         helmod_flag, helmod_dl )
use omp_lib
! openmp によるスレッド並列が可能.
! ガウス・ザイデル法ではアルゴリズムの点から並列化が困難と思われたので,
! 並列計算によるポアソン方程式の求積が必要となるなら,
! ヤコビ法のものを使用されたい.
! 各オプション配列は, ポアソン系の各微分項の値. デフォルトはゼロで設定される.
! $$xa\dfrac{\partial ^2\psi}{\partial x^2} +ya\dfrac{\partial ^2\psi}{\partial y^2} +za\dfrac{\partial ^2\psi}{\partial z^2} +a\dfrac{\partial ^2\psi}{\partial x\partial y} +b\dfrac{\partial ^2\psi}{\partial y\partial z} +c\dfrac{\partial ^2\psi}{\partial z\partial x} +d\dfrac{\partial \psi}{\partial x} +e\dfrac{\partial \psi}{\partial y} +f\dfrac{\partial \psi}{\partial z} +g\psi =\rho $$
! の各係数に対応している.
  implicit none
  double precision, intent(in) :: x(:)  ! 領域の x 座標
  double precision, intent(in) :: y(:)  ! 領域の y 座標
  double precision, intent(in) :: z(:)  ! 領域の z 座標
  double precision, intent(in) :: rho(size(x),size(y),size(z))  ! ポアソン方程式の強制項
                   ! rho =0 でラプラス方程式も求積可能
  double precision, intent(in) :: eps  ! 収束条件
  character(6), intent(in) :: boundary  ! 境界条件
                ! 4 文字で各辺の境界条件を与える.
                ! 1 文字目 : x 下端, 2 文字目 : y 左端, 3 文字目 : x 上端,
                ! 4 文字目 : y 右端
                ! boundary は 1 : 固定端境界, 2 : 自由端境界, 3 : 周期境界
  double precision, intent(in), optional :: bound_opt(size(x),size(y),size(z))
                             ! 境界での強制
                             ! ノイマン境界の場合 : フラックス値
  double precision, intent(in), optional :: xa(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: ya(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: za(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: a(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: b(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: c(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: d(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: e(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: f(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: g(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(inout) :: psi(size(x),size(y),size(z))  ! ポアソン方程式の解
  double precision, intent(in), optional :: undef  ! 未定義値
  integer, intent(in), optional :: inner_bound(size(x),size(y),size(z))
                             ! 内部領域の境界. 値に応じてその格子点で境界値計算
                             ! 1 = 固定端境界, 10 = 境界の内側.
                             ! 2 = y 方向自由端境界 (フラックスは上向き)
                             ! -2 = y 方向自由端境界 (フラックスは下向き)
                             ! 4 = x 方向自由端境界 (フラックスは右向き)
                             ! -4 = x 方向自由端境界 (フラックスは左向き)
                             ! 3 = 周期境界
                             ! 8 = |_, ~| で両方とも自由境界条件
                             ! -8 = |~, _| で両方とも自由境界条件
                             ! この引数が与えられなければ全領域を計算する.
                             ! 境界の内側格子点 (10) は反復計算を行わず,
                             ! undef で設定された値もしくはゼロが入る.
                             ! このときの境界値は bound_opt の値が用いられる.
  logical, intent(in), optional :: init_flag  ! psi の値をゼロで初期化するか.
                             ! .true. = 初期化する. .false. = 初期化しない.
                             ! デフォルトでは初期化する.
  double precision, intent(in), optional :: accel  ! SOR の加速係数 (0 < accel < 2)
                             ! デフォルト = 1
  integer, intent(in), optional :: ln  ! 反復回数
                             ! この値が与えられるとき, eps の値に無関係に
                             ! ln 回ループさせる.
  logical, intent(in), optional :: check_flag  ! true の場合, 最高階数の
                             ! 係数が負号でも stop する.
                             ! default = .false. (係数が 0 になる場合のみ stop)
  logical, intent(in), optional :: helmod_flag  ! g が与えられたときのヘルムホルツ
                             ! 方程式の収束判定を満たすために, g の値を調整する.
                             ! .true.: 調整してメッセージを出力する.
                             ! .false.: 調整せず, 収束判定を満たさなければメッセージを出して終了する
                             ! default = .false.
  logical, intent(in), optional :: helmod_dl  ! helmod_flag = .true. のときのデバッグ出力
                             ! .true.: g が調整されたときに逐次メッセージを出力する.
                             ! .false.: ルーチンが起動したときに一度だけメッセージを出力する.
                             ! default = .false.

  integer :: i, j, k, ix, jy, kz, nl, counter
  integer :: nx  ! x 方向の配列要素
  integer :: ny  ! y 方向の配列要素
  integer :: nz  ! z 方向の配列要素
  integer :: nmax  ! nx, ny, nz の最大値
  integer :: signa, signb, signc  ! 各係数を計算するかどうか
  integer, dimension(size(x),size(y),size(z)) :: ib

  double precision :: defun
  double precision :: err, err_max, accc
  double precision :: dcosinv_nmax, hel_fact, h2, tmpg, acccinv
  double precision :: bnd(size(x),size(y),size(z))
  double precision :: dx(size(x)), dy(size(y)), dx2(size(x)), dy2(size(y))
  double precision :: dz(size(z)), dz2(size(z))
  double precision, dimension(size(x),size(y)) :: dxdy
  double precision, dimension(size(y),size(z)) :: dydz
  double precision, dimension(size(x),size(z)) :: dxdz
  double precision, dimension(size(x),size(y),size(z)) :: xt, yt, zt, at, bt, ct, dt, et, ft, gt
  double precision, dimension(size(x),size(y),size(z)) :: xdp, xdm, yep, yem, zfp, zfm, xyz, divi
  double precision, dimension(size(x),size(y),size(z)) :: tmp, tmp_b

  character(6) :: bound
  logical :: sor_flag, hel_flag, hel_dl
  logical, dimension(size(x),size(y),size(z)) :: inner_flag

  bound(1:6)=boundary(1:6)

  nx=size(x)
  ny=size(y)
  nz=size(z)

!-- 応答関数の初期化

  if(present(init_flag))then
     if(init_flag.eqv..true.)then
        psi = 0.0d0
     end if
  else
     psi = 0.0d0
  end if

!-- 内部境界の判別フラグの設定

  if(present(inner_bound))then
     call set_bound_3d( bound, ib, inner_flag, inner_bound )
  else
     call set_bound_3d( bound, ib, inner_flag )
  end if

!-- 領域・内部境界における境界値の設定

  if(present(bound_opt))then
     call setval_bound_3dd( ib, bnd, psi, bound_opt )
  else
     call setval_bound_3dd( ib, bnd, psi )
  end if

!-- 未定義値の設定

  if(present(undef))then
     defun=undef
  else
     defun=0.0d0
  end if

!-- 係数の代入
!-- xa, ya, za については, 値が入れられていなければ, 全配列に 1 を代入する.
  if(present(xa))then
     call set_coe_3dd( xt, ext=xa )
  else
     call set_coe_3dd( xt, def=1.0d0 )
  end if

  if(present(ya))then
     call set_coe_3dd( yt, ext=ya )
  else
     call set_coe_3dd( yt, def=1.0d0 )
  end if

  if(present(za))then
     call set_coe_3dd( zt, ext=za )
  else
     call set_coe_3dd( zt, def=1.0d0 )
  end if

  if(present(a))then
     call set_coe_3dd( at, ext=a )
     signa=1
  else
     call set_coe_3dd( at, def=0.0d0 )
     signa=0
  end if

  if(present(b))then
     call set_coe_3dd( bt, ext=b )
     signb=1
  else
     call set_coe_3dd( bt, def=0.0d0 )
     signb=0
  end if

  if(present(c))then
     signc=1
     call set_coe_3dd( ct, ext=c )
  else
     call set_coe_3dd( ct, def=0.0d0 )
     signc=0
  end if

  if(present(d))then
     call set_coe_3dd( dt, ext=d )
  else
     call set_coe_3dd( dt, def=0.0d0 )
  end if

  if(present(e))then
     call set_coe_3dd( et, ext=e )
  else
     call set_coe_3dd( et, def=0.0d0 )
  end if

  if(present(f))then
     call set_coe_3dd( ft, ext=f )
  else
     call set_coe_3dd( ft, def=0.0d0 )
  end if

!-- 最高階数における係数チェック. (係数がゼロでないか調べる.)

  if(present(check_flag))then
     if(check_flag.eqv..true.)then
        call check_le_coe_3dd( xt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
        call check_le_coe_3dd( yt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
        call check_le_coe_3dd( zt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
     else
        call check_coe_3dd( xt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
        call check_coe_3dd( yt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
        call check_coe_3dd( zt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
     end if
  else
     call check_coe_3dd( xt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
     call check_coe_3dd( yt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
     call check_coe_3dd( zt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
  end if

!-- 加速係数の判定

  if(present(accel))then
     accc=accel
     sor_flag=.true.
  else
     accc=1.0d0
     sor_flag=.false.
  end if

!-- ln の処理
  if(present(ln))then
     nl=ln
  else
     nl=0
  end if

!-- ヘルムホルツ方程式収束判定

  hel_flag=.false.
  hel_dl=.false.
  if(present(g))then
     call set_coe_3dd( gt, ext=g )
     nmax=max(max(nx,ny),nz)
     dcosinv_nmax=1.0d0/dcos(pi_dp/dble(nmax))
     if(present(helmod_flag))then
        hel_flag=helmod_flag
        if((hel_flag.eqv..true.).and.present(helmod_dl))then
           hel_dl=helmod_dl
           if(hel_dl.eqv..false.)then
              write(*,*) "*** MESSAGE (Ellip_GauSei_3dd) ***: helmod_flag is active"
           end if
        end if
     end if
     acccinv=1.0d0/accc
  else
     call set_coe_3dd( gt, def=0.0d0 )
  end if

!-- 以下で先に格子間隔等の 1 回計算でよいものを求めておく.
!-- これらは 1 方向のみで変化すればよい.
!-- 格子点間隔の計算
  do i=2,nx-1
     dx(i)=(x(i+1)-x(i-1))*0.5d0
     dx2(i)=dx(i)**2
  end do
  do j=2,ny-1
     dy(j)=(y(j+1)-y(j-1))*0.5d0
     dy2(j)=dy(j)**2
  end do
  do k=2,nz-1
     dz(k)=(z(k+1)-z(k-1))*0.5d0
     dz2(k)=dz(k)**2
  end do

  dx(1)=(x(2)-x(1))
  dx(nx)=(x(nx)-x(nx-1))
  dy(1)=(y(2)-y(1))
  dy(ny)=(y(ny)-y(ny-1))
  dz(1)=(z(2)-z(1))
  dz(nz)=(z(nz)-z(nz-1))

  do j=1,ny
     do i=1,nx
        dxdy(i,j)=dx(i)*dy(j)
     end do
  end do
  do k=1,nz
     do j=1,ny
        dydz(j,k)=dy(j)*dz(k)
     end do
  end do
  do k=1,nz
     do i=1,nx
        dxdz(i,k)=dx(i)*dz(k)
     end do
  end do

!-- ポアソン係数の計算
!-- 実際にポアソン係数が使用されるのは, 境界を除いた 1 回り内側なので,
!-- 計算量削減のため, ループをそのようにしておく.

  xyz=0.0d0
  xdp=0.0d0
  xdm=0.0d0
  yep=0.0d0
  yem=0.0d0
  zfp=0.0d0
  zfm=0.0d0

!-- 最高次数係数 ac の計算
!$omp parallel default(shared)

  if(present(g))then
  ! g が指定された場合, ヘルムホルツ方程式の収束判定も実施 (David et al. 2014)

!$omp do schedule(dynamic) private(i,j,k,hel_fact,h2,tmpg)

     do k=2,nz-1
        do j=2,ny-1
           do i=2,nx-1
              tmpg=gt(i,j,k)
              if(gt(i,j,k)>0.0d0)then
                 h2=1.0d0/(2.0d0*(xt(i,j,k)/dx2(i)+yt(i,j,k)/dy2(j)+zt(i,j,k)/dz2(k)))
                 hel_fact=acccinv*dcosinv_nmax*(1.0d0-gt(i,j,k)*h2)
                 if(dabs(hel_fact)<=1.0d0)then
                    if(hel_flag.eqv..true.)then
                       tmpg=(1.0d0-(10.0d0/9.0d0)/(acccinv*dcosinv_nmax))/h2
                       if(hel_dl.eqv..true.)then
                          write(*,*) "*** MESSAGE (Ellip_GauSei_3d) ***: Detect the critical value for the Helmholtz solver."
                          write(*,*) "Forced the grid value to the smaller than the critical value."
                          write(*,'(a12,1P2E16.8,3i5)') "Original vs Modified: ", gt(i,j,k), tmpg, i, j, k
                       end if
                    else
                       write(*,*) "*** ERROR (Ellip_GauSei_3d) ***: Detect critical value for the Helmholtz solver."
                       write(*,*) "Critical value (1-f/c), i, j, k:", hel_fact, gt(i,j,k), i, j, k
                       !write(*,*) "Critical value (1-f/c), i, j, k:", hel_fact, i, j, k
                       stop
                    end if
                 end if
              end if
              xyz(i,j,k)=1.0d0/(2.0d0*(xt(i,j,k)/dx2(i)+yt(i,j,k)/dy2(j)+zt(i,j,k)/dz2(k))-tmpg)
           end do
        end do
     end do

!$omp end do

  else

!$omp do schedule(runtime) private(i,j,k)

     do k=2,nz-1
        do j=2,ny-1
           do i=2,nx-1
              xyz(i,j,k)=1.0d0/(2.0d0*(xt(i,j,k)/dx2(i)+yt(i,j,k)/dy2(j)+zt(i,j,k)/dz2(k))-gt(i,j,k))
           end do
        end do
     end do

!$omp end do

  end if

!$omp barrier
!$omp do schedule(dynamic) private(i,j,k)

  do k=2,nz-1
     do j=2,ny-1
        do i=2,nx-1
           xdp(i,j,k)=(xt(i,j,k)/(dx2(i))+0.5d0*dt(i,j,k)/dx(i))*xyz(i,j,k)
           xdm(i,j,k)=(xt(i,j,k)/(dx2(i))-0.5d0*dt(i,j,k)/dx(i))*xyz(i,j,k)
           yep(i,j,k)=(yt(i,j,k)/(dy2(j))+0.5d0*et(i,j,k)/dy(j))*xyz(i,j,k)
           yem(i,j,k)=(yt(i,j,k)/(dy2(j))-0.5d0*et(i,j,k)/dy(j))*xyz(i,j,k)
           zfp(i,j,k)=(zt(i,j,k)/(dz2(k))+0.5d0*ft(i,j,k)/dz(k))*xyz(i,j,k)
           zfm(i,j,k)=(zt(i,j,k)/(dz2(k))-0.5d0*ft(i,j,k)/dz(k))*xyz(i,j,k)
           if(signa==1)then
              at(i,j,k)=0.25d0*at(i,j,k)/(dxdy(i,j))*xyz(i,j,k)
           else
              at(i,j,k)=0.0d0
           end if
           if(signb==1)then
              bt(i,j,k)=0.25d0*bt(i,j,k)/(dydz(j,k))*xyz(i,j,k)
           else
              bt(i,j,k)=0.0d0
           end if
           if(signc==1)then
              ct(i,j,k)=0.25d0*ct(i,j,k)/(dxdz(i,k))*xyz(i,j,k)
           else
              ct(i,j,k)=0.0d0
           end if
        end do
     end do
  end do

!$omp end do
!$omp end parallel

  err_max=eps  ! while に入るための便宜的措置
  counter=0

!-- 実際のソルバ ---
  do while(err_max>=eps)
     err_max=0.0d0

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k,ix,jy,kz)
     do k=1,nz
        do j=1,ny
           do i=1,nx

!-- 以降, 反復計算に必要な点の周囲 8 点についてそれぞれ
!-- inner_flag のチェックと同時に適切な値をそれぞれ逐次計算する,
!-- 下で, 各方向の格子計算を行っているが, ib=8,-8 は 4 隅格子のときにしか
!-- case select しない. なぜなら, 真横や上下に隅格子がくることはありえない.
              if(inner_flag(i,j,k).eqv..false.)then  ! .false. なら領域計算開始

                 tmp(i,j,k)=-rho(i,j,k)*xyz(i,j,k)
                 tmp(i,j,k)=tmp(i,j,k)+xdp(i,j,k)*psi(i+1,j,k)  &
  &                                   +xdm(i,j,k)*psi(i-1,j,k)  &
  &                                   +yep(i,j,k)*psi(i,j+1,k)  &
  &                                   +yem(i,j,k)*psi(i,j-1,k)  &
  &                                   +zfp(i,j,k)*psi(i,j,k+1)  &
  &                                   +zfm(i,j,k)*psi(i,j,k-1)


                 if(signa==1)then
                    tmp_b(i,j,k)=at(i,j,k)*(psi(i+1,j+1,k)+psi(i-1,j-1,k)  &
  &                                        -psi(i+1,j-1,k)-psi(i-1,j+1,k))
                 else
                    tmp_b(i,j,k)=0.0d0
                 end if
                 if(signb==1)then  ! そもそも bt = 0 なら計算しない.
                    tmp_b(i,j,k)=tmp_b(i,j,k)  &
  &                             +bt(i,j,k)*(psi(i,j+1,k+1)+psi(i,j-1,k-1)  &
  &                                        -psi(i,j-1,k+1)-psi(i,j+1,k-1))
                 end if
                 if(signc==1)then  ! そもそも bt = 0 なら計算しない.
                    tmp_b(i,j,k)=tmp_b(i,j,k)  &
  &                             +ct(i,j,k)*(psi(i+1,j,k+1)+psi(i-1,j,k-1)  &
  &                                        -psi(i-1,j,k+1)-psi(i+1,j,k-1))
                 end if
              
                 tmp(i,j,k)=tmp(i,j,k)+tmp_b(i,j,k)

              else   ! .true. なら境界計算開始.

                 select case (ib(i,j,k))
                 case (1)
                    tmp(i,j,k)=bnd(i,j,k)

                 case (2)  ! x 方向にフラックス一定, 上側が参照値
                    tmp(i,j,k)=psi(i+1,j,k)-bnd(i,j,k)*dx(i)

                 case (-2)  ! x 方向にフラックス一定, 下側が参照値
                    tmp(i,j,k)=psi(i-1,j,k)+bnd(i,j,k)*dx(i)

                 case (4)  ! y 方向にフラックス一定, 右側が参照値
                    tmp(i,j,k)=psi(i,j+1,k)-bnd(i,j,k)*dy(j)

                 case (-4)  ! y 方向にフラックス一定, 左側が参照値
                    tmp(i,j,k)=psi(i,j-1,k)+bnd(i,j,k)*dy(j)

                 case (6)  ! y 方向にフラックス一定, 右側が参照値
                    tmp(i,j,k)=psi(i,j,k+1)-bnd(i,j,k)*dz(k)

                 case (-6)  ! y 方向にフラックス一定, 左側が参照値
                    tmp(i,j,k)=psi(i,j,k-1)+bnd(i,j,k)*dz(k)

                 case (3)  ! 12 辺, もしくは 8 点で周期境界を判断
                    if(i==1)then
                       ix=nx-1
                    else if(i==nx)then
                       ix=2
                    else
                       ix=i
                    end if
                    if(j==1)then
                       jy=ny-1
                    else if(j==ny)then
                       jy=2
                    else
                       jy=j
                    end if
                    if(k==1)then
                       kz=nz-1
                    else if(k==nz)then
                       kz=2
                    else
                       kz=k
                    end if
                    tmp(i,j,k)=psi(ix,jy,kz)

                 case (8)  ! 両方フラックス一定で z 面の x, y 右上か左下角.
                    if(i==1.and.j==1)then  ! -- 評価 1
                       tmp(i,j,k)=psi(i+1,j+1,k)  &
  &                        -0.5d0*(bnd(i+1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                    else if(i==nx.and.j==ny)then  ! -- 評価 2
                       tmp(i,j,k)=psi(i-1,j-1,k)  &
  &                        +0.5d0*(bnd(i-1,j,k)*dy(j)+bnd(i,j-1,k)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j-1,k)==10)then
                       ! -- 評価 1 と同じ
                       tmp(i,j,k)=psi(i+1,j+1,k)  &
  &                        -0.5d0*(bnd(i+1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j+1,k)==10)then
                       ! -- 評価 2 と同じ
                       tmp(i,j,k)=psi(i-1,j-1,k)  &
  &                        +0.5d0*(bnd(i-1,j,k)*dy(j)+bnd(i,j-1,k)*dx(i))

                    end if

                 case (-8)  ! 両方フラックス一定で z 面の x, y 右下か左上角.
                    if(i==1.and.j==ny)then  ! -- 評価 1
                       tmp(i,j,k)=psi(i+1,j-1,k)  &
  &                        +0.5d0*(bnd(i+1,j,k)*dy(j)-bnd(i,j-1,k)*dx(i))

                    else if(i==nx.and.j==1)then  ! -- 評価 2
                       tmp(i,j,k)=psi(i-1,j+1,k)  &
  &                        +0.5d0*(-bnd(i-1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j+1,k)==10)then
                       ! -- 評価 1 と同じ
                       tmp(i,j,k)=psi(i+1,j-1,k)  &
  &                        +0.5d0*(bnd(i+1,j,k)*dy(j)-bnd(i,j-1,k)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j-1,k)==10)then
                       ! -- 評価 2 と同じ
                       tmp(i,j,k)=psi(i-1,j+1,k)  &
  &                        +0.5d0*(-bnd(i-1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))
                    end if

                 case (12)  ! 両方フラックス一定で y 面の x, z 右上か左下角.
                    if(i==1.and.k==1)then  ! -- 評価 1
                       tmp(i,j,k)=psi(i+1,j,k+1)  &
  &                        -0.5d0*(bnd(i+1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    else if(i==nx.and.k==nz)then  ! -- 評価 2
                       tmp(i,j,k)=psi(i-1,j,k-1)  &
  &                        +0.5d0*(bnd(i-1,j,k)*dz(k)+bnd(i,j,k-1)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 1 と同じ
                       tmp(i,j,k)=psi(i+1,j,k+1)  &
  &                        -0.5d0*(bnd(i+1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 2 と同じ
                       tmp(i,j,k)=psi(i-1,j,k-1)  &
  &                        +0.5d0*(bnd(i-1,j,k)*dz(k)+bnd(i,j,k-1)*dx(i))

                    end if

                 case (-12)  ! 両方フラックス一定で y 面の x, z 右下か左上角.
                    if(i==1.and.k==nz)then  ! -- 評価 1
                       tmp(i,j,k)=psi(i+1,j,k-1)  &
  &                        +0.5d0*(bnd(i+1,j,k)*dz(k)-bnd(i,j,k-1)*dx(i))

                    else if(i==nx.and.k==1)then  ! -- 評価 2
                       tmp(i,j,k)=psi(i-1,j,k+1)  &
  &                        +0.5d0*(-bnd(i-1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 1 と同じ
                       tmp(i,j,k)=psi(i+1,j,k-1)  &
  &                        +0.5d0*(bnd(i+1,j,k)*dz(k)-bnd(i,j,k-1)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 2 と同じ
                       tmp(i,j,k)=psi(i-1,j,k+1)  &
  &                        +0.5d0*(-bnd(i-1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    end if

                 case (24)  ! 両方フラックス一定で x 面の y, z 右上か左下角.
                    if(j==1.and.k==1)then  ! -- 評価 1
                       tmp(i,j,k)=psi(i,j+1,k+1)  &
  &                        -0.5d0*(bnd(i,j+1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    else if(j==ny.and.k==nz)then  ! -- 評価 2
                       tmp(i,j,k)=psi(i,j-1,k-1)  &
  &                        +0.5d0*(bnd(i,j-1,k)*dz(k)+bnd(i,j,k-1)*dy(j))

                    else if(ib(i,j-1,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 1 と同じ
                       tmp(i,j,k)=psi(i,j+1,k+1)  &
  &                        -0.5d0*(bnd(i,j+1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    else if(ib(i,j+1,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 2 と同じ
                       tmp(i,j,k)=psi(i,j-1,k-1)  &
  &                        +0.5d0*(bnd(i,j-1,k)*dz(k)+bnd(i,j,k-1)*dy(j))

                    end if

                 case (-24)  ! 両方フラックス一定で x 面の y, z 右下か左上角.
                    if(j==1.and.k==nz)then  ! -- 評価 1
                       tmp(i,j,k)=psi(i,j+1,k-1)  &
  &                        +0.5d0*(bnd(i,j+1,k)*dz(k)-bnd(i,j,k-1)*dy(j))

                    else if(j==ny.and.k==1)then  ! -- 評価 2
                       tmp(i,j,k)=psi(i,j-1,k+1)  &
  &                        +0.5d0*(-bnd(i,j-1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    else if(ib(i,j-1,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 1 と同じ
                       tmp(i,j,k)=psi(i,j+1,k-1)  &
  &                        +0.5d0*(bnd(i,j+1,k)*dz(k)-bnd(i,j,k-1)*dy(j))

                    else if(ib(i,j+1,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 2 と同じ
                       tmp(i,j,k)=psi(i,j-1,k+1)  &
  &                        +0.5d0*(-bnd(i,j-1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    end if

                 !-- 以降, 隅領域なので, 個別に設定.
                 case (11)  ! 両方フラックス一定で (1,1,1) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i+1,j+1,k+1)  &
  &                           -(bnd(i,j+1,k+1)*dx(i)  &
  &                            +bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j+1,k)*dz(k))/3.0d0

                 case (13)  ! 両方フラックス一定で (nx,1,1) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i-1,j+1,k+1)  &
  &                           -(-bnd(i,j+1,k+1)*dx(i)  &
  &                             +bnd(i-1,j,k+1)*dy(j)  &
  &                             +bnd(i-1,j+1,k)*dz(k))/3.0d0

                 case (17)  ! 両方フラックス一定で (1,ny,1) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i+1,j-1,k+1)  &
  &                           -(bnd(i,j-1,k+1)*dx(i)  &
  &                            -bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j-1,k)*dz(k))/3.0d0

                 case (19)  ! 両方フラックス一定で (nx,ny,1) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i-1,j-1,k+1)  &
  &                           -(-bnd(i,j-1,k+1)*dx(i)  &
  &                             -bnd(i-1,j,k+1)*dy(j)  &
  &                             +bnd(i-1,j-1,k)*dz(k))/3.0d0

                 case (23)  ! 両方フラックス一定で (1,1,nz) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i+1,j+1,k-1)  &
  &                           -(bnd(i,j+1,k-1)*dx(i)  &
  &                            +bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j+1,k)*dz(k))/3.0d0

                 case (29)  ! 両方フラックス一定で (nx,1,nz) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i-1,j+1,k-1)  &
  &                           -(-bnd(i,j+1,k-1)*dx(i)  &
  &                             +bnd(i-1,j,k-1)*dy(j)  &
  &                             -bnd(i-1,j+1,k)*dz(k))/3.0d0

                 case (31)  ! 両方フラックス一定で (1,ny,nz) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i+1,j-1,k-1)  &
  &                           -(bnd(i,j-1,k-1)*dx(i)  &
  &                            -bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j-1,k)*dz(k))/3.0d0

                 case (37)  ! 両方フラックス一定で (nx,ny,nz) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i-1,j-1,k-1)  &
  &                           -(-bnd(i,j-1,k-1)*dx(i)  &
  &                             -bnd(i-1,j,k-1)*dy(j)  &
  &                             -bnd(i-1,j-1,k)*dz(k))/3.0d0

                 case (-11)  ! 両方フラックス一定で (i+1,j+1,k+1) に 10 が設定.
                    tmp(i,j,k)=(psi(i-1,j+1,k+1)+psi(i+1,j-1,k+1)+psi(i+1,j+1,k-1)  &
  &                            +bnd(i,j+1,k+1)*dx(i)+bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j+1,k)*dz(k))/3.0d0

                 case (-13)  ! 両方フラックス一定で (i-1,j+1,k+1) に 10 が設定.
                    tmp(i,j,k)=(psi(i+1,j+1,k+1)+psi(i-1,j-1,k+1)+psi(i-1,j+1,k-1)  &
  &                            -bnd(i,j+1,k+1)*dx(i)+bnd(i-1,j,k+1)*dy(j)  &
  &                            +bnd(i-1,j+1,k)*dz(k))/3.0d0

                 case (-17)  ! 両方フラックス一定で (i+1,j-1,k+1) に 10 が設定.
                    tmp(i,j,k)=(psi(i-1,j-1,k+1)+psi(i+1,j+1,k+1)+psi(i+1,j-1,k-1)  &
  &                            +bnd(i,j-1,k+1)*dx(i)-bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j-1,k)*dz(k))/3.0d0

                 case (-19)  ! 両方フラックス一定で (i-1,j-1,k+1) に 10 が設定.
                    tmp(i,j,k)=(psi(i+1,j-1,k+1)+psi(i-1,j+1,k+1)+psi(i-1,j-1,k-1)  &
  &                            -bnd(i,j-1,k+1)*dx(i)-bnd(i-1,j,k+1)*dy(j)  &
  &                            +bnd(i-1,j-1,k)*dz(k))/3.0d0

                 case (-23)  ! 両方フラックス一定で (i+1,j+1,k-1) に 10 が設定.
                    tmp(i,j,k)=(psi(i-1,j+1,k-1)+psi(i+1,j-1,k-1)+psi(i+1,j+1,k+1)  &
  &                            +bnd(i,j+1,k-1)*dx(i)+bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j+1,k)*dz(k))/3.0d0

                 case (-29)  ! 両方フラックス一定で (i-1,j+1,k-1) に 10 が設定.
                    tmp(i,j,k)=(psi(i+1,j+1,k-1)+psi(i-1,j-1,k-1)+psi(i-1,j+1,k+1)  &
  &                            -bnd(i,j+1,k-1)*dx(i)+bnd(i-1,j,k-1)*dy(j)  &
  &                            -bnd(i-1,j+1,k)*dz(k))/3.0d0

                 case (-31)  ! 両方フラックス一定で (i+1,j-1,k-1) に 10 が設定.
                    tmp(i,j,k)=(psi(i-1,j-1,k-1)+psi(i+1,j+1,k-1)+psi(i+1,j-1,k+1)  &
  &                            +bnd(i,j-1,k-1)*dx(i)-bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j-1,k)*dz(k))/3.0d0

                 case (-37)  ! 両方フラックス一定で (i-1,j-1,k-1) に 10 が設定.
                    tmp(i,j,k)=(psi(i+1,j-1,k-1)+psi(i-1,j+1,k-1)+psi(i-1,j-1,k+1)  &
  &                            -bnd(i,j-1,k-1)*dx(i)-bnd(i-1,j,k-1)*dy(j)  &
  &                            -bnd(i-1,j-1,k)*dz(k))/3.0d0

                 end select

              end if

              if(sor_flag.eqv..true.)then
                 tmp(i,j,k)=(1.0d0-accc)*psi(i,j,k)+tmp(i,j,k)*accc
              end if

           end do
        end do
     end do
!$omp end do
!$omp end parallel

!-- 最大誤差の更新
     do k=2,nz-1
        do j=2,ny-1
           do i=2,nx-1
              err=abs(tmp(i,j,k)-psi(i,j,k))
              if(err_max<=err)then
                 err_max=err
              end if
           end do
        end do
     end do

!-- 一斉更新

     do k=1,nz
        do j=1,ny
           do i=1,nx
              psi(i,j,k)=tmp(i,j,k)
           end do
        end do
     end do

     if(nl/=0)then
        counter=counter+1
        if(counter==nl)then
           exit
        else
           err_max=eps
        end if
     end if

  end do

!-- 境界の設定

  call calculate_bound_3dd( ib, dx, dy, dz, bnd, psi )

!-- 未定義領域には undef を代入する.

  do k=1,nz
     do j=1,ny
        do i=1,nx
           if(ib(i,j,k)==10)then
              psi(i,j,k)=defun
           end if
        end do
     end do
  end do

end subroutine Ellip_Jacobi_3dd

!----------------------------------
!----------------------------------

subroutine Full_Multi_Grid_2df( level, x, y, rho, eps, boundary, psi,  &
  &                             a, b, c, d, e, f, conv_num, add_itr, rlu_err,  &
  &                             accel, helmod_flag, helmod_dl )
  ! Perform a V-cycle from the level grid to 1st grid (i.e., coarsest grid)
  implicit none
  integer, intent(in) :: level               ! Top grid level (M)
  real, intent(in) :: x(:)                  ! X grid on the level grid
  real, intent(in) :: y(:)                  ! Y grid on the level grid
  real, intent(in) :: rho(size(x),size(y))  ! The forcing on the level grid
  real, intent(in) :: eps        ! Threshold for iteration of GauSei
  character(4), intent(in) :: boundary       ! The boundary conditions
  real, intent(out) :: psi(size(x),size(y)) ! The updated u on the level grid
  real, intent(in), optional :: a(size(x),size(y))  ! coefficient in PDE
  real, intent(in), optional :: b(size(x),size(y))  ! coefficient in PDE
  real, intent(in), optional :: c(size(x),size(y))  ! coefficient in PDE
  real, intent(in), optional :: d(size(x),size(y))  ! coefficient in PDE
  real, intent(in), optional :: e(size(x),size(y))  ! coefficient in PDE
  real, intent(in), optional :: f(size(x),size(y))  ! coefficient in PDE
  real, intent(out), optional :: rlu_err(size(x),size(y))  ! the final error
  integer, intent(in) :: conv_num  ! number of the threshold for iteration
  integer, intent(in), optional :: add_itr  ! number of the threshold for additional iteration
  real, intent(in), optional :: accel  ! Acceleration coefficient for Successive Over Relaxation
                             ! (0 < accel < 2), default = 1
  logical, intent(in), optional :: helmod_flag
                             ! Adjustment of "g" for the stability in the Helmholtz equation on the coarse grids.
                             ! .true.: Adjusting "g" and displaying the message
                             ! .false.: No adjusting "g" and terminating with the error message
                             ! default = .false.
  logical, intent(in), optional :: helmod_dl  ! Debug output for helmod_flag = .true.
                             ! .true.: successive output in adjusting g
                             ! .false.: once output in calling this routine
                             ! default = .false.

  integer :: ii, l, sk, nxu, nyu, nxd, nyd, nxl, nyl
  real :: dx, dy, dxu, dyu, dxd, dyd
  real, allocatable, dimension(:) :: xlu, ylu, xld, yld
  real, allocatable, dimension(:,:) :: rlu, rld, ulu, uld, flu, fld

  integer :: nx, ny, nxm, nym, nxg, nyg
  integer :: i, j
  real :: dxm, dym, dxg, dyg
  real :: accc
  real, allocatable, dimension(:) :: xm, ym, xg, yg
  real, allocatable, dimension(:,:) :: rhom, psim, rhog, psig, psinew
  real, allocatable, dimension(:,:) :: am, bm, cm, dm, em, fm
  real, allocatable, dimension(:,:) :: ag, bg, cg, dg, eg, fg
  integer :: skipnum, skipnug
  logical :: ter_flag, hel_flag(level), hel_dl

  nx=size(x)
  ny=size(y)

  dx=x(2)-x(1)
  dy=y(2)-y(1)

  hel_flag=.false.
  hel_dl=.false.
  if(present(helmod_flag))then
     !-- the finest grid corresponds to hel_flag(level)
     hel_flag(1:level-1)=helmod_flag
     if(present(helmod_dl))then
        hel_dl=helmod_dl
     end if
  end if

!-- check "accel"

  if(present(accel))then
     accc=accel
  else
     accc=1.0
  end if

  !-- 1. Solving on the coasest grid
  !-- Set and Allocate variables on the coarsest grid
  skipnum=2**(level-1)
  nxm=(nx-1)/skipnum+1
  nym=(ny-1)/skipnum+1
  dxm=dx*dble(skipnum)
  dym=dy*dble(skipnum)
  allocate(xm(nxm))
  allocate(ym(nym))
  allocate(rhom(nxm,nym))
  allocate(psim(nxm,nym))
  allocate(am(nxm,nym))
  allocate(bm(nxm,nym))
  allocate(cm(nxm,nym))
  allocate(dm(nxm,nym))
  allocate(em(nxm,nym))
  allocate(fm(nxm,nym))
  xm=(/(dxm*(i-1),i=1,nxm)/)
  ym=(/(dym*(i-1),i=1,nym)/)
  am=1.0
  cm=1.0
  bm=0.0
  dm=0.0
  em=0.0
  fm=0.0
!  write(*,*) "skipnum = ", skipnum, "XY=", (nx-1)/skipnum, (ny-1)/skipnum

  !-- 2. Calculation of the initial u^l before the V-cycle
  call forward_interpo_2df( rho, rhom, level, 1 )
  if(present(a))then
     call forward_interpo_2df( a, am, level, 1 )
  end if
  if(present(b))then
     call forward_interpo_2df( b, bm, level, 1 )
  end if
  if(present(c))then
     call forward_interpo_2df( c, cm, level, 1 )
  end if
  if(present(d))then
     call forward_interpo_2df( d, dm, level, 1 )
  end if
  if(present(e))then
     call forward_interpo_2df( e, em, level, 1 )
  end if
  if(present(f))then
     call forward_interpo_2df( f, fm, level, 1 )
  end if

  call Ellip_GauSei_2df( xm, ym, rhom, eps, boundary, psim,  &
  &                      a=am, b=bm, c=cm, d=dm, e=em, f=fm, ln=conv_num,  &
  &                      accel=accc, helmod_flag=hel_flag(1), helmod_dl=hel_dl )
!write(*,*) "temporary stop", psim

  deallocate(am)
  deallocate(bm)
  deallocate(cm)
  deallocate(dm)
  deallocate(em)
  deallocate(fm)

  !-- 3. Enter cycles
  do l=2,level
     !-- Set and Allocate variables on the l grid
     skipnug=2**(level-l)
     nxg=(nx-1)/skipnug+1
     nyg=(ny-1)/skipnug+1
     dxg=dx*real(skipnug)
     dyg=dy*real(skipnug)
     allocate(xg(nxg))
     allocate(yg(nyg))
     allocate(rhog(nxg,nyg))
     allocate(psig(nxg,nyg))
     allocate(psinew(nxg,nyg))
     allocate(ag(nxg,nyg))
     allocate(bg(nxg,nyg))
     allocate(cg(nxg,nyg))
     allocate(dg(nxg,nyg))
     allocate(eg(nxg,nyg))
     allocate(fg(nxg,nyg))
     if(l==level)then
        xg(1:nxg)=x(1:nx)
        yg(1:nyg)=y(1:ny)
     else
        xg=(/(dxg*(i-1),i=1,nxg)/)
        yg=(/(dyg*(i-1),i=1,nyg)/)
     end if
     ag=1.0
     cg=1.0
     bg=0.0
     dg=0.0
     eg=0.0
     fg=0.0
!     write(*,*) "skipnum = ", skipnug, "XY=", (nx-1)/skipnug, (ny-1)/skipnug

     !-- Interpolation of the initial u^l
     call backward_interpo_2df( psim, psig, l-1, l )  ! From l-1 to l
     call forward_interpo_2df( rho, rhog, level, l )  ! From level to l
     if(present(a))then
        call forward_interpo_2df( a, ag, level, l )
     end if
     if(present(b))then
        call forward_interpo_2df( b, bg, level, l )
     end if
     if(present(c))then
        call forward_interpo_2df( c, cg, level, l )
     end if
     if(present(d))then
        call forward_interpo_2df( d, dg, level, l )
     end if
     if(present(e))then
        call forward_interpo_2df( e, eg, level, l )
     end if
     if(present(f))then
        call forward_interpo_2df( f, fg, level, l )
     end if

     !-- Enter the V-cycle from the l grid
     call GauSei_Vcycle_2df( l, xg, yg, boundary, psig, rhog, psinew, conv_num,  &
  &                          at=ag, bt=bg, ct=cg, dt=dg, et=eg, ft=fg,  &
  &                          accel=accc, helmod_flag=hel_flag(1:l), helmod_dl=hel_dl )
!write(*,*) "temporary stop", psig
!stop

     !-- Reallocate psim from l-1 to l
     deallocate(psim)
     nxm=nxg
     nym=nyg
     allocate(psim(nxm,nym))

     !-- Store psinew to psim
     psim(1:nxm,1:nym)=psinew(1:nxg,1:nyg)

     !-- Deallocate variables
     deallocate(xg)
     deallocate(yg)
     deallocate(rhog)
     deallocate(psig)
     deallocate(psinew)
     deallocate(ag)
     deallocate(bg)
     deallocate(cg)
     deallocate(dg)
     deallocate(eg)
     deallocate(fg)
  end do

  if(nxm/=nx.or.nym/=ny)then
     write(*,*) "error"
     stop
  end if

  if(present(add_itr))then
     allocate(psinew(nx,ny))
     allocate(am(nxm,nym))
     allocate(bm(nxm,nym))
     allocate(cm(nxm,nym))
     allocate(dm(nxm,nym))
     allocate(em(nxm,nym))
     allocate(fm(nxm,nym))
     am=1.0
     cm=1.0
     bm=0.0
     dm=0.0
     em=0.0
     fm=0.0
     if(present(a))then
        am=a
     end if
     if(present(b))then
        bm=b
     end if
     if(present(c))then
        cm=c
     end if
     if(present(d))then
        dm=d
     end if
     if(present(e))then
        em=e
     end if
     if(present(f))then
        fm=f
     end if
     do l=1,add_itr
        call GauSei_Vcycle_2df( level, x, y, boundary, psim, rho, psinew, conv_num,  &
  &                             at=am, bt=bm, ct=cm, dt=dm, et=em, ft=fm,  &
  &                             accel=accc, helmod_flag=hel_flag(1:level), helmod_dl=hel_dl )
        psim=psinew
     end do
  end if

  !-- 4. Set psi
!  write(*,*) "check final grid", nx, ny, nxm, nym
  psi(1:nx,1:ny)=psim(1:nxm,1:nym)

  if(present(rlu_err))then
     allocate(rlu(nx,ny))
     if(present(add_itr).eqv..false.)then  ! 以下は add_itr が有効なら割付済み
        allocate(psinew(nx,ny))
        allocate(am(nxm,nym))
        allocate(bm(nxm,nym))
        allocate(cm(nxm,nym))
        allocate(dm(nxm,nym))
        allocate(em(nxm,nym))
        allocate(fm(nxm,nym))
        am=1.0
        cm=1.0
        bm=0.0
        dm=0.0
        em=0.0
        fm=0.0
        if(present(a))then
           am=a
        end if
        if(present(b))then
           bm=b
        end if
        if(present(c))then
           cm=c
        end if
        if(present(d))then
           dm=d
        end if
        if(present(e))then
           em=e
        end if
        if(present(f))then
           fm=f
        end if
     end if

     call calc_error_2df( x, y, psim, rho, rlu, a=am, b=bm, c=cm, d=dm, e=em, f=fm )
     rlu_err=rlu
  end if

end subroutine Full_Multi_Grid_2df

!----------------------------------
!----------------------------------

subroutine Full_Multi_Grid_2dd( level, x, y, rho, eps, boundary, psi,  &
  &                             a, b, c, d, e, f, conv_num, add_itr, rlu_err,  &
  &                             accel, helmod_flag, helmod_dl )
  ! Perform a V-cycle from the level grid to 1st grid (i.e., coarsest grid)
  implicit none
  integer, intent(in) :: level               ! Top grid level (M)
  double precision, intent(in) :: x(:)                  ! X grid on the level grid
  double precision, intent(in) :: y(:)                  ! Y grid on the level grid
  double precision, intent(in) :: rho(size(x),size(y))  ! The forcing on the level grid
  double precision, intent(in) :: eps        ! Threshold for iteration of GauSei
  character(4), intent(in) :: boundary       ! The boundary conditions
  double precision, intent(out) :: psi(size(x),size(y)) ! The updated u on the level grid
  double precision, intent(in), optional :: a(size(x),size(y))  ! coefficient in PDE
  double precision, intent(in), optional :: b(size(x),size(y))  ! coefficient in PDE
  double precision, intent(in), optional :: c(size(x),size(y))  ! coefficient in PDE
  double precision, intent(in), optional :: d(size(x),size(y))  ! coefficient in PDE
  double precision, intent(in), optional :: e(size(x),size(y))  ! coefficient in PDE
  double precision, intent(in), optional :: f(size(x),size(y))  ! coefficient in PDE
  double precision, intent(out), optional :: rlu_err(size(x),size(y))  ! the final error
  integer, intent(in) :: conv_num  ! number of the threshold for iteration
  integer, intent(in), optional :: add_itr  ! number of the threshold for additional iteration
  double precision, intent(in), optional :: accel  ! Acceleration coefficient for Successive Over Relaxation
                             ! (0 < accel < 2), default = 1
  logical, intent(in), optional :: helmod_flag
                             ! Adjustment of "g" for the stability in the Helmholtz equation on the coarse grids.
                             ! .true.: Adjusting "g" and displaying the message
                             ! .false.: No adjusting "g" and terminating with the error message
                             ! default = .false.
  logical, intent(in), optional :: helmod_dl  ! Debug output for helmod_flag = .true.
                             ! .true.: successive output in adjusting g
                             ! .false.: once output in calling this routine
                             ! default = .false.

  integer :: ii, l, sk, nxu, nyu, nxd, nyd, nxl, nyl
  double precision :: dx, dy, dxu, dyu, dxd, dyd
  double precision, allocatable, dimension(:) :: xlu, ylu, xld, yld
  double precision, allocatable, dimension(:,:) :: rlu, rld, ulu, uld, flu, fld

  integer :: nx, ny, nxm, nym, nxg, nyg
  integer :: i, j
  double precision :: dxm, dym, dxg, dyg
  double precision :: accc
  double precision, allocatable, dimension(:) :: xm, ym, xg, yg
  double precision, allocatable, dimension(:,:) :: rhom, psim, rhog, psig, psinew
  double precision, allocatable, dimension(:,:) :: am, bm, cm, dm, em, fm
  double precision, allocatable, dimension(:,:) :: ag, bg, cg, dg, eg, fg
  integer :: skipnum, skipnug
  logical :: ter_flag, hel_flag(level), hel_dl

  nx=size(x)
  ny=size(y)

  dx=x(2)-x(1)
  dy=y(2)-y(1)

  hel_flag=.false.
  hel_dl=.false.
  if(present(helmod_flag))then
     !-- the finest grid corresponds to hel_flag(level)
     hel_flag(1:level-1)=helmod_flag
     if(present(helmod_dl))then
        hel_dl=helmod_dl
     end if
  end if

!-- check "accel"

  if(present(accel))then
     accc=accel
  else
     accc=1.0d0
  end if

  !-- 1. Solving on the coasest grid
  !-- Set and Allocate variables on the coarsest grid
  skipnum=2**(level-1)
  nxm=(nx-1)/skipnum+1
  nym=(ny-1)/skipnum+1
  dxm=dx*dble(skipnum)
  dym=dy*dble(skipnum)
  allocate(xm(nxm))
  allocate(ym(nym))
  allocate(rhom(nxm,nym))
  allocate(psim(nxm,nym))
  allocate(am(nxm,nym))
  allocate(bm(nxm,nym))
  allocate(cm(nxm,nym))
  allocate(dm(nxm,nym))
  allocate(em(nxm,nym))
  allocate(fm(nxm,nym))
  xm=(/(dxm*(i-1),i=1,nxm)/)
  ym=(/(dym*(i-1),i=1,nym)/)
  am=1.0d0
  cm=1.0d0
  bm=0.0d0
  dm=0.0d0
  em=0.0d0
  fm=0.0d0
!  write(*,*) "skipnum = ", skipnum, "XY=", (nx-1)/skipnum, (ny-1)/skipnum

  !-- 2. Calculation of the initial u^l before the V-cycle
  call forward_interpo_2dd( rho, rhom, level, 1 )
  if(present(a))then
     call forward_interpo_2dd( a, am, level, 1 )
  end if
  if(present(b))then
     call forward_interpo_2dd( b, bm, level, 1 )
  end if
  if(present(c))then
     call forward_interpo_2dd( c, cm, level, 1 )
  end if
  if(present(d))then
     call forward_interpo_2dd( d, dm, level, 1 )
  end if
  if(present(e))then
     call forward_interpo_2dd( e, em, level, 1 )
  end if
  if(present(f))then
     call forward_interpo_2dd( f, fm, level, 1 )
  end if

  call Ellip_GauSei_2dd( xm, ym, rhom, eps, boundary, psim,  &
  &                      a=am, b=bm, c=cm, d=dm, e=em, f=fm, ln=conv_num,  &
  &                      accel=accc, helmod_flag=hel_flag(1), helmod_dl=hel_dl )
!write(*,*) "temporary stop", psim

  deallocate(am)
  deallocate(bm)
  deallocate(cm)
  deallocate(dm)
  deallocate(em)
  deallocate(fm)

  !-- 3. Enter cycles
  do l=2,level
     !-- Set and Allocate variables on the l grid
     skipnug=2**(level-l)
     nxg=(nx-1)/skipnug+1
     nyg=(ny-1)/skipnug+1
     dxg=dx*real(skipnug)
     dyg=dy*real(skipnug)
     allocate(xg(nxg))
     allocate(yg(nyg))
     allocate(rhog(nxg,nyg))
     allocate(psig(nxg,nyg))
     allocate(psinew(nxg,nyg))
     allocate(ag(nxg,nyg))
     allocate(bg(nxg,nyg))
     allocate(cg(nxg,nyg))
     allocate(dg(nxg,nyg))
     allocate(eg(nxg,nyg))
     allocate(fg(nxg,nyg))
     if(l==level)then
        xg(1:nxg)=x(1:nx)
        yg(1:nyg)=y(1:ny)
     else
        xg=(/(dxg*(i-1),i=1,nxg)/)
        yg=(/(dyg*(i-1),i=1,nyg)/)
     end if
     ag=1.0d0
     cg=1.0d0
     bg=0.0d0
     dg=0.0d0
     eg=0.0d0
     fg=0.0d0
!     write(*,*) "skipnum = ", skipnug, "XY=", (nx-1)/skipnug, (ny-1)/skipnug

     !-- Interpolation of the initial u^l
     call backward_interpo_2dd( psim, psig, l-1, l )  ! From l-1 to l
     call forward_interpo_2dd( rho, rhog, level, l )  ! From level to l
     if(present(a))then
        call forward_interpo_2dd( a, ag, level, l )
     end if
     if(present(b))then
        call forward_interpo_2dd( b, bg, level, l )
     end if
     if(present(c))then
        call forward_interpo_2dd( c, cg, level, l )
     end if
     if(present(d))then
        call forward_interpo_2dd( d, dg, level, l )
     end if
     if(present(e))then
        call forward_interpo_2dd( e, eg, level, l )
     end if
     if(present(f))then
        call forward_interpo_2dd( f, fg, level, l )
     end if

     !-- Enter the V-cycle from the l grid
     call GauSei_Vcycle_2dd( l, xg, yg, boundary, psig, rhog, psinew, conv_num,  &
  &                          at=ag, bt=bg, ct=cg, dt=dg, et=eg, ft=fg,  &
  &                          accel=accc, helmod_flag=hel_flag(1:l), helmod_dl=hel_dl )
!write(*,*) "temporary stop", psig
!stop

     !-- Reallocate psim from l-1 to l
     deallocate(psim)
     nxm=nxg
     nym=nyg
     allocate(psim(nxm,nym))

     !-- Store psinew to psim
     psim(1:nxm,1:nym)=psinew(1:nxg,1:nyg)

     !-- Deallocate variables
     deallocate(xg)
     deallocate(yg)
     deallocate(rhog)
     deallocate(psig)
     deallocate(psinew)
     deallocate(ag)
     deallocate(bg)
     deallocate(cg)
     deallocate(dg)
     deallocate(eg)
     deallocate(fg)
  end do

  if(nxm/=nx.or.nym/=ny)then
     write(*,*) "error"
     stop
  end if

  if(present(add_itr))then
     allocate(psinew(nx,ny))
     allocate(am(nxm,nym))
     allocate(bm(nxm,nym))
     allocate(cm(nxm,nym))
     allocate(dm(nxm,nym))
     allocate(em(nxm,nym))
     allocate(fm(nxm,nym))
     am=1.0d0
     cm=1.0d0
     bm=0.0d0
     dm=0.0d0
     em=0.0d0
     fm=0.0d0
     if(present(a))then
        am=a
     end if
     if(present(b))then
        bm=b
     end if
     if(present(c))then
        cm=c
     end if
     if(present(d))then
        dm=d
     end if
     if(present(e))then
        em=e
     end if
     if(present(f))then
        fm=f
     end if
     do l=1,add_itr
        call GauSei_Vcycle_2dd( level, x, y, boundary, psim, rho, psinew, conv_num,  &
  &                             at=am, bt=bm, ct=cm, dt=dm, et=em, ft=fm,  &
  &                             accel=accc, helmod_flag=hel_flag(1:level), helmod_dl=hel_dl )
        psim=psinew
     end do
  end if

  !-- 4. Set psi
!  write(*,*) "check final grid", nx, ny, nxm, nym
  psi(1:nx,1:ny)=psim(1:nxm,1:nym)

  if(present(rlu_err))then
     allocate(rlu(nx,ny))
     if(present(add_itr).eqv..false.)then  ! 以下は add_itr が有効なら割付済み
        allocate(psinew(nx,ny))
        allocate(am(nxm,nym))
        allocate(bm(nxm,nym))
        allocate(cm(nxm,nym))
        allocate(dm(nxm,nym))
        allocate(em(nxm,nym))
        allocate(fm(nxm,nym))
        am=1.0d0
        cm=1.0d0
        bm=0.0d0
        dm=0.0d0
        em=0.0d0
        fm=0.0d0
        if(present(a))then
           am=a
        end if
        if(present(b))then
           bm=b
        end if
        if(present(c))then
           cm=c
        end if
        if(present(d))then
           dm=d
        end if
        if(present(e))then
           em=e
        end if
        if(present(f))then
           fm=f
        end if
     end if

     call calc_error_2dd( x, y, psim, rho, rlu, a=am, b=bm, c=cm, d=dm, e=em, f=fm )
     rlu_err=rlu
  end if

end subroutine Full_Multi_Grid_2dd

!----------------------------------
!----------------------------------

subroutine Full_Multi_Grid_3df( level, levelz, x, y, z, rho, eps, boundary, psi,  &
  &                             xa, ya, za, a, b, c, d, e, f, g,  &
  &                             conv_num, add_itr, rlu_err, accel, helmod_flag, helmod_dl )
  ! Perform a V-cycle from the level grid to 1st grid (i.e., coarsest grid)
  implicit none
  integer, intent(in) :: level              ! Top grid level (M)
  integer, intent(in) :: levelz(level)      ! Level for vertical grid level (M)
  real, intent(in) :: x(:)                  ! X grid on the level grid
  real, intent(in) :: y(:)                  ! Y grid on the level grid
  real, intent(in) :: z(:)                  ! Z grid on the level grid
  real, intent(in) :: rho(size(x),size(y),size(z))  ! The forcing on the level grid
  real, intent(in) :: eps        ! Threshold for iteration of GauSei
  character(6), intent(in) :: boundary       ! The boundary conditions
  real, intent(out) :: psi(size(x),size(y),size(z)) ! The updated u on the level grid
  real, intent(in), optional :: xa(size(x),size(y),size(z)) ! coefficient in PDE
  real, intent(in), optional :: ya(size(x),size(y),size(z)) ! coefficient in PDE
  real, intent(in), optional :: za(size(x),size(y),size(z)) ! coefficient in PDE
  real, intent(in), optional :: a(size(x),size(y),size(z))  ! coefficient in PDE
  real, intent(in), optional :: b(size(x),size(y),size(z))  ! coefficient in PDE
  real, intent(in), optional :: c(size(x),size(y),size(z))  ! coefficient in PDE
  real, intent(in), optional :: d(size(x),size(y),size(z))  ! coefficient in PDE
  real, intent(in), optional :: e(size(x),size(y),size(z))  ! coefficient in PDE
  real, intent(in), optional :: f(size(x),size(y),size(z))  ! coefficient in PDE
  real, intent(in), optional :: g(size(x),size(y),size(z))  ! coefficient in PDE
  integer, intent(in) :: conv_num  ! number of the threshold for iteration
  integer, intent(in), optional :: add_itr  ! number of the threshold for additional iteration
  real, intent(out), optional :: rlu_err(size(x),size(y),size(z))  ! the final error
  real, intent(in), optional :: accel  ! Acceleration coefficient for Successive Over Relaxation
                             ! (0 < accel < 2), default = 1
  logical, intent(in), optional :: helmod_flag
                             ! Adjustment of "g" for the stability in the Helmholtz equation on the coarse grids.
                             ! .true.: Adjusting "g" and displaying the message
                             ! .false.: No adjusting "g" and terminating with the error message
                             ! default = .false.
  logical, intent(in), optional :: helmod_dl  ! Debug output for helmod_flag = .true.
                             ! .true.: successive output in adjusting g
                             ! .false.: once output in calling this routine
                             ! default = .false.

  integer :: ii, l, sk, skz, nxu, nyu, nzu, nxd, nyd, nzd, nxl, nyl, nzl
  real :: dx, dy, dz, dxu, dyu, dzu, dxd, dyd, dzd
  real, allocatable, dimension(:) :: xlu, ylu, zlu, xld, yld, zld
  real, allocatable, dimension(:,:,:) :: rlu, rld, ulu, uld, flu, fld

  integer :: nx, ny, nz, nxm, nym, nzm, nxg, nyg, nzg
  integer :: i, j, k
  real :: dxm, dym, dzm, dxg, dyg, dzg
  real :: accc
  real, allocatable, dimension(:) :: xm, ym, zm, xg, yg, zg
  real, allocatable, dimension(:,:,:) :: rhom, psim, rhog, psig, psinew
  real, allocatable, dimension(:,:,:) :: xam, yam, zam, am, bm, cm, dm, em, fm, gm
  real, allocatable, dimension(:,:,:) :: xag, yag, zag, ag, bg, cg, dg, eg, fg, gg
  integer :: skipnum, skipnumz, skipnug, skipnugz
  logical :: ter_flag, hel_flag(level), hel_dl

  nx=size(x)
  ny=size(y)
  nz=size(z)

  dx=x(2)-x(1)
  dy=y(2)-y(1)
  dz=z(2)-z(1)

  hel_flag=.false.
  hel_dl=.false.
  if(present(helmod_flag))then
     !-- the finest grid corresponds to hel_flag(level)
     hel_flag(1:level-1)=helmod_flag
     if(present(helmod_dl))then
        hel_dl=helmod_dl
     end if
  end if

!-- check "accel"

  if(present(accel))then
     accc=accel
  else
     accc=1.0
  end if

  !-- 1. Solving on the coasest grid
  !-- Set and Allocate variables on the coarsest grid
  skipnum=2**(level-1)
  skipnumz=2**(levelz(level)-1)
  nxm=(nx-1)/skipnum+1
  nym=(ny-1)/skipnum+1
  nzm=(nz-1)/skipnumz+1
  dxm=dx*real(skipnum)
  dym=dy*real(skipnum)
  dzm=dz*real(skipnumz)
  allocate(xm(nxm))
  allocate(ym(nym))
  allocate(zm(nzm))
  allocate(rhom(nxm,nym,nzm))
  allocate(psim(nxm,nym,nzm))
  allocate(xam(nxm,nym,nzm))
  allocate(yam(nxm,nym,nzm))
  allocate(zam(nxm,nym,nzm))
  allocate(am(nxm,nym,nzm))
  allocate(bm(nxm,nym,nzm))
  allocate(cm(nxm,nym,nzm))
  allocate(dm(nxm,nym,nzm))
  allocate(em(nxm,nym,nzm))
  allocate(fm(nxm,nym,nzm))
  allocate(gm(nxm,nym,nzm))
  xm=(/(dxm*(i-1),i=1,nxm)/)
  ym=(/(dym*(i-1),i=1,nym)/)
  zm=(/(dzm*(i-1),i=1,nzm)/)
  xam=1.0
  yam=1.0
  zam=1.0
  am=0.0
  bm=0.0
  cm=0.0
  dm=0.0
  em=0.0
  fm=0.0
  gm=0.0
!  write(*,*) "skipnum = ", skipnum, "XY=", (nx-1)/skipnum, (ny-1)/skipnum

  !-- 2. Calculation of the initial u^l before the V-cycle
  call forward_interpo_3df( rho, rhom, level, levelz(1:level), 1 )
  if(present(xa))then
     call forward_interpo_3df( xa, xam, level, levelz(1:level), 1 )
  end if
  if(present(ya))then
     call forward_interpo_3df( ya, yam, level, levelz(1:level), 1 )
  end if
  if(present(za))then
     call forward_interpo_3df( za, zam, level, levelz(1:level), 1 )
  end if
  if(present(a))then
     call forward_interpo_3df( a, am, level, levelz(1:level), 1 )
  end if
  if(present(b))then
     call forward_interpo_3df( b, bm, level, levelz(1:level), 1 )
  end if
  if(present(c))then
     call forward_interpo_3df( c, cm, level, levelz(1:level), 1 )
  end if
  if(present(d))then
     call forward_interpo_3df( d, dm, level, levelz(1:level), 1 )
  end if
  if(present(e))then
     call forward_interpo_3df( e, em, level, levelz(1:level), 1 )
  end if
  if(present(f))then
     call forward_interpo_3df( f, fm, level, levelz(1:level), 1 )
  end if
  if(present(g))then
     call forward_interpo_3df( g, gm, level, levelz(1:level), 1 )
  end if

  call Ellip_GauSei_3df( xm, ym, zm, rhom, eps, boundary, psim,  &
  &                      xa=xam, ya=yam, za=zam, a=am, b=bm, c=cm, d=dm,  &
  &                      e=em, f=fm, g=gm, ln=conv_num,  &
  &                      accel=accc, helmod_flag=hel_flag(1), helmod_dl=hel_dl )
!write(*,*) "temporary stop", psim

  deallocate(xam)
  deallocate(yam)
  deallocate(zam)
  deallocate(am)
  deallocate(bm)
  deallocate(cm)
  deallocate(dm)
  deallocate(em)
  deallocate(fm)
  deallocate(gm)

  !-- 3. Enter cycles
  do l=2,level
     !-- Set and Allocate variables on the l grid
     skipnug=2**(level-l)
     skipnugz=2**(levelz(level)-levelz(l))
     nxg=(nx-1)/skipnug+1
     nyg=(ny-1)/skipnug+1
     nzg=(nz-1)/skipnugz+1
     dxg=dx*real(skipnug)
     dyg=dy*real(skipnug)
     dzg=dz*real(skipnugz)
     allocate(xg(nxg))
     allocate(yg(nyg))
     allocate(zg(nzg))
     allocate(rhog(nxg,nyg,nzg))
     allocate(psig(nxg,nyg,nzg))
     allocate(psinew(nxg,nyg,nzg))
     allocate(xag(nxg,nyg,nzg))
     allocate(yag(nxg,nyg,nzg))
     allocate(zag(nxg,nyg,nzg))
     allocate(ag(nxg,nyg,nzg))
     allocate(bg(nxg,nyg,nzg))
     allocate(cg(nxg,nyg,nzg))
     allocate(dg(nxg,nyg,nzg))
     allocate(eg(nxg,nyg,nzg))
     allocate(fg(nxg,nyg,nzg))
     allocate(gg(nxg,nyg,nzg))
     if(l==level)then
        xg(1:nxg)=x(1:nx)
        yg(1:nyg)=y(1:ny)
        zg(1:nzg)=z(1:nz)
     else
        xg=(/(dxg*(i-1),i=1,nxg)/)
        yg=(/(dyg*(i-1),i=1,nyg)/)
        zg=(/(dzg*(i-1),i=1,nzg)/)
     end if
     xag=1.0
     yag=1.0
     zag=1.0
     ag=0.0
     bg=0.0
     cg=0.0
     dg=0.0
     eg=0.0
     fg=0.0
     gg=0.0
!     write(*,*) "skipnum = ", skipnug, "XY=", (nx-1)/skipnug, (ny-1)/skipnug

     !-- Interpolation of the initial u^l
     call backward_interpo_3df( psim, psig, l-1, levelz(1:level), l )  ! From l-1 to l
     call forward_interpo_3df( rho, rhog, level, levelz(1:level), l )  ! From level to l
     if(present(xa))then
        call forward_interpo_3df( xa, xag, level, levelz(1:level), l )
     end if
     if(present(ya))then
        call forward_interpo_3df( ya, yag, level, levelz(1:level), l )
     end if
     if(present(xa))then
        call forward_interpo_3df( za, zag, level, levelz(1:level), l )
     end if
     if(present(a))then
        call forward_interpo_3df( a, ag, level, levelz(1:level), l )
     end if
     if(present(b))then
        call forward_interpo_3df( b, bg, level, levelz(1:level), l )
     end if
     if(present(c))then
        call forward_interpo_3df( c, cg, level, levelz(1:level), l )
     end if
     if(present(d))then
        call forward_interpo_3df( d, dg, level, levelz(1:level), l )
     end if
     if(present(e))then
        call forward_interpo_3df( e, eg, level, levelz(1:level), l )
     end if
     if(present(f))then
        call forward_interpo_3df( f, fg, level, levelz(1:level), l )
     end if
     if(present(g))then
        call forward_interpo_3df( g, gg, level, levelz(1:level), l )
     end if

     !-- Enter the V-cycle from the l grid
     call GauSei_Vcycle_3df( l, levelz(1:l), xg, yg, zg, boundary, psig,  &
  &                          rhog, psinew, conv_num, xat=xag, yat=yag, zat=zag,  &
  &                          at=ag, bt=bg, ct=cg, dt=dg, et=eg, ft=fg, gt=gg,  &
  &                          accel=accc, helmod_flag=hel_flag(1:l), helmod_dl=hel_dl )
!write(*,*) "temporary stop", psig
!stop

     !-- Reallocate psim from l-1 to l
     deallocate(psim)
     nxm=nxg
     nym=nyg
     nzm=nzg
     allocate(psim(nxm,nym,nzm))

     !-- Store psinew to psim
     psim(1:nxm,1:nym,1:nzm)=psinew(1:nxg,1:nyg,1:nzg)

     !-- Deallocate variables
     deallocate(xg)
     deallocate(yg)
     deallocate(zg)
     deallocate(rhog)
     deallocate(psig)
     deallocate(psinew)
     deallocate(xag)
     deallocate(yag)
     deallocate(zag)
     deallocate(ag)
     deallocate(bg)
     deallocate(cg)
     deallocate(dg)
     deallocate(eg)
     deallocate(fg)
     deallocate(gg)
  end do

  if(nxm/=nx.or.nym/=ny.or.nzm/=nz)then
     write(*,*) "error"
     stop
  end if

  if(present(add_itr))then
     allocate(psinew(nx,ny,nz))
     allocate(xam(nxm,nym,nzm))
     allocate(yam(nxm,nym,nzm))
     allocate(zam(nxm,nym,nzm))
     allocate(am(nxm,nym,nzm))
     allocate(bm(nxm,nym,nzm))
     allocate(cm(nxm,nym,nzm))
     allocate(dm(nxm,nym,nzm))
     allocate(em(nxm,nym,nzm))
     allocate(fm(nxm,nym,nzm))
     allocate(gm(nxm,nym,nzm))
     xam=1.0
     yam=1.0
     zam=1.0
     am=0.0
     bm=0.0
     cm=0.0
     dm=0.0
     em=0.0
     fm=0.0
     gm=0.0
     if(present(xa))then
        xam=xa
     end if
     if(present(ya))then
        yam=ya
     end if
     if(present(za))then
        zam=za
     end if
     if(present(a))then
        am=a
     end if
     if(present(b))then
        bm=b
     end if
     if(present(c))then
        cm=c
     end if
     if(present(d))then
        dm=d
     end if
     if(present(e))then
        em=e
     end if
     if(present(f))then
        fm=f
     end if
     if(present(g))then
        gm=g
     end if
     do l=1,add_itr
        write(*,*) "*** MESSAGE (FMG_3d): Enter additional iteration", l
        call GauSei_Vcycle_3df( level, levelz(1:level), x, y, z, boundary,  &
  &                             psim, rho, psinew, conv_num,  &
  &                             xat=xam, yat=yam, zat=zam,  &
  &                             at=am, bt=bm, ct=cm, dt=dm, et=em, ft=fm, gt=gm,  &
  &                             accel=accc, helmod_flag=hel_flag(1:level), helmod_dl=hel_dl )
        psim=psinew
     end do
  end if

  !-- 4. Set psi
!  write(*,*) "check final grid", nx, ny, nxm, nym
  psi(1:nx,1:ny,1:nz)=psim(1:nxm,1:nym,1:nzm)

  if(present(rlu_err))then
     allocate(rlu(nx,ny,nz))
     if(present(add_itr).eqv..false.)then  ! 以下は add_itr が有効なら割付済み
        allocate(psinew(nx,ny,nz))
        allocate(xam(nxm,nym,nzm))
        allocate(yam(nxm,nym,nzm))
        allocate(zam(nxm,nym,nzm))
        allocate(am(nxm,nym,nzm))
        allocate(bm(nxm,nym,nzm))
        allocate(cm(nxm,nym,nzm))
        allocate(dm(nxm,nym,nzm))
        allocate(em(nxm,nym,nzm))
        allocate(fm(nxm,nym,nzm))
        allocate(gm(nxm,nym,nzm))
        xam=1.0
        yam=1.0
        zam=1.0
        am=0.0
        bm=0.0
        cm=0.0
        dm=0.0
        em=0.0
        fm=0.0
        gm=0.0
        if(present(xa))then
           xam=xa
        end if
        if(present(ya))then
           yam=ya
        end if
        if(present(za))then
           zam=za
        end if
        if(present(a))then
           am=a
        end if
        if(present(b))then
           bm=b
        end if
        if(present(c))then
           cm=c
        end if
        if(present(d))then
           dm=d
        end if
        if(present(e))then
           em=e
        end if
        if(present(f))then
           fm=f
        end if
        if(present(g))then
           gm=g
        end if
     end if

     call calc_error_3df( x, y, z, psim, rho, rlu, xa=xam, ya=yam, za=zam,  &
  &                       a=am, b=bm, c=cm, d=dm, e=em, f=fm, g=gm )
     rlu_err=rlu
  end if

end subroutine Full_Multi_Grid_3df

!----------------------------------
!----------------------------------

subroutine Full_Multi_Grid_3dd( level, levelz, x, y, z, rho, eps, boundary, psi,  &
  &                             xa, ya, za, a, b, c, d, e, f, g,  &
  &                             conv_num, add_itr, rlu_err, accel, helmod_flag, helmod_dl )
  ! Perform a V-cycle from the level grid to 1st grid (i.e., coarsest grid)
  implicit none
  integer, intent(in) :: level              ! Top grid level (M)
  integer, intent(in) :: levelz(level)      ! Level for vertical grid level (M)
  double precision, intent(in) :: x(:)                  ! X grid on the level grid
  double precision, intent(in) :: y(:)                  ! Y grid on the level grid
  double precision, intent(in) :: z(:)                  ! Z grid on the level grid
  double precision, intent(in) :: rho(size(x),size(y),size(z))  ! The forcing on the level grid
  double precision, intent(in) :: eps        ! Threshold for iteration of GauSei
  character(6), intent(in) :: boundary       ! The boundary conditions
  double precision, intent(out) :: psi(size(x),size(y),size(z)) ! The updated u on the level grid
  double precision, intent(in), optional :: xa(size(x),size(y),size(z)) ! coefficient in PDE
  double precision, intent(in), optional :: ya(size(x),size(y),size(z)) ! coefficient in PDE
  double precision, intent(in), optional :: za(size(x),size(y),size(z)) ! coefficient in PDE
  double precision, intent(in), optional :: a(size(x),size(y),size(z))  ! coefficient in PDE
  double precision, intent(in), optional :: b(size(x),size(y),size(z))  ! coefficient in PDE
  double precision, intent(in), optional :: c(size(x),size(y),size(z))  ! coefficient in PDE
  double precision, intent(in), optional :: d(size(x),size(y),size(z))  ! coefficient in PDE
  double precision, intent(in), optional :: e(size(x),size(y),size(z))  ! coefficient in PDE
  double precision, intent(in), optional :: f(size(x),size(y),size(z))  ! coefficient in PDE
  double precision, intent(in), optional :: g(size(x),size(y),size(z))  ! coefficient in PDE
  integer, intent(in) :: conv_num  ! number of the threshold for iteration
  integer, intent(in), optional :: add_itr  ! number of the threshold for additional iteration
  double precision, intent(out), optional :: rlu_err(size(x),size(y),size(z))  ! the final error
  double precision, intent(in), optional :: accel  ! Acceleration coefficient for Successive Over Relaxation
                             ! (0 < accel < 2), default = 1
  logical, intent(in), optional :: helmod_flag
                             ! Adjustment of "g" for the stability in the Helmholtz equation on the coarse grids.
                             ! .true.: Adjusting "g" and displaying the message
                             ! .false.: No adjusting "g" and terminating with the error message
                             ! default = .false.
  logical, intent(in), optional :: helmod_dl  ! Debug output for helmod_flag = .true.
                             ! .true.: successive output in adjusting g
                             ! .false.: once output in calling this routine
                             ! default = .false.

  integer :: ii, l, sk, skz, nxu, nyu, nzu, nxd, nyd, nzd, nxl, nyl, nzl
  double precision :: dx, dy, dz, dxu, dyu, dzu, dxd, dyd, dzd
  double precision, allocatable, dimension(:) :: xlu, ylu, zlu, xld, yld, zld
  double precision, allocatable, dimension(:,:,:) :: rlu, rld, ulu, uld, flu, fld

  integer :: nx, ny, nz, nxm, nym, nzm, nxg, nyg, nzg
  integer :: i, j, k
  double precision :: dxm, dym, dzm, dxg, dyg, dzg
  double precision :: accc
  double precision, allocatable, dimension(:) :: xm, ym, zm, xg, yg, zg
  double precision, allocatable, dimension(:,:,:) :: rhom, psim, rhog, psig, psinew
  double precision, allocatable, dimension(:,:,:) :: xam, yam, zam, am, bm, cm, dm, em, fm, gm
  double precision, allocatable, dimension(:,:,:) :: xag, yag, zag, ag, bg, cg, dg, eg, fg, gg
  integer :: skipnum, skipnumz, skipnug, skipnugz
  logical :: ter_flag, hel_flag(level), hel_dl

  nx=size(x)
  ny=size(y)
  nz=size(z)

  dx=x(2)-x(1)
  dy=y(2)-y(1)
  dz=z(2)-z(1)

  hel_flag=.false.
  hel_dl=.false.
  if(present(helmod_flag))then
     !-- the finest grid corresponds to hel_flag(level)
     hel_flag(1:level-1)=helmod_flag
     if(present(helmod_dl))then
        hel_dl=helmod_dl
     end if
  end if

!-- check "accel"

  if(present(accel))then
     accc=accel
  else
     accc=1.0d0
  end if

  !-- 1. Solving on the coasest grid
  !-- Set and Allocate variables on the coarsest grid
  skipnum=2**(level-1)
  skipnumz=2**(levelz(level)-1)
  nxm=(nx-1)/skipnum+1
  nym=(ny-1)/skipnum+1
  nzm=(nz-1)/skipnumz+1
  dxm=dx*real(skipnum)
  dym=dy*real(skipnum)
  dzm=dz*real(skipnumz)
  allocate(xm(nxm))
  allocate(ym(nym))
  allocate(zm(nzm))
  allocate(rhom(nxm,nym,nzm))
  allocate(psim(nxm,nym,nzm))
  allocate(xam(nxm,nym,nzm))
  allocate(yam(nxm,nym,nzm))
  allocate(zam(nxm,nym,nzm))
  allocate(am(nxm,nym,nzm))
  allocate(bm(nxm,nym,nzm))
  allocate(cm(nxm,nym,nzm))
  allocate(dm(nxm,nym,nzm))
  allocate(em(nxm,nym,nzm))
  allocate(fm(nxm,nym,nzm))
  allocate(gm(nxm,nym,nzm))
  xm=(/(dxm*(i-1),i=1,nxm)/)
  ym=(/(dym*(i-1),i=1,nym)/)
  zm=(/(dzm*(i-1),i=1,nzm)/)
  xam=1.0d0
  yam=1.0d0
  zam=1.0d0
  am=0.0d0
  bm=0.0d0
  cm=0.0d0
  dm=0.0d0
  em=0.0d0
  fm=0.0d0
  gm=0.0d0
!  write(*,*) "skipnum = ", skipnum, "XY=", (nx-1)/skipnum, (ny-1)/skipnum

  !-- 2. Calculation of the initial u^l before the V-cycle
  call forward_interpo_3dd( rho, rhom, level, levelz(1:level), 1 )
  if(present(xa))then
     call forward_interpo_3dd( xa, xam, level, levelz(1:level), 1 )
  end if
  if(present(ya))then
     call forward_interpo_3dd( ya, yam, level, levelz(1:level), 1 )
  end if
  if(present(za))then
     call forward_interpo_3dd( za, zam, level, levelz(1:level), 1 )
  end if
  if(present(a))then
     call forward_interpo_3dd( a, am, level, levelz(1:level), 1 )
  end if
  if(present(b))then
     call forward_interpo_3dd( b, bm, level, levelz(1:level), 1 )
  end if
  if(present(c))then
     call forward_interpo_3dd( c, cm, level, levelz(1:level), 1 )
  end if
  if(present(d))then
     call forward_interpo_3dd( d, dm, level, levelz(1:level), 1 )
  end if
  if(present(e))then
     call forward_interpo_3dd( e, em, level, levelz(1:level), 1 )
  end if
  if(present(f))then
     call forward_interpo_3dd( f, fm, level, levelz(1:level), 1 )
  end if
  if(present(g))then
     call forward_interpo_3dd( g, gm, level, levelz(1:level), 1 )
  end if

  call Ellip_GauSei_3dd( xm, ym, zm, rhom, eps, boundary, psim,  &
  &                      xa=xam, ya=yam, za=zam, a=am, b=bm, c=cm, d=dm,  &
  &                      e=em, f=fm, g=gm, ln=conv_num,  &
  &                      accel=accc, helmod_flag=hel_flag(1), helmod_dl=hel_dl )
!write(*,*) "temporary stop", psim

  deallocate(xam)
  deallocate(yam)
  deallocate(zam)
  deallocate(am)
  deallocate(bm)
  deallocate(cm)
  deallocate(dm)
  deallocate(em)
  deallocate(fm)
  deallocate(gm)

  !-- 3. Enter cycles
  do l=2,level
     !-- Set and Allocate variables on the l grid
     skipnug=2**(level-l)
     skipnugz=2**(levelz(level)-levelz(l))
     nxg=(nx-1)/skipnug+1
     nyg=(ny-1)/skipnug+1
     nzg=(nz-1)/skipnugz+1
     dxg=dx*real(skipnug)
     dyg=dy*real(skipnug)
     dzg=dz*real(skipnugz)
     allocate(xg(nxg))
     allocate(yg(nyg))
     allocate(zg(nzg))
     allocate(rhog(nxg,nyg,nzg))
     allocate(psig(nxg,nyg,nzg))
     allocate(psinew(nxg,nyg,nzg))
     allocate(xag(nxg,nyg,nzg))
     allocate(yag(nxg,nyg,nzg))
     allocate(zag(nxg,nyg,nzg))
     allocate(ag(nxg,nyg,nzg))
     allocate(bg(nxg,nyg,nzg))
     allocate(cg(nxg,nyg,nzg))
     allocate(dg(nxg,nyg,nzg))
     allocate(eg(nxg,nyg,nzg))
     allocate(fg(nxg,nyg,nzg))
     allocate(gg(nxg,nyg,nzg))
     if(l==level)then
        xg(1:nxg)=x(1:nx)
        yg(1:nyg)=y(1:ny)
        zg(1:nzg)=z(1:nz)
     else
        xg=(/(dxg*(i-1),i=1,nxg)/)
        yg=(/(dyg*(i-1),i=1,nyg)/)
        zg=(/(dzg*(i-1),i=1,nzg)/)
     end if
     xag=1.0d0
     yag=1.0d0
     zag=1.0d0
     ag=0.0d0
     bg=0.0d0
     cg=0.0d0
     dg=0.0d0
     eg=0.0d0
     fg=0.0d0
     gg=0.0d0
!     write(*,*) "skipnum = ", skipnug, "XY=", (nx-1)/skipnug, (ny-1)/skipnug

     !-- Interpolation of the initial u^l
     call backward_interpo_3dd( psim, psig, l-1, levelz(1:level), l )  ! From l-1 to l
     call forward_interpo_3dd( rho, rhog, level, levelz(1:level), l )  ! From level to l
     if(present(xa))then
        call forward_interpo_3dd( xa, xag, level, levelz(1:level), l )
     end if
     if(present(ya))then
        call forward_interpo_3dd( ya, yag, level, levelz(1:level), l )
     end if
     if(present(xa))then
        call forward_interpo_3dd( za, zag, level, levelz(1:level), l )
     end if
     if(present(a))then
        call forward_interpo_3dd( a, ag, level, levelz(1:level), l )
     end if
     if(present(b))then
        call forward_interpo_3dd( b, bg, level, levelz(1:level), l )
     end if
     if(present(c))then
        call forward_interpo_3dd( c, cg, level, levelz(1:level), l )
     end if
     if(present(d))then
        call forward_interpo_3dd( d, dg, level, levelz(1:level), l )
     end if
     if(present(e))then
        call forward_interpo_3dd( e, eg, level, levelz(1:level), l )
     end if
     if(present(f))then
        call forward_interpo_3dd( f, fg, level, levelz(1:level), l )
     end if
     if(present(g))then
        call forward_interpo_3dd( g, gg, level, levelz(1:level), l )
     end if

     !-- Enter the V-cycle from the l grid
     call GauSei_Vcycle_3dd( l, levelz(1:l), xg, yg, zg, boundary, psig,  &
  &                          rhog, psinew, conv_num, xat=xag, yat=yag, zat=zag,  &
  &                          at=ag, bt=bg, ct=cg, dt=dg, et=eg, ft=fg, gt=gg,  &
  &                          accel=accc, helmod_flag=hel_flag(1:l), helmod_dl=hel_dl )
!write(*,*) "temporary stop", psig
!stop

     !-- Reallocate psim from l-1 to l
     deallocate(psim)
     nxm=nxg
     nym=nyg
     nzm=nzg
     allocate(psim(nxm,nym,nzm))

     !-- Store psinew to psim
     psim(1:nxm,1:nym,1:nzm)=psinew(1:nxg,1:nyg,1:nzg)

     !-- Deallocate variables
     deallocate(xg)
     deallocate(yg)
     deallocate(zg)
     deallocate(rhog)
     deallocate(psig)
     deallocate(psinew)
     deallocate(xag)
     deallocate(yag)
     deallocate(zag)
     deallocate(ag)
     deallocate(bg)
     deallocate(cg)
     deallocate(dg)
     deallocate(eg)
     deallocate(fg)
     deallocate(gg)
  end do

  if(nxm/=nx.or.nym/=ny.or.nzm/=nz)then
     write(*,*) "error"
     stop
  end if

  if(present(add_itr))then
     allocate(psinew(nx,ny,nz))
     allocate(xam(nxm,nym,nzm))
     allocate(yam(nxm,nym,nzm))
     allocate(zam(nxm,nym,nzm))
     allocate(am(nxm,nym,nzm))
     allocate(bm(nxm,nym,nzm))
     allocate(cm(nxm,nym,nzm))
     allocate(dm(nxm,nym,nzm))
     allocate(em(nxm,nym,nzm))
     allocate(fm(nxm,nym,nzm))
     allocate(gm(nxm,nym,nzm))
     xam=1.0d0
     yam=1.0d0
     zam=1.0d0
     am=0.0d0
     bm=0.0d0
     cm=0.0d0
     dm=0.0d0
     em=0.0d0
     fm=0.0d0
     gm=0.0d0
     if(present(xa))then
        xam=xa
     end if
     if(present(ya))then
        yam=ya
     end if
     if(present(za))then
        zam=za
     end if
     if(present(a))then
        am=a
     end if
     if(present(b))then
        bm=b
     end if
     if(present(c))then
        cm=c
     end if
     if(present(d))then
        dm=d
     end if
     if(present(e))then
        em=e
     end if
     if(present(f))then
        fm=f
     end if
     if(present(g))then
        gm=g
     end if
     do l=1,add_itr
        write(*,*) "*** MESSAGE (FMG_3d): Enter additional iteration", l
        call GauSei_Vcycle_3dd( level, levelz(1:level), x, y, z, boundary,  &
  &                             psim, rho, psinew, conv_num,  &
  &                             xat=xam, yat=yam, zat=zam,  &
  &                             at=am, bt=bm, ct=cm, dt=dm, et=em, ft=fm, gt=gm,  &
  &                             accel=accc, helmod_flag=hel_flag(1:level), helmod_dl=hel_dl )
        psim=psinew
     end do
  end if

  !-- 4. Set psi
!  write(*,*) "check final grid", nx, ny, nxm, nym
  psi(1:nx,1:ny,1:nz)=psim(1:nxm,1:nym,1:nzm)

  if(present(rlu_err))then
     allocate(rlu(nx,ny,nz))
     if(present(add_itr).eqv..false.)then  ! 以下は add_itr が有効なら割付済み
        allocate(psinew(nx,ny,nz))
        allocate(xam(nxm,nym,nzm))
        allocate(yam(nxm,nym,nzm))
        allocate(zam(nxm,nym,nzm))
        allocate(am(nxm,nym,nzm))
        allocate(bm(nxm,nym,nzm))
        allocate(cm(nxm,nym,nzm))
        allocate(dm(nxm,nym,nzm))
        allocate(em(nxm,nym,nzm))
        allocate(fm(nxm,nym,nzm))
        allocate(gm(nxm,nym,nzm))
        xam=1.0d0
        yam=1.0d0
        zam=1.0d0
        am=0.0d0
        bm=0.0d0
        cm=0.0d0
        dm=0.0d0
        em=0.0d0
        fm=0.0d0
        gm=0.0d0
        if(present(xa))then
           xam=xa
        end if
        if(present(ya))then
           yam=ya
        end if
        if(present(za))then
           zam=za
        end if
        if(present(a))then
           am=a
        end if
        if(present(b))then
           bm=b
        end if
        if(present(c))then
           cm=c
        end if
        if(present(d))then
           dm=d
        end if
        if(present(e))then
           em=e
        end if
        if(present(f))then
           fm=f
        end if
        if(present(g))then
           gm=g
        end if
     end if

     call calc_error_3dd( x, y, z, psim, rho, rlu, xa=xam, ya=yam, za=zam,  &
  &                       a=am, b=bm, c=cm, d=dm, e=em, f=fm, g=gm )
     rlu_err=rlu
  end if

end subroutine Full_Multi_Grid_3dd

!----------------------------------
!----------------------------------

subroutine GauSei_Vcycle_2df( level, xl, yl, boundary, ul, fl, ul_new, conv_num,  &
  &                           at, bt, ct, dt, et, ft, rlu_err, accel, helmod_flag, helmod_dl )
  ! Perform a V-cycle from the level grid to 1st grid (i.e., coarsest grid)
  implicit none
  integer, intent(in) :: level               ! Top grid level (M)
  real, intent(in) :: xl(:)                  ! X grid on the level grid
  real, intent(in) :: yl(:)                  ! Y grid on the level grid
  character(4), intent(in) :: boundary       ! Boundary conditions
  real, intent(in) :: ul(size(xl),size(yl))  ! Initial guess for u on the level grid
  real, intent(in) :: fl(size(xl),size(yl))  ! Forcing on the level grid
  real, intent(out) :: ul_new(size(xl),size(yl)) ! The updated u on the level grid
  integer, intent(in) :: conv_num  ! number of the threshold for iteration
  real, intent(in), optional :: at(size(xl),size(yl))  ! coefficient in PDE
  real, intent(in), optional :: bt(size(xl),size(yl))  ! coefficient in PDE
  real, intent(in), optional :: ct(size(xl),size(yl))  ! coefficient in PDE
  real, intent(in), optional :: dt(size(xl),size(yl))  ! coefficient in PDE
  real, intent(in), optional :: et(size(xl),size(yl))  ! coefficient in PDE
  real, intent(in), optional :: ft(size(xl),size(yl))  ! coefficient in PDE
  real, intent(out), optional :: rlu_err(size(xl),size(yl))  ! the final error
  real, intent(in), optional :: accel  ! Acceleration coefficient for Successive Over Relaxation
                             ! (0 < accel < 2), default = 1
  logical, intent(in), optional :: helmod_flag(level)
                             ! Adjustment of "g" for the stability in the Helmholtz equation on the coarse grids.
                             ! .true.: Adjusting "g" and displaying the message
                             ! .false.: No adjusting "g" and terminating with the error message
                             ! default = .false.
  logical, intent(in), optional :: helmod_dl  ! Debug output for helmod_flag = .true.
                             ! .true.: successive output in adjusting g
                             ! .false.: once output in calling this routine
                             ! default = .false.

  integer :: ii, l, sk, nxu, nyu, nxd, nyd, nxl, nyl
  real :: dx, dy, dxu, dyu, dxd, dyd
  real :: accc
  real, allocatable, dimension(:) :: xlu, ylu, xld, yld
  real, allocatable, dimension(:,:) :: rlu, rld, ulu, uld, flu, fld
  real, allocatable, dimension(:,:) :: au, bu, cu, du, eu, fu, ad, bd, cd, dd, ed, fd
  real :: tmpv(size(xl),size(yl),level+1)
  logical :: hel_flag(level), hel_dl

  hel_flag=.false.
  hel_dl=.false.
  if(present(helmod_flag))then
     !-- the finest grid corresponds to hel_flag(level)
     hel_flag(1:level)=helmod_flag(1:level)
     if(present(helmod_dl))then
        hel_dl=helmod_dl
     end if
  end if

!-- check "accel"

  if(present(accel))then
     accc=accel
  else
     accc=1.0
  end if

  !-- Set and Allocate variables
  nxl=size(xl)
  nyl=size(yl)
  dx=xl(2)-xl(1)
  dy=yl(2)-yl(1)
  nxu=nxl
  nyu=nyl
  dxu=dx
  dyu=dy

  tmpv=0.0

  tmpv(1:nxl,1:nyl,level+1)=ul(1:nxl,1:nyl)

  allocate(xlu(nxu))
  allocate(ylu(nyu))
  allocate(ulu(nxu,nyu))
  allocate(flu(nxu,nyu))
  allocate(rlu(nxu,nyu))
  allocate(au(nxu,nyu))
  allocate(bu(nxu,nyu))
  allocate(cu(nxu,nyu))
  allocate(du(nxu,nyu))
  allocate(eu(nxu,nyu))
  allocate(fu(nxu,nyu))

  au=1.0
  cu=1.0
  bu=0.0
  du=0.0
  eu=0.0
  fu=0.0

  if(present(at))then
     au(1:nxu,1:nyu)=at(1:nxu,1:nyu)
  end if
  if(present(bt))then
     bu(1:nxu,1:nyu)=bt(1:nxu,1:nyu)
  end if
  if(present(ct))then
     cu(1:nxu,1:nyu)=ct(1:nxu,1:nyu)
  end if
  if(present(dt))then
     du(1:nxu,1:nyu)=dt(1:nxu,1:nyu)
  end if
  if(present(et))then
     eu(1:nxu,1:nyu)=et(1:nxu,1:nyu)
  end if
  if(present(ft))then
     fu(1:nxu,1:nyu)=ft(1:nxu,1:nyu)
  end if

  !-- Calculation of error on the finest grid
  call calc_error_2df( xl, yl, ul, fl, rlu, a=au, b=bu, c=cu, d=du, e=eu, f=fu )

  !-- Relaxing cycle on level (i.e., the finest grid) (v=(L)^{-1}r)
  call Ellip_GauSei_2df( xl, yl, rlu, 1.0e-8, boundary, ulu,  &
  &                      a=au, b=bu, c=cu, d=du, e=eu, f=fu, ln=conv_num,  &
  &                      accel=accc, helmod_flag=hel_flag(level), helmod_dl=hel_dl )

  xlu(1:nxu)=xl(1:nxu)
  ylu(1:nyu)=yl(1:nyu)
!  ulu(1:nxu,1:nyu)=ul(1:nxu,1:nyu)
  flu(1:nxu,1:nyu)=rlu(1:nxu,1:nyu)
  tmpv(1:nxl,1:nyl,level)=ulu(1:nxl,1:nyl)

  !-- 1. Downward cycle (Relaxing cycle)
  do l=level,2,-1
     !-- Set and Allocate variables for downward (backward)
     sk=2**(level-l+1)
     nxd=(nxl-1)/sk+1
     nyd=(nyl-1)/sk+1
     dxd=dx*real(sk)
     dyd=dy*real(sk)
     allocate(xld(nxd))
     allocate(yld(nyd))
     allocate(uld(nxd,nyd))
     allocate(fld(nxd,nyd))
     allocate(rld(nxd,nyd))
     allocate(ad(nxd,nyd))
     allocate(bd(nxd,nyd))
     allocate(cd(nxd,nyd))
     allocate(dd(nxd,nyd))
     allocate(ed(nxd,nyd))
     allocate(fd(nxd,nyd))
     xld=(/(dxd*(ii-1),ii=1,nxd)/)
     yld=(/(dyd*(ii-1),ii=1,nyd)/)
     ad=1.0
     cd=1.0
     bd=0.0
     dd=0.0
     ed=0.0
     fd=0.0

     !-- Calculation error (r^l=f^l-L^lu^l) on the l grid
     call calc_error_2df( xlu, ylu, ulu, flu, rlu, a=au, b=bu, c=cu, d=du, e=eu, f=fu )
!write(*,*) "check rlu", l, rlu
!write(*,*) "check", nxd, nyd, dxd, dyd, sk!, xld(1), xld(nxd), yld(1), yld(nyd)

     !-- Interpolation rl to rl-1 (rlu -> rld)
!write(*,*) "check", l, size(rlu,1), size(rlu,2), l-1, size(rld,1), size(rld,2)
     call forward_interpo_2df( rlu, rld, l, l-1 )
     if(present(at))then
        call forward_interpo_2df( at, ad, level, l-1 )
     end if
     if(present(bt))then
        call forward_interpo_2df( bt, bd, level, l-1 )
     end if
     if(present(ct))then
        call forward_interpo_2df( ct, cd, level, l-1 )
     end if
     if(present(dt))then
        call forward_interpo_2df( dt, dd, level, l-1 )
     end if
     if(present(et))then
        call forward_interpo_2df( et, ed, level, l-1 )
     end if
     if(present(ft))then
        call forward_interpo_2df( ft, fd, level, l-1 )
     end if

     !-- Relaxing cycle on l-1 (v^{l-1}=(L^{l-1})^{-1}r^{l-1})
     call Ellip_GauSei_2df( xld, yld, rld, 1.0e-8, boundary, uld,  &
  &                         a=ad, b=bd, c=cd, d=dd, e=ed, f=fd, ln=conv_num,  &
  &                         accel=accc, helmod_flag=hel_flag(l-1), helmod_dl=hel_dl )

     !-- Restore v^{l-1}
     tmpv(1:nxd,1:nyd,l-1)=uld(1:nxd,1:nyd)

     !-- Reset *lu variables on the l-1 grid
     nxu=nxd
     nyu=nyd
     dxu=dxd
     dyu=dyd

     !-- Reallocate *lu variables on the l-1 grid
     deallocate(xlu)
     deallocate(ylu)
     deallocate(ulu)
     deallocate(flu)
     deallocate(rlu)
     deallocate(au)
     deallocate(bu)
     deallocate(cu)
     deallocate(du)
     deallocate(eu)
     deallocate(fu)
     allocate(xlu(nxu))
     allocate(ylu(nyu))
     allocate(ulu(nxu,nyu))
     allocate(flu(nxu,nyu))
     allocate(rlu(nxu,nyu))
     allocate(au(nxu,nyu))
     allocate(bu(nxu,nyu))
     allocate(cu(nxu,nyu))
     allocate(du(nxu,nyu))
     allocate(eu(nxu,nyu))
     allocate(fu(nxu,nyu))
     xlu=(/(dxu*(ii-1),ii=1,nxu)/)
     ylu=(/(dyu*(ii-1),ii=1,nyu)/)
     ulu(1:nxu,1:nyu)=uld(1:nxd,1:nyd)
     flu(1:nxu,1:nyu)=rld(1:nxd,1:nyd)
     au(1:nxu,1:nyu)=ad(1:nxd,1:nyd)
     bu(1:nxu,1:nyu)=bd(1:nxd,1:nyd)
     cu(1:nxu,1:nyu)=cd(1:nxd,1:nyd)
     du(1:nxu,1:nyu)=dd(1:nxd,1:nyd)
     eu(1:nxu,1:nyu)=ed(1:nxd,1:nyd)
     fu(1:nxu,1:nyu)=fd(1:nxd,1:nyd)

     !-- Deallocate
     deallocate(xld)
     deallocate(yld)
     deallocate(uld)
     deallocate(fld)
     deallocate(rld)
     deallocate(ad)
     deallocate(bd)
     deallocate(cd)
     deallocate(dd)
     deallocate(ed)
     deallocate(fd)

  end do

  deallocate(ulu)

  !-- 2. Upward cycle (smoothing and interpolating cycle)
  do l=2,level
     !-- Set and Allocate variables for upward (forward)
     sk=2**(level-l+1)
     nxd=(nxl-1)/sk+1
     nyd=(nyl-1)/sk+1
     sk=2**(level-l)
     nxu=(nxl-1)/sk+1
     nyu=(nyl-1)/sk+1
     allocate(uld(nxd,nyd))
     allocate(ulu(nxu,nyu))

     uld(1:nxd,1:nyd)=tmpv(1:nxd,1:nyd,l-1)

     !-- Interpolation vl-1 to vl
     call backward_interpo_2df( uld, ulu, l-1, l )

     !-- Add the vl to the previous vl on l grid
     tmpv(1:nxu,1:nyu,l)=tmpv(1:nxu,1:nyu,l)+ulu(1:nxu,1:nyu)

     !-- Deallocate ul
     deallocate(uld)
     deallocate(ulu)
  end do

  tmpv(1:nxl,1:nyl,level+1)=tmpv(1:nxl,1:nyl,level+1)+tmpv(1:nxl,1:nyl,level)
  ul_new(1:nxl,1:nyl)=tmpv(1:nxl,1:nyl,level+1)

  if(present(rlu_err))then
     deallocate(rlu)
     deallocate(au)
     deallocate(bu)
     deallocate(cu)
     deallocate(du)
     deallocate(eu)
     deallocate(fu)
     allocate(rlu(nxl,nyl))
     allocate(au(nxl,nyl))
     allocate(bu(nxl,nyl))
     allocate(cu(nxl,nyl))
     allocate(du(nxl,nyl))
     allocate(eu(nxl,nyl))
     allocate(fu(nxl,nyl))

     au=1.0
     cu=1.0
     bu=0.0
     du=0.0
     eu=0.0
     fu=0.0

     if(present(at))then
        au=at
     end if
     if(present(bt))then
        bu=bt
     end if
     if(present(ct))then
        cu=ct
     end if
     if(present(dt))then
        du=dt
     end if
     if(present(et))then
        eu=et
     end if
     if(present(ft))then
        fu=ft
     end if

     call calc_error_2df( xl, yl, tmpv(1:nxl,1:nyl,level+1),  &
  &                       fl, rlu, a=au, b=bu, c=cu, d=du, e=eu, f=fu )
     rlu_err=rlu
  end if

end subroutine GauSei_Vcycle_2df

!----------------------------------
!----------------------------------

subroutine GauSei_Vcycle_2dd( level, xl, yl, boundary, ul, fl, ul_new, conv_num,  &
  &                           at, bt, ct, dt, et, ft, rlu_err, accel, helmod_flag, helmod_dl )
  ! Perform a V-cycle from the level grid to 1st grid (i.e., coarsest grid)
  implicit none
  integer, intent(in) :: level               ! Top grid level (M)
  double precision, intent(in) :: xl(:)                  ! X grid on the level grid
  double precision, intent(in) :: yl(:)                  ! Y grid on the level grid
  character(4), intent(in) :: boundary       ! Boundary conditions
  double precision, intent(in) :: ul(size(xl),size(yl))  ! Initial guess for u on the level grid
  double precision, intent(in) :: fl(size(xl),size(yl))  ! Forcing on the level grid
  double precision, intent(out) :: ul_new(size(xl),size(yl)) ! The updated u on the level grid
  integer, intent(in) :: conv_num  ! number of the threshold for iteration
  double precision, intent(in), optional :: at(size(xl),size(yl))  ! coefficient in PDE
  double precision, intent(in), optional :: bt(size(xl),size(yl))  ! coefficient in PDE
  double precision, intent(in), optional :: ct(size(xl),size(yl))  ! coefficient in PDE
  double precision, intent(in), optional :: dt(size(xl),size(yl))  ! coefficient in PDE
  double precision, intent(in), optional :: et(size(xl),size(yl))  ! coefficient in PDE
  double precision, intent(in), optional :: ft(size(xl),size(yl))  ! coefficient in PDE
  double precision, intent(out), optional :: rlu_err(size(xl),size(yl))  ! the final error
  double precision, intent(in), optional :: accel  ! Acceleration coefficient for Successive Over Relaxation
                             ! (0 < accel < 2), default = 1
  logical, intent(in), optional :: helmod_flag(level)
                             ! Adjustment of "g" for the stability in the Helmholtz equation on the coarse grids.
                             ! .true.: Adjusting "g" and displaying the message
                             ! .false.: No adjusting "g" and terminating with the error message
                             ! default = .false.
  logical, intent(in), optional :: helmod_dl  ! Debug output for helmod_flag = .true.
                             ! .true.: successive output in adjusting g
                             ! .false.: once output in calling this routine
                             ! default = .false.

  integer :: ii, l, sk, nxu, nyu, nxd, nyd, nxl, nyl
  double precision :: dx, dy, dxu, dyu, dxd, dyd
  double precision :: accc
  double precision, allocatable, dimension(:) :: xlu, ylu, xld, yld
  double precision, allocatable, dimension(:,:) :: rlu, rld, ulu, uld, flu, fld
  double precision, allocatable, dimension(:,:) :: au, bu, cu, du, eu, fu, ad, bd, cd, dd, ed, fd
  double precision :: tmpv(size(xl),size(yl),level+1)
  logical :: hel_flag(level), hel_dl

  hel_flag=.false.
  hel_dl=.false.
  if(present(helmod_flag))then
     !-- the finest grid corresponds to hel_flag(level)
     hel_flag(1:level)=helmod_flag(1:level)
     if(present(helmod_dl))then
        hel_dl=helmod_dl
     end if
  end if

!-- check "accel"

  if(present(accel))then
     accc=accel
  else
     accc=1.0d0
  end if

  !-- Set and Allocate variables
  nxl=size(xl)
  nyl=size(yl)
  dx=xl(2)-xl(1)
  dy=yl(2)-yl(1)
  nxu=nxl
  nyu=nyl
  dxu=dx
  dyu=dy

  tmpv=0.0d0

  tmpv(1:nxl,1:nyl,level+1)=ul(1:nxl,1:nyl)

  allocate(xlu(nxu))
  allocate(ylu(nyu))
  allocate(ulu(nxu,nyu))
  allocate(flu(nxu,nyu))
  allocate(rlu(nxu,nyu))
  allocate(au(nxu,nyu))
  allocate(bu(nxu,nyu))
  allocate(cu(nxu,nyu))
  allocate(du(nxu,nyu))
  allocate(eu(nxu,nyu))
  allocate(fu(nxu,nyu))

  au=1.0d0
  cu=1.0d0
  bu=0.0d0
  du=0.0d0
  eu=0.0d0
  fu=0.0d0

  if(present(at))then
     au(1:nxu,1:nyu)=at(1:nxu,1:nyu)
  end if
  if(present(bt))then
     bu(1:nxu,1:nyu)=bt(1:nxu,1:nyu)
  end if
  if(present(ct))then
     cu(1:nxu,1:nyu)=ct(1:nxu,1:nyu)
  end if
  if(present(dt))then
     du(1:nxu,1:nyu)=dt(1:nxu,1:nyu)
  end if
  if(present(et))then
     eu(1:nxu,1:nyu)=et(1:nxu,1:nyu)
  end if
  if(present(ft))then
     fu(1:nxu,1:nyu)=ft(1:nxu,1:nyu)
  end if

  !-- Calculation of error on the finest grid
  call calc_error_2dd( xl, yl, ul, fl, rlu, a=au, b=bu, c=cu, d=du, e=eu, f=fu )

  !-- Relaxing cycle on level (i.e., the finest grid) (v=(L)^{-1}r)
  call Ellip_GauSei_2dd( xl, yl, rlu, 1.0d-8, boundary, ulu,  &
  &                      a=au, b=bu, c=cu, d=du, e=eu, f=fu, ln=conv_num,  &
  &                      accel=accc, helmod_flag=hel_flag(level), helmod_dl=hel_dl )

  xlu(1:nxu)=xl(1:nxu)
  ylu(1:nyu)=yl(1:nyu)
!  ulu(1:nxu,1:nyu)=ul(1:nxu,1:nyu)
  flu(1:nxu,1:nyu)=rlu(1:nxu,1:nyu)
  tmpv(1:nxl,1:nyl,level)=ulu(1:nxl,1:nyl)

  !-- 1. Downward cycle (Relaxing cycle)
  do l=level,2,-1
     !-- Set and Allocate variables for downward (backward)
     sk=2**(level-l+1)
     nxd=(nxl-1)/sk+1
     nyd=(nyl-1)/sk+1
     dxd=dx*real(sk)
     dyd=dy*real(sk)
     allocate(xld(nxd))
     allocate(yld(nyd))
     allocate(uld(nxd,nyd))
     allocate(fld(nxd,nyd))
     allocate(rld(nxd,nyd))
     allocate(ad(nxd,nyd))
     allocate(bd(nxd,nyd))
     allocate(cd(nxd,nyd))
     allocate(dd(nxd,nyd))
     allocate(ed(nxd,nyd))
     allocate(fd(nxd,nyd))
     xld=(/(dxd*(ii-1),ii=1,nxd)/)
     yld=(/(dyd*(ii-1),ii=1,nyd)/)
     ad=1.0d0
     cd=1.0d0
     bd=0.0d0
     dd=0.0d0
     ed=0.0d0
     fd=0.0d0

     !-- Calculation error (r^l=f^l-L^lu^l) on the l grid
     call calc_error_2dd( xlu, ylu, ulu, flu, rlu, a=au, b=bu, c=cu, d=du, e=eu, f=fu )
!write(*,*) "check rlu", l, rlu
!write(*,*) "check", nxd, nyd, dxd, dyd, sk!, xld(1), xld(nxd), yld(1), yld(nyd)

     !-- Interpolation rl to rl-1 (rlu -> rld)
!write(*,*) "check", l, size(rlu,1), size(rlu,2), l-1, size(rld,1), size(rld,2)
     call forward_interpo_2dd( rlu, rld, l, l-1 )
     if(present(at))then
        call forward_interpo_2dd( at, ad, level, l-1 )
     end if
     if(present(bt))then
        call forward_interpo_2dd( bt, bd, level, l-1 )
     end if
     if(present(ct))then
        call forward_interpo_2dd( ct, cd, level, l-1 )
     end if
     if(present(dt))then
        call forward_interpo_2dd( dt, dd, level, l-1 )
     end if
     if(present(et))then
        call forward_interpo_2dd( et, ed, level, l-1 )
     end if
     if(present(ft))then
        call forward_interpo_2dd( ft, fd, level, l-1 )
     end if

     !-- Relaxing cycle on l-1 (v^{l-1}=(L^{l-1})^{-1}r^{l-1})
     call Ellip_GauSei_2dd( xld, yld, rld, 1.0d-8, boundary, uld,  &
  &                         a=ad, b=bd, c=cd, d=dd, e=ed, f=fd, ln=conv_num,  &
  &                         accel=accc, helmod_flag=hel_flag(l-1), helmod_dl=hel_dl )

     !-- Restore v^{l-1}
     tmpv(1:nxd,1:nyd,l-1)=uld(1:nxd,1:nyd)

     !-- Reset *lu variables on the l-1 grid
     nxu=nxd
     nyu=nyd
     dxu=dxd
     dyu=dyd

     !-- Reallocate *lu variables on the l-1 grid
     deallocate(xlu)
     deallocate(ylu)
     deallocate(ulu)
     deallocate(flu)
     deallocate(rlu)
     deallocate(au)
     deallocate(bu)
     deallocate(cu)
     deallocate(du)
     deallocate(eu)
     deallocate(fu)
     allocate(xlu(nxu))
     allocate(ylu(nyu))
     allocate(ulu(nxu,nyu))
     allocate(flu(nxu,nyu))
     allocate(rlu(nxu,nyu))
     allocate(au(nxu,nyu))
     allocate(bu(nxu,nyu))
     allocate(cu(nxu,nyu))
     allocate(du(nxu,nyu))
     allocate(eu(nxu,nyu))
     allocate(fu(nxu,nyu))
     xlu=(/(dxu*(ii-1),ii=1,nxu)/)
     ylu=(/(dyu*(ii-1),ii=1,nyu)/)
     ulu(1:nxu,1:nyu)=uld(1:nxd,1:nyd)
     flu(1:nxu,1:nyu)=rld(1:nxd,1:nyd)
     au(1:nxu,1:nyu)=ad(1:nxd,1:nyd)
     bu(1:nxu,1:nyu)=bd(1:nxd,1:nyd)
     cu(1:nxu,1:nyu)=cd(1:nxd,1:nyd)
     du(1:nxu,1:nyu)=dd(1:nxd,1:nyd)
     eu(1:nxu,1:nyu)=ed(1:nxd,1:nyd)
     fu(1:nxu,1:nyu)=fd(1:nxd,1:nyd)

     !-- Deallocate
     deallocate(xld)
     deallocate(yld)
     deallocate(uld)
     deallocate(fld)
     deallocate(rld)
     deallocate(ad)
     deallocate(bd)
     deallocate(cd)
     deallocate(dd)
     deallocate(ed)
     deallocate(fd)

  end do

  deallocate(ulu)

  !-- 2. Upward cycle (smoothing and interpolating cycle)
  do l=2,level
     !-- Set and Allocate variables for upward (forward)
     sk=2**(level-l+1)
     nxd=(nxl-1)/sk+1
     nyd=(nyl-1)/sk+1
     sk=2**(level-l)
     nxu=(nxl-1)/sk+1
     nyu=(nyl-1)/sk+1
     allocate(uld(nxd,nyd))
     allocate(ulu(nxu,nyu))

     uld(1:nxd,1:nyd)=tmpv(1:nxd,1:nyd,l-1)

     !-- Interpolation vl-1 to vl
     call backward_interpo_2dd( uld, ulu, l-1, l )

     !-- Add the vl to the previous vl on l grid
     tmpv(1:nxu,1:nyu,l)=tmpv(1:nxu,1:nyu,l)+ulu(1:nxu,1:nyu)

     !-- Deallocate ul
     deallocate(uld)
     deallocate(ulu)
  end do

  tmpv(1:nxl,1:nyl,level+1)=tmpv(1:nxl,1:nyl,level+1)+tmpv(1:nxl,1:nyl,level)
  ul_new(1:nxl,1:nyl)=tmpv(1:nxl,1:nyl,level+1)

  if(present(rlu_err))then
     deallocate(rlu)
     deallocate(au)
     deallocate(bu)
     deallocate(cu)
     deallocate(du)
     deallocate(eu)
     deallocate(fu)
     allocate(rlu(nxl,nyl))
     allocate(au(nxl,nyl))
     allocate(bu(nxl,nyl))
     allocate(cu(nxl,nyl))
     allocate(du(nxl,nyl))
     allocate(eu(nxl,nyl))
     allocate(fu(nxl,nyl))

     au=1.0d0
     cu=1.0d0
     bu=0.0d0
     du=0.0d0
     eu=0.0d0
     fu=0.0d0

     if(present(at))then
        au=at
     end if
     if(present(bt))then
        bu=bt
     end if
     if(present(ct))then
        cu=ct
     end if
     if(present(dt))then
        du=dt
     end if
     if(present(et))then
        eu=et
     end if
     if(present(ft))then
        fu=ft
     end if

     call calc_error_2dd( xl, yl, tmpv(1:nxl,1:nyl,level+1),  &
  &                       fl, rlu, a=au, b=bu, c=cu, d=du, e=eu, f=fu )
     rlu_err=rlu
  end if

end subroutine GauSei_Vcycle_2dd

!----------------------------------
!----------------------------------

subroutine GauSei_Vcycle_3df( level, levelz, xl, yl, zl, boundary,  &
  &                           ul, fl, ul_new, conv_num,  &
  &                           xat, yat, zat, at, bt, ct, dt, et, ft, gt,  &
  &                           rlu_err, accel, helmod_flag, helmod_dl )
  ! Perform a V-cycle from the level grid to 1st grid (i.e., coarsest grid)
  implicit none
  integer, intent(in) :: level               ! Top grid level (M)
  integer, intent(in) :: levelz(level)       ! Vertical grid level (M)
  real, intent(in) :: xl(:)                  ! X grid on the level grid
  real, intent(in) :: yl(:)                  ! Y grid on the level grid
  real, intent(in) :: zl(:)                  ! Z grid on the level grid
  character(6), intent(in) :: boundary       ! Boundary conditions
  real, intent(in) :: ul(size(xl),size(yl),size(zl))  ! Initial guess for u on the level grid
  real, intent(in) :: fl(size(xl),size(yl),size(zl))  ! Forcing on the level grid
  real, intent(out) :: ul_new(size(xl),size(yl),size(zl)) ! The updated u on the level grid
  integer, intent(in) :: conv_num  ! number of the threshold for iteration
  real, intent(in), optional :: xat(size(xl),size(yl),size(zl))  ! coefficient in PDE
  real, intent(in), optional :: yat(size(xl),size(yl),size(zl))  ! coefficient in PDE
  real, intent(in), optional :: zat(size(xl),size(yl),size(zl))  ! coefficient in PDE
  real, intent(in), optional :: at(size(xl),size(yl),size(zl))  ! coefficient in PDE
  real, intent(in), optional :: bt(size(xl),size(yl),size(zl))  ! coefficient in PDE
  real, intent(in), optional :: ct(size(xl),size(yl),size(zl))  ! coefficient in PDE
  real, intent(in), optional :: dt(size(xl),size(yl),size(zl))  ! coefficient in PDE
  real, intent(in), optional :: et(size(xl),size(yl),size(zl))  ! coefficient in PDE
  real, intent(in), optional :: ft(size(xl),size(yl),size(zl))  ! coefficient in PDE
  real, intent(in), optional :: gt(size(xl),size(yl),size(zl))  ! coefficient in PDE
  real, intent(out), optional :: rlu_err(size(xl),size(yl),size(zl))  ! the final error
  real, intent(in), optional :: accel  ! Acceleration coefficient for Successive Over Relaxation
                             ! (0 < accel < 2), default = 1
  logical, intent(in), optional :: helmod_flag(level)
                             ! Adjustment of "g" for the stability in the Helmholtz equation on the coarse grids.
                             ! .true.: Adjusting "g" and displaying the message
                             ! .false.: No adjusting "g" and terminating with the error message
                             ! default = .false.
  logical, intent(in), optional :: helmod_dl  ! Debug output for helmod_flag = .true.
                             ! .true.: successive output in adjusting g
                             ! .false.: once output in calling this routine
                             ! default = .false.

  integer :: ii, l, sk, skz, nxu, nyu, nzu, nxd, nyd, nzd, nxl, nyl, nzl
  real :: dx, dy, dz, dxu, dyu, dzu, dxd, dyd, dzd
  real :: accc
  real, allocatable, dimension(:) :: xlu, ylu, zlu, xld, yld, zld
  real, allocatable, dimension(:,:,:) :: rlu, rld, ulu, uld, flu, fld
  real, allocatable, dimension(:,:,:) :: xau, yau, zau, au, bu, cu, du, eu, fu, gu
  real, allocatable, dimension(:,:,:) :: xad, yad, zad, ad, bd, cd, dd, ed, fd, gd
  real :: tmpv(size(xl),size(yl),size(zl),level+1)
  logical :: hel_flag(level), hel_dl

  hel_flag=.false.
  hel_dl=.false.
  if(present(helmod_flag))then
     !-- the finest grid corresponds to hel_flag(level)
     hel_flag(1:level)=helmod_flag(1:level)
     if(present(helmod_dl))then
        hel_dl=helmod_dl
     end if
  end if

!-- check "accel"

  if(present(accel))then
     accc=accel
  else
     accc=1.0
  end if

  !-- Set and Allocate variables
  nxl=size(xl)
  nyl=size(yl)
  nzl=size(zl)
  dx=xl(2)-xl(1)
  dy=yl(2)-yl(1)
  dz=zl(2)-zl(1)
  nxu=nxl
  nyu=nyl
  nzu=nzl
  dxu=dx
  dyu=dy
  dzu=dz

  tmpv=0.0

  tmpv(1:nxl,1:nyl,1:nzl,level+1)=ul(1:nxl,1:nyl,1:nzl)

  allocate(xlu(nxu))
  allocate(ylu(nyu))
  allocate(zlu(nzu))
  allocate(ulu(nxu,nyu,nzu))
  allocate(flu(nxu,nyu,nzu))
  allocate(rlu(nxu,nyu,nzu))
  allocate(xau(nxu,nyu,nzu))
  allocate(yau(nxu,nyu,nzu))
  allocate(zau(nxu,nyu,nzu))
  allocate(au(nxu,nyu,nzu))
  allocate(bu(nxu,nyu,nzu))
  allocate(cu(nxu,nyu,nzu))
  allocate(du(nxu,nyu,nzu))
  allocate(eu(nxu,nyu,nzu))
  allocate(fu(nxu,nyu,nzu))
  allocate(gu(nxu,nyu,nzu))

  xau=1.0
  yau=1.0
  zau=1.0
  au=0.0
  bu=0.0
  cu=0.0
  du=0.0
  eu=0.0
  fu=0.0
  gu=0.0

  if(present(xat))then
     xau(1:nxu,1:nyu,1:nzu)=xat(1:nxu,1:nyu,1:nzu)
  end if
  if(present(yat))then
     yau(1:nxu,1:nyu,1:nzu)=yat(1:nxu,1:nyu,1:nzu)
  end if
  if(present(zat))then
     zau(1:nxu,1:nyu,1:nzu)=zat(1:nxu,1:nyu,1:nzu)
  end if
  if(present(at))then
     au(1:nxu,1:nyu,1:nzu)=at(1:nxu,1:nyu,1:nzu)
  end if
  if(present(bt))then
     bu(1:nxu,1:nyu,1:nzu)=bt(1:nxu,1:nyu,1:nzu)
  end if
  if(present(ct))then
     cu(1:nxu,1:nyu,1:nzu)=ct(1:nxu,1:nyu,1:nzu)
  end if
  if(present(dt))then
     du(1:nxu,1:nyu,1:nzu)=dt(1:nxu,1:nyu,1:nzu)
  end if
  if(present(et))then
     eu(1:nxu,1:nyu,1:nzu)=et(1:nxu,1:nyu,1:nzu)
  end if
  if(present(ft))then
     fu(1:nxu,1:nyu,1:nzu)=ft(1:nxu,1:nyu,1:nzu)
  end if
  if(present(gt))then
     gu(1:nxu,1:nyu,1:nzu)=gt(1:nxu,1:nyu,1:nzu)
  end if

  !-- Calculation of error on the finest grid
  call calc_error_3df( xl, yl, zl, ul, fl, rlu, xa=xau, ya=yau, za=zau,  &
  &                    a=au, b=bu, c=cu, d=du, e=eu, f=fu, g=gu )

  !-- Relaxing cycle on level (i.e., the finest grid) (v=(L)^{-1}r)
  call Ellip_GauSei_3df( xl, yl, zl, rlu, 1.0e-8, boundary, ulu,  &
  &                      xa=xau, ya=yau, za=zau,  &
  &                      a=au, b=bu, c=cu, d=du, e=eu, f=fu, g=gu, ln=conv_num,  &
  &                      accel=accc, helmod_flag=hel_flag(level), helmod_dl=hel_dl )

  xlu(1:nxu)=xl(1:nxu)
  ylu(1:nyu)=yl(1:nyu)
  zlu(1:nzu)=zl(1:nzu)
!  ulu(1:nxu,1:nyu,1:nzu)=ul(1:nxu,1:nyu,1:nzu)
  flu(1:nxu,1:nyu,1:nzu)=rlu(1:nxu,1:nyu,1:nzu)
  tmpv(1:nxl,1:nyl,1:nzl,level)=ulu(1:nxl,1:nyl,1:nzl)

  !-- 1. Downward cycle (Relaxing cycle)
  do l=level,2,-1
     !-- Set and Allocate variables for downward (backward)
     sk=2**(level-l+1)
     skz=2**(levelz(level)-levelz(l-1))
     nxd=(nxl-1)/sk+1
     nyd=(nyl-1)/sk+1
     nzd=(nzl-1)/skz+1
     dxd=dx*real(sk)
     dyd=dy*real(sk)
     dzd=dz*real(skz)
     allocate(xld(nxd))
     allocate(yld(nyd))
     allocate(zld(nzd))
     allocate(uld(nxd,nyd,nzd))
     allocate(fld(nxd,nyd,nzd))
     allocate(rld(nxd,nyd,nzd))
     allocate(xad(nxd,nyd,nzd))
     allocate(yad(nxd,nyd,nzd))
     allocate(zad(nxd,nyd,nzd))
     allocate(ad(nxd,nyd,nzd))
     allocate(bd(nxd,nyd,nzd))
     allocate(cd(nxd,nyd,nzd))
     allocate(dd(nxd,nyd,nzd))
     allocate(ed(nxd,nyd,nzd))
     allocate(fd(nxd,nyd,nzd))
     allocate(gd(nxd,nyd,nzd))
     xld=(/(dxd*(ii-1),ii=1,nxd)/)
     yld=(/(dyd*(ii-1),ii=1,nyd)/)
     zld=(/(dzd*(ii-1),ii=1,nzd)/)
     xad=1.0
     yad=1.0
     zad=1.0
     ad=0.0
     bd=0.0
     cd=0.0
     dd=0.0
     ed=0.0
     fd=0.0
     gd=0.0

     !-- Calculation error (r^l=f^l-L^lu^l) on the l grid
     call calc_error_3df( xlu, ylu, zlu, ulu, flu, rlu,  &
     &                    xa=xau, ya=yau, za=zau,  &
     &                    a=au, b=bu, c=cu, d=du, e=eu, f=fu, g=gu )
!write(*,*) "check rlu", l, rlu
!write(*,*) "check", nxd, nyd, dxd, dyd, sk!, xld(1), xld(nxd), yld(1), yld(nyd)

     !-- Interpolation rl to rl-1 (rlu -> rld)
!write(*,*) "check", l, size(rlu,1), size(rlu,2), l-1, size(rld,1), size(rld,2)
     call forward_interpo_3df( rlu, rld, l, levelz(1:level), l-1 )
     if(present(xat))then
        call forward_interpo_3df( xat, xad, level, levelz(1:level), l-1 )
     end if
     if(present(yat))then
        call forward_interpo_3df( yat, yad, level, levelz(1:level), l-1 )
     end if
     if(present(zat))then
        call forward_interpo_3df( zat, zad, level, levelz(1:level), l-1 )
     end if
     if(present(at))then
        call forward_interpo_3df( at, ad, level, levelz(1:level), l-1 )
     end if
     if(present(bt))then
        call forward_interpo_3df( bt, bd, level, levelz(1:level), l-1 )
     end if
     if(present(ct))then
        call forward_interpo_3df( ct, cd, level, levelz(1:level), l-1 )
     end if
     if(present(dt))then
        call forward_interpo_3df( dt, dd, level, levelz(1:level), l-1 )
     end if
     if(present(et))then
        call forward_interpo_3df( et, ed, level, levelz(1:level), l-1 )
     end if
     if(present(ft))then
        call forward_interpo_3df( ft, fd, level, levelz(1:level), l-1 )
     end if
     if(present(gt))then
        call forward_interpo_3df( gt, gd, level, levelz(1:level), l-1 )
     end if

     !-- Relaxing cycle on l-1 (v^{l-1}=(L^{l-1})^{-1}r^{l-1})
     call Ellip_GauSei_3df( xld, yld, zld, rld, 1.0e-8, boundary, uld,  &
  &                         xa=xad, ya=yad, za=zad,  &
  &                         a=ad, b=bd, c=cd, d=dd, e=ed, f=fd, g=gd, ln=conv_num,  &
  &                         accel=accc, helmod_flag=hel_flag(l-1), helmod_dl=hel_dl )

     !-- Restore v^{l-1}
     tmpv(1:nxd,1:nyd,1:nzd,l-1)=uld(1:nxd,1:nyd,1:nzd)

     !-- Reset *lu variables on the l-1 grid
     nxu=nxd
     nyu=nyd
     nzu=nzd
     dxu=dxd
     dyu=dyd
     dzu=dzd

     !-- Reallocate *lu variables on the l-1 grid
     deallocate(xlu)
     deallocate(ylu)
     deallocate(zlu)
     deallocate(ulu)
     deallocate(flu)
     deallocate(rlu)
     deallocate(xau)
     deallocate(yau)
     deallocate(zau)
     deallocate(au)
     deallocate(bu)
     deallocate(cu)
     deallocate(du)
     deallocate(eu)
     deallocate(fu)
     deallocate(gu)
     allocate(xlu(nxu))
     allocate(ylu(nyu))
     allocate(ulu(nxu,nyu,nzu))
     allocate(flu(nxu,nyu,nzu))
     allocate(rlu(nxu,nyu,nzu))
     allocate(xau(nxu,nyu,nzu))
     allocate(yau(nxu,nyu,nzu))
     allocate(zau(nxu,nyu,nzu))
     allocate(au(nxu,nyu,nzu))
     allocate(bu(nxu,nyu,nzu))
     allocate(cu(nxu,nyu,nzu))
     allocate(du(nxu,nyu,nzu))
     allocate(eu(nxu,nyu,nzu))
     allocate(fu(nxu,nyu,nzu))
     allocate(gu(nxu,nyu,nzu))
     xlu=(/(dxu*(ii-1),ii=1,nxu)/)
     ylu=(/(dyu*(ii-1),ii=1,nyu)/)
     zlu=(/(dzu*(ii-1),ii=1,nzu)/)
     ulu(1:nxu,1:nyu,1:nzu)=uld(1:nxd,1:nyd,1:nzd)
     flu(1:nxu,1:nyu,1:nzu)=rld(1:nxd,1:nyd,1:nzd)
     xau(1:nxu,1:nyu,1:nzu)=xad(1:nxd,1:nyd,1:nzd)
     yau(1:nxu,1:nyu,1:nzu)=yad(1:nxd,1:nyd,1:nzd)
     zau(1:nxu,1:nyu,1:nzu)=zad(1:nxd,1:nyd,1:nzd)
     au(1:nxu,1:nyu,1:nzu)=ad(1:nxd,1:nyd,1:nzd)
     bu(1:nxu,1:nyu,1:nzu)=bd(1:nxd,1:nyd,1:nzd)
     cu(1:nxu,1:nyu,1:nzu)=cd(1:nxd,1:nyd,1:nzd)
     du(1:nxu,1:nyu,1:nzu)=dd(1:nxd,1:nyd,1:nzd)
     eu(1:nxu,1:nyu,1:nzu)=ed(1:nxd,1:nyd,1:nzd)
     fu(1:nxu,1:nyu,1:nzu)=fd(1:nxd,1:nyd,1:nzd)
     gu(1:nxu,1:nyu,1:nzu)=gd(1:nxd,1:nyd,1:nzd)

     !-- Deallocate
     deallocate(xld)
     deallocate(yld)
     deallocate(zld)
     deallocate(uld)
     deallocate(fld)
     deallocate(rld)
     deallocate(xad)
     deallocate(yad)
     deallocate(zad)
     deallocate(ad)
     deallocate(bd)
     deallocate(cd)
     deallocate(dd)
     deallocate(ed)
     deallocate(fd)
     deallocate(gd)

  end do

  deallocate(ulu)

  !-- 2. Upward cycle (smoothing and interpolating cycle)
  do l=2,level
     !-- Set and Allocate variables for upward (forward)
     sk=2**(level-l+1)
     skz=2**(levelz(level)-levelz(l-1))
     nxd=(nxl-1)/sk+1
     nyd=(nyl-1)/sk+1
     nzd=(nzl-1)/skz+1
     sk=2**(level-l)
     skz=2**(levelz(level)-levelz(l))
     nxu=(nxl-1)/sk+1
     nyu=(nyl-1)/sk+1
     nzu=(nzl-1)/skz+1
     allocate(uld(nxd,nyd,nzd))
     allocate(ulu(nxu,nyu,nzu))

     uld(1:nxd,1:nyd,1:nzd)=tmpv(1:nxd,1:nyd,1:nzd,l-1)

     !-- Interpolation vl-1 to vl
     call backward_interpo_3df( uld, ulu, l-1, levelz(1:level), l )

     !-- Add the vl to the previous vl on l grid
     tmpv(1:nxu,1:nyu,1:nzu,l)=tmpv(1:nxu,1:nyu,1:nzu,l)+ulu(1:nxu,1:nyu,1:nzu)

     !-- Deallocate ul
     deallocate(uld)
     deallocate(ulu)
  end do

  tmpv(1:nxl,1:nyl,1:nzl,level+1)=tmpv(1:nxl,1:nyl,1:nzl,level+1)+tmpv(1:nxl,1:nyl,1:nzl,level)
  ul_new(1:nxl,1:nyl,1:nzl)=tmpv(1:nxl,1:nyl,1:nzl,level+1)

  if(present(rlu_err))then
     deallocate(rlu)
     deallocate(xau)
     deallocate(yau)
     deallocate(zau)
     deallocate(au)
     deallocate(bu)
     deallocate(cu)
     deallocate(du)
     deallocate(eu)
     deallocate(fu)
     deallocate(gu)
     allocate(rlu(nxl,nyl,nzl))
     allocate(xau(nxl,nyl,nzl))
     allocate(yau(nxl,nyl,nzl))
     allocate(zau(nxl,nyl,nzl))
     allocate(au(nxl,nyl,nzl))
     allocate(bu(nxl,nyl,nzl))
     allocate(cu(nxl,nyl,nzl))
     allocate(du(nxl,nyl,nzl))
     allocate(eu(nxl,nyl,nzl))
     allocate(fu(nxl,nyl,nzl))
     allocate(gu(nxl,nyl,nzl))

     xau=1.0
     yau=1.0
     zau=1.0
     au=0.0
     cu=0.0
     bu=0.0
     du=0.0
     eu=0.0
     fu=0.0
     gu=0.0

     if(present(xat))then
        xau=xat
     end if
     if(present(yat))then
        yau=yat
     end if
     if(present(zat))then
        zau=zat
     end if
     if(present(at))then
        au=at
     end if
     if(present(bt))then
        bu=bt
     end if
     if(present(ct))then
        cu=ct
     end if
     if(present(dt))then
        du=dt
     end if
     if(present(et))then
        eu=et
     end if
     if(present(ft))then
        fu=ft
     end if
     if(present(gt))then
        gu=gt
     end if

     call calc_error_3df( xl, yl, zl, tmpv(1:nxl,1:nyl,1:nzl,level+1),  &
  &                       fl, rlu, xa=xau, ya=yau, za=zau,  &
  &                       a=au, b=bu, c=cu, d=du, e=eu, f=fu, g=gu )
     rlu_err=rlu
  end if

end subroutine GauSei_Vcycle_3df

!----------------------------------
!----------------------------------

subroutine GauSei_Vcycle_3dd( level, levelz, xl, yl, zl, boundary,  &
  &                           ul, fl, ul_new, conv_num,  &
  &                           xat, yat, zat, at, bt, ct, dt, et, ft, gt,  &
  &                           rlu_err, accel, helmod_flag, helmod_dl )
  ! Perform a V-cycle from the level grid to 1st grid (i.e., coarsest grid)
  implicit none
  integer, intent(in) :: level               ! Top grid level (M)
  integer, intent(in) :: levelz(level)       ! Vertical grid level (M)
  double precision, intent(in) :: xl(:)                  ! X grid on the level grid
  double precision, intent(in) :: yl(:)                  ! Y grid on the level grid
  double precision, intent(in) :: zl(:)                  ! Z grid on the level grid
  character(6), intent(in) :: boundary       ! Boundary conditions
  double precision, intent(in) :: ul(size(xl),size(yl),size(zl))  ! Initial guess for u on the level grid
  double precision, intent(in) :: fl(size(xl),size(yl),size(zl))  ! Forcing on the level grid
  double precision, intent(out) :: ul_new(size(xl),size(yl),size(zl)) ! The updated u on the level grid
  integer, intent(in) :: conv_num  ! number of the threshold for iteration
  double precision, intent(in), optional :: xat(size(xl),size(yl),size(zl))  ! coefficient in PDE
  double precision, intent(in), optional :: yat(size(xl),size(yl),size(zl))  ! coefficient in PDE
  double precision, intent(in), optional :: zat(size(xl),size(yl),size(zl))  ! coefficient in PDE
  double precision, intent(in), optional :: at(size(xl),size(yl),size(zl))  ! coefficient in PDE
  double precision, intent(in), optional :: bt(size(xl),size(yl),size(zl))  ! coefficient in PDE
  double precision, intent(in), optional :: ct(size(xl),size(yl),size(zl))  ! coefficient in PDE
  double precision, intent(in), optional :: dt(size(xl),size(yl),size(zl))  ! coefficient in PDE
  double precision, intent(in), optional :: et(size(xl),size(yl),size(zl))  ! coefficient in PDE
  double precision, intent(in), optional :: ft(size(xl),size(yl),size(zl))  ! coefficient in PDE
  double precision, intent(in), optional :: gt(size(xl),size(yl),size(zl))  ! coefficient in PDE
  double precision, intent(out), optional :: rlu_err(size(xl),size(yl),size(zl))  ! the final error
  double precision, intent(in), optional :: accel  ! Acceleration coefficient for Successive Over Relaxation
                             ! (0 < accel < 2), default = 1
  logical, intent(in), optional :: helmod_flag(level)
                             ! Adjustment of "g" for the stability in the Helmholtz equation on the coarse grids.
                             ! .true.: Adjusting "g" and displaying the message
                             ! .false.: No adjusting "g" and terminating with the error message
                             ! default = .false.
  logical, intent(in), optional :: helmod_dl  ! Debug output for helmod_flag = .true.
                             ! .true.: successive output in adjusting g
                             ! .false.: once output in calling this routine
                             ! default = .false.

  integer :: ii, l, sk, skz, nxu, nyu, nzu, nxd, nyd, nzd, nxl, nyl, nzl
  double precision :: dx, dy, dz, dxu, dyu, dzu, dxd, dyd, dzd
  double precision :: accc
  double precision, allocatable, dimension(:) :: xlu, ylu, zlu, xld, yld, zld
  double precision, allocatable, dimension(:,:,:) :: rlu, rld, ulu, uld, flu, fld
  double precision, allocatable, dimension(:,:,:) :: xau, yau, zau, au, bu, cu, du, eu, fu, gu
  double precision, allocatable, dimension(:,:,:) :: xad, yad, zad, ad, bd, cd, dd, ed, fd, gd
  double precision :: tmpv(size(xl),size(yl),size(zl),level+1)
  logical :: hel_flag(level), hel_dl

  hel_flag=.false.
  hel_dl=.false.
  if(present(helmod_flag))then
     !-- the finest grid corresponds to hel_flag(level)
     hel_flag(1:level)=helmod_flag(1:level)
     if(present(helmod_dl))then
        hel_dl=helmod_dl
     end if
  end if

!-- check "accel"

  if(present(accel))then
     accc=accel
  else
     accc=1.0d0
  end if

  !-- Set and Allocate variables
  nxl=size(xl)
  nyl=size(yl)
  nzl=size(zl)
  dx=xl(2)-xl(1)
  dy=yl(2)-yl(1)
  dz=zl(2)-zl(1)
  nxu=nxl
  nyu=nyl
  nzu=nzl
  dxu=dx
  dyu=dy
  dzu=dz

  tmpv=0.0d0

  tmpv(1:nxl,1:nyl,1:nzl,level+1)=ul(1:nxl,1:nyl,1:nzl)

  allocate(xlu(nxu))
  allocate(ylu(nyu))
  allocate(zlu(nzu))
  allocate(ulu(nxu,nyu,nzu))
  allocate(flu(nxu,nyu,nzu))
  allocate(rlu(nxu,nyu,nzu))
  allocate(xau(nxu,nyu,nzu))
  allocate(yau(nxu,nyu,nzu))
  allocate(zau(nxu,nyu,nzu))
  allocate(au(nxu,nyu,nzu))
  allocate(bu(nxu,nyu,nzu))
  allocate(cu(nxu,nyu,nzu))
  allocate(du(nxu,nyu,nzu))
  allocate(eu(nxu,nyu,nzu))
  allocate(fu(nxu,nyu,nzu))
  allocate(gu(nxu,nyu,nzu))

  xau=1.0d0
  yau=1.0d0
  zau=1.0d0
  au=0.0d0
  bu=0.0d0
  cu=0.0d0
  du=0.0d0
  eu=0.0d0
  fu=0.0d0
  gu=0.0d0

  if(present(xat))then
     xau(1:nxu,1:nyu,1:nzu)=xat(1:nxu,1:nyu,1:nzu)
  end if
  if(present(yat))then
     yau(1:nxu,1:nyu,1:nzu)=yat(1:nxu,1:nyu,1:nzu)
  end if
  if(present(zat))then
     zau(1:nxu,1:nyu,1:nzu)=zat(1:nxu,1:nyu,1:nzu)
  end if
  if(present(at))then
     au(1:nxu,1:nyu,1:nzu)=at(1:nxu,1:nyu,1:nzu)
  end if
  if(present(bt))then
     bu(1:nxu,1:nyu,1:nzu)=bt(1:nxu,1:nyu,1:nzu)
  end if
  if(present(ct))then
     cu(1:nxu,1:nyu,1:nzu)=ct(1:nxu,1:nyu,1:nzu)
  end if
  if(present(dt))then
     du(1:nxu,1:nyu,1:nzu)=dt(1:nxu,1:nyu,1:nzu)
  end if
  if(present(et))then
     eu(1:nxu,1:nyu,1:nzu)=et(1:nxu,1:nyu,1:nzu)
  end if
  if(present(ft))then
     fu(1:nxu,1:nyu,1:nzu)=ft(1:nxu,1:nyu,1:nzu)
  end if
  if(present(gt))then
     gu(1:nxu,1:nyu,1:nzu)=gt(1:nxu,1:nyu,1:nzu)
  end if

  !-- Calculation of error on the finest grid
  call calc_error_3dd( xl, yl, zl, ul, fl, rlu, xa=xau, ya=yau, za=zau,  &
  &                    a=au, b=bu, c=cu, d=du, e=eu, f=fu, g=gu )

  !-- Relaxing cycle on level (i.e., the finest grid) (v=(L)^{-1}r)
  call Ellip_GauSei_3dd( xl, yl, zl, rlu, 1.0d-8, boundary, ulu,  &
  &                      xa=xau, ya=yau, za=zau,  &
  &                      a=au, b=bu, c=cu, d=du, e=eu, f=fu, g=gu, ln=conv_num,  &
  &                      accel=accc, helmod_flag=hel_flag(level), helmod_dl=hel_dl )

  xlu(1:nxu)=xl(1:nxu)
  ylu(1:nyu)=yl(1:nyu)
  zlu(1:nzu)=zl(1:nzu)
!  ulu(1:nxu,1:nyu,1:nzu)=ul(1:nxu,1:nyu,1:nzu)
  flu(1:nxu,1:nyu,1:nzu)=rlu(1:nxu,1:nyu,1:nzu)
  tmpv(1:nxl,1:nyl,1:nzl,level)=ulu(1:nxl,1:nyl,1:nzl)

  !-- 1. Downward cycle (Relaxing cycle)
  do l=level,2,-1
     !-- Set and Allocate variables for downward (backward)
     sk=2**(level-l+1)
     skz=2**(levelz(level)-levelz(l-1))
     nxd=(nxl-1)/sk+1
     nyd=(nyl-1)/sk+1
     nzd=(nzl-1)/skz+1
     dxd=dx*real(sk)
     dyd=dy*real(sk)
     dzd=dz*real(skz)
     allocate(xld(nxd))
     allocate(yld(nyd))
     allocate(zld(nzd))
     allocate(uld(nxd,nyd,nzd))
     allocate(fld(nxd,nyd,nzd))
     allocate(rld(nxd,nyd,nzd))
     allocate(xad(nxd,nyd,nzd))
     allocate(yad(nxd,nyd,nzd))
     allocate(zad(nxd,nyd,nzd))
     allocate(ad(nxd,nyd,nzd))
     allocate(bd(nxd,nyd,nzd))
     allocate(cd(nxd,nyd,nzd))
     allocate(dd(nxd,nyd,nzd))
     allocate(ed(nxd,nyd,nzd))
     allocate(fd(nxd,nyd,nzd))
     allocate(gd(nxd,nyd,nzd))
     xld=(/(dxd*(ii-1),ii=1,nxd)/)
     yld=(/(dyd*(ii-1),ii=1,nyd)/)
     zld=(/(dzd*(ii-1),ii=1,nzd)/)
     xad=1.0d0
     yad=1.0d0
     zad=1.0d0
     ad=0.0d0
     bd=0.0d0
     cd=0.0d0
     dd=0.0d0
     ed=0.0d0
     fd=0.0d0
     gd=0.0d0

     !-- Calculation error (r^l=f^l-L^lu^l) on the l grid
     call calc_error_3dd( xlu, ylu, zlu, ulu, flu, rlu,  &
     &                    xa=xau, ya=yau, za=zau,  &
     &                    a=au, b=bu, c=cu, d=du, e=eu, f=fu, g=gu )
!write(*,*) "check rlu", l, rlu
!write(*,*) "check", nxd, nyd, dxd, dyd, sk!, xld(1), xld(nxd), yld(1), yld(nyd)

     !-- Interpolation rl to rl-1 (rlu -> rld)
!write(*,*) "check", l, size(rlu,1), size(rlu,2), l-1, size(rld,1), size(rld,2)
     call forward_interpo_3dd( rlu, rld, l, levelz(1:level), l-1 )
     if(present(xat))then
        call forward_interpo_3dd( xat, xad, level, levelz(1:level), l-1 )
     end if
     if(present(yat))then
        call forward_interpo_3dd( yat, yad, level, levelz(1:level), l-1 )
     end if
     if(present(zat))then
        call forward_interpo_3dd( zat, zad, level, levelz(1:level), l-1 )
     end if
     if(present(at))then
        call forward_interpo_3dd( at, ad, level, levelz(1:level), l-1 )
     end if
     if(present(bt))then
        call forward_interpo_3dd( bt, bd, level, levelz(1:level), l-1 )
     end if
     if(present(ct))then
        call forward_interpo_3dd( ct, cd, level, levelz(1:level), l-1 )
     end if
     if(present(dt))then
        call forward_interpo_3dd( dt, dd, level, levelz(1:level), l-1 )
     end if
     if(present(et))then
        call forward_interpo_3dd( et, ed, level, levelz(1:level), l-1 )
     end if
     if(present(ft))then
        call forward_interpo_3dd( ft, fd, level, levelz(1:level), l-1 )
     end if
     if(present(gt))then
        call forward_interpo_3dd( gt, gd, level, levelz(1:level), l-1 )
     end if

     !-- Relaxing cycle on l-1 (v^{l-1}=(L^{l-1})^{-1}r^{l-1})
     call Ellip_GauSei_3dd( xld, yld, zld, rld, 1.0d-8, boundary, uld,  &
  &                         xa=xad, ya=yad, za=zad,  &
  &                         a=ad, b=bd, c=cd, d=dd, e=ed, f=fd, g=gd,  &
  &                         accel=accc, ln=conv_num, helmod_flag=hel_flag(l-1), helmod_dl=hel_dl )

     !-- Restore v^{l-1}
     tmpv(1:nxd,1:nyd,1:nzd,l-1)=uld(1:nxd,1:nyd,1:nzd)

     !-- Reset *lu variables on the l-1 grid
     nxu=nxd
     nyu=nyd
     nzu=nzd
     dxu=dxd
     dyu=dyd
     dzu=dzd

     !-- Reallocate *lu variables on the l-1 grid
     deallocate(xlu)
     deallocate(ylu)
     deallocate(zlu)
     deallocate(ulu)
     deallocate(flu)
     deallocate(rlu)
     deallocate(xau)
     deallocate(yau)
     deallocate(zau)
     deallocate(au)
     deallocate(bu)
     deallocate(cu)
     deallocate(du)
     deallocate(eu)
     deallocate(fu)
     deallocate(gu)
     allocate(xlu(nxu))
     allocate(ylu(nyu))
     allocate(ulu(nxu,nyu,nzu))
     allocate(flu(nxu,nyu,nzu))
     allocate(rlu(nxu,nyu,nzu))
     allocate(xau(nxu,nyu,nzu))
     allocate(yau(nxu,nyu,nzu))
     allocate(zau(nxu,nyu,nzu))
     allocate(au(nxu,nyu,nzu))
     allocate(bu(nxu,nyu,nzu))
     allocate(cu(nxu,nyu,nzu))
     allocate(du(nxu,nyu,nzu))
     allocate(eu(nxu,nyu,nzu))
     allocate(fu(nxu,nyu,nzu))
     allocate(gu(nxu,nyu,nzu))
     xlu=(/(dxu*(ii-1),ii=1,nxu)/)
     ylu=(/(dyu*(ii-1),ii=1,nyu)/)
     zlu=(/(dzu*(ii-1),ii=1,nzu)/)
     ulu(1:nxu,1:nyu,1:nzu)=uld(1:nxd,1:nyd,1:nzd)
     flu(1:nxu,1:nyu,1:nzu)=rld(1:nxd,1:nyd,1:nzd)
     xau(1:nxu,1:nyu,1:nzu)=xad(1:nxd,1:nyd,1:nzd)
     yau(1:nxu,1:nyu,1:nzu)=yad(1:nxd,1:nyd,1:nzd)
     zau(1:nxu,1:nyu,1:nzu)=zad(1:nxd,1:nyd,1:nzd)
     au(1:nxu,1:nyu,1:nzu)=ad(1:nxd,1:nyd,1:nzd)
     bu(1:nxu,1:nyu,1:nzu)=bd(1:nxd,1:nyd,1:nzd)
     cu(1:nxu,1:nyu,1:nzu)=cd(1:nxd,1:nyd,1:nzd)
     du(1:nxu,1:nyu,1:nzu)=dd(1:nxd,1:nyd,1:nzd)
     eu(1:nxu,1:nyu,1:nzu)=ed(1:nxd,1:nyd,1:nzd)
     fu(1:nxu,1:nyu,1:nzu)=fd(1:nxd,1:nyd,1:nzd)
     gu(1:nxu,1:nyu,1:nzu)=gd(1:nxd,1:nyd,1:nzd)

     !-- Deallocate
     deallocate(xld)
     deallocate(yld)
     deallocate(zld)
     deallocate(uld)
     deallocate(fld)
     deallocate(rld)
     deallocate(xad)
     deallocate(yad)
     deallocate(zad)
     deallocate(ad)
     deallocate(bd)
     deallocate(cd)
     deallocate(dd)
     deallocate(ed)
     deallocate(fd)
     deallocate(gd)

  end do

  deallocate(ulu)

  !-- 2. Upward cycle (smoothing and interpolating cycle)
  do l=2,level
     !-- Set and Allocate variables for upward (forward)
     sk=2**(level-l+1)
     skz=2**(levelz(level)-levelz(l-1))
     nxd=(nxl-1)/sk+1
     nyd=(nyl-1)/sk+1
     nzd=(nzl-1)/skz+1
     sk=2**(level-l)
     skz=2**(levelz(level)-levelz(l))
     nxu=(nxl-1)/sk+1
     nyu=(nyl-1)/sk+1
     nzu=(nzl-1)/skz+1
     allocate(uld(nxd,nyd,nzd))
     allocate(ulu(nxu,nyu,nzu))

     uld(1:nxd,1:nyd,1:nzd)=tmpv(1:nxd,1:nyd,1:nzd,l-1)

     !-- Interpolation vl-1 to vl
     call backward_interpo_3dd( uld, ulu, l-1, levelz(1:level), l )

     !-- Add the vl to the previous vl on l grid
     tmpv(1:nxu,1:nyu,1:nzu,l)=tmpv(1:nxu,1:nyu,1:nzu,l)+ulu(1:nxu,1:nyu,1:nzu)

     !-- Deallocate ul
     deallocate(uld)
     deallocate(ulu)
  end do

  tmpv(1:nxl,1:nyl,1:nzl,level+1)=tmpv(1:nxl,1:nyl,1:nzl,level+1)+tmpv(1:nxl,1:nyl,1:nzl,level)
  ul_new(1:nxl,1:nyl,1:nzl)=tmpv(1:nxl,1:nyl,1:nzl,level+1)

  if(present(rlu_err))then
     deallocate(rlu)
     deallocate(xau)
     deallocate(yau)
     deallocate(zau)
     deallocate(au)
     deallocate(bu)
     deallocate(cu)
     deallocate(du)
     deallocate(eu)
     deallocate(fu)
     deallocate(gu)
     allocate(rlu(nxl,nyl,nzl))
     allocate(xau(nxl,nyl,nzl))
     allocate(yau(nxl,nyl,nzl))
     allocate(zau(nxl,nyl,nzl))
     allocate(au(nxl,nyl,nzl))
     allocate(bu(nxl,nyl,nzl))
     allocate(cu(nxl,nyl,nzl))
     allocate(du(nxl,nyl,nzl))
     allocate(eu(nxl,nyl,nzl))
     allocate(fu(nxl,nyl,nzl))
     allocate(gu(nxl,nyl,nzl))

     xau=1.0d0
     yau=1.0d0
     zau=1.0d0
     au=0.0d0
     cu=0.0d0
     bu=0.0d0
     du=0.0d0
     eu=0.0d0
     fu=0.0d0
     gu=0.0d0

     if(present(xat))then
        xau=xat
     end if
     if(present(yat))then
        yau=yat
     end if
     if(present(zat))then
        zau=zat
     end if
     if(present(at))then
        au=at
     end if
     if(present(bt))then
        bu=bt
     end if
     if(present(ct))then
        cu=ct
     end if
     if(present(dt))then
        du=dt
     end if
     if(present(et))then
        eu=et
     end if
     if(present(ft))then
        fu=ft
     end if
     if(present(gt))then
        gu=gt
     end if

     call calc_error_3dd( xl, yl, zl, tmpv(1:nxl,1:nyl,1:nzl,level+1),  &
  &                       fl, rlu, xa=xau, ya=yau, za=zau,  &
  &                       a=au, b=bu, c=cu, d=du, e=eu, f=fu, g=gu )
     rlu_err=rlu
  end if

end subroutine GauSei_Vcycle_3dd

!----------------------------------
!-----------------------------------
! 以下, private subroutine
!-----------------------------------

subroutine set_bound( bound, ib, inner_flag, inner_bound )
! 各反復法ルーチンにおいて設定される境界条件のフラグをチェック, 設定する.
  implicit none
  character(4), intent(in) :: bound   ! 領域境界のフラグ
  integer, intent(inout) :: ib(:,:)   ! 全境界の判別整数
  logical, intent(inout) :: inner_flag(size(ib,1),size(ib,2))
                            ! 全領域境界フラグ
  integer, intent(in), optional :: inner_bound(size(ib,1),size(ib,2))
                            ! 内部領域境界の判別整数
  integer :: i, j, nx, ny

  nx=size(ib,1)
  ny=size(ib,2)

!-- 出力変数の初期化

  ib=0
  inner_flag=.true.

!-- 周期境界の設定確認.
!-- 周期境界なので, 両端とも 3 が設定されていないといけない.
  if(bound(1:1)=='3')then
     if(bound(3:3)/='3')then
        write(*,*) "### ERROR ###"
        write(*,*) "if bound = 3, bound(1:1)==bound(3:3). STOP."
        stop
     end if
  end if

  if(bound(3:3)=='3')then
     if(bound(1:1)/='3')then
        write(*,*) "### ERROR ###"
        write(*,*) "if bound = 3, bound(1:1)==bound(3:3). STOP."
        stop
     end if
  end if

  if(bound(2:2)=='3')then
     if(bound(4:4)/='3')then
        write(*,*) "### ERROR ###"
        write(*,*) "if bound = 3, bound(2:2)==bound(4:4). STOP."
        stop
     end if
  end if

  if(bound(4:4)=='3')then
     if(bound(2:2)/='3')then
        write(*,*) "### ERROR ###"
        write(*,*) "if bound = 3, bound(2:2)==bound(4:4). STOP."
        stop
     end if
  end if

  select case (bound(1:1))
  case ('1')
     do i=2,nx-1
        ib(i,1)=1
     end do

  case ('2')
     do i=2,nx-1
        ib(i,1)=4  ! y 方向のフラックスで参照値は上側
     end do

  case ('3')
     do i=2,nx-1
        ib(i,1)=3  ! y 方向 (x 軸) 周期境界
     end do
  end select

  select case (bound(2:2))
  case ('1')
     do j=2,ny-1
        ib(1,j)=1
     end do

  case ('2')
     do j=2,ny-1
        ib(1,j)=2  ! x 方向のフラックスで参照値は右側
     end do

  case ('3')
     do j=2,ny-1
        ib(1,j)=3  ! x 方向 (y 軸) 周期境界
     end do
  end select

  select case (bound(3:3))
  case ('1')
     do i=2,nx-1
        ib(i,ny)=1
     end do

  case ('2')
     do i=2,nx-1
        ib(i,ny)=-4  ! y 方向のフラックスで参照値は下側
     end do

  case ('3')
     do i=2,nx-1
        ib(i,ny)=3  ! y 方向 (x 軸) 周期境界
     end do
  end select

  select case (bound(4:4))
  case ('1')
     do j=2,ny-1
        ib(nx,j)=1
     end do

  case ('2')
     do j=2,ny-1
        ib(nx,j)=-2  ! x 方向のフラックスで参照値は左側
     end do

  case ('3')
     do j=2,ny-1
        ib(nx,j)=3  ! x 方向 (y 軸) 周期境界
     end do
  end select

!-- 領域隅境界は 2 辺が重なるので, 境界での強制方法は以下の順に優先する.
!-- (1) どちらかが 1 のとき, 隅領域は 1,
!-- (2) (1) 以外でどちらかが 3 のとき, 隅領域は 3.
!-- (3) (1), (2) 以外で 2. つまり, 両軸とも 2 なら 2 となる.

  if(bound(1:1)=='1')then
     ib(1,1)=1
     ib(nx,1)=1
  end if
  if(bound(2:2)=='1')then
     ib(1,1)=1
     ib(1,ny)=1
  end if
  if(bound(3:3)=='1')then
     ib(1,ny)=1
     ib(nx,ny)=1
  end if
  if(bound(4:4)=='1')then
     ib(nx,1)=1
     ib(nx,ny)=1
  end if

!-- 4 隅とも周期境界の場合
  if(bound(1:2)=='33')then
     ib(1,1)=3
     ib(1,ny)=3
     ib(nx,1)=3
     ib(nx,ny)=3
  end if

!-- どちらかが周期境界の場合, ib の符号が設定されていなければ -3 を指定.
  if(bound(1:1)=='3'.or.bound(2:2)=='3')then
     if(ib(1,1)==0)then
        ib(1,1)=3
     end if
     if(ib(nx,1)==0)then
        ib(nx,1)=3
     end if
     if(ib(1,ny)==0)then
        ib(1,ny)=3
     end if
     if(ib(nx,ny)==0)then
        ib(nx,ny)=3
     end if
  end if

!-- 領域の隅であることを 8, -8 で示す.
!-- 8 の場合, 左下か右上であることを, -8 の場合, 左上か右下であることを示す.

  if(ib(1,1)==0)then
     ib(1,1)=8
  end if
  if(ib(nx,1)==0)then
     ib(nx,1)=-8
  end if
  if(ib(1,ny)==0)then
     ib(1,ny)=-8
  end if
  if(ib(nx,ny)==0)then
     ib(nx,ny)=8
  end if

!-- 内部境界の設定

  if(present(inner_bound))then
     do j=2,ny-1
        do i=2,nx-1
           ib(i,j)=inner_bound(i,j)
           if(ib(i,j)==0)then
              inner_flag(i,j)=.false.
           end if
        end do
     end do
  else   ! 内部領域が設定されていない場合は, 内部全て計算する.
     do j=2,ny-1
        do i=2,nx-1
           inner_flag(i,j)=.false.
        end do
     end do
  end if

end subroutine set_bound



subroutine setval_boundf( ib, bnd, psi, bound_opt )
! 境界値を境界条件判別整数をもとに設定する.
  implicit none
  integer, intent(in) :: ib(:,:)  ! 境界条件判別整数
  real, intent(inout) :: bnd(size(ib,1),size(ib,2))  ! 境界での値
  real, intent(inout) :: psi(size(ib,1),size(ib,2))  ! 応答
  real, intent(in), optional :: bound_opt(size(ib,1),size(ib,2))  ! 境界での値
  integer :: i, j, nx, ny

  nx=size(ib,1)
  ny=size(ib,2)

!-- 境界値の設定
  if(present(bound_opt))then
     do j=1,ny
        do i=1,nx
           bnd(i,j)=bound_opt(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           bnd(i,j)=0.0
        end do
     end do
  end if

!-- 境界条件の代入 "ib(i,j)==1 の場合のみ"
!-- 内部領域についてもここで代入してしまう.

  do j=1,ny
     do i=1,nx
        if(ib(i,j)==1)then
           psi(i,j)=bnd(i,j)
        end if
     end do
  end do

end subroutine setval_boundf


subroutine setval_boundd( ib, bnd, psi, bound_opt )
! 境界値を境界条件判別整数をもとに設定する.
  implicit none
  integer, intent(in) :: ib(:,:)  ! 境界条件判別整数
  double precision, intent(inout) :: bnd(size(ib,1),size(ib,2))  ! 境界での値
  double precision, intent(inout) :: psi(size(ib,1),size(ib,2))  ! 応答
  double precision, intent(in), optional :: bound_opt(size(ib,1),size(ib,2))  ! 境界での値
  integer :: i, j, nx, ny

  nx=size(ib,1)
  ny=size(ib,2)

!-- 境界値の設定
  if(present(bound_opt))then
     do j=1,ny
        do i=1,nx
           bnd(i,j)=bound_opt(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           bnd(i,j)=0.0d0
        end do
     end do
  end if

!-- 境界条件の代入 "ib(i,j)==1 の場合のみ"
!-- 内部領域についてもここで代入してしまう.

  do j=1,ny
     do i=1,nx
        if(ib(i,j)==1)then
           psi(i,j)=bnd(i,j)
        end if
     end do
  end do

end subroutine setval_boundd


subroutine set_coef( coe, ext, def )
! 2 次元配列に ext で指定された値もしくは def で指定された一定値を代入する.
! ext, def どちらも optional であるが, 必ずどちらかは指定されていないといけない.
  implicit none
  real, intent(inout) :: coe(:,:)  ! 代入される配列
  real, intent(in), optional :: ext(size(coe,1),size(coe,2))  ! 代入する配列
  real, intent(in), optional :: def  ! 代入する一定値
  integer :: i, j, nx, ny

  nx=size(coe,1)
  ny=size(coe,2)

  if(present(ext))then
     do j=1,ny
        do i=1,nx
           coe(i,j)=ext(i,j)
        end do
     end do
  else if(present(def))then
     do j=1,ny
        do i=1,nx
           coe(i,j)=def
        end do
     end do
  else
     write(*,*) "### ERROR ###"
     write(*,*) "subroutine set_coe must be set optional argument 'ext' or 'def'"
     write(*,*) "STOP."
     stop
  end if

end subroutine set_coef

subroutine set_coed( coe, ext, def )
! 2 次元配列に ext で指定された値もしくは def で指定された一定値を代入する.
! ext, def どちらも optional であるが, 必ずどちらかは指定されていないといけない.
  implicit none
  double precision, intent(inout) :: coe(:,:)  ! 代入される配列
  double precision, intent(in), optional :: ext(size(coe,1),size(coe,2))  ! 代入する配列
  double precision, intent(in), optional :: def  ! 代入する一定値
  integer :: i, j, nx, ny

  nx=size(coe,1)
  ny=size(coe,2)

  if(present(ext))then
     do j=1,ny
        do i=1,nx
           coe(i,j)=ext(i,j)
        end do
     end do
  else if(present(def))then
     do j=1,ny
        do i=1,nx
           coe(i,j)=def
        end do
     end do
  else
     write(*,*) "### ERROR ###"
     write(*,*) "subroutine set_coe must be set optional argument 'ext' or 'def'"
     write(*,*) "STOP."
     stop
  end if

end subroutine set_coed


subroutine check_coef( coe, aval, undeff )
! 2 次元配列に aval が代入されていないかをチェックする.
  implicit none
  real, intent(inout) :: coe(:,:)  ! 代入される配列
  real, intent(in) :: aval         ! チェックされる値
  real, intent(in), optional :: undeff
  integer :: i, j, nx, ny

  nx=size(coe,1)
  ny=size(coe,2)

  if(present(undeff))then
     do j=1,ny
        do i=1,nx
           if(coe(i,j)/=undeff)then
              if(coe(i,j)==aval)then
                 write(*,*) "### ERROR (Ellip_Slv module) ###"
                 write(*,*) "Detect a certain value", aval, i, j
                 write(*,*) "STOP."
                 stop
              end if
           end if
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           if(coe(i,j)==aval)then
              write(*,*) "### ERROR (Ellip_Slv module) ###"
              write(*,*) "Detect a certain value", aval, i, j
              write(*,*) "STOP."
              stop
           end if
        end do
     end do
  end if

end subroutine check_coef


subroutine check_coed( coe, aval, undeff )
! 2 次元配列に aval が代入されていないかをチェックする.
  implicit none
  double precision, intent(inout) :: coe(:,:)  ! 代入される配列
  double precision, intent(in) :: aval         ! チェックされる値
  double precision, intent(in), optional :: undeff
  integer :: i, j, nx, ny

  nx=size(coe,1)
  ny=size(coe,2)

  if(present(undeff))then
     do j=1,ny
        do i=1,nx
           if(coe(i,j)/=undeff)then
              if(coe(i,j)==aval)then
                 write(*,*) "### ERROR (Ellip_Slv module) ###"
                 write(*,*) "Detect a certain value", aval, i, j
                 write(*,*) "STOP."
                 stop
              end if
           end if
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           if(coe(i,j)==aval)then
              write(*,*) "### ERROR (Ellip_Slv module) ###"
              write(*,*) "Detect a certain value", aval, i, j
              write(*,*) "STOP."
              stop
           end if
        end do
     end do
  end if

end subroutine check_coed


subroutine check_le_coef( coe, aval, undeff )
! 2 次元配列に less equal aval をチェックする.
  implicit none
  real, intent(inout) :: coe(:,:)  ! 代入される配列
  real, intent(in) :: aval         ! チェックされる値
  real, intent(in), optional :: undeff
  integer :: i, j, nx, ny

  nx=size(coe,1)
  ny=size(coe,2)

  if(present(undeff))then
     do j=1,ny
        do i=1,nx
           if(coe(i,j)/=undeff)then
              if(coe(i,j)<=aval)then
                 write(*,*) "### ERROR (Ellip_Slv module) ###"
                 write(*,*) "Detect a certain value", coe(i,j), i, j
                 write(*,*) "STOP."
                 stop
              end if
           end if
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           if(coe(i,j)<=aval)then
              write(*,*) "### ERROR (Ellip_Slv module) ###"
              write(*,*) "Detect a certain value", coe(i,j), i, j
              write(*,*) "STOP."
              stop
           end if
        end do
     end do
  end if

end subroutine check_le_coef


subroutine check_le_coed( coe, aval, undeff )
! 2 次元配列に less equal aval をチェックする.
  implicit none
  double precision, intent(inout) :: coe(:,:)  ! 代入される配列
  double precision, intent(in) :: aval         ! チェックされる値
  double precision, intent(in), optional :: undeff
  integer :: i, j, nx, ny

  nx=size(coe,1)
  ny=size(coe,2)

  if(present(undeff))then
     do j=1,ny
        do i=1,nx
           if(coe(i,j)/=undeff)then
              if(coe(i,j)<=aval)then
                 write(*,*) "### ERROR (Ellip_Slv module) ###"
                 write(*,*) "Detect a certain value", coe(i,j), i, j
                 write(*,*) "STOP."
                 stop
              end if
           end if
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           if(coe(i,j)<=aval)then
              write(*,*) "### ERROR (Ellip_Slv module) ###"
              write(*,*) "Detect a certain value", coe(i,j), i, j
              write(*,*) "STOP."
              stop
           end if
        end do
     end do
  end if

end subroutine check_le_coed


subroutine calculate_boundf( ib, dx, dy, bnd, psi )
! ib 判別を元に, ノイマン型, 周期境界型について境界値を計算する.
  integer, intent(in) :: ib(:,:)  ! 境界整数判別
  real, intent(in) :: dx(size(ib,1))  ! x 方向の格子解像度
  real, intent(in) :: dy(size(ib,2))  ! y 方向の格子解像度
  real, intent(in) :: bnd(size(ib,1),size(ib,2))  ! 境界値(ノイマン型のみ使用)
  real, intent(inout) :: psi(size(ib,1),size(ib,2))  ! 応答関数
  integer :: i, j, nx, ny, ix, jy

  nx=size(ib,1)
  ny=size(ib,2)

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,ix,jy)
  do j=1,ny
     do i=1,nx
        if(ib(i,j)/=0)then   ! 計算領域点が select 文に入るのを防ぐ.
           select case (ib(i,j))
           case (1)
              psi(i,j)=bnd(i,j)

           case (2)  ! x 方向にフラックス一定, 上側が参照値
              psi(i,j)=psi(i+1,j)-bnd(i,j)*dx(i)

           case (-2)  ! x 方向にフラックス一定, 下側が参照値
              psi(i,j)=psi(i-1,j)+bnd(i,j)*dx(i)

           case (4)  ! y 方向にフラックス一定, 右側が参照値
              psi(i,j)=psi(i,j+1)-bnd(i,j)*dy(j)

           case (-4)  ! y 方向にフラックス一定, 左側が参照値
              psi(i,j)=psi(i,j-1)+bnd(i,j)*dy(j)

           case (3)  ! 周期境界
              if(i==1)then
                 ix=nx-1
              else if(i==nx)then
                 ix=2
              else
                 ix=i
              end if
              if(j==1)then
                 jy=ny-1
              else if(j==ny)then
                 jy=2
              else
                 jy=j
              end if

              psi(i,j)=psi(ix,jy)

           case (7)  ! 両方フラックス一定で内部境界限定.
              if((ib(i+1,j+1)==10).and.(ib(i+1,j)/=10).and.  &
  &              (ib(i,j+1)/=10))then
                 psi(i,j)=0.5*(psi(i-1,j)+psi(i,j-1))  &
  &                       +0.5*bnd(i,j)*(dy(j)+dx(i))

              else if((ib(i-1,j-1)==10).and.(ib(i-1,j)/=10).and.  &
  &                   (ib(i,j-1)/=10))then
                 psi(i,j)=0.5*(psi(i+1,j)+psi(i,j+1))  &
  &                       -0.5*bnd(i,j)*(dy(j)+dx(i))

              end if

           case (-7)  ! 両方フラックス一定で内部境界限定.
              if((ib(i-1,j+1)==10).and.(ib(i-1,j)/=10).and.  &
  &              (ib(i,j+1)/=10))then
                 psi(i,j)=0.5*(psi(i+1,j)+psi(i,j-1))  &
  &                       +0.5*bnd(i,j)*(dy(j)-dx(i))

              else if((ib(i+1,j-1)==10).and.(ib(i+1,j)/=10).and.  &
  &              (ib(i,j-1)/=10))then
                 psi(i,j)=0.5*(psi(i-1,j)+psi(i,j+1))  &
  &                       +0.5*bnd(i,j)*(-dy(j)+dx(i))

              end if

           case (8)  ! 両方フラックス一定で左下角か右上角, もしくは内部境界.
              if(i==1.and.j==1)then  ! -- 評価 1
                 psi(i,j)=psi(i+1,j+1)-0.5*(bnd(i+1,j)*dy(j)+bnd(i,j+1)*dx(i))

              else if(i==nx.and.j==ny)then  ! -- 評価 2
                 psi(i,j)=psi(i-1,j-1)+0.5*(bnd(i-1,j)*dy(j)+bnd(i,j-1)*dx(i))

              else if(ib(i-1,j)==10.and.ib(i,j-1)==10)then
                 ! -- 評価 1 と同じ
                 psi(i,j)=psi(i+1,j+1)-0.5*(bnd(i+1,j)*dy(j)+bnd(i,j+1)*dx(i))

              else if(ib(i+1,j)==10.and.ib(i,j+1)==10)then
                 ! -- 評価 2 と同じ
                 psi(i,j)=psi(i-1,j-1)+0.5*(bnd(i-1,j)*dy(j)+bnd(i,j-1)*dx(i))

              end if

           case (-8)  ! 両方フラックス一定で右下角か左上角
              if(i==1.and.j==ny)then  ! -- 評価 1
                 psi(i,j)=psi(i+1,j-1)+0.5*(bnd(i+1,j)*dy(j)-bnd(i,j-1)*dx(i))

              else if(i==nx.and.j==1)then  ! -- 評価 2
                 psi(i,j)=psi(i-1,j+1)+0.5*(-bnd(i-1,j)*dy(j)+bnd(i,j+1)*dx(i))

              else if(ib(i-1,j)==10.and.ib(i,j+1)==10)then
                 ! -- 評価 1 と同じ
                 psi(i,j)=psi(i+1,j-1)+0.5*(bnd(i+1,j)*dy(j)-bnd(i,j-1)*dx(i))

              else if(ib(i+1,j)==10.and.ib(i,j-1)==10)then
                 ! -- 評価 2 と同じ
                 psi(i,j)=psi(i-1,j+1)+0.5*(-bnd(i-1,j)*dy(j)+bnd(i,j+1)*dx(i))

              end if
           end select

        end if
     end do
  end do
!$omp end do
!$omp end parallel

end subroutine calculate_boundf


subroutine calculate_boundd( ib, dx, dy, bnd, psi )
! ib 判別を元に, ノイマン型, 周期境界型について境界値を計算する.
  integer, intent(in) :: ib(:,:)  ! 境界整数判別
  double precision, intent(in) :: dx(size(ib,1))  ! x 方向の格子解像度
  double precision, intent(in) :: dy(size(ib,2))  ! y 方向の格子解像度
  double precision, intent(in) :: bnd(size(ib,1),size(ib,2))  ! 境界値(ノイマン型のみ使用)
  double precision, intent(inout) :: psi(size(ib,1),size(ib,2))  ! 応答関数
  integer :: i, j, nx, ny, ix, jy

  nx=size(ib,1)
  ny=size(ib,2)

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,ix,jy)
  do j=1,ny
     do i=1,nx
        if(ib(i,j)/=0)then   ! 計算領域点が select 文に入るのを防ぐ.
           select case (ib(i,j))
           case (1)
              psi(i,j)=bnd(i,j)

           case (2)  ! x 方向にフラックス一定, 上側が参照値
              psi(i,j)=psi(i+1,j)-bnd(i,j)*dx(i)

           case (-2)  ! x 方向にフラックス一定, 下側が参照値
              psi(i,j)=psi(i-1,j)+bnd(i,j)*dx(i)

           case (4)  ! y 方向にフラックス一定, 右側が参照値
              psi(i,j)=psi(i,j+1)-bnd(i,j)*dy(j)

           case (-4)  ! y 方向にフラックス一定, 左側が参照値
              psi(i,j)=psi(i,j-1)+bnd(i,j)*dy(j)

           case (3)  ! 周期境界
              if(i==1)then
                 ix=nx-1
              else if(i==nx)then
                 ix=2
              else
                 ix=i
              end if
              if(j==1)then
                 jy=ny-1
              else if(j==ny)then
                 jy=2
              else
                 jy=j
              end if

              psi(i,j)=psi(ix,jy)

           case (7)  ! 両方フラックス一定で内部境界限定.
              if((ib(i+1,j+1)==10).and.(ib(i+1,j)/=10).and.  &
  &              (ib(i,j+1)/=10))then
                 psi(i,j)=0.5d0*(psi(i-1,j)+psi(i,j-1))  &
  &                       +0.5d0*bnd(i,j)*(dy(j)+dx(i))

              else if((ib(i-1,j-1)==10).and.(ib(i-1,j)/=10).and.  &
  &                   (ib(i,j-1)/=10))then
                 psi(i,j)=0.5d0*(psi(i+1,j)+psi(i,j+1))  &
  &                       -0.5d0*bnd(i,j)*(dy(j)+dx(i))

              end if

           case (-7)  ! 両方フラックス一定で内部境界限定.
              if((ib(i-1,j+1)==10).and.(ib(i-1,j)/=10).and.  &
  &              (ib(i,j+1)/=10))then
                 psi(i,j)=0.5d0*(psi(i+1,j)+psi(i,j-1))  &
  &                       +0.5d0*bnd(i,j)*(dy(j)-dx(i))

              else if((ib(i+1,j-1)==10).and.(ib(i+1,j)/=10).and.  &
  &              (ib(i,j-1)/=10))then
                 psi(i,j)=0.5d0*(psi(i-1,j)+psi(i,j+1))  &
  &                       +0.5d0*bnd(i,j)*(-dy(j)+dx(i))

              end if

           case (8)  ! 両方フラックス一定で左下角か右上角, もしくは内部境界.
              if(i==1.and.j==1)then  ! -- 評価 1
                 psi(i,j)=psi(i+1,j+1)-0.5d0*(bnd(i+1,j)*dy(j)+bnd(i,j+1)*dx(i))

              else if(i==nx.and.j==ny)then  ! -- 評価 2
                 psi(i,j)=psi(i-1,j-1)+0.5d0*(bnd(i-1,j)*dy(j)+bnd(i,j-1)*dx(i))

              else if(ib(i-1,j)==10.and.ib(i,j-1)==10)then
                 ! -- 評価 1 と同じ
                 psi(i,j)=psi(i+1,j+1)-0.5d0*(bnd(i+1,j)*dy(j)+bnd(i,j+1)*dx(i))

              else if(ib(i+1,j)==10.and.ib(i,j+1)==10)then
                 ! -- 評価 2 と同じ
                 psi(i,j)=psi(i-1,j-1)+0.5d0*(bnd(i-1,j)*dy(j)+bnd(i,j-1)*dx(i))

              end if

           case (-8)  ! 両方フラックス一定で右下角か左上角
              if(i==1.and.j==ny)then  ! -- 評価 1
                 psi(i,j)=psi(i+1,j-1)+0.5d0*(bnd(i+1,j)*dy(j)-bnd(i,j-1)*dx(i))

              else if(i==nx.and.j==1)then  ! -- 評価 2
                 psi(i,j)=psi(i-1,j+1)+0.5d0*(-bnd(i-1,j)*dy(j)+bnd(i,j+1)*dx(i))

              else if(ib(i-1,j)==10.and.ib(i,j+1)==10)then
                 ! -- 評価 1 と同じ
                 psi(i,j)=psi(i+1,j-1)+0.5d0*(bnd(i+1,j)*dy(j)-bnd(i,j-1)*dx(i))

              else if(ib(i+1,j)==10.and.ib(i,j-1)==10)then
                 ! -- 評価 2 と同じ
                 psi(i,j)=psi(i-1,j+1)+0.5d0*(-bnd(i-1,j)*dy(j)+bnd(i,j+1)*dx(i))

              end if
           end select

        end if
     end do
  end do
!$omp end do
!$omp end parallel

end subroutine calculate_boundd


subroutine set_bound_3d( bound, ib, inner_flag, inner_bound )
! 各反復法ルーチンにおいて設定される境界条件のフラグをチェック, 設定する.
  implicit none
  character(6), intent(in) :: bound     ! 領域境界のフラグ
  integer, intent(inout) :: ib(:,:,:)   ! 全境界の判別整数
  logical, intent(inout) :: inner_flag(size(ib,1),size(ib,2),size(ib,3))
                            ! 全領域境界フラグ
  integer, intent(in), optional :: inner_bound(size(ib,1),size(ib,2),size(ib,3))
                            ! 内部領域境界の判別整数
  integer :: i, j, k, nx, ny, nz

  nx=size(ib,1)
  ny=size(ib,2)
  nz=size(ib,3)

!-- 出力変数の初期化

  ib=0
  inner_flag=.true.

!-- 周期境界の設定確認.
!-- 周期境界なので, 両端とも 3 が設定されていないといけない.
  if(bound(1:1)=='3')then
     if(bound(3:3)/='3')then
        write(*,*) "### ERROR ###"
        write(*,*) "if bound = 3, bound(1:1)==bound(3:3). STOP."
        stop
     end if
  end if

  if(bound(3:3)=='3')then
     if(bound(1:1)/='3')then
        write(*,*) "### ERROR ###"
        write(*,*) "if bound = 3, bound(1:1)==bound(3:3). STOP."
        stop
     end if
  end if

  if(bound(2:2)=='3')then
     if(bound(4:4)/='3')then
        write(*,*) "### ERROR ###"
        write(*,*) "if bound = 3, bound(2:2)==bound(4:4). STOP."
        stop
     end if
  end if

  if(bound(4:4)=='3')then
     if(bound(2:2)/='3')then
        write(*,*) "### ERROR ###"
        write(*,*) "if bound = 3, bound(2:2)==bound(4:4). STOP."
        stop
     end if
  end if

  if(bound(5:5)=='3')then
     if(bound(6:6)/='3')then
        write(*,*) "### ERROR ###"
        write(*,*) "if bound = 3, bound(5:5)==bound(6:6). STOP."
        stop
     end if
  end if

  if(bound(6:6)=='3')then
     if(bound(5:5)/='3')then
        write(*,*) "### ERROR ###"
        write(*,*) "if bound = 3, bound(5:5)==bound(6:6). STOP."
        stop
     end if
  end if

  select case (bound(1:1))
  case ('1')
     do k=2,nz-1
        do i=2,nx-1
           ib(i,1,k)=1
        end do
     end do

  case ('2')
     do k=2,nz-1
        do i=2,nx-1
           ib(i,1,k)=4  ! y 方向のフラックスで参照値は上側
        end do
     end do

  case ('3')
     do k=2,nz-1
        do i=2,nx-1
           ib(i,1,k)=3  ! y 方向 (x 軸) 周期境界
        end do
     end do
  end select

  select case (bound(2:2))
  case ('1')
     do k=2,nz-1
        do j=2,ny-1
           ib(1,j,k)=1
        end do
     end do

  case ('2')
     do k=2,nz-1
        do j=2,ny-1
           ib(1,j,k)=2  ! x 方向のフラックスで参照値は右側
        end do
     end do

  case ('3')
     do k=2,nz-1
        do j=2,ny-1
           ib(1,j,k)=3  ! x 方向 (y 軸) 周期境界
        end do
     end do
  end select

  select case (bound(3:3))
  case ('1')
     do k=2,nz-1
        do i=2,nx-1
           ib(i,ny,k)=1
        end do
     end do

  case ('2')
     do k=2,nz-1
        do i=2,nx-1
           ib(i,ny,k)=-4  ! y 方向のフラックスで参照値は下側
        end do
     end do

  case ('3')
     do k=2,nz-1
        do i=2,nx-1
           ib(i,ny,k)=3  ! y 方向 (x 軸) 周期境界
        end do
     end do
  end select

  select case (bound(4:4))
  case ('1')
     do k=2,nz-1
        do j=2,ny-1
           ib(nx,j,k)=1
        end do
     end do

  case ('2')
     do k=2,nz-1
        do j=2,ny-1
           ib(nx,j,k)=-2  ! x 方向のフラックスで参照値は左側
        end do
     end do

  case ('3')
     do k=2,nz-1
        do j=2,ny-1
           ib(nx,j,k)=3  ! x 方向 (y 軸) 周期境界
        end do
     end do
  end select

  select case (bound(5:5))
  case ('1')
     do j=2,ny-1
        do i=2,nx-1
           ib(i,j,1)=1
        end do
     end do

  case ('2')
     do j=2,ny-1
        do i=2,nx-1
           ib(i,j,1)=6  ! z 方向のフラックスで参照値は下面
        end do
     end do

  case ('3')
     do j=2,ny-1
        do i=2,nx-1
           ib(i,j,1)=3
        end do
     end do
  end select

  select case (bound(6:6))
  case ('1')
     do j=2,ny-1
        do i=2,nx-1
           ib(i,j,nz)=1
        end do
     end do

  case ('2')
     do j=2,ny-1
        do i=2,nx-1
           ib(i,j,nz)=-6  ! z 方向のフラックスで参照値は上面
        end do
     end do

  case ('3')
     do j=2,ny-1
        do i=2,nx-1
           ib(i,j,nz)=3
        end do
     end do
  end select

!-- 領域隅境界は 2 辺が重なるので, 境界での強制方法は以下の順に優先する.
!-- (1) どちらかが 1 のとき, 隅領域は 1,
!-- (2) (1) 以外でどちらかが 3 のとき, 隅領域は 3.
!-- (3) (1), (2) 以外で 2. つまり, 両軸とも 2 なら 2 となる.
!-- 境界辺は 12 本あるので, 12 箇所設定.
!-- 各境界辺は以下のように x, y, z で指定.
!-- (1,1,0) -> z 軸に伸びる nx=1, ny=1 面を通る辺.

!-- [注意] : 1, 3 の場合は境界辺だけでなく, 境界点も同時に設定できるので
!--          ここで一緒に設定してしまう.

  if(bound(1:1)=='1')then  ! (1,1,0), (nx,1,0), (0,1,1), (0,1,nz) 辺
     ib(1,1,1:nz)=1
     ib(nx,1,1:nz)=1
     ib(1:nx,1,1)=1
     ib(1:nx,1,nz)=1
  end if
  if(bound(2:2)=='1')then  ! (1,1,0), (1,ny,0), (1,0,1), (1,0,nz) 辺
     ib(1,1,1:nz)=1
     ib(1,ny,1:nz)=1
     ib(1,1:ny,1)=1
     ib(1,1:ny,nz)=1
  end if
  if(bound(3:3)=='1')then  ! (1,ny,0), (nx,ny,0), (0,ny,1), (0,ny,nz) 辺
     ib(1,ny,1:nz)=1
     ib(nx,ny,1:nz)=1
     ib(1:nx,ny,1)=1
     ib(1:nx,ny,nz)=1
  end if
  if(bound(4:4)=='1')then  ! (nx,1,0), (nx,ny,0), (nx,0,1), (nx,0,nz) 辺
     ib(nx,1,1:nz)=1
     ib(nx,ny,1:nz)=1
     ib(nx,1:ny,1)=1
     ib(nx,1:ny,nz)=1
  end if
  if(bound(5:5)=='1')then  ! (0,1,1), (0,ny,1), (1,0,1), (nx,0,1) 辺
     ib(1:nx,1,1)=1
     ib(1:nx,ny,1)=1
     ib(1,1:ny,1)=1
     ib(nx,1:ny,1)=1
  end if
  if(bound(6:6)=='1')then  ! (0,1,nz), (0,ny,nz), (1,0,nz), (nx,0,nz) 辺
     ib(1:nx,1,nz)=1
     ib(1:nx,ny,nz)=1
     ib(1,1:ny,nz)=1
     ib(nx,1:ny,nz)=1
  end if

!-- 12 本とも周期境界の場合
  if(bound(1:1)=='3'.and.bound(2:2)=='3'.and.bound(5:5)=='3')then
     ib(1,1,1:nz)=3
     ib(1,ny,1:nz)=3
     ib(nx,1,1:nz)=3
     ib(nx,ny,1:nz)=3
     ib(1:nx,1,1)=3
     ib(1:nx,1,nz)=3
     ib(1:nx,ny,1)=3
     ib(1:nx,ny,nz)=3
     ib(1,1:ny,1)=3
     ib(nx,1:ny,1)=3
     ib(1,1:ny,nz)=3
     ib(nx,1:ny,nz)=3
  end if

!-- 12 辺のうち, いずれかで周期境界の場合, ib の値が設定されていなければ
!-- -3 を設定する. -3 は固定境界 1 より優先度が小さいので, ここで設定.
  if(bound(1:1)=='3'.or.bound(2:2)=='3'.or.bound(5:5)=='3')then
     do k=1,nz
        do i=1,nx
           if(ib(i,1,k)==0)then
              ib(i,1,k)=3
           end if
           if(ib(i,ny,k)==0)then
              ib(i,ny,k)=3
           end if
        end do
     end do
     do k=1,nz
        do j=1,ny
           if(ib(1,j,k)==0)then
              ib(1,j,k)=3
           end if
           if(ib(nx,j,k)==0)then
              ib(nx,j,k)=3
           end if
        end do
     end do
     do j=1,ny
        do i=1,nx
           if(ib(i,j,1)==0)then
              ib(i,j,1)=3
           end if
           if(ib(i,j,nz)==0)then
              ib(i,j,nz)=3
           end if
        end do
     end do
  end if

!-- 自由境界条件の設定
!-- [注意] : ここでは, 境界辺と境界点で設定値が異なるので,
!--          別々に設定する.
!-- 境界辺の設定には, ib = 0 であることを確認した後,
!-- 隣接 2 面に設定されている値の積となるように設定する.
!-- 符号も含めて設定する.
!-- 各面は符号を省いて 2, 4, 6 で設定されているので
!-- それぞれの面が隣接する 12 辺は
!-- 8, 12, 24 という値が符号つきで設定される.

  do k=2,nz-1
     if(ib(1,1,k)==0)then
        ib(1,1,k)=ib(2,1,k)*ib(1,2,k)
     end if
     if(ib(1,ny,k)==0)then
        ib(1,ny,k)=ib(2,ny,k)*ib(1,ny-1,k)
     end if
     if(ib(nx,1,k)==0)then
        ib(nx,1,k)=ib(nx-1,1,k)*ib(nx,2,k)
     end if
     if(ib(nx,ny,k)==0)then
        ib(nx,ny,k)=ib(nx-1,ny,k)*ib(nx,ny-1,k)
     end if
  end do

  do j=2,ny-1
     if(ib(1,j,1)==0)then
        ib(1,j,1)=ib(2,j,1)*ib(1,j,2)
     end if
     if(ib(nx,j,1)==0)then
        ib(nx,j,1)=ib(nx-1,j,1)*ib(nx,j,2)
     end if
     if(ib(1,j,nz)==0)then
        ib(1,j,nz)=ib(2,j,nz)*ib(1,j,nz-1)
     end if
     if(ib(nx,j,nz)==0)then
        ib(nx,j,nz)=ib(nx-1,j,nz)*ib(nx,j,nz-1)
     end if
  end do

  do i=2,nx-1
     if(ib(i,1,1)==0)then
        ib(i,1,1)=ib(i,1,2)*ib(i,2,1)
     end if
     if(ib(i,ny,1)==0)then
        ib(i,ny,1)=ib(i,ny,2)*ib(i,ny-1,1)
     end if
     if(ib(i,1,nz)==0)then
        ib(i,1,nz)=ib(i,1,nz-1)*ib(i,2,nz)
     end if
     if(ib(i,ny,nz)==0)then
        ib(i,ny,nz)=ib(i,ny,nz-1)*ib(i,ny-1,nz)
     end if
  end do

!-- 8 境界点の設定.
!-- 8 点しかないので, 以下の素数を各点に対応させる.
!-- (1,1,1) = 11
!-- (nx,1,1) = 13
!-- (1,ny,1) = 17
!-- (nx,ny,1) = 19
!-- (1,1,nz) = 23
!-- (nx,1,nz) = 29
!-- (1,ny,nz) = 31
!-- (nx,ny,nz) = 33

  if(ib(1,1,1)==0)then
     ib(1,1,1)=11
  end if
  if(ib(nx,1,1)==0)then
     ib(nx,1,1)=13
  end if
  if(ib(1,ny,1)==0)then
     ib(1,ny,1)=17
  end if
  if(ib(nx,ny,1)==0)then
     ib(nx,ny,1)=19
  end if
  if(ib(1,1,nz)==0)then
     ib(1,1,nz)=23
  end if
  if(ib(nx,1,nz)==0)then
     ib(nx,1,nz)=29
  end if
  if(ib(1,ny,nz)==0)then
     ib(1,ny,nz)=31
  end if
  if(ib(nx,ny,nz)==0)then
     ib(nx,ny,nz)=37
  end if

!-- 内部境界の設定

  if(present(inner_bound))then
     do k=2,nz-1
        do j=2,ny-1
           do i=2,nx-1
              ib(i,j,k)=inner_bound(i,j,k)
              if(ib(i,j,k)==0)then
                 inner_flag(i,j,k)=.false.
              end if
           end do
        end do
     end do
  else   ! 内部領域が設定されていない場合は内部全て計算する.
     do k=2,nz-1
        do j=2,ny-1
           do i=2,nx-1
              inner_flag(i,j,k)=.false.
           end do
        end do
     end do
  end if

end subroutine set_bound_3d


subroutine setval_bound_3df( ib, bnd, psi, bound_opt )
! 境界値を境界条件判別整数をもとに設定する.
  implicit none
  integer, intent(in) :: ib(:,:,:)  ! 境界条件判別整数
  real, intent(inout) :: bnd(size(ib,1),size(ib,2),size(ib,3))  ! 境界での値
  real, intent(inout) :: psi(size(ib,1),size(ib,2),size(ib,3))  ! 応答
  real, intent(in), optional :: bound_opt(size(ib,1),size(ib,2),size(ib,3))  ! 境界での値
  integer :: i, j, k, nx, ny, nz

  nx=size(ib,1)
  ny=size(ib,2)
  nz=size(ib,3)

!-- 境界値の設定
  if(present(bound_opt))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              bnd(i,j,k)=bound_opt(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              bnd(i,j,k)=0.0
           end do
        end do
     end do
  end if

!-- 境界条件の代入 "ib(i,j)==1 の場合のみ"
!-- 内部領域についてもここで代入してしまう.

  do k=1,nz
     do j=1,ny
        do i=1,nx
           if(ib(i,j,k)==1)then
              psi(i,j,k)=bnd(i,j,k)
           end if
        end do
     end do
  end do

end subroutine setval_bound_3df


subroutine setval_bound_3dd( ib, bnd, psi, bound_opt )
! 境界値を境界条件判別整数をもとに設定する.
  implicit none
  integer, intent(in) :: ib(:,:,:)  ! 境界条件判別整数
  double precision, intent(inout) :: bnd(size(ib,1),size(ib,2),size(ib,3))  ! 境界での値
  double precision, intent(inout) :: psi(size(ib,1),size(ib,2),size(ib,3))  ! 応答
  double precision, intent(in), optional :: bound_opt(size(ib,1),size(ib,2),size(ib,3))  ! 境界での値
  integer :: i, j, k, nx, ny, nz

  nx=size(ib,1)
  ny=size(ib,2)
  nz=size(ib,3)

!-- 境界値の設定
  if(present(bound_opt))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              bnd(i,j,k)=bound_opt(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              bnd(i,j,k)=0.0d0
           end do
        end do
     end do
  end if

!-- 境界条件の代入 "ib(i,j)==1 の場合のみ"
!-- 内部領域についてもここで代入してしまう.

  do k=1,nz
     do j=1,ny
        do i=1,nx
           if(ib(i,j,k)==1)then
              psi(i,j,k)=bnd(i,j,k)
           end if
        end do
     end do
  end do

end subroutine setval_bound_3dd


subroutine set_coe_3df( coe, ext, def )
! 2 次元配列に ext で指定された値もしくは def で指定された一定値を代入する.
! ext, def どちらも optional であるが, 必ずどちらかは指定されていないといけない.
  implicit none
  real, intent(inout) :: coe(:,:,:)  ! 代入される配列
  real, intent(in), optional :: ext(size(coe,1),size(coe,2),size(coe,3))  ! 代入する配列
  real, intent(in), optional :: def  ! 代入する一定値
  integer :: i, j, k, nx, ny, nz

  nx=size(coe,1)
  ny=size(coe,2)
  nz=size(coe,3)

  if(present(ext))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              coe(i,j,k)=ext(i,j,k)
           end do
        end do
     end do
  else if(present(def))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              coe(i,j,k)=def
           end do
        end do
     end do
  else
     write(*,*) "### ERROR ###"
     write(*,*) "subroutine set_coe must be set optional argument 'ext' or 'def'"
     write(*,*) "STOP."
     stop
  end if

end subroutine set_coe_3df


subroutine set_coe_3dd( coe, ext, def )
! 2 次元配列に ext で指定された値もしくは def で指定された一定値を代入する.
! ext, def どちらも optional であるが, 必ずどちらかは指定されていないといけない.
  implicit none
  double precision, intent(inout) :: coe(:,:,:)  ! 代入される配列
  double precision, intent(in), optional :: ext(size(coe,1),size(coe,2),size(coe,3))  ! 代入する配列
  double precision, intent(in), optional :: def  ! 代入する一定値
  integer :: i, j, k, nx, ny, nz

  nx=size(coe,1)
  ny=size(coe,2)
  nz=size(coe,3)

  if(present(ext))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              coe(i,j,k)=ext(i,j,k)
           end do
        end do
     end do
  else if(present(def))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              coe(i,j,k)=def
           end do
        end do
     end do
  else
     write(*,*) "### ERROR ###"
     write(*,*) "subroutine set_coe must be set optional argument 'ext' or 'def'"
     write(*,*) "STOP."
     stop
  end if

end subroutine set_coe_3dd


subroutine check_coe_3df( coe, aval, undeff )
! 3 次元配列に aval で指定された値が入っていないかを検出する.
  implicit none
  real, intent(inout) :: coe(:,:,:)  ! 代入される配列
  real, intent(in) :: aval           ! 検出される値
  real, intent(in), optional :: undeff
  integer :: i, j, k, nx, ny, nz

  nx=size(coe,1)
  ny=size(coe,2)
  nz=size(coe,3)

  if(present(undeff))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(coe(i,j,k)/=undeff)then
                 if(coe(i,j,k)==aval)then
                    write(*,*) "### ERROR (Ellip_Slv module) ###"
                    write(*,*) "Detect a certain value", aval, i, j, k
                    write(*,*) "STOP."
                    stop
                 end if
              end if
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(coe(i,j,k)==aval)then
                 write(*,*) "### ERROR (Ellip_Slv module) ###"
                 write(*,*) "Detect a certain value", aval, i, j, k
                 write(*,*) "STOP."
                 stop
              end if
           end do
        end do
     end do
  end if

end subroutine check_coe_3df


subroutine check_coe_3dd( coe, aval, undeff )
! 3 次元配列に aval で指定された値が入っていないかを検出する.
  implicit none
  double precision, intent(inout) :: coe(:,:,:)  ! 代入される配列
  double precision, intent(in) :: aval           ! 検出される値
  double precision, intent(in), optional :: undeff
  integer :: i, j, k, nx, ny, nz

  nx=size(coe,1)
  ny=size(coe,2)
  nz=size(coe,3)

  if(present(undeff))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(coe(i,j,k)/=undeff)then
                 if(coe(i,j,k)==aval)then
                    write(*,*) "### ERROR (Ellip_Slv module) ###"
                    write(*,*) "Detect a certain value", aval, i, j, k
                    write(*,*) "STOP."
                    stop
                 end if
              end if
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(coe(i,j,k)==aval)then
                 write(*,*) "### ERROR (Ellip_Slv module) ###"
                 write(*,*) "Detect a certain value", aval, i, j, k
                 write(*,*) "STOP."
                 stop
              end if
           end do
        end do
     end do
  end if

end subroutine check_coe_3dd


subroutine check_le_coe_3df( coe, aval, undeff )
! 3 次元配列に less equal aval を検出する.
  implicit none
  real, intent(inout) :: coe(:,:,:)  ! 代入される配列
  real, intent(in) :: aval           ! 検出される値
  real, intent(in), optional :: undeff
  integer :: i, j, k, nx, ny, nz

  nx=size(coe,1)
  ny=size(coe,2)
  nz=size(coe,3)

  if(present(undeff))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(coe(i,j,k)/=undeff)then
                 if(coe(i,j,k)<=aval)then
                    write(*,*) "### ERROR (Ellip_Slv module) ###"
                    write(*,*) "Detect a certain value", coe(i,j,k), i, j, k
                    write(*,*) "STOP."
                    stop
                 end if
              end if
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(coe(i,j,k)<=aval)then
                 write(*,*) "### ERROR (Ellip_Slv module) ###"
                 write(*,*) "Detect a certain value", coe(i,j,k), i, j, k
                 write(*,*) "STOP."
                 stop
              end if
           end do
        end do
     end do
  end if

end subroutine check_le_coe_3df


subroutine check_le_coe_3dd( coe, aval, undeff )
! 3 次元配列に less equal aval を検出する.
  implicit none
  double precision, intent(inout) :: coe(:,:,:)  ! 代入される配列
  double precision, intent(in) :: aval           ! 検出される値
  double precision, intent(in), optional :: undeff
  integer :: i, j, k, nx, ny, nz

  nx=size(coe,1)
  ny=size(coe,2)
  nz=size(coe,3)

  if(present(undeff))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(coe(i,j,k)/=undeff)then
                 if(coe(i,j,k)<=aval)then
                    write(*,*) "### ERROR (Ellip_Slv module) ###"
                    write(*,*) "Detect a certain value", coe(i,j,k), i, j, k
                    write(*,*) "STOP."
                    stop
                 end if
              end if
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(coe(i,j,k)<=aval)then
                 write(*,*) "### ERROR (Ellip_Slv module) ###"
                 write(*,*) "Detect a certain value", coe(i,j,k), i, j, k
                 write(*,*) "STOP."
                 stop
              end if
           end do
        end do
     end do
  end if

end subroutine check_le_coe_3dd


subroutine calculate_bound_3df( ib, dx, dy, dz, bnd, psi )
! ib 判別を元に, ノイマン型, 周期境界型について境界値を計算する.
  integer, intent(in) :: ib(:,:,:)  ! 境界整数判別
  real, intent(in) :: dx(size(ib,1))  ! x 方向の格子解像度
  real, intent(in) :: dy(size(ib,2))  ! y 方向の格子解像度
  real, intent(in) :: dz(size(ib,3))  ! z 方向の格子解像度
  real, intent(in) :: bnd(size(ib,1),size(ib,2),size(ib,3))
                                      ! 境界値(ノイマン型のみ使用)
  real, intent(inout) :: psi(size(ib,1),size(ib,2),size(ib,3))  ! 応答関数
  integer :: i, j, k, nx, ny, nz, ix, jy, kz

  nx=size(ib,1)
  ny=size(ib,2)
  nz=size(ib,3)

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k,ix,jy,kz)
  do k=1,nz
     do j=1,ny
        do i=1,nx
           if(ib(i,j,k)/=0)then
              select case (ib(i,j,k))
              case (1)
                 psi(i,j,k)=bnd(i,j,k)

              case (2)  ! x 方向にフラックス一定, 上側が参照値
                 psi(i,j,k)=psi(i+1,j,k)-bnd(i,j,k)*dx(i)

              case (-2)  ! x 方向にフラックス一定, 下側が参照値
                 psi(i,j,k)=psi(i-1,j,k)+bnd(i,j,k)*dx(i)

              case (4)  ! y 方向にフラックス一定, 右側が参照値
                 psi(i,j,k)=psi(i,j+1,k)-bnd(i,j,k)*dy(j)

              case (-4)  ! y 方向にフラックス一定, 左側が参照値
                 psi(i,j,k)=psi(i,j-1,k)+bnd(i,j,k)*dy(j)

              case (6)  ! y 方向にフラックス一定, 右側が参照値
                 psi(i,j,k)=psi(i,j,k+1)-bnd(i,j,k)*dz(k)

              case (-6)  ! y 方向にフラックス一定, 左側が参照値
                 psi(i,j,k)=psi(i,j,k-1)+bnd(i,j,k)*dz(k)

              case (3)  ! 12 辺, もしくは 8 点で周期境界を判断
                 if(i==1)then
                    ix=nx-1
                 else if(i==nx)then
                    ix=2
                 else
                    ix=i
                 end if
                 if(j==1)then
                    jy=ny-1
                 else if(j==ny)then
                    jy=2
                 else
                    jy=j
                 end if
                 if(k==1)then
                    kz=nz-1
                 else if(k==nz)then
                    kz=2
                 else
                    kz=k
                 end if
                 psi(i,j,k)=psi(ix,jy,kz)

              case (8)  ! 両方フラックス一定で z 面の x, y 右上か左下角.
                 if(i==1.and.j==1)then  ! -- 評価 1
                    psi(i,j,k)=psi(i+1,j+1,k)-0.5*(bnd(i+1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                 else if(i==nx.and.j==ny)then  ! -- 評価 2
                    psi(i,j,k)=psi(i-1,j-1,k)+0.5*(bnd(i-1,j,k)*dy(j)+bnd(i,j-1,k)*dx(i))

                 else if(ib(i-1,j,k)==10.and.ib(i,j-1,k)==10)then
                    ! -- 評価 1 と同じ
                    psi(i,j,k)=psi(i+1,j+1,k)-0.5*(bnd(i+1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                 else if(ib(i+1,j,k)==10.and.ib(i,j+1,k)==10)then
                    ! -- 評価 2 と同じ
                    psi(i,j,k)=psi(i-1,j-1,k)+0.5*(bnd(i-1,j,k)*dy(j)+bnd(i,j-1,k)*dx(i))

                 end if

              case (-8)  ! 両方フラックス一定で z 面の x, y 右下か左上角.
                 if(i==1.and.j==ny)then  ! -- 評価 1
                    psi(i,j,k)=psi(i+1,j-1,k)+0.5*(bnd(i+1,j,k)*dy(j)-bnd(i,j-1,k)*dx(i))

                 else if(i==nx.and.j==1)then  ! -- 評価 2
                    psi(i,j,k)=psi(i-1,j+1,k)+0.5*(-bnd(i-1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                 else if(ib(i-1,j,k)==10.and.ib(i,j+1,k)==10)then
                    ! -- 評価 1 と同じ
                    psi(i,j,k)=psi(i+1,j-1,k)+0.5*(bnd(i+1,j,k)*dy(j)-bnd(i,j-1,k)*dx(i))

                 else if(ib(i+1,j,k)==10.and.ib(i,j-1,k)==10)then
                    ! -- 評価 2 と同じ
                    psi(i,j,k)=psi(i-1,j+1,k)+0.5*(-bnd(i-1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))
                 end if

              case (12)  ! 両方フラックス一定で y 面の x, z 右上か左下角.
                 if(i==1.and.k==1)then  ! -- 評価 1
                    psi(i,j,k)=psi(i+1,j,k+1)-0.5*(bnd(i+1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                 else if(i==nx.and.k==nz)then  ! -- 評価 2
                    psi(i,j,k)=psi(i-1,j,k-1)+0.5*(bnd(i-1,j,k)*dz(k)+bnd(i,j,k-1)*dx(i))

                 else if(ib(i-1,j,k)==10.and.ib(i,j,k-1)==10)then
                    ! -- 評価 1 と同じ
                    psi(i,j,k)=psi(i+1,j,k+1)-0.5*(bnd(i+1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                 else if(ib(i+1,j,k)==10.and.ib(i,j,k+1)==10)then
                    ! -- 評価 2 と同じ
                    psi(i,j,k)=psi(i-1,j,k-1)+0.5*(bnd(i-1,j,k)*dz(k)+bnd(i,j,k-1)*dx(i))

                 end if

              case (-12)  ! 両方フラックス一定で y 面の x, z 右下か左上角.
                 if(i==1.and.k==nz)then  ! -- 評価 1
                    psi(i,j,k)=psi(i+1,j,k-1)+0.5*(bnd(i+1,j,k)*dz(k)-bnd(i,j,k-1)*dx(i))

                 else if(i==nx.and.k==1)then  ! -- 評価 2
                    psi(i,j,k)=psi(i-1,j,k+1)+0.5*(-bnd(i-1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                 else if(ib(i-1,j,k)==10.and.ib(i,j,k+1)==10)then
                    ! -- 評価 1 と同じ
                    psi(i,j,k)=psi(i+1,j,k-1)+0.5*(bnd(i+1,j,k)*dz(k)-bnd(i,j,k-1)*dx(i))

                 else if(ib(i+1,j,k)==10.and.ib(i,j,k-1)==10)then
                    ! -- 評価 2 と同じ
                    psi(i,j,k)=psi(i-1,j,k+1)+0.5*(-bnd(i-1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                 end if

              case (24)  ! 両方フラックス一定で x 面の y, z 右上か左下角.
                 if(j==1.and.k==1)then  ! -- 評価 1
                    psi(i,j,k)=psi(i,j+1,k+1)-0.5*(bnd(i,j+1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                 else if(j==ny.and.k==nz)then  ! -- 評価 2
                    psi(i,j,k)=psi(i,j-1,k-1)+0.5*(bnd(i,j-1,k)*dz(k)+bnd(i,j,k-1)*dy(j))

                 else if(ib(i,j-1,k)==10.and.ib(i,j,k-1)==10)then
                    ! -- 評価 1 と同じ
                    psi(i,j,k)=psi(i,j+1,k+1)-0.5*(bnd(i,j+1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                 else if(ib(i,j+1,k)==10.and.ib(i,j,k+1)==10)then
                    ! -- 評価 2 と同じ
                    psi(i,j,k)=psi(i,j-1,k-1)+0.5*(bnd(i,j-1,k)*dz(k)+bnd(i,j,k-1)*dy(j))

                 end if

              case (-24)  ! 両方フラックス一定で x 面の y, z 右下か左上角.
                 if(j==1.and.k==nz)then  ! -- 評価 1
                    psi(i,j,k)=psi(i,j+1,k-1)+0.5*(bnd(i,j+1,k)*dz(k)-bnd(i,j,k-1)*dy(j))

                 else if(j==ny.and.k==1)then  ! -- 評価 2
                    psi(i,j,k)=psi(i,j-1,k+1)+0.5*(-bnd(i,j-1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                 else if(ib(i,j-1,k)==10.and.ib(i,j,k+1)==10)then
                    ! -- 評価 1 と同じ
                    psi(i,j,k)=psi(i,j+1,k-1)+0.5*(bnd(i,j+1,k)*dz(k)-bnd(i,j,k-1)*dy(j))

                 else if(ib(i,j+1,k)==10.and.ib(i,j,k-1)==10)then
                    ! -- 評価 2 と同じ
                    psi(i,j,k)=psi(i,j-1,k+1)+0.5*(-bnd(i,j-1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                 end if

              !-- 以降, 隅領域なので, 個別に設定.
              case (11)  ! 両方フラックス一定で (1,1,1) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i+1,j+1,k+1)  &
  &                        -(bnd(i,j+1,k+1)*dx(i)  &
  &                         +bnd(i+1,j,k+1)*dy(j)  &
  &                         +bnd(i+1,j+1,k)*dz(k))/3.0

              case (13)  ! 両方フラックス一定で (nx,1,1) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i-1,j+1,k+1)  &
  &                        -(-bnd(i,j+1,k+1)*dx(i)  &
  &                          +bnd(i-1,j,k+1)*dy(j)  &
  &                          +bnd(i-1,j+1,k)*dz(k))/3.0

              case (17)  ! 両方フラックス一定で (1,ny,1) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i+1,j-1,k+1)  &
  &                        -(bnd(i,j-1,k+1)*dx(i)  &
  &                         -bnd(i+1,j,k+1)*dy(j)  &
  &                         +bnd(i+1,j-1,k)*dz(k))/3.0

              case (19)  ! 両方フラックス一定で (nx,ny,1) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i-1,j-1,k+1)  &
  &                        -(-bnd(i,j-1,k+1)*dx(i)  &
  &                          -bnd(i-1,j,k+1)*dy(j)  &
  &                          +bnd(i-1,j-1,k)*dz(k))/3.0

              case (23)  ! 両方フラックス一定で (1,1,nz) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i+1,j+1,k-1)  &
  &                        -(bnd(i,j+1,k-1)*dx(i)  &
  &                         +bnd(i+1,j,k-1)*dy(j)  &
  &                         -bnd(i+1,j+1,k)*dz(k))/3.0

              case (29)  ! 両方フラックス一定で (nx,1,nz) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i-1,j+1,k-1)  &
  &                        -(-bnd(i,j+1,k-1)*dx(i)  &
  &                          +bnd(i-1,j,k-1)*dy(j)  &
  &                          -bnd(i-1,j+1,k)*dz(k))/3.0

              case (31)  ! 両方フラックス一定で (1,ny,nz) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i+1,j-1,k-1)  &
  &                        -(bnd(i,j-1,k-1)*dx(i)  &
  &                         -bnd(i+1,j,k-1)*dy(j)  &
  &                         -bnd(i+1,j-1,k)*dz(k))/3.0

              case (37)  ! 両方フラックス一定で (nx,ny,nz) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i-1,j-1,k-1)  &
  &                        -(-bnd(i,j-1,k-1)*dx(i)  &
  &                          -bnd(i-1,j,k-1)*dy(j)  &
  &                          -bnd(i-1,j-1,k)*dz(k))/3.0

              case (-11)  ! 両方フラックス一定で (i+1,j+1,k+1) に 10 が設定.
                 psi(i,j,k)=(psi(i-1,j+1,k+1)+psi(i+1,j-1,k+1)+psi(i+1,j+1,k-1)  &
  &                         +bnd(i,j+1,k+1)*dx(i)+bnd(i+1,j,k+1)*dy(j)  &
  &                         +bnd(i+1,j+1,k)*dz(k))/3.0

              case (-13)  ! 両方フラックス一定で (i-1,j+1,k+1) に 10 が設定.
                 psi(i,j,k)=(psi(i+1,j+1,k+1)+psi(i-1,j-1,k+1)+psi(i-1,j+1,k-1)  &
  &                         -bnd(i,j+1,k+1)*dx(i)+bnd(i-1,j,k+1)*dy(j)  &
  &                         +bnd(i-1,j+1,k)*dz(k))/3.0

              case (-17)  ! 両方フラックス一定で (i+1,j-1,k+1) に 10 が設定.
                 psi(i,j,k)=(psi(i-1,j-1,k+1)+psi(i+1,j+1,k+1)+psi(i+1,j-1,k-1)  &
  &                         +bnd(i,j-1,k+1)*dx(i)-bnd(i+1,j,k+1)*dy(j)  &
  &                         +bnd(i+1,j-1,k)*dz(k))/3.0

              case (-19)  ! 両方フラックス一定で (i-1,j-1,k+1) に 10 が設定.
                 psi(i,j,k)=(psi(i+1,j-1,k+1)+psi(i-1,j+1,k+1)+psi(i-1,j-1,k-1)  &
  &                         -bnd(i,j-1,k+1)*dx(i)-bnd(i-1,j,k+1)*dy(j)  &
  &                         +bnd(i-1,j-1,k)*dz(k))/3.0

              case (-23)  ! 両方フラックス一定で (i+1,j+1,k-1) に 10 が設定.
                 psi(i,j,k)=(psi(i-1,j+1,k-1)+psi(i+1,j-1,k-1)+psi(i+1,j+1,k+1)  &
  &                         +bnd(i,j+1,k-1)*dx(i)+bnd(i+1,j,k-1)*dy(j)  &
  &                         -bnd(i+1,j+1,k)*dz(k))/3.0

              case (-29)  ! 両方フラックス一定で (i-1,j+1,k-1) に 10 が設定.
                 psi(i,j,k)=(psi(i+1,j+1,k-1)+psi(i-1,j-1,k-1)+psi(i-1,j+1,k+1)  &
  &                         -bnd(i,j+1,k-1)*dx(i)+bnd(i-1,j,k-1)*dy(j)  &
  &                         -bnd(i-1,j+1,k)*dz(k))/3.0

              case (-31)  ! 両方フラックス一定で (i+1,j-1,k-1) に 10 が設定.
                 psi(i,j,k)=(psi(i-1,j-1,k-1)+psi(i+1,j+1,k-1)+psi(i+1,j-1,k+1)  &
  &                         +bnd(i,j-1,k-1)*dx(i)-bnd(i+1,j,k-1)*dy(j)  &
  &                         -bnd(i+1,j-1,k)*dz(k))/3.0

              case (-37)  ! 両方フラックス一定で (i-1,j-1,k-1) に 10 が設定.
                 psi(i,j,k)=(psi(i+1,j-1,k-1)+psi(i-1,j+1,k-1)+psi(i-1,j-1,k+1)  &
  &                         -bnd(i,j-1,k-1)*dx(i)-bnd(i-1,j,k-1)*dy(j)  &
  &                         -bnd(i-1,j-1,k)*dz(k))/3.0

              end select

           end if
        end do
     end do
  end do
!$omp end do
!$omp end parallel

end subroutine calculate_bound_3df


subroutine calculate_bound_3dd( ib, dx, dy, dz, bnd, psi )
! ib 判別を元に, ノイマン型, 周期境界型について境界値を計算する.
  integer, intent(in) :: ib(:,:,:)  ! 境界整数判別
  double precision, intent(in) :: dx(size(ib,1))  ! x 方向の格子解像度
  double precision, intent(in) :: dy(size(ib,2))  ! y 方向の格子解像度
  double precision, intent(in) :: dz(size(ib,3))  ! z 方向の格子解像度
  double precision, intent(in) :: bnd(size(ib,1),size(ib,2),size(ib,3))
                                      ! 境界値(ノイマン型のみ使用)
  double precision, intent(inout) :: psi(size(ib,1),size(ib,2),size(ib,3))  ! 応答関数
  integer :: i, j, k, nx, ny, nz, ix, jy, kz

  nx=size(ib,1)
  ny=size(ib,2)
  nz=size(ib,3)

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k,ix,jy,kz)
  do k=1,nz
     do j=1,ny
        do i=1,nx
           if(ib(i,j,k)/=0)then
              select case (ib(i,j,k))
              case (1)
                 psi(i,j,k)=bnd(i,j,k)

              case (2)  ! x 方向にフラックス一定, 上側が参照値
                 psi(i,j,k)=psi(i+1,j,k)-bnd(i,j,k)*dx(i)

              case (-2)  ! x 方向にフラックス一定, 下側が参照値
                 psi(i,j,k)=psi(i-1,j,k)+bnd(i,j,k)*dx(i)

              case (4)  ! y 方向にフラックス一定, 右側が参照値
                 psi(i,j,k)=psi(i,j+1,k)-bnd(i,j,k)*dy(j)

              case (-4)  ! y 方向にフラックス一定, 左側が参照値
                 psi(i,j,k)=psi(i,j-1,k)+bnd(i,j,k)*dy(j)

              case (6)  ! y 方向にフラックス一定, 右側が参照値
                 psi(i,j,k)=psi(i,j,k+1)-bnd(i,j,k)*dz(k)

              case (-6)  ! y 方向にフラックス一定, 左側が参照値
                 psi(i,j,k)=psi(i,j,k-1)+bnd(i,j,k)*dz(k)

              case (3)  ! 12 辺, もしくは 8 点で周期境界を判断
                 if(i==1)then
                    ix=nx-1
                 else if(i==nx)then
                    ix=2
                 else
                    ix=i
                 end if
                 if(j==1)then
                    jy=ny-1
                 else if(j==ny)then
                    jy=2
                 else
                    jy=j
                 end if
                 if(k==1)then
                    kz=nz-1
                 else if(k==nz)then
                    kz=2
                 else
                    kz=k
                 end if
                 psi(i,j,k)=psi(ix,jy,kz)

              case (8)  ! 両方フラックス一定で z 面の x, y 右上か左下角.
                 if(i==1.and.j==1)then  ! -- 評価 1
                    psi(i,j,k)=psi(i+1,j+1,k)-0.5d0*(bnd(i+1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                 else if(i==nx.and.j==ny)then  ! -- 評価 2
                    psi(i,j,k)=psi(i-1,j-1,k)+0.5d0*(bnd(i-1,j,k)*dy(j)+bnd(i,j-1,k)*dx(i))

                 else if(ib(i-1,j,k)==10.and.ib(i,j-1,k)==10)then
                    ! -- 評価 1 と同じ
                    psi(i,j,k)=psi(i+1,j+1,k)-0.5d0*(bnd(i+1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                 else if(ib(i+1,j,k)==10.and.ib(i,j+1,k)==10)then
                    ! -- 評価 2 と同じ
                    psi(i,j,k)=psi(i-1,j-1,k)+0.5d0*(bnd(i-1,j,k)*dy(j)+bnd(i,j-1,k)*dx(i))

                 end if

              case (-8)  ! 両方フラックス一定で z 面の x, y 右下か左上角.
                 if(i==1.and.j==ny)then  ! -- 評価 1
                    psi(i,j,k)=psi(i+1,j-1,k)+0.5d0*(bnd(i+1,j,k)*dy(j)-bnd(i,j-1,k)*dx(i))

                 else if(i==nx.and.j==1)then  ! -- 評価 2
                    psi(i,j,k)=psi(i-1,j+1,k)+0.5d0*(-bnd(i-1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                 else if(ib(i-1,j,k)==10.and.ib(i,j+1,k)==10)then
                    ! -- 評価 1 と同じ
                    psi(i,j,k)=psi(i+1,j-1,k)+0.5d0*(bnd(i+1,j,k)*dy(j)-bnd(i,j-1,k)*dx(i))

                 else if(ib(i+1,j,k)==10.and.ib(i,j-1,k)==10)then
                    ! -- 評価 2 と同じ
                    psi(i,j,k)=psi(i-1,j+1,k)+0.5d0*(-bnd(i-1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))
                 end if

              case (12)  ! 両方フラックス一定で y 面の x, z 右上か左下角.
                 if(i==1.and.k==1)then  ! -- 評価 1
                    psi(i,j,k)=psi(i+1,j,k+1)-0.5d0*(bnd(i+1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                 else if(i==nx.and.k==nz)then  ! -- 評価 2
                    psi(i,j,k)=psi(i-1,j,k-1)+0.5d0*(bnd(i-1,j,k)*dz(k)+bnd(i,j,k-1)*dx(i))

                 else if(ib(i-1,j,k)==10.and.ib(i,j,k-1)==10)then
                    ! -- 評価 1 と同じ
                    psi(i,j,k)=psi(i+1,j,k+1)-0.5d0*(bnd(i+1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                 else if(ib(i+1,j,k)==10.and.ib(i,j,k+1)==10)then
                    ! -- 評価 2 と同じ
                    psi(i,j,k)=psi(i-1,j,k-1)+0.5d0*(bnd(i-1,j,k)*dz(k)+bnd(i,j,k-1)*dx(i))

                 end if

              case (-12)  ! 両方フラックス一定で y 面の x, z 右下か左上角.
                 if(i==1.and.k==nz)then  ! -- 評価 1
                    psi(i,j,k)=psi(i+1,j,k-1)+0.5d0*(bnd(i+1,j,k)*dz(k)-bnd(i,j,k-1)*dx(i))

                 else if(i==nx.and.k==1)then  ! -- 評価 2
                    psi(i,j,k)=psi(i-1,j,k+1)+0.5d0*(-bnd(i-1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                 else if(ib(i-1,j,k)==10.and.ib(i,j,k+1)==10)then
                    ! -- 評価 1 と同じ
                    psi(i,j,k)=psi(i+1,j,k-1)+0.5d0*(bnd(i+1,j,k)*dz(k)-bnd(i,j,k-1)*dx(i))

                 else if(ib(i+1,j,k)==10.and.ib(i,j,k-1)==10)then
                    ! -- 評価 2 と同じ
                    psi(i,j,k)=psi(i-1,j,k+1)+0.5d0*(-bnd(i-1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                 end if

              case (24)  ! 両方フラックス一定で x 面の y, z 右上か左下角.
                 if(j==1.and.k==1)then  ! -- 評価 1
                    psi(i,j,k)=psi(i,j+1,k+1)-0.5d0*(bnd(i,j+1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                 else if(j==ny.and.k==nz)then  ! -- 評価 2
                    psi(i,j,k)=psi(i,j-1,k-1)+0.5d0*(bnd(i,j-1,k)*dz(k)+bnd(i,j,k-1)*dy(j))

                 else if(ib(i,j-1,k)==10.and.ib(i,j,k-1)==10)then
                    ! -- 評価 1 と同じ
                    psi(i,j,k)=psi(i,j+1,k+1)-0.5d0*(bnd(i,j+1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                 else if(ib(i,j+1,k)==10.and.ib(i,j,k+1)==10)then
                    ! -- 評価 2 と同じ
                    psi(i,j,k)=psi(i,j-1,k-1)+0.5d0*(bnd(i,j-1,k)*dz(k)+bnd(i,j,k-1)*dy(j))

                 end if

              case (-24)  ! 両方フラックス一定で x 面の y, z 右下か左上角.
                 if(j==1.and.k==nz)then  ! -- 評価 1
                    psi(i,j,k)=psi(i,j+1,k-1)+0.5d0*(bnd(i,j+1,k)*dz(k)-bnd(i,j,k-1)*dy(j))

                 else if(j==ny.and.k==1)then  ! -- 評価 2
                    psi(i,j,k)=psi(i,j-1,k+1)+0.5d0*(-bnd(i,j-1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                 else if(ib(i,j-1,k)==10.and.ib(i,j,k+1)==10)then
                    ! -- 評価 1 と同じ
                    psi(i,j,k)=psi(i,j+1,k-1)+0.5d0*(bnd(i,j+1,k)*dz(k)-bnd(i,j,k-1)*dy(j))

                 else if(ib(i,j+1,k)==10.and.ib(i,j,k-1)==10)then
                    ! -- 評価 2 と同じ
                    psi(i,j,k)=psi(i,j-1,k+1)+0.5d0*(-bnd(i,j-1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                 end if

              !-- 以降, 隅領域なので, 個別に設定.
              case (11)  ! 両方フラックス一定で (1,1,1) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i+1,j+1,k+1)  &
  &                        -(bnd(i,j+1,k+1)*dx(i)  &
  &                         +bnd(i+1,j,k+1)*dy(j)  &
  &                         +bnd(i+1,j+1,k)*dz(k))/3.0d0

              case (13)  ! 両方フラックス一定で (nx,1,1) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i-1,j+1,k+1)  &
  &                        -(-bnd(i,j+1,k+1)*dx(i)  &
  &                          +bnd(i-1,j,k+1)*dy(j)  &
  &                          +bnd(i-1,j+1,k)*dz(k))/3.0d0

              case (17)  ! 両方フラックス一定で (1,ny,1) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i+1,j-1,k+1)  &
  &                        -(bnd(i,j-1,k+1)*dx(i)  &
  &                         -bnd(i+1,j,k+1)*dy(j)  &
  &                         +bnd(i+1,j-1,k)*dz(k))/3.0d0

              case (19)  ! 両方フラックス一定で (nx,ny,1) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i-1,j-1,k+1)  &
  &                        -(-bnd(i,j-1,k+1)*dx(i)  &
  &                          -bnd(i-1,j,k+1)*dy(j)  &
  &                          +bnd(i-1,j-1,k)*dz(k))/3.0d0

              case (23)  ! 両方フラックス一定で (1,1,nz) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i+1,j+1,k-1)  &
  &                        -(bnd(i,j+1,k-1)*dx(i)  &
  &                         +bnd(i+1,j,k-1)*dy(j)  &
  &                         -bnd(i+1,j+1,k)*dz(k))/3.0d0

              case (29)  ! 両方フラックス一定で (nx,1,nz) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i-1,j+1,k-1)  &
  &                        -(-bnd(i,j+1,k-1)*dx(i)  &
  &                          +bnd(i-1,j,k-1)*dy(j)  &
  &                          -bnd(i-1,j+1,k)*dz(k))/3.0d0

              case (31)  ! 両方フラックス一定で (1,ny,nz) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i+1,j-1,k-1)  &
  &                        -(bnd(i,j-1,k-1)*dx(i)  &
  &                         -bnd(i+1,j,k-1)*dy(j)  &
  &                         -bnd(i+1,j-1,k)*dz(k))/3.0d0

              case (37)  ! 両方フラックス一定で (nx,ny,nz) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i-1,j-1,k-1)  &
  &                        -(-bnd(i,j-1,k-1)*dx(i)  &
  &                          -bnd(i-1,j,k-1)*dy(j)  &
  &                          -bnd(i-1,j-1,k)*dz(k))/3.0d0

              case (-11)  ! 両方フラックス一定で (i+1,j+1,k+1) に 10 が設定.
                 psi(i,j,k)=(psi(i-1,j+1,k+1)+psi(i+1,j-1,k+1)+psi(i+1,j+1,k-1)  &
  &                         +bnd(i,j+1,k+1)*dx(i)+bnd(i+1,j,k+1)*dy(j)  &
  &                         +bnd(i+1,j+1,k)*dz(k))/3.0d0

              case (-13)  ! 両方フラックス一定で (i-1,j+1,k+1) に 10 が設定.
                 psi(i,j,k)=(psi(i+1,j+1,k+1)+psi(i-1,j-1,k+1)+psi(i-1,j+1,k-1)  &
  &                         -bnd(i,j+1,k+1)*dx(i)+bnd(i-1,j,k+1)*dy(j)  &
  &                         +bnd(i-1,j+1,k)*dz(k))/3.0d0

              case (-17)  ! 両方フラックス一定で (i+1,j-1,k+1) に 10 が設定.
                 psi(i,j,k)=(psi(i-1,j-1,k+1)+psi(i+1,j+1,k+1)+psi(i+1,j-1,k-1)  &
  &                         +bnd(i,j-1,k+1)*dx(i)-bnd(i+1,j,k+1)*dy(j)  &
  &                         +bnd(i+1,j-1,k)*dz(k))/3.0d0

              case (-19)  ! 両方フラックス一定で (i-1,j-1,k+1) に 10 が設定.
                 psi(i,j,k)=(psi(i+1,j-1,k+1)+psi(i-1,j+1,k+1)+psi(i-1,j-1,k-1)  &
  &                         -bnd(i,j-1,k+1)*dx(i)-bnd(i-1,j,k+1)*dy(j)  &
  &                         +bnd(i-1,j-1,k)*dz(k))/3.0d0

              case (-23)  ! 両方フラックス一定で (i+1,j+1,k-1) に 10 が設定.
                 psi(i,j,k)=(psi(i-1,j+1,k-1)+psi(i+1,j-1,k-1)+psi(i+1,j+1,k+1)  &
  &                         +bnd(i,j+1,k-1)*dx(i)+bnd(i+1,j,k-1)*dy(j)  &
  &                         -bnd(i+1,j+1,k)*dz(k))/3.0d0

              case (-29)  ! 両方フラックス一定で (i-1,j+1,k-1) に 10 が設定.
                 psi(i,j,k)=(psi(i+1,j+1,k-1)+psi(i-1,j-1,k-1)+psi(i-1,j+1,k+1)  &
  &                         -bnd(i,j+1,k-1)*dx(i)+bnd(i-1,j,k-1)*dy(j)  &
  &                         -bnd(i-1,j+1,k)*dz(k))/3.0d0

              case (-31)  ! 両方フラックス一定で (i+1,j-1,k-1) に 10 が設定.
                 psi(i,j,k)=(psi(i-1,j-1,k-1)+psi(i+1,j+1,k-1)+psi(i+1,j-1,k+1)  &
  &                         +bnd(i,j-1,k-1)*dx(i)-bnd(i+1,j,k-1)*dy(j)  &
  &                         -bnd(i+1,j-1,k)*dz(k))/3.0d0

              case (-37)  ! 両方フラックス一定で (i-1,j-1,k-1) に 10 が設定.
                 psi(i,j,k)=(psi(i+1,j-1,k-1)+psi(i-1,j+1,k-1)+psi(i-1,j-1,k+1)  &
  &                         -bnd(i,j-1,k-1)*dx(i)-bnd(i-1,j,k-1)*dy(j)  &
  &                         -bnd(i-1,j-1,k)*dz(k))/3.0d0

              end select

           end if
        end do
     end do
  end do
!$omp end do
!$omp end parallel

end subroutine calculate_bound_3dd


real function check_diff_error_2df( x, y, psi, rho, at, bt, ct, dt, et, ft, ib )
  implicit none
  real, intent(in) :: x(:)
  real, intent(in) :: y(:)
  real, intent(in) :: psi(size(x),size(y))
  real, intent(in) :: rho(size(x),size(y))
  real, intent(in) :: at(size(x),size(y))
  real, intent(in) :: bt(size(x),size(y))
  real, intent(in) :: ct(size(x),size(y))
  real, intent(in) :: dt(size(x),size(y))
  real, intent(in) :: et(size(x),size(y))
  real, intent(in) :: ft(size(x),size(y))
  integer, intent(in) :: ib(size(x),size(y))
  integer :: nx, ny, i, j
  real, parameter :: undef=-999.0
  real :: emax
  real, dimension(size(x),size(y)) :: dp2dx2, dp2dy2, dpdx, dpdy, dp2dxy
  real, dimension(size(x),size(y)) :: xi, errmax

  nx=size(x)
  ny=size(y)
  emax=0.0

  xi=psi

  do j=1,ny
     do i=1,nx
        if(ib(i,j)==10)then
           xi(i,j)=undef
        end if
     end do
  end do

  call grad_2d( x, y, xi, dpdx, dpdy, undeff=undef )

  do j=1,ny
     call laplacian_1d( x, xi(:,j), dp2dx2(:,j), undef=undef )
     call grad_1d( x, dpdy(:,j), dp2dxy(:,j), undef=undef )
  end do
  do i=1,nx
     call laplacian_1d( y, xi(i,:), dp2dy2(i,:), undef=undef )
  end do

  do j=1,ny
     do i=1,nx
        if(xi(i,j)/=undef)then
           errmax(i,j)=at(i,j)*dp2dx2(i,j)  &
  &                   +ct(i,j)*dp2dy2(i,j)  &
  &                   +bt(i,j)*dp2dxy(i,j)  &
  &                   +dt(i,j)*dpdx(i,j)  &
  &                   +et(i,j)*dpdy(i,j)  &
  &                   +ft(i,j)*xi(i,j)  &
  &                   -rho(i,j)
           errmax(i,j)=abs(errmax(i,j))
        else
           errmax(i,j)=0.0
        end if
        if(emax<errmax(i,j))then
           emax=errmax(i,j)
        end if
     end do
  end do

  check_diff_error_2df=emax

  return

end function check_diff_error_2df


double precision function check_diff_error_2dd( x, y, psi, rho, at, bt, ct, dt, et, ft, ib )
  implicit none
  double precision, intent(in) :: x(:)
  double precision, intent(in) :: y(:)
  double precision, intent(in) :: psi(size(x),size(y))
  double precision, intent(in) :: rho(size(x),size(y))
  double precision, intent(in) :: at(size(x),size(y))
  double precision, intent(in) :: bt(size(x),size(y))
  double precision, intent(in) :: ct(size(x),size(y))
  double precision, intent(in) :: dt(size(x),size(y))
  double precision, intent(in) :: et(size(x),size(y))
  double precision, intent(in) :: ft(size(x),size(y))
  integer, intent(in) :: ib(size(x),size(y))
  integer :: nx, ny, i, j
  double precision, parameter :: undef=-999.0
  double precision :: emax
  double precision, dimension(size(x),size(y)) :: dp2dx2, dp2dy2, dpdx, dpdy, dp2dxy
  double precision, dimension(size(x),size(y)) :: xi, errmax

  nx=size(x)
  ny=size(y)
  emax=0.0

  xi=psi

  do j=1,ny
     do i=1,nx
        if(ib(i,j)==10)then
           xi(i,j)=undef
        end if
     end do
  end do

  call grad_2d( x, y, xi, dpdx, dpdy, undeff=undef )

  do j=1,ny
     call laplacian_1d( x, xi(:,j), dp2dx2(:,j), undef=undef )
     call grad_1d( x, dpdy(:,j), dp2dxy(:,j), undef=undef )
  end do
  do i=1,nx
     call laplacian_1d( y, xi(i,:), dp2dy2(i,:), undef=undef )
  end do

  do j=1,ny
     do i=1,nx
        if(xi(i,j)/=undef)then
           errmax(i,j)=at(i,j)*dp2dx2(i,j)  &
  &                   +ct(i,j)*dp2dy2(i,j)  &
  &                   +bt(i,j)*dp2dxy(i,j)  &
  &                   +dt(i,j)*dpdx(i,j)  &
  &                   +et(i,j)*dpdy(i,j)  &
  &                   +ft(i,j)*xi(i,j)  &
  &                   -rho(i,j)
           errmax(i,j)=abs(errmax(i,j))
        else
           errmax(i,j)=0.0
        end if
        if(emax<errmax(i,j))then
           emax=errmax(i,j)
        end if
     end do
  end do

  check_diff_error_2dd=emax

  return

end function check_diff_error_2dd


subroutine forward_interpo_2df( ival, oval, ilev, olev )
  ! interpolation of fine grid (ilev) to coarse grid (olev)
  implicit none
  real, intent(in) :: ival(:,:)
  real, intent(out) :: oval(:,:)
  integer, intent(in) :: ilev  ! fine grid level
  integer, intent(in) :: olev  ! coarse grid level
  integer :: ii, jj, sk, nxi, nyi, nxo, nyo

  if(olev>ilev)then
     write(*,*) "*** ERROR (forward_interpo_2d) ***: olev < ilev. stop."
     stop
  end if

  sk=2**(ilev-olev)

  nxi=size(ival,1)
  nyi=size(ival,2)
  nxo=size(oval,1)
  nyo=size(oval,2)

  if(ilev==olev)then

     oval(1:nxo,1:nyo)=ival(1:nxi,1:nyi)

  else

     do jj=1,nyo
        do ii=1,nxo
           oval(ii,jj)=ival(1+sk*(ii-1),1+sk*(jj-1))
        end do
     end do

  end if

end subroutine forward_interpo_2df


subroutine forward_interpo_2dd( ival, oval, ilev, olev )
  ! interpolation of fine grid (ilev) to coarse grid (olev)
  implicit none
  double precision, intent(in) :: ival(:,:)
  double precision, intent(out) :: oval(:,:)
  integer, intent(in) :: ilev  ! fine grid level
  integer, intent(in) :: olev  ! coarse grid level
  integer :: ii, jj, sk, nxi, nyi, nxo, nyo

  if(olev>ilev)then
     write(*,*) "*** ERROR (forward_interpo_2d) ***: olev < ilev. stop."
     stop
  end if

  sk=2**(ilev-olev)

  nxi=size(ival,1)
  nyi=size(ival,2)
  nxo=size(oval,1)
  nyo=size(oval,2)

  if(ilev==olev)then

     oval(1:nxo,1:nyo)=ival(1:nxi,1:nyi)

  else

     do jj=1,nyo
        do ii=1,nxo
           oval(ii,jj)=ival(1+sk*(ii-1),1+sk*(jj-1))
        end do
     end do

  end if

end subroutine forward_interpo_2dd


subroutine forward_interpo_3df( ival, oval, ilev, ilevz, olev )
  ! interpolation of fine grid (ilev) to coarse grid (olev)
  implicit none
  real, intent(in) :: ival(:,:,:)
  real, intent(out) :: oval(:,:,:)
  integer, intent(in) :: ilev      ! fine grid level
  integer, intent(in) :: ilevz(:)  ! fine grid level in vertical
  integer, intent(in) :: olev      ! coarse grid level
  integer :: ii, jj, kk, sk, skz, nxi, nyi, nzi, nxo, nyo, nzo

  if(olev>ilev)then
     write(*,*) "*** ERROR (forward_interpo_3d) ***: olev < ilev. stop."
     stop
  end if

  sk=2**(ilev-olev)
  skz=2**(ilevz(ilev)-ilevz(olev))

  nxi=size(ival,1)
  nyi=size(ival,2)
  nzi=size(ival,3)
  nxo=size(oval,1)
  nyo=size(oval,2)
  nzo=size(oval,3)

  if(ilev==olev)then

     oval(1:nxo,1:nyo,1:nzo)=ival(1:nxi,1:nyi,1:nzi)

  else

     do kk=1,nzo
        do jj=1,nyo
           do ii=1,nxo
              oval(ii,jj,kk)=ival(1+sk*(ii-1),1+sk*(jj-1),1+skz*(kk-1))
           end do
        end do
     end do

  end if

end subroutine forward_interpo_3df


subroutine forward_interpo_3dd( ival, oval, ilev, ilevz, olev )
  ! interpolation of fine grid (ilev) to coarse grid (olev)
  implicit none
  double precision, intent(in) :: ival(:,:,:)
  double precision, intent(out) :: oval(:,:,:)
  integer, intent(in) :: ilev      ! fine grid level
  integer, intent(in) :: ilevz(:)  ! fine grid level in vertical
  integer, intent(in) :: olev      ! coarse grid level
  integer :: ii, jj, kk, sk, skz, nxi, nyi, nzi, nxo, nyo, nzo

  if(olev>ilev)then
     write(*,*) "*** ERROR (forward_interpo_3d) ***: olev < ilev. stop."
     stop
  end if

  sk=2**(ilev-olev)
  skz=2**(ilevz(ilev)-ilevz(olev))

  nxi=size(ival,1)
  nyi=size(ival,2)
  nzi=size(ival,3)
  nxo=size(oval,1)
  nyo=size(oval,2)
  nzo=size(oval,3)

  if(ilev==olev)then

     oval(1:nxo,1:nyo,1:nzo)=ival(1:nxi,1:nyi,1:nzi)

  else

!write(*,*) "skz check", sk, skz, ilevz(ilev), ilevz(olev), nzi, nzo
     do kk=1,nzo
        do jj=1,nyo
           do ii=1,nxo
              oval(ii,jj,kk)=ival(1+sk*(ii-1),1+sk*(jj-1),1+skz*(kk-1))
           end do
        end do
     end do

  end if

end subroutine forward_interpo_3dd


subroutine backward_interpo_2df( ival, oval, ilev, olev )
  ! interpolation of coarse grid (ilev) to fine grid (olev)
  implicit none
  real, intent(in) :: ival(:,:)
  real, intent(out) :: oval(:,:)
  integer, intent(in) :: ilev  ! fine grid level
  integer, intent(in) :: olev  ! coarse grid level
  integer :: ii, jj, sk, nxi, nyi, nxo, nyo
  integer :: ijj, iii
  real :: alp, bet

  if(olev<ilev)then
     write(*,*) "*** ERROR (backward_interpo_2d) ***: olev > ilev. stop."
     stop
  end if

  sk=2**(olev-ilev)

  nxi=size(ival,1)
  nyi=size(ival,2)
  nxo=size(oval,1)
  nyo=size(oval,2)

  if(olev==ilev)then

     oval(1:nxo,1:nyo)=ival(1:nxi,1:nyi)

  else

     do jj=1,nyo-1
        bet=real(mod(jj-1,sk))/real(sk)
        ijj=(jj-1)/sk+1
        do ii=1,nxo-1
           alp=real(mod(ii-1,sk))/real(sk)
           iii=(ii-1)/sk+1
           oval(ii,jj)=(1.0-bet)*(1.0-alp)*ival(iii,ijj)  &
  &                   +(1.0-bet)*alp*ival(iii+1,ijj)  &
  &                   +bet*(1.0-alp)*ival(iii,ijj+1)  &
  &                   +bet*alp*ival(iii+1,ijj+1)
        end do
     end do

     !-- on nyo
     ijj=(nyo-1)/sk+1
     do ii=1,nxo-1
        alp=real(mod(ii-1,sk))/real(sk)
        iii=(ii-1)/sk+1
        oval(ii,nyo)=(1.0-alp)*ival(iii,ijj)  &
  &                  +alp*ival(iii+1,ijj)
     end do

     !-- on nxo
     iii=(nxo-1)/sk+1
     do jj=1,nyo-1
        bet=real(mod(jj-1,sk))/real(sk)
        ijj=(jj-1)/sk+1
        oval(nxo,jj)=(1.0-bet)*ival(iii,ijj)  &
  &                  +bet*ival(iii,ijj+1)
     end do

     !-- on NE corner
     oval(nxo,nyo)=ival(nxi,nyi)

  end if

end subroutine backward_interpo_2df


subroutine backward_interpo_2dd( ival, oval, ilev, olev )
  ! interpolation of coarse grid (ilev) to fine grid (olev)
  implicit none
  double precision, intent(in) :: ival(:,:)
  double precision, intent(out) :: oval(:,:)
  integer, intent(in) :: ilev  ! fine grid level
  integer, intent(in) :: olev  ! coarse grid level
  integer :: ii, jj, sk, nxi, nyi, nxo, nyo
  integer :: ijj, iii
  double precision :: alp, bet

  if(olev<ilev)then
     write(*,*) "*** ERROR (backward_interpo_2d) ***: olev > ilev. stop."
     stop
  end if

  sk=2**(olev-ilev)

  nxi=size(ival,1)
  nyi=size(ival,2)
  nxo=size(oval,1)
  nyo=size(oval,2)

  if(olev==ilev)then

     oval(1:nxo,1:nyo)=ival(1:nxi,1:nyi)

  else

     do jj=1,nyo-1
        bet=dble(mod(jj-1,sk))/dble(sk)
        ijj=(jj-1)/sk+1
        do ii=1,nxo-1
           alp=dble(mod(ii-1,sk))/dble(sk)
           iii=(ii-1)/sk+1
           oval(ii,jj)=(1.0-bet)*(1.0-alp)*ival(iii,ijj)  &
  &                   +(1.0-bet)*alp*ival(iii+1,ijj)  &
  &                   +bet*(1.0-alp)*ival(iii,ijj+1)  &
  &                   +bet*alp*ival(iii+1,ijj+1)
        end do
     end do

     !-- on nyo
     ijj=(nyo-1)/sk+1
     do ii=1,nxo-1
        alp=dble(mod(ii-1,sk))/dble(sk)
        iii=(ii-1)/sk+1
        oval(ii,nyo)=(1.0-alp)*ival(iii,ijj)  &
  &                  +alp*ival(iii+1,ijj)
     end do

     !-- on nxo
     iii=(nxo-1)/sk+1
     do jj=1,nyo-1
        bet=dble(mod(jj-1,sk))/dble(sk)
        ijj=(jj-1)/sk+1
        oval(nxo,jj)=(1.0-bet)*ival(iii,ijj)  &
  &                  +bet*ival(iii,ijj+1)
     end do

     !-- on NE corner
     oval(nxo,nyo)=ival(nxi,nyi)

  end if

end subroutine backward_interpo_2dd


subroutine backward_interpo_3df( ival, oval, ilev, ilevz, olev )
  ! interpolation of coarse grid (ilev) to fine grid (olev)
  implicit none
  real, intent(in) :: ival(:,:,:)
  real, intent(out) :: oval(:,:,:)
  integer, intent(in) :: ilev  ! fine grid level
  integer, intent(in) :: ilevz(:)  ! fine grid level in vertical
  integer, intent(in) :: olev  ! coarse grid level
  integer :: ii, jj, kk, sk, skz, nxi, nyi, nzi, nxo, nyo, nzo
  integer :: ijj, iii, ikk
  real :: alp, bet, gam

  if(olev<ilev)then
     write(*,*) "*** ERROR (backward_interpo_3d) ***: olev > ilev. stop."
     stop
  end if
  if(olev>size(ilevz))then
     write(*,*) "*** ERROR (backward_interpo_3d) ***: olev <= size(ilevz). stop."
  end if

  sk=2**(olev-ilev)
  skz=2**(ilevz(olev)-ilevz(ilev))

  nxi=size(ival,1)
  nyi=size(ival,2)
  nzi=size(ival,3)
  nxo=size(oval,1)
  nyo=size(oval,2)
  nzo=size(oval,3)

  if(olev==ilev)then

     oval(1:nxo,1:nyo,1:nzo)=ival(1:nxi,1:nyi,1:nzi)

  else

     do kk=1,nzo-1
        gam=real(mod(kk-1,skz))/real(skz)
        ikk=(kk-1)/skz+1
        do jj=1,nyo-1
           bet=real(mod(jj-1,sk))/real(sk)
           ijj=(jj-1)/sk+1
           do ii=1,nxo-1
              alp=real(mod(ii-1,sk))/real(sk)
              iii=(ii-1)/sk+1
              oval(ii,jj,kk)=(1.0-gam)*(1.0-bet)*(1.0-alp)*ival(iii,ijj,ikk)  &
  &                         +(1.0-gam)*(1.0-bet)*alp*ival(iii+1,ijj,ikk)  &
  &                         +(1.0-gam)*bet*(1.0-alp)*ival(iii,ijj+1,ikk)  &
  &                         +gam*(1.0-bet)*(1.0-alp)*ival(iii,ijj,ikk+1)  &
  &                         +(1.0-gam)*bet*alp*ival(iii+1,ijj+1,ikk)  &
  &                         +gam*(1.0-bet)*alp*ival(iii+1,ijj,ikk+1)  &
  &                         +gam*bet*(1.0-alp)*ival(iii,ijj+1,ikk+1)  &
  &                         +gam*bet*alp*ival(iii+1,ijj+1,ikk+1)
           end do
        end do
     end do

     !-- on nzo
     ikk=(nzo-1)/skz+1
     do jj=1,nyo-1
        bet=real(mod(jj-1,sk))/real(sk)
        ijj=(jj-1)/sk+1
        do ii=1,nxo-1
           alp=real(mod(ii-1,sk))/real(sk)
           iii=(ii-1)/sk+1
           oval(ii,jj,kk)=(1.0-bet)*(1.0-alp)*ival(iii,ijj,ikk)  &
  &                      +(1.0-bet)*alp*ival(iii+1,ijj,ikk)  &
  &                      +bet*(1.0-alp)*ival(iii,ijj+1,ikk)  &
  &                      +bet*alp*ival(iii+1,ijj+1,ikk)
        end do
     end do

     !-- on nyo
     ijj=(nyo-1)/sk+1
     do kk=1,nzo-1
        gam=real(mod(kk-1,skz))/real(skz)
        ikk=(kk-1)/skz+1
        do ii=1,nxo-1
           alp=real(mod(ii-1,sk))/real(sk)
           iii=(ii-1)/sk+1
           oval(ii,jj,kk)=(1.0-gam)*(1.0-alp)*ival(iii,ijj,ikk)  &
  &                      +(1.0-gam)*alp*ival(iii+1,ijj,ikk)  &
  &                      +gam*(1.0-alp)*ival(iii,ijj,ikk+1)  &
  &                      +gam*alp*ival(iii+1,ijj,ikk+1)
        end do
     end do

     !-- on nxo
     iii=(nxo-1)/sk+1
     do kk=1,nzo-1
        gam=real(mod(kk-1,skz))/real(skz)
        ikk=(kk-1)/skz+1
        do jj=1,nyo-1
           bet=real(mod(jj-1,sk))/real(sk)
           ijj=(jj-1)/sk+1
           oval(ii,jj,kk)=(1.0-gam)*(1.0-bet)*ival(iii,ijj,ikk)  &
  &                      +(1.0-gam)*bet*ival(iii,ijj+1,ikk)  &
  &                      +gam*(1.0-bet)*ival(iii,ijj,ikk+1)  &
  &                      +gam*bet*ival(iii,ijj+1,ikk+1)
        end do
     end do

     !-- on nxo and nyo
     iii=(nxo-1)/sk+1
     ijj=(nyo-1)/sk+1
     do kk=1,nzo-1
        gam=real(mod(kk-1,skz))/real(skz)
        ikk=(kk-1)/skz+1
        oval(ii,jj,kk)=(1.0-gam)*ival(iii,ijj,ikk)  &
  &                   +gam*ival(iii,ijj,ikk+1)
     end do

     !-- on nxo and nzo
     ikk=(nzo-1)/skz+1
     iii=(nxo-1)/sk+1
     do jj=1,nyo-1
        bet=real(mod(jj-1,sk))/real(sk)
        ijj=(jj-1)/sk+1
        oval(ii,jj,kk)=(1.0-bet)*ival(iii,ijj,ikk)  &
  &                   +bet*ival(iii,ijj+1,ikk)
     end do

     !-- on nyo and nzo
     ikk=(nzo-1)/skz+1
     ijj=(nyo-1)/sk+1
     do ii=1,nxo-1
        alp=real(mod(ii-1,sk))/real(sk)
        iii=(ii-1)/sk+1
        oval(ii,jj,kk)=(1.0-alp)*ival(iii,ijj,ikk)  &
  &                   +alp*ival(iii+1,ijj,ikk)
     end do

     !-- on NE corner
     oval(nxo,nyo,nzo)=ival(nxi,nyi,nzi)

  end if

end subroutine backward_interpo_3df


subroutine backward_interpo_3dd( ival, oval, ilev, ilevz, olev )
  ! interpolation of coarse grid (ilev) to fine grid (olev)
  implicit none
  double precision, intent(in) :: ival(:,:,:)
  double precision, intent(out) :: oval(:,:,:)
  integer, intent(in) :: ilev  ! fine grid level
  integer, intent(in) :: ilevz(:)  ! fine grid level in vertical
  integer, intent(in) :: olev  ! coarse grid level
  integer :: ii, jj, kk, sk, skz, nxi, nyi, nzi, nxo, nyo, nzo
  integer :: ijj, iii, ikk
  double precision :: alp, bet, gam

  if(olev<ilev)then
     write(*,*) "*** ERROR (backward_interpo_3d) ***: olev > ilev. stop."
     stop
  end if
  if(olev>size(ilevz))then
     write(*,*) "*** ERROR (backward_interpo_3d) ***: olev <= size(ilevz). stop."
  end if

  sk=2**(olev-ilev)
  skz=2**(ilevz(olev)-ilevz(ilev))

  nxi=size(ival,1)
  nyi=size(ival,2)
  nzi=size(ival,3)
  nxo=size(oval,1)
  nyo=size(oval,2)
  nzo=size(oval,3)

  if(olev==ilev)then

     oval(1:nxo,1:nyo,1:nzo)=ival(1:nxi,1:nyi,1:nzi)

  else

     do kk=1,nzo-1
        gam=dble(mod(kk-1,skz))/dble(skz)
        ikk=(kk-1)/skz+1
        do jj=1,nyo-1
           bet=dble(mod(jj-1,sk))/dble(sk)
           ijj=(jj-1)/sk+1
           do ii=1,nxo-1
              alp=dble(mod(ii-1,sk))/dble(sk)
              iii=(ii-1)/sk+1
              oval(ii,jj,kk)=(1.0d0-gam)*(1.0d0-bet)*(1.0d0-alp)*ival(iii,ijj,ikk)  &
  &                         +(1.0d0-gam)*(1.0d0-bet)*alp*ival(iii+1,ijj,ikk)  &
  &                         +(1.0d0-gam)*bet*(1.0d0-alp)*ival(iii,ijj+1,ikk)  &
  &                         +gam*(1.0d0-bet)*(1.0d0-alp)*ival(iii,ijj,ikk+1)  &
  &                         +(1.0d0-gam)*bet*alp*ival(iii+1,ijj+1,ikk)  &
  &                         +gam*(1.0d0-bet)*alp*ival(iii+1,ijj,ikk+1)  &
  &                         +gam*bet*(1.0d0-alp)*ival(iii,ijj+1,ikk+1)  &
  &                         +gam*bet*alp*ival(iii+1,ijj+1,ikk+1)
           end do
        end do
     end do

     !-- on nzo
     ikk=(nzo-1)/skz+1
     do jj=1,nyo-1
        bet=dble(mod(jj-1,sk))/dble(sk)
        ijj=(jj-1)/sk+1
        do ii=1,nxo-1
           alp=dble(mod(ii-1,sk))/dble(sk)
           iii=(ii-1)/sk+1
           oval(ii,jj,kk)=(1.0d0-bet)*(1.0d0-alp)*ival(iii,ijj,ikk)  &
  &                      +(1.0d0-bet)*alp*ival(iii+1,ijj,ikk)  &
  &                      +bet*(1.0d0-alp)*ival(iii,ijj+1,ikk)  &
  &                      +bet*alp*ival(iii+1,ijj+1,ikk)
        end do
     end do

     !-- on nyo
     ijj=(nyo-1)/sk+1
     do kk=1,nzo-1
        gam=dble(mod(kk-1,skz))/dble(skz)
        ikk=(kk-1)/skz+1
        do ii=1,nxo-1
           alp=dble(mod(ii-1,sk))/dble(sk)
           iii=(ii-1)/sk+1
           oval(ii,jj,kk)=(1.0d0-gam)*(1.0d0-alp)*ival(iii,ijj,ikk)  &
  &                      +(1.0d0-gam)*alp*ival(iii+1,ijj,ikk)  &
  &                      +gam*(1.0d0-alp)*ival(iii,ijj,ikk+1)  &
  &                      +gam*alp*ival(iii+1,ijj,ikk+1)
        end do
     end do

     !-- on nxo
     iii=(nxo-1)/sk+1
     do kk=1,nzo-1
        gam=dble(mod(kk-1,skz))/dble(skz)
        ikk=(kk-1)/skz+1
        do jj=1,nyo-1
           bet=dble(mod(jj-1,sk))/dble(sk)
           ijj=(jj-1)/sk+1
           oval(ii,jj,kk)=(1.0d0-gam)*(1.0d0-bet)*ival(iii,ijj,ikk)  &
  &                      +(1.0d0-gam)*bet*ival(iii,ijj+1,ikk)  &
  &                      +gam*(1.0d0-bet)*ival(iii,ijj,ikk+1)  &
  &                      +gam*bet*ival(iii,ijj+1,ikk+1)
        end do
     end do

     !-- on nxo and nyo
     iii=(nxo-1)/sk+1
     ijj=(nyo-1)/sk+1
     do kk=1,nzo-1
        gam=dble(mod(kk-1,skz))/dble(skz)
        ikk=(kk-1)/skz+1
        oval(ii,jj,kk)=(1.0d0-gam)*ival(iii,ijj,ikk)  &
  &                   +gam*ival(iii,ijj,ikk+1)
     end do

     !-- on nxo and nzo
     ikk=(nzo-1)/skz+1
     iii=(nxo-1)/sk+1
     do jj=1,nyo-1
        bet=dble(mod(jj-1,sk))/dble(sk)
        ijj=(jj-1)/sk+1
        oval(ii,jj,kk)=(1.0d0-bet)*ival(iii,ijj,ikk)  &
  &                   +bet*ival(iii,ijj+1,ikk)
     end do

     !-- on nyo and nzo
     ikk=(nzo-1)/skz+1
     ijj=(nyo-1)/sk+1
     do ii=1,nxo-1
        alp=dble(mod(ii-1,sk))/dble(sk)
        iii=(ii-1)/sk+1
        oval(ii,jj,kk)=(1.0d0-alp)*ival(iii,ijj,ikk)  &
  &                   +alp*ival(iii+1,ijj,ikk)
     end do

     !-- on NE corner
     oval(nxo,nyo,nzo)=ival(nxi,nyi,nzi)

  end if

end subroutine backward_interpo_3dd


subroutine calc_error_2df( xl, yl, ul, fl, error, a, b, c, d, e, f )
!-- Calculation error for f^l - L^lu^l
  implicit none
  real, intent(in) :: xl(:)                  ! X grid on the level grid
  real, intent(in) :: yl(:)                  ! Y grid on the level grid
  real, intent(in) :: ul(size(xl),size(yl))  ! The initial guess for u on the level grid
  real, intent(in) :: fl(size(xl),size(yl))  ! The forcing on the level grid
  real, intent(out) :: error(size(xl),size(yl)) ! The error for fl - Ll(ul)
  real, intent(in), optional :: a(size(xl),size(yl))  ! coefficient in PDE
  real, intent(in), optional :: b(size(xl),size(yl))  ! coefficient in PDE
  real, intent(in), optional :: c(size(xl),size(yl))  ! coefficient in PDE
  real, intent(in), optional :: d(size(xl),size(yl))  ! coefficient in PDE
  real, intent(in), optional :: e(size(xl),size(yl))  ! coefficient in PDE
  real, intent(in), optional :: f(size(xl),size(yl))  ! coefficient in PDE
  integer :: ii, jj, nxl, nyl
  real :: dxi, dyi
  real, dimension(size(xl),size(yl)) :: at, bt, ct, dt, et, ft

  !-- Set and Allocate variables
  nxl=size(xl)
  nyl=size(yl)
  dxi=1.0/(xl(2)-xl(1))
  dyi=1.0/(yl(2)-yl(1))

  at=1.0
  ct=1.0
  bt=0.0
  dt=0.0
  et=0.0
  ft=0.0

  if(present(a))then
     at(1:nxl,1:nyl)=a(1:nxl,1:nyl)
  end if
  if(present(b))then
     bt(1:nxl,1:nyl)=b(1:nxl,1:nyl)
  end if
  if(present(c))then
     ct(1:nxl,1:nyl)=c(1:nxl,1:nyl)
  end if
  if(present(d))then
     dt(1:nxl,1:nyl)=d(1:nxl,1:nyl)
  end if
  if(present(e))then
     et(1:nxl,1:nyl)=e(1:nxl,1:nyl)
  end if
  if(present(f))then
     ft(1:nxl,1:nyl)=f(1:nxl,1:nyl)
  end if

  error=0.0

  do jj=2,nyl-1
     do ii=2,nxl-1
        error(ii,jj)=fl(ii,jj)  &
  &                 -at(ii,jj)*(ul(ii+1,jj)+ul(ii-1,jj)-2.0*ul(ii,jj))*dxi*dxi  &
  &                 -ct(ii,jj)*(ul(ii,jj+1)+ul(ii,jj-1)-2.0*ul(ii,jj))*dyi*dyi  &
  &                 -bt(ii,jj)*(ul(ii+1,jj+1)+ul(ii-1,jj-1)  &
  &                           -(ul(ii-1,jj+1)+ul(ii+1,jj-1)))*0.25*dxi*dyi  &
  &                 -dt(ii,jj)*(ul(ii+1,jj)-ul(ii-1,jj))*0.5*dxi  &
  &                 -et(ii,jj)*(ul(ii,jj+1)-ul(ii,jj-1))*0.5*dyi  &
  &                 -ft(ii,jj)*ul(ii,jj)
     end do
  end do

end subroutine calc_error_2df


subroutine calc_error_2dd( xl, yl, ul, fl, error, a, b, c, d, e, f )
!-- Calculation error for f^l - L^lu^l
  implicit none
  double precision, intent(in) :: xl(:)                  ! X grid on the level grid
  double precision, intent(in) :: yl(:)                  ! Y grid on the level grid
  double precision, intent(in) :: ul(size(xl),size(yl))  ! The initial guess for u on the level grid
  double precision, intent(in) :: fl(size(xl),size(yl))  ! The forcing on the level grid
  double precision, intent(out) :: error(size(xl),size(yl)) ! The error for fl - Ll(ul)
  double precision, intent(in), optional :: a(size(xl),size(yl))  ! coefficient in PDE
  double precision, intent(in), optional :: b(size(xl),size(yl))  ! coefficient in PDE
  double precision, intent(in), optional :: c(size(xl),size(yl))  ! coefficient in PDE
  double precision, intent(in), optional :: d(size(xl),size(yl))  ! coefficient in PDE
  double precision, intent(in), optional :: e(size(xl),size(yl))  ! coefficient in PDE
  double precision, intent(in), optional :: f(size(xl),size(yl))  ! coefficient in PDE
  integer :: ii, jj, nxl, nyl
  double precision :: dxi, dyi
  double precision, dimension(size(xl),size(yl)) :: at, bt, ct, dt, et, ft

  !-- Set and Allocate variables
  nxl=size(xl)
  nyl=size(yl)
  dxi=1.0d0/(xl(2)-xl(1))
  dyi=1.0d0/(yl(2)-yl(1))

  at=1.0d0
  ct=1.0d0
  bt=0.0d0
  dt=0.0d0
  et=0.0d0
  ft=0.0d0

  if(present(a))then
     at(1:nxl,1:nyl)=a(1:nxl,1:nyl)
  end if
  if(present(b))then
     bt(1:nxl,1:nyl)=b(1:nxl,1:nyl)
  end if
  if(present(c))then
     ct(1:nxl,1:nyl)=c(1:nxl,1:nyl)
  end if
  if(present(d))then
     dt(1:nxl,1:nyl)=d(1:nxl,1:nyl)
  end if
  if(present(e))then
     et(1:nxl,1:nyl)=e(1:nxl,1:nyl)
  end if
  if(present(f))then
     ft(1:nxl,1:nyl)=f(1:nxl,1:nyl)
  end if

  error=0.0

  do jj=2,nyl-1
     do ii=2,nxl-1
        error(ii,jj)=fl(ii,jj)  &
  &                 -at(ii,jj)*(ul(ii+1,jj)+ul(ii-1,jj)-2.0d0*ul(ii,jj))*dxi*dxi  &
  &                 -ct(ii,jj)*(ul(ii,jj+1)+ul(ii,jj-1)-2.0d0*ul(ii,jj))*dyi*dyi  &
  &                 -bt(ii,jj)*(ul(ii+1,jj+1)+ul(ii-1,jj-1)  &
  &                           -(ul(ii-1,jj+1)+ul(ii+1,jj-1)))*0.25d0*dxi*dyi  &
  &                 -dt(ii,jj)*(ul(ii+1,jj)-ul(ii-1,jj))*0.5d0*dxi  &
  &                 -et(ii,jj)*(ul(ii,jj+1)-ul(ii,jj-1))*0.5d0*dyi  &
  &                 -ft(ii,jj)*ul(ii,jj)
     end do
  end do

end subroutine calc_error_2dd


subroutine calc_error_3df( xl, yl, zl, ul, fl, error, xa, ya, za,  &
  &                        a, b, c, d, e, f, g )
!-- Calculation error for f^l - L^lu^l
  implicit none
  real, intent(in) :: xl(:)                  ! X grid on the level grid
  real, intent(in) :: yl(:)                  ! Y grid on the level grid
  real, intent(in) :: zl(:)                  ! Z grid on the level grid
  real, intent(in) :: ul(size(xl),size(yl),size(zl))  ! The initial guess for u on the level grid
  real, intent(in) :: fl(size(xl),size(yl),size(zl))  ! The forcing on the level grid
  real, intent(out) :: error(size(xl),size(yl),size(zl)) ! The error for fl - Ll(ul)
  real, intent(in), optional :: xa(size(xl),size(yl),size(zl))  ! coefficient in PDE
  real, intent(in), optional :: ya(size(xl),size(yl),size(zl))  ! coefficient in PDE
  real, intent(in), optional :: za(size(xl),size(yl),size(zl))  ! coefficient in PDE
  real, intent(in), optional :: a(size(xl),size(yl),size(zl))  ! coefficient in PDE
  real, intent(in), optional :: b(size(xl),size(yl),size(zl))  ! coefficient in PDE
  real, intent(in), optional :: c(size(xl),size(yl),size(zl))  ! coefficient in PDE
  real, intent(in), optional :: d(size(xl),size(yl),size(zl))  ! coefficient in PDE
  real, intent(in), optional :: e(size(xl),size(yl),size(zl))  ! coefficient in PDE
  real, intent(in), optional :: f(size(xl),size(yl),size(zl))  ! coefficient in PDE
  real, intent(in), optional :: g(size(xl),size(yl),size(zl))  ! coefficient in PDE
  integer :: ii, jj, kk, nxl, nyl, nzl
  real :: dxi, dyi, dzi
  real, dimension(size(xl),size(yl),size(zl)) :: xat, yat, zat, at, bt, ct, dt, et, ft, gt

  !-- Set and Allocate variables
  nxl=size(xl)
  nyl=size(yl)
  nzl=size(zl)
  dxi=1.0/(xl(2)-xl(1))
  dyi=1.0/(yl(2)-yl(1))
  dzi=1.0/(zl(2)-zl(1))

  xat=1.0
  yat=1.0
  zat=1.0
  at=0.0
  bt=0.0
  ct=0.0
  dt=0.0
  et=0.0
  ft=0.0
  gt=0.0

  if(present(xa))then
     xat(1:nxl,1:nyl,1:nzl)=xa(1:nxl,1:nyl,1:nzl)
  end if
  if(present(ya))then
     yat(1:nxl,1:nyl,1:nzl)=ya(1:nxl,1:nyl,1:nzl)
  end if
  if(present(za))then
     zat(1:nxl,1:nyl,1:nzl)=za(1:nxl,1:nyl,1:nzl)
  end if
  if(present(a))then
     at(1:nxl,1:nyl,1:nzl)=a(1:nxl,1:nyl,1:nzl)
  end if
  if(present(b))then
     bt(1:nxl,1:nyl,1:nzl)=b(1:nxl,1:nyl,1:nzl)
  end if
  if(present(c))then
     ct(1:nxl,1:nyl,1:nzl)=c(1:nxl,1:nyl,1:nzl)
  end if
  if(present(d))then
     dt(1:nxl,1:nyl,1:nzl)=d(1:nxl,1:nyl,1:nzl)
  end if
  if(present(e))then
     et(1:nxl,1:nyl,1:nzl)=e(1:nxl,1:nyl,1:nzl)
  end if
  if(present(f))then
     ft(1:nxl,1:nyl,1:nzl)=f(1:nxl,1:nyl,1:nzl)
  end if
  if(present(g))then
     gt(1:nxl,1:nyl,1:nzl)=g(1:nxl,1:nyl,1:nzl)
  end if

  error=0.0

  do kk=2,nzl-1
     do jj=2,nyl-1
        do ii=2,nxl-1
           error(ii,jj,kk)=fl(ii,jj,kk)  &
  &                       -xat(ii,jj,kk)*(ul(ii+1,jj,kk)+ul(ii-1,jj,kk)-2.0*ul(ii,jj,kk))*dxi*dxi  &
  &                       -yat(ii,jj,kk)*(ul(ii,jj+1,kk)+ul(ii,jj-1,kk)-2.0*ul(ii,jj,kk))*dyi*dyi  &
  &                       -zat(ii,jj,kk)*(ul(ii,jj,kk+1)+ul(ii,jj,kk-1)-2.0*ul(ii,jj,kk))*dzi*dzi  &
  &                       -at(ii,jj,kk)*(ul(ii+1,jj+1,kk)+ul(ii-1,jj-1,kk)  &
  &                                    -(ul(ii-1,jj+1,kk)+ul(ii+1,jj-1,kk)))*0.25*dxi*dyi  &
  &                       -bt(ii,jj,kk)*(ul(ii,jj+1,kk+1)+ul(ii,jj-1,kk-1)  &
  &                                    -(ul(ii,jj-1,kk+1)+ul(ii,jj+1,kk-1)))*0.25*dyi*dzi  &
  &                       -ct(ii,jj,kk)*(ul(ii+1,jj,kk+1)+ul(ii-1,jj,kk-1)  &
  &                                    -(ul(ii+1,jj,kk-1)+ul(ii-1,jj,kk+1)))*0.25*dzi*dxi  &
  &                       -dt(ii,jj,kk)*(ul(ii+1,jj,kk)-ul(ii-1,jj,kk))*0.5*dxi  &
  &                       -et(ii,jj,kk)*(ul(ii,jj+1,kk)-ul(ii,jj-1,kk))*0.5*dyi  &
  &                       -ft(ii,jj,kk)*(ul(ii,jj,kk+1)-ul(ii,jj,kk-1))*0.5*dzi  &
  &                       -gt(ii,jj,kk)*ul(ii,jj,kk)
        end do
     end do
  end do

end subroutine calc_error_3df


subroutine calc_error_3dd( xl, yl, zl, ul, fl, error, xa, ya, za,  &
  &                        a, b, c, d, e, f, g )
!-- Calculation error for f^l - L^lu^l
  implicit none
  double precision, intent(in) :: xl(:)                  ! X grid on the level grid
  double precision, intent(in) :: yl(:)                  ! Y grid on the level grid
  double precision, intent(in) :: zl(:)                  ! Z grid on the level grid
  double precision, intent(in) :: ul(size(xl),size(yl),size(zl))  ! The initial guess for u on the level grid
  double precision, intent(in) :: fl(size(xl),size(yl),size(zl))  ! The forcing on the level grid
  double precision, intent(out) :: error(size(xl),size(yl),size(zl)) ! The error for fl - Ll(ul)
  double precision, intent(in), optional :: xa(size(xl),size(yl),size(zl))  ! coefficient in PDE
  double precision, intent(in), optional :: ya(size(xl),size(yl),size(zl))  ! coefficient in PDE
  double precision, intent(in), optional :: za(size(xl),size(yl),size(zl))  ! coefficient in PDE
  double precision, intent(in), optional :: a(size(xl),size(yl),size(zl))  ! coefficient in PDE
  double precision, intent(in), optional :: b(size(xl),size(yl),size(zl))  ! coefficient in PDE
  double precision, intent(in), optional :: c(size(xl),size(yl),size(zl))  ! coefficient in PDE
  double precision, intent(in), optional :: d(size(xl),size(yl),size(zl))  ! coefficient in PDE
  double precision, intent(in), optional :: e(size(xl),size(yl),size(zl))  ! coefficient in PDE
  double precision, intent(in), optional :: f(size(xl),size(yl),size(zl))  ! coefficient in PDE
  double precision, intent(in), optional :: g(size(xl),size(yl),size(zl))  ! coefficient in PDE
  integer :: ii, jj, kk, nxl, nyl, nzl
  double precision :: dxi, dyi, dzi
  double precision, dimension(size(xl),size(yl),size(zl)) :: xat, yat, zat, at, bt, ct, dt, et, ft, gt

  !-- Set and Allocate variables
  nxl=size(xl)
  nyl=size(yl)
  nzl=size(zl)
  dxi=1.0d0/(xl(2)-xl(1))
  dyi=1.0d0/(yl(2)-yl(1))
  dzi=1.0d0/(zl(2)-zl(1))

  xat=1.0d0
  yat=1.0d0
  zat=1.0d0
  at=0.0d0
  bt=0.0d0
  ct=0.0d0
  dt=0.0d0
  et=0.0d0
  ft=0.0d0
  gt=0.0d0

  if(present(xa))then
     xat(1:nxl,1:nyl,1:nzl)=xa(1:nxl,1:nyl,1:nzl)
  end if
  if(present(ya))then
     yat(1:nxl,1:nyl,1:nzl)=ya(1:nxl,1:nyl,1:nzl)
  end if
  if(present(za))then
     zat(1:nxl,1:nyl,1:nzl)=za(1:nxl,1:nyl,1:nzl)
  end if
  if(present(a))then
     at(1:nxl,1:nyl,1:nzl)=a(1:nxl,1:nyl,1:nzl)
  end if
  if(present(b))then
     bt(1:nxl,1:nyl,1:nzl)=b(1:nxl,1:nyl,1:nzl)
  end if
  if(present(c))then
     ct(1:nxl,1:nyl,1:nzl)=c(1:nxl,1:nyl,1:nzl)
  end if
  if(present(d))then
     dt(1:nxl,1:nyl,1:nzl)=d(1:nxl,1:nyl,1:nzl)
  end if
  if(present(e))then
     et(1:nxl,1:nyl,1:nzl)=e(1:nxl,1:nyl,1:nzl)
  end if
  if(present(f))then
     ft(1:nxl,1:nyl,1:nzl)=f(1:nxl,1:nyl,1:nzl)
  end if
  if(present(g))then
     gt(1:nxl,1:nyl,1:nzl)=g(1:nxl,1:nyl,1:nzl)
  end if

  error=0.0d0

  do kk=2,nzl-1
     do jj=2,nyl-1
        do ii=2,nxl-1
           error(ii,jj,kk)=fl(ii,jj,kk)  &
  &                       -xat(ii,jj,kk)*(ul(ii+1,jj,kk)+ul(ii-1,jj,kk)-2.0d0*ul(ii,jj,kk))*dxi*dxi  &
  &                       -yat(ii,jj,kk)*(ul(ii,jj+1,kk)+ul(ii,jj-1,kk)-2.0d0*ul(ii,jj,kk))*dyi*dyi  &
  &                       -zat(ii,jj,kk)*(ul(ii,jj,kk+1)+ul(ii,jj,kk-1)-2.0d0*ul(ii,jj,kk))*dzi*dzi  &
  &                       -at(ii,jj,kk)*(ul(ii+1,jj+1,kk)+ul(ii-1,jj-1,kk)  &
  &                                    -(ul(ii-1,jj+1,kk)+ul(ii+1,jj-1,kk)))*0.25d0*dxi*dyi  &
  &                       -bt(ii,jj,kk)*(ul(ii,jj+1,kk+1)+ul(ii,jj-1,kk-1)  &
  &                                    -(ul(ii,jj-1,kk+1)+ul(ii,jj+1,kk-1)))*0.25d0*dyi*dzi  &
  &                       -ct(ii,jj,kk)*(ul(ii+1,jj,kk+1)+ul(ii-1,jj,kk-1)  &
  &                                    -(ul(ii+1,jj,kk-1)+ul(ii-1,jj,kk+1)))*0.25d0*dzi*dxi  &
  &                       -dt(ii,jj,kk)*(ul(ii+1,jj,kk)-ul(ii-1,jj,kk))*0.5d0*dxi  &
  &                       -et(ii,jj,kk)*(ul(ii,jj+1,kk)-ul(ii,jj-1,kk))*0.5d0*dyi  &
  &                       -ft(ii,jj,kk)*(ul(ii,jj,kk+1)-ul(ii,jj,kk-1))*0.5d0*dzi  &
  &                       -gt(ii,jj,kk)*ul(ii,jj,kk)
        end do
     end do
  end do

end subroutine calc_error_3dd


end module Ellip_Slv
