!---------------------------------------------------------------
! Copyright (C) 2009-2015 GFD Dennou Club. All rights reserved.
!---------------------------------------------------------------

module file_operate  ! CReSS の計算結果出力ファイルデータの操作用モジュール

public :: read_file_gtool3_header,  &
  &       read_file,  &
  &       read_file_3d,  &
  &       read_file_part,  &
  &       read_file_part_3d,  &
  &       write_file,  &
  &       write_file_3d

interface read_file_gtool3_header

  module procedure read_file_gtool3_header_c,  &
  &                read_file_gtool3_header_i,  &
  &                read_file_gtool3_header_r

end interface read_file_gtool3_header

interface read_file

  module procedure read_file_i1,  &
  &                read_file_i2,  &
  &                read_file_i4,  &
  &                read_file_f4,  &
  &                read_file_d8

end interface read_file

interface read_file_3d

  module procedure read_file_3di1,  &
  &                read_file_3di2,  &
  &                read_file_3di4,  &
  &                read_file_3df4,  &
  &                read_file_3dd8

end interface read_file_3d

interface read_file_part

  module procedure read_file_part_i1,  &
  &                read_file_part_i2,  &
  &                read_file_part_i4,  &
  &                read_file_part_f4,  &
  &                read_file_part_d8

end interface read_file_part

interface write_file

  module procedure write_file_i1,  &
  &                write_file_i2,  &
  &                write_file_i4,  &
  &                write_file_f4,  &
  &                write_file_d8

end interface write_file

interface write_file_3d

  module procedure write_file_3di1,  &
  &                write_file_3di2,  &
  &                write_file_3di4,  &
  &                write_file_3df4,  &
  &                write_file_3dd8

end interface write_file_3d

private :: check_err

private :: read_file_i1,  &
  &        read_file_i2,  &
  &        read_file_i4,  &
  &        read_file_f4,  &
  &        read_file_d8,  &
  &        read_file_3di1,  &
  &        read_file_3di2,  &
  &        read_file_3di4,  &
  &        read_file_3df4,  &
  &        read_file_3dd8,  &
  &        write_file_i1,  &
  &        write_file_i2,  &
  &        write_file_i4,  &
  &        write_file_f4,  &
  &        write_file_d8,  &
  &        write_file_3di1,  &
  &        write_file_3di2,  &
  &        write_file_3di4,  &
  &        write_file_3df4,  &
  &        write_file_3dd8

contains

subroutine read_file_i1( file_name, nx, ny, rec_num, var, offset, funit )
  ! 1 バイト符号付き整数ファイル読み取りルーチン
  implicit none
  integer, intent(in) :: nx  ! データの x 方向の個数
  integer, intent(in) :: ny  ! データの y 方向の個数
  integer, intent(in) :: rec_num  ! 読み出すデータのレコード番号
  character(*), intent(in) :: file_name  ! 読み出すデータファイル名
  integer(1), intent(inout) :: var(nx,ny)  ! 読み出すデータ
  integer, intent(in), optional :: offset  ! 読み込み開始位置指定
                       ! このバイト位置まではスキップする.
                       ! 例えば, offset = 1 なら, 先頭 1 byte を飛ばし,
                       ! 次の 1 byte から読み始める.
                       ! なお, これが指定された場合, rec_num は無視する.
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: i, j, k, err, unitn   ! 作業用配列
  integer, parameter :: bnum=1

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  err=0
  if(present(offset))then
     open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &       recl=bnum, status='old', iostat=err)
        call check_err( err, 'File Not Found : '//trim(adjustl(file_name)) )
        k=offset
        do j=1,ny
           do i=1,nx
              k=k+1
              read(unitn,rec=k,iostat=err) var(i,j)
              call check_err( err, 'Can not read : '//trim(adjustl(file_name)))
           end do
        end do
     close(unit=unitn)
  else
     open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &       recl=bnum*nx*ny, status='old', iostat=err)
        call check_err( err, 'File Not Found : '//trim(adjustl(file_name)) )

        read(unitn,rec=rec_num,iostat=err) ((var(i,j),i=1,nx),j=1,ny)

        call check_err( err, 'Can not read : '//trim(adjustl(file_name)))
     close(unit=unitn)
  end if

end subroutine read_file_i1

!-----------------------------------------------
!-----------------------------------------------

subroutine read_file_i2( file_name, nx, ny, rec_num, var, offset, funit )
  ! 2 バイト整数ファイル読み取りルーチン
  implicit none
  integer, intent(in) :: nx  ! データの x 方向の個数
  integer, intent(in) :: ny  ! データの y 方向の個数
  integer, intent(in) :: rec_num  ! 読み出すデータのレコード番号
  character(*), intent(in) :: file_name  ! 読み出すデータファイル名
  integer(2), intent(inout) :: var(nx,ny)  ! 読み出すデータ
  integer, intent(in), optional :: offset  ! 読み込み開始位置指定
                       ! このバイト位置まではスキップする.
                       ! 例えば, offset = 1 なら, 先頭 2 byte を飛ばし,
                       ! 次の 2 byte から読み始める.
                       ! なお, これが指定された場合, rec_num は無視する.
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: i, j, k, err, unitn   ! 作業用配列
  integer, parameter :: bnum=2

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  err=0
  if(present(offset))then
     open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &       recl=bnum, status='old', iostat=err)
        call check_err( err, 'File Not Found : '//trim(adjustl(file_name)) )
        k=offset
        do j=1,ny
           do i=1,nx
              k=k+1
              read(unitn,rec=k,iostat=err) var(i,j)
              call check_err( err, 'Can not read : '//trim(adjustl(file_name)))
           end do
        end do
     close(unit=unitn)
  else
     open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &       recl=bnum*nx*ny, status='old', iostat=err)
        call check_err( err, 'File Not Found : '//trim(adjustl(file_name)) )

        read(unitn,rec=rec_num,iostat=err) ((var(i,j),i=1,nx),j=1,ny)

        call check_err( err, 'Can not read : '//trim(adjustl(file_name)))
     close(unit=unitn)
  end if

end subroutine read_file_i2

!-----------------------------------------------
!-----------------------------------------------

subroutine read_file_i4( file_name, nx, ny, rec_num, var, offset, funit )
  ! 4 バイト整数ファイル読み取りルーチン
  implicit none
  integer, intent(in) :: nx  ! データの x 方向の個数
  integer, intent(in) :: ny  ! データの y 方向の個数
  integer, intent(in) :: rec_num  ! 読み出すデータのレコード番号
  character(*), intent(in) :: file_name  ! 読み出すデータファイル名
  integer, intent(inout) :: var(nx,ny)  ! 読み出すデータ
  integer, intent(in), optional :: offset  ! 読み込み開始位置指定
                       ! このバイト位置まではスキップする.
                       ! 例えば, offset = 1 なら, 先頭 4 byte を飛ばし,
                       ! 次の 4 byte から読み始める.
                       ! なお, これが指定された場合, rec_num は無視する.
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: i, j, k, err, unitn   ! 作業用配列
  integer, parameter :: bnum=4

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  err=0
  if(present(offset))then
     open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &       recl=bnum, status='old', iostat=err)
        call check_err( err, 'File Not Found : '//trim(adjustl(file_name)) )
        k=offset
        do j=1,ny
           do i=1,nx
              k=k+1
              read(unitn,rec=k,iostat=err) var(i,j)
              call check_err( err, 'Can not read : '//trim(adjustl(file_name)))
           end do
        end do
     close(unit=unitn)
  else
     open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &       recl=bnum*nx*ny, status='old', iostat=err)
        call check_err( err, 'File Not Found : '//trim(adjustl(file_name)) )

        read(unitn,rec=rec_num,iostat=err) ((var(i,j),i=1,nx),j=1,ny)

        call check_err( err, 'Can not read : '//trim(adjustl(file_name)))
     close(unit=unitn)
  end if

end subroutine read_file_i4

!-----------------------------------------------
!-----------------------------------------------

subroutine read_file_f4( file_name, nx, ny, rec_num, var, offset, funit )
  ! 4 バイト実数ファイル読み取りルーチン
  implicit none
  integer, intent(in) :: nx  ! データの x 方向の個数
  integer, intent(in) :: ny  ! データの y 方向の個数
  integer, intent(in) :: rec_num  ! 読み出すデータのレコード番号
  character(*), intent(in) :: file_name  ! 読み出すデータファイル名
  real, intent(inout) :: var(nx,ny)  ! 読み出すデータ
  integer, intent(in), optional :: offset  ! 読み込み開始位置指定
                       ! このバイト位置まではスキップする.
                       ! 例えば, offset = 1 なら, 先頭 4 byte を飛ばし,
                       ! 次の 4 byte から読み始める.
                       ! なお, これが指定された場合, rec_num は無視する.
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: i, j, k, err, unitn   ! 作業用配列
  integer, parameter :: bnum=4

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  err=0
  if(present(offset))then
     open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &       recl=bnum, status='old', iostat=err)
        call check_err( err, 'File Not Found : '//trim(adjustl(file_name)) )
        k=offset
        do j=1,ny
           do i=1,nx
              k=k+1
              read(unitn,rec=k,iostat=err) var(i,j)
              call check_err( err, 'Can not read : '//trim(adjustl(file_name)))
           end do
        end do
     close(unit=unitn)
  else
     open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &       recl=bnum*nx*ny, status='old', iostat=err)
        call check_err( err, 'File Not Found : '//trim(adjustl(file_name)) )

        read(unitn,rec=rec_num,iostat=err) ((var(i,j),i=1,nx),j=1,ny)

        call check_err( err, 'Can not read : '//trim(adjustl(file_name)))
     close(unit=unitn)
  end if

end subroutine read_file_f4

!-----------------------------------------------
!-----------------------------------------------

subroutine read_file_d8( file_name, nx, ny, rec_num, var, offset, funit )
  ! 8 バイト実数ファイル読み取りルーチン
  implicit none
  integer, intent(in) :: nx  ! データの x 方向の個数
  integer, intent(in) :: ny  ! データの y 方向の個数
  integer, intent(in) :: rec_num  ! 読み出すデータのレコード番号
  character(*), intent(in) :: file_name  ! 読み出すデータファイル名
  double precision, intent(inout) :: var(nx,ny)  ! 読み出すデータ
  integer, intent(in), optional :: offset  ! 読み込み開始位置指定
                       ! このバイト位置まではスキップする.
                       ! 例えば, offset = 1 なら, 先頭 8 byte を飛ばし,
                       ! 次の 8 byte から読み始める.
                       ! なお, これが指定された場合, rec_num は無視する.
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: i, j, k, err, unitn   ! 作業用配列
  integer, parameter :: bnum=8

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  err=0
  if(present(offset))then
     open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &       recl=bnum, status='old', iostat=err)
        call check_err( err, 'File Not Found : '//trim(adjustl(file_name)) )
        k=offset
        do j=1,ny
           do i=1,nx
              k=k+1
              read(unitn,rec=k,iostat=err) var(i,j)
              call check_err( err, 'Can not read : '//trim(adjustl(file_name)))
           end do
        end do
     close(unit=unitn)
  else
     open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &       recl=bnum*nx*ny, status='old', iostat=err)
        call check_err( err, 'File Not Found : '//trim(adjustl(file_name)) )

        read(unitn,rec=rec_num,iostat=err) ((var(i,j),i=1,nx),j=1,ny)

        call check_err( err, 'Can not read : '//trim(adjustl(file_name)))
     close(unit=unitn)
  end if

end subroutine read_file_d8

!-----------------------------------------------
!-----------------------------------------------

subroutine read_file_3di1( file_name, nx, ny, nz, rec_num, var, offset, funit )
  ! read_file 3 次元版.
  implicit none
  integer, intent(in) :: nx  ! データの x 方向の個数
  integer, intent(in) :: ny  ! データの y 方向の個数
  integer, intent(in) :: nz  ! データの z 方向の個数
  integer, intent(in) :: rec_num  ! 読み出しを開始するデータのレコード番号
  character(*), intent(in) :: file_name  ! 読み出すデータファイル名
  integer(1), intent(inout) :: var(nx,ny,nz)  ! 読み出すデータ
  integer, intent(in), optional :: offset  ! 読み込み開始位置指定
                       ! このバイト位置まではスキップする.
                       ! 例えば, offset = 1 なら, 先頭 1 byte を飛ばし,
                       ! 次の 1 byte から読み始める.
                       ! なお, これが指定された場合, rec_num は無視する.
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: i, j, k, l, err, unitn  ! 作業用配列
  integer, parameter :: bnum=1

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  err=0
  if(present(offset))then
     open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &       recl=bnum, status='old', iostat=err)
        call check_err( err, 'File Not Found : '//trim(adjustl(file_name)) )
        l=offset
        do k=1,nz
           do j=1,ny
              do i=1,nx
                 l=l+1
                 read(unitn,rec=l,iostat=err) var(i,j,k)
                 call check_err( err, 'Can not read : '//trim(adjustl(file_name)) )
              end do
           end do
        end do
     close(unit=unitn)
  else
     open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &       recl=bnum*nx*ny, status='old', iostat=err)
        call check_err( err, 'File Not Found : '//trim(adjustl(file_name)) )
        do k=1,nz
           read(unitn,rec=rec_num+k-1,iostat=err) ((var(i,j,k),i=1,nx),j=1,ny)
           call check_err( err, 'Can not read : '//trim(adjustl(file_name)) )
        end do
     close(unit=unitn)
  end if

end subroutine read_file_3di1

!-----------------------------------------------
!-----------------------------------------------

subroutine read_file_3di2( file_name, nx, ny, nz, rec_num, var, offset, funit )
  ! read_file 3 次元版.
  implicit none
  integer, intent(in) :: nx  ! データの x 方向の個数
  integer, intent(in) :: ny  ! データの y 方向の個数
  integer, intent(in) :: nz  ! データの z 方向の個数
  integer, intent(in) :: rec_num  ! 読み出しを開始するデータのレコード番号
  character(*), intent(in) :: file_name  ! 読み出すデータファイル名
  integer(2), intent(inout) :: var(nx,ny,nz)  ! 読み出すデータ
  integer, intent(in), optional :: offset  ! 読み込み開始位置指定
                       ! このバイト位置まではスキップする.
                       ! 例えば, offset = 1 なら, 先頭 2 byte を飛ばし,
                       ! 次の 2 byte から読み始める.
                       ! なお, これが指定された場合, rec_num は無視する.
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: i, j, k, l, err, unitn  ! 作業用配列
  integer, parameter :: bnum=2

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  err=0
  if(present(offset))then
     open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &       recl=bnum, status='old', iostat=err)
        call check_err( err, 'File Not Found : '//trim(adjustl(file_name)) )
        l=offset
        do k=1,nz
           do j=1,ny
              do i=1,nx
                 l=l+1
                 read(unitn,rec=l,iostat=err) var(i,j,k)
                 call check_err( err, 'Can not read : '//trim(adjustl(file_name)) )
              end do
           end do
        end do
     close(unit=unitn)
  else
     open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &       recl=bnum*nx*ny, status='old', iostat=err)
        call check_err( err, 'File Not Found : '//trim(adjustl(file_name)) )
        do k=1,nz
           read(unitn,rec=rec_num+k-1,iostat=err) ((var(i,j,k),i=1,nx),j=1,ny)
           call check_err( err, 'Can not read : '//trim(adjustl(file_name)) )
        end do
     close(unit=unitn)
  end if

end subroutine read_file_3di2

!-----------------------------------------------
!-----------------------------------------------

subroutine read_file_3di4( file_name, nx, ny, nz, rec_num, var, offset, funit )
  ! read_file 3 次元版.
  implicit none
  integer, intent(in) :: nx  ! データの x 方向の個数
  integer, intent(in) :: ny  ! データの y 方向の個数
  integer, intent(in) :: nz  ! データの z 方向の個数
  integer, intent(in) :: rec_num  ! 読み出しを開始するデータのレコード番号
  character(*), intent(in) :: file_name  ! 読み出すデータファイル名
  integer(4), intent(inout) :: var(nx,ny,nz)  ! 読み出すデータ
  integer, intent(in), optional :: offset  ! 読み込み開始位置指定
                       ! このバイト位置まではスキップする.
                       ! 例えば, offset = 1 なら, 先頭 4 byte を飛ばし,
                       ! 次の 4 byte から読み始める.
                       ! なお, これが指定された場合, rec_num は無視する.
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: i, j, k, l, err, unitn  ! 作業用配列
  integer, parameter :: bnum=4

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  err=0
  if(present(offset))then
     open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &       recl=bnum, status='old', iostat=err)
        call check_err( err, 'File Not Found : '//trim(adjustl(file_name)) )
        l=offset
        do k=1,nz
           do j=1,ny
              do i=1,nx
                 l=l+1
                 read(unitn,rec=l,iostat=err) var(i,j,k)
                 call check_err( err, 'Can not read : '//trim(adjustl(file_name)) )
              end do
           end do
        end do
     close(unit=unitn)
  else
     open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &       recl=bnum*nx*ny, status='old', iostat=err)
        call check_err( err, 'File Not Found : '//trim(adjustl(file_name)) )
        do k=1,nz
           read(unitn,rec=rec_num+k-1,iostat=err) ((var(i,j,k),i=1,nx),j=1,ny)
           call check_err( err, 'Can not read : '//trim(adjustl(file_name)) )
        end do
     close(unit=unitn)
  end if

end subroutine read_file_3di4

!-----------------------------------------------
!-----------------------------------------------

subroutine read_file_3df4( file_name, nx, ny, nz, rec_num, var, offset, funit )
  ! read_file 3 次元版.
  implicit none
  integer, intent(in) :: nx  ! データの x 方向の個数
  integer, intent(in) :: ny  ! データの y 方向の個数
  integer, intent(in) :: nz  ! データの z 方向の個数
  integer, intent(in) :: rec_num  ! 読み出しを開始するデータのレコード番号
  character(*), intent(in) :: file_name  ! 読み出すデータファイル名
  real, intent(inout) :: var(nx,ny,nz)  ! 読み出すデータ
  integer, intent(in), optional :: offset  ! 読み込み開始位置指定
                       ! このバイト位置まではスキップする.
                       ! 例えば, offset = 1 なら, 先頭 4 byte を飛ばし,
                       ! 次の 4 byte から読み始める.
                       ! なお, これが指定された場合, rec_num は無視する.
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: i, j, k, l, err, unitn  ! 作業用配列
  integer, parameter :: bnum=4

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  err=0
  if(present(offset))then
     open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &       recl=bnum, status='old', iostat=err)
        call check_err( err, 'File Not Found : '//trim(adjustl(file_name)) )
        l=offset
        do k=1,nz
           do j=1,ny
              do i=1,nx
                 l=l+1
                 read(unitn,rec=l,iostat=err) var(i,j,k)
                 call check_err( err, 'Can not read : '//trim(adjustl(file_name)) )
              end do
           end do
        end do
     close(unit=unitn)
  else
     open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &       recl=bnum*nx*ny, status='old', iostat=err)
        call check_err( err, 'File Not Found : '//trim(adjustl(file_name)) )
        do k=1,nz
           read(unitn,rec=rec_num+k-1,iostat=err) ((var(i,j,k),i=1,nx),j=1,ny)
           call check_err( err, 'Can not read : '//trim(adjustl(file_name)) )
        end do
     close(unit=unitn)
  end if

end subroutine read_file_3df4

!-----------------------------------------------
!-----------------------------------------------

subroutine read_file_3dd8( file_name, nx, ny, nz, rec_num, var, offset, funit )
  ! read_file 3 次元版.
  implicit none
  integer, intent(in) :: nx  ! データの x 方向の個数
  integer, intent(in) :: ny  ! データの y 方向の個数
  integer, intent(in) :: nz  ! データの z 方向の個数
  integer, intent(in) :: rec_num  ! 読み出しを開始するデータのレコード番号
  character(*), intent(in) :: file_name  ! 読み出すデータファイル名
  double precision, intent(inout) :: var(nx,ny,nz)  ! 読み出すデータ
  integer, intent(in), optional :: offset  ! 読み込み開始位置指定
                       ! このバイト位置まではスキップする.
                       ! 例えば, offset = 1 なら, 先頭 8 byte を飛ばし,
                       ! 次の 8 byte から読み始める.
                       ! なお, これが指定された場合, rec_num は無視する.
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: i, j, k, l, err, unitn  ! 作業用配列
  integer, parameter :: bnum=8

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  err=0
  if(present(offset))then
     open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &       recl=bnum, status='old', iostat=err)
        call check_err( err, 'File Not Found : '//trim(adjustl(file_name)) )
        l=offset
        do k=1,nz
           do j=1,ny
              do i=1,nx
                 l=l+1
                 read(unitn,rec=l,iostat=err) var(i,j,k)
                 call check_err( err, 'Can not read : '//trim(adjustl(file_name)) )
              end do
           end do
        end do
     close(unit=unitn)
  else
     open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &       recl=bnum*nx*ny, status='old', iostat=err)
        call check_err( err, 'File Not Found : '//trim(adjustl(file_name)) )
        do k=1,nz
           read(unitn,rec=rec_num+k-1,iostat=err) ((var(i,j,k),i=1,nx),j=1,ny)
           call check_err( err, 'Can not read : '//trim(adjustl(file_name)) )
        end do
     close(unit=unitn)
  end if

end subroutine read_file_3dd8

!-----------------------------------------------
!-----------------------------------------------

subroutine read_file_part_i1( file_name, nx, ny, nnx, nny, rec_num, var,  &
  &                           funit )
  ! 1 バイト符号付き整数ファイル部分読み取りルーチン
  implicit none
  integer, intent(in) :: nx  ! データの x 方向の個数
  integer, intent(in) :: ny  ! データの y 方向の個数
  integer, intent(in) :: nnx(2)  ! 読みだすデータの x 方向の格子番号
  integer, intent(in) :: nny(2)  ! 読みだすデータの y 方向の格子番号
  integer, intent(in) :: rec_num  ! 読み出すデータのレコード番号 (nx,ny 基準)
  character(*), intent(in) :: file_name  ! 読み出すデータファイル名
  integer(1), intent(inout) :: var(nnx(2)-nnx(1)+1,nny(2)-nny(1)+1)
                       ! 読み出すデータ
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: i, j, k, err, unitn, l   ! 作業用配列
  integer, parameter :: bnum=1

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  err=0

  open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &    recl=bnum, status='old', iostat=err)
     call check_err( err, 'File Not Found : '//trim(adjustl(file_name)) )

     l=nx*ny*(rec_num-1)
     do j=1,ny
        do i=1,nx
           l=l+1
           if(i>=nnx(1).and.i<=nnx(2).and.j>=nny(1).and.j<=nny(2))then
              read(unitn,rec=l,iostat=err) var(i-nnx(1)+1,j-nny(1)+1)
              call check_err( err, 'Can not read : '//trim(adjustl(file_name)) )
           end if
        end do
     end do

  close(unit=unitn)

end subroutine read_file_part_i1

!-----------------------------------------------
!-----------------------------------------------

subroutine read_file_part_i2( file_name, nx, ny, nnx, nny, rec_num, var,  &
  &                           funit )
  ! 2 バイト符号付き整数ファイル部分読み取りルーチン
  implicit none
  integer, intent(in) :: nx  ! データの x 方向の個数
  integer, intent(in) :: ny  ! データの y 方向の個数
  integer, intent(in) :: nnx(2)  ! 読みだすデータの x 方向の格子番号
  integer, intent(in) :: nny(2)  ! 読みだすデータの y 方向の格子番号
  integer, intent(in) :: rec_num  ! 読み出すデータのレコード番号 (nx,ny 基準)
  character(*), intent(in) :: file_name  ! 読み出すデータファイル名
  integer(2), intent(inout) :: var(nnx(2)-nnx(1)+1,nny(2)-nny(1)+1)
                       ! 読み出すデータ
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: i, j, k, err, unitn, l   ! 作業用配列
  integer, parameter :: bnum=2

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  err=0

  open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &    recl=bnum, status='old', iostat=err)
     call check_err( err, 'File Not Found : '//trim(adjustl(file_name)) )

     l=nx*ny*(rec_num-1)
     do j=1,ny
        do i=1,nx
           l=l+1
           if(i>=nnx(1).and.i<=nnx(2).and.j>=nny(1).and.j<=nny(2))then
              read(unitn,rec=l,iostat=err) var(i-nnx(1)+1,j-nny(1)+1)
              call check_err( err, 'Can not read : '//trim(adjustl(file_name)) )
           end if
        end do
     end do

  close(unit=unitn)

end subroutine read_file_part_i2

!-----------------------------------------------
!-----------------------------------------------

subroutine read_file_part_i4( file_name, nx, ny, nnx, nny, rec_num, var,  &
  &                           funit )
  ! 4 バイト符号付き整数ファイル部分読み取りルーチン
  implicit none
  integer, intent(in) :: nx  ! データの x 方向の個数
  integer, intent(in) :: ny  ! データの y 方向の個数
  integer, intent(in) :: nnx(2)  ! 読みだすデータの x 方向の格子番号
  integer, intent(in) :: nny(2)  ! 読みだすデータの y 方向の格子番号
  integer, intent(in) :: rec_num  ! 読み出すデータのレコード番号 (nx,ny 基準)
  character(*), intent(in) :: file_name  ! 読み出すデータファイル名
  integer, intent(inout) :: var(nnx(2)-nnx(1)+1,nny(2)-nny(1)+1)
                       ! 読み出すデータ
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: i, j, k, err, unitn, l   ! 作業用配列
  integer, parameter :: bnum=4

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  err=0

  open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &    recl=bnum, status='old', iostat=err)
     call check_err( err, 'File Not Found : '//trim(adjustl(file_name)) )

     l=nx*ny*(rec_num-1)
     do j=1,ny
        do i=1,nx
           l=l+1
           if(i>=nnx(1).and.i<=nnx(2).and.j>=nny(1).and.j<=nny(2))then
              read(unitn,rec=l,iostat=err) var(i-nnx(1)+1,j-nny(1)+1)
              call check_err( err, 'Can not read : '//trim(adjustl(file_name)) )
           end if
        end do
     end do

  close(unit=unitn)

end subroutine read_file_part_i4

!-----------------------------------------------
!-----------------------------------------------

subroutine read_file_part_f4( file_name, nx, ny, nnx, nny, rec_num, var,  &
  &                           funit )
  ! 単精度実数ファイル部分読み取りルーチン
  implicit none
  integer, intent(in) :: nx  ! データの x 方向の個数
  integer, intent(in) :: ny  ! データの y 方向の個数
  integer, intent(in) :: nnx(2)  ! 読みだすデータの x 方向の格子番号
  integer, intent(in) :: nny(2)  ! 読みだすデータの y 方向の格子番号
  integer, intent(in) :: rec_num  ! 読み出すデータのレコード番号 (nx,ny 基準)
  character(*), intent(in) :: file_name  ! 読み出すデータファイル名
  real, intent(inout) :: var(nnx(2)-nnx(1)+1,nny(2)-nny(1)+1)
                       ! 読み出すデータ
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: i, j, k, err, unitn, l   ! 作業用配列
  integer, parameter :: bnum=4

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  err=0

  open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &    recl=bnum, status='old', iostat=err)
     call check_err( err, 'File Not Found : '//trim(adjustl(file_name)) )

     l=nx*ny*(rec_num-1)
     do j=1,ny
        do i=1,nx
           l=l+1
           if(i>=nnx(1).and.i<=nnx(2).and.j>=nny(1).and.j<=nny(2))then
              read(unitn,rec=l,iostat=err) var(i-nnx(1)+1,j-nny(1)+1)
              call check_err( err, 'Can not read : '//trim(adjustl(file_name)) )
           end if
        end do
     end do

  close(unit=unitn)

end subroutine read_file_part_f4

!-----------------------------------------------
!-----------------------------------------------

subroutine read_file_part_d8( file_name, nx, ny, nnx, nny, rec_num, var,  &
  &                           funit )
  ! 倍精度実数ファイル部分読み取りルーチン
  implicit none
  integer, intent(in) :: nx  ! データの x 方向の個数
  integer, intent(in) :: ny  ! データの y 方向の個数
  integer, intent(in) :: nnx(2)  ! 読みだすデータの x 方向の格子番号
  integer, intent(in) :: nny(2)  ! 読みだすデータの y 方向の格子番号
  integer, intent(in) :: rec_num  ! 読み出すデータのレコード番号 (nx,ny 基準)
  character(*), intent(in) :: file_name  ! 読み出すデータファイル名
  double precision, intent(inout) :: var(nnx(2)-nnx(1)+1,nny(2)-nny(1)+1)
                       ! 読み出すデータ
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: i, j, k, err, unitn, l   ! 作業用配列
  integer, parameter :: bnum=8

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  err=0

  open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &    recl=bnum, status='old', iostat=err)
     call check_err( err, 'File Not Found : '//trim(adjustl(file_name)) )

     l=nx*ny*(rec_num-1)
     do j=1,ny
        do i=1,nx
           l=l+1
           if(i>=nnx(1).and.i<=nnx(2).and.j>=nny(1).and.j<=nny(2))then
              read(unitn,rec=l,iostat=err) var(i-nnx(1)+1,j-nny(1)+1)
              call check_err( err, 'Can not read : '//trim(adjustl(file_name)) )
           end if
        end do
     end do

  close(unit=unitn)

end subroutine read_file_part_d8

!-----------------------------------------------
!-----------------------------------------------

subroutine write_file_i1( file_name, nx, ny, rec_num, var, mode, funit )
  ! ファイル出力ルーチン
  implicit none
  integer, intent(in) :: nx  ! データの x 方向の個数
  integer, intent(in) :: ny  ! データの y 方向の個数
  integer, intent(in) :: rec_num  ! 書き出すデータのレコード番号
  character(*), intent(in) :: file_name  ! 書き出すデータファイル名
  integer(1), intent(in) :: var(nx,ny)  ! 書き出すデータ
  character(*), optional, intent(in) :: mode  ! ファイルの書き出しオプション
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: i, j, unitn  ! 作業用配列
  character(10) :: cmode
  integer, parameter :: bnum=1

  cmode=''

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  if(present(mode))then
     cmode=mode
  else
     cmode='unknown'
  end if

  open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &    recl=bnum*nx*ny, status=cmode)
     write(unitn,rec=rec_num) ((var(i,j),i=1,nx),j=1,ny)
  close(unit=unitn)

end subroutine write_file_i1

!-----------------------------------------------
!-----------------------------------------------

subroutine write_file_i2( file_name, nx, ny, rec_num, var, mode, funit )
  ! ファイル出力ルーチン
  implicit none
  integer, intent(in) :: nx  ! データの x 方向の個数
  integer, intent(in) :: ny  ! データの y 方向の個数
  integer, intent(in) :: rec_num  ! 書き出すデータのレコード番号
  character(*), intent(in) :: file_name  ! 書き出すデータファイル名
  integer(2), intent(in) :: var(nx,ny)  ! 書き出すデータ
  character(*), optional, intent(in) :: mode  ! ファイルの書き出しオプション
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: i, j, unitn  ! 作業用配列
  character(10) :: cmode
  integer, parameter :: bnum=2

  cmode=''

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  if(present(mode))then
     cmode=mode
  else
     cmode='unknown'
  end if

  open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &    recl=bnum*nx*ny, status=cmode)
     write(unitn,rec=rec_num) ((var(i,j),i=1,nx),j=1,ny)
  close(unit=unitn)

end subroutine write_file_i2

!-----------------------------------------------
!-----------------------------------------------

subroutine write_file_i4( file_name, nx, ny, rec_num, var, mode, funit )
  ! ファイル出力ルーチン
  implicit none
  integer, intent(in) :: nx  ! データの x 方向の個数
  integer, intent(in) :: ny  ! データの y 方向の個数
  integer, intent(in) :: rec_num  ! 書き出すデータのレコード番号
  character(*), intent(in) :: file_name  ! 書き出すデータファイル名
  integer, intent(in) :: var(nx,ny)  ! 書き出すデータ
  character(*), optional, intent(in) :: mode  ! ファイルの書き出しオプション
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: i, j, unitn  ! 作業用配列
  character(10) :: cmode
  integer, parameter :: bnum=4

  cmode=''

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  if(present(mode))then
     cmode=mode
  else
     cmode='unknown'
  end if

  open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &    recl=bnum*nx*ny, status=cmode)
     write(unitn,rec=rec_num) ((var(i,j),i=1,nx),j=1,ny)
  close(unit=unitn)

end subroutine write_file_i4

!-----------------------------------------------
!-----------------------------------------------

subroutine write_file_f4( file_name, nx, ny, rec_num, var, mode, funit )
  ! ファイル出力ルーチン
  implicit none
  integer, intent(in) :: nx  ! データの x 方向の個数
  integer, intent(in) :: ny  ! データの y 方向の個数
  integer, intent(in) :: rec_num  ! 書き出すデータのレコード番号
  character(*), intent(in) :: file_name  ! 書き出すデータファイル名
  real, intent(in) :: var(nx,ny)  ! 書き出すデータ
  character(*), optional, intent(in) :: mode  ! ファイルの書き出しオプション
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: i, j, unitn  ! 作業用配列
  character(10) :: cmode
  integer, parameter :: bnum=4

  cmode=''

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  if(present(mode))then
     cmode=mode
  else
     cmode='unknown'
  end if

  open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &    recl=bnum*nx*ny, status=cmode)
     write(unitn,rec=rec_num) ((var(i,j),i=1,nx),j=1,ny)
  close(unit=unitn)

end subroutine write_file_f4

!-----------------------------------------------
!-----------------------------------------------

subroutine write_file_d8( file_name, nx, ny, rec_num, var, mode, funit )
  ! ファイル出力ルーチン
  implicit none
  integer, intent(in) :: nx  ! データの x 方向の個数
  integer, intent(in) :: ny  ! データの y 方向の個数
  integer, intent(in) :: rec_num  ! 書き出すデータのレコード番号
  character(*), intent(in) :: file_name  ! 書き出すデータファイル名
  double precision, intent(in) :: var(nx,ny)  ! 書き出すデータ
  character(*), optional, intent(in) :: mode  ! ファイルの書き出しオプション
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: i, j, unitn  ! 作業用配列
  character(10) :: cmode
  integer, parameter :: bnum=8

  cmode=''

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  if(present(mode))then
     cmode=mode
  else
     cmode='unknown'
  end if

  open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &    recl=bnum*nx*ny, status=cmode)
     write(unitn,rec=rec_num) ((var(i,j),i=1,nx),j=1,ny)
  close(unit=unitn)

end subroutine write_file_d8

!-----------------------------------------------
!-----------------------------------------------

subroutine write_file_3di1( file_name, nx, ny, nz, rec_num, var, mode, funit )
  ! write_file 3 次元版.
  implicit none
  integer, intent(in) :: nx  ! データの x 方向の個数
  integer, intent(in) :: ny  ! データの y 方向の個数
  integer, intent(in) :: nz  ! データの z 方向の個数
  integer, intent(in) :: rec_num  ! 読み出しを開始するデータのレコード番号
  character(*), intent(in) :: file_name  ! 読み出すデータファイル名
  integer(1), intent(in) :: var(nx,ny,nz)  ! 読み出すデータ
  character(*), optional, intent(in) :: mode  ! ファイルの書き出しオプション
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: i, j, k, unitn  ! 作業用配列
  character(10) :: cmode
  integer, parameter :: bnum=1

  cmode=''

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  if(present(mode))then
     cmode=mode
  else
     cmode='unknown'
  end if

  open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &    recl=bnum*nx*ny, status=trim(cmode))
     do k=1,nz
        write(unitn,rec=rec_num+k-1) ((var(i,j,k),i=1,nx),j=1,ny)
     end do
  close(unit=unitn)

end subroutine write_file_3di1

!-----------------------------------------------
!-----------------------------------------------

subroutine write_file_3di2( file_name, nx, ny, nz, rec_num, var, mode, funit )
  ! write_file 3 次元版.
  implicit none
  integer, intent(in) :: nx  ! データの x 方向の個数
  integer, intent(in) :: ny  ! データの y 方向の個数
  integer, intent(in) :: nz  ! データの z 方向の個数
  integer, intent(in) :: rec_num  ! 読み出しを開始するデータのレコード番号
  character(*), intent(in) :: file_name  ! 読み出すデータファイル名
  integer(2), intent(in) :: var(nx,ny,nz)  ! 読み出すデータ
  character(*), optional, intent(in) :: mode  ! ファイルの書き出しオプション
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: i, j, k, unitn  ! 作業用配列
  character(10) :: cmode
  integer, parameter :: bnum=2

  cmode=''

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  if(present(mode))then
     cmode=mode
  else
     cmode='unknown'
  end if

  open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &    recl=bnum*nx*ny, status=trim(cmode))
     do k=1,nz
        write(unitn,rec=rec_num+k-1) ((var(i,j,k),i=1,nx),j=1,ny)
     end do
  close(unit=unitn)

end subroutine write_file_3di2

!-----------------------------------------------
!-----------------------------------------------

subroutine write_file_3di4( file_name, nx, ny, nz, rec_num, var, mode, funit )
  ! write_file 3 次元版.
  implicit none
  integer, intent(in) :: nx  ! データの x 方向の個数
  integer, intent(in) :: ny  ! データの y 方向の個数
  integer, intent(in) :: nz  ! データの z 方向の個数
  integer, intent(in) :: rec_num  ! 読み出しを開始するデータのレコード番号
  character(*), intent(in) :: file_name  ! 読み出すデータファイル名
  integer, intent(in) :: var(nx,ny,nz)  ! 読み出すデータ
  character(*), optional, intent(in) :: mode  ! ファイルの書き出しオプション
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: i, j, k, unitn  ! 作業用配列
  character(10) :: cmode
  integer, parameter :: bnum=4

  cmode=''

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  if(present(mode))then
     cmode=mode
  else
     cmode='unknown'
  end if

  open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &    recl=bnum*nx*ny, status=trim(cmode))
     do k=1,nz
        write(unitn,rec=rec_num+k-1) ((var(i,j,k),i=1,nx),j=1,ny)
     end do
  close(unit=unitn)

end subroutine write_file_3di4

!-----------------------------------------------
!-----------------------------------------------

subroutine write_file_3df4( file_name, nx, ny, nz, rec_num, var, mode, funit )
  ! write_file 3 次元版.
  implicit none
  integer, intent(in) :: nx  ! データの x 方向の個数
  integer, intent(in) :: ny  ! データの y 方向の個数
  integer, intent(in) :: nz  ! データの z 方向の個数
  integer, intent(in) :: rec_num  ! 読み出しを開始するデータのレコード番号
  character(*), intent(in) :: file_name  ! 読み出すデータファイル名
  real, intent(in) :: var(nx,ny,nz)  ! 読み出すデータ
  character(*), optional, intent(in) :: mode  ! ファイルの書き出しオプション
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: i, j, k, unitn  ! 作業用配列
  character(10) :: cmode
  integer, parameter :: bnum=4

  cmode=''

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  if(present(mode))then
     cmode=mode
  else
     cmode='unknown'
  end if

  open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &    recl=bnum*nx*ny, status=trim(cmode))
     do k=1,nz
        write(unitn,rec=rec_num+k-1) ((var(i,j,k),i=1,nx),j=1,ny)
     end do
  close(unit=unitn)

end subroutine write_file_3df4

!-----------------------------------------------
!-----------------------------------------------

subroutine write_file_3dd8( file_name, nx, ny, nz, rec_num, var, mode, funit )
  ! write_file 3 次元版.
  implicit none
  integer, intent(in) :: nx  ! データの x 方向の個数
  integer, intent(in) :: ny  ! データの y 方向の個数
  integer, intent(in) :: nz  ! データの z 方向の個数
  integer, intent(in) :: rec_num  ! 読み出しを開始するデータのレコード番号
  character(*), intent(in) :: file_name  ! 読み出すデータファイル名
  double precision, intent(in) :: var(nx,ny,nz)  ! 読み出すデータ
  character(*), optional, intent(in) :: mode  ! ファイルの書き出しオプション
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: i, j, k, unitn  ! 作業用配列
  character(10) :: cmode
  integer, parameter :: bnum=8

  cmode=''

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  if(present(mode))then
     cmode=mode
  else
     cmode='unknown'
  end if

  open(unit=unitn, file=trim(adjustl(file_name)), access='direct',  &
  &    recl=bnum*nx*ny, status=trim(cmode))
     do k=1,nz
        write(unitn,rec=rec_num+k-1) ((var(i,j,k),i=1,nx),j=1,ny)
     end do
  close(unit=unitn)

end subroutine write_file_3dd8

!-----------------------------------------------
!-----------------------------------------------

!subroutine read_file_valbyte( file_name, nx, ny, rec_num, byte, var )
!  ! 任意バイト数のデータを読み出すルーチン
!  implicit none
!  integer, intent(in) :: nx  ! データの x 方向の個数
!  integer, intent(in) :: ny  ! データの y 方向の個数
!  integer, intent(in) :: rec_num  ! 読み出すデータのレコード番号
!  integer, intent(in) :: byte  ! 読み出すバイト数
!  character(*), intent(in) :: file_name  ! 読み出すデータファイル名
!  real(kind=byte), intent(inout) :: var(nx,ny)  ! 読み出すデータ
!  integer :: i, j  ! 作業用配列
!
!  open(unit=11, file=file_name, access='direct', recl=byte*nx*ny, status='old')
!     read(11,rec=rec_num) ((var(i,j),i=1,nx),j=1,ny)
!  close(unit=11, status='keep')
!
!end subroutine read_file_valbyte

!-----------------------------------------------
!-----------------------------------------------

subroutine request_dim_grads( fname, dim_len, funit )
! GrADS のコントロールファイルから次元要素数を取得する.
  implicit none
  character(*), intent(in) :: fname  ! GrADS のコントロールファイル
  integer, intent(inout), dimension(4) :: dim_len  ! x,y,z,t 方向の要素数
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  character(1000), allocatable, dimension(:) :: ctl_cval
  character(100), allocatable, dimension(:,:) :: ctl_hval
  integer, dimension(5) :: cont_flag
  integer :: nl, i, unitn
  character(1000) :: tmpc
  logical :: pdeflag

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  nl=line_number_counter( trim(fname), funit=unitn )

  allocate(ctl_cval(nl))
  allocate(ctl_hval(10,1))

  pdeflag=.false.
  ctl_cval=''
  ctl_hval=''
  tmpc=''

!-- 先頭単語を変数へ格納, 以下の def チェック用
!-- 各座標データが何行目に入っているかを調べる.

  call read_file_text( trim(fname), 1, nl, ctl_cval, forma='(a1000)',  &
  &                    funit=unitn )

  do i=1,nl
     tmpc=trim(adjustl(ctl_cval(i)))
     if(tmpc(1:4)=='xdef')then
        cont_flag(1)=i
     end if
     if(tmpc(1:4)=='ydef')then
        cont_flag(2)=i
     end if
     if(tmpc(1:4)=='zdef')then
        cont_flag(3)=i
     end if
     if(tmpc(1:4)=='tdef')then
        cont_flag(4)=i
     end if
     if(tmpc(1:4)=='pdef')then
        cont_flag(5)=i
        pdeflag=.true.
     end if
  end do

!-- これらのチェックされた行番号データから実際にデータを読み込む.


  if(pdeflag.eqv..false.)then

     call read_file_text( trim(fname), 2, 1, ctl_hval(1:2,1),  &
  &                       skip=cont_flag(1)-1, funit=unitn )
     read(ctl_hval(2,1),*) dim_len(1)
     call read_file_text( trim(fname), 2, 1, ctl_hval(1:2,1),  &
  &                       skip=cont_flag(2)-1, funit=unitn )
     read(ctl_hval(2,1),*) dim_len(2)
     call read_file_text( trim(fname), 2, 1, ctl_hval(1:2,1),  &
  &                       skip=cont_flag(3)-1, funit=unitn )
     read(ctl_hval(2,1),*) dim_len(3)
     call read_file_text( trim(fname), 2, 1, ctl_hval(1:2,1),  &
  &                       skip=cont_flag(4)-1, funit=unitn )
     read(ctl_hval(2,1),*) dim_len(4)

  else

     call read_file_text( trim(fname), 3, 1, ctl_hval(1:2,1),  &
  &                       skip=cont_flag(5)-1, funit=unitn )
     read(ctl_hval(2,1),*) dim_len(1)
     read(ctl_hval(3,1),*) dim_len(2)
     call read_file_text( trim(fname), 2, 1, ctl_hval(1:2,1),  &
  &                       skip=cont_flag(3)-1, funit=unitn )
     read(ctl_hval(2,1),*) dim_len(3)
     call read_file_text( trim(fname), 2, 1, ctl_hval(1:2,1),  &
  &                       skip=cont_flag(4)-1, funit=unitn )
     read(ctl_hval(2,1),*) dim_len(4)

  end if

end subroutine

!-----------------------------------------------
!-----------------------------------------------

subroutine request_axis_grads( fname, axis_name, x, y, stime, funit )
! GrADS のコントロールファイルから 1 次元方向の軸データを取得する.
! 現在, linear, levels タイプのみ対応.
  implicit none
  character(*), intent(in) :: fname  ! GrADS のコントロールファイル
  character(*), intent(in) :: axis_name  ! 読み出す軸の名前.
                              ! xdef, ydef, zdef, tdef, pdef
  real, intent(inout) :: x(:)  ! 軸の座標値
  real, optional, intent(inout) :: y(:)  ! pdef の場合の y 軸の座標値
  character(*), optional, intent(inout) :: stime  ! initial date for tdef
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  character(1000), allocatable, dimension(:) :: ctl_cval
  character(100), allocatable, dimension(:) :: ctl_vval
  character(100), allocatable, dimension(:,:) :: ctl_hval
  integer :: nl, i, j, nx, ny, unitn
  real :: vmin, dv
  character(1000) :: tmpc, tmptmp
  character(100), allocatable, dimension(:) :: dummy
  character(3), dimension(12) :: mname
  type tmpdate
     integer :: yyyy
     integer :: mm
     integer :: dd
     integer :: hh
     integer :: mn
  end type

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  mname=(/'Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec'/)

  nl=line_number_counter( trim(fname), funit=unitn )
  nx=size(x)
  if(present(y))then
     if(trim(axis_name)=='pdef')then
        ny=size(y)
     else
        write(*,*) "ERROR : request_axis_grads."
        write(*,*) "If axis_name is pdef, argument y must need."
        write(*,*) "STOP."
        stop
     end if
  end if

  allocate(ctl_cval(nl))
  allocate(ctl_hval(2,nl))
  allocate(ctl_vval(13))
  allocate(dummy(nx+3))

  ctl_cval=''
  ctl_vval=''
  ctl_hval=''
  tmpc=''
  dummy=''

  call read_file_text( trim(fname), 1, nl, ctl_cval, forma='(a1000)',  &
  &                    funit=unitn )
  call read_file_text( trim(fname), 2, nl-1, ctl_hval(1:2,1:nl-1), funit=unitn )

  do i=1,nl
     tmpc=trim(adjustl(ctl_cval(i)))
     if(tmpc(1:4)==trim(axis_name))then
        if(tmpc(1:4)/='pdef')then
           call read_file_text( trim(fname), 5, 1, ctl_vval(1:5), skip=i-1,  &
  &                             funit=unitn )
           if(tmpc(1:4)/='tdef')then
              if(trim(ctl_vval(3))=='linear')then
                 read(ctl_vval(4),*) vmin
                 read(ctl_vval(5),*) dv
                 x=(/((vmin+dv*(i-1)),i=1,nx)/)
              else
                 if(trim(ctl_vval(3))=='levels')then
                    call read_file_text( trim(fname), nx+3, 1, dummy, skip=i-1,  &
  &                                      funit=unitn )
                    do j=4,nx+3
                       read(dummy(j),*) x(j-3)
                    end do
                 else
                    write(*,*) "ERROR (request_axis_grads) : ", trim(ctl_vval(3))
                    write(*,*) "Mode is neither 'levels' nor 'linear'."
                    write(*,*) "STOP."
                    stop
                 end if
              end if
           else
              if(trim(ctl_vval(3))=='linear')then  ! 現在, linear のみ対応.
                 tmptmp=trim(adjustl(ctl_vval(5)))
                 select case (tmptmp(len_trim(tmptmp)-1:len_trim(tmptmp)))
                 case ('mn')
                    tmptmp=trim(adjustl(ctl_vval(4)))
                    read(tmptmp(3:4),*) vmin
                 case ('hr')
                    tmptmp=trim(adjustl(ctl_vval(4)))
                    read(tmptmp(1:2),*) vmin
                 case ('dy')
                    tmptmp=trim(adjustl(ctl_vval(4)))
                    select case (len_trim(tmptmp))
                    case(12)
                       read(tmptmp(4:5),*) vmin
                    case(15)
                       read(tmptmp(7:8),*) vmin
                    end select
                 case ('yr')
                    tmptmp=trim(adjustl(ctl_vval(4)))
                    select case (len_trim(tmptmp))
                    case(12)
                       read(tmptmp(9:12),*) vmin
                    case(15)
                       read(tmptmp(12:15),*) vmin
                    end select
                 end select

                 tmptmp=trim(adjustl(ctl_vval(5)))
                 read(tmptmp(1:len_trim(tmptmp)-2),*) dv
                 x=(/((vmin+dv*(i-1)),i=1,nx)/)
                 stime=trim(adjustl(ctl_vval(4)))

!              else
!                 if(trim(ctl_vval(3))=='levels')then
!                    call read_file_text( trim(fname), nx+3, 1, dummy, skip=i-1 )
!                    do j=4,nx+3
!                       read(dummy(j),*) x(j-3)
!                    end do
!                 else
!                    write(*,*) "ERROR (request_axis_grads) : ", trim(ctl_vval(3))
!                    write(*,*) "Mode is neither 'levels' nor 'linear'."
!                    write(*,*) "STOP."
!                    stop
!                 end if
              end if
           end if
        else
           call read_file_text( trim(fname), 13, 1, ctl_vval(1:13), skip=i-1,  &
  &                             funit=unitn )
           read(ctl_vval(7),*) vmin
           read(ctl_vval(12),*) dv
           x=(/((dv*(i-int(vmin))),i=1,nx)/)
           read(ctl_vval(8),*) vmin
           read(ctl_vval(13),*) dv
           y=(/((dv*(i-int(vmin))),i=1,ny)/)
        end if
     end if
  end do

end subroutine

!-----------------------------------------------
!-----------------------------------------------

subroutine request_vardim_grads( fname, var_name, nz, funit )
! GrADS のコントロールファイルから var_name で指定された変数の高度方向の要素数
! を取得する.
  implicit none
  character(*), intent(in) :: fname  ! GrADS のコントロールファイル
  character(*), intent(in) :: var_name  ! 要素数を取得する変数の名前.
  integer, intent(inout) :: nz  ! 要素数
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  character(1000), allocatable, dimension(:) :: ctl_cval, ctl_vval
  character(100), allocatable, dimension(:,:) :: ctl_hval
  integer :: nl, i, counter, unitn
  character(1000) :: tmpc

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  nl=line_number_counter( trim(fname), funit=unitn )

  allocate(ctl_cval(nl))
  allocate(ctl_hval(2,nl))
  allocate(ctl_vval(5))

  ctl_vval=''
  ctl_hval=''
  ctl_cval=''
  tmpc=''

  call read_file_text( trim(fname), 1, nl, ctl_cval, forma='(a1000)',  &
  &                    funit=unitn )
  call read_file_text( trim(fname), 2, nl-1, ctl_hval(1:2,1:nl-1), funit=unitn )

  counter=0

  do i=1,nl
     tmpc=trim(adjustl(ctl_hval(1,i)))
     if(trim(tmpc)=='vars')then
        counter=1
     end if
     if(trim(tmpc)==trim(var_name).and.counter==1)then
        read(ctl_hval(2,i),*) nz
        exit
     end if
  end do

end subroutine

!-----------------------------------------------
!-----------------------------------------------

subroutine read_file_grads( fname, val_name, dim_len, intvz, intvt, val, funit )
! GrADS のコントロールファイルを読み込んで, val_name で指定した変数のデータを
! 取り出す.
  implicit none
  character(*), intent(in) :: fname  ! GrADS のコントロールファイル
  character(*), intent(in) :: val_name  ! 読み出す変数名
  integer, intent(in), dimension(4) :: dim_len  ! 各次元要素の数 (x,y,z,t) の順
  integer, intent(in), dimension(2) :: intvz  ! 高度方向の読み出す要素区間
  integer, intent(in), dimension(2) :: intvt  ! 時間方向の読み出す要素区間
  real, intent(inout),  &
  & dimension(dim_len(1),dim_len(2),intvz(1):intvz(2),intvt(1):intvt(2))  &
  &  :: val  ! 読み出した値が格納される変数.
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  character(1000), allocatable, dimension(:) :: ctl_cval
  character(100), allocatable, dimension(:,:) :: ctl_hval
  integer :: nl, nx, ny, nz, nt, mon_num
  integer :: i, j, unitn
  character(1000) :: tmpc, ctmp, header, footer, iname
  integer, dimension(10) :: cont_flag
  integer :: t_fact, year, month, day, hour, counter_start, read_flag, counter
  integer :: year_r, month_r, day_r, hour_r
  logical, dimension(4) :: t_fact_flag, date_flag

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  cont_flag=0

  nl=line_number_counter( trim(fname), funit=unitn )
  nx=dim_len(1)
  ny=dim_len(2)
  nz=dim_len(3)
  nt=dim_len(4)

  allocate(ctl_cval(nl))
  allocate(ctl_hval(10,nl))

  ctl_cval=''
  ctl_hval=''
  tmpc=''
  ctmp=''
  header=''
  footer=''
  iname=''

!-- ctl file reading

  call read_file_text( trim(fname), 1, nl, ctl_cval, forma='(a1000)',  &
  &                    funit=unitn )
  call read_file_text( trim(fname), 2, nl-1, ctl_hval(1:2,1:nl-1), funit=unitn )

!-- essential factor estimating

  do i=1,nl
     tmpc=trim(adjustl(ctl_cval(i)))
     if(tmpc(1:4)=='vars')then
        cont_flag(1)=i
     end if
     if(tmpc(1:7)=='endvars')then
        cont_flag(2)=i
     end if
     if(tmpc(1:4)=='dset')then
        cont_flag(4)=i
     end if
     if(tmpc(1:4)=='tdef')then
        cont_flag(5)=i
     end if
  end do

  call read_file_text( trim(fname), 4, 1,  &
  &                    ctl_hval(1:4,cont_flag(5):cont_flag(5)),  &
  &                    skip=cont_flag(5)-1, funit=unitn )

  do i=cont_flag(1),cont_flag(2)
     tmpc=trim(adjustl(ctl_hval(1,i)))
     if(trim(tmpc)==trim(val_name))then
        cont_flag(3)=i
     end if
  end do

  if(cont_flag(3)==0)then
     write(*,*) "*** ERROR (read_file_grads) *** : "  &
  &           //'"'//trim(val_name)//'" is not found. STOP.'
     stop
  end if

!-- reading array estimate

  read_flag=0
  do i=cont_flag(1)+1,cont_flag(3)-1
     read(ctl_hval(2,i),*) counter
     if(counter==0)then
        read_flag=read_flag+1
     else
        read_flag=read_flag+counter
     end if
  end do

!-- t_fact caluculate

  t_fact_flag(1:4)=.false.
  call read_file_text( trim(fname), 5, 1, ctl_hval(1:5,cont_flag(5)),  &
  &                    skip=cont_flag(5)-1, funit=unitn )
  tmpc=trim(ctl_hval(5,cont_flag(5)))
  select case (tmpc(len_trim(adjustl(tmpc))-1:len_trim(adjustl(tmpc))))
  case ('yr')
     t_fact_flag(1)=.true.
     read(tmpc(1:len_trim(adjustl(tmpc))-2),*) t_fact
  case ('mn')
     t_fact_flag(2)=.true.
     read(tmpc(1:len_trim(adjustl(tmpc))-2),*) t_fact
  case ('dy')
     t_fact_flag(3)=.true.
     read(tmpc(1:len_trim(adjustl(tmpc))-2),*) t_fact
  case ('hr')
     t_fact_flag(4)=.true.
     read(tmpc(1:len_trim(adjustl(tmpc))-2),*) t_fact
  end select

!-- file header estimate

  counter=0
  date_flag(1:4)=.false.
  tmpc=ctl_hval(2,cont_flag(4))
  do i=2,len_trim(adjustl(ctl_hval(2,cont_flag(4))))
     if(tmpc(i:i)=='%')then
        counter=counter+1
        if(counter==1)then
           counter_start=i
           header(1:i-2)=tmpc(2:i-1)
        end if
        select case (tmpc(i+1:i+2))
        case ('y4')
           date_flag(1)=.true.
           if(t_fact_flag(1).eqv..true.)then
              footer=tmpc(i+3:len_trim(adjustl(ctl_hval(2,cont_flag(4)))))
           end if
        case ('m2')
           date_flag(2)=.true.
           if(t_fact_flag(2).eqv..true.)then
              footer=tmpc(i+3:len_trim(adjustl(ctl_hval(2,cont_flag(4)))))
           end if
        case ('d2')
           date_flag(3)=.true.
           if(t_fact_flag(3).eqv..true.)then
              footer=tmpc(i+3:len_trim(adjustl(ctl_hval(2,cont_flag(4)))))
           end if
        case ('h2')
           date_flag(4)=.true.
           if(t_fact_flag(4).eqv..true.)then
              footer=tmpc(i+3:len_trim(adjustl(ctl_hval(2,cont_flag(4)))))
           end if
        end select
     end if
  end do

  if(counter==0)then
     tmpc=trim(adjustl(ctl_hval(2,cont_flag(4))))
     header=trim(tmpc(2:len_trim(adjustl(ctl_hval(2,cont_flag(4))))))
     footer=''
  end if
!-- initial time reading

  tmpc=trim(ctl_hval(4,cont_flag(5)))
  read(tmpc(1:2),*) hour
  select case (tmpc(3:3))
  case (':')
     read(tmpc(7:8),*) day
     read(tmpc(12:15),*) year
     mon_num=9  ! 月を表す文字が始まる位置
  case ('Z')
     read(tmpc(4:5),*) day
     read(tmpc(9:12),*) year
     mon_num=6  ! 月を表す文字が始まる位置
  end select

  select case (tmpc(mon_num:mon_num+2))
  case('Jan')
     month=1
  case('Feb')
     month=2
  case('Mar')
     month=3
  case('Apr')
     month=4
  case('May')
     month=5
  case('Jun')
     month=6
  case('Jul')
     month=7
  case('Aug')
     month=8
  case('Sep')
     month=9
  case('Oct')
     month=10
  case('Nov')
     month=11
  case('Dec')
     month=12
  end select

  year_r=year
  month_r=month
  day_r=day
  hour_r=hour

!-- reading data

  do i=intvt(1),intvt(2)

!-- default value setting

     iname=trim(header)
     do j=4,1,-1  ! 時間から足し合わせた方が都合がよいためループ逆回し
        select case (j)
        case (1)
           if(t_fact_flag(1).eqv..true.)then
              year=year_r+t_fact*(i-1)
           end if
        case (2)
           if(t_fact_flag(2).eqv..true.)then
              month=month_r+t_fact*(i-1)
           end if
        case (3)
           if(t_fact_flag(3).eqv..true.)then
              day=day_r+t_fact*(i-1)
           end if
        case (4)
           if(t_fact_flag(4).eqv..true.)then
              hour=hour_r+t_fact*(i-1)
           end if
        end select

        do while (hour>24)
           hour=hour-24
           day=day+1
        end do

        do while (day>28)
           do while (month>12)
              year=year+1
              month=month-12
           end do
           select case (month)
           case (1)
              if(day>31)then
                 month=month+1
                 day=day-31
              else
                 exit
              end if
           case (3)
              if(day>31)then
                 month=month+1
                 day=day-31
              else
                 exit
              end if
           case (5)
              if(day>31)then
                 month=month+1
                 day=day-31
              else
                 exit
              end if
           case (7)
              if(day>31)then
                 month=month+1
                 day=day-31
              else
                 exit
              end if
           case (8)
              if(day>31)then
                 month=month+1
                 day=day-31
              else
                 exit
              end if
           case (10)
              if(day>31)then
                 month=month+1
                 day=day-31
              else
                 exit
              end if
           case (12)
              if(day>31)then
                 month=month+1
                 day=day-31
              else
                 exit
              end if
           case (4)
              if(day>30)then
                 month=month+1
                 day=day-30
              else
                 exit
              end if
           case (6)
              if(day>30)then
                 month=month+1
                 day=day-30
              else
                 exit
              end if
           case (9)
              if(day>30)then
                 month=month+1
                 day=day-30
              else
                 exit
              end if
           case (11)
              if(day>30)then
                 month=month+1
                 day=day-30
              else
                 exit
              end if
           case (2)
              if(day>29.and.mod(year,4)==0)then
                 month=month+1
                 day=day-29
              else
                 if(day>28.and.mod(year,4)/=0)then
                    month=month+1
                    day=day-28
                 else
                    exit
                 end if
              end if
          end select
        end do
     end do

     do j=1,4
        if(date_flag(j).eqv..true.)then
           select case (j)
           case (1)
              write(ctmp,'(i4.4)') year
              iname=trim(iname)//trim(adjustl(ctmp))
           case (2)
              write(ctmp,'(i2.2)') month
              iname=trim(iname)//trim(adjustl(ctmp))
           case (3)
              write(ctmp,'(i2.2)') day
              iname=trim(iname)//trim(adjustl(ctmp))
           case (4)
              write(ctmp,'(i2.2)') hour
              iname=trim(iname)//trim(adjustl(ctmp))
           end select
        end if
     end do

     iname=trim(iname)//trim(footer)

     do j=intvz(1),intvz(2)
        call read_file( trim(iname), nx, ny, read_flag+j,  &
  &                     val(:,:,j,i) )
     end do
     write(*,*) "finishing to read file", trim(iname), nx, ny, read_flag+intvz(1), read_flag+intvz(2), i
  end do

end subroutine

!-----------------------------------------------
!-----------------------------------------------

subroutine auto_read_file( file_name, d2n, d3n, d2val, d3val, ad2val, ad3val,  &
  &                        nx, ny, nz, d2var, d3var, specnz, intvnz, funit )
! 読み込む変数の個数を指定することで, 不要な変数データはメモリに読み込まない
! ルーチン. (For CReSS)
! このルーチンを呼び出す前に必ず, val_counter で用意する変数の総数を求めておく.
  implicit none
  character(*), intent(in) :: file_name  ! 読み出すファイル名
  integer, intent(in) :: d2n  ! d2 変数用に用意すべき配列総数 (変数の種類数)
  integer, intent(in) :: d3n  ! d3 変数用に用意すべき配列総数 (変数の種類数)
  character(*), intent(in) :: d2val  ! CReSS のダンプファイルの最初の d2 データ
  character(*), intent(in) :: d3val  ! CReSS のダンプファイルの最初の d3 データ
  character(*), intent(in) :: ad2val  ! CReSS のダンプファイルの後の d2 データ
  character(*), intent(in) :: ad3val  ! CReSS のダンプファイルの後の d3 データ
  integer, intent(in) :: nx  ! x 方向の要素数
  integer, intent(in) :: ny  ! y 方向の要素数
  integer, intent(in) :: nz  ! z 方向の要素数
  integer, intent(in), optional :: specnz  ! nz について指定高度のみ取り出す
                      ! この場合, 必ず nz = 1 でなければならない.
  integer, intent(in), dimension(2), optional :: intvnz
                      ! nz について intvnz(1) から intvnz(2) までの高度を取り出す.
                      ! この場合, nz はその変数の全要素数でなければならない.
  real, dimension(nx,ny,d2n), intent(inout) :: d2var  ! d2 変数用配列
  real, dimension(nx,ny,nz,d3n), intent(inout) :: d3var  ! d3 変数用配列
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: d2m, d3m, d2am, d3am, i, j, k, all_count, d2_count, d3_count
  integer :: unitn

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  d2m=len_trim(adjustl(d2val))
  d3m=len_trim(adjustl(d3val))
  d2am=len_trim(adjustl(ad2val))
  d3am=len_trim(adjustl(ad3val))
write(*,*) "len=", d2m, d3m, d2am, d3am
  all_count=0
!-- ファイルの 1 レコード目から順に読むかどうかの判断を行って読む ---
!-- まず, 最初の d2 データについて
  j=0
  if(d2m>=1)then
  do i=1,d2m
     if(d2val(i:i)=='1')then
        j=j+1
        if(d2n<j)then
           write(*,*) "*** ERROR ***"
           write(*,*) "read array exceeds d2n. STOP."
           stop
        else
           call read_file( trim(adjustl(file_name)), nx, ny, i, d2var(:,:,j),  &
  &                        funit=unitn )
        end if
     end if
  end do
  end if
  d2_count=j  ! 最初の d2 データでいくら入ったか (読み飛ばしを入れない)
  all_count=all_count+d2m  ! 読み飛ばした分も含めて何レコード分読んだかのカウント

!-- 次, 最初の d3 データについて
  j=0
  if(d3m>=1)then
  do i=1,d3m
     if(d3val(i:i)=='1')then
        j=j+1
        if(d3n<j)then
           write(*,*) "*** ERROR ***"
           write(*,*) "read array exceeds d3n. STOP."
           stop
        else
           if(present(specnz))then
              call read_file( trim(adjustl(file_name)), nx, ny,  &
  &                           all_count+specnz, d3var(:,:,specnz,j),  &
  &                           funit=unitn )
           else
              if(present(intvnz))then
                 do k=intvnz(1),intvnz(2)
                    call read_file( trim(adjustl(file_name)), nx, ny,  &
  &                                 all_count+k, d3var(:,:,k,j),  &
  &                                 funit=unitn )
                 end do
              else
                 do k=1,nz
                    call read_file( trim(adjustl(file_name)), nx, ny,  &
  &                                 all_count+k, d3var(:,:,k,j),  &
  &                                 funit=unitn )
                 end do
              end if
           end if
        end if
     end if
     all_count=all_count+nz  ! if 文で読んでいなくても nz 行分カウント
  end do
  end if
  d3_count=j  ! 最初の d3 データでいくら入ったか (読み飛ばしを入れない)

!-- 続いて, 後の d2 データについて
  j=d2_count
  if(d2am>=1)then
  do i=1,d2am
     if(ad2val(i:i)=='1')then
        j=j+1
        if(d2n<j)then
           write(*,*) "*** ERROR ***"
           write(*,*) "read array exceeds d2n. STOP."
           stop
        else
           call read_file( trim(adjustl(file_name)), nx, ny,  &
  &                        all_count+i, d2var(:,:,j), funit=unitn )
        end if
     end if
  end do
  end if
  all_count=all_count+d2am  ! 読み飛ばした分も含めて何レコード分読んだかのカウント

!-- 最後に, 後の d3 データについて
  j=d3_count
  if(d3am>=1)then
  do i=1,d3am
     if(ad3val(i:i)=='1')then
        j=j+1
        if(d3n<j)then
           write(*,*) "*** ERROR ***"
           write(*,*) "read array exceeds d3n. STOP."
           stop
        else
           if(present(specnz))then
              call read_file( trim(adjustl(file_name)), nx, ny,  &
  &                           all_count+specnz, d3var(:,:,specnz,j),  &
  &                           funit=unitn )
           else
              if(present(intvnz))then
                 do k=intvnz(1),intvnz(2)
                    call read_file( trim(adjustl(file_name)), nx, ny,  &
  &                                 all_count+k, d3var(:,:,k,j),  &
  &                                 funit=unitn )
                 end do
              else
                 do k=1,nz
                    call read_file( trim(adjustl(file_name)), nx, ny,  &
  &                                 all_count+k, d3var(:,:,k,j),  &
  &                                 funit=unitn )
                 end do
              end if
           end if
        end if
     end if
     all_count=all_count+nz  ! if 文で読んでいなくても nz 行分カウント
  end do
  end if

end subroutine

!-----------------------------------
!-----------------------------------

subroutine read_file_gtool3( fname, nx, ny, nz, tstr, tstep, tnum, val, funit )
! gtool3 形式のデータを読み出す.
  implicit none
  character(*), intent(in) :: fname  ! 読み込むデータファイル
  integer, intent(in) :: nx  ! x 方向の配列数
  integer, intent(in) :: ny  ! y 方向の配列数
  integer, intent(in) :: nz  ! z 方向の配列数
  integer, intent(in) :: tstr  ! 読み込む時刻 (初期値から何ステップ目かで指定).
  integer, intent(in) :: tstep  ! 読み飛ばす時間間隔 (1 で毎時読み込み)
  integer, intent(in) :: tnum  ! 読み込む時間の数
  real, intent(inout) :: val(nx,ny,nz,tnum)
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  character(16) :: header(64)  ! ヘッダー情報読み飛ばし用
  integer :: i, j, k, l, tmax, counter, unitn
  real :: skip(nx,ny,nz)

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  tmax=tstr+tstep*(tnum-1)

  open(unit=unitn,file=trim(fname),form='unformatted',access='sequential',status='old')
     if(tstr/=1)then
        do i=1,tstr-1
           read(unitn) header
           read(unitn) skip
        end do
     end if

     counter=0

     do i=tstr,tmax

        read(unitn) header
        read(unitn) skip

        if(mod((i-tstr),tstep)==0)then
           counter=counter+1
           do l=1,nz
              do k=1,ny
                 do j=1,nx
                    val(j,k,l,counter)=skip(j,k,l)
                 end do
              end do
           end do
        end if

     end do
  close(unit=unitn)

end subroutine

!-----------------------------------
!-----------------------------------

subroutine read_file_gtool3_header_c( fname, header, val, funit )
! gtool3 形式のデータのヘッダーから, 必要な変数名を入力し, その値を返す.
! 参照変数名は http://www.riam.kyushu-u.ac.jp/taikai/lab/others/Gtool/node111.html 参照.
! ただし, Fortran の予約語である unit, size については, uni, siz を使用.
  implicit none
  character(*), intent(in) :: fname
  character(*), intent(in) :: header(:)
  character(16), intent(inout) :: val(size(header))
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer, parameter :: vnum=43
  character(16) :: raw_dat(64)
  character(6) :: signc(vnum)
  integer :: signi(vnum)
  integer :: i, j, unitn

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  signc=(/'dset  ',  &
  &       'item  ',  &
  &       'edit1 ',  &
  &       'edit2 ',  &
  &       'edit3 ',  &
  &       'edit4 ',  &
  &       'edit5 ',  &
  &       'edit6 ',  &
  &       'edit7 ',  &
  &       'edit8 ',  &
  &       'titl1 ',  &
  &       'titl2 ',  &
  &       'uni   ',  &
  &       'ettl1 ',  &
  &       'ettl2 ',  &
  &       'ettl3 ',  &
  &       'ettl4 ',  &
  &       'ettl5 ',  &
  &       'ettl6 ',  &
  &       'ettl7 ',  &
  &       'ettl8 ',  &
  &       'date  ',  &
  &       'utim  ',  &
  &       'aitm1 ',  &
  &       'aitm2 ',  &
  &       'aitm3 ',  &
  &       'dfmt  ',  &
  &       'coptn ',  &
  &       'utim2 ',  &
  &       'memo01',  &
  &       'memo02',  &
  &       'memo03',  &
  &       'memo04',  &
  &       'memo05',  &
  &       'memo06',  &
  &       'memo07',  &
  &       'memo08',  &
  &       'memo09',  &
  &       'memo10',  &
  &       'cdate ',  &
  &       'csign ',  &
  &       'mdate ',  &
  &       'msign ' /)

  signi=(/2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 14, 15, 16, 17, 18, 19, 20, 21,  &
  &       22, 23, 24, 26, 27, 29, 32, 35, 38, 45, 49, 50, 51, 52, 53, 54,  &
  &       55, 56, 57, 58, 59, 60, 61, 62, 63 /)

  open(unit=unitn,file=trim(fname),form='unformatted',access='sequential',status='old')
     read(unitn) raw_dat
  close(unit=unitn)

  do i=1,size(header)
     do j=1,vnum
        if(trim(header(i))==trim(signc(j)))then
           val(i)=raw_dat(signi(j))
           exit
        end if
     end do
  end do

end subroutine

!-----------------------------------
!-----------------------------------

subroutine read_file_gtool3_header_i( fname, header, val, funit )
! gtool3 形式のデータのヘッダーから, 必要な変数名を入力し, その値を返す.
! 参照変数名は http://www.riam.kyushu-u.ac.jp/taikai/lab/others/Gtool/node111.html 参照.
! ただし, Fortran の予約語である unit, size については, uni, siz を使用.
  implicit none
  character(*), intent(in) :: fname
  character(*), intent(in) :: header(:)
  integer, intent(inout) :: val(size(header))
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer, parameter :: vnum=15
  character(16) :: raw_dat(64)
  character(6) :: signc(vnum)
  integer :: signi(vnum)
  integer :: i, j, unitn

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  signc=(/'idfm  ',  &
  &       'fnum  ',  &
  &       'dnum  ',  &
  &       'time  ',  &
  &       'tdur  ',  &
  &       'astr1 ',  &
  &       'aend1 ',  &
  &       'astr2 ',  &
  &       'aend2 ',  &
  &       'astr3 ',  &
  &       'aend3 ',  &
  &       'styp  ',  &
  &       'ioptn ',  &
  &       'time2 ',  &
  &       'size  ' /)

  signi=(/1, 12, 13, 25, 28, 30, 31, 33, 34, 36, 37, 44, 46, 48, 64 /)

  open(unit=unitn,file=trim(fname),form='unformatted',access='sequential',status='old')
     read(unitn) raw_dat
  close(unit=unitn)

  do i=1,size(header)
     do j=1,vnum
        if(trim(header(i))==trim(signc(j)))then
           read(raw_dat(signi(j)),'(i16)') val(i)
           exit
        end if
     end do
  end do

end subroutine

!-----------------------------------
!-----------------------------------

subroutine read_file_gtool3_header_r( fname, header, val, funit )
! gtool3 形式のデータのヘッダーから, 必要な変数名を入力し, その値を返す.
! 参照変数名は http://www.riam.kyushu-u.ac.jp/taikai/lab/others/Gtool/node111.html 参照.
! ただし, Fortran の予約語である unit, size については, uni, siz を使用.
  implicit none
  character(*), intent(in) :: fname
  character(*), intent(in) :: header(:)
  real, intent(inout) :: val(size(header))
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer, parameter :: vnum=6
  character(16) :: raw_dat(64)
  character(10) :: signc(vnum)
  integer :: signi(vnum)
  integer :: i, j, unitn

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  signc=(/'miss  ',  &
  &       'dmin  ',  &
  &       'dmax  ',  &
  &       'divs  ',  &
  &       'divl  ',  &
  &       'roptn ' /)

  signi=(/39, 40, 41, 42, 43, 47 /)

  open(unit=unitn,file=trim(fname),form='unformatted',access='sequential',status='old')
     read(unitn) raw_dat
  close(unit=unitn)

  do i=1,size(header)
     do j=1,vnum
        if(trim(header(i))==trim(signc(j)))then
           if(trim(header(i))=='roptn')then
              read(raw_dat(signi(j)),'(e16.7)') val(i)
           else
              read(raw_dat(signi(j)),'(e15.7)') val(i)
           end if
           exit
        end if
     end do
  end do

end subroutine

!-----------------------------------
!-----------------------------------

integer function line_number_counter( fname, funit )
  ! テキストデータの行数を計算する関数
  implicit none
  character(*), intent(in) :: fname  ! 読み取るテキストファイル
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
!  integer, intent(inout) :: res
  integer :: i, err, unitn
  character(1) :: dummy

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  i=0
  open(unit=unitn,file=trim(fname),iostat=err,status='old')
  do while(err==0)
     read(unitn,*,iostat=err) dummy
     i=i+1
  end do
  close(unit=unitn)

  line_number_counter=i-1
  return

end function

!-----------------------------------
!-----------------------------------

subroutine read_file_text( fname, nx, ny, val, skip, forma, funit )
! テキスト形式のデータを読み込むルーチン
! forma と nx, ny の関係は 4x3 の i3 (空白込み) の行列データの場合,
! forma = '(3a3)' という form で読み込むなら, nx = 3, ny = 4
! forma = '(a)' という form で読み込むなら, nx = 1, ny = 4 となる.
  implicit none
  character(*), intent(in) :: fname  ! 読み込むテキストファイル
  integer, intent(in) :: nx  ! 読み込むファイルの列数
  integer, intent(in) :: ny  ! 読み込むファイルの行数
  character(*), intent(inout) :: val(nx,ny)  ! 文字列データにも対応するため, 文字列で結果を返す.
  integer, intent(in), optional :: skip  ! 行頭何行を読み飛ばすか. default は 0.
  character(*), intent(in), optional :: forma  ! read のフォーマット '(f4.1)' など. デフォルトでは read のデフォルトフォーマット使用.
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  integer :: i, j, unitn
  character(1) :: dummy

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  open(unit=unitn, file=trim(fname), status='old')
  if(present(skip))then
     do i=1,skip
        read(unitn,*) dummy
     end do
  end if

  if(present(forma))then
     do j=1,ny
        read(unitn,forma) (val(i,j),i=1,nx)
     end do
  else
     do j=1,ny
        read(unitn,*) (val(i,j),i=1,nx)
     end do
  end if
  close(unit=unitn)

end subroutine

!-----------------------------------
!-----------------------------------

subroutine read_mgdsst( fname, sst, funit )
! mgdsst データを読み込むためのルーチン
! 読み込み用データは 1440 x 720 で固定.
! これは, mgdsst のフォーマットからくるものである.
  implicit none
  integer, parameter :: nx=1440  ! 経度方向の格子点数
  integer, parameter :: ny=720  ! 緯度方向の格子点数
  character(*), intent(in) :: fname  ! 読み込む mgdsst ルーチン
  real, intent(inout), dimension(nx,ny) :: sst  ! 読み込んだ SST データ [K].
                              ! ただし, 999.0 が未定義, 888.0 が海氷.
  integer, intent(in), optional :: funit   ! ファイルユニット番号.
                       ! デフォルト = 11.
  character(3), dimension(nx,ny) :: sstc
  character(10) :: formatt
  integer :: j, k, unitn

  if(present(funit))then
     unitn=funit
  else
     unitn=11
  end if

  formatt='(1440a3)'

  call read_file_text( trim(fname), nx, ny, sstc, skip=1, forma=trim(formatt),  &
  &                    funit=unitn )
!  open(unit=12,file=trim(dat_list(i)),status='old')
!     read(12,*) sstc(1,1)
  do k=1,ny
!     read(12,formatt) (sstc(j,k),j=1,nx)
     do j=1,nx
        if(sstc(j,k)=='888')then
           sst(j,ny-k+1)=263.15
        else if(sstc(j,k)=='999')then
           sst(j,ny-k+1)=999.0
        else
           read(sstc(j,k),*) sst(j,ny-k+1)
           sst(j,ny-k+1)=sst(j,ny-k+1)/10.0+273.15
!           write(*,*) j,k,sstc(j,k),sst(j,k)
        end if
!        tmp(j,ny-k+1)=sst(j,k)
!        sst(j,k)=tmp(j,k)
     end do
  end do

end subroutine

!-----------------------------------
!  private routine
!-----------------------------------

subroutine check_err( err, chara )
! outputting file reading error message and stopping main program.
  integer, intent(in) :: err
  character(*), intent(in) :: chara

  if(err/=0)then
     write(*,*) "*** MESSAGE (file_operate) ***"
     write(*,*) trim(adjustl(chara))
     stop
  end if

end subroutine

end module
