require "numru/gphys"
require "numru/ganalysis/ganalysis_ext"

module NumRu
  module GAnalysis
    module PDE
      module_function
      #
      # = Solve 2-dim 2nd order PDE by the succesive over-relaxation method
      # 
      # ARGUMENTS.
      # * z [in-out, 2D dfloat NArray] : When in, initial values. When out,
      #   the result. Boundary values are not changed (Dirichlet problem)
      #   (developper's memo: Neuman cond can be supported as an option)
      # * a, b, c, d, e, f [in, 2D dfloat NArray with the same shape as z] :
      #   Specifies the PDF as
      #      a z_xx + 2*b z_xy + c z_yy + d z_x + e z_y = f
      #               ^^^
      #   Please note that 2*b is the coefficient of z_xy.
      #   x is the first (0th) NArray dimension.
      # * dx, dy [Float] : grid spacings
      # * ome [Float] : the over-relaxation parameter 1<=ome<2.
      #   For a large-scale poisson equation, should be ~ 2 (e.g. 1.9).
      # * eps [optional, Float] : conversion threshold (e.g. 1e-5).
      #   Compared with |increment|/|F| (| | is the L2 norm).
      #   When |f| == 0 (homogenous), compared simply with |increment|.
      # * maxi [optional, Integer]: max iteration times(default, 5*[nx,ny].max).
      #   Since SOR (or the Gauss-Seidel method) acts like a stepwise diffusion,
      #   it must be greater than the along-x and along-y grid sizes.
      # * ignore_non_eliptic : if true (non-nil,non-false), do not raise
      #   error when the coeeficients are not entirely eliptic.
      # 
      # RETURN VALUE
      # * |last increment|/|F| of |last increment| (to be compared with eps)
      # 
      def SOR_2D_2ndorder(z, a, b, c, d, e, f, dx, dy, ome, eps: nil, maxi: nil,
                          ignore_non_eliptic: false)
        lf = Math.sqrt( (f**2).sum )  # L2 norm of f
        lf = 1.0 if lf==0   # when |f|==0 (homogeneous), no-division by |f|.
        res = nil
        if !maxi
          maxi = [ z.shape[0], z.shape[1] ].max * 5  # default max itr times
        end
        convd = false
        for i in 0...maxi
          res = SOR_2D_2ndorder_1step(z, a, b, c, d, e, f, dx, dy, ome,
                                      ignore_non_eliptic)
          #p res/lf, z
          if (eps && res/lf < eps)
            convd = true
            break
          end
        end
        if (eps && !convd)
          raise("Max iteration times (#{maxi}) reached before convergence" +
                " (#{res/lf} > #{eps}). Change ome (#{ome}) or specify maxi.")
        end
        #p i,res/lf #, z
        res/lf
      end
    end
  end
end

################################################
##### test part ######
if $0 == __FILE__
  require "numru/ggraph"
  include NumRu

  #nx = ny = 9
  nx = ny = 91
  z = NArray.float(nx,ny)
  a = c = NArray.float(nx,ny).fill!(1.0)
  b = d = e = NArray.float(nx,ny)
  f = NArray.float(nx,ny)
  f[nx/2,ny/2] = 1.0
  dx = dy = 1.0
  #ome = 1.9
  #GAnalysis::PDE.SOR_2D_2ndorder(z, a, b, c, d, e, f, dx, dy, ome)
=begin
  (1.0..1.91).step(0.1){|ome|
    puts "ome = #{ome}"
    GAnalysis::PDE.SOR_2D_2ndorder(z, a, b, c, d, e, f, dx, dy, ome)
  }
=end
  eps = 1e-6
  ome = 1.95
  GAnalysis::PDE.SOR_2D_2ndorder(z, a, b, c, d, e, f, dx, dy, ome, eps: eps)
end
