#lang scribble/doc
@(require scribble/manual scribble/eval "guide-utils.rkt"
          (for-label racket/flonum racket/future))

@(define future-eval (make-base-eval))
@(interaction-eval #:eval future-eval (require racket/future 
                                               future-visualizer/private/visualizer-drawing 
                                               future-visualizer/private/visualizer-data
                                               pict))

@(interaction-eval #:eval future-eval
   (define (show-timeline log
                          #:selected-event-index [sel #f]
                          #:scale [s 0.8]
                          #:width [width 700]
                          #:height [height 300])
      (scale
           (timeline-pict log 
                          #:x 0 
                          #:y 0 
                          #:width width
                          #:height height
                          #:timeline-width 500
                          #:selected-event-index sel)
            s)))

@title[#:tag "effective-futures"]{Parallelism with Futures}

The @racketmodname[racket/future] library provides support for
performance improvement through parallelism with @deftech{futures} and the @racket[future]
and @racket[touch] functions. Achieving parallelism through
those constructs depends on avoiding @tech{blocking} operations: anything that inspects
the full continuation or requires atomic execution relative to Racket threads. The caveats
in @secref["DrRacket-perf"] also apply to futures; notably,
debugging instrumentation currently defeats futures.

@margin-note{Other functions, such as @racket[thread], support the
creation of reliably concurrent tasks. However, threads never run truly
in parallel, even if the hardware and operating system support
parallelism.}

As a starting example, the @racket[any-double?] function below takes a
list of numbers and determines whether any number in the list has a
double that is also in the list:

@racketblock[
(define (any-double? l)
  (for/or ([i (in-list l)])
    (for/or ([i2 (in-list l)])
      (= i2 (* 2 i)))))
]

This function runs in quadratic time, so it can take a long time (on
the order of a second) on large lists like @racket[l1] and
@racket[l2]:

@racketblock[
(define l1 (for/list ([i (in-range 5000)]) 
             (+ (* 2 i) 1)))
(define l2 (for/list ([i (in-range 5000)]) 
             (- (* 2 i) 1)))
(or (any-double? l1)
    (any-double? l2))
]

The best way to speed up @racket[any-double?]  is to use a different
algorithm. However, on a machine that offers at least two processing
units, the example above can run in about half the time using
@racket[future] and @racket[touch]:

@racketblock[
(let ([f (future (lambda () (any-double? l2)))])
  (or (any-double? l1)
      (touch f)))
]

The future @racket[f] runs @racket[(any-double? l2)] in parallel to
@racket[(any-double? l1)], and the result for @racket[(any-double?
l2)] becomes available about the same time that it is demanded by
@racket[(touch f)].

Futures run in parallel as long as they can do so safely and
independent of any continuation context that a @racket[touch] might
provide. Continuation context can include an exception handler,
parameter value, continuation prompts, or other values accessible via
continuation marks. Safety concerns shared mutable objects, such as
input ports, output ports, and @racket[equal?]-based hash tables,
where concurrent access is managed either internally or explicitly via
locks. The remainder of this section works through an example to illustrate
safety and continuation obstacles, and it also shows how the
@defterm{futures visualizer} (provided by @racketmodname[future-visualizer #:indirect])
can help shed light on those obstacles.

Consider the following core of a Mandelbrot-set computation:

@racketblock[
(define (mandelbrot iterations x y n)
  (printf "starting\n")
  (let ([ci (- (/ (* 2.0 y) n) 1.0)]
        [cr (- (/ (* 2.0 x) n) 1.5)])
    (let loop ([i 0] [zr 0.0] [zi 0.0])
      (if (> i iterations)
          i
          (let ([zrq (* zr zr)]
                [ziq (* zi zi)])
            (cond
              [(> (+ zrq ziq) 4) i]
              [else (loop (add1 i)
                          (+ (- zrq ziq) cr)
                          (+ (* 2 zr zi) ci))]))))))
]

The expressions @racket[(mandelbrot 10000000 62 500 1000)] and
@racket[(mandelbrot 10000000 62 501 1000)] each take a while to
produce an answer. Computing them both, of course, takes twice as
long:

@racketblock[
(list (mandelbrot 10000000 62 500 1000)
      (mandelbrot 10000000 62 501 1000))
]

Unfortunately, attempting to run the two computations in parallel by
using one @racket[future] does not improve performance:

@racketblock[
 (let ([f (future (lambda () (mandelbrot 10000000 62 501 1000)))])
   (list (mandelbrot 10000000 62 500 1000)
         (touch f)))
]

To see why, use the futures visualizer from @racketmodname[future-visualizer #:indirect] to
visualize the execution of the above program.
@;
The visualizer opens a window showing a graphical view of a trace of the computation.
The upper-left portion of the window contains an execution timeline:

@(interaction-eval 
  #:eval future-eval 
  (define bad-log 
    (list
(indexed-future-event 0 '#s(future-event #f 0 create 1677475790052.766 #f 1))
(indexed-future-event 1 '#s(future-event 1 1 start-work 1677475790052.928 #f #f))
(indexed-future-event 2 '#s(future-event 1 1 block 1677475790052.93 #f #f))
(indexed-future-event 3 '#s(future-event 1 1 suspend 1677475790052.935 #f #f))
(indexed-future-event 4 '#s(future-event 1 1 end-work 1677475790052.937 #f #f))
(indexed-future-event 5 '#s(gc-info minor 239724368 360290056 0 227775472 360290056 2361 2362 1677475790055.524 1677475790056.868))
(indexed-future-event 6 '#s(gc-info minor 236174928 360290056 0 227777600 360290056 2365 2365 1677475790059.686 1677475790059.709))
(indexed-future-event 7 '#s(gc-info minor 236172144 360290056 0 227779552 360290056 2368 2368 1677475790062.466 1677475790062.484))
(indexed-future-event 8 '#s(gc-info minor 236174064 360290056 0 227792672 360290056 2370 2371 1677475790065.233 1677475790065.85))
(indexed-future-event 9 '#s(gc-info minor 236187280 360290056 0 227775328 360290056 2374 2374 1677475790068.628 1677475790068.651))
(indexed-future-event 10 '#s(gc-info minor 236169824 360290056 0 227777280 360290056 2376 2376 1677475790071.401 1677475790071.417))
(indexed-future-event 11 '#s(gc-info minor 236176640 360290056 0 227779312 360290056 2379 2379 1677475790074.2 1677475790074.216))
(indexed-future-event 12 '#s(gc-info minor 236173808 360290056 0 227777360 360290056 2382 2382 1677475790076.968 1677475790076.993))
(indexed-future-event 13 '#s(gc-info minor 236171904 360290056 0 227779280 360290056 2385 2385 1677475790079.748 1677475790079.764))
(indexed-future-event 14 '#s(gc-info minor 236173792 360290056 0 227781232 360290056 2387 2387 1677475790082.511 1677475790082.526))
(indexed-future-event 15 '#s(gc-info minor 236180640 360290056 0 227783296 360290056 2390 2390 1677475790085.305 1677475790085.32))
(indexed-future-event 16 '#s(gc-info minor 236177792 360290056 0 227781264 360290056 2393 2393 1677475790088.101 1677475790088.131))
(indexed-future-event 17 '#s(gc-info minor 236175824 360290056 0 227783168 360290056 2396 2396 1677475790091.046 1677475790091.08))
(indexed-future-event 18 '#s(gc-info minor 236177664 360290056 0 227785152 360290056 2398 2398 1677475790093.853 1677475790093.869))
(indexed-future-event 19 '#s(gc-info minor 236184544 360290056 0 227787216 360290056 2401 2401 1677475790096.647 1677475790096.663))
(indexed-future-event 20 '#s(gc-info minor 236181712 360290056 0 228090016 360290056 2404 2408 1677475790099.411 1677475790103.387))
(indexed-future-event 21 '#s(gc-info minor 236484592 360290056 0 227777104 360290056 2411 2411 1677475790106.259 1677475790106.442))
(indexed-future-event 22 '#s(gc-info minor 236171600 360290056 0 227779056 360290056 2413 2414 1677475790109.239 1677475790109.257))
(indexed-future-event 23 '#s(gc-info minor 236173616 360290056 0 227780992 360290056 2416 2416 1677475790112.028 1677475790112.044))
(indexed-future-event 24 '#s(gc-info minor 236180272 360290056 0 227779104 360290056 2419 2419 1677475790114.822 1677475790114.848))
(indexed-future-event 25 '#s(gc-info minor 236173712 360290056 0 227781040 360290056 2422 2422 1677475790117.599 1677475790117.615))
(indexed-future-event 26 '#s(gc-info minor 236177376 360290056 0 227848400 360290056 2424 2424 1677475790120.364 1677475790120.381))
(indexed-future-event 27 '#s(gc-info minor 236242976 360290056 0 227850368 360290056 2427 2427 1677475790123.133 1677475790123.148))
(indexed-future-event 28 '#s(gc-info minor 236249648 360290056 0 227846832 360290056 2430 2430 1677475790125.928 1677475790125.948))
(indexed-future-event 29 '#s(gc-info minor 236241376 360290056 0 227848752 360290056 2433 2433 1677475790128.7 1677475790128.715))
(indexed-future-event 30 '#s(gc-info minor 236243264 360290056 0 227850704 360290056 2435 2435 1677475790131.465 1677475790131.479))
(indexed-future-event 31 '#s(gc-info minor 236245312 360290056 0 227852672 360290056 2438 2438 1677475790134.234 1677475790134.249))
(indexed-future-event 32 '#s(gc-info minor 236247168 360290056 0 227850640 360290056 2441 2441 1677475790137.014 1677475790137.032))
(indexed-future-event 33 '#s(gc-info minor 236250000 360290056 0 227852640 360290056 2443 2443 1677475790139.813 1677475790139.828))
(indexed-future-event 34 '#s(gc-info minor 236247136 360290056 0 227854624 360290056 2446 2446 1677475790142.589 1677475790142.604))
(indexed-future-event 35 '#s(gc-info minor 236249168 360290056 0 227856576 360290056 2449 2449 1677475790145.362 1677475790145.376))
(indexed-future-event 36 '#s(gc-info minor 236251088 360290056 0 227848592 360290056 2452 2452 1677475790148.134 1677475790148.16))
(indexed-future-event 37 '#s(gc-info minor 236248000 360290056 0 227850592 360290056 2454 2454 1677475790150.943 1677475790150.958))
(indexed-future-event 38 '#s(gc-info minor 236245088 360290056 0 227852560 360290056 2457 2457 1677475790153.707 1677475790153.721))
(indexed-future-event 39 '#s(gc-info minor 236247120 360290056 0 227854496 360290056 2460 2460 1677475790156.469 1677475790156.484))
(indexed-future-event 40 '#s(gc-info minor 236248992 360290056 0 227852544 360290056 2462 2463 1677475790159.231 1677475790159.249))
(indexed-future-event 41 '#s(gc-info minor 236251936 360290056 0 227854576 360290056 2465 2465 1677475790162.026 1677475790162.042))
(indexed-future-event 42 '#s(gc-info minor 236249072 360290056 0 227856528 360290056 2468 2468 1677475790164.791 1677475790164.806))
(indexed-future-event 43 '#s(gc-info minor 236251104 360290056 0 227858496 360290056 2471 2471 1677475790167.557 1677475790167.572))
(indexed-future-event 44 '#s(gc-info minor 236252992 360290056 0 227856448 360290056 2473 2473 1677475790170.321 1677475790170.338))
(indexed-future-event 45 '#s(gc-info minor 236251008 360290056 0 227858352 360290056 2476 2476 1677475790173.098 1677475790173.113))
(indexed-future-event 46 '#s(gc-info minor 236257632 360290056 0 227860400 360290056 2479 2479 1677475790175.935 1677475790175.952))
(indexed-future-event 47 '#s(gc-info minor 236255008 360290056 0 227862368 360290056 2482 2482 1677475790178.751 1677475790178.767))
(indexed-future-event 48 '#s(gc-info minor 236256864 360290056 0 227860384 360290056 2484 2484 1677475790181.559 1677475790181.578))
(indexed-future-event 49 '#s(gc-info minor 236254960 360290056 0 227862320 360290056 2487 2487 1677475790184.332 1677475790184.347))
(indexed-future-event 50 '#s(gc-info minor 236261600 360290056 0 227864400 360290056 2490 2490 1677475790187.123 1677475790187.138))
(indexed-future-event 51 '#s(gc-info minor 236258944 360290056 0 227866352 360290056 2493 2493 1677475790189.886 1677475790189.901))
(indexed-future-event 52 '#s(gc-info minor 236260864 360290056 0 227864352 360290056 2495 2495 1677475790192.66 1677475790192.684))
(indexed-future-event 53 '#s(gc-info minor 236258960 360290056 0 227866256 360290056 2498 2498 1677475790195.433 1677475790195.448))
(indexed-future-event 54 '#s(gc-info minor 236265552 360290056 0 227868304 360290056 2501 2501 1677475790198.224 1677475790198.24))
(indexed-future-event 55 '#s(gc-info minor 236262864 360290056 0 227870240 360290056 2503 2503 1677475790200.996 1677475790201.011))
(indexed-future-event 56 '#s(gc-info minor 236264736 360290056 0 227868288 360290056 2506 2506 1677475790203.77 1677475790203.786))
(indexed-future-event 57 '#s(gc-info minor 236262832 360290056 0 227870208 360290056 2509 2509 1677475790206.536 1677475790206.551))
(indexed-future-event 58 '#s(gc-info minor 236264720 360290056 0 227872160 360290056 2512 2512 1677475790209.316 1677475790209.33))
(indexed-future-event 59 '#s(gc-info minor 236271568 360290056 0 227874224 360290056 2514 2514 1677475790212.106 1677475790212.12))
(indexed-future-event 60 '#s(gc-info minor 236268720 360290056 0 227872192 360290056 2517 2517 1677475790214.867 1677475790214.884))
(indexed-future-event 61 '#s(gc-info minor 236266752 360290056 0 227874096 360290056 2520 2520 1677475790217.635 1677475790217.651))
(indexed-future-event 62 '#s(gc-info minor 236268592 360290056 0 227876080 360290056 2522 2522 1677475790220.397 1677475790220.411))
(indexed-future-event 63 '#s(gc-info minor 236275472 360290056 0 227878144 360290056 2525 2525 1677475790223.186 1677475790223.201))
(indexed-future-event 64 '#s(gc-info minor 236272640 360290056 0 227876128 360290056 2528 2528 1677475790225.95 1677475790225.968))
(indexed-future-event 65 '#s(gc-info minor 236270704 360290056 0 227878064 360290056 2531 2531 1677475790228.716 1677475790228.73))
(indexed-future-event 66 '#s(gc-info minor 236272560 360290056 0 227880016 360290056 2533 2533 1677475790231.475 1677475790231.491))
(indexed-future-event 67 '#s(gc-info minor 236279344 360290056 0 227882064 360290056 2536 2536 1677475790234.267 1677475790234.283))
(indexed-future-event 68 '#s(gc-info minor 236276576 360290056 0 228230432 360290056 2539 2547 1677475790237.03 1677475790245.166))
(indexed-future-event 69 '#s(gc-info minor 236625040 360290056 0 227850928 360290056 2550 2550 1677475790248.041 1677475790248.313))
(indexed-future-event 70 '#s(gc-info minor 236245424 360290056 0 227852880 360290056 2553 2553 1677475790251.07 1677475790251.087))
(indexed-future-event 71 '#s(gc-info minor 236247456 360290056 0 227854864 360290056 2555 2555 1677475790253.835 1677475790253.85))
(indexed-future-event 72 '#s(gc-info minor 236254144 360290056 0 227852976 360290056 2558 2558 1677475790256.625 1677475790256.65))
(indexed-future-event 73 '#s(gc-info minor 236247520 360290056 0 227854896 360290056 2561 2561 1677475790259.401 1677475790259.416))
(indexed-future-event 74 '#s(gc-info minor 236249408 360290056 0 227856848 360290056 2563 2563 1677475790262.166 1677475790262.18))
(indexed-future-event 75 '#s(gc-info minor 236251456 360290056 0 227858816 360290056 2566 2566 1677475790264.927 1677475790264.942))
(indexed-future-event 76 '#s(gc-info minor 236258112 360290056 0 227856880 360290056 2569 2569 1677475790267.72 1677475790267.739))
(indexed-future-event 77 '#s(gc-info minor 236251440 360290056 0 227858784 360290056 2572 2572 1677475790270.486 1677475790270.501))
(indexed-future-event 78 '#s(gc-info minor 236253280 360290056 0 227860768 360290056 2574 2574 1677475790273.246 1677475790273.26))
(indexed-future-event 79 '#s(gc-info minor 236255312 360290056 0 227862720 360290056 2577 2577 1677475790276.009 1677475790276.024))
(indexed-future-event 80 '#s(gc-info minor 236262000 360290056 0 227860816 360290056 2580 2580 1677475790278.797 1677475790278.817))
(indexed-future-event 81 '#s(gc-info minor 236255392 360290056 0 227862752 360290056 2582 2582 1677475790281.564 1677475790281.578))
(indexed-future-event 82 '#s(gc-info minor 236257248 360290056 0 227864704 360290056 2585 2585 1677475790284.326 1677475790284.341))
(indexed-future-event 83 '#s(gc-info minor 236259264 360290056 0 227866640 360290056 2588 2588 1677475790287.086 1677475790287.1))
(indexed-future-event 84 '#s(gc-info minor 236261136 360290056 0 227864720 360290056 2591 2591 1677475790289.852 1677475790289.873))
(indexed-future-event 85 '#s(gc-info minor 236264112 360290056 0 227866720 360290056 2593 2593 1677475790292.655 1677475790292.669))
(indexed-future-event 86 '#s(gc-info minor 236261216 360290056 0 227868672 360290056 2596 2596 1677475790295.416 1677475790295.431))
(indexed-future-event 87 '#s(gc-info minor 236263248 360290056 0 227870640 360290056 2599 2599 1677475790298.181 1677475790298.197))
(indexed-future-event 88 '#s(gc-info minor 236265136 360290056 0 227868592 360290056 2601 2601 1677475790300.942 1677475790300.961))
(indexed-future-event 89 '#s(gc-info minor 236267920 360290056 0 227870608 360290056 2604 2604 1677475790303.735 1677475790303.751))
(indexed-future-event 90 '#s(gc-info minor 236265120 360290056 0 227872560 360290056 2607 2607 1677475790306.498 1677475790306.512))
(indexed-future-event 91 '#s(gc-info minor 236267168 360290056 0 227874528 360290056 2610 2610 1677475790309.261 1677475790309.277))
(indexed-future-event 92 '#s(gc-info minor 236269024 360290056 0 227872528 360290056 2612 2612 1677475790312.022 1677475790312.041))
(indexed-future-event 93 '#s(gc-info minor 236267104 360290056 0 227874464 360290056 2615 2615 1677475790314.788 1677475790314.803))
(indexed-future-event 94 '#s(gc-info minor 236273840 360290056 0 227876512 360290056 2618 2618 1677475790317.578 1677475790317.593))
(indexed-future-event 95 '#s(gc-info minor 236271024 360290056 0 227878464 360290056 2620 2620 1677475790320.341 1677475790320.355))
(indexed-future-event 96 '#s(gc-info minor 236273072 360290056 0 227876480 360290056 2623 2623 1677475790323.102 1677475790323.12))
(indexed-future-event 97 '#s(gc-info minor 236270976 360290056 0 227878400 360290056 2626 2626 1677475790325.865 1677475790325.879))
(indexed-future-event 98 '#s(gc-info minor 236277760 360290056 0 227880432 360290056 2629 2629 1677475790328.657 1677475790328.672))
(indexed-future-event 99 '#s(gc-info minor 236274928 360290056 0 227882416 360290056 2631 2631 1677475790331.417 1677475790331.431))
(indexed-future-event 100 '#s(gc-info minor 236276960 360290056 0 227880432 360290056 2634 2634 1677475790334.176 1677475790334.197))
(indexed-future-event 101 '#s(gc-info minor 236274944 360290056 0 227882320 360290056 2637 2637 1677475790336.948 1677475790336.963))
(indexed-future-event 102 '#s(gc-info minor 236281728 360290056 0 227884384 360290056 2639 2639 1677475790339.737 1677475790339.752))
(indexed-future-event 103 '#s(gc-info minor 236278880 360290056 0 227886336 360290056 2642 2642 1677475790342.504 1677475790342.518))
(indexed-future-event 104 '#s(gc-info minor 236280896 360290056 0 227884320 360290056 2645 2645 1677475790345.267 1677475790345.285))
(indexed-future-event 105 '#s(gc-info minor 236278816 360290056 0 227886272 360290056 2648 2648 1677475790348.03 1677475790348.045))
(indexed-future-event 106 '#s(gc-info minor 236280816 360290056 0 227888224 360290056 2650 2650 1677475790350.791 1677475790350.806))
(indexed-future-event 107 '#s(gc-info minor 236287504 360290056 0 227890288 360290056 2653 2653 1677475790353.587 1677475790353.602))
(indexed-future-event 108 '#s(gc-info minor 236284864 360290056 0 227888304 360290056 2656 2656 1677475790356.349 1677475790356.367))
(indexed-future-event 109 '#s(gc-info minor 236284640 360290056 0 227955328 360290056 2658 2658 1677475790359.104 1677475790359.12))
(indexed-future-event 110 '#s(gc-info minor 236349888 360290056 0 227957264 360290056 2661 2661 1677475790361.885 1677475790361.9))
(indexed-future-event 111 '#s(gc-info minor 236356544 360290056 0 227959312 360290056 2664 2664 1677475790364.679 1677475790364.693))
(indexed-future-event 112 '#s(gc-info minor 236353920 360290056 0 227954928 360290056 2667 2667 1677475790367.443 1677475790367.46))
(indexed-future-event 113 '#s(gc-info minor 236349424 360290056 0 227956848 360290056 2669 2669 1677475790370.226 1677475790370.241))
(indexed-future-event 114 '#s(gc-info minor 236351424 360290056 0 227958816 360290056 2672 2672 1677475790372.986 1677475790373.0))
(indexed-future-event 115 '#s(gc-info minor 236358096 360290056 0 227960896 360290056 2675 2675 1677475790375.775 1677475790375.791))
(indexed-future-event 116 '#s(gc-info minor 236355440 360290056 0 227955808 360290056 2677 2677 1677475790378.543 1677475790378.562))
(indexed-future-event 117 '#s(gc-info minor 236350320 360290056 0 227957696 360290056 2680 2680 1677475790381.312 1677475790381.326))
(indexed-future-event 118 '#s(gc-info minor 236352304 360290056 0 227959664 360290056 2683 2683 1677475790384.088 1677475790384.102))
(indexed-future-event 119 '#s(gc-info minor 236354160 360290056 0 227961616 360290056 2686 2686 1677475790386.851 1677475790386.865))
(indexed-future-event 120 '#s(gc-info minor 236360976 360290056 0 227959696 360290056 2688 2688 1677475790389.646 1677475790389.662))
(indexed-future-event 121 '#s(gc-info minor 236354192 360290056 0 227961648 360290056 2691 2691 1677475790392.416 1677475790392.43))
(indexed-future-event 122 '#s(gc-info minor 236356192 360290056 0 227963600 360290056 2694 2694 1677475790395.186 1677475790395.2))
(indexed-future-event 123 '#s(gc-info minor 236358112 360290056 0 227965552 360290056 2697 2697 1677475790397.982 1677475790397.997))
(indexed-future-event 124 '#s(gc-info minor 236364960 360290056 0 227963664 360290056 2699 2699 1677475790400.893 1677475790400.91))
(indexed-future-event 125 '#s(gc-info minor 236358160 360290056 0 227965584 360290056 2702 2702 1677475790403.668 1677475790403.682))
(indexed-future-event 126 '#s(gc-info minor 236360144 360290056 0 227967520 360290056 2705 2705 1677475790406.432 1677475790406.446))
(indexed-future-event 127 '#s(gc-info minor 236362016 360290056 0 227969504 360290056 2708 2708 1677475790409.194 1677475790409.208))
(indexed-future-event 128 '#s(gc-info minor 236368896 360290056 0 227967600 360290056 2710 2710 1677475790411.985 1677475790412.001))
(indexed-future-event 129 '#s(gc-info minor 236362096 360290056 0 227969520 360290056 2713 2713 1677475790414.755 1677475790414.77))
(indexed-future-event 130 '#s(gc-info minor 236364096 360290056 0 227971488 360290056 2716 2716 1677475790417.53 1677475790417.544))
(indexed-future-event 131 '#s(gc-info minor 236365984 360290056 0 227973440 360290056 2718 2718 1677475790420.301 1677475790420.315))
(indexed-future-event 132 '#s(gc-info minor 236368000 360290056 0 227959136 360290056 2721 2721 1677475790423.065 1677475790423.094))
(indexed-future-event 133 '#s(gc-info minor 236358416 360290056 0 227961088 360290056 2724 2724 1677475790425.879 1677475790425.895))
(indexed-future-event 134 '#s(gc-info minor 236355696 360290056 0 227963056 360290056 2727 2727 1677475790428.648 1677475790428.662))
(indexed-future-event 135 '#s(gc-info minor 236357552 360290056 0 227965008 360290056 2729 2729 1677475790431.413 1677475790431.428))
(indexed-future-event 136 '#s(gc-info minor 236359584 360290056 0 227963040 360290056 2732 2732 1677475790434.194 1677475790434.21))
(indexed-future-event 137 '#s(gc-info minor 236362320 360290056 0 227965088 360290056 2735 2735 1677475790436.987 1677475790437.002))
(indexed-future-event 138 '#s(gc-info minor 236359632 360290056 0 227967040 360290056 2737 2737 1677475790439.752 1677475790439.767))
(indexed-future-event 139 '#s(gc-info minor 236361552 360290056 0 227968992 360290056 2740 2740 1677475790442.523 1677475790442.537))
(indexed-future-event 140 '#s(gc-info minor 236363600 360290056 0 227966976 360290056 2743 2743 1677475790445.294 1677475790445.31))
(indexed-future-event 141 '#s(gc-info minor 236361472 360290056 0 227968896 360290056 2746 2746 1677475790448.06 1677475790448.075))
(indexed-future-event 142 '#s(gc-info minor 236368240 360290056 0 227970976 360290056 2748 2748 1677475790450.85 1677475790450.865))
(indexed-future-event 143 '#s(gc-info minor 236365520 360290056 0 227972928 360290056 2751 2751 1677475790453.632 1677475790453.647))
(indexed-future-event 144 '#s(gc-info minor 236367440 360290056 0 227970912 360290056 2754 2754 1677475790456.396 1677475790456.412))
(indexed-future-event 145 '#s(gc-info minor 236365520 360290056 0 227972848 360290056 2756 2757 1677475790459.166 1677475790459.181))
(indexed-future-event 146 '#s(gc-info minor 236372288 360290056 0 227974928 360290056 2759 2759 1677475790461.962 1677475790461.977))
(indexed-future-event 147 '#s(gc-info minor 236369488 360290056 0 227976864 360290056 2762 2762 1677475790464.737 1677475790464.751))
(indexed-future-event 148 '#s(gc-info minor 236371360 360290056 0 227974944 360290056 2765 2765 1677475790467.499 1677475790467.517))
(indexed-future-event 149 '#s(gc-info minor 236369488 360290056 0 227976832 360290056 2767 2767 1677475790470.268 1677475790470.282))
(indexed-future-event 150 '#s(gc-info minor 236376112 360290056 0 227978880 360290056 2770 2770 1677475790473.06 1677475790473.074))
(indexed-future-event 151 '#s(gc-info minor 236373456 360290056 0 227980848 360290056 2773 2773 1677475790475.827 1677475790475.842))
(indexed-future-event 152 '#s(gc-info minor 236375344 360290056 0 227978816 360290056 2776 2776 1677475790478.595 1677475790478.61))
(indexed-future-event 153 '#s(gc-info minor 236373376 360290056 0 227980720 360290056 2778 2778 1677475790481.36 1677475790481.375))
(indexed-future-event 154 '#s(gc-info minor 236375216 360290056 0 227982704 360290056 2781 2781 1677475790484.135 1677475790484.149))
(indexed-future-event 155 '#s(gc-info minor 236382096 360290056 0 227984768 360290056 2784 2784 1677475790486.934 1677475790486.949))
(indexed-future-event 156 '#s(gc-info minor 236379264 360290056 0 227982752 360290056 2786 2786 1677475790489.702 1677475790489.718))
(indexed-future-event 157 '#s(gc-info minor 236377328 360290056 0 227984688 360290056 2789 2789 1677475790492.47 1677475790492.492))
(indexed-future-event 158 '#s(future-event 1 0 block 1677475790492.703 continuation-mark-set-first #f))
(indexed-future-event 159 '#s(future-event 1 0 result 1677475790492.716 #f #f))
(indexed-future-event 160 '#s(future-event 1 0 start-work 1677475790492.717 #f #f))
(indexed-future-event 161 '#s(gc-info minor 236381488 360290056 0 227989008 360290056 2792 2792 1677475790495.328 1677475790495.349))
(indexed-future-event 162 '#s(gc-info minor 236388288 360290056 0 227991024 360290056 2795 2795 1677475790498.127 1677475790498.142))
(indexed-future-event 163 '#s(gc-info minor 236385600 360290056 0 227989072 360290056 2797 2797 1677475790500.926 1677475790500.947))
(indexed-future-event 164 '#s(gc-info minor 236383568 360290056 0 227990960 360290056 2800 2800 1677475790503.699 1677475790503.714))
(indexed-future-event 165 '#s(gc-info minor 236385520 360290056 0 227992896 360290056 2803 2803 1677475790506.464 1677475790506.479))
(indexed-future-event 166 '#s(gc-info minor 236392176 360290056 0 227994944 360290056 2806 2806 1677475790509.269 1677475790509.284))
(indexed-future-event 167 '#s(gc-info minor 236389552 360290056 0 227993040 360290056 2808 2808 1677475790512.038 1677475790512.059))
(indexed-future-event 168 '#s(gc-info minor 236387536 360290056 0 227994896 360290056 2811 2811 1677475790514.807 1677475790514.822))
(indexed-future-event 169 '#s(gc-info minor 236389472 360290056 0 227996864 360290056 2814 2814 1677475790517.578 1677475790517.592))
(indexed-future-event 170 '#s(gc-info minor 236391360 360290056 0 227998816 360290056 2816 2816 1677475790520.344 1677475790520.358))
(indexed-future-event 171 '#s(gc-info minor 236398144 360290056 0 227996896 360290056 2819 2819 1677475790523.137 1677475790523.153))
(indexed-future-event 172 '#s(gc-info minor 236391408 360290056 0 227998816 360290056 2822 2822 1677475790525.903 1677475790525.919))
(indexed-future-event 173 '#s(gc-info minor 236393424 360290056 0 228000784 360290056 2825 2825 1677475790528.841 1677475790528.889))
(indexed-future-event 174 '#s(gc-info minor 236395280 360290056 0 228002736 360290056 2828 2828 1677475790531.715 1677475790531.739))
(indexed-future-event 175 '#s(gc-info minor 236402096 360290056 0 228000816 360290056 2830 2830 1677475790534.529 1677475790534.557))
(indexed-future-event 176 '#s(gc-info minor 236395312 360290056 0 228002768 360290056 2833 2833 1677475790537.306 1677475790537.322))
(indexed-future-event 177 '#s(gc-info minor 236397312 360290056 0 228004720 360290056 2836 2836 1677475790540.075 1677475790540.09))
(indexed-future-event 178 '#s(gc-info minor 236399232 360290056 0 228006672 360290056 2838 2839 1677475790542.848 1677475790542.863))
(indexed-future-event 179 '#s(gc-info minor 236406080 360290056 0 228004784 360290056 2841 2841 1677475790545.64 1677475790545.66))
(indexed-future-event 180 '#s(gc-info minor 236399280 360290056 0 228006704 360290056 2844 2844 1677475790548.425 1677475790548.442))
(indexed-future-event 181 '#s(gc-info minor 236401264 360290056 0 228008640 360290056 2847 2847 1677475790551.191 1677475790551.206))
(indexed-future-event 182 '#s(gc-info minor 236403136 360290056 0 228010624 360290056 2849 2849 1677475790553.955 1677475790553.971))
(indexed-future-event 183 '#s(gc-info minor 236405168 360290056 0 228008656 360290056 2852 2852 1677475790556.709 1677475790556.732))
(indexed-future-event 184 '#s(gc-info minor 236407936 360290056 0 228010640 360290056 2855 2855 1677475790559.511 1677475790559.525))
(indexed-future-event 185 '#s(gc-info minor 236405216 360290056 0 228012608 360290056 2858 2858 1677475790562.271 1677475790562.286))
(indexed-future-event 186 '#s(gc-info minor 236407104 360290056 0 228014560 360290056 2860 2860 1677475790565.053 1677475790565.069))
(indexed-future-event 187 '#s(gc-info minor 236409120 360290056 0 228012544 360290056 2863 2863 1677475790567.818 1677475790567.835))
(indexed-future-event 188 '#s(gc-info minor 236411824 360290056 0 228014560 360290056 2866 2866 1677475790570.6 1677475790570.614))
(indexed-future-event 189 '#s(gc-info minor 236409168 360290056 0 228016528 360290056 2868 2868 1677475790573.363 1677475790573.378))
(indexed-future-event 190 '#s(gc-info minor 236411024 360290056 0 228018480 360290056 2871 2871 1677475790576.117 1677475790576.132))
(indexed-future-event 191 '#s(gc-info minor 236413056 360290056 0 228016512 360290056 2874 2874 1677475790578.871 1677475790578.887))
(indexed-future-event 192 '#s(gc-info minor 236415792 360290056 0 228018560 360290056 2877 2877 1677475790581.653 1677475790581.668))
(indexed-future-event 193 '#s(gc-info minor 236413104 360290056 0 228020512 360290056 2879 2879 1677475790584.411 1677475790584.426))
(indexed-future-event 194 '#s(gc-info minor 236415024 360290056 0 228022464 360290056 2882 2882 1677475790587.163 1677475790587.178))
(indexed-future-event 195 '#s(gc-info minor 236417072 360290056 0 228020448 360290056 2885 2885 1677475790589.927 1677475790589.943))
(indexed-future-event 196 '#s(gc-info minor 236414944 360290056 0 228022368 360290056 2887 2887 1677475790592.709 1677475790592.724))
(indexed-future-event 197 '#s(gc-info minor 236421728 360290056 0 228024400 360290056 2890 2890 1677475790595.498 1677475790595.513))
(indexed-future-event 198 '#s(gc-info minor 236418896 360290056 0 228026384 360290056 2893 2893 1677475790598.259 1677475790598.273))
(indexed-future-event 199 '#s(gc-info minor 236420928 360290056 0 228024448 360290056 2895 2896 1677475790601.018 1677475790601.052))
(indexed-future-event 200 '#s(gc-info minor 236418960 360290056 0 228026272 360290056 2898 2898 1677475790603.797 1677475790603.812))
(indexed-future-event 201 '#s(gc-info minor 236427520 360290056 0 228093264 360290056 2901 2901 1677475790606.572 1677475790606.589))
(indexed-future-event 202 '#s(gc-info minor 236487760 360290056 0 228095216 360290056 2904 2904 1677475790609.335 1677475790609.349))
(indexed-future-event 203 '#s(gc-info minor 236489776 360290056 0 228091952 360290056 2906 2906 1677475790612.091 1677475790612.109))
(indexed-future-event 204 '#s(gc-info minor 236486448 360290056 0 228093904 360290056 2909 2909 1677475790614.846 1677475790614.86))
(indexed-future-event 205 '#s(gc-info minor 236488448 360290056 0 228095856 360290056 2912 2912 1677475790617.606 1677475790617.62))
(indexed-future-event 206 '#s(gc-info minor 236495136 360290056 0 228097904 360290056 2914 2914 1677475790620.385 1677475790620.399))
(indexed-future-event 207 '#s(gc-info minor 236492480 360290056 0 228095920 360290056 2917 2917 1677475790623.141 1677475790623.157))
(indexed-future-event 208 '#s(gc-info minor 236490416 360290056 0 228097840 360290056 2920 2920 1677475790625.894 1677475790625.908))
(indexed-future-event 209 '#s(gc-info minor 236492400 360290056 0 228099776 360290056 2923 2923 1677475790628.648 1677475790628.662))
(indexed-future-event 210 '#s(gc-info minor 236499056 360290056 0 228101824 360290056 2925 2925 1677475790631.443 1677475790631.458))
(indexed-future-event 211 '#s(gc-info minor 236496432 360290056 0 228099856 360290056 2928 2928 1677475790634.194 1677475790634.211))
(indexed-future-event 212 '#s(gc-info minor 236494352 360290056 0 228101776 360290056 2931 2931 1677475790636.956 1677475790636.971))
(indexed-future-event 213 '#s(gc-info minor 236496352 360290056 0 228103744 360290056 2933 2933 1677475790639.71 1677475790639.724))
(indexed-future-event 214 '#s(gc-info minor 236503024 360290056 0 228105824 360290056 2936 2936 1677475790642.495 1677475790642.509))
(indexed-future-event 215 '#s(gc-info minor 236500368 360290056 0 228096032 360290056 2939 2939 1677475790645.259 1677475790645.278))
(indexed-future-event 216 '#s(gc-info minor 236490544 360290056 0 228097920 360290056 2942 2942 1677475790648.02 1677475790648.034))
(indexed-future-event 217 '#s(gc-info minor 236492528 360290056 0 228099888 360290056 2944 2944 1677475790650.776 1677475790650.79))
(indexed-future-event 218 '#s(gc-info minor 236494384 360290056 0 228101840 360290056 2947 2947 1677475790653.533 1677475790653.547))
(indexed-future-event 219 '#s(gc-info minor 236501200 360290056 0 228099936 360290056 2950 2950 1677475790656.315 1677475790656.331))
(indexed-future-event 220 '#s(gc-info minor 236494432 360290056 0 228101888 360290056 2952 2952 1677475790659.066 1677475790659.081))
(indexed-future-event 221 '#s(gc-info minor 236496432 360290056 0 228103840 360290056 2955 2955 1677475790661.823 1677475790661.837))
(indexed-future-event 222 '#s(gc-info minor 236498352 360290056 0 228105792 360290056 2958 2958 1677475790664.578 1677475790664.592))
(indexed-future-event 223 '#s(gc-info minor 236505200 360290056 0 228103904 360290056 2961 2961 1677475790667.365 1677475790667.381))
(indexed-future-event 224 '#s(gc-info minor 236498400 360290056 0 228105824 360290056 2963 2963 1677475790670.129 1677475790670.143))
(indexed-future-event 225 '#s(gc-info minor 236500384 360290056 0 228107760 360290056 2966 2966 1677475790672.888 1677475790672.902))
(indexed-future-event 226 '#s(gc-info minor 236502256 360290056 0 228109744 360290056 2969 2969 1677475790675.645 1677475790675.659))
(indexed-future-event 227 '#s(gc-info minor 236509136 360290056 0 228107840 360290056 2971 2971 1677475790678.46 1677475790678.489))
(indexed-future-event 228 '#s(gc-info minor 236502336 360290056 0 228109760 360290056 2974 2974 1677475790681.24 1677475790681.255))
(indexed-future-event 229 '#s(gc-info minor 236504336 360290056 0 228111728 360290056 2977 2977 1677475790683.992 1677475790684.006))
(indexed-future-event 230 '#s(gc-info minor 236506224 360290056 0 228113680 360290056 2980 2980 1677475790686.743 1677475790686.757))
(indexed-future-event 231 '#s(gc-info minor 236508240 360290056 0 228111680 360290056 2982 2982 1677475790689.498 1677475790689.516))
(indexed-future-event 232 '#s(gc-info minor 236510960 360290056 0 228113664 360290056 2985 2985 1677475790692.285 1677475790692.298))
(indexed-future-event 233 '#s(gc-info minor 236508272 360290056 0 228115632 360290056 2988 2988 1677475790695.069 1677475790695.083))
(indexed-future-event 234 '#s(gc-info minor 236510128 360290056 0 228117584 360290056 2990 2990 1677475790697.828 1677475790697.842))
(indexed-future-event 235 '#s(gc-info minor 236512160 360290056 0 228115616 360290056 2993 2993 1677475790700.58 1677475790700.595))
(indexed-future-event 236 '#s(gc-info minor 236514896 360290056 0 228117664 360290056 2996 2996 1677475790703.372 1677475790703.387))
(indexed-future-event 237 '#s(gc-info minor 236512208 360290056 0 228119616 360290056 2999 2999 1677475790706.127 1677475790706.141))
(indexed-future-event 238 '#s(gc-info minor 236514128 360290056 0 228121568 360290056 3001 3001 1677475790708.889 1677475790708.903))
(indexed-future-event 239 '#s(gc-info minor 236516176 360290056 0 228119552 360290056 3004 3004 1677475790711.644 1677475790711.66))
(indexed-future-event 240 '#s(gc-info minor 236518848 360290056 0 228121568 360290056 3007 3007 1677475790714.426 1677475790714.441))
(indexed-future-event 241 '#s(gc-info minor 236516128 360290056 0 228123504 360290056 3009 3009 1677475790717.176 1677475790717.19))
(indexed-future-event 242 '#s(gc-info minor 236518000 360290056 0 228125488 360290056 3012 3012 1677475790719.941 1677475790719.955))
(indexed-future-event 243 '#s(gc-info minor 236520032 360290056 0 228123472 360290056 3015 3015 1677475790722.69 1677475790722.706))
(indexed-future-event 244 '#s(gc-info minor 236517984 360290056 0 228125392 360290056 3017 3017 1677475790725.443 1677475790725.457))
(indexed-future-event 245 '#s(gc-info minor 236524800 360290056 0 228127456 360290056 3020 3020 1677475790728.229 1677475790728.243))
(indexed-future-event 246 '#s(gc-info minor 236521952 360290056 0 228129408 360290056 3023 3023 1677475790730.99 1677475790731.004))
(indexed-future-event 247 '#s(gc-info minor 236523968 360290056 0 228127424 360290056 3026 3026 1677475790733.745 1677475790733.763))
(indexed-future-event 248 '#s(gc-info minor 236521920 360290056 0 228129344 360290056 3028 3028 1677475790736.508 1677475790736.522))
(indexed-future-event 249 '#s(gc-info minor 236528736 360290056 0 228131408 360290056 3031 3031 1677475790739.291 1677475790739.306))
(indexed-future-event 250 '#s(gc-info minor 236525904 360290056 0 228133360 360290056 3034 3034 1677475790742.051 1677475790742.066))
(indexed-future-event 251 '#s(gc-info minor 236527936 360290056 0 228131360 360290056 3036 3036 1677475790744.814 1677475790744.833))
(indexed-future-event 252 '#s(gc-info minor 236525856 360290056 0 228133280 360290056 3039 3039 1677475790747.572 1677475790747.586))
(indexed-future-event 253 '#s(gc-info minor 236532608 360290056 0 228135328 360290056 3042 3042 1677475790750.363 1677475790750.379))
(indexed-future-event 254 '#s(gc-info minor 236529840 360290056 0 228137280 360290056 3045 3045 1677475790753.129 1677475790753.144))
(indexed-future-event 255 '#s(gc-info minor 236531888 360290056 0 228135312 360290056 3047 3047 1677475790755.882 1677475790755.899))
(indexed-future-event 256 '#s(gc-info minor 236529808 360290056 0 228137232 360290056 3050 3050 1677475790758.639 1677475790758.653))
(indexed-future-event 257 '#s(gc-info minor 236531808 360290056 0 228139200 360290056 3053 3053 1677475790761.45 1677475790761.464))
(indexed-future-event 258 '#s(gc-info minor 236538480 360290056 0 228141280 360290056 3055 3055 1677475790764.233 1677475790764.248))
(indexed-future-event 259 '#s(gc-info minor 236535824 360290056 0 228139232 360290056 3058 3058 1677475790767.0 1677475790767.018))
(indexed-future-event 260 '#s(gc-info minor 236533744 360290056 0 228141152 360290056 3061 3061 1677475790769.771 1677475790769.786))
(indexed-future-event 261 '#s(gc-info minor 236535760 360290056 0 228143120 360290056 3064 3064 1677475790772.533 1677475790772.548))
(indexed-future-event 262 '#s(gc-info minor 236542416 360290056 0 228145168 360290056 3066 3066 1677475790775.316 1677475790775.334))
(indexed-future-event 263 '#s(gc-info minor 236539728 360290056 0 228111936 360290056 3069 3069 1677475790778.09 1677475790778.12))
(indexed-future-event 264 '#s(gc-info minor 236506432 360290056 0 228113824 360290056 3072 3072 1677475790780.882 1677475790780.897))
(indexed-future-event 265 '#s(gc-info minor 236508368 360290056 0 228115776 360290056 3074 3074 1677475790783.644 1677475790783.659))
(indexed-future-event 266 '#s(gc-info minor 236515056 360290056 0 228117824 360290056 3077 3077 1677475790786.44 1677475790786.455))
(indexed-future-event 267 '#s(gc-info minor 236512400 360290056 0 228115840 360290056 3080 3080 1677475790789.204 1677475790789.224))
(indexed-future-event 268 '#s(gc-info minor 236510336 360290056 0 228117760 360290056 3083 3083 1677475790791.965 1677475790791.98))
(indexed-future-event 269 '#s(gc-info minor 236512320 360290056 0 228119696 360290056 3085 3085 1677475790794.735 1677475790794.75))
(indexed-future-event 270 '#s(gc-info minor 236514192 360290056 0 228121680 360290056 3088 3088 1677475790797.493 1677475790797.507))
(indexed-future-event 271 '#s(gc-info minor 236521072 360290056 0 228119776 360290056 3091 3091 1677475790800.306 1677475790800.323))
(indexed-future-event 272 '#s(gc-info minor 236514272 360290056 0 228121696 360290056 3094 3094 1677475790803.07 1677475790803.085))
(indexed-future-event 273 '#s(gc-info minor 236516272 360290056 0 228123664 360290056 3096 3096 1677475790805.833 1677475790805.848))
(indexed-future-event 274 '#s(gc-info minor 236518160 360290056 0 228125616 360290056 3099 3099 1677475790808.596 1677475790808.613))
(indexed-future-event 275 '#s(gc-info minor 236524944 360290056 0 228123696 360290056 3102 3102 1677475790811.398 1677475790811.414))
(indexed-future-event 276 '#s(gc-info minor 236518208 360290056 0 228125616 360290056 3104 3104 1677475790814.162 1677475790814.177))
(indexed-future-event 277 '#s(gc-info minor 236520224 360290056 0 228127584 360290056 3107 3107 1677475790816.926 1677475790816.941))
(indexed-future-event 278 '#s(gc-info minor 236522080 360290056 0 228129536 360290056 3110 3110 1677475790819.693 1677475790819.708))
(indexed-future-event 279 '#s(gc-info minor 236528896 360290056 0 228127648 360290056 3113 3113 1677475790822.479 1677475790822.497))
(indexed-future-event 280 '#s(gc-info minor 236522144 360290056 0 228129568 360290056 3115 3115 1677475790825.264 1677475790825.285))
(indexed-future-event 281 '#s(gc-info minor 236524112 360290056 0 228131520 360290056 3118 3118 1677475790828.043 1677475790828.059))
(indexed-future-event 282 '#s(gc-info minor 236526032 360290056 0 228133472 360290056 3121 3121 1677475790830.802 1677475790830.817))
(indexed-future-event 283 '#s(gc-info minor 236528080 360290056 0 228131488 360290056 3123 3123 1677475790833.56 1677475790833.577))
(indexed-future-event 284 '#s(gc-info minor 236530784 360290056 0 228133504 360290056 3126 3126 1677475790836.363 1677475790836.377))
(indexed-future-event 285 '#s(gc-info minor 236528064 360290056 0 228135440 360290056 3129 3129 1677475790839.124 1677475790839.139))
(indexed-future-event 286 '#s(gc-info minor 236529936 360290056 0 228137424 360290056 3132 3132 1677475790841.896 1677475790841.913))
(indexed-future-event 287 '#s(gc-info minor 236531968 360290056 0 228135408 360290056 3134 3134 1677475790844.667 1677475790844.684))
(indexed-future-event 288 '#s(gc-info minor 236534688 360290056 0 228137424 360290056 3137 3137 1677475790847.458 1677475790847.473))
(indexed-future-event 289 '#s(gc-info minor 236532000 360290056 0 228139392 360290056 3140 3140 1677475790850.227 1677475790850.245))
(indexed-future-event 290 '#s(gc-info minor 236533888 360290056 0 228141344 360290056 3142 3142 1677475790853.0 1677475790853.015))
(indexed-future-event 291 '#s(gc-info minor 236535904 360290056 0 228139344 360290056 3145 3145 1677475790855.765 1677475790855.785))
(indexed-future-event 292 '#s(gc-info minor 236533840 360290056 0 228141296 360290056 3148 3148 1677475790858.533 1677475790858.547))
(indexed-future-event 293 '#s(gc-info minor 236540608 360290056 0 228143344 360290056 3151 3151 1677475790861.331 1677475790861.346))
(indexed-future-event 294 '#s(gc-info minor 236537920 360290056 0 228145312 360290056 3153 3153 1677475790864.108 1677475790864.124))
(indexed-future-event 295 '#s(gc-info minor 236539808 360290056 0 228143328 360290056 3156 3156 1677475790866.87 1677475790866.89))
(indexed-future-event 296 '#s(gc-info minor 236537888 360290056 0 228145200 360290056 3159 3159 1677475790869.64 1677475790869.655))
(indexed-future-event 297 '#s(gc-info minor 236544480 360290056 0 228147248 360290056 3161 3161 1677475790872.435 1677475790872.449))
(indexed-future-event 298 '#s(gc-info minor 236541856 360290056 0 228149216 360290056 3164 3164 1677475790875.202 1677475790875.224))
(indexed-future-event 299 '#s(gc-info minor 236545552 360290056 0 228212544 360290056 3167 3167 1677475790877.989 1677475790878.008))
(indexed-future-event 300 '#s(gc-info minor 236607120 360290056 0 228214480 360290056 3170 3170 1677475790880.746 1677475790880.763))
(indexed-future-event 301 '#s(gc-info minor 236613760 360290056 0 228216560 360290056 3172 3172 1677475790883.53 1677475790883.545))
(indexed-future-event 302 '#s(gc-info minor 236611104 360290056 0 228218512 360290056 3175 3175 1677475790886.3 1677475790886.315))
(indexed-future-event 303 '#s(gc-info minor 236613024 360290056 0 228213888 360290056 3178 3178 1677475790889.081 1677475790889.099))
(indexed-future-event 304 '#s(gc-info minor 236608496 360290056 0 228215824 360290056 3180 3180 1677475790891.861 1677475790891.876))
(indexed-future-event 305 '#s(gc-info minor 236610320 360290056 0 228217776 360290056 3183 3183 1677475790894.64 1677475790894.654))
(indexed-future-event 306 '#s(gc-info minor 236617136 360290056 0 228219808 360290056 3186 3186 1677475790897.428 1677475790897.443))
(indexed-future-event 307 '#s(gc-info minor 236614304 360290056 0 228217856 360290056 3189 3189 1677475790900.19 1677475790900.207))
(indexed-future-event 308 '#s(gc-info minor 236612400 360290056 0 228219776 360290056 3191 3191 1677475790902.973 1677475790902.988))
(indexed-future-event 309 '#s(gc-info minor 236614288 360290056 0 228221728 360290056 3194 3194 1677475790905.734 1677475790905.749))
(indexed-future-event 310 '#s(gc-info minor 236621136 360290056 0 228223792 360290056 3197 3197 1677475790908.534 1677475790908.554))
(indexed-future-event 311 '#s(gc-info minor 236618288 360290056 0 228216544 360290056 3200 3200 1677475790911.319 1677475790911.339))
(indexed-future-event 312 '#s(gc-info minor 236611104 360290056 0 228218416 360290056 3202 3202 1677475790914.075 1677475790914.09))
(indexed-future-event 313 '#s(gc-info minor 236612912 360290056 0 228220400 360290056 3205 3205 1677475790916.891 1677475790916.906))
(indexed-future-event 314 '#s(future-event 1 0 complete 1677475790917.071 #f #f))
(indexed-future-event 315 '#s(future-event 1 0 end-work 1677475790917.084 #f #f))
          )))
                   
@interaction-eval-show[
    #:eval future-eval
           (show-timeline bad-log)
]

Each horizontal row represents a parallel task, and the colored 
dots represent important events in the execution of the program; they are 
color-coded to distinguish one event type from another.  The upper-left blue 
dot in the timeline represents the future's creation.  The future 
executes for a brief period (represented by a green bar in the second line) on thread 
1. It then pauses, because the Racket thread will need to perform a
future-unsafe operation (as represented by a red dot). That pause is long,
because the Racket thread is performing its own copy of the calculation
before it @racket[touch]es the future. Meanwhile, the
pink vertical lines represent garbage-collection events, which imply a
synchronization across parallel tasks.

A future-unsafe operation or an operation that needs continuation context is
a @deftech{blocking} operation.
A blocking operation halts evaluation of a future and
will not allow it to continue until the future is touched. A @racket[touch] of
the future in a thread causes its work to be evaluated sequentially by the
thread, which provides a continuation context and can synchronize with
Racket threads.@margin-note*{In the @tech{BC} implementation of Racket, a
@deftech{synchronized} operation also halts the future. A background
thread may perform the operation at any time and, once completed, the
future may continue running in parallel. The @tech{CS} implementation
can perform synchronized operations without stopping a future or relying
on a background thread.}

When you move your mouse over an event, the visualizer shows you 
detailed information about the event and draws arrows
connecting all of the events in the corresponding future.
This image shows those connections for our future.

@interaction-eval-show[
     #:eval future-eval
            (show-timeline bad-log 
                           #:selected-event-index 1)
]

A dotted blue line connects the first event in the future to
the future that created it, a red line connects a future blocking
event to its resumptions, and the purple lines connect adjacent
events within the future. 

The reason that we see no parallelism is that the @racket[printf]
operation before the loop in @racket[mandelbrot] needs to look up the
@racket[current-output-port] parameter's value, which depends on the
evaluation context of the @racket[touch]. Even if that is fixed by
using @racket[fprintf] and variable that refers directly to the port,
writing to a port is also a blocking operation, because it must take a
lock on the port. Removing the @racket[printf] avoids both problems.

@interaction-eval[
    #:eval future-eval            
    (define better-log 
      (list
(indexed-future-event 0 '#s(future-event #f 0 create 1677477536732.404 #f 1))
(indexed-future-event 1 '#s(future-event 1 6 start-work 1677477536732.562 #f #f))
(indexed-future-event 2 '#s(gc-info minor 246656656 392484616 0 234224784 392484616 6833 6835 1677477536733.533 1677477536735.401))
(indexed-future-event 3 '#s(gc-info minor 242628320 392484616 0 234247696 392484616 6837 6837 1677477536736.484 1677477536736.539))
(indexed-future-event 4 '#s(gc-info minor 242639168 392484616 0 234249456 392484616 6839 6839 1677477536737.566 1677477536737.595))
(indexed-future-event 5 '#s(gc-info minor 242650576 392484616 0 234242960 392484616 6841 6842 1677477536738.611 1677477536739.218))
(indexed-future-event 6 '#s(gc-info minor 242634816 392484616 0 234230320 392484616 6844 6844 1677477536740.286 1677477536740.329))
(indexed-future-event 7 '#s(gc-info minor 242631056 392484616 0 234231984 392484616 6846 6846 1677477536741.389 1677477536741.417))
(indexed-future-event 8 '#s(gc-info minor 242639072 392484616 0 234233744 392484616 6848 6848 1677477536742.461 1677477536742.491))
(indexed-future-event 9 '#s(gc-info minor 242636608 392484616 0 234231520 392484616 6850 6850 1677477536743.527 1677477536743.559))
(indexed-future-event 10 '#s(gc-info minor 242637744 392484616 0 234233296 392484616 6852 6852 1677477536744.599 1677477536744.626))
(indexed-future-event 11 '#s(gc-info minor 242640736 392484616 0 234235056 392484616 6854 6854 1677477536745.665 1677477536745.691))
(indexed-future-event 12 '#s(gc-info minor 242651824 392484616 0 234236704 392484616 6856 6856 1677477536746.715 1677477536746.741))
(indexed-future-event 13 '#s(gc-info minor 242627760 392484616 0 234234448 392484616 6858 6858 1677477536747.776 1677477536747.806))
(indexed-future-event 14 '#s(gc-info minor 242635984 392484616 0 234236064 392484616 6860 6860 1677477536748.838 1677477536748.863))
(indexed-future-event 15 '#s(gc-info minor 242626736 392484616 0 234237824 392484616 6862 6862 1677477536749.897 1677477536749.921))
(indexed-future-event 16 '#s(gc-info minor 242639744 392484616 0 234239472 392484616 6864 6864 1677477536750.963 1677477536750.988))
(indexed-future-event 17 '#s(gc-info minor 242630144 392484616 0 234249056 392484616 6866 6867 1677477536752.018 1677477536753.099))
(indexed-future-event 18 '#s(gc-info minor 242649824 392484616 0 234234416 392484616 6870 6870 1677477536754.559 1677477536754.601))
(indexed-future-event 19 '#s(gc-info minor 242640640 392484616 0 234236176 392484616 6873 6873 1677477536756.055 1677477536756.083))
(indexed-future-event 20 '#s(gc-info minor 242652176 392484616 0 234237824 392484616 6876 6876 1677477536757.52 1677477536757.546))
(indexed-future-event 21 '#s(gc-info minor 242644496 392484616 0 234235680 392484616 6879 6879 1677477536759.02 1677477536759.065))
(indexed-future-event 22 '#s(gc-info minor 242638000 392484616 0 234237408 392484616 6882 6882 1677477536760.529 1677477536760.563))
(indexed-future-event 23 '#s(gc-info minor 242642656 392484616 0 234239184 392484616 6885 6885 1677477536762.027 1677477536762.052))
(indexed-future-event 24 '#s(gc-info minor 242628720 392484616 0 234240944 392484616 6888 6888 1677477536763.499 1677477536763.523))
(indexed-future-event 25 '#s(gc-info minor 242641376 392484616 0 234238576 392484616 6890 6891 1677477536764.965 1677477536764.996))
(indexed-future-event 26 '#s(gc-info minor 242644880 392484616 0 234240304 392484616 6893 6893 1677477536766.46 1677477536766.486))
(indexed-future-event 27 '#s(gc-info minor 242640736 392484616 0 234241984 392484616 6896 6896 1677477536767.927 1677477536767.952))
(indexed-future-event 28 '#s(gc-info minor 242647904 392484616 0 234243744 392484616 6899 6899 1677477536769.407 1677477536769.431))
(indexed-future-event 29 '#s(gc-info minor 242644176 392484616 0 234241456 392484616 6902 6902 1677477536770.881 1677477536770.908))
(indexed-future-event 30 '#s(gc-info minor 242647376 392484616 0 234243184 392484616 6905 6905 1677477536772.358 1677477536772.385))
(indexed-future-event 31 '#s(gc-info minor 242644592 392484616 0 234244912 392484616 6908 6908 1677477536773.83 1677477536773.86))
(indexed-future-event 32 '#s(gc-info minor 242650336 392484616 0 234246688 392484616 6911 6911 1677477536775.307 1677477536775.332))
(indexed-future-event 33 '#s(gc-info minor 242636240 392484616 0 234244480 392484616 6914 6914 1677477536776.781 1677477536776.811))
(indexed-future-event 34 '#s(gc-info minor 242644512 392484616 0 234246096 392484616 6916 6916 1677477536778.247 1677477536778.272))
(indexed-future-event 35 '#s(gc-info minor 242637184 392484616 0 234247888 392484616 6919 6919 1677477536779.743 1677477536779.768))
(indexed-future-event 36 '#s(gc-info minor 242647920 392484616 0 234249568 392484616 6922 6922 1677477536781.209 1677477536781.236))
(indexed-future-event 37 '#s(gc-info minor 242655504 392484616 0 234247328 392484616 6925 6925 1677477536782.693 1677477536782.721))
(indexed-future-event 38 '#s(gc-info minor 242647360 392484616 0 234248976 392484616 6928 6928 1677477536784.155 1677477536784.183))
(indexed-future-event 39 '#s(gc-info minor 242654528 392484616 0 234250768 392484616 6931 6931 1677477536785.627 1677477536785.651))
(indexed-future-event 40 '#s(gc-info minor 242656368 392484616 0 234252544 392484616 6934 6934 1677477536787.108 1677477536787.131))
(indexed-future-event 41 '#s(gc-info minor 242668912 392484616 0 234250208 392484616 6937 6937 1677477536788.582 1677477536788.61))
(indexed-future-event 42 '#s(gc-info minor 242656192 392484616 0 234251952 392484616 6939 6940 1677477536790.062 1677477536790.088))
(indexed-future-event 43 '#s(gc-info minor 242651936 392484616 0 234253648 392484616 6942 6942 1677477536791.525 1677477536791.548))
(indexed-future-event 44 '#s(gc-info minor 242643248 392484616 0 234255424 392484616 6945 6945 1677477536793.01 1677477536793.034))
(indexed-future-event 45 '#s(gc-info minor 242671408 392484616 0 234253088 392484616 6948 6948 1677477536794.477 1677477536794.506))
(indexed-future-event 46 '#s(gc-info minor 242642304 392484616 0 234254832 392484616 6951 6951 1677477536795.971 1677477536795.999))
(indexed-future-event 47 '#s(gc-info minor 242655600 392484616 0 234256736 392484616 6954 6954 1677477536797.442 1677477536797.467))
(indexed-future-event 48 '#s(gc-info minor 242647456 392484616 0 234258496 392484616 6957 6957 1677477536798.924 1677477536798.951))
(indexed-future-event 49 '#s(gc-info minor 242648464 392484616 0 234256080 392484616 6960 6960 1677477536800.407 1677477536800.438))
(indexed-future-event 50 '#s(gc-info minor 242672464 392484616 0 234257680 392484616 6963 6963 1677477536801.873 1677477536801.896))
(indexed-future-event 51 '#s(gc-info minor 242663264 392484616 0 234259456 392484616 6965 6965 1677477536803.353 1677477536803.377))
(indexed-future-event 52 '#s(gc-info minor 242659456 392484616 0 234261120 392484616 6968 6968 1677477536804.817 1677477536804.839))
(indexed-future-event 53 '#s(gc-info minor 242667088 392484616 0 234258928 392484616 6971 6971 1677477536806.292 1677477536806.32))
(indexed-future-event 54 '#s(gc-info minor 242658928 392484616 0 234260560 392484616 6974 6974 1677477536807.763 1677477536807.788))
(indexed-future-event 55 '#s(gc-info minor 242650144 392484616 0 234262336 392484616 6977 6977 1677477536809.241 1677477536809.266))
(indexed-future-event 56 '#s(gc-info minor 242667216 392484616 0 234264448 392484616 6980 6980 1677477536810.718 1677477536810.746))
(indexed-future-event 57 '#s(gc-info minor 242665888 392484616 0 234261792 392484616 6983 6983 1677477536812.182 1677477536812.211))
(indexed-future-event 58 '#s(gc-info minor 242651328 392484616 0 234263520 392484616 6985 6986 1677477536813.657 1677477536813.682))
(indexed-future-event 59 '#s(gc-info minor 242679504 392484616 0 234265184 392484616 6988 6988 1677477536815.125 1677477536815.15))
(indexed-future-event 60 '#s(gc-info minor 242671104 392484616 0 234266944 392484616 6991 6991 1677477536816.604 1677477536816.627))
(indexed-future-event 61 '#s(gc-info minor 242682928 392484616 0 234264672 392484616 6994 6994 1677477536818.08 1677477536818.107))
(indexed-future-event 62 '#s(gc-info minor 242670592 392484616 0 234266400 392484616 6997 6997 1677477536819.558 1677477536819.583))
(indexed-future-event 63 '#s(gc-info minor 242682384 392484616 0 234268064 392484616 7000 7000 1677477536821.027 1677477536821.053))
(indexed-future-event 64 '#s(gc-info minor 242657984 392484616 0 234269824 392484616 7003 7003 1677477536822.504 1677477536822.532))
(indexed-future-event 65 '#s(gc-info minor 242675760 392484616 0 234640448 392484616 7006 7016 1677477536823.985 1677477536833.862))
(indexed-future-event 66 '#s(gc-info minor 243042720 392484616 0 234249136 392484616 7018 7019 1677477536835.356 1677477536835.671))
(indexed-future-event 67 '#s(gc-info minor 242655456 392484616 0 234250896 392484616 7022 7022 1677477536837.121 1677477536837.152))
(indexed-future-event 68 '#s(gc-info minor 242667248 392484616 0 234252560 392484616 7024 7025 1677477536838.59 1677477536838.618))
(indexed-future-event 69 '#s(gc-info minor 242642112 392484616 0 234250384 392484616 7027 7027 1677477536840.07 1677477536840.098))
(indexed-future-event 70 '#s(gc-info minor 242650352 392484616 0 234252016 392484616 7030 7030 1677477536841.524 1677477536841.551))
(indexed-future-event 71 '#s(gc-info minor 242641184 392484616 0 234253776 392484616 7033 7033 1677477536843.002 1677477536843.033))
(indexed-future-event 72 '#s(gc-info minor 242653744 392484616 0 234255440 392484616 7036 7036 1677477536844.477 1677477536844.504))
(indexed-future-event 73 '#s(gc-info minor 242662912 392484616 0 234253264 392484616 7039 7039 1677477536845.964 1677477536845.992))
(indexed-future-event 74 '#s(gc-info minor 242659232 392484616 0 234255008 392484616 7042 7042 1677477536847.438 1677477536847.465))
(indexed-future-event 75 '#s(gc-info minor 242655392 392484616 0 234256672 392484616 7045 7045 1677477536848.892 1677477536848.919))
(indexed-future-event 76 '#s(gc-info minor 242662256 392484616 0 234258448 392484616 7047 7047 1677477536850.364 1677477536850.389))
(indexed-future-event 77 '#s(gc-info minor 242658448 392484616 0 234256144 392484616 7050 7050 1677477536851.823 1677477536851.848))
(indexed-future-event 78 '#s(gc-info minor 242661728 392484616 0 234257888 392484616 7053 7053 1677477536853.289 1677477536853.315))
(indexed-future-event 79 '#s(gc-info minor 242673888 392484616 0 234259552 392484616 7056 7056 1677477536854.744 1677477536854.77))
(indexed-future-event 80 '#s(gc-info minor 242649136 392484616 0 234261328 392484616 7059 7059 1677477536856.215 1677477536856.242))
(indexed-future-event 81 '#s(gc-info minor 242666592 392484616 0 234259504 392484616 7062 7062 1677477536857.688 1677477536857.722))
(indexed-future-event 82 '#s(gc-info minor 242660960 392484616 0 234261136 392484616 7065 7065 1677477536859.154 1677477536859.181))
(indexed-future-event 83 '#s(gc-info minor 242650288 392484616 0 234262896 392484616 7067 7067 1677477536860.628 1677477536860.654))
(indexed-future-event 84 '#s(gc-info minor 242662496 392484616 0 234264560 392484616 7070 7070 1677477536862.086 1677477536862.11))
(indexed-future-event 85 '#s(gc-info minor 242670480 392484616 0 234262000 392484616 7073 7073 1677477536863.561 1677477536863.586))
(indexed-future-event 86 '#s(gc-info minor 242677984 392484616 0 234263632 392484616 7076 7076 1677477536865.017 1677477536865.039))
(indexed-future-event 87 '#s(gc-info minor 242653168 392484616 0 234265392 392484616 7079 7079 1677477536866.479 1677477536866.507))
(indexed-future-event 88 '#s(gc-info minor 242665760 392484616 0 234267056 392484616 7082 7082 1677477536867.941 1677477536867.97))
(indexed-future-event 89 '#s(gc-info minor 242673360 392484616 0 234264880 392484616 7085 7085 1677477536869.415 1677477536869.447))
(indexed-future-event 90 '#s(gc-info minor 242670816 392484616 0 234266608 392484616 7087 7087 1677477536870.885 1677477536870.909))
(indexed-future-event 91 '#s(gc-info minor 242682576 392484616 0 234268272 392484616 7090 7090 1677477536872.346 1677477536872.375))
(indexed-future-event 92 '#s(gc-info minor 242674208 392484616 0 234270032 392484616 7093 7093 1677477536873.814 1677477536873.84))
(indexed-future-event 93 '#s(gc-info minor 242686768 392484616 0 234267760 392484616 7096 7096 1677477536875.28 1677477536875.307))
(indexed-future-event 94 '#s(gc-info minor 242673696 392484616 0 234269488 392484616 7099 7099 1677477536876.746 1677477536876.772))
(indexed-future-event 95 '#s(gc-info minor 242669456 392484616 0 234271152 392484616 7102 7102 1677477536878.199 1677477536878.224))
(indexed-future-event 96 '#s(gc-info minor 242676704 392484616 0 234272912 392484616 7105 7105 1677477536879.667 1677477536879.695))
(indexed-future-event 97 '#s(gc-info minor 242678864 392484616 0 234270784 392484616 7107 7107 1677477536881.148 1677477536881.178))
(indexed-future-event 98 '#s(gc-info minor 242671552 392484616 0 234272384 392484616 7110 7110 1677477536882.609 1677477536882.636))
(indexed-future-event 99 '#s(gc-info minor 242678736 392484616 0 234274160 392484616 7113 7113 1677477536884.092 1677477536884.124))
(indexed-future-event 100 '#s(gc-info minor 242674544 392484616 0 234275824 392484616 7116 7116 1677477536885.554 1677477536885.578))
(indexed-future-event 101 '#s(gc-info minor 242681792 392484616 0 234273632 392484616 7119 7119 1677477536887.02 1677477536887.045))
(indexed-future-event 102 '#s(gc-info minor 242689632 392484616 0 234275264 392484616 7122 7122 1677477536888.472 1677477536888.5))
(indexed-future-event 103 '#s(gc-info minor 242680848 392484616 0 234277040 392484616 7125 7125 1677477536889.941 1677477536889.963))
(indexed-future-event 104 '#s(gc-info minor 242677040 392484616 0 234278704 392484616 7127 7127 1677477536891.394 1677477536891.418))
(indexed-future-event 105 '#s(gc-info minor 242684624 392484616 0 234276512 392484616 7130 7130 1677477536892.861 1677477536892.886))
(indexed-future-event 106 '#s(gc-info minor 242682448 392484616 0 234278272 392484616 7133 7133 1677477536894.328 1677477536894.354))
(indexed-future-event 107 '#s(gc-info minor 242678304 392484616 0 234279952 392484616 7136 7136 1677477536895.781 1677477536895.804))
(indexed-future-event 108 '#s(gc-info minor 242685504 392484616 0 234281744 392484616 7139 7139 1677477536897.24 1677477536897.263))
(indexed-future-event 109 '#s(gc-info minor 242681776 392484616 0 234279392 392484616 7142 7142 1677477536898.694 1677477536898.722))
(indexed-future-event 110 '#s(gc-info minor 242669328 392484616 0 234281152 392484616 7145 7145 1677477536900.165 1677477536900.19))
(indexed-future-event 111 '#s(gc-info minor 242681184 392484616 0 234282832 392484616 7147 7147 1677477536901.626 1677477536901.65))
(indexed-future-event 112 '#s(gc-info minor 242678912 392484616 0 234284624 392484616 7150 7150 1677477536903.185 1677477536903.209))
(indexed-future-event 113 '#s(gc-info minor 242689264 392484616 0 234282304 392484616 7153 7153 1677477536904.698 1677477536904.728))
(indexed-future-event 114 '#s(gc-info minor 242688240 392484616 0 234284032 392484616 7156 7156 1677477536906.165 1677477536906.192))
(indexed-future-event 115 '#s(gc-info minor 242684064 392484616 0 234285712 392484616 7159 7159 1677477536907.613 1677477536907.638))
(indexed-future-event 116 '#s(gc-info minor 242676336 392484616 0 234352880 392484616 7162 7162 1677477536909.082 1677477536909.114))
(indexed-future-event 117 '#s(gc-info minor 242754448 392484616 0 234350208 392484616 7165 7165 1677477536910.554 1677477536910.581))
(indexed-future-event 118 '#s(gc-info minor 242756144 392484616 0 234351968 392484616 7168 7168 1677477536912.018 1677477536912.042))
(indexed-future-event 119 '#s(gc-info minor 242741568 392484616 0 234353744 392484616 7170 7170 1677477536913.482 1677477536913.505))
(indexed-future-event 120 '#s(gc-info minor 242753728 392484616 0 234355440 392484616 7173 7173 1677477536914.934 1677477536914.962))
(indexed-future-event 121 '#s(gc-info minor 242745424 392484616 0 234353184 392484616 7176 7176 1677477536916.399 1677477536916.425))
(indexed-future-event 122 '#s(gc-info minor 242753168 392484616 0 234354848 392484616 7179 7179 1677477536917.855 1677477536917.882))
(indexed-future-event 123 '#s(gc-info minor 242744832 392484616 0 234356624 392484616 7182 7182 1677477536919.317 1677477536919.343))
(indexed-future-event 124 '#s(gc-info minor 242772560 392484616 0 234358320 392484616 7185 7185 1677477536920.774 1677477536920.798))
(indexed-future-event 125 '#s(gc-info minor 242747920 392484616 0 234356064 392484616 7188 7188 1677477536922.238 1677477536922.265))
(indexed-future-event 126 '#s(gc-info minor 242757024 392484616 0 234357776 392484616 7190 7190 1677477536923.691 1677477536923.717))
(indexed-future-event 127 '#s(gc-info minor 242762384 392484616 0 234359536 392484616 7193 7193 1677477536925.149 1677477536925.174))
(indexed-future-event 128 '#s(gc-info minor 242766272 392484616 0 234361312 392484616 7196 7196 1677477536926.636 1677477536926.658))
(indexed-future-event 129 '#s(gc-info minor 242761312 392484616 0 234354256 392484616 7199 7199 1677477536928.089 1677477536928.132))
(indexed-future-event 130 '#s(gc-info minor 242743840 392484616 0 234355936 392484616 7202 7202 1677477536929.567 1677477536929.59))
(indexed-future-event 131 '#s(gc-info minor 242771936 392484616 0 234357600 392484616 7205 7205 1677477536931.026 1677477536931.051))
(indexed-future-event 132 '#s(gc-info minor 242747184 392484616 0 234359376 392484616 7208 7208 1677477536932.49 1677477536932.515))
(indexed-future-event 133 '#s(gc-info minor 242759376 392484616 0 234357072 392484616 7210 7210 1677477536933.944 1677477536933.972))
(indexed-future-event 134 '#s(gc-info minor 242747024 392484616 0 234358832 392484616 7213 7213 1677477536935.411 1677477536935.438))
(indexed-future-event 135 '#s(gc-info minor 242765088 392484616 0 234360624 392484616 7216 7216 1677477536936.884 1677477536936.907))
(indexed-future-event 136 '#s(gc-info minor 242776656 392484616 0 234362304 392484616 7219 7219 1677477536938.339 1677477536938.365))
(indexed-future-event 137 '#s(gc-info minor 242767856 392484616 0 234360096 392484616 7222 7222 1677477536939.798 1677477536939.828))
(indexed-future-event 138 '#s(gc-info minor 242760128 392484616 0 234361744 392484616 7225 7225 1677477536941.25 1677477536941.273))
(indexed-future-event 139 '#s(gc-info minor 242751296 392484616 0 234363536 392484616 7227 7227 1677477536942.717 1677477536942.743))
(indexed-future-event 140 '#s(gc-info minor 242763568 392484616 0 234365216 392484616 7230 7230 1677477536944.174 1677477536944.201))
(indexed-future-event 141 '#s(gc-info minor 242755152 392484616 0 234362976 392484616 7233 7233 1677477536945.636 1677477536945.663))
(indexed-future-event 142 '#s(gc-info minor 242763984 392484616 0 234364672 392484616 7236 7236 1677477536947.094 1677477536947.117))
(indexed-future-event 143 '#s(gc-info minor 242769280 392484616 0 234366432 392484616 7239 7239 1677477536948.551 1677477536948.576))
(indexed-future-event 144 '#s(gc-info minor 242755984 392484616 0 234368192 392484616 7242 7242 1677477536950.01 1677477536950.035))
(indexed-future-event 145 '#s(gc-info minor 242768160 392484616 0 234365856 392484616 7245 7245 1677477536951.465 1677477536951.495))
(indexed-future-event 146 '#s(gc-info minor 242772176 392484616 0 234367552 392484616 7247 7247 1677477536952.941 1677477536952.963))
(indexed-future-event 147 '#s(gc-info minor 242767520 392484616 0 234369216 392484616 7250 7250 1677477536954.395 1677477536954.418))
(indexed-future-event 148 '#s(gc-info minor 242758768 392484616 0 234370976 392484616 7253 7253 1677477536955.858 1677477536955.9))
(indexed-future-event 149 '#s(gc-info minor 242771328 392484616 0 234368704 392484616 7256 7256 1677477536957.321 1677477536957.349))
(indexed-future-event 150 '#s(gc-info minor 242758256 392484616 0 234370432 392484616 7259 7259 1677477536958.791 1677477536958.814))
(indexed-future-event 151 '#s(gc-info minor 242760016 392484616 0 234372208 392484616 7262 7262 1677477536960.25 1677477536960.274))
(indexed-future-event 152 '#s(gc-info minor 242788208 392484616 0 234373872 392484616 7264 7265 1677477536961.71 1677477536961.737))
(indexed-future-event 153 '#s(gc-info minor 242780224 392484616 0 234371696 392484616 7267 7267 1677477536963.178 1677477536963.204))
(indexed-future-event 154 '#s(gc-info minor 242787696 392484616 0 234373328 392484616 7270 7270 1677477536964.634 1677477536964.659))
(indexed-future-event 155 '#s(gc-info minor 242762912 392484616 0 234375104 392484616 7273 7273 1677477536966.097 1677477536966.119))
(indexed-future-event 156 '#s(gc-info minor 242774720 392484616 0 234376768 392484616 7276 7276 1677477536967.546 1677477536967.569))
(indexed-future-event 157 '#s(gc-info minor 242782352 392484616 0 234374576 392484616 7279 7279 1677477536969.02 1677477536969.046))
(indexed-future-event 158 '#s(gc-info minor 242764304 392484616 0 234376352 392484616 7282 7282 1677477536970.487 1677477536970.512))
(indexed-future-event 159 '#s(gc-info minor 242777344 392484616 0 234378016 392484616 7284 7284 1677477536971.942 1677477536971.969))
(indexed-future-event 160 '#s(gc-info minor 242783936 392484616 0 234379776 392484616 7287 7287 1677477536973.413 1677477536973.44))
(indexed-future-event 161 '#s(gc-info minor 242796144 392484616 0 234377472 392484616 7290 7290 1677477536974.872 1677477536974.9))
(indexed-future-event 162 '#s(gc-info minor 242783392 392484616 0 234379168 392484616 7293 7293 1677477536976.338 1677477536976.365))
(indexed-future-event 163 '#s(gc-info minor 242778768 392484616 0 234380832 392484616 7296 7296 1677477536977.792 1677477536977.819))
(indexed-future-event 164 '#s(gc-info minor 242786736 392484616 0 234382592 392484616 7299 7299 1677477536979.259 1677477536979.285))
(indexed-future-event 165 '#s(gc-info minor 242798960 392484616 0 234380320 392484616 7302 7302 1677477536980.713 1677477536980.739))
(indexed-future-event 166 '#s(gc-info minor 242786240 392484616 0 234382048 392484616 7304 7304 1677477536982.179 1677477536982.202))
(indexed-future-event 167 '#s(gc-info minor 242771600 392484616 0 234383808 392484616 7307 7307 1677477536983.645 1677477536983.677))
(indexed-future-event 168 '#s(gc-info minor 242783392 392484616 0 234385472 392484616 7310 7310 1677477536985.107 1677477536985.131))
(indexed-future-event 169 '#s(gc-info minor 242791024 392484616 0 234383296 392484616 7313 7313 1677477536986.57 1677477536986.597))
(indexed-future-event 170 '#s(gc-info minor 242783264 392484616 0 234384928 392484616 7316 7316 1677477536988.022 1677477536988.047))
(indexed-future-event 171 '#s(gc-info minor 242774480 392484616 0 234386688 392484616 7319 7319 1677477536989.483 1677477536989.508))
(indexed-future-event 172 '#s(gc-info minor 242802656 392484616 0 234388352 392484616 7321 7322 1677477536990.946 1677477536990.969))
(indexed-future-event 173 '#s(gc-info minor 242795440 392484616 0 234386176 392484616 7324 7324 1677477536992.43 1677477536992.457))
(indexed-future-event 174 '#s(gc-info minor 242774640 392484616 0 234388256 392484616 7327 7327 1677477536993.889 1677477536993.914))
(indexed-future-event 175 '#s(gc-info minor 242789424 392484616 0 234390000 392484616 7330 7330 1677477536995.346 1677477536995.372))
(indexed-future-event 176 '#s(gc-info minor 242795904 392484616 0 234391760 392484616 7333 7333 1677477536996.82 1677477536996.843))
(indexed-future-event 177 '#s(gc-info minor 242792144 392484616 0 234389072 392484616 7336 7336 1677477536998.274 1677477536998.303))
(indexed-future-event 178 '#s(gc-info minor 242794976 392484616 0 234390768 392484616 7339 7339 1677477536999.739 1677477536999.765))
(indexed-future-event 179 '#s(gc-info minor 242790768 392484616 0 234392416 392484616 7341 7341 1677477537001.193 1677477537001.216))
(indexed-future-event 180 '#s(gc-info minor 242797936 392484616 0 234394176 392484616 7344 7344 1677477537002.661 1677477537002.688))
(indexed-future-event 181 '#s(gc-info minor 242794560 392484616 0 234391920 392484616 7347 7347 1677477537004.113 1677477537004.142))
(indexed-future-event 182 '#s(gc-info minor 242797824 392484616 0 234393648 392484616 7350 7350 1677477537005.58 1677477537005.605))
(indexed-future-event 183 '#s(gc-info minor 242799568 392484616 0 234395408 392484616 7353 7353 1677477537007.052 1677477537007.078))
(indexed-future-event 184 '#s(gc-info minor 242795776 392484616 0 234397072 392484616 7356 7356 1677477537008.507 1677477537008.533))
(indexed-future-event 185 '#s(gc-info minor 242786608 392484616 0 234394912 392484616 7359 7359 1677477537009.968 1677477537009.995))
(indexed-future-event 186 '#s(gc-info minor 242794896 392484616 0 234396544 392484616 7361 7361 1677477537011.418 1677477537011.444))
(indexed-future-event 187 '#s(gc-info minor 242786080 392484616 0 234398304 392484616 7364 7364 1677477537012.879 1677477537012.904))
(indexed-future-event 188 '#s(gc-info minor 242814288 392484616 0 234399968 392484616 7367 7367 1677477537014.335 1677477537014.357))
(indexed-future-event 189 '#s(gc-info minor 242789504 392484616 0 234397792 392484616 7370 7370 1677477537015.797 1677477537015.821))
(indexed-future-event 190 '#s(gc-info minor 242786272 392484616 0 234399872 392484616 7373 7373 1677477537017.258 1677477537017.283))
(indexed-future-event 191 '#s(gc-info minor 242801360 392484616 0 234401568 392484616 7376 7376 1677477537018.712 1677477537018.741))
(indexed-future-event 192 '#s(gc-info minor 242807168 392484616 0 234403312 392484616 7378 7379 1677477537020.182 1677477537020.209))
(indexed-future-event 193 '#s(gc-info minor 242803296 392484616 0 234377376 392484616 7381 7381 1677477537021.643 1677477537021.684))
(indexed-future-event 194 '#s(gc-info minor 242767744 392484616 0 234379024 392484616 7384 7384 1677477537023.117 1677477537023.143))
(indexed-future-event 195 '#s(gc-info minor 242795392 392484616 0 234380688 392484616 7387 7387 1677477537024.582 1677477537024.608))
(indexed-future-event 196 '#s(gc-info minor 242770672 392484616 0 234382432 392484616 7390 7390 1677477537026.052 1677477537026.076))
(indexed-future-event 197 '#s(gc-info minor 242783184 392484616 0 234380192 392484616 7393 7393 1677477537027.511 1677477537027.537))
(indexed-future-event 198 '#s(gc-info minor 242769408 392484616 0 234381904 392484616 7396 7396 1677477537028.981 1677477537029.009))
(indexed-future-event 199 '#s(gc-info minor 242788192 392484616 0 234383664 392484616 7399 7399 1677477537030.462 1677477537030.485))
(indexed-future-event 200 '#s(gc-info minor 242783664 392484616 0 234385312 392484616 7401 7401 1677477537031.914 1677477537031.94))
(indexed-future-event 201 '#s(gc-info minor 242774832 392484616 0 234383168 392484616 7404 7404 1677477537033.394 1677477537033.421))
(indexed-future-event 202 '#s(gc-info minor 242783168 392484616 0 234384784 392484616 7407 7407 1677477537034.843 1677477537034.871))
(indexed-future-event 203 '#s(gc-info minor 242774304 392484616 0 234386560 392484616 7410 7410 1677477537036.322 1677477537036.35))
(indexed-future-event 204 '#s(gc-info minor 242802944 392484616 0 234388208 392484616 7413 7413 1677477537037.777 1677477537037.804))
(indexed-future-event 205 '#s(gc-info minor 242778112 392484616 0 234386064 392484616 7416 7416 1677477537039.248 1677477537039.276))
(indexed-future-event 206 '#s(gc-info minor 242786432 392484616 0 234387680 392484616 7419 7419 1677477537040.701 1677477537040.729))
(indexed-future-event 207 '#s(gc-info minor 242793584 392484616 0 234389440 392484616 7421 7421 1677477537042.17 1677477537042.193))
(indexed-future-event 208 '#s(gc-info minor 242778976 392484616 0 234391200 392484616 7424 7424 1677477537043.63 1677477537043.655))
(indexed-future-event 209 '#s(gc-info minor 242807568 392484616 0 234388976 392484616 7427 7427 1677477537045.081 1677477537045.108))
(indexed-future-event 210 '#s(gc-info minor 242778512 392484616 0 234390672 392484616 7430 7430 1677477537046.544 1677477537046.567))
(indexed-future-event 211 '#s(gc-info minor 242790656 392484616 0 234392336 392484616 7433 7433 1677477537047.994 1677477537048.021))
(indexed-future-event 212 '#s(gc-info minor 242782256 392484616 0 234394096 392484616 7436 7436 1677477537049.454 1677477537049.476))
(indexed-future-event 213 '#s(gc-info minor 242810080 392484616 0 234391824 392484616 7438 7438 1677477537050.908 1677477537050.933))
(indexed-future-event 214 '#s(gc-info minor 242781360 392484616 0 234393552 392484616 7441 7441 1677477537052.367 1677477537052.389))
(indexed-future-event 215 '#s(gc-info minor 242783104 392484616 0 234395312 392484616 7444 7444 1677477537053.826 1677477537053.85))
(indexed-future-event 216 '#s(gc-info minor 242811664 392484616 0 234396976 392484616 7447 7447 1677477537055.277 1677477537055.3))
(indexed-future-event 217 '#s(gc-info minor 242786464 392484616 0 234394800 392484616 7450 7450 1677477537056.742 1677477537056.768))
(indexed-future-event 218 '#s(gc-info minor 242795152 392484616 0 234396432 392484616 7453 7453 1677477537058.198 1677477537058.222))
(indexed-future-event 219 '#s(gc-info minor 242802368 392484616 0 234398192 392484616 7455 7456 1677477537059.658 1677477537059.685))
(indexed-future-event 220 '#s(gc-info minor 242798304 392484616 0 234399888 392484616 7458 7458 1677477537061.121 1677477537061.145))
(indexed-future-event 221 '#s(gc-info minor 242805824 392484616 0 234397712 392484616 7461 7461 1677477537062.583 1677477537062.61))
(indexed-future-event 222 '#s(gc-info minor 242797680 392484616 0 234399344 392484616 7464 7464 1677477537064.04 1677477537064.063))
(indexed-future-event 223 '#s(gc-info minor 242788896 392484616 0 234401104 392484616 7467 7467 1677477537065.504 1677477537065.529))
(indexed-future-event 224 '#s(gc-info minor 242807072 392484616 0 234402880 392484616 7470 7470 1677477537066.97 1677477537066.997))
(indexed-future-event 225 '#s(gc-info minor 242802880 392484616 0 234400624 392484616 7473 7473 1677477537068.429 1677477537068.46))
(indexed-future-event 226 '#s(gc-info minor 242806592 392484616 0 234402336 392484616 7475 7475 1677477537069.907 1677477537069.931))
(indexed-future-event 227 '#s(gc-info minor 242802336 392484616 0 234404000 392484616 7478 7478 1677477537071.363 1677477537071.388))
(indexed-future-event 228 '#s(gc-info minor 242793584 392484616 0 234405776 392484616 7481 7481 1677477537072.883 1677477537072.917))
(indexed-future-event 229 '#s(gc-info minor 242806160 392484616 0 234403472 392484616 7484 7484 1677477537074.358 1677477537074.396))
(indexed-future-event 230 '#s(gc-info minor 242793824 392484616 0 234405216 392484616 7487 7487 1677477537075.834 1677477537075.86))
(indexed-future-event 231 '#s(gc-info minor 242805984 392484616 0 234406880 392484616 7490 7490 1677477537077.283 1677477537077.306))
(indexed-future-event 232 '#s(gc-info minor 242797616 392484616 0 234408656 392484616 7493 7493 1677477537078.744 1677477537078.769))
(indexed-future-event 233 '#s(gc-info minor 242814976 392484616 0 234406480 392484616 7496 7496 1677477537080.203 1677477537080.232))
(indexed-future-event 234 '#s(gc-info minor 242807264 392484616 0 234408128 392484616 7498 7498 1677477537081.666 1677477537081.692))
(indexed-future-event 235 '#s(gc-info minor 242814448 392484616 0 234409920 392484616 7501 7501 1677477537083.131 1677477537083.158))
(indexed-future-event 236 '#s(gc-info minor 242810336 392484616 0 234411600 392484616 7504 7504 1677477537084.589 1677477537084.613))
(indexed-future-event 237 '#s(gc-info minor 242801152 392484616 0 234409360 392484616 7507 7507 1677477537086.053 1677477537086.081))
(indexed-future-event 238 '#s(gc-info minor 242809776 392484616 0 234411008 392484616 7510 7510 1677477537087.502 1677477537087.527))
(indexed-future-event 239 '#s(gc-info minor 242800560 392484616 0 234412800 392484616 7513 7513 1677477537088.97 1677477537088.994))
(indexed-future-event 240 '#s(gc-info minor 242813216 392484616 0 234414480 392484616 7516 7516 1677477537090.426 1677477537090.451))
(indexed-future-event 241 '#s(gc-info minor 242804032 392484616 0 234412272 392484616 7518 7518 1677477537091.897 1677477537091.928))
(indexed-future-event 242 '#s(gc-info minor 242802208 392484616 0 234413984 392484616 7521 7521 1677477537093.369 1677477537093.393))
(indexed-future-event 243 '#s(gc-info minor 242813968 392484616 0 234415680 392484616 7524 7524 1677477537094.828 1677477537094.853))
(indexed-future-event 244 '#s(gc-info minor 242822048 392484616 0 234417456 392484616 7527 7527 1677477537096.302 1677477537096.328))
(indexed-future-event 245 '#s(gc-info minor 242833824 392484616 0 234415120 392484616 7530 7530 1677477537097.762 1677477537097.788))
(indexed-future-event 246 '#s(gc-info minor 242804656 392484616 0 234416864 392484616 7533 7533 1677477537099.23 1677477537099.257))
(indexed-future-event 247 '#s(gc-info minor 242816848 392484616 0 234418560 392484616 7536 7536 1677477537100.691 1677477537100.719))
(indexed-future-event 248 '#s(gc-info minor 242824928 392484616 0 234420336 392484616 7538 7538 1677477537102.164 1677477537102.189))
(indexed-future-event 249 '#s(gc-info minor 242820704 392484616 0 234418000 392484616 7541 7541 1677477537103.617 1677477537103.644))
(indexed-future-event 250 '#s(gc-info minor 242824736 392484616 0 234419744 392484616 7544 7544 1677477537105.097 1677477537105.12))
(indexed-future-event 251 '#s(gc-info minor 242825648 392484616 0 234421504 392484616 7547 7547 1677477537106.569 1677477537106.595))
(indexed-future-event 252 '#s(gc-info minor 242821936 392484616 0 234423184 392484616 7550 7550 1677477537108.026 1677477537108.051))
(indexed-future-event 253 '#s(gc-info minor 242812720 392484616 0 234420944 392484616 7553 7553 1677477537109.492 1677477537109.518))
(indexed-future-event 254 '#s(gc-info minor 242821376 392484616 0 234422592 392484616 7556 7556 1677477537110.946 1677477537110.971))
(indexed-future-event 255 '#s(gc-info minor 242828512 392484616 0 234424352 392484616 7558 7558 1677477537112.408 1677477537112.43))
(indexed-future-event 256 '#s(gc-info minor 242840400 392484616 0 234426032 392484616 7561 7561 1677477537113.86 1677477537113.884))
(indexed-future-event 257 '#s(gc-info minor 242815568 392484616 0 234423888 392484616 7564 7564 1677477537115.321 1677477537115.363))
(indexed-future-event 258 '#s(gc-info minor 242812432 392484616 0 234425920 392484616 7567 7567 1677477537116.807 1677477537116.838))
(indexed-future-event 259 '#s(gc-info minor 242843392 392484616 0 234427632 392484616 7570 7570 1677477537118.283 1677477537118.308))
(indexed-future-event 260 '#s(gc-info minor 242833568 392484616 0 234429424 392484616 7573 7573 1677477537119.75 1677477537119.773))
(indexed-future-event 261 '#s(gc-info minor 242829456 392484616 0 234426720 392484616 7575 7575 1677477537121.201 1677477537121.226))
(indexed-future-event 262 '#s(gc-info minor 242816272 392484616 0 234428480 392484616 7578 7578 1677477537122.663 1677477537122.688))
(indexed-future-event 263 '#s(gc-info minor 242828896 392484616 0 234430160 392484616 7581 7581 1677477537124.117 1677477537124.141))
(indexed-future-event 264 '#s(gc-info minor 242819712 392484616 0 234431952 392484616 7584 7584 1677477537125.583 1677477537125.61))
(indexed-future-event 265 '#s(gc-info minor 242832368 392484616 0 234429600 392484616 7587 7587 1677477537127.034 1677477537127.064))
(indexed-future-event 266 '#s(gc-info minor 242835536 392484616 0 234431360 392484616 7590 7590 1677477537128.506 1677477537128.53))
(indexed-future-event 267 '#s(gc-info minor 242820960 392484616 0 234433136 392484616 7593 7593 1677477537129.97 1677477537129.996))
(indexed-future-event 268 '#s(gc-info minor 242833120 392484616 0 234434832 392484616 7595 7595 1677477537131.428 1677477537131.456))
(indexed-future-event 269 '#s(gc-info minor 242841200 392484616 0 234432576 392484616 7598 7598 1677477537132.895 1677477537132.922))
(indexed-future-event 270 '#s(gc-info minor 242832560 392484616 0 234434240 392484616 7601 7601 1677477537134.347 1677477537134.372))
(indexed-future-event 271 '#s(gc-info minor 242823840 392484616 0 234436032 392484616 7604 7604 1677477537135.811 1677477537135.836))
(indexed-future-event 272 '#s(gc-info minor 242852016 392484616 0 234437728 392484616 7607 7607 1677477537137.259 1677477537137.284))
(indexed-future-event 273 '#s(gc-info minor 242843328 392484616 0 234435504 392484616 7610 7610 1677477537138.728 1677477537138.759))
(indexed-future-event 274 '#s(gc-info minor 242839312 392484616 0 234437264 392484616 7612 7613 1677477537140.2 1677477537140.229))
(indexed-future-event 275 '#s(gc-info minor 242856560 392484616 0 234438976 392484616 7615 7615 1677477537141.663 1677477537141.686))
(indexed-future-event 276 '#s(gc-info minor 242828560 392484616 0 234440752 392484616 7618 7618 1677477537143.13 1677477537143.155))
(indexed-future-event 277 '#s(gc-info minor 242840752 392484616 0 234438336 392484616 7621 7621 1677477537144.584 1677477537144.612))
(indexed-future-event 278 '#s(gc-info minor 242844688 392484616 0 234440080 392484616 7624 7624 1677477537146.053 1677477537146.079))
(indexed-future-event 279 '#s(gc-info minor 242840464 392484616 0 234441744 392484616 7627 7627 1677477537147.504 1677477537147.528))
(indexed-future-event 280 '#s(gc-info minor 242847328 392484616 0 234443520 392484616 7630 7630 1677477537148.968 1677477537148.992))
(indexed-future-event 281 '#s(gc-info minor 242859520 392484616 0 234441216 392484616 7632 7632 1677477537150.424 1677477537150.453))
(indexed-future-event 282 '#s(gc-info minor 242847568 392484616 0 234442960 392484616 7635 7635 1677477537151.893 1677477537151.918))
(indexed-future-event 283 '#s(gc-info minor 242832896 392484616 0 234444752 392484616 7638 7638 1677477537153.354 1677477537153.378))
(indexed-future-event 284 '#s(gc-info minor 242844784 392484616 0 234446432 392484616 7641 7641 1677477537154.809 1677477537154.837))
(indexed-future-event 285 '#s(gc-info minor 242852752 392484616 0 234444224 392484616 7644 7644 1677477537156.283 1677477537156.318))
(indexed-future-event 286 '#s(gc-info minor 242845392 392484616 0 234511120 392484616 7647 7647 1677477537157.747 1677477537157.773))
(indexed-future-event 287 '#s(gc-info minor 242901056 392484616 0 234512912 392484616 7650 7650 1677477537159.211 1677477537159.235))
(indexed-future-event 288 '#s(gc-info minor 242913392 392484616 0 234563904 392484616 7652 7652 1677477537160.658 1677477537160.683))
(indexed-future-event 289 '#s(gc-info minor 242953456 392484616 0 234560320 392484616 7655 7655 1677477537162.123 1677477537162.154))
(indexed-future-event 290 '#s(gc-info minor 242960736 392484616 0 234561936 392484616 7658 7658 1677477537163.584 1677477537163.611))
(indexed-future-event 291 '#s(gc-info minor 242968256 392484616 0 234563728 392484616 7661 7661 1677477537165.051 1677477537165.075))
(indexed-future-event 292 '#s(gc-info minor 242953328 392484616 0 234565504 392484616 7664 7664 1677477537166.511 1677477537166.536))
(indexed-future-event 293 '#s(gc-info minor 242965472 392484616 0 234563168 392484616 7667 7667 1677477537167.967 1677477537167.998))
(indexed-future-event 294 '#s(gc-info minor 242969536 392484616 0 234564912 392484616 7669 7670 1677477537169.447 1677477537169.472))
(indexed-future-event 295 '#s(gc-info minor 242981664 392484616 0 234566608 392484616 7672 7672 1677477537170.902 1677477537170.929))
(indexed-future-event 296 '#s(gc-info minor 242956208 392484616 0 234568384 392484616 7675 7675 1677477537172.366 1677477537172.391))
(indexed-future-event 297 '#s(gc-info minor 242968368 392484616 0 234566048 392484616 7678 7678 1677477537173.82 1677477537173.846))
(indexed-future-event 298 '#s(gc-info minor 242955648 392484616 0 234567792 392484616 7681 7681 1677477537175.284 1677477537175.312))
(indexed-future-event 299 '#s(gc-info minor 242968160 392484616 0 234569488 392484616 7684 7684 1677477537176.74 1677477537176.766))
(indexed-future-event 300 '#s(future-event 1 6 complete 1677477537177.549 #f #f))
(indexed-future-event 301 '#s(future-event 1 6 end-work 1677477537177.552 #f #f))
(indexed-future-event 302 '#s(gc-info minor 242970080 392484616 0 234577168 392484616 7687 7687 1677477537178.838 1677477537178.869))
(indexed-future-event 303 '#s(gc-info minor 242971664 392484616 0 234590496 392484616 7689 7689 1677477537181.656 1677477537181.687))
(indexed-future-event 304 '#s(gc-info minor 242989888 392484616 0 234592528 392484616 7692 7692 1677477537184.502 1677477537184.521))
(indexed-future-event 305 '#s(gc-info minor 242987024 392484616 0 234594480 392484616 7695 7695 1677477537187.321 1677477537187.339))
(indexed-future-event 306 '#s(gc-info minor 242989056 392484616 0 234596448 392484616 7698 7698 1677477537190.134 1677477537190.15))
(indexed-future-event 307 '#s(gc-info minor 242990944 392484616 0 234583744 392484616 7700 7700 1677477537192.938 1677477537192.964))
(indexed-future-event 308 '#s(gc-info minor 242983072 392484616 0 234585728 392484616 7703 7703 1677477537195.787 1677477537195.804))
))
]

@interaction-eval-show[
    #:eval future-eval
           (show-timeline better-log )
]

More generally, we can create @racket[N] futures to perform the same
computation, and they will run in parallel:

@racketblock[
  (define fs
    (for/list ([i (in-range N)])
      (future (lambda () (mandelbrot 10000000 62 500 1000)))))
  (for/list ([f (in-list fs)])
    (touch f))
]

@interaction-eval[
    #:eval future-eval            
    (define four-log 
      (list
(indexed-future-event 0 '#s(future-event #f 0 create 1677477808230.212 #f 1))
(indexed-future-event 1 '#s(future-event #f 0 create 1677477808230.347 #f 2))
(indexed-future-event 2 '#s(future-event #f 0 create 1677477808230.351 #f 3))
(indexed-future-event 3 '#s(future-event #f 0 create 1677477808230.355 #f 4))
(indexed-future-event 4 '#s(future-event 1 0 start-work 1677477808230.359 #f #f))
(indexed-future-event 5 '#s(future-event 2 1 start-work 1677477808230.375 #f #f))
(indexed-future-event 6 '#s(future-event 4 2 start-work 1677477808230.375 #f #f))
(indexed-future-event 7 '#s(future-event 3 5 start-work 1677477808230.376 #f #f))
(indexed-future-event 8 '#s(gc-info minor 236598680 371021576 0 223732856 371021576 1210 1212 1677477808231.074 1677477808232.63))
(indexed-future-event 9 '#s(gc-info minor 232129320 371021576 0 223739816 371021576 1215 1215 1677477808233.419 1677477808233.49))
(indexed-future-event 10 '#s(gc-info minor 232151016 371021576 0 223741368 371021576 1218 1218 1677477808234.268 1677477808234.323))
(indexed-future-event 11 '#s(gc-info minor 232138488 371021576 0 223701464 371021576 1221 1222 1677477808235.088 1677477808235.708))
(indexed-future-event 12 '#s(gc-info minor 232105704 371021576 0 223690056 371021576 1225 1225 1677477808236.503 1677477808236.56))
(indexed-future-event 13 '#s(gc-info minor 232084888 371021576 0 223691640 371021576 1228 1228 1677477808237.332 1677477808237.383))
(indexed-future-event 14 '#s(gc-info minor 232119608 371021576 0 223693208 371021576 1231 1231 1677477808238.151 1677477808238.199))
(indexed-future-event 15 '#s(gc-info minor 232111864 371021576 0 223690888 371021576 1234 1234 1677477808238.979 1677477808239.051))
(indexed-future-event 16 '#s(gc-info minor 232102104 371021576 0 223692424 371021576 1237 1237 1677477808239.829 1677477808239.873))
(indexed-future-event 17 '#s(gc-info minor 232088824 371021576 0 223694008 371021576 1240 1240 1677477808240.666 1677477808240.712))
(indexed-future-event 18 '#s(gc-info minor 232088824 371021576 0 223695576 371021576 1243 1243 1677477808241.479 1677477808241.54))
(indexed-future-event 19 '#s(gc-info minor 232114264 371021576 0 223693272 371021576 1246 1246 1677477808242.3 1677477808242.343))
(indexed-future-event 20 '#s(gc-info minor 232121624 371021576 0 223694840 371021576 1249 1249 1677477808243.115 1677477808243.165))
(indexed-future-event 21 '#s(gc-info minor 232089640 371021576 0 223696408 371021576 1252 1252 1677477808243.925 1677477808243.986))
(indexed-future-event 22 '#s(gc-info minor 232114424 371021576 0 223698424 371021576 1255 1255 1677477808244.747 1677477808244.808))
(indexed-future-event 23 '#s(gc-info minor 232095496 371021576 0 223709896 371021576 1258 1259 1677477808245.569 1677477808246.51))
(indexed-future-event 24 '#s(gc-info minor 232154616 371021576 0 223694216 371021576 1262 1262 1677477808247.304 1677477808247.36))
(indexed-future-event 25 '#s(gc-info minor 232122568 371021576 0 223695816 371021576 1265 1265 1677477808248.121 1677477808248.158))
(indexed-future-event 26 '#s(gc-info minor 232099304 371021576 0 223697496 371021576 1268 1268 1677477808248.91 1677477808248.959))
(indexed-future-event 27 '#s(gc-info minor 232093080 371021576 0 223695096 371021576 1271 1271 1677477808249.727 1677477808249.775))
(indexed-future-event 28 '#s(gc-info minor 232107464 371021576 0 223696648 371021576 1274 1274 1677477808250.55 1677477808250.603))
(indexed-future-event 29 '#s(gc-info minor 232107080 371021576 0 223698216 371021576 1277 1277 1677477808251.38 1677477808251.434))
(indexed-future-event 30 '#s(gc-info minor 232101640 371021576 0 223699880 371021576 1280 1280 1677477808252.201 1677477808252.251))
(indexed-future-event 31 '#s(gc-info minor 232110712 371021576 0 223697240 371021576 1283 1283 1677477808253.025 1677477808253.082))
(indexed-future-event 32 '#s(gc-info minor 232093256 371021576 0 223699048 371021576 1286 1286 1677477808253.835 1677477808253.884))
(indexed-future-event 33 '#s(gc-info minor 232100968 371021576 0 223700712 371021576 1289 1289 1677477808254.662 1677477808254.704))
(indexed-future-event 34 '#s(gc-info minor 232113064 371021576 0 223702264 371021576 1292 1292 1677477808255.481 1677477808255.529))
(indexed-future-event 35 '#s(gc-info minor 232113464 371021576 0 223699928 371021576 1295 1295 1677477808256.287 1677477808256.345))
(indexed-future-event 36 '#s(gc-info minor 232095496 371021576 0 223701464 371021576 1298 1298 1677477808257.122 1677477808257.172))
(indexed-future-event 37 '#s(gc-info minor 232119752 371021576 0 223703128 371021576 1301 1301 1677477808257.934 1677477808257.976))
(indexed-future-event 38 '#s(gc-info minor 232114344 371021576 0 223704712 371021576 1304 1304 1677477808258.745 1677477808258.788))
(indexed-future-event 39 '#s(gc-info minor 232115144 371021576 0 223702328 371021576 1307 1307 1677477808259.556 1677477808259.604))
(indexed-future-event 40 '#s(gc-info minor 232104216 371021576 0 223703928 371021576 1310 1310 1677477808260.379 1677477808260.426))
(indexed-future-event 41 '#s(gc-info minor 232115144 371021576 0 223705496 371021576 1313 1313 1677477808261.192 1677477808261.234))
(indexed-future-event 42 '#s(gc-info minor 232116360 371021576 0 223707080 371021576 1316 1316 1677477808262.006 1677477808262.045))
(indexed-future-event 43 '#s(gc-info minor 232101512 371021576 0 223704696 371021576 1319 1319 1677477808262.81 1677477808262.854))
(indexed-future-event 44 '#s(gc-info minor 232106664 371021576 0 223706360 371021576 1322 1322 1677477808263.625 1677477808263.668))
(indexed-future-event 45 '#s(gc-info minor 232116392 371021576 0 223707928 371021576 1325 1325 1677477808264.448 1677477808264.488))
(indexed-future-event 46 '#s(gc-info minor 232134760 371021576 0 223709496 371021576 1328 1328 1677477808265.271 1677477808265.31))
(indexed-future-event 47 '#s(gc-info minor 232111464 371021576 0 223707208 371021576 1331 1331 1677477808266.082 1677477808266.136))
(indexed-future-event 48 '#s(gc-info minor 232103176 371021576 0 223708744 371021576 1334 1334 1677477808266.888 1677477808266.936))
(indexed-future-event 49 '#s(gc-info minor 232119560 371021576 0 223710296 371021576 1336 1337 1677477808267.7 1677477808267.754))
(indexed-future-event 50 '#s(gc-info minor 232127816 371021576 0 223711704 371021576 1340 1340 1677477808268.545 1677477808268.601))
(indexed-future-event 51 '#s(gc-info minor 232140104 371021576 0 223709608 371021576 1343 1343 1677477808269.361 1677477808269.415))
(indexed-future-event 52 '#s(gc-info minor 232105576 371021576 0 223711144 371021576 1345 1346 1677477808270.165 1677477808270.218))
(indexed-future-event 53 '#s(gc-info minor 232122360 371021576 0 223712728 371021576 1348 1349 1677477808270.983 1677477808271.034))
(indexed-future-event 54 '#s(gc-info minor 232130632 371021576 0 223714424 371021576 1351 1352 1677477808271.806 1677477808271.848))
(indexed-future-event 55 '#s(gc-info minor 232125992 371021576 0 223712024 371021576 1354 1355 1677477808272.617 1677477808272.669))
(indexed-future-event 56 '#s(gc-info minor 232123240 371021576 0 223713528 371021576 1357 1358 1677477808273.438 1677477808273.484))
(indexed-future-event 57 '#s(gc-info minor 232114728 371021576 0 223715208 371021576 1360 1360 1677477808274.259 1677477808274.304))
(indexed-future-event 58 '#s(gc-info minor 232110792 371021576 0 223716776 371021576 1363 1363 1677477808275.064 1677477808275.112))
(indexed-future-event 59 '#s(gc-info minor 232143208 371021576 0 223714376 371021576 1366 1366 1677477808275.885 1677477808275.939))
(indexed-future-event 60 '#s(gc-info minor 232128264 371021576 0 223715944 371021576 1369 1369 1677477808276.707 1677477808276.766))
(indexed-future-event 61 '#s(gc-info minor 232117128 371021576 0 223717624 371021576 1372 1372 1677477808277.54 1677477808277.594))
(indexed-future-event 62 '#s(gc-info minor 232129208 371021576 0 223719192 371021576 1375 1375 1677477808278.363 1677477808278.406))
(indexed-future-event 63 '#s(gc-info minor 232113976 371021576 0 223716808 371021576 1378 1378 1677477808279.168 1677477808279.224))
(indexed-future-event 64 '#s(gc-info minor 232118712 371021576 0 223718472 371021576 1381 1381 1677477808279.99 1677477808280.034))
(indexed-future-event 65 '#s(gc-info minor 232113272 371021576 0 223720040 371021576 1384 1384 1677477808280.802 1677477808280.844))
(indexed-future-event 66 '#s(gc-info minor 232114488 371021576 0 223721608 371021576 1387 1387 1677477808281.611 1677477808281.668))
(indexed-future-event 67 '#s(gc-info minor 232117240 371021576 0 223719208 371021576 1390 1390 1677477808282.422 1677477808282.471))
(indexed-future-event 68 '#s(gc-info minor 232121128 371021576 0 223720840 371021576 1393 1393 1677477808283.237 1677477808283.284))
(indexed-future-event 69 '#s(gc-info minor 232131656 371021576 0 223722392 371021576 1396 1396 1677477808284.061 1677477808284.11))
(indexed-future-event 70 '#s(gc-info minor 232134360 371021576 0 223723992 371021576 1399 1399 1677477808284.861 1677477808284.903))
(indexed-future-event 71 '#s(gc-info minor 232125128 371021576 0 223813656 371021576 1402 1404 1677477808285.68 1677477808287.671))
(indexed-future-event 72 '#s(gc-info minor 232225624 371021576 0 223711496 371021576 1407 1407 1677477808288.461 1677477808288.569))
(indexed-future-event 73 '#s(gc-info minor 232107864 371021576 0 223713080 371021576 1410 1410 1677477808289.316 1677477808289.376))
(indexed-future-event 74 '#s(gc-info minor 232109000 371021576 0 223714648 371021576 1413 1413 1677477808290.13 1677477808290.178))
(indexed-future-event 75 '#s(gc-info minor 232116152 371021576 0 223712344 371021576 1416 1416 1677477808290.946 1677477808290.998))
(indexed-future-event 76 '#s(gc-info minor 232107944 371021576 0 223713880 371021576 1419 1419 1677477808291.761 1677477808291.814))
(indexed-future-event 77 '#s(gc-info minor 232141512 371021576 0 223715464 371021576 1422 1422 1677477808292.577 1677477808292.624))
(indexed-future-event 78 '#s(gc-info minor 232133000 371021576 0 223717128 371021576 1425 1425 1677477808293.4 1677477808293.442))
(indexed-future-event 79 '#s(gc-info minor 232111560 371021576 0 223714728 371021576 1428 1428 1677477808294.209 1677477808294.259))
(indexed-future-event 80 '#s(gc-info minor 232125544 371021576 0 223716296 371021576 1431 1431 1677477808295.023 1677477808295.07))
(indexed-future-event 81 '#s(gc-info minor 232114712 371021576 0 223718280 371021576 1434 1434 1677477808295.839 1677477808295.909))
(indexed-future-event 82 '#s(gc-info minor 232117848 371021576 0 223719992 371021576 1437 1437 1677477808296.675 1677477808296.721))
(indexed-future-event 83 '#s(gc-info minor 232130808 371021576 0 223717144 371021576 1440 1440 1677477808297.486 1677477808297.53))
(indexed-future-event 84 '#s(gc-info minor 232128328 371021576 0 223718664 371021576 1443 1443 1677477808298.295 1677477808298.337))
(indexed-future-event 85 '#s(gc-info minor 232118264 371021576 0 223720328 371021576 1446 1446 1677477808299.154 1677477808299.199))
(indexed-future-event 86 '#s(gc-info minor 232147576 371021576 0 223721912 371021576 1449 1449 1677477808299.968 1677477808300.009))
(indexed-future-event 87 '#s(gc-info minor 232132344 371021576 0 223719576 371021576 1452 1452 1677477808300.779 1677477808300.827))
(indexed-future-event 88 '#s(gc-info minor 232121112 371021576 0 223721160 371021576 1455 1455 1677477808301.598 1677477808301.641))
(indexed-future-event 89 '#s(gc-info minor 232131976 371021576 0 223722760 371021576 1458 1458 1677477808302.411 1677477808302.459))
(indexed-future-event 90 '#s(gc-info minor 232133560 371021576 0 223724328 371021576 1461 1461 1677477808303.224 1677477808303.269))
(indexed-future-event 91 '#s(gc-info minor 232125464 371021576 0 223722040 371021576 1464 1464 1677477808304.039 1677477808304.093))
(indexed-future-event 92 '#s(gc-info minor 232116872 371021576 0 223723592 371021576 1467 1467 1677477808304.858 1677477808304.903))
(indexed-future-event 93 '#s(gc-info minor 232134024 371021576 0 223725160 371021576 1470 1470 1677477808305.671 1677477808305.71))
(indexed-future-event 94 '#s(gc-info minor 232119976 371021576 0 223726712 371021576 1473 1473 1677477808306.477 1677477808306.521))
(indexed-future-event 95 '#s(gc-info minor 232175848 371021576 0 223724184 371021576 1476 1476 1677477808307.321 1677477808307.362))
(indexed-future-event 96 '#s(gc-info minor 232135048 371021576 0 223725992 371021576 1479 1479 1677477808308.131 1677477808308.171))
(indexed-future-event 97 '#s(gc-info minor 232136424 371021576 0 223727304 371021576 1482 1482 1677477808308.946 1677477808309.0))
(indexed-future-event 98 '#s(gc-info minor 232128840 371021576 0 223729208 371021576 1485 1485 1677477808309.778 1677477808309.824))
(indexed-future-event 99 '#s(gc-info minor 232140408 371021576 0 223726872 371021576 1488 1488 1677477808310.591 1677477808310.637))
(indexed-future-event 100 '#s(gc-info minor 232137672 371021576 0 223728408 371021576 1491 1491 1677477808311.4 1677477808311.444))
(indexed-future-event 101 '#s(gc-info minor 232129160 371021576 0 223730072 371021576 1494 1494 1677477808312.218 1677477808312.256))
(indexed-future-event 102 '#s(gc-info minor 232126440 371021576 0 223731656 371021576 1497 1497 1677477808312.997 1677477808313.038))
(indexed-future-event 103 '#s(gc-info minor 232126088 371021576 0 223729272 371021576 1500 1500 1677477808313.802 1677477808313.85))
(indexed-future-event 104 '#s(gc-info minor 232140856 371021576 0 223730760 371021576 1503 1503 1677477808314.611 1677477808314.649))
(indexed-future-event 105 '#s(gc-info minor 232150584 371021576 0 223732424 371021576 1506 1506 1677477808315.445 1677477808315.485))
(indexed-future-event 106 '#s(gc-info minor 232143992 371021576 0 223734008 371021576 1509 1509 1677477808316.251 1677477808316.292))
(indexed-future-event 107 '#s(gc-info minor 232145208 371021576 0 223731640 371021576 1512 1512 1677477808317.064 1677477808317.105))
(indexed-future-event 108 '#s(gc-info minor 232132408 371021576 0 223733256 371021576 1514 1515 1677477808317.878 1677477808317.924))
(indexed-future-event 109 '#s(gc-info minor 232144840 371021576 0 223734856 371021576 1517 1518 1677477808318.687 1677477808318.73))
(indexed-future-event 110 '#s(gc-info minor 232145272 371021576 0 223736424 371021576 1520 1520 1677477808319.499 1677477808319.541))
(indexed-future-event 111 '#s(gc-info minor 232132232 371021576 0 223734088 371021576 1523 1523 1677477808320.31 1677477808320.361))
(indexed-future-event 112 '#s(gc-info minor 232151064 371021576 0 223735720 371021576 1526 1526 1677477808321.122 1677477808321.172))
(indexed-future-event 113 '#s(gc-info minor 232146536 371021576 0 223737272 371021576 1529 1529 1677477808321.938 1677477808321.979))
(indexed-future-event 114 '#s(gc-info minor 232148088 371021576 0 223738616 371021576 1532 1532 1677477808322.745 1677477808322.787))
(indexed-future-event 115 '#s(gc-info minor 232140184 371021576 0 223736552 371021576 1535 1535 1677477808323.564 1677477808323.638))
(indexed-future-event 116 '#s(gc-info minor 232132520 371021576 0 223738088 371021576 1538 1538 1677477808324.381 1677477808324.435))
(indexed-future-event 117 '#s(gc-info minor 232166072 371021576 0 223739416 371021576 1541 1541 1677477808325.19 1677477808325.245))
(indexed-future-event 118 '#s(gc-info minor 232140536 371021576 0 223741704 371021576 1544 1544 1677477808326.013 1677477808326.062))
(indexed-future-event 119 '#s(gc-info minor 232138120 371021576 0 223738968 371021576 1547 1547 1677477808326.82 1677477808326.874))
(indexed-future-event 120 '#s(gc-info minor 232134168 371021576 0 223740488 371021576 1550 1550 1677477808327.632 1677477808327.68))
(indexed-future-event 121 '#s(gc-info minor 232151688 371021576 0 223742056 371021576 1553 1553 1677477808328.444 1677477808328.481))
(indexed-future-event 122 '#s(gc-info minor 232143208 371021576 0 223743704 371021576 1556 1556 1677477808329.258 1677477808329.3))
(indexed-future-event 123 '#s(gc-info minor 232138136 371021576 0 223741368 371021576 1559 1559 1677477808330.066 1677477808330.111))
(indexed-future-event 124 '#s(gc-info minor 232152168 371021576 0 223742920 371021576 1562 1562 1677477808330.873 1677477808330.916))
(indexed-future-event 125 '#s(gc-info minor 232143672 371021576 0 223744584 371021576 1565 1565 1677477808331.698 1677477808331.744))
(indexed-future-event 126 '#s(gc-info minor 232140184 371021576 0 223746168 371021576 1568 1568 1677477808332.5 1677477808332.542))
(indexed-future-event 127 '#s(gc-info minor 232172984 371021576 0 223743752 371021576 1571 1571 1677477808333.31 1677477808333.352))
(indexed-future-event 128 '#s(gc-info minor 232156856 371021576 0 223745352 371021576 1574 1574 1677477808334.134 1677477808334.177))
(indexed-future-event 129 '#s(gc-info minor 232145448 371021576 0 223747032 371021576 1577 1577 1677477808334.946 1677477808334.987))
(indexed-future-event 130 '#s(gc-info minor 232173848 371021576 0 223748600 371021576 1580 1580 1677477808335.761 1677477808335.8))
(indexed-future-event 131 '#s(gc-info minor 232175432 371021576 0 223746168 371021576 1583 1583 1677477808336.574 1677477808336.629))
(indexed-future-event 132 '#s(gc-info minor 232148840 371021576 0 223747832 371021576 1586 1586 1677477808337.399 1677477808337.443))
(indexed-future-event 133 '#s(gc-info minor 232159016 371021576 0 223749400 371021576 1589 1589 1677477808338.211 1677477808338.253))
(indexed-future-event 134 '#s(gc-info minor 232176616 371021576 0 223750968 371021576 1592 1592 1677477808339.018 1677477808339.062))
(indexed-future-event 135 '#s(gc-info minor 232167784 371021576 0 223748728 371021576 1595 1595 1677477808339.836 1677477808339.896))
(indexed-future-event 136 '#s(gc-info minor 232159544 371021576 0 223750200 371021576 1598 1598 1677477808340.665 1677477808340.709))
(indexed-future-event 137 '#s(gc-info minor 232161016 371021576 0 223751752 371021576 1601 1601 1677477808341.475 1677477808341.515))
(indexed-future-event 138 '#s(gc-info minor 220778936 371021576 0 212375880 371021576 1604 1604 1677477808342.289 1677477808342.331))
(indexed-future-event 139 '#s(gc-info minor 232167496 371021576 0 223749992 371021576 1607 1607 1677477808343.1 1677477808343.15))
(indexed-future-event 140 '#s(gc-info minor 232176824 371021576 0 223752056 371021576 1610 1610 1677477808343.923 1677477808343.962))
(indexed-future-event 141 '#s(gc-info minor 232148024 371021576 0 223753624 371021576 1613 1613 1677477808344.702 1677477808344.742))
(indexed-future-event 142 '#s(gc-info minor 232154360 371021576 0 223755288 371021576 1616 1616 1677477808345.521 1677477808345.573))
(indexed-future-event 143 '#s(gc-info minor 232166888 371021576 0 223752744 371021576 1619 1619 1677477808346.333 1677477808346.385))
(indexed-future-event 144 '#s(gc-info minor 232147992 371021576 0 223754552 371021576 1622 1622 1677477808347.143 1677477808347.183))
(indexed-future-event 145 '#s(gc-info minor 232154248 371021576 0 223756568 371021576 1624 1625 1677477808347.956 1677477808348.001))
(indexed-future-event 146 '#s(gc-info minor 232152888 371021576 0 223758200 371021576 1627 1628 1677477808348.77 1677477808348.81))
(indexed-future-event 147 '#s(gc-info minor 232185000 371021576 0 223755400 371021576 1630 1630 1677477808349.583 1677477808349.63))
(indexed-future-event 148 '#s(gc-info minor 232150232 371021576 0 223756936 371021576 1633 1633 1677477808350.398 1677477808350.44))
(indexed-future-event 149 '#s(gc-info minor 232157752 371021576 0 223758616 371021576 1636 1636 1677477808351.215 1677477808351.257))
(indexed-future-event 150 '#s(gc-info minor 232185816 371021576 0 223760184 371021576 1639 1639 1677477808352.033 1677477808352.08))
(indexed-future-event 151 '#s(gc-info minor 232171384 371021576 0 223757560 371021576 1642 1642 1677477808352.867 1677477808352.917))
(indexed-future-event 152 '#s(gc-info minor 232159848 371021576 0 223759416 371021576 1645 1645 1677477808353.694 1677477808353.737))
(indexed-future-event 153 '#s(gc-info minor 232170664 371021576 0 223761000 371021576 1648 1648 1677477808354.504 1677477808354.546))
(indexed-future-event 154 '#s(gc-info minor 232172200 371021576 0 223762568 371021576 1651 1651 1677477808355.308 1677477808355.354))
(indexed-future-event 155 '#s(gc-info minor 232158760 371021576 0 223760264 371021576 1654 1654 1677477808356.116 1677477808356.167))
(indexed-future-event 156 '#s(gc-info minor 232161224 371021576 0 223761896 371021576 1657 1657 1677477808356.935 1677477808356.984))
(indexed-future-event 157 '#s(gc-info minor 232173112 371021576 0 223763464 371021576 1660 1660 1677477808357.751 1677477808357.802))
(indexed-future-event 158 '#s(gc-info minor 232175096 371021576 0 223765048 371021576 1663 1663 1677477808358.555 1677477808358.598))
(indexed-future-event 159 '#s(gc-info minor 232166584 371021576 0 223762696 371021576 1666 1666 1677477808359.369 1677477808359.416))
(indexed-future-event 160 '#s(gc-info minor 232173880 371021576 0 223764216 371021576 1669 1669 1677477808360.185 1677477808360.227))
(indexed-future-event 161 '#s(gc-info minor 232175416 371021576 0 223765816 371021576 1672 1672 1677477808360.994 1677477808361.041))
(indexed-future-event 162 '#s(gc-info minor 232166648 371021576 0 223767832 371021576 1675 1675 1677477808361.816 1677477808361.861))
(indexed-future-event 163 '#s(gc-info minor 232180216 371021576 0 223765112 371021576 1678 1678 1677477808362.626 1677477808362.681))
(indexed-future-event 164 '#s(gc-info minor 232176696 371021576 0 223766648 371021576 1681 1681 1677477808363.442 1677477808363.497))
(indexed-future-event 165 '#s(gc-info minor 232178232 371021576 0 223768200 371021576 1684 1684 1677477808364.256 1677477808364.3))
(indexed-future-event 166 '#s(gc-info minor 232170424 371021576 0 223769864 371021576 1687 1687 1677477808365.081 1677477808365.128))
(indexed-future-event 167 '#s(gc-info minor 232180728 371021576 0 223767544 371021576 1690 1690 1677477808365.904 1677477808365.955))
(indexed-future-event 168 '#s(gc-info minor 232178744 371021576 0 223769048 371021576 1693 1693 1677477808366.715 1677477808366.758))
(indexed-future-event 169 '#s(gc-info minor 232186584 371021576 0 223770696 371021576 1696 1696 1677477808367.527 1677477808367.571))
(indexed-future-event 170 '#s(gc-info minor 232181896 371021576 0 223772296 371021576 1699 1699 1677477808368.336 1677477808368.378))
(indexed-future-event 171 '#s(gc-info minor 232183096 371021576 0 223769896 371021576 1702 1702 1677477808369.146 1677477808369.2))
(indexed-future-event 172 '#s(gc-info minor 232165496 371021576 0 223771432 371021576 1705 1705 1677477808369.953 1677477808369.997))
(indexed-future-event 173 '#s(gc-info minor 232173400 371021576 0 223773112 371021576 1708 1708 1677477808370.761 1677477808370.803))
(indexed-future-event 174 '#s(gc-info minor 232183544 371021576 0 223774680 371021576 1711 1711 1677477808371.571 1677477808371.613))
(indexed-future-event 175 '#s(gc-info minor 232185112 371021576 0 223772280 371021576 1714 1714 1677477808372.387 1677477808372.435))
(indexed-future-event 176 '#s(gc-info minor 232173400 371021576 0 223773912 371021576 1717 1717 1677477808373.213 1677477808373.263))
(indexed-future-event 177 '#s(gc-info minor 232185544 371021576 0 223775496 371021576 1720 1720 1677477808374.017 1677477808374.061))
(indexed-future-event 178 '#s(gc-info minor 232202312 371021576 0 223777064 371021576 1723 1723 1677477808374.825 1677477808374.871))
(indexed-future-event 179 '#s(gc-info minor 232177832 371021576 0 223774776 371021576 1726 1726 1677477808375.652 1677477808375.695))
(indexed-future-event 180 '#s(gc-info minor 232186664 371021576 0 223841688 371021576 1729 1729 1677477808376.454 1677477808376.505))
(indexed-future-event 181 '#s(gc-info minor 232236872 371021576 0 223843512 371021576 1732 1732 1677477808377.26 1677477808377.303))
(indexed-future-event 182 '#s(gc-info minor 232254344 371021576 0 223845080 371021576 1734 1735 1677477808378.072 1677477808378.117))
(indexed-future-event 183 '#s(gc-info minor 232246280 371021576 0 223842264 371021576 1737 1737 1677477808378.884 1677477808378.932))
(indexed-future-event 184 '#s(gc-info minor 232237848 371021576 0 223843768 371021576 1740 1740 1677477808379.694 1677477808379.738))
(indexed-future-event 185 '#s(gc-info minor 232270968 371021576 0 223845064 371021576 1743 1743 1677477808380.507 1677477808380.553))
(indexed-future-event 186 '#s(gc-info minor 232246200 371021576 0 223846984 371021576 1746 1746 1677477808381.323 1677477808381.371))
(indexed-future-event 187 '#s(gc-info minor 232241848 371021576 0 223844632 371021576 1749 1749 1677477808382.135 1677477808382.185))
(indexed-future-event 188 '#s(gc-info minor 232271448 371021576 0 223846168 371021576 1752 1752 1677477808382.959 1677477808383.008))
(indexed-future-event 189 '#s(gc-info minor 232264472 371021576 0 223847816 371021576 1755 1755 1677477808383.78 1677477808383.824))
(indexed-future-event 190 '#s(gc-info minor 232242632 371021576 0 223849416 371021576 1758 1758 1677477808384.595 1677477808384.644))
(indexed-future-event 191 '#s(gc-info minor 232244600 371021576 0 223847016 371021576 1761 1761 1677477808385.402 1677477808385.443))
(indexed-future-event 192 '#s(gc-info minor 232241848 371021576 0 223848552 371021576 1764 1764 1677477808386.213 1677477808386.255))
(indexed-future-event 193 '#s(gc-info minor 232249752 371021576 0 223850232 371021576 1767 1767 1677477808387.031 1677477808387.076))
(indexed-future-event 194 '#s(gc-info minor 232245048 371021576 0 223851800 371021576 1770 1770 1677477808387.84 1677477808387.887))
(indexed-future-event 195 '#s(gc-info minor 232262616 371021576 0 223849400 371021576 1773 1773 1677477808388.657 1677477808388.697))
(indexed-future-event 196 '#s(gc-info minor 232250536 371021576 0 223851032 371021576 1776 1776 1677477808389.469 1677477808389.511))
(indexed-future-event 197 '#s(gc-info minor 232280568 371021576 0 223852616 371021576 1779 1779 1677477808390.255 1677477808390.307))
(indexed-future-event 198 '#s(gc-info minor 232263432 371021576 0 223854184 371021576 1782 1782 1677477808391.077 1677477808391.118))
(indexed-future-event 199 '#s(gc-info minor 232249608 371021576 0 223848840 371021576 1785 1785 1677477808391.885 1677477808391.943))
(indexed-future-event 200 '#s(gc-info minor 232247880 371021576 0 223850664 371021576 1788 1788 1677477808392.722 1677477808392.767))
(indexed-future-event 201 '#s(gc-info minor 232278264 371021576 0 223852232 371021576 1791 1791 1677477808393.537 1677477808393.581))
(indexed-future-event 202 '#s(gc-info minor 232247096 371021576 0 223853816 371021576 1794 1794 1677477808394.35 1677477808394.39))
(indexed-future-event 203 '#s(gc-info minor 232255352 371021576 0 223851464 371021576 1797 1797 1677477808395.17 1677477808395.212))
(indexed-future-event 204 '#s(gc-info minor 232246280 371021576 0 223852984 371021576 1800 1800 1677477808395.985 1677477808396.028))
(indexed-future-event 205 '#s(gc-info minor 232247800 371021576 0 223854584 371021576 1803 1803 1677477808396.79 1677477808396.833))
(indexed-future-event 206 '#s(gc-info minor 232272152 371021576 0 223856264 371021576 1806 1806 1677477808397.606 1677477808397.644))
(indexed-future-event 207 '#s(gc-info minor 232267080 371021576 0 223853864 371021576 1809 1809 1677477808398.41 1677477808398.456))
(indexed-future-event 208 '#s(gc-info minor 232248312 371021576 0 223855416 371021576 1812 1812 1677477808399.222 1677477808399.293))
(indexed-future-event 209 '#s(gc-info minor 232267752 371021576 0 223856984 371021576 1815 1815 1677477808400.045 1677477808400.092))
(indexed-future-event 210 '#s(gc-info minor 232258920 371021576 0 223858648 371021576 1818 1818 1677477808400.865 1677477808400.908))
(indexed-future-event 211 '#s(gc-info minor 232285896 371021576 0 223856280 371021576 1821 1821 1677477808401.685 1677477808401.736))
(indexed-future-event 212 '#s(gc-info minor 232251096 371021576 0 223857816 371021576 1824 1824 1677477808402.497 1677477808402.548))
(indexed-future-event 213 '#s(gc-info minor 232259352 371021576 0 223859464 371021576 1827 1827 1677477808403.327 1677477808403.374))
(indexed-future-event 214 '#s(gc-info minor 232254216 371021576 0 223861064 371021576 1830 1830 1677477808404.136 1677477808404.179))
(indexed-future-event 215 '#s(gc-info minor 232303096 371021576 0 223858696 371021576 1833 1833 1677477808404.967 1677477808405.02))
(indexed-future-event 216 '#s(gc-info minor 232260216 371021576 0 223860296 371021576 1836 1836 1677477808405.786 1677477808405.833))
(indexed-future-event 217 '#s(gc-info minor 232255512 371021576 0 223861880 371021576 1839 1839 1677477808406.604 1677477808406.649))
(indexed-future-event 218 '#s(gc-info minor 232257080 371021576 0 223863448 371021576 1842 1842 1677477808407.413 1677477808407.467))
(indexed-future-event 219 '#s(gc-info minor 232275032 371021576 0 223861048 371021576 1845 1845 1677477808408.223 1677477808408.274))
(indexed-future-event 220 '#s(gc-info minor 232280104 371021576 0 223862680 371021576 1848 1848 1677477808409.048 1677477808409.103))
(indexed-future-event 221 '#s(gc-info minor 232289928 371021576 0 223864264 371021576 1851 1851 1677477808409.873 1677477808409.924))
(indexed-future-event 222 '#s(gc-info minor 232275848 371021576 0 223865832 371021576 1854 1854 1677477808410.681 1677477808410.724))
(indexed-future-event 223 '#s(gc-info minor 232267368 371021576 0 223863544 371021576 1857 1857 1677477808411.491 1677477808411.552))
(indexed-future-event 224 '#s(gc-info minor 232276280 371021576 0 223865112 371021576 1860 1860 1677477808412.306 1677477808412.352))
(indexed-future-event 225 '#s(gc-info minor 232275528 371021576 0 223866680 371021576 1863 1863 1677477808413.121 1677477808413.175))
(indexed-future-event 226 '#s(gc-info minor 232279048 371021576 0 223868248 371021576 1865 1866 1677477808413.929 1677477808413.975))
(indexed-future-event 227 '#s(gc-info minor 232286600 371021576 0 223865976 371021576 1868 1869 1677477808414.746 1677477808414.814))
(indexed-future-event 228 '#s(gc-info minor 232278712 371021576 0 223867512 371021576 1871 1871 1677477808415.56 1677477808415.603))
(indexed-future-event 229 '#s(gc-info minor 232278328 371021576 0 223869064 371021576 1874 1874 1677477808416.37 1677477808416.42))
(indexed-future-event 230 '#s(gc-info minor 232302584 371021576 0 223870728 371021576 1877 1877 1677477808417.197 1677477808417.244))
(indexed-future-event 231 '#s(gc-info minor 232281928 371021576 0 223868408 371021576 1880 1880 1677477808418.008 1677477808418.07))
(indexed-future-event 232 '#s(gc-info minor 232279992 371021576 0 223869912 371021576 1883 1883 1677477808418.823 1677477808418.866))
(indexed-future-event 233 '#s(gc-info minor 232281528 371021576 0 223871704 371021576 1886 1886 1677477808419.635 1677477808419.682))
(indexed-future-event 234 '#s(gc-info minor 232273240 371021576 0 223873368 371021576 1889 1889 1677477808420.447 1677477808420.491))
(indexed-future-event 235 '#s(gc-info minor 232299816 371021576 0 223870760 371021576 1892 1892 1677477808421.265 1677477808421.313))
(indexed-future-event 236 '#s(gc-info minor 232298392 371021576 0 223872312 371021576 1895 1895 1677477808422.074 1677477808422.122))
(indexed-future-event 237 '#s(gc-info minor 232273848 371021576 0 223873976 371021576 1898 1898 1677477808422.893 1677477808422.943))
(indexed-future-event 238 '#s(gc-info minor 232268792 371021576 0 223875528 371021576 1901 1901 1677477808423.704 1677477808423.75))
(indexed-future-event 239 '#s(gc-info minor 232269960 371021576 0 223873192 371021576 1904 1904 1677477808424.516 1677477808424.56))
(indexed-future-event 240 '#s(gc-info minor 232274376 371021576 0 223874840 371021576 1907 1907 1677477808425.323 1677477808425.367))
(indexed-future-event 241 '#s(gc-info minor 232285272 371021576 0 223876408 371021576 1910 1910 1677477808426.137 1677477808426.179))
(indexed-future-event 242 '#s(gc-info minor 232287560 371021576 0 223877992 371021576 1913 1913 1677477808426.946 1677477808426.993))
(indexed-future-event 243 '#s(gc-info minor 232290168 371021576 0 223875608 371021576 1916 1916 1677477808427.76 1677477808427.813))
(indexed-future-event 244 '#s(gc-info minor 232293288 371021576 0 223877240 371021576 1919 1919 1677477808428.582 1677477808428.629))
(indexed-future-event 245 '#s(gc-info minor 232272104 371021576 0 223878824 371021576 1922 1922 1677477808429.391 1677477808429.434))
(indexed-future-event 246 '#s(gc-info minor 232289640 371021576 0 223880392 371021576 1925 1925 1677477808430.201 1677477808430.245))
(indexed-future-event 247 '#s(gc-info minor 232281528 371021576 0 223878088 371021576 1928 1928 1677477808431.016 1677477808431.065))
(indexed-future-event 248 '#s(gc-info minor 232290056 371021576 0 223879624 371021576 1931 1931 1677477808431.839 1677477808431.883))
(indexed-future-event 249 '#s(gc-info minor 232291192 371021576 0 223881192 371021576 1934 1934 1677477808432.647 1677477808432.691))
(indexed-future-event 250 '#s(gc-info minor 232282120 371021576 0 223882856 371021576 1937 1937 1677477808433.476 1677477808433.519))
(indexed-future-event 251 '#s(gc-info minor 232278328 371021576 0 223880472 371021576 1940 1940 1677477808434.287 1677477808434.328))
(indexed-future-event 252 '#s(gc-info minor 232291288 371021576 0 223881992 371021576 1943 1943 1677477808435.093 1677477808435.139))
(indexed-future-event 253 '#s(gc-info minor 232292808 371021576 0 223883592 371021576 1946 1946 1677477808435.904 1677477808435.943))
(indexed-future-event 254 '#s(gc-info minor 232285544 371021576 0 223885272 371021576 1949 1949 1677477808436.715 1677477808436.757))
(indexed-future-event 255 '#s(gc-info minor 232296088 371021576 0 223882872 371021576 1952 1952 1677477808437.519 1677477808437.564))
(indexed-future-event 256 '#s(gc-info minor 232293704 371021576 0 223884424 371021576 1955 1955 1677477808438.329 1677477808438.377))
(indexed-future-event 257 '#s(gc-info minor 232286328 371021576 0 223886120 371021576 1958 1958 1677477808439.141 1677477808439.184))
(indexed-future-event 258 '#s(gc-info minor 232313688 371021576 0 223887688 371021576 1961 1961 1677477808439.961 1677477808440.012))
(indexed-future-event 259 '#s(gc-info minor 232283288 371021576 0 223885272 371021576 1964 1964 1677477808440.77 1677477808440.812))
(indexed-future-event 260 '#s(gc-info minor 232280728 371021576 0 223886872 371021576 1967 1967 1677477808441.588 1677477808441.632))
(indexed-future-event 261 '#s(gc-info minor 232287480 371021576 0 223888520 371021576 1970 1970 1677477808442.397 1677477808442.432))
(indexed-future-event 262 '#s(gc-info minor 232335176 371021576 0 223890120 371021576 1973 1973 1677477808443.225 1677477808443.267))
(indexed-future-event 263 '#s(gc-info minor 232301304 371021576 0 223876088 371021576 1976 1976 1677477808444.046 1677477808444.104))
(indexed-future-event 264 '#s(gc-info minor 232278760 371021576 0 223877656 371021576 1979 1979 1677477808444.862 1677477808444.907))
(indexed-future-event 265 '#s(gc-info minor 232304872 371021576 0 223879240 371021576 1982 1982 1677477808445.679 1677477808445.722))
(indexed-future-event 266 '#s(gc-info minor 232290008 371021576 0 223880808 371021576 1984 1985 1677477808446.492 1677477808446.53))
(indexed-future-event 267 '#s(gc-info minor 232292600 371021576 0 223878456 371021576 1987 1987 1677477808447.31 1677477808447.35))
(indexed-future-event 268 '#s(gc-info minor 232278712 371021576 0 223880104 371021576 1990 1990 1677477808448.119 1677477808448.151))
(indexed-future-event 269 '#s(gc-info minor 232292840 371021576 0 223881672 371021576 1993 1993 1677477808448.901 1677477808448.943))
(indexed-future-event 270 '#s(gc-info minor 232292120 371021576 0 223883256 371021576 1996 1996 1677477808449.706 1677477808449.762))
(indexed-future-event 271 '#s(gc-info minor 232300776 371021576 0 223880952 371021576 1999 1999 1677477808450.536 1677477808450.582))
(indexed-future-event 272 '#s(gc-info minor 232275752 371021576 0 223882488 371021576 2002 2002 1677477808451.346 1677477808451.39))
(indexed-future-event 273 '#s(gc-info minor 232277320 371021576 0 223884056 371021576 2005 2005 1677477808452.156 1677477808452.206))
(indexed-future-event 274 '#s(gc-info minor 232301256 371021576 0 223885736 371021576 2008 2008 1677477808452.992 1677477808453.038))
(indexed-future-event 275 '#s(gc-info minor 232312936 371021576 0 223883320 371021576 2011 2011 1677477808453.807 1677477808453.852))
(indexed-future-event 276 '#s(gc-info minor 232278136 371021576 0 223884840 371021576 2014 2014 1677477808454.62 1677477808454.662))
(indexed-future-event 277 '#s(gc-info minor 232296056 371021576 0 223886648 371021576 2017 2017 1677477808455.425 1677477808455.469))
(indexed-future-event 278 '#s(gc-info minor 232303800 371021576 0 223888312 371021576 2020 2020 1677477808456.236 1677477808456.278))
(indexed-future-event 279 '#s(gc-info minor 232315912 371021576 0 223885768 371021576 2023 2023 1677477808457.054 1677477808457.115))
(indexed-future-event 280 '#s(gc-info minor 232295816 371021576 0 223887272 371021576 2026 2026 1677477808457.895 1677477808457.94))
(indexed-future-event 281 '#s(gc-info minor 232288776 371021576 0 223888936 371021576 2029 2029 1677477808458.712 1677477808458.753))
(indexed-future-event 282 '#s(gc-info minor 232299768 371021576 0 223890504 371021576 2032 2032 1677477808459.543 1677477808459.593))
(indexed-future-event 283 '#s(gc-info minor 232301752 371021576 0 223888136 371021576 2035 2035 1677477808460.387 1677477808460.442))
(indexed-future-event 284 '#s(gc-info minor 232303368 371021576 0 223889768 371021576 2038 2038 1677477808461.265 1677477808461.308))
(indexed-future-event 285 '#s(gc-info minor 232301352 371021576 0 223891320 371021576 2041 2041 1677477808462.081 1677477808462.122))
(indexed-future-event 286 '#s(gc-info minor 232302136 371021576 0 223892920 371021576 2044 2044 1677477808462.89 1677477808462.932))
(indexed-future-event 287 '#s(gc-info minor 232294104 371021576 0 223890632 371021576 2047 2047 1677477808463.708 1677477808463.748))
(indexed-future-event 288 '#s(gc-info minor 232286984 371021576 0 223892168 371021576 2050 2050 1677477808464.488 1677477808464.535))
(indexed-future-event 289 '#s(gc-info minor 232303384 371021576 0 223893752 371021576 2053 2053 1677477808465.301 1677477808465.34))
(indexed-future-event 290 '#s(gc-info minor 232290104 371021576 0 223895320 371021576 2056 2056 1677477808466.08 1677477808466.12))
(indexed-future-event 291 '#s(gc-info minor 232296440 371021576 0 223893016 371021576 2059 2059 1677477808466.893 1677477808466.936))
(indexed-future-event 292 '#s(gc-info minor 232287400 371021576 0 223894552 371021576 2062 2062 1677477808467.711 1677477808467.752))
(indexed-future-event 293 '#s(gc-info minor 232288648 371021576 0 223896136 371021576 2065 2065 1677477808468.519 1677477808468.564))
(indexed-future-event 294 '#s(gc-info minor 232296904 371021576 0 223897800 371021576 2068 2068 1677477808469.343 1677477808469.384))
(indexed-future-event 295 '#s(gc-info minor 232309000 371021576 0 223895432 371021576 2071 2071 1677477808470.15 1677477808470.202))
(indexed-future-event 296 '#s(gc-info minor 232306200 371021576 0 223896968 371021576 2074 2074 1677477808470.974 1677477808471.022))
(indexed-future-event 297 '#s(gc-info minor 232293112 371021576 0 223898584 371021576 2077 2077 1677477808471.785 1677477808471.826))
(indexed-future-event 298 '#s(gc-info minor 232314488 371021576 0 223900328 371021576 2080 2080 1677477808472.596 1677477808472.637))
(indexed-future-event 299 '#s(gc-info minor 232311144 371021576 0 223897816 371021576 2083 2083 1677477808473.413 1677477808473.457))
(indexed-future-event 300 '#s(gc-info minor 232292632 371021576 0 223899336 371021576 2086 2086 1677477808474.226 1677477808474.266))
(indexed-future-event 301 '#s(gc-info minor 232316472 371021576 0 223901000 371021576 2089 2089 1677477808475.041 1677477808475.089))
(indexed-future-event 302 '#s(gc-info minor 232312632 371021576 0 223902584 371021576 2092 2092 1677477808475.848 1677477808475.899))
(indexed-future-event 303 '#s(gc-info minor 232298168 371021576 0 223900216 371021576 2095 2095 1677477808476.661 1677477808476.708))
(indexed-future-event 304 '#s(gc-info minor 232301752 371021576 0 223901832 371021576 2098 2098 1677477808477.486 1677477808477.525))
(indexed-future-event 305 '#s(gc-info minor 232297800 371021576 0 223903432 371021576 2101 2101 1677477808478.271 1677477808478.309))
(indexed-future-event 306 '#s(gc-info minor 232313848 371021576 0 223905000 371021576 2104 2104 1677477808479.079 1677477808479.12))
(indexed-future-event 307 '#s(gc-info minor 232315448 371021576 0 223902616 371021576 2107 2107 1677477808479.881 1677477808479.926))
(indexed-future-event 308 '#s(gc-info minor 232304184 371021576 0 223904248 371021576 2110 2110 1677477808480.697 1677477808480.74))
(indexed-future-event 309 '#s(gc-info minor 232299064 371021576 0 223905800 371021576 2113 2113 1677477808481.511 1677477808481.554))
(indexed-future-event 310 '#s(gc-info minor 232333000 371021576 0 223907144 371021576 2116 2116 1677477808482.321 1677477808482.36))
(indexed-future-event 311 '#s(gc-info minor 232308712 371021576 0 223905160 371021576 2119 2119 1677477808483.129 1677477808483.176))
(indexed-future-event 312 '#s(gc-info minor 232315976 371021576 0 223906664 371021576 2122 2122 1677477808483.951 1677477808483.998))
(indexed-future-event 313 '#s(gc-info minor 232350648 371021576 0 223908248 371021576 2125 2125 1677477808484.776 1677477808484.816))
(indexed-future-event 314 '#s(gc-info minor 232325384 371021576 0 223909944 371021576 2128 2128 1677477808485.601 1677477808485.653))
(indexed-future-event 315 '#s(gc-info minor 232304744 371021576 0 223907512 371021576 2130 2131 1677477808486.425 1677477808486.471))
(indexed-future-event 316 '#s(gc-info minor 232302344 371021576 0 223909048 371021576 2133 2134 1677477808487.239 1677477808487.28))
(indexed-future-event 317 '#s(gc-info minor 232303928 371021576 0 223910840 371021576 2136 2136 1677477808488.048 1677477808488.09))
(indexed-future-event 318 '#s(gc-info minor 232312408 371021576 0 223912488 371021576 2139 2139 1677477808488.862 1677477808488.906))
(indexed-future-event 319 '#s(gc-info minor 232323304 371021576 0 223909944 371021576 2142 2142 1677477808489.686 1677477808489.73))
(indexed-future-event 320 '#s(gc-info minor 232321512 371021576 0 223911480 371021576 2145 2145 1677477808490.503 1677477808490.546))
(indexed-future-event 321 '#s(gc-info minor 232313000 371021576 0 223913144 371021576 2148 2148 1677477808491.321 1677477808491.377))
(indexed-future-event 322 '#s(gc-info minor 232325128 371021576 0 223914728 371021576 2151 2151 1677477808492.132 1677477808492.173))
(indexed-future-event 323 '#s(gc-info minor 232340392 371021576 0 223912312 371021576 2154 2154 1677477808492.958 1677477808493.003))
(indexed-future-event 324 '#s(gc-info minor 232329816 371021576 0 223913944 371021576 2157 2157 1677477808493.779 1677477808493.823))
(indexed-future-event 325 '#s(gc-info minor 232309544 371021576 0 223915512 371021576 2160 2160 1677477808494.584 1677477808494.636))
(indexed-future-event 326 '#s(gc-info minor 232310376 371021576 0 223917096 371021576 2163 2163 1677477808495.396 1677477808495.431))
(indexed-future-event 327 '#s(gc-info minor 232313448 371021576 0 223914776 371021576 2166 2166 1677477808496.171 1677477808496.235))
(indexed-future-event 328 '#s(gc-info minor 232317464 371021576 0 223916328 371021576 2169 2169 1677477808496.991 1677477808497.04))
(indexed-future-event 329 '#s(gc-info minor 232311144 371021576 0 223917928 371021576 2172 2172 1677477808497.802 1677477808497.854))
(indexed-future-event 330 '#s(gc-info minor 232332184 371021576 0 223919496 371021576 2175 2175 1677477808498.636 1677477808498.679))
(indexed-future-event 331 '#s(gc-info minor 232314920 371021576 0 223917160 371021576 2178 2178 1677477808499.452 1677477808499.49))
(indexed-future-event 332 '#s(gc-info minor 232317368 371021576 0 223918792 371021576 2181 2181 1677477808500.256 1677477808500.298))
(indexed-future-event 333 '#s(gc-info minor 232345608 371021576 0 223920344 371021576 2184 2184 1677477808501.101 1677477808501.146))
(indexed-future-event 334 '#s(gc-info minor 232331160 371021576 0 223921944 371021576 2187 2187 1677477808501.913 1677477808501.958))
(indexed-future-event 335 '#s(gc-info minor 232322744 371021576 0 223919624 371021576 2190 2190 1677477808502.722 1677477808502.771))
(indexed-future-event 336 '#s(gc-info minor 232314824 371021576 0 223921160 371021576 2193 2193 1677477808503.534 1677477808503.576))
(indexed-future-event 337 '#s(gc-info minor 232331992 371021576 0 223922744 371021576 2196 2196 1677477808504.343 1677477808504.389))
(indexed-future-event 338 '#s(gc-info minor 232341032 371021576 0 223924440 371021576 2199 2199 1677477808505.167 1677477808505.222))
(indexed-future-event 339 '#s(gc-info minor 232320008 371021576 0 223922008 371021576 2202 2202 1677477808505.978 1677477808506.029))
(indexed-future-event 340 '#s(gc-info minor 232317608 371021576 0 223923544 371021576 2205 2205 1677477808506.788 1677477808506.831))
(indexed-future-event 341 '#s(gc-info minor 232350408 371021576 0 223924872 371021576 2208 2208 1677477808507.603 1677477808507.649))
(indexed-future-event 342 '#s(gc-info minor 232342792 371021576 0 223926792 371021576 2211 2211 1677477808508.424 1677477808508.474))
(indexed-future-event 343 '#s(gc-info minor 232353992 371021576 0 223924424 371021576 2214 2214 1677477808509.238 1677477808509.288))
(indexed-future-event 344 '#s(gc-info minor 232351624 371021576 0 223925960 371021576 2217 2217 1677477808510.058 1677477808510.107))
(indexed-future-event 345 '#s(gc-info minor 232328680 371021576 0 223927640 371021576 2220 2220 1677477808510.883 1677477808510.925))
(indexed-future-event 346 '#s(gc-info minor 232338456 371021576 0 223929208 371021576 2223 2223 1677477808511.7 1677477808511.741))
(indexed-future-event 347 '#s(gc-info minor 232356040 371021576 0 223926568 371021576 2226 2226 1677477808512.527 1677477808512.574))
(indexed-future-event 348 '#s(gc-info minor 232328072 371021576 0 223928488 371021576 2229 2229 1677477808513.348 1677477808513.393))
(indexed-future-event 349 '#s(gc-info minor 232339288 371021576 0 223930056 371021576 2232 2232 1677477808514.162 1677477808514.209))
(indexed-future-event 350 '#s(gc-info minor 232341272 371021576 0 223931624 371021576 2235 2235 1677477808514.979 1677477808515.026))
(indexed-future-event 351 '#s(gc-info minor 232342488 371021576 0 223929224 371021576 2238 2238 1677477808515.791 1677477808515.84))
(indexed-future-event 352 '#s(gc-info minor 232330760 371021576 0 223930856 371021576 2241 2241 1677477808516.605 1677477808516.651))
(indexed-future-event 353 '#s(gc-info minor 232341672 371021576 0 223932408 371021576 2244 2244 1677477808517.414 1677477808517.464))
(indexed-future-event 354 '#s(gc-info minor 232343224 371021576 0 223934008 371021576 2247 2247 1677477808518.228 1677477808518.27))
(indexed-future-event 355 '#s(gc-info minor 232335176 371021576 0 223931720 371021576 2250 2250 1677477808519.044 1677477808519.092))
(indexed-future-event 356 '#s(gc-info minor 232326152 371021576 0 223933256 371021576 2253 2253 1677477808519.854 1677477808519.898))
(indexed-future-event 357 '#s(gc-info minor 232360088 371021576 0 223934840 371021576 2256 2256 1677477808520.666 1677477808520.711))
(indexed-future-event 358 '#s(gc-info minor 232352744 371021576 0 223936536 371021576 2259 2259 1677477808521.489 1677477808521.533))
(indexed-future-event 359 '#s(gc-info minor 232331336 371021576 0 223934136 371021576 2261 2262 1677477808522.301 1677477808522.355))
(indexed-future-event 360 '#s(gc-info minor 232329736 371021576 0 223935640 371021576 2264 2265 1677477808523.115 1677477808523.156))
(indexed-future-event 361 '#s(gc-info minor 232364424 371021576 0 223937224 371021576 2268 2268 1677477808523.977 1677477808524.033))
(indexed-future-event 362 '#s(gc-info minor 232372680 371021576 0 223938888 371021576 2271 2271 1677477808524.799 1677477808524.852))
(indexed-future-event 363 '#s(gc-info minor 232334856 371021576 0 223936488 371021576 2273 2274 1677477808525.601 1677477808525.649))
(indexed-future-event 364 '#s(gc-info minor 232347240 371021576 0 223938056 371021576 2276 2277 1677477808526.414 1677477808526.455))
(indexed-future-event 365 '#s(gc-info minor 232372776 371021576 0 223939480 371021576 2279 2279 1677477808527.261 1677477808527.301))
(indexed-future-event 366 '#s(gc-info minor 232350680 371021576 0 223941304 371021576 2282 2282 1677477808528.065 1677477808528.117))
(indexed-future-event 367 '#s(gc-info minor 232352136 371021576 0 223938920 371021576 2285 2285 1677477808528.883 1677477808528.932))
(indexed-future-event 368 '#s(gc-info minor 232350120 371021576 0 223940456 371021576 2288 2288 1677477808529.693 1677477808529.737))
(indexed-future-event 369 '#s(gc-info minor 232357960 371021576 0 223942120 371021576 2291 2291 1677477808530.514 1677477808530.569))
(indexed-future-event 370 '#s(gc-info minor 232352952 371021576 0 223943704 371021576 2294 2294 1677477808531.34 1677477808531.391))
(indexed-future-event 371 '#s(gc-info minor 232338952 371021576 0 223941336 371021576 2297 2297 1677477808532.148 1677477808532.197))
(indexed-future-event 372 '#s(gc-info minor 232342488 371021576 0 223942968 371021576 2300 2300 1677477808532.983 1677477808533.044))
(indexed-future-event 373 '#s(gc-info minor 232338936 371021576 0 223944520 371021576 2303 2303 1677477808533.792 1677477808533.833))
(indexed-future-event 374 '#s(gc-info minor 232355320 371021576 0 223946120 371021576 2306 2306 1677477808534.602 1677477808534.642))
(indexed-future-event 375 '#s(gc-info minor 232346072 371021576 0 223944216 371021576 2309 2309 1677477808535.413 1677477808535.455))
(indexed-future-event 376 '#s(gc-info minor 232356712 371021576 0 223945816 371021576 2312 2312 1677477808536.222 1677477808536.26))
(indexed-future-event 377 '#s(gc-info minor 232356632 371021576 0 223947384 371021576 2315 2315 1677477808537.03 1677477808537.069))
(indexed-future-event 378 '#s(gc-info minor 232358200 371021576 0 223948936 371021576 2318 2318 1677477808537.846 1677477808537.887))
(indexed-future-event 379 '#s(gc-info minor 232367992 371021576 0 223946216 371021576 2321 2321 1677477808538.687 1677477808538.733))
(indexed-future-event 380 '#s(gc-info minor 232357464 371021576 0 223947512 371021576 2324 2324 1677477808539.501 1677477808539.536))
(indexed-future-event 381 '#s(gc-info minor 232341944 371021576 0 223949336 371021576 2327 2327 1677477808540.303 1677477808540.341))
(indexed-future-event 382 '#s(gc-info minor 232352024 371021576 0 223950984 371021576 2330 2330 1677477808541.09 1677477808541.129))
(indexed-future-event 383 '#s(gc-info minor 232394184 371021576 0 223948648 371021576 2333 2333 1677477808541.912 1677477808541.95))
(indexed-future-event 384 '#s(gc-info minor 232343064 371021576 0 223950184 371021576 2336 2336 1677477808542.721 1677477808542.763))
(indexed-future-event 385 '#s(gc-info minor 232361016 371021576 0 223951752 371021576 2339 2339 1677477808543.534 1677477808543.57))
(indexed-future-event 386 '#s(gc-info minor 232354856 371021576 0 223953416 371021576 2342 2342 1677477808544.311 1677477808544.348))
(indexed-future-event 387 '#s(gc-info minor 232349768 371021576 0 223951000 371021576 2345 2345 1677477808545.087 1677477808545.132))
(indexed-future-event 388 '#s(gc-info minor 232362200 371021576 0 223952568 371021576 2348 2348 1677477808545.902 1677477808545.945))
(indexed-future-event 389 '#s(gc-info minor 232370136 371021576 0 223954248 371021576 2351 2351 1677477808546.717 1677477808546.761))
(indexed-future-event 390 '#s(gc-info minor 232364680 371021576 0 223955560 371021576 2354 2354 1677477808547.535 1677477808547.571))
(indexed-future-event 391 '#s(gc-info minor 232396472 371021576 0 223953496 371021576 2357 2357 1677477808548.397 1677477808548.451))
(indexed-future-event 392 '#s(gc-info minor 232355400 371021576 0 223955096 371021576 2360 2360 1677477808549.229 1677477808549.274))
(indexed-future-event 393 '#s(gc-info minor 232349896 371021576 0 223956664 371021576 2363 2363 1677477808550.043 1677477808550.097))
(indexed-future-event 394 '#s(gc-info minor 232351112 371021576 0 223958232 371021576 2366 2366 1677477808550.868 1677477808550.911))
(indexed-future-event 395 '#s(gc-info minor 232369096 371021576 0 223955832 371021576 2369 2369 1677477808551.679 1677477808551.727))
(indexed-future-event 396 '#s(gc-info minor 232357368 371021576 0 223957464 371021576 2372 2372 1677477808552.491 1677477808552.535))
(indexed-future-event 397 '#s(gc-info minor 232367896 371021576 0 223959016 371021576 2375 2375 1677477808553.307 1677477808553.348))
(indexed-future-event 398 '#s(gc-info minor 232353448 371021576 0 223960616 371021576 2378 2378 1677477808554.123 1677477808554.165))
(indexed-future-event 399 '#s(gc-info minor 232361800 371021576 0 223958328 371021576 2381 2381 1677477808554.935 1677477808554.992))
(indexed-future-event 400 '#s(gc-info minor 232370680 371021576 0 223959864 371021576 2384 2384 1677477808555.743 1677477808555.791))
(indexed-future-event 401 '#s(gc-info minor 232354696 371021576 0 223961448 371021576 2387 2387 1677477808556.557 1677477808556.599))
(indexed-future-event 402 '#s(gc-info minor 232395352 371021576 0 223963144 371021576 2390 2390 1677477808557.386 1677477808557.424))
(indexed-future-event 403 '#s(gc-info minor 232390696 371021576 0 223960712 371021576 2393 2393 1677477808558.194 1677477808558.247))
(indexed-future-event 404 '#s(gc-info minor 232372696 371021576 0 223962248 371021576 2396 2396 1677477808558.997 1677477808559.037))
(indexed-future-event 405 '#s(gc-info minor 232373112 371021576 0 223963832 371021576 2398 2399 1677477808559.803 1677477808559.847))
(indexed-future-event 406 '#s(gc-info minor 232366136 371021576 0 223965496 371021576 2401 2401 1677477808560.62 1677477808560.663))
(indexed-future-event 407 '#s(gc-info minor 232359928 371021576 0 223963128 371021576 2404 2404 1677477808561.43 1677477808561.478))
(indexed-future-event 408 '#s(gc-info minor 232358328 371021576 0 223964664 371021576 2407 2407 1677477808562.245 1677477808562.283))
(indexed-future-event 409 '#s(gc-info minor 232396696 371021576 0 223966088 371021576 2410 2410 1677477808563.093 1677477808563.133))
(indexed-future-event 410 '#s(gc-info minor 232378440 371021576 0 223967912 371021576 2413 2413 1677477808563.908 1677477808563.949))
(indexed-future-event 411 '#s(gc-info minor 232378744 371021576 0 223965528 371021576 2416 2416 1677477808564.721 1677477808564.765))
(indexed-future-event 412 '#s(gc-info minor 232365880 371021576 0 223967528 371021576 2419 2419 1677477808565.535 1677477808565.576))
(indexed-future-event 413 '#s(gc-info minor 232363560 371021576 0 223969112 371021576 2422 2422 1677477808566.346 1677477808566.385))
(indexed-future-event 414 '#s(gc-info minor 232379208 371021576 0 223970696 371021576 2425 2425 1677477808567.157 1677477808567.202))
(indexed-future-event 415 '#s(gc-info minor 232381128 371021576 0 223967912 371021576 2428 2428 1677477808567.968 1677477808568.01))
(indexed-future-event 416 '#s(gc-info minor 232369064 371021576 0 223969528 371021576 2431 2431 1677477808568.784 1677477808568.822))
(indexed-future-event 417 '#s(gc-info minor 232399032 371021576 0 223971128 371021576 2434 2434 1677477808569.568 1677477808569.612))
(indexed-future-event 418 '#s(gc-info minor 232382696 371021576 0 223972696 371021576 2437 2437 1677477808570.38 1677477808570.421))
(indexed-future-event 419 '#s(gc-info minor 232373832 371021576 0 223970408 371021576 2440 2440 1677477808571.193 1677477808571.238))
(indexed-future-event 420 '#s(gc-info minor 232381624 371021576 0 223971960 371021576 2443 2443 1677477808572.005 1677477808572.047))
(indexed-future-event 421 '#s(gc-info minor 232398776 371021576 0 223973528 371021576 2446 2446 1677477808572.813 1677477808572.852))
(indexed-future-event 422 '#s(gc-info minor 232388168 371021576 0 223975208 371021576 2449 2449 1677477808573.645 1677477808573.697))
(indexed-future-event 423 '#s(gc-info minor 232389384 371021576 0 223972856 371021576 2452 2452 1677477808574.463 1677477808574.51))
(indexed-future-event 424 '#s(gc-info minor 232369208 371021576 0 223974360 371021576 2455 2455 1677477808575.247 1677477808575.286))
(indexed-future-event 425 '#s(gc-info minor 232382504 371021576 0 223975944 371021576 2458 2458 1677477808576.125 1677477808576.172))
(indexed-future-event 426 '#s(gc-info minor 232377848 371021576 0 223977640 371021576 2461 2461 1677477808576.941 1677477808576.995))
(indexed-future-event 427 '#s(gc-info minor 232389592 371021576 0 223975208 371021576 2464 2464 1677477808577.779 1677477808577.82))
(indexed-future-event 428 '#s(gc-info minor 232387192 371021576 0 223976744 371021576 2467 2467 1677477808578.582 1677477808578.631))
(indexed-future-event 429 '#s(gc-info minor 232394712 371021576 0 223978424 371021576 2470 2470 1677477808579.409 1677477808579.451))
(indexed-future-event 430 '#s(gc-info minor 232389240 371021576 0 223979992 371021576 2473 2473 1677477808580.219 1677477808580.263))
(indexed-future-event 431 '#s(gc-info minor 232390808 371021576 0 223977592 371021576 2476 2476 1677477808581.029 1677477808581.079))
(indexed-future-event 432 '#s(gc-info minor 232388408 371021576 0 223979160 371021576 2479 2479 1677477808581.849 1677477808581.892))
(indexed-future-event 433 '#s(gc-info minor 232396712 371021576 0 223980824 371021576 2482 2482 1677477808582.663 1677477808582.719))
(indexed-future-event 434 '#s(gc-info minor 232392424 371021576 0 223982408 371021576 2485 2485 1677477808583.48 1677477808583.524))
(indexed-future-event 435 '#s(gc-info minor 232392840 371021576 0 223980008 371021576 2488 2488 1677477808584.289 1677477808584.34))
(indexed-future-event 436 '#s(gc-info minor 232381496 371021576 0 223981640 371021576 2491 2491 1677477808585.114 1677477808585.156))
(indexed-future-event 437 '#s(gc-info minor 232392472 371021576 0 223983208 371021576 2494 2494 1677477808585.926 1677477808585.98))
(indexed-future-event 438 '#s(gc-info minor 232395224 371021576 0 223984792 371021576 2497 2497 1677477808586.738 1677477808586.782))
(indexed-future-event 439 '#s(gc-info minor 232418328 371021576 0 223982536 371021576 2500 2500 1677477808587.557 1677477808587.607))
(indexed-future-event 440 '#s(gc-info minor 232378120 371021576 0 223984024 371021576 2503 2503 1677477808588.365 1677477808588.408))
(indexed-future-event 441 '#s(gc-info minor 232379992 371021576 0 223985624 371021576 2506 2506 1677477808589.182 1677477808589.233))
(indexed-future-event 442 '#s(gc-info minor 232413192 371021576 0 223987192 371021576 2509 2509 1677477808590.011 1677477808590.057))
(indexed-future-event 443 '#s(gc-info minor 232388712 371021576 0 223984904 371021576 2512 2512 1677477808590.826 1677477808590.873))
(indexed-future-event 444 '#s(gc-info minor 232395736 371021576 0 223986456 371021576 2515 2515 1677477808591.641 1677477808591.682))
(indexed-future-event 445 '#s(gc-info minor 232380888 371021576 0 223988024 371021576 2518 2518 1677477808592.445 1677477808592.491))
(indexed-future-event 446 '#s(gc-info minor 232389144 371021576 0 223989688 371021576 2521 2521 1677477808593.265 1677477808593.309))
(indexed-future-event 447 '#s(gc-info minor 232400904 371021576 0 223987304 371021576 2524 2524 1677477808594.078 1677477808594.127))
(indexed-future-event 448 '#s(gc-info minor 232414920 371021576 0 223988856 371021576 2527 2527 1677477808594.889 1677477808594.936))
(indexed-future-event 449 '#s(gc-info minor 232403400 371021576 0 223990824 371021576 2530 2530 1677477808595.729 1677477808595.773))
(indexed-future-event 450 '#s(gc-info minor 232389064 371021576 0 223992504 371021576 2533 2533 1677477808596.544 1677477808596.588))
(indexed-future-event 451 '#s(gc-info minor 232387304 371021576 0 223989704 371021576 2536 2536 1677477808597.351 1677477808597.398))
(indexed-future-event 452 '#s(gc-info minor 232400904 371021576 0 223991240 371021576 2538 2539 1677477808598.162 1677477808598.218))
(indexed-future-event 453 '#s(gc-info minor 232393592 371021576 0 223992920 371021576 2541 2542 1677477808598.976 1677477808599.025))
(indexed-future-event 454 '#s(gc-info minor 232404120 371021576 0 223994488 371021576 2544 2545 1677477808599.79 1677477808599.834))
(indexed-future-event 455 '#s(gc-info minor 232405304 371021576 0 223992152 371021576 2547 2547 1677477808600.601 1677477808600.665))
(indexed-future-event 456 '#s(gc-info minor 232394152 371021576 0 223994104 371021576 2550 2550 1677477808601.429 1677477808601.472))
(indexed-future-event 457 '#s(gc-info minor 232390008 371021576 0 223995704 371021576 2553 2553 1677477808602.235 1677477808602.286))
(indexed-future-event 458 '#s(gc-info minor 232407304 371021576 0 223997288 371021576 2556 2556 1677477808603.042 1677477808603.091))
(indexed-future-event 459 '#s(gc-info minor 232408872 371021576 0 223994504 371021576 2559 2559 1677477808603.86 1677477808603.913))
(indexed-future-event 460 '#s(gc-info minor 232413160 371021576 0 223996136 371021576 2562 2562 1677477808604.672 1677477808604.735))
(indexed-future-event 461 '#s(gc-info minor 232391736 371021576 0 223997704 371021576 2565 2565 1677477808605.498 1677477808605.541))
(indexed-future-event 462 '#s(gc-info minor 232424952 371021576 0 223999288 371021576 2568 2568 1677477808606.312 1677477808606.349))
(indexed-future-event 463 '#s(gc-info minor 232400824 371021576 0 223997000 371021576 2571 2571 1677477808607.124 1677477808607.168))
(indexed-future-event 464 '#s(gc-info minor 232391816 371021576 0 223998520 371021576 2574 2574 1677477808607.937 1677477808607.975))
(indexed-future-event 465 '#s(gc-info minor 232409336 371021576 0 224000120 371021576 2577 2577 1677477808608.746 1677477808608.784))
(indexed-future-event 466 '#s(gc-info minor 232394536 371021576 0 224001688 371021576 2580 2580 1677477808609.553 1677477808609.588))
(indexed-future-event 467 '#s(gc-info minor 232420360 371021576 0 223999400 371021576 2583 2583 1677477808610.36 1677477808610.405))
(indexed-future-event 468 '#s(gc-info minor 232394232 371021576 0 224000952 371021576 2586 2586 1677477808611.176 1677477808611.215))
(indexed-future-event 469 '#s(gc-info minor 232395768 371021576 0 224002520 371021576 2589 2589 1677477808611.984 1677477808612.022))
(indexed-future-event 470 '#s(gc-info minor 232405176 371021576 0 224004184 371021576 2592 2592 1677477808612.807 1677477808612.85))
(indexed-future-event 471 '#s(gc-info minor 232431016 371021576 0 224001832 371021576 2595 2595 1677477808613.62 1677477808613.663))
(indexed-future-event 472 '#s(gc-info minor 232412696 371021576 0 224003352 371021576 2598 2598 1677477808614.427 1677477808614.468))
(indexed-future-event 473 '#s(gc-info minor 232415912 371021576 0 224004712 371021576 2601 2601 1677477808615.239 1677477808615.278))
(indexed-future-event 474 '#s(gc-info minor 232404552 371021576 0 224006664 371021576 2604 2604 1677477808616.053 1677477808616.093))
(indexed-future-event 475 '#s(gc-info minor 232417464 371021576 0 224004200 371021576 2607 2607 1677477808616.864 1677477808616.907))
(indexed-future-event 476 '#s(gc-info minor 232399032 371021576 0 224005736 371021576 2610 2610 1677477808617.673 1677477808617.712))
(indexed-future-event 477 '#s(gc-info minor 232406936 371021576 0 224007416 371021576 2613 2613 1677477808618.491 1677477808618.529))
(indexed-future-event 478 '#s(gc-info minor 232402232 371021576 0 224008984 371021576 2616 2616 1677477808619.3 1677477808619.34))
(indexed-future-event 479 '#s(gc-info minor 232419800 371021576 0 224006584 371021576 2619 2619 1677477808620.106 1677477808620.15))
(indexed-future-event 480 '#s(gc-info minor 232408872 371021576 0 224008216 371021576 2622 2622 1677477808620.93 1677477808620.973))
(indexed-future-event 481 '#s(gc-info minor 232403016 371021576 0 224009800 371021576 2625 2625 1677477808621.739 1677477808621.78))
(indexed-future-event 482 '#s(gc-info minor 232421384 371021576 0 224011368 371021576 2628 2628 1677477808622.556 1677477808622.6))
(indexed-future-event 483 '#s(gc-info minor 232421816 371021576 0 224009016 371021576 2631 2631 1677477808623.37 1677477808623.414))
(indexed-future-event 484 '#s(gc-info minor 232410568 371021576 0 224010648 371021576 2634 2634 1677477808624.181 1677477808624.217))
(indexed-future-event 485 '#s(gc-info minor 232405096 371021576 0 224012216 371021576 2637 2637 1677477808624.986 1677477808625.02))
(indexed-future-event 486 '#s(gc-info minor 232408616 371021576 0 224013800 371021576 2640 2640 1677477808625.756 1677477808625.794))
(indexed-future-event 487 '#s(gc-info minor 232414952 371021576 0 224011528 371021576 2642 2643 1677477808626.569 1677477808626.611))
(indexed-future-event 488 '#s(gc-info minor 232439112 371021576 0 224013016 371021576 2645 2645 1677477808627.373 1677477808627.408))
(indexed-future-event 489 '#s(gc-info minor 232425368 371021576 0 224014616 371021576 2648 2648 1677477808628.148 1677477808628.185))
(indexed-future-event 490 '#s(gc-info minor 232410776 371021576 0 224016232 371021576 2651 2651 1677477808628.951 1677477808628.988))
(indexed-future-event 491 '#s(gc-info minor 232416136 371021576 0 224013912 371021576 2654 2654 1677477808629.757 1677477808629.797))
(indexed-future-event 492 '#s(gc-info minor 232408728 371021576 0 224015448 371021576 2657 2657 1677477808630.56 1677477808630.596))
(indexed-future-event 493 '#s(gc-info minor 232411800 371021576 0 224017000 371021576 2660 2660 1677477808631.331 1677477808631.368))
(indexed-future-event 494 '#s(gc-info minor 232419672 371021576 0 224018680 371021576 2663 2663 1677477808632.116 1677477808632.152))
(indexed-future-event 495 '#s(gc-info minor 232413544 371021576 0 224016328 371021576 2666 2666 1677477808632.918 1677477808632.958))
(indexed-future-event 496 '#s(gc-info minor 232427144 371021576 0 224017864 371021576 2669 2669 1677477808633.723 1677477808633.76))
(indexed-future-event 497 '#s(gc-info minor 232418232 371021576 0 224019912 371021576 2672 2672 1677477808634.532 1677477808634.569))
(indexed-future-event 498 '#s(gc-info minor 232431928 371021576 0 224021496 371021576 2675 2675 1677477808635.335 1677477808635.373))
(indexed-future-event 499 '#s(gc-info minor 232432312 371021576 0 224018712 371021576 2678 2678 1677477808636.141 1677477808636.181))
(indexed-future-event 500 '#s(gc-info minor 232413960 371021576 0 224020264 371021576 2681 2681 1677477808636.943 1677477808636.983))
(indexed-future-event 501 '#s(gc-info minor 232421032 371021576 0 224021928 371021576 2684 2684 1677477808637.765 1677477808637.802))
(indexed-future-event 502 '#s(gc-info minor 232460136 371021576 0 224023480 371021576 2687 2687 1677477808638.66 1677477808638.698))
(indexed-future-event 503 '#s(gc-info minor 232434296 371021576 0 224021176 371021576 2690 2690 1677477808639.462 1677477808639.506))
(indexed-future-event 504 '#s(gc-info minor 232439128 371021576 0 224022792 371021576 2693 2693 1677477808640.275 1677477808640.316))
(indexed-future-event 505 '#s(gc-info minor 232433992 371021576 0 224024360 371021576 2696 2696 1677477808641.079 1677477808641.116))
(indexed-future-event 506 '#s(gc-info minor 232436728 371021576 0 224025944 371021576 2699 2699 1677477808641.851 1677477808641.886))
(indexed-future-event 507 '#s(gc-info minor 232428616 371021576 0 224023640 371021576 2702 2702 1677477808642.632 1677477808642.672))
(indexed-future-event 508 '#s(gc-info minor 232418824 371021576 0 224025176 371021576 2705 2705 1677477808643.44 1677477808643.476))
(indexed-future-event 509 '#s(gc-info minor 232419624 371021576 0 224026744 371021576 2708 2708 1677477808644.242 1677477808644.284))
(indexed-future-event 510 '#s(gc-info minor 232421608 371021576 0 224028328 371021576 2710 2711 1677477808645.05 1677477808645.089))
(indexed-future-event 511 '#s(gc-info minor 232446280 371021576 0 224025992 371021576 2713 2714 1677477808645.866 1677477808645.909))
(indexed-future-event 512 '#s(gc-info minor 232420808 371021576 0 224027560 371021576 2716 2716 1677477808646.673 1677477808646.711))
(indexed-future-event 513 '#s(gc-info minor 232421976 371021576 0 224029128 371021576 2719 2719 1677477808647.477 1677477808647.514))
(indexed-future-event 514 '#s(gc-info minor 232432904 371021576 0 224030792 371021576 2722 2722 1677477808648.272 1677477808648.309))
(indexed-future-event 515 '#s(gc-info minor 232427496 371021576 0 224028424 371021576 2725 2725 1677477808649.052 1677477808649.096))
(indexed-future-event 516 '#s(gc-info minor 232439240 371021576 0 224029960 371021576 2728 2728 1677477808649.861 1677477808649.901))
(indexed-future-event 517 '#s(gc-info minor 232440776 371021576 0 224031512 371021576 2731 2731 1677477808650.677 1677477808650.713))
(indexed-future-event 518 '#s(gc-info minor 232432264 371021576 0 224033176 371021576 2734 2734 1677477808651.49 1677477808651.528))
(indexed-future-event 519 '#s(gc-info minor 232442504 371021576 0 224030888 371021576 2737 2737 1677477808652.325 1677477808652.386))
(indexed-future-event 520 '#s(gc-info minor 232427624 371021576 0 224032360 371021576 2740 2740 1677477808653.14 1677477808653.182))
(indexed-future-event 521 '#s(gc-info minor 232450280 371021576 0 224034008 371021576 2743 2743 1677477808653.971 1677477808654.011))
(indexed-future-event 522 '#s(gc-info minor 232428440 371021576 0 224035608 371021576 2746 2746 1677477808654.789 1677477808654.827))
(indexed-future-event 523 '#s(gc-info minor 232430408 371021576 0 224033208 371021576 2749 2749 1677477808655.594 1677477808655.64))
(indexed-future-event 524 '#s(gc-info minor 232451880 371021576 0 224034840 371021576 2752 2752 1677477808656.427 1677477808656.464))
(indexed-future-event 525 '#s(gc-info minor 232446056 371021576 0 224036424 371021576 2755 2755 1677477808657.232 1677477808657.27))
(indexed-future-event 526 '#s(gc-info minor 232430472 371021576 0 224037992 371021576 2758 2758 1677477808658.039 1677477808658.075))
(indexed-future-event 527 '#s(gc-info minor 232434344 371021576 0 224035592 371021576 2761 2761 1677477808658.813 1677477808658.854))
(indexed-future-event 528 '#s(gc-info minor 232453496 371021576 0 224037224 371021576 2764 2764 1677477808659.635 1677477808659.673))
(indexed-future-event 529 '#s(gc-info minor 232431704 371021576 0 224038808 371021576 2767 2767 1677477808660.446 1677477808660.488))
(indexed-future-event 530 '#s(gc-info minor 232449624 371021576 0 224040376 371021576 2770 2770 1677477808661.255 1677477808661.291))
(indexed-future-event 531 '#s(gc-info minor 232443064 371021576 0 224038088 371021576 2773 2773 1677477808662.043 1677477808662.079))
(indexed-future-event 532 '#s(gc-info minor 232435512 371021576 0 224105192 371021576 2776 2776 1677477808662.813 1677477808662.853))
(indexed-future-event 533 '#s(gc-info minor 232515592 371021576 0 224106760 371021576 2779 2779 1677477808663.626 1677477808663.662))
(indexed-future-event 534 '#s(gc-info minor 232502360 371021576 0 224108328 371021576 2782 2782 1677477808664.433 1677477808664.475))
(indexed-future-event 535 '#s(gc-info minor 232527064 371021576 0 224105512 371021576 2785 2785 1677477808665.284 1677477808665.338))
(indexed-future-event 536 '#s(gc-info minor 232533864 371021576 0 224107016 371021576 2788 2788 1677477808666.094 1677477808666.135))
(indexed-future-event 537 '#s(gc-info minor 232534216 371021576 0 224108568 371021576 2791 2791 1677477808666.905 1677477808666.944))
(indexed-future-event 538 '#s(gc-info minor 232509704 371021576 0 224110232 371021576 2794 2794 1677477808667.724 1677477808667.764))
(indexed-future-event 539 '#s(gc-info minor 232505096 371021576 0 224107880 371021576 2797 2797 1677477808668.53 1677477808668.576))
(indexed-future-event 540 '#s(gc-info minor 232518696 371021576 0 224109416 371021576 2800 2800 1677477808669.337 1677477808669.372))
(indexed-future-event 541 '#s(gc-info minor 232506552 371021576 0 224111000 371021576 2803 2803 1677477808670.118 1677477808670.158))
(indexed-future-event 542 '#s(gc-info minor 232511752 371021576 0 224112696 371021576 2806 2806 1677477808670.933 1677477808670.969))
(indexed-future-event 543 '#s(gc-info minor 232523496 371021576 0 224110264 371021576 2809 2809 1677477808671.733 1677477808671.77))
(indexed-future-event 544 '#s(gc-info minor 232504328 371021576 0 224111800 371021576 2811 2811 1677477808672.535 1677477808672.571))
(indexed-future-event 545 '#s(gc-info minor 232514920 371021576 0 224113480 371021576 2814 2814 1677477808673.32 1677477808673.36))
(indexed-future-event 546 '#s(gc-info minor 232507912 371021576 0 224115048 371021576 2817 2817 1677477808674.121 1677477808674.156))
(indexed-future-event 547 '#s(gc-info minor 232511400 371021576 0 224112648 371021576 2820 2820 1677477808674.898 1677477808674.935))
(indexed-future-event 548 '#s(gc-info minor 232525000 371021576 0 224114216 371021576 2823 2823 1677477808675.674 1677477808675.715))
(indexed-future-event 549 '#s(gc-info minor 232516168 371021576 0 224115896 371021576 2826 2826 1677477808676.486 1677477808676.529))
(indexed-future-event 550 '#s(gc-info minor 232526328 371021576 0 224117464 371021576 2829 2829 1677477808677.303 1677477808677.345))
(indexed-future-event 551 '#s(gc-info minor 232512296 371021576 0 224112248 371021576 2832 2832 1677477808678.121 1677477808678.174))
(indexed-future-event 552 '#s(gc-info minor 232530536 371021576 0 224113880 371021576 2835 2835 1677477808678.961 1677477808679.001))
(indexed-future-event 553 '#s(gc-info minor 232524680 371021576 0 224115448 371021576 2838 2838 1677477808679.763 1677477808679.803))
(indexed-future-event 554 '#s(gc-info minor 232524744 371021576 0 224117016 371021576 2841 2841 1677477808680.598 1677477808680.643))
(indexed-future-event 555 '#s(gc-info minor 232518216 371021576 0 224114712 371021576 2844 2844 1677477808681.421 1677477808681.46))
(indexed-future-event 556 '#s(gc-info minor 232525912 371021576 0 224116248 371021576 2847 2847 1677477808682.227 1677477808682.265))
(indexed-future-event 557 '#s(gc-info minor 232527432 371021576 0 224117800 371021576 2850 2850 1677477808683.071 1677477808683.11))
(indexed-future-event 558 '#s(gc-info minor 232512616 371021576 0 224119400 371021576 2853 2853 1677477808683.873 1677477808683.915))
(indexed-future-event 559 '#s(gc-info minor 232520968 371021576 0 224117112 371021576 2856 2856 1677477808684.681 1677477808684.724))
(indexed-future-event 560 '#s(gc-info minor 232543928 371021576 0 224118648 371021576 2859 2859 1677477808685.501 1677477808685.541))
(indexed-future-event 561 '#s(gc-info minor 232529080 371021576 0 224120232 371021576 2862 2862 1677477808686.309 1677477808686.348))
(indexed-future-event 562 '#s(gc-info minor 232537752 371021576 0 224121928 371021576 2865 2865 1677477808687.127 1677477808687.165))
(indexed-future-event 563 '#s(gc-info minor 232517112 371021576 0 224119496 371021576 2868 2868 1677477808687.936 1677477808687.979))
(indexed-future-event 564 '#s(gc-info minor 232530328 371021576 0 224121032 371021576 2871 2871 1677477808688.75 1677477808688.796))
(indexed-future-event 565 '#s(gc-info minor 232522232 371021576 0 224122712 371021576 2874 2874 1677477808689.57 1677477808689.611))
(indexed-future-event 566 '#s(gc-info minor 232534296 371021576 0 224124280 371021576 2877 2877 1677477808690.419 1677477808690.461))
(indexed-future-event 567 '#s(gc-info minor 232537016 371021576 0 224121912 371021576 2880 2880 1677477808691.207 1677477808691.262))
(indexed-future-event 568 '#s(gc-info minor 232533880 371021576 0 224123448 371021576 2883 2883 1677477808692.023 1677477808692.064))
(indexed-future-event 569 '#s(gc-info minor 232557016 371021576 0 224125128 371021576 2886 2886 1677477808692.838 1677477808692.878))
(indexed-future-event 570 '#s(gc-info minor 232519944 371021576 0 224126696 371021576 2889 2889 1677477808693.641 1677477808693.694))
(indexed-future-event 571 '#s(gc-info minor 232538296 371021576 0 224124312 371021576 2892 2892 1677477808694.455 1677477808694.502))
(indexed-future-event 572 '#s(gc-info minor 232526984 371021576 0 224125976 371021576 2895 2895 1677477808695.289 1677477808695.332))
(indexed-future-event 573 '#s(gc-info minor 232537160 371021576 0 224127544 371021576 2898 2898 1677477808696.094 1677477808696.138))
(indexed-future-event 574 '#s(gc-info minor 232522376 371021576 0 224129112 371021576 2901 2901 1677477808696.904 1677477808696.983))
(indexed-future-event 575 '#s(gc-info minor 232541896 371021576 0 224126712 371021576 2904 2904 1677477808697.727 1677477808697.777))
(indexed-future-event 576 '#s(gc-info minor 232529016 371021576 0 224128344 371021576 2907 2907 1677477808698.55 1677477808698.592))
(indexed-future-event 577 '#s(gc-info minor 232522776 371021576 0 224129896 371021576 2910 2910 1677477808699.36 1677477808699.404))
(indexed-future-event 578 '#s(gc-info minor 232524712 371021576 0 224131496 371021576 2913 2913 1677477808700.177 1677477808700.223))
(indexed-future-event 579 '#s(gc-info minor 232532680 371021576 0 224129208 371021576 2916 2916 1677477808700.989 1677477808701.04))
(indexed-future-event 580 '#s(gc-info minor 232524408 371021576 0 224130744 371021576 2918 2919 1677477808701.793 1677477808701.838))
(indexed-future-event 581 '#s(gc-info minor 232525576 371021576 0 224132328 371021576 2921 2921 1677477808702.598 1677477808702.653))
(indexed-future-event 582 '#s(gc-info minor 232527912 371021576 0 224133896 371021576 2924 2924 1677477808703.412 1677477808703.457))
(indexed-future-event 583 '#s(gc-info minor 232535768 371021576 0 224125704 371021576 2927 2927 1677477808704.227 1677477808704.286))
(indexed-future-event 584 '#s(gc-info minor 232521688 371021576 0 224127176 371021576 2930 2930 1677477808705.039 1677477808705.079))
(indexed-future-event 585 '#s(gc-info minor 232554040 371021576 0 224128760 371021576 2933 2933 1677477808705.909 1677477808705.949))
(indexed-future-event 586 '#s(gc-info minor 232547448 371021576 0 224130168 371021576 2936 2937 1677477808706.739 1677477808706.783))
(indexed-future-event 587 '#s(gc-info minor 232557752 371021576 0 224177176 371021576 2939 2939 1677477808707.546 1677477808707.594))
(indexed-future-event 588 '#s(gc-info minor 232571992 371021576 0 224195112 371021576 2942 2942 1677477808708.356 1677477808708.408))
(indexed-future-event 589 '#s(gc-info minor 232590680 371021576 0 224196680 371021576 2945 2945 1677477808709.158 1677477808709.197))
(indexed-future-event 590 '#s(gc-info minor 232614584 371021576 0 224247688 371021576 2948 2948 1677477808709.978 1677477808710.027))
(indexed-future-event 591 '#s(gc-info minor 232686744 371021576 0 224294280 371021576 2951 2951 1677477808710.872 1677477808710.917))
(indexed-future-event 592 '#s(gc-info minor 232689096 371021576 0 224295816 371021576 2954 2954 1677477808711.678 1677477808711.718))
(indexed-future-event 593 '#s(gc-info minor 232696920 371021576 0 224297480 371021576 2957 2957 1677477808712.486 1677477808712.526))
(indexed-future-event 594 '#s(gc-info minor 232708696 371021576 0 224299048 371021576 2960 2960 1677477808713.301 1677477808713.342))
(indexed-future-event 595 '#s(gc-info minor 232693912 371021576 0 224280312 371021576 2963 2963 1677477808714.103 1677477808714.152))
(indexed-future-event 596 '#s(gc-info minor 232681848 371021576 0 224281944 371021576 2966 2966 1677477808714.925 1677477808714.966))
(indexed-future-event 597 '#s(gc-info minor 232692760 371021576 0 224283496 371021576 2969 2969 1677477808715.732 1677477808715.772))
(indexed-future-event 598 '#s(gc-info minor 232726312 371021576 0 224285096 371021576 2972 2972 1677477808716.56 1677477808716.604))
(indexed-future-event 599 '#s(gc-info minor 232686664 371021576 0 224282840 371021576 2975 2975 1677477808717.373 1677477808717.429))
(indexed-future-event 600 '#s(gc-info minor 232678808 371021576 0 224284344 371021576 2978 2978 1677477808718.184 1677477808718.232))
(indexed-future-event 601 '#s(gc-info minor 232679560 371021576 0 224285928 371021576 2981 2981 1677477808719.024 1677477808719.075))
(indexed-future-event 602 '#s(gc-info minor 232730616 371021576 0 224287496 371021576 2984 2984 1677477808719.853 1677477808719.902))
(indexed-future-event 603 '#s(gc-info minor 232689384 371021576 0 224285192 371021576 2987 2987 1677477808720.671 1677477808720.726))
(indexed-future-event 604 '#s(gc-info minor 232697944 371021576 0 224286728 371021576 2990 2990 1677477808721.484 1677477808721.531))
(indexed-future-event 605 '#s(gc-info minor 232681576 371021576 0 224288312 371021576 2993 2993 1677477808722.297 1677477808722.345))
(indexed-future-event 606 '#s(gc-info minor 232689848 371021576 0 224289976 371021576 2996 2996 1677477808723.123 1677477808723.172))
(indexed-future-event 607 '#s(gc-info minor 232701176 371021576 0 224287576 371021576 2999 2999 1677477808723.951 1677477808723.998))
(indexed-future-event 608 '#s(gc-info minor 232683160 371021576 0 224289144 371021576 3002 3002 1677477808724.758 1677477808724.805))
(indexed-future-event 609 '#s(gc-info minor 232699944 371021576 0 224290712 371021576 3005 3005 1677477808725.575 1677477808725.619))
(indexed-future-event 610 '#s(gc-info minor 232692232 371021576 0 224292376 371021576 3008 3008 1677477808726.389 1677477808726.431))
(indexed-future-event 611 '#s(gc-info minor 232718824 371021576 0 224290008 371021576 3011 3011 1677477808727.207 1677477808727.254))
(indexed-future-event 612 '#s(gc-info minor 232700824 371021576 0 224291544 371021576 3014 3014 1677477808728.012 1677477808728.059))
(indexed-future-event 613 '#s(gc-info minor 232693048 371021576 0 224293208 371021576 3017 3017 1677477808728.844 1677477808728.889))
(indexed-future-event 614 '#s(gc-info minor 232704808 371021576 0 224294776 371021576 3020 3020 1677477808729.649 1677477808729.695))
(indexed-future-event 615 '#s(future-event 2 1 complete 1677477808729.82 #f #f))
(indexed-future-event 616 '#s(future-event 2 1 end-work 1677477808729.826 #f #f))
(indexed-future-event 617 '#s(future-event 4 2 complete 1677477808729.891 #f #f))
(indexed-future-event 618 '#s(future-event 4 2 end-work 1677477808729.895 #f #f))
(indexed-future-event 619 '#s(gc-info minor 232695624 371021576 0 224293864 371021576 3023 3023 1677477808731.043 1677477808731.105))
(indexed-future-event 620 '#s(future-event 3 5 complete 1677477808731.333 #f #f))
(indexed-future-event 621 '#s(future-event 3 5 end-work 1677477808731.342 #f #f))
(indexed-future-event 622 '#s(gc-info minor 232688936 371021576 0 224302040 371021576 3026 3026 1677477808733.776 1677477808733.806))
(indexed-future-event 623 '#s(gc-info minor 232696600 371021576 0 224303976 371021576 3029 3029 1677477808736.6 1677477808736.621))
(indexed-future-event 624 '#s(future-event 1 0 complete 1677477808737.807 #f #f))
(indexed-future-event 625 '#s(future-event 1 0 end-work 1677477808737.811 #f #f))
))
]

With @racket[N] as @racket[4] on a machine with at least 4 processing units:

@interaction-eval-show[
    #:eval future-eval
           (show-timeline four-log #:height 600 #:width 1300 #:scale 0.4)
]

Most every arithmetic operation in this example produces an inexact
number whose storage must be allocated, and that triggers frequent
garbage collections as reflected by dense pink lines, effectively
giving the whole graph a pink background. Garbage collection is not
necessarily a problem, but since a garbage collection requires
synchronization across parallel tasks, it can sometimes limit
performance.

By using @tech{flonum}-specific operations (see
@secref["fixnums+flonums"]), we can re-write @racket[mandelbrot] to use
much less allocation:

@interaction-eval[
    #:eval future-eval 
    (define good-log 
      (list
(indexed-future-event 0 '#s(future-event #f 0 create 1677478641063.649 #f 5))
(indexed-future-event 1 '#s(future-event #f 0 create 1677478641063.663 #f 6))
(indexed-future-event 2 '#s(future-event #f 0 create 1677478641063.666 #f 7))
(indexed-future-event 3 '#s(future-event #f 0 create 1677478641063.668 #f 8))
(indexed-future-event 4 '#s(future-event 5 0 start-work 1677478641063.672 #f #f))
(indexed-future-event 5 '#s(future-event 7 4 start-work 1677478641063.699 #f #f))
(indexed-future-event 6 '#s(future-event 6 3 start-work 1677478641063.704 #f #f))
(indexed-future-event 7 '#s(future-event 8 1 start-work 1677478641063.708 #f #f))
(indexed-future-event 8 '#s(future-event 8 1 complete 1677478641104.635 #f #f))
(indexed-future-event 9 '#s(future-event 8 1 end-work 1677478641104.636 #f #f))
(indexed-future-event 10 '#s(future-event 7 4 complete 1677478641104.649 #f #f))
(indexed-future-event 11 '#s(future-event 7 4 end-work 1677478641104.649 #f #f))
(indexed-future-event 12 '#s(future-event 6 3 complete 1677478641104.654 #f #f))
(indexed-future-event 13 '#s(future-event 6 3 end-work 1677478641104.655 #f #f))
(indexed-future-event 14 '#s(future-event 5 0 complete 1677478641107.166 #f #f))
(indexed-future-event 15 '#s(future-event 5 0 end-work 1677478641107.169 #f #f))
            ))
]

@racketblock[
(define (mandelbrot iterations x y n)
  (let ([ci (fl- (fl/ (* 2.0 (->fl y)) (->fl n)) 1.0)]
        [cr (fl- (fl/ (* 2.0 (->fl x)) (->fl n)) 1.5)])
    (let loop ([i 0] [zr 0.0] [zi 0.0])
      (if (> i iterations)
          i
          (let ([zrq (fl* zr zr)]
                [ziq (fl* zi zi)])
            (cond
              [(fl> (fl+ zrq ziq) 4.0) i]
              [else (loop (add1 i)
                          (fl+ (fl- zrq ziq) cr)
                          (fl+ (fl* 2.0 (fl* zr zi)) ci))]))))))
]

This conversion can speed @racket[mandelbrot] by a factor of 10 or so,
even in sequential mode, but avoiding allocation also allows
@racket[mandelbrot] to run more consistently in parallel. Executing
this program yields the following pink-free result the visualizer (not
to scale relative to previous graphs):

@interaction-eval-show[
    #:eval future-eval 
           (show-timeline good-log #:height 600)
]


As a general guideline, an operation is @tech{blocking} if it needs to
consult the continuation (such as obtaining a parameter value) or if
it interacts with Racket's thread system, such as taking a lock within
the implementation of an output port or an @racket[equal?]-based hash
table. In the @tech{CS} implementation of Racket, most primitives are
non-blocking, while the @tech{BC} implementation includes many more
blocking or @tech{synchronized} operations.

@close-eval[future-eval]
