/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var isnan = require( './../../../../base/assert/is-nan' );
var isInteger = require( './../../../../base/assert/is-integer' );
var ln = require( './../../../../base/special/ln' );
var floor = require( './../../../../base/special/floor' );
var PHI = require( '@stdlib/constants/float64/phi' );
var PINF = require( '@stdlib/constants/float64/pinf' );


// VARIABLES //

var SQRT_5 = 2.23606797749979;
var LN_PHI = ln( PHI );


// MAIN //

/**
* Computes the nth non-Fibonacci number.
*
* ## References
*
* -   Gould, H.W. 1965. "Non-Fibonacci Numbers." _Fibonacci Quarterly_, no. 3: 177–83. <http://www.fq.math.ca/Scanned/3-3/gould.pdf>.
* -   Farhi, Bakir. 2011. "An explicit formula generating the non-Fibonacci numbers." _arXiv_ abs/1105.1127 \[Math.NT\] (May): 1–5. <https://arxiv.org/abs/1105.1127>.
*
*
* @param {NonNegativeInteger} n - the non-Fibonacci number to compute
* @returns {NonNegativeInteger} non-Fibonacci number
*
* @example
* var v = nonfibonacci( 1 );
* // returns 4
*
* @example
* var v = nonfibonacci( 2 );
* // returns 6
*
* @example
* var v = nonfibonacci( 3 );
* // returns 7
*
* @example
* var v = nonfibonacci( NaN );
* // returns NaN
*
* @example
* var v = nonfibonacci( 3.14 );
* // returns NaN
*
* @example
* var v = nonfibonacci( -1 );
* // returns NaN
*/
function nonfibonacci( n ) {
	var a;
	var b;
	if (
		isnan( n ) ||
		isInteger( n ) === false ||
		n < 1 ||
		n === PINF
	) {
		return NaN;
	}
	n += 1;
	a = ln( n * SQRT_5 ) / LN_PHI;
	b = ln( (SQRT_5 * (n+a)) - 5.0 + (3.0/n) ) / LN_PHI;
	return floor( n + b - 2.0 );
}


// EXPORTS //

module.exports = nonfibonacci;
