# Copyright 2012-2025, Paul Johnson (paul@pjcj.net)

# This software is free.  It is licensed under the same terms as Perl itself.

# The latest version of this software should be available from my homepage:
# https://pjcj.net

package Devel::Cover::Report::Nvim;

use strict;
use warnings;

our $VERSION;

BEGIN {
  our $VERSION = '1.51'; # VERSION
}

use Devel::Cover::Inc ();

BEGIN { $VERSION //= $Devel::Cover::Inc::VERSION }

use Getopt::Long  qw( GetOptions );
use Template 2.00 ();

sub get_options {
  my ($self, $opt) = @_;
  $opt->{outputfile} = "coverage.lua";
  die "Invalid command line options" unless GetOptions(
    $opt, qw(
      outputfile=s
    )
  );
}

sub report {
  my ($pkg, $db, $options) = @_;

  my $template = Template->new({
    LOAD_TEMPLATES =>
      [ Devel::Cover::Report::Nvim::Template::Provider->new({}) ]
  });

  my $vars = {
    runs => [
      map {
        run      => $_->run,
          perl   => $_->perl,
          OS     => $_->OS,
          start  => scalar gmtime $_->start,
          finish => scalar gmtime $_->finish,
      },
      sort { $a->start <=> $b->start } $db->runs,
    ],
    cov_time => do {
      my $time = 0;
      for ($db->runs) {
        $time = $_->finish if $_->finish > $time;
      }
      int $time
    },
    version => $VERSION,
    files   => $options->{file},
    cover   => $db->cover,
    types   => [ grep $_ ne "time", keys %{ $options->{show} } ],
  };

  my $out = "$options->{outputdir}/$options->{outputfile}";
  $template->process("nvim", $vars, $out) or die $template->error();

  print "Neovim Lua script written to $out\n" unless $options->{silent};
}

1;

package Devel::Cover::Report::Nvim::Template::Provider;

use strict;
use warnings;

our $VERSION = '1.51'; # VERSION

use base "Template::Provider";

my %Templates;

sub fetch {
  my $self = shift;
  my ($name) = @_;
  # print "Looking for <$name>\n";
  $self->SUPER::fetch(exists $Templates{$name} ? \$Templates{$name} : $name)
}

$Templates{nvim} = <<'EOT';
-- This file was generated by Devel::Cover Version [% version %]
-- Devel::Cover is copyright 2001-2025, Paul Johnson (paul@pjcj.net)
-- Devel::Cover is free. It is licensed under the same terms as Perl itself.
-- The latest version of Devel::Cover should be available from my homepage:
-- https://pjcj.net

[% FOREACH r = runs %]
-- Run:          [% r.run    %]
-- Perl version: [% r.perl   %]
-- OS:           [% r.OS     %]
-- Start:        [% r.start  %]
-- Finish:       [% r.finish %]

[% END %]

local M = {}

-- Configuration defaults - can be overridden via vim.g variables
local config = {
  -- Colors
  cover_fg = vim.g.devel_cover_fg or "Green",
  error_fg = vim.g.devel_cover_error_fg or "Red",
  cover_bg = vim.g.devel_cover_bg or nil,
  error_bg = vim.g.devel_cover_error_bg or nil,
  valid_bg = vim.g.devel_cover_valid_bg or nil,
  old_bg = vim.g.devel_cover_old_bg or nil,

  -- Sign text - can be customized
  signs = vim.g.devel_cover_signs or {
    pod = "P ",
    subroutine = "R ",
    statement = "S ",
    branch = "B ",
    condition = "C "
  },

  -- Line highlight groups
  linehl_cover = vim.g.devel_cover_linehl or "cov",
  linehl_error = vim.g.devel_cover_linehl_error or "err",

  -- Additional highlight options
  cterm = vim.g.devel_cover_cterm or "bold",
  gui = vim.g.devel_cover_gui or "bold",

  -- Sign column configuration
  sign_group = vim.g.devel_cover_sign_group or "DevelCover",
  sign_priority = vim.g.devel_cover_sign_priority or 1,
}

-- Build highlight group definitions dynamically
local function create_highlight_groups()
  local types = { "pod", "subroutine", "statement", "branch", "condition" }

  for _, type_name in ipairs(types) do
    -- Normal coverage highlight
    local hl_opts = {
      fg = config.cover_fg,
      bold = config.cterm == "bold" or config.gui == "bold",
    }
    if config.cover_bg then
      hl_opts.bg = config.cover_bg
    end
    vim.api.nvim_set_hl(0, "cov_" .. type_name, hl_opts)

    -- Error coverage highlight
    local err_hl_opts = {
      fg = config.error_fg,
      bold = config.cterm == "bold" or config.gui == "bold",
    }
    if config.error_bg then
      err_hl_opts.bg = config.error_bg
    end
    vim.api.nvim_set_hl(0, "cov_" .. type_name .. "_error", err_hl_opts)
  end
end

-- Create highlight groups
create_highlight_groups()

-- Create line highlight groups for covered/error lines
if config.cover_bg then
  vim.api.nvim_set_hl(0, config.linehl_cover, { bg = config.cover_bg })
end
if config.error_bg then
  vim.api.nvim_set_hl(0, config.linehl_error, { bg = config.error_bg })
end

-- Build sign definitions dynamically
local function create_sign_definitions()
  local types = { "pod", "subroutine", "statement", "branch", "condition" }

  for _, type_name in ipairs(types) do
    -- Normal coverage sign
    vim.fn.sign_define(type_name, {
      text = config.signs[type_name],
      linehl = config.linehl_cover,
      texthl = "cov_" .. type_name
    })

    -- Error coverage sign
    vim.fn.sign_define(type_name .. "_error", {
      text = config.signs[type_name],
      linehl = config.linehl_error,
      texthl = "cov_" .. type_name .. "_error"
    })
  end
end

-- Create sign definitions
create_sign_definitions()

-- Functions for handling coverage age
function M.coverage_old(filename)
  -- This function is called when coverage data is older than the file
  if config.old_bg then
    M.set_background(config.old_bg)
  end
end

function M.coverage_valid(filename)
  -- This function is called when coverage data is current
  if config.valid_bg then
    M.set_background(config.valid_bg)
  end
end

-- Helper function to set background colors for all coverage types
function M.set_background(bg)
  local types = { "pod", "subroutine", "statement", "branch", "condition" }

  -- Update highlight groups for coverage sign characters with new background
  for _, type_name in ipairs(types) do
    -- Get existing highlight and create new options table for normal coverage
    local hl = vim.api.nvim_get_hl(0, { name = "cov_" .. type_name })
    local new_hl = {
      fg = hl.fg,
      bg = bg,
      bold = hl.bold,
      italic = hl.italic,
      underline = hl.underline,
    }
    vim.api.nvim_set_hl(0, "cov_" .. type_name, new_hl)

    -- Get existing highlight and create new options table for error coverage
    local err_hl = vim.api.nvim_get_hl(0, { name = "cov_" .. type_name .. "_error" })
    local new_err_hl = {
      fg = err_hl.fg,
      bg = bg,
      bold = err_hl.bold,
      italic = err_hl.italic,
      underline = err_hl.underline,
    }
    vim.api.nvim_set_hl(0, "cov_" .. type_name .. "_error", new_err_hl)
  end

  -- Redefine all signs with updated texthl (no numhl to avoid affecting line numbers)
  for _, type_name in ipairs(types) do
    -- Normal coverage sign
    vim.fn.sign_define(type_name, {
      text = config.signs[type_name],
      linehl = config.linehl_cover,
      texthl = "cov_" .. type_name
    })

    -- Error coverage sign
    vim.fn.sign_define(type_name .. "_error", {
      text = config.signs[type_name],
      linehl = config.linehl_error,
      texthl = "cov_" .. type_name .. "_error"
    })
  end

  -- Force a refresh by issuing redraw commands
  vim.cmd("redraw!")
  vim.cmd("redrawstatus!")
end

-- Load local configuration if it exists
local config_path = vim.fn.stdpath("config") .. "/lua/devel-cover.lua"
if vim.fn.filereadable(config_path) == 1 then
  print("Reading local config from " .. config_path)
  dofile(config_path)
end

local types = {
[%- FOREACH type = types -%] "[%- type -%]",[%- END -%]
[%- FOREACH type = types -%] "[%- type -%]_error",[%- END -%]
}

[%- MACRO criterion(file, crit, error) BLOCK %]
      [% crit %][% error ? "_error" : "" %] = {
    [%- criteria = cover.file("$file").$crit -%]
    [%- FOREACH loc = criteria.items.nsort -%]
        [%- cov = 0 -%]
        [%- FOREACH l = criteria.location("$loc") -%]
            [%- IF error ? l.error : l.covered -%] [% loc -%],[%- cov = 1; LAST; END -%]
        [%- LAST IF cov; END -%]
    [%- END -%]
      },
[%- END %]

local coverage = {
[% FOREACH file = files %]
  ["[% file %]"] = {
[%- FOREACH type = types; criterion(file, type, 0); criterion(file, type, 1); END %]
  },
[% END %]
}

local cov_time = [% cov_time %]

-- Find coverage data for a given filename
local function coverage_for(filename)
  local fn_len = string.len(filename)
  for cf, _ in pairs(coverage) do
    local f = string.gsub(cf, "^blib/", "")
    local match_pos = string.find(filename, f .. "$")
    if match_pos and match_pos <= fn_len then
      return coverage[cf]
    end
  end

  print("No coverage recorded for " .. filename)
  return {}
end

local signs = {}
local sign_num = 1

-- Remove coverage signs from a file
local function del_coverage_signs(filename)
  if signs[filename] then
    local s = signs[filename]
    for line, id in pairs(s) do
      vim.fn.sign_unplace(config.sign_group, { id = id })
    end
    signs[filename] = {}
  end
end

-- Add coverage signs to a file
local function add_coverage_signs(filename)
  local cov = coverage_for(filename)
  if vim.tbl_isempty(cov) then
    return
  end

  local file_time = vim.fn.getftime(filename)
  local needs_refresh = false

  -- Check if we already have signs and background is changing
  if signs[filename] and not vim.tbl_isempty(signs[filename]) then
    needs_refresh = true
  end

  if file_time > cov_time then
    print("File is newer than coverage run of " .. os.date("%c", cov_time))
    M.coverage_old(filename)
  else
    M.coverage_valid(filename)
  end

  -- If background changed and we had existing signs, refresh them
  if needs_refresh then
    del_coverage_signs(filename)
  end

  if not signs[filename] then
    signs[filename] = {}
  end
  local s = signs[filename]

  -- Process types in reverse order for proper priority
  local reversed_types = {}
  for i = #types, 1, -1 do
    table.insert(reversed_types, types[i])
  end

  for _, type_name in ipairs(reversed_types) do
    if cov[type_name] then
      for _, line in ipairs(cov[type_name]) do
        if not s[line] then
          local id = sign_num
          sign_num = sign_num + 1
          s[line] = id

          -- Validate parameters before placing sign
          local bufnr = vim.fn.bufnr(filename)
          if bufnr == -1 then
            -- Buffer doesn't exist, skip this sign
            goto continue
          end

          -- Validate line number
          local line_count = vim.api.nvim_buf_line_count(bufnr)
          if line > line_count then
            -- Line doesn't exist in buffer, skip this sign
            goto continue
          end

          -- Place the sign with buffer number instead of filename for better reliability
          local success, err = pcall(vim.fn.sign_place, id, config.sign_group, type_name, bufnr, { lnum = line, priority = config.sign_priority })
          if not success then
            print("Warning: Failed to place coverage sign: " .. err)
          end

          ::continue::
        end
      end
    end
  end
end


-- User commands
vim.api.nvim_create_user_command("Cov", function()
  add_coverage_signs(vim.fn.expand("%:p"))
end, {})

vim.api.nvim_create_user_command("Uncov", function()
  del_coverage_signs(vim.fn.expand("%:p"))
end, {})

-- Clear all existing Devel::Cover signs from all buffers when loading new coverage data
-- This ensures old signs don't persist when coverage.lua is reloaded
local function clear_all_coverage_signs()
  -- Get all existing signs in the Devel::Cover group across all buffers
  vim.fn.sign_unplace(config.sign_group)

  -- Clear the signs tracking table
  signs = {}
  sign_num = 1
end

-- Clear old signs when script loads
clear_all_coverage_signs()

-- Auto commands for coverage display
local augroup = vim.api.nvim_create_augroup("devel-cover", { clear = true })

-- Show signs automatically for all known files
for filename, _ in pairs(coverage) do
  vim.api.nvim_create_autocmd("BufReadPost", {
    group = augroup,
    pattern = filename,
    callback = function()
      add_coverage_signs(vim.fn.expand("%:p"))
    end
  })

  local f = string.gsub(filename, "^blib/", "")
  if filename ~= f then
    vim.api.nvim_create_autocmd("BufReadPost", {
      group = augroup,
      pattern = f,
      callback = function()
        add_coverage_signs(vim.fn.expand("%:p"))
      end
    })
  end
end

-- Apply coverage to current buffer
add_coverage_signs(vim.fn.expand("%:p"))

return M
EOT

1

__END__

=head1 NAME

Devel::Cover::Report::Nvim - Backend for displaying coverage data in Neovim

=head1 VERSION

version 1.51

=head1 SYNOPSIS

 cover -report nvim

=head1 DESCRIPTION

This module provides a reporting mechanism for displaying coverage data in
Neovim using Lua.  It is designed to be called from the C<cover> program.

By default, the output of this report is a file named C<coverage.lua> in the
directory of the coverage database.  To use it, run

 :luafile cover_db/coverage.lua

and you should see signs in the left column indicating the coverage status of
that line.

The signs are as follows:

 P - Pod coverage
 S - Statement coverage
 R - Subroutine coverage
 B - Branch coverage
 C - Condition coverage

The last of the criteria, in the order given above, is the one which is
displayed.  Correctly covered criteria are shown in green.  Incorrectly
covered criteria are shown in red.  Any incorrectly covered criterion will
override a correctly covered criterion.

If the coverage for the file being displayed is out of date the function
called coverage_old() is called and passed the name of the file.  Similarly,
for current coverage data file coverage_valid() is called.

Colors and signs can be customized using global vim variables. Set these in your
init.lua or init.vim file before loading the coverage script.

Available configuration variables:

 vim.g.devel_cover_fg         -- Foreground color for covered lines (default: "Green")
 vim.g.devel_cover_error_fg   -- Foreground color for uncovered lines (default: "Red")
 vim.g.devel_cover_bg         -- Background color for covered lines (default: nil)
 vim.g.devel_cover_error_bg   -- Background color for uncovered lines (default: nil)
 vim.g.devel_cover_valid_bg   -- Background when coverage is current (default: nil)
 vim.g.devel_cover_old_bg     -- Background when coverage is old (default: nil)
 vim.g.devel_cover_cterm      -- Terminal styling (default: "bold")
 vim.g.devel_cover_gui        -- GUI styling (default: "bold")
 vim.g.devel_cover_linehl     -- Line highlight group for covered (default: "cov")
 vim.g.devel_cover_linehl_error -- Line highlight group for errors (default: "err")
 vim.g.devel_cover_signs      -- Custom sign text table
 vim.g.devel_cover_sign_group -- Sign group name for dedicated column (default: "DevelCover")
 vim.g.devel_cover_sign_priority -- Sign priority within group (default: 10)

Example configuration for solarized theme in init.lua:

 -- Set colors before loading coverage
 vim.g.devel_cover_fg = "#859900"        -- solarized green
 vim.g.devel_cover_error_fg = "#dc322f"  -- solarized red
 vim.g.devel_cover_valid_bg = "#073642"  -- solarized base02
 vim.g.devel_cover_old_bg = "#342a2a"    -- darker red-tinted background
 vim.g.devel_cover_cterm = "NONE"
 vim.g.devel_cover_gui = "NONE"

 -- Custom sign characters (optional)
 -- Note: The same character is used for both covered/uncovered
 -- The difference is shown through fg/bg colors defined above
 vim.g.devel_cover_signs = {
   pod = "P ",
   subroutine = "R ",
   statement = "· ",  -- Subtle dot for statements (most common)
   branch = "B ",
   condition = "C "
 }

 -- Or for a minimal look that relies purely on colors
 vim.g.devel_cover_bg = "#002b36"        -- dark for covered
 vim.g.devel_cover_error_bg = "#2d1616"  -- dark red for uncovered
 vim.g.devel_cover_signs = {
   pod = "  ",
   subroutine = "  ",
   statement = "│ ",  -- Thin vertical bar (unobtrusive)
   branch = "  ",
   condition = "  "
 }

 -- Or use subtle Unicode characters
 vim.g.devel_cover_signs = {
   pod = "¶ ",
   subroutine = "ƒ ",
   statement = "• ",  -- Small bullet (unobtrusive)
   branch = "» ",
   condition = "? "
 }

 -- For a dedicated coverage column (recommended)
 -- This keeps coverage signs separate from other signs like diagnostics
 vim.g.devel_cover_sign_group = "DevelCover"  -- Creates its own column

 -- You can also configure the sign column appearance
 vim.o.signcolumn = "yes:2"  -- Show 2 sign columns (1 for diagnostics, 1 for coverage)

Alternative: You can still override via devel-cover.lua in your config directory
for more complex customizations

This configuration sets the background colour of the signs to a dark red when
the coverage data is out of date.

coverage.lua adds two user commands: :Cov and :Uncov which can be used to
toggle the state of coverage signs.

The idea and the lua template is adapted from the Vim version which was
shamelessly stolen from Simplecov-Vim.  See
https://github.com/nyarly/Simplecov-Vim

=head1 SEE ALSO

 Devel::Cover
 Devel::Cover::Report::Vim
 Simplecov-Vim (https://github.com/nyarly/Simplecov-Vim)

=head1 BUGS

Huh?

=head1 LICENCE

Copyright 2012-2025, Paul Johnson (paul@pjcj.net)

This software is free.  It is licensed under the same terms as Perl itself.

The latest version of this software should be available from my homepage:
https://pjcj.net

The template is adapted from the Vim version which was copied from Simplecov-Vim
(https://github.com/nyarly/Simplecov-Vim) and is under the MIT Licence.


The MIT License

Copyright (c) 2011 Judson Lester

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.

=cut
