// Copyright (c) Microsoft Corporation.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception


// Copyright 2018 Ulf Adams
// Copyright (c) Microsoft Corporation. All rights reserved.

// Boost Software License - Version 1.0 - August 17th, 2003

// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:

// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.


// This file contains test cases derived from:
// https://github.com/ulfjack/ryu
// See xcharconv_ryu.h for the exact commit.
// (Keep the cgmanifest.json commitHash in sync.)


#ifndef FLOAT_FIXED_PRECISION_TO_CHARS_TEST_CASES_HPP
#define FLOAT_FIXED_PRECISION_TO_CHARS_TEST_CASES_HPP

#include <charconv>

#include "test.hpp"
using namespace std;

inline constexpr FloatPrecisionToCharsTestCase float_fixed_precision_to_chars_test_cases[] = {
    // Test special cases (zero, inf, nan) and an ordinary case. Also test negative signs.
    {0.0f, chars_format::fixed, 4, "0.0000"},
    {-0.0f, chars_format::fixed, 4, "-0.0000"},
    {float_inf, chars_format::fixed, 4, "inf"},
    {-float_inf, chars_format::fixed, 4, "-inf"},
    {float_nan, chars_format::fixed, 4, "nan"},
    {-float_nan, chars_format::fixed, 4, "-nan(ind)"},
    {float_nan_payload, chars_format::fixed, 4, "nan"},
    {-float_nan_payload, chars_format::fixed, 4, "-nan"},
    {1.729f, chars_format::fixed, 4, "1.7290"},
    {-1.729f, chars_format::fixed, 4, "-1.7290"},

    // Ryu Printf Zero
    {0.0f, chars_format::fixed, 4, "0.0000"},
    {0.0f, chars_format::fixed, 3, "0.000"},
    {0.0f, chars_format::fixed, 2, "0.00"},
    {0.0f, chars_format::fixed, 1, "0.0"},
    {0.0f, chars_format::fixed, 0, "0"},

    // Test corner cases.
    {0x0.000002p-126f, chars_format::fixed, 149, // min subnormal
        "0."
        "0000000000000000000000000000000000000000000014012984643248170709237295832899161312802619418765157717570682"
        "8388979108268586060148663818836212158203125"},
    {0x0.fffffep-126f, chars_format::fixed, 149, // max subnormal
        "0."
        "0000000000000000000000000000000000000117549421069244107548702944484928734882705242874589333385717453057158"
        "8870475618904265502351336181163787841796875"},
    {0x1p-126f, chars_format::fixed, 126, // min normal
        "0."
        "0000000000000000000000000000000000000117549435082228750796873653722224567781866555677208752150875170627841"
        "72594547271728515625"},
    {0x1.fffffep+127f, chars_format::fixed, 0, // max normal
        "340282346638528859811704183484516925440"},

    // Ryu Printf AllPowersOfTen
    // These values test every power of ten that's within the range of floats.
    {1e-44f, chars_format::fixed, 149,
        "0."
        "0000000000000000000000000000000000000000000098090892502737194964661070830294129189618335931356104022994779"
        "8722853757880102421040646731853485107421875"},
    {1e-43f, chars_format::fixed, 149,
        "0."
        "0000000000000000000000000000000000000000000994921909670620120355848004135840453208985978732326197947518481"
        "5617516687069610270555131137371063232421875"},
    {1e-42f, chars_format::fixed, 148,
        "0."
        "0000000000000000000000000000000000000000010005271035279193886395429224690001177341070264998322610345467546"
        "973108330377044694614596664905548095703125"},
    {1e-41f, chars_format::fixed, 144,
        "0."
        "0000000000000000000000000000000000000000099996658414218946181117343063568415128159492172308165472584392738"
        "37549166046301252208650112152099609375"},
    {1e-40f, chars_format::fixed, 148,
        "0."
        "0000000000000000000000000000000000000000999994610111475958152591905227349949604220526961919185041279068749"
        "432712426283842432894743978977203369140625"},
    {1e-39f, chars_format::fixed, 146,
        "0."
        "0000000000000000000000000000000000000010000002153053332574208756001456831092687456480096866911043660970225"
        "6827159061458587530069053173065185546875"},
    {1e-38f, chars_format::fixed, 148,
        "0."
        "0000000000000000000000000000000000000099999993504564039245746141539976645128551939195729831580121174560891"
        "149363239804870318039320409297943115234375"},
    {1e-37f, chars_format::fixed, 145,
        "0."
        "0000000000000000000000000000000000000999999991097578965450144252348949782882164643167775990861842615891642"
        "849224041356137604452669620513916015625"},
    {1e-36f, chars_format::fixed, 143,
        "0."
        "0000000000000000000000000000000000010000000359391298238442905219082964481594808441361581309103473121178279"
        "3369735600208514370024204254150390625"},
    {1e-35f, chars_format::fixed, 139,
        "0."
        "0000000000000000000000000000000000100000001800250948048663201408455778204855436374880527489094543362735389"
        "990803014370612800121307373046875"},
    {1e-34f, chars_format::fixed, 136,
        "0."
        "0000000000000000000000000000000001000000046701102029858885626602539647826036732368569844521988439212112353"
        "970951517112553119659423828125"},
    {1e-33f, chars_format::fixed, 131,
        "0."
        "0000000000000000000000000000000010000000237422279903610827365881541552040508374727581888171540347443055907"
        "2061441838741302490234375"},
    {1e-32f, chars_format::fixed, 130,
        "0."
        "0000000000000000000000000000000100000002374222799036108273658815415520405083747275818881715403474430559072"
        "061441838741302490234375"},
    {1e-31f, chars_format::fixed, 126,
        "0."
        "0000000000000000000000000000000999999979661189834525301187760534009369837919272799809863871978166116605279"
        "96718883514404296875"},
    {1e-30f, chars_format::fixed, 118,
        "0."
        "0000000000000000000000000000010000000031710768509710513471352647538147514756461109453056224083411507308483"
        "123779296875"},
    {1e-29f, chars_format::fixed, 117,
        "0."
        "0000000000000000000000000000100000000317107685097105134713526475381475147564611094530562240834115073084831"
        "23779296875"},
    {1e-28f, chars_format::fixed, 116,
        "0."
        "0000000000000000000000000001000000003171076850971051347135264753814751475646110945305622408341150730848312"
        "3779296875"},
    {1e-27f, chars_format::fixed, 112,
        "0."
        "0000000000000000000000000010000000272452011558114995103349890361263429573723815479979748488403856754302978"
        "515625"},
    {1e-26f, chars_format::fixed, 109,
        "0."
        "0000000000000000000000000099999998872660226806678244921543018442779658661034858369021094404160976409912109"
        "375"},
    {1e-25f, chars_format::fixed, 104,
        "0."
        "00000000000000000000000010000000195414813782625560981110772657866336832199749551364220678806304931640625"},
    {1e-24f, chars_format::fixed, 103,
        "0."
        "0000000000000000000000010000000195414813782625560981110772657866336832199749551364220678806304931640625"},
    {1e-23f, chars_format::fixed, 99,
        "0.000000000000000000000009999999998199587477372609628178631337169779413898140774108469486236572265625"},
    {1e-22f, chars_format::fixed, 97,
        "0.0000000000000000000001000000031374394956577733179287005745028427128318071481771767139434814453125"},
    {1e-21f, chars_format::fixed, 88,
        "0.0000000000000000000009999999682655225388967887463487205224055287544615566730499267578125"},
    {1e-20f, chars_format::fixed, 87,
        "0.000000000000000000009999999682655225388967887463487205224055287544615566730499267578125"},
    {1e-19f, chars_format::fixed, 86,
        "0.00000000000000000009999999682655225388967887463487205224055287544615566730499267578125"},
    {1e-18f, chars_format::fixed, 83,
        "0.00000000000000000100000004581370496574313326554328540396454627625644207000732421875"},
    {1e-17f, chars_format::fixed, 79,
        "0.0000000000000000099999998377515902426605765018763349871733225882053375244140625"},
    {1e-16f, chars_format::fixed, 77,
        "0.00000000000000010000000168623835263871646450439811815158464014530181884765625"},
    {1e-15f, chars_format::fixed, 73, "0.0000000000000010000000036274937255387218471014421083964407444000244140625"},
    {1e-14f, chars_format::fixed, 68, "0.00000000000000999999982451670044181213370393379591405391693115234375"},
    {1e-13f, chars_format::fixed, 67, "0.0000000000000999999982451670044181213370393379591405391693115234375"},
    {1e-12f, chars_format::fixed, 61, "0.0000000000009999999960041972002500187954865396022796630859375"},
    {1e-11f, chars_format::fixed, 60, "0.000000000009999999960041972002500187954865396022796630859375"},
    {1e-10f, chars_format::fixed, 57, "0.000000000100000001335143196001808973960578441619873046875"},
    {1e-09f, chars_format::fixed, 53, "0.00000000099999997171806853657471947371959686279296875"},
    {1e-08f, chars_format::fixed, 50, "0.00000000999999993922529029077850282192230224609375"},
    {1e-07f, chars_format::fixed, 47, "0.00000010000000116860974230803549289703369140625"},
    {1e-06f, chars_format::fixed, 43, "0.0000009999999974752427078783512115478515625"},
    {1e-05f, chars_format::fixed, 38, "0.00000999999974737875163555145263671875"},
    {1e-04f, chars_format::fixed, 37, "0.0000999999974737875163555145263671875"},
    {1e-03f, chars_format::fixed, 33, "0.001000000047497451305389404296875"},
    {1e-02f, chars_format::fixed, 29, "0.00999999977648258209228515625"},
    {1e-01f, chars_format::fixed, 27, "0.100000001490116119384765625"},
    {1e+00f, chars_format::fixed, 0, "1"},
    {1e+01f, chars_format::fixed, 0, "10"},
    {1e+02f, chars_format::fixed, 0, "100"},
    {1e+03f, chars_format::fixed, 0, "1000"},
    {1e+04f, chars_format::fixed, 0, "10000"},
    {1e+05f, chars_format::fixed, 0, "100000"},
    {1e+06f, chars_format::fixed, 0, "1000000"},
    {1e+07f, chars_format::fixed, 0, "10000000"},
    {1e+08f, chars_format::fixed, 0, "100000000"},
    {1e+09f, chars_format::fixed, 0, "1000000000"},
    {1e+10f, chars_format::fixed, 0, "10000000000"},
    {1e+11f, chars_format::fixed, 0, "99999997952"},
    {1e+12f, chars_format::fixed, 0, "999999995904"},
    {1e+13f, chars_format::fixed, 0, "9999999827968"},
    {1e+14f, chars_format::fixed, 0, "100000000376832"},
    {1e+15f, chars_format::fixed, 0, "999999986991104"},
    {1e+16f, chars_format::fixed, 0, "10000000272564224"},
    {1e+17f, chars_format::fixed, 0, "99999998430674944"},
    {1e+18f, chars_format::fixed, 0, "999999984306749440"},
    {1e+19f, chars_format::fixed, 0, "9999999980506447872"},
    {1e+20f, chars_format::fixed, 0, "100000002004087734272"},
    {1e+21f, chars_format::fixed, 0, "1000000020040877342720"},
    {1e+22f, chars_format::fixed, 0, "9999999778196308361216"},
    {1e+23f, chars_format::fixed, 0, "99999997781963083612160"},
    {1e+24f, chars_format::fixed, 0, "1000000013848427855085568"},
    {1e+25f, chars_format::fixed, 0, "9999999562023526247432192"},
    {1e+26f, chars_format::fixed, 0, "100000002537764290115403776"},
    {1e+27f, chars_format::fixed, 0, "999999988484154753734934528"},
    {1e+28f, chars_format::fixed, 0, "9999999442119689768320106496"},
    {1e+29f, chars_format::fixed, 0, "100000001504746621987668885504"},
    {1e+30f, chars_format::fixed, 0, "1000000015047466219876688855040"},
    {1e+31f, chars_format::fixed, 0, "9999999848243207295109594873856"},
    {1e+32f, chars_format::fixed, 0, "100000003318135351409612647563264"},
    {1e+33f, chars_format::fixed, 0, "999999994495727286427992885035008"},
    {1e+34f, chars_format::fixed, 0, "9999999790214767953607394487959552"},
    {1e+35f, chars_format::fixed, 0, "100000004091847875962975319375216640"},
    {1e+36f, chars_format::fixed, 0, "999999961690316245365415600208216064"},
    {1e+37f, chars_format::fixed, 0, "9999999933815812510711506376257961984"},
    {1e+38f, chars_format::fixed, 0, "99999996802856924650656260769173209088"},

    // Ryu Printf RoundToEven
    {0.125f, chars_format::fixed, 3, "0.125"},
    {0.125f, chars_format::fixed, 2, "0.12"},
    {0.375f, chars_format::fixed, 3, "0.375"},
    {0.375f, chars_format::fixed, 2, "0.38"},

    // Ryu Printf RoundToEvenInteger
    {2.5f, chars_format::fixed, 1, "2.5"},
    {2.5f, chars_format::fixed, 0, "2"},
    {3.5f, chars_format::fixed, 1, "3.5"},
    {3.5f, chars_format::fixed, 0, "4"},

    // Ryu Printf NonRoundToEvenScenarios
    {0.748046875f, chars_format::fixed, 3, "0.748"},
    {0.748046875f, chars_format::fixed, 2, "0.75"},
    {0.748046875f, chars_format::fixed, 1, "0.7"}, // 0.75 would round to "0.8", but this is smaller

    {0.2509765625f, chars_format::fixed, 3, "0.251"},
    {0.2509765625f, chars_format::fixed, 2, "0.25"},
    {0.2509765625f, chars_format::fixed, 1, "0.3"}, // 0.25 would round to "0.2", but this is larger

    {0x1.000002p-2f, chars_format::fixed, 25, "0.2500000298023223876953125"},
    {0x1.000002p-2f, chars_format::fixed, 3, "0.250"},
    {0x1.000002p-2f, chars_format::fixed, 2, "0.25"},
    {0x1.000002p-2f, chars_format::fixed, 1, "0.3"}, // 0.25 would round to "0.2", but this is larger (again)

    // More rounding tests.
    {9.5f, chars_format::fixed, 1, "9.5"},
    {9.5f, chars_format::fixed, 0, "10"},
    {10.5f, chars_format::fixed, 1, "10.5"},
    {10.5f, chars_format::fixed, 0, "10"},

    {1.241f, chars_format::fixed, 3, "1.241"},
    {1.241f, chars_format::fixed, 1, "1.2"},
    {1.251f, chars_format::fixed, 3, "1.251"},
    {1.251f, chars_format::fixed, 1, "1.3"},
    {1.261f, chars_format::fixed, 3, "1.261"},
    {1.261f, chars_format::fixed, 1, "1.3"},
    {1.341f, chars_format::fixed, 3, "1.341"},
    {1.341f, chars_format::fixed, 1, "1.3"},
    {1.351f, chars_format::fixed, 3, "1.351"},
    {1.351f, chars_format::fixed, 1, "1.4"},
    {1.361f, chars_format::fixed, 3, "1.361"},
    {1.361f, chars_format::fixed, 1, "1.4"},

    {2.41f, chars_format::fixed, 2, "2.41"},
    {2.41f, chars_format::fixed, 0, "2"},
    {2.51f, chars_format::fixed, 2, "2.51"},
    {2.51f, chars_format::fixed, 0, "3"},
    {2.61f, chars_format::fixed, 2, "2.61"},
    {2.61f, chars_format::fixed, 0, "3"},
    {3.41f, chars_format::fixed, 2, "3.41"},
    {3.41f, chars_format::fixed, 0, "3"},
    {3.51f, chars_format::fixed, 2, "3.51"},
    {3.51f, chars_format::fixed, 0, "4"},
    {3.61f, chars_format::fixed, 2, "3.61"},
    {3.61f, chars_format::fixed, 0, "4"},

    // Ryu Printf VaryingPrecision
    {1.142857f, chars_format::fixed, 28, "1.1428569555282592773437500000"},
    {1.142857f, chars_format::fixed, 27, "1.142856955528259277343750000"},
    {1.142857f, chars_format::fixed, 26, "1.14285695552825927734375000"},
    {1.142857f, chars_format::fixed, 25, "1.1428569555282592773437500"},
    {1.142857f, chars_format::fixed, 24, "1.142856955528259277343750"},
    {1.142857f, chars_format::fixed, 23, "1.14285695552825927734375"},
    {1.142857f, chars_format::fixed, 22, "1.1428569555282592773438"},
    {1.142857f, chars_format::fixed, 21, "1.142856955528259277344"},
    {1.142857f, chars_format::fixed, 20, "1.14285695552825927734"},
    {1.142857f, chars_format::fixed, 19, "1.1428569555282592773"},
    {1.142857f, chars_format::fixed, 18, "1.142856955528259277"},
    {1.142857f, chars_format::fixed, 17, "1.14285695552825928"},
    {1.142857f, chars_format::fixed, 16, "1.1428569555282593"},
    {1.142857f, chars_format::fixed, 15, "1.142856955528259"},
    {1.142857f, chars_format::fixed, 14, "1.14285695552826"},
    {1.142857f, chars_format::fixed, 13, "1.1428569555283"},
    {1.142857f, chars_format::fixed, 12, "1.142856955528"},
    {1.142857f, chars_format::fixed, 11, "1.14285695553"},
    {1.142857f, chars_format::fixed, 10, "1.1428569555"},
    {1.142857f, chars_format::fixed, 9, "1.142856956"},
    {1.142857f, chars_format::fixed, 8, "1.14285696"},
    {1.142857f, chars_format::fixed, 7, "1.1428570"},
    {1.142857f, chars_format::fixed, 6, "1.142857"},
    {1.142857f, chars_format::fixed, 5, "1.14286"},
    {1.142857f, chars_format::fixed, 4, "1.1429"},
    {1.142857f, chars_format::fixed, 3, "1.143"},
    {1.142857f, chars_format::fixed, 2, "1.14"},
    {1.142857f, chars_format::fixed, 1, "1.1"},
    {1.142857f, chars_format::fixed, 0, "1"},

    // Negative precision requests 6 digits of precision.
    {1.142857f, chars_format::fixed, -1, "1.142857"},
    {1.142857f, chars_format::fixed, -2, "1.142857"},
    {1.142857f, chars_format::fixed, -3, "1.142857"},

    // Ryu Printf Carrying
    {0.0009f, chars_format::fixed, 4, "0.0009"},
    {0.0009f, chars_format::fixed, 3, "0.001"},
    {0.0029f, chars_format::fixed, 4, "0.0029"},
    {0.0029f, chars_format::fixed, 3, "0.003"},
    {0.0099f, chars_format::fixed, 4, "0.0099"},
    {0.0099f, chars_format::fixed, 3, "0.010"},
    {0.0299f, chars_format::fixed, 4, "0.0299"},
    {0.0299f, chars_format::fixed, 3, "0.030"},
    {0.0999f, chars_format::fixed, 4, "0.0999"},
    {0.0999f, chars_format::fixed, 3, "0.100"},
    {0.2999f, chars_format::fixed, 4, "0.2999"},
    {0.2999f, chars_format::fixed, 3, "0.300"},
    {0.9999f, chars_format::fixed, 4, "0.9999"},
    {0.9999f, chars_format::fixed, 3, "1.000"},
    {2.9999f, chars_format::fixed, 4, "2.9999"},
    {2.9999f, chars_format::fixed, 3, "3.000"},
    {9.9999f, chars_format::fixed, 4, "9.9999"},
    {9.9999f, chars_format::fixed, 3, "10.000"},
    {29.9999f, chars_format::fixed, 4, "29.9999"},
    {29.9999f, chars_format::fixed, 3, "30.000"},
    {99.9999f, chars_format::fixed, 4, "99.9999"},
    {99.9999f, chars_format::fixed, 3, "100.000"},
    {299.9999f, chars_format::fixed, 4, "299.9999"},
    {299.9999f, chars_format::fixed, 3, "300.000"},

    {0.09f, chars_format::fixed, 2, "0.09"},
    {0.09f, chars_format::fixed, 1, "0.1"},
    {0.29f, chars_format::fixed, 2, "0.29"},
    {0.29f, chars_format::fixed, 1, "0.3"},
    {0.99f, chars_format::fixed, 2, "0.99"},
    {0.99f, chars_format::fixed, 1, "1.0"},
    {2.99f, chars_format::fixed, 2, "2.99"},
    {2.99f, chars_format::fixed, 1, "3.0"},
    {9.99f, chars_format::fixed, 2, "9.99"},
    {9.99f, chars_format::fixed, 1, "10.0"},
    {29.99f, chars_format::fixed, 2, "29.99"},
    {29.99f, chars_format::fixed, 1, "30.0"},
    {99.99f, chars_format::fixed, 2, "99.99"},
    {99.99f, chars_format::fixed, 1, "100.0"},
    {299.99f, chars_format::fixed, 2, "299.99"},
    {299.99f, chars_format::fixed, 1, "300.0"},

    {0.9f, chars_format::fixed, 1, "0.9"},
    {0.9f, chars_format::fixed, 0, "1"},
    {2.9f, chars_format::fixed, 1, "2.9"},
    {2.9f, chars_format::fixed, 0, "3"},
    {9.9f, chars_format::fixed, 1, "9.9"},
    {9.9f, chars_format::fixed, 0, "10"},
    {29.9f, chars_format::fixed, 1, "29.9"},
    {29.9f, chars_format::fixed, 0, "30"},
    {99.9f, chars_format::fixed, 1, "99.9"},
    {99.9f, chars_format::fixed, 0, "100"},
    {299.9f, chars_format::fixed, 1, "299.9"},
    {299.9f, chars_format::fixed, 0, "300"},

    // Ryu Printf RoundingResultZero
    {0.004f, chars_format::fixed, 3, "0.004"},
    {0.004f, chars_format::fixed, 2, "0.00"},
    {0.4f, chars_format::fixed, 1, "0.4"},
    {0.4f, chars_format::fixed, 0, "0"},
    {0.5f, chars_format::fixed, 1, "0.5"},
    {0.5f, chars_format::fixed, 0, "0"},

    // Ryu Printf AllBinaryExponents
    // These values test every binary exponent.
    // The mantissas were randomly generated.
    {0x0.bafab0p-126f, chars_format::fixed, 146,
        "0."
        "0000000000000000000000000000000000000085856660078164374052571520381239855817217629811131320365461649230225"
        "8101698697601023013703525066375732421875"},
    {0x1.2c4906p-126f, chars_format::fixed, 149,
        "0."
        "0000000000000000000000000000000000000137884223604447257991705553959882023830165636017936204355992973751997"
        "2013648494879589634365402162075042724609375"},
    {0x1.da6c8cp-125f, chars_format::fixed, 147,
        "0."
        "0000000000000000000000000000000000000435689644606144922962341034916717454461784748180572306511365470365953"
        "68653788540314053534530103206634521484375"},
    {0x1.094fd8p-124f, chars_format::fixed, 145,
        "0."
        "0000000000000000000000000000000000000487300980449569406486898593097235018686258578182781609022211792895361"
        "293087574949822737835347652435302734375"},
    {0x1.1fba2ap-123f, chars_format::fixed, 146,
        "0."
        "0000000000000000000000000000000000001056942819182250793988103906000418825212997860175004035399177620979400"
        "2661102041429330711252987384796142578125"},
    {0x1.05c066p-122f, chars_format::fixed, 145,
        "0."
        "0000000000000000000000000000000000001923046724143804860574903666495417184272104431219282642086921032167044"
        "776084452450959361158311367034912109375"},
    {0x1.aa97aep-121f, chars_format::fixed, 144,
        "0."
        "0000000000000000000000000000000000006268213405227838203431757007863521825324812101203856469220875149766722"
        "40063493291017948649823665618896484375"},
    {0x1.dd39a8p-120f, chars_format::fixed, 141,
        "0."
        "0000000000000000000000000000000000014024388746462477508516570165943074639858849697711014853285027082288594"
        "08023936339304782450199127197265625"},
    {0x1.d47ee4p-119f, chars_format::fixed, 141,
        "0."
        "0000000000000000000000000000000000027535700468003209195416842287342046227746219166975594715835834154840644"
        "49429979504202492535114288330078125"},
    {0x1.3d3c36p-118f, chars_format::fixed, 141,
        "0."
        "0000000000000000000000000000000000037290818427663765497432907740725131368928567215367442737882934305701509"
        "99159295679419301450252532958984375"},
    {0x1.1661f4p-117f, chars_format::fixed, 139,
        "0."
        "0000000000000000000000000000000000065447441644065192772010189083715139205202243608571574700187600294454259"
        "852727773250080645084381103515625"},
    {0x1.b68df4p-116f, chars_format::fixed, 138,
        "0."
        "0000000000000000000000000000000000206207336977375818320192367186588031181149116107584831794039036470533865"
        "49714559805579483509063720703125"},
    {0x1.d99cbcp-115f, chars_format::fixed, 137,
        "0."
        "0000000000000000000000000000000000445382813514879099167344241437088992337862296399353856123689687079858501"
        "3062620419077575206756591796875"},
    {0x1.fd046ep-114f, chars_format::fixed, 137,
        "0."
        "0000000000000000000000000000000000957355143513621934625335631305232671680527244248210039119495722709229923"
        "4527672524563968181610107421875"},
    {0x1.89834cp-113f, chars_format::fixed, 135,
        "0."
        "0000000000000000000000000000000001480230929779647770398330978023641189899501899250006144284413575618053471"
        "31667076610028743743896484375"},
    {0x1.44f9f6p-112f, chars_format::fixed, 135,
        "0."
        "0000000000000000000000000000000002444850777614032698558632186222413041911856813461843446368257012912827974"
        "56005937419831752777099609375"},
    {0x1.610156p-111f, chars_format::fixed, 134,
        "0."
        "0000000000000000000000000000000005311432194104638958491823018983379637847618995394862767718806989547530861"
        "2731168977916240692138671875"},
    {0x1.1c4ce0p-110f, chars_format::fixed, 129,
        "0."
        "0000000000000000000000000000000008555350741040305433153411782350510989661057055423470539198826934068620175"
        "82170665264129638671875"},
    {0x1.c8846ap-109f, chars_format::fixed, 132,
        "0."
        "0000000000000000000000000000000027475632104005746766942313987412370292282555616009485081234833825369889837"
        "02030964195728302001953125"},
    {0x1.49aaa6p-108f, chars_format::fixed, 131,
        "0."
        "0000000000000000000000000000000039682172991165697309827799919973022713504068615668209476284727932338114442"
        "2455690801143646240234375"},
    {0x1.f5603cp-107f, chars_format::fixed, 129,
        "0."
        "0000000000000000000000000000000120701861138584576157150361758540470962569680234040676814963322094342856871"
        "66266143321990966796875"},
    {0x1.b7bbf8p-106f, chars_format::fixed, 127,
        "0."
        "0000000000000000000000000000000211724341322508937840915176712265363597160619557838059749677039889093066449"
        "277102947235107421875"},
    {0x1.6d305cp-105f, chars_format::fixed, 127,
        "0."
        "0000000000000000000000000000000351664122601460292174574136500884378151845989294218826175242309517443572985"
        "939681529998779296875"},
    {0x1.dd9944p-104f, chars_format::fixed, 126,
        "0."
        "0000000000000000000000000000000919821625881433083724268012282371631728046815243377909779298740033937065163"
        "62726688385009765625"},
    {0x1.0f4254p-103f, chars_format::fixed, 125,
        "0."
        "0000000000000000000000000000001044852024561729954450605502201132293228119140390439904125807757395705266389"
        "9958133697509765625"},
    {0x1.049450p-102f, chars_format::fixed, 122,
        "0."
        "0000000000000000000000000000002007430259113927348388472759172849833669167623529955984951200775867619086056"
        "9477081298828125"},
    {0x1.28d030p-101f, chars_format::fixed, 121,
        "0."
        "0000000000000000000000000000004573131937693259427041496124538667251427229422876784281637441154089174233376"
        "979827880859375"},
    {0x1.28a2bep-100f, chars_format::fixed, 123,
        "0."
        "0000000000000000000000000000009140793594875532256935908936727717060368161903718573976784789181238011224195"
        "36113739013671875"},
    {0x1.e674d2p-99f, chars_format::fixed, 122,
        "0."
        "0000000000000000000000000000029980185962354845050897219390986524441297232842758378051906120731473492924124"
        "0024566650390625"},
    {0x1.227314p-98f, chars_format::fixed, 120,
        "0."
        "0000000000000000000000000000035800667869547456776694564979794758804336827210561614021067100566142471507191"
        "65802001953125"},
    {0x1.735b6cp-97f, chars_format::fixed, 119,
        "0."
        "0000000000000000000000000000091546597262378319603799081529207906703837108826262575211885064163652714341878"
        "8909912109375"},
    {0x1.ef60b4p-96f, chars_format::fixed, 118,
        "0."
        "0000000000000000000000000000244240085996903849216356078751341022854748745722804070812372856380534358322620"
        "391845703125"},
    {0x1.f58d34p-95f, chars_format::fixed, 117,
        "0."
        "0000000000000000000000000000494568036548015750964222103779193000963396430755616983709899159293854609131813"
        "04931640625"},
    {0x1.a9fa8ap-94f, chars_format::fixed, 117,
        "0."
        "0000000000000000000000000000840094794528154864080325730507983995095850536733231295194457288744160905480384"
        "82666015625"},
    {0x1.2ebd9ap-93f, chars_format::fixed, 116,
        "0."
        "0000000000000000000000000001194101241497498690354073604120044306874891791320908440710013564967084676027297"
        "9736328125"},
    {0x1.1c25bep-92f, chars_format::fixed, 115,
        "0."
        "0000000000000000000000000002241527991772840369420073304083191365256388471842441401093992681126110255718231"
        "201171875"},
    {0x1.80d526p-91f, chars_format::fixed, 114,
        "0."
        "0000000000000000000000000006071588038765549904506806942923684386898144178454361785668425000039860606193542"
        "48046875"},
    {0x1.16cdd0p-90f, chars_format::fixed, 110,
        "0."
        "0000000000000000000000000008797501615285119946834874987404311536605839433322628906353202182799577713012695"
        "3125"},
    {0x1.be00c0p-89f, chars_format::fixed, 107,
        "0."
        "0000000000000000000000000028146741987560362391732368583172300827762585262448169487470295280218124389648437"
        "5"},
    {0x1.dbe376p-88f, chars_format::fixed, 111,
        "0."
        "0000000000000000000000000060065575697458565955014820557088473772922162727261330417150020366534590721130371"
        "09375"},
    {0x1.75b358p-87f, chars_format::fixed, 108,
        "0."
        "0000000000000000000000000094335284238393382638914933670753282739891362740358715655020205304026603698730468"
        "75"},
    {0x1.5e56fap-86f, chars_format::fixed, 109,
        "0."
        "0000000000000000000000000176876373794073549186243776242822499413496518949617808402763330377638339996337890"
        "625"},
    {0x1.1542e6p-85f, chars_format::fixed, 108,
        "0."
        "0000000000000000000000000279962390364982552136537973532622361314990341862873890477203531190752983093261718"
        "75"},
    {0x1.37b7a6p-84f, chars_format::fixed, 107,
        "0."
        "0000000000000000000000000629508229027247503098930994487426088983880040350626927647681441158056259155273437"
        "5"},
    {0x1.31f62ap-83f, chars_format::fixed, 106,
        "0."
        "000000000000000000000000123576897369666466985856753932114511864998745993560191891447175294160842895507812"
        "5"},
    {0x1.ac3560p-82f, chars_format::fixed, 101,
        "0.00000000000000000000000034590406845628797200186450581230516131137076030199750675819814205169677734375"},
    {0x1.a7db5cp-81f, chars_format::fixed, 103,
        "0."
        "0000000000000000000000006847777099176331341674240101847394713956151957034990118700079619884490966796875"},
    {0x1.40189cp-80f, chars_format::fixed, 102,
        "0.000000000000000000000001034286379672715366987641733210033664035198963659922810620628297328948974609375"},
    {0x1.aad1eep-79f, chars_format::fixed, 102,
        "0.000000000000000000000002758259846499093682487211692864773559218105614121441249153576791286468505859375"},
    {0x1.49824cp-78f, chars_format::fixed, 100,
        "0.0000000000000000000000042588036474940459085811637121780459484809977510622047702781856060028076171875"},
    {0x1.955292p-77f, chars_format::fixed, 100,
        "0.0000000000000000000000104773420985315373838626628182169411331037256474019159213639795780181884765625"},
    {0x1.d8ca0cp-76f, chars_format::fixed, 98,
        "0.00000000000000000000002444263111177596802332967266437459101513507420122550684027373790740966796875"},
    {0x1.28b5aap-75f, chars_format::fixed, 98,
        "0.00000000000000000000003067905619497844072028707730390270809472941238027487997896969318389892578125"},
    {0x1.e5fda8p-74f, chars_format::fixed, 95,
        "0.00000000000000000000010050055115902033206854316793794380802129495577901252545416355133056640625"},
    {0x1.fd929cp-73f, chars_format::fixed, 95,
        "0.00000000000000000000021075432611470358337541921610390309449467594049565377645194530487060546875"},
    {0x1.c0b84cp-72f, chars_format::fixed, 94,
        "0.0000000000000000000003711724097438896357340602997067040280665395357573288492858409881591796875"},
    {0x1.5cfeaep-71f, chars_format::fixed, 94,
        "0.0000000000000000000005773635352424624465965559338086719731730767080080113373696804046630859375"},
    {0x1.bcce4ap-70f, chars_format::fixed, 93,
        "0.000000000000000000001471738991536079335112024613790339400143380998997599817812442779541015625"},
    {0x1.edf106p-69f, chars_format::fixed, 92,
        "0.00000000000000000000326863064574260634910627773444362353938430487687583081424236297607421875"},
    {0x1.30b422p-68f, chars_format::fixed, 91,
        "0.0000000000000000000040327191475944672156035895296995186232180685692583210766315460205078125"},
    {0x1.7aa8d8p-67f, chars_format::fixed, 88,
        "0.0000000000000000000100230347240102665385544432156972316505516573670320212841033935546875"},
    {0x1.4ad4e0p-66f, chars_format::fixed, 85,
        "0.0000000000000000000175140760553442509348562143578487138029231573455035686492919921875"},
    {0x1.dde636p-65f, chars_format::fixed, 88,
        "0.0000000000000000000505995524921864861016965251964971894693690046551637351512908935546875"},
    {0x1.5df870p-64f, chars_format::fixed, 84,
        "0.000000000000000000074109127331368847396687003781234892585416673682630062103271484375"},
    {0x1.c346fap-63f, chars_format::fixed, 86,
        "0.00000000000000000019112335047873604296656620434025075638828639057464897632598876953125"},
    {0x1.58d2eap-62f, chars_format::fixed, 85,
        "0.0000000000000000002920771899491385068938311721231659845443573431111872196197509765625"},
    {0x1.0d4824p-61f, chars_format::fixed, 83,
        "0.00000000000000000045618111223383324851561766710705825289551285095512866973876953125"},
    {0x1.04585cp-60f, chars_format::fixed, 82,
        "0.0000000000000000008820836917354691955064048547452415505176759324967861175537109375"},
    {0x1.55cf7ap-59f, chars_format::fixed, 82,
        "0.0000000000000000023161977389916240139687737820128887733517331071197986602783203125"},
    {0x1.1fd8ecp-58f, chars_format::fixed, 80,
        "0.00000000000000000390105904223582084021197668999292318403604440391063690185546875"},
    {0x1.0bc866p-57f, chars_format::fixed, 80,
        "0.00000000000000000725826751123333980988787395016714754092390649020671844482421875"},
    {0x1.4dfa86p-56f, chars_format::fixed, 79,
        "0.0000000000000000181050165732891247031242920595417444928898476064205169677734375"},
    {0x1.335daep-55f, chars_format::fixed, 78,
        "0.000000000000000033324681586205479543426333233213654239079914987087249755859375"},
    {0x1.5bc756p-54f, chars_format::fixed, 77,
        "0.00000000000000007541247487712833172911197632259927559061907231807708740234375"},
    {0x1.9eb052p-53f, chars_format::fixed, 76,
        "0.0000000000000001798425779915148827771409489884035792783834040164947509765625"},
    {0x1.13b6d2p-52f, chars_format::fixed, 75,
        "0.000000000000000239143897259270284301468922905087310937233269214630126953125"},
    {0x1.260438p-51f, chars_format::fixed, 72,
        "0.000000000000000510037289299151118393549353413618518970906734466552734375"},
    {0x1.9e6b44p-50f, chars_format::fixed, 72,
        "0.000000000000001437804758404521467129999479084290214814245700836181640625"},
    {0x1.89c0bcp-49f, chars_format::fixed, 71,
        "0.00000000000000273220937993773164975674916377101908437907695770263671875"},
    {0x1.e30610p-48f, chars_format::fixed, 68,
        "0.00000000000000670330015995791728133923470522859133780002593994140625"},
    {0x1.48b6e8p-47f, chars_format::fixed, 68,
        "0.00000000000000912365953728740131101204724473063834011554718017578125"},
    {0x1.41382ep-46f, chars_format::fixed, 69,
        "0.000000000000017831261573081173821275768887062440626323223114013671875"},
    {0x1.383b8ep-45f, chars_format::fixed, 68,
        "0.00000000000003466478609693256218715617933412431739270687103271484375"},
    {0x1.1e6564p-44f, chars_format::fixed, 66, "0.000000000000063592699357274684590635160930105485022068023681640625"},
    {0x1.c35e62p-43f, chars_format::fixed, 66, "0.000000000000200447961722950707130763703389675356447696685791015625"},
    {0x1.2a2f4ep-42f, chars_format::fixed, 65, "0.00000000000026484129017449731247069166784058324992656707763671875"},
    {0x1.69fae2p-41f, chars_format::fixed, 64, "0.0000000000006430056682417417679431537180789746344089508056640625"},
    {0x1.4ccefep-40f, chars_format::fixed, 63, "0.000000000001182373535017766652543969030375592410564422607421875"},
    {0x1.aa9bf6p-39f, chars_format::fixed, 62, "0.00000000000303124083993189241681420753593556582927703857421875"},
    {0x1.3b9744p-38f, chars_format::fixed, 60, "0.000000000004484816164274096905728583806194365024566650390625"},
    {0x1.b2fc6ap-37f, chars_format::fixed, 60, "0.000000000012363045483188006556929394719190895557403564453125"},
    {0x1.7bc418p-36f, chars_format::fixed, 57, "0.000000000021587197307493255493682227097451686859130859375"},
    {0x1.f4a74cp-35f, chars_format::fixed, 57, "0.000000000056917713597837149563929415307939052581787109375"},
    {0x1.89f248p-34f, chars_format::fixed, 55, "0.0000000000895730434269381703416001982986927032470703125"},
    {0x1.60ac54p-33f, chars_format::fixed, 55, "0.0000000001603771837555001411601551808416843414306640625"},
    {0x1.2f6d0ep-32f, chars_format::fixed, 55, "0.0000000002759643347172158200919511727988719940185546875"},
    {0x1.748684p-31f, chars_format::fixed, 53, "0.00000000067761984912095840627443976700305938720703125"},
    {0x1.b4fa00p-30f, chars_format::fixed, 45, "0.000000001589711473570787347853183746337890625"},
    {0x1.c204d8p-29f, chars_format::fixed, 50, "0.00000000327431859403759517590515315532684326171875"},
    {0x1.50029ep-28f, chars_format::fixed, 51, "0.000000004889592286616561978007666766643524169921875"},
    {0x1.56cf38p-27f, chars_format::fixed, 48, "0.000000009977068060607052757404744625091552734375"},
    {0x1.0b5a5cp-26f, chars_format::fixed, 48, "0.000000015561990807100301026366651058197021484375"},
    {0x1.fc8250p-25f, chars_format::fixed, 45, "0.000000059198242752245278097689151763916015625"},
    {0x1.c66674p-24f, chars_format::fixed, 46, "0.0000001057982927932243910618126392364501953125"},
    {0x1.4da57ep-23f, chars_format::fixed, 46, "0.0000001553662372089092968963086605072021484375"},
    {0x1.4fcdacp-22f, chars_format::fixed, 44, "0.00000031274129241865011863410472869873046875"},
    {0x1.5eaff4p-21f, chars_format::fixed, 43, "0.0000006532060297104180790483951568603515625"},
    {0x1.d2f696p-20f, chars_format::fixed, 43, "0.0000017395735767422593198716640472412109375"},
    {0x1.e4400cp-19f, chars_format::fixed, 41, "0.00000360794501830241642892360687255859375"},
    {0x1.03e624p-18f, chars_format::fixed, 40, "0.0000038727966966689564287662506103515625"},
    {0x1.bdb65ep-17f, chars_format::fixed, 40, "0.0000132832637973478995263576507568359375"},
    {0x1.57fb84p-16f, chars_format::fixed, 38, "0.00002050295370281673967838287353515625"},
    {0x1.fd2d62p-15f, chars_format::fixed, 38, "0.00006069866140023805201053619384765625"},
    {0x1.ca0c58p-14f, chars_format::fixed, 35, "0.00010920720524154603481292724609375"},
    {0x1.988f70p-13f, chars_format::fixed, 33, "0.000194816733710467815399169921875"},
    {0x1.032dd6p-12f, chars_format::fixed, 35, "0.00024717240012250840663909912109375"},
    {0x1.571b08p-11f, chars_format::fixed, 32, "0.00065442197956144809722900390625"},
    {0x1.53bedap-10f, chars_format::fixed, 33, "0.001296026282943785190582275390625"},
    {0x1.ab2f36p-9f, chars_format::fixed, 32, "0.00325915846042335033416748046875"},
    {0x1.7293dap-8f, chars_format::fixed, 31, "0.0056545645929872989654541015625"},
    {0x1.825eb6p-7f, chars_format::fixed, 30, "0.011791075579822063446044921875"},
    {0x1.f45aa0p-6f, chars_format::fixed, 25, "0.0305391848087310791015625"},
    {0x1.854d96p-5f, chars_format::fixed, 28, "0.0475223474204540252685546875"},
    {0x1.5650cep-4f, chars_format::fixed, 27, "0.083573155105113983154296875"},
    {0x1.03acdap-3f, chars_format::fixed, 26, "0.12679453194141387939453125"},
    {0x1.6b9416p-2f, chars_format::fixed, 25, "0.3550570905208587646484375"},
    {0x1.a8544ap-1f, chars_format::fixed, 24, "0.828768074512481689453125"},
    {0x1.0693f6p+0f, chars_format::fixed, 23, "1.02569520473480224609375"},
    {0x1.b9476ep+1f, chars_format::fixed, 22, "3.4474923610687255859375"},
    {0x1.3cb752p+2f, chars_format::fixed, 21, "4.948688983917236328125"},
    {0x1.bb8a64p+3f, chars_format::fixed, 19, "13.8606433868408203125"},
    {0x1.1de906p+4f, chars_format::fixed, 19, "17.8693904876708984375"},
    {0x1.d8e834p+5f, chars_format::fixed, 17, "59.11338043212890625"},
    {0x1.27cd38p+6f, chars_format::fixed, 15, "73.950408935546875"},
    {0x1.3cdcd6p+7f, chars_format::fixed, 16, "158.4313201904296875"},
    {0x1.392656p+8f, chars_format::fixed, 15, "313.149749755859375"},
    {0x1.c96aa8p+9f, chars_format::fixed, 12, "914.833251953125"},
    {0x1.28b6b2p+10f, chars_format::fixed, 13, "1186.8546142578125"},
    {0x1.786090p+11f, chars_format::fixed, 9, "3011.017578125"},
    {0x1.79c6f6p+12f, chars_format::fixed, 11, "6044.43505859375"},
    {0x1.ef1840p+13f, chars_format::fixed, 5, "15843.03125"},
    {0x1.539fd0p+14f, chars_format::fixed, 6, "21735.953125"},
    {0x1.b31804p+15f, chars_format::fixed, 7, "55692.0078125"},
    {0x1.ad4a9cp+16f, chars_format::fixed, 6, "109898.609375"},
    {0x1.4c43a6p+17f, chars_format::fixed, 6, "170119.296875"},
    {0x1.5598c6p+18f, chars_format::fixed, 5, "349795.09375"},
    {0x1.73695ep+19f, chars_format::fixed, 4, "760650.9375"},
    {0x1.234f2ap+20f, chars_format::fixed, 3, "1193202.625"},
    {0x1.0a4cc8p+21f, chars_format::fixed, 0, "2181529"},
    {0x1.90abd2p+22f, chars_format::fixed, 1, "6564596.5"},
    {0x1.62dde8p+23f, chars_format::fixed, 0, "11628276"},
    {0x1.9e3a8cp+24f, chars_format::fixed, 0, "27146892"},
    {0x1.53a3eap+25f, chars_format::fixed, 0, "44517332"},
    {0x1.41a1cep+26f, chars_format::fixed, 0, "84313912"},
    {0x1.8fdda4p+27f, chars_format::fixed, 0, "209644832"},
    {0x1.d0322ap+28f, chars_format::fixed, 0, "486744736"},
    {0x1.cdb764p+29f, chars_format::fixed, 0, "968289408"},
    {0x1.7620d8p+30f, chars_format::fixed, 0, "1569207808"},
    {0x1.c18df6p+31f, chars_format::fixed, 0, "3771136768"},
    {0x1.240cf8p+32f, chars_format::fixed, 0, "4899796992"},
    {0x1.81669ap+33f, chars_format::fixed, 0, "12931904512"},
    {0x1.3be30cp+34f, chars_format::fixed, 0, "21198811136"},
    {0x1.d1e6e4p+35f, chars_format::fixed, 0, "62532296704"},
    {0x1.06b274p+36f, chars_format::fixed, 0, "70517211136"},
    {0x1.a74284p+37f, chars_format::fixed, 0, "227235889152"},
    {0x1.9fd3e6p+38f, chars_format::fixed, 0, "446491623424"},
    {0x1.e2cec4p+39f, chars_format::fixed, 0, "1036821594112"},
    {0x1.3d5d32p+40f, chars_format::fixed, 0, "1363068190720"},
    {0x1.accccap+41f, chars_format::fixed, 0, "3683363586048"},
    {0x1.a120ccp+42f, chars_format::fixed, 0, "7166206410752"},
    {0x1.55a028p+43f, chars_format::fixed, 0, "11738166591488"},
    {0x1.035296p+44f, chars_format::fixed, 0, "17820513468416"},
    {0x1.22d1aap+45f, chars_format::fixed, 0, "39969859043328"},
    {0x1.eb8eaep+46f, chars_format::fixed, 0, "135118253457408"},
    {0x1.490d0ep+47f, chars_format::fixed, 0, "180897697497088"},
    {0x1.9da088p+48f, chars_format::fixed, 0, "454787778740224"},
    {0x1.e7fab4p+49f, chars_format::fixed, 0, "1073077848899584"},
    {0x1.98a534p+50f, chars_format::fixed, 0, "1797241144606720"},
    {0x1.93aeeap+51f, chars_format::fixed, 0, "3550835489374208"},
    {0x1.3df680p+52f, chars_format::fixed, 0, "5593662327095296"},
    {0x1.c763f6p+53f, chars_format::fixed, 0, "16022627827056640"},
    {0x1.8b669ep+54f, chars_format::fixed, 0, "27823861147893760"},
    {0x1.73e5b6p+55f, chars_format::fixed, 0, "52339893103230976"},
    {0x1.a13d18p+56f, chars_format::fixed, 0, "117442238576852992"},
    {0x1.a0797ep+57f, chars_format::fixed, 0, "234454344768946176"},
    {0x1.c07a80p+58f, chars_format::fixed, 0, "504941918963105792"},
    {0x1.729388p+59f, chars_format::fixed, 0, "834463629662224384"},
    {0x1.edfb70p+60f, chars_format::fixed, 0, "2224697951572197376"},
    {0x1.3d6782p+61f, chars_format::fixed, 0, "2858924021141995520"},
    {0x1.b121e8p+62f, chars_format::fixed, 0, "7802620494837972992"},
    {0x1.0efc5ap+63f, chars_format::fixed, 0, "9763290520209063936"},
    {0x1.b7dba0p+64f, chars_format::fixed, 0, "31695102724410441728"},
    {0x1.ec2306p+65f, chars_format::fixed, 0, "70924388975830368256"},
    {0x1.2e2d28p+66f, chars_format::fixed, 0, "87096415015485308928"},
    {0x1.e02208p+67f, chars_format::fixed, 0, "276777792668052750336"},
    {0x1.402636p+68f, chars_format::fixed, 0, "369106968238077509632"},
    {0x1.11f97cp+69f, chars_format::fixed, 0, "631742296991907971072"},
    {0x1.74db2ap+70f, chars_format::fixed, 0, "1719495307615820316672"},
    {0x1.94a32ap+71f, chars_format::fixed, 0, "3732120907777931476992"},
    {0x1.c272dcp+72f, chars_format::fixed, 0, "8309311323384498356224"},
    {0x1.36ca40p+73f, chars_format::fixed, 0, "11466128622488263852032"},
    {0x1.5f6fbep+74f, chars_format::fixed, 0, "25931436172223350571008"},
    {0x1.95ec4ep+75f, chars_format::fixed, 0, "59903671176748022628352"},
    {0x1.6b3912p+76f, chars_format::fixed, 0, "107204487170660958732288"},
    {0x1.10992ap+77f, chars_format::fixed, 0, "160913632700346331561984"},
    {0x1.74a25ep+78f, chars_format::fixed, 0, "439928869395322133020672"},
    {0x1.43f462p+79f, chars_format::fixed, 0, "764916220582548125777920"},
    {0x1.f12ca2p+80f, chars_format::fixed, 0, "2347839472055691035803648"},
    {0x1.2b7f18p+81f, chars_format::fixed, 0, "2828664088515283884441600"},
    {0x1.a40704p+82f, chars_format::fixed, 0, "7934093352976572433301504"},
    {0x1.35d5f8p+83f, chars_format::fixed, 0, "11705266159821935293235200"},
    {0x1.c2c9d2p+84f, chars_format::fixed, 0, "34060605519118462894473216"},
    {0x1.47bf20p+85f, chars_format::fixed, 0, "49527663163502775133798400"},
    {0x1.60b728p+86f, chars_format::fixed, 0, "106601704860119390738186240"},
    {0x1.3354c8p+87f, chars_format::fixed, 0, "185770297377533474371534848"},
    {0x1.e9e512p+88f, chars_format::fixed, 0, "592246479757524141957185536"},
    {0x1.c4b6cap+89f, chars_format::fixed, 0, "1094595334815995103451021312"},
    {0x1.799cb8p+90f, chars_format::fixed, 0, "1826020469467809704300249088"},
    {0x1.1afa36p+91f, chars_format::fixed, 0, "2736789351009782551090823168"},
    {0x1.80c214p+92f, chars_format::fixed, 0, "7442304364233212615194574848"},
    {0x1.657890p+93f, chars_format::fixed, 0, "13828987453168434783077793792"},
    {0x1.5ce17cp+94f, chars_format::fixed, 0, "26993344325171312829134798848"},
    {0x1.3f1e9ap+95f, chars_format::fixed, 0, "49381356576017938861904625664"},
    {0x1.874612p+96f, chars_format::fixed, 0, "121093348650115637567232671744"},
    {0x1.5f4d5ep+97f, chars_format::fixed, 0, "217445539275703670631001227264"},
    {0x1.45b1bep+98f, chars_format::fixed, 0, "403190021246562727728269754368"},
    {0x1.a570f4p+99f, chars_format::fixed, 0, "1043437928672039460753056464896"},
    {0x1.f5106ep+100f, chars_format::fixed, 0, "2481149635102733266542145830912"},
    {0x1.d84424p+101f, chars_format::fixed, 0, "4677097651091265616934539886592"},
    {0x1.3d6c56p+102f, chars_format::fixed, 0, "6287213966425746785671183335424"},
    {0x1.9d8cf0p+103f, chars_format::fixed, 0, "16382424580981433623378525159424"},
    {0x1.e2e73ep+104f, chars_format::fixed, 0, "38259540322544957537972440268800"},
    {0x1.2d6594p+105f, chars_format::fixed, 0, "47758227647613648865431576903680"},
    {0x1.ce43bap+106f, chars_format::fixed, 0, "146497485749802409635393442938880"},
    {0x1.b3ea00p+107f, chars_format::fixed, 0, "276293361488025452794185737306112"},
    {0x1.03a052p+108f, chars_format::fixed, 0, "329115373194929392757058784198656"},
    {0x1.6f59e0p+109f, chars_format::fixed, 0, "931345619455766569116232623063040"},
    {0x1.05adacp+110f, chars_format::fixed, 0, "1326867152522435745601434087849984"},
    {0x1.2cdef0p+111f, chars_format::fixed, 0, "3051192904788012466473218045116416"},
    {0x1.e81552p+112f, chars_format::fixed, 0, "9899505055765620068271358482579456"},
    {0x1.bfa8f4p+113f, chars_format::fixed, 0, "18159245876954178992833811110166528"},
    {0x1.a14810p+114f, chars_format::fixed, 0, "33853896736735722962455354188759040"},
    {0x1.f18b10p+115f, chars_format::fixed, 0, "80731001914916160681187088757948416"},
    {0x1.8d6e30p+116f, chars_format::fixed, 0, "128973545052908058560090358153216000"},
    {0x1.9480c2p+117f, chars_format::fixed, 0, "262537431192608192877759864086986752"},
    {0x1.60975cp+118f, chars_format::fixed, 0, "457689606761340509948952337218273280"},
    {0x1.ab1bb2p+119f, chars_format::fixed, 0, "1108836243133298765768030079592431616"},
    {0x1.6a0c80p+120f, chars_format::fixed, 0, "1879864992909653247408339011818749952"},
    {0x1.2cac2cp+121f, chars_format::fixed, 0, "3122362236102854007005843883842076672"},
    {0x1.0baaf6p+122f, chars_format::fixed, 0, "5559243043957593079267046257728684032"},
    {0x1.098282p+123f, chars_format::fixed, 0, "11028845443370647144636654644992409600"},
    {0x1.122f8ap+124f, chars_format::fixed, 0, "22778456735621461875293910785310326784"},
    {0x1.57f4c6p+125f, chars_format::fixed, 0, "57149517363101270672263900542030315520"},
    {0x1.05e028p+126f, chars_format::fixed, 0, "87023098173139747570875357950241669120"},
    {0x1.9d8424p+127f, chars_format::fixed, 0, "274828637805621292108186801756142829568"},

    // Test the maximum mantissa, which generates the most digits for each exponent.
    {0x0.fffffep-126f, chars_format::fixed, 149,
        "0."
        "0000000000000000000000000000000000000117549421069244107548702944484928734882705242874589333385717453057158"
        "8870475618904265502351336181163787841796875"},
    {0x1.fffffep-126f, chars_format::fixed, 149,
        "0."
        "0000000000000000000000000000000000000235098856151472858345576598207153302664571798551798085536592623685000"
        "6129930346077117064851336181163787841796875"},
    {0x1.fffffep-125f, chars_format::fixed, 148,
        "0."
        "0000000000000000000000000000000000000470197712302945716691153196414306605329143597103596171073185247370001"
        "225986069215423412970267236232757568359375"},
    {0x1.fffffep-124f, chars_format::fixed, 147,
        "0."
        "0000000000000000000000000000000000000940395424605891433382306392828613210658287194207192342146370494740002"
        "45197213843084682594053447246551513671875"},
    {0x1.fffffep-123f, chars_format::fixed, 146,
        "0."
        "0000000000000000000000000000000000001880790849211782866764612785657226421316574388414384684292740989480004"
        "9039442768616936518810689449310302734375"},
    {0x1.fffffep-122f, chars_format::fixed, 145,
        "0."
        "0000000000000000000000000000000000003761581698423565733529225571314452842633148776828769368585481978960009"
        "807888553723387303762137889862060546875"},
    {0x1.fffffep-121f, chars_format::fixed, 144,
        "0."
        "0000000000000000000000000000000000007523163396847131467058451142628905685266297553657538737170963957920019"
        "61577710744677460752427577972412109375"},
    {0x1.fffffep-120f, chars_format::fixed, 143,
        "0."
        "0000000000000000000000000000000000015046326793694262934116902285257811370532595107315077474341927915840039"
        "2315542148935492150485515594482421875"},
    {0x1.fffffep-119f, chars_format::fixed, 142,
        "0."
        "0000000000000000000000000000000000030092653587388525868233804570515622741065190214630154948683855831680078"
        "463108429787098430097103118896484375"},
    {0x1.fffffep-118f, chars_format::fixed, 141,
        "0."
        "0000000000000000000000000000000000060185307174777051736467609141031245482130380429260309897367711663360156"
        "92621685957419686019420623779296875"},
    {0x1.fffffep-117f, chars_format::fixed, 140,
        "0."
        "0000000000000000000000000000000000120370614349554103472935218282062490964260760858520619794735423326720313"
        "8524337191483937203884124755859375"},
    {0x1.fffffep-116f, chars_format::fixed, 139,
        "0."
        "0000000000000000000000000000000000240741228699108206945870436564124981928521521717041239589470846653440627"
        "704867438296787440776824951171875"},
    {0x1.fffffep-115f, chars_format::fixed, 138,
        "0."
        "0000000000000000000000000000000000481482457398216413891740873128249963857043043434082479178941693306881255"
        "40973487659357488155364990234375"},
    {0x1.fffffep-114f, chars_format::fixed, 137,
        "0."
        "0000000000000000000000000000000000962964914796432827783481746256499927714086086868164958357883386613762510"
        "8194697531871497631072998046875"},
    {0x1.fffffep-113f, chars_format::fixed, 136,
        "0."
        "0000000000000000000000000000000001925929829592865655566963492512999855428172173736329916715766773227525021"
        "638939506374299526214599609375"},
    {0x1.fffffep-112f, chars_format::fixed, 135,
        "0."
        "0000000000000000000000000000000003851859659185731311133926985025999710856344347472659833431533546455050043"
        "27787901274859905242919921875"},
    {0x1.fffffep-111f, chars_format::fixed, 134,
        "0."
        "0000000000000000000000000000000007703719318371462622267853970051999421712688694945319666863067092910100086"
        "5557580254971981048583984375"},
    {0x1.fffffep-110f, chars_format::fixed, 133,
        "0."
        "0000000000000000000000000000000015407438636742925244535707940103998843425377389890639333726134185820200173"
        "111516050994396209716796875"},
    {0x1.fffffep-109f, chars_format::fixed, 132,
        "0."
        "0000000000000000000000000000000030814877273485850489071415880207997686850754779781278667452268371640400346"
        "22303210198879241943359375"},
    {0x1.fffffep-108f, chars_format::fixed, 131,
        "0."
        "0000000000000000000000000000000061629754546971700978142831760415995373701509559562557334904536743280800692"
        "4460642039775848388671875"},
    {0x1.fffffep-107f, chars_format::fixed, 130,
        "0."
        "0000000000000000000000000000000123259509093943401956285663520831990747403019119125114669809073486561601384"
        "892128407955169677734375"},
    {0x1.fffffep-106f, chars_format::fixed, 129,
        "0."
        "0000000000000000000000000000000246519018187886803912571327041663981494806038238250229339618146973123202769"
        "78425681591033935546875"},
    {0x1.fffffep-105f, chars_format::fixed, 128,
        "0."
        "0000000000000000000000000000000493038036375773607825142654083327962989612076476500458679236293946246405539"
        "5685136318206787109375"},
    {0x1.fffffep-104f, chars_format::fixed, 127,
        "0."
        "0000000000000000000000000000000986076072751547215650285308166655925979224152953000917358472587892492811079"
        "137027263641357421875"},
    {0x1.fffffep-103f, chars_format::fixed, 126,
        "0."
        "0000000000000000000000000000001972152145503094431300570616333311851958448305906001834716945175784985622158"
        "27405452728271484375"},
    {0x1.fffffep-102f, chars_format::fixed, 125,
        "0."
        "0000000000000000000000000000003944304291006188862601141232666623703916896611812003669433890351569971244316"
        "5481090545654296875"},
    {0x1.fffffep-101f, chars_format::fixed, 124,
        "0."
        "0000000000000000000000000000007888608582012377725202282465333247407833793223624007338867780703139942488633"
        "096218109130859375"},
    {0x1.fffffep-100f, chars_format::fixed, 123,
        "0."
        "0000000000000000000000000000015777217164024755450404564930666494815667586447248014677735561406279884977266"
        "19243621826171875"},
    {0x1.fffffep-99f, chars_format::fixed, 122,
        "0."
        "0000000000000000000000000000031554434328049510900809129861332989631335172894496029355471122812559769954532"
        "3848724365234375"},
    {0x1.fffffep-98f, chars_format::fixed, 121,
        "0."
        "0000000000000000000000000000063108868656099021801618259722665979262670345788992058710942245625119539909064"
        "769744873046875"},
    {0x1.fffffep-97f, chars_format::fixed, 120,
        "0."
        "0000000000000000000000000000126217737312198043603236519445331958525340691577984117421884491250239079818129"
        "53948974609375"},
    {0x1.fffffep-96f, chars_format::fixed, 119,
        "0."
        "0000000000000000000000000000252435474624396087206473038890663917050681383155968234843768982500478159636259"
        "0789794921875"},
    {0x1.fffffep-95f, chars_format::fixed, 118,
        "0."
        "0000000000000000000000000000504870949248792174412946077781327834101362766311936469687537965000956319272518"
        "157958984375"},
    {0x1.fffffep-94f, chars_format::fixed, 117,
        "0."
        "0000000000000000000000000001009741898497584348825892155562655668202725532623872939375075930001912638545036"
        "31591796875"},
    {0x1.fffffep-93f, chars_format::fixed, 116,
        "0."
        "0000000000000000000000000002019483796995168697651784311125311336405451065247745878750151860003825277090072"
        "6318359375"},
    {0x1.fffffep-92f, chars_format::fixed, 115,
        "0."
        "0000000000000000000000000004038967593990337395303568622250622672810902130495491757500303720007650554180145"
        "263671875"},
    {0x1.fffffep-91f, chars_format::fixed, 114,
        "0."
        "0000000000000000000000000008077935187980674790607137244501245345621804260990983515000607440015301108360290"
        "52734375"},
    {0x1.fffffep-90f, chars_format::fixed, 113,
        "0."
        "0000000000000000000000000016155870375961349581214274489002490691243608521981967030001214880030602216720581"
        "0546875"},
    {0x1.fffffep-89f, chars_format::fixed, 112,
        "0."
        "0000000000000000000000000032311740751922699162428548978004981382487217043963934060002429760061204433441162"
        "109375"},
    {0x1.fffffep-88f, chars_format::fixed, 111,
        "0."
        "0000000000000000000000000064623481503845398324857097956009962764974434087927868120004859520122408866882324"
        "21875"},
    {0x1.fffffep-87f, chars_format::fixed, 110,
        "0."
        "0000000000000000000000000129246963007690796649714195912019925529948868175855736240009719040244817733764648"
        "4375"},
    {0x1.fffffep-86f, chars_format::fixed, 109,
        "0."
        "0000000000000000000000000258493926015381593299428391824039851059897736351711472480019438080489635467529296"
        "875"},
    {0x1.fffffep-85f, chars_format::fixed, 108,
        "0."
        "0000000000000000000000000516987852030763186598856783648079702119795472703422944960038876160979270935058593"
        "75"},
    {0x1.fffffep-84f, chars_format::fixed, 107,
        "0."
        "0000000000000000000000001033975704061526373197713567296159404239590945406845889920077752321958541870117187"
        "5"},
    {0x1.fffffep-83f, chars_format::fixed, 106,
        "0."
        "000000000000000000000000206795140812305274639542713459231880847918189081369177984015550464391708374023437"
        "5"},
    {0x1.fffffep-82f, chars_format::fixed, 105,
        "0."
        "00000000000000000000000041359028162461054927908542691846376169583637816273835596803110092878341674804687"
        "5"},
    {0x1.fffffep-81f, chars_format::fixed, 104,
        "0."
        "00000000000000000000000082718056324922109855817085383692752339167275632547671193606220185756683349609375"},
    {0x1.fffffep-80f, chars_format::fixed, 103,
        "0."
        "0000000000000000000000016543611264984421971163417076738550467833455126509534238721244037151336669921875"},
    {0x1.fffffep-79f, chars_format::fixed, 102,
        "0.000000000000000000000003308722252996884394232683415347710093566691025301906847744248807430267333984375"},
    {0x1.fffffep-78f, chars_format::fixed, 101,
        "0.00000000000000000000000661744450599376878846536683069542018713338205060381369548849761486053466796875"},
    {0x1.fffffep-77f, chars_format::fixed, 100,
        "0.0000000000000000000000132348890119875375769307336613908403742667641012076273909769952297210693359375"},
    {0x1.fffffep-76f, chars_format::fixed, 99,
        "0.000000000000000000000026469778023975075153861467322781680748533528202415254781953990459442138671875"},
    {0x1.fffffep-75f, chars_format::fixed, 98,
        "0.00000000000000000000005293955604795015030772293464556336149706705640483050956390798091888427734375"},
    {0x1.fffffep-74f, chars_format::fixed, 97,
        "0.0000000000000000000001058791120959003006154458692911267229941341128096610191278159618377685546875"},
    {0x1.fffffep-73f, chars_format::fixed, 96,
        "0.000000000000000000000211758224191800601230891738582253445988268225619322038255631923675537109375"},
    {0x1.fffffep-72f, chars_format::fixed, 95,
        "0.00000000000000000000042351644838360120246178347716450689197653645123864407651126384735107421875"},
    {0x1.fffffep-71f, chars_format::fixed, 94,
        "0.0000000000000000000008470328967672024049235669543290137839530729024772881530225276947021484375"},
    {0x1.fffffep-70f, chars_format::fixed, 93,
        "0.000000000000000000001694065793534404809847133908658027567906145804954576306045055389404296875"},
    {0x1.fffffep-69f, chars_format::fixed, 92,
        "0.00000000000000000000338813158706880961969426781731605513581229160990915261209011077880859375"},
    {0x1.fffffep-68f, chars_format::fixed, 91,
        "0.0000000000000000000067762631741376192393885356346321102716245832198183052241802215576171875"},
    {0x1.fffffep-67f, chars_format::fixed, 90,
        "0.000000000000000000013552526348275238478777071269264220543249166439636610448360443115234375"},
    {0x1.fffffep-66f, chars_format::fixed, 89,
        "0.00000000000000000002710505269655047695755414253852844108649833287927322089672088623046875"},
    {0x1.fffffep-65f, chars_format::fixed, 88,
        "0.0000000000000000000542101053931009539151082850770568821729966657585464417934417724609375"},
    {0x1.fffffep-64f, chars_format::fixed, 87,
        "0.000000000000000000108420210786201907830216570154113764345993331517092883586883544921875"},
    {0x1.fffffep-63f, chars_format::fixed, 86,
        "0.00000000000000000021684042157240381566043314030822752869198666303418576717376708984375"},
    {0x1.fffffep-62f, chars_format::fixed, 85,
        "0.0000000000000000004336808431448076313208662806164550573839733260683715343475341796875"},
    {0x1.fffffep-61f, chars_format::fixed, 84,
        "0.000000000000000000867361686289615262641732561232910114767946652136743068695068359375"},
    {0x1.fffffep-60f, chars_format::fixed, 83,
        "0.00000000000000000173472337257923052528346512246582022953589330427348613739013671875"},
    {0x1.fffffep-59f, chars_format::fixed, 82,
        "0.0000000000000000034694467451584610505669302449316404590717866085469722747802734375"},
    {0x1.fffffep-58f, chars_format::fixed, 81,
        "0.000000000000000006938893490316922101133860489863280918143573217093944549560546875"},
    {0x1.fffffep-57f, chars_format::fixed, 80,
        "0.00000000000000001387778698063384420226772097972656183628714643418788909912109375"},
    {0x1.fffffep-56f, chars_format::fixed, 79,
        "0.0000000000000000277555739612676884045354419594531236725742928683757781982421875"},
    {0x1.fffffep-55f, chars_format::fixed, 78,
        "0.000000000000000055511147922535376809070883918906247345148585736751556396484375"},
    {0x1.fffffep-54f, chars_format::fixed, 77,
        "0.00000000000000011102229584507075361814176783781249469029717147350311279296875"},
    {0x1.fffffep-53f, chars_format::fixed, 76,
        "0.0000000000000002220445916901415072362835356756249893805943429470062255859375"},
    {0x1.fffffep-52f, chars_format::fixed, 75,
        "0.000000000000000444089183380283014472567071351249978761188685894012451171875"},
    {0x1.fffffep-51f, chars_format::fixed, 74,
        "0.00000000000000088817836676056602894513414270249995752237737178802490234375"},
    {0x1.fffffep-50f, chars_format::fixed, 73,
        "0.0000000000000017763567335211320578902682854049999150447547435760498046875"},
    {0x1.fffffep-49f, chars_format::fixed, 72,
        "0.000000000000003552713467042264115780536570809999830089509487152099609375"},
    {0x1.fffffep-48f, chars_format::fixed, 71,
        "0.00000000000000710542693408452823156107314161999966017901897430419921875"},
    {0x1.fffffep-47f, chars_format::fixed, 70,
        "0.0000000000000142108538681690564631221462832399993203580379486083984375"},
    {0x1.fffffep-46f, chars_format::fixed, 69,
        "0.000000000000028421707736338112926244292566479998640716075897216796875"},
    {0x1.fffffep-45f, chars_format::fixed, 68,
        "0.00000000000005684341547267622585248858513295999728143215179443359375"},
    {0x1.fffffep-44f, chars_format::fixed, 67, "0.0000000000001136868309453524517049771702659199945628643035888671875"},
    {0x1.fffffep-43f, chars_format::fixed, 66, "0.000000000000227373661890704903409954340531839989125728607177734375"},
    {0x1.fffffep-42f, chars_format::fixed, 65, "0.00000000000045474732378140980681990868106367997825145721435546875"},
    {0x1.fffffep-41f, chars_format::fixed, 64, "0.0000000000009094946475628196136398173621273599565029144287109375"},
    {0x1.fffffep-40f, chars_format::fixed, 63, "0.000000000001818989295125639227279634724254719913005828857421875"},
    {0x1.fffffep-39f, chars_format::fixed, 62, "0.00000000000363797859025127845455926944850943982601165771484375"},
    {0x1.fffffep-38f, chars_format::fixed, 61, "0.0000000000072759571805025569091185388970188796520233154296875"},
    {0x1.fffffep-37f, chars_format::fixed, 60, "0.000000000014551914361005113818237077794037759304046630859375"},
    {0x1.fffffep-36f, chars_format::fixed, 59, "0.00000000002910382872201022763647415558807551860809326171875"},
    {0x1.fffffep-35f, chars_format::fixed, 58, "0.0000000000582076574440204552729483111761510372161865234375"},
    {0x1.fffffep-34f, chars_format::fixed, 57, "0.000000000116415314888040910545896622352302074432373046875"},
    {0x1.fffffep-33f, chars_format::fixed, 56, "0.00000000023283062977608182109179324470460414886474609375"},
    {0x1.fffffep-32f, chars_format::fixed, 55, "0.0000000004656612595521636421835864894092082977294921875"},
    {0x1.fffffep-31f, chars_format::fixed, 54, "0.000000000931322519104327284367172978818416595458984375"},
    {0x1.fffffep-30f, chars_format::fixed, 53, "0.00000000186264503820865456873434595763683319091796875"},
    {0x1.fffffep-29f, chars_format::fixed, 52, "0.0000000037252900764173091374686919152736663818359375"},
    {0x1.fffffep-28f, chars_format::fixed, 51, "0.000000007450580152834618274937383830547332763671875"},
    {0x1.fffffep-27f, chars_format::fixed, 50, "0.00000001490116030566923654987476766109466552734375"},
    {0x1.fffffep-26f, chars_format::fixed, 49, "0.0000000298023206113384730997495353221893310546875"},
    {0x1.fffffep-25f, chars_format::fixed, 48, "0.000000059604641222676946199499070644378662109375"},
    {0x1.fffffep-24f, chars_format::fixed, 47, "0.00000011920928244535389239899814128875732421875"},
    {0x1.fffffep-23f, chars_format::fixed, 46, "0.0000002384185648907077847979962825775146484375"},
    {0x1.fffffep-22f, chars_format::fixed, 45, "0.000000476837129781415569595992565155029296875"},
    {0x1.fffffep-21f, chars_format::fixed, 44, "0.00000095367425956283113919198513031005859375"},
    {0x1.fffffep-20f, chars_format::fixed, 43, "0.0000019073485191256622783839702606201171875"},
    {0x1.fffffep-19f, chars_format::fixed, 42, "0.000003814697038251324556767940521240234375"},
    {0x1.fffffep-18f, chars_format::fixed, 41, "0.00000762939407650264911353588104248046875"},
    {0x1.fffffep-17f, chars_format::fixed, 40, "0.0000152587881530052982270717620849609375"},
    {0x1.fffffep-16f, chars_format::fixed, 39, "0.000030517576306010596454143524169921875"},
    {0x1.fffffep-15f, chars_format::fixed, 38, "0.00006103515261202119290828704833984375"},
    {0x1.fffffep-14f, chars_format::fixed, 37, "0.0001220703052240423858165740966796875"},
    {0x1.fffffep-13f, chars_format::fixed, 36, "0.000244140610448084771633148193359375"},
    {0x1.fffffep-12f, chars_format::fixed, 35, "0.00048828122089616954326629638671875"},
    {0x1.fffffep-11f, chars_format::fixed, 34, "0.0009765624417923390865325927734375"},
    {0x1.fffffep-10f, chars_format::fixed, 33, "0.001953124883584678173065185546875"},
    {0x1.fffffep-9f, chars_format::fixed, 32, "0.00390624976716935634613037109375"},
    {0x1.fffffep-8f, chars_format::fixed, 31, "0.0078124995343387126922607421875"},
    {0x1.fffffep-7f, chars_format::fixed, 30, "0.015624999068677425384521484375"},
    {0x1.fffffep-6f, chars_format::fixed, 29, "0.03124999813735485076904296875"},
    {0x1.fffffep-5f, chars_format::fixed, 28, "0.0624999962747097015380859375"},
    {0x1.fffffep-4f, chars_format::fixed, 27, "0.124999992549419403076171875"},
    {0x1.fffffep-3f, chars_format::fixed, 26, "0.24999998509883880615234375"},
    {0x1.fffffep-2f, chars_format::fixed, 25, "0.4999999701976776123046875"},
    {0x1.fffffep-1f, chars_format::fixed, 24, "0.999999940395355224609375"},
    {0x1.fffffep+0f, chars_format::fixed, 23, "1.99999988079071044921875"},
    {0x1.fffffep+1f, chars_format::fixed, 22, "3.9999997615814208984375"},
    {0x1.fffffep+2f, chars_format::fixed, 21, "7.999999523162841796875"},
    {0x1.fffffep+3f, chars_format::fixed, 20, "15.99999904632568359375"},
    {0x1.fffffep+4f, chars_format::fixed, 19, "31.9999980926513671875"},
    {0x1.fffffep+5f, chars_format::fixed, 18, "63.999996185302734375"},
    {0x1.fffffep+6f, chars_format::fixed, 17, "127.99999237060546875"},
    {0x1.fffffep+7f, chars_format::fixed, 16, "255.9999847412109375"},
    {0x1.fffffep+8f, chars_format::fixed, 15, "511.999969482421875"},
    {0x1.fffffep+9f, chars_format::fixed, 14, "1023.99993896484375"},
    {0x1.fffffep+10f, chars_format::fixed, 13, "2047.9998779296875"},
    {0x1.fffffep+11f, chars_format::fixed, 12, "4095.999755859375"},
    {0x1.fffffep+12f, chars_format::fixed, 11, "8191.99951171875"},
    {0x1.fffffep+13f, chars_format::fixed, 10, "16383.9990234375"},
    {0x1.fffffep+14f, chars_format::fixed, 9, "32767.998046875"},
    {0x1.fffffep+15f, chars_format::fixed, 8, "65535.99609375"},
    {0x1.fffffep+16f, chars_format::fixed, 7, "131071.9921875"},
    {0x1.fffffep+17f, chars_format::fixed, 6, "262143.984375"},
    {0x1.fffffep+18f, chars_format::fixed, 5, "524287.96875"},
    {0x1.fffffep+19f, chars_format::fixed, 4, "1048575.9375"},
    {0x1.fffffep+20f, chars_format::fixed, 3, "2097151.875"},
    {0x1.fffffep+21f, chars_format::fixed, 2, "4194303.75"},
    {0x1.fffffep+22f, chars_format::fixed, 1, "8388607.5"},
    {0x1.fffffep+23f, chars_format::fixed, 0, "16777215"},
    {0x1.fffffep+24f, chars_format::fixed, 0, "33554430"},
    {0x1.fffffep+25f, chars_format::fixed, 0, "67108860"},
    {0x1.fffffep+26f, chars_format::fixed, 0, "134217720"},
    {0x1.fffffep+27f, chars_format::fixed, 0, "268435440"},
    {0x1.fffffep+28f, chars_format::fixed, 0, "536870880"},
    {0x1.fffffep+29f, chars_format::fixed, 0, "1073741760"},
    {0x1.fffffep+30f, chars_format::fixed, 0, "2147483520"},
    {0x1.fffffep+31f, chars_format::fixed, 0, "4294967040"},
    {0x1.fffffep+32f, chars_format::fixed, 0, "8589934080"},
    {0x1.fffffep+33f, chars_format::fixed, 0, "17179868160"},
    {0x1.fffffep+34f, chars_format::fixed, 0, "34359736320"},
    {0x1.fffffep+35f, chars_format::fixed, 0, "68719472640"},
    {0x1.fffffep+36f, chars_format::fixed, 0, "137438945280"},
    {0x1.fffffep+37f, chars_format::fixed, 0, "274877890560"},
    {0x1.fffffep+38f, chars_format::fixed, 0, "549755781120"},
    {0x1.fffffep+39f, chars_format::fixed, 0, "1099511562240"},
    {0x1.fffffep+40f, chars_format::fixed, 0, "2199023124480"},
    {0x1.fffffep+41f, chars_format::fixed, 0, "4398046248960"},
    {0x1.fffffep+42f, chars_format::fixed, 0, "8796092497920"},
    {0x1.fffffep+43f, chars_format::fixed, 0, "17592184995840"},
    {0x1.fffffep+44f, chars_format::fixed, 0, "35184369991680"},
    {0x1.fffffep+45f, chars_format::fixed, 0, "70368739983360"},
    {0x1.fffffep+46f, chars_format::fixed, 0, "140737479966720"},
    {0x1.fffffep+47f, chars_format::fixed, 0, "281474959933440"},
    {0x1.fffffep+48f, chars_format::fixed, 0, "562949919866880"},
    {0x1.fffffep+49f, chars_format::fixed, 0, "1125899839733760"},
    {0x1.fffffep+50f, chars_format::fixed, 0, "2251799679467520"},
    {0x1.fffffep+51f, chars_format::fixed, 0, "4503599358935040"},
    {0x1.fffffep+52f, chars_format::fixed, 0, "9007198717870080"},
    {0x1.fffffep+53f, chars_format::fixed, 0, "18014397435740160"},
    {0x1.fffffep+54f, chars_format::fixed, 0, "36028794871480320"},
    {0x1.fffffep+55f, chars_format::fixed, 0, "72057589742960640"},
    {0x1.fffffep+56f, chars_format::fixed, 0, "144115179485921280"},
    {0x1.fffffep+57f, chars_format::fixed, 0, "288230358971842560"},
    {0x1.fffffep+58f, chars_format::fixed, 0, "576460717943685120"},
    {0x1.fffffep+59f, chars_format::fixed, 0, "1152921435887370240"},
    {0x1.fffffep+60f, chars_format::fixed, 0, "2305842871774740480"},
    {0x1.fffffep+61f, chars_format::fixed, 0, "4611685743549480960"},
    {0x1.fffffep+62f, chars_format::fixed, 0, "9223371487098961920"},
    {0x1.fffffep+63f, chars_format::fixed, 0, "18446742974197923840"},
    {0x1.fffffep+64f, chars_format::fixed, 0, "36893485948395847680"},
    {0x1.fffffep+65f, chars_format::fixed, 0, "73786971896791695360"},
    {0x1.fffffep+66f, chars_format::fixed, 0, "147573943793583390720"},
    {0x1.fffffep+67f, chars_format::fixed, 0, "295147887587166781440"},
    {0x1.fffffep+68f, chars_format::fixed, 0, "590295775174333562880"},
    {0x1.fffffep+69f, chars_format::fixed, 0, "1180591550348667125760"},
    {0x1.fffffep+70f, chars_format::fixed, 0, "2361183100697334251520"},
    {0x1.fffffep+71f, chars_format::fixed, 0, "4722366201394668503040"},
    {0x1.fffffep+72f, chars_format::fixed, 0, "9444732402789337006080"},
    {0x1.fffffep+73f, chars_format::fixed, 0, "18889464805578674012160"},
    {0x1.fffffep+74f, chars_format::fixed, 0, "37778929611157348024320"},
    {0x1.fffffep+75f, chars_format::fixed, 0, "75557859222314696048640"},
    {0x1.fffffep+76f, chars_format::fixed, 0, "151115718444629392097280"},
    {0x1.fffffep+77f, chars_format::fixed, 0, "302231436889258784194560"},
    {0x1.fffffep+78f, chars_format::fixed, 0, "604462873778517568389120"},
    {0x1.fffffep+79f, chars_format::fixed, 0, "1208925747557035136778240"},
    {0x1.fffffep+80f, chars_format::fixed, 0, "2417851495114070273556480"},
    {0x1.fffffep+81f, chars_format::fixed, 0, "4835702990228140547112960"},
    {0x1.fffffep+82f, chars_format::fixed, 0, "9671405980456281094225920"},
    {0x1.fffffep+83f, chars_format::fixed, 0, "19342811960912562188451840"},
    {0x1.fffffep+84f, chars_format::fixed, 0, "38685623921825124376903680"},
    {0x1.fffffep+85f, chars_format::fixed, 0, "77371247843650248753807360"},
    {0x1.fffffep+86f, chars_format::fixed, 0, "154742495687300497507614720"},
    {0x1.fffffep+87f, chars_format::fixed, 0, "309484991374600995015229440"},
    {0x1.fffffep+88f, chars_format::fixed, 0, "618969982749201990030458880"},
    {0x1.fffffep+89f, chars_format::fixed, 0, "1237939965498403980060917760"},
    {0x1.fffffep+90f, chars_format::fixed, 0, "2475879930996807960121835520"},
    {0x1.fffffep+91f, chars_format::fixed, 0, "4951759861993615920243671040"},
    {0x1.fffffep+92f, chars_format::fixed, 0, "9903519723987231840487342080"},
    {0x1.fffffep+93f, chars_format::fixed, 0, "19807039447974463680974684160"},
    {0x1.fffffep+94f, chars_format::fixed, 0, "39614078895948927361949368320"},
    {0x1.fffffep+95f, chars_format::fixed, 0, "79228157791897854723898736640"},
    {0x1.fffffep+96f, chars_format::fixed, 0, "158456315583795709447797473280"},
    {0x1.fffffep+97f, chars_format::fixed, 0, "316912631167591418895594946560"},
    {0x1.fffffep+98f, chars_format::fixed, 0, "633825262335182837791189893120"},
    {0x1.fffffep+99f, chars_format::fixed, 0, "1267650524670365675582379786240"},
    {0x1.fffffep+100f, chars_format::fixed, 0, "2535301049340731351164759572480"},
    {0x1.fffffep+101f, chars_format::fixed, 0, "5070602098681462702329519144960"},
    {0x1.fffffep+102f, chars_format::fixed, 0, "10141204197362925404659038289920"},
    {0x1.fffffep+103f, chars_format::fixed, 0, "20282408394725850809318076579840"},
    {0x1.fffffep+104f, chars_format::fixed, 0, "40564816789451701618636153159680"},
    {0x1.fffffep+105f, chars_format::fixed, 0, "81129633578903403237272306319360"},
    {0x1.fffffep+106f, chars_format::fixed, 0, "162259267157806806474544612638720"},
    {0x1.fffffep+107f, chars_format::fixed, 0, "324518534315613612949089225277440"},
    {0x1.fffffep+108f, chars_format::fixed, 0, "649037068631227225898178450554880"},
    {0x1.fffffep+109f, chars_format::fixed, 0, "1298074137262454451796356901109760"},
    {0x1.fffffep+110f, chars_format::fixed, 0, "2596148274524908903592713802219520"},
    {0x1.fffffep+111f, chars_format::fixed, 0, "5192296549049817807185427604439040"},
    {0x1.fffffep+112f, chars_format::fixed, 0, "10384593098099635614370855208878080"},
    {0x1.fffffep+113f, chars_format::fixed, 0, "20769186196199271228741710417756160"},
    {0x1.fffffep+114f, chars_format::fixed, 0, "41538372392398542457483420835512320"},
    {0x1.fffffep+115f, chars_format::fixed, 0, "83076744784797084914966841671024640"},
    {0x1.fffffep+116f, chars_format::fixed, 0, "166153489569594169829933683342049280"},
    {0x1.fffffep+117f, chars_format::fixed, 0, "332306979139188339659867366684098560"},
    {0x1.fffffep+118f, chars_format::fixed, 0, "664613958278376679319734733368197120"},
    {0x1.fffffep+119f, chars_format::fixed, 0, "1329227916556753358639469466736394240"},
    {0x1.fffffep+120f, chars_format::fixed, 0, "2658455833113506717278938933472788480"},
    {0x1.fffffep+121f, chars_format::fixed, 0, "5316911666227013434557877866945576960"},
    {0x1.fffffep+122f, chars_format::fixed, 0, "10633823332454026869115755733891153920"},
    {0x1.fffffep+123f, chars_format::fixed, 0, "21267646664908053738231511467782307840"},
    {0x1.fffffep+124f, chars_format::fixed, 0, "42535293329816107476463022935564615680"},
    {0x1.fffffep+125f, chars_format::fixed, 0, "85070586659632214952926045871129231360"},
    {0x1.fffffep+126f, chars_format::fixed, 0, "170141173319264429905852091742258462720"},
    {0x1.fffffep+127f, chars_format::fixed, 0, "340282346638528859811704183484516925440"},
};

#endif // FLOAT_FIXED_PRECISION_TO_CHARS_TEST_CASES_HPP
