/**************************************************************************\
 gatos (General ATI TV and Overlay Software)

  Project Coordinated By Insomnia (Steaphan Greene)
  (insomnia@core.binghamton.edu)

  Copyright (C) 1999 Steaphan Greene, yvind Aabling, Octavian Purdila, 
	Vladimir Dergachev and Christian Lupien.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.

\**************************************************************************/

#define GATOS_ATIR128_C 1

/* Define to 0 for normal operation, define to 1 for
 * display of capture buffers in upper left corner */
#define VISIBLE_BUFFERS 0

#if VISIBLE_BUFFERS
#define BUF0		0
#define BUF1		0
#define BYTESTRIDE	(gatos.xdim*stride)
#define PIXELSTRIDE	((gatos.xdim*stride)>>1)
#else
#define BUF0		gatos.buffer0
#define BUF1		gatos.buffer1
#define BYTESTRIDE	(gatos.xcapt+gatos.xcapt)
#define PIXELSTRIDE	(gatos.xcapt)
#endif

#include "gatos.h"
#include "ati.h"
#include "atir128.h"
#include "r128regs.h"

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <time.h>

/* Private global vars */
static u32 scale_cntl=DISABLE_R128_VIDEO ;
static int stride=0 ;	/* FrameBuffer pixel stride (bytes per pixel) */
static u32 save_offset=0, save_h_total_disp=0, save_v_total_disp=0 ;
static u8 ecp_div=0 ;
static int firstpoll=1 ;
#if 0
static int atishow=0 ;
#endif

static int step_by=2;

/* ------------------------------------------------------------------------ */
/* Initialization routines */
void r128_ResetEngine(void)
{
fprintf(stderr,"Resetting r128 engine.. Aiee !\n");
}

void r128_WaitForFifo(int entries)
{
time_t t;
if(entries>64)entries=64;
t=time(NULL);
while((R128_GUI_STAT & 0xfff)<entries){
	if(t-time(NULL)>1){
		r128_ResetEngine();
		}
	usleep(10);
	}
}


int r128_inita(void) {

  u16 id=R128_DEVICE_ID ; u8 rev=R128_REVISION_ID ; int d, v, m ;

  /* Chip register aperture enabled (chip in use by X server) ? */
  if (id == 0xFFFF) RETURN(ENXIO) ;

  /* Check the FrameBuffer address */
  if ( (R128_CONFIG_APER_0_BASE & 0xFC000000) != gatos.fbaddr ) RETURN(EINVAL) ;

  /* ATI chip device id and chip revision */
  d = gatos.ati.deviceid = id ;			/* Device ID */
  v = gatos.ati.revision = (rev & 0xF0) >> 4 ;	/* Chip Version */
  m = (rev & 0x0F) ;				/* Chip Version Minor */

  /* Amount of Video RAM in kilobytes */
  gatos.videoram = R128_CONFIG_MEMSIZE >> 10 ;

  R128_VIDEOMUX_CNTL=0x50781;
  RETURN0 ; }


int r128_initb(void) {

  /* Place video capture buffers at top of video memory */
  // hack - we need twice as many ram for weave
  gatos.buffermem *= 2 ;
  gatos.captbufsize = (gatos.buffermem) >> 1 ;
  gatos.buffer0 = 1024*(gatos.videoram) - 1024*gatos.captbufsize ;
  gatos.buffer1 = gatos.buffer0         - 1024*gatos.captbufsize ;

  r128_WaitForFifo(64) ;

  CAP0_BUF0_OFFSET = BUF0 ; CAP0_BUF0_EVEN_OFFSET = BUF0+2*gatos.xcapt ;
  CAP0_BUF1_OFFSET = BUF1 ; CAP0_BUF1_EVEN_OFFSET = BUF1-2*gatos.xcapt ;
  CAP0_ONESHOT_BUF_OFFSET = BUF0 ;


  /* Initial ATI register programming */

  scale_cntl     = ENABLE_R128_VIDEO ;
  CAP0_TRIG_CNTL = 0x00000000L ;
  OV0_SCALE_CNTL = DISABLE_R128_VIDEO ;

  OV0_VIDEO_KEY_CLR = 0x00000000L ;
  OV0_VIDEO_KEY_MSK = 0x00000000L ;
  OV0_KEY_CNTL      = 0x00000050L ;

  /* ??? */
  OV0_FILTER_CNTL     = 0x0000000FL ;

//OV0_AUTO_FLIP_CNTRL = 0x00040000L;

  OV0_COLOUR_CNTL     = 0x00101000L ;
//OV0_COLOUR_CNTL     = 0x00FFFFFFL ;

  OV0_FOUR_TAP_COEF_0 = 0x00002000L ;
  OV0_FOUR_TAP_COEF_1 = 0x0D06200DL ;
  OV0_FOUR_TAP_COEF_2 = 0x0D0A1C0DL ;
  OV0_FOUR_TAP_COEF_3 = 0x0C0E1A0CL ;
  OV0_FOUR_TAP_COEF_4 = 0x0C14140CL ;

  /* Suspicious (No it isn't, ATI sample code says do this /AA) */
  /* we need 1 just in case bit 0 has been unset */
  R128_VIDEOMUX_CNTL |= 2 | 1;

  CAP0_PORT_MODE_CNTL = 0 ;
  CAP0_BUF_PITCH = BYTESTRIDE ;
  CAP0_H_WINDOW  = BYTESTRIDE<<16 ;
  
  OV0_VID_BUF0_BASE_ADRS =   CAP0_BUF1_EVEN_OFFSET & (~3) ;
  OV0_VID_BUF1_BASE_ADRS = ( CAP0_BUF1_EVEN_OFFSET & (~3) ) | 1 ;
  OV0_VID_BUF2_BASE_ADRS = ( CAP0_BUF1_EVEN_OFFSET & (~3) ) | 1 ;
  
  OV0_VID_BUF3_BASE_ADRS =  gatos.buffer1 & (~3) ;
  OV0_VID_BUF4_BASE_ADRS = (gatos.buffer1 & (~3) ) | 1 ;
  OV0_VID_BUF5_BASE_ADRS = (gatos.buffer1 & (~3) ) | 1 ;
  
//  OV0_VID_BUF_PITCH0_VALUE = BYTESTRIDE<<1 ;
//  OV0_VID_BUF_PITCH1_VALUE = BYTESTRIDE<<1 ;

  OV0_VID_BUF_PITCH0_VALUE = PIXELSTRIDE<<2 ;

  OV0_VID_BUF_PITCH1_VALUE = PIXELSTRIDE<<2 ;

  step_by=2;
  OV0_STEP_BY = 2 | (2<<8) ;


//OV0_DEINTERLACE_PATTERN=0x20000000 | (3 | (3 << 2) | (3 <<4));
  OV0_DEINTERLACE_PATTERN = 1 ;

//CAP0_V_WINDOW  =(( v-1+(gatos.ycapt>>1)) <<16 )| (v) ;
//CAP0_TRIG_CNTL = 0x00000011L ;

  CAP0_DEBUG=0;


  #if 0
  VIDEO_FORMAT      = 0x000B000BL ;
  #endif
  ati_setcolorkey() ; ati_pollscreen(0) ; RETURN0 ; }

void stream_enable(int enable) {
  R128_CLOCK_CNTL_INDEX0 = (FCP_CNTL & 0x1F) | 0x80 ;
  if (enable) {
    R128_CLOCK_CNTL_DATA = 0x101 ;
    CAP0_TRIG_CNTL = 0x00000011L ; }	
  else {
    CAP0_TRIG_CNTL = 0x00000000L ;
    R128_CLOCK_CNTL_DATA = 0x404 ; } }

/* ------------------------------------------------------------------------ */
/* Public routines */

/* Overlay Window Colorkeying */
int r128_setcolorkey(void) {
  if ((R128_CRTC_GEN_CNTL & 0x01000000L))	/* Extended mode */
    switch ((R128_CRTC_GEN_CNTL & 0x00000700L) >> 8) {
      case 1: OV0_GRAPHICS_KEY_MSK = 0x0000000FL ; stride = 1 ; break ;
      case 2: OV0_GRAPHICS_KEY_MSK = 0x000000FFL ; stride = 1 ; break ;
      case 3: OV0_GRAPHICS_KEY_MSK = 0x00007FFFL ; stride = 2 ; break ;
      case 4: OV0_GRAPHICS_KEY_MSK = 0x0000FFFFL ; stride = 2 ; break ;
      case 5: OV0_GRAPHICS_KEY_MSK = 0x00FFFFFFL ; stride = 4 ; break ;
      case 6: OV0_GRAPHICS_KEY_MSK = 0x00FFFFFFL ; stride = 4 ; break ;
      default: RETURN(ENODEV) ; break ; }
  else {					/* VGA mode: assume 8bpp */
    OV0_GRAPHICS_KEY_MSK = 0x000000FFL ; stride = 1 ; }
  OV0_GRAPHICS_KEY_CLR = gatos.colorkey ;
  RETURN0 ; }

/* Enable/display video capture to capture buffers */
void r128_enable_capture(int enable) {
  CAP0_CONFIG = (enable) ? ENABLE_R128_CAPTURE : 0x00000000L ;
}

/* Enable/disable video display in Overlay Window */
void r128_enable_video(int enable) {
  
   if (!enable) {
    OV0_EXCLUSIVE_HORZ = OV0_EXCLUSIVE_VERT = 0x00000000 ; }
  OV0_SCALE_CNTL = (enable) ? scale_cntl : DISABLE_R128_VIDEO ;
  if(enable){
//  	OV0_SCALE_CNTL|=0x80000000L;
//  	OV0_REG_LOAD_CNTL&=~1;
  CAP0_CONFIG=ENABLE_R128_CAPTURE;
  CAP0_TRIG_CNTL = 0x00000011L ;
  	}
  stream_enable(enable);
}

/* Capture Buffer Mode */
void r128_buffer_mode(void) {
  /* TODO: Single/double buffer, field/frame capture */
}

/* ------------------------------------------------------------------------ */
/* Size in pixels of incoming video stream (full frame) */
int r128_setcaptsize(void) {

  int v=0 ;

//  *** Have to come up with a better way to do this  --ISG
//  if (gatos.xcapt > hactive[gatos.hcaptmax]) RETURN(EOVERFLOW) ;
//  if (gatos.ycapt > vactive[gatos.vcaptmax]) RETURN(EOVERFLOW) ;

  /* NTSC, PAL or SECAM ? */
  switch (gatos.format) {
    case 1: case 2: case 4:			/* NTSC */
      v = 23 ;
//  *** Have to come up with a better way to do this  --ISG
//      if (gatos.xcapt > 640) RETURN(EINVAL) ;
//      if (gatos.ycapt > 480) RETURN(EINVAL) ;
      if (gatos.ycapt <= 360) v = 22 ;
      if (gatos.ycapt <= 240) v = 21 ;
      if (gatos.ycapt <= 120) v = 20 ;
      if (gatos.ycapt <=  60) v = 19 ; break ;
    case 3: case 5: case 7:			/* PAL */
      v = 24 ;
      if (gatos.xcapt > 768) RETURN(EINVAL) ;
      if (gatos.ycapt > 576) RETURN(EINVAL) ;
      if (gatos.ycapt <= 432) v = 23 ;
      if (gatos.ycapt <= 288) v = 21 ;
      if (gatos.ycapt <= 144) v = 19 ;
      if (gatos.ycapt <=  72) v = 18 ; break ;
    case 6:					/* SECAM */
      v = 25 ;
      if (gatos.xcapt > 768) RETURN(EINVAL) ;
      if (gatos.ycapt > 576) RETURN(EINVAL) ;
      if (gatos.ycapt <= 432) v = 24 ;
      if (gatos.ycapt <= 288) v = 22 ;
      if (gatos.ycapt <= 144) v = 20 ;
      if (gatos.ycapt <=  72) v = 19 ; break ;
    default: RETURN(EINVAL) ; }

#if defined(GATOSBUTTONS) && GATOSBUTTONS >= 1
  v += gatosbuttons[0] ;
#endif

  OV0_SCALE_CNTL = DISABLE_R128_VIDEO ;
  CAP0_BUF_PITCH = 2*2*gatos.xcapt ;
  CAP0_H_WINDOW  = (2*gatos.xcapt)<<16 ;
  CAP0_V_WINDOW  = (( v-1+(gatos.ycapt>>1)) <<16 )| (v) ;

/* we only know what capture size actually is here */
  OV0_VID_BUF_PITCH0_VALUE = PIXELSTRIDE*2*2 ;
  OV0_VID_BUF_PITCH1_VALUE = PIXELSTRIDE*2*2 ;
  CAP0_BUF0_EVEN_OFFSET = BUF0+2*gatos.xcapt ;
  CAP0_BUF1_EVEN_OFFSET = BUF1-2*gatos.xcapt ;

  OV0_VID_BUF0_BASE_ADRS =   CAP0_BUF0_OFFSET & (~3) ;
  OV0_VID_BUF1_BASE_ADRS = ( CAP0_BUF0_EVEN_OFFSET & (~3) ) | 1 ;
  OV0_VID_BUF2_BASE_ADRS = ( CAP0_BUF0_EVEN_OFFSET & (~3) ) | 1 ;
  
  OV0_VID_BUF3_BASE_ADRS =   CAP0_BUF1_EVEN_OFFSET & (~3) ;
  OV0_VID_BUF4_BASE_ADRS = ( CAP0_BUF1_OFFSET & (~3) ) | 1 ;
  OV0_VID_BUF5_BASE_ADRS = ( CAP0_BUF1_OFFSET & (~3) ) | 1 ;
//  OV0_VID_BUF3_BASE_ADRS =  gatos.buffer1 & (~3) ;
//  OV0_VID_BUF4_BASE_ADRS = (gatos.buffer1 & (~3) ) | 1 ;
//  OV0_VID_BUF5_BASE_ADRS = (gatos.buffer1 & (~3) ) | 1 ;


  OV0_P1_X_START_END=(gatos.xcapt+0)|(0<<16);
//OV0_P1_X_START_END=0;  // disable for debugging
  OV0_P3_X_START_END= // this register is redundant
  OV0_P2_X_START_END=((gatos.xcapt>>1)+1)|(1<<16);

  CAP0_TRIG_CNTL = 0x00000011L ;
  if (gatos.video) OV0_SCALE_CNTL = scale_cntl ;

  RETURN0 ; }

/* ------------------------------------------------------------------------ */

int r128_pollscreen(int sig) {

  u32 vclk_ecp_cntl, pll_ref_div, vclk_post_div, vclk_fb_div ;
  u32 disp, pitch ; int i ; u8 save ;

  /* Return if called from signal handler and nothing changed */
  if ( sig && ((R128_CRTC_OFFSET&0x01FFFFFFL) == save_offset)
    && (R128_CRTC_H_TOTAL_DISP == save_h_total_disp)
    && (R128_CRTC_V_TOTAL_DISP == save_v_total_disp) ) return 0 ;

  /* Remember new values for next time */
  save_offset = R128_CRTC_OFFSET ;
  save_h_total_disp = R128_CRTC_H_TOTAL_DISP ;
  save_v_total_disp = R128_CRTC_V_TOTAL_DISP ;

  /* Get monitor dotclock, check for Overlay Scaler clock limit */
  save = R128_CLOCK_CNTL_INDEX0 ; i = R128_CLOCK_CNTL_INDEX1 & 3 ;
  R128_CLOCK_CNTL_INDEX0 = 3 ;
  pll_ref_div = R128_CLOCK_CNTL_DATA & 0x000003FFL ;
  R128_CLOCK_CNTL_INDEX0 = 8 ;
  vclk_ecp_cntl = R128_CLOCK_CNTL_DATA ;
  R128_CLOCK_CNTL_INDEX0 = 4+i ;
  vclk_fb_div = R128_CLOCK_CNTL_DATA & 0x000007FFL ;
  vclk_post_div = (R128_CLOCK_CNTL_DATA & 0x00070000L) >> 16 ;
  gatos.dotclock = /* 2.0* */ gatos.refclock*vclk_fb_div
                 / (pll_ref_div*(1<<vclk_post_div)) ;

  /* ecp_div: 0=dotclock, 1=dotclock/2, 2=dotclock/4 */
  ecp_div = gatos.dotclock /
    (gatos.aticard[gatos.cardidx].dotclock + gatos.overclock) ;
  if (ecp_div>2) ecp_div = 2 ;
  /* Force a scaler clock factor of 1 if refclock *
   * is unknown (VCLK_SRC not PLLVCLK)       /AA */
  if ((vclk_ecp_cntl & 0x00000003L) != 0x00000003L)
    gatos.dotclock = ecp_div = 0 ;
  if ((vclk_ecp_cntl & 0x00000300L) != ecp_div<<8) {
    R128_CLOCK_CNTL_INDEX0 = 0x88 ;
    R128_CLOCK_CNTL_DATA = (vclk_ecp_cntl&0xFFFFFCFFL) | (ecp_div<<8) ; }

  /* Restore PLL Register Index */
  R128_CLOCK_CNTL_INDEX0 = save ;

  /* Screen (monitor) dimensions */
  if ((R128_CRTC_GEN_CNTL & 0x01000000L)) {	/* Extended mode */
    gatos.xdim = (R128_CRTC_H_TOTAL_DISP>>16) + 1 ; gatos.xdim *= 8 ;
    gatos.ydim = (R128_CRTC_V_TOTAL_DISP>>16) + 1 ; }
  else {				/* VGA mode */
    /* TODO: decode VGA registers */
    gatos.xdim = 1280 ; gatos.ydim = 1024 ; }

  /* Desktop panning */
  disp = (R128_CRTC_OFFSET & 0x01FFFFFFL) / stride ;
  pitch = 8 * (R128_CRTC_PITCH & 0x000003FFL) ;
  gatos.ypan = disp/pitch ; gatos.xpan = disp - pitch*gatos.ypan ;
/*  printf("%d,%d (%d,%d)\n", disp, pitch, gatos.xpan, gatos.ypan); */

  /* Tell user about the new values */
  if (GATOSASYNC || (VERBOSE && firstpoll)) fprintf(stderr,
    "%s Screen %dx%d+%d+%d, Dotclock %f MHz, Scaler Clock Factor %d\n",
      (firstpoll)?"==>":"***", gatos.xdim, gatos.ydim,
      gatos.xpan, gatos.ypan, gatos.dotclock, 1<<ecp_div) ;
  firstpoll = 0 ;

  /* Caller should call ati_setgeometry() */
  return 1 ; }

/* ------------------------------------------------------------------------ */
/* Size and position of Overlay Scaler window */
void r128_setgeometry(void) {
  long tmp;
  int xs, ys, xc, yc, x0, y0, x1, x2, y1, y2, xscale,xscale_color, yscale, offset=0 ;
  int xss=4096, yss=2048, ysize, ypos, ypan ; double xsf, ysf ;
  static int port=1;
#if 0
  int h1, h2, h3 ;
#endif

  /* Subgeometry scale factors */
  xsf = (gatos.xend-gatos.xstart)/(double)gatos.xcapt ; xss *= xsf ;
  ysf = (gatos.yend-gatos.ystart)/(double)gatos.ycapt ; yss *= ysf ;

  /* Doublescan mode ? */
  ysize = gatos.ysize ;  ypos = gatos.ypos ; ypan = gatos.ypan ;
  if (R128_CRTC_GEN_CNTL&1) { ysize *= 2 ; ypos *= 2 ; ypan *= 2 ; }

  /* Window size and position */
  xs = gatos.xsize ; x0 = gatos.xpos - gatos.xpan ;
  ys = ysize ; y0 = ypos - ypan ;

  /* Disable video before reprogramming registers */
  OV0_SCALE_CNTL = DISABLE_R128_VIDEO ;

  /* Return (with video disabled) if window totally outside screen */
  if (x0+gatos.xsize <= 0 || y0+ysize <= 0 ||
      x0 >= gatos.xdim || y0 >= gatos.ydim) return ;

  /* Number of pixels outside screen:
   * x1=left, x2=right, y1=top, y2=bottom */
  x1 = -x0 ; x2 = x0+xs-gatos.xdim ; if (x1<0) x1 = 0 ; if (x2<0) x2 = 0 ;
  y1 = -y0 ; y2 = y0+ys-gatos.ydim ; if (y1<0) y1 = 0 ; if (y2<0) y2 = 0 ;

  /* Size and position of on-screen part of window */
  xs -= x1+x2 ; if (x0<0) x0 = 0 ;
  ys -= y1+y2 ; if (y0<0) y0 = 0 ;

  /* Video scaling factors */
  xc = gatos.xcapt*xs/gatos.xsize ; xscale = xc*xss*(1<<ecp_div)/xs ;
   xscale_color=xc*xss*(1<<ecp_div)/(xs<<1) ;
  yc = gatos.ycapt*ys/ysize ; yscale = 2*yc*yss/ys ;

  /* Offset Scaler buffers from Capture buffers *
   * if upper left window corner outside screen */
  if (x1+y1) offset = 2*x1*xsf*gatos.xcapt/gatos.xsize +
    (((int)(y1*ysf*gatos.ycapt/ysize))&~1) * gatos.xcapt * 2;

  /* Subgeometry offset factors */
  if (gatos.xstart)
    offset += 2*gatos.xstart ;
  if (gatos.ystart)
    offset += (gatos.ystart&~1)*gatos.xcapt*2;

  /* Exclusive Overlay Region (colorkey ignored) TEST/AA */
#if 0
  h1 = (x0>>3) + 1 ; h2 = (x0+xs)>>3 ; h3 = (gatos.xdim-8*h2)>>3 ;
  OVERLAY_EXCLUSIVE_HORZ = 0x00000000 ;
  if (gatos.visibility==0 && !atishow && (h2-h1)>=8) {
    OV0_EXCLUSIVE_VERT = XY(y0+ys-1,y0) ;
    OV0_EXCLUSIVE_HORZ = 0x80000000 | XYZ(h3,h2,h1) ; }
#endif

  #if 0
  /* Program the chip */
  SCALER_BUF0_OFFSET = gatos.buffer0+offset ;
  SCALER_BUF1_OFFSET = gatos.buffer1+offset ;
  SCALER_HEIGHT_WIDTH = XY((int)(xc*xsf),((int)(yc*ysf))>>1) ;

  OV0_H_INC = xscale | (xscale << 16); /* ??? */
  OV0_V_INC = yscale << ( 7 + ((R128_CRTC_GEN_CNTL&2)==2) ) ; /* ??? */
  #endif

  OV0_H_INC = (xscale/step_by) | ((xscale_color/step_by) << 16); /* ??? */
  OV0_V_INC = yscale << ( 8 + ((R128_CRTC_GEN_CNTL&2)==2) ) ; /* ??? */
  
//OV0_H_INC = (1<<12) | ((1<<12)<<16) ; /* ??? */
//OV0_V_INC = (1<<12)<<8 ; /* ??? */
//OV0_P1_BLANK_LINES_AT_TOP =    (((int)(xc*xsf)-1)<<16) | 0x0FFF ;
//OV0_P23_BLANK_LINES_AT_TOP =    (((int)(xc*xsf/2)-1)<<16) | 0x0FFF ;

  OV0_Y_X_START = XY(x0,y0) ;
  OV0_Y_X_END = XY(x0+xs-1,y0+ys-1) ; 

/* set up offset (for zoom) */

  OV0_VID_BUF0_BASE_ADRS =   (CAP0_BUF0_OFFSET +offset) & (~3) ;
  OV0_VID_BUF1_BASE_ADRS = ( (CAP0_BUF0_EVEN_OFFSET+offset) & (~3) ) | 1 ;
  OV0_VID_BUF2_BASE_ADRS = ( (CAP0_BUF0_EVEN_OFFSET+offset) & (~3) ) | 1 ;
  
  OV0_VID_BUF3_BASE_ADRS =   (CAP0_BUF1_EVEN_OFFSET+offset) & (~3) ;
  OV0_VID_BUF4_BASE_ADRS = ( (CAP0_BUF1_OFFSET+offset) & (~3) ) | 1 ;
  OV0_VID_BUF5_BASE_ADRS = ( (CAP0_BUF1_OFFSET+offset) & (~3) ) | 1 ;

  OV0_P1_X_START_END=(gatos.xcapt+0)|((offset & 3)<<16);
//OV0_P1_X_START_END=0;  // disable for debugging
  OV0_P3_X_START_END= // this register is redundant
  OV0_P2_X_START_END=((gatos.xcapt)+0)|((offset & 3)<<16);
//OV0_P3_X_START_END= // disable for debugging
//OV0_P2_X_START_END=0;
  
  OV0_P1_H_ACCUM_INIT=0x20000000 | (0);
  OV0_P23_H_ACCUM_INIT=0x20000000 | (0);
  
  OV0_P1_BLANK_LINES_AT_TOP = (((gatos.ycapt)-1) <<16) | 0xFFF ;
  OV0_P23_BLANK_LINES_AT_TOP = (((gatos.ycapt)-1) <<16) | 0xFFF ;

  port = 0 ; /* which port to use 0 , 1 -hardware, 2 - software */

  OV0_AUTO_FLIP_CNTRL = (port<<8)|(1<<6)|(0) ;
  tmp = OV0_AUTO_FLIP_CNTRL ;
  OV0_AUTO_FLIP_CNTRL = (port<<8)|(0<<6)|(0) ;
  
  OV0_TEST = 0 ;
  if (gatos.video) OV0_SCALE_CNTL = scale_cntl ;

}

/* ------------------------------------------------------------------------ */

/* Note:
 * Gamma and Red Temp correction only affects
 * video displayed in the Overlay Scaler window.
 * Data in the capture buffers is not affected.
 */

/* Overlay Scaler window gamma correction */
int r128_setgamma(void) {
  scale_cntl = ENABLE_R128_VIDEO ;
  if (gatos.video) OV0_SCALE_CNTL = scale_cntl ;
  RETURN0 ; }

/* Overlay Scaler window red color temp (cold=9800K or warm=6500K) */
int r128_setcold(void) {
  scale_cntl = ENABLE_R128_VIDEO ;
  if (gatos.video) OV0_SCALE_CNTL = scale_cntl ;
  RETURN0 ; }

/* ------------------------------------------------------------------------ */
/* Capture and playback routines */

void r128_rgbcapture(unsigned char *rgb) {
  unsigned char *buf0, *buf1;
  volatile unsigned char *abuf0=ATIFB+CAP0_BUF0_OFFSET,
                         *abuf1=ATIFB+CAP0_BUF1_OFFSET ;
  unsigned char buf[4];
  int i, j, y, cr, cb, r, g, b ;

  buf0 = (unsigned char*)malloc(PIXELSTRIDE*gatos.ycapt*2);
  buf1 = (unsigned char*)malloc(PIXELSTRIDE*gatos.ycapt*2);
  memcpy(buf0, (char*)abuf0, PIXELSTRIDE*gatos.ycapt*2);
  memcpy(buf1, (char*)abuf1, PIXELSTRIDE*gatos.ycapt*2);

  /* Capture (no sync to data arrival; requires kernel support :-( */
  for ( i=0 ; i<(gatos.ycapt*2) ; i+=4 ) {
    for ( j=0 ; j<gatos.xcapt ; j+=2 ) {
      memcpy(buf, (void*)(buf0+i*PIXELSTRIDE+j*2), 4) ;
      cb = buf[0] - 128 ; cr = buf[2] - 128 ;

      y = buf[1] - 16 ;
      r = ( 76284*y + 104595*cr             )>>16 ;
      g = ( 76284*y -  53281*cr -  25624*cb )>>16 ;
      b = ( 76284*y             + 132252*cb )>>16 ;
      *rgb++ = LIMIT(r, 0, 255) ;
      *rgb++ = LIMIT(g, 0, 255) ;
      *rgb++ = LIMIT(b, 0, 255) ;

      y = buf[3] - 16 ;
      r = ( 76284*y + 104595*cr             )>>16 ;
      g = ( 76284*y -  53281*cr -  25624*cb )>>16 ;
      b = ( 76284*y             + 132252*cb )>>16 ;
      *rgb++ = LIMIT(r, 0, 255) ;
      *rgb++ = LIMIT(g, 0, 255) ;
      *rgb++ = LIMIT(b, 0, 255) ;
      }
    for ( j=0 ; j<gatos.xcapt ; j+=2 ) {
      memcpy(buf, (void*)(buf1+i*PIXELSTRIDE+j*2), 4) ;
      cb = buf[0] - 128 ; cr = buf[2] - 128 ;

      y = buf[1] - 16 ;
      r = ( 76284*y + 104595*cr             )>>16 ;
      g = ( 76284*y -  53281*cr -  25624*cb )>>16 ;
      b = ( 76284*y             + 132252*cb )>>16 ;
      *rgb++ = LIMIT(r, 0, 255) ;
      *rgb++ = LIMIT(g, 0, 255) ;
      *rgb++ = LIMIT(b, 0, 255) ;

      y = buf[3] - 16 ;
      r = ( 76284*y + 104595*cr             )>>16 ;
      g = ( 76284*y -  53281*cr -  25624*cb )>>16 ;
      b = ( 76284*y             + 132252*cb )>>16 ;
      *rgb++ = LIMIT(r, 0, 255) ;
      *rgb++ = LIMIT(g, 0, 255) ;
      *rgb++ = LIMIT(b, 0, 255) ;
      }
    }
  free(buf0);
  free(buf1);
  }

void r128_capture(FILE *file, int wait) {

  volatile void *buf0=ATIFB+CAP0_BUF0_OFFSET,
                *buf1=ATIFB+CAP0_BUF1_OFFSET ;
  int i ;

  /* Capture (no sync to data arrival; requires kernel support :-( */
  while (1) {
    fprintf(file,"%3d %3d\n",gatos.xcapt,gatos.ycapt*2) ;
    for ( i=0 ; i<(gatos.ycapt*2) ; i+=2 ) {
      fwrite((void*)(buf0+i*PIXELSTRIDE),2,gatos.xcapt,file) ;
      fwrite((void*)(buf1+i*PIXELSTRIDE),2,gatos.xcapt,file) ; }
    fflush(file) ; if (gatos.stop) break ; if (wait) usleep(wait) ; }

}

void r128_playback(FILE *file, int wait) {

  int i, h, v, c, xsave=gatos.xcapt, ysave=gatos.ycapt, res ; char buf[9] ;
  volatile void *buf0=ATIFB+CAP0_BUF0_OFFSET,
                *buf1=ATIFB+CAP0_BUF1_OFFSET ;

  c = (CAP0_CONFIG) ? 1 : 0;
  gatos_enable_capture(0);
	
  /* Playback (no sync; requires kernel support :-( */
  while ( fread(buf,1,8,file) == 8 ) {
    res = sscanf(buf,"%3d %3d\n",&h,&v) ;
    h /= 2;
    if (h!=gatos.xcapt || v!=gatos.ycapt) {
      gatos_setcapturesize(gatos.xcapt,gatos.ycapt) ;
      gatos_setgeometry(gatos.xsize,gatos.ysize,gatos.xpos,gatos.ypos) ; }
    for ( i=0 ; i<(gatos.ycapt*2) ; i+=2 ) {
      fread((void*)(buf0+i*PIXELSTRIDE),2,gatos.xcapt,file) ;
      fread((void*)(buf1+i*PIXELSTRIDE),2,gatos.xcapt,file) ; }
    if (gatos.stop) break ; if (wait) usleep(wait) ; }

  if (gatos.xcapt!=xsave || gatos.ycapt!=ysave) {
    gatos_setcapturesize(xsave,ysave) ;
    gatos_setgeometry(gatos.xsize,gatos.ysize,gatos.xpos,gatos.ypos) ; }

  /* Done */
  gatos_enable_capture(c);

}

/* ------------------------------------------------------------------------ */
/* Debug and report routines */

#define DUMPREG(MEMADDR,NAME)	\
	fprintf(stderr,"%s: %-26s (%s) = 0x%08X\n", \
	gatos.ati.ident,#NAME,MEMADDR,NAME)

void r128_dumpregs(void) {
  int i ; u8 tmp ;
  DUMPREG("MMR_0008",R128_CLOCK_CNTL_INDEX) ;
  DUMPREG("MMR_000C",R128_CLOCK_CNTL_DATA) ;
  DUMPREG("MMR_0050",R128_CRTC_GEN_CNTL) ;
  DUMPREG("MMR_0070",R128_MPP_TB_ADDR) ;
  DUMPREG("MMR_0074",R128_MPP_TB_DATA) ;
  DUMPREG("MMR_0078",R128_MPP_GP_ADDR) ;
  DUMPREG("MMR_007C",R128_MPP_GP_DATA) ;
  DUMPREG("MMR_0088",R128_MPP_GP_ALT_REG_ADDR) ;
  DUMPREG("MMR_008C",R128_MPP_GP_ALT_REG_DATA) ;
  DUMPREG("MMR_0090",R128_I2C_CNTL_0) ;
  DUMPREG("MMR_0094",R128_I2C_CNTL_1) ;
  DUMPREG("MMR_0098",R128_I2C_DATA) ;
  DUMPREG("MMR_00F8",R128_CONFIG_MEMSIZE) ;
  DUMPREG("MMR_0100",R128_CONFIG_APER_0_BASE) ;
  DUMPREG("MMR_0194",R128_AMCGPIO_MASK) ;
  DUMPREG("MMR_0198",R128_MDGPIO_MASK) ;
  DUMPREG("MMR_01A0",R128_AMCGPIO_A_REG) ;
  DUMPREG("MMR_01A4",R128_AMCGPIO_Y_REG) ;
  DUMPREG("MMR_01A8",R128_AMCGPIO_EN_REG) ;
  DUMPREG("MMR_01AC",R128_MDGPIO_A_REG) ;
  DUMPREG("MMR_01B0",R128_MDGPIO_EN_REG) ;
  DUMPREG("MMR_01B4",R128_MDGPIO_Y_REG) ;
  DUMPREG("MMR_01C0",R128_MPP_TB_CONFIG) ;
  DUMPREG("MMR_01C4",R128_MPP_TB_STROBE_SEQ) ;
  DUMPREG("MMR_01C8",R128_MPP_GP_CONFIG) ;
  DUMPREG("MMR_01CC",R128_MPP_GP_STROBE_SEQ) ;
  DUMPREG("MMR_0200",R128_CRTC_H_TOTAL_DISP) ;
  DUMPREG("MMR_0204",R128_CRTC_H_SYNC_STRT_WID) ;
  DUMPREG("MMR_0208",R128_CRTC_V_TOTAL_DISP) ;
  DUMPREG("MMR_020C",R128_CRTC_V_SYNC_STRT_WID) ;
  DUMPREG("MMR_0224",R128_CRTC_OFFSET) ;
  DUMPREG("MMR_022C",R128_CRTC_PITCH) ;
  DUMPREG("MMR_0190",R128_VIDEOMUX_CNTL) ;
  DUMPREG("MMR_0400",OV0_Y_X_START) ;
  DUMPREG("MMR_0404",OV0_Y_X_END) ;
  DUMPREG("MMR_0408",OV0_EXCLUSIVE_HORZ) ;
  DUMPREG("MMR_040C",OV0_EXCLUSIVE_VERT) ;
  DUMPREG("MMR_0410",OV0_REG_LOAD_CNTL) ;
  DUMPREG("MMR_0420",OV0_SCALE_CNTL) ;
  DUMPREG("MMR_0424",OV0_V_INC) ;
  DUMPREG("MMR_0428",OV0_P1_V_ACCUM_INIT) ;
  DUMPREG("MMR_042C",OV0_P23_V_ACCUM_INIT) ;
  DUMPREG("MMR_0430",OV0_P1_BLANK_LINES_AT_TOP) ;
  DUMPREG("MMR_0434",OV0_P23_BLANK_LINES_AT_TOP) ;
  DUMPREG("MMR_0440",OV0_VID_BUF0_BASE_ADRS) ;
  DUMPREG("MMR_0444",OV0_VID_BUF1_BASE_ADRS) ;
  DUMPREG("MMR_0448",OV0_VID_BUF2_BASE_ADRS) ;
  DUMPREG("MMR_044C",OV0_VID_BUF3_BASE_ADRS) ;
  DUMPREG("MMR_0450",OV0_VID_BUF4_BASE_ADRS) ;
  DUMPREG("MMR_0454",OV0_VID_BUF5_BASE_ADRS) ;
  DUMPREG("MMR_0460",OV0_VID_BUF_PITCH0_VALUE) ;
  DUMPREG("MMR_0464",OV0_VID_BUF_PITCH1_VALUE) ;
  DUMPREG("MMR_0470",OV0_AUTO_FLIP_CNTRL) ;
  DUMPREG("MMR_0474",OV0_DEINTERLACE_PATTERN) ;
  DUMPREG("MMR_0480",OV0_H_INC) ;
  DUMPREG("MMR_0484",OV0_STEP_BY) ;
  DUMPREG("MMR_0488",OV0_P1_H_ACCUM_INIT) ;
  DUMPREG("MMR_048C",OV0_P23_H_ACCUM_INIT) ;
  DUMPREG("MMR_0494",OV0_P1_X_START_END) ;
  DUMPREG("MMR_0498",OV0_P2_X_START_END) ;
  DUMPREG("MMR_049C",OV0_P3_X_START_END) ;
  DUMPREG("MMR_04A0",OV0_FILTER_CNTL) ;
  DUMPREG("MMR_04B0",OV0_FOUR_TAP_COEF_0) ;
  DUMPREG("MMR_04B4",OV0_FOUR_TAP_COEF_1) ;
  DUMPREG("MMR_04B8",OV0_FOUR_TAP_COEF_2) ;
  DUMPREG("MMR_04BC",OV0_FOUR_TAP_COEF_3) ;
  DUMPREG("MMR_04C0",OV0_FOUR_TAP_COEF_4) ;
  DUMPREG("MMR_04E0",OV0_COLOUR_CNTL) ;
  DUMPREG("MMR_04E4",OV0_VIDEO_KEY_CLR) ;
  DUMPREG("MMR_04E8",OV0_VIDEO_KEY_MSK) ;
  DUMPREG("MMR_04EC",OV0_GRAPHICS_KEY_CLR) ;
  DUMPREG("MMR_04F0",OV0_GRAPHICS_KEY_MSK) ;
  DUMPREG("MMR_04F4",OV0_KEY_CNTL) ;
  DUMPREG("MMR_04F8",OV0_TEST) ;
  DUMPREG("MMR_0900",VID_BUFFER_CONTROL) ;
  DUMPREG("MMR_0908",CAP_INT_CNTL) ;
  DUMPREG("MMR_090C",CAP_INT_STATUS) ;
  DUMPREG("MMR_0920",CAP0_BUF0_OFFSET) ;
  DUMPREG("MMR_0924",CAP0_BUF1_OFFSET) ;
  DUMPREG("MMR_0928",CAP0_BUF0_EVEN_OFFSET) ;
  DUMPREG("MMR_092C",CAP0_BUF1_EVEN_OFFSET) ;
  DUMPREG("MMR_0930",CAP0_BUF_PITCH) ;
  DUMPREG("MMR_0934",CAP0_V_WINDOW) ;
  DUMPREG("MMR_0938",CAP0_H_WINDOW) ;
  DUMPREG("MMR_093C",CAP0_VBI_ODD_OFFSET) ;
  DUMPREG("MMR_0940",CAP0_VBI_EVEN_OFFSET) ;
  DUMPREG("MMR_0944",CAP0_VBI_V_WINDOW) ;
  DUMPREG("MMR_0948",CAP0_VBI_H_WINDOW) ;
  DUMPREG("MMR_094C",CAP0_PORT_MODE_CNTL) ;
  DUMPREG("MMR_0950",CAP0_TRIG_CNTL) ;
  DUMPREG("MMR_0954",CAP0_DEBUG) ;
  DUMPREG("MMR_0958",CAP0_CONFIG) ;
  DUMPREG("MMR_095C",CAP0_ANC_ODD_OFFSET) ;
  DUMPREG("MMR_0960",CAP0_ANC_EVEN_OFFSET) ;
  DUMPREG("MMR_0964",CAP0_ANC_H_WINDOW) ;
  DUMPREG("MMR_0968",CAP0_VIDEO_SYNC_TEST) ;
  DUMPREG("MMR_096C",CAP0_ONESHOT_BUF_OFFSET) ;
  DUMPREG("MMR_0970",CAP0_BUF_STATUS) ;
  DUMPREG("MMR_0978",CAP0_DWNSC_XRATIO) ;
  DUMPREG("MMR_097C",CAP0_XSHARPNESS) ;
  DUMPREG("MMR_0990",CAP1_BUF0_OFFSET) ;
  DUMPREG("MMR_0994",CAP1_BUF1_OFFSET) ;
  DUMPREG("MMR_0998",CAP1_BUF0_EVEN_OFFSET) ;
  DUMPREG("MMR_099C",CAP1_BUF1_EVEN_OFFSET) ;
  DUMPREG("MMR_09A0",CAP1_BUF_PITCH) ;
  DUMPREG("MMR_09A4",CAP1_V_WINDOW) ;
  DUMPREG("MMR_09A8",CAP1_H_WINDOW) ;
  DUMPREG("MMR_09AC",CAP1_VBI_ODD_OFFSET) ;
  DUMPREG("MMR_09B0",CAP1_VBI_EVEN_OFFSET) ;
  DUMPREG("MMR_09B4",CAP1_VBI_V_WINDOW) ;
  DUMPREG("MMR_09B8",CAP1_VBI_H_WINDOW) ;
  DUMPREG("MMR_09BC",CAP1_PORT_MODE_CNTL) ;
  DUMPREG("MMR_09C0",CAP1_TRIG_CNTL) ;
  DUMPREG("MMR_09C4",CAP1_DEBUG) ;
  DUMPREG("MMR_09C8",CAP1_CONFIG) ;
  DUMPREG("MMR_09CC",CAP1_ANC_ODD_OFFSET) ;
  DUMPREG("MMR_09D0",CAP1_ANC_EVEN_OFFSET) ;
  DUMPREG("MMR_09D4",CAP1_ANC_H_WINDOW) ;
  DUMPREG("MMR_09D8",CAP1_VIDEO_SYNC_TEST) ;
  DUMPREG("MMR_09DC",CAP1_ONESHOT_BUF_OFFSET) ;
  DUMPREG("MMR_09E0",CAP1_BUF_STATUS) ;
  DUMPREG("MMR_09E8",CAP1_DWNSC_XRATIO) ;
  DUMPREG("MMR_09EC",CAP1_XSHARPNESS) ;
  DUMPREG("MMR_0F02",R128_DEVICE_ID) ;
  DUMPREG("MMR_0F08",R128_REVISION_ID) ;
  tmp = R128_CLOCK_CNTL_INDEX0 ;
  for ( i=0 ; i<20 ; i++ ) {
    if (i%4==0) fprintf(stderr,"%s: PLL[%02X..%02X] =",gatos.ati.ident,i,i+3) ;
    R128_CLOCK_CNTL_INDEX0 = i ; fprintf(stderr," %08X",R128_CLOCK_CNTL_DATA) ;
    if ((i+1)%4==0) fprintf(stderr,"\n") ; }
  R128_CLOCK_CNTL_INDEX0 = tmp ;
}
