/**
 * \file      test-mac-hmac-sha256.c
 *
 * \brief     The HMAC-SHA-2-256 message authentication code unit-tests
 *
 * \copyright 2022 Joseph Benden <joe@benden.us>
 *
 * \license   GPL-2.0-OR-LATER
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <stdlib.h>

#include "aircrack-ng/crypto/crypto.h"
#include "tests.h"

// clang-format off
/*
 * RFC 6234 test vectors
 */
struct {
	uint8_t         key[136];
	size_t          key_len;
	uint8_t         data[164];
	size_t          data_len;
	uint8_t         digest[DIGEST_SHA256_MAC_LEN];
} sha256_tests[] = {
	{
		{0x0b, 0x0b, 0x0b, 0x0b,
		 0x0b, 0x0b, 0x0b, 0x0b,
		 0x0b, 0x0b, 0x0b, 0x0b,
		 0x0b, 0x0b, 0x0b, 0x0b,
		 0x0b, 0x0b, 0x0b, 0x0b},
		20,
		"Hi There",
		8,
		{0xB0, 0x34, 0x4C, 0x61, 0xD8, 0xDB, 0x38, 0x53,
		 0x5C, 0xA8, 0xAF, 0xCE, 0xAF, 0x0B, 0xF1, 0x2B,
		 0x88, 0x1D, 0xC2, 0x00, 0xC9, 0x83, 0x3D, 0xA7,
		 0x26, 0xE9, 0x37, 0x6C, 0x2E, 0x32, 0xCF, 0xF7}
	},
	{
		"Jefe",
		4,
		"what do ya want for nothing?",
		28,
		{0x5B, 0xDC, 0xC1, 0x46, 0xBF, 0x60, 0x75, 0x4E,
		 0x6A, 0x04, 0x24, 0x26, 0x08, 0x95, 0x75, 0xC7,
		 0x5A, 0x00, 0x3F, 0x08, 0x9D, 0x27, 0x39, 0x83,
		 0x9D, 0xEC, 0x58, 0xB9, 0x64, 0xEC, 0x38, 0x43}
#ifdef EXPENSIVE_TESTS
	},
	{
		{0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa},
		20,
		{0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
		 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
		 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
		 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
		 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
		 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
		 0xdd, 0xdd},
		50,
		{0x77, 0x3E, 0xA9, 0x1E, 0x36, 0x80, 0x0E, 0x46,
		 0x85, 0x4D, 0xB8, 0xEB, 0xD0, 0x91, 0x81, 0xA7,
		 0x29, 0x59, 0x09, 0x8B, 0x3E, 0xF8, 0xC1, 0x22,
		 0xD9, 0x63, 0x55, 0x14, 0xCE, 0xD5, 0x65, 0xFE}
	},
	{
		{0x01, 0x02, 0x03, 0x04,
		 0x05, 0x06, 0x07, 0x08,
		 0x09, 0x0a, 0x0b, 0x0c,
		 0x0d, 0x0e, 0x0f, 0x10,
		 0x11, 0x12, 0x13, 0x14,
		 0x15, 0x16, 0x17, 0x18,
		 0x19},
		25,
		{0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
		 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
		 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
		 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
		 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
		 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
		 0xcd, 0xcd},
		50,
		{0x82, 0x55, 0x8A, 0x38, 0x9A, 0x44, 0x3C, 0x0E,
		 0xA4, 0xCC, 0x81, 0x98, 0x99, 0xF2, 0x08, 0x3A,
		 0x85, 0xF0, 0xFA, 0xA3, 0xE5, 0x78, 0xF8, 0x07,
		 0x7A, 0x2E, 0x3F, 0xF4, 0x67, 0x29, 0x66, 0x5B}
	},
	{
		{0x0c, 0x0c, 0x0c, 0x0c,
		 0x0c, 0x0c, 0x0c, 0x0c,
		 0x0c, 0x0c, 0x0c, 0x0c,
		 0x0c, 0x0c, 0x0c, 0x0c,
		 0x0c, 0x0c, 0x0c, 0x0c},
		20,
		"Test With Truncation",
		20,
		{0xA3, 0xB6, 0x16, 0x74, 0x73, 0x10, 0x0E, 0xE0,
		 0x6E, 0x0C, 0x79, 0x6C, 0x29, 0x55, 0x55, 0x2B,
		 0XFA, 0X6F, 0X7C, 0X0A, 0X6A, 0X8A, 0XEF, 0X8B,
		 0X93, 0XF8, 0X60, 0XAA, 0XB0, 0XCD, 0X20, 0XC5}
	},
	{
		{0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, //32
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, //64
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
 		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
 		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
 		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, //96
 		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
 		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
 		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
 		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, //128
		 0xaa, 0xaa, 0xaa}, //131
		131,
		"Test Using Larger Than Block-Size Key - Hash Key First",
		54,
		{0x60, 0xE4, 0x31, 0x59, 0x1E, 0xE0, 0xB6, 0x7F,
		 0x0D, 0x8A, 0x26, 0xAA, 0xCB, 0xF5, 0xB7, 0x7F,
		 0x8E, 0x0B, 0xC6, 0x21, 0x37, 0x28, 0xC5, 0x14,
		 0x05, 0x46, 0x04, 0x0F, 0x0E, 0xE3, 0x7F, 0x54}
	},
	{
		{0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, //32
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, //64
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
 		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
 		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
 		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, //96
 		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
 		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
 		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
 		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, //128
		 0xaa, 0xaa, 0xaa}, //131
		131,
		"This is a test using a larger than block-size key and "
		"a larger than block-size data. The key needs to be hashed "
		"before being used by the HMAC algorithm.",
		152,
		{0x9B, 0x09, 0xFF, 0xA7, 0x1B, 0x94, 0x2F, 0xCB,
		 0x27, 0x63, 0x5F, 0xBC, 0xD5, 0xB0, 0xE9, 0x44,
		 0xBF, 0xDC, 0x63, 0x64, 0x4F, 0x07, 0x13, 0x93,
		 0x8A, 0x7F, 0x51, 0x53, 0x5C, 0x3A, 0x35, 0xE2}
#endif
	}
};
// clang-format on

STATIC_ASSERT(ArrayCount(sha256_tests) >= 2u, ensure_at_least_2_entries);

int main(int argc, char ** argv)
{
	size_t i;
	int error = 0;
	uint8_t sha256sum[DIGEST_SHA256_MAC_LEN];

	(void) argc;

	for (i = 0; i < ArrayCount(sha256_tests); i++)
	{
		error |= MAC_HMAC_SHA256(sha256_tests[i].key_len,
								 sha256_tests[i].key,
								 sha256_tests[i].data_len,
								 sha256_tests[i].data,
								 sha256sum);
		error |= test(
			sha256sum, sha256_tests[i].digest, DIGEST_SHA256_MAC_LEN, argv[0]);
	}

	for (i = 0; i < ArrayCount(sha256_tests); i++)
	{
		error |= MAC_HMAC_SHA256_Vector(
			sha256_tests[i].key_len,
			sha256_tests[i].key,
			1,
			(const uint8_t * []){(uint8_t *) &sha256_tests[i].data},
			&sha256_tests[i].data_len,
			sha256sum);
		error |= test(
			sha256sum, sha256_tests[i].digest, DIGEST_SHA256_MAC_LEN, argv[0]);
	}

	return error;
}
